import { createConfig } from "@arcjet/rollup-config";
import fs from "node:fs/promises";

function generateJs(wasm) {
  const disclaimer = `
/**
 * This file contains an Arcjet Wasm binary inlined as a base64
 * [Data URL](https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/Data_URLs)
 * with the application/wasm MIME type.
 *
 * This was chosen to save on storage space over inlining the file directly as
 * a Uint8Array, which would take up ~3x the space of the Wasm file. See
 * https://blobfolio.com/2019/better-binary-batter-mixing-base64-and-uint8array/
 * for more details.
 *
 * It is then decoded into an ArrayBuffer to be used directly via WebAssembly's
 * \`compile()\` function in our entry point file.
 *
 * This is all done to avoid trying to read or bundle the Wasm asset in various
 * ways based on the platform or bundler a user is targeting. One example being
 * that Next.js requires special \`asyncWebAssembly\` webpack config to load our
 * Wasm file if we don't do this.
 *
 * In the future, we hope to do away with this workaround when all bundlers
 * properly support consistent asset bundling techniques.
 */
`;

  return `// @generated by wasm2module - DO NOT EDIT
/* eslint-disable */
// @ts-nocheck
${disclaimer}
const wasmBase64 = "data:application/wasm;base64,${wasm.toString("base64")}";
/**
 * Returns a WebAssembly.Module for an Arcjet Wasm binary, decoded from a base64
 * Data URL.
 */
// TODO: Switch back to top-level await when our platforms all support it
export async function wasm() {
  // This uses fetch to decode the wasm data url, but disabling cache so files
  // larger than 2mb don't fail to parse in the Next.js App Router
  const wasmDecode = await fetch(wasmBase64, { cache: "no-store" });
  const buf = await wasmDecode.arrayBuffer();
  // And then we return it as a WebAssembly.Module
  return WebAssembly.compile(buf);
}
`;
}

function wasmToModule() {
  const idToWasmPath = new Map();

  return {
    name: "base64-wasm",
    resolveId(source) {
      if (source.endsWith(".wasm?js")) {
        // Slice off the `?js` to make it a valid path
        const filepath = source.slice(0, -3);
        // Create a "virtual module", prefixed with `\0` as per the Rollup docs,
        // for our replacement import
        const id = `\0${filepath.replace(/\.wasm$/, ".js")}`;
        // Store the actual Wasm path against the virtual module ID.
        idToWasmPath.set(id, filepath);
        return id;
      }

      return null;
    },
    async load(id) {
      const wasmPath = idToWasmPath.get(id);
      // If we resolved this `id` during the `resolveId` phase, generate the
      // JavaScript file with the base64 Wasm and loading helper
      if (wasmPath) {
        const wasm = await fs.readFile(wasmPath);
        return generateJs(wasm);
      }

      return null;
    },
  };
}

export default createConfig(import.meta.url, {
  plugins: [wasmToModule()],
});
