//
//  WhisperView.swift
//  cookingmama
//
//  Created by Andrew Park on 2/17/24.
//

import SwiftUI
import WhisperKit
#if canImport(UIKit)
    import UIKit
#elseif canImport(AppKit)
    import AppKit
#endif
import AVFoundation

struct MenuItem: Identifiable, Hashable {
    var id = UUID()
    var name: String
    var image: String
}

struct WhisperView: View {
    var cookVM: CookingVM = .init()
    
    @State private var contentText: String = ""

    @State var isRecording: Bool = false

    @State private var modelState: ModelState = .unloaded

//    @State private var messages: [Message] = [
//        .init(
//        role: "system",
//        content: "Hello"
//        content: "Hello"
//    ),
//        .init(
//        role: "user",
//        content: "Hiiiiiii!"
//    )
//    ]
    @State private var messages: [Message] = []
    @State private var audioPlayer: AVAudioPlayer?
    
    @State private var reResponse: RecipeResponse?

    // MARK: ViewssendApiRequest

    var body: some View {
        NavigationStack {
            VStack{
                HStack{
                    Image(.mama)
                        .resizable()
                        .scaledToFit()
                        .frame(width: 64)
                        .background(
                            Image(.mama)
                                .resizable()
                                .scaledToFit()
                                .frame(width: 64)
                                .blur(radius: 32)
                        )
                    Text("Cooking Mama")
                        .font(.extraLargeTitle)
                    
                    Spacer()
                }
                .padding()
                HStack{
                    VStack{
                        if let model = reResponse {
                            ScrollView {
                                VStack(alignment: .leading){
                                    Text("Your Recipe")
                                        .font(.extraLargeTitle)
                                        .bold()
                                    Text("Ingredients: \(model.ingredients)")
                                        .foregroundStyle(.secondary)
                                        .font(.body)
                                    
                                    // List the shopping items
                                    ForEach(model.instacartAndShoppingList, id: \.item) { item in
                                        VStack(alignment: .leading) {
                                            Text(item.item)
                                                .font(.headline)
                                            Divider()
                                            HStack{
                                                Text("Store")
                                                    .bold()
                                                Text("\(item.store)")
                                                    .foregroundStyle(.secondary)
                                                Spacer()
                                            }
                                            HStack{
                                                Text("Price")
                                                    .bold()
                                                Text("\(item.price)")
                                                    .foregroundStyle(.secondary)
                                                Spacer()
                                            }
                                        }
                                        .padding()
                                        .background(.black.opacity(0.25))
                                        .clipShape(.rect(cornerRadius: 12))
                                        .padding(6)
                                    }
                                    
                                    Text(reResponse?.generatedRecipe ?? "")
                                        .font(.title3)
                                }.padding(32)
                            }
                        }else{
                            ContentUnavailableView("Ask for a recipe first", systemImage: "book")
                                .font(.largeTitle)
                            
                        }
                        
                    }
                    //                    Divider()
                    ScrollView {
                        VStack{
                            ForEach(messages) { message in
                                VStack(alignment: message.role == "user" ? .trailing : .leading) {
                                    HStack{
                                        Group{
                                            if message.role == "user" {
                                                Image(systemName: "person.circle.fill")
                                            }else{
                                                Image(systemName: "face.smiling")
                                            }
                                        }
                                        .foregroundStyle(.tertiary)
                                        Text(message.role == "user" ? "User" : "Vision Mama")
                                            .font(.headline)
                                    }
                                    Text(message.content)
                                        .font(.body)
                                }
                                .frame(maxWidth: .infinity, alignment: message.role == "user" ? .trailing : .leading)
                            }
                        }
                        .padding()
                        .padding(32)
                    }
                    
                    
                }
            }
//            .navigationTitle("Cooking Mama")
            .toolbar{
                ToolbarItem(placement: .bottomOrnament) {
                    VStack(alignment: .leading) {
                        HStack {
                            TextField("What's on your mind?", text: $contentText, axis: .vertical)
                                .font(.largeTitle)
                                .frame(maxWidth: 500, minHeight: 100, maxHeight: .infinity)
                                .fixedSize(horizontal: false, vertical: true)
                            
                            Spacer()
                            
                            Button {
                                withAnimation {
//                                    #if !DEBUG
                                    cookVM.toggleRecording(shouldLoop: true)
//                                    #else
//                                    cookVM.isRecording.toggle()
//                                    #endif
                                }
                                
                            } label: {
                                Image(
                                    systemName: !cookVM.isRecording ? "waveform" : "stop.circle.fill"
                                )
                                .font(.extraLargeTitle2)
                                .foregroundStyle(cookVM.isRecording ? .red : .primary)
                                .foregroundColor(modelState != .loaded ? .secondary : .primary)
                            }
                            .buttonBorderShape(.circle)
                        }
                        .padding(.horizontal)
                        
                        
                        Button {
                            Task {
                                await sendApiRequest(with: cookVM.myText)
                            }
                        } label: {
                            HStack{
                                Spacer()
                                HStack{
                                    Image(systemName: "arrow.up")
                                        .symbolVariant(.circle)
                                        .symbolVariant(.fill)
                                        .padding(.trailing)
                                    Text("Send")
                                        
                                }.font(.largeTitle)
                                    .fontWeight(.semibold)
                                    
                                    
                                Spacer()
                            }
                            .frame(width: 500)
                            .padding(32)
                            .clipShape(.rect(cornerRadius: 64))
                        }
                        .buttonStyle(.bordered)
                        .padding(.vertical)
                    }
                    .padding(16)

                }
            }
        }
        .background(.pink.opacity(0.25))
        .onChange(of: contentText, { _, newValue in
            cookVM.myText = newValue
        })
        .onChange(of: cookVM.myText, { _, newValue in
            self.contentText = newValue
        })
        .task {
            cookVM.whisperKit = try? await WhisperKit(
                verbose: true,
                logLevel: .debug
            )
            cookVM.loadModel("base")

            cookVM.fetchModels()
        }
    }
    
    func updateUIWithTranscription(_ transcription: String) {
        let newMessage = Message(role: "user", content: transcription)
        DispatchQueue.main.async {
            self.messages.append(newMessage)
        }
    }
    


    func sendApiRequest(with text: String) async {
        if text.isEmpty {
            return
        }
        print("sending")
        self.audioPlayer?.stop()
            
        if text.lowercased().contains("make me") {
            let foodText = text.lowercased().replacingOccurrences(of: "make me ", with: "")
            print("FOOD TEXT", foodText)
            let jsonData = try? JSONSerialization.data(withJSONObject: ["desiredFood": foodText], options: [])
            contentText = ""
            DispatchQueue.main.async {
                self.messages.append(Message(role: "user", content: text))
            }
            guard let url = URL(string: "https://eohr4l2odp1g8eo.m.pipedream.net") else {
                print("Invalid URL")
                return
            }
            
            var request = URLRequest(url: url)
            request.httpMethod = "POST"
            request.addValue("application/json", forHTTPHeaderField: "Content-Type")
            request.httpBody = jsonData
            
            do {
                print("url session")
                let (data, response) = try await URLSession.shared.upload(for: request, from: jsonData ?? Data())
                guard let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 else {
                    print("Error: Unexpected response")
                    return
                }
                print("httpresponse")
                let decoded = try? JSONDecoder().decode(RecipeResponse.self, from: data)
                print(String(describing: decoded))
                self.reResponse = decoded
                
                let elevenTalk = reResponse?.recipeTranscript(text: foodText)
                print("eleventalk", elevenTalk ?? "no eleven")
                
                Task {
                    do {
                        let url = try await cookVM.elevenApi.textToSpeech(voice_id: "21m00Tcm4TlvDq8ikWAM", text: elevenTalk ?? "")
                        self.audioPlayer = try AVAudioPlayer(contentsOf: url)
                        self.audioPlayer?.play()
                        print("url", url)
                    } catch {
                        // Handle errors
                        print("Error performing text to speech: \(error)")
                    }
                }
                
                DispatchQueue.main.async {
                    self.messages.append(Message(role: "system", content: elevenTalk  ?? ""))
                    cookVM.resetState()
                }

                
            } catch {
                print("Error: \(error.localizedDescription)")
            }
            return
        }


        print("new user message")
        let newUserMessage = Message(role: "user", content: text)
        messages.append(newUserMessage)

        // Specify the URL for the OpenAI ChatGPT API
        let url = URL(string: "https://api.openai.com/v1/chat/completions")!
        var request = URLRequest(url: url)
        request.httpMethod = "POST"
        request.addValue("application/json", forHTTPHeaderField: "Content-Type")
        request.addValue("application/json", forHTTPHeaderField: "Accept")

        // Use your OpenAI API key
        let openAIKey = ""
        request.addValue("Bearer \(openAIKey)", forHTTPHeaderField: "Authorization")

        // Prepare the messages array for the requestBody including all history
        let messagesRequestBody = messages.map { message -> [String: String] in
            ["role": message.role, "content": message.content]
        }

        // Create the requestBody with the required model, messages, and any other parameters like temperature
        let requestBody = [
            "model": "gpt-3.5-turbo",
            "messages": messagesRequestBody,
            "temperature": 0.7,
            "max_tokens": 100
        ] as [String: Any]

        do {
            request.httpBody = try JSONSerialization.data(withJSONObject: requestBody, options: [])
            let (data, response) = try await URLSession.shared.data(for: request)
            // let (data, response) = try await URLSession.shared.data(for: request)

            guard let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 else {
                print("Error: Unexpected response")
                return
            }
            print("httpResponse", httpResponse)

            guard let jsonResponse = try? JSONSerialization.jsonObject(with: data, options: []) as? [String: Any],
                  let choices = jsonResponse["choices"] as? [[String: Any]],
                  let firstChoice = choices.first,
                  let messageContent = firstChoice["message"] as? [String: String]
            else {
                print("Error: Unable to parse JSON")
                return
            }

            // UI updates must be performed on the main thread
            DispatchQueue.main.async {
                self.messages.append(Message(role: messageContent["role"] ?? "", content: messageContent["content"] ?? ""))
                cookVM.resetState()
                self.contentText = ""
            }

            Task {
                do {
                    let url = try await cookVM.elevenApi.textToSpeech(voice_id: "21m00Tcm4TlvDq8ikWAM", text: messageContent["content"] ?? "")
                    self.audioPlayer = try AVAudioPlayer(contentsOf: url)
                    self.audioPlayer?.play()
                    print("url", url)
                } catch {
                    // Handle errors
                    print("Error performing text to speech: \(error)")
                }
            }
        } catch {
            print("Error: \(error.localizedDescription)")
        }
    }

    // MARK: - Transcription

//    var transcriptionView: some View {
//        VStack {
//            
//                HStack(spacing: 1) {
//                    let startIndex = max(cookVM.bufferEnergy.count - 300, 0)
//                    ForEach(Array(cookVM.bufferEnergy.enumerated())[startIndex...], id: \.element) { _, energy in
//                        ZStack {
//                            RoundedRectangle(cornerRadius: 2)
//                                .frame(width: 2, height: CGFloat(energy) * 24)
//                        }
//                        .frame(maxHeight: 24)
//                        .background(energy > Float(cookVM.silenceThreshold) ? Color.green.opacity(0.2) : Color.red.opacity(0.2))
//                    }
//                }
//            
//            .defaultScrollAnchor(.trailing)
//            .frame(height: 32)
//            .scrollIndicators(.never)
//            
//                VStack(alignment: .leading) {
//                    ForEach(Array(cookVM.confirmedSegments.enumerated()), id: \.element) { _, segment in
//                        Text(segment.text)
//                            .font(.headline)
//                            .fontWeight(.bold)
//                            .tint(.green)
//                            .multilineTextAlignment(.leading)
//                            .frame(maxWidth: .infinity, alignment: .leading)
//                    }
//                    ForEach(Array(cookVM.unconfirmedSegments.enumerated()), id: \.element) { _, segment in
//                        Text(segment.text)
//                            .font(.headline)
//                            .fontWeight(.light)
//                            .multilineTextAlignment(.leading)
//                            .frame(maxWidth: .infinity, alignment: .leading)
//                    }
//                }
//            
//            .frame(maxWidth: .infinity)
//            .defaultScrollAnchor(.bottom)
//            .padding()
//        }
//    }


    struct InfoButton: View {
        var infoText: String
        @State private var showInfo = false

        init(_ infoText: String) {
            self.infoText = infoText
        }

        var body: some View {
            Button(action: {
                self.showInfo = true
            }) {
                Image(systemName: "info.circle")
                    .foregroundColor(.blue)
            }
            .popover(isPresented: $showInfo) {
                Text(infoText)
                    .padding()
            }
            .buttonStyle(BorderlessButtonStyle())
        }
    }
}

#Preview {
    WhisperView()
    #if os(macOS)
        .frame(width: 800, height: 500)
    #endif
}
