"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getProfileArguments = exports.runTests = void 0;
const cp = require("child_process");
const path = require("path");
const download_1 = require("./download");
const util_1 = require("./util");
/**
 * Run VS Code extension test
 *
 * @returns The exit code of the command to launch VS Code extension test
 */
async function runTests(options) {
    if (!options.vscodeExecutablePath) {
        options.vscodeExecutablePath = await download_1.downloadAndUnzipVSCode(options);
    }
    let args = [
        // https://github.com/microsoft/vscode/issues/84238
        '--no-sandbox',
        // https://github.com/microsoft/vscode-test/issues/221
        '--disable-gpu-sandbox',
        // https://github.com/microsoft/vscode-test/issues/120
        '--disable-updates',
        '--skip-welcome',
        '--skip-release-notes',
        '--disable-workspace-trust',
        '--extensionTestsPath=' + options.extensionTestsPath,
    ];
    if (Array.isArray(options.extensionDevelopmentPath)) {
        args.push(...options.extensionDevelopmentPath.map((devPath) => `--extensionDevelopmentPath=${devPath}`));
    }
    else {
        args.push(`--extensionDevelopmentPath=${options.extensionDevelopmentPath}`);
    }
    if (options.launchArgs) {
        args = options.launchArgs.concat(args);
    }
    if (!options.reuseMachineInstall) {
        args.push(...getProfileArguments(args));
    }
    return innerRunTests(options.vscodeExecutablePath, args, options.extensionTestsEnv);
}
exports.runTests = runTests;
/** Adds the extensions and user data dir to the arguments for the VS Code CLI */
function getProfileArguments(args) {
    const out = [];
    if (!hasArg('extensions-dir', args)) {
        out.push(`--extensions-dir=${path.join(download_1.defaultCachePath, 'extensions')}`);
    }
    if (!hasArg('user-data-dir', args)) {
        out.push(`--user-data-dir=${path.join(download_1.defaultCachePath, 'user-data')}`);
    }
    return out;
}
exports.getProfileArguments = getProfileArguments;
function hasArg(argName, argList) {
    return argList.some((a) => a === `--${argName}` || a.startsWith(`--${argName}=`));
}
const SIGINT = 'SIGINT';
async function innerRunTests(executable, args, testRunnerEnv) {
    const fullEnv = Object.assign({}, process.env, testRunnerEnv);
    const cmd = cp.spawn(executable, args, { env: fullEnv });
    let exitRequested = false;
    const ctrlc1 = () => {
        process.removeListener(SIGINT, ctrlc1);
        process.on(SIGINT, ctrlc2);
        console.log('Closing VS Code gracefully. Press Ctrl+C to force close.');
        exitRequested = true;
        cmd.kill(SIGINT); // this should cause the returned promise to resolve
    };
    const ctrlc2 = () => {
        console.log('Closing VS Code forcefully.');
        process.removeListener(SIGINT, ctrlc2);
        exitRequested = true;
        util_1.killTree(cmd.pid, true);
    };
    const prom = new Promise((resolve, reject) => {
        if (cmd.pid) {
            process.on(SIGINT, ctrlc1);
        }
        cmd.stdout.on('data', (d) => process.stdout.write(d));
        cmd.stderr.on('data', (d) => process.stderr.write(d));
        cmd.on('error', function (data) {
            console.log('Test error: ' + data.toString());
        });
        let finished = false;
        function onProcessClosed(code, signal) {
            if (finished) {
                return;
            }
            finished = true;
            console.log(`Exit code:   ${code !== null && code !== void 0 ? code : signal}`);
            // fix: on windows, it seems like these descriptors can linger for an
            // indeterminate amount of time, causing the process to hang.
            cmd.stdout.destroy();
            cmd.stderr.destroy();
            if (code === null) {
                reject(signal);
            }
            else if (code !== 0) {
                reject('Failed');
            }
            else {
                console.log('Done\n');
                resolve(code !== null && code !== void 0 ? code : -1);
            }
        }
        cmd.on('close', onProcessClosed);
        cmd.on('exit', onProcessClosed);
    });
    let code;
    try {
        code = await prom;
    }
    finally {
        process.removeListener(SIGINT, ctrlc1);
        process.removeListener(SIGINT, ctrlc2);
    }
    // exit immediately if we handled a SIGINT and no one else did
    if (exitRequested && process.listenerCount(SIGINT) === 0) {
        process.exit(1);
    }
    return code;
}
