import { DownloadVersion, DownloadPlatform } from './download';
import { ProgressReporter } from './progress';
export interface TestOptions {
    /**
     * The VS Code executable path used for testing.
     *
     * If not passed, will use `options.version` to download a copy of VS Code for testing.
     * If `version` is not specified either, will download and use latest stable release.
     */
    vscodeExecutablePath?: string;
    /**
     * The VS Code version to download. Valid versions are:
     * - `'stable'`
     * - `'insiders'`
     * - `'1.32.0'`, `'1.31.1'`, etc
     *
     * Defaults to `stable`, which is latest stable version.
     *
     * *If a local copy exists at `.vscode-test/vscode-<VERSION>`, skip download.*
     */
    version?: DownloadVersion;
    /**
     * The VS Code platform to download. If not specified, it defaults to the
     * current platform.
     *
     * Possible values are:
     * 	- `win32-x64-archive`
     * 	- `win32-arm64-archive		`
     * 	- `darwin`
     * 	- `darwin-arm64`
     * 	- `linux-x64`
     * 	- `linux-arm64`
     * 	- `linux-armhf`
     */
    platform?: DownloadPlatform;
    /**
     * Whether VS Code should be launched using default settings and extensions
     * installed on this machine. If `false`, then separate directories will be
     * used inside the `.vscode-test` folder within the project.
     *
     * Defaults to `false`.
     */
    reuseMachineInstall?: boolean;
    /**
     * Absolute path to the extension root. Passed to `--extensionDevelopmentPath`.
     * Must include a `package.json` Extension Manifest.
     */
    extensionDevelopmentPath: string | string[];
    /**
     * Absolute path to the extension tests runner. Passed to `--extensionTestsPath`.
     * Can be either a file path or a directory path that contains an `index.js`.
     * Must export a `run` function of the following signature:
     *
     * ```ts
     * function run(): Promise<void>;
     * ```
     *
     * When running the extension test, the Extension Development Host will call this function
     * that runs the test suite. This function should throws an error if any test fails.
     *
     * The first argument is the path to the file specified in `extensionTestsPath`.
     *
     */
    extensionTestsPath: string;
    /**
     * Environment variables being passed to the extension test script.
     */
    extensionTestsEnv?: {
        [key: string]: string | undefined;
    };
    /**
     * A list of launch arguments passed to VS Code executable, in addition to `--extensionDevelopmentPath`
     * and `--extensionTestsPath` which are provided by `extensionDevelopmentPath` and `extensionTestsPath`
     * options.
     *
     * If the first argument is a path to a file/folder/workspace, the launched VS Code instance
     * will open it.
     *
     * See `code --help` for possible arguments.
     */
    launchArgs?: string[];
    /**
     * Progress reporter to use while VS Code is downloaded. Defaults to a
     * console reporter. A {@link SilentReporter} is also available, and you
     * may implement your own.
     */
    reporter?: ProgressReporter;
    /**
     * Whether the downloaded zip should be synchronously extracted. Should be
     * omitted unless you're experiencing issues installing VS Code versions.
     */
    extractSync?: boolean;
    /**
     * Number of milliseconds after which to time out if no data is received from
     * the remote when downloading VS Code. Note that this is an 'idle' timeout
     * and does not enforce the total time VS Code may take to download.
     */
    timeout?: number;
}
/**
 * Run VS Code extension test
 *
 * @returns The exit code of the command to launch VS Code extension test
 */
export declare function runTests(options: TestOptions): Promise<number>;
/** Adds the extensions and user data dir to the arguments for the VS Code CLI */
export declare function getProfileArguments(args: readonly string[]): string[];
