"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeoutError = exports.TimeoutController = exports.getJSON = exports.getStream = void 0;
const https = require("https");
const util_1 = require("./util");
async function getStream(api, timeout) {
    const ctrl = new TimeoutController(timeout);
    return new Promise((resolve, reject) => {
        ctrl.signal.addEventListener('abort', () => {
            reject(new TimeoutError(timeout));
            req.destroy();
        });
        const req = https.get(api, util_1.urlToOptions(api), (res) => resolve(res)).on('error', reject);
    }).finally(() => ctrl.dispose());
}
exports.getStream = getStream;
async function getJSON(api, timeout) {
    const ctrl = new TimeoutController(timeout);
    return new Promise((resolve, reject) => {
        ctrl.signal.addEventListener('abort', () => {
            reject(new TimeoutError(timeout));
            req.destroy();
        });
        const req = https
            .get(api, util_1.urlToOptions(api), (res) => {
            if (res.statusCode !== 200) {
                reject('Failed to get JSON');
            }
            let data = '';
            res.on('data', (chunk) => {
                ctrl.touch();
                data += chunk;
            });
            res.on('end', () => {
                ctrl.dispose();
                try {
                    const jsonData = JSON.parse(data);
                    resolve(jsonData);
                }
                catch (err) {
                    console.error(`Failed to parse response from ${api} as JSON`);
                    reject(err);
                }
            });
            res.on('error', reject);
        })
            .on('error', reject);
    }).finally(() => ctrl.dispose());
}
exports.getJSON = getJSON;
class TimeoutController {
    constructor(timeout) {
        this.timeout = timeout;
        this.ctrl = new AbortController();
        this.reject = () => {
            this.ctrl.abort();
        };
        this.handle = setTimeout(this.reject, timeout);
    }
    get signal() {
        return this.ctrl.signal;
    }
    touch() {
        clearTimeout(this.handle);
        this.handle = setTimeout(this.reject, this.timeout);
    }
    dispose() {
        clearTimeout(this.handle);
    }
}
exports.TimeoutController = TimeoutController;
class TimeoutError extends Error {
    constructor(duration) {
        super(`@vscode/test-electron request timeout out after ${duration}ms`);
    }
}
exports.TimeoutError = TimeoutError;
