"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConsoleReporter = exports.SilentReporter = exports.ProgressReportStage = void 0;
/** Stages of progress while downloading VS Code */
var ProgressReportStage;
(function (ProgressReportStage) {
    /** Initial fetch of the latest version if not explicitly given */
    ProgressReportStage["FetchingVersion"] = "fetchingVersion";
    /** Always fired when the version is determined. */
    ProgressReportStage["ResolvedVersion"] = "resolvedVersion";
    /** Fired before fetching info about the latest Insiders version, when requesting insiders builds */
    ProgressReportStage["FetchingInsidersMetadata"] = "fetchingInsidersMetadata";
    /** Fired if the current Insiders is out of date */
    ProgressReportStage["ReplacingOldInsiders"] = "replacingOldInsiders";
    /** Fired when an existing install is found which does not require a download */
    ProgressReportStage["FoundMatchingInstall"] = "foundMatchingInstall";
    /** Fired before the URL to the download zip or tarball is looked up */
    ProgressReportStage["ResolvingCDNLocation"] = "resolvingCDNLocation";
    /** Fired continuously while a download happens */
    ProgressReportStage["Downloading"] = "downloading";
    /** Fired when the command is issued to do a synchronous extraction. May not fire depending on the platform and options. */
    ProgressReportStage["ExtractingSynchonrously"] = "extractingSynchonrously";
    /** Fired when the download fails and a retry will be attempted */
    ProgressReportStage["Retrying"] = "retrying";
    /** Fired after folder is downloaded and unzipped */
    ProgressReportStage["NewInstallComplete"] = "newInstallComplete";
})(ProgressReportStage = exports.ProgressReportStage || (exports.ProgressReportStage = {}));
/** Silent progress reporter */
class SilentReporter {
    report() {
        // no-op
    }
    error() {
        // no-op
    }
}
exports.SilentReporter = SilentReporter;
/** Default progress reporter that logs VS Code download progress to console */
class ConsoleReporter {
    constructor(showDownloadProgress) {
        this.showDownloadProgress = showDownloadProgress;
    }
    report(report) {
        switch (report.stage) {
            case ProgressReportStage.ResolvedVersion:
                this.version = report.version;
                break;
            case ProgressReportStage.ReplacingOldInsiders:
                console.log(`Removing outdated Insiders at ${report.downloadedPath} and re-downloading.`);
                console.log(`Old: ${report.oldHash} | ${report.oldDate.toISOString()}`);
                console.log(`New: ${report.newHash} | ${report.newDate.toISOString()}`);
                break;
            case ProgressReportStage.FoundMatchingInstall:
                console.log(`Found existing install in ${report.downloadedPath}. Skipping download`);
                break;
            case ProgressReportStage.ResolvingCDNLocation:
                console.log(`Downloading VS Code ${this.version} from ${report.url}`);
                break;
            case ProgressReportStage.Downloading:
                if (!this.showDownloadProgress && report.bytesSoFar === 0) {
                    console.log(`Downloading VS Code (${report.totalBytes}B)`);
                }
                else if (!this.downloadReport) {
                    this.downloadReport = { timeout: setTimeout(() => this.reportDownload(), 100), report };
                }
                else {
                    this.downloadReport.report = report;
                }
                break;
            case ProgressReportStage.Retrying:
                this.flushDownloadReport();
                console.log(`Error downloading, retrying (attempt ${report.attempt} of ${report.totalAttempts}): ${report.error.message}`);
                break;
            case ProgressReportStage.NewInstallComplete:
                this.flushDownloadReport();
                console.log(`Downloaded VS Code into ${report.downloadedPath}`);
                break;
        }
    }
    error(err) {
        console.error(err);
    }
    flushDownloadReport() {
        if (this.showDownloadProgress) {
            this.reportDownload();
            console.log('');
        }
    }
    reportDownload() {
        if (!this.downloadReport) {
            return;
        }
        const { totalBytes, bytesSoFar } = this.downloadReport.report;
        this.downloadReport = undefined;
        const percent = Math.max(0, Math.min(1, bytesSoFar / totalBytes));
        const progressBarSize = 30;
        const barTicks = Math.floor(percent * progressBarSize);
        const progressBar = '='.repeat(barTicks) + '-'.repeat(progressBarSize - barTicks);
        process.stdout.write(`\x1b[G\x1b[0KDownloading VS Code [${progressBar}] ${(percent * 100).toFixed()}%`);
    }
}
exports.ConsoleReporter = ConsoleReporter;
