"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const vitest_1 = require("vitest");
const download_1 = require("./download");
const progress_1 = require("./progress");
const util_1 = require("./util");
const platforms = [
    'darwin',
    'darwin-arm64',
    'win32-x64-archive',
    'win32-arm64-archive',
    'linux-x64',
    'linux-arm64',
    'linux-armhf',
];
vitest_1.describe('sane downloads', () => {
    const testTempDir = path_1.join(os_1.tmpdir(), 'vscode-test-download');
    vitest_1.beforeAll(async () => {
        await fs_1.promises.mkdir(testTempDir, { recursive: true });
    });
    for (const quality of ['insiders', 'stable']) {
        for (const platform of platforms) {
            vitest_1.test.concurrent(`${quality}/${platform}`, async () => {
                const location = await download_1.downloadAndUnzipVSCode({
                    platform,
                    version: quality,
                    cachePath: testTempDir,
                    reporter: new progress_1.SilentReporter(),
                });
                if (!fs_1.existsSync(location)) {
                    throw new Error(`expected ${location} to exist for ${platform}`);
                }
                const exePath = util_1.resolveCliPathFromVSCodeExecutablePath(location, platform);
                if (!fs_1.existsSync(exePath)) {
                    throw new Error(`expected ${exePath} to from ${location}`);
                }
                if (platform === util_1.systemDefaultPlatform) {
                    const version = child_process_1.spawnSync(exePath, ['--version']);
                    vitest_1.expect(version.status).to.equal(0);
                    vitest_1.expect(version.stdout.toString().trim()).to.not.be.empty;
                }
            });
        }
    }
    vitest_1.afterAll(async () => {
        try {
            await fs_1.promises.rmdir(testTempDir, { recursive: true });
        }
        catch {
            // ignored
        }
    });
});
vitest_1.describe.skip('fetchTargetInferredVersion', () => {
    let stable;
    let insiders;
    let extensionsDevelopmentPath = path_1.join(os_1.tmpdir(), 'vscode-test-tmp-workspace');
    vitest_1.beforeAll(async () => {
        [stable, insiders] = await Promise.all([download_1.fetchStableVersions(5000), download_1.fetchInsiderVersions(5000)]);
    });
    vitest_1.afterEach(async () => {
        await fs_1.promises.rm(extensionsDevelopmentPath, { recursive: true, force: true });
    });
    const writeJSON = async (path, contents) => {
        const target = path_1.join(extensionsDevelopmentPath, path);
        await fs_1.promises.mkdir(path_1.dirname(target), { recursive: true });
        await fs_1.promises.writeFile(target, JSON.stringify(contents));
    };
    const doFetch = (paths = ['./']) => download_1.fetchTargetInferredVersion({
        cachePath: path_1.join(extensionsDevelopmentPath, '.cache'),
        platform: 'win32-x64-archive',
        timeout: 5000,
        extensionsDevelopmentPath: paths.map((p) => path_1.join(extensionsDevelopmentPath, p)),
    });
    vitest_1.test('matches stable if no workspace', async () => {
        const version = await doFetch();
        vitest_1.expect(version).to.equal(stable[0]);
    });
    vitest_1.test('matches stable by default', async () => {
        await writeJSON('package.json', {});
        const version = await doFetch();
        vitest_1.expect(version).to.equal(stable[0]);
    });
    vitest_1.test('matches if stable is defined', async () => {
        await writeJSON('package.json', { engines: { vscode: '^1.50.0' } });
        const version = await doFetch();
        vitest_1.expect(version).to.equal(stable[0]);
    });
    vitest_1.test('matches best', async () => {
        await writeJSON('package.json', { engines: { vscode: '<=1.60.5' } });
        const version = await doFetch();
        vitest_1.expect(version).to.equal('1.60.2');
    });
    vitest_1.test('matches multiple workspaces', async () => {
        await writeJSON('a/package.json', { engines: { vscode: '<=1.60.5' } });
        await writeJSON('b/package.json', { engines: { vscode: '<=1.55.5' } });
        const version = await doFetch(['a', 'b']);
        vitest_1.expect(version).to.equal('1.55.2');
    });
    vitest_1.test('matches insiders to better stable if there is one', async () => {
        await writeJSON('package.json', { engines: { vscode: '^1.60.0-insider' } });
        const version = await doFetch();
        vitest_1.expect(version).to.equal(stable[0]);
    });
    vitest_1.test('matches current insiders', async () => {
        await writeJSON('package.json', { engines: { vscode: `^${insiders[0]}` } });
        const version = await doFetch();
        vitest_1.expect(version).to.equal(insiders[0]);
    });
    vitest_1.test('matches insiders to exact', async () => {
        await writeJSON('package.json', { engines: { vscode: '1.60.0-insider' } });
        const version = await doFetch();
        vitest_1.expect(version).to.equal('1.60.0-insider');
    });
});
