"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fetchNodeBinary = void 0;
const path = require("path");
const fs = require("fs-extra");
const stream_1 = require("stream");
const log_1 = require("../log");
const child_process_1 = require("child_process");
const util_1 = require("util");
const got_1 = require("got");
const retry = require("async-retry");
const util_2 = require("../util");
const pipeline = (0, util_1.promisify)(stream_1.pipeline);
const exec = (0, util_1.promisify)(child_process_1.exec);
const RETRY_TIMEOUT_MS = 1000;
async function fetchNodeBinary({ nodeVersion, output, platform, arch, tmp }) {
    if (arch === 'arm')
        arch = 'armv7l';
    let nodeBase = `node-v${nodeVersion}-${platform}-${arch}`;
    let tarball = path.join(tmp, 'node', `${nodeBase}.tar.xz`);
    let url = `https://nodejs.org/dist/v${nodeVersion}/${nodeBase}.tar.xz`;
    if (platform === 'win32') {
        await (0, util_2.checkFor7Zip)();
        nodeBase = `node-v${nodeVersion}-win-${arch}`;
        tarball = path.join(tmp, 'node', `${nodeBase}.7z`);
        url = `https://nodejs.org/dist/v${nodeVersion}/${nodeBase}.7z`;
        output += '.exe';
    }
    let cache = path.join(tmp, 'cache', `node-v${nodeVersion}-${platform}-${arch}`);
    if (platform === 'win32')
        cache += '.exe';
    const download = async () => {
        (0, log_1.log)(`downloading ${nodeBase}`);
        await Promise.all([
            fs.ensureDir(path.join(tmp, 'cache', nodeVersion)),
            fs.ensureDir(path.join(tmp, 'node')),
        ]);
        const shasums = path.join(tmp, 'cache', nodeVersion, 'SHASUMS256.txt.asc');
        if (!fs.existsSync(shasums)) {
            await pipeline(got_1.default.stream(`https://nodejs.org/dist/v${nodeVersion}/SHASUMS256.txt.asc`), fs.createWriteStream(shasums));
        }
        const basedir = path.dirname(tarball);
        await fs.promises.mkdir(basedir, { recursive: true });
        await pipeline(got_1.default.stream(url), fs.createWriteStream(tarball));
        if (platform !== 'win32')
            await exec(`grep "${path.basename(tarball)}" "${shasums}" | shasum -a 256 -c -`, { cwd: basedir });
    };
    const extract = async () => {
        (0, log_1.log)(`extracting ${nodeBase}`);
        const nodeTmp = path.join(tmp, 'node');
        await fs.promises.mkdir(nodeTmp, { recursive: true });
        await fs.promises.mkdir(path.dirname(cache), { recursive: true });
        if (platform === 'win32') {
            await exec(`7z x -bd -y "${tarball}"`, { cwd: nodeTmp });
            await fs.move(path.join(nodeTmp, nodeBase, 'node.exe'), path.join(cache, 'node.exe'));
        }
        else {
            await exec(`tar -C "${tmp}/node" -xJf "${tarball}"`);
            await fs.move(path.join(nodeTmp, nodeBase, 'bin', 'node'), path.join(cache, 'node'));
        }
    };
    if (!fs.existsSync(cache)) {
        await retry(download, {
            retries: 3,
            factor: 1,
            maxTimeout: RETRY_TIMEOUT_MS,
            minTimeout: RETRY_TIMEOUT_MS,
            onRetry(_e, attempt) {
                (0, log_1.log)(`retrying node download (attempt ${attempt})`);
            },
        });
        await extract();
    }
    await fs.copy(path.join(cache, getFilename(platform)), output);
    return output;
}
exports.fetchNodeBinary = fetchNodeBinary;
const getFilename = (platform) => {
    return platform === 'win32' ? 'node.exe' : 'node';
};
