"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const fs = require("fs");
const path = require("path");
const aws_1 = require("../../aws");
const log_1 = require("../../log");
const Tarballs = require("../../tarballs");
const upload_util_1 = require("../../upload-util");
class UploadDeb extends core_1.Command {
    async run() {
        const { flags } = await this.parse(UploadDeb);
        const buildConfig = await Tarballs.buildConfig(flags.root);
        const { s3Config, config } = buildConfig;
        const dist = (f) => buildConfig.dist(path.join('deb', f));
        const S3Options = {
            Bucket: s3Config.bucket,
            ACL: s3Config.acl || 'public-read',
        };
        if (!fs.existsSync(dist('Release')))
            this.error('Cannot find debian artifacts', {
                suggestions: ['Run "oclif pack deb" before uploading'],
            });
        const cloudKeyBase = (0, upload_util_1.commitAWSDir)(config.pjson.version, buildConfig.gitSha, s3Config);
        const upload = (file) => {
            const cloudKey = `${cloudKeyBase}/apt/${file}`;
            return aws_1.default.s3.uploadFile(dist(file), Object.assign(Object.assign({}, S3Options), { CacheControl: 'max-age=86400', Key: cloudKey }));
        };
        // apt expects ../apt/dists/versionName/[artifacts] but oclif wants versions/sha/apt/[artifacts]
        // see https://github.com/oclif/oclif/issues/347 for the AWS-redirect that solves this
        // this workaround puts the code in both places that the redirect was doing
        // with this, the docs are correct. The copies are all done in parallel so it shouldn't be too costly.
        const uploadWorkaround = (file) => {
            const cloudKey = `${cloudKeyBase}/apt/./${file}`;
            return aws_1.default.s3.uploadFile(dist(file), Object.assign(Object.assign({}, S3Options), { CacheControl: 'max-age=86400', Key: cloudKey }));
        };
        const uploadDeb = async (arch) => {
            const deb = (0, upload_util_1.templateShortKey)('deb', { bin: config.bin, versionShaRevision: (0, upload_util_1.debVersion)(buildConfig), arch: arch });
            if (fs.existsSync(dist(deb)))
                await Promise.all([upload(deb), uploadWorkaround(deb)]);
        };
        (0, log_1.log)(`starting upload of deb artifacts for v${config.version}-${buildConfig.gitSha}`);
        const arches = buildConfig.targets.filter(t => t.platform === 'linux');
        await Promise.all([
            ...arches.map(a => uploadDeb((0, upload_util_1.debArch)(a.arch))),
            upload('Packages.gz'),
            upload('Packages.xz'),
            upload('Packages.bz2'),
            upload('Release'),
            uploadWorkaround('Packages.gz'),
            uploadWorkaround('Packages.xz'),
            uploadWorkaround('Packages.bz2'),
            uploadWorkaround('Release'),
        ].concat(fs.existsSync(dist('InRelease')) ? [upload('InRelease'), uploadWorkaround('InRelease')] : [], fs.existsSync(dist('Release.gpg')) ? [upload('Release.gpg'), uploadWorkaround('Release.gpg')] : []));
        (0, log_1.log)(`done uploading deb artifacts for v${config.version}-${buildConfig.gitSha}`);
    }
}
exports.default = UploadDeb;
UploadDeb.description = 'upload deb package built with pack:deb';
UploadDeb.flags = {
    root: core_1.Flags.string({ char: 'r', description: 'path to oclif CLI root', default: '.', required: true }),
};
