"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const fs = require("fs-extra");
const _ = require("lodash");
const path = require("path");
const url_1 = require("url");
const util_1 = require("../util");
const help_compatibility_1 = require("../help-compatibility");
const normalize = require('normalize-package-data');
const columns = Number.parseInt(process.env.COLUMNS, 10) || 120;
const slugify = new (require('github-slugger'))();
class Readme extends core_1.Command {
    async run() {
        var _a, _b;
        this.flags = (await this.parse(Readme)).flags;
        const cwd = process.cwd();
        const readmePath = path.resolve(cwd, 'README.md');
        const tsConfigPath = path.resolve(cwd, 'tsconfig.json');
        const tsConfig = await fs.readJSON(tsConfigPath).catch(() => ({}));
        const outDir = (_b = (_a = tsConfig.compilerOptions) === null || _a === void 0 ? void 0 : _a.outDir) !== null && _b !== void 0 ? _b : 'lib';
        if (!await fs.pathExists(outDir)) {
            this.warn(`No compiled source found at ${outDir}. Some commands may be missing.`);
        }
        const config = await core_1.Config.load({ root: cwd, devPlugins: false, userPlugins: false });
        try {
            const p = require.resolve('@oclif/plugin-legacy', { paths: [cwd] });
            const plugin = new core_1.Plugin({ root: p, type: 'core' });
            await plugin.load();
            config.plugins.push(plugin);
        }
        catch (_c) { }
        await config.runHook('init', { id: 'readme', argv: this.argv });
        this.HelpClass = await (0, core_1.loadHelpClass)(config);
        let readme = await fs.readFile(readmePath, 'utf8');
        let commands = config.commands
            .filter(c => !c.hidden && c.pluginType === 'core')
            .filter(c => this.flags.aliases ? true : !c.aliases.includes(c.id))
            .map(c => c.id === '.' ? Object.assign(Object.assign({}, c), { id: '' }) : c);
        this.debug('commands:', commands.map(c => c.id).length);
        commands = (0, util_1.uniqBy)(commands, c => c.id);
        commands = (0, util_1.sortBy)(commands, c => c.id);
        readme = this.replaceTag(readme, 'usage', this.usage(config));
        readme = this.replaceTag(readme, 'commands', this.flags.multi ? this.multiCommands(config, commands, this.flags.dir) : this.commands(config, commands));
        readme = this.replaceTag(readme, 'toc', this.toc(config, readme));
        readme = readme.trimEnd();
        readme += '\n';
        await fs.outputFile(readmePath, readme);
    }
    replaceTag(readme, tag, body) {
        if (readme.includes(`<!-- ${tag} -->`)) {
            if (readme.includes(`<!-- ${tag}stop -->`)) {
                readme = readme.replace(new RegExp(`<!-- ${tag} -->(.|\n)*<!-- ${tag}stop -->`, 'm'), `<!-- ${tag} -->`);
            }
            this.log(`replacing <!-- ${tag} --> in README.md`);
        }
        return readme.replace(`<!-- ${tag} -->`, `<!-- ${tag} -->\n${body}\n<!-- ${tag}stop -->`);
    }
    toc(__, readme) {
        return readme.split('\n').filter(l => l.startsWith('# '))
            .map(l => l.trim().slice(2))
            .map(l => `* [${l}](#${slugify.slug(l)})`)
            .join('\n');
    }
    usage(config) {
        var _a;
        const versionFlags = ['--version', ...((_a = config.pjson.oclif.additionalVersionFlags) !== null && _a !== void 0 ? _a : []).sort()];
        const versionFlagsString = `(${versionFlags.join('|')})`;
        return [
            `\`\`\`sh-session
$ npm install -g ${config.name}
$ ${config.bin} COMMAND
running command...
$ ${config.bin} ${versionFlagsString}
${config.name}/${this.flags.version || config.version} ${process.platform}-${process.arch} node-v${process.versions.node}
$ ${config.bin} --help [COMMAND]
USAGE
  $ ${config.bin} COMMAND
...
\`\`\`\n`,
        ].join('\n').trim();
    }
    multiCommands(config, commands, dir) {
        let topics = config.topics;
        topics = topics.filter(t => !t.hidden && !t.name.includes(':'));
        topics = topics.filter(t => commands.find(c => c.id.startsWith(t.name)));
        topics = (0, util_1.sortBy)(topics, t => t.name);
        topics = (0, util_1.uniqBy)(topics, t => t.name);
        for (const topic of topics) {
            this.createTopicFile(path.join('.', dir, topic.name.replace(/:/g, '/') + '.md'), config, topic, commands.filter(c => c.id === topic.name || c.id.startsWith(topic.name + ':')));
        }
        return [
            '# Command Topics\n',
            ...topics.map(t => {
                return (0, util_1.compact)([
                    `* [\`${config.bin} ${t.name}\`](${dir}/${t.name.replace(/:/g, '/')}.md)`,
                    (0, util_1.template)({ config })(t.description || '').trim().split('\n')[0],
                ]).join(' - ');
            }),
        ].join('\n').trim() + '\n';
    }
    createTopicFile(file, config, topic, commands) {
        const bin = `\`${config.bin} ${topic.name}\``;
        const doc = [
            bin,
            '='.repeat(bin.length),
            '',
            (0, util_1.template)({ config })(topic.description || '').trim(),
            '',
            this.commands(config, commands),
        ].join('\n').trim() + '\n';
        fs.outputFileSync(file, doc);
    }
    commands(config, commands) {
        return [
            ...commands.map(c => {
                const usage = this.commandUsage(config, c);
                return usage ? `* [\`${config.bin} ${usage}\`](#${slugify.slug(`${config.bin}-${usage}`)})` : `* [\`${config.bin}\`](#${slugify.slug(`${config.bin}`)})`;
            }),
            '',
            ...commands.map(c => this.renderCommand(config, c)).map(s => s.trim() + '\n'),
        ].join('\n').trim();
    }
    renderCommand(config, c) {
        this.debug('rendering command', c.id);
        const title = (0, util_1.template)({ config, command: c })(c.summary || c.description || '').trim().split('\n')[0];
        const help = new this.HelpClass(config, { stripAnsi: true, maxWidth: columns });
        const wrapper = new help_compatibility_1.HelpCompatibilityWrapper(help);
        const header = () => {
            const usage = this.commandUsage(config, c);
            return usage ? `## \`${config.bin} ${usage}\`` : `## \`${config.bin}\``;
        };
        try {
            // copy c to keep the command ID with colons, see:
            // https://github.com/oclif/oclif/pull/1165#discussion_r1282305242
            const command = Object.assign({}, c);
            return (0, util_1.compact)([
                header(),
                title,
                '```\n' + wrapper.formatCommand(c).trim() + '\n```',
                this.commandCode(config, command),
            ]).join('\n\n');
        }
        catch (error) {
            this.error(error.message);
        }
    }
    commandCode(config, c) {
        const pluginName = c.pluginName;
        if (!pluginName)
            return;
        const plugin = config.plugins.find(p => p.name === c.pluginName);
        if (!plugin)
            return;
        const repo = this.repo(plugin);
        if (!repo)
            return;
        let label = plugin.name;
        let version = plugin.version;
        const commandPath = this.commandPath(plugin, c);
        if (!commandPath)
            return;
        if (config.name === plugin.name) {
            label = commandPath;
            version = this.flags.version || version;
        }
        const template = this.flags['repository-prefix'] || plugin.pjson.oclif.repositoryPrefix || '<%- repo %>/blob/v<%- version %>/<%- commandPath %>';
        return `_See code: [${label}](${_.template(template)({ repo, version, commandPath, config, c })})_`;
    }
    repo(plugin) {
        const pjson = Object.assign({}, plugin.pjson);
        normalize(pjson);
        const repo = pjson.repository && pjson.repository.url;
        if (!repo)
            return;
        const url = new url_1.URL(repo);
        if (!['github.com', 'gitlab.com'].includes(url.hostname) && !pjson.oclif.repositoryPrefix && !this.flags['repository-prefix'])
            return;
        return `https://${url.hostname}${url.pathname.replace(/\.git$/, '')}`;
    }
    // eslint-disable-next-line valid-jsdoc
    /**
     * fetches the path to a command
     */
    commandPath(plugin, c) {
        var _a, _b;
        const commandsDir = plugin.pjson.oclif.commands;
        if (!commandsDir)
            return;
        const hasTypescript = ((_a = plugin.pjson.devDependencies) === null || _a === void 0 ? void 0 : _a.typescript) || ((_b = plugin.pjson.dependencies) === null || _b === void 0 ? void 0 : _b.typescript);
        let p = path.join(plugin.root, commandsDir, ...c.id.split(':'));
        const outDir = path.dirname(commandsDir.replace(`.${path.sep}`, ''));
        const outDirRegex = new RegExp('^' + outDir + (path.sep === '\\' ? '\\\\' : path.sep));
        if (fs.pathExistsSync(path.join(p, 'index.js'))) {
            p = path.join(p, 'index.js');
        }
        else if (fs.pathExistsSync(p + '.js')) {
            p += '.js';
        }
        else if (hasTypescript) {
            // check if non-compiled scripts are available
            const base = p.replace(plugin.root + path.sep, '');
            p = path.join(plugin.root, base.replace(outDirRegex, 'src' + path.sep));
            if (fs.pathExistsSync(path.join(p, 'index.ts'))) {
                p = path.join(p, 'index.ts');
            }
            else if (fs.pathExistsSync(p + '.ts')) {
                p += '.ts';
            }
            else
                return;
        }
        else
            return;
        p = p.replace(plugin.root + path.sep, '');
        if (hasTypescript) {
            p = p.replace(outDirRegex, 'src' + path.sep).replace(/\.js$/, '.ts');
        }
        p = p.replace(/\\/g, '/'); // Replace windows '\' by '/'
        return p;
    }
    commandUsage(config, command) {
        const arg = (arg) => {
            const name = arg.name.toUpperCase();
            if (arg.required)
                return `${name}`;
            return `[${name}]`;
        };
        const id = (0, core_1.toConfiguredId)(command.id, config);
        const defaultUsage = () => {
            return (0, util_1.compact)([
                id,
                Object.values(command.args).filter(a => !a.hidden).map(a => arg(a)).join(' '),
            ]).join(' ');
        };
        const usages = (0, util_1.castArray)(command.usage);
        return (0, util_1.template)({ config, command })(usages.length === 0 ? defaultUsage() : usages[0]);
    }
}
exports.default = Readme;
Readme.description = `adds commands to README.md in current directory
The readme must have any of the following tags inside of it for it to be replaced or else it will do nothing:
# Usage
<!-- usage -->
# Commands
<!-- commands -->
# Table of contents
<!-- toc -->

Customize the code URL prefix by setting oclif.repositoryPrefix in package.json.
`;
Readme.flags = {
    dir: core_1.Flags.string({ description: 'output directory for multi docs', default: 'docs', required: true }),
    multi: core_1.Flags.boolean({ description: 'create a different markdown page for each topic' }),
    aliases: core_1.Flags.boolean({ description: 'include aliases in the command list', allowNo: true, default: true }),
    'repository-prefix': core_1.Flags.string({ description: 'a template string used to build links to the source code' }),
    version: core_1.Flags.string({ description: 'version to use in readme links. defaults to the version in package.json', env: 'OCLIF_NEXT_VERSION' }),
};
