"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const fs = require("fs-extra");
const path = require("path");
const os = require("os");
const semver = require("semver");
const shelljs_1 = require("shelljs");
const got_1 = require("got");
const util_1 = require("util");
const stream_1 = require("stream");
const util_2 = require("../util");
const pipeline = (0, util_1.promisify)(stream_1.pipeline);
async function fileExists(filePath) {
    try {
        await fs.access(filePath);
        return true;
    }
    catch (_a) {
        return false;
    }
}
class Manifest extends core_1.Command {
    async run() {
        var _a;
        const { flags } = await this.parse(Manifest);
        try {
            fs.unlinkSync('oclif.manifest.json');
        }
        catch (_b) { }
        const { args } = await this.parse(Manifest);
        const root = path.resolve(args.path);
        const packageJson = fs.readJSONSync('package.json');
        let jitPluginManifests = [];
        if (flags.jit && ((_a = packageJson.oclif) === null || _a === void 0 ? void 0 : _a.jitPlugins)) {
            this.debug('jitPlugins: %s', packageJson.oclif.jitPlugins);
            const tmpDir = os.tmpdir();
            const promises = Object.entries(packageJson.oclif.jitPlugins).map(async ([jitPlugin, version]) => {
                const pluginDir = jitPlugin.replace('/', '-').replace('@', '');
                const fullPath = path.join(tmpDir, pluginDir);
                if (await fileExists(fullPath))
                    await fs.remove(fullPath);
                await fs.mkdir(fullPath, { recursive: true });
                const resolvedVersion = this.getVersion(jitPlugin, version);
                const tarballUrl = this.getTarballUrl(jitPlugin, resolvedVersion);
                const tarball = path.join(fullPath, path.basename(tarballUrl));
                await pipeline(got_1.default.stream(tarballUrl), fs.createWriteStream(tarball));
                if (process.platform === 'win32') {
                    await (0, util_2.checkFor7Zip)();
                    (0, shelljs_1.exec)(`7z x -bd -y "${tarball}"`, { cwd: fullPath });
                }
                else {
                    (0, shelljs_1.exec)(`tar -xzf "${tarball}"`, { cwd: fullPath });
                }
                const manifest = await fs.readJSON(path.join(fullPath, 'package', 'oclif.manifest.json'));
                for (const command of Object.values(manifest.commands)) {
                    command.pluginType = 'jit';
                }
                return manifest;
            });
            core_1.ux.action.start('Generating JIT plugin manifests');
            jitPluginManifests = await Promise.all(promises);
            core_1.ux.action.stop();
        }
        let plugin = new core_1.Plugin({ root, type: 'core', ignoreManifest: true, errorOnManifestCreate: true });
        if (!plugin)
            throw new Error('plugin not found');
        await plugin.load(true);
        if (!plugin.valid) {
            const p = require.resolve('@oclif/plugin-legacy', { paths: [process.cwd()] });
            const { PluginLegacy } = require(p);
            plugin = new PluginLegacy(this.config, plugin);
            await plugin.load();
        }
        if (process.env.OCLIF_NEXT_VERSION) {
            plugin.manifest.version = process.env.OCLIF_NEXT_VERSION;
        }
        const dotfile = plugin.pjson.files.find((f) => f.endsWith('.oclif.manifest.json'));
        const file = path.join(plugin.root, `${dotfile ? '.' : ''}oclif.manifest.json`);
        for (const manifest of jitPluginManifests) {
            plugin.manifest.commands = Object.assign(Object.assign({}, plugin.manifest.commands), manifest.commands);
        }
        fs.writeFileSync(file, JSON.stringify(plugin.manifest, null, 2));
        this.log(`wrote manifest to ${file}`);
    }
    getVersion(plugin, version) {
        var _a;
        if (version.startsWith('^') || version.startsWith('~')) {
            // Grab latest from npm to get all the versions so we can find the max satisfying version.
            // We explicitly ask for latest since this command is typically run inside of `npm prepack`,
            // which sets the npm_config_tag env var, which is used as the default anytime a tag isn't
            // provided to `npm view`. This can be problematic if you're building the `nightly` version
            // of a CLI and all the JIT plugins don't have a `nightly` tag themselves.
            // TL;DR - always ask for latest to avoid potentially requesting a non-existent tag.
            const { versions } = JSON.parse(this.executeCommand(`npm view ${plugin}@latest --json`).stdout);
            return (_a = semver.maxSatisfying(versions, version)) !== null && _a !== void 0 ? _a : version.replace('^', '').replace('~', '');
        }
        return version;
    }
    getTarballUrl(plugin, version) {
        const { dist } = JSON.parse(this.executeCommand(`npm view ${plugin}@${version} --json`).stdout);
        return dist.tarball;
    }
    executeCommand(command, options) {
        const debugString = (options === null || options === void 0 ? void 0 : options.cwd) ? `executing command: ${command} in ${options.cwd}` : `executing command: ${command}`;
        this.debug(debugString);
        const result = (0, shelljs_1.exec)(command, Object.assign(Object.assign({}, options), { silent: true, async: false }));
        if (result.code !== 0) {
            this.error(result.stderr);
        }
        this.debug(result.stdout);
        return result;
    }
}
exports.default = Manifest;
Manifest.description = 'generates plugin manifest json';
Manifest.args = {
    path: core_1.Args.string({ description: 'path to plugin', default: '.' }),
};
Manifest.flags = {
    jit: core_1.Flags.boolean({
        allowNo: true,
        summary: 'append commands from JIT plugins in manifest',
        default: true,
    }),
};
