import { publicRuntimeConfig } from "next.config"
import Image from "next/image"
import Link from "next/link"
import { useEffect } from "react"

export function AppHeader() {
    useEffect(() => {
        if (typeof window !== 'undefined') {
            const HSThemeAppearance = {
                init() {
                    const defaultTheme = 'default'
                    let theme = localStorage.getItem('hs_theme') || defaultTheme

                    if (document.querySelector('html').classList.contains('dark')) return
                    this.setAppearance(theme)
                },
                _resetStylesOnLoad() {
                    const $resetStyles = document.createElement('style')
                    $resetStyles.innerText = `*{transition: unset !important;}`
                    $resetStyles.setAttribute('data-hs-appearance-onload-styles', '')
                    document.head.appendChild($resetStyles)
                    return $resetStyles
                },
                setAppearance(theme, saveInStore = true, dispatchEvent = true) {
                    const $resetStylesEl = this._resetStylesOnLoad()

                    if (saveInStore) {
                        localStorage.setItem('hs_theme', theme)
                    }

                    if (theme === 'auto') {
                        theme = window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'default'
                    }

                    document.querySelector('html').classList.remove('dark')
                    document.querySelector('html').classList.remove('default')
                    document.querySelector('html').classList.remove('auto')

                    document.querySelector('html').classList.add(theme)

                    setTimeout(() => {
                        $resetStylesEl.remove()
                    })

                    if (dispatchEvent) {
                        window.dispatchEvent(new CustomEvent('on-hs-appearance-change', { detail: theme }))
                    }

                    $resetStylesEl.remove()
                },
                getAppearance() {
                    let theme = this.getOriginalAppearance()
                    if (theme === 'auto') {
                        theme = window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'default'
                    }
                    return theme
                },
                getOriginalAppearance() {
                    const defaultTheme = 'default'
                    return localStorage.getItem('hs_theme') || defaultTheme
                }
            }

            HSThemeAppearance.init()

            window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', () => {
                if (HSThemeAppearance.getOriginalAppearance() === 'auto') {
                    HSThemeAppearance.setAppearance('auto', false)
                }
            })

            const $clickableThemes = document.querySelectorAll('[data-hs-theme-click-value]')
            const $switchableThemes = document.querySelectorAll('[data-hs-theme-switch]')

            $clickableThemes.forEach(function ($item) {
                $item.addEventListener('click', function () {
                    HSThemeAppearance.setAppearance($item.getAttribute('data-hs-theme-click-value'), true, $item)
                })
            })

            $switchableThemes.forEach($item => {
                $item.addEventListener('change', (e) => {
                    HSThemeAppearance.setAppearance(e.target.checked ? 'dark' : 'default')
                })

                $item.checked = HSThemeAppearance.getAppearance() === 'dark'
            })

            window.addEventListener('on-hs-appearance-change', e => {
                $switchableThemes.forEach($item => {
                    $item.checked = e.detail === 'dark'
                })
            })
        }
    }, [])

    return (
        <header class="flex flex-wrap md:justify-start md:flex-nowrap z-50 w-full text-sm">
            <nav class="shadow-sm shadow-gray-100 dark:shadow-gray-600 mt-6 relative max-w-7xl w-full bg-gradient-to-r from-violet-300 to-violet-100 border dark:bg-gradient-to-r dark:from-violet-900 dark:to-violet-300 border-gray-200 dark:border-gray-600 rounded-[25px] mx-2 py-3 px-4 md:flex md:items-center md:justify-between md:py-0 md:px-6 lg:px-8 xl:mx-auto dark:bg-gray-800 dark:border-gray-700" aria-label="Global">
                <div class="flex-none">
                    <div class="flex flex-wrap items-center justify-between">
                        <div class="flex items-center gap-5">
                            <Link href="/" className={`flex items-center justify-between gap-2 text-xl font-semibold text-gray-800 dark:text-white`} aria-label="Brand">
                                <Image
                                    src="icon.svg"
                                    alt="LinkSnatch Icon"
                                    width={25}
                                    height={25}
                                />
                                <span>{publicRuntimeConfig.app_name}</span>
                            </Link>
                            <div class="hs-dropdown" data-hs-dropdown-placement="bottom-right" data-hs-dropdown-offset="30">
                                <a class="hs-dropdown-toggle hs-dark-mode group flex items-center text-gray-700 hover:text-purple-700 font-medium dark:text-gray-400 dark:hover:text-gray-500" href="javascript:;">
                                    <svg class="hs-dark-mode-active:hidden block w-4 h-4" xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                                        <path d="M6 .278a.768.768 0 0 1 .08.858 7.208 7.208 0 0 0-.878 3.46c0 4.021 3.278 7.277 7.318 7.277.527 0 1.04-.055 1.533-.16a.787.787 0 0 1 .81.316.733.733 0 0 1-.031.893A8.349 8.349 0 0 1 8.344 16C3.734 16 0 12.286 0 7.71 0 4.266 2.114 1.312 5.124.06A.752.752 0 0 1 6 .278zM4.858 1.311A7.269 7.269 0 0 0 1.025 7.71c0 4.02 3.279 7.276 7.319 7.276a7.316 7.316 0 0 0 5.205-2.162c-.337.042-.68.063-1.029.063-4.61 0-8.343-3.714-8.343-8.29 0-1.167.242-2.278.681-3.286z" />
                                    </svg>
                                    <svg class="hs-dark-mode-active:block hidden w-4 h-4 dark:stroke-gray-100" xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                                        <path d="M8 11a3 3 0 1 1 0-6 3 3 0 0 1 0 6zm0 1a4 4 0 1 0 0-8 4 4 0 0 0 0 8zM8 0a.5.5 0 0 1 .5.5v2a.5.5 0 0 1-1 0v-2A.5.5 0 0 1 8 0zm0 13a.5.5 0 0 1 .5.5v2a.5.5 0 0 1-1 0v-2A.5.5 0 0 1 8 13zm8-5a.5.5 0 0 1-.5.5h-2a.5.5 0 0 1 0-1h2a.5.5 0 0 1 .5.5zM3 8a.5.5 0 0 1-.5.5h-2a.5.5 0 0 1 0-1h2A.5.5 0 0 1 3 8zm10.657-5.657a.5.5 0 0 1 0 .707l-1.414 1.415a.5.5 0 1 1-.707-.708l1.414-1.414a.5.5 0 0 1 .707 0zm-9.193 9.193a.5.5 0 0 1 0 .707L3.05 13.657a.5.5 0 0 1-.707-.707l1.414-1.414a.5.5 0 0 1 .707 0zm9.193 2.121a.5.5 0 0 1-.707 0l-1.414-1.414a.5.5 0 0 1 .707-.707l1.414 1.414a.5.5 0 0 1 0 .707zM4.464 4.465a.5.5 0 0 1-.707 0L2.343 3.05a.5.5 0 1 1 .707-.707l1.414 1.414a.5.5 0 0 1 0 .708z" />
                                    </svg>
                                </a>

                                <div id="selectThemeDropdown" class="hs-dropdown-menu hs-dropdown-open:opacity-100 mt-2 hidden z-10 transition-[margin,opacity] opacity-0 duration-300 mb-2 origin-bottom-left bg-white shadow-md rounded-lg p-2 space-y-1 dark:bg-gray-800 dark:border dark:border-gray-700 dark:divide-gray-700">
                                    <a class="hs-auto-mode-active:bg-gray-100 flex items-center gap-x-3.5 py-2 px-3 rounded-md text-sm text-gray-800 hover:bg-gray-100 dark:text-gray-400 dark:hover:bg-gray-700 dark:hover:text-gray-300" href="javascript:;" data-hs-theme-click-value="auto">
                                        Auto (system default)
                                    </a>
                                    <a class="hs-default-mode-active:bg-gray-100 flex items-center gap-x-3.5 py-2 px-3 rounded-md text-sm text-gray-800 hover:bg-gray-100 dark:text-gray-400 dark:hover:bg-gray-700 dark:hover:text-gray-300" href="javascript:;" data-hs-theme-click-value="default">
                                        Default (light mode)
                                    </a>
                                    <a class="hs-dark-mode-active:bg-gray-700 flex items-center gap-x-3.5 py-2 px-3 rounded-md text-sm text-gray-800 hover:bg-gray-100 dark:text-gray-400 dark:hover:bg-gray-700 dark:hover:text-gray-300" href="javascript:;" data-hs-theme-click-value="dark">
                                        Dark
                                    </a>
                                </div>
                            </div>
                        </div>
                        <div class="md:hidden">
                            <button type="button" class="hs-collapse-toggle p-2 inline-flex justify-center items-center gap-2 rounded-full border font-medium bg-white text-gray-700 shadow-sm align-middle hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-offset-white focus:ring-blue-600 transition-all text-sm dark:bg-purple-900 dark:hover:bg-purple-800 dark:border-gray-700 dark:text-gray-400 dark:hover:text-white dark:focus:ring-offset-gray-800" data-hs-collapse="#navbar-collapse-with-animation" aria-controls="navbar-collapse-with-animation" aria-label="Toggle navigation">
                                <svg class="hs-collapse-open:hidden w-4 h-4" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                                    <path fillRule="evenodd" d="M2.5 12a.5.5 0 0 1 .5-.5h10a.5.5 0 0 1 0 1H3a.5.5 0 0 1-.5-.5zm0-4a.5.5 0 0 1 .5-.5h10a.5.5 0 0 1 0 1H3a.5.5 0 0 1-.5-.5zm0-4a.5.5 0 0 1 .5-.5h10a.5.5 0 0 1 0 1H3a.5.5 0 0 1-.5-.5z" />
                                </svg>
                                <svg class="hs-collapse-open:block hidden w-4 h-4" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                                    <path d="M4.646 4.646a.5.5 0 0 1 .708 0L8 7.293l2.646-2.647a.5.5 0 0 1 .708.708L8.707 8l2.647 2.646a.5.5 0 0 1-.708.708L8 8.707l-2.646 2.647a.5.5 0 0 1-.708-.708L7.293 8 4.646 5.354a.5.5 0 0 1 0-.708z" />
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                <div id="navbar-collapse-with-animation" class="hs-collapse hidden overflow-hidden transition-all duration-300 basis-full grow md:block">
                    <div class="flex flex-col gap-y-4 gap-x-0 mt-5 md:flex-row md:items-center md:justify-end md:gap-y-0 md:gap-x-7 md:mt-0 md:pl-7">
                        <Link href="/about" className={`font-medium text-violet-800 md:py-6 dark:text-violet-100 dark:md:text-gray-900`} aria-current="page">About</Link>
                    </div>
                </div>
            </nav>
        </header>
    )
}