# coding=utf-8
# Copyright 2021 The Alibaba DAMO NLP Team Authors.
# Copyright 2018 The Google AI Language Team Authors.
# Copyright (c) 2018, NVIDIA CORPORATION.  All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Run masked LM/next sentence masked_lm pre-training for BERT."""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import os
import time
import modeling
import modeling_labert
import optimization
import tensorflow as tf
import shutil
import random
try:
  import horovod.tensorflow as hvd
except ImportError:
  hvd = None

tf.flags = tf.compat.v1.flags
flags = tf.flags

FLAGS = flags.FLAGS

## Required parameters
flags.DEFINE_string(
  "labert_config_file", None,
  "The config json file corresponding to the pre-trained BERT model. "
  "This specifies the model architecture.")

flags.DEFINE_string(
  "input_file", None,
  "Input TF example files (can be a glob or comma separated).")

flags.DEFINE_string(
  "eval_file", None,
  " TF example files for eval (can be a glob or comma separated).")

flags.DEFINE_string(
  "output_dir", None,
  "The output directory where the model checkpoints will be written.")

## Other parameters
flags.DEFINE_string(
  "init_checkpoint", None,
  "Initial checkpoint (usually from a pre-trained BERT model).")

flags.DEFINE_integer(
  "max_seq_length", 128,
  "The maximum total input sequence length after WordPiece tokenization. "
  "Sequences longer than this will be truncated, and sequences shorter "
  "than this will be padded. Must match data generation.")

flags.DEFINE_integer(
  "max_predictions_per_seq", 20,
  "Maximum number of masked LM predictions per sequence. "
  "Must match data generation.")

flags.DEFINE_integer(
  "next_sentence_type", 3,
  "Next sentence task: support 2 or 3 only. Must match data generation.")

flags.DEFINE_bool("do_train", False, "Whether to run training.")

flags.DEFINE_bool("do_eval", False, "Whether to run eval on the dev set.")

flags.DEFINE_integer(
  "train_batch_size", 32,
  "Total batch size for training. [NOTE] This number means the real batch "
  "size on each GPU core. If use_horovod is configured, the algorithmic "
  "batch size equals to train_batch_size * hvd.size()")

flags.DEFINE_integer("eval_batch_size", 8, "Total batch size for eval.")

flags.DEFINE_float("learning_rate", 5e-5, "The initial learning rate for Adam.")

flags.DEFINE_integer(
  "num_train_steps", 100000,
  "Number of training steps. [NOTE] This number is the algorithmic training steps.")

flags.DEFINE_integer(
  "num_warmup_steps", 10000,
  "Number of warmup steps. [NOTE] This number is the algorithmic steps.")

flags.DEFINE_integer(
  "num_decay_steps", None,
  "Number of decay steps. If unset, it equals to the num_traing_steps. "
  "[NOTE] This number is the algorithmic steps.")

flags.DEFINE_integer("num_log_steps", 100, "Log every n steps.")

flags.DEFINE_integer("save_checkpoints_steps", 1000,
                     "How often to save the model checkpoint.")

flags.DEFINE_integer("iterations_per_loop", 1000,
                     "How many steps to make in each estimator call.")

flags.DEFINE_integer("max_eval_steps", 100, "Maximum number of eval steps.")

flags.DEFINE_bool("use_tpu", False, "Whether to use TPU or GPU/CPU.")

tf.flags.DEFINE_string(
  "tpu_name", None,
  "The Cloud TPU to use for training. This should be either the name "
  "used when creating the Cloud TPU, or a grpc://ip.address.of.tpu:8470 "
  "url.")

tf.flags.DEFINE_string(
  "tpu_zone", None,
  "[Optional] GCE zone where the Cloud TPU is located in. If not "
  "specified, we will attempt to automatically detect the GCE project from "
  "metadata.")

tf.flags.DEFINE_string(
  "gcp_project", None,
  "[Optional] Project name for the Cloud TPU-enabled project. If not "
  "specified, we will attempt to automatically detect the GCE project from "
  "metadata.")

tf.flags.DEFINE_string("master", None, "[Optional] TensorFlow master URL.")

flags.DEFINE_integer(
  "num_tpu_cores", 8,
  "Only used if `use_tpu` is True. Total number of TPU cores to use.")

flags.DEFINE_bool("use_horovod", False, "Whether to use Horovod for multi-gpu training.")

flags.DEFINE_bool("use_fp16", False, "Whether to use fp16 for mix-precision gradient.")

flags.DEFINE_integer(
  "num_accumulate_steps", 1,
  "The number of accumulated gradients. Supposing the batch size is 256 and "
  "num_accumulated_steps is set to 32, the final batch size equals to 8K. "
  "[Note] if you run continuous training, this number should be manually kept "
  "consistent between different runs.")

flags.DEFINE_string("transformer_model_type", "post-ln", "The transformer model type.")
flags.DEFINE_bool("do_share_parameter_across_layers", False, "Use to specify to share weight across layers.")

flags.DEFINE_string("optimizer_type", "adam", "Optimizer used for training - LAMB or ADAM")
flags.DEFINE_float("adam_beta1", 0.9, "The beta1 for adam.")
flags.DEFINE_float("adam_beta2", 0.999, "The beta2 for adam.")
flags.DEFINE_float("adam_epsilon", 1e-6, "The epsilon for adam.")
flags.DEFINE_bool("allreduce_post_accumulation", False,
                  "Whether to all reduce after accumulation of N steps or after each step")

flags.DEFINE_bool("ignore_pooler", False,
                  "Whether adopt ignore_pooler in create_optimizer")

flags.DEFINE_bool("do_return_all_attention_maps", False, "do_return_all_attention_maps")

flags.DEFINE_bool(
  "scaling_up_lr", False,
  "For the multiple GPU case, scaling up the learning rate.")

flags.DEFINE_bool("adopt_mask_matrix", False, "Whether to use mask matrix in pre-training.")


class LossHook(tf.estimator.SessionRunHook):
  def __init__(self, interval):
    self.interval = interval
    self.fetches = tf.estimator.SessionRunArgs(
      fetches=[
        "global_step:0",
        "total_loss:0",
        "masked_lm_loss:0",
        "next_sentence_loss:0",
        "learning_rate:0",
      ])
    self.step_start_time = -1
    self.logged = None
    self.numb = 0

  def before_run(self, run_context):
    self.step_start_time = time.time()
    return self.fetches

  def after_run(self, run_context, run_values):
    global_step, total_loss, masked_lm_loss, next_sentence_loss, learning_rate = run_values.results
    if (not FLAGS.use_horovod or hvd.rank() == 0) and global_step % self.interval == 0 and self.logged != global_step:
      tf.compat.v1.logging.info(
        'global_step=%d\ttotal_loss=%2.5f\tmasked_lm_loss=%2.5f\t'
        'next_sentence_loss=%2.5f\tlearning_rate=%.5e' % (global_step,
                                                          total_loss, masked_lm_loss, next_sentence_loss,
                                                          learning_rate))
      self.logged = global_step


def model_fn_builder(bert_config, init_checkpoint, learning_rate, beta1, beta2, epsilon,
                     num_decay_steps, num_warmup_steps, num_accumulate_steps, use_horovod, use_fp16,
                     use_one_hot_embeddings, do_return_all_attention_maps):
  """Returns `model_fn` closure for TPUEstimator."""

  def model_fn(features, labels, mode, params):  # pylint: disable=unused-argument
    """The `model_fn` for TPUEstimator."""

    if not use_horovod or hvd.rank() == 0:
      tf.compat.v1.logging.info("*** Features ***")
      for name in sorted(features.keys()):
        tf.compat.v1.logging.info("  name = %s, shape = %s" % (name, features[name].shape))

    input_ids = features["input_ids"]
    input_mask = features["input_mask"]
    segment_ids = features["segment_ids"]
    masked_lm_positions = features["masked_lm_positions"]
    masked_lm_ids = features["masked_lm_ids"]
    masked_lm_weights = features["masked_lm_weights"]
    next_sentence_labels = features["next_sentence_labels"]
    positional_embeddings_start = features['positional_embeddings_start']
    positional_embeddings_end = features['positional_embeddings_end']

    is_training = (mode == tf.estimator.ModeKeys.TRAIN)

    model = modeling_labert.LaBertModel(
      config=bert_config,
      is_training=is_training,
      input_ids=input_ids,
      input_mask=input_mask,
      token_type_ids=segment_ids,
      start_positions=positional_embeddings_start,
      end_positions=positional_embeddings_end,
      use_one_hot_embeddings=use_one_hot_embeddings,
      transformer_model_type=FLAGS.transformer_model_type,
      do_share_parameter_across_layers=FLAGS.do_share_parameter_across_layers,
      compute_type=tf.float16 if use_fp16 else tf.float32,
      do_return_all_attention_maps=do_return_all_attention_maps,)

    (masked_lm_loss,
     masked_lm_example_loss, masked_lm_log_probs) = get_masked_lm_output(
      bert_config, model.get_sequence_output(), model.get_embedding_table(),
      masked_lm_positions, masked_lm_ids, masked_lm_weights)

    (next_sentence_loss, next_sentence_example_loss,
     next_sentence_log_probs) = get_next_sentence_output(
      bert_config, model.get_pooled_output(), next_sentence_labels, FLAGS.next_sentence_type)

    if do_return_all_attention_maps:
      for i in range(19):
        print('$$$start$$$')
      attention_maps = model.get_all_encoder_attention_maps()
      print(type(attention_maps), len(attention_maps))
      for t in attention_maps:
        print(t.shape)

    masked_lm_loss = tf.identity(masked_lm_loss, name="masked_lm_loss")
    next_sentence_loss = tf.identity(next_sentence_loss, name="next_sentence_loss")

    total_loss = masked_lm_loss + next_sentence_loss
    total_loss = tf.identity(total_loss, name='total_loss')

    tvars = tf.compat.v1.trainable_variables()

    initialized_variable_names = {}
    if init_checkpoint and (not use_horovod or hvd.rank() == 0):
      (assignment_map, initialized_variable_names
       ) = modeling.get_assignment_map_from_checkpoint(tvars, init_checkpoint)

      tf.compat.v1.train.init_from_checkpoint(init_checkpoint, assignment_map)

    if not use_horovod or hvd.rank() == 0:
      tf.compat.v1.logging.info("**** Trainable Variables ****")
      for var in tvars:
        init_string = ""
        if var.name in initialized_variable_names:
          init_string = ", *INIT_FROM_CKPT*"
        tf.compat.v1.logging.info("  name = %s, shape = %s%s", var.name, var.shape, init_string)

    output_spec = None
    if mode == tf.estimator.ModeKeys.TRAIN:
      train_op = optimization.create_optimizer(
        total_loss, learning_rate, beta1, beta2, epsilon, num_decay_steps, num_warmup_steps,
        hvd if use_horovod else None, use_fp16, num_accumulate_steps,
        FLAGS.optimizer_type, FLAGS.allreduce_post_accumulation, ignore_pooler=FLAGS.ignore_pooler)

      output_spec = tf.estimator.EstimatorSpec(
        mode=mode,
        loss=total_loss,
        train_op=train_op)
    elif mode == tf.estimator.ModeKeys.EVAL:

      def metric_fn(masked_lm_example_loss, masked_lm_log_probs, masked_lm_ids,
                    masked_lm_weights, next_sentence_example_loss,
                    next_sentence_log_probs, next_sentence_labels):
        """Computes the loss and accuracy of the model."""
        masked_lm_log_probs = tf.reshape(masked_lm_log_probs,
                                         [-1, masked_lm_log_probs.shape[-1]])
        masked_lm_predictions = tf.argmax(
          masked_lm_log_probs, axis=-1, output_type=tf.int32)
        masked_lm_example_loss = tf.reshape(masked_lm_example_loss, [-1])
        masked_lm_ids = tf.reshape(masked_lm_ids, [-1])
        masked_lm_weights = tf.reshape(masked_lm_weights, [-1])
        masked_lm_accuracy = tf.compat.v1.metrics.accuracy(
          labels=masked_lm_ids,
          predictions=masked_lm_predictions,
          weights=masked_lm_weights)
        masked_lm_mean_loss = tf.compat.v1.metrics.mean(
          values=masked_lm_example_loss, weights=masked_lm_weights)

        next_sentence_log_probs = tf.reshape(
          next_sentence_log_probs, [-1, next_sentence_log_probs.shape[-1]])
        next_sentence_predictions = tf.argmax(
          next_sentence_log_probs, axis=-1, output_type=tf.int32)
        next_sentence_labels = tf.reshape(next_sentence_labels, [-1])
        next_sentence_accuracy = tf.compat.v1.metrics.accuracy(
          labels=next_sentence_labels, predictions=next_sentence_predictions)
        next_sentence_mean_loss = tf.compat.v1.metrics.mean(
          values=next_sentence_example_loss)

        return {
          "masked_lm_accuracy": masked_lm_accuracy,
          "masked_lm_loss": masked_lm_mean_loss,
          "next_sentence_accuracy": next_sentence_accuracy,
          "next_sentence_loss": next_sentence_mean_loss,
        }

      eval_metrics = metric_fn(
        masked_lm_example_loss, masked_lm_log_probs, masked_lm_ids,
        masked_lm_weights, next_sentence_example_loss,
        next_sentence_log_probs, next_sentence_labels)
      output_spec = tf.estimator.EstimatorSpec(
        mode=mode,
        loss=total_loss,
        eval_metric_ops=eval_metrics)
    else:
      raise ValueError("Only TRAIN and EVAL modes are supported: %s" % (mode))

    return output_spec

  return model_fn


def get_masked_lm_output(labert_config, input_tensor, output_weights, positions,
                         label_ids, label_weights):
  """Get loss and log probs for the masked LM."""
  input_tensor = gather_indexes(input_tensor, positions)

  with tf.compat.v1.variable_scope("cls/predictions"):
    # We apply one more non-linear transformation before the output layer.
    # This matrix is not used after pre-training.
    with tf.compat.v1.variable_scope("transform"):
      input_tensor = tf.layers.dense(
        input_tensor,
        units=labert_config.embedding_size,
        activation=modeling.get_activation(labert_config.hidden_act),
        kernel_initializer=modeling.create_initializer(
          labert_config.initializer_range))
      input_tensor = modeling.layer_norm(input_tensor)

    # The output weights are the same as the input embeddings, but there is
    # an output-only bias for each token.
    output_bias = tf.compat.v1.get_variable(
      "output_bias",
      shape=[labert_config.vocab_size],
      initializer=tf.zeros_initializer())

    logits = tf.matmul(tf.cast(input_tensor, tf.float32), output_weights, transpose_b=True)
    logits = tf.nn.bias_add(logits, output_bias)
    log_probs = tf.nn.log_softmax(logits, axis=-1)

    label_ids = tf.reshape(label_ids, [-1])
    label_weights = tf.reshape(label_weights, [-1])

    one_hot_labels = tf.one_hot(
      label_ids, depth=labert_config.vocab_size, dtype=tf.float32)

    # The `positions` tensor might be zero-padded (if the sequence is too
    # short to have the maximum number of predictions). The `label_weights`
    # tensor has a value of 1.0 for every real prediction and 0.0 for the
    # padding predictions.
    per_example_loss = -tf.reduce_sum(log_probs * one_hot_labels, axis=[-1])
    numerator = tf.reduce_sum(label_weights * per_example_loss)
    denominator = tf.reduce_sum(label_weights) + 1e-5
    loss = numerator / denominator

  return (loss, per_example_loss, log_probs)


def get_next_sentence_output(bert_config, input_tensor, labels, num_classes):
  """Get loss and log probs for the next sentence prediction."""

  # Simple binary classification. Note that 0 is "next sentence" and 1 is
  # "random sentence". This weight matrix is not used after pre-training.
  with tf.compat.v1.variable_scope("cls/seq_relationship"):
    output_weights = tf.compat.v1.get_variable(
      "output_weights",
      shape=[num_classes, bert_config.hidden_size],
      initializer=modeling.create_initializer(bert_config.initializer_range))
    output_bias = tf.compat.v1.get_variable(
      "output_bias", shape=[num_classes], initializer=tf.zeros_initializer())

    logits = tf.matmul(tf.cast(input_tensor, tf.float32), output_weights, transpose_b=True)
    logits = tf.nn.bias_add(logits, output_bias)
    log_probs = tf.nn.log_softmax(logits, axis=-1)
    labels = tf.reshape(labels, [-1])
    one_hot_labels = tf.one_hot(labels, depth=num_classes, dtype=tf.float32)
    per_example_loss = -tf.reduce_sum(one_hot_labels * log_probs, axis=-1)
    loss = tf.reduce_mean(per_example_loss)
    return (loss, per_example_loss, log_probs)


def gather_indexes(sequence_tensor, positions):
  """Gathers the vectors at the specific positions over a minibatch."""
  sequence_shape = modeling.get_shape_list(sequence_tensor, expected_rank=3)
  batch_size = sequence_shape[0]
  seq_length = sequence_shape[1]
  width = sequence_shape[2]

  flat_offsets = tf.reshape(
    tf.range(0, batch_size, dtype=tf.int32) * seq_length, [-1, 1])
  flat_positions = tf.reshape(positions + flat_offsets, [-1])
  flat_sequence_tensor = tf.reshape(sequence_tensor,
                                    [batch_size * seq_length, width])
  output_tensor = tf.gather(flat_sequence_tensor, flat_positions)
  return output_tensor


def input_fn_builder(input_files,
                     max_seq_length,
                     max_predictions_per_seq,
                     is_training,
                     num_cpu_threads=4):
  """Creates an `input_fn` closure to be passed to TPUEstimator."""

  def input_fn(params):
    """The actual input function."""
    batch_size = FLAGS.train_batch_size if is_training else FLAGS.eval_batch_size

    name_to_features = {
      "input_ids":
        tf.io.FixedLenFeature([max_seq_length], tf.int64),
      "input_mask":
        tf.io.FixedLenFeature([max_seq_length], tf.int64),
      "segment_ids":
        tf.io.FixedLenFeature([max_seq_length], tf.int64),
      "positional_embeddings_start":
        tf.io.FixedLenFeature([max_seq_length], tf.int64),
      "positional_embeddings_end":
        tf.io.FixedLenFeature([max_seq_length], tf.int64),
      "masked_lm_positions":
        tf.io.FixedLenFeature([max_predictions_per_seq], tf.int64),
      "masked_lm_ids":
        tf.io.FixedLenFeature([max_predictions_per_seq], tf.int64),
      "masked_lm_weights":
        tf.io.FixedLenFeature([max_predictions_per_seq], tf.float32),
      "next_sentence_labels":
        tf.io.FixedLenFeature([1], tf.int64),
    }
    if FLAGS.adopt_mask_matrix:
      name_to_features['visibility_matrix'] = tf.io.SparseFeature(
        index_key=['visibility_matrix_i', 'visibility_matrix_j'],
        value_key='visibility_matrix_values',
        dtype=tf.int64, size=[max_seq_length, max_seq_length])

    # For training, we want a lot of parallel reading and shuffling.
    # For eval, we want no shuffling and parallel reading doesn't matter.
    if is_training:
      d = tf.data.Dataset.from_tensor_slices(tf.constant(input_files))
      if FLAGS.use_horovod:
        d = d.shard(hvd.size(), hvd.rank())
      d = d.repeat()
      d = d.shuffle(buffer_size=len(input_files))

      # `cycle_length` is the number of parallel files that get read.
      cycle_length = min(num_cpu_threads, len(input_files))

      # `sloppy` mode means that the interleaving is not exact. This adds
      # even more randomness to the training pipeline.
      d = d.apply(
        tf.contrib.data.parallel_interleave(
          tf.data.TFRecordDataset,
          sloppy=is_training,
          cycle_length=cycle_length))
      d = d.shuffle(buffer_size=100)
    else:
      d = tf.data.TFRecordDataset(input_files)
      # Since we evaluate for a fixed number of steps we don't want to encounter
      # out-of-range exceptions.
      d = d.repeat()

    # We must `drop_remainder` on training because the TPU requires fixed
    # size dimensions. For eval, we assume we are evaluating on the CPU or GPU
    # and we *don't* want to drop the remainder, otherwise we wont cover
    # every sample.
    d = d.apply(
      tf.contrib.data.map_and_batch(
        lambda record: _decode_record(record, name_to_features),
        batch_size=batch_size,
        num_parallel_batches=num_cpu_threads,
        drop_remainder=True))
    return d

  return input_fn


def _decode_record(record, name_to_features):
  """Decodes a record to a TensorFlow example."""
  example = tf.io.parse_single_example(record, name_to_features)

  # tf.Example only supports tf.int64, but the TPU only supports tf.int32.
  # So cast all int64 to int32.
  for name in list(example.keys()):
    t = example[name]
    if t.dtype == tf.int64:
      t = tf.cast(t, tf.int32)
    example[name] = t

  return example


def main(_):
  tf.compat.v1.logging.set_verbosity(tf.compat.v1.logging.INFO)
  if FLAGS.use_horovod:
    hvd.init()

  if not FLAGS.use_horovod or hvd.rank() == 0:
    tf.compat.v1.logging.info('config: \n' + tf.flags.FLAGS.flags_into_string())

  if not FLAGS.do_train and not FLAGS.do_eval:
    raise ValueError("At least one of `do_train` or `do_eval` must be True.")

  bert_config = modeling_labert.LaBertConfig.from_json_file(FLAGS.labert_config_file)

  tf.io.gfile.makedirs(FLAGS.output_dir)

  input_files, eval_files = [], []
  for input_pattern in FLAGS.input_file.split(","):
    input_files.extend(tf.io.gfile.glob(input_pattern))

  for input_pattern in FLAGS.eval_file.split(","):
    eval_files.extend(tf.io.gfile.glob(input_pattern))

  if not FLAGS.use_horovod or hvd.rank() == 0:
    tf.compat.v1.logging.info("*** Input Files ***")
    for input_file in input_files:
      tf.compat.v1.logging.info("  %s" % input_file)

    tf.compat.v1.logging.info("*** Evaluate Files ***")
    for eval_file in eval_files:
      tf.compat.v1.logging.info("  %s" % eval_file)

  config = tf.compat.v1.ConfigProto()
  if FLAGS.use_horovod:
    config.gpu_options.visible_device_list = str(hvd.local_rank())

  save_checkpoints_steps = None
  if not FLAGS.use_horovod or hvd.rank() == 0:
    save_checkpoints_steps = FLAGS.save_checkpoints_steps
  else:
    # If the current process is not the zero ranked process, never save checkpoints.
    save_checkpoints_steps = None
  run_config = tf.estimator.RunConfig(
    session_config=config,
    log_step_count_steps=1 << 25,
    model_dir=FLAGS.output_dir,
    save_checkpoints_steps=save_checkpoints_steps,
    keep_checkpoint_max=20)

  # according to https://arxiv.org/pdf/1706.02677.pdf, "When the minibatch size is
  # multiplied by k, multiply the learning rate by k." If it is true, the number
  # should be multiplied by the number of accumulated steps, because it also increases
  # the minibatch size.
  #
  # However,
  # https://lambdalabs.com/blog/bert-multi-gpu-implementation-using-tensorflow-and-horovod-with-code/
  # suggested that it is highly optional because "It caused significant performance
  # drop in several tasks we tested."
  lr = FLAGS.learning_rate
  if FLAGS.use_horovod and FLAGS.scaling_up_lr:
    lr = FLAGS.learning_rate * hvd.size()
  if FLAGS.num_accumulate_steps > 1 and FLAGS.scaling_up_lr:
    lr *= FLAGS.num_accumulate_steps

  model_fn = model_fn_builder(
    bert_config=bert_config,
    init_checkpoint=FLAGS.init_checkpoint,
    learning_rate=lr,
    beta1=FLAGS.adam_beta1,
    beta2=FLAGS.adam_beta2,
    epsilon=FLAGS.adam_epsilon,
    num_decay_steps=FLAGS.num_decay_steps if FLAGS.num_decay_steps else FLAGS.num_train_steps,
    num_warmup_steps=FLAGS.num_warmup_steps,
    num_accumulate_steps=FLAGS.num_accumulate_steps,
    use_one_hot_embeddings=FLAGS.use_tpu,
    use_horovod=FLAGS.use_horovod,
    use_fp16=FLAGS.use_fp16,
    do_return_all_attention_maps=FLAGS.do_return_all_attention_maps,)

  estimator = tf.estimator.Estimator(
    model_fn=model_fn,
    config=run_config)

  if FLAGS.do_train:
    training_hooks = []
    if FLAGS.use_horovod and hvd.size() > 1:
      training_hooks.append(hvd.BroadcastGlobalVariablesHook(0))

    training_hooks.append(LossHook(interval=FLAGS.num_log_steps))
    tf.compat.v1.logging.info("***** Running training *****")
    tf.compat.v1.logging.info("  Batch size = %d", FLAGS.train_batch_size)
    random.shuffle(input_files)
    train_input_fn = input_fn_builder(
      input_files=input_files,
      max_seq_length=FLAGS.max_seq_length,
      max_predictions_per_seq=FLAGS.max_predictions_per_seq,
      is_training=True)

    estimator.train(input_fn=train_input_fn, hooks=training_hooks, max_steps=FLAGS.num_train_steps)

    shutil.copy(FLAGS.labert_config_file, os.path.join(FLAGS.output_dir, "labert_config.json"))

  if FLAGS.do_eval and (not FLAGS.use_horovod or hvd.rank() == 0):
    eval_input_fn = input_fn_builder(
      input_files=eval_files,
      max_seq_length=FLAGS.max_seq_length,
      max_predictions_per_seq=FLAGS.max_predictions_per_seq,
      is_training=False)

    if FLAGS.do_return_all_attention_maps:
      training_hooks = []
      training_hooks.append(LossHook(interval=FLAGS.num_log_steps))
      result = estimator.evaluate(input_fn=eval_input_fn, hooks=training_hooks, steps=FLAGS.max_eval_steps)
    else:
      result = estimator.evaluate(input_fn=eval_input_fn, steps=FLAGS.max_eval_steps)

    output_eval_file = os.path.join(FLAGS.output_dir, "eval_results.txt")
    with tf.io.gfile.GFile(output_eval_file, "w") as writer:
      tf.compat.v1.logging.info("***** Eval results *****")
      for key in sorted(result.keys()):
        tf.compat.v1.logging.info("  %s = %s", key, str(result[key]))
        writer.write("%s = %s\n" % (key, str(result[key])))


if __name__ == "__main__":
  flags.mark_flag_as_required("input_file")
  flags.mark_flag_as_required("labert_config_file")
  flags.mark_flag_as_required("output_dir")
  tf.compat.v1.app.run()
