import type StorageManager from '@worldbrain/storex'
import { extractIdFromAnnotationUrl } from '@worldbrain/memex-common/lib/personal-cloud/backend/translation-layer/utils'
import { TEST_USER } from '@worldbrain/memex-common/lib/authentication/dev'
import { AnnotationPrivacyLevels } from '@worldbrain/memex-common/lib/annotations/types'
import { SYNCED_SETTING_KEYS } from '@worldbrain/memex-common/lib/synced-settings/constants'
import {
    ContentLocatorType,
    LocationSchemeType,
    ContentLocatorFormat,
    FingerprintSchemeType,
} from '@worldbrain/memex-common/lib/personal-cloud/storage/types'
import { buildMaterializedPath } from '@worldbrain/memex-common/lib/content-sharing/utils'
import { ROOT_NODE_PARENT_ID } from '@worldbrain/memex-common/lib/content-sharing/tree-utils'
import {
    DEFAULT_KEY,
    DEFAULT_SPACE_BETWEEN,
} from '@worldbrain/memex-common/lib/utils/item-ordering'

export async function insertTestPages(storageManager: StorageManager) {
    await storageManager
        .collection('pages')
        .createObject(LOCAL_TEST_DATA_V24.pages.first)
    await storageManager
        .collection('pages')
        .createObject(LOCAL_TEST_DATA_V24.pages.second)
}

export async function insertReadwiseAPIKey(
    storageManager: StorageManager,
    userId: string,
) {
    await storageManager.collection('personalMemexSetting').createObject({
        name: SYNCED_SETTING_KEYS.ReadwiseAPIKey,
        value: 'test-key',
        user: userId,
    })
}

const REMOTE_DEVICES_V24 = {
    first: {
        id: 1,
        user: TEST_USER.id,
    },
}

const LOCAL_PAGES_V24 = {
    first: {
        url: 'getmemexed.com/test',
        fullUrl: 'https://www.getmemexed.com/test',
        domain: 'getmemexed.com',
        hostname: 'getmemexed.com',
        fullTitle: 'getmemexed.com title',
        text: '',
        lang: 'en-GB',
        canonicalUrl: 'https://www.getmemexed.com/test',
    },
    second: {
        url: 'notionized.com/foo',
        fullUrl: 'https://www.notionized.com/foo',
        domain: 'notionized.com',
        hostname: 'notionized.com',
        fullTitle: 'notionized.com/foo title',
        text: '',
        lang: 'en-US',
        canonicalUrl: 'https://www.notionized.com/foo',
    },
    third: {
        url: 'memex.cloud/ct/test1.pdf',
        fullUrl: 'https://memex.cloud/ct/test1.pdf',
        domain: 'memex.cloud',
        hostname: 'memex.cloud',
        fullTitle: 'https://memex.cloud/ct/test1.pdf',
        text: '',
        lang: undefined,
        canonicalUrl: 'https://memex.cloud/ct/test1.pdf',
    },
    fourth: {
        url: 'memex.cloud/ct/test2.pdf',
        fullUrl: 'https://memex.cloud/ct/test2.pdf',
        domain: 'memex.cloud',
        hostname: 'memex.cloud',
        fullTitle: 'https://memex.cloud/ct/test2.pdf',
        text: '',
        lang: undefined,
        canonicalUrl: 'https://memex.cloud/ct/test2.pdf',
    },
    twitter_a: {
        url: 'twitter.com/zzzzz/status/1111011338575481374',
        fullUrl: 'https://twitter.com/zzzzz/status/1111011338575481374',
        domain: 'twitter.com',
        hostname: 'twitter.com',
        fullTitle: 'twitter.com/zzzzz/status/1111011338575481374',
        text: '',
        lang: undefined,
        canonicalUrl: 'https://twitter.com/zzzzz/status/1111011338575481374',
    },
    twitter_b: {
        url: 'mobile.twitter.com/zzzzz/status/1111011338575481355',
        fullUrl: 'https://mobile.twitter.com/zzzzz/status/1111011338575481355',
        domain: 'twitter.com',
        hostname: 'mobile.twitter.com',
        fullTitle: 'mobile.twitter.com/zzzzz/status/1111011338575481355',
        text: '',
        lang: undefined,
        canonicalUrl:
            'https://mobile.twitter.com/zzzzz/status/1111011338575481355',
    },
}

const LOCAL_ANNOTATIONS_V24 = {
    first: {
        url: LOCAL_PAGES_V24.first.url + '/#111111111',
        pageUrl: LOCAL_PAGES_V24.first.url,
        pageTitle: LOCAL_PAGES_V24.first.fullTitle,
        body: 'This is a test highlight',
        comment: 'This is a test comment',
        createdWhen: new Date('2020-10-10'),
        lastEdited: new Date('2020-10-10'),
        color: '#ffffff',
        selector: {
            quote: 'This is a test highlight',
            descriptor: { strategy: 'hyp-anchoring', content: [] },
        },
    },
    second: {
        url: LOCAL_PAGES_V24.first.url + '/#111111112',
        pageUrl: LOCAL_PAGES_V24.first.url,
        pageTitle: LOCAL_PAGES_V24.first.fullTitle,
        comment: 'This is another test comment',
        createdWhen: new Date('2020-10-11'),
        lastEdited: new Date('2020-10-20'),
    },
    third: {
        url: LOCAL_PAGES_V24.third.url + '/#111111113',
        pageUrl: LOCAL_PAGES_V24.third.url,
        pageTitle: LOCAL_PAGES_V24.third.fullTitle,
        comment: 'This is a test comment on a PDF page!',
        createdWhen: new Date('2020-10-12'),
        lastEdited: new Date('2020-10-12'),
    },
    fourth: {
        url: LOCAL_PAGES_V24.third.url + '/#111111114',
        pageUrl: LOCAL_PAGES_V24.third.url,
        pageTitle: LOCAL_PAGES_V24.third.fullTitle,
        comment: 'This is another test comment on a PDF page',
        createdWhen: new Date('2020-10-13'),
        lastEdited: new Date('2020-10-13'),
    },
    fifth: {
        url: LOCAL_PAGES_V24.fourth.url + '/#111111115',
        pageUrl: LOCAL_PAGES_V24.fourth.url,
        pageTitle: LOCAL_PAGES_V24.fourth.fullTitle,
        comment: 'This is a comment on a local PDF page',
        createdWhen: new Date('2020-10-14'),
        lastEdited: new Date('2020-10-14'),
    },
}

const LOCAL_LISTS_V24 = {
    first: {
        id: 1619414286452,
        name: 'Test list A',
        searchableName: 'Test list A',
        createdAt: new Date(1619414286452),
        isDeletable: true,
        isNestable: true,
    },
    second: {
        id: 1619414286456,
        name: 'Test list B',
        searchableName: 'Test list B',
        createdAt: new Date(1619414286456),
        isDeletable: true,
        isNestable: true,
    },
    third: {
        id: 1698120000000,
        name: 'Test list C',
        searchableName: 'Test list C',
        createdAt: new Date(1698120000000),
        isDeletable: true,
        isNestable: true,
    },
    fourth: {
        id: 1698120000001,
        name: 'Test list D',
        searchableName: 'Test list D',
        createdAt: new Date(1698120000001),
        isDeletable: true,
        isNestable: true,
    },
}

const LOCAL_SHARED_LIST_METADATA_V24 = {
    first: {
        localId: LOCAL_LISTS_V24.first.id,
        remoteId: 'test-1',
        private: false,
    },
    second: {
        localId: LOCAL_LISTS_V24.second.id,
        remoteId: 'test-2',
        private: false,
    },
    third: {
        localId: LOCAL_LISTS_V24.third.id,
        remoteId: 'test-3',
        private: false,
    },
    fourth: {
        localId: LOCAL_LISTS_V24.fourth.id,
        remoteId: 'test-4',
        private: false,
    },
}

export const LOCAL_TEST_DATA_V24 = {
    pages: LOCAL_PAGES_V24,
    pageMetadata: {
        first: {
            normalizedPageUrl: LOCAL_PAGES_V24.first.url,
            doi: 'test-doi',
            title: 'test title',
            // annotation?: string
            // sourceName?: string
            // journalName?: string
            // journalPage?: string
            // journalIssue?: string
            // journalVolume?: string
            // releaseDate?: number
            accessDate: 1709782039294,
        },
    },
    pageEntities: {
        first: {
            id: 1,
            normalizedPageUrl: LOCAL_PAGES_V24.first.url,
            name: 'Smith',
            isPrimary: true,
            additionalName: 'John D.',
            order: DEFAULT_KEY,
        },
        second: {
            id: 2,
            normalizedPageUrl: LOCAL_PAGES_V24.first.url,
            name: 'Google',
            isPrimary: false,
            order: DEFAULT_KEY + 50,
        },
    },
    bookmarks: {
        first: {
            url: LOCAL_PAGES_V24.first.url,
            time: 1622010270273,
        },
    },
    annotations: LOCAL_ANNOTATIONS_V24,
    annotationPrivacyLevels: {
        first: {
            id: 1,
            annotation: LOCAL_ANNOTATIONS_V24.first.url,
            privacyLevel: AnnotationPrivacyLevels.SHARED,
            createdWhen: new Date(1625190554983),
            updatedWhen: new Date(1625190554983),
        },
        first_private: {
            id: 1,
            annotation: LOCAL_ANNOTATIONS_V24.first.url,
            privacyLevel: AnnotationPrivacyLevels.PRIVATE,
            createdWhen: new Date(1625190554983),
            updatedWhen: new Date(1625190554983),
        },
        second: {
            id: 2,
            annotation: LOCAL_ANNOTATIONS_V24.second.url,
            privacyLevel: AnnotationPrivacyLevels.SHARED_PROTECTED,
            createdWhen: new Date(1625190554984),
            updatedWhen: new Date(1625190554984),
        },
        third: {
            id: 3,
            annotation: LOCAL_ANNOTATIONS_V24.third.url,
            privacyLevel: AnnotationPrivacyLevels.SHARED,
            createdWhen: new Date(1625190554985),
            updatedWhen: new Date(1625190554985),
        },
        fifth: {
            id: 4,
            annotation: LOCAL_ANNOTATIONS_V24.fifth.url,
            privacyLevel: AnnotationPrivacyLevels.SHARED,
            createdWhen: new Date(1625190554986),
            updatedWhen: new Date(1625190554986),
        },
    },
    sharedAnnotationMetadata: {
        first: {
            excludeFromLists: false,
            localId: LOCAL_ANNOTATIONS_V24.first.url,
            remoteId: 'test-1',
        },
        second: {
            excludeFromLists: true,
            localId: LOCAL_ANNOTATIONS_V24.second.url,
            remoteId: 'test-2',
        },
        third: {
            excludeFromLists: false,
            localId: LOCAL_ANNOTATIONS_V24.third.url,
            remoteId: 'test-3',
        },
        fifth: {
            excludeFromLists: false,
            localId: LOCAL_ANNOTATIONS_V24.fifth.url,
            remoteId: 'test-4',
        },
    },
    locators: {
        // NOTE: first and second are skipped, as the corresponding pages are not PDFs.
        //   Only pages that are PDFs will have local extension locators.
        third: {
            id: 1,
            fingerprint: 'test-fingerprint-1',
            fingerprintScheme: FingerprintSchemeType.PdfV1,
            location: 'getmemex.com/test1.pdf',
            locationType: ContentLocatorType.Remote,
            locationScheme: LocationSchemeType.NormalizedUrlV1,
            normalizedUrl: LOCAL_PAGES_V24.third.url,
            originalLocation: 'https://getmemex.com/test1.pdf',
            format: ContentLocatorFormat.PDF,
            lastVisited: 1635927733923,
            primary: true,
            valid: true,
            version: 0,
            deviceId: REMOTE_DEVICES_V24.first.id,
        },
        // NOTE: all the fourth_* locators point to the same page
        fourth_a: {
            id: 2,
            fingerprint: 'test-fingerprint-2',
            fingerprintScheme: FingerprintSchemeType.PdfV1,
            location: '/home/user/test2.pdf',
            locationType: ContentLocatorType.Local,
            locationScheme: LocationSchemeType.FilesystemPathV1,
            normalizedUrl: LOCAL_PAGES_V24.fourth.url,
            originalLocation: 'file:///home/user/test2.pdf',
            format: ContentLocatorFormat.PDF,
            lastVisited: 1635927733925,
            primary: true,
            valid: true,
            version: 0,
            deviceId: REMOTE_DEVICES_V24.first.id,
        },
        fourth_b: {
            id: 3,
            fingerprint: 'test-fingerprint-3',
            fingerprintScheme: FingerprintSchemeType.PdfV1,
            location: 'getmemex.com/test2.pdf',
            locationType: ContentLocatorType.Remote,
            locationScheme: LocationSchemeType.NormalizedUrlV1,
            normalizedUrl: LOCAL_PAGES_V24.fourth.url,
            originalLocation: 'https://getmemex.com/test2.pdf',
            format: ContentLocatorFormat.PDF,
            lastVisited: 1635927743925,
            primary: true,
            valid: true,
            version: 0,
            deviceId: REMOTE_DEVICES_V24.first.id,
        },
        fourth_uploading: {
            id: 4,
            fingerprint: null,
            fingerprintScheme: null,
            location: 'test-upload-id-1',
            locationType: ContentLocatorType.Remote,
            locationScheme: LocationSchemeType.UploadStorage,
            normalizedUrl: LOCAL_PAGES_V24.fourth.url,
            originalLocation: LOCAL_PAGES_V24.fourth.fullUrl,
            format: ContentLocatorFormat.PDF,
            lastVisited: 1635927743925,
            status: 'uploading',
            primary: true,
            valid: true,
            version: 0,
            deviceId: REMOTE_DEVICES_V24.first.id,
        },
        fourth_uploaded: {
            id: 4,
            fingerprint: null,
            fingerprintScheme: null,
            location: 'test-upload-id-1',
            locationType: ContentLocatorType.Remote,
            locationScheme: LocationSchemeType.UploadStorage,
            normalizedUrl: LOCAL_PAGES_V24.fourth.url,
            originalLocation: LOCAL_PAGES_V24.fourth.fullUrl,
            format: ContentLocatorFormat.PDF,
            lastVisited: 1635927743925,
            status: 'uploaded',
            primary: true,
            valid: true,
            version: 0,
            deviceId: REMOTE_DEVICES_V24.first.id,
        },
    },
    visits: {
        first: {
            url: LOCAL_PAGES_V24.first.url,
            time: 4545,
            duration: 1122,
            scrollMaxPerc: 100,
            scrollMaxPx: 500,
            scrollPerc: 50,
            scrollPx: 250,
        },
        second: {
            url: LOCAL_PAGES_V24.second.url,
            time: 5545,
            duration: 174,
            scrollMaxPerc: 70,
            scrollMaxPx: 320,
            scrollPerc: 30,
            scrollPx: 130,
        },
    },
    tags: {
        firstPageTag: {
            url: LOCAL_PAGES_V24.first.url,
            name: 'foo-tag',
        },
        secondPageTag: {
            url: LOCAL_PAGES_V24.second.url,
            name: 'foo-tag',
        },
        firstAnnotationTag: {
            url: LOCAL_ANNOTATIONS_V24.first.url,
            name: 'annot-tag',
        },
        secondAnnotationTag: {
            url: LOCAL_ANNOTATIONS_V24.second.url,
            name: 'annot-tag',
        },
    },
    customLists: LOCAL_LISTS_V24,
    customListTrees: {
        first: {
            id: 1,
            listId: LOCAL_LISTS_V24.first.id,
            linkTarget: null,
            parentListId: ROOT_NODE_PARENT_ID,
            path: null,
            order: 5,
            createdWhen: 1698124786131,
            updatedWhen: 1698124786131,
        },
        second: {
            id: 2,
            listId: LOCAL_LISTS_V24.second.id,
            linkTarget: null,
            parentListId: LOCAL_LISTS_V24.first.id,
            path: buildMaterializedPath(LOCAL_LISTS_V24.first.id),
            order: 10,
            createdWhen: 1698124786132,
            updatedWhen: 1698124786132,
        },
        third: {
            id: 3,
            listId: LOCAL_LISTS_V24.third.id,
            linkTarget: null,
            parentListId: LOCAL_LISTS_V24.first.id,
            path: buildMaterializedPath(LOCAL_LISTS_V24.first.id),
            order: 20,
            createdWhen: 1698124786133,
            updatedWhen: 1698124786133,
        },
        fourth: {
            id: 4,
            listId: LOCAL_LISTS_V24.fourth.id,
            linkTarget: null,
            parentListId: LOCAL_LISTS_V24.third.id,
            path: buildMaterializedPath(
                LOCAL_LISTS_V24.first.id,
                LOCAL_LISTS_V24.third.id,
            ),
            order: 30,
            createdWhen: 1698124786134,
            updatedWhen: 1698124786134,
        },
        fifth: {
            id: 5,
            listId: null,
            linkTarget: LOCAL_LISTS_V24.first.id,
            parentListId: LOCAL_LISTS_V24.second.id,
            path: buildMaterializedPath(
                LOCAL_LISTS_V24.first.id,
                LOCAL_LISTS_V24.second.id,
            ),
            order: 40,
            createdWhen: 1698124786135,
            updatedWhen: 1698124786135,
        },
    },
    customListDescriptions: {
        first: {
            listId: LOCAL_LISTS_V24.first.id,
            description: 'test description first list',
        },
        second: {
            listId: LOCAL_LISTS_V24.second.id,
            description: 'test description second list',
        },
    },
    sharedListMetadata: LOCAL_SHARED_LIST_METADATA_V24,
    pageListEntries: {
        first: {
            createdAt: new Date(1625190554480),
            fullUrl: LOCAL_PAGES_V24.first.fullUrl,
            pageUrl: LOCAL_PAGES_V24.first.url,
            listId: LOCAL_LISTS_V24.first.id,
        },
        second: {
            createdAt: new Date(1625190554986),
            fullUrl: LOCAL_PAGES_V24.second.fullUrl,
            pageUrl: LOCAL_PAGES_V24.second.url,
            listId: LOCAL_LISTS_V24.first.id,
        },
        third: {
            createdAt: new Date(1625190554990),
            fullUrl: LOCAL_PAGES_V24.third.fullUrl,
            pageUrl: LOCAL_PAGES_V24.third.url,
            listId: LOCAL_LISTS_V24.first.id,
        },
        fourth: {
            createdAt: new Date(1625190554990),
            fullUrl: LOCAL_PAGES_V24.fourth.fullUrl,
            pageUrl: LOCAL_PAGES_V24.fourth.url,
            listId: LOCAL_LISTS_V24.first.id,
        },
        fifth: {
            createdAt: new Date(1625190554991),
            fullUrl: LOCAL_PAGES_V24.second.fullUrl,
            pageUrl: LOCAL_PAGES_V24.second.url,
            listId: LOCAL_LISTS_V24.second.id,
        },
    },
    annotationListEntries: {
        first: {
            listId: LOCAL_LISTS_V24.first.id,
            url: LOCAL_ANNOTATIONS_V24.first.url,
            createdAt: new Date(1625190554990),
        },
        second: {
            listId: LOCAL_LISTS_V24.first.id,
            url: LOCAL_ANNOTATIONS_V24.second.url,
            createdAt: new Date(1625190554991),
        },
    },
    followedList: {
        first: {
            creator: TEST_USER.id,
            name: LOCAL_LISTS_V24.first.name,
            sharedList: LOCAL_SHARED_LIST_METADATA_V24.first.remoteId,
        },
    },
    templates: {
        first: {
            id: 1,
            isFavourite: false,
            title: 'Roam Markdown',
            code: '[[{{{PageTitle}}}]]',
            outputFormat: 'markdown',
            order: DEFAULT_KEY,
        },
        second: {
            id: 2,
            isFavourite: true,
            title: 'Other Markdown',
            code: '[[{{{PageUrl}}}]]',
            outputFormat: 'markdown',
            order: DEFAULT_KEY + DEFAULT_SPACE_BETWEEN,
        },
    },
    settings: {
        first: {
            key: 'test-1',
            value: 123,
        },
        second: {
            key: 'test-2',
            value: '123',
        },
        third: {
            key: 'test-3',
            value: { sub: 'hi' },
        },
    },
}

const REMOTE_METADATA_V24 = {
    first: {
        id: 1,
        createdWhen: 555,
        updatedWhen: 555,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        canonicalUrl: LOCAL_TEST_DATA_V24.pages.first.canonicalUrl,
        title: LOCAL_TEST_DATA_V24.pages.first.fullTitle,
        lang: LOCAL_TEST_DATA_V24.pages.first.lang,
    },
    second: {
        id: 2,
        createdWhen: 557,
        updatedWhen: 557,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        canonicalUrl: LOCAL_TEST_DATA_V24.pages.second.canonicalUrl,
        title: LOCAL_TEST_DATA_V24.pages.second.fullTitle,
        lang: LOCAL_TEST_DATA_V24.pages.second.lang,
    },
    third: {
        id: 3,
        createdWhen: 559,
        updatedWhen: 559,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        canonicalUrl: LOCAL_TEST_DATA_V24.pages.third.canonicalUrl,
        title: LOCAL_TEST_DATA_V24.pages.third.fullTitle,
        lang: LOCAL_TEST_DATA_V24.pages.third.lang ?? null,
    },
    fourth: {
        id: 4,
        createdWhen: 561,
        updatedWhen: 561,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        canonicalUrl: LOCAL_TEST_DATA_V24.pages.fourth.canonicalUrl,
        title: LOCAL_TEST_DATA_V24.pages.fourth.fullTitle,
        lang: LOCAL_TEST_DATA_V24.pages.fourth.lang ?? null,
    },
    twitter_a: {
        id: 1,
        createdWhen: 598,
        updatedWhen: 598,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        canonicalUrl: LOCAL_TEST_DATA_V24.pages.twitter_a.canonicalUrl,
        title: LOCAL_TEST_DATA_V24.pages.twitter_a.fullTitle,
        lang: LOCAL_TEST_DATA_V24.pages.twitter_a.lang ?? null,
    },
    twitter_b: {
        id: 2,
        createdWhen: 599,
        updatedWhen: 599,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        canonicalUrl: LOCAL_TEST_DATA_V24.pages.twitter_b.canonicalUrl,
        title: LOCAL_TEST_DATA_V24.pages.twitter_b.fullTitle,
        lang: LOCAL_TEST_DATA_V24.pages.twitter_b.lang ?? null,
    },
}

// Contains fields that would appear in the first personalContentMetadata if local pageMetadata was also created
export const REMOTE_METADATA_V24_AMENDED = {
    first: {
        doi: LOCAL_TEST_DATA_V24.pageMetadata.first.doi,
        metadataTitle: LOCAL_TEST_DATA_V24.pageMetadata.first.title,
        accessDate: LOCAL_TEST_DATA_V24.pageMetadata.first.accessDate,
    },
}

const REMOTE_LOCATORS_V24 = {
    first: {
        id: 1,
        createdWhen: 556,
        updatedWhen: 556,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.first.id,
        format: ContentLocatorFormat.HTML,
        location: LOCAL_TEST_DATA_V24.pages.first.url,
        locationScheme: LocationSchemeType.NormalizedUrlV1,
        locationType: ContentLocatorType.Remote,
        originalLocation: LOCAL_TEST_DATA_V24.pages.first.fullUrl,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: 0,
        localId: null,
        // contentSize: null,
        // fingerprint: null,
    },
    second: {
        id: 2,
        createdWhen: 558,
        updatedWhen: 558,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.second.id,
        format: ContentLocatorFormat.HTML,
        location: LOCAL_TEST_DATA_V24.pages.second.url,
        locationScheme: LocationSchemeType.NormalizedUrlV1,
        locationType: ContentLocatorType.Remote,
        originalLocation: LOCAL_TEST_DATA_V24.pages.second.fullUrl,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: 0,
        localId: null,
        // contentSize: null,
        // fingerprint: null,
    },
    // As the third page is a PDF, an extra base locator exists to point to the contentMetadata
    //  memex.cloud/ct/ page that gets created locally for the PDF
    third_base: {
        id: 3,
        createdWhen: 560,
        updatedWhen: 560,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.third.id,
        localId: null,
        format: ContentLocatorFormat.PDF,
        location: LOCAL_TEST_DATA_V24.pages.third.url,
        locationScheme: LocationSchemeType.NormalizedUrlV1,
        locationType: ContentLocatorType.MemexCloud,
        originalLocation: LOCAL_TEST_DATA_V24.pages.third.fullUrl,
        primary: true,
        valid: true,
        version: 0,
        fingerprintScheme: null,
        contentSize: null,
        fingerprint: null,
        lastVisited: 0,
    },
    third: {
        id: 4,
        createdWhen: 562,
        updatedWhen: 562,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.third.id,
        localId: LOCAL_TEST_DATA_V24.locators.third.id,
        format: LOCAL_TEST_DATA_V24.locators.third.format,
        location: LOCAL_TEST_DATA_V24.locators.third.location,
        locationScheme: LOCAL_TEST_DATA_V24.locators.third.locationScheme,
        locationType: LOCAL_TEST_DATA_V24.locators.third.locationType,
        originalLocation: LOCAL_TEST_DATA_V24.locators.third.originalLocation,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: LOCAL_TEST_DATA_V24.locators.third.lastVisited,
        // contentSize: null,
        fingerprint: LOCAL_TEST_DATA_V24.locators.third.fingerprint,
        fingerprintScheme: LOCAL_TEST_DATA_V24.locators.third.fingerprintScheme,
    },
    fourth_base: {
        id: 5,
        createdWhen: 564,
        updatedWhen: 564,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.fourth.id,
        localId: null,
        format: ContentLocatorFormat.PDF,
        location: LOCAL_TEST_DATA_V24.pages.fourth.url,
        locationScheme: LocationSchemeType.NormalizedUrlV1,
        locationType: ContentLocatorType.MemexCloud,
        originalLocation: LOCAL_TEST_DATA_V24.pages.fourth.fullUrl,
        primary: true,
        valid: true,
        version: 0,
        fingerprintScheme: null,
        contentSize: null,
        fingerprint: null,
        lastVisited: 0,
    },
    fourth_a: {
        id: 6,
        createdWhen: 566,
        updatedWhen: 566,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.fourth.id,
        localId: LOCAL_TEST_DATA_V24.locators.fourth_a.id,
        format: LOCAL_TEST_DATA_V24.locators.fourth_a.format,
        location: LOCAL_TEST_DATA_V24.locators.fourth_a.location,
        locationScheme: LOCAL_TEST_DATA_V24.locators.fourth_a.locationScheme,
        locationType: LOCAL_TEST_DATA_V24.locators.fourth_a.locationType,
        originalLocation:
            LOCAL_TEST_DATA_V24.locators.fourth_a.originalLocation,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: LOCAL_TEST_DATA_V24.locators.fourth_a.lastVisited,
        // contentSize: null,
        fingerprint: LOCAL_TEST_DATA_V24.locators.fourth_a.fingerprint,
        fingerprintScheme:
            LOCAL_TEST_DATA_V24.locators.fourth_a.fingerprintScheme,
    },
    fourth_b: {
        id: 7,
        createdWhen: 566,
        updatedWhen: 566,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.fourth.id,
        localId: LOCAL_TEST_DATA_V24.locators.fourth_b.id,
        format: LOCAL_TEST_DATA_V24.locators.fourth_b.format,
        location: LOCAL_TEST_DATA_V24.locators.fourth_b.location,
        locationScheme: LOCAL_TEST_DATA_V24.locators.fourth_b.locationScheme,
        locationType: LOCAL_TEST_DATA_V24.locators.fourth_b.locationType,
        originalLocation:
            LOCAL_TEST_DATA_V24.locators.fourth_b.originalLocation,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: LOCAL_TEST_DATA_V24.locators.fourth_b.lastVisited,
        // contentSize: null,
        fingerprint: LOCAL_TEST_DATA_V24.locators.fourth_b.fingerprint,
        fingerprintScheme:
            LOCAL_TEST_DATA_V24.locators.fourth_b.fingerprintScheme,
    },
    fourth_c_uploading: {
        id: 8,
        createdWhen: 566,
        updatedWhen: 566,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.fourth.id,
        localId: LOCAL_TEST_DATA_V24.locators.fourth_uploading.id,
        format: LOCAL_TEST_DATA_V24.locators.fourth_uploading.format,
        location: LOCAL_TEST_DATA_V24.locators.fourth_uploading.location,
        locationScheme:
            LOCAL_TEST_DATA_V24.locators.fourth_uploading.locationScheme,
        locationType:
            LOCAL_TEST_DATA_V24.locators.fourth_uploading.locationType,
        originalLocation:
            LOCAL_TEST_DATA_V24.locators.fourth_uploading.originalLocation,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: LOCAL_TEST_DATA_V24.locators.fourth_uploading.lastVisited,
        // contentSize: null,
        status: 'uploading',
    },
    fourth_c_uploaded: {
        id: 8,
        createdWhen: 566,
        updatedWhen: 566,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.fourth.id,
        localId: LOCAL_TEST_DATA_V24.locators.fourth_uploading.id,
        format: LOCAL_TEST_DATA_V24.locators.fourth_uploading.format,
        location: LOCAL_TEST_DATA_V24.locators.fourth_uploading.location,
        locationScheme:
            LOCAL_TEST_DATA_V24.locators.fourth_uploading.locationScheme,
        locationType:
            LOCAL_TEST_DATA_V24.locators.fourth_uploading.locationType,
        originalLocation:
            LOCAL_TEST_DATA_V24.locators.fourth_uploading.originalLocation,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: LOCAL_TEST_DATA_V24.locators.fourth_uploading.lastVisited,
        // contentSize: null,
        status: 'uploaded',
    },
    twitter_a: {
        id: 1,
        createdWhen: 598,
        updatedWhen: 598,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.twitter_a.id,
        format: ContentLocatorFormat.HTML,
        location: LOCAL_TEST_DATA_V24.pages.twitter_a.url,
        locationScheme: LocationSchemeType.NormalizedUrlV1,
        locationType: ContentLocatorType.Remote,
        originalLocation: LOCAL_TEST_DATA_V24.pages.twitter_a.fullUrl,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: 0,
        localId: null,
    },
    twitter_b: {
        id: 2,
        createdWhen: 599,
        updatedWhen: 599,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalContentMetadata: REMOTE_METADATA_V24.twitter_b.id,
        format: ContentLocatorFormat.HTML,
        location: LOCAL_TEST_DATA_V24.pages.twitter_b.url,
        locationScheme: LocationSchemeType.NormalizedUrlV1,
        locationType: ContentLocatorType.Remote,
        originalLocation: LOCAL_TEST_DATA_V24.pages.twitter_b.fullUrl,
        primary: true,
        valid: true,
        version: 0,
        lastVisited: 0,
        localId: null,
    },
}

const REMOTE_ANNOTATIONS_V24 = {
    first: {
        id: 1,
        personalContentMetadata: REMOTE_METADATA_V24.first.id,
        localId: extractIdFromAnnotationUrl(
            LOCAL_TEST_DATA_V24.annotations.first.url,
        ),
        body: LOCAL_TEST_DATA_V24.annotations.first.body,
        comment: LOCAL_TEST_DATA_V24.annotations.first.comment,
        createdWhen: LOCAL_TEST_DATA_V24.annotations.first.createdWhen.getTime(),
        updatedWhen: LOCAL_TEST_DATA_V24.annotations.first.createdWhen.getTime(),
        color: LOCAL_TEST_DATA_V24.annotations.first.color,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
    },
    second: {
        id: 2,
        personalContentMetadata: REMOTE_METADATA_V24.first.id,
        localId: extractIdFromAnnotationUrl(
            LOCAL_TEST_DATA_V24.annotations.second.url,
        ),
        comment: LOCAL_TEST_DATA_V24.annotations.second.comment,
        createdWhen: LOCAL_TEST_DATA_V24.annotations.second.createdWhen.getTime(),
        updatedWhen: LOCAL_TEST_DATA_V24.annotations.second.createdWhen.getTime(),
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
    },
    third: {
        id: 1,
        personalContentMetadata: REMOTE_METADATA_V24.third.id,
        localId: extractIdFromAnnotationUrl(
            LOCAL_TEST_DATA_V24.annotations.third.url,
        ),
        comment: LOCAL_TEST_DATA_V24.annotations.third.comment,
        createdWhen: LOCAL_TEST_DATA_V24.annotations.third.createdWhen.getTime(),
        updatedWhen: LOCAL_TEST_DATA_V24.annotations.third.createdWhen.getTime(),
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
    },
    fifth: {
        id: 1,
        personalContentMetadata: REMOTE_METADATA_V24.third.id,
        localId: extractIdFromAnnotationUrl(
            LOCAL_TEST_DATA_V24.annotations.fifth.url,
        ),
        comment: LOCAL_TEST_DATA_V24.annotations.fifth.comment,
        createdWhen: LOCAL_TEST_DATA_V24.annotations.fifth.createdWhen.getTime(),
        updatedWhen: LOCAL_TEST_DATA_V24.annotations.fifth.createdWhen.getTime(),
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
    },
}

const REMOTE_ANNOTATION_SELECTORS_V24 = {
    first: {
        id: 1,
        createdWhen: 560,
        updatedWhen: 560,
        user: TEST_USER.id,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        personalAnnotation: REMOTE_ANNOTATIONS_V24.first.id,
        selector: {
            ...LOCAL_TEST_DATA_V24.annotations.first.selector,
        },
    },
}

const REMOTE_TAGS_V24 = {
    firstPageTag: {
        id: 1,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        createdWhen: 559,
        updatedWhen: 559,
        user: TEST_USER.id,
        name: LOCAL_TEST_DATA_V24.tags.firstPageTag.name,
    },
    firstAnnotationTag: {
        id: 1,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        createdWhen: 560,
        updatedWhen: 560,
        user: TEST_USER.id,
        name: LOCAL_TEST_DATA_V24.tags.firstAnnotationTag.name,
    },
    secondAnnotationTag: {
        id: 2,
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        createdWhen: 561,
        updatedWhen: 561,
        user: TEST_USER.id,
        name: LOCAL_TEST_DATA_V24.tags.secondAnnotationTag.name,
    },
}

const REMOTE_LISTS_V24 = {
    first: {
        id: 1,
        localId: LOCAL_LISTS_V24.first.id,
        name: LOCAL_LISTS_V24.first.name,
        isDeletable: LOCAL_LISTS_V24.first.isDeletable,
        isNestable: LOCAL_LISTS_V24.first.isNestable,
        createdWhen: LOCAL_LISTS_V24.first.createdAt.getTime(),
        updatedWhen: LOCAL_LISTS_V24.first.createdAt.getTime(),
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        user: TEST_USER.id,
    },
    second: {
        id: 2,
        localId: LOCAL_LISTS_V24.second.id,
        name: LOCAL_LISTS_V24.second.name,
        isDeletable: LOCAL_LISTS_V24.second.isDeletable,
        isNestable: LOCAL_LISTS_V24.second.isNestable,
        createdWhen: LOCAL_LISTS_V24.second.createdAt.getTime(),
        updatedWhen: LOCAL_LISTS_V24.second.createdAt.getTime(),
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        user: TEST_USER.id,
    },
    third: {
        id: 3,
        localId: LOCAL_LISTS_V24.third.id,
        name: LOCAL_LISTS_V24.third.name,
        isDeletable: LOCAL_LISTS_V24.third.isDeletable,
        isNestable: LOCAL_LISTS_V24.third.isNestable,
        createdWhen: LOCAL_LISTS_V24.third.createdAt.getTime(),
        updatedWhen: LOCAL_LISTS_V24.third.createdAt.getTime(),
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        user: TEST_USER.id,
    },
    fourth: {
        id: 4,
        localId: LOCAL_LISTS_V24.fourth.id,
        name: LOCAL_LISTS_V24.fourth.name,
        isDeletable: LOCAL_LISTS_V24.fourth.isDeletable,
        isNestable: LOCAL_LISTS_V24.fourth.isNestable,
        createdWhen: LOCAL_LISTS_V24.fourth.createdAt.getTime(),
        updatedWhen: LOCAL_LISTS_V24.fourth.createdAt.getTime(),
        createdByDevice: REMOTE_DEVICES_V24.first.id,
        user: TEST_USER.id,
    },
}

export const REMOTE_TEST_DATA_V24 = {
    personalDeviceInfo: REMOTE_DEVICES_V24,
    personalContentMetadata: REMOTE_METADATA_V24,
    personalContentLocator: REMOTE_LOCATORS_V24,
    personalContentRead: {
        first: {
            id: 1,
            createdWhen: 559,
            updatedWhen: 559,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            personalContentMetadata: REMOTE_METADATA_V24.first.id,
            personalContentLocator: REMOTE_LOCATORS_V24.first.id,
            readWhen: LOCAL_TEST_DATA_V24.visits.first.time,
            readDuration: LOCAL_TEST_DATA_V24.visits.first.duration,
            scrollMaxPercentage: LOCAL_TEST_DATA_V24.visits.first.scrollMaxPerc,
            scrollEndPercentage: LOCAL_TEST_DATA_V24.visits.first.scrollPerc,
            scrollMaxPixel: LOCAL_TEST_DATA_V24.visits.first.scrollMaxPx,
            scrollEndPixel: LOCAL_TEST_DATA_V24.visits.first.scrollPx,
            // pageTotal: null,
            // pageProgress: null,
        },
        second: {
            id: 2,
            createdWhen: 560,
            updatedWhen: 560,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            personalContentMetadata: REMOTE_METADATA_V24.first.id,
            personalContentLocator: REMOTE_LOCATORS_V24.first.id,
            readWhen: LOCAL_TEST_DATA_V24.visits.second.time,
            readDuration: LOCAL_TEST_DATA_V24.visits.second.duration,
            scrollMaxPercentage:
                LOCAL_TEST_DATA_V24.visits.second.scrollMaxPerc,
            scrollEndPercentage: LOCAL_TEST_DATA_V24.visits.second.scrollPerc,
            scrollMaxPixel: LOCAL_TEST_DATA_V24.visits.second.scrollMaxPx,
            scrollEndPixel: LOCAL_TEST_DATA_V24.visits.second.scrollPx,
            // pageTotal: null,
            // pageProgress: null,
        },
    },
    personalContentEntity: {
        first: {
            id: 1,
            user: TEST_USER.id,
            localId: LOCAL_TEST_DATA_V24.pageEntities.first.id,
            personalContentMetadata: REMOTE_METADATA_V24.first.id,
            order: LOCAL_TEST_DATA_V24.pageEntities.first.order,
            name: LOCAL_TEST_DATA_V24.pageEntities.first.name,
            isPrimary: LOCAL_TEST_DATA_V24.pageEntities.first.isPrimary,
            additionalName:
                LOCAL_TEST_DATA_V24.pageEntities.first.additionalName,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
        },
        second: {
            id: 2,
            user: TEST_USER.id,
            localId: LOCAL_TEST_DATA_V24.pageEntities.second.id,
            personalContentMetadata: REMOTE_METADATA_V24.first.id,
            order: LOCAL_TEST_DATA_V24.pageEntities.second.order,
            name: LOCAL_TEST_DATA_V24.pageEntities.second.name,
            isPrimary: LOCAL_TEST_DATA_V24.pageEntities.second.isPrimary,
            additionalName: null,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
        },
    },
    personalBookmark: {
        first: {
            id: 1,
            personalContentMetadata: REMOTE_METADATA_V24.first.id,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: LOCAL_TEST_DATA_V24.bookmarks.first.time,
            updatedWhen: LOCAL_TEST_DATA_V24.bookmarks.first.time,
        },
    },
    personalTag: REMOTE_TAGS_V24,
    personalTagConnection: {
        firstPageTag: {
            id: 1,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 560,
            updatedWhen: 560,
            collection: 'personalContentMetadata',
            objectId: REMOTE_METADATA_V24.first.id,
            personalTag: REMOTE_TAGS_V24.firstPageTag.id,
            user: TEST_USER.id,
        },
        secondPageTag: {
            id: 2,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 562,
            updatedWhen: 562,
            collection: 'personalContentMetadata',
            objectId: REMOTE_METADATA_V24.second.id,
            personalTag: REMOTE_TAGS_V24.firstPageTag.id,
            user: TEST_USER.id,
        },
        firstAnnotationTag: {
            id: 1,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 561,
            updatedWhen: 561,
            collection: 'personalAnnotation',
            objectId: REMOTE_ANNOTATIONS_V24.first.id,
            personalTag: REMOTE_TAGS_V24.firstPageTag.id,
            user: TEST_USER.id,
        },
        secondAnnotationTag: {
            id: 2,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 563,
            updatedWhen: 563,
            collection: 'personalAnnotation',
            objectId: REMOTE_ANNOTATIONS_V24.second.id,
            personalTag: REMOTE_TAGS_V24.firstPageTag.id,
            user: TEST_USER.id,
        },
    },
    personalAnnotation: REMOTE_ANNOTATIONS_V24,
    personalAnnotationSelector: REMOTE_ANNOTATION_SELECTORS_V24,
    personalAnnotationPrivacyLevel: {
        first: {
            id: 1,
            localId: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.first.id,
            privacyLevel:
                LOCAL_TEST_DATA_V24.annotationPrivacyLevels.first.privacyLevel,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.first.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.first.createdWhen.getTime(),
            updatedWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.first.createdWhen.getTime(),
        },
        second: {
            id: 2,
            localId: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.second.id,
            privacyLevel:
                LOCAL_TEST_DATA_V24.annotationPrivacyLevels.second.privacyLevel,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.second.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.second.createdWhen.getTime(),
            updatedWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.second.createdWhen.getTime(),
        },
        third: {
            id: 1,
            localId: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.third.id,
            privacyLevel:
                LOCAL_TEST_DATA_V24.annotationPrivacyLevels.third.privacyLevel,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.third.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.third.createdWhen.getTime(),
            updatedWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.third.createdWhen.getTime(),
        },
        fifth: {
            id: 1,
            localId: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.fifth.id,
            privacyLevel:
                LOCAL_TEST_DATA_V24.annotationPrivacyLevels.fifth.privacyLevel,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.fifth.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.fifth.createdWhen.getTime(),
            updatedWhen: LOCAL_TEST_DATA_V24.annotationPrivacyLevels.fifth.createdWhen.getTime(),
        },
    },
    personalAnnotationShare: {
        first: {
            id: 1,
            remoteId:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.first.remoteId,
            excludeFromLists:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.first
                    .excludeFromLists,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.first.id,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 565,
            updatedWhen: 565,
        },
        second: {
            id: 2,
            remoteId:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.second.remoteId,
            excludeFromLists:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.second
                    .excludeFromLists,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.second.id,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 565,
            updatedWhen: 565,
        },
        third: {
            id: 1,
            remoteId:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.third.remoteId,
            excludeFromLists:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.third
                    .excludeFromLists,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.third.id,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 567,
            updatedWhen: 567,
        },
        fifth: {
            id: 1,
            remoteId:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.fifth.remoteId,
            excludeFromLists:
                LOCAL_TEST_DATA_V24.sharedAnnotationMetadata.fifth
                    .excludeFromLists,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.fifth.id,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 569,
            updatedWhen: 569,
        },
    },
    personalList: REMOTE_LISTS_V24,
    personalListTree: {
        first: {
            id: 1,
            personalList: REMOTE_LISTS_V24.first.id,
            parentListId: ROOT_NODE_PARENT_ID,
            path: null,
            localId: LOCAL_TEST_DATA_V24.customListTrees.first.id,
            localPath: LOCAL_TEST_DATA_V24.customListTrees.first.path,
            localListId: LOCAL_TEST_DATA_V24.customListTrees.first.listId,
            localParentListId:
                LOCAL_TEST_DATA_V24.customListTrees.first.parentListId,
            localLinkTarget: null,
            order: LOCAL_TEST_DATA_V24.customListTrees.first.order,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: LOCAL_TEST_DATA_V24.customListTrees.first.createdWhen,
            updatedWhen: LOCAL_TEST_DATA_V24.customListTrees.first.updatedWhen,
        },
        second: {
            id: 2,
            personalList: REMOTE_LISTS_V24.second.id,
            parentListId: REMOTE_LISTS_V24.first.id,
            path: buildMaterializedPath(REMOTE_LISTS_V24.first.id),
            localId: LOCAL_TEST_DATA_V24.customListTrees.second.id,
            localPath: LOCAL_TEST_DATA_V24.customListTrees.second.path,
            localListId: LOCAL_TEST_DATA_V24.customListTrees.second.listId,
            localParentListId:
                LOCAL_TEST_DATA_V24.customListTrees.second.parentListId,
            localLinkTarget: null,
            order: LOCAL_TEST_DATA_V24.customListTrees.second.order,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: LOCAL_TEST_DATA_V24.customListTrees.second.createdWhen,
            updatedWhen: LOCAL_TEST_DATA_V24.customListTrees.second.updatedWhen,
        },
        third: {
            id: 3,
            personalList: REMOTE_LISTS_V24.third.id,
            parentListId: REMOTE_LISTS_V24.first.id,
            path: buildMaterializedPath(REMOTE_LISTS_V24.first.id),
            localId: LOCAL_TEST_DATA_V24.customListTrees.third.id,
            localPath: LOCAL_TEST_DATA_V24.customListTrees.third.path,
            localListId: LOCAL_TEST_DATA_V24.customListTrees.third.listId,
            localParentListId:
                LOCAL_TEST_DATA_V24.customListTrees.third.parentListId,
            localLinkTarget: null,
            order: LOCAL_TEST_DATA_V24.customListTrees.third.order,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: LOCAL_TEST_DATA_V24.customListTrees.third.createdWhen,
            updatedWhen: LOCAL_TEST_DATA_V24.customListTrees.third.updatedWhen,
        },
        fourth: {
            id: 4,
            personalList: REMOTE_LISTS_V24.fourth.id,
            parentListId: REMOTE_LISTS_V24.third.id,
            path: buildMaterializedPath(
                REMOTE_LISTS_V24.first.id,
                REMOTE_LISTS_V24.third.id,
            ),
            localId: LOCAL_TEST_DATA_V24.customListTrees.fourth.id,
            localPath: LOCAL_TEST_DATA_V24.customListTrees.fourth.path,
            localListId: LOCAL_TEST_DATA_V24.customListTrees.fourth.listId,
            localParentListId:
                LOCAL_TEST_DATA_V24.customListTrees.fourth.parentListId,
            localLinkTarget: null,
            order: LOCAL_TEST_DATA_V24.customListTrees.fourth.order,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: LOCAL_TEST_DATA_V24.customListTrees.fourth.createdWhen,
            updatedWhen: LOCAL_TEST_DATA_V24.customListTrees.fourth.updatedWhen,
        },
        fifth: {
            id: 5,
            linkTarget: REMOTE_LISTS_V24.first.id,
            parentListId: REMOTE_LISTS_V24.second.id,
            path: buildMaterializedPath(
                REMOTE_LISTS_V24.first.id,
                REMOTE_LISTS_V24.second.id,
            ),
            localId: LOCAL_TEST_DATA_V24.customListTrees.fifth.id,
            localPath: LOCAL_TEST_DATA_V24.customListTrees.fifth.path,
            localListId: null,
            localParentListId:
                LOCAL_TEST_DATA_V24.customListTrees.fifth.parentListId,
            localLinkTarget:
                LOCAL_TEST_DATA_V24.customListTrees.fifth.linkTarget,
            order: LOCAL_TEST_DATA_V24.customListTrees.fifth.order,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: LOCAL_TEST_DATA_V24.customListTrees.fifth.createdWhen,
            updatedWhen: LOCAL_TEST_DATA_V24.customListTrees.fifth.updatedWhen,
        },
    },
    personalListDescription: {
        first: {
            id: 1,
            personalList: REMOTE_LISTS_V24.first.id,
            description:
                LOCAL_TEST_DATA_V24.customListDescriptions.first.description,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 565,
            updatedWhen: 565,
        },
        second: {
            id: 2,
            personalList: REMOTE_LISTS_V24.second.id,
            description:
                LOCAL_TEST_DATA_V24.customListDescriptions.second.description,
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            createdWhen: 566,
            updatedWhen: 566,
        },
    },
    personalListShare: {
        first: {
            id: 1,
            personalList: REMOTE_LISTS_V24.first.id,
            remoteId: LOCAL_SHARED_LIST_METADATA_V24.first.remoteId,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            private: false,
            createdWhen: 565,
            updatedWhen: 565,
        },
        second: {
            id: 2,
            personalList: REMOTE_LISTS_V24.second.id,
            remoteId: LOCAL_SHARED_LIST_METADATA_V24.second.remoteId,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            private: false,
            createdWhen: 565,
            updatedWhen: 565,
        },
        third: {
            id: 3,
            personalList: REMOTE_LISTS_V24.third.id,
            remoteId: LOCAL_SHARED_LIST_METADATA_V24.third.remoteId,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            private: false,
            createdWhen: 565,
            updatedWhen: 565,
        },
        fourth: {
            id: 4,
            personalList: REMOTE_LISTS_V24.fourth.id,
            remoteId: LOCAL_SHARED_LIST_METADATA_V24.fourth.remoteId,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            private: false,
            createdWhen: 565,
            updatedWhen: 565,
        },
    },
    personalListEntry: {
        first: {
            id: 1,
            personalContentMetadata: REMOTE_METADATA_V24.first.id,
            personalList: REMOTE_LISTS_V24.first.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 563,
            updatedWhen: 563,
        },
        second: {
            id: 2,
            personalContentMetadata: REMOTE_METADATA_V24.second.id,
            personalList: REMOTE_LISTS_V24.first.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 563,
            updatedWhen: 563,
        },
        third: {
            id: 1,
            personalContentMetadata: REMOTE_METADATA_V24.third.id,
            personalList: REMOTE_LISTS_V24.first.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 565,
            updatedWhen: 565,
        },
        fourth: {
            id: 1,
            personalContentMetadata: REMOTE_METADATA_V24.fourth.id,
            personalList: REMOTE_LISTS_V24.first.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 565,
            updatedWhen: 565,
        },
        fifth: {
            id: 1,
            personalContentMetadata: REMOTE_METADATA_V24.second.id,
            personalList: REMOTE_LISTS_V24.second.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 566,
            updatedWhen: 566,
        },
    },
    personalAnnotationListEntry: {
        first: {
            id: 1,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            personalList: REMOTE_LISTS_V24.first.id,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.first.id,
        },
        second: {
            id: 2,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            personalList: REMOTE_LISTS_V24.first.id,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.second.id,
        },
    },
    personalFollowedList: {
        first: {
            id: 1,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            sharedList: LOCAL_SHARED_LIST_METADATA_V24.first.remoteId,
        },
    },
    personalTextTemplate: {
        first: {
            id: 1,
            isFavourite: LOCAL_TEST_DATA_V24.templates.first.isFavourite,
            title: LOCAL_TEST_DATA_V24.templates.first.title,
            localId: LOCAL_TEST_DATA_V24.templates.first.id,
            code: LOCAL_TEST_DATA_V24.templates.first.code,
            order: LOCAL_TEST_DATA_V24.templates.first.order,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            outputFormat: 'markdown',
            user: TEST_USER.id,
            createdWhen: 561,
            updatedWhen: 561,
        },
        second: {
            id: 2,
            isFavourite: LOCAL_TEST_DATA_V24.templates.second.isFavourite,
            title: LOCAL_TEST_DATA_V24.templates.second.title,
            localId: LOCAL_TEST_DATA_V24.templates.second.id,
            code: LOCAL_TEST_DATA_V24.templates.second.code,
            order: LOCAL_TEST_DATA_V24.templates.second.order,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            outputFormat: 'markdown',
            user: TEST_USER.id,
            createdWhen: 562,
            updatedWhen: 562,
        },
    },
    personalMemexSetting: {
        first: {
            id: 1,
            name: LOCAL_TEST_DATA_V24.settings.first.key,
            value: LOCAL_TEST_DATA_V24.settings.first.value,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 561,
            updatedWhen: 561,
        },
        second: {
            id: 2,
            name: LOCAL_TEST_DATA_V24.settings.second.key,
            value: LOCAL_TEST_DATA_V24.settings.second.value,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 562,
            updatedWhen: 562,
        },
        third: {
            id: 3,
            name: LOCAL_TEST_DATA_V24.settings.third.key,
            value: LOCAL_TEST_DATA_V24.settings.third.value,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
            user: TEST_USER.id,
            createdWhen: 563,
            updatedWhen: 564,
        },
    },
    personalReadwiseAction: {
        first: {
            id: 1,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.first.id,
            createdWhen: new Date(1625634720653),
            updatedWhen: new Date(1625634720653),
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
        },
        second: {
            id: 2,
            personalAnnotation: REMOTE_ANNOTATIONS_V24.second.id,
            createdWhen: new Date(1625634720654),
            updatedWhen: new Date(1625634720654),
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
        },
    },
    personalTwitterAction: {
        first: {
            id: 1,
            personalContentMetadata: REMOTE_METADATA_V24.twitter_a.id,
            createdWhen: new Date(1625634720653),
            updatedWhen: new Date(1625634720653),
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
        },
        second: {
            id: 2,
            personalContentMetadata: REMOTE_METADATA_V24.twitter_b.id,
            createdWhen: new Date(1625634720654),
            updatedWhen: new Date(1625634720654),
            user: TEST_USER.id,
            createdByDevice: REMOTE_DEVICES_V24.first.id,
        },
    },
}
