
#include "node.h"
#include "private/node_priv.h"
#include "private/identity_priv.h"
#include "memory.h"

#define NODE_ARRAY_TYPE_ID 1

wickr_node_t *wickr_node_create(wickr_buffer_t *dev_id, wickr_identity_chain_t *id_chain, wickr_ephemeral_keypair_t *ephemeral_keypair)
{
    if (!dev_id || !id_chain) {
        return NULL;
    }
    
    wickr_node_t *new_node = wickr_alloc_zero(sizeof(wickr_node_t));
    
    if (!new_node) {
        return NULL;
    }
    
    new_node->dev_id = dev_id;
    new_node->id_chain = id_chain;
    new_node->ephemeral_keypair = ephemeral_keypair;
    
    return new_node;
}

bool wickr_node_rotate_keypair(wickr_node_t *node, wickr_ephemeral_keypair_t *new_keypair, bool copy)
{
    if (!node || !new_keypair) {
        return false;
    }
    
    wickr_ephemeral_keypair_destroy(&node->ephemeral_keypair);
    
    node->status = NODE_STATUS_UNKNOWN;
    
    if (copy) {
        wickr_ephemeral_keypair_t *copy_keypair = wickr_ephemeral_keypair_copy(new_keypair);
        if (!copy_keypair) {
            return false;
        }
        node->ephemeral_keypair = copy_keypair;
    }
    else {
        node->ephemeral_keypair = new_keypair;
    }
    
    return true;
}

bool wickr_node_verify_signature_chain(wickr_node_t *node, const wickr_crypto_engine_t *engine)
{
    if (!node || !engine) {
        return false;
    }
    
    /* Check to see if we need to recalculate the status of the node */
    if (wickr_node_has_valid_cache(node, engine)) {
        return node->status == NODE_STATUS_VALID;
    }
    
    /* If the id_chain status is invalid, then return false without continuing */
    if (!wickr_identity_chain_validate(node->id_chain, engine)) {
        node->status = NODE_STATUS_INVALID;
        wickr_node_update_status_cache(node, engine);
        return false;
    }
    
    /* If the key pair ownership can't be verified by the node signature key in the id_chain, return false */
    bool has_valid_ephemeral_key = wickr_ephemeral_keypair_verify_owner(node->ephemeral_keypair, engine, node->id_chain->node);
    
    node->status = has_valid_ephemeral_key ? NODE_STATUS_VALID : NODE_STATUS_INVALID;
    wickr_node_update_status_cache(node, engine);
    
    return node->status == NODE_STATUS_VALID && node->id_chain->status == IDENTITY_CHAIN_STATUS_VALID;
}

wickr_node_t *wickr_node_copy(const wickr_node_t *source)
{
    if (!source) {
        return NULL;
    }
    
    wickr_buffer_t *dev_id_copy = wickr_buffer_copy(source->dev_id);
    
    if (!dev_id_copy) {
        return NULL;
    }
    
    wickr_identity_chain_t *id_chain_copy = wickr_identity_chain_copy(source->id_chain);
    
    if (!id_chain_copy) {
        wickr_buffer_destroy(&dev_id_copy);
        return NULL;
    }
    
    wickr_ephemeral_keypair_t *keypair_copy = wickr_ephemeral_keypair_copy(source->ephemeral_keypair);
    
    if (!keypair_copy && source->ephemeral_keypair) {
        wickr_buffer_destroy(&dev_id_copy);
        wickr_identity_chain_destroy(&id_chain_copy);
        return NULL;
    }
    
    wickr_buffer_t *status_copy = wickr_buffer_copy(source->_status_cache);
    
    if (source->_status_cache && !status_copy) {
        wickr_buffer_destroy(&dev_id_copy);
        wickr_identity_chain_destroy(&id_chain_copy);
        wickr_ephemeral_keypair_destroy(&keypair_copy);
        return NULL;
    }
    
    wickr_node_t *node_copy = wickr_node_create(dev_id_copy, id_chain_copy, keypair_copy);
    
    if (!node_copy) {
        wickr_buffer_destroy(&dev_id_copy);
        wickr_identity_chain_destroy(&id_chain_copy);
        wickr_ephemeral_keypair_destroy(&keypair_copy);
        wickr_buffer_destroy(&status_copy);
        return NULL;
    }
    
    node_copy->_status_cache = status_copy;
    node_copy->status = source->status;
    
    return node_copy;
}

void wickr_node_destroy(wickr_node_t **node)
{
    if (!node || !*node) {
        return;
    }
    
    wickr_buffer_destroy(&(*node)->dev_id);
    wickr_identity_chain_destroy(&(*node)->id_chain);
    wickr_ephemeral_keypair_destroy(&(*node)->ephemeral_keypair);
    wickr_buffer_destroy(&(*node)->_status_cache);
    wickr_free(*node);
    *node = NULL;
}

wickr_node_array_t *wickr_node_array_new(uint32_t node_count)
{
    return wickr_array_new(node_count, NODE_ARRAY_TYPE_ID, (wickr_array_copy_func)wickr_node_copy,
                           (wickr_array_destroy_func)wickr_node_destroy);
}

bool wickr_node_array_set_item(wickr_array_t *array, uint32_t index, wickr_node_t *node)
{
    return wickr_array_set_item(array, index, node, false);
}

wickr_node_t *wickr_node_array_fetch_item(const wickr_array_t *array, uint32_t index)
{
    return wickr_array_fetch_item(array, index, false);
}

wickr_node_array_t *wickr_node_array_copy(const wickr_node_array_t *array)
{
    return wickr_array_copy(array, true);
}

void wickr_node_array_destroy(wickr_node_array_t **array)
{
    if (!array || !*array) {
        return;
    }
    
    wickr_array_destroy(array, false);
}

wickr_buffer_t *wickr_node_serialize(const wickr_node_t *node)
{
    if (!node) {
        return NULL;
    }
    
    Wickr__Proto__Node *proto_node = wickr_node_to_proto(node);
    
    if (!proto_node) {
        return NULL;
    }
    
    size_t packed_size = wickr__proto__node__get_packed_size(proto_node);
    
    wickr_buffer_t *packed_buffer = wickr_buffer_create_empty(packed_size);
    
    if (!packed_buffer) {
        wickr_node_proto_free(proto_node);
        return NULL;
    }
    
    wickr__proto__node__pack(proto_node, packed_buffer->bytes);
    wickr_node_proto_free(proto_node);
    
    return packed_buffer;
}

wickr_node_t *wickr_node_create_from_buffer(const wickr_buffer_t *buffer, const wickr_crypto_engine_t *engine)
{
    if (!buffer) {
        return NULL;
    }
    
    Wickr__Proto__Node *proto_node = wickr__proto__node__unpack(NULL, buffer->length, buffer->bytes);
    
    if (!proto_node) {
        return NULL;
    }
    
    wickr_node_t *return_node = wickr_node_create_from_proto(proto_node, engine);
    wickr__proto__node__free_unpacked(proto_node, NULL);
    
    return return_node;
}
