/*
* Copyright © 2012-2020 Wickr Inc.  All rights reserved.
*
* This code is being released for EDUCATIONAL, ACADEMIC, AND CODE REVIEW PURPOSES
* ONLY.  COMMERCIAL USE OF THE CODE IS EXPRESSLY PROHIBITED.  For additional details,
* please see LICENSE
*
* THE CODE IS MADE AVAILABLE "AS-IS" AND WITHOUT ANY EXPRESS OR
* IMPLIED GUARANTEES AS TO FITNESS, MERCHANTABILITY, NON-
* INFRINGEMENT OR OTHERWISE. IT IS NOT BEING PROVIDED IN TRADE BUT ON
* A VOLUNTARY BASIS ON BEHALF OF THE AUTHOR’S PART FOR THE BENEFIT
* OF THE LICENSEE AND IS NOT MADE AVAILABLE FOR CONSUMER USE OR ANY
* OTHER USE OUTSIDE THE TERMS OF THIS LICENSE. ANYONE ACCESSING THE
* CODE SHOULD HAVE THE REQUISITE EXPERTISE TO SECURE THEIR SYSTEM
* AND DEVICES AND TO ACCESS AND USE THE CODE FOR REVIEW PURPOSES
* ONLY. LICENSEE BEARS THE RISK OF ACCESSING AND USING THE CODE. IN
* PARTICULAR, AUTHOR BEARS NO LIABILITY FOR ANY INTERFERENCE WITH OR
* ADVERSE EFFECT THAT MAY OCCUR AS A RESULT OF THE LICENSEE
* ACCESSING AND/OR USING THE CODE ON LICENSEE’S SYSTEM.
*/

#ifndef transport_packet_h
#define transport_packet_h

#include "buffer.h"
#include "identity.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
@addtogroup wickr_transport_packet
*/

typedef enum {
    TRANSPORT_PAYLOAD_TYPE_HANDSHAKE, /* Payload is a handshake control packet */
    TRANSPORT_PAYLOAD_TYPE_CIPHERTEXT /* Payload contains encrypted application data */
} wickr_transport_packet_payload_type;

typedef enum {
    TRANSPORT_MAC_TYPE_NONE, /* Packet contains no authentication */
    TRANSPORT_MAC_TYPE_AUTH_CIPHER, /* Packet body contains ciphertext generated by an authenticated cipher such as AES-GCM */
    TRANSPORT_MAC_TYPE_EC_P521 /* Packet was signed by a key with the P521 curve */
} wickr_transport_packet_mac_type;

/**
@addtogroup wickr_transport_packet_meta
*/

/**
 @ingroup wickr_transport_packet_meta
 
 @struct wickr_transport_handshake_meta
 
 @brief Metadata specifically for handshake packets within a transport
 
 @var wickr_transport_handshake_meta::protocol_version
 the current version of this handshake, currently always 1. Will be used in the future to enable additional functionality
 @var wickr_transport_handshake_meta::flags
 future use buffer to add additional fields to the handshake metadata if needed
 */

struct wickr_transport_handshake_meta {
    uint8_t protocol_version;
    uint64_t flags; /* Future use */
};

typedef struct wickr_transport_handshake_meta wickr_transport_handshake_meta_t;

/**
@ingroup wickr_transport_packet_meta

@struct wickr_transport_data_meta

@brief Metadata specifically for user data packets within a transport

@var wickr_transport_data_meta::sequence_number
the sequence number within the transport that is associated with this packet

*/
struct wickr_transport_data_meta {
    uint64_t sequence_number;
};

typedef struct wickr_transport_data_meta wickr_transport_data_meta_t;

/**
@ingroup wickr_transport_packet_meta

@struct wickr_transport_packet_meta

@brief Packet metadata included with all transport packets

@var wickr_transport_packet_meta::body_meta
union that can either hold data unique to a handshake or user data packet
@var wickr_transport_packet_meta::body_type
a field that indicates if this packet's body contains handshake or user data information
@var wickr_transport_packet_meta::mac_type
the type of data authentication the packet has

*/
struct wickr_transport_packet_meta {
    union {
        wickr_transport_handshake_meta_t handshake;
        wickr_transport_data_meta_t data;
    } body_meta; /* Set based on body_type */
    wickr_transport_packet_payload_type body_type;
    wickr_transport_packet_mac_type mac_type;
};

typedef struct wickr_transport_packet_meta wickr_transport_packet_meta_t;

/**
 @ingroup wickr_transport_packet_meta
 
 Initialize packet metadata for a handshake packet
 
 @param meta_out a pointer to initialize for the handshake metadata
 @param protocol_version the protocol version the handshake will be using (default 1)
 @param mac_type the type of mac or signature to be used to authenticate the body data of the packet
 */
void wickr_transport_packet_meta_initialize_handshake(wickr_transport_packet_meta_t *meta_out,
                                                      uint8_t protocol_version,
                                                      wickr_transport_packet_mac_type mac_type);

/**
 @ingroup wickr_transport_packet_meta
 
 Initialize packet metadata for a data packet
 
 @param meta_out a pointer to initialize for the handshake metadata
 @param sequence_number the sequence number of this packet within the current stream of data
 @param mac_type the type of mac or signature to be used to authenticate the body data of the packet
 */
void wickr_transport_packet_meta_initialize_data(wickr_transport_packet_meta_t *meta_out,
                                                 uint64_t sequence_number,
                                                 wickr_transport_packet_mac_type mac_type);

/**
 @ingroup wickr_transport_packet_meta
 
 Initialize packet metadata with a serialized packet meta buffer
 
 @param meta_out a pointer to initialize for the handshake metadata
 @param buffer a buffer containing a serialized packet
 @return the number of bytes from buffer that had to be read to initialize `meta_out` or a negative int to indicate failure
 */
int wickr_transport_packet_meta_initialize_buffer(wickr_transport_packet_meta_t *meta_out,
                                                  const wickr_buffer_t *buffer);

/**
 @ingroup wickr_transport_packet_meta
 
 @param meta the metadata to serialize
 @return a serialized representation of `meta` or NULL if there was a failure during serialization
 */
wickr_buffer_t *wickr_transport_packet_meta_serialize(const wickr_transport_packet_meta_t *meta);

/**
 @ingroup wickr_transport_packet
 
 @struct wickr_transport_packet
 @brief A packet used within the wickr_transport_ctx to package up data and related metadata + authentication in one serializable structure

 @var wickr_transport_packet::network_buffer
 the serialized representation of the packet, will be null unless `wickr_transport_packet_create_from_buffer` or `wickr_transport_packet_sign` has been called
 @var wickr_transport_packet::body
 the main content data of the packet
 @var wickr_transport_packet::mac
 authentication data used to verify that the body, and meta data have not been tampered with
 @var wickr_transport_packet::meta
 descriptive elements of the packet that also differentiates it's type as a handshake or data packet
*/
struct wickr_transport_packet {
    wickr_buffer_t *network_buffer;
    wickr_buffer_t *body;
    wickr_buffer_t *mac;
    wickr_transport_packet_meta_t meta;
};

typedef struct wickr_transport_packet wickr_transport_packet_t;

/**
 @ingroup wickr_transport_packet
 
 Create a transport packet
 
 @param meta metadata describing `body`
 @param body the main contents of the packet
 */
wickr_transport_packet_t *wickr_transport_packet_create(wickr_transport_packet_meta_t meta, wickr_buffer_t *body);

/**
 @ingroup wickr_transport_packet
 
 Make a deep copy of a transport packet
 
 @param pkt the transport packet to make a copy of
 @return a newly allocated transport packet holding a deep copy of the properties of 'pkt'
 */
wickr_transport_packet_t *wickr_transport_packet_copy(const wickr_transport_packet_t *pkt);

/**
 @ingroup wickr_transport_packet

 Destroy a transport packet
 @param pkt a pointer to the transport packet to destroy. All properties of `*pkt` will also be destroyed
*/
void wickr_transport_packet_destroy(wickr_transport_packet_t **pkt);

/**
 @ingroup wickr_transport_packet
 
 Serialize a transport packet down to a buffer for transfer
 
 @param pkt the packet to serialize
 @return the byte representation of `pkt` or NULL if serialization fails
 */
wickr_buffer_t *wickr_transport_packet_serialize(const wickr_transport_packet_t *pkt);

/**
 @ingroup wickr_transport_packet
 
 Create a transport packet from a serialized transport packet
 
 @param buffer a buffer that was created by a call to `wickr_transport_packet_serialize`
 @return a newly allocated transport packet from the data contained in `buffer` or NULL if the buffer is corrupt or does not represent a transport packet
 */
wickr_transport_packet_t *wickr_transport_packet_create_from_buffer(const wickr_buffer_t *buffer);

/**
 @ingroup wickr_transport_packet
 
 Sign a transport packet to authenticate it using an identity chain. Upon successful signing, the mac type of `pkt` will be set based on the type of key used for the signature
 
 @param pkt the packet to sign
 @param engine a pointer to a crypto engine that support digital signatures
 @param identity_chain an identity chain that at minimum contains the private key for a node identity in order to sign the packet
 @return true if signing is successful
 */
bool wickr_transport_packet_sign(wickr_transport_packet_t *pkt,
                                 const wickr_crypto_engine_t *engine,
                                 const wickr_identity_chain_t *identity_chain);

/**
 @ingroup wickr_transport_packet
 
 Verify a packet that was signed by a call to `wickr_transport_packet_sign`
 
 @param packet the packet to verify
 @param engine a pointer to a crypto engine that supports verifying digital signatures
 @param identity_chain the public identity chain that is believed to have signed `packet`
 @return true if `packet` was signed by `identity_chain`
 */
bool wickr_transport_packet_verify(const wickr_transport_packet_t *packet,
                                   const wickr_crypto_engine_t *engine,
                                   wickr_identity_chain_t *identity_chain);

#ifdef __cplusplus
}
#endif

#endif /* transport_packet_h */
