/*
 * Copyright © 2012-2020 Wickr Inc.  All rights reserved.
 *
 * This code is being released for EDUCATIONAL, ACADEMIC, AND CODE REVIEW PURPOSES
 * ONLY.  COMMERCIAL USE OF THE CODE IS EXPRESSLY PROHIBITED.  For additional details,
 * please see LICENSE
 *
 * THE CODE IS MADE AVAILABLE "AS-IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED GUARANTEES AS TO FITNESS, MERCHANTABILITY, NON-
 * INFRINGEMENT OR OTHERWISE. IT IS NOT BEING PROVIDED IN TRADE BUT ON
 * A VOLUNTARY BASIS ON BEHALF OF THE AUTHOR’S PART FOR THE BENEFIT
 * OF THE LICENSEE AND IS NOT MADE AVAILABLE FOR CONSUMER USE OR ANY
 * OTHER USE OUTSIDE THE TERMS OF THIS LICENSE. ANYONE ACCESSING THE
 * CODE SHOULD HAVE THE REQUISITE EXPERTISE TO SECURE THEIR SYSTEM
 * AND DEVICES AND TO ACCESS AND USE THE CODE FOR REVIEW PURPOSES
 * ONLY. LICENSEE BEARS THE RISK OF ACCESSING AND USING THE CODE. IN
 * PARTICULAR, AUTHOR BEARS NO LIABILITY FOR ANY INTERFERENCE WITH OR
 * ADVERSE EFFECT THAT MAY OCCUR AS A RESULT OF THE LICENSEE
 * ACCESSING AND/OR USING THE CODE ON LICENSEE’S SYSTEM.
 */

#ifndef storage_h
#define storage_h

#include <stdlib.h>
#include "buffer.h"
#include "crypto_engine.h"

#ifdef __cplusplus
extern "C" {
#endif
/**
 @addtogroup wickr_storage_keys
 */
/**
 
 @ingroup wickr_storage_keys
 
 @struct wickr_storage_keys
 
 @brief Key's used for the client to encrypt data meant for itself, and not for other clients
 
 @var wickr_storage_keys::local
 cipher key to be used for encrypting / decrypting data that does not leave the local client. Local keys are generated by merging device information with a root storage keys, and thus are specific to the local environment.
 @var wickr_storage_keys::remote 
 cipher key to be used for encrypting / decrypting data that is to be synchronized between multiple client instances.
 */
struct wickr_storage_keys {
    wickr_cipher_key_t *local;
    wickr_cipher_key_t *remote;
};

typedef struct wickr_storage_keys wickr_storage_keys_t;

/**
 
 @ingroup wickr_storage_keys
 
 Create a storage key set from components

 @param local see 'wickr_storage_keys' property documentation
 @param remote see 'wickr_storage_keys' property documentation
 @return a newly allocated storage key set. Takes ownership of the passed inputs
 */
wickr_storage_keys_t *wickr_storage_keys_create(wickr_cipher_key_t *local, wickr_cipher_key_t *remote);

/**
 
 @ingroup wickr_storage_keys
 
 Copy a storage key set
 
 @param source the storage key set to copy
 @return a newly allocated storage key set holding a deep copy of the properties of 'source'
 */
wickr_storage_keys_t *wickr_storage_keys_copy(const wickr_storage_keys_t *source);

/**
 
 @ingroup wickr_storage_keys
 
 Create a storage key set from a previously serialized storage key set

 @param buffer a buffer containing the output of 'wickr_storage_keys_serialize'
 @return a newly allocated storage key set or NULL if parsing fails
 */
wickr_storage_keys_t *wickr_storage_keys_create_from_buffer(const wickr_buffer_t *buffer);

/**
 
 @ingroup wickr_storage_keys
 
 Serialize a storage key set to a buffer

 @param keys the keys to serialize
 @return a buffer containing the properties of storage keys in the following format:
    | CIPHER_ID (LOCAL) | KEY_DATA (LOCAL) | CIPHER_ID (REMOTE) | KEY_DATA (REMOTE) |
 */
wickr_buffer_t *wickr_storage_keys_serialize(const wickr_storage_keys_t *keys);

/**
 
 @ingroup wickr_storage_keys
 
 Destroy a storage key set
 
 @param keys a pointer to the storage key set to destroy. All properties of '*keys' will also be destroyed
 */
void wickr_storage_keys_destroy(wickr_storage_keys_t **keys);

#ifdef __cplusplus
}
#endif

#endif /* storage_h */
