#version 330 core

// ----------------- in out

out vec4 FragColor;

in VS_OUT {
    vec3 FragPos;
    vec3 Normal;
    vec2 TexCoords;
    vec3 Tangent;
} fs_in;

// ----------------- const

#define MAX_POINT_LIGHTS 8
const float PI = 3.14159265359;
const float INV_PI = 0.31830988618;

// array of offset direction for sampling
const vec3 gridSamplingDisk[20] = vec3[]
(
   vec3(1, 1,  1), vec3( 1, -1,  1), vec3(-1, -1,  1), vec3(-1, 1,  1), 
   vec3(1, 1, -1), vec3( 1, -1, -1), vec3(-1, -1, -1), vec3(-1, 1, -1),
   vec3(1, 1,  0), vec3( 1, -1,  0), vec3(-1, -1,  0), vec3(-1, 1,  0),
   vec3(1, 0,  1), vec3(-1,  0,  1), vec3( 1,  0, -1), vec3(-1, 0, -1),
   vec3(0, 1,  1), vec3( 0, -1,  1), vec3( 0, -1, -1), vec3( 0, 1, -1)
);

// ----------------- struct

struct BSDF_FrostedGlass {
	vec3 colorFactor;
	bool haveColorTexture;
    sampler2D colorTexture;

	float roughnessFactor;
	bool haveRoughnessTexture;
    sampler2D roughnessTexture;
	
	bool haveAOTexture;
    sampler2D aoTexture;

	bool haveNormalTexture;
	sampler2D normalTexture;

	float ior;
};

// ----------------- Uniform

// 48
struct PointLight {
    vec3 position;	// 12	0
    vec3 L;			// 12	16
    float linear;	// 4	28
    float quadratic;// 4	32
};

// 160
layout (std140) uniform Camera{
	mat4 view;			// 64	0	64
	mat4 projection;	// 64	64	64
	vec3 viewPos;		// 12	128	144
	float nearPlane;	// 4	144	148
	float farPlane;		// 4	148	152
	float fov;			// 4	152	156
	float ar;			// 4	156	160
};

// 400
layout (std140) uniform PointLights{
	int numLight;// 16
	PointLight pointLights[MAX_POINT_LIGHTS];// 48 * MAX_POINT_LIGHTS = 48 * 8
};

uniform BSDF_FrostedGlass bsdf;

uniform samplerCube pointLightDepthMap0;
uniform samplerCube pointLightDepthMap1;
uniform samplerCube pointLightDepthMap2;
uniform samplerCube pointLightDepthMap3;
uniform samplerCube pointLightDepthMap4;
uniform samplerCube pointLightDepthMap5;
uniform samplerCube pointLightDepthMap6;
uniform samplerCube pointLightDepthMap7;

uniform float lightFar;

uniform int mode;

uniform float interpolateRatio; // (0, 1]

// ----------------- declaration

vec3 CalcBumpedNormal(vec3 normal, vec3 tangent, sampler2D normalTexture, vec2 texcoord);

float GGX_D(vec3 norm, vec3 h, float alpha);
float Fr(vec3 v, vec3 h, float ior);
float GGX_G1(vec3 norm, vec3 v, vec3 h, float alpha);
float GGX_G(vec3 norm, vec3 wo, vec3 wi, vec3 h, float alpha);
vec3 BSDF(vec3 norm, vec3 wo, vec3 wi, vec3 color, float roughness, float ao, float ior);

float Visibility(vec3 lightToFrag, int id);
float Visibility(vec3 lightToFrag, samplerCube depthMap);

float tanh(float x) {
	float expZ = exp(x);
    float invExpZ = 1 / expZ;
    return (expZ - invExpZ) / (expZ + invExpZ);
}

float smootherstep(float x){
	return ((6*x - 15)*x + 10) * x*x*x;
}

// template declaration
void Model
(
	in float directIllum_R,
	in float directIllum_G,
	in float directIllum_B,
	in float pos_x,
	in float pos_y,
	in float pos_z,
	in float viewDir_x,
	in float viewDir_y,
	in float viewDir_z,
	in float normal_x,
	in float normal_y,
	in float normal_z,
	in float matColor_R,
	in float matColor_G,
	in float matColor_B,
	in float IOR,
	in float roughness,
	
	out float indirectIllum_R,
	out float indirectIllum_G,
	out float indirectIllum_B
);

// ----------------- main

void main() {
	// get attr
	vec3 color = bsdf.colorFactor;
	if(bsdf.haveColorTexture) {
		color *= texture(bsdf.colorTexture, fs_in.TexCoords).xyz;
	}

	float roughness = bsdf.roughnessFactor;
	if(bsdf.haveRoughnessTexture) {
		roughness *= texture(bsdf.roughnessTexture, fs_in.TexCoords).x;
	}

	float ao = 1.0f;
	if(bsdf.haveAOTexture) {
		ao *= texture(bsdf.aoTexture, fs_in.TexCoords).x;
	}

	vec3 wo = normalize(viewPos - fs_in.FragPos);

	vec3 norm = normalize(fs_in.Normal);
	if(bsdf.haveNormalTexture) {
		norm = CalcBumpedNormal(norm, normalize(fs_in.Tangent), bsdf.normalTexture, fs_in.TexCoords);
	}
	
	// sample point light
	vec3 result = vec3(0);
    for(int i = 0; i < numLight; i++) {
		vec3 fragToLight = pointLights[i].position - fs_in.FragPos;
		float dist2 = dot(fragToLight, fragToLight);
		float dist = sqrt(dist2);
		vec3 wi = fragToLight / dist;
		
		float visibility = Visibility(-fragToLight, i);
		if(visibility==0)
			continue;

		vec3 f = BSDF(norm, wo, wi, color, roughness, ao, bsdf.ior);

		float cosTheta = max(dot(wi, norm), 0);
		
		float attenuation = 1.0f + pointLights[i].linear * dist + pointLights[i].quadratic * dist2;
		
		result += visibility * cosTheta / attenuation * f * pointLights[i].L;
	}
	
	vec3 indirectIllum;
	
	Model
	(
		// input
		result.r,
		result.g,
		result.b,
		fs_in.FragPos.x,
		fs_in.FragPos.y,
		fs_in.FragPos.z,
		wo.x,
		wo.y,
		wo.z,
		norm.x,
		norm.y,
		norm.z,
		color.r,
		color.g,
		color.b,
		bsdf.ior,
		roughness,
		
		// output
		indirectIllum.r,
		indirectIllum.g,
		indirectIllum.b
	);
	
	indirectIllum = clamp(indirectIllum, 0, 1);
	
	// gamma correction
	vec3 finalColor;
	if(mode==0)
		finalColor = result;
	else if(mode==1)
		finalColor = indirectIllum;
	else
		finalColor = result + indirectIllum;
		
    FragColor = vec4(sqrt(finalColor), 1.0);
}

// ----------------- definition

float GGX_D(vec3 norm, vec3 h, float alpha) {
	float HoN = dot(h, norm);
	if (HoN <= 0)
		return 0;

	float cos2 = HoN * HoN;
	float alpha2 = alpha * alpha;

	float root = alpha / (cos2 * (alpha2 - 1) + 1);

	return INV_PI * root * root;
}

float Fr(vec3 v, vec3 h, float ior) {
	// angle in low ior media
	float cosTheta = dot(v, h);
	bool entering = cosTheta > 0.0f;
	
	if (!entering) {
		vec3 vInLowIOR = refract(-v, -h, ior);
		if (vInLowIOR == vec3(0))
			return 1;

		cosTheta = dot(vInLowIOR, h);
	}

	float R0 = pow((ior - 1) / (ior + 1), 2);

	float Fr = R0 + (1 - R0) * pow((1 - cosTheta), 5);

	return Fr;
}

float GGX_G1(vec3 norm, vec3 v, vec3 h, float alpha) {
	float NoV = dot(norm, v);
	float HoV = dot(v, h);
	if (NoV * HoV <= 0)
		return 0;

	float alpha2 = alpha * alpha;
	float tan2 = 1.f / (NoV * NoV) - 1.f;
	return 2.f / (1.f + sqrt(1 + alpha2 * tan2));
}

float GGX_G(vec3 norm, vec3 wo, vec3 wi, vec3 h, float alpha) {
	return GGX_G1(norm, wo, h, alpha) * GGX_G1(norm, wi, h, alpha);
}

vec3 BSDF(vec3 norm, vec3 wo, vec3 wi, vec3 color, float roughness, float ao, float ior) {
	float NoWo = dot(norm, wo);
	float NoWi = dot(norm, wi);
	if(NoWo == 0 || NoWi == 0)
		return vec3(0);

	bool entering = NoWo > 0;
	bool isReflect = NoWo * NoWi > 0;

	float alpha = roughness * roughness;

	if (isReflect) {
		vec3 h = normalize(wo + wi);

		h *= sign(dot(h, norm));// let h point to outside
		float bsdfVal = Fr(wo, h, ior) * GGX_D(norm, h, alpha) * GGX_G(norm, wo, wi, h, alpha) / abs(4.f * NoWo * NoWi);
		return bsdfVal * color;
	}
	else {
		float etai = 1.f, etat = ior;
		if (!entering){
			// swap(etai, etat);
			float tmp = etai;
			etai = etat;
			etat = tmp;
		}

		vec3 h = -normalize(etai * wo + etat * wi);

		float HoWo = dot(h, wo);
		float HoWi = dot(h, wi);
		float sqrtDenom = etai * HoWo + etat * HoWi;

		float factor = abs(HoWo * HoWi / (NoWo * NoWi));

		h *= sign(dot(h,norm));// let h point to outside
		float bsdfVal = factor * ((1 - Fr(wo, h, ior)) * GGX_D(norm, h, alpha) * GGX_G(norm, wo, wi, h, alpha) * etat * etat) / (sqrtDenom * sqrtDenom);
		return bsdfVal * color;
	}
}

vec3 CalcBumpedNormal(vec3 normal, vec3 tangent, sampler2D normalTexture, vec2 texcoord) {
    tangent = normalize(tangent - dot(tangent, normal) * normal);
    vec3 bitangent = cross(tangent, normal);
    vec3 bumpMapNormal = texture(normalTexture, texcoord).xyz;
    bumpMapNormal = 2.0 * bumpMapNormal - 1.0;
    mat3 TBN = mat3(tangent, bitangent, normal);
    vec3 newNormal = TBN * bumpMapNormal;
    newNormal = normalize(newNormal);
    return newNormal;
}

float Visibility(vec3 lightToFrag, int id){
	if(id == 0) {
		return Visibility(lightToFrag, pointLightDepthMap0);
	} else if(id == 1) {
		return Visibility(lightToFrag, pointLightDepthMap1);
	} else if(id == 2) {
		return Visibility(lightToFrag, pointLightDepthMap2);
	} else if(id == 3) {
		return Visibility(lightToFrag, pointLightDepthMap3);
	} else if(id == 4) {
		return Visibility(lightToFrag, pointLightDepthMap4);
	} else if(id == 5) {
		return Visibility(lightToFrag, pointLightDepthMap5);
	} else if(id == 6) {
		return Visibility(lightToFrag, pointLightDepthMap6);
	} else if(id == 7) {
		return Visibility(lightToFrag, pointLightDepthMap7);
	}else 
		return 1;// not support id
}

float Visibility(vec3 lightToFrag, samplerCube depthMap) {
	float currentDepth = length(lightToFrag);
	float bias = 0.08;
	int samples = 20;
	float shadow = 0.0;
	float viewDistance = length(viewPos - fs_in.FragPos);
	float diskRadius = (1.0 + (viewDistance / lightFar)) / 50.0;
	for(int i = 0; i < samples; ++i) {
		float closestDepth = lightFar * texture(depthMap, lightToFrag + gridSamplingDisk[i] * diskRadius).r;
		shadow += smoothstep(closestDepth, closestDepth + bias, currentDepth);
	}
	shadow /= float(samples);
	return 1 - shadow;
}

void FNN_3_3_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(0.284661)+x1*(-0.168943)+x2*(0.307368)+x3*(-0.271415)+x4*(0.0663434)+x5*(-5.97016)+x6*(-1.07716)+x7*(-1.45702)+x8*(-5.02254)+x9*(0.0191183)+x10*(0.382511)+x11*(-6.3344)+x12*(0.102452)+x13*(0.0713574)+x14*(0.264581)+x15*(0.231289)+x16*(0.328775)+(0.312238)); 
    h1 = tanh(x0*(-8.33239)+x1*(-8.53577)+x2*(-8.53288)+x3*(0.506842)+x4*(0.712016)+x5*(0.597135)+x6*(-1.09368)+x7*(-1.47246)+x8*(1.69425)+x9*(-0.537709)+x10*(-0.282733)+x11*(0.559631)+x12*(-0.137574)+x13*(0.15223)+x14*(0.224986)+x15*(-0.254898)+x16*(-0.177696)+(-0.560481)); 
    h2 = tanh(x0*(-0.11997)+x1*(0.0282287)+x2*(-0.535154)+x3*(0.157576)+x4*(0.929573)+x5*(0.00607486)+x6*(-0.681435)+x7*(-2.69714)+x8*(2.29406)+x9*(0.356619)+x10*(0.0293324)+x11*(-0.0951615)+x12*(-0.190374)+x13*(-0.0309575)+x14*(-0.145842)+x15*(0.272821)+x16*(0.289494)+(0.106431)); 
    h3 = tanh(x0*(-0.55667)+x1*(-0.304162)+x2*(-0.300087)+x3*(0.0637149)+x4*(-0.873086)+x5*(2.09932)+x6*(0.761787)+x7*(-0.592253)+x8*(-0.58144)+x9*(-0.199556)+x10*(-0.0968169)+x11*(5.62154)+x12*(-0.00914347)+x13*(-0.38094)+x14*(-0.284653)+x15*(0.342094)+x16*(0.250259)+(-0.172732)); 
    h4 = tanh(x0*(-0.180805)+x1*(-0.0884696)+x2*(-0.0392641)+x3*(-0.357166)+x4*(-0.0533273)+x5*(0.321651)+x6*(0.891511)+x7*(0.106766)+x8*(-2.14408)+x9*(-0.147938)+x10*(-0.230911)+x11*(-1.1875)+x12*(0.138901)+x13*(0.197483)+x14*(0.0891038)+x15*(0.0576328)+x16*(-0.155022)+(-0.13428)); 
    h5 = tanh(x0*(-1.12922)+x1*(-1.32117)+x2*(-0.980611)+x3*(-1.80464)+x4*(0.41103)+x5*(-0.844509)+x6*(2.75397)+x7*(-1.3133)+x8*(1.33622)+x9*(-0.935289)+x10*(-0.663627)+x11*(-2.64239)+x12*(0.0657026)+x13*(-0.215727)+x14*(0.0172811)+x15*(-0.0745023)+x16*(0.232034)+(-0.77179)); 
    h6 = tanh(x0*(-0.106715)+x1*(0.504242)+x2*(0.350797)+x3*(-0.443339)+x4*(-0.224066)+x5*(-0.766253)+x6*(-0.230871)+x7*(-0.843967)+x8*(0.324735)+x9*(-0.443646)+x10*(-0.0621596)+x11*(-2.51166)+x12*(0.303014)+x13*(0.146605)+x14*(-0.140073)+x15*(0.334945)+x16*(0.395876)+(-0.1744)); 
    h7 = tanh(x0*(-0.594258)+x1*(-0.780454)+x2*(-0.898894)+x3*(-1.14982)+x4*(-0.223219)+x5*(-0.185079)+x6*(1.06174)+x7*(2.02207)+x8*(-1.8989)+x9*(-0.245548)+x10*(-0.64278)+x11*(-0.255858)+x12*(0.0454656)+x13*(-0.0719088)+x14*(0.117479)+x15*(-0.276539)+x16*(-0.361317)+(-0.315689)); 
    h8 = tanh(x0*(0.646248)+x1*(0.419375)+x2*(1.03952)+x3*(-0.11226)+x4*(-0.86312)+x5*(-0.560752)+x6*(0.381195)+x7*(0.278965)+x8*(-4.49588)+x9*(0.10417)+x10*(0.334554)+x11*(-1.27195)+x12*(0.195742)+x13*(0.386633)+x14*(0.400836)+x15*(-0.208185)+x16*(0.101954)+(0.00644789)); 
    h9 = tanh(x0*(0.684406)+x1*(0.422838)+x2*(0.132128)+x3*(0.982623)+x4*(-0.292089)+x5*(1.65083)+x6*(-1.11052)+x7*(2.23562)+x8*(4.04737)+x9*(-0.831212)+x10*(-0.834408)+x11*(4.99724)+x12*(0.386445)+x13*(-0.390053)+x14*(0.214983)+x15*(0.339401)+x16*(0.147368)+(-0.547253)); 
    h10 = tanh(x0*(-0.159243)+x1*(-0.196235)+x2*(0.374568)+x3*(0.997969)+x4*(-0.54283)+x5*(-0.42839)+x6*(-3.31705)+x7*(0.268332)+x8*(-0.829407)+x9*(0.0159866)+x10*(-0.14333)+x11*(-0.9201)+x12*(-0.361003)+x13*(0.38666)+x14*(0.264905)+x15*(-0.251065)+x16*(-0.320869)+(0.0940232)); 
    h11 = tanh(x0*(0.466595)+x1*(0.518197)+x2*(0.341809)+x3*(1.29966)+x4*(0.863066)+x5*(1.30904)+x6*(-2.69671)+x7*(-1.13142)+x8*(-1.60244)+x9*(1.15994)+x10*(0.852764)+x11*(6.47818)+x12*(-0.204065)+x13*(0.193944)+x14*(0.334839)+x15*(-0.0616772)+x16*(0.308358)+(1.03893)); 
    h12 = tanh(x0*(-6.78545)+x1*(-6.95836)+x2*(-6.80495)+x3*(0.519756)+x4*(-0.694652)+x5*(0.0877993)+x6*(-1.04977)+x7*(1.91271)+x8*(-3.29146)+x9*(-1.49102)+x10*(-0.219575)+x11*(-0.41558)+x12*(-0.348428)+x13*(0.328561)+x14*(-0.0438982)+x15*(-0.140404)+x16*(-0.146071)+(-0.714232)); 
    h13 = tanh(x0*(0.469477)+x1*(0.457969)+x2*(0.29392)+x3*(0.113084)+x4*(0.246469)+x5*(-0.267005)+x6*(0.00323499)+x7*(0.33386)+x8*(-2.29425)+x9*(0.525681)+x10*(0.39345)+x11*(-0.990887)+x12*(0.111247)+x13*(-0.155932)+x14*(-0.075639)+x15*(0.165444)+x16*(-0.150989)+(0.0990027)); 
    h14 = tanh(x0*(-0.485333)+x1*(-0.431249)+x2*(0.0254683)+x3*(-0.788429)+x4*(-1.17881)+x5*(0.0114553)+x6*(0.651314)+x7*(2.64158)+x8*(-2.39344)+x9*(-0.385214)+x10*(-0.0999452)+x11*(-0.237994)+x12*(-0.168357)+x13*(0.374728)+x14*(-0.209556)+x15*(0.334932)+x16*(0.339687)+(-0.424553)); 
    h15 = tanh(x0*(0.499173)+x1*(0.922931)+x2*(0.856594)+x3*(0.288473)+x4*(-0.259846)+x5*(0.0231394)+x6*(0.0733)+x7*(-0.285896)+x8*(-0.0277038)+x9*(-0.0896544)+x10*(0.108253)+x11*(-0.00113387)+x12*(0.280544)+x13*(0.275887)+x14*(-0.302862)+x15*(-0.271225)+x16*(-0.324519)+(-0.198107)); 
    h16 = tanh(x0*(0.808858)+x1*(0.316371)+x2*(0.315332)+x3*(-0.259632)+x4*(-0.180166)+x5*(1.79941)+x6*(-0.723082)+x7*(-1.46245)+x8*(4.24796)+x9*(1.07443)+x10*(0.860527)+x11*(2.63598)+x12*(-0.235638)+x13*(-0.0264941)+x14*(0.120375)+x15*(-0.010813)+x16*(0.0900996)+(0.831968)); 
    h17 = tanh(x0*(-1.13254)+x1*(-0.977224)+x2*(-0.450414)+x3*(-0.94528)+x4*(-0.269719)+x5*(-0.0689991)+x6*(-0.197821)+x7*(-1.43212)+x8*(-0.907266)+x9*(-0.0917746)+x10*(0.689461)+x11*(0.375089)+x12*(-0.241656)+x13*(0.0760486)+x14*(-0.056371)+x15*(0.368781)+x16*(-0.0672431)+(0.189518)); 
    h18 = tanh(x0*(0.0894602)+x1*(0.0587961)+x2*(0.503699)+x3*(1.05488)+x4*(-0.489494)+x5*(0.159066)+x6*(-1.77957)+x7*(2.80441)+x8*(-1.64448)+x9*(-0.207814)+x10*(-0.203783)+x11*(0.180471)+x12*(-0.0581705)+x13*(-0.255144)+x14*(0.0355695)+x15*(0.127005)+x16*(-0.31211)+(-0.063084)); 
    h19 = tanh(x0*(-0.140692)+x1*(0.305936)+x2*(0.0371038)+x3*(-0.25396)+x4*(0.675463)+x5*(0.385465)+x6*(1.45333)+x7*(-0.626602)+x8*(-0.562507)+x9*(0.11831)+x10*(-0.095387)+x11*(1.70887)+x12*(0.0606929)+x13*(0.0931314)+x14*(0.368823)+x15*(-0.234239)+x16*(-0.0792973)+(0.0901418)); 
}

void FNN_3_3_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.0413246)+x1*(0.371637)+x2*(0.0336614)+x3*(0.00453012)+x4*(0.328177)+x5*(0.444268)+x6*(0.165831)+x7*(-0.335639)+x8*(-1.04639)+x9*(0.241729)+x10*(0.423418)+x11*(-0.0190057)+x12*(-0.549924)+x13*(0.0738298)+x14*(-0.136209)+x15*(0.509999)+x16*(0.543548)+x17*(0.289583)+x18*(0.58922)+x19*(-0.793206)+(-0.173221)); 
    h1 = tanh(x0*(0.510904)+x1*(-1.13102)+x2*(-0.184869)+x3*(0.472018)+x4*(-0.668477)+x5*(-1.01068)+x6*(0.0966633)+x7*(1.14824)+x8*(0.752279)+x9*(-1.02734)+x10*(-0.270647)+x11*(-0.220909)+x12*(0.245109)+x13*(-0.287639)+x14*(-0.662989)+x15*(0.0362452)+x16*(-1.13938)+x17*(0.586633)+x18*(1.12468)+x19*(0.621046)+(-0.154612)); 
    h2 = tanh(x0*(1.23297)+x1*(0.985928)+x2*(2.24131)+x3*(-0.409398)+x4*(0.0944432)+x5*(-1.31188)+x6*(-0.237207)+x7*(0.357146)+x8*(0.301286)+x9*(-0.394173)+x10*(0.54038)+x11*(0.744115)+x12*(-0.102534)+x13*(0.463209)+x14*(-1.13355)+x15*(0.222465)+x16*(0.0899688)+x17*(0.60749)+x18*(-0.220943)+x19*(-1.31296)+(0.143365)); 
    h3 = tanh(x0*(-1.96846)+x1*(-0.198236)+x2*(-0.775278)+x3*(-0.275408)+x4*(0.60244)+x5*(0.657956)+x6*(0.24032)+x7*(0.271737)+x8*(-0.291137)+x9*(1.32226)+x10*(0.392053)+x11*(-1.18294)+x12*(1.06202)+x13*(0.554614)+x14*(1.01573)+x15*(-0.423656)+x16*(0.977581)+x17*(-0.933992)+x18*(0.525278)+x19*(-0.919096)+(-0.0223216)); 
    h4 = tanh(x0*(2.048)+x1*(0.188784)+x2*(0.953134)+x3*(0.374621)+x4*(-0.378502)+x5*(-0.644326)+x6*(-0.158923)+x7*(-0.294611)+x8*(0.30354)+x9*(-1.26606)+x10*(-0.454834)+x11*(1.19629)+x12*(-1.09675)+x13*(-0.44426)+x14*(-1.05413)+x15*(0.342489)+x16*(-0.950392)+x17*(1.00146)+x18*(-0.476512)+x19*(0.887215)+(0.182486)); 
    h5 = tanh(x0*(0.22045)+x1*(-0.29058)+x2*(-1.55514)+x3*(-0.568099)+x4*(0.302545)+x5*(-0.386506)+x6*(-0.993547)+x7*(0.0418745)+x8*(-0.186007)+x9*(0.627754)+x10*(0.304906)+x11*(-1.48406)+x12*(-0.308647)+x13*(0.0169548)+x14*(0.00125012)+x15*(-0.535538)+x16*(-0.128401)+x17*(-0.671684)+x18*(0.111487)+x19*(-1.22427)+(0.315071)); 
    h6 = tanh(x0*(0.439332)+x1*(-0.195362)+x2*(-0.0862463)+x3*(0.223919)+x4*(0.925085)+x5*(0.804263)+x6*(0.669219)+x7*(-0.397444)+x8*(-0.598196)+x9*(1.1081)+x10*(0.297799)+x11*(-0.582374)+x12*(-0.628976)+x13*(-0.609635)+x14*(-0.35971)+x15*(-0.0592845)+x16*(0.609338)+x17*(-1.11427)+x18*(0.291865)+x19*(0.515526)+(-0.313355)); 
    h7 = tanh(x0*(1.29258)+x1*(-0.00504076)+x2*(-1.21818)+x3*(-0.142645)+x4*(-0.47733)+x5*(0.991498)+x6*(-0.0248363)+x7*(-1.36484)+x8*(-0.617244)+x9*(-1.22183)+x10*(0.202718)+x11*(0.225644)+x12*(-0.224984)+x13*(-0.125628)+x14*(-0.98484)+x15*(-0.779325)+x16*(0.877862)+x17*(-0.15992)+x18*(-1.21271)+x19*(0.488755)+(-0.0599906)); 
    h8 = tanh(x0*(0.252068)+x1*(-0.164236)+x2*(0.425198)+x3*(0.077006)+x4*(0.176109)+x5*(-0.911651)+x6*(0.310901)+x7*(0.742547)+x8*(0.929652)+x9*(-0.584738)+x10*(-1.34171)+x11*(0.922535)+x12*(0.606134)+x13*(0.012454)+x14*(0.215785)+x15*(0.625265)+x16*(-1.00507)+x17*(0.335953)+x18*(-0.609693)+x19*(1.20863)+(-0.0632501)); 
    h9 = tanh(x0*(1.04805)+x1*(0.0337118)+x2*(-1.24702)+x3*(0.199799)+x4*(0.381905)+x5*(0.634908)+x6*(-0.111126)+x7*(0.000546311)+x8*(-1.3399)+x9*(1.04456)+x10*(0.280895)+x11*(-1.73013)+x12*(-0.808649)+x13*(0.423821)+x14*(-0.192567)+x15*(0.567928)+x16*(0.62347)+x17*(-0.682805)+x18*(-0.895023)+x19*(0.733407)+(-0.151695)); 
}

void FNN_3_3_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-0.453205)+x1*(-0.633588)+x2*(0.324429)+x3*(-0.397406)+x4*(0.483065)+x5*(0.698753)+x6*(-0.730645)+x7*(-0.816765)+x8*(-0.528827)+x9*(0.750054)+(0.289108); 
    h1 = x0*(-0.33742)+x1*(-0.602265)+x2*(0.316266)+x3*(-1.03017)+x4*(-0.100209)+x5*(0.696418)+x6*(-0.7819)+x7*(-0.855326)+x8*(-0.504977)+x9*(0.772251)+(0.327167); 
    h2 = x0*(-0.190744)+x1*(-0.622508)+x2*(0.298107)+x3*(-0.115458)+x4*(0.766103)+x5*(0.706548)+x6*(-0.759676)+x7*(-0.824971)+x8*(-0.413458)+x9*(0.74539)+(0.36164); 
}

void FNN_3_3(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_3_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_3_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_3_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_3(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_3_5_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(0.285777)+x1*(-0.139379)+x2*(0.0215598)+x3*(-0.110973)+x4*(0.108165)+x5*(1.08466)+x6*(0.15921)+x7*(0.0616863)+x8*(-1.59246)+x9*(0.102186)+x10*(-0.307216)+x11*(-1.73499)+x12*(-0.314332)+x13*(0.263967)+x14*(0.185359)+x15*(-0.314642)+x16*(0.185111)+(-0.1858)); 
    h1 = tanh(x0*(-0.75082)+x1*(-0.438132)+x2*(-0.579074)+x3*(0.431574)+x4*(-0.234048)+x5*(-4.52022)+x6*(1.04331)+x7*(-0.235127)+x8*(-1.55465)+x9*(-0.44206)+x10*(-0.275277)+x11*(-3.89166)+x12*(0.0777281)+x13*(-0.0191149)+x14*(0.0696949)+x15*(-0.175921)+x16*(-0.0955278)+(-0.26391)); 
    h2 = tanh(x0*(0.357702)+x1*(0.622564)+x2*(0.0915322)+x3*(-1.26711)+x4*(-0.669714)+x5*(-1.51992)+x6*(3.3808)+x7*(2.12686)+x8*(-3.24648)+x9*(0.0898825)+x10*(-0.714116)+x11*(-4.56659)+x12*(-0.0840614)+x13*(-0.230766)+x14*(-0.183521)+x15*(0.379397)+x16*(-0.250942)+(-0.410328)); 
    h3 = tanh(x0*(-0.350107)+x1*(-0.284083)+x2*(-0.595579)+x3*(0.865033)+x4*(0.167315)+x5*(-0.252115)+x6*(-1.45024)+x7*(-0.878428)+x8*(-1.50564)+x9*(0.314252)+x10*(0.16248)+x11*(1.39917)+x12*(0.104253)+x13*(-0.216457)+x14*(0.334984)+x15*(-0.132767)+x16*(-0.105027)+(-0.03253)); 
    h4 = tanh(x0*(-0.0190637)+x1*(-0.581293)+x2*(-0.574616)+x3*(0.545231)+x4*(-0.470647)+x5*(1.55403)+x6*(-1.63159)+x7*(2.06114)+x8*(2.29116)+x9*(-0.539619)+x10*(-0.128623)+x11*(4.07006)+x12*(0.288096)+x13*(-0.109263)+x14*(0.10701)+x15*(0.039765)+x16*(0.213541)+(-0.37746)); 
    h5 = tanh(x0*(-1.23465)+x1*(-0.654467)+x2*(-1.09336)+x3*(-0.0584)+x4*(-0.981659)+x5*(-0.243616)+x6*(-0.632836)+x7*(-0.0183349)+x8*(-3.47421)+x9*(-0.0369575)+x10*(-0.404376)+x11*(-1.91818)+x12*(-0.145395)+x13*(0.362713)+x14*(-0.202933)+x15*(0.335931)+x16*(0.349995)+(-0.335971)); 
    h6 = tanh(x0*(0.678993)+x1*(0.242424)+x2*(0.367649)+x3*(0.247943)+x4*(0.236059)+x5*(2.52517)+x6*(-0.408472)+x7*(0.311865)+x8*(0.389661)+x9*(-0.108933)+x10*(0.0992142)+x11*(0.164629)+x12*(0.103536)+x13*(-0.173923)+x14*(-0.303126)+x15*(-0.370283)+x16*(-0.262146)+(0.026285)); 
    h7 = tanh(x0*(0.121596)+x1*(0.240866)+x2*(-0.34846)+x3*(0.132341)+x4*(-0.594558)+x5*(-2.48852)+x6*(0.161773)+x7*(0.0523036)+x8*(-0.529784)+x9*(-0.0951959)+x10*(-0.290882)+x11*(-2.91706)+x12*(0.0293808)+x13*(-0.0866495)+x14*(0.350347)+x15*(-0.0172265)+x16*(0.358204)+(-0.193144)); 
    h8 = tanh(x0*(0.436287)+x1*(0.908215)+x2*(0.925226)+x3*(0.739519)+x4*(-0.607987)+x5*(0.486422)+x6*(-0.822846)+x7*(0.878479)+x8*(3.54357)+x9*(-0.566789)+x10*(-0.53578)+x11*(1.26658)+x12*(0.179984)+x13*(-0.219381)+x14*(-0.1495)+x15*(-0.201763)+x16*(0.238971)+(-0.432053)); 
    h9 = tanh(x0*(-0.885657)+x1*(-0.465007)+x2*(-0.325452)+x3*(-1.25979)+x4*(0.397992)+x5*(3.06994)+x6*(-1.85747)+x7*(0.263526)+x8*(-0.487005)+x9*(-0.159781)+x10*(0.0768771)+x11*(5.83267)+x12*(-0.144872)+x13*(-0.154541)+x14*(0.221967)+x15*(0.314795)+x16*(-0.323606)+(0.136823)); 
    h10 = tanh(x0*(0.613621)+x1*(0.0674413)+x2*(0.468505)+x3*(-0.0169907)+x4*(0.237977)+x5*(0.0350479)+x6*(1.41364)+x7*(0.210457)+x8*(0.107013)+x9*(-0.0883358)+x10*(-0.413862)+x11*(-0.920052)+x12*(-0.0678231)+x13*(-0.0674959)+x14*(-0.39952)+x15*(0.291169)+x16*(-0.302609)+(-0.180929)); 
    h11 = tanh(x0*(0.0182466)+x1*(0.630175)+x2*(0.366586)+x3*(0.0932677)+x4*(0.170535)+x5*(1.77793)+x6*(-0.120377)+x7*(-0.423161)+x8*(-1.43687)+x9*(0.277506)+x10*(-0.257701)+x11*(4.01247)+x12*(0.372254)+x13*(-0.111451)+x14*(-0.0477667)+x15*(-0.146589)+x16*(-0.19474)+(-0.0686846)); 
    h12 = tanh(x0*(4.27589)+x1*(4.2074)+x2*(4.45752)+x3*(-0.488516)+x4*(0.10353)+x5*(-0.424867)+x6*(1.67524)+x7*(0.769615)+x8*(2.01574)+x9*(0.410225)+x10*(-0.0528024)+x11*(-1.89784)+x12*(0.219293)+x13*(0.129958)+x14*(0.392782)+x15*(-0.0273888)+x16*(-0.182806)+(0.0405396)); 
    h13 = tanh(x0*(0.283775)+x1*(0.43343)+x2*(-0.141507)+x3*(-0.0647382)+x4*(0.140589)+x5*(-0.787055)+x6*(-0.686883)+x7*(0.287324)+x8*(-0.526969)+x9*(0.123423)+x10*(-0.0175919)+x11*(-1.86301)+x12*(0.0630632)+x13*(-0.28633)+x14*(0.395991)+x15*(0.0506536)+x16*(-0.052409)+(-0.0293416)); 
    h14 = tanh(x0*(0.326507)+x1*(0.55077)+x2*(0.548833)+x3*(1.46088)+x4*(0.330914)+x5*(-0.190518)+x6*(-1.29315)+x7*(-0.598792)+x8*(-2.73874)+x9*(0.560078)+x10*(0.101745)+x11*(-1.60275)+x12*(-0.263581)+x13*(0.0907346)+x14*(0.198868)+x15*(0.373207)+x16*(-0.0900322)+(0.690555)); 
    h15 = tanh(x0*(-0.0534039)+x1*(0.0660158)+x2*(-0.295102)+x3*(-0.972856)+x4*(-0.683378)+x5*(-2.51647)+x6*(-2.07471)+x7*(-0.470586)+x8*(0.733589)+x9*(0.283402)+x10*(-0.087792)+x11*(-2.47767)+x12*(0.129972)+x13*(-0.228185)+x14*(0.125392)+x15*(0.183776)+x16*(0.302265)+(0.143404)); 
    h16 = tanh(x0*(-0.93911)+x1*(-1.11025)+x2*(-1.57329)+x3*(-0.769636)+x4*(-0.395307)+x5*(-3.38899)+x6*(2.22738)+x7*(-0.970537)+x8*(1.96789)+x9*(0.5252)+x10*(0.0506988)+x11*(-5.95359)+x12*(-0.159429)+x13*(-0.196338)+x14*(0.0204865)+x15*(0.142839)+x16*(-0.376596)+(0.123037)); 
    h17 = tanh(x0*(-0.807185)+x1*(-0.387933)+x2*(-0.442054)+x3*(0.550423)+x4*(-0.476375)+x5*(-1.8958)+x6*(-0.312398)+x7*(0.89144)+x8*(-0.650355)+x9*(-0.225762)+x10*(-0.229551)+x11*(-4.02699)+x12*(-0.109346)+x13*(0.296865)+x14*(0.215455)+x15*(-0.34652)+x16*(-0.0493589)+(0.0546542)); 
    h18 = tanh(x0*(0.580073)+x1*(0.287282)+x2*(0.0510334)+x3*(-0.748728)+x4*(0.31256)+x5*(-0.667879)+x6*(1.92525)+x7*(-0.720643)+x8*(-0.778525)+x9*(-0.236537)+x10*(-0.0195467)+x11*(-2.64011)+x12*(-0.169831)+x13*(-0.0147846)+x14*(0.331971)+x15*(-0.0141903)+x16*(-0.36003)+(-0.181141)); 
    h19 = tanh(x0*(-0.72562)+x1*(-0.0841694)+x2*(-0.562653)+x3*(-0.44761)+x4*(0.371784)+x5*(-0.28023)+x6*(0.423431)+x7*(0.616631)+x8*(0.361961)+x9*(0.201944)+x10*(0.224057)+x11*(-1.79023)+x12*(0.256776)+x13*(-0.1261)+x14*(-0.171178)+x15*(-0.0617976)+x16*(0.188236)+(-0.306211)); 
}

void FNN_3_5_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.355592)+x1*(0.555521)+x2*(0.133112)+x3*(-0.824186)+x4*(-0.582309)+x5*(-0.45097)+x6*(-0.918139)+x7*(0.64013)+x8*(1.02001)+x9*(-1.54347)+x10*(0.471557)+x11*(0.2154)+x12*(0.320403)+x13*(0.103926)+x14*(-1.01466)+x15*(-0.528682)+x16*(1.2711)+x17*(-0.000966236)+x18*(1.02074)+x19*(1.12421)+(-0.105541)); 
    h1 = tanh(x0*(0.0160356)+x1*(0.506674)+x2*(0.468665)+x3*(-1.82944)+x4*(0.417468)+x5*(0.241348)+x6*(0.18639)+x7*(-0.319236)+x8*(0.0922374)+x9*(-1.97599)+x10*(1.45201)+x11*(-0.509903)+x12*(-0.838221)+x13*(-0.661679)+x14*(-0.47194)+x15*(-0.137115)+x16*(1.99139)+x17*(-0.306464)+x18*(1.2506)+x19*(0.255717)+(0.0882759)); 
    h2 = tanh(x0*(-0.418723)+x1*(0.36704)+x2*(0.441786)+x3*(0.0606247)+x4*(-0.148006)+x5*(-0.332046)+x6*(-0.201106)+x7*(-0.147148)+x8*(-0.213748)+x9*(0.0377732)+x10*(-0.124859)+x11*(0.432509)+x12*(0.40165)+x13*(-0.605991)+x14*(0.0234617)+x15*(-0.530787)+x16*(-0.0656532)+x17*(-0.493435)+x18*(0.231386)+x19*(0.501667)+(0.0108844)); 
    h3 = tanh(x0*(1.57435)+x1*(0.89265)+x2*(1.42001)+x3*(0.00299687)+x4*(-0.0900297)+x5*(0.795889)+x6*(0.145798)+x7*(0.265004)+x8*(-0.728695)+x9*(-0.0516724)+x10*(0.0612226)+x11*(0.434908)+x12*(0.0302095)+x13*(-0.0789155)+x14*(0.279166)+x15*(0.0903874)+x16*(-0.0842591)+x17*(-0.246223)+x18*(0.0201991)+x19*(0.238794)+(0.0692021)); 
    h4 = tanh(x0*(0.522327)+x1*(0.593342)+x2*(0.776712)+x3*(-0.647127)+x4*(0.0914132)+x5*(0.200126)+x6*(0.0363897)+x7*(0.000235783)+x8*(-1.60565)+x9*(0.813319)+x10*(0.698767)+x11*(0.988618)+x12*(-0.348801)+x13*(0.367617)+x14*(0.757514)+x15*(0.196558)+x16*(-0.702528)+x17*(0.949218)+x18*(0.376519)+x19*(-0.462573)+(-0.166049)); 
    h5 = tanh(x0*(0.678761)+x1*(-0.0888746)+x2*(0.169573)+x3*(-0.307811)+x4*(-0.440026)+x5*(0.038181)+x6*(-0.674146)+x7*(0.459442)+x8*(-0.555562)+x9*(-0.116121)+x10*(-0.194262)+x11*(0.0817852)+x12*(-0.641659)+x13*(0.914766)+x14*(0.285199)+x15*(0.394279)+x16*(0.549557)+x17*(1.19945)+x18*(0.653939)+x19*(-0.5621)+(-0.159044)); 
    h6 = tanh(x0*(0.628165)+x1*(0.600237)+x2*(3.39541)+x3*(0.266514)+x4*(2.33145)+x5*(1.14223)+x6*(0.883115)+x7*(0.28927)+x8*(1.15437)+x9*(0.061633)+x10*(0.454935)+x11*(-0.797199)+x12*(0.560332)+x13*(0.668622)+x14*(-1.0474)+x15*(0.217125)+x16*(-0.754152)+x17*(0.78438)+x18*(-1.44491)+x19*(0.762282)+(-0.0509421)); 
    h7 = tanh(x0*(1.5241)+x1*(0.667185)+x2*(1.53113)+x3*(0.190703)+x4*(0.0186224)+x5*(0.5777)+x6*(-0.453991)+x7*(0.0162624)+x8*(-0.650998)+x9*(-0.130485)+x10*(0.0262043)+x11*(0.200565)+x12*(-0.120409)+x13*(-0.017572)+x14*(0.377551)+x15*(0.205023)+x16*(-0.171512)+x17*(0.0742254)+x18*(0.485014)+x19*(0.0622721)+(0.0476768)); 
    h8 = tanh(x0*(0.524597)+x1*(0.567935)+x2*(-0.565455)+x3*(-0.724436)+x4*(0.341475)+x5*(0.556205)+x6*(-0.243078)+x7*(0.325377)+x8*(0.879968)+x9*(-0.427716)+x10*(-0.671143)+x11*(0.0242341)+x12*(-0.277256)+x13*(1.19516)+x14*(-0.206889)+x15*(0.450164)+x16*(-1.2434)+x17*(0.631517)+x18*(-0.191207)+x19*(0.699652)+(-0.323119)); 
    h9 = tanh(x0*(-0.327649)+x1*(-0.799908)+x2*(1.273)+x3*(-0.315108)+x4*(2.66094)+x5*(0.108083)+x6*(0.278443)+x7*(-0.247238)+x8*(0.858624)+x9*(0.157236)+x10*(-0.203302)+x11*(-0.590285)+x12*(-0.59955)+x13*(0.66747)+x14*(-0.0858532)+x15*(-0.187446)+x16*(-0.492858)+x17*(1.90232)+x18*(0.181563)+x19*(-0.0414809)+(0.190302)); 
}

void FNN_3_5_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-1.30085)+x1*(0.952754)+x2*(-0.521007)+x3*(0.524313)+x4*(-0.952386)+x5*(-0.532363)+x6*(0.479181)+x7*(0.239599)+x8*(-0.493557)+x9*(-0.599415)+(0.0879578); 
    h1 = x0*(-1.29489)+x1*(0.952566)+x2*(-0.511914)+x3*(0.0298246)+x4*(-0.896111)+x5*(-0.542648)+x6*(0.475805)+x7*(0.685902)+x8*(-0.489495)+x9*(-0.599763)+(0.0879645); 
    h2 = x0*(-1.29249)+x1*(0.90399)+x2*(-0.672478)+x3*(0.792575)+x4*(-0.993051)+x5*(-0.5659)+x6*(0.469589)+x7*(0.0427096)+x8*(-0.540366)+x9*(-0.588091)+(0.188647); 
}

void FNN_3_5(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_5_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_5_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_5_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_5(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_3_6_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.334122)+x1*(-0.120617)+x2*(-0.235446)+x3*(-0.388833)+x4*(-0.223152)+x5*(-1.54323)+x6*(-0.203985)+x7*(-1.04047)+x8*(-3.27095)+x9*(0.204862)+x10*(0.22276)+x11*(-1.4816)+x12*(-0.227947)+x13*(0.157362)+x14*(-0.107413)+x15*(-0.174499)+x16*(0.285151)+(0.328525)); 
    h1 = tanh(x0*(0.731362)+x1*(0.316308)+x2*(0.483551)+x3*(0.0751731)+x4*(-0.113241)+x5*(-0.888899)+x6*(-0.169441)+x7*(-0.464086)+x8*(-2.2369)+x9*(0.368317)+x10*(-0.0542635)+x11*(-0.9917)+x12*(-0.0962552)+x13*(0.128387)+x14*(0.318399)+x15*(0.170445)+x16*(0.186643)+(0.0443826)); 
    h2 = tanh(x0*(-0.593728)+x1*(-0.284678)+x2*(-0.104618)+x3*(-0.101711)+x4*(0.501248)+x5*(-0.56059)+x6*(-0.293235)+x7*(-0.199782)+x8*(-0.890899)+x9*(-0.173698)+x10*(0.388735)+x11*(-1.49667)+x12*(-0.359555)+x13*(0.164031)+x14*(0.387264)+x15*(-0.348229)+x16*(0.309159)+(0.172597)); 
    h3 = tanh(x0*(-0.491267)+x1*(-0.470701)+x2*(0.175309)+x3*(0.897393)+x4*(-0.333226)+x5*(0.0371765)+x6*(-1.31832)+x7*(1.03117)+x8*(-2.21599)+x9*(-0.0426725)+x10*(0.120956)+x11*(-0.885056)+x12*(0.0460417)+x13*(-0.311416)+x14*(-0.0517268)+x15*(-0.150376)+x16*(0.161178)+(0.137664)); 
    h4 = tanh(x0*(-0.832085)+x1*(-0.768874)+x2*(-1.30171)+x3*(0.0130774)+x4*(0.296152)+x5*(-0.525998)+x6*(0.153217)+x7*(-1.06499)+x8*(1.92952)+x9*(0.327376)+x10*(0.300048)+x11*(-0.360941)+x12*(-0.18414)+x13*(-0.0442093)+x14*(-0.136577)+x15*(0.383923)+x16*(0.216861)+(-0.0692946)); 
    h5 = tanh(x0*(1.33994)+x1*(1.19187)+x2*(0.834346)+x3*(0.00341062)+x4*(0.315102)+x5*(-0.205418)+x6*(0.0180457)+x7*(0.264016)+x8*(-0.305491)+x9*(0.0766972)+x10*(0.191514)+x11*(0.128239)+x12*(-0.365676)+x13*(-0.173826)+x14*(-0.174207)+x15*(0.164753)+x16*(-0.184197)+(0.121375)); 
    h6 = tanh(x0*(0.0510027)+x1*(-0.141342)+x2*(-0.0308987)+x3*(-0.671929)+x4*(-0.63186)+x5*(-1.26798)+x6*(-0.40067)+x7*(0.275886)+x8*(-0.916728)+x9*(0.0202232)+x10*(-0.343743)+x11*(-1.46891)+x12*(0.25702)+x13*(-0.278402)+x14*(0.120266)+x15*(-0.214508)+x16*(0.351265)+(-0.219637)); 
    h7 = tanh(x0*(-0.336178)+x1*(-0.494953)+x2*(-0.300892)+x3*(-0.482917)+x4*(-0.351386)+x5*(0.7116)+x6*(0.969586)+x7*(0.446337)+x8*(0.204909)+x9*(0.130234)+x10*(0.345339)+x11*(0.217218)+x12*(0.164057)+x13*(0.376818)+x14*(0.0998549)+x15*(0.229693)+x16*(0.0103422)+(0.158713)); 
    h8 = tanh(x0*(-0.167507)+x1*(-0.0200146)+x2*(-0.0259514)+x3*(-1.1934)+x4*(-0.0644854)+x5*(0.0279617)+x6*(0.785171)+x7*(-0.735187)+x8*(0.748003)+x9*(-0.345266)+x10*(0.0897156)+x11*(-0.314822)+x12*(0.384214)+x13*(-0.22668)+x14*(0.0409448)+x15*(0.0507469)+x16*(0.0412453)+(-0.23278)); 
    h9 = tanh(x0*(0.175329)+x1*(0.659153)+x2*(0.734013)+x3*(-0.293233)+x4*(-0.350052)+x5*(1.04999)+x6*(-0.158915)+x7*(1.05788)+x8*(1.00146)+x9*(0.297938)+x10*(-0.387214)+x11*(1.31265)+x12*(-0.11036)+x13*(-0.183675)+x14*(-0.286722)+x15*(-0.102699)+x16*(-0.213287)+(-0.112531)); 
    h10 = tanh(x0*(-0.344453)+x1*(-0.00749871)+x2*(-0.270709)+x3*(0.288099)+x4*(0.291171)+x5*(-1.07613)+x6*(0.348707)+x7*(0.0972856)+x8*(1.09478)+x9*(-0.607074)+x10*(0.143117)+x11*(-0.00762214)+x12*(-0.0390517)+x13*(-0.271632)+x14*(-0.169974)+x15*(0.186432)+x16*(-0.104605)+(-0.397024)); 
    h11 = tanh(x0*(0.364747)+x1*(0.117119)+x2*(-0.14511)+x3*(0.757291)+x4*(-0.219993)+x5*(-0.421687)+x6*(-1.79352)+x7*(1.26843)+x8*(-3.34187)+x9*(0.141324)+x10*(0.581879)+x11*(-1.22999)+x12*(-0.122188)+x13*(0.0384884)+x14*(0.0174293)+x15*(0.236775)+x16*(0.208126)+(0.322214)); 
    h12 = tanh(x0*(0.197313)+x1*(-0.107366)+x2*(0.0853656)+x3*(0.100849)+x4*(-0.650773)+x5*(0.0915402)+x6*(-0.85546)+x7*(0.514787)+x8*(-0.78682)+x9*(0.0902645)+x10*(-0.366076)+x11*(-0.463611)+x12*(0.366455)+x13*(0.308729)+x14*(0.394279)+x15*(-0.265936)+x16*(0.190676)+(-0.0829221)); 
    h13 = tanh(x0*(-0.166221)+x1*(0.24423)+x2*(-0.258915)+x3*(0.0146246)+x4*(-0.235021)+x5*(2.25449)+x6*(-0.585473)+x7*(1.09386)+x8*(0.931432)+x9*(-0.125841)+x10*(-0.38676)+x11*(1.48546)+x12*(0.282705)+x13*(0.327962)+x14*(0.240089)+x15*(0.123486)+x16*(0.117732)+(-0.201173)); 
    h14 = tanh(x0*(-3.95109)+x1*(-3.59252)+x2*(-3.53823)+x3*(-1.62)+x4*(-0.684856)+x5*(0.174868)+x6*(1.59147)+x7*(0.320153)+x8*(3.88761)+x9*(-0.56472)+x10*(-0.366734)+x11*(1.04588)+x12*(0.148993)+x13*(-0.325358)+x14*(-0.276654)+x15*(-0.266098)+x16*(0.110345)+(-0.722666)); 
    h15 = tanh(x0*(0.158153)+x1*(-0.247938)+x2*(-0.289848)+x3*(0.215276)+x4*(0.0864283)+x5*(0.0846262)+x6*(-0.0917113)+x7*(-0.281647)+x8*(-0.956261)+x9*(0.055139)+x10*(0.2991)+x11*(0.0921603)+x12*(0.347006)+x13*(0.0897046)+x14*(0.0292262)+x15*(0.315076)+x16*(0.22516)+(0.276822)); 
    h16 = tanh(x0*(-0.609685)+x1*(-0.30161)+x2*(-0.0336721)+x3*(0.248412)+x4*(-0.632426)+x5*(0.13697)+x6*(0.138118)+x7*(1.27376)+x8*(1.23626)+x9*(-0.687936)+x10*(-0.493937)+x11*(1.65019)+x12*(0.213133)+x13*(0.204836)+x14*(0.163299)+x15*(-0.320065)+x16*(0.124753)+(-0.637355)); 
    h17 = tanh(x0*(0.283442)+x1*(0.902669)+x2*(0.630419)+x3*(0.464323)+x4*(0.513247)+x5*(-0.29816)+x6*(1.01996)+x7*(-2.03791)+x8*(-3.27143)+x9*(-0.183051)+x10*(-0.333514)+x11*(-1.48833)+x12*(-0.234927)+x13*(0.0760754)+x14*(-0.29776)+x15*(-0.362345)+x16*(0.155828)+(0.0681723)); 
    h18 = tanh(x0*(-0.367946)+x1*(-0.685293)+x2*(-0.250224)+x3*(1.44575)+x4*(0.533195)+x5*(-0.238183)+x6*(-1.77199)+x7*(-1.88302)+x8*(-1.41895)+x9*(0.199331)+x10*(0.104823)+x11*(0.378296)+x12*(0.232208)+x13*(-0.191109)+x14*(0.186587)+x15*(-0.12434)+x16*(-0.108356)+(0.19117)); 
    h19 = tanh(x0*(1.01724)+x1*(0.889792)+x2*(1.06421)+x3*(1.11479)+x4*(-0.147226)+x5*(0.062214)+x6*(0.532359)+x7*(-0.126212)+x8*(0.367181)+x9*(0.394928)+x10*(0.0275512)+x11*(0.747967)+x12*(0.383721)+x13*(0.218982)+x14*(-0.198788)+x15*(0.157136)+x16*(0.24307)+(0.155086)); 
}

void FNN_3_6_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.613995)+x1*(-0.74196)+x2*(-0.591981)+x3*(-0.0346866)+x4*(0.337041)+x5*(-0.204547)+x6*(0.299477)+x7*(-0.767038)+x8*(0.627026)+x9*(-0.0223534)+x10*(0.110223)+x11*(0.0491056)+x12*(0.422055)+x13*(-0.15811)+x14*(0.517608)+x15*(0.325865)+x16*(-0.268399)+x17*(-1.15263)+x18*(0.429698)+x19*(0.405273)+(0.0900716)); 
    h1 = tanh(x0*(-0.308392)+x1*(-0.41154)+x2*(-0.314838)+x3*(0.688094)+x4*(0.361816)+x5*(0.246456)+x6*(-0.206506)+x7*(-0.118287)+x8*(0.498686)+x9*(0.567548)+x10*(-0.32561)+x11*(0.203695)+x12*(0.129459)+x13*(0.27532)+x14*(0.523688)+x15*(-0.363925)+x16*(0.33187)+x17*(-0.650496)+x18*(0.540302)+x19*(-0.077399)+(0.0775052)); 
    h2 = tanh(x0*(-0.144801)+x1*(0.287882)+x2*(-0.128681)+x3*(-0.0366755)+x4*(-0.172609)+x5*(0.237665)+x6*(0.312051)+x7*(0.0828968)+x8*(-0.606608)+x9*(0.232615)+x10*(0.0766488)+x11*(-0.241326)+x12*(0.373683)+x13*(-0.599783)+x14*(-0.616046)+x15*(0.637081)+x16*(-0.0894585)+x17*(0.435109)+x18*(0.247014)+x19*(-0.0199004)+(0.175982)); 
    h3 = tanh(x0*(0.256667)+x1*(0.139757)+x2*(0.289693)+x3*(-1.38361)+x4*(0.509835)+x5*(-0.355515)+x6*(-0.932624)+x7*(-0.278856)+x8*(0.0804751)+x9*(-0.812518)+x10*(-1.09374)+x11*(-1.9503)+x12*(-0.511493)+x13*(-1.61462)+x14*(0.528775)+x15*(0.407929)+x16*(-0.710424)+x17*(1.13438)+x18*(1.88627)+x19*(0.546126)+(0.303102)); 
    h4 = tanh(x0*(0.806566)+x1*(0.0283563)+x2*(0.98972)+x3*(0.530349)+x4*(0.578177)+x5*(0.148618)+x6*(0.151451)+x7*(0.349552)+x8*(-0.333008)+x9*(0.13173)+x10*(-0.0368301)+x11*(0.496184)+x12*(0.177604)+x13*(-0.0233811)+x14*(-0.294985)+x15*(0.64679)+x16*(-1.38365)+x17*(1.17)+x18*(-0.279912)+x19*(0.774464)+(0.235735)); 
    h5 = tanh(x0*(0.735892)+x1*(-0.0419637)+x2*(0.279256)+x3*(-0.917796)+x4*(0.102242)+x5*(-0.121855)+x6*(0.259826)+x7*(0.2786)+x8*(-0.274417)+x9*(-0.209278)+x10*(0.444855)+x11*(0.135363)+x12*(-0.464288)+x13*(-0.437794)+x14*(-0.695214)+x15*(0.335294)+x16*(-0.11025)+x17*(0.984669)+x18*(-0.370277)+x19*(-0.00639592)+(-0.0204396)); 
    h6 = tanh(x0*(0.495537)+x1*(0.0400898)+x2*(0.928751)+x3*(0.876165)+x4*(-0.742458)+x5*(0.632521)+x6*(0.419975)+x7*(0.0942772)+x8*(0.242289)+x9*(-0.319068)+x10*(-0.699079)+x11*(0.331386)+x12*(0.812861)+x13*(-0.254231)+x14*(-1.18202)+x15*(0.762818)+x16*(-0.00612652)+x17*(0.260335)+x18*(0.19687)+x19*(0.487282)+(-0.0054175)); 
    h7 = tanh(x0*(1.96843)+x1*(0.349576)+x2*(0.556732)+x3*(-1.26003)+x4*(0.962306)+x5*(0.247198)+x6*(0.183592)+x7*(-0.411822)+x8*(0.141332)+x9*(-0.930295)+x10*(0.483206)+x11*(0.0287692)+x12*(-0.650346)+x13*(-1.36511)+x14*(-0.904258)+x15*(0.90441)+x16*(-0.484044)+x17*(2.12054)+x18*(0.645291)+x19*(-0.635549)+(-0.188915)); 
    h8 = tanh(x0*(0.875363)+x1*(1.18163)+x2*(0.588811)+x3*(0.921411)+x4*(-0.0756447)+x5*(0.117347)+x6*(0.0661911)+x7*(-0.114995)+x8*(-1.32102)+x9*(-0.113125)+x10*(-0.494502)+x11*(0.370095)+x12*(0.122557)+x13*(0.0879176)+x14*(-0.637299)+x15*(0.211512)+x16*(-0.776592)+x17*(-0.802778)+x18*(0.832501)+x19*(-0.123344)+(-0.00189378)); 
    h9 = tanh(x0*(-0.218678)+x1*(-0.391811)+x2*(-0.0665978)+x3*(0.383717)+x4*(-0.498351)+x5*(-0.431597)+x6*(-0.0635886)+x7*(0.0428335)+x8*(-0.504354)+x9*(0.328268)+x10*(-0.0416704)+x11*(1.32942)+x12*(-0.0516931)+x13*(1.4777)+x14*(0.530351)+x15*(-0.324544)+x16*(1.04627)+x17*(-1.17072)+x18*(0.625273)+x19*(-0.552841)+(-0.0623905)); 
}

void FNN_3_6_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(1.11777)+x1*(0.148888)+x2*(0.692665)+x3*(-0.578033)+x4*(0.832557)+x5*(-0.813536)+x6*(0.635011)+x7*(0.641513)+x8*(-1.14855)+x9*(-0.652153)+(0.208046); 
    h1 = x0*(0.955679)+x1*(0.450755)+x2*(0.65228)+x3*(-0.621491)+x4*(0.843434)+x5*(-0.613183)+x6*(0.689379)+x7*(0.672355)+x8*(-1.20636)+x9*(-0.649245)+(0.19374); 
    h2 = x0*(0.955727)+x1*(0.457387)+x2*(0.661919)+x3*(-0.618758)+x4*(0.847299)+x5*(-0.626466)+x6*(0.685755)+x7*(0.671898)+x8*(-1.20887)+x9*(-0.654547)+(0.184288); 
}

void FNN_3_6(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_6_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_6_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_6_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_6(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_4(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x6 < lowBound ) {
        ModelKDTree_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x6 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x6 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void ModelKDTree_2(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x7 < lowBound ) {
        ModelKDTree_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x7 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x7 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void FNN_3_7_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.0320182)+x1*(-0.0876655)+x2*(-0.196538)+x3*(-0.0233527)+x4*(1.51935)+x5*(-0.182289)+x6*(0.264071)+x7*(-0.146662)+x8*(0.346444)+x9*(0.189901)+x10*(-0.167835)+x11*(-0.24533)+x12*(-0.18717)+x13*(0.0566531)+x14*(-0.0121487)+x15*(-0.335607)+x16*(-0.210212)+(0.112747)); 
    h1 = tanh(x0*(0.102757)+x1*(-0.1605)+x2*(-0.178525)+x3*(0.363137)+x4*(0.953577)+x5*(0.211799)+x6*(-0.492585)+x7*(-1.79875)+x8*(0.279412)+x9*(0.792026)+x10*(0.144577)+x11*(-0.0427912)+x12*(-0.389263)+x13*(-0.262857)+x14*(-0.188232)+x15*(-0.356711)+x16*(-0.054777)+(0.358311)); 
    h2 = tanh(x0*(-0.598564)+x1*(-0.42153)+x2*(0.0656925)+x3*(-0.219398)+x4*(0.0660328)+x5*(0.191613)+x6*(-0.188412)+x7*(-0.488251)+x8*(-0.463884)+x9*(0.0785832)+x10*(-0.660379)+x11*(-0.0496442)+x12*(0.0293558)+x13*(0.29701)+x14*(-0.232474)+x15*(-0.257219)+x16*(0.294565)+(0.0834964)); 
    h3 = tanh(x0*(0.254104)+x1*(0.360767)+x2*(0.144561)+x3*(1.51856)+x4*(1.23663)+x5*(0.280528)+x6*(-2.96542)+x7*(-2.7915)+x8*(-0.0806324)+x9*(-0.612754)+x10*(0.26984)+x11*(-0.130589)+x12*(-0.0850258)+x13*(0.306462)+x14*(0.395607)+x15*(-0.039032)+x16*(-0.342154)+(0.940989)); 
    h4 = tanh(x0*(-0.317827)+x1*(-0.499049)+x2*(-0.667329)+x3*(-0.568637)+x4*(0.43318)+x5*(-0.10061)+x6*(1.18467)+x7*(-1.09508)+x8*(0.091114)+x9*(0.246179)+x10*(-0.0420747)+x11*(0.667513)+x12*(0.395627)+x13*(-0.388582)+x14*(-0.0684183)+x15*(-0.301584)+x16*(0.174634)+(-0.358691)); 
    h5 = tanh(x0*(0.414043)+x1*(0.458354)+x2*(0.308709)+x3*(0.145371)+x4*(0.350167)+x5*(-0.237098)+x6*(0.248169)+x7*(-0.474229)+x8*(0.104968)+x9*(-0.343728)+x10*(-0.551974)+x11*(-0.617102)+x12*(0.0650494)+x13*(0.118407)+x14*(-0.104089)+x15*(0.245548)+x16*(0.205123)+(-0.403558)); 
    h6 = tanh(x0*(0.0953316)+x1*(-0.202912)+x2*(-0.270601)+x3*(-0.133394)+x4*(-0.147497)+x5*(0.366661)+x6*(-0.114109)+x7*(-0.74565)+x8*(-0.491196)+x9*(-0.145129)+x10*(0.155739)+x11*(0.0518877)+x12*(-0.220019)+x13*(0.0140573)+x14*(0.36036)+x15*(0.0684528)+x16*(0.176445)+(0.251524)); 
    h7 = tanh(x0*(-0.315952)+x1*(-0.227298)+x2*(0.104397)+x3*(-0.18707)+x4*(0.0390057)+x5*(-0.0208166)+x6*(-0.648029)+x7*(0.459099)+x8*(0.118765)+x9*(-0.153618)+x10*(0.272831)+x11*(0.0293219)+x12*(0.0865715)+x13*(0.194135)+x14*(0.0488646)+x15*(-0.35685)+x16*(0.204493)+(0.160147)); 
    h8 = tanh(x0*(-0.275372)+x1*(-0.554299)+x2*(-0.284318)+x3*(0.320025)+x4*(0.188389)+x5*(-0.221141)+x6*(0.483092)+x7*(-0.45377)+x8*(0.351136)+x9*(0.289176)+x10*(-0.99137)+x11*(0.0884692)+x12*(-0.146578)+x13*(-0.343978)+x14*(0.225134)+x15*(-0.388146)+x16*(0.0819555)+(-0.0154584)); 
    h9 = tanh(x0*(0.434787)+x1*(0.0425674)+x2*(0.635613)+x3*(0.11989)+x4*(0.268139)+x5*(-0.317451)+x6*(-0.216215)+x7*(-0.115505)+x8*(-0.149628)+x9*(0.0107901)+x10*(-0.303933)+x11*(0.166751)+x12*(-0.171565)+x13*(-0.304934)+x14*(-0.207344)+x15*(-0.247229)+x16*(-0.13289)+(0.341873)); 
    h10 = tanh(x0*(-0.0381473)+x1*(0.169144)+x2*(-0.349111)+x3*(0.675719)+x4*(0.0882815)+x5*(0.320519)+x6*(-1.76332)+x7*(-0.469985)+x8*(0.53439)+x9*(-0.448173)+x10*(0.489392)+x11*(-0.241195)+x12*(-0.283588)+x13*(0.0646971)+x14*(-0.123082)+x15*(0.18714)+x16*(-0.301701)+(0.50831)); 
    h11 = tanh(x0*(-0.237478)+x1*(0.110027)+x2*(-0.204685)+x3*(0.54795)+x4*(-0.291231)+x5*(-0.205077)+x6*(1.38613)+x7*(0.920067)+x8*(0.38459)+x9*(0.44572)+x10*(-0.077647)+x11*(-0.253059)+x12*(-0.0815933)+x13*(-0.0371286)+x14*(-0.277131)+x15*(0.30313)+x16*(-0.0179966)+(-0.587761)); 
    h12 = tanh(x0*(-0.0162214)+x1*(-0.138192)+x2*(0.129788)+x3*(0.315114)+x4*(0.769451)+x5*(-0.156191)+x6*(0.0257796)+x7*(-0.261847)+x8*(-0.697829)+x9*(0.210844)+x10*(0.109636)+x11*(-0.500063)+x12*(0.176554)+x13*(0.294786)+x14*(-0.272396)+x15*(-0.310509)+x16*(-0.0401962)+(-0.391236)); 
    h13 = tanh(x0*(-0.177307)+x1*(-0.570949)+x2*(-0.693148)+x3*(-0.81795)+x4*(-2.76951)+x5*(-0.187601)+x6*(1.03193)+x7*(4.10455)+x8*(-0.81314)+x9*(-0.338272)+x10*(-0.00950498)+x11*(-0.0307886)+x12*(0.312566)+x13*(-0.167035)+x14*(0.184602)+x15*(-0.340586)+x16*(-0.0753386)+(-0.0891507)); 
    h14 = tanh(x0*(-0.523879)+x1*(-0.276278)+x2*(-0.0271632)+x3*(0.928786)+x4*(-0.134455)+x5*(-0.196948)+x6*(-1.26867)+x7*(1.16896)+x8*(-0.460398)+x9*(-0.711253)+x10*(0.577771)+x11*(-0.203559)+x12*(0.178323)+x13*(0.00679642)+x14*(0.291684)+x15*(-0.22987)+x16*(-0.286349)+(0.439303)); 
    h15 = tanh(x0*(0.48749)+x1*(0.543038)+x2*(0.408925)+x3*(-0.0761953)+x4*(0.164645)+x5*(0.0761674)+x6*(-0.274765)+x7*(1.60877)+x8*(-0.608065)+x9*(-0.032593)+x10*(0.232606)+x11*(0.278147)+x12*(-0.251796)+x13*(-0.0760672)+x14*(-0.222366)+x15*(-0.181633)+x16*(0.19637)+(0.313129)); 
    h16 = tanh(x0*(-0.541549)+x1*(-0.696309)+x2*(-0.0190485)+x3*(0.413278)+x4*(2.64539)+x5*(0.136193)+x6*(-0.678847)+x7*(-4.32568)+x8*(0.0495433)+x9*(-0.00233889)+x10*(-0.293468)+x11*(-0.165193)+x12*(0.212691)+x13*(-0.10052)+x14*(0.1045)+x15*(0.0446427)+x16*(0.35167)+(0.365874)); 
    h17 = tanh(x0*(-0.867794)+x1*(-1.51423)+x2*(-1.37265)+x3*(1.69045)+x4*(-0.528621)+x5*(-0.115358)+x6*(0.519012)+x7*(-0.930033)+x8*(0.327576)+x9*(0.155086)+x10*(0.281759)+x11*(-0.375048)+x12*(0.025439)+x13*(0.0932276)+x14*(0.200749)+x15*(0.363124)+x16*(-0.142512)+(-0.103803)); 
    h18 = tanh(x0*(0.457153)+x1*(0.749818)+x2*(0.541523)+x3*(0.294768)+x4*(-0.16715)+x5*(0.112133)+x6*(-0.236259)+x7*(-0.620996)+x8*(0.104053)+x9*(0.0580859)+x10*(0.20103)+x11*(-0.171689)+x12*(-0.00402501)+x13*(-0.144476)+x14*(-0.06485)+x15*(0.160897)+x16*(-0.219846)+(0.0567043)); 
    h19 = tanh(x0*(-0.061997)+x1*(0.0934796)+x2*(-0.319229)+x3*(0.559001)+x4*(0.308739)+x5*(0.178044)+x6*(-0.837028)+x7*(0.165648)+x8*(0.246487)+x9*(-0.182289)+x10*(0.148118)+x11*(0.0785056)+x12*(0.322878)+x13*(0.320025)+x14*(0.151636)+x15*(-0.306378)+x16*(-0.123607)+(0.0811355)); 
}

void FNN_3_7_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(0.33505)+x1*(0.550455)+x2*(-0.493398)+x3*(-0.941657)+x4*(0.221087)+x5*(0.648588)+x6*(-0.229657)+x7*(0.00117012)+x8*(-0.266123)+x9*(-0.0225452)+x10*(0.0424316)+x11*(0.459736)+x12*(-0.260304)+x13*(-0.0215838)+x14*(-0.841031)+x15*(-0.665772)+x16*(0.0230437)+x17*(0.172612)+x18*(0.0813199)+x19*(0.144577)+(-0.0183915)); 
    h1 = tanh(x0*(0.75507)+x1*(0.593403)+x2*(0.29842)+x3*(-0.230855)+x4*(0.58525)+x5*(-0.20004)+x6*(-0.430143)+x7*(-0.268478)+x8*(-0.627014)+x9*(-0.325002)+x10*(-1.58812)+x11*(0.768548)+x12*(-0.197602)+x13*(1.00909)+x14*(-1.36083)+x15*(-0.114174)+x16*(1.11682)+x17*(-0.27665)+x18*(0.400053)+x19*(-0.846379)+(-0.0660852)); 
    h2 = tanh(x0*(-0.148086)+x1*(-0.183933)+x2*(-0.485611)+x3*(1.13182)+x4*(-1.144)+x5*(-0.58889)+x6*(0.154819)+x7*(0.124877)+x8*(-0.349696)+x9*(-0.109958)+x10*(-1.13155)+x11*(0.25249)+x12*(-0.19811)+x13*(1.16904)+x14*(0.509509)+x15*(0.190777)+x16*(0.910414)+x17*(0.0354063)+x18*(-0.452196)+x19*(0.0335202)+(0.122611)); 
    h3 = tanh(x0*(-0.353949)+x1*(0.456207)+x2*(-0.335062)+x3*(-0.581438)+x4*(-0.741387)+x5*(-0.0846817)+x6*(-0.243959)+x7*(0.138871)+x8*(-0.0394278)+x9*(-0.398801)+x10*(0.733927)+x11*(-0.572128)+x12*(-0.554276)+x13*(-0.164627)+x14*(0.187384)+x15*(-0.224315)+x16*(-0.31494)+x17*(0.318508)+x18*(-0.669337)+x19*(0.702672)+(0.186739)); 
    h4 = tanh(x0*(-0.117042)+x1*(-0.924031)+x2*(-0.125276)+x3*(0.813727)+x4*(0.176774)+x5*(0.296906)+x6*(-0.542706)+x7*(-0.130665)+x8*(-0.538392)+x9*(-0.482655)+x10*(0.743105)+x11*(0.288651)+x12*(0.292047)+x13*(1.59682)+x14*(0.517513)+x15*(1.0604)+x16*(0.160749)+x17*(0.167107)+x18*(0.257457)+x19*(0.127502)+(-0.261654)); 
    h5 = tanh(x0*(0.0653024)+x1*(-0.159992)+x2*(0.159701)+x3*(-0.474807)+x4*(-0.838355)+x5*(-0.67085)+x6*(-0.383602)+x7*(0.269898)+x8*(-0.0835513)+x9*(-0.29705)+x10*(1.2203)+x11*(-0.188603)+x12*(-0.333626)+x13*(0.0663204)+x14*(0.881357)+x15*(0.989675)+x16*(-0.886347)+x17*(0.439571)+x18*(-0.141775)+x19*(-0.686082)+(0.0284917)); 
    h6 = tanh(x0*(0.156716)+x1*(-0.416164)+x2*(0.0618659)+x3*(0.196309)+x4*(-1.39098)+x5*(-0.0398102)+x6*(0.109685)+x7*(-0.575954)+x8*(0.120392)+x9*(0.755964)+x10*(1.92537)+x11*(0.138437)+x12*(-0.147995)+x13*(-0.491682)+x14*(1.29102)+x15*(-0.734384)+x16*(-0.877446)+x17*(0.710417)+x18*(0.641429)+x19*(0.668106)+(0.115832)); 
    h7 = tanh(x0*(0.284399)+x1*(-0.18509)+x2*(-0.0157067)+x3*(0.562438)+x4*(0.598054)+x5*(-0.00519068)+x6*(-0.171222)+x7*(-0.917522)+x8*(-0.127701)+x9*(0.0147596)+x10*(-1.2754)+x11*(-0.0472166)+x12*(0.643837)+x13*(0.137331)+x14*(-0.36643)+x15*(-0.120389)+x16*(-0.0349419)+x17*(-0.195296)+x18*(0.309332)+x19*(-0.192532)+(-0.180329)); 
    h8 = tanh(x0*(-0.342791)+x1*(-1.21522)+x2*(-0.467536)+x3*(-0.444522)+x4*(-0.0477664)+x5*(-0.8196)+x6*(-0.325487)+x7*(0.556402)+x8*(-1.13652)+x9*(0.177782)+x10*(-0.00883281)+x11*(-0.116889)+x12*(0.530157)+x13*(0.667798)+x14*(-0.169126)+x15*(0.877278)+x16*(-0.942345)+x17*(0.789983)+x18*(0.738258)+x19*(0.116011)+(0.0440825)); 
    h9 = tanh(x0*(0.0296757)+x1*(0.33172)+x2*(-0.0120692)+x3*(0.235865)+x4*(-1.5352)+x5*(0.43677)+x6*(0.208321)+x7*(0.167505)+x8*(-0.370082)+x9*(0.642048)+x10*(1.34515)+x11*(0.175309)+x12*(-0.254136)+x13*(0.312991)+x14*(1.05757)+x15*(0.0283371)+x16*(0.239247)+x17*(0.61494)+x18*(0.208348)+x19*(-0.0456613)+(0.0922172)); 
}

void FNN_3_7_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(0.387581)+x1*(-0.850667)+x2*(0.938847)+x3*(0.384734)+x4*(-0.584653)+x5*(-0.679334)+x6*(0.470795)+x7*(-0.0519208)+x8*(0.355262)+x9*(-1.27379)+(0.0177348); 
    h1 = x0*(0.389563)+x1*(-0.662592)+x2*(0.635317)+x3*(-0.484523)+x4*(-0.429623)+x5*(-0.378913)+x6*(0.392535)+x7*(-0.740723)+x8*(0.271031)+x9*(-1.03324)+(0.10262); 
    h2 = x0*(0.269583)+x1*(-0.80945)+x2*(0.875878)+x3*(0.0744176)+x4*(-0.546364)+x5*(-0.63426)+x6*(0.503799)+x7*(-0.327127)+x8*(0.351685)+x9*(-1.30189)+(0.0849948); 
}

void FNN_3_7(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_7_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_7_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_7_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_7(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_7(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_1(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x5 < lowBound ) {
        ModelKDTree_2(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x5 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_2(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_7(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x5 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_7(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void FNN_3_10_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(0.0981119)+x1*(0.160606)+x2*(0.190752)+x3*(-0.257476)+x4*(0.246386)+x5*(-0.378935)+x6*(-0.915789)+x7*(0.0636019)+x8*(-1.38652)+x9*(-0.258654)+x10*(0.0569102)+x11*(-3.10942)+x12*(0.377387)+x13*(-0.0359279)+x14*(0.100212)+x15*(0.334474)+x16*(0.227016)+(0.0103681)); 
    h1 = tanh(x0*(-0.520472)+x1*(-0.0670346)+x2*(0.132117)+x3*(0.818348)+x4*(0.457694)+x5*(-0.99593)+x6*(1.36377)+x7*(0.627624)+x8*(1.97528)+x9*(-0.365031)+x10*(-0.197823)+x11*(0.822348)+x12*(-0.0413485)+x13*(-0.275922)+x14*(0.0245942)+x15*(0.129171)+x16*(-0.341255)+(0.00916328)); 
    h2 = tanh(x0*(-0.517098)+x1*(-0.490635)+x2*(-0.867548)+x3*(-0.206991)+x4*(-1.59984)+x5*(1.44151)+x6*(-0.600029)+x7*(-2.29552)+x8*(-1.95859)+x9*(0.484246)+x10*(0.704159)+x11*(4.62974)+x12*(-0.137747)+x13*(0.371584)+x14*(0.378711)+x15*(-0.0410023)+x16*(0.355784)+(0.565043)); 
    h3 = tanh(x0*(0.0446795)+x1*(0.291446)+x2*(0.281198)+x3*(-0.294972)+x4*(1.4331)+x5*(-1.6017)+x6*(0.593931)+x7*(-3.21674)+x8*(0.541957)+x9*(-0.19021)+x10*(0.470031)+x11*(-2.8748)+x12*(-0.205631)+x13*(-0.318009)+x14*(0.184562)+x15*(0.197773)+x16*(-0.207626)+(-0.0899593)); 
    h4 = tanh(x0*(-0.391758)+x1*(-0.198376)+x2*(-0.913818)+x3*(0.327485)+x4*(0.258303)+x5*(0.03886)+x6*(-0.416131)+x7*(-0.190664)+x8*(-0.951484)+x9*(0.263771)+x10*(-0.0994741)+x11*(1.61436)+x12*(0.27817)+x13*(0.332441)+x14*(0.23284)+x15*(0.281416)+x16*(0.258875)+(0.110714)); 
    h5 = tanh(x0*(-0.0902223)+x1*(-0.0441027)+x2*(0.199852)+x3*(-0.764165)+x4*(-0.481937)+x5*(-6.49443)+x6*(3.03702)+x7*(0.618585)+x8*(0.589052)+x9*(-0.571572)+x10*(-0.111445)+x11*(-6.74307)+x12*(-0.250788)+x13*(-0.236925)+x14*(0.321935)+x15*(-0.085082)+x16*(-0.38751)+(-0.350907)); 
    h6 = tanh(x0*(-0.111423)+x1*(-0.476627)+x2*(-0.358577)+x3*(1.48091)+x4*(-0.246531)+x5*(1.2491)+x6*(-1.64483)+x7*(-0.272831)+x8*(3.35294)+x9*(-0.606584)+x10*(-0.166949)+x11*(4.07124)+x12*(-0.104186)+x13*(-0.235984)+x14*(0.337803)+x15*(-0.333326)+x16*(-0.272108)+(-0.559375)); 
    h7 = tanh(x0*(0.174329)+x1*(0.303776)+x2*(0.417896)+x3*(0.261509)+x4*(0.296635)+x5*(-0.64493)+x6*(0.218202)+x7*(-0.256048)+x8*(-1.13503)+x9*(-0.265821)+x10*(-0.262611)+x11*(-2.51758)+x12*(-0.387723)+x13*(-0.294262)+x14*(-0.374724)+x15*(0.0502563)+x16*(0.26003)+(-0.158055)); 
    h8 = tanh(x0*(-8.75263)+x1*(-8.55692)+x2*(-8.48844)+x3*(-0.762553)+x4*(0.146793)+x5*(4.0945)+x6*(2.63086)+x7*(-0.507177)+x8*(1.73785)+x9*(-0.959647)+x10*(-0.489967)+x11*(3.29746)+x12*(0.205039)+x13*(0.0409889)+x14*(-0.284136)+x15*(0.117678)+x16*(-0.0737868)+(-0.858813)); 
    h9 = tanh(x0*(1.20834)+x1*(0.732951)+x2*(0.941436)+x3*(-0.487357)+x4*(-0.509753)+x5*(-2.56659)+x6*(1.13721)+x7*(0.727198)+x8*(-1.42485)+x9*(0.0750412)+x10*(0.188356)+x11*(-4.39778)+x12*(-0.332633)+x13*(-0.387184)+x14*(-0.274659)+x15*(-0.365367)+x16*(0.145979)+(0.344715)); 
    h10 = tanh(x0*(0.0715183)+x1*(0.0361033)+x2*(-0.283745)+x3*(0.343312)+x4*(-0.305005)+x5*(-0.206645)+x6*(-0.772554)+x7*(-0.139444)+x8*(0.226484)+x9*(-0.0473143)+x10*(-0.220783)+x11*(0.86321)+x12*(0.135032)+x13*(-0.316753)+x14*(0.332803)+x15*(-0.129188)+x16*(-0.363013)+(-0.0950886)); 
    h11 = tanh(x0*(-0.243446)+x1*(0.0837427)+x2*(-0.457753)+x3*(0.227257)+x4*(-0.351848)+x5*(-4.0255)+x6*(-0.271584)+x7*(0.0237329)+x8*(0.422618)+x9*(0.381252)+x10*(0.369289)+x11*(-6.56728)+x12*(-0.33204)+x13*(0.0995334)+x14*(0.334327)+x15*(0.256953)+x16*(0.0814505)+(0.111134)); 
    h12 = tanh(x0*(-0.049914)+x1*(-0.235533)+x2*(-0.50981)+x3*(0.377234)+x4*(0.0241138)+x5*(-3.68039)+x6*(-0.123831)+x7*(0.187385)+x8*(-6.52094)+x9*(-0.032589)+x10*(-0.302808)+x11*(-1.96374)+x12*(-0.134391)+x13*(0.376857)+x14*(-0.0179202)+x15*(-0.170605)+x16*(0.371718)+(-0.0932679)); 
    h13 = tanh(x0*(-0.239839)+x1*(-0.689395)+x2*(-0.323445)+x3*(0.69499)+x4*(0.698781)+x5*(-0.570418)+x6*(0.64204)+x7*(1.01372)+x8*(1.1318)+x9*(0.109694)+x10*(-0.30889)+x11*(-2.08748)+x12*(0.0216834)+x13*(-0.305539)+x14*(0.0112581)+x15*(0.395158)+x16*(0.00855917)+(0.000978482)); 
    h14 = tanh(x0*(0.0915167)+x1*(0.154843)+x2*(0.212195)+x3*(0.724566)+x4*(-0.984646)+x5*(-0.456763)+x6*(-1.35532)+x7*(1.56078)+x8*(-0.0965869)+x9*(0.214582)+x10*(0.203965)+x11*(-1.47485)+x12*(-0.209136)+x13*(0.00938609)+x14*(0.16907)+x15*(0.145481)+x16*(0.079078)+(0.164703)); 
    h15 = tanh(x0*(0.453893)+x1*(0.386894)+x2*(0.183369)+x3*(0.618036)+x4*(-0.57163)+x5*(2.98054)+x6*(0.922077)+x7*(-0.0219073)+x8*(0.553814)+x9*(0.403132)+x10*(-0.170259)+x11*(4.32072)+x12*(0.0270349)+x13*(-0.180573)+x14*(-0.279574)+x15*(0.257129)+x16*(-0.3009)+(0.231538)); 
    h16 = tanh(x0*(-0.578963)+x1*(0.124103)+x2*(-0.177956)+x3*(-0.600587)+x4*(0.199742)+x5*(3.40617)+x6*(-3.27063)+x7*(0.124232)+x8*(-1.3788)+x9*(0.218192)+x10*(0.110935)+x11*(6.62245)+x12*(0.0785826)+x13*(0.334074)+x14*(-0.287819)+x15*(0.00171417)+x16*(0.222045)+(0.302759)); 
    h17 = tanh(x0*(-0.158316)+x1*(0.353552)+x2*(-0.151995)+x3*(0.13936)+x4*(1.01129)+x5*(-0.322615)+x6*(0.197499)+x7*(1.03747)+x8*(0.115693)+x9*(-0.137992)+x10*(-0.198225)+x11*(-0.615913)+x12*(0.346354)+x13*(0.352671)+x14*(0.166043)+x15*(-0.319423)+x16*(-0.00365719)+(-0.233001)); 
    h18 = tanh(x0*(0.322722)+x1*(0.0299455)+x2*(0.135027)+x3*(0.157539)+x4*(-0.128548)+x5*(4.79897)+x6*(0.26152)+x7*(0.402732)+x8*(-3.11555)+x9*(-0.0379551)+x10*(-0.210805)+x11*(9.04883)+x12*(0.302861)+x13*(0.287957)+x14*(-0.114519)+x15*(-0.0120181)+x16*(-0.0718458)+(-0.343467)); 
    h19 = tanh(x0*(-0.0331761)+x1*(0.0926876)+x2*(0.175565)+x3*(0.389985)+x4*(1.61614)+x5*(0.0315609)+x6*(-0.640911)+x7*(-2.36171)+x8*(-0.608676)+x9*(0.827332)+x10*(0.51961)+x11*(-1.77375)+x12*(0.196341)+x13*(0.352716)+x14*(-0.0329221)+x15*(0.363772)+x16*(0.34034)+(0.485797)); 
}

void FNN_3_10_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.0941137)+x1*(1.21673)+x2*(-0.957122)+x3*(1.61601)+x4*(-0.266223)+x5*(1.63034)+x6*(0.259458)+x7*(0.0415376)+x8*(1.34822)+x9*(0.371823)+x10*(-0.780528)+x11*(0.0101873)+x12*(-0.109053)+x13*(0.522667)+x14*(-0.486264)+x15*(-0.259541)+x16*(-1.39821)+x17*(-0.277308)+x18*(-0.553506)+x19*(-1.43755)+(-0.0469252)); 
    h1 = tanh(x0*(0.120228)+x1*(0.151927)+x2*(-0.277747)+x3*(-0.0536785)+x4*(-0.178741)+x5*(0.14138)+x6*(-0.0311385)+x7*(-0.258759)+x8*(0.58071)+x9*(0.406502)+x10*(0.351719)+x11*(0.179305)+x12*(-0.627881)+x13*(-0.141768)+x14*(0.245664)+x15*(-0.232491)+x16*(-0.449935)+x17*(-0.0272036)+x18*(-0.29638)+x19*(0.199623)+(0.123329)); 
    h2 = tanh(x0*(-0.569313)+x1*(-0.274695)+x2*(0.674181)+x3*(-0.241268)+x4*(-0.124252)+x5*(0.7023)+x6*(-0.628447)+x7*(0.722067)+x8*(-1.05144)+x9*(1.22482)+x10*(0.298872)+x11*(-0.57757)+x12*(0.676304)+x13*(-0.493847)+x14*(-0.867721)+x15*(-0.612564)+x16*(-0.412185)+x17*(-0.0616155)+x18*(1.01178)+x19*(0.216607)+(-0.295302)); 
    h3 = tanh(x0*(-0.00127773)+x1*(0.198912)+x2*(-0.213373)+x3*(0.419274)+x4*(-0.288808)+x5*(0.561886)+x6*(0.15915)+x7*(0.447857)+x8*(0.316439)+x9*(0.234532)+x10*(-0.508182)+x11*(0.704955)+x12*(-0.358873)+x13*(0.19104)+x14*(-0.322447)+x15*(0.035374)+x16*(-0.332135)+x17*(0.582025)+x18*(-0.929991)+x19*(-0.0040306)+(0.0434768)); 
    h4 = tanh(x0*(0.834168)+x1*(-0.729906)+x2*(-0.0576429)+x3*(0.0912245)+x4*(0.0164033)+x5*(-0.59171)+x6*(-0.236435)+x7*(0.0230475)+x8*(-0.385333)+x9*(-0.00822285)+x10*(0.300433)+x11*(0.651513)+x12*(0.875391)+x13*(-0.269071)+x14*(0.527155)+x15*(-0.22837)+x16*(-0.0794678)+x17*(0.363722)+x18*(0.215599)+x19*(-0.587408)+(-0.327764)); 
    h5 = tanh(x0*(1.08562)+x1*(-0.781509)+x2*(0.619271)+x3*(0.110161)+x4*(0.866611)+x5*(1.63082)+x6*(-1.06301)+x7*(0.832742)+x8*(-1.0345)+x9*(0.48389)+x10*(0.0997546)+x11*(-0.115352)+x12*(1.00845)+x13*(0.101599)+x14*(0.816295)+x15*(-0.689352)+x16*(-0.3234)+x17*(-0.459429)+x18*(0.591302)+x19*(0.93586)+(-0.181407)); 
    h6 = tanh(x0*(0.0976578)+x1*(-0.993861)+x2*(0.32069)+x3*(-1.26234)+x4*(0.37236)+x5*(-1.86723)+x6*(0.915823)+x7*(0.00213158)+x8*(-1.53634)+x9*(-0.166351)+x10*(-0.49953)+x11*(-0.793026)+x12*(0.53058)+x13*(-0.467191)+x14*(0.809991)+x15*(-0.734897)+x16*(1.3095)+x17*(-0.727109)+x18*(0.31054)+x19*(0.882072)+(-0.0595818)); 
    h7 = tanh(x0*(0.691796)+x1*(0.117833)+x2*(-1.06917)+x3*(0.622339)+x4*(-0.64682)+x5*(1.93404)+x6*(-0.42208)+x7*(-0.113668)+x8*(-0.667274)+x9*(0.622097)+x10*(-0.513935)+x11*(0.270486)+x12*(-1.70698)+x13*(-0.234837)+x14*(-0.464711)+x15*(-0.146756)+x16*(-0.718726)+x17*(-0.0375333)+x18*(-0.517422)+x19*(0.109095)+(0.10516)); 
    h8 = tanh(x0*(-0.998447)+x1*(1.54784)+x2*(-1.29743)+x3*(1.87994)+x4*(-0.498052)+x5*(1.29464)+x6*(1.22454)+x7*(-1.19267)+x8*(0.695117)+x9*(-0.89091)+x10*(-0.179537)+x11*(0.384136)+x12*(-1.55369)+x13*(0.169541)+x14*(-1.68484)+x15*(0.726628)+x16*(-1.89926)+x17*(0.919106)+x18*(0.165519)+x19*(-2.39922)+(0.289949)); 
    h9 = tanh(x0*(0.540966)+x1*(0.332152)+x2*(-0.358574)+x3*(0.570374)+x4*(-0.0564235)+x5*(0.743533)+x6*(0.664424)+x7*(0.441993)+x8*(0.421755)+x9*(0.892989)+x10*(-0.700901)+x11*(0.247831)+x12*(-0.404462)+x13*(0.453312)+x14*(-0.343223)+x15*(0.0717955)+x16*(-0.274505)+x17*(0.0969621)+x18*(-0.820882)+x19*(-0.00780904)+(0.16392)); 
}

void FNN_3_10_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-1.05734)+x1*(-0.260813)+x2*(0.71435)+x3*(0.949827)+x4*(0.77853)+x5*(-0.953844)+x6*(-1.28025)+x7*(-0.621261)+x8*(1.06629)+x9*(-0.303688)+(-0.134491); 
    h1 = x0*(-1.05991)+x1*(-0.272241)+x2*(0.689945)+x3*(0.640327)+x4*(0.759729)+x5*(-0.929071)+x6*(-1.27038)+x7*(-0.545172)+x8*(1.07565)+x9*(-0.0520318)+(-0.1555); 
    h2 = x0*(-1.03994)+x1*(-0.282877)+x2*(0.684108)+x3*(0.111165)+x4*(0.825021)+x5*(-0.938582)+x6*(-1.12052)+x7*(-0.366486)+x8*(1.12076)+x9*(0.433041)+(-0.143299); 
}

void FNN_3_10(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_10_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_10_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_10_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_10(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_10(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_3_12_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-1.03272)+x1*(-0.91507)+x2*(-0.559596)+x3*(1.8591)+x4*(-0.564055)+x5*(1.92153)+x6*(0.124034)+x7*(0.399847)+x8*(1.80931)+x9*(0.193252)+x10*(-0.193843)+x11*(1.34374)+x12*(0.151201)+x13*(0.027835)+x14*(-0.401085)+x15*(-0.368758)+x16*(-0.226116)+(0.267368)); 
    h1 = tanh(x0*(-0.853997)+x1*(-0.815669)+x2*(-0.906851)+x3*(0.0868032)+x4*(0.230682)+x5*(-1.55103)+x6*(-0.4697)+x7*(-0.159775)+x8*(-0.988212)+x9*(-0.0609249)+x10*(-0.00899505)+x11*(-2.11531)+x12*(-0.223338)+x13*(-0.00224683)+x14*(0.266504)+x15*(-0.00662908)+x16*(-0.0604149)+(-0.0663167)); 
    h2 = tanh(x0*(0.30344)+x1*(0.331032)+x2*(0.0494532)+x3*(0.224054)+x4*(0.0349442)+x5*(0.573381)+x6*(-0.449504)+x7*(0.0450687)+x8*(0.562233)+x9*(0.264117)+x10*(0.231215)+x11*(0.282488)+x12*(0.334587)+x13*(-0.164415)+x14*(0.277152)+x15*(-0.244004)+x16*(0.142572)+(0.152135)); 
    h3 = tanh(x0*(0.095994)+x1*(-0.283555)+x2*(-0.366942)+x3*(-0.253196)+x4*(-0.0504077)+x5*(0.00207391)+x6*(0.250036)+x7*(0.325457)+x8*(-0.0351748)+x9*(0.259184)+x10*(0.330876)+x11*(-0.491991)+x12*(0.11659)+x13*(-0.257197)+x14*(-0.131015)+x15*(-0.0637012)+x16*(-0.281309)+(0.103189)); 
    h4 = tanh(x0*(0.493718)+x1*(0.160088)+x2*(0.661078)+x3*(0.36736)+x4*(-0.309496)+x5*(-2.37506)+x6*(-0.347246)+x7*(-0.692295)+x8*(-1.52758)+x9*(-0.115016)+x10*(0.445099)+x11*(-3.33938)+x12*(-0.22416)+x13*(0.211375)+x14*(0.377027)+x15*(-0.213863)+x16*(0.373088)+(0.155857)); 
    h5 = tanh(x0*(2.42092)+x1*(2.68283)+x2*(2.52618)+x3*(0.0408133)+x4*(1.89835)+x5*(-0.548134)+x6*(0.288282)+x7*(-3.49228)+x8*(-1.03013)+x9*(0.799725)+x10*(0.675134)+x11*(-1.12468)+x12*(-0.272056)+x13*(-0.267968)+x14*(-0.15694)+x15*(0.0757695)+x16*(0.389041)+(0.892881)); 
    h6 = tanh(x0*(0.348693)+x1*(0.361933)+x2*(0.0193642)+x3*(-0.713413)+x4*(1.39423)+x5*(0.291042)+x6*(-0.786521)+x7*(0.872986)+x8*(-0.0270654)+x9*(0.476204)+x10*(0.346628)+x11*(1.14139)+x12*(0.192217)+x13*(-0.390293)+x14*(-0.303332)+x15*(-0.132172)+x16*(-0.140409)+(0.194555)); 
    h7 = tanh(x0*(0.323541)+x1*(0.604981)+x2*(0.246338)+x3*(0.706605)+x4*(1.0222)+x5*(-0.00285497)+x6*(0.177222)+x7*(0.094115)+x8*(1.53942)+x9*(0.249046)+x10*(-0.0849822)+x11*(0.572771)+x12*(0.160175)+x13*(-0.291414)+x14*(0.2242)+x15*(-0.0128611)+x16*(-0.350563)+(0.185188)); 
    h8 = tanh(x0*(-0.659671)+x1*(-0.596098)+x2*(-0.65573)+x3*(0.0491954)+x4*(0.0850293)+x5*(-1.64263)+x6*(0.0414263)+x7*(-0.452816)+x8*(-7.29325)+x9*(-0.264866)+x10*(0.0319365)+x11*(-3.20962)+x12*(0.0055486)+x13*(0.164483)+x14*(-0.0197875)+x15*(0.0727683)+x16*(-0.197686)+(-0.0585617)); 
    h9 = tanh(x0*(-0.169358)+x1*(-0.172754)+x2*(-0.0692528)+x3*(0.00552848)+x4*(0.534737)+x5*(0.103888)+x6*(0.207083)+x7*(-0.0778314)+x8*(0.0771704)+x9*(0.2379)+x10*(-0.251959)+x11*(-0.344975)+x12*(0.360178)+x13*(-0.0723029)+x14*(-0.198994)+x15*(0.114871)+x16*(0.0755773)+(-0.191541)); 
    h10 = tanh(x0*(-0.143656)+x1*(0.263714)+x2*(0.0205896)+x3*(0.658514)+x4*(0.316618)+x5*(1.30662)+x6*(0.0289835)+x7*(-0.229966)+x8*(0.601986)+x9*(-0.56143)+x10*(-0.109715)+x11*(1.21783)+x12*(-0.221608)+x13*(-0.0280673)+x14*(0.279)+x15*(0.119532)+x16*(0.0875742)+(-0.232389)); 
    h11 = tanh(x0*(-0.11839)+x1*(-0.307852)+x2*(-0.700799)+x3*(-0.228482)+x4*(0.095627)+x5*(-0.112537)+x6*(0.988381)+x7*(-0.288862)+x8*(1.50627)+x9*(0.0734758)+x10*(-0.729392)+x11*(0.906057)+x12*(-0.369032)+x13*(0.00679404)+x14*(-0.210268)+x15*(0.38566)+x16*(0.168823)+(-0.302451)); 
    h12 = tanh(x0*(-0.545056)+x1*(-0.312501)+x2*(-0.137267)+x3*(-0.94508)+x4*(-0.401823)+x5*(-0.58766)+x6*(0.948857)+x7*(0.0853784)+x8*(2.18961)+x9*(-0.646029)+x10*(-0.347875)+x11*(-0.691314)+x12*(-0.1206)+x13*(-0.244344)+x14*(-0.206946)+x15*(0.315006)+x16*(0.374522)+(-0.351362)); 
    h13 = tanh(x0*(-0.823321)+x1*(-0.966731)+x2*(-0.966076)+x3*(-0.411945)+x4*(0.0904809)+x5*(-2.20993)+x6*(1.71107)+x7*(-0.071369)+x8*(-0.480801)+x9*(-0.775881)+x10*(-0.468752)+x11*(-3.1996)+x12*(-0.13453)+x13*(0.235619)+x14*(-0.063208)+x15*(0.32767)+x16*(0.0471864)+(-0.683049)); 
    h14 = tanh(x0*(-0.159295)+x1*(-0.133335)+x2*(-0.23049)+x3*(0.0646736)+x4*(1.27969)+x5*(-0.0227866)+x6*(-0.318703)+x7*(-1.82834)+x8*(0.649836)+x9*(-0.208843)+x10*(-0.333415)+x11*(-0.11287)+x12*(-0.186265)+x13*(-0.0624622)+x14*(0.172204)+x15*(0.361687)+x16*(0.267235)+(0.0295087)); 
    h15 = tanh(x0*(-0.564706)+x1*(-0.259936)+x2*(-0.547799)+x3*(0.100828)+x4*(0.549001)+x5*(-1.27693)+x6*(0.62939)+x7*(-0.807465)+x8*(-1.47771)+x9*(-0.0718484)+x10*(0.262655)+x11*(-1.27684)+x12*(-0.302758)+x13*(0.134721)+x14*(0.311502)+x15*(-0.356158)+x16*(-0.399787)+(0.0247598)); 
    h16 = tanh(x0*(0.53151)+x1*(0.0277526)+x2*(0.666371)+x3*(0.243401)+x4*(-1.02685)+x5*(0.488629)+x6*(0.671842)+x7*(2.6222)+x8*(-2.21887)+x9*(-0.355278)+x10*(-0.371653)+x11*(0.0595186)+x12*(-0.384522)+x13*(0.339221)+x14*(-0.209887)+x15*(0.174223)+x16*(-0.084361)+(-0.105753)); 
    h17 = tanh(x0*(0.380427)+x1*(-0.0454512)+x2*(0.161293)+x3*(-0.788813)+x4*(0.364707)+x5*(0.399095)+x6*(0.212157)+x7*(0.101007)+x8*(0.914947)+x9*(0.177865)+x10*(0.170096)+x11*(1.68846)+x12*(0.163236)+x13*(0.385568)+x14*(-0.397331)+x15*(-0.3736)+x16*(0.0678053)+(-0.15333)); 
    h18 = tanh(x0*(-0.856274)+x1*(-0.704599)+x2*(-0.431844)+x3*(0.324088)+x4*(-0.306129)+x5*(-0.681087)+x6*(-1.69576)+x7*(0.0938584)+x8*(0.124844)+x9*(0.917599)+x10*(0.620475)+x11*(-0.75616)+x12*(-0.126362)+x13*(0.245121)+x14*(-0.376896)+x15*(-0.0902435)+x16*(-0.0513227)+(0.647726)); 
    h19 = tanh(x0*(0.101224)+x1*(0.403497)+x2*(0.121866)+x3*(-0.283696)+x4*(-0.214655)+x5*(-0.315247)+x6*(-0.163242)+x7*(-0.133827)+x8*(-1.65182)+x9*(-0.508137)+x10*(-0.327215)+x11*(0.576549)+x12*(0.138417)+x13*(0.0880406)+x14*(0.274244)+x15*(-0.290983)+x16*(-0.283013)+(-0.0984351)); 
}

void FNN_3_12_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(0.531334)+x1*(0.419991)+x2*(-0.333125)+x3*(-0.22115)+x4*(0.592244)+x5*(0.246935)+x6*(0.14754)+x7*(-0.306686)+x8*(0.403604)+x9*(-0.680287)+x10*(-0.441023)+x11*(-0.811241)+x12*(-0.140757)+x13*(0.417231)+x14*(-0.641814)+x15*(0.630188)+x16*(0.223444)+x17*(-0.48235)+x18*(-0.451159)+x19*(0.463962)+(-0.116524)); 
    h1 = tanh(x0*(0.1119)+x1*(0.0197411)+x2*(0.774612)+x3*(-0.0134225)+x4*(0.528229)+x5*(-1.25773)+x6*(0.718176)+x7*(0.726441)+x8*(-0.579443)+x9*(-0.275207)+x10*(-0.417744)+x11*(-1.02175)+x12*(-1.35473)+x13*(-2.05112)+x14*(0.918691)+x15*(-1.16389)+x16*(-1.59515)+x17*(0.081529)+x18*(2.27914)+x19*(0.10102)+(0.258445)); 
    h2 = tanh(x0*(0.0267706)+x1*(0.756429)+x2*(-0.0635999)+x3*(-0.414988)+x4*(0.367634)+x5*(0.180877)+x6*(-0.0263351)+x7*(-0.402248)+x8*(0.280163)+x9*(0.191302)+x10*(0.0169871)+x11*(-0.297343)+x12*(-0.210517)+x13*(-0.590227)+x14*(0.156446)+x15*(-0.14699)+x16*(0.488169)+x17*(-0.639093)+x18*(0.318803)+x19*(0.351626)+(-0.154989)); 
    h3 = tanh(x0*(0.305905)+x1*(0.334308)+x2*(-0.250054)+x3*(-0.216561)+x4*(0.304534)+x5*(0.807538)+x6*(-0.728105)+x7*(-0.0906731)+x8*(1.50018)+x9*(0.664988)+x10*(0.216161)+x11*(-0.120785)+x12*(-0.437003)+x13*(0.677624)+x14*(-0.806426)+x15*(0.928983)+x16*(0.860282)+x17*(0.271145)+x18*(-1.08514)+x19*(0.52141)+(-0.119157)); 
    h4 = tanh(x0*(-0.792552)+x1*(0.059984)+x2*(-0.266015)+x3*(-0.0768022)+x4*(0.187864)+x5*(-0.0982316)+x6*(0.214145)+x7*(0.00572012)+x8*(0.566312)+x9*(0.122225)+x10*(-0.982503)+x11*(0.545137)+x12*(0.527256)+x13*(1.02437)+x14*(0.287231)+x15*(-0.689002)+x16*(0.0826447)+x17*(-0.439514)+x18*(-0.647379)+x19*(0.259605)+(0.104716)); 
    h5 = tanh(x0*(0.244783)+x1*(-0.762617)+x2*(-0.228926)+x3*(0.282358)+x4*(-0.25979)+x5*(-0.0662584)+x6*(0.316345)+x7*(0.561891)+x8*(0.143713)+x9*(-0.0823246)+x10*(-0.213176)+x11*(0.98023)+x12*(0.0423667)+x13*(0.547045)+x14*(-0.383403)+x15*(-0.197692)+x16*(-0.589868)+x17*(0.228569)+x18*(-0.611529)+x19*(-0.146251)+(0.0799746)); 
    h6 = tanh(x0*(0.63222)+x1*(-1.53312)+x2*(-0.155452)+x3*(0.296069)+x4*(-0.334654)+x5*(-0.562187)+x6*(0.040971)+x7*(0.487998)+x8*(0.0073795)+x9*(-0.0484964)+x10*(0.0138939)+x11*(0.658272)+x12*(0.406336)+x13*(1.21507)+x14*(0.265927)+x15*(-0.695197)+x16*(-0.871885)+x17*(0.4484)+x18*(-0.962692)+x19*(-0.294429)+(-0.000433959)); 
    h7 = tanh(x0*(0.661458)+x1*(-0.302493)+x2*(-0.268249)+x3*(-0.332693)+x4*(-0.127437)+x5*(0.385702)+x6*(0.155699)+x7*(0.245899)+x8*(0.0102834)+x9*(-0.159088)+x10*(0.758229)+x11*(0.00610077)+x12*(-0.144658)+x13*(-0.92901)+x14*(-0.351549)+x15*(0.387104)+x16*(0.165162)+x17*(0.325878)+x18*(0.7611)+x19*(-0.0429602)+(-0.0154431)); 
    h8 = tanh(x0*(-0.243082)+x1*(-1.12407)+x2*(-0.303461)+x3*(-0.275615)+x4*(-0.849179)+x5*(-0.779392)+x6*(-0.340599)+x7*(0.501566)+x8*(-1.08517)+x9*(-0.514796)+x10*(0.388773)+x11*(1.04485)+x12*(0.572992)+x13*(-0.314252)+x14*(1.46492)+x15*(-0.727796)+x16*(-0.156637)+x17*(0.233102)+x18*(-0.631227)+x19*(0.238128)+(-0.126536)); 
    h9 = tanh(x0*(-0.383242)+x1*(0.607777)+x2*(0.237209)+x3*(-0.525756)+x4*(-0.450081)+x5*(-1.31642)+x6*(0.300534)+x7*(-0.366046)+x8*(-2.20356)+x9*(0.506792)+x10*(0.016854)+x11*(-1.12867)+x12*(-0.658583)+x13*(-1.45811)+x14*(0.515538)+x15*(-1.09593)+x16*(-1.71729)+x17*(0.372583)+x18*(1.77748)+x19*(-0.165726)+(-0.129834)); 
}

void FNN_3_12_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-0.798569)+x1*(0.892683)+x2*(0.690117)+x3*(-1.47091)+x4*(-0.346711)+x5*(-0.772425)+x6*(0.929178)+x7*(-1.04276)+x8*(-0.474735)+x9*(-0.997477)+(-0.17578); 
    h1 = x0*(-0.801163)+x1*(0.997806)+x2*(0.621024)+x3*(-1.39418)+x4*(0.250146)+x5*(-0.84079)+x6*(0.971196)+x7*(-0.634058)+x8*(-0.514697)+x9*(-0.98132)+(-0.0497698); 
    h2 = x0*(-0.787122)+x1*(1.01302)+x2*(0.431249)+x3*(-1.38678)+x4*(0.630022)+x5*(-1.03836)+x6*(1.01428)+x7*(-0.264829)+x8*(-0.517805)+x9*(-0.990054)+(-0.0542746); 
}

void FNN_3_12(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_12_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_12_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_12_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_12(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_12(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_3_13_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.200611)+x1*(-0.200821)+x2*(0.344301)+x3*(-0.790321)+x4*(0.564712)+x5*(-2.05677)+x6*(1.37655)+x7*(-0.495254)+x8*(-0.30174)+x9*(-0.531189)+x10*(-0.329221)+x11*(-0.947965)+x12*(-0.0381054)+x13*(0.20529)+x14*(-0.112186)+x15*(-0.373668)+x16*(-0.0536792)+(-0.289246)); 
    h1 = tanh(x0*(0.103842)+x1*(-0.576871)+x2*(0.0917446)+x3*(-0.736071)+x4*(0.135964)+x5*(0.868064)+x6*(-0.486082)+x7*(0.114616)+x8*(6.06482)+x9*(0.0249392)+x10*(0.0580104)+x11*(1.48999)+x12*(-0.290453)+x13*(0.38244)+x14*(-0.372328)+x15*(-0.326309)+x16*(-0.156719)+(0.00662726)); 
    h2 = tanh(x0*(0.362935)+x1*(0.34737)+x2*(0.193056)+x3*(-0.844554)+x4*(-0.31507)+x5*(-0.479524)+x6*(0.318632)+x7*(1.45719)+x8*(-0.563752)+x9*(-0.163239)+x10*(-0.0668397)+x11*(0.0254205)+x12*(0.0518056)+x13*(-0.145508)+x14*(-0.145052)+x15*(-0.155936)+x16*(0.207939)+(-0.363652)); 
    h3 = tanh(x0*(-0.548443)+x1*(-0.305762)+x2*(0.00883051)+x3*(-1.05364)+x4*(0.0201896)+x5*(0.950453)+x6*(1.97328)+x7*(0.415951)+x8*(4.50249)+x9*(-0.788471)+x10*(-0.698159)+x11*(2.20996)+x12*(0.343453)+x13*(-0.0406636)+x14*(-0.152572)+x15*(0.0567753)+x16*(0.189644)+(-0.49422)); 
    h4 = tanh(x0*(-0.917678)+x1*(-0.536271)+x2*(-0.890974)+x3*(-0.100555)+x4*(0.229161)+x5*(-0.625636)+x6*(0.0423241)+x7*(0.582293)+x8*(-0.827946)+x9*(-0.30746)+x10*(-0.516908)+x11*(-0.780732)+x12*(-0.0764294)+x13*(0.0156043)+x14*(0.371279)+x15*(0.352154)+x16*(-0.259891)+(-0.176337)); 
    h5 = tanh(x0*(-0.0971311)+x1*(0.337492)+x2*(0.596181)+x3*(-0.20604)+x4*(-0.0376736)+x5*(-0.155577)+x6*(1.72735)+x7*(1.26767)+x8*(-2.57539)+x9*(-0.838484)+x10*(-0.645404)+x11*(-0.569486)+x12*(0.346647)+x13*(0.242016)+x14*(-0.137916)+x15*(0.246026)+x16*(-0.291518)+(-0.505417)); 
    h6 = tanh(x0*(0.234358)+x1*(-0.332644)+x2*(-0.281946)+x3*(0.138315)+x4*(-0.415665)+x5*(0.590245)+x6*(0.244894)+x7*(-0.569907)+x8*(2.64476)+x9*(-0.44845)+x10*(0.254876)+x11*(1.5395)+x12*(-0.166699)+x13*(-0.21916)+x14*(-0.126218)+x15*(0.317406)+x16*(0.213363)+(-0.118761)); 
    h7 = tanh(x0*(-0.0671833)+x1*(0.129978)+x2*(-0.193989)+x3*(0.116929)+x4*(-0.0612394)+x5*(0.422018)+x6*(0.829021)+x7*(-0.288193)+x8*(1.20814)+x9*(-0.25687)+x10*(-0.630763)+x11*(0.44441)+x12*(0.393577)+x13*(0.319744)+x14*(-0.188187)+x15*(0.299251)+x16*(-0.246469)+(-0.452791)); 
    h8 = tanh(x0*(-0.294207)+x1*(0.0792154)+x2*(0.268165)+x3*(-0.332751)+x4*(0.662756)+x5*(-0.0990413)+x6*(0.365645)+x7*(0.0772858)+x8*(-0.881727)+x9*(0.396387)+x10*(-0.0819028)+x11*(-0.568392)+x12*(-0.17172)+x13*(-0.342096)+x14*(-0.382319)+x15*(0.342269)+x16*(0.0127954)+(0.114551)); 
    h9 = tanh(x0*(-0.170017)+x1*(-0.0912975)+x2*(-0.194067)+x3*(0.197071)+x4*(0.0137359)+x5*(-0.845309)+x6*(0.354031)+x7*(-0.199921)+x8*(1.80568)+x9*(-0.00935599)+x10*(0.232245)+x11*(-0.962512)+x12*(0.211502)+x13*(-0.312245)+x14*(0.155393)+x15*(0.18714)+x16*(-0.235743)+(0.0168068)); 
    h10 = tanh(x0*(0.734594)+x1*(1.28179)+x2*(1.17238)+x3*(-0.215456)+x4*(0.998669)+x5*(0.00680841)+x6*(-0.92853)+x7*(0.66373)+x8*(4.62297)+x9*(0.361772)+x10*(0.164383)+x11*(0.214508)+x12*(-0.00759879)+x13*(-0.296578)+x14*(-0.280945)+x15*(0.0869845)+x16*(0.353866)+(0.109541)); 
    h11 = tanh(x0*(0.101054)+x1*(-0.0782355)+x2*(-0.0369895)+x3*(0.829637)+x4*(-0.0561464)+x5*(-1.54246)+x6*(-1.7663)+x7*(1.65609)+x8*(0.991224)+x9*(-0.0831063)+x10*(-0.42075)+x11*(-4.8353)+x12*(0.0765105)+x13*(0.117398)+x14*(-0.329241)+x15*(0.0486702)+x16*(-0.198423)+(-0.0989452)); 
    h12 = tanh(x0*(1.33408)+x1*(1.37613)+x2*(0.973076)+x3*(0.100038)+x4*(-1.02133)+x5*(-1.2553)+x6*(-0.913638)+x7*(1.0271)+x8*(0.872167)+x9*(-0.18404)+x10*(-0.15269)+x11*(-1.60133)+x12*(-0.279632)+x13*(-0.111845)+x14*(0.214968)+x15*(-0.0699644)+x16*(-0.116169)+(0.0686528)); 
    h13 = tanh(x0*(-0.553557)+x1*(-0.751103)+x2*(-0.351397)+x3*(-0.959671)+x4*(-0.00653951)+x5*(0.715931)+x6*(0.420054)+x7*(-0.394909)+x8*(0.391707)+x9*(0.112829)+x10*(-0.201373)+x11*(0.661435)+x12*(-0.287774)+x13*(0.318384)+x14*(-0.0990494)+x15*(-0.23968)+x16*(0.0560635)+(-0.268082)); 
    h14 = tanh(x0*(-0.783289)+x1*(-0.586548)+x2*(-0.484034)+x3*(-0.209917)+x4*(0.0813009)+x5*(0.716224)+x6*(0.145782)+x7*(-0.467628)+x8*(-1.15671)+x9*(-0.280757)+x10*(-0.38009)+x11*(0.32069)+x12*(0.320284)+x13*(0.00536644)+x14*(-0.0682966)+x15*(0.0396822)+x16*(0.365955)+(-0.0869427)); 
    h15 = tanh(x0*(0.638409)+x1*(0.617633)+x2*(0.240817)+x3*(0.955905)+x4*(-0.0957525)+x5*(1.13663)+x6*(0.518464)+x7*(0.371807)+x8*(-0.167917)+x9*(-0.09048)+x10*(-0.0477122)+x11*(-0.763808)+x12*(0.373836)+x13*(-0.391189)+x14*(-0.312989)+x15*(-0.26743)+x16*(0.188589)+(0.162243)); 
    h16 = tanh(x0*(-0.409592)+x1*(-0.482955)+x2*(-0.768411)+x3*(0.246624)+x4*(-0.0290387)+x5*(0.667469)+x6*(0.241545)+x7*(-0.126236)+x8*(0.751761)+x9*(0.186876)+x10*(0.4093)+x11*(1.55137)+x12*(0.179327)+x13*(0.208078)+x14*(-0.303961)+x15*(0.0663021)+x16*(0.145341)+(0.229135)); 
    h17 = tanh(x0*(0.0631238)+x1*(-0.0221235)+x2*(0.0815527)+x3*(-0.839664)+x4*(-0.349272)+x5*(0.687496)+x6*(0.0821618)+x7*(-0.0432018)+x8*(0.929849)+x9*(-0.0629046)+x10*(0.365355)+x11*(1.9397)+x12*(-0.377489)+x13*(-0.18257)+x14*(-0.249768)+x15*(-0.347907)+x16*(-0.075566)+(0.0121749)); 
    h18 = tanh(x0*(9.23144)+x1*(9.51356)+x2*(9.65079)+x3*(0.33349)+x4*(-1.10773)+x5*(-8.04357)+x6*(0.70139)+x7*(1.79147)+x8*(-1.67152)+x9*(-0.372399)+x10*(0.225664)+x11*(-9.22652)+x12*(-0.214578)+x13*(0.167423)+x14*(0.00580072)+x15*(-0.382513)+x16*(-0.0348158)+(-0.0825567)); 
    h19 = tanh(x0*(-0.570954)+x1*(-0.829286)+x2*(-0.621605)+x3*(-0.438938)+x4*(0.121736)+x5*(-0.104613)+x6*(-0.783956)+x7*(0.518451)+x8*(1.2883)+x9*(0.429249)+x10*(0.330135)+x11*(-3.40079)+x12*(0.289142)+x13*(-0.0135314)+x14*(0.0463413)+x15*(0.148814)+x16*(-0.258132)+(0.149469)); 
}

void FNN_3_13_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(0.91286)+x1*(-1.12616)+x2*(-1.61074)+x3*(0.226071)+x4*(-0.868733)+x5*(-2.02831)+x6*(-0.732161)+x7*(0.235489)+x8*(0.809047)+x9*(0.630494)+x10*(0.261624)+x11*(-1.15735)+x12*(-0.0290336)+x13*(-0.0777938)+x14*(0.141828)+x15*(-0.287428)+x16*(0.360357)+x17*(0.0970376)+x18*(-1.15175)+x19*(-0.302072)+(0.0977985)); 
    h1 = tanh(x0*(0.599081)+x1*(-0.993288)+x2*(-0.0138058)+x3*(0.381809)+x4*(0.240484)+x5*(0.365773)+x6*(-0.479461)+x7*(-0.0812017)+x8*(-0.242735)+x9*(-0.58532)+x10*(-0.448323)+x11*(0.0980834)+x12*(-0.246188)+x13*(0.253091)+x14*(-0.450528)+x15*(0.555522)+x16*(0.195107)+x17*(-0.32698)+x18*(0.712602)+x19*(-0.829199)+(0.0243264)); 
    h2 = tanh(x0*(-1.307)+x1*(0.783452)+x2*(-0.0627646)+x3*(-2.32069)+x4*(0.0568686)+x5*(-1.89145)+x6*(-0.205281)+x7*(-0.709854)+x8*(0.437884)+x9*(0.0020316)+x10*(0.830743)+x11*(0.57462)+x12*(0.567575)+x13*(-0.686901)+x14*(0.23321)+x15*(0.172252)+x16*(-0.0588488)+x17*(0.00345135)+x18*(-1.01153)+x19*(1.62082)+(0.255004)); 
    h3 = tanh(x0*(0.0594054)+x1*(0.577606)+x2*(0.0764159)+x3*(-0.0892026)+x4*(-0.340202)+x5*(-1.58705)+x6*(0.0080134)+x7*(-0.606853)+x8*(-0.126312)+x9*(-0.206569)+x10*(-0.0968985)+x11*(0.486905)+x12*(0.346896)+x13*(0.0714572)+x14*(-0.555872)+x15*(-0.398333)+x16*(-0.0233319)+x17*(-0.0997972)+x18*(-0.663387)+x19*(0.758852)+(0.144499)); 
    h4 = tanh(x0*(0.795468)+x1*(0.450056)+x2*(-0.387936)+x3*(0.784193)+x4*(-0.344849)+x5*(-0.561912)+x6*(0.428076)+x7*(0.464997)+x8*(-0.443025)+x9*(0.390413)+x10*(0.519066)+x11*(1.27348)+x12*(0.558534)+x13*(-0.00586137)+x14*(-0.415463)+x15*(0.372453)+x16*(0.655931)+x17*(0.23274)+x18*(-1.12263)+x19*(-0.0733801)+(0.303733)); 
    h5 = tanh(x0*(1.52137)+x1*(0.244527)+x2*(0.986046)+x3*(0.891028)+x4*(0.10381)+x5*(1.36008)+x6*(0.357873)+x7*(2.31851)+x8*(-0.189787)+x9*(0.146103)+x10*(0.516127)+x11*(-0.769944)+x12*(0.154917)+x13*(0.268723)+x14*(0.695993)+x15*(-0.536916)+x16*(-0.345553)+x17*(-0.146966)+x18*(0.189862)+x19*(-1.41735)+(-0.417267)); 
    h6 = tanh(x0*(0.61889)+x1*(-1.37248)+x2*(0.623562)+x3*(-0.0730977)+x4*(0.146603)+x5*(0.613168)+x6*(-0.224734)+x7*(0.0236871)+x8*(-0.00940027)+x9*(-0.519731)+x10*(-0.488796)+x11*(-0.0197628)+x12*(-0.84389)+x13*(0.452916)+x14*(0.0469468)+x15*(0.19457)+x16*(-0.215805)+x17*(0.309603)+x18*(1.01761)+x19*(-0.884714)+(-0.0381965)); 
    h7 = tanh(x0*(0.133376)+x1*(-0.78031)+x2*(-0.000553514)+x3*(-0.209222)+x4*(0.706693)+x5*(0.837739)+x6*(-0.958306)+x7*(0.320362)+x8*(0.30924)+x9*(-0.39308)+x10*(-0.372254)+x11*(-0.457587)+x12*(-0.193012)+x13*(-0.170461)+x14*(0.844791)+x15*(0.271503)+x16*(-0.556405)+x17*(-0.298085)+x18*(1.06525)+x19*(-0.364235)+(-0.429272)); 
    h8 = tanh(x0*(-1.56528)+x1*(-1.10317)+x2*(1.64682)+x3*(2.71657)+x4*(-0.180867)+x5*(0.450277)+x6*(-0.685849)+x7*(0.150588)+x8*(0.209678)+x9*(-0.363008)+x10*(-0.748283)+x11*(1.49995)+x12*(-0.0289074)+x13*(-0.798468)+x14*(-0.172275)+x15*(-0.119774)+x16*(-0.841546)+x17*(0.355278)+x18*(1.03266)+x19*(0.517635)+(-0.330841)); 
    h9 = tanh(x0*(0.27489)+x1*(0.35635)+x2*(-0.0460475)+x3*(-0.181299)+x4*(-0.262075)+x5*(-0.715038)+x6*(1.09019)+x7*(0.318137)+x8*(-0.267477)+x9*(0.463659)+x10*(0.829308)+x11*(0.532778)+x12*(0.276328)+x13*(0.459806)+x14*(-0.883441)+x15*(-0.0923442)+x16*(0.427464)+x17*(1.03916)+x18*(-0.836463)+x19*(0.626815)+(0.386483)); 
}

void FNN_3_13_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-1.28328)+x1*(0.943552)+x2*(-0.549931)+x3*(0.627857)+x4*(0.627134)+x5*(-0.488411)+x6*(0.767596)+x7*(-0.0568476)+x8*(-1.33059)+x9*(0.596121)+(0.258276); 
    h1 = x0*(-1.36019)+x1*(0.871023)+x2*(-0.567824)+x3*(0.65431)+x4*(0.632816)+x5*(-0.591682)+x6*(0.895963)+x7*(-0.288802)+x8*(-1.40389)+x9*(0.382137)+(0.195236); 
    h2 = x0*(-1.36827)+x1*(0.890922)+x2*(-0.568729)+x3*(0.668848)+x4*(0.646458)+x5*(-0.585816)+x6*(0.9042)+x7*(-0.272413)+x8*(-1.41188)+x9*(0.404558)+(0.194422); 
}

void FNN_3_13(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_13_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_13_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_13_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_13(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_13(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_11(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x4 < lowBound ) {
        ModelKDTree_12(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x4 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_12(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_13(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x4 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_13(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void ModelKDTree_9(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x6 < lowBound ) {
        ModelKDTree_10(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x6 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_10(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_11(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x6 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_11(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void FNN_3_15_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.879886)+x1*(-1.00678)+x2*(-0.978135)+x3*(-0.575291)+x4*(0.0190083)+x5*(-0.093122)+x6*(2.28637)+x7*(-0.411276)+x8*(0.498155)+x9*(-0.809196)+x10*(-0.194801)+x11*(-0.0968144)+x12*(0.0020546)+x13*(0.245176)+x14*(-0.257697)+x15*(0.143115)+x16*(0.0721054)+(-0.870415)); 
    h1 = tanh(x0*(0.309706)+x1*(0.393897)+x2*(0.498281)+x3*(-0.0481082)+x4*(-0.205509)+x5*(0.0812157)+x6*(0.128576)+x7*(0.154728)+x8*(-0.0655421)+x9*(-0.429224)+x10*(-0.244106)+x11*(-0.300407)+x12*(0.111481)+x13*(0.051808)+x14*(0.390966)+x15*(0.111808)+x16*(-0.0947161)+(0.0254417)); 
    h2 = tanh(x0*(-1.06834)+x1*(-0.962678)+x2*(-0.660585)+x3*(0.609739)+x4*(-0.774506)+x5*(-0.305794)+x6*(-0.212113)+x7*(0.388159)+x8*(-0.186716)+x9*(0.3105)+x10*(0.0164148)+x11*(-0.371184)+x12*(0.250351)+x13*(-0.317566)+x14*(0.0998731)+x15*(0.240604)+x16*(0.369043)+(0.127988)); 
    h3 = tanh(x0*(0.417994)+x1*(0.128097)+x2*(0.176762)+x3*(-0.0393828)+x4*(0.019888)+x5*(0.166184)+x6*(0.634269)+x7*(0.487869)+x8*(-0.390967)+x9*(-0.118929)+x10*(0.422562)+x11*(0.051741)+x12*(0.100002)+x13*(-0.303964)+x14*(0.233066)+x15*(-0.382015)+x16*(0.0949779)+(0.0967372)); 
    h4 = tanh(x0*(-0.627195)+x1*(-0.555832)+x2*(-0.507262)+x3*(1.25012)+x4*(1.50487)+x5*(0.0442381)+x6*(-1.16843)+x7*(-1.36113)+x8*(-0.227573)+x9*(0.159105)+x10*(0.0274527)+x11*(-0.46543)+x12*(-0.0687755)+x13*(0.334345)+x14*(0.353403)+x15*(-0.319528)+x16*(-0.213908)+(0.32522)); 
    h5 = tanh(x0*(-0.863245)+x1*(-0.616292)+x2*(-0.350555)+x3*(-0.134733)+x4*(-0.486239)+x5*(-0.315479)+x6*(0.416679)+x7*(0.37048)+x8*(-0.208376)+x9*(-0.0946361)+x10*(0.329129)+x11*(0.0763792)+x12*(-0.179018)+x13*(-0.0231177)+x14*(-0.0896024)+x15*(0.0922382)+x16*(-0.325676)+(-0.258301)); 
    h6 = tanh(x0*(-0.288929)+x1*(-0.134481)+x2*(-0.0504028)+x3*(0.707539)+x4*(-0.425959)+x5*(-0.0391457)+x6*(0.209076)+x7*(0.86545)+x8*(0.160028)+x9*(0.0950385)+x10*(0.0494194)+x11*(-0.233833)+x12*(0.377009)+x13*(0.0324374)+x14*(-0.32968)+x15*(-0.394333)+x16*(-0.0922575)+(-0.0103197)); 
    h7 = tanh(x0*(-0.34153)+x1*(0.317021)+x2*(0.137397)+x3*(-0.387413)+x4*(-0.0125633)+x5*(-0.0483554)+x6*(0.190196)+x7*(0.119544)+x8*(-0.642045)+x9*(0.119789)+x10*(-0.000218517)+x11*(-0.330064)+x12*(0.198925)+x13*(0.361539)+x14*(-0.100084)+x15*(-0.0193803)+x16*(-0.0589128)+(-0.0869178)); 
    h8 = tanh(x0*(-0.0593331)+x1*(0.0716575)+x2*(0.189798)+x3*(-0.815042)+x4*(0.999298)+x5*(0.0662374)+x6*(1.29336)+x7*(-1.5441)+x8*(0.177926)+x9*(-0.263572)+x10*(-0.385358)+x11*(0.121884)+x12*(-0.369963)+x13*(0.30689)+x14*(-0.0752611)+x15*(0.400217)+x16*(0.330955)+(-0.0247108)); 
    h9 = tanh(x0*(0.151138)+x1*(0.203894)+x2*(0.166429)+x3*(-0.587435)+x4*(0.0647379)+x5*(-0.442382)+x6*(0.838053)+x7*(1.51504)+x8*(-0.161685)+x9*(-0.691783)+x10*(0.0282624)+x11*(-0.0534057)+x12*(0.335061)+x13*(-0.159227)+x14*(0.0919852)+x15*(-0.12653)+x16*(-0.370149)+(-0.474241)); 
    h10 = tanh(x0*(-0.206466)+x1*(-0.394842)+x2*(-0.0977284)+x3*(-1.19068)+x4*(0.314808)+x5*(0.193844)+x6*(1.10141)+x7*(0.0291678)+x8*(-0.104344)+x9*(-0.417281)+x10*(-0.400589)+x11*(0.152255)+x12*(0.350322)+x13*(0.310653)+x14*(0.278957)+x15*(-0.0859638)+x16*(0.106868)+(-0.192312)); 
    h11 = tanh(x0*(0.333249)+x1*(-0.12892)+x2*(0.0064354)+x3*(0.655069)+x4*(-0.430161)+x5*(0.00193904)+x6*(-0.939712)+x7*(1.22146)+x8*(-0.381361)+x9*(0.627126)+x10*(-0.42906)+x11*(-0.116699)+x12*(-0.343342)+x13*(-0.223567)+x14*(-0.184098)+x15*(-0.249721)+x16*(-0.279483)+(0.363132)); 
    h12 = tanh(x0*(0.704278)+x1*(0.24165)+x2*(0.0235873)+x3*(-0.136414)+x4*(-1.48837)+x5*(-0.177415)+x6*(-0.684996)+x7*(1.93294)+x8*(0.478231)+x9*(0.183883)+x10*(0.127965)+x11*(0.176129)+x12*(-0.121838)+x13*(-0.258537)+x14*(-0.003977)+x15*(-0.374583)+x16*(0.0932259)+(0.0376925)); 
    h13 = tanh(x0*(0.072654)+x1*(0.172562)+x2*(-0.0765846)+x3*(0.182456)+x4*(0.167026)+x5*(0.117057)+x6*(-0.545708)+x7*(0.78095)+x8*(0.0339929)+x9*(-0.442692)+x10*(-0.474514)+x11*(0.518074)+x12*(0.095443)+x13*(-0.356982)+x14*(-0.382679)+x15*(0.0884856)+x16*(-0.178755)+(-0.0570698)); 
    h14 = tanh(x0*(0.199814)+x1*(-0.0273768)+x2*(0.4049)+x3*(0.993202)+x4*(0.124414)+x5*(0.0904996)+x6*(-1.58978)+x7*(-0.28301)+x8*(-0.135598)+x9*(0.078256)+x10*(-0.145587)+x11*(0.228278)+x12*(-0.175177)+x13*(-0.321998)+x14*(0.372116)+x15*(-0.199578)+x16*(0.110593)+(0.403736)); 
    h15 = tanh(x0*(-1.0969)+x1*(-0.432071)+x2*(-1.07196)+x3*(0.301841)+x4*(0.0222871)+x5*(0.367978)+x6*(-0.698793)+x7*(0.568757)+x8*(-0.327167)+x9*(-0.118294)+x10*(-0.234217)+x11*(0.410618)+x12*(-0.279709)+x13*(-0.29299)+x14*(0.277382)+x15*(-0.20986)+x16*(-0.0729611)+(0.103864)); 
    h16 = tanh(x0*(-0.268009)+x1*(-0.401648)+x2*(0.224284)+x3*(-0.762246)+x4*(-0.909935)+x5*(-0.0767283)+x6*(0.793868)+x7*(1.10237)+x8*(0.703119)+x9*(-0.247632)+x10*(-0.462888)+x11*(-0.324356)+x12*(0.185438)+x13*(0.034146)+x14*(0.386109)+x15*(0.198108)+x16*(-0.160416)+(-0.416299)); 
    h17 = tanh(x0*(0.0235299)+x1*(0.00394765)+x2*(0.263122)+x3*(-0.864151)+x4*(0.170752)+x5*(0.154811)+x6*(-0.522507)+x7*(0.3087)+x8*(-0.779465)+x9*(0.0025018)+x10*(-0.246201)+x11*(-0.148749)+x12*(0.26603)+x13*(0.202729)+x14*(-0.268977)+x15*(0.300275)+x16*(0.0332532)+(-0.0436261)); 
    h18 = tanh(x0*(0.348582)+x1*(0.0813669)+x2*(0.0284423)+x3*(0.369718)+x4*(-1.11397)+x5*(0.148208)+x6*(-0.217655)+x7*(-0.0638752)+x8*(-0.165259)+x9*(-0.151334)+x10*(0.149768)+x11*(-0.184674)+x12*(0.106128)+x13*(-0.260404)+x14*(-0.194501)+x15*(0.0777292)+x16*(0.0574582)+(0.063115)); 
    h19 = tanh(x0*(-0.314256)+x1*(0.149078)+x2*(-0.259059)+x3*(-1.26993)+x4*(0.666193)+x5*(0.0727966)+x6*(1.8371)+x7*(-1.05)+x8*(0.705541)+x9*(-0.115158)+x10*(-0.150314)+x11*(0.14527)+x12*(0.380282)+x13*(0.325806)+x14*(0.319284)+x15*(-0.0155445)+x16*(0.107088)+(-0.297325)); 
}

void FNN_3_15_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.130593)+x1*(0.537324)+x2*(0.00343698)+x3*(-0.0758166)+x4*(-0.755129)+x5*(0.984041)+x6*(0.787899)+x7*(0.436528)+x8*(-0.901403)+x9*(0.614326)+x10*(-0.462119)+x11*(0.429515)+x12*(0.227282)+x13*(0.223257)+x14*(-0.341524)+x15*(0.483285)+x16*(0.319643)+x17*(0.580945)+x18*(-0.0707952)+x19*(0.187706)+(-0.0574983)); 
    h1 = tanh(x0*(-0.114028)+x1*(0.288152)+x2*(-0.191718)+x3*(0.123097)+x4*(-0.187034)+x5*(0.410262)+x6*(-0.331063)+x7*(0.305735)+x8*(0.456045)+x9*(-0.510645)+x10*(-0.415649)+x11*(0.276037)+x12*(-0.361043)+x13*(-0.0206849)+x14*(0.00179116)+x15*(0.176467)+x16*(-0.426125)+x17*(-0.042058)+x18*(-0.206173)+x19*(-0.00457743)+(0.0349514)); 
    h2 = tanh(x0*(0.558892)+x1*(0.148251)+x2*(0.344178)+x3*(1.02243)+x4*(-1.38852)+x5*(0.892686)+x6*(1.4483)+x7*(-0.206163)+x8*(-0.148223)+x9*(0.136054)+x10*(1.55714)+x11*(-0.460094)+x12*(1.319)+x13*(0.565305)+x14*(-1.4893)+x15*(0.290822)+x16*(-0.735507)+x17*(0.263394)+x18*(0.117342)+x19*(-0.145569)+(0.00804131)); 
    h3 = tanh(x0*(0.210473)+x1*(0.236307)+x2*(-0.0936848)+x3*(-0.0901405)+x4*(-1.19608)+x5*(-0.0303237)+x6*(1.72618)+x7*(-0.642082)+x8*(-0.565784)+x9*(0.137656)+x10*(0.359258)+x11*(0.840844)+x12*(0.861057)+x13*(1.30505)+x14*(-0.524666)+x15*(0.778365)+x16*(0.988433)+x17*(-0.340519)+x18*(-0.652502)+x19*(-0.489593)+(0.143226)); 
    h4 = tanh(x0*(-0.405304)+x1*(0.187327)+x2*(0.173362)+x3*(0.493462)+x4*(-0.158305)+x5*(0.0658893)+x6*(0.363441)+x7*(0.648718)+x8*(0.123658)+x9*(-0.413901)+x10*(0.063367)+x11*(-0.458932)+x12*(-0.301816)+x13*(0.224056)+x14*(0.360392)+x15*(-0.0450384)+x16*(-0.0985111)+x17*(-0.116306)+x18*(-0.126727)+x19*(0.261628)+(-0.132187)); 
    h5 = tanh(x0*(0.172392)+x1*(-0.207613)+x2*(-0.02929)+x3*(-0.016217)+x4*(0.834814)+x5*(-1.17936)+x6*(-1.17626)+x7*(-0.276329)+x8*(1.10063)+x9*(-0.61664)+x10*(0.383945)+x11*(-0.19923)+x12*(-0.360583)+x13*(-0.626235)+x14*(0.378468)+x15*(-0.39578)+x16*(-0.120459)+x17*(-1.02723)+x18*(-0.274616)+x19*(-0.438284)+(0.252477)); 
    h6 = tanh(x0*(0.65332)+x1*(0.498835)+x2*(-0.185388)+x3*(-0.26212)+x4*(-0.782143)+x5*(-0.0716651)+x6*(0.679073)+x7*(0.451842)+x8*(0.239453)+x9*(0.540426)+x10*(0.40206)+x11*(-0.323422)+x12*(0.684182)+x13*(-0.0805623)+x14*(-1.2158)+x15*(-0.641701)+x16*(0.261378)+x17*(-0.0440307)+x18*(-0.676997)+x19*(0.677631)+(-0.156271)); 
    h7 = tanh(x0*(-0.0371779)+x1*(0.0969293)+x2*(0.448342)+x3*(0.136067)+x4*(0.316782)+x5*(-0.228454)+x6*(-0.0881753)+x7*(0.311773)+x8*(1.92136)+x9*(-0.483802)+x10*(0.671566)+x11*(-0.546713)+x12*(-0.313824)+x13*(-0.458897)+x14*(-0.773891)+x15*(-0.455397)+x16*(-0.322797)+x17*(0.184821)+x18*(-0.605658)+x19*(0.212774)+(0.0211204)); 
    h8 = tanh(x0*(0.350328)+x1*(-0.620717)+x2*(-0.314497)+x3*(0.40018)+x4*(-1.51142)+x5*(0.311474)+x6*(-0.303922)+x7*(-0.145564)+x8*(1.48129)+x9*(0.321104)+x10*(-0.0377673)+x11*(-0.649165)+x12*(-1.11309)+x13*(0.031795)+x14*(-0.765527)+x15*(0.225438)+x16*(0.435841)+x17*(-0.814722)+x18*(-0.613043)+x19*(0.489771)+(-0.0956744)); 
    h9 = tanh(x0*(-0.165583)+x1*(-0.235243)+x2*(0.530112)+x3*(0.413538)+x4*(-0.514499)+x5*(0.0453118)+x6*(0.0476414)+x7*(0.123864)+x8*(0.225482)+x9*(1.25301)+x10*(0.249498)+x11*(0.246475)+x12*(0.224274)+x13*(0.109889)+x14*(-0.392926)+x15*(0.480398)+x16*(0.839393)+x17*(0.163363)+x18*(0.381425)+x19*(-1.59488)+(0.131805)); 
}

void FNN_3_15_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-0.394541)+x1*(-0.39677)+x2*(0.552945)+x3*(-0.548163)+x4*(-0.15276)+x5*(1.06228)+x6*(-0.471512)+x7*(-0.724248)+x8*(0.502075)+x9*(0.885484)+(0.14352); 
    h1 = x0*(-0.482328)+x1*(-0.846851)+x2*(0.461564)+x3*(-0.367019)+x4*(0.574431)+x5*(0.715704)+x6*(-0.426892)+x7*(-0.580517)+x8*(0.408806)+x9*(0.645457)+(0.107859); 
    h2 = x0*(-0.709718)+x1*(-0.0100697)+x2*(0.548434)+x3*(-0.53982)+x4*(-0.727781)+x5*(0.759646)+x6*(-0.467008)+x7*(-0.716596)+x8*(0.517375)+x9*(0.892099)+(0.201025); 
}

void FNN_3_15(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_15_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_15_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_15_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_15(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_15(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_3_16_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.409079)+x1*(-0.712445)+x2*(-0.760963)+x3*(0.293843)+x4*(-0.213609)+x5*(0.442083)+x6*(0.030446)+x7*(0.332055)+x8*(0.116414)+x9*(-0.378053)+x10*(-0.211633)+x11*(0.245891)+x12*(-0.311686)+x13*(-0.188064)+x14*(0.184537)+x15*(-0.0399733)+x16*(-0.219178)+(0.0378052)); 
    h1 = tanh(x0*(0.101261)+x1*(0.38518)+x2*(0.327299)+x3*(0.530264)+x4*(-0.718647)+x5*(0.272879)+x6*(0.346866)+x7*(-0.121412)+x8*(-0.609832)+x9*(0.477414)+x10*(-0.403597)+x11*(0.332486)+x12*(-0.181417)+x13*(-0.153774)+x14*(-0.0275164)+x15*(-0.128316)+x16*(-0.281406)+(-0.110425)); 
    h2 = tanh(x0*(0.0453095)+x1*(0.288453)+x2*(0.530758)+x3*(0.637901)+x4*(-0.839362)+x5*(0.383558)+x6*(-0.523444)+x7*(0.407822)+x8*(0.160184)+x9*(-0.886816)+x10*(-0.389876)+x11*(0.307815)+x12*(0.0818315)+x13*(0.0670975)+x14*(0.339839)+x15*(0.0817507)+x16*(0.292788)+(0.0836425)); 
    h3 = tanh(x0*(-0.901418)+x1*(-0.521318)+x2*(-0.367208)+x3*(-0.180405)+x4*(-0.584007)+x5*(0.134269)+x6*(-0.838747)+x7*(-0.432234)+x8*(-0.26912)+x9*(0.176576)+x10*(0.0734006)+x11*(0.414302)+x12*(0.0318242)+x13*(0.192769)+x14*(-0.0623709)+x15*(0.344187)+x16*(0.164264)+(0.0280544)); 
    h4 = tanh(x0*(-3.86194)+x1*(-3.88916)+x2*(-3.73023)+x3*(0.116223)+x4*(0.258966)+x5*(-0.542619)+x6*(-0.409842)+x7*(0.570232)+x8*(-0.0459349)+x9*(-0.360588)+x10*(0.0802649)+x11*(-0.161084)+x12*(0.363436)+x13*(0.267138)+x14*(0.177004)+x15*(-0.252568)+x16*(-0.219105)+(-0.210861)); 
    h5 = tanh(x0*(-0.872162)+x1*(-1.31226)+x2*(-0.640352)+x3*(-0.570265)+x4*(-1.08266)+x5*(-0.104794)+x6*(0.836476)+x7*(1.02492)+x8*(0.0211118)+x9*(0.155226)+x10*(-0.0297042)+x11*(-0.328603)+x12*(0.214672)+x13*(-0.221069)+x14*(0.176474)+x15*(0.312785)+x16*(0.401501)+(0.061612)); 
    h6 = tanh(x0*(0.361216)+x1*(0.301463)+x2*(-0.274029)+x3*(-0.280512)+x4*(-3.12214)+x5*(-0.581149)+x6*(0.379665)+x7*(4.79178)+x8*(0.385235)+x9*(-0.150826)+x10*(0.328858)+x11*(0.186936)+x12*(-0.301857)+x13*(-0.304839)+x14*(-0.222607)+x15*(0.227811)+x16*(-0.130112)+(-0.326477)); 
    h7 = tanh(x0*(-0.0592542)+x1*(-0.0586953)+x2*(0.153553)+x3*(0.0164562)+x4*(-0.139509)+x5*(-0.16905)+x6*(0.0649679)+x7*(-0.0453014)+x8*(0.169387)+x9*(-0.157847)+x10*(-0.23928)+x11*(0.0384535)+x12*(-0.137737)+x13*(0.29282)+x14*(-0.0709404)+x15*(-0.137886)+x16*(0.302543)+(-0.184861)); 
    h8 = tanh(x0*(-0.156267)+x1*(0.329018)+x2*(0.476714)+x3*(-0.367701)+x4*(-0.166104)+x5*(-0.119005)+x6*(-0.164498)+x7*(-0.106929)+x8*(0.503633)+x9*(0.247598)+x10*(0.25404)+x11*(-0.170384)+x12*(0.0418873)+x13*(-0.18861)+x14*(-0.00330964)+x15*(0.352062)+x16*(0.0330431)+(-0.0533178)); 
    h9 = tanh(x0*(0.0231002)+x1*(0.201973)+x2*(0.6271)+x3*(-0.395138)+x4*(-2.48245)+x5*(-0.276562)+x6*(0.421395)+x7*(3.45872)+x8*(-0.755145)+x9*(0.238053)+x10*(-0.0804901)+x11*(0.0266854)+x12*(0.163623)+x13*(-0.359895)+x14*(-0.0231195)+x15*(0.0569877)+x16*(0.0979602)+(0.0700325)); 
    h10 = tanh(x0*(-0.264033)+x1*(-0.134903)+x2*(-0.0932651)+x3*(0.302127)+x4*(-0.0752487)+x5*(0.194351)+x6*(-0.934679)+x7*(0.636269)+x8*(-0.222847)+x9*(-0.585534)+x10*(0.155122)+x11*(-0.0582933)+x12*(0.299915)+x13*(0.0849614)+x14*(-0.187679)+x15*(-0.228655)+x16*(0.390806)+(0.107111)); 
    h11 = tanh(x0*(0.935429)+x1*(0.843442)+x2*(0.708368)+x3*(0.664255)+x4*(-0.0716428)+x5*(-0.135637)+x6*(-0.960646)+x7*(0.408417)+x8*(0.401738)+x9*(-0.520196)+x10*(0.203721)+x11*(0.0657679)+x12*(0.132772)+x13*(0.140028)+x14*(0.0798019)+x15*(-0.178811)+x16*(0.372598)+(0.0872215)); 
    h12 = tanh(x0*(-0.427808)+x1*(-0.0209146)+x2*(-0.476181)+x3*(-0.74508)+x4*(-0.116874)+x5*(0.107225)+x6*(0.335078)+x7*(-0.0120861)+x8*(1.08946)+x9*(-0.0715185)+x10*(-0.0997901)+x11*(0.502829)+x12*(0.251704)+x13*(-0.3388)+x14*(0.320585)+x15*(0.112392)+x16*(-0.286851)+(0.145216)); 
    h13 = tanh(x0*(0.198817)+x1*(0.522386)+x2*(0.110923)+x3*(0.295972)+x4*(0.582127)+x5*(0.0712897)+x6*(0.0787466)+x7*(0.414445)+x8*(0.26919)+x9*(0.264352)+x10*(0.0753941)+x11*(-0.101678)+x12*(-0.084118)+x13*(-0.106833)+x14*(0.0346671)+x15*(-0.131444)+x16*(-0.0228086)+(-0.0490882)); 
    h14 = tanh(x0*(0.169899)+x1*(0.315076)+x2*(0.370071)+x3*(-0.0571102)+x4*(-0.669963)+x5*(0.0415913)+x6*(-0.00509477)+x7*(-0.822691)+x8*(0.385079)+x9*(-0.445964)+x10*(0.572774)+x11*(0.0727248)+x12*(-0.152186)+x13*(-0.0134979)+x14*(0.165135)+x15*(0.321894)+x16*(-0.0215027)+(0.120135)); 
    h15 = tanh(x0*(0.154384)+x1*(-0.376879)+x2*(-0.358358)+x3*(-0.500545)+x4*(0.137058)+x5*(-0.0718249)+x6*(0.336468)+x7*(0.350255)+x8*(0.242215)+x9*(-0.657064)+x10*(0.255046)+x11*(0.43018)+x12*(-0.0222905)+x13*(0.376679)+x14*(0.314033)+x15*(0.212013)+x16*(-0.0941511)+(-0.340965)); 
    h16 = tanh(x0*(0.122559)+x1*(-0.152341)+x2*(-0.171687)+x3*(-0.162497)+x4*(-0.165048)+x5*(0.1516)+x6*(0.586084)+x7*(-0.663737)+x8*(-0.307013)+x9*(0.872511)+x10*(0.0499561)+x11*(0.513493)+x12*(-0.0409022)+x13*(-0.39945)+x14*(0.174922)+x15*(0.284163)+x16*(0.31776)+(0.0513213)); 
    h17 = tanh(x0*(-0.546329)+x1*(-1.12154)+x2*(-1.20358)+x3*(1.15651)+x4*(-0.214378)+x5*(0.0101569)+x6*(-1.36501)+x7*(-0.11186)+x8*(0.765189)+x9*(-1.06842)+x10*(0.514298)+x11*(0.458481)+x12*(0.253327)+x13*(0.0670653)+x14*(-0.28537)+x15*(0.364893)+x16*(-0.0779149)+(0.20084)); 
    h18 = tanh(x0*(0.420511)+x1*(-0.00731322)+x2*(0.28056)+x3*(0.0187112)+x4*(-0.204895)+x5*(-0.255999)+x6*(-0.256163)+x7*(-0.346246)+x8*(0.135705)+x9*(-0.560491)+x10*(-0.523418)+x11*(-0.240719)+x12*(-0.328024)+x13*(-0.326912)+x14*(-0.316177)+x15*(-0.0540892)+x16*(0.159563)+(0.0554718)); 
    h19 = tanh(x0*(-0.194267)+x1*(-0.323656)+x2*(-0.504876)+x3*(-0.525745)+x4*(-2.11566)+x5*(0.090409)+x6*(0.736704)+x7*(1.24545)+x8*(-0.066115)+x9*(-0.25324)+x10*(-0.147339)+x11*(0.101752)+x12*(0.379289)+x13*(-0.223658)+x14*(0.0754272)+x15*(0.00290909)+x16*(0.390339)+(-0.200694)); 
}

void FNN_3_16_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.29857)+x1*(0.251347)+x2*(1.65009)+x3*(0.767896)+x4*(0.189173)+x5*(-1.06702)+x6*(0.86871)+x7*(-0.257184)+x8*(-0.50886)+x9*(-0.977174)+x10*(2.19989)+x11*(0.392973)+x12*(0.140397)+x13*(-0.0736363)+x14*(0.239956)+x15*(-0.0762356)+x16*(-0.98066)+x17*(1.17569)+x18*(0.133628)+x19*(-0.23014)+(0.0741689)); 
    h1 = tanh(x0*(0.147676)+x1*(0.151227)+x2*(-0.790887)+x3*(0.0517088)+x4*(-0.91578)+x5*(-0.433151)+x6*(-0.226575)+x7*(-0.130036)+x8*(0.670604)+x9*(-1.42736)+x10*(-1.17666)+x11*(-0.779796)+x12*(0.317026)+x13*(-0.0684479)+x14*(0.104957)+x15*(-0.945492)+x16*(0.685418)+x17*(-0.384942)+x18*(-0.0160248)+x19*(-0.565243)+(0.0668365)); 
    h2 = tanh(x0*(-0.399513)+x1*(0.20085)+x2*(-1.53762)+x3*(-0.183937)+x4*(0.218477)+x5*(0.188627)+x6*(0.786444)+x7*(-0.00504111)+x8*(-0.0220409)+x9*(-0.846131)+x10*(-2.01598)+x11*(-1.18389)+x12*(0.321198)+x13*(-0.010678)+x14*(-0.787506)+x15*(0.429206)+x16*(1.53132)+x17*(-1.55799)+x18*(-0.546291)+x19*(1.12872)+(0.0476793)); 
    h3 = tanh(x0*(-0.141937)+x1*(0.0407546)+x2*(0.088397)+x3*(0.416555)+x4*(-0.743465)+x5*(-0.05012)+x6*(2.53034)+x7*(0.119293)+x8*(0.237991)+x9*(-2.1539)+x10*(0.586079)+x11*(-0.539736)+x12*(0.113368)+x13*(0.290945)+x14*(-0.900358)+x15*(-0.887892)+x16*(0.202423)+x17*(-0.86884)+x18*(-0.427932)+x19*(-0.385117)+(-0.00972022)); 
    h4 = tanh(x0*(0.407465)+x1*(-0.638142)+x2*(0.547532)+x3*(0.311972)+x4*(-0.0175746)+x5*(-0.553324)+x6*(0.0266221)+x7*(0.418851)+x8*(0.978998)+x9*(-0.0826214)+x10*(-0.486181)+x11*(0.391348)+x12*(-0.45668)+x13*(0.109226)+x14*(0.156292)+x15*(0.538215)+x16*(-0.138118)+x17*(0.834167)+x18*(0.154047)+x19*(-0.0123846)+(-0.245479)); 
    h5 = tanh(x0*(-0.0738582)+x1*(-0.555651)+x2*(1.35679)+x3*(0.152896)+x4*(-0.118728)+x5*(0.594793)+x6*(-0.85547)+x7*(0.32366)+x8*(-0.825339)+x9*(2.36359)+x10*(0.433295)+x11*(0.595536)+x12*(0.100074)+x13*(-0.160697)+x14*(-1.52829)+x15*(0.443183)+x16*(-2.00934)+x17*(0.0521171)+x18*(0.295227)+x19*(0.141783)+(-0.185786)); 
    h6 = tanh(x0*(0.128186)+x1*(0.26509)+x2*(0.61727)+x3*(0.439793)+x4*(-0.116048)+x5*(1.07882)+x6*(-0.631989)+x7*(0.425622)+x8*(-0.311852)+x9*(-0.304389)+x10*(0.790042)+x11*(0.0124883)+x12*(-0.681509)+x13*(-0.231099)+x14*(-0.378456)+x15*(0.706745)+x16*(0.125534)+x17*(0.673913)+x18*(0.249992)+x19*(0.069894)+(-0.27869)); 
    h7 = tanh(x0*(0.197738)+x1*(-0.0359803)+x2*(0.633563)+x3*(0.211424)+x4*(0.239545)+x5*(0.0111623)+x6*(-0.351092)+x7*(0.333162)+x8*(0.192253)+x9*(0.045372)+x10*(0.486313)+x11*(0.0724398)+x12*(-0.323192)+x13*(-0.0664926)+x14*(0.387089)+x15*(0.854043)+x16*(-0.480625)+x17*(0.552348)+x18*(0.319313)+x19*(-0.146086)+(0.0105083)); 
    h8 = tanh(x0*(0.316634)+x1*(0.0172578)+x2*(0.232274)+x3*(0.00912867)+x4*(0.280511)+x5*(-0.322582)+x6*(-0.146347)+x7*(-0.0515232)+x8*(0.74698)+x9*(0.00158545)+x10*(0.475497)+x11*(0.0863368)+x12*(-0.0243191)+x13*(-0.333383)+x14*(0.184122)+x15*(0.599337)+x16*(-0.0625579)+x17*(0.592081)+x18*(0.291141)+x19*(0.0604243)+(-0.315162)); 
    h9 = tanh(x0*(0.028886)+x1*(-0.57265)+x2*(-0.124055)+x3*(0.462519)+x4*(-0.109161)+x5*(0.0883943)+x6*(0.248163)+x7*(-0.311968)+x8*(0.194347)+x9*(-0.151186)+x10*(0.465675)+x11*(-0.0478678)+x12*(-0.0390892)+x13*(-0.210315)+x14*(0.372781)+x15*(-0.473389)+x16*(-0.297777)+x17*(0.172526)+x18*(0.273972)+x19*(-0.0793781)+(0.18876)); 
}

void FNN_3_16_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(1.27671)+x1*(0.376821)+x2*(1.05988)+x3*(-0.889987)+x4*(-0.517571)+x5*(0.468585)+x6*(-0.881964)+x7*(-0.519625)+x8*(0.382161)+x9*(0.426563)+(0.314196); 
    h1 = x0*(1.07977)+x1*(0.225759)+x2*(0.907877)+x3*(-0.636212)+x4*(0.141035)+x5*(0.36455)+x6*(-0.163639)+x7*(-0.945392)+x8*(-0.217844)+x9*(0.409792)+(0.252464); 
    h2 = x0*(1.30755)+x1*(0.364348)+x2*(1.09971)+x3*(-0.967352)+x4*(-0.216361)+x5*(0.518514)+x6*(-0.825764)+x7*(-0.11092)+x8*(-0.565754)+x9*(0.635596)+(0.333598); 
}

void FNN_3_16(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_3_16_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_3_16_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_3_16_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_16(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_3_16(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_14(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x6 < lowBound ) {
        ModelKDTree_15(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x6 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_15(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_16(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x6 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_16(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void ModelKDTree_8(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x5 < lowBound ) {
        ModelKDTree_9(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x5 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_9(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_14(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x5 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_14(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void ModelKDTree_0(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x9 < lowBound ) {
        ModelKDTree_1(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x9 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_1(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_8(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x9 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_8(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void Model(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,    
out float h0,out float h1,out float h2)
{
    // map to unit hyperbox
    x0 = (x0 - (0.000148)) / (0.999852);
    x1 = (x1 - (0.000148)) / (0.999852);
    x2 = (x2 - (0.000148)) / (0.999852);
    x3 = (x3 - (-0.375)) / (0.75);
    x4 = (x4 - (0.374756)) / (0.750244);
    x5 = (x5 - (-0.012497)) / (0.024994);
    x6 = (x6 - (-0.999512)) / (1.99805);
    x7 = (x7 - (-0.996094)) / (1.99512);
    x8 = (x8 - (-0.999512)) / (1.99902);
    x9 = (x9 - (-1)) / (2);
    x10 = (x10 - (-1)) / (2);
    x11 = (x11 - (-1)) / (2);
    x12 = (x12 - (1)) / (1);
    x13 = (x13 - (1)) / (1);
    x14 = (x14 - (1)) / (1);
    x15 = (x15 - (1.5)) / (1);
    x16 = (x16 - (0.299805)) / (1);

    ModelKDTree_0(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);

    // map back
    h0 = h0 * (0.998883) + (0);
    h1 = h1 * (0.998818) + (0);
    h2 = h2 * (0.998882) + (0);
}

