#version 330 core

// ----------------- in out

out vec4 FragColor;

in VS_OUT {
    vec3 FragPos;
    vec3 Normal;
    vec2 TexCoords;
    vec3 Tangent;
} fs_in;

// ----------------- const

#define MAX_POINT_LIGHTS 8
const float PI = 3.14159265359;
const float INV_PI = 0.31830988618;

// array of offset direction for sampling
const vec3 gridSamplingDisk[20] = vec3[]
(
   vec3(1, 1,  1), vec3( 1, -1,  1), vec3(-1, -1,  1), vec3(-1, 1,  1), 
   vec3(1, 1, -1), vec3( 1, -1, -1), vec3(-1, -1, -1), vec3(-1, 1, -1),
   vec3(1, 1,  0), vec3( 1, -1,  0), vec3(-1, -1,  0), vec3(-1, 1,  0),
   vec3(1, 0,  1), vec3(-1,  0,  1), vec3( 1,  0, -1), vec3(-1, 0, -1),
   vec3(0, 1,  1), vec3( 0, -1,  1), vec3( 0, -1, -1), vec3( 0, 1, -1)
);

// ----------------- struct

struct BSDF_FrostedGlass {
	vec3 colorFactor;
	bool haveColorTexture;
    sampler2D colorTexture;

	float roughnessFactor;
	bool haveRoughnessTexture;
    sampler2D roughnessTexture;
	
	bool haveAOTexture;
    sampler2D aoTexture;

	bool haveNormalTexture;
	sampler2D normalTexture;

	float ior;
};

// ----------------- Uniform

// 48
struct PointLight {
    vec3 position;	// 12	0
    vec3 L;			// 12	16
    float linear;	// 4	28
    float quadratic;// 4	32
};

// 160
layout (std140) uniform Camera{
	mat4 view;			// 64	0	64
	mat4 projection;	// 64	64	64
	vec3 viewPos;		// 12	128	144
	float nearPlane;	// 4	144	148
	float farPlane;		// 4	148	152
	float fov;			// 4	152	156
	float ar;			// 4	156	160
};

// 400
layout (std140) uniform PointLights{
	int numLight;// 16
	PointLight pointLights[MAX_POINT_LIGHTS];// 48 * MAX_POINT_LIGHTS = 48 * 8
};

uniform BSDF_FrostedGlass bsdf;

uniform samplerCube pointLightDepthMap0;
uniform samplerCube pointLightDepthMap1;
uniform samplerCube pointLightDepthMap2;
uniform samplerCube pointLightDepthMap3;
uniform samplerCube pointLightDepthMap4;
uniform samplerCube pointLightDepthMap5;
uniform samplerCube pointLightDepthMap6;
uniform samplerCube pointLightDepthMap7;

uniform float lightFar;

uniform int mode;

uniform float interpolateRatio; // (0, 1]

// ----------------- declaration

vec3 CalcBumpedNormal(vec3 normal, vec3 tangent, sampler2D normalTexture, vec2 texcoord);

float GGX_D(vec3 norm, vec3 h, float alpha);
float Fr(vec3 v, vec3 h, float ior);
float GGX_G1(vec3 norm, vec3 v, vec3 h, float alpha);
float GGX_G(vec3 norm, vec3 wo, vec3 wi, vec3 h, float alpha);
vec3 BSDF(vec3 norm, vec3 wo, vec3 wi, vec3 color, float roughness, float ao, float ior);

float Visibility(vec3 lightToFrag, int id);
float Visibility(vec3 lightToFrag, samplerCube depthMap);

float tanh(float x) {
	float expZ = exp(x);
    float invExpZ = 1 / expZ;
    return (expZ - invExpZ) / (expZ + invExpZ);
}

float smootherstep(float x){
	return ((6*x - 15)*x + 10) * x*x*x;
}

// template declaration
void Model
(
	in float directIllum_R,
	in float directIllum_G,
	in float directIllum_B,
	in float pos_x,
	in float pos_y,
	in float pos_z,
	in float viewDir_x,
	in float viewDir_y,
	in float viewDir_z,
	in float normal_x,
	in float normal_y,
	in float normal_z,
	in float matColor_R,
	in float matColor_G,
	in float matColor_B,
	in float IOR,
	in float roughness,
	
	out float indirectIllum_R,
	out float indirectIllum_G,
	out float indirectIllum_B
);

// ----------------- main

void main() {
	// get attr
	vec3 color = bsdf.colorFactor;
	if(bsdf.haveColorTexture) {
		color *= texture(bsdf.colorTexture, fs_in.TexCoords).xyz;
	}

	float roughness = bsdf.roughnessFactor;
	if(bsdf.haveRoughnessTexture) {
		roughness *= texture(bsdf.roughnessTexture, fs_in.TexCoords).x;
	}

	float ao = 1.0f;
	if(bsdf.haveAOTexture) {
		ao *= texture(bsdf.aoTexture, fs_in.TexCoords).x;
	}

	vec3 wo = normalize(viewPos - fs_in.FragPos);

	vec3 norm = normalize(fs_in.Normal);
	if(bsdf.haveNormalTexture) {
		norm = CalcBumpedNormal(norm, normalize(fs_in.Tangent), bsdf.normalTexture, fs_in.TexCoords);
	}
	
	// sample point light
	vec3 result = vec3(0);
    for(int i = 0; i < numLight; i++) {
		vec3 fragToLight = pointLights[i].position - fs_in.FragPos;
		float dist2 = dot(fragToLight, fragToLight);
		float dist = sqrt(dist2);
		vec3 wi = fragToLight / dist;
		
		float visibility = Visibility(-fragToLight, i);
		if(visibility==0)
			continue;

		vec3 f = BSDF(norm, wo, wi, color, roughness, ao, bsdf.ior);

		float cosTheta = max(dot(wi, norm), 0);
		
		float attenuation = 1.0f + pointLights[i].linear * dist + pointLights[i].quadratic * dist2;
		
		result += visibility * cosTheta / attenuation * f * pointLights[i].L;
	}
	
	vec3 indirectIllum;
	
	Model
	(
		// input
		result.r,
		result.g,
		result.b,
		fs_in.FragPos.x,
		fs_in.FragPos.y,
		fs_in.FragPos.z,
		wo.x,
		wo.y,
		wo.z,
		norm.x,
		norm.y,
		norm.z,
		color.r,
		color.g,
		color.b,
		bsdf.ior,
		roughness,
		
		// output
		indirectIllum.r,
		indirectIllum.g,
		indirectIllum.b
	);
	
	indirectIllum = clamp(indirectIllum, 0, 1);
	
	// gamma correction
	vec3 finalColor;
	if(mode==0)
		finalColor = result;
	else if(mode==1)
		finalColor = indirectIllum;
	else
		finalColor = result + indirectIllum;
		
    FragColor = vec4(sqrt(finalColor), 1.0);
}

// ----------------- definition

float GGX_D(vec3 norm, vec3 h, float alpha) {
	float HoN = dot(h, norm);
	if (HoN <= 0)
		return 0;

	float cos2 = HoN * HoN;
	float alpha2 = alpha * alpha;

	float root = alpha / (cos2 * (alpha2 - 1) + 1);

	return INV_PI * root * root;
}

float Fr(vec3 v, vec3 h, float ior) {
	// angle in low ior media
	float cosTheta = dot(v, h);
	bool entering = cosTheta > 0.0f;
	
	if (!entering) {
		vec3 vInLowIOR = refract(-v, -h, ior);
		if (vInLowIOR == vec3(0))
			return 1;

		cosTheta = dot(vInLowIOR, h);
	}

	float R0 = pow((ior - 1) / (ior + 1), 2);

	float Fr = R0 + (1 - R0) * pow((1 - cosTheta), 5);

	return Fr;
}

float GGX_G1(vec3 norm, vec3 v, vec3 h, float alpha) {
	float NoV = dot(norm, v);
	float HoV = dot(v, h);
	if (NoV * HoV <= 0)
		return 0;

	float alpha2 = alpha * alpha;
	float tan2 = 1.f / (NoV * NoV) - 1.f;
	return 2.f / (1.f + sqrt(1 + alpha2 * tan2));
}

float GGX_G(vec3 norm, vec3 wo, vec3 wi, vec3 h, float alpha) {
	return GGX_G1(norm, wo, h, alpha) * GGX_G1(norm, wi, h, alpha);
}

vec3 BSDF(vec3 norm, vec3 wo, vec3 wi, vec3 color, float roughness, float ao, float ior) {
	float NoWo = dot(norm, wo);
	float NoWi = dot(norm, wi);
	if(NoWo == 0 || NoWi == 0)
		return vec3(0);

	bool entering = NoWo > 0;
	bool isReflect = NoWo * NoWi > 0;

	float alpha = roughness * roughness;

	if (isReflect) {
		vec3 h = normalize(wo + wi);

		h *= sign(dot(h, norm));// let h point to outside
		float bsdfVal = Fr(wo, h, ior) * GGX_D(norm, h, alpha) * GGX_G(norm, wo, wi, h, alpha) / abs(4.f * NoWo * NoWi);
		return bsdfVal * color;
	}
	else {
		float etai = 1.f, etat = ior;
		if (!entering){
			// swap(etai, etat);
			float tmp = etai;
			etai = etat;
			etat = tmp;
		}

		vec3 h = -normalize(etai * wo + etat * wi);

		float HoWo = dot(h, wo);
		float HoWi = dot(h, wi);
		float sqrtDenom = etai * HoWo + etat * HoWi;

		float factor = abs(HoWo * HoWi / (NoWo * NoWi));

		h *= sign(dot(h,norm));// let h point to outside
		float bsdfVal = factor * ((1 - Fr(wo, h, ior)) * GGX_D(norm, h, alpha) * GGX_G(norm, wo, wi, h, alpha) * etat * etat) / (sqrtDenom * sqrtDenom);
		return bsdfVal * color;
	}
}

vec3 CalcBumpedNormal(vec3 normal, vec3 tangent, sampler2D normalTexture, vec2 texcoord) {
    tangent = normalize(tangent - dot(tangent, normal) * normal);
    vec3 bitangent = cross(tangent, normal);
    vec3 bumpMapNormal = texture(normalTexture, texcoord).xyz;
    bumpMapNormal = 2.0 * bumpMapNormal - 1.0;
    mat3 TBN = mat3(tangent, bitangent, normal);
    vec3 newNormal = TBN * bumpMapNormal;
    newNormal = normalize(newNormal);
    return newNormal;
}

float Visibility(vec3 lightToFrag, int id){
	if(id == 0) {
		return Visibility(lightToFrag, pointLightDepthMap0);
	} else if(id == 1) {
		return Visibility(lightToFrag, pointLightDepthMap1);
	} else if(id == 2) {
		return Visibility(lightToFrag, pointLightDepthMap2);
	} else if(id == 3) {
		return Visibility(lightToFrag, pointLightDepthMap3);
	} else if(id == 4) {
		return Visibility(lightToFrag, pointLightDepthMap4);
	} else if(id == 5) {
		return Visibility(lightToFrag, pointLightDepthMap5);
	} else if(id == 6) {
		return Visibility(lightToFrag, pointLightDepthMap6);
	} else if(id == 7) {
		return Visibility(lightToFrag, pointLightDepthMap7);
	}else 
		return 1;// not support id
}

float Visibility(vec3 lightToFrag, samplerCube depthMap) {
	float currentDepth = length(lightToFrag);
	float bias = 0.08;
	int samples = 20;
	float shadow = 0.0;
	float viewDistance = length(viewPos - fs_in.FragPos);
	float diskRadius = (1.0 + (viewDistance / lightFar)) / 50.0;
	for(int i = 0; i < samples; ++i) {
		float closestDepth = lightFar * texture(depthMap, lightToFrag + gridSamplingDisk[i] * diskRadius).r;
		shadow += smoothstep(closestDepth, closestDepth + bias, currentDepth);
	}
	shadow /= float(samples);
	return 1 - shadow;
}

void FNN_14_2_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.235911)+x1*(-0.0547483)+x2*(-0.0456602)+x3*(0.175681)+x4*(0.183908)+x5*(-0.0412029)+x6*(0.258398)+x7*(0.351)+x8*(-0.0217651)+x9*(0.302821)+x10*(-0.133983)+x11*(0.153032)+x12*(0.198041)+x13*(-0.374279)+x14*(-0.332853)+x15*(-0.239093)+x16*(-0.158456)+(-0.000734218)); 
    h1 = tanh(x0*(-0.0065267)+x1*(-0.0447796)+x2*(-0.0118241)+x3*(-0.262991)+x4*(-0.218338)+x5*(-0.0628891)+x6*(0.332473)+x7*(-0.330964)+x8*(-0.129613)+x9*(-0.0927485)+x10*(0.124188)+x11*(-0.286416)+x12*(-0.0391234)+x13*(0.214179)+x14*(-0.107105)+x15*(0.389548)+x16*(-0.143385)+(-0.0626868)); 
    h2 = tanh(x0*(0.364388)+x1*(0.391028)+x2*(0.332607)+x3*(0.2627)+x4*(-0.196433)+x5*(-0.170178)+x6*(-0.0380621)+x7*(0.219622)+x8*(0.280194)+x9*(-0.151052)+x10*(0.293876)+x11*(-0.0111342)+x12*(-0.0451015)+x13*(0.0407822)+x14*(6.05881e-05)+x15*(-0.296693)+x16*(0.0904821)+(0.0985503)); 
    h3 = tanh(x0*(-0.21647)+x1*(0.215384)+x2*(0.105355)+x3*(0.217636)+x4*(0.0792968)+x5*(0.0453273)+x6*(-0.137451)+x7*(0.00864726)+x8*(0.0365099)+x9*(0.31232)+x10*(-0.226821)+x11*(0.103448)+x12*(0.0415535)+x13*(0.119936)+x14*(0.20398)+x15*(0.200001)+x16*(0.0822753)+(-0.0306785)); 
    h4 = tanh(x0*(0.111467)+x1*(-0.475565)+x2*(-0.0675612)+x3*(-0.18748)+x4*(-0.400026)+x5*(-0.181962)+x6*(0.162545)+x7*(0.014155)+x8*(0.346387)+x9*(0.0344927)+x10*(0.330471)+x11*(0.178514)+x12*(0.0527844)+x13*(-0.206775)+x14*(0.0671901)+x15*(0.364593)+x16*(-0.0323828)+(0.128771)); 
    h5 = tanh(x0*(-0.209271)+x1*(0.168471)+x2*(-0.27824)+x3*(0.0692692)+x4*(0.0156999)+x5*(-0.275264)+x6*(-0.28897)+x7*(0.0147223)+x8*(-0.0659221)+x9*(0.00755095)+x10*(0.25922)+x11*(0.37567)+x12*(-0.392521)+x13*(0.0425934)+x14*(-0.0705835)+x15*(-0.121625)+x16*(0.275179)+(-0.0982983)); 
    h6 = tanh(x0*(0.22197)+x1*(-0.134796)+x2*(-0.280338)+x3*(0.305096)+x4*(0.23483)+x5*(-0.256676)+x6*(0.167122)+x7*(0.0616041)+x8*(-0.2985)+x9*(-0.170965)+x10*(0.11129)+x11*(0.220147)+x12*(0.155004)+x13*(0.00356245)+x14*(0.308962)+x15*(0.135771)+x16*(0.181289)+(0.0633812)); 
    h7 = tanh(x0*(0.344367)+x1*(-0.347654)+x2*(0.24633)+x3*(-0.281444)+x4*(0.122285)+x5*(0.225636)+x6*(-0.263604)+x7*(-0.447721)+x8*(-0.247775)+x9*(0.186453)+x10*(-0.0924139)+x11*(0.129128)+x12*(-0.0360158)+x13*(0.137178)+x14*(0.0477923)+x15*(0.309423)+x16*(0.163826)+(0.106963)); 
    h8 = tanh(x0*(-0.169409)+x1*(-0.382417)+x2*(-0.0740488)+x3*(-0.184552)+x4*(0.22947)+x5*(0.129)+x6*(-0.0291621)+x7*(-0.0571585)+x8*(-0.270607)+x9*(-0.151213)+x10*(0.366421)+x11*(0.225435)+x12*(0.162529)+x13*(0.0830123)+x14*(0.379148)+x15*(0.232082)+x16*(0.261812)+(0.0878817)); 
    h9 = tanh(x0*(0.11298)+x1*(0.300968)+x2*(-0.253091)+x3*(-0.104728)+x4*(0.173111)+x5*(-0.320205)+x6*(0.341249)+x7*(0.231444)+x8*(0.293934)+x9*(-0.289421)+x10*(-0.0679665)+x11*(-0.0828508)+x12*(0.313134)+x13*(0.237056)+x14*(0.3951)+x15*(0.371995)+x16*(-0.15537)+(-0.0425063)); 
    h10 = tanh(x0*(-0.309443)+x1*(0.401783)+x2*(-0.251418)+x3*(0.35267)+x4*(0.109369)+x5*(-0.274796)+x6*(-0.245798)+x7*(-0.150835)+x8*(-0.28901)+x9*(0.378514)+x10*(0.358072)+x11*(-0.0270629)+x12*(0.342598)+x13*(-0.34821)+x14*(0.0531289)+x15*(0.145837)+x16*(0.392089)+(0.0103913)); 
    h11 = tanh(x0*(0.307256)+x1*(-0.24747)+x2*(0.164256)+x3*(-0.20791)+x4*(-0.157645)+x5*(-0.198309)+x6*(-0.347142)+x7*(0.129538)+x8*(0.0871494)+x9*(0.0147117)+x10*(0.099948)+x11*(-0.0153146)+x12*(-0.0895194)+x13*(-0.384797)+x14*(0.0873303)+x15*(0.247476)+x16*(0.368545)+(-0.102825)); 
    h12 = tanh(x0*(-0.154506)+x1*(-0.307771)+x2*(0.371704)+x3*(0.199392)+x4*(-0.255071)+x5*(0.0736042)+x6*(-0.375605)+x7*(-0.00565691)+x8*(-0.0211234)+x9*(-0.0818463)+x10*(-0.0662479)+x11*(-0.367603)+x12*(-0.0884925)+x13*(0.0480271)+x14*(0.104917)+x15*(-0.114717)+x16*(0.147188)+(-0.00348778)); 
    h13 = tanh(x0*(0.0311415)+x1*(-0.235688)+x2*(0.0654697)+x3*(0.0291918)+x4*(0.217784)+x5*(0.21457)+x6*(-0.22818)+x7*(-0.294515)+x8*(0.131761)+x9*(-0.0668963)+x10*(0.153197)+x11*(0.111347)+x12*(-0.0600942)+x13*(-0.29891)+x14*(0.294479)+x15*(-0.0162744)+x16*(0.233799)+(0.0488558)); 
    h14 = tanh(x0*(0.269101)+x1*(-0.0510415)+x2*(0.0362084)+x3*(-0.252862)+x4*(-0.488602)+x5*(-0.109636)+x6*(0.141395)+x7*(-0.108454)+x8*(0.207334)+x9*(0.280433)+x10*(0.0945842)+x11*(-0.0779129)+x12*(0.0220541)+x13*(0.364275)+x14*(-0.19987)+x15*(0.372239)+x16*(0.00404027)+(0.0183271)); 
    h15 = tanh(x0*(-0.056276)+x1*(0.171617)+x2*(-0.0312825)+x3*(-0.225535)+x4*(-0.289719)+x5*(-0.0101006)+x6*(-0.117055)+x7*(0.359159)+x8*(-0.0172854)+x9*(0.117914)+x10*(0.0751704)+x11*(0.3562)+x12*(0.188728)+x13*(-0.29982)+x14*(-0.268869)+x15*(-0.100894)+x16*(-0.229348)+(-0.049115)); 
    h16 = tanh(x0*(-0.0540817)+x1*(0.223963)+x2*(0.320295)+x3*(-0.425883)+x4*(-0.141378)+x5*(0.206724)+x6*(0.230215)+x7*(0.104063)+x8*(-0.219079)+x9*(0.0663482)+x10*(-0.209488)+x11*(-0.139993)+x12*(-0.373465)+x13*(-0.143104)+x14*(-0.0447292)+x15*(-0.168042)+x16*(0.394865)+(0.0184347)); 
    h17 = tanh(x0*(0.25163)+x1*(0.00593512)+x2*(0.355821)+x3*(-0.248228)+x4*(0.538328)+x5*(0.161581)+x6*(0.320633)+x7*(0.456662)+x8*(-0.232089)+x9*(-0.315856)+x10*(-0.212058)+x11*(-0.0831923)+x12*(0.227901)+x13*(-0.0245547)+x14*(0.268134)+x15*(0.367562)+x16*(-0.317141)+(0.00135047)); 
    h18 = tanh(x0*(0.217614)+x1*(-0.41073)+x2*(0.248235)+x3*(0.222282)+x4*(0.150916)+x5*(-0.387455)+x6*(0.196553)+x7*(-0.0703585)+x8*(-0.0358314)+x9*(-0.0211196)+x10*(-0.331013)+x11*(-0.346995)+x12*(-0.369396)+x13*(0.32479)+x14*(0.109224)+x15*(-0.0186099)+x16*(-0.24745)+(-0.0465034)); 
    h19 = tanh(x0*(-0.364482)+x1*(-0.239437)+x2*(0.345946)+x3*(0.0721991)+x4*(-0.267345)+x5*(0.0142685)+x6*(-0.170814)+x7*(-0.0132151)+x8*(0.360557)+x9*(-0.169305)+x10*(-0.0363523)+x11*(-0.337921)+x12*(-0.388171)+x13*(0.11282)+x14*(-0.0260725)+x15*(-0.223304)+x16*(0.0348131)+(-0.00953413)); 
}

void FNN_14_2_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(0.171984)+x1*(-0.00518183)+x2*(-0.246982)+x3*(-0.301534)+x4*(0.374106)+x5*(-0.312793)+x6*(0.337358)+x7*(-0.264389)+x8*(0.0986533)+x9*(-0.294978)+x10*(0.277034)+x11*(-0.201421)+x12*(-0.29881)+x13*(0.0735718)+x14*(0.0102647)+x15*(0.237274)+x16*(-0.256589)+x17*(0.28544)+x18*(0.119329)+x19*(-0.0604105)+(0.0289271)); 
    h1 = tanh(x0*(-0.25862)+x1*(-0.159555)+x2*(-0.24945)+x3*(0.271649)+x4*(0.219504)+x5*(-0.26441)+x6*(-0.160039)+x7*(0.387887)+x8*(0.258273)+x9*(-0.310444)+x10*(0.273477)+x11*(0.0679348)+x12*(0.00598925)+x13*(-0.433801)+x14*(-0.245851)+x15*(0.351488)+x16*(-0.150165)+x17*(-0.199344)+x18*(0.193149)+x19*(0.0192596)+(-0.0488802)); 
    h2 = tanh(x0*(0.155923)+x1*(-0.0175649)+x2*(0.242742)+x3*(0.22187)+x4*(-0.0641175)+x5*(0.261595)+x6*(0.234538)+x7*(-0.297899)+x8*(-0.109775)+x9*(0.0824842)+x10*(0.0805508)+x11*(-0.0786369)+x12*(-0.346464)+x13*(-0.311413)+x14*(0.404548)+x15*(0.227954)+x16*(0.328259)+x17*(0.167958)+x18*(-0.063703)+x19*(0.0455705)+(-0.090337)); 
    h3 = tanh(x0*(0.291439)+x1*(0.125836)+x2*(0.142756)+x3*(-0.194124)+x4*(-0.129963)+x5*(0.172964)+x6*(0.0478564)+x7*(0.214964)+x8*(-0.248658)+x9*(-0.300074)+x10*(-0.339887)+x11*(-0.178448)+x12*(0.285477)+x13*(0.0511625)+x14*(-0.117319)+x15*(0.436451)+x16*(-0.357221)+x17*(0.0516407)+x18*(-0.205772)+x19*(-0.137983)+(0.0461323)); 
    h4 = tanh(x0*(-0.379129)+x1*(0.21859)+x2*(-0.516973)+x3*(0.278918)+x4*(0.334797)+x5*(-0.157631)+x6*(-0.289431)+x7*(0.407875)+x8*(0.338843)+x9*(-0.448854)+x10*(-0.405241)+x11*(0.0942697)+x12*(0.0446347)+x13*(-0.233805)+x14*(-0.327695)+x15*(-0.202248)+x16*(0.385164)+x17*(0.0427047)+x18*(0.164657)+x19*(-0.348999)+(0.0208935)); 
    h5 = tanh(x0*(0.288824)+x1*(0.0673746)+x2*(-0.091965)+x3*(0.0571574)+x4*(0.0635175)+x5*(0.390631)+x6*(-0.260384)+x7*(0.374952)+x8*(-0.157325)+x9*(-0.385638)+x10*(-0.0539394)+x11*(0.425158)+x12*(-0.0954145)+x13*(-0.260751)+x14*(-0.404991)+x15*(-0.0320552)+x16*(0.289716)+x17*(0.48494)+x18*(-0.253305)+x19*(-0.513664)+(-0.0509453)); 
    h6 = tanh(x0*(-0.112557)+x1*(-0.366193)+x2*(0.437087)+x3*(0.031578)+x4*(-0.202119)+x5*(0.094635)+x6*(-0.351113)+x7*(0.328058)+x8*(0.1805)+x9*(-0.286968)+x10*(0.204949)+x11*(0.127751)+x12*(-0.363751)+x13*(0.144084)+x14*(0.464375)+x15*(-0.327885)+x16*(0.212725)+x17*(-0.356772)+x18*(-0.257734)+x19*(-0.353451)+(0.037869)); 
    h7 = tanh(x0*(0.191992)+x1*(-0.346794)+x2*(0.211733)+x3*(-0.243321)+x4*(0.4308)+x5*(-0.283129)+x6*(0.3007)+x7*(0.243905)+x8*(0.0289611)+x9*(0.272882)+x10*(-0.129185)+x11*(-0.511011)+x12*(0.100163)+x13*(0.00468839)+x14*(-0.071255)+x15*(-0.28179)+x16*(0.0667567)+x17*(0.45325)+x18*(0.159659)+x19*(0.259418)+(0.0937941)); 
    h8 = tanh(x0*(-0.424419)+x1*(-0.292356)+x2*(0.146467)+x3*(0.0271693)+x4*(0.39795)+x5*(0.402188)+x6*(-0.0180702)+x7*(0.0716861)+x8*(0.136765)+x9*(-0.38882)+x10*(0.246351)+x11*(0.280573)+x12*(-0.196863)+x13*(0.115178)+x14*(0.383343)+x15*(-0.0531838)+x16*(-0.0152957)+x17*(0.109769)+x18*(0.0611976)+x19*(-0.0352106)+(0.0256217)); 
    h9 = tanh(x0*(0.163798)+x1*(-0.220005)+x2*(0.321404)+x3*(0.3207)+x4*(0.223778)+x5*(-0.00244513)+x6*(0.195825)+x7*(0.227421)+x8*(0.286543)+x9*(-0.122643)+x10*(-0.186296)+x11*(-0.360027)+x12*(0.0531476)+x13*(-0.0209793)+x14*(-0.200407)+x15*(0.145163)+x16*(0.370098)+x17*(0.0873442)+x18*(-0.280085)+x19*(-0.153514)+(0.0716135)); 
}

void FNN_14_2_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(0.0384842)+x1*(-0.314902)+x2*(-0.269492)+x3*(0.193443)+x4*(0.452803)+x5*(-0.390841)+x6*(0.22328)+x7*(0.427633)+x8*(0.257738)+x9*(0.550582)+(0.00749968); 
    h1 = x0*(0.403882)+x1*(-0.262769)+x2*(-0.119469)+x3*(0.144059)+x4*(0.0735709)+x5*(-0.198094)+x6*(0.601475)+x7*(0.375786)+x8*(0.184396)+x9*(0.0696738)+(0.0265727); 
    h2 = x0*(-0.0893505)+x1*(0.407992)+x2*(-0.0578007)+x3*(0.0302812)+x4*(0.223267)+x5*(-0.505402)+x6*(0.346986)+x7*(0.3528)+x8*(-0.337468)+x9*(0.494636)+(0.0356783); 
}

void FNN_14_2(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_14_2_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_14_2_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_14_2_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_2(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_14_2(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_14_4_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(0.41279)+x1*(0.374115)+x2*(-0.264548)+x3*(-0.151608)+x4*(0.163206)+x5*(-0.446224)+x6*(0.0211551)+x7*(-0.179413)+x8*(-0.249509)+x9*(-0.0383325)+x10*(-0.191549)+x11*(-0.153639)+x12*(0.086836)+x13*(-0.380276)+x14*(-0.140492)+x15*(0.340705)+x16*(-0.163529)+(-0.0709142)); 
    h1 = tanh(x0*(-0.0453019)+x1*(-0.352677)+x2*(0.285672)+x3*(0.254436)+x4*(-0.0587886)+x5*(-0.167155)+x6*(-0.15845)+x7*(-0.201229)+x8*(-0.00416644)+x9*(-0.28705)+x10*(0.125947)+x11*(-0.316791)+x12*(-0.175008)+x13*(0.307602)+x14*(0.188365)+x15*(-0.234597)+x16*(0.140603)+(0.0934155)); 
    h2 = tanh(x0*(-0.315705)+x1*(0.332958)+x2*(0.451018)+x3*(0.268754)+x4*(-0.0731065)+x5*(-0.504719)+x6*(0.242939)+x7*(-0.126478)+x8*(-0.10876)+x9*(-0.00209051)+x10*(-0.186091)+x11*(0.268069)+x12*(0.187966)+x13*(-0.38303)+x14*(-0.339609)+x15*(0.117854)+x16*(-0.182345)+(0.0839576)); 
    h3 = tanh(x0*(-0.172106)+x1*(-0.0104635)+x2*(0.0980472)+x3*(-0.544196)+x4*(-0.0713865)+x5*(0.0964254)+x6*(-0.102702)+x7*(-0.460873)+x8*(0.135751)+x9*(0.149461)+x10*(0.221704)+x11*(-0.381434)+x12*(0.183158)+x13*(0.0506696)+x14*(-0.389897)+x15*(0.21748)+x16*(0.11975)+(0.0971011)); 
    h4 = tanh(x0*(-0.228526)+x1*(0.0203769)+x2*(0.0128227)+x3*(-0.34009)+x4*(0.23115)+x5*(-0.39665)+x6*(-0.263173)+x7*(-0.0205459)+x8*(-0.0840657)+x9*(0.14823)+x10*(0.321683)+x11*(-0.260747)+x12*(-0.0501482)+x13*(0.116248)+x14*(-0.299768)+x15*(0.37023)+x16*(0.375681)+(0.0582295)); 
    h5 = tanh(x0*(-0.290418)+x1*(-0.202558)+x2*(-0.116416)+x3*(0.355151)+x4*(0.0277197)+x5*(-0.359386)+x6*(0.308056)+x7*(0.244047)+x8*(-0.206075)+x9*(0.19078)+x10*(0.0440255)+x11*(-0.113673)+x12*(-0.0208725)+x13*(0.305975)+x14*(-0.10143)+x15*(-0.399969)+x16*(0.259585)+(0.162206)); 
    h6 = tanh(x0*(0.854827)+x1*(0.877102)+x2*(0.512827)+x3*(-0.0725952)+x4*(0.384493)+x5*(0.297405)+x6*(-0.00650401)+x7*(0.128431)+x8*(-0.154224)+x9*(0.178372)+x10*(0.332917)+x11*(0.16989)+x12*(-0.000426292)+x13*(0.266925)+x14*(0.064689)+x15*(-0.150804)+x16*(-0.0468812)+(-0.0209833)); 
    h7 = tanh(x0*(-0.154531)+x1*(-0.148688)+x2*(-0.101944)+x3*(0.191606)+x4*(0.295092)+x5*(-0.055843)+x6*(-0.140027)+x7*(0.0958303)+x8*(0.0849403)+x9*(-0.137445)+x10*(-0.277953)+x11*(-0.29311)+x12*(-0.171494)+x13*(-0.0498202)+x14*(-0.30735)+x15*(0.351184)+x16*(-0.300518)+(0.020303)); 
    h8 = tanh(x0*(0.0665445)+x1*(0.0266686)+x2*(-0.085197)+x3*(-0.207944)+x4*(-0.0636073)+x5*(0.388789)+x6*(-0.105903)+x7*(0.226764)+x8*(-0.325104)+x9*(-0.174835)+x10*(0.192008)+x11*(-0.161576)+x12*(0.0268742)+x13*(0.400163)+x14*(-0.377869)+x15*(-0.161403)+x16*(0.00216511)+(-0.138377)); 
    h9 = tanh(x0*(0.0298484)+x1*(0.035105)+x2*(0.329721)+x3*(0.411269)+x4*(0.192395)+x5*(0.178418)+x6*(-0.0157396)+x7*(-0.00666634)+x8*(0.324796)+x9*(-0.281428)+x10*(-0.312665)+x11*(0.136434)+x12*(0.35592)+x13*(-0.343379)+x14*(-0.324542)+x15*(-0.258564)+x16*(0.0975587)+(0.117336)); 
    h10 = tanh(x0*(0.00712222)+x1*(-0.0743808)+x2*(-0.146091)+x3*(-0.36353)+x4*(-0.0197582)+x5*(-0.227436)+x6*(-0.117683)+x7*(-0.184129)+x8*(-0.357427)+x9*(0.326704)+x10*(-0.240408)+x11*(-0.123261)+x12*(0.0602721)+x13*(0.0623197)+x14*(0.245043)+x15*(-0.147995)+x16*(0.175679)+(0.0881127)); 
    h11 = tanh(x0*(-0.00941678)+x1*(0.257915)+x2*(0.42224)+x3*(0.0136519)+x4*(0.0361391)+x5*(0.599432)+x6*(-0.0978864)+x7*(0.127915)+x8*(0.122827)+x9*(0.185439)+x10*(-0.263079)+x11*(-0.32015)+x12*(0.32436)+x13*(-0.109652)+x14*(0.339929)+x15*(-0.263627)+x16*(0.290862)+(0.0394639)); 
    h12 = tanh(x0*(-0.0989632)+x1*(0.275801)+x2*(-0.228242)+x3*(-0.00745353)+x4*(0.0315761)+x5*(-0.208001)+x6*(-0.0401965)+x7*(0.0814305)+x8*(0.252792)+x9*(0.0176831)+x10*(-0.250885)+x11*(-0.0681074)+x12*(-0.342789)+x13*(-0.0622965)+x14*(-0.209069)+x15*(0.0935407)+x16*(0.266977)+(0.0666313)); 
    h13 = tanh(x0*(0.419172)+x1*(0.365001)+x2*(-0.0549243)+x3*(0.283624)+x4*(0.0391724)+x5*(-0.774371)+x6*(0.0731626)+x7*(-0.368401)+x8*(0.0568997)+x9*(-0.156676)+x10*(-0.13606)+x11*(-0.306506)+x12*(-0.0496321)+x13*(-0.319439)+x14*(-0.0397065)+x15*(-0.0988432)+x16*(-0.200123)+(0.211019)); 
    h14 = tanh(x0*(-0.0234496)+x1*(0.0542073)+x2*(-0.246062)+x3*(0.0249757)+x4*(-0.361617)+x5*(-0.233353)+x6*(0.12767)+x7*(0.28564)+x8*(0.147658)+x9*(-0.38604)+x10*(0.334431)+x11*(-0.251831)+x12*(-0.225565)+x13*(-0.20361)+x14*(-0.25635)+x15*(-0.0724788)+x16*(-0.00251257)+(0.0482143)); 
    h15 = tanh(x0*(-0.322193)+x1*(0.159584)+x2*(0.368406)+x3*(0.145682)+x4*(0.336401)+x5*(-0.405785)+x6*(-0.157518)+x7*(0.157938)+x8*(-0.0470984)+x9*(-0.134292)+x10*(-0.239686)+x11*(0.174878)+x12*(0.197414)+x13*(0.354893)+x14*(-0.0377887)+x15*(0.263971)+x16*(-0.00425571)+(0.0618386)); 
    h16 = tanh(x0*(0.466191)+x1*(-0.118419)+x2*(0.417966)+x3*(0.398589)+x4*(0.104089)+x5*(-0.326397)+x6*(-0.354042)+x7*(0.447073)+x8*(-0.0924174)+x9*(-0.133731)+x10*(-0.0563618)+x11*(0.0916981)+x12*(0.279822)+x13*(0.0944874)+x14*(0.00440186)+x15*(-0.0817289)+x16*(-0.2414)+(-0.138896)); 
    h17 = tanh(x0*(-0.22152)+x1*(-0.160313)+x2*(0.223506)+x3*(-0.217078)+x4*(0.101814)+x5*(-0.35605)+x6*(-0.288628)+x7*(0.148275)+x8*(-0.0459089)+x9*(-0.376027)+x10*(0.193762)+x11*(-0.178561)+x12*(0.336535)+x13*(-0.352591)+x14*(-0.385667)+x15*(0.383617)+x16*(-0.181064)+(0.1536)); 
    h18 = tanh(x0*(0.146032)+x1*(-0.176838)+x2*(0.152581)+x3*(0.164007)+x4*(0.181459)+x5*(0.213784)+x6*(0.354924)+x7*(0.112176)+x8*(-0.076958)+x9*(-0.333641)+x10*(-0.0417154)+x11*(-0.251681)+x12*(0.307732)+x13*(0.374405)+x14*(0.220076)+x15*(0.383492)+x16*(0.303195)+(-0.0290535)); 
    h19 = tanh(x0*(-0.088694)+x1*(0.298199)+x2*(0.0310084)+x3*(-0.0775141)+x4*(0.127461)+x5*(0.00401711)+x6*(-0.354872)+x7*(0.557094)+x8*(0.312552)+x9*(-0.271714)+x10*(0.317062)+x11*(0.279597)+x12*(0.0299532)+x13*(-0.384736)+x14*(-0.0650887)+x15*(-0.238421)+x16*(0.394505)+(-0.105373)); 
}

void FNN_14_4_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(0.322731)+x1*(-0.235028)+x2*(-0.471256)+x3*(-0.360857)+x4*(-0.162941)+x5*(-0.733317)+x6*(-0.484449)+x7*(-0.300377)+x8*(0.592341)+x9*(-0.159292)+x10*(-0.119853)+x11*(-0.310392)+x12*(-0.218345)+x13*(-0.162123)+x14*(-0.172198)+x15*(-0.0741852)+x16*(-0.348375)+x17*(-0.463515)+x18*(0.141414)+x19*(0.283188)+(-0.108244)); 
    h1 = tanh(x0*(0.131069)+x1*(-0.201499)+x2*(-0.0133144)+x3*(-0.670517)+x4*(0.38103)+x5*(-0.216743)+x6*(0.272034)+x7*(-0.143911)+x8*(0.257259)+x9*(0.221056)+x10*(-0.0593965)+x11*(-0.231279)+x12*(-0.162765)+x13*(-0.160872)+x14*(0.487666)+x15*(0.0911659)+x16*(0.126376)+x17*(-0.339024)+x18*(-0.0248281)+x19*(-0.0130924)+(-0.00318273)); 
    h2 = tanh(x0*(-0.182379)+x1*(-0.364749)+x2*(-0.209459)+x3*(-0.504961)+x4*(-0.274132)+x5*(0.330895)+x6*(0.0587815)+x7*(-0.307203)+x8*(0.216394)+x9*(0.301112)+x10*(-0.398662)+x11*(-0.144191)+x12*(0.32612)+x13*(0.00599594)+x14*(0.609769)+x15*(0.0429817)+x16*(0.00393409)+x17*(0.160448)+x18*(0.0597228)+x19*(0.536429)+(-0.053384)); 
    h3 = tanh(x0*(0.227982)+x1*(0.189303)+x2*(0.451274)+x3*(-0.136975)+x4*(0.231632)+x5*(0.15763)+x6*(-0.0694314)+x7*(0.2182)+x8*(-0.477582)+x9*(-0.415922)+x10*(0.234075)+x11*(0.238101)+x12*(0.310746)+x13*(0.0546769)+x14*(-0.309665)+x15*(-0.195527)+x16*(-0.124241)+x17*(0.311531)+x18*(-0.297928)+x19*(0.0844376)+(-0.0149253)); 
    h4 = tanh(x0*(0.0166765)+x1*(0.123934)+x2*(0.198114)+x3*(-0.189244)+x4*(0.0455322)+x5*(0.234425)+x6*(0.312945)+x7*(-0.213509)+x8*(-0.344453)+x9*(0.380101)+x10*(-0.227977)+x11*(-0.279445)+x12*(-0.0235209)+x13*(0.16304)+x14*(0.0566997)+x15*(0.195753)+x16*(-0.10647)+x17*(0.35905)+x18*(-0.392403)+x19*(-0.123285)+(0.0498557)); 
    h5 = tanh(x0*(-0.377929)+x1*(0.50214)+x2*(0.0819654)+x3*(0.269502)+x4*(0.215058)+x5*(0.240217)+x6*(-0.0411484)+x7*(0.123122)+x8*(0.194627)+x9*(0.183426)+x10*(-0.291339)+x11*(-0.0785595)+x12*(0.260973)+x13*(0.148109)+x14*(-0.127838)+x15*(0.18361)+x16*(-0.390562)+x17*(0.117244)+x18*(-0.195202)+x19*(-0.163871)+(0.0120315)); 
    h6 = tanh(x0*(-0.260836)+x1*(-0.28192)+x2*(0.237758)+x3*(-0.0564427)+x4*(-0.205821)+x5*(0.296484)+x6*(-0.0597055)+x7*(0.245865)+x8*(-0.231918)+x9*(0.179931)+x10*(-0.414323)+x11*(0.14951)+x12*(-0.195433)+x13*(-0.165729)+x14*(0.0705238)+x15*(-0.409847)+x16*(0.406652)+x17*(-0.216225)+x18*(-0.210847)+x19*(0.371727)+(0.0235473)); 
    h7 = tanh(x0*(0.0504437)+x1*(-0.597034)+x2*(-0.0199128)+x3*(-0.473794)+x4*(0.213962)+x5*(-0.195194)+x6*(0.294953)+x7*(-0.0395641)+x8*(-0.207446)+x9*(-0.129442)+x10*(-0.255398)+x11*(-0.29793)+x12*(-0.427873)+x13*(0.169133)+x14*(-0.0708799)+x15*(-0.329323)+x16*(0.335868)+x17*(-0.335631)+x18*(-0.058346)+x19*(-0.376422)+(-0.00154143)); 
    h8 = tanh(x0*(0.33095)+x1*(-0.196339)+x2*(-0.181012)+x3*(-0.0436238)+x4*(-0.380914)+x5*(-0.56865)+x6*(-0.231331)+x7*(-0.0750704)+x8*(0.0556733)+x9*(-0.112317)+x10*(-0.146265)+x11*(-0.208993)+x12*(-0.0248638)+x13*(-0.331968)+x14*(-0.188166)+x15*(-0.14623)+x16*(-0.0141581)+x17*(0.0504754)+x18*(-0.0262158)+x19*(-0.179628)+(-0.0363249)); 
    h9 = tanh(x0*(0.465471)+x1*(-0.454581)+x2*(0.73223)+x3*(-0.301722)+x4*(0.187596)+x5*(0.324077)+x6*(-0.322419)+x7*(0.20984)+x8*(-0.228859)+x9*(-0.177072)+x10*(-0.34171)+x11*(-0.465413)+x12*(0.154053)+x13*(0.628519)+x14*(0.253177)+x15*(0.803804)+x16*(0.376889)+x17*(0.118855)+x18*(-0.0980676)+x19*(0.232273)+(-0.0252662)); 
}

void FNN_14_4_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-0.476793)+x1*(-0.427998)+x2*(-0.117228)+x3*(-0.565833)+x4*(0.27946)+x5*(-0.247004)+x6*(-0.586128)+x7*(-0.164889)+x8*(-0.687941)+x9*(-0.589584)+(0.00737669); 
    h1 = x0*(-0.0753729)+x1*(0.611151)+x2*(-0.521663)+x3*(-0.209362)+x4*(0.732774)+x5*(0.328528)+x6*(-0.173257)+x7*(-0.226097)+x8*(-0.646711)+x9*(-0.623751)+(-0.00520426); 
    h2 = x0*(-0.665741)+x1*(0.185983)+x2*(-0.453267)+x3*(0.628306)+x4*(-0.294238)+x5*(0.174445)+x6*(0.524548)+x7*(0.188459)+x8*(-0.118454)+x9*(-0.532734)+(0.0978435); 
}

void FNN_14_4(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_14_4_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_14_4_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_14_4_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_4(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_14_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_14_5_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.289214)+x1*(-0.403649)+x2*(-0.470475)+x3*(0.263478)+x4*(0.374313)+x5*(0.190583)+x6*(0.203066)+x7*(-0.311687)+x8*(-0.0522893)+x9*(-0.356541)+x10*(-0.265569)+x11*(0.35154)+x12*(0.210896)+x13*(0.267719)+x14*(-0.00591025)+x15*(0.390655)+x16*(0.201951)+(0.0447165)); 
    h1 = tanh(x0*(-0.283897)+x1*(0.187546)+x2*(0.321102)+x3*(0.0155244)+x4*(-0.0466132)+x5*(0.748137)+x6*(-0.180416)+x7*(-0.25379)+x8*(0.266651)+x9*(-0.249297)+x10*(-0.0994819)+x11*(-0.0724432)+x12*(0.391829)+x13*(0.282855)+x14*(0.356221)+x15*(-0.0558463)+x16*(-0.0354584)+(-0.553958)); 
    h2 = tanh(x0*(0.117387)+x1*(0.0132532)+x2*(-0.219862)+x3*(-0.016873)+x4*(0.160307)+x5*(0.219681)+x6*(0.130168)+x7*(-0.12388)+x8*(0.057121)+x9*(0.0193158)+x10*(0.310513)+x11*(0.223451)+x12*(-0.288427)+x13*(0.15987)+x14*(-0.209283)+x15*(-0.102916)+x16*(0.322993)+(-0.136227)); 
    h3 = tanh(x0*(-0.338244)+x1*(0.198601)+x2*(-0.259401)+x3*(0.146902)+x4*(0.024119)+x5*(-0.188498)+x6*(0.125343)+x7*(-0.699408)+x8*(-0.0416986)+x9*(-0.40016)+x10*(0.39931)+x11*(-0.0335461)+x12*(-0.0897276)+x13*(-0.344961)+x14*(-0.316899)+x15*(0.0382206)+x16*(-0.238788)+(0.278758)); 
    h4 = tanh(x0*(-0.182976)+x1*(-0.332962)+x2*(0.00211631)+x3*(-0.0552366)+x4*(-0.134784)+x5*(0.507872)+x6*(0.00653096)+x7*(-0.0736797)+x8*(-0.137155)+x9*(0.0811919)+x10*(-0.312031)+x11*(0.128723)+x12*(-0.0529777)+x13*(-0.377913)+x14*(-0.244493)+x15*(0.306556)+x16*(0.266402)+(0.0275012)); 
    h5 = tanh(x0*(0.201374)+x1*(-0.209333)+x2*(0.314776)+x3*(-0.0089273)+x4*(-0.0408453)+x5*(0.0266441)+x6*(0.121744)+x7*(0.162138)+x8*(0.0625243)+x9*(0.0936293)+x10*(-0.107449)+x11*(-0.0804539)+x12*(-0.147967)+x13*(0.141236)+x14*(-0.393184)+x15*(-0.165645)+x16*(0.12809)+(-0.113956)); 
    h6 = tanh(x0*(0.243982)+x1*(-0.279844)+x2*(-0.264055)+x3*(-0.186822)+x4*(-0.128966)+x5*(-0.177403)+x6*(-0.000968665)+x7*(0.380679)+x8*(-0.210828)+x9*(0.252233)+x10*(-0.300384)+x11*(-0.220988)+x12*(-0.340277)+x13*(0.233048)+x14*(-0.32254)+x15*(0.287027)+x16*(0.16715)+(0.0627492)); 
    h7 = tanh(x0*(0.154604)+x1*(0.253612)+x2*(0.0397838)+x3*(0.321576)+x4*(0.0413463)+x5*(-0.056787)+x6*(0.248977)+x7*(-0.225319)+x8*(-0.294502)+x9*(0.186184)+x10*(0.207699)+x11*(-0.215033)+x12*(-0.194232)+x13*(0.145473)+x14*(-0.035024)+x15*(-0.184659)+x16*(-0.0751315)+(-0.114459)); 
    h8 = tanh(x0*(-0.256884)+x1*(0.192051)+x2*(0.114826)+x3*(0.228008)+x4*(-0.245858)+x5*(0.220613)+x6*(-0.326842)+x7*(-0.245394)+x8*(-0.0503471)+x9*(0.223083)+x10*(0.205791)+x11*(-0.169462)+x12*(0.330968)+x13*(0.161748)+x14*(-0.115702)+x15*(0.186809)+x16*(0.0820804)+(-0.155294)); 
    h9 = tanh(x0*(-0.442011)+x1*(-0.422758)+x2*(-0.187428)+x3*(-0.0580633)+x4*(0.328862)+x5*(0.180606)+x6*(0.0709201)+x7*(0.255596)+x8*(0.170345)+x9*(-0.0400564)+x10*(-0.246125)+x11*(0.117832)+x12*(0.323264)+x13*(-0.236561)+x14*(-0.188637)+x15*(0.342904)+x16*(0.288106)+(-0.022324)); 
    h10 = tanh(x0*(-0.353414)+x1*(-0.145378)+x2*(0.269322)+x3*(0.0534741)+x4*(0.0326555)+x5*(-0.306241)+x6*(0.16308)+x7*(-0.376375)+x8*(0.421629)+x9*(0.292619)+x10*(0.0650328)+x11*(0.236998)+x12*(-0.0839425)+x13*(0.189024)+x14*(-0.0732288)+x15*(-0.171299)+x16*(0.0611541)+(0.219738)); 
    h11 = tanh(x0*(0.353934)+x1*(0.0460325)+x2*(0.0470534)+x3*(0.222653)+x4*(0.0731137)+x5*(0.454277)+x6*(0.0111412)+x7*(0.110179)+x8*(0.104563)+x9*(0.251327)+x10*(0.304211)+x11*(-0.11433)+x12*(0.228793)+x13*(-0.0127476)+x14*(0.284993)+x15*(-0.386969)+x16*(-0.316344)+(-0.228623)); 
    h12 = tanh(x0*(0.032243)+x1*(-0.00287052)+x2*(0.479526)+x3*(0.0261886)+x4*(0.282087)+x5*(0.0918711)+x6*(0.272354)+x7*(0.029929)+x8*(-0.0834554)+x9*(-0.253778)+x10*(-0.0435822)+x11*(-0.274587)+x12*(0.0710947)+x13*(0.242382)+x14*(0.311833)+x15*(-0.197675)+x16*(0.1595)+(-0.124406)); 
    h13 = tanh(x0*(0.535264)+x1*(0.68926)+x2*(-0.0831526)+x3*(0.475879)+x4*(-0.341919)+x5*(0.411466)+x6*(-0.058444)+x7*(0.104392)+x8*(-0.00688265)+x9*(0.375916)+x10*(0.278359)+x11*(-0.0711994)+x12*(-0.293431)+x13*(-0.134421)+x14*(-0.25823)+x15*(-0.0442247)+x16*(0.0808101)+(0.233243)); 
    h14 = tanh(x0*(0.157149)+x1*(-0.173528)+x2*(0.0938835)+x3*(0.0896429)+x4*(-0.00116661)+x5*(0.097084)+x6*(0.322235)+x7*(0.477569)+x8*(-0.160629)+x9*(0.398286)+x10*(-0.0964206)+x11*(0.320026)+x12*(0.316872)+x13*(0.0353722)+x14*(-0.126839)+x15*(0.333994)+x16*(-0.256959)+(-0.112712)); 
    h15 = tanh(x0*(-0.400137)+x1*(0.121982)+x2*(-0.245349)+x3*(-0.027168)+x4*(-0.347403)+x5*(0.0977411)+x6*(-0.199581)+x7*(0.300061)+x8*(-0.133076)+x9*(0.234615)+x10*(-0.386322)+x11*(-0.32158)+x12*(0.260891)+x13*(0.377632)+x14*(0.0753738)+x15*(-0.0114948)+x16*(-0.0962503)+(-0.000187775)); 
    h16 = tanh(x0*(0.415653)+x1*(-0.242698)+x2*(-0.173195)+x3*(-0.0302581)+x4*(-0.365303)+x5*(-0.2152)+x6*(-0.297098)+x7*(0.0309858)+x8*(-0.216708)+x9*(0.144905)+x10*(-0.24684)+x11*(0.290756)+x12*(-0.392632)+x13*(0.203125)+x14*(-0.139968)+x15*(-0.204353)+x16*(0.261103)+(0.140391)); 
    h17 = tanh(x0*(0.232458)+x1*(0.21466)+x2*(-0.354451)+x3*(-0.0567089)+x4*(-0.319652)+x5*(-0.368167)+x6*(-0.449061)+x7*(0.151406)+x8*(-0.130359)+x9*(-0.0162542)+x10*(0.063284)+x11*(0.138565)+x12*(0.310243)+x13*(0.200061)+x14*(0.20388)+x15*(0.186176)+x16*(0.361654)+(-0.0878242)); 
    h18 = tanh(x0*(-0.325738)+x1*(0.107805)+x2*(-0.267655)+x3*(0.471351)+x4*(0.0533415)+x5*(0.056145)+x6*(0.205654)+x7*(0.0325016)+x8*(0.0705061)+x9*(0.012179)+x10*(-0.0638813)+x11*(-0.318406)+x12*(-0.402629)+x13*(-0.106591)+x14*(-0.0172482)+x15*(-0.0261809)+x16*(-0.0238419)+(-0.233029)); 
    h19 = tanh(x0*(0.350193)+x1*(0.307532)+x2*(-0.134595)+x3*(-0.284984)+x4*(0.00156996)+x5*(-0.487595)+x6*(-0.102432)+x7*(-0.15363)+x8*(0.187336)+x9*(-0.083055)+x10*(0.267741)+x11*(-0.228095)+x12*(-0.0233268)+x13*(-0.370893)+x14*(0.207174)+x15*(0.183516)+x16*(0.392931)+(0.212365)); 
}

void FNN_14_5_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.166636)+x1*(-0.264748)+x2*(-0.334376)+x3*(0.633756)+x4*(0.0943724)+x5*(-0.208608)+x6*(0.238628)+x7*(-0.277506)+x8*(-0.0849668)+x9*(0.352562)+x10*(0.421973)+x11*(0.175855)+x12*(0.0172809)+x13*(0.255953)+x14*(-0.360055)+x15*(0.142812)+x16*(-0.0176005)+x17*(-0.351749)+x18*(-0.344332)+x19*(-0.262615)+(0.0807909)); 
    h1 = tanh(x0*(-0.159336)+x1*(0.382205)+x2*(0.272479)+x3*(0.16528)+x4*(0.158882)+x5*(0.411965)+x6*(0.12612)+x7*(0.189171)+x8*(0.0956783)+x9*(-0.126422)+x10*(0.190622)+x11*(-0.0396973)+x12*(-0.227302)+x13*(-0.0924917)+x14*(0.0889304)+x15*(0.166025)+x16*(-0.378246)+x17*(-0.179273)+x18*(0.253827)+x19*(-0.228396)+(-0.0746258)); 
    h2 = tanh(x0*(0.00952327)+x1*(-0.00219696)+x2*(-0.343328)+x3*(-0.0373082)+x4*(0.29194)+x5*(0.100974)+x6*(-0.0623375)+x7*(-0.277614)+x8*(-0.214725)+x9*(-0.0902663)+x10*(-0.169207)+x11*(-0.374884)+x12*(-0.269598)+x13*(-0.072566)+x14*(-0.174638)+x15*(0.152571)+x16*(0.200779)+x17*(-0.367514)+x18*(0.255443)+x19*(-0.0318608)+(0.0453034)); 
    h3 = tanh(x0*(0.239731)+x1*(-1.03831)+x2*(-0.399527)+x3*(0.812892)+x4*(-0.427579)+x5*(-0.120602)+x6*(0.407654)+x7*(-0.462999)+x8*(-0.615987)+x9*(-0.0969398)+x10*(0.70731)+x11*(-0.341407)+x12*(-0.500145)+x13*(0.0861893)+x14*(-0.121427)+x15*(-0.0449778)+x16*(0.307757)+x17*(0.30555)+x18*(0.00833125)+x19*(0.416713)+(0.0598592)); 
    h4 = tanh(x0*(0.227091)+x1*(-0.284893)+x2*(0.345392)+x3*(-0.259805)+x4*(-0.0373671)+x5*(0.351517)+x6*(-0.200888)+x7*(0.361567)+x8*(0.194539)+x9*(0.210395)+x10*(-0.486283)+x11*(-0.21388)+x12*(-0.257359)+x13*(0.120108)+x14*(0.320838)+x15*(-0.278796)+x16*(-0.268398)+x17*(0.241346)+x18*(-0.303674)+x19*(-0.0386307)+(-0.0776278)); 
    h5 = tanh(x0*(-0.10382)+x1*(-0.223367)+x2*(0.382119)+x3*(0.627595)+x4*(0.441911)+x5*(-0.284763)+x6*(0.493239)+x7*(0.0830019)+x8*(0.373326)+x9*(-0.170814)+x10*(0.128443)+x11*(0.347087)+x12*(0.138174)+x13*(-0.124787)+x14*(0.32114)+x15*(0.33963)+x16*(-0.316847)+x17*(0.128698)+x18*(0.100855)+x19*(-0.180038)+(0.0165031)); 
    h6 = tanh(x0*(-0.200889)+x1*(0.0695969)+x2*(-0.277596)+x3*(-0.0834984)+x4*(-0.0600475)+x5*(-0.368766)+x6*(0.335444)+x7*(0.0282551)+x8*(-0.398234)+x9*(0.058528)+x10*(0.370069)+x11*(-0.321536)+x12*(0.118704)+x13*(0.366112)+x14*(0.366325)+x15*(-0.363117)+x16*(0.0944366)+x17*(0.194692)+x18*(-0.0363674)+x19*(0.0585291)+(0.025992)); 
    h7 = tanh(x0*(0.329311)+x1*(0.0703091)+x2*(0.212)+x3*(0.476013)+x4*(0.191876)+x5*(-0.198157)+x6*(-0.383038)+x7*(-0.247396)+x8*(-0.0202147)+x9*(-0.505517)+x10*(0.281749)+x11*(-0.148536)+x12*(0.13444)+x13*(-0.169294)+x14*(-0.135413)+x15*(-0.394622)+x16*(0.406815)+x17*(0.0986474)+x18*(0.412655)+x19*(-0.243198)+(0.0235964)); 
    h8 = tanh(x0*(0.14065)+x1*(0.789592)+x2*(0.109726)+x3*(-0.31812)+x4*(0.529162)+x5*(-0.112836)+x6*(-0.141096)+x7*(0.0867853)+x8*(0.0771104)+x9*(0.0847356)+x10*(-0.220165)+x11*(0.075321)+x12*(0.320834)+x13*(-0.415314)+x14*(0.0104152)+x15*(-0.200143)+x16*(0.260425)+x17*(0.217231)+x18*(0.314882)+x19*(-0.444315)+(-0.0771699)); 
    h9 = tanh(x0*(0.00803841)+x1*(0.240434)+x2*(0.00539191)+x3*(-0.35379)+x4*(-0.168036)+x5*(0.142607)+x6*(0.0355873)+x7*(0.100939)+x8*(-0.056)+x9*(-0.177611)+x10*(-0.0710086)+x11*(-0.140214)+x12*(0.336833)+x13*(0.0743095)+x14*(0.146447)+x15*(0.227743)+x16*(-0.312204)+x17*(0.105868)+x18*(0.349153)+x19*(0.0329866)+(0.000415753)); 
}

void FNN_14_5_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(0.441831)+x1*(-0.0639829)+x2*(0.434671)+x3*(-0.675489)+x4*(0.582906)+x5*(-0.357579)+x6*(0.201015)+x7*(0.55248)+x8*(-0.524243)+x9*(-0.473942)+(0.0384049); 
    h1 = x0*(0.169348)+x1*(0.200842)+x2*(-0.200892)+x3*(-0.514619)+x4*(-0.423482)+x5*(0.4459)+x6*(-0.133029)+x7*(-0.181868)+x8*(-0.754441)+x9*(-0.670495)+(-0.00653071); 
    h2 = x0*(0.230457)+x1*(-0.516678)+x2*(0.2482)+x3*(-0.497357)+x4*(-0.58914)+x5*(0.390394)+x6*(-0.0839626)+x7*(0.461271)+x8*(-0.499414)+x9*(0.603308)+(0.0259034); 
}

void FNN_14_5(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_14_5_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_14_5_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_14_5_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_5(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_14_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_3(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x3 < lowBound ) {
        ModelKDTree_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x3 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x3 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void ModelKDTree_1(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x5 < lowBound ) {
        ModelKDTree_2(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x5 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_2(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x5 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void FNN_14_6_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.0555044)+x1*(0.103831)+x2*(0.180034)+x3*(0.300072)+x4*(-0.0443707)+x5*(0.933327)+x6*(-0.162774)+x7*(-0.375172)+x8*(-0.559587)+x9*(-0.263562)+x10*(0.369013)+x11*(0.200583)+x12*(-0.250221)+x13*(0.0710695)+x14*(-0.110981)+x15*(-0.24939)+x16*(-0.243666)+(-0.304087)); 
    h1 = tanh(x0*(-0.192664)+x1*(0.270974)+x2*(-0.110722)+x3*(0.0688704)+x4*(0.162829)+x5*(-0.327706)+x6*(-0.378424)+x7*(-0.211144)+x8*(-0.0391582)+x9*(-0.369824)+x10*(-0.333565)+x11*(-0.336196)+x12*(0.0591144)+x13*(0.379192)+x14*(-0.249587)+x15*(0.308029)+x16*(-0.109508)+(-0.116596)); 
    h2 = tanh(x0*(-0.275047)+x1*(-0.0669861)+x2*(0.471452)+x3*(0.302309)+x4*(-0.0205249)+x5*(0.0248823)+x6*(-0.20403)+x7*(-0.269309)+x8*(-0.149484)+x9*(-0.217974)+x10*(0.348431)+x11*(-0.395026)+x12*(-0.0982687)+x13*(0.25119)+x14*(-0.381684)+x15*(-0.360006)+x16*(-0.289329)+(0.112791)); 
    h3 = tanh(x0*(0.0210556)+x1*(0.263813)+x2*(0.10697)+x3*(0.19171)+x4*(0.096375)+x5*(-0.018209)+x6*(0.0632739)+x7*(0.230882)+x8*(-0.182882)+x9*(0.137594)+x10*(0.291301)+x11*(0.08002)+x12*(0.177669)+x13*(-0.358754)+x14*(0.185329)+x15*(0.335517)+x16*(0.0664223)+(0.0420433)); 
    h4 = tanh(x0*(-0.199911)+x1*(0.140765)+x2*(0.296653)+x3*(-0.227316)+x4*(-0.202656)+x5*(0.44885)+x6*(0.105891)+x7*(-0.142981)+x8*(0.0359886)+x9*(0.377074)+x10*(-0.2847)+x11*(-0.171881)+x12*(-0.328288)+x13*(0.242461)+x14*(0.216726)+x15*(-0.177609)+x16*(0.0567709)+(0.0311515)); 
    h5 = tanh(x0*(0.300559)+x1*(-0.0903008)+x2*(0.180162)+x3*(-0.389136)+x4*(-0.153546)+x5*(0.341255)+x6*(-0.251779)+x7*(-0.340994)+x8*(0.270005)+x9*(0.364033)+x10*(0.0134693)+x11*(-0.371212)+x12*(0.206098)+x13*(0.0686032)+x14*(0.328777)+x15*(-0.210639)+x16*(0.0893124)+(-0.014784)); 
    h6 = tanh(x0*(-0.287204)+x1*(-0.160249)+x2*(0.182445)+x3*(0.335055)+x4*(-0.097002)+x5*(0.696251)+x6*(-0.206762)+x7*(0.154487)+x8*(-0.351078)+x9*(0.339543)+x10*(-0.0412119)+x11*(0.36433)+x12*(-0.285)+x13*(0.39623)+x14*(-0.0428927)+x15*(-0.0631169)+x16*(-0.180832)+(-0.152284)); 
    h7 = tanh(x0*(0.169875)+x1*(0.131023)+x2*(-0.0772801)+x3*(0.0730803)+x4*(-0.34866)+x5*(-0.289198)+x6*(-0.0556083)+x7*(-0.41696)+x8*(0.141075)+x9*(-0.393606)+x10*(-0.101754)+x11*(-0.145477)+x12*(0.176481)+x13*(-0.0416884)+x14*(-0.253406)+x15*(0.246823)+x16*(0.383332)+(0.0485616)); 
    h8 = tanh(x0*(-0.297476)+x1*(-0.0480662)+x2*(-0.170718)+x3*(-0.258943)+x4*(-0.144435)+x5*(0.289183)+x6*(-0.127358)+x7*(-0.0319645)+x8*(0.00512667)+x9*(0.197129)+x10*(0.132792)+x11*(0.386137)+x12*(-0.396313)+x13*(-0.0509875)+x14*(-0.130631)+x15*(-0.0522405)+x16*(-0.220931)+(0.0216719)); 
    h9 = tanh(x0*(0.127542)+x1*(0.0711419)+x2*(0.0921259)+x3*(-0.187861)+x4*(0.422915)+x5*(0.0548554)+x6*(-0.286739)+x7*(0.313706)+x8*(-0.450572)+x9*(-0.293217)+x10*(0.26341)+x11*(-0.0731856)+x12*(-0.293067)+x13*(-0.113637)+x14*(0.220958)+x15*(0.19519)+x16*(0.236955)+(0.0462179)); 
    h10 = tanh(x0*(0.311575)+x1*(0.0844999)+x2*(0.236174)+x3*(0.207272)+x4*(0.203623)+x5*(0.106051)+x6*(0.371119)+x7*(-0.462719)+x8*(0.423912)+x9*(-0.256501)+x10*(0.323725)+x11*(0.278675)+x12*(0.00945991)+x13*(0.274336)+x14*(0.0289491)+x15*(0.162512)+x16*(0.315254)+(0.117821)); 
    h11 = tanh(x0*(-0.114819)+x1*(-0.341629)+x2*(-0.0807053)+x3*(-0.0450628)+x4*(0.252037)+x5*(0.513995)+x6*(-0.332871)+x7*(-0.217194)+x8*(0.360139)+x9*(0.289492)+x10*(-0.385474)+x11*(-0.362434)+x12*(0.174693)+x13*(-0.373698)+x14*(-0.302392)+x15*(0.195545)+x16*(0.160491)+(-0.00702563)); 
    h12 = tanh(x0*(0.179045)+x1*(0.200426)+x2*(-0.3622)+x3*(-0.294982)+x4*(0.0638842)+x5*(0.359125)+x6*(0.0754685)+x7*(0.334028)+x8*(-0.290309)+x9*(0.147106)+x10*(0.109439)+x11*(-0.187664)+x12*(-0.0732373)+x13*(0.310952)+x14*(-0.177786)+x15*(-0.381639)+x16*(-0.209117)+(-0.125059)); 
    h13 = tanh(x0*(0.0130952)+x1*(-0.261506)+x2*(0.308271)+x3*(0.0415191)+x4*(0.0181715)+x5*(0.433552)+x6*(0.0570925)+x7*(0.00231996)+x8*(-0.223233)+x9*(-0.28719)+x10*(0.294322)+x11*(0.291924)+x12*(0.00578949)+x13*(0.0332623)+x14*(-0.012992)+x15*(0.345564)+x16*(-0.158727)+(-0.179574)); 
    h14 = tanh(x0*(-0.0966966)+x1*(-0.194202)+x2*(0.0871314)+x3*(-0.273998)+x4*(-0.00514754)+x5*(-0.309003)+x6*(-0.177081)+x7*(0.182612)+x8*(0.0824451)+x9*(-0.401361)+x10*(-0.283712)+x11*(0.0770018)+x12*(0.214302)+x13*(0.0165313)+x14*(-0.217028)+x15*(-0.393968)+x16*(-0.401076)+(0.0663814)); 
    h15 = tanh(x0*(-0.0439079)+x1*(-0.157427)+x2*(-0.0499638)+x3*(-0.221651)+x4*(0.117012)+x5*(0.316002)+x6*(-0.262653)+x7*(-0.194898)+x8*(-0.373048)+x9*(-0.167919)+x10*(-0.0707249)+x11*(-0.347397)+x12*(-0.395473)+x13*(0.0449467)+x14*(-0.153881)+x15*(-0.358496)+x16*(-0.310635)+(-0.115709)); 
    h16 = tanh(x0*(-0.108144)+x1*(0.235814)+x2*(0.101738)+x3*(0.10815)+x4*(0.347529)+x5*(0.207462)+x6*(-0.0264121)+x7*(-0.176539)+x8*(-0.280023)+x9*(0.363379)+x10*(-0.209664)+x11*(-0.114033)+x12*(-0.105518)+x13*(-0.143537)+x14*(-0.259047)+x15*(0.117678)+x16*(-0.232261)+(0.137658)); 
    h17 = tanh(x0*(-0.258838)+x1*(-0.229437)+x2*(0.251444)+x3*(0.291391)+x4*(-0.206356)+x5*(-0.223667)+x6*(-0.0946549)+x7*(0.240659)+x8*(0.084668)+x9*(-0.244478)+x10*(0.0544975)+x11*(-0.0726663)+x12*(0.245984)+x13*(-0.0670866)+x14*(0.281024)+x15*(0.127624)+x16*(-0.169207)+(0.045439)); 
    h18 = tanh(x0*(0.271072)+x1*(0.122645)+x2*(-0.292864)+x3*(-0.333753)+x4*(-0.217541)+x5*(0.549016)+x6*(-0.243759)+x7*(0.239548)+x8*(-0.176314)+x9*(0.332134)+x10*(-0.11176)+x11*(0.259182)+x12*(-0.201476)+x13*(-0.399383)+x14*(-0.292716)+x15*(0.123432)+x16*(0.0798725)+(0.0625367)); 
    h19 = tanh(x0*(-0.223536)+x1*(0.377229)+x2*(0.513203)+x3*(-0.11692)+x4*(0.00179932)+x5*(-0.240778)+x6*(-0.264277)+x7*(0.438986)+x8*(-0.260636)+x9*(0.103878)+x10*(-0.0238322)+x11*(-0.214993)+x12*(-0.166695)+x13*(-0.118973)+x14*(0.100035)+x15*(0.151916)+x16*(-0.279673)+(-0.0472093)); 
}

void FNN_14_6_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.164671)+x1*(-0.00335768)+x2*(-0.429945)+x3*(-0.0374479)+x4*(-0.271952)+x5*(-0.286709)+x6*(0.391968)+x7*(-0.398272)+x8*(-0.347206)+x9*(0.180979)+x10*(0.323618)+x11*(0.0438777)+x12*(0.0812584)+x13*(-0.391163)+x14*(-0.0590311)+x15*(-0.318681)+x16*(-0.502063)+x17*(0.399927)+x18*(0.301388)+x19*(0.293348)+(-0.0833718)); 
    h1 = tanh(x0*(-0.210707)+x1*(0.242756)+x2*(0.09023)+x3*(-0.313563)+x4*(0.379813)+x5*(0.0295197)+x6*(0.260161)+x7*(-0.182983)+x8*(-0.353445)+x9*(0.155063)+x10*(0.0757243)+x11*(0.0584855)+x12*(0.172404)+x13*(0.254261)+x14*(-0.224989)+x15*(0.0037733)+x16*(-0.464223)+x17*(-0.23516)+x18*(0.379514)+x19*(-0.05061)+(-0.0824091)); 
    h2 = tanh(x0*(-0.258329)+x1*(-0.24725)+x2*(0.470177)+x3*(-0.433894)+x4*(0.112128)+x5*(-0.26675)+x6*(-0.0414183)+x7*(0.0455094)+x8*(-0.355841)+x9*(-0.196805)+x10*(0.055296)+x11*(0.220271)+x12*(-0.201139)+x13*(-0.213886)+x14*(-0.0985673)+x15*(-0.307937)+x16*(-0.0966221)+x17*(0.29336)+x18*(-0.04808)+x19*(0.0326411)+(0.113492)); 
    h3 = tanh(x0*(0.197809)+x1*(0.340147)+x2*(-0.169343)+x3*(0.329206)+x4*(0.505976)+x5*(-0.180848)+x6*(-0.28989)+x7*(0.22554)+x8*(-0.100831)+x9*(0.110945)+x10*(0.403741)+x11*(0.0880724)+x12*(0.215244)+x13*(0.2577)+x14*(-0.315761)+x15*(-0.35185)+x16*(0.394998)+x17*(-0.26167)+x18*(0.461451)+x19*(-0.0111166)+(0.0847079)); 
    h4 = tanh(x0*(0.366501)+x1*(-0.00658234)+x2*(0.0782797)+x3*(-0.00204821)+x4*(0.179265)+x5*(-0.14472)+x6*(0.207403)+x7*(-0.00402609)+x8*(0.0803237)+x9*(-0.0838737)+x10*(-0.19928)+x11*(0.199549)+x12*(0.333682)+x13*(0.337104)+x14*(-0.257953)+x15*(0.155262)+x16*(-0.120634)+x17*(-0.119669)+x18*(0.103438)+x19*(0.377344)+(-0.112421)); 
    h5 = tanh(x0*(-0.150899)+x1*(0.061933)+x2*(-0.238199)+x3*(0.039927)+x4*(-0.417148)+x5*(-0.190177)+x6*(0.235845)+x7*(0.309044)+x8*(0.388703)+x9*(0.433489)+x10*(-0.358718)+x11*(-0.0832703)+x12*(-0.185582)+x13*(0.135446)+x14*(0.260729)+x15*(-0.245547)+x16*(-0.41727)+x17*(0.281421)+x18*(0.242818)+x19*(0.0702877)+(0.0179656)); 
    h6 = tanh(x0*(0.29609)+x1*(-0.0534579)+x2*(0.165935)+x3*(-0.155471)+x4*(-0.286001)+x5*(0.144301)+x6*(-0.0137615)+x7*(-0.263984)+x8*(0.190083)+x9*(0.0234684)+x10*(0.236659)+x11*(0.0107739)+x12*(-0.0367464)+x13*(-0.13953)+x14*(-0.0538027)+x15*(-0.365497)+x16*(-0.284537)+x17*(0.103187)+x18*(0.259968)+x19*(-0.194393)+(-0.175627)); 
    h7 = tanh(x0*(-0.760126)+x1*(0.327747)+x2*(0.0934519)+x3*(-0.173053)+x4*(-0.431057)+x5*(-0.151055)+x6*(-0.582089)+x7*(-0.0297203)+x8*(0.096042)+x9*(-0.239503)+x10*(0.119488)+x11*(-0.256278)+x12*(-0.438624)+x13*(-0.116995)+x14*(-0.0321181)+x15*(-0.261719)+x16*(-0.443974)+x17*(0.239552)+x18*(-0.629289)+x19*(0.574538)+(-0.063204)); 
    h8 = tanh(x0*(0.000379884)+x1*(0.23059)+x2*(0.00508397)+x3*(0.00471651)+x4*(0.297794)+x5*(0.189666)+x6*(-0.0534993)+x7*(-0.391514)+x8*(0.129156)+x9*(0.220046)+x10*(-0.0431894)+x11*(-0.191071)+x12*(-0.0169401)+x13*(-0.0215829)+x14*(0.187516)+x15*(-0.273546)+x16*(0.138461)+x17*(0.219379)+x18*(0.435288)+x19*(-0.145412)+(0.0641924)); 
    h9 = tanh(x0*(-0.156733)+x1*(-0.0832244)+x2*(0.115388)+x3*(-0.279397)+x4*(-0.672996)+x5*(-0.0654862)+x6*(-0.492227)+x7*(0.482543)+x8*(-0.0404371)+x9*(0.185857)+x10*(0.296547)+x11*(-0.343999)+x12*(0.00803384)+x13*(-0.5177)+x14*(0.587666)+x15*(0.345065)+x16*(-0.00211821)+x17*(0.342117)+x18*(-0.533923)+x19*(0.392363)+(-0.164319)); 
}

void FNN_14_6_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(-0.334148)+x1*(-0.0914298)+x2*(0.398697)+x3*(-0.148738)+x4*(-0.552449)+x5*(-0.0192528)+x6*(-0.446265)+x7*(-0.359281)+x8*(0.346597)+x9*(-0.149095)+(0.106743); 
    h1 = x0*(-0.248429)+x1*(0.21745)+x2*(0.55396)+x3*(0.344896)+x4*(-0.573923)+x5*(0.359507)+x6*(-0.27404)+x7*(-0.472435)+x8*(-0.25924)+x9*(0.157762)+(0.082869); 
    h2 = x0*(-0.220167)+x1*(-0.532981)+x2*(0.02064)+x3*(0.449804)+x4*(-0.555519)+x5*(0.433439)+x6*(0.447731)+x7*(-0.277281)+x8*(-0.203208)+x9*(-0.146696)+(0.0684771); 
}

void FNN_14_6(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_14_6_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_14_6_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_14_6_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_6(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_14_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_0(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x8 < lowBound ) {
        ModelKDTree_1(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x8 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_1(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x8 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void Model(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,    
out float h0,out float h1,out float h2)
{
    // map to unit hyperbox
    x0 = (x0 - (0.104431)) / (0.895569);
    x1 = (x1 - (0.104431)) / (0.895569);
    x2 = (x2 - (0.104431)) / (0.895569);
    x3 = (x3 - (-0.999512)) / (1.99902);
    x4 = (x4 - (0)) / (1e-06);
    x5 = (x5 - (-0.999512)) / (1.99902);
    x6 = (x6 - (-0.995605)) / (1.98828);
    x7 = (x7 - (0.028336)) / (0.971176);
    x8 = (x8 - (-0.999023)) / (1.98926);
    x9 = (x9 - (0)) / (1);
    x10 = (x10 - (1)) / (1);
    x11 = (x11 - (0)) / (1);
    x12 = (x12 - (1)) / (1);
    x13 = (x13 - (1)) / (1);
    x14 = (x14 - (1)) / (1);
    x15 = (x15 - (20)) / (1);
    x16 = (x16 - (1)) / (1);

    ModelKDTree_0(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);

    // map back
    h0 = h0 * (0.311386) + (0);
    h1 = h1 * (0.334915) + (0);
    h2 = h2 * (0.37065) + (0);
}

