#version 330 core

// ----------------- in out

out vec4 FragColor;

in VS_OUT {
    vec3 FragPos;
    vec3 Normal;
    vec2 TexCoords;
    vec3 Tangent;
} fs_in;

// ----------------- const

#define MAX_POINT_LIGHTS 8
const float PI = 3.14159265359;
const float INV_PI = 0.31830988618;

// array of offset direction for sampling
const vec3 gridSamplingDisk[20] = vec3[]
(
   vec3(1, 1,  1), vec3( 1, -1,  1), vec3(-1, -1,  1), vec3(-1, 1,  1), 
   vec3(1, 1, -1), vec3( 1, -1, -1), vec3(-1, -1, -1), vec3(-1, 1, -1),
   vec3(1, 1,  0), vec3( 1, -1,  0), vec3(-1, -1,  0), vec3(-1, 1,  0),
   vec3(1, 0,  1), vec3(-1,  0,  1), vec3( 1,  0, -1), vec3(-1, 0, -1),
   vec3(0, 1,  1), vec3( 0, -1,  1), vec3( 0, -1, -1), vec3( 0, 1, -1)
);

// ----------------- struct

struct BSDF_FrostedGlass {
	vec3 colorFactor;
	bool haveColorTexture;
    sampler2D colorTexture;

	float roughnessFactor;
	bool haveRoughnessTexture;
    sampler2D roughnessTexture;
	
	bool haveAOTexture;
    sampler2D aoTexture;

	bool haveNormalTexture;
	sampler2D normalTexture;

	float ior;
};

// ----------------- Uniform

// 48
struct PointLight {
    vec3 position;	// 12	0
    vec3 L;			// 12	16
    float linear;	// 4	28
    float quadratic;// 4	32
};

// 160
layout (std140) uniform Camera{
	mat4 view;			// 64	0	64
	mat4 projection;	// 64	64	64
	vec3 viewPos;		// 12	128	144
	float nearPlane;	// 4	144	148
	float farPlane;		// 4	148	152
	float fov;			// 4	152	156
	float ar;			// 4	156	160
};

// 400
layout (std140) uniform PointLights{
	int numLight;// 16
	PointLight pointLights[MAX_POINT_LIGHTS];// 48 * MAX_POINT_LIGHTS = 48 * 8
};

uniform BSDF_FrostedGlass bsdf;

uniform samplerCube pointLightDepthMap0;
uniform samplerCube pointLightDepthMap1;
uniform samplerCube pointLightDepthMap2;
uniform samplerCube pointLightDepthMap3;
uniform samplerCube pointLightDepthMap4;
uniform samplerCube pointLightDepthMap5;
uniform samplerCube pointLightDepthMap6;
uniform samplerCube pointLightDepthMap7;

uniform float lightFar;

uniform int mode;

uniform float interpolateRatio; // (0, 1]

// ----------------- declaration

vec3 CalcBumpedNormal(vec3 normal, vec3 tangent, sampler2D normalTexture, vec2 texcoord);

float GGX_D(vec3 norm, vec3 h, float alpha);
float Fr(vec3 v, vec3 h, float ior);
float GGX_G1(vec3 norm, vec3 v, vec3 h, float alpha);
float GGX_G(vec3 norm, vec3 wo, vec3 wi, vec3 h, float alpha);
vec3 BSDF(vec3 norm, vec3 wo, vec3 wi, vec3 color, float roughness, float ao, float ior);

float Visibility(vec3 lightToFrag, int id);
float Visibility(vec3 lightToFrag, samplerCube depthMap);

float tanh(float x) {
	float expZ = exp(x);
    float invExpZ = 1 / expZ;
    return (expZ - invExpZ) / (expZ + invExpZ);
}

float smootherstep(float x){
	return ((6*x - 15)*x + 10) * x*x*x;
}

// template declaration
void Model
(
	in float directIllum_R,
	in float directIllum_G,
	in float directIllum_B,
	in float pos_x,
	in float pos_y,
	in float pos_z,
	in float viewDir_x,
	in float viewDir_y,
	in float viewDir_z,
	in float normal_x,
	in float normal_y,
	in float normal_z,
	in float matColor_R,
	in float matColor_G,
	in float matColor_B,
	in float IOR,
	in float roughness,
	
	out float indirectIllum_R,
	out float indirectIllum_G,
	out float indirectIllum_B
);

// ----------------- main

void main() {
	// get attr
	vec3 color = bsdf.colorFactor;
	if(bsdf.haveColorTexture) {
		color *= texture(bsdf.colorTexture, fs_in.TexCoords).xyz;
	}

	float roughness = bsdf.roughnessFactor;
	if(bsdf.haveRoughnessTexture) {
		roughness *= texture(bsdf.roughnessTexture, fs_in.TexCoords).x;
	}

	float ao = 1.0f;
	if(bsdf.haveAOTexture) {
		ao *= texture(bsdf.aoTexture, fs_in.TexCoords).x;
	}

	vec3 wo = normalize(viewPos - fs_in.FragPos);

	vec3 norm = normalize(fs_in.Normal);
	if(bsdf.haveNormalTexture) {
		norm = CalcBumpedNormal(norm, normalize(fs_in.Tangent), bsdf.normalTexture, fs_in.TexCoords);
	}
	
	// sample point light
	vec3 result = vec3(0);
    for(int i = 0; i < numLight; i++) {
		vec3 fragToLight = pointLights[i].position - fs_in.FragPos;
		float dist2 = dot(fragToLight, fragToLight);
		float dist = sqrt(dist2);
		vec3 wi = fragToLight / dist;
		
		float visibility = Visibility(-fragToLight, i);
		if(visibility==0)
			continue;

		vec3 f = BSDF(norm, wo, wi, color, roughness, ao, bsdf.ior);

		float cosTheta = max(dot(wi, norm), 0);
		
		float attenuation = 1.0f + pointLights[i].linear * dist + pointLights[i].quadratic * dist2;
		
		result += visibility * cosTheta / attenuation * f * pointLights[i].L;
	}
	
	vec3 indirectIllum;
	
	Model
	(
		// input
		result.r,
		result.g,
		result.b,
		fs_in.FragPos.x,
		fs_in.FragPos.y,
		fs_in.FragPos.z,
		wo.x,
		wo.y,
		wo.z,
		norm.x,
		norm.y,
		norm.z,
		color.r,
		color.g,
		color.b,
		bsdf.ior,
		roughness,
		
		// output
		indirectIllum.r,
		indirectIllum.g,
		indirectIllum.b
	);
	
	indirectIllum = clamp(indirectIllum, 0, 1);
	
	// gamma correction
	vec3 finalColor;
	if(mode==0)
		finalColor = result;
	else if(mode==1)
		finalColor = indirectIllum;
	else
		finalColor = result + indirectIllum;
		
    FragColor = vec4(sqrt(finalColor), 1.0);
}

// ----------------- definition

float GGX_D(vec3 norm, vec3 h, float alpha) {
	float HoN = dot(h, norm);
	if (HoN <= 0)
		return 0;

	float cos2 = HoN * HoN;
	float alpha2 = alpha * alpha;

	float root = alpha / (cos2 * (alpha2 - 1) + 1);

	return INV_PI * root * root;
}

float Fr(vec3 v, vec3 h, float ior) {
	// angle in low ior media
	float cosTheta = dot(v, h);
	bool entering = cosTheta > 0.0f;
	
	if (!entering) {
		vec3 vInLowIOR = refract(-v, -h, ior);
		if (vInLowIOR == vec3(0))
			return 1;

		cosTheta = dot(vInLowIOR, h);
	}

	float R0 = pow((ior - 1) / (ior + 1), 2);

	float Fr = R0 + (1 - R0) * pow((1 - cosTheta), 5);

	return Fr;
}

float GGX_G1(vec3 norm, vec3 v, vec3 h, float alpha) {
	float NoV = dot(norm, v);
	float HoV = dot(v, h);
	if (NoV * HoV <= 0)
		return 0;

	float alpha2 = alpha * alpha;
	float tan2 = 1.f / (NoV * NoV) - 1.f;
	return 2.f / (1.f + sqrt(1 + alpha2 * tan2));
}

float GGX_G(vec3 norm, vec3 wo, vec3 wi, vec3 h, float alpha) {
	return GGX_G1(norm, wo, h, alpha) * GGX_G1(norm, wi, h, alpha);
}

vec3 BSDF(vec3 norm, vec3 wo, vec3 wi, vec3 color, float roughness, float ao, float ior) {
	float NoWo = dot(norm, wo);
	float NoWi = dot(norm, wi);
	if(NoWo == 0 || NoWi == 0)
		return vec3(0);

	bool entering = NoWo > 0;
	bool isReflect = NoWo * NoWi > 0;

	float alpha = roughness * roughness;

	if (isReflect) {
		vec3 h = normalize(wo + wi);

		h *= sign(dot(h, norm));// let h point to outside
		float bsdfVal = Fr(wo, h, ior) * GGX_D(norm, h, alpha) * GGX_G(norm, wo, wi, h, alpha) / abs(4.f * NoWo * NoWi);
		return bsdfVal * color;
	}
	else {
		float etai = 1.f, etat = ior;
		if (!entering){
			// swap(etai, etat);
			float tmp = etai;
			etai = etat;
			etat = tmp;
		}

		vec3 h = -normalize(etai * wo + etat * wi);

		float HoWo = dot(h, wo);
		float HoWi = dot(h, wi);
		float sqrtDenom = etai * HoWo + etat * HoWi;

		float factor = abs(HoWo * HoWi / (NoWo * NoWi));

		h *= sign(dot(h,norm));// let h point to outside
		float bsdfVal = factor * ((1 - Fr(wo, h, ior)) * GGX_D(norm, h, alpha) * GGX_G(norm, wo, wi, h, alpha) * etat * etat) / (sqrtDenom * sqrtDenom);
		return bsdfVal * color;
	}
}

vec3 CalcBumpedNormal(vec3 normal, vec3 tangent, sampler2D normalTexture, vec2 texcoord) {
    tangent = normalize(tangent - dot(tangent, normal) * normal);
    vec3 bitangent = cross(tangent, normal);
    vec3 bumpMapNormal = texture(normalTexture, texcoord).xyz;
    bumpMapNormal = 2.0 * bumpMapNormal - 1.0;
    mat3 TBN = mat3(tangent, bitangent, normal);
    vec3 newNormal = TBN * bumpMapNormal;
    newNormal = normalize(newNormal);
    return newNormal;
}

float Visibility(vec3 lightToFrag, int id){
	if(id == 0) {
		return Visibility(lightToFrag, pointLightDepthMap0);
	} else if(id == 1) {
		return Visibility(lightToFrag, pointLightDepthMap1);
	} else if(id == 2) {
		return Visibility(lightToFrag, pointLightDepthMap2);
	} else if(id == 3) {
		return Visibility(lightToFrag, pointLightDepthMap3);
	} else if(id == 4) {
		return Visibility(lightToFrag, pointLightDepthMap4);
	} else if(id == 5) {
		return Visibility(lightToFrag, pointLightDepthMap5);
	} else if(id == 6) {
		return Visibility(lightToFrag, pointLightDepthMap6);
	} else if(id == 7) {
		return Visibility(lightToFrag, pointLightDepthMap7);
	}else 
		return 1;// not support id
}

float Visibility(vec3 lightToFrag, samplerCube depthMap) {
	float currentDepth = length(lightToFrag);
	float bias = 0.08;
	int samples = 20;
	float shadow = 0.0;
	float viewDistance = length(viewPos - fs_in.FragPos);
	float diskRadius = (1.0 + (viewDistance / lightFar)) / 50.0;
	for(int i = 0; i < samples; ++i) {
		float closestDepth = lightFar * texture(depthMap, lightToFrag + gridSamplingDisk[i] * diskRadius).r;
		shadow += smoothstep(closestDepth, closestDepth + bias, currentDepth);
	}
	shadow /= float(samples);
	return 1 - shadow;
}

void FNN_13_3_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(-0.0913082)+x1*(-0.224651)+x2*(-0.279435)+x3*(0.238582)+x4*(-0.31455)+x5*(-0.134132)+x6*(0.340369)+x7*(0.0593144)+x8*(-0.329835)+x9*(0.306985)+x10*(0.0132863)+x11*(-0.354564)+x12*(0.338017)+x13*(-0.216994)+x14*(-0.206493)+x15*(0.330907)+x16*(0.278955)+(0.00691864)); 
    h1 = tanh(x0*(0.118439)+x1*(-0.132403)+x2*(0.419734)+x3*(0.38312)+x4*(0.100103)+x5*(0.104092)+x6*(0.454107)+x7*(-0.258549)+x8*(0.389055)+x9*(0.263999)+x10*(0.368552)+x11*(0.401751)+x12*(0.285593)+x13*(0.0379975)+x14*(0.318066)+x15*(0.156944)+x16*(0.283347)+(0.0262278)); 
    h2 = tanh(x0*(0.39469)+x1*(0.313943)+x2*(-0.298142)+x3*(-0.213959)+x4*(0.207602)+x5*(-0.216969)+x6*(0.147042)+x7*(0.266)+x8*(-0.057629)+x9*(-0.341234)+x10*(0.35774)+x11*(0.31798)+x12*(-0.300763)+x13*(-0.331707)+x14*(-0.0608462)+x15*(-0.271133)+x16*(-0.225583)+(0.0159632)); 
    h3 = tanh(x0*(-0.123698)+x1*(0.120652)+x2*(0.358541)+x3*(0.36093)+x4*(0.3377)+x5*(-0.133391)+x6*(-0.130321)+x7*(-0.255101)+x8*(0.41158)+x9*(-0.316754)+x10*(-0.255267)+x11*(0.30791)+x12*(-0.391349)+x13*(-0.355977)+x14*(0.0725176)+x15*(-0.113146)+x16*(-0.244296)+(-0.00147904)); 
    h4 = tanh(x0*(0.124929)+x1*(0.459584)+x2*(-0.26337)+x3*(-0.207986)+x4*(0.00387004)+x5*(-0.216137)+x6*(0.233949)+x7*(-0.117039)+x8*(-0.263359)+x9*(-0.383657)+x10*(-0.0746652)+x11*(0.234543)+x12*(-0.35909)+x13*(-0.264703)+x14*(0.345445)+x15*(0.384781)+x16*(0.0972434)+(0.0874162)); 
    h5 = tanh(x0*(0.129931)+x1*(0.293795)+x2*(-0.229672)+x3*(-0.406871)+x4*(0.156081)+x5*(-0.0418075)+x6*(0.121018)+x7*(0.0316664)+x8*(0.034392)+x9*(-0.129219)+x10*(0.289077)+x11*(-0.123074)+x12*(0.200046)+x13*(0.0719703)+x14*(0.0892091)+x15*(-0.0542266)+x16*(0.205079)+(-0.0424574)); 
    h6 = tanh(x0*(0.358692)+x1*(0.353281)+x2*(-0.0517298)+x3*(-0.105156)+x4*(0.0897885)+x5*(0.0412711)+x6*(0.23487)+x7*(-0.000520212)+x8*(0.211445)+x9*(0.250732)+x10*(-0.0103994)+x11*(-0.38988)+x12*(-0.211858)+x13*(0.304424)+x14*(0.351038)+x15*(0.339029)+x16*(0.379945)+(0.00538447)); 
    h7 = tanh(x0*(-0.0719018)+x1*(-0.0633959)+x2*(-0.240797)+x3*(0.144202)+x4*(0.32085)+x5*(-0.353282)+x6*(-0.0871573)+x7*(-0.358589)+x8*(-0.255646)+x9*(-0.162378)+x10*(0.346664)+x11*(0.145365)+x12*(0.398629)+x13*(0.289399)+x14*(-0.315125)+x15*(-0.0834263)+x16*(-0.25286)+(-0.0113002)); 
    h8 = tanh(x0*(-0.194451)+x1*(-0.248914)+x2*(0.263193)+x3*(-0.170515)+x4*(-0.24923)+x5*(0.000661419)+x6*(-0.304989)+x7*(0.20497)+x8*(0.327481)+x9*(0.0786234)+x10*(0.00145081)+x11*(0.0145399)+x12*(0.109379)+x13*(0.0970389)+x14*(-0.246559)+x15*(0.180766)+x16*(0.360628)+(-0.0169209)); 
    h9 = tanh(x0*(0.418377)+x1*(-0.345259)+x2*(0.0497663)+x3*(-0.309061)+x4*(0.37076)+x5*(-0.215075)+x6*(0.372138)+x7*(-0.240277)+x8*(0.337983)+x9*(0.338655)+x10*(0.213078)+x11*(0.113714)+x12*(-0.23469)+x13*(0.140727)+x14*(-0.0105839)+x15*(-0.0673251)+x16*(-0.0491303)+(0.0395818)); 
    h10 = tanh(x0*(-0.347585)+x1*(-0.348552)+x2*(0.125479)+x3*(-0.0743629)+x4*(0.0236227)+x5*(-0.309684)+x6*(-0.196452)+x7*(-0.331623)+x8*(0.32069)+x9*(-0.167819)+x10*(0.10481)+x11*(0.120285)+x12*(-0.112268)+x13*(-0.298594)+x14*(-0.359158)+x15*(-0.341734)+x16*(0.250987)+(0.0180761)); 
    h11 = tanh(x0*(0.205415)+x1*(0.220992)+x2*(0.104849)+x3*(-0.0856851)+x4*(0.284054)+x5*(-0.262375)+x6*(-0.17831)+x7*(-0.256581)+x8*(0.0582711)+x9*(-0.109409)+x10*(0.13954)+x11*(-0.397882)+x12*(-0.265976)+x13*(0.149645)+x14*(-0.29831)+x15*(0.0740137)+x16*(-0.103518)+(-0.00672813)); 
    h12 = tanh(x0*(-0.00904473)+x1*(0.28871)+x2*(-0.289113)+x3*(0.168429)+x4*(0.157361)+x5*(-0.353593)+x6*(-0.0617664)+x7*(-0.311913)+x8*(0.181063)+x9*(-0.253016)+x10*(-0.0551818)+x11*(-0.314981)+x12*(-0.153535)+x13*(-0.171084)+x14*(0.287131)+x15*(0.333315)+x16*(0.386377)+(0.0046254)); 
    h13 = tanh(x0*(-0.112567)+x1*(0.139096)+x2*(-0.0266165)+x3*(-0.197333)+x4*(-0.0974751)+x5*(-0.119121)+x6*(-0.151444)+x7*(0.0755109)+x8*(0.224711)+x9*(0.306373)+x10*(-0.1045)+x11*(0.166406)+x12*(0.305884)+x13*(0.371119)+x14*(-0.0802397)+x15*(0.220968)+x16*(-0.128877)+(-0.000619932)); 
    h14 = tanh(x0*(-0.173843)+x1*(-0.0392904)+x2*(-0.105026)+x3*(-0.0938852)+x4*(0.341523)+x5*(0.0632053)+x6*(-0.301655)+x7*(-0.260073)+x8*(0.0911912)+x9*(0.402646)+x10*(-0.024834)+x11*(-0.360708)+x12*(0.215797)+x13*(-0.0280026)+x14*(-0.397813)+x15*(-0.110358)+x16*(-0.283283)+(0.0423321)); 
    h15 = tanh(x0*(-0.0185075)+x1*(0.0201556)+x2*(-0.137955)+x3*(0.179752)+x4*(-0.18321)+x5*(0.338455)+x6*(-0.191569)+x7*(0.136117)+x8*(0.126273)+x9*(-0.0970242)+x10*(-0.155187)+x11*(-0.116225)+x12*(-0.211306)+x13*(-0.0313964)+x14*(-0.383144)+x15*(0.377962)+x16*(-0.250901)+(0.00333812)); 
    h16 = tanh(x0*(0.0522604)+x1*(-0.189247)+x2*(-0.39538)+x3*(-0.272691)+x4*(-0.203892)+x5*(-0.236967)+x6*(-0.291483)+x7*(0.114135)+x8*(0.16095)+x9*(-0.161716)+x10*(0.00958973)+x11*(0.174454)+x12*(0.277653)+x13*(0.326635)+x14*(0.329458)+x15*(-0.110295)+x16*(-0.12639)+(0.00772098)); 
    h17 = tanh(x0*(-0.0435399)+x1*(-0.269598)+x2*(0.31091)+x3*(-0.0305765)+x4*(-0.235901)+x5*(-0.179582)+x6*(-0.0340713)+x7*(-0.217742)+x8*(-0.406075)+x9*(0.30386)+x10*(-0.37729)+x11*(-0.275612)+x12*(0.369104)+x13*(0.316774)+x14*(0.240805)+x15*(0.106466)+x16*(-0.11181)+(0.0049675)); 
    h18 = tanh(x0*(-0.349775)+x1*(0.420266)+x2*(-0.367654)+x3*(0.228109)+x4*(0.126039)+x5*(-0.134196)+x6*(-0.260929)+x7*(0.304774)+x8*(0.27003)+x9*(0.261146)+x10*(-0.00736663)+x11*(-0.0151096)+x12*(-0.0851879)+x13*(0.167153)+x14*(-0.0148647)+x15*(0.153967)+x16*(-0.0873468)+(0.0125723)); 
    h19 = tanh(x0*(0.0370718)+x1*(0.0708673)+x2*(0.229903)+x3*(0.0875964)+x4*(-0.169535)+x5*(-0.167387)+x6*(-0.202792)+x7*(-0.134082)+x8*(0.348184)+x9*(-0.177717)+x10*(-0.126333)+x11*(0.126611)+x12*(-0.150734)+x13*(-0.381166)+x14*(-0.154733)+x15*(0.0516313)+x16*(0.185992)+(0.00273203)); 
}

void FNN_13_3_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.313667)+x1*(-0.327878)+x2*(0.275906)+x3*(0.121074)+x4*(-0.307563)+x5*(0.479426)+x6*(-0.0385303)+x7*(0.301647)+x8*(-0.0158154)+x9*(-0.495004)+x10*(0.160638)+x11*(0.1024)+x12*(-0.575017)+x13*(-0.462909)+x14*(0.386138)+x15*(0.0702752)+x16*(-0.38143)+x17*(-0.0486875)+x18*(0.365494)+x19*(-0.408148)+(0.0207091)); 
    h1 = tanh(x0*(-0.125906)+x1*(-0.103331)+x2*(0.0237562)+x3*(-0.0718202)+x4*(-0.223247)+x5*(-0.169403)+x6*(0.259022)+x7*(0.11424)+x8*(0.377858)+x9*(0.248661)+x10*(0.0311171)+x11*(-0.0223402)+x12*(0.298758)+x13*(-0.320148)+x14*(-0.047766)+x15*(0.220996)+x16*(0.457342)+x17*(0.0916259)+x18*(0.11259)+x19*(0.0350298)+(-0.0100437)); 
    h2 = tanh(x0*(-0.178159)+x1*(-0.132145)+x2*(-0.200914)+x3*(-0.134718)+x4*(0.267142)+x5*(-0.388655)+x6*(0.193845)+x7*(-0.272207)+x8*(0.319288)+x9*(-0.34197)+x10*(0.263591)+x11*(-0.276843)+x12*(0.449988)+x13*(0.210592)+x14*(-0.175053)+x15*(0.1173)+x16*(-0.293798)+x17*(0.326876)+x18*(-0.0898968)+x19*(-0.133324)+(-0.0225864)); 
    h3 = tanh(x0*(0.182465)+x1*(-0.00656106)+x2*(0.205822)+x3*(0.0641613)+x4*(-0.391971)+x5*(0.0731042)+x6*(0.374316)+x7*(0.358265)+x8*(-0.237062)+x9*(0.169914)+x10*(-0.339329)+x11*(-0.0591733)+x12*(0.434105)+x13*(0.327012)+x14*(0.196705)+x15*(-0.349484)+x16*(0.172342)+x17*(-0.268717)+x18*(0.0809721)+x19*(-0.203095)+(-0.0385452)); 
    h4 = tanh(x0*(-0.353343)+x1*(-0.339645)+x2*(0.0514981)+x3*(-0.0566632)+x4*(-0.0298518)+x5*(-0.33066)+x6*(0.266813)+x7*(0.380713)+x8*(0.202109)+x9*(0.253533)+x10*(-0.11585)+x11*(0.238485)+x12*(-0.157241)+x13*(-0.161647)+x14*(-0.181669)+x15*(-0.423069)+x16*(0.040185)+x17*(0.0516439)+x18*(0.214612)+x19*(0.327069)+(-0.0108898)); 
    h5 = tanh(x0*(-0.0966381)+x1*(-0.11613)+x2*(-0.349724)+x3*(0.0623183)+x4*(-0.406816)+x5*(0.155072)+x6*(0.2257)+x7*(0.254874)+x8*(-0.345609)+x9*(-0.0412446)+x10*(0.285583)+x11*(0.480397)+x12*(0.648746)+x13*(-0.0896855)+x14*(0.0725116)+x15*(0.182168)+x16*(-0.268464)+x17*(-0.317364)+x18*(0.124509)+x19*(-0.18951)+(-0.014768)); 
    h6 = tanh(x0*(0.16095)+x1*(0.0334438)+x2*(-0.167949)+x3*(-0.246569)+x4*(0.312847)+x5*(0.333193)+x6*(0.290639)+x7*(-0.128625)+x8*(0.367086)+x9*(0.0773487)+x10*(-0.140956)+x11*(0.469644)+x12*(-0.147212)+x13*(-0.330855)+x14*(-0.234534)+x15*(0.308167)+x16*(-0.109479)+x17*(0.00853394)+x18*(-0.333028)+x19*(0.388102)+(-0.00720247)); 
    h7 = tanh(x0*(0.211033)+x1*(-0.24202)+x2*(0.0698457)+x3*(-0.353153)+x4*(-0.0972918)+x5*(-0.0908947)+x6*(-0.227417)+x7*(-0.106733)+x8*(-0.0924297)+x9*(-0.258871)+x10*(-0.219965)+x11*(0.202072)+x12*(0.0611087)+x13*(-0.163259)+x14*(0.450607)+x15*(-0.158167)+x16*(-0.17546)+x17*(0.0858422)+x18*(-0.253263)+x19*(0.423355)+(-0.0173848)); 
    h8 = tanh(x0*(0.20564)+x1*(-0.196332)+x2*(0.00216838)+x3*(-0.286816)+x4*(0.27763)+x5*(-0.100755)+x6*(0.374076)+x7*(0.113311)+x8*(-0.216347)+x9*(-0.231024)+x10*(0.0616209)+x11*(-0.105577)+x12*(0.196983)+x13*(-0.107126)+x14*(0.213879)+x15*(-0.435789)+x16*(0.442047)+x17*(0.367493)+x18*(0.0514566)+x19*(0.267752)+(0.00792031)); 
    h9 = tanh(x0*(-0.325782)+x1*(-0.341088)+x2*(-0.222992)+x3*(0.211771)+x4*(-0.0305719)+x5*(0.340326)+x6*(-0.355529)+x7*(0.439019)+x8*(-0.0633618)+x9*(-0.027648)+x10*(-0.365049)+x11*(0.207468)+x12*(-0.230587)+x13*(0.214121)+x14*(0.20122)+x15*(-0.355686)+x16*(-0.0494944)+x17*(0.41122)+x18*(-0.164651)+x19*(-0.310003)+(-0.0572883)); 
}

void FNN_13_3_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(0.475827)+x1*(0.201054)+x2*(-0.319803)+x3*(-0.352712)+x4*(-0.409556)+x5*(-0.0117598)+x6*(0.3979)+x7*(-0.338736)+x8*(0.461202)+x9*(-0.561954)+(0.0376661); 
    h1 = x0*(-0.314846)+x1*(0.04433)+x2*(0.287955)+x3*(-0.204414)+x4*(0.287618)+x5*(-0.460692)+x6*(-0.0935907)+x7*(0.511202)+x8*(-0.211058)+x9*(-0.562189)+(0.024915); 
    h2 = x0*(0.284706)+x1*(-0.404599)+x2*(-0.0589204)+x3*(-0.0204006)+x4*(-0.0528151)+x5*(-0.156952)+x6*(-0.285202)+x7*(-0.34351)+x8*(-0.418673)+x9*(-0.0590856)+(0.0135947); 
}

void FNN_13_3(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_13_3_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_13_3_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_13_3_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_3(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_13_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void FNN_13_4_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(0.144851)+x1*(-0.167769)+x2*(-0.194743)+x3*(-0.0583936)+x4*(0.352958)+x5*(0.115613)+x6*(0.0221209)+x7*(0.01527)+x8*(0.342002)+x9*(-0.0242409)+x10*(-0.112071)+x11*(-0.301752)+x12*(-0.0419802)+x13*(-0.376029)+x14*(-0.295762)+x15*(0.316299)+x16*(0.346864)+(0.0219909)); 
    h1 = tanh(x0*(-0.348843)+x1*(0.406785)+x2*(-0.00772821)+x3*(0.294819)+x4*(-0.393929)+x5*(0.325492)+x6*(0.10149)+x7*(-0.0111425)+x8*(0.161219)+x9*(-0.0300352)+x10*(-0.14215)+x11*(-0.301156)+x12*(0.101447)+x13*(0.313851)+x14*(0.265948)+x15*(-0.206822)+x16*(-0.361399)+(0.0443479)); 
    h2 = tanh(x0*(-0.30047)+x1*(0.276362)+x2*(0.391847)+x3*(0.247525)+x4*(0.198388)+x5*(0.140623)+x6*(-0.0413931)+x7*(-0.0190756)+x8*(1.61824e-05)+x9*(0.0617833)+x10*(0.0228438)+x11*(-0.196064)+x12*(-0.145995)+x13*(0.0511212)+x14*(0.197893)+x15*(-0.247259)+x16*(-0.293272)+(0.0113919)); 
    h3 = tanh(x0*(-0.104427)+x1*(0.348573)+x2*(0.0989152)+x3*(-0.0525461)+x4*(-0.357154)+x5*(0.355058)+x6*(-0.103692)+x7*(0.0601819)+x8*(-0.147821)+x9*(0.272477)+x10*(0.0903504)+x11*(0.401766)+x12*(0.106031)+x13*(-0.136252)+x14*(-0.373665)+x15*(-0.336259)+x16*(0.082994)+(0.0307921)); 
    h4 = tanh(x0*(0.305213)+x1*(-0.0522353)+x2*(0.331727)+x3*(0.053945)+x4*(-0.102598)+x5*(-0.144364)+x6*(-0.344673)+x7*(0.0815855)+x8*(0.12055)+x9*(-0.375242)+x10*(-0.291659)+x11*(-0.102962)+x12*(-0.354108)+x13*(0.300634)+x14*(-0.276777)+x15*(0.286434)+x16*(-0.341748)+(-0.0673068)); 
    h5 = tanh(x0*(-0.0792323)+x1*(-0.00786525)+x2*(-0.399373)+x3*(-0.0914444)+x4*(0.261783)+x5*(-0.212464)+x6*(-0.419004)+x7*(0.226518)+x8*(-0.0895902)+x9*(0.152465)+x10*(-0.167716)+x11*(-0.0934868)+x12*(-0.39696)+x13*(-0.111173)+x14*(0.147575)+x15*(0.226325)+x16*(-0.358942)+(-0.001478)); 
    h6 = tanh(x0*(0.243002)+x1*(0.233189)+x2*(-0.356692)+x3*(-0.372121)+x4*(-0.133297)+x5*(0.360586)+x6*(-0.0465193)+x7*(-0.288311)+x8*(0.0179377)+x9*(-0.270666)+x10*(-0.295493)+x11*(-0.192503)+x12*(-0.167519)+x13*(-0.30646)+x14*(-0.149512)+x15*(-0.264837)+x16*(0.0368147)+(0.0257364)); 
    h7 = tanh(x0*(-0.13308)+x1*(0.208824)+x2*(0.198584)+x3*(0.342294)+x4*(-0.231244)+x5*(-0.0462319)+x6*(0.0210188)+x7*(-0.180219)+x8*(-0.126489)+x9*(-0.345389)+x10*(-0.270381)+x11*(0.39828)+x12*(0.124341)+x13*(-0.0961423)+x14*(0.197356)+x15*(0.282097)+x16*(-0.119343)+(-0.0545539)); 
    h8 = tanh(x0*(0.24236)+x1*(-0.10968)+x2*(0.0223718)+x3*(-0.133613)+x4*(-0.392553)+x5*(0.0413042)+x6*(-0.297011)+x7*(0.272829)+x8*(-0.278476)+x9*(0.276891)+x10*(-0.0238431)+x11*(0.331002)+x12*(-0.376945)+x13*(0.0547229)+x14*(-0.109012)+x15*(-0.22648)+x16*(-0.400881)+(-0.00118615)); 
    h9 = tanh(x0*(0.0223029)+x1*(-0.185752)+x2*(-0.294336)+x3*(-0.453718)+x4*(-0.237784)+x5*(-0.228055)+x6*(-0.149081)+x7*(-0.0909522)+x8*(0.236657)+x9*(0.1854)+x10*(0.334348)+x11*(0.358962)+x12*(0.157665)+x13*(0.108753)+x14*(-0.0678932)+x15*(-0.113765)+x16*(0.0739116)+(-0.044937)); 
    h10 = tanh(x0*(-0.125975)+x1*(0.0624278)+x2*(0.242179)+x3*(-0.210259)+x4*(-0.344288)+x5*(-0.149811)+x6*(-0.119532)+x7*(-0.100538)+x8*(-0.27386)+x9*(0.110282)+x10*(-0.228932)+x11*(-0.0162759)+x12*(0.401127)+x13*(0.276142)+x14*(0.169639)+x15*(-0.314133)+x16*(0.273169)+(-0.0400991)); 
    h11 = tanh(x0*(-0.124366)+x1*(-0.20303)+x2*(-0.0665562)+x3*(0.143761)+x4*(-0.359278)+x5*(-0.175793)+x6*(-0.223788)+x7*(-0.187529)+x8*(-0.170649)+x9*(0.0318691)+x10*(0.383569)+x11*(-0.171579)+x12*(0.066215)+x13*(0.139847)+x14*(0.175547)+x15*(0.293706)+x16*(-0.17561)+(-0.044729)); 
    h12 = tanh(x0*(0.0985345)+x1*(0.197813)+x2*(-0.169422)+x3*(-0.423136)+x4*(0.0640085)+x5*(-0.378254)+x6*(0.094256)+x7*(0.230572)+x8*(-0.112076)+x9*(0.117829)+x10*(-0.0864882)+x11*(0.399801)+x12*(0.263935)+x13*(0.21563)+x14*(0.215361)+x15*(-0.204288)+x16*(0.000411302)+(0.0130931)); 
    h13 = tanh(x0*(-0.339146)+x1*(-0.234033)+x2*(-0.254522)+x3*(0.031231)+x4*(0.20077)+x5*(0.344475)+x6*(0.317281)+x7*(-0.0771741)+x8*(-0.313133)+x9*(-0.375046)+x10*(-0.165538)+x11*(0.0314763)+x12*(-0.291716)+x13*(-0.140815)+x14*(-0.389529)+x15*(-0.368567)+x16*(-0.0321948)+(0.0257654)); 
    h14 = tanh(x0*(0.158438)+x1*(0.149085)+x2*(0.1909)+x3*(0.179351)+x4*(0.168797)+x5*(0.11126)+x6*(0.0564005)+x7*(-0.447399)+x8*(-0.024576)+x9*(0.210604)+x10*(-0.163266)+x11*(0.162512)+x12*(0.161383)+x13*(0.214957)+x14*(0.194513)+x15*(0.383408)+x16*(0.291245)+(-0.0314127)); 
    h15 = tanh(x0*(-0.328576)+x1*(-0.259861)+x2*(0.130005)+x3*(-0.333429)+x4*(-0.0766641)+x5*(-0.00251307)+x6*(0.0292086)+x7*(0.0448198)+x8*(0.333291)+x9*(-0.0029138)+x10*(0.0957323)+x11*(0.0725238)+x12*(-0.252392)+x13*(-0.402435)+x14*(-0.114758)+x15*(-0.117093)+x16*(-0.293357)+(-0.0329385)); 
    h16 = tanh(x0*(0.213377)+x1*(0.199521)+x2*(0.20075)+x3*(0.114102)+x4*(0.235336)+x5*(-0.0411904)+x6*(0.17096)+x7*(-0.430388)+x8*(0.14576)+x9*(-0.0415481)+x10*(-0.181239)+x11*(0.0489988)+x12*(0.0747458)+x13*(0.253561)+x14*(0.160558)+x15*(0.392551)+x16*(0.0469976)+(-0.033847)); 
    h17 = tanh(x0*(-0.267129)+x1*(-0.0600936)+x2*(0.402117)+x3*(-0.309807)+x4*(-0.118812)+x5*(0.379027)+x6*(0.348499)+x7*(-0.349874)+x8*(0.153877)+x9*(0.157933)+x10*(0.053572)+x11*(-0.125268)+x12*(-0.283053)+x13*(-0.052095)+x14*(-0.177206)+x15*(-0.0384273)+x16*(-0.0800263)+(-0.0130226)); 
    h18 = tanh(x0*(0.229117)+x1*(-0.29423)+x2*(-0.159905)+x3*(-0.25426)+x4*(0.348885)+x5*(0.472688)+x6*(-0.139605)+x7*(-0.090505)+x8*(-0.0809031)+x9*(-0.229789)+x10*(0.0326282)+x11*(0.107702)+x12*(0.257316)+x13*(-0.349761)+x14*(-0.00406295)+x15*(0.0823364)+x16*(-0.178409)+(0.0402962)); 
    h19 = tanh(x0*(0.306105)+x1*(0.177286)+x2*(-0.0560945)+x3*(-0.108945)+x4*(-0.240911)+x5*(-0.172845)+x6*(0.0862985)+x7*(0.0524447)+x8*(-0.0642817)+x9*(-0.127701)+x10*(0.113714)+x11*(-0.0752591)+x12*(0.0527142)+x13*(-0.380019)+x14*(0.354845)+x15*(-0.114763)+x16*(0.297907)+(0.0271792)); 
}

void FNN_13_4_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(0.315841)+x1*(0.0636597)+x2*(0.252732)+x3*(-0.0222507)+x4*(-0.0591666)+x5*(-0.288321)+x6*(0.243938)+x7*(-0.0947485)+x8*(0.0526332)+x9*(-0.338366)+x10*(-0.418364)+x11*(0.041599)+x12*(0.0212007)+x13*(-0.419457)+x14*(0.326945)+x15*(-0.193395)+x16*(0.394539)+x17*(-0.0406887)+x18*(0.294367)+x19*(0.39598)+(0.0383422)); 
    h1 = tanh(x0*(0.401686)+x1*(-0.21209)+x2*(0.28918)+x3*(0.133244)+x4*(-0.492021)+x5*(-0.0870023)+x6*(-0.0315744)+x7*(-0.230049)+x8*(-0.123962)+x9*(-0.0323642)+x10*(-0.268285)+x11*(0.0950596)+x12*(0.0647401)+x13*(0.0953866)+x14*(-0.483021)+x15*(0.258067)+x16*(-0.229777)+x17*(-0.237896)+x18*(-0.0177694)+x19*(0.342673)+(0.0403841)); 
    h2 = tanh(x0*(-0.272599)+x1*(-0.286112)+x2*(0.331295)+x3*(0.020794)+x4*(-0.16381)+x5*(0.0928153)+x6*(-0.300608)+x7*(-0.406183)+x8*(-0.178085)+x9*(0.0489566)+x10*(-0.214875)+x11*(-0.0807989)+x12*(0.053779)+x13*(0.0525394)+x14*(-0.205628)+x15*(0.150804)+x16*(-0.268538)+x17*(-0.0748354)+x18*(-0.184649)+x19*(0.28579)+(-0.0421691)); 
    h3 = tanh(x0*(-0.233246)+x1*(0.22423)+x2*(-0.0167191)+x3*(0.312057)+x4*(-0.195315)+x5*(-0.0429546)+x6*(0.187238)+x7*(-0.371808)+x8*(-0.320545)+x9*(0.126885)+x10*(-0.426515)+x11*(-0.310472)+x12*(-0.306163)+x13*(0.35875)+x14*(-0.218772)+x15*(-0.173897)+x16*(-0.0643583)+x17*(-0.362804)+x18*(-0.344897)+x19*(-0.149008)+(0.0639533)); 
    h4 = tanh(x0*(0.248045)+x1*(-0.0512387)+x2*(0.375847)+x3*(-0.189894)+x4*(0.196739)+x5*(0.0319329)+x6*(0.308633)+x7*(-0.0419842)+x8*(0.0699867)+x9*(0.318689)+x10*(-0.279485)+x11*(0.294465)+x12*(0.330623)+x13*(0.145652)+x14*(0.254382)+x15*(-0.151826)+x16*(-0.339561)+x17*(-0.306992)+x18*(-0.158683)+x19*(0.0923861)+(0.00185067)); 
    h5 = tanh(x0*(-0.300148)+x1*(-0.236478)+x2*(0.301954)+x3*(0.0780754)+x4*(0.502336)+x5*(0.389434)+x6*(-0.0854558)+x7*(-0.313949)+x8*(-0.333044)+x9*(0.197548)+x10*(0.16074)+x11*(-0.277138)+x12*(0.166518)+x13*(0.348823)+x14*(0.0187068)+x15*(0.34258)+x16*(0.29597)+x17*(-0.38004)+x18*(0.0428765)+x19*(-0.445811)+(-0.0175612)); 
    h6 = tanh(x0*(0.268878)+x1*(-0.216387)+x2*(-0.48068)+x3*(0.327515)+x4*(-0.420334)+x5*(-0.166799)+x6*(0.161909)+x7*(-0.302187)+x8*(0.0173625)+x9*(-0.219953)+x10*(0.349928)+x11*(0.119156)+x12*(0.0353491)+x13*(0.164844)+x14*(-0.0326859)+x15*(0.215882)+x16*(-0.0994521)+x17*(0.0996817)+x18*(0.262214)+x19*(0.0744282)+(0.00211967)); 
    h7 = tanh(x0*(-0.100701)+x1*(-0.392109)+x2*(0.049918)+x3*(-0.357174)+x4*(-0.328134)+x5*(-0.0398418)+x6*(-0.105592)+x7*(0.373842)+x8*(-0.00334094)+x9*(0.273929)+x10*(-0.0628091)+x11*(-0.386971)+x12*(-0.0792758)+x13*(-0.184233)+x14*(-0.0579046)+x15*(-0.394228)+x16*(-0.204857)+x17*(0.302257)+x18*(0.268447)+x19*(-0.125873)+(-0.00631768)); 
    h8 = tanh(x0*(0.12)+x1*(-0.31017)+x2*(-0.0634587)+x3*(-0.13501)+x4*(-0.412565)+x5*(-0.402452)+x6*(-0.205152)+x7*(0.106017)+x8*(0.00932685)+x9*(-0.0531172)+x10*(0.180279)+x11*(-0.146401)+x12*(-0.343487)+x13*(0.0745709)+x14*(0.472204)+x15*(0.322576)+x16*(0.223637)+x17*(-0.003642)+x18*(-0.02665)+x19*(-0.417897)+(-0.0419611)); 
    h9 = tanh(x0*(-0.335341)+x1*(0.0152174)+x2*(0.181721)+x3*(-0.120225)+x4*(0.15735)+x5*(0.00505128)+x6*(0.0115702)+x7*(-0.291545)+x8*(-0.1769)+x9*(0.250804)+x10*(0.0441245)+x11*(0.394172)+x12*(0.110407)+x13*(-0.180581)+x14*(-0.409017)+x15*(-0.221147)+x16*(0.124981)+x17*(-0.0903621)+x18*(-0.433129)+x19*(-0.145018)+(-0.0427168)); 
}

void FNN_13_4_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(0.199851)+x1*(0.120922)+x2*(0.134164)+x3*(0.206283)+x4*(0.564478)+x5*(-0.233949)+x6*(0.0636808)+x7*(-0.382522)+x8*(-0.364702)+x9*(-0.668415)+(0.0277466); 
    h1 = x0*(0.391731)+x1*(0.238546)+x2*(-0.14535)+x3*(0.310091)+x4*(0.559993)+x5*(-0.473965)+x6*(0.03166)+x7*(0.211795)+x8*(-0.462862)+x9*(0.0853229)+(0.0168469); 
    h2 = x0*(-0.0818022)+x1*(0.113119)+x2*(-0.270172)+x3*(0.537595)+x4*(-0.430384)+x5*(0.278925)+x6*(-0.253265)+x7*(0.136422)+x8*(-0.635571)+x9*(-0.438257)+(0.0257648); 
}

void FNN_13_4(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_13_4_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_13_4_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_13_4_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_4(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_13_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_2(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x6 < lowBound ) {
        ModelKDTree_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x6 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_3(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x6 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_4(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void FNN_13_5_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(0.23426)+x1*(-0.0256565)+x2*(0.096815)+x3*(-0.088648)+x4*(0.125174)+x5*(-0.179812)+x6*(0.252818)+x7*(0.36111)+x8*(-0.151341)+x9*(0.135046)+x10*(0.028885)+x11*(0.326724)+x12*(-0.348585)+x13*(0.237947)+x14*(0.0840446)+x15*(0.361801)+x16*(-0.391188)+(0.011218)); 
    h1 = tanh(x0*(-0.0713037)+x1*(0.25402)+x2*(0.126138)+x3*(-0.103751)+x4*(0.0516665)+x5*(0.443312)+x6*(-0.301979)+x7*(-0.0111356)+x8*(-0.00167689)+x9*(-0.0527996)+x10*(-0.24015)+x11*(0.390676)+x12*(0.196361)+x13*(-0.0571808)+x14*(-0.357258)+x15*(-0.28366)+x16*(0.263431)+(0.0188183)); 
    h2 = tanh(x0*(-0.158316)+x1*(0.314207)+x2*(-0.315575)+x3*(0.0725701)+x4*(-0.337998)+x5*(0.0718698)+x6*(0.145875)+x7*(0.363474)+x8*(-0.151465)+x9*(-0.384623)+x10*(-0.00678712)+x11*(-0.393462)+x12*(-0.373291)+x13*(0.148547)+x14*(-0.207845)+x15*(0.384247)+x16*(0.246827)+(0.0191663)); 
    h3 = tanh(x0*(0.295653)+x1*(-0.154419)+x2*(0.327622)+x3*(0.122241)+x4*(-0.0676036)+x5*(-0.413867)+x6*(-0.0470196)+x7*(0.0834002)+x8*(-0.0234675)+x9*(0.06237)+x10*(0.337492)+x11*(-0.14958)+x12*(-0.272154)+x13*(0.0684871)+x14*(0.174841)+x15*(0.290193)+x16*(-0.0528095)+(-0.00223313)); 
    h4 = tanh(x0*(-0.374516)+x1*(0.000874308)+x2*(-0.29423)+x3*(0.110604)+x4*(-0.391025)+x5*(-0.167226)+x6*(-0.234796)+x7*(0.00598418)+x8*(-0.179948)+x9*(0.0617051)+x10*(0.229605)+x11*(-0.176937)+x12*(0.025501)+x13*(-0.183697)+x14*(0.0789695)+x15*(-0.288212)+x16*(-0.156453)+(0.0176919)); 
    h5 = tanh(x0*(-0.314096)+x1*(-0.165838)+x2*(-0.377906)+x3*(0.0258865)+x4*(-0.151801)+x5*(-0.0037946)+x6*(-0.0353394)+x7*(-0.239684)+x8*(-0.0269828)+x9*(-0.370255)+x10*(-0.34561)+x11*(-0.0301636)+x12*(0.301552)+x13*(-0.118592)+x14*(-0.328504)+x15*(-0.0151969)+x16*(0.277752)+(-0.078688)); 
    h6 = tanh(x0*(0.339457)+x1*(0.301346)+x2*(0.0867165)+x3*(0.370302)+x4*(0.318494)+x5*(0.320018)+x6*(0.280486)+x7*(0.152561)+x8*(0.0536627)+x9*(0.117711)+x10*(0.114459)+x11*(0.326871)+x12*(0.0202199)+x13*(0.253352)+x14*(-0.365478)+x15*(-0.191538)+x16*(0.0770272)+(-0.0194203)); 
    h7 = tanh(x0*(-0.166368)+x1*(0.328658)+x2*(-0.249449)+x3*(0.0867386)+x4*(-0.127447)+x5*(0.264507)+x6*(0.279276)+x7*(0.171021)+x8*(-0.20082)+x9*(-0.314429)+x10*(-0.293712)+x11*(0.363109)+x12*(0.095706)+x13*(-0.111387)+x14*(-0.0885375)+x15*(-0.37068)+x16*(-0.245539)+(-0.0360411)); 
    h8 = tanh(x0*(0.0663122)+x1*(0.363579)+x2*(-0.0361933)+x3*(-0.0981123)+x4*(0.276718)+x5*(0.0138305)+x6*(0.0274313)+x7*(-0.299065)+x8*(-0.11663)+x9*(0.286689)+x10*(-0.223285)+x11*(0.182218)+x12*(-0.256806)+x13*(0.260118)+x14*(0.292379)+x15*(0.209944)+x16*(0.375906)+(-0.00271404)); 
    h9 = tanh(x0*(0.334363)+x1*(-0.121515)+x2*(-0.352598)+x3*(-0.252037)+x4*(0.386134)+x5*(-0.302681)+x6*(0.275113)+x7*(-0.182609)+x8*(0.329356)+x9*(-0.109453)+x10*(0.360995)+x11*(0.344697)+x12*(0.321561)+x13*(0.284468)+x14*(-0.276457)+x15*(-0.29826)+x16*(-0.032048)+(-0.0236366)); 
    h10 = tanh(x0*(0.370251)+x1*(-0.0778158)+x2*(-0.270544)+x3*(-0.338146)+x4*(0.0614213)+x5*(0.0266434)+x6*(-0.195999)+x7*(-0.301528)+x8*(-0.389259)+x9*(0.118521)+x10*(-0.0709529)+x11*(0.176917)+x12*(0.143735)+x13*(-0.286839)+x14*(-0.0554905)+x15*(-0.0356588)+x16*(-0.212235)+(-0.0253253)); 
    h11 = tanh(x0*(0.335185)+x1*(-0.056821)+x2*(-0.238043)+x3*(-0.0961)+x4*(-0.250288)+x5*(0.109639)+x6*(0.235384)+x7*(0.135007)+x8*(-0.162746)+x9*(0.352627)+x10*(-0.106724)+x11*(0.149779)+x12*(0.150361)+x13*(-0.299107)+x14*(-0.101639)+x15*(-0.370637)+x16*(-0.27714)+(-0.0264062)); 
    h12 = tanh(x0*(-0.0288598)+x1*(-0.231612)+x2*(0.0615662)+x3*(0.12715)+x4*(-0.380478)+x5*(-0.0378416)+x6*(0.173971)+x7*(-0.000670048)+x8*(0.329047)+x9*(0.317428)+x10*(-0.235825)+x11*(0.214559)+x12*(-0.368434)+x13*(-0.163224)+x14*(-0.303991)+x15*(0.0051367)+x16*(0.176247)+(0.0645947)); 
    h13 = tanh(x0*(0.0587826)+x1*(0.333824)+x2*(-0.319696)+x3*(-0.186777)+x4*(0.187495)+x5*(0.278572)+x6*(0.14524)+x7*(0.0551653)+x8*(0.0659397)+x9*(0.0396806)+x10*(0.294111)+x11*(-0.312595)+x12*(-0.382459)+x13*(0.149602)+x14*(0.10146)+x15*(0.10603)+x16*(0.287102)+(0.0280556)); 
    h14 = tanh(x0*(0.255537)+x1*(-0.166387)+x2*(-0.167332)+x3*(0.0289069)+x4*(-0.331011)+x5*(-0.321758)+x6*(0.257922)+x7*(0.398898)+x8*(-0.241091)+x9*(-0.0885895)+x10*(0.0351776)+x11*(-0.154313)+x12*(0.236838)+x13*(0.389145)+x14*(-0.314295)+x15*(0.0598424)+x16*(-0.280445)+(-0.0137809)); 
    h15 = tanh(x0*(-0.290117)+x1*(-0.186246)+x2*(-0.386869)+x3*(-0.25235)+x4*(-0.217028)+x5*(0.180228)+x6*(-0.397509)+x7*(-0.0861467)+x8*(-0.420322)+x9*(-0.34384)+x10*(-0.168041)+x11*(0.0127056)+x12*(0.157045)+x13*(0.205731)+x14*(0.328083)+x15*(0.295405)+x16*(0.0158576)+(-0.0310437)); 
    h16 = tanh(x0*(-0.213257)+x1*(-0.35833)+x2*(-0.208993)+x3*(-0.118671)+x4*(0.177576)+x5*(-0.119225)+x6*(0.0884709)+x7*(-0.0775379)+x8*(-0.0873221)+x9*(0.10926)+x10*(0.205358)+x11*(0.220851)+x12*(0.0164327)+x13*(-0.158803)+x14*(-0.261449)+x15*(-0.217594)+x16*(0.100118)+(-0.0194359)); 
    h17 = tanh(x0*(-0.146838)+x1*(-0.338209)+x2*(0.370436)+x3*(0.0425749)+x4*(-0.27912)+x5*(-0.299992)+x6*(0.0139275)+x7*(0.169005)+x8*(0.326351)+x9*(0.0749864)+x10*(-0.40044)+x11*(0.399819)+x12*(-0.003142)+x13*(0.329805)+x14*(0.0269044)+x15*(0.0803438)+x16*(0.115167)+(0.0663853)); 
    h18 = tanh(x0*(0.217112)+x1*(-0.324291)+x2*(0.086367)+x3*(-0.313203)+x4*(0.360803)+x5*(0.0273686)+x6*(0.28771)+x7*(0.17595)+x8*(-0.00876048)+x9*(0.286208)+x10*(-0.143509)+x11*(-0.133306)+x12*(0.049706)+x13*(0.0931846)+x14*(0.330147)+x15*(0.173726)+x16*(0.00337541)+(0.0573634)); 
    h19 = tanh(x0*(0.252322)+x1*(0.337204)+x2*(0.117272)+x3*(0.205667)+x4*(0.360321)+x5*(-0.123411)+x6*(0.179504)+x7*(-0.163771)+x8*(-0.167807)+x9*(-0.286591)+x10*(-0.344601)+x11*(-0.130387)+x12*(-0.395476)+x13*(-0.0926524)+x14*(-0.219687)+x15*(0.0983483)+x16*(-0.161128)+(-0.00779618)); 
}

void FNN_13_5_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(0.143067)+x1*(-0.047928)+x2*(0.137237)+x3*(-0.0643565)+x4*(0.346403)+x5*(-0.427878)+x6*(0.135381)+x7*(0.35276)+x8*(-0.122376)+x9*(0.107604)+x10*(0.369877)+x11*(-0.314939)+x12*(0.125785)+x13*(0.338534)+x14*(0.301461)+x15*(-0.32417)+x16*(0.352005)+x17*(0.286565)+x18*(0.0638553)+x19*(-0.288905)+(0.0246364)); 
    h1 = tanh(x0*(0.141527)+x1*(-0.10334)+x2*(-0.247261)+x3*(0.158076)+x4*(-0.0242977)+x5*(-0.178509)+x6*(-0.288136)+x7*(0.496061)+x8*(-0.244795)+x9*(0.475643)+x10*(0.0796618)+x11*(0.389402)+x12*(-0.250248)+x13*(-0.0707843)+x14*(0.199452)+x15*(-0.14727)+x16*(-0.393468)+x17*(-0.0992765)+x18*(0.0535257)+x19*(0.456144)+(0.0180503)); 
    h2 = tanh(x0*(0.0915853)+x1*(0.478959)+x2*(-0.0418842)+x3*(-0.457648)+x4*(0.272167)+x5*(0.431585)+x6*(0.0725506)+x7*(0.182107)+x8*(-0.429649)+x9*(-0.355719)+x10*(0.32953)+x11*(0.215695)+x12*(-0.33606)+x13*(-0.0155852)+x14*(0.0316719)+x15*(-0.33008)+x16*(-0.247012)+x17*(0.125123)+x18*(0.3715)+x19*(-0.364047)+(0.0275877)); 
    h3 = tanh(x0*(0.292835)+x1*(0.100685)+x2*(0.0204605)+x3*(-0.339263)+x4*(0.406304)+x5*(0.145762)+x6*(0.0239851)+x7*(0.301471)+x8*(0.27817)+x9*(-0.0944642)+x10*(0.14392)+x11*(0.128769)+x12*(0.170164)+x13*(-0.295936)+x14*(0.0148044)+x15*(0.0317424)+x16*(-0.141835)+x17*(0.365866)+x18*(-0.301094)+x19*(-0.162909)+(-0.0232296)); 
    h4 = tanh(x0*(0.243636)+x1*(0.0561184)+x2*(-0.367)+x3*(-0.136531)+x4*(0.0528566)+x5*(0.220987)+x6*(0.112099)+x7*(0.289839)+x8*(0.11245)+x9*(0.260087)+x10*(0.381827)+x11*(0.243553)+x12*(-0.348764)+x13*(0.362386)+x14*(0.262637)+x15*(0.290605)+x16*(0.153878)+x17*(0.433761)+x18*(-0.0476712)+x19*(-0.134589)+(-0.0241004)); 
    h5 = tanh(x0*(-0.433721)+x1*(-0.0376664)+x2*(0.0205733)+x3*(-0.0133885)+x4*(-0.260392)+x5*(-0.266452)+x6*(-0.251758)+x7*(-0.0311159)+x8*(-0.383422)+x9*(-0.239576)+x10*(-0.225486)+x11*(0.385572)+x12*(0.393945)+x13*(0.0951772)+x14*(0.198537)+x15*(-0.408901)+x16*(0.0720069)+x17*(-0.110504)+x18*(0.0828134)+x19*(0.427228)+(0.0216543)); 
    h6 = tanh(x0*(-0.27048)+x1*(0.047815)+x2*(-0.321877)+x3*(-0.168266)+x4*(0.0118967)+x5*(0.190552)+x6*(0.416268)+x7*(0.167909)+x8*(-0.00988642)+x9*(0.305199)+x10*(0.078185)+x11*(0.38602)+x12*(-0.0444968)+x13*(-0.135575)+x14*(0.257083)+x15*(0.237487)+x16*(0.0661363)+x17*(-0.327046)+x18*(0.262529)+x19*(-0.256954)+(-0.028616)); 
    h7 = tanh(x0*(0.110648)+x1*(-0.505166)+x2*(-0.228787)+x3*(0.171327)+x4*(-0.342292)+x5*(0.294031)+x6*(-0.149149)+x7*(0.254499)+x8*(-0.389982)+x9*(0.0849156)+x10*(0.346598)+x11*(-0.226449)+x12*(-0.00192335)+x13*(0.381761)+x14*(-0.194523)+x15*(0.00155022)+x16*(-0.0732487)+x17*(0.289136)+x18*(-0.308204)+x19*(-0.072535)+(-0.0230877)); 
    h8 = tanh(x0*(0.0621774)+x1*(0.212897)+x2*(0.234546)+x3*(-0.464096)+x4*(-0.24539)+x5*(0.380613)+x6*(-0.382131)+x7*(0.0817119)+x8*(-0.21731)+x9*(-0.421494)+x10*(-0.386017)+x11*(0.458053)+x12*(-0.307283)+x13*(0.246242)+x14*(0.322288)+x15*(-0.0840058)+x16*(-0.294163)+x17*(-0.355617)+x18*(-0.160663)+x19*(-0.427352)+(0.0182095)); 
    h9 = tanh(x0*(-0.0377471)+x1*(-0.140781)+x2*(0.246302)+x3*(0.384818)+x4*(-0.314236)+x5*(-0.295913)+x6*(0.220484)+x7*(0.0866452)+x8*(-0.207855)+x9*(-0.306025)+x10*(0.405286)+x11*(0.269215)+x12*(-0.0902809)+x13*(-0.256303)+x14*(-0.111079)+x15*(0.0372189)+x16*(0.451089)+x17*(0.23075)+x18*(-0.215774)+x19*(-0.0806179)+(-0.023496)); 
}

void FNN_13_5_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(0.0373894)+x1*(-0.277882)+x2*(0.329011)+x3*(-0.560798)+x4*(0.405613)+x5*(0.660479)+x6*(-0.302904)+x7*(0.119005)+x8*(0.289434)+x9*(0.116302)+(0.0376351); 
    h1 = x0*(0.423864)+x1*(-0.0350577)+x2*(-0.221243)+x3*(-0.2568)+x4*(-0.327898)+x5*(-0.283811)+x6*(-0.220581)+x7*(-0.56668)+x8*(0.478475)+x9*(-0.457809)+(0.0257986); 
    h2 = x0*(0.284537)+x1*(0.302777)+x2*(0.425078)+x3*(0.427892)+x4*(-0.557176)+x5*(-0.0234519)+x6*(-0.272396)+x7*(0.088043)+x8*(0.204821)+x9*(-0.395551)+(0.025252); 
}

void FNN_13_5(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_13_5_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_13_5_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_13_5_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_5(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_13_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_1(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x5 < lowBound ) {
        ModelKDTree_2(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x5 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_2(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x5 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_5(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void FNN_13_6_Layer_0_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9,out float h10,out float h11,out float h12,out float h13,out float h14,out float h15,out float h16,out float h17,out float h18,out float h19)
{
    h0 = tanh(x0*(0.152452)+x1*(0.234603)+x2*(0.249836)+x3*(-0.189114)+x4*(0.13102)+x5*(-0.502304)+x6*(-0.129887)+x7*(-0.189203)+x8*(-0.38521)+x9*(-0.162675)+x10*(0.188425)+x11*(-0.24005)+x12*(0.129546)+x13*(0.150547)+x14*(-0.25779)+x15*(-0.276691)+x16*(-0.0758526)+(-0.0817585)); 
    h1 = tanh(x0*(0.277727)+x1*(0.300049)+x2*(-0.362403)+x3*(0.230293)+x4*(-0.453059)+x5*(0.691635)+x6*(0.13122)+x7*(-0.307233)+x8*(0.208525)+x9*(0.323219)+x10*(-0.0851973)+x11*(0.161867)+x12*(0.0179093)+x13*(-0.348764)+x14*(0.326896)+x15*(0.214268)+x16*(-0.216364)+(-0.117209)); 
    h2 = tanh(x0*(-0.0081543)+x1*(-0.0729366)+x2*(0.0161713)+x3*(0.0761399)+x4*(0.500233)+x5*(-0.519074)+x6*(0.111967)+x7*(-0.289951)+x8*(-0.26057)+x9*(-0.162422)+x10*(0.302963)+x11*(-0.324204)+x12*(-0.397238)+x13*(0.399856)+x14*(-0.264226)+x15*(-0.262101)+x16*(0.0506279)+(0.107635)); 
    h3 = tanh(x0*(0.128959)+x1*(0.220873)+x2*(0.188419)+x3*(-0.0740575)+x4*(0.391552)+x5*(0.342102)+x6*(0.314987)+x7*(0.16737)+x8*(0.054554)+x9*(0.0484948)+x10*(0.208819)+x11*(-0.154391)+x12*(0.0692805)+x13*(0.00122508)+x14*(-0.0757466)+x15*(0.0653404)+x16*(-0.124539)+(0.0653089)); 
    h4 = tanh(x0*(0.325422)+x1*(-0.410333)+x2*(0.318816)+x3*(-0.088778)+x4*(-0.226228)+x5*(-0.0142792)+x6*(-0.130645)+x7*(0.0937002)+x8*(-0.0969335)+x9*(-0.218905)+x10*(-0.379334)+x11*(-0.11547)+x12*(0.340703)+x13*(0.264095)+x14*(0.106404)+x15*(-0.178506)+x16*(0.308899)+(0.00355437)); 
    h5 = tanh(x0*(0.394184)+x1*(0.112613)+x2*(0.0430186)+x3*(-0.0247047)+x4*(-0.189787)+x5*(0.11543)+x6*(0.141948)+x7*(-0.51888)+x8*(0.377199)+x9*(0.1781)+x10*(-0.383575)+x11*(0.319178)+x12*(0.337282)+x13*(0.374778)+x14*(0.107891)+x15*(-0.309737)+x16*(0.183294)+(0.0623761)); 
    h6 = tanh(x0*(-0.455684)+x1*(0.135566)+x2*(-0.33677)+x3*(0.308741)+x4*(0.248427)+x5*(0.25384)+x6*(0.144673)+x7*(-0.40081)+x8*(0.469066)+x9*(0.0573543)+x10*(0.12236)+x11*(-0.279186)+x12*(0.338184)+x13*(-0.343628)+x14*(-0.382927)+x15*(0.291294)+x16*(0.185071)+(0.0478802)); 
    h7 = tanh(x0*(0.0626376)+x1*(0.316716)+x2*(0.420212)+x3*(0.169886)+x4*(-0.0073833)+x5*(0.509255)+x6*(0.00515226)+x7*(0.0203302)+x8*(0.0296467)+x9*(-0.383844)+x10*(0.0605658)+x11*(-0.235854)+x12*(0.291685)+x13*(-0.218007)+x14*(0.122518)+x15*(0.294221)+x16*(0.229441)+(-0.0697777)); 
    h8 = tanh(x0*(-0.332324)+x1*(0.378625)+x2*(-0.0892036)+x3*(0.251905)+x4*(-0.162941)+x5*(-0.342481)+x6*(-0.287119)+x7*(0.181518)+x8*(0.309807)+x9*(0.064649)+x10*(-0.213318)+x11*(0.0785899)+x12*(0.366326)+x13*(0.0260545)+x14*(0.0831052)+x15*(-0.136962)+x16*(-0.232015)+(0.0539429)); 
    h9 = tanh(x0*(0.13503)+x1*(-0.266263)+x2*(-0.350761)+x3*(0.261826)+x4*(0.261502)+x5*(-0.326461)+x6*(0.258258)+x7*(-0.127954)+x8*(0.194954)+x9*(-0.159015)+x10*(0.0458555)+x11*(0.202097)+x12*(0.00229779)+x13*(-0.257437)+x14*(0.0600727)+x15*(0.0496373)+x16*(-0.194248)+(0.0364876)); 
    h10 = tanh(x0*(-0.0476861)+x1*(-0.0456387)+x2*(-0.0828341)+x3*(0.221444)+x4*(-0.339765)+x5*(0.0604621)+x6*(0.333834)+x7*(-0.092862)+x8*(0.345805)+x9*(-0.308732)+x10*(0.214371)+x11*(0.0214811)+x12*(0.0989069)+x13*(-0.0731245)+x14*(0.0306008)+x15*(0.121541)+x16*(0.221178)+(0.0300433)); 
    h11 = tanh(x0*(0.00692206)+x1*(0.0957523)+x2*(0.146325)+x3*(0.607841)+x4*(0.0829252)+x5*(0.168425)+x6*(-0.204162)+x7*(-0.345426)+x8*(0.0326255)+x9*(0.0450019)+x10*(0.0237696)+x11*(-0.293987)+x12*(0.240036)+x13*(0.371229)+x14*(-0.230672)+x15*(-0.257527)+x16*(0.292491)+(0.129858)); 
    h12 = tanh(x0*(-0.263573)+x1*(0.341623)+x2*(-0.153733)+x3*(0.114372)+x4*(-0.314623)+x5*(-0.00588009)+x6*(-0.239276)+x7*(-0.126649)+x8*(-0.177587)+x9*(0.396619)+x10*(0.24995)+x11*(-0.390595)+x12*(0.376483)+x13*(-0.375449)+x14*(0.182724)+x15*(0.0909608)+x16*(-0.267995)+(-0.0121665)); 
    h13 = tanh(x0*(-0.253745)+x1*(0.231299)+x2*(0.0464338)+x3*(0.438452)+x4*(-0.274646)+x5*(-0.227148)+x6*(0.158733)+x7*(-0.0373777)+x8*(0.104579)+x9*(-0.366193)+x10*(-0.195182)+x11*(-0.0201446)+x12*(-0.138898)+x13*(0.396372)+x14*(-0.331151)+x15*(-0.211041)+x16*(-0.346028)+(-0.018574)); 
    h14 = tanh(x0*(-0.0784919)+x1*(0.417802)+x2*(0.13933)+x3*(-0.0880527)+x4*(-0.282159)+x5*(-0.603966)+x6*(-0.253691)+x7*(-0.13083)+x8*(-0.0737455)+x9*(0.226143)+x10*(0.247404)+x11*(0.00834948)+x12*(0.384763)+x13*(-0.00670704)+x14*(-0.183551)+x15*(0.24504)+x16*(-0.35246)+(0.03346)); 
    h15 = tanh(x0*(0.170639)+x1*(-0.289503)+x2*(-0.0515672)+x3*(-0.280842)+x4*(0.407401)+x5*(-0.122075)+x6*(-0.362156)+x7*(0.327764)+x8*(-0.335529)+x9*(0.107839)+x10*(0.0865853)+x11*(0.0376667)+x12*(-0.127597)+x13*(-0.281752)+x14*(0.0492927)+x15*(-0.0239291)+x16*(0.219048)+(0.0305092)); 
    h16 = tanh(x0*(-0.157432)+x1*(-0.00800844)+x2*(-0.280333)+x3*(-0.1499)+x4*(0.441327)+x5*(-0.411536)+x6*(0.137365)+x7*(0.122343)+x8*(0.0935035)+x9*(-0.0424552)+x10*(-0.26206)+x11*(0.306834)+x12*(0.0851425)+x13*(-0.361094)+x14*(-0.00593129)+x15*(-0.271912)+x16*(0.131215)+(0.0475906)); 
    h17 = tanh(x0*(-0.321831)+x1*(-0.314115)+x2*(0.149782)+x3*(-0.0361685)+x4*(0.422965)+x5*(-0.554152)+x6*(-0.0207671)+x7*(0.358389)+x8*(0.488635)+x9*(-0.0468155)+x10*(0.227087)+x11*(0.370904)+x12*(0.199939)+x13*(0.0580944)+x14*(0.348102)+x15*(-0.275438)+x16*(-0.124933)+(0.0777455)); 
    h18 = tanh(x0*(0.292411)+x1*(0.0761664)+x2*(0.11714)+x3*(-0.225004)+x4*(-0.311329)+x5*(0.14898)+x6*(-0.322751)+x7*(-0.0943206)+x8*(-0.144003)+x9*(0.218405)+x10*(0.243245)+x11*(-0.10342)+x12*(-0.274388)+x13*(0.115357)+x14*(-0.0306088)+x15*(0.0540698)+x16*(0.351576)+(-0.0174328)); 
    h19 = tanh(x0*(0.0599437)+x1*(0.27761)+x2*(-0.382247)+x3*(-0.39701)+x4*(0.454846)+x5*(0.252538)+x6*(0.251601)+x7*(0.232389)+x8*(0.255483)+x9*(0.0393983)+x10*(0.31308)+x11*(0.374649)+x12*(-0.371128)+x13*(0.394039)+x14*(0.328945)+x15*(0.233019)+x16*(0.363313)+(0.0922499)); 
}

void FNN_13_6_Layer_1_Dense_tanh(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,float x17,float x18,float x19,
    out float h0,out float h1,out float h2,out float h3,out float h4,out float h5,out float h6,out float h7,out float h8,out float h9)
{
    h0 = tanh(x0*(-0.220481)+x1*(-0.230219)+x2*(0.220962)+x3*(0.0809818)+x4*(0.257598)+x5*(-0.246092)+x6*(0.308094)+x7*(-0.210837)+x8*(0.175649)+x9*(0.0291918)+x10*(-0.179267)+x11*(-0.267562)+x12*(-0.299923)+x13*(-0.00325437)+x14*(-0.259588)+x15*(0.360886)+x16*(0.107365)+x17*(-0.0360397)+x18*(0.392371)+x19*(-0.155707)+(0.000778231)); 
    h1 = tanh(x0*(0.274283)+x1*(-0.03622)+x2*(-0.219394)+x3*(-0.25886)+x4*(-0.0722293)+x5*(-0.159578)+x6*(0.20687)+x7*(0.0159237)+x8*(-0.360336)+x9*(-0.191395)+x10*(0.0264899)+x11*(0.109154)+x12*(0.0228468)+x13*(0.353973)+x14*(0.19568)+x15*(0.264353)+x16*(-0.178709)+x17*(-0.0449943)+x18*(0.481683)+x19*(-0.0487619)+(-0.018871)); 
    h2 = tanh(x0*(-0.113047)+x1*(-0.0789383)+x2*(-0.00736216)+x3*(0.118931)+x4*(0.229599)+x5*(-0.119229)+x6*(0.165926)+x7*(-0.0283235)+x8*(0.0369002)+x9*(-0.375707)+x10*(-0.181687)+x11*(0.176295)+x12*(0.169721)+x13*(-0.41988)+x14*(0.0140809)+x15*(0.325674)+x16*(0.156565)+x17*(0.0299971)+x18*(-0.140112)+x19*(-0.36731)+(0.00877221)); 
    h3 = tanh(x0*(0.118579)+x1*(0.484058)+x2*(-1.24657)+x3*(-0.298668)+x4*(-0.0674136)+x5*(-0.362161)+x6*(0.270628)+x7*(-0.15086)+x8*(-0.497801)+x9*(-0.285068)+x10*(0.011206)+x11*(0.0686605)+x12*(0.142329)+x13*(-0.312629)+x14*(0.422641)+x15*(-0.0168856)+x16*(-0.335433)+x17*(-0.505843)+x18*(0.211318)+x19*(0.0632146)+(-0.0396924)); 
    h4 = tanh(x0*(-0.196356)+x1*(-0.186282)+x2*(0.118063)+x3*(-0.158149)+x4*(0.108165)+x5*(0.339181)+x6*(0.219625)+x7*(0.0850101)+x8*(-0.425664)+x9*(-0.0586596)+x10*(-0.0077199)+x11*(-0.0031904)+x12*(0.029316)+x13*(-0.291787)+x14*(0.0862942)+x15*(-0.221323)+x16*(0.163774)+x17*(0.267263)+x18*(-0.339122)+x19*(0.193671)+(0.0389905)); 
    h5 = tanh(x0*(-0.199333)+x1*(0.146665)+x2*(0.384967)+x3*(0.228704)+x4*(-0.415174)+x5*(0.477601)+x6*(0.394788)+x7*(-0.426283)+x8*(-0.0881141)+x9*(0.182117)+x10*(0.175858)+x11*(0.19494)+x12*(-0.124203)+x13*(-0.170719)+x14*(0.210187)+x15*(-0.0027044)+x16*(-0.255639)+x17*(-0.372944)+x18*(0.361385)+x19*(0.400813)+(0.0359246)); 
    h6 = tanh(x0*(-0.54587)+x1*(0.945273)+x2*(-0.631883)+x3*(-0.0648507)+x4*(-0.333468)+x5*(0.0558801)+x6*(-0.0864031)+x7*(0.465521)+x8*(-0.248041)+x9*(-0.565406)+x10*(0.200626)+x11*(-0.116916)+x12*(-0.196186)+x13*(0.173822)+x14*(-0.448282)+x15*(-0.0887164)+x16*(-0.65486)+x17*(-0.695412)+x18*(0.0224935)+x19*(0.410527)+(0.000319133)); 
    h7 = tanh(x0*(0.107007)+x1*(-0.739116)+x2*(0.930419)+x3*(-0.429075)+x4*(0.195268)+x5*(-0.157929)+x6*(-0.159541)+x7*(-0.493069)+x8*(0.652859)+x9*(0.147243)+x10*(-0.129205)+x11*(0.172102)+x12*(0.256982)+x13*(0.0225205)+x14*(0.126152)+x15*(0.388209)+x16*(0.119896)+x17*(0.567641)+x18*(-0.285491)+x19*(0.0976194)+(0.0176182)); 
    h8 = tanh(x0*(0.0285307)+x1*(0.402864)+x2*(-0.496763)+x3*(-0.0985813)+x4*(0.02095)+x5*(0.150974)+x6*(-0.222244)+x7*(-0.0896904)+x8*(-0.458898)+x9*(-0.514625)+x10*(0.136968)+x11*(-0.310012)+x12*(0.214921)+x13*(-0.0404264)+x14*(-0.00600566)+x15*(-0.36406)+x16*(-0.0810151)+x17*(-0.143025)+x18*(0.474541)+x19*(-0.233842)+(-0.0219678)); 
    h9 = tanh(x0*(0.216696)+x1*(-0.247809)+x2*(-0.380353)+x3*(0.356559)+x4*(-0.0879247)+x5*(-0.0552669)+x6*(0.403505)+x7*(-0.126564)+x8*(0.300197)+x9*(0.308185)+x10*(0.173839)+x11*(0.162009)+x12*(-0.171502)+x13*(0.18917)+x14*(-0.0191915)+x15*(-0.125683)+x16*(0.0302257)+x17*(-0.573082)+x18*(0.320329)+x19*(-0.436628)+(-0.0161469)); 
}

void FNN_13_6_Layer_2_Dense_Identity(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,
    out float h0,out float h1,out float h2)
{
    h0 = x0*(0.103605)+x1*(-0.134237)+x2*(0.39323)+x3*(-0.441905)+x4*(0.462486)+x5*(-0.592737)+x6*(0.0530505)+x7*(-0.309212)+x8*(0.0981119)+x9*(0.138042)+(-0.00546871); 
    h1 = x0*(-0.23012)+x1*(0.538771)+x2*(-0.000130461)+x3*(-0.00875075)+x4*(0.556094)+x5*(-0.555572)+x6*(0.319859)+x7*(0.0779475)+x8*(-0.672811)+x9*(-0.184155)+(-0.00207773); 
    h2 = x0*(0.210162)+x1*(-0.0964915)+x2*(-0.166016)+x3*(-0.207084)+x4*(-0.264039)+x5*(-0.268181)+x6*(0.479466)+x7*(0.161062)+x8*(-0.319591)+x9*(-0.069054)+(0.00571438); 
}

void FNN_13_6(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h_3_0,out float h_3_1,out float h_3_2)
{
    float h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19;
    FNN_13_6_Layer_0_Dense_tanh(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19);

    float h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9;
    FNN_13_6_Layer_1_Dense_tanh(h_1_0,h_1_1,h_1_2,h_1_3,h_1_4,h_1_5,h_1_6,h_1_7,h_1_8,h_1_9,h_1_10,h_1_11,h_1_12,h_1_13,h_1_14,h_1_15,h_1_16,h_1_17,h_1_18,h_1_19,h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9);

    FNN_13_6_Layer_2_Dense_Identity(h_2_0,h_2_1,h_2_2,h_2_3,h_2_4,h_2_5,h_2_6,h_2_7,h_2_8,h_2_9,h_3_0,h_3_1,h_3_2);

}

void ModelKDTree_6(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    FNN_13_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
}

void ModelKDTree_0(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,
    out float h0,out float h1,out float h2)
{
    float interpolateExtent = 1* interpolateRatio;
    float delta = interpolateExtent / 2;
    float lowBound = 0.5 - delta;
    float highBound = 0.5 + delta;
    if ( x4 < lowBound ) {
        ModelKDTree_1(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
    else if ( x4 < highBound ) {
        float left_h0;
        float left_h1;
        float left_h2;
        float right_h0;
        float right_h1;
        float right_h2;
        
        ModelKDTree_1(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,left_h0,left_h1,left_h2);
        ModelKDTree_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,right_h0,right_h1,right_h2);
        
        float t = 0.5 + ( x4 - 0.5 ) / interpolateExtent;
        t = smootherstep(t);
        h0= mix(left_h0,right_h0, t);
        h1= mix(left_h1,right_h1, t);
        h2= mix(left_h2,right_h2, t);
    }
    else {
        ModelKDTree_6(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);
    }
}

void Model(float x0,float x1,float x2,float x3,float x4,float x5,float x6,float x7,float x8,float x9,float x10,float x11,float x12,float x13,float x14,float x15,float x16,    
out float h0,out float h1,out float h2)
{
    // map to unit hyperbox
    x0 = (x0 - (0.106384)) / (0.893616);
    x1 = (x1 - (0.106384)) / (0.893616);
    x2 = (x2 - (0.106384)) / (0.893616);
    x3 = (x3 - (-0.999512)) / (1.99902);
    x4 = (x4 - (1.49902)) / (0.000977);
    x5 = (x5 - (-0.999512)) / (1.99902);
    x6 = (x6 - (-0.970703)) / (1.96973);
    x7 = (x7 - (-0.999512)) / (0.967194);
    x8 = (x8 - (-0.999023)) / (1.98486);
    x9 = (x9 - (0)) / (1);
    x10 = (x10 - (-1)) / (1);
    x11 = (x11 - (0)) / (1);
    x12 = (x12 - (1)) / (1);
    x13 = (x13 - (1)) / (1);
    x14 = (x14 - (1)) / (1);
    x15 = (x15 - (20)) / (1);
    x16 = (x16 - (1)) / (1);

    ModelKDTree_0(x0,x1,x2,x3,x4,x5,x6,x7,x8,x9,x10,x11,x12,x13,x14,x15,x16,h0,h1,h2);

    // map back
    h0 = h0 * (0.354262) + (0);
    h1 = h1 * (0.346419) + (0);
    h2 = h2 * (0.34068) + (0);
}

