﻿/********************************************************************
*
* Copyright (c) 2002 Sven Wiegand <mail@sven-wiegand.de>
*
* You can use this and modify this in any way you want,
* BUT LEAVE THIS HEADER INTACT.
*
* Redistribution is appreciated.
*
* $Workfile:$
* $Revision:$
* $Modtime:$
* $Author:$
*
* Revision History:
*   $History:$
*
*********************************************************************/


#if !defined(AFX_TREEPROPSHEET_H__50695CFB_FCE4_4188_ADB4_BF05A5488E41__INCLUDED_)
#define AFX_TREEPROPSHEET_H__50695CFB_FCE4_4188_ADB4_BF05A5488E41__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "PropPageFrame.h"
#include <afxtempl.h>

namespace TreePropSheet
{

/**
A property sheet, which can use a tree control instead of a tab
control, to give the user access to the different pages.

You can use it exactly the same way, as a CPropertySheet object.
Simply create CPropertyPage objects and add them via AddPage() to
the sheet. If you would like to use the tree view mode (default),
you can specify the path of the pages in the tree, by their name:
The names of the pages can contain
double colons ("::"), which will specify the path of that page in the
tree control. I.e. if you have three pages with the following names:
1. L"Appearance::Toolbars"
2. L"Appearance::Menus"
3. L"Directories"
the tree would look as follow:
\verbatim
Appearance
|
+-Toolbars
|
+-Menus

Directories
\endverbatim
If you would like to use a double colon, which should not be
interpreted as a path separator, prefix it with a backslash ("\\::").

To disable tree view mode and use the standard tabbed mode, call
the SetTreeViewMode() method. This also allows you, to enable page
captions and tree images for tree view mode. If you would like to
have images in the tree, but not all of your pages specify images or
there are tree view items, which are not attached to a page (only
parent items for real page items), you have to set default images
using the SetTreeDefaultImages() method -- otherwise their may appear
display errors.

If the user selects a tree view item, which does not belong to a page,
because it is just a parent item for real page items, no page will
be displayed, instead a message will be displayed, that can be set
via SetEmptyPageText().

@author Sven Wiegand
*/
class /*AFX_EXT_CLASS*/ CTreePropSheet : public CPropertySheet
{
	DECLARE_DYNAMIC(CTreePropSheet)

// Construction/Destruction
public:
	CTreePropSheet();
	CTreePropSheet(UINT nIDCaption, CWnd* pParentWnd = nullptr, UINT iSelectPage = 0);
	CTreePropSheet(LPCWSTR pszCaption, CWnd* pParentWnd = nullptr, UINT iSelectPage = 0);
	virtual ~CTreePropSheet();

// Operations
public:
	/**
	Call this method, if you would like to use a tree control to browse
	the pages, instead of the tab control.

	This method needs to becalled, before DoModal() or Create(). If the
	window has already been created, the method will fail.

	@param bTreeViewMode
		Pass TRUE to provide a tree view control instead of a tab control
		to browse the pages, pass FALSE to use the normal tab control.
	@param bPageCaption
		TRUE if a caption should be displayed for each page. The caption
		contains the page title and an icon if specified with the page.
		Ignored if bTreeViewMode is FALSE.
	@param bTreeImages
		TRUE if the page icons should be displayed in the page tree,
		FALSE if there should be no icons in the page tree. Ignored if
		bTreeViewMode is FALSE. If not all of your pages are containing
		icons, or if there will be empty pages (parent nodes without a
		related page, you need to call SetTreeDefaultImages() to avoid
		display errors.

	@return
		TRUE on success or FALSE, if the window has already been created.
	*/
	BOOL SetTreeViewMode(BOOL bTreeViewMode = TRUE, BOOL bPageCaption = FALSE, BOOL bTreeImages = FALSE);

	/**
	Specifies the width of the tree control, when the sheet is in tree
	view mode. The default value (if this method is not called) is 150
	pixels.

	This method needs to be called, before DoModeal() or Create().
	Otherwise it will fail.

	@param nWidth
		The width in pixels for the page tree.
		The window automatically scales the width according to current DPI.

	@return
		TRUE on success, FALSE otherwise (if the window has already been
		created).
	*/
	BOOL SetTreeWidth(int nWidth);

	void SetParentPage(CPropertyPage* pParentPage, CPropertyPage *pPage);

	/**
	Specifies the text to be drawn on empty pages (pages for tree view
	items, that are not related to a page, because they are only
	parents for other items). This is only needed in tree view mode.

	The specified text can contains a single "%s" placeholder which
	will be replaced with the title of the empty page.
	*/
	void SetEmptyPageText(LPCWSTR lpszEmptyPageText);

	/**
	Allows you to specify, how the empty page message (see
	SetEmptyPageText()) should be drawn.

	@param dwFormat
		A combination of the DT_* flags available for the Win32-API
		function DrawText(), that should be used to draw the text.
		The default value is:
		\code
		DT_CENTER|DT_VCENTER|DT_NOPREFIX|DT_SINGLELINE
		\endcode

	@return
		The previous format.
	*/
	DWORD SetEmptyPageTextFormat(DWORD dwFormat);

	//@{
	/**
	Defines the images, that should be used for pages without icons and
	for empty parent nodes. The list contains exactly to images:
	1. An image that should be used for parent tree nodes, without a
	   page asignd.
	2. An image that should be used for pages, which are not specifying
	   any icons.
	Standard image size is 16x16 Pixels, but if you call this method
	before creating the sheet, the size of image 0 in this list will
	be assumed as your preferred image size and all other icons must
	have the same size.

	@param pImages
		Pointer to an image list with exactly to images, that should be
		used as default images. The images are copied to an internal
		list, so that the given list can be deleted after this call.
	@param unBitmapID
		Resource identifier for the bitmap, that contains the default
		images. The resource should contain exactly to images.
	@param cx
		Width of a singe image in pixels.
	@param crMask
		Color that should be interpreted as transparent.

	@return
		TRUE on success, FALSE otherwise.
	*/
	BOOL SetTreeDefaultImages(CImageList *pImages);
	BOOL SetTreeDefaultImages(UINT unBitmapID, int cx, COLORREF crMask);
	//@}

	/**
	Returns a pointer to the tree control, when the sheet is in
	tree view mode, nullptr otherwise.
	*/
	CTreeCtrl* GetPageTreeControl();

	/**
	 * set the dark theme mode
	 */
	void SetTheme(bool bDark);

// Public helpers
public:
	//@{
	/**
	This helper allows you to easily set the icon of a property page.

	This static method does nothing more, than extracting the specified
	image as an icon from the given image list and assign the
	icon-handle to the hIcon property of the pages PROPSHEETPAGE
	structure (m_psp) and modify the structures flags, so that the
	image will be recognized.

	You need to call this method for a page, before adding the page
	to a property sheet.

	@important
	If you are using the CImageList-version, you are responsible for
	destroying the extracted icon with DestroyIcon() or the static
	DestroyPageIcon() method.

	@see DestroyPageIcon()

	@param pPage
		Property page to set the image for.
	@param hIcon
		Handle to icon that should be set for the page.
	@param unIconId
		Ressource identifier for the icon to set.
	@param Images
		Reference of the image list to extract the icon from.
	@param nImage
		Zero based index of the image in pImages, that should be used
		as an icon.

	@return
		TRUE on success, FALSE if an error occurred.
	*/
	static BOOL SetPageIcon(CPropertyPage *pPage, HICON hIcon);
	static BOOL SetPageIcon(CPropertyPage *pPage, UINT unIconId);
	static BOOL SetPageIcon(CPropertyPage *pPage, CImageList &Images, int nImage);
	//@}

	/**
	Checks, if the PSP_USEHICON flag is set in the PROPSHEETPAGE struct;
	If this is the case, the flag will be removed and the icon
	specified by the hIcon attribute of the PROPSHEETPAGE struct will
	be destroyed using DestroyIcon().

	@note
	You only have to call DestroyIcon() for icons, that have been
	created using CreateIconIndirect() (i.e. used by
	CImageList::ExtractIcon()).

	@return
		TRUE on success, FALSE if the PSP_USEHICON flag was not set or
		if the icon handle was nullptr.
	*/
	static BOOL DestroyPageIcon(CPropertyPage *pPage);

// Overridable implementation helpers
protected:
	/**
	Will be called to generate the message, that should be displayed on
	an empty page, when the sheet is in tree view mode

	This default implementation simply returns lpszEmptyPageMessage
	with the optional "%s" placeholder replaced by lpszCaption.

	@param lpszEmptyPageMessage
		The string, set by SetEmptyPageMessage(). This string may contain
		a "%s" placeholder.
	@param lpszCaption
		The title of the empty page.
	*/
	virtual CString GenerateEmptyPageMessage(LPCWSTR lpszEmptyPageMessage, LPCWSTR lpszCaption);

	/**
	Will be called during creation process, to create the CTreeCtrl
	object (the object, not the window!).

	Allows you to inject your own CTreeCtrl-derived classes.

	This default implementation simply creates a CTreeCtrl with new
	and returns it.
	*/
	virtual CTreeCtrl* CreatePageTreeObject();

	/**
	Will be called during creation process, to create the object, that
	is responsible for drawing the frame around the pages, drawing the
	empty page message and the caption.

	Allows you to inject your own CPropPageFrame-derived classes.

	This default implementation simply creates a CPropPageFrameTab with
	new and returns it.
	*/
	virtual CPropPageFrame* CreatePageFrame();

// Implementation helpers
protected:
	/**
	Moves all childs by the specified amount of pixels.

	@param nDx
		Pixels to move the childs in horizontal direction (can be
		negative).
	@param nDy
		Pixels to move the childs in vertical direction (can be
		negative).
	*/
	void MoveChildWindows(int nDx, int nDy);

	/**
	Refills the tree that contains the entries for the several pages.
	*/
	void RefillPageTree();

	/**
	Creates the specified path in the page tree and returns the handle
	of the most child item created.

	@param lpszPath
		Path of the item to create (see description of this class).
	@param hParentItem
		Handle of the item under which the path should be created or
		TVI_ROOT to start from the root.
	*/
	HTREEITEM CreatePageTreeItem(LPCWSTR lpszPath, HTREEITEM hParent = TVI_ROOT);

	/**
	Splits the given path into the topmost item and the rest. See
	description of this class for detailed path information.

	I.e. when given the string "Appearance::Toolbars::Customize", the
	method will return "Appearance" and after the call strRest will
	be "Toolbars::Customize".
	*/
	CString SplitPageTreePath(CString &strRest);

	/**
	Tries to deactivate the current page, and hides it if successful,
	so that an empty page becomes visible.

	@return
		TRUE if the current page has been deactivated successfully,
		FALSE if the currently active page prevents a page change.
	*/
	BOOL KillActiveCurrentPage();

	/**
	Returns the page tree item, that representates the specified page
	or nullptr, if no such icon exists.

	@param nPage
		Zero based page index, for which the item to retrieve.
	@param hRoot
		Item to start the search at or TVI_ROOT to search the whole
		tree.
	*/
	HTREEITEM GetPageTreeItem(int nPage, HTREEITEM hRoot = TVI_ROOT);

	/**
	Selects and shows the item, representing the specified page.

	@param nPage
		Zero based page index.

	@return
		TRUE on success, FALSE if no item does exist for the specified
		page.
	*/
	BOOL SelectPageTreeItem(int nPage);

	/**
	Selects and shows the tree item for the currently active page.

	@return
		TRUE on success, FALSE if no item exists for the currently active
		page or if it was not possible to get information about the
		currently active page.
	*/
	BOOL SelectCurrentPageTreeItem();

	/**
	Updates the caption for the currently selected page (if the caption
	is enabled).
	*/
	void UpdateCaption();

	/**
	Activates the previous page in the page order or the last one, if
	the current one is the first.

	This method does never fail.
	*/
	void ActivatePreviousPage();

	/**
	Activates the next page in the page order or the first one, if the
	current one is the last.

	This method does never fail.
	*/
	void ActivateNextPage();

// Overridings
protected:
	//{{AFX_VIRTUAL(CTreePropSheet)
	public:
	virtual BOOL OnInitDialog();
	void BuildPropPageArray() override
	{
		CPropertySheet::BuildPropPageArray();

		// create a copy of existing PROPSHEETPAGE array which can be modified
		int nPages = static_cast<int>(m_pages.GetSize());
		int nBytes = 0;
		for (decltype(nPages) i = 0; i < nPages; ++i)
		{
			auto pPage = GetPage(i);
			nBytes += pPage->m_psp.dwSize;
		}
		auto ppsp0 = static_cast<LPPROPSHEETPAGE>(malloc(nBytes));
		Checked::memcpy_s(ppsp0, nBytes, m_psh.ppsp, nBytes);
		auto ppsp = ppsp0;
		for (decltype(nPages) i = 0; i < nPages; ++i)
		{
			const DLGTEMPLATE* pResource = ppsp->pResource;
			CDialogTemplate dlgTemplate(pResource);
			dlgTemplate.SetFont(L"Segoe UI", 9);
			HGLOBAL hNew = GlobalAlloc(GPTR, dlgTemplate.m_dwTemplateSize);
			ppsp->pResource = static_cast<DLGTEMPLATE*>(GlobalLock(hNew));
			Checked::memcpy_s(const_cast<void*>(static_cast<const void*>(ppsp->pResource)), dlgTemplate.m_dwTemplateSize, dlgTemplate.m_hTemplate, dlgTemplate.m_dwTemplateSize);
			GlobalUnlock(hNew);
			reinterpret_cast<BYTE*&>(ppsp) += ppsp->dwSize;
		}
		// free existing PROPSHEETPAGE array and assign the new one
		free((void*)m_psh.ppsp);
		m_psh.ppsp = ppsp0;
	}
	//}}AFX_VIRTUAL

// Message handlers
protected:
	//{{AFX_MSG(CTreePropSheet)
	afx_msg void OnDestroy();
	//}}AFX_MSG
	afx_msg LRESULT OnAddPage(WPARAM wParam, LPARAM lParam);
	afx_msg LRESULT OnRemovePage(WPARAM wParam, LPARAM lParam);
	afx_msg LRESULT OnSetCurSel(WPARAM wParam, LPARAM lParam);
	afx_msg LRESULT OnSetCurSelId(WPARAM wParam, LPARAM lParam);
	afx_msg LRESULT OnIsDialogMessage(WPARAM wParam, LPARAM lParam);

	afx_msg void OnPageTreeSelChanging(NMHDR *pNotifyStruct, LRESULT *plResult);
	afx_msg void OnPageTreeSelChanged(NMHDR *pNotifyStruct, LRESULT *plResult);
	DECLARE_MESSAGE_MAP()

// Properties
private:
	/** TRUE if we should use the tree control instead of the tab ctrl. */
	BOOL m_bTreeViewMode= TRUE;

	/** The tree control */
	CTreeCtrl* m_pwndPageTree = nullptr;

	/** The frame around the pages */
	CPropPageFrame* m_pFrame = nullptr;

	std::map<const CPropertyPage*, CPropertyPage*> parentsMap;

	/**
	TRUE, if a tree item selection by OnPageTreeSelChanged() is
	performed currently.
	*/
	BOOL m_bPageTreeSelChangedActive = FALSE;

	/** TRUE if a page caption should be displayed, FALSE otherwise. */
	BOOL m_bPageCaption = FALSE;

	/** TRUE if images should be displayed in the tree. */
	BOOL m_bTreeImages = FALSE;

	/** Images to be displayed in the tree control. */
	CImageList m_Images;

	/** Default images. */
	CImageList m_DefaultImages;

	/**
	Message to be displayed on empty pages. May contain a "%s"
	placeholder which will be replaced by the caption of the empty
	page.
	*/
	CString m_strEmptyPageMessage;

	/** The width of the page tree control in pixels. */
	int m_nPageTreeWidth = 150;

	bool m_bDark = false;

// Static Properties
private:
	/** The id of the tree view control, that shows the pages. */
	static const UINT s_unPageTreeId;
};


} //namespace TreePropSheet

/////////////////////////////////////////////////////////////////////////////

//{{AFX_INSERT_LOCATION}}
// Microsoft Visual C++ fügt unmittelbar vor der vorhergehenden Zeile zusätzliche Deklarationen ein.

#endif // AFX_TREEPROPSHEET_H__50695CFB_FCE4_4188_ADB4_BF05A5488E41__INCLUDED_
