﻿using NUnit.Framework;
using Programmerare.CrsTransformations.Test.CrsTransformations.Utils;
using System.Linq;
using System.Collections.Generic;
using System.IO;
using System.Text;
using Programmerare.CrsTransformations.Coordinate;
using Programmerare.CrsTransformations.Identifier;
using Programmerare.CrsTransformations.Adapter.DotSpatial;
using Programmerare.CrsTransformations.Adapter.ProjNet;
using System;

namespace Programmerare.CrsTransformations.Test.CrsTransformations.WktTest.WktTestUsingCsvFile {

// This class below generates a file with coordinate transformation results,
// which can be used for then retrieving different results with desired accuracy
// by directly selecting columns from the file i.e. by applying a filter.
// One row in that file correponds to the class 'TransformResult' and that file ('TransformResult.cs')
// contains the documentation of the fields in the file.
// The file that can be generated by this class below is then read by the class 'CrsTransformationAdapterTransformationResultTest.cs'

[TestFixture]
[Ignore("Does not need to be executed all the time i.e. generate new files but you can disabled this Ignore attribute if you want to execute it")] 
class FileGeneratorForCsvFileWithWktResults {

    private static Dictionary<int, EpsgAndWkt> _manyEpsgAndWktFromProjNetCsvFile;
    private static Dictionary<int, CoordinateAndTargetEpsg> _coordinatesPerEpsgArea;

    static List<ICrsTransformationAdapter> _allCrsAdapters;

    [OneTimeSetUp] // https://docs.nunit.org/articles/nunit/writing-tests/attributes/onetimesetup.html
    public static void SetUp() {
        string csvFilePathRelativeToSomeBaseDirectoryWithEpsgAndWkt = "Programmerare.CrsTransformations.Adapter.ProjNet/SRID_ShippedWithProjNet_2_0_0.csv";
        var lines = ReadLinesFromFile(csvFilePathRelativeToSomeBaseDirectoryWithEpsgAndWkt);
        _manyEpsgAndWktFromProjNetCsvFile = lines.Select(line => {
            // example of a line from the iterated file:
            // 3006;PROJCS["SWEREF99 TM",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.0174532925199433,AUTHORITY["EPSG","9122"]],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AUTHORITY["EPSG","3006"]]
            var array = line.Split(';');
            int epsgNumber = int.Parse(array[0]); // e.g. 3006 as in the above example
            string wkt = array[1];
            return new EpsgAndWkt(epsgNumber, wkt);
        }).ToDictionary(e => e.EpsgNumber);

        string csvFilePathRelativeToSomeBaseDirectoryWithCoordinatesPerEpsgArea = "TestAndExampleProjects/Programmerare.CrsTransformations.Test/resources/generated/CoordinateTestDataGeneratedFromEpsgDatabase.csv";
        lines = ReadLinesFromFile(csvFilePathRelativeToSomeBaseDirectoryWithCoordinatesPerEpsgArea);
        _coordinatesPerEpsgArea = lines.Select(line => { 
            // example of a line from the iterated file:
            // 3006|1225|Sweden|17.083659606206545|61.98770256318016
            var array = line.Split('|');
            int epsgNumber = int.Parse(array[0]);
            double x = double.Parse(array[3]);
            double y = double.Parse(array[4]);
            return new CoordinateAndTargetEpsg(CrsCoordinateFactory.LatLon(y, x), epsgNumber);
        }).ToDictionary(e => e.epsg);

        _allCrsAdapters = new List<ICrsTransformationAdapter>(){
            new CrsTransformationAdapterProjNet(),
            new CrsTransformationAdapterDotSpatial()
        };
    }


    [Test]
    public void GenerateCsvFileWithTransformResults() {
        var results = CreateTransformResults();

        results.Sort((x, y) => x.epsgNumber.CompareTo(y.epsgNumber));
        WriteToFile(results, "results_sorted_by_epsg.csv");
        // the above file is not used programmatically, but may be convenient to have a file which can be
        // opened in an editor and already be sorted by EPSG number


        // the below sorted and created file is used from 'CrsTransformationAdapterTransformationResultTest.cs'
        // and it is sorted with the "best" results at the top, i.e. succeeding transformations
        // (succeed both in the transform from WGS84 to target CRS and transformed back to WGS84)
        // and small maximal differences between the different combinations of CRS adapter and CRS identifiers
        // (i.e. identifier created by EPSG or WKT)
        // Please note that it is a VERY ROUGH comparison, actually comparing very different units, 
        // for example degrees versus meters or feet...
        // So, this is *almost* a useless sorting, however some values are very large, for example EPSG 5638 
        // with the max X diff value around 446635 and the max Y diff value around 158477.
        // This mentioned EPSG 5638 with big differences have been chosen in the example test file "CrsTransformationAdapterCompositeWktTest"
        // which are also providing some tests that use the native libraries directly to verify that those 
        // very different values are indeed retrieved.
        results.Sort((a, b) => {
            if(a.diffMaxTargetCrsExists && !b.diffMaxTargetCrsExists) return -1;
            if(!a.diffMaxTargetCrsExists && b.diffMaxTargetCrsExists) return 1;
            if(!a.diffMaxTargetCrsExists && !b.diffMaxTargetCrsExists) return 0;
            if(a.diffMaxWgs84Exists&& !b.diffMaxWgs84Exists) return -1;
            if(!a.diffMaxWgs84Exists && b.diffMaxWgs84Exists) return 1;
            if(!a.diffMaxWgs84Exists && !b.diffMaxWgs84Exists) return 0;
            // the above sortings puts the failures at the bottom 
            // and the below sorting puts those wih the smalles max differences at the top
            double aMax = Math.Max(a.xDiffMaxTargetCrs, a.yDiffMaxTargetCrs);
            double bMax = Math.Max(b.xDiffMaxTargetCrs, b.yDiffMaxTargetCrs);
            double diff = (aMax - bMax);
            if(diff == 0) return 0; // before this 'return 0' was added, got this exception: System.ArgumentException : Unable to sort because the IComparer.Compare() method returns inconsistent results. Either a value does not compare equal to itself, or one value repeatedly compared to another value yields different results. IComparer: 'System.Comparison`1[TransformResult]'.
            return diff > 0 ? 1 : -1;
        });
        WriteToFile(results, Filename_results_sorted_with_best_results_first);
    }

    // Note that the sorting, at the creation of this "best results first" file is not very useful,
    // since the sorting does not consider the very different units, e.g. degrees vs meters.
    // But still somewhat useful to see the very worst (largest) differences far down in the file.
    private const string Filename_results_sorted_with_best_results_first = "results_sorted_with_best_results_first.csv";

    public static FileInfo Get_file_with_results_sorted_with_best_results_first(
        string fileName = Filename_results_sorted_with_best_results_first,
        bool createDirectoryIfNotAlreadyExisting = true
    ) {
        var directory_resources = FileUtility.FindDirectory(@"Programmerare.CrsTransformations.Test/resources");
        var directory_wkt_transformation_results = new DirectoryInfo(Path.Combine(directory_resources.FullName, "wkt_transformation_results"));
        if(!directory_wkt_transformation_results.Exists) {
            if(createDirectoryIfNotAlreadyExisting) {
                directory_wkt_transformation_results.Create();
            }
        }
        string filePath = Path.Combine(directory_wkt_transformation_results.FullName, fileName);
        return new FileInfo(filePath);
    }

    private void WriteToFile(List<TransformResult> results, string fileName) {
        var fileToWrite = Get_file_with_results_sorted_with_best_results_first(fileName);
        File.WriteAllLines(fileToWrite.FullName, results.Select(item => item.GetAsRowForFile()));
    }

    private List<TransformResult> CreateTransformResults() {
        var transformResults = new List<TransformResult>();
        var enumerator = _manyEpsgAndWktFromProjNetCsvFile.GetEnumerator();    
        while(enumerator.MoveNext()) {
            int epsgNumber = enumerator.Current.Key;
            string wkt = enumerator.Current.Value.Wkt;
            if (_coordinatesPerEpsgArea.ContainsKey(epsgNumber)) {
                CrsCoordinate crsCoordinateForCentroid = _coordinatesPerEpsgArea[epsgNumber].crsCoordinate;
                CrsIdentifier crsEpsg   = CrsIdentifierFactory.CreateFromEpsgNumber(epsgNumber);
                CrsIdentifier crsWkt    = CrsIdentifierFactory.CreateFromWktCrs(wkt);
                    
                var transformResultItems = new List<TransformResultItem>();
                foreach(var crsAapter in _allCrsAdapters) {
                    transformResultItems.Add(CreateWktResultItem(crsAapter, crsCoordinateForCentroid, crsEpsg));
                    transformResultItems.Add(CreateWktResultItem(crsAapter, crsCoordinateForCentroid, crsWkt));
                }

                transformResults.Add(
                    TransformResult.Create(
                        epsgNumber,
                        crsCoordinateForCentroid.X,
                        crsCoordinateForCentroid.Y,
                        transformResultItems,
                        wkt
                    )
                );
            }
        } // while loop ends
        Assert.That(transformResults.Count, Is.GreaterThan(5000)); // 5190
        return transformResults;
    }

    private TransformResultItem CreateWktResultItem(
        ICrsTransformationAdapter crsAdapter,
        CrsCoordinate crsCoordinateForCentroid,
        CrsIdentifier crs
    ) {
        string epsgOrWkt = crs.isEpsgCode ? "EPSG" : "WKT";
        string adapterName = crsAdapter.ShortNameOfImplementation;

        bool successTargetCrs; // will always be initialized further below, unlike the five below variables
        double xTargetCrs = 0.0;
        double yTargetCrs = 0.0;
        bool successWgs84 = false;
        double xWgs84 = 0.0;
        double yWgs84 = 0.0;
            
        var resultTargetCrs = crsAdapter.Transform(crsCoordinateForCentroid, crs);
        successTargetCrs = resultTargetCrs.isSuccess;
        if(successTargetCrs) {
            var targetCoord = resultTargetCrs.outputCoordinate;
            xTargetCrs = targetCoord.X;
            yTargetCrs = targetCoord.Y;
            var resultWgs84 = crsAdapter.Transform(targetCoord, CrsCoordinateFactory.COORDINATE_REFERENCE_SYSTEM_WGS84);
            successWgs84 = resultWgs84.isSuccess;
            if(successWgs84) {
                var wgs84Coord = resultWgs84.outputCoordinate;
                xWgs84 = wgs84Coord.X;
                yWgs84 = wgs84Coord.Y;
            }
        }
        return new TransformResultItem(
            epsgOrWkt,
            adapterName,
            successTargetCrs,
            xTargetCrs,
            yTargetCrs,
            successWgs84,
            xWgs84,
            yWgs84
        );
    }

    private static List<string> ReadLinesFromFile(
        string filePathRelativeToSomeBaseDirectory
    ) {
        FileInfo file = FileUtility.FindFile(filePathRelativeToSomeBaseDirectory);
        string[] lines = File.ReadAllLines(file.FullName, Encoding.UTF8);
        return lines.ToList();
    }

} // class

} // namespace