﻿using Programmerare.CrsTransformations.Coordinate;
using Programmerare.CrsTransformations.CompositeTransformations;
using NUnit.Framework;
using Programmerare.CrsConstants.ConstantsByAreaNameNumber.v10_036;
using System.Collections.Generic;

namespace Programmerare.CrsTransformations.NuGetClientExampleProjectCSharpe {

    // This C# class is used as a code example for 
    // the libraries that should be retrieved with NuGet
    // i.e. Project Dependencies are not used from this project.
    // There are three projects with similar code as below but 
    // for the different languages C# , F# and VB.NET :
    //  NuGetClientExampleProjectCSharpe ( Programmerare.CrsTransformations.NuGetClientExampleProjectCSharpe )
    //  NuGetClientExampleProjectFSharpe ( Programmerare.CrsTransformations.NuGetClientExampleProjectFSharpe )
    //  NuGetClientExampleProjectVBnet ( Programmerare.CrsTransformations.NuGetClientExampleProjectVBnet )
    // The main purpose for these projects is to verify that NuGet retrieval works including to show that the assembly deployed to NuGet works fine.
    // For the F# and VB.NET projects, another purpose is to verify usage of the code from those languages.
    // Since the library is implemented with F# it should not be a surprise that the code works for F#
    // but for "completeness" I created a similar library for all these three languages, although 
    // it is most interesting with the VB.NET code since the above VB.NET library 
    // is the only code with VB.NET while there are lots of tests for C# 
    // in the project "Programmerare.CrsTransformations.Test"

    [TestFixture]
    public class CSharpeExampleUsingNuGetDependencies {

        private const double SmallDeltaValue = 0.0000001;
        
        [Test]
        public void CSharpeExampleCode() {
            int epsgWgs84  = EpsgNumber.WORLD__WGS_84__4326;
            int epsgSweRef = EpsgNumber.SWEDEN__SWEREF99_TM__3006;
            Assert.AreEqual(4326, epsgWgs84);
            Assert.AreEqual(3006, epsgSweRef);
         
            CrsCoordinate centralStockholmWgs84 = CrsCoordinateFactory.LatLon(59.330231, 18.059196, epsgWgs84);
        
            ICrsTransformationAdapter crsTransformationAdapter = CrsTransformationAdapterCompositeFactory.Create().CreateCrsTransformationMedian();
            CrsTransformationResult centralStockholmResultSweRef = crsTransformationAdapter.Transform(centralStockholmWgs84, epsgSweRef);
        
            Assert.IsNotNull(centralStockholmResultSweRef);
            Assert.IsTrue(centralStockholmResultSweRef.IsSuccess);
            IList<CrsTransformationResult> transformationResultChildren = centralStockholmResultSweRef.TransformationResultChildren;

            // Reason for the below assertion with value 3 :
            // If the NuGet configuration includes all (currently three) adapter implementations, then the 
            // above created 'Composite' implementation will below use all three 'leaf' implementations 
            // and return a coordinate with a median longitude and a median latitude
            Assert.AreEqual(3, transformationResultChildren.Count);

            // Console.WriteLine(centralStockholmResultSweRef.OutputCoordinate);
            // Console output from the above code row: 
            // CrsCoordinate(xEastingLongitude=674032.357177155, yNorthingLatitude=6580821.99121561, crsIdentifier=CrsIdentifier(crsCode='EPSG:3006', isEpsgCode=True, epsgNumber=3006))
            var outputCoordinate = centralStockholmResultSweRef.OutputCoordinate;
            Assert.IsNotNull(outputCoordinate);
            Assert.AreEqual(674032.357177155, outputCoordinate.XEastingLongitude, SmallDeltaValue);
            Assert.AreEqual(6580821.99121561, outputCoordinate.YNorthingLatitude, SmallDeltaValue);

            CrsTransformationResultStatistic crsTransformationResultStatistic = centralStockholmResultSweRef.CrsTransformationResultStatistic;
            var medianCoordinate = crsTransformationResultStatistic.CoordinateMedian;
            // the median values have already been tested above since we used 'CreateCrsTransformationMedian'
            // for creating the main result.
            var averageCoordinate = crsTransformationResultStatistic.CoordinateAverage;
            Assert.AreEqual(674032.35716645606, averageCoordinate.XEastingLongitude, SmallDeltaValue);
            Assert.AreEqual(6580821.9921956062, averageCoordinate.YNorthingLatitude, SmallDeltaValue);

            Assert.IsTrue(crsTransformationResultStatistic.IsStatisticsAvailable);
            Assert.AreEqual(3, crsTransformationResultStatistic.NumberOfPotentiallySuccesfulResults);
            Assert.That(crsTransformationResultStatistic.MaxDifferenceForXEastingLongitude, Is.LessThan(0.01));
            Assert.That(crsTransformationResultStatistic.MaxDifferenceForYNorthingLatitude, Is.LessThan(0.01));

            // "Reliable True" below since there should be three sucesful results 
            // and the absolute value for the difference between longitudes and longitudes 
            // should be less than 0.01
            Assert.IsTrue(
                centralStockholmResultSweRef.IsReliable(
                    3,  // minimumNumberOfSuccesfulResults
                    0.01// maxDeltaValueForXLongitudeAndYLatitude
                )                                                
            );

            // "Reliable False" below because too extreme requirements of equal values for all the results 
            // i.e. very small tolerance for differences 
            Assert.IsFalse(
                centralStockholmResultSweRef.IsReliable(
                    3,  // minimumNumberOfSuccesfulResults
                    0.000000000000000000001// maxDeltaValueForXLongitudeAndYLatitude
                )                                                
            );

            // "Reliable False" below because can not require 4 succesful values 
            // when there are only 3 implementations
            Assert.IsFalse(
                centralStockholmResultSweRef.IsReliable(
                    4,  // minimumNumberOfSuccesfulResults
                    0.01// maxDeltaValueForXLongitudeAndYLatitude
                )                                                
            );

            ICrsTransformationAdapter crsTransformationAdapterResultSource = centralStockholmResultSweRef.CrsTransformationAdapterResultSource;
            CrsTransformationAdapteeType adapteeType = crsTransformationAdapterResultSource.AdapteeType;
            Assert.AreEqual(CrsTransformationAdapteeType.COMPOSITE_MEDIAN, adapteeType);
            var dict = new Dictionary<CrsTransformationAdapteeType, bool>();
            foreach (CrsTransformationResult crsTransformationResultLeaf in transformationResultChildren) {
                Assert.IsTrue(crsTransformationResultLeaf.IsSuccess);
                dict.Add(crsTransformationResultLeaf.CrsTransformationAdapterResultSource.AdapteeType, true);

                // Leafs always only have one result and thus there are zero difference between max and min result.
                // Therefore the below assertion should succeed
                Assert.IsTrue(
                    crsTransformationResultLeaf.IsReliable(
                        1,  // minimumNumberOfSuccesfulResults
                        0.0000000000000000000000000000001// maxDeltaValueForXLongitudeAndYLatitude
                    )                                                
                );

                // Leafs always only have one result and thus the below tested method should return False
                Assert.IsFalse(
                    crsTransformationResultLeaf.IsReliable(
                        2,  // minimumNumberOfSuccesfulResults
                        0.1// maxDeltaValueForXLongitudeAndYLatitude
                    )                                                
                );

                CrsTransformationResultStatistic leafResultStatistic = crsTransformationResultLeaf.CrsTransformationResultStatistic;
                Assert.IsTrue(leafResultStatistic.IsStatisticsAvailable);
                Assert.AreEqual(1, leafResultStatistic.NumberOfPotentiallySuccesfulResults);
                Assert.That(leafResultStatistic.MaxDifferenceForXEastingLongitude, Is.LessThan(0.01));
                Assert.That(leafResultStatistic.MaxDifferenceForYNorthingLatitude, Is.LessThan(0.01));
            }
            Assert.AreEqual(3, dict.Count);
            Assert.IsTrue(dict.ContainsKey(CrsTransformationAdapteeType.LEAF_MIGHTY_LITTLE_GEODESY_1_0_2));
            Assert.IsTrue(dict.ContainsKey(CrsTransformationAdapteeType.LEAF_DOT_SPATIAL_2_0_0_RC1));
            Assert.IsTrue(dict.ContainsKey(CrsTransformationAdapteeType.LEAF_PROJ_NET_2_0_0));
        }
    }
}