package ai.timefold.solver.core.impl.domain.common.accessor.gizmo;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatCode;

import java.lang.reflect.Member;
import java.lang.reflect.Method;

import ai.timefold.solver.core.api.domain.entity.PlanningPin;
import ai.timefold.solver.core.api.domain.lookup.PlanningId;
import ai.timefold.solver.core.api.domain.variable.PlanningVariable;
import ai.timefold.solver.core.impl.domain.common.accessor.MemberAccessor;
import ai.timefold.solver.core.impl.testdata.domain.TestdataEntity;
import ai.timefold.solver.core.impl.testdata.domain.TestdataValue;
import ai.timefold.solver.core.impl.testdata.domain.gizmo.GizmoTestdataEntity;

import org.junit.jupiter.api.Test;

class GizmoMemberAccessorImplementorTest {

    @Test
    void testGeneratedMemberAccessorForMethod() throws NoSuchMethodException {
        Member member = TestdataEntity.class.getMethod("getValue");
        MemberAccessor memberAccessor =
                GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true, new GizmoClassLoader());
        assertThat(memberAccessor.getName()).isEqualTo("value");
        assertThat(memberAccessor.getType()).isEqualTo(TestdataValue.class);
        assertThat(memberAccessor.getSpeedNote()).isEqualTo("Fast access with generated bytecode");
        assertThat(memberAccessor.getDeclaringClass()).isEqualTo(TestdataEntity.class);
        assertThat(memberAccessor.supportSetter()).isTrue();
        assertThat(memberAccessor.getAnnotation(PlanningVariable.class)).isNotNull();

        TestdataEntity testdataEntity = new TestdataEntity();
        TestdataValue testdataValue1 = new TestdataValue("A");
        testdataEntity.setValue(testdataValue1);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(testdataValue1);

        TestdataValue testdataValue2 = new TestdataValue("B");
        memberAccessor.executeSetter(testdataEntity, testdataValue2);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(testdataValue2);
    }

    @Test
    void testGeneratedMemberAccessorForMethodWithoutSetter() throws NoSuchMethodException {
        Member member = GizmoTestdataEntity.class.getMethod("getId");
        MemberAccessor memberAccessor =
                GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningId.class, true, new GizmoClassLoader());
        assertThat(memberAccessor.getName()).isEqualTo("id");
        assertThat(memberAccessor.getType()).isEqualTo(String.class);
        assertThat(memberAccessor.getSpeedNote()).isEqualTo("Fast access with generated bytecode");
        assertThat(memberAccessor.getDeclaringClass()).isEqualTo(GizmoTestdataEntity.class);
        assertThat(memberAccessor.supportSetter()).isFalse();
        assertThat(memberAccessor.getAnnotation(PlanningId.class)).isNotNull();

        GizmoTestdataEntity testdataEntity = new GizmoTestdataEntity("A", null, false);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo("A");
    }

    @Test
    void testGeneratedMemberAccessorForField() throws NoSuchFieldException {
        Member member = GizmoTestdataEntity.class.getField("value");
        MemberAccessor memberAccessor =
                GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true, new GizmoClassLoader());
        assertThat(memberAccessor.getName()).isEqualTo("value");
        assertThat(memberAccessor.getType()).isEqualTo(TestdataValue.class);
        assertThat(memberAccessor.getSpeedNote()).isEqualTo("Fast access with generated bytecode");
        assertThat(memberAccessor.getDeclaringClass()).isEqualTo(GizmoTestdataEntity.class);
        assertThat(memberAccessor.supportSetter()).isTrue();
        assertThat(memberAccessor.getAnnotation(PlanningVariable.class)).isNotNull();

        GizmoTestdataEntity testdataEntity = new GizmoTestdataEntity("A", null, false);
        TestdataValue testdataValue1 = new TestdataValue("A");
        testdataEntity.setValue(testdataValue1);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(testdataValue1);

        TestdataValue testdataValue2 = new TestdataValue("B");
        memberAccessor.executeSetter(testdataEntity, testdataValue2);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(testdataValue2);
    }

    @Test
    void testGeneratedMemberAccessorForPrimitiveField() throws NoSuchFieldException {
        Member member = GizmoTestdataEntity.class.getField("isPinned");
        MemberAccessor memberAccessor =
                GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningPin.class, true, new GizmoClassLoader());
        assertThat(memberAccessor.getName()).isEqualTo("isPinned");
        assertThat(memberAccessor.getType()).isEqualTo(boolean.class);
        assertThat(memberAccessor.getSpeedNote()).isEqualTo("Fast access with generated bytecode");
        assertThat(memberAccessor.getDeclaringClass()).isEqualTo(GizmoTestdataEntity.class);
        assertThat(memberAccessor.supportSetter()).isTrue();

        GizmoTestdataEntity testdataEntity = new GizmoTestdataEntity("A", null, false);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(false);

        memberAccessor.executeSetter(testdataEntity, true);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(true);
    }

    @Test
    void testGeneratedMemberAccessorSameClass() throws NoSuchMethodException {
        GizmoClassLoader gizmoClassLoader = new GizmoClassLoader();
        Member member = TestdataEntity.class.getMethod("getValue");
        MemberAccessor memberAccessor1 =
                GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true, gizmoClassLoader);
        MemberAccessor memberAccessor2 =
                GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true, gizmoClassLoader);

        assertThat(memberAccessor1.getClass()).isEqualTo(memberAccessor2.getClass());
    }

    @Test
    void testGeneratedMemberAccessorReturnVoid() throws NoSuchMethodException {
        Method member = TestdataEntity.class.getMethod("updateValue");
        MemberAccessor memberAccessor =
                GizmoMemberAccessorFactory.buildGizmoMemberAccessor(member, null, false,
                        new GizmoClassLoader());

        TestdataEntity entity = new TestdataEntity();
        TestdataValue value = new TestdataValue("A");
        entity.setValue(value);

        memberAccessor.executeGetter(entity);
        assertThat(entity.getValue().getCode()).isEqualTo("A/A");

        assertThat(memberAccessor.supportSetter()).isFalse();
        assertThat(memberAccessor.getDeclaringClass()).isEqualTo(TestdataEntity.class);
        assertThat(memberAccessor.getGenericType()).isNull();
        assertThat(memberAccessor.getType()).isEqualTo(void.class);
        assertThat(memberAccessor.getType()).isEqualTo(member.getReturnType());
        assertThat(memberAccessor.getName()).isEqualTo("updateValue");
        assertThat(memberAccessor.getSpeedNote()).isEqualTo("Fast access with generated bytecode");
    }

    @Test
    void testThrowsWhenGetterMethodHasParameters() throws NoSuchMethodException {
        Member member = GizmoTestdataEntity.class.getMethod("methodWithParameters", String.class);
        assertThatCode(() -> {
            GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true, new GizmoClassLoader());
        }).hasMessage("The getterMethod (methodWithParameters) with a PlanningVariable annotation " +
                "must not have any parameters, but has parameters ([Ljava/lang/String;]).");
    }

    @Test
    void testThrowsWhenGetterMethodReturnVoid() throws NoSuchMethodException {
        Member member = GizmoTestdataEntity.class.getMethod("getVoid");
        assertThatCode(() -> {
            GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true, new GizmoClassLoader());
        }).hasMessage("The getterMethod (getVoid) with a PlanningVariable annotation must have a non-void return type.");
    }

    @Test
    void testThrowsWhenReadMethodReturnVoid() throws NoSuchMethodException {
        Member member = GizmoTestdataEntity.class.getMethod("voidMethod");
        assertThatCode(() -> {
            GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true, new GizmoClassLoader());
        }).hasMessage("The readMethod (voidMethod) with a PlanningVariable annotation must have a non-void return type.");
    }

    @Test
    void testGeneratedMemberAccessorForBooleanMethod() throws NoSuchMethodException {
        Member member = GizmoTestdataEntity.class.getMethod("isPinned");
        MemberAccessor memberAccessor =
                GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningPin.class, true, new GizmoClassLoader());
        assertThat(memberAccessor.getName()).isEqualTo("pinned");
        assertThat(memberAccessor.getType()).isEqualTo(boolean.class);
        assertThat(memberAccessor.getSpeedNote()).isEqualTo("Fast access with generated bytecode");
        assertThat(memberAccessor.getDeclaringClass()).isEqualTo(GizmoTestdataEntity.class);
        assertThat(memberAccessor.supportSetter()).isTrue();
        assertThat(memberAccessor.getAnnotation(PlanningPin.class)).isNotNull();

        GizmoTestdataEntity testdataEntity = new GizmoTestdataEntity("A", null, false);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(false);

        memberAccessor.executeSetter(testdataEntity, true);
        assertThat(memberAccessor.executeGetter(testdataEntity)).isEqualTo(true);
    }

    @Test
    void testThrowsWhenGetBooleanReturnsNonBoolean() throws NoSuchMethodException {
        Member member = GizmoTestdataEntity.class.getMethod("isAMethodThatHasABadName");
        assertThatCode(
                () -> GizmoMemberAccessorImplementor.createAccessorFor(member, PlanningVariable.class, true,
                        new GizmoClassLoader()))
                .hasMessage("""
                        The getterMethod (isAMethodThatHasABadName) with a PlanningVariable annotation \
                        must have a primitive boolean return type but returns (L%s;).
                        Maybe rename the method (getAMethodThatHasABadName)?"""
                        .formatted(String.class.getName().replace('.', '/')));
    }
}
