#include <memory>

#include <windows.h>
#include <streams.h>
#include <dvdmedia.h>
#include <commctrl.h>
#include <tchar.h>
#include <initguid.h>

#include "resource.h" // Generated by CMake

#include "AC/Core.hpp"

#define SET_PLANAR_FORMAT(t) (((t) << 8) | 0)
#define SET_PACKED_FORMAT(t) (((t) << 8) | 1)
#define IS_PLANAR_FORMAT(f) (((f) & 0xff) == 0)
#define GET_FORMAT_TYPE(f) ((f) >> 8)

#define gRegArgument (RegArgument::instance())

DEFINE_GUID(CLSID_AC_FILTER, 0x731ae2e9, 0xeeed, 0x4e29, 0xb8, 0x1b, 0x5d, 0xc8, 0xa0, 0xd6, 0xa3, 0x07);
class Filter : public CTransformFilter, public ISpecifyPropertyPages
{
public:
    static CUnknown* WINAPI CreateInstance(LPUNKNOWN punk, HRESULT* phr);
public:
    DECLARE_IUNKNOWN;

    STDMETHODIMP NonDelegatingQueryInterface(REFIID riid, void** ppv) override;

    HRESULT CheckInputType(const CMediaType* mtIn) override;
    HRESULT CheckTransform(const CMediaType* mtIn, const CMediaType* mtOut) override;
    HRESULT GetMediaType(int pos, CMediaType* mt) override;
    HRESULT DecideBufferSize(IMemAllocator* alloctor, ALLOCATOR_PROPERTIES* request) override;
    HRESULT Transform(IMediaSample* in, IMediaSample* out) override;

    STDMETHODIMP GetPages(CAUUID* pages) override;
private:
    Filter(TCHAR* name, LPUNKNOWN punk, HRESULT* phr);
private:
    struct { struct { LONG width, height; } src, dst, limit; } size{};
private:
    int format{};
    double factor{};
    std::shared_ptr<ac::core::Processor> processor{};
};
DEFINE_GUID(CLSID_AC_PROPERTY_PAGE, 0x2548fd44, 0x5370, 0x40f6, 0x99, 0x66, 0x48, 0x5d, 0x7b, 0xac, 0x71, 0x3c);
class PropertyPage : public CBasePropertyPage
{
public:
    static CUnknown* WINAPI CreateInstance(LPUNKNOWN punk, HRESULT* phr);
private:
    INT_PTR OnReceiveMessage(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam) override;
    HRESULT OnActivate() override;
    HRESULT OnDeactivate() override;
    HRESULT OnApplyChanges() override;
private:
    PropertyPage(TCHAR* name, LPUNKNOWN punk, HRESULT* phr);
private:
    bool isInitialized = false;
};
class RegArgument
{
private:
    RegArgument();
    ~RegArgument();
public:
    double getFactor();
    int getProcessorId();
    int getDevice();
    int getLimitWidth();
    int getLimitHeight();
    const TCHAR* getModelName();

    void setFactor(double v) const;
    void setProcessorId(int v) const;
    void setDevice(int v) const;
    void setLimitWidth(int v) const;
    void setLimitHeight(int v) const;
    void setModelName(const TCHAR* v) const;
public:
    static RegArgument& instance();
public:
    static constexpr int ModelNameMaxSize = 32;

    static constexpr double FactorDefault = 2.0;
    static constexpr int ProcessorIdDefault = ac::core::Processor::CPU;
    static constexpr int DeviceDefault = 0;
    static constexpr int LimitWidthDefault = 1920;
    static constexpr int LimitHeightDefault = 1080;
    static constexpr TCHAR* ModelNameDefault = TEXT("acnet-hdn0");
private:
    static constexpr TCHAR* FactorValueName = TEXT("Factor");
    static constexpr TCHAR* ProcessorIdValueName = TEXT("ProcessorId");
    static constexpr TCHAR* DeviceValueName = TEXT("Device");
    static constexpr TCHAR* LimitWidthValueName = TEXT("LimitWidth");
    static constexpr TCHAR* LimitHeightValueName = TEXT("LimitHeight");
    static constexpr TCHAR* ModelNameValueName = TEXT("ModelName");
private:
    HKEY key{};
    double factor{};
    int processorId{};
    int device{};
    int limitWidth{};
    int limitHeight{};
    TCHAR modelName[ModelNameMaxSize]{};
};

const AMOVIESETUP_MEDIATYPE sudPinTypes[] =
{
    {
        &MEDIATYPE_Video,
        &MEDIASUBTYPE_IYUV
    },
    {
        &MEDIATYPE_Video,
        &MEDIASUBTYPE_YV12
    },
    {
        &MEDIATYPE_Video,
        &MEDIASUBTYPE_NV12
    },
    {
        &MEDIATYPE_Video,
        &MEDIASUBTYPE_P010
    },
    {
        &MEDIATYPE_Video,
        &MEDIASUBTYPE_P016
    }
};
const AMOVIESETUP_PIN sudpPins[] =
{
    {
        L"Input",
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        &CLSID_NULL,
        NULL,
        NUMELMS(sudPinTypes),
        sudPinTypes
    },
    {
        L"Output",
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        &CLSID_NULL,
        NULL,
        NUMELMS(sudPinTypes),
        sudPinTypes
    }
};
const AMOVIESETUP_FILTER sudFilter =
{
    &CLSID_AC_FILTER,
    L"Anime4KCPP for DirectShow",
    MERIT_DO_NOT_USE,
    2,
    sudpPins
};
CFactoryTemplate g_Templates[] = {
    {
        L"Anime4KCPP for DirectShow",
        &CLSID_AC_FILTER,
        Filter::CreateInstance,
        nullptr,
        &sudFilter
    },
    {
        L"Anime4KCPP Settings",
        &CLSID_AC_PROPERTY_PAGE,
        PropertyPage::CreateInstance
    }
};
int g_cTemplates = NUMELMS(g_Templates);

STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}

extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}

CUnknown* Filter::CreateInstance(LPUNKNOWN punk, HRESULT* phr)
{
    auto object = new Filter(NAME("Anime4KCPP for DirectShow"), punk, phr);
    if (!object && phr) *phr = E_OUTOFMEMORY;
    return object;
}
Filter::Filter(TCHAR* name, LPUNKNOWN punk, HRESULT* /*phr*/) : CTransformFilter(name, punk, CLSID_AC_FILTER)
{
    factor = gRegArgument.getFactor();
    size.limit.width = gRegArgument.getLimitWidth();
    size.limit.height = gRegArgument.getLimitHeight();
    auto processorId = gRegArgument.getProcessorId();
    auto device = gRegArgument.getDevice();
    auto modelName = gRegArgument.getModelName();

    processor = [&]() {
        ac::core::model::ACNet model{ [&]() {
            for (auto p = modelName; *p != TEXT('\0'); p++)
            {
                if (*p == TEXT('1')) return ac::core::model::ACNet::Variant::HDN1;
                if (*p == TEXT('2')) return ac::core::model::ACNet::Variant::HDN2;
                if (*p == TEXT('3')) return ac::core::model::ACNet::Variant::HDN3;
            }
            return ac::core::model::ACNet::Variant::HDN0;
        }() };

#       ifdef AC_CORE_WITH_OPENCL
            if (processorId == ac::core::Processor::OpenCL) return ac::core::Processor::create<ac::core::Processor::OpenCL>(device, model);
#       endif
#       ifdef AC_CORE_WITH_CUDA
            if (processorId == ac::core::Processor::CUDA) return ac::core::Processor::create<ac::core::Processor::CUDA>(device, model);
#       endif
        return ac::core::Processor::create<ac::core::Processor::CPU>(device, model);
    }();
}
STDMETHODIMP Filter::NonDelegatingQueryInterface(REFIID riid, void** ppv)
{
    CheckPointer(ppv, E_POINTER);

    if (riid == IID_ISpecifyPropertyPages)
        return GetInterface(static_cast<ISpecifyPropertyPages*>(this), ppv);
    else
        return CTransformFilter::NonDelegatingQueryInterface(riid, ppv);
}
HRESULT Filter::CheckInputType(const CMediaType* mtIn)
{
    CheckPointer(mtIn, E_POINTER);

    if (!IsEqualGUID(*mtIn->FormatType(), FORMAT_VideoInfo2) &&
        !IsEqualGUID(*mtIn->FormatType(), FORMAT_VideoInfo))
        return VFW_E_TYPE_NOT_ACCEPTED;

    format = [&]() -> int {
        // planar: YYYYUUVV
        if (IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_IYUV) ||
            IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_YV12)) return SET_PLANAR_FORMAT(ac::core::Image::UInt8);
        // packed: YYYYUVUV
        if (IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_NV12)) return SET_PACKED_FORMAT(ac::core::Image::UInt8);
        if (IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_P010) ||
            IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_P016)) return SET_PACKED_FORMAT(ac::core::Image::UInt16);
        return 0;
        }();

    if (!format) return VFW_E_TYPE_NOT_ACCEPTED;

    auto checkVideoInfo = [&](auto vi) -> HRESULT {
        if (vi->bmiHeader.biWidth > size.limit.width || vi->bmiHeader.biHeight > size.limit.height)
            return VFW_E_TYPE_NOT_ACCEPTED;
        return S_OK;
    };
    if (IsEqualGUID(*mtIn->FormatType(), FORMAT_VideoInfo2))
        return checkVideoInfo(reinterpret_cast<VIDEOINFOHEADER2*>(mtIn->pbFormat));
    else
        return checkVideoInfo(reinterpret_cast<VIDEOINFOHEADER*>(mtIn->pbFormat));
}
HRESULT Filter::CheckTransform(const CMediaType* mtIn, const CMediaType* mtOut)
{
    CheckPointer(mtIn, E_POINTER);
    CheckPointer(mtOut, E_POINTER);

    if (!IsEqualGUID(*mtOut->FormatType(), FORMAT_VideoInfo2) &&
        !IsEqualGUID(*mtOut->FormatType(), FORMAT_VideoInfo))
        return VFW_E_TYPE_NOT_ACCEPTED;
    if (!IsEqualGUID(*mtIn->Subtype(), *mtOut->Subtype()))
        return VFW_E_TYPE_NOT_ACCEPTED;
    return S_OK;
}
HRESULT Filter::GetMediaType(int pos, CMediaType* mt)
{
    if (m_pInput->IsConnected() == FALSE) return E_UNEXPECTED;
    if (pos < 0) return E_INVALIDARG;
    if (pos > 0) return VFW_S_NO_MORE_ITEMS;

    CheckPointer(mt, E_POINTER);
    *mt = m_pInput->CurrentMediaType();

    auto setVideoInfo = [&](auto vi) {
        size.src.width = vi->bmiHeader.biWidth;
        size.src.height = vi->bmiHeader.biHeight;
        size.dst.width = static_cast<decltype(size.dst.width)>(size.src.width * factor);
        size.dst.height = static_cast<decltype(size.dst.height)>(size.src.height * factor);
        vi->bmiHeader.biWidth = size.dst.width;
        vi->bmiHeader.biHeight = size.dst.height;
        vi->bmiHeader.biSizeImage = DIBSIZE(vi->bmiHeader);
        mt->SetSampleSize(vi->bmiHeader.biSizeImage);
        SetRect(&vi->rcSource, 0, 0, size.dst.width, size.dst.height);
        SetRect(&vi->rcTarget, 0, 0, size.dst.width, size.dst.height);
    };
    if (IsEqualGUID(*mt->FormatType(), FORMAT_VideoInfo2))
        setVideoInfo(reinterpret_cast<VIDEOINFOHEADER2*>(mt->pbFormat));
    else
        setVideoInfo(reinterpret_cast<VIDEOINFOHEADER*>(mt->pbFormat));

    return S_OK;
}
HRESULT Filter::DecideBufferSize(IMemAllocator* alloctor, ALLOCATOR_PROPERTIES* request)
{
    if (!m_pInput->IsConnected()) return E_UNEXPECTED;
    CheckPointer(alloctor, E_POINTER);
    CheckPointer(request, E_POINTER);

    request->cbBuffer = static_cast<decltype(request->cbBuffer)>(m_pOutput->CurrentMediaType().GetSampleSize());

    if (request->cbAlign == 0) request->cbAlign = 1;
    if (request->cBuffers == 0) request->cBuffers = 1;

    ALLOCATOR_PROPERTIES actual{};
    auto err = alloctor->SetProperties(request, &actual);
    if (err != S_OK) return err;

    if (request->cBuffers > actual.cBuffers || request->cbBuffer > actual.cbBuffer)
        return E_FAIL;

    return S_OK;
}
HRESULT Filter::Transform(IMediaSample* in, IMediaSample* out)
{
    CheckPointer(in, E_POINTER);
    CheckPointer(out, E_POINTER);

    BYTE* src{};
    BYTE* dst{};
    in->GetPointer(&src);
    out->GetPointer(&dst);

    ac::core::Image srcy{ size.src.width, size.src.height, 1, GET_FORMAT_TYPE(format), src, ((in->GetActualDataLength() * 2) / 3) / size.src.height };
    ac::core::Image dsty{ size.dst.width, size.dst.height, 1, GET_FORMAT_TYPE(format), dst, ((out->GetActualDataLength() * 2) / 3) / size.dst.height };
    processor->process(srcy, dsty, factor);

    int channels = IS_PLANAR_FORMAT(format) ? 1 : 2;
    ac::core::Image srcuv{ size.src.width / 2, size.src.height / channels, channels, GET_FORMAT_TYPE(format), src + srcy.size(), ((in->GetActualDataLength() * channels) / 3) / size.src.height };
    ac::core::Image dstuv{ size.dst.width / 2, size.dst.height / channels, channels, GET_FORMAT_TYPE(format), dst + dsty.size(), ((out->GetActualDataLength() * channels) / 3) / size.dst.height };
    ac::core::resize(srcuv, dstuv, 0.0, 0.0);

    return S_OK;
}
STDMETHODIMP Filter::GetPages(CAUUID* pages)
{
    CheckPointer(pages, E_POINTER);

    pages->cElems = 1;
    pages->pElems = static_cast<GUID*>(CoTaskMemAlloc(sizeof(GUID)));
    if (!pages->pElems) return E_OUTOFMEMORY;
    pages->pElems[0] = CLSID_AC_PROPERTY_PAGE;
    return S_OK;
}

CUnknown* PropertyPage::CreateInstance(LPUNKNOWN punk, HRESULT* phr)
{
    auto object = new PropertyPage(NAME("Anime4KCPP for DirectShow Property Page"), punk, phr);
    if (!object && phr) *phr = E_OUTOFMEMORY;
    return object;
}
PropertyPage::PropertyPage(TCHAR* name, LPUNKNOWN punk, HRESULT* /*phr*/) : CBasePropertyPage(name, punk, IDD_PROPPAGE, IDS_TITLE) {}
INT_PTR PropertyPage::OnReceiveMessage(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_COMMAND:
    {
        if (isInitialized)
        {
            m_bDirty = TRUE;
            if (m_pPageSite)
                m_pPageSite->OnStatusChange(PROPPAGESTATUS_DIRTY);
        }
        return (LRESULT)1;
    }
    }
    return CBasePropertyPage::OnReceiveMessage(hwnd, uMsg, wParam, lParam);
}
HRESULT PropertyPage::OnActivate()
{
    TCHAR buffer[STR_MAX_LENGTH] = {};

    auto factor = gRegArgument.getFactor();
    _stprintf_s(buffer, NUMELMS(buffer), TEXT("%.2lf"), factor);
    Edit_SetText(GetDlgItem(m_Dlg, IDC_EDIT_FACTOR), buffer);

    const TCHAR* processorList[] = { TEXT("cpu"),
#       ifdef AC_CORE_WITH_OPENCL
            TEXT("opencl"),
#       else
            nullptr,
#       endif
#       ifdef AC_CORE_WITH_CUDA
            TEXT("cuda"),
#       else
            nullptr,
#       endif
    };
    for (int i = 0; i < NUMELMS(processorList); i++)
        if (processorList[i])
            ComboBox_AddString(GetDlgItem(m_Dlg, IDC_COMBO_PROCESSOR), processorList[i]);

    auto processorId = gRegArgument.getProcessorId();
    if (processorId < 0 || processorId >= NUMELMS(processorList) || !processorList[processorId])
        processorId = ac::core::Processor::CPU;
    ComboBox_SelectString(GetDlgItem(m_Dlg, IDC_COMBO_PROCESSOR), 0, processorList[processorId]);

    auto device = gRegArgument.getDevice();
    _stprintf_s(buffer, NUMELMS(buffer), TEXT("%d"), device);
    Edit_SetText(GetDlgItem(m_Dlg, IDC_EDIT_DEVICE), buffer);

    auto modelName = gRegArgument.getModelName();
    Edit_SetText(GetDlgItem(m_Dlg, IDC_EDIT_MODEL), modelName);

    auto limitWidth = gRegArgument.getLimitWidth();
    _stprintf_s(buffer, NUMELMS(buffer), TEXT("%d"), limitWidth);
    Edit_SetText(GetDlgItem(m_Dlg, IDC_EDIT_LIMIT_WIDTH), buffer);

    auto limitHeight = gRegArgument.getLimitHeight();
    _stprintf_s(buffer, NUMELMS(buffer), TEXT("%d"), limitHeight);
    Edit_SetText(GetDlgItem(m_Dlg, IDC_EDIT_LIMIT_HEIGHT), buffer);

    int count = 0;
    for (auto p = ac::core::Processor::info<ac::core::Processor::CPU>(); *p != '\0' && (count < NUMELMS(buffer) - 1); p++)
    {
        TCHAR ch = *p;
        if (ch == TEXT('\n')) buffer[count++] = TEXT('\r');
        buffer[count++] = ch;
    }
#   ifdef AC_CORE_WITH_OPENCL
        for (auto p = ac::core::Processor::info<ac::core::Processor::OpenCL>(); *p != '\0' && (count < NUMELMS(buffer) - 1); p++)
        {
            TCHAR ch = *p;
            if (ch == TEXT('\n')) buffer[count++] = TEXT('\r');
            buffer[count++] = ch;
        }
#   endif
#   ifdef AC_CORE_WITH_CUDA
        for (auto p = ac::core::Processor::info<ac::core::Processor::CUDA>(); *p != '\0' && (count < NUMELMS(buffer) - 1); p++)
        {
            TCHAR ch = *p;
            if (ch == TEXT('\n')) buffer[count++] = TEXT('\r');
            buffer[count++] = ch;
        }
#   endif
    buffer[count] = TEXT('\0');
    Edit_SetText(GetDlgItem(m_Dlg, IDC_EDIT_INFO), buffer);

    Static_SetText(GetDlgItem(m_Dlg, IDC_STATIC_VERSION), TEXT(AC_CORE_VERSION_STR));
    Static_SetText(GetDlgItem(m_Dlg, IDC_STATIC_COPYRIGHT),
        TEXT("Copyright (c) 2020-") TEXT(AC_BUILD_YEAR) TEXT(" the Anime4KCPP project"));

    isInitialized = true;
    return S_OK;
}
HRESULT PropertyPage::OnDeactivate()
{
    isInitialized = false;
    return S_OK;
}
HRESULT PropertyPage::OnApplyChanges()
{
    TCHAR buffer[STR_MAX_LENGTH] = {};
    TCHAR* endptr{};

    Edit_GetText(GetDlgItem(m_Dlg, IDC_EDIT_FACTOR), buffer, NUMELMS(buffer));
    auto factor = _tcstod(buffer, &endptr);
    if (endptr == buffer) factor = RegArgument::FactorDefault;
    gRegArgument.setFactor(factor);

    ComboBox_GetText(GetDlgItem(m_Dlg, IDC_COMBO_PROCESSOR), buffer, NUMELMS(buffer));
#   ifdef AC_CORE_WITH_OPENCL
        if (!_tcscmp(buffer, TEXT("opencl"))) gRegArgument.setProcessorId(ac::core::Processor::OpenCL);
        else
#   endif
#   ifdef AC_CORE_WITH_CUDA
        if (!_tcscmp(buffer, TEXT("cuda"))) gRegArgument.setProcessorId(ac::core::Processor::CUDA);
        else
#   endif
    gRegArgument.setProcessorId(ac::core::Processor::CPU);

    Edit_GetText(GetDlgItem(m_Dlg, IDC_EDIT_DEVICE), buffer, NUMELMS(buffer));
    auto device = _tcstol(buffer, &endptr, 10);
    if (endptr == buffer) device = RegArgument::DeviceDefault;
    gRegArgument.setDevice(device);

    Edit_GetText(GetDlgItem(m_Dlg, IDC_EDIT_LIMIT_WIDTH), buffer, NUMELMS(buffer));
    auto limitWidth = _tcstol(buffer, &endptr, 10);
    if (endptr == buffer) limitWidth = RegArgument::LimitWidthDefault;
    gRegArgument.setLimitWidth(limitWidth);

    Edit_GetText(GetDlgItem(m_Dlg, IDC_EDIT_LIMIT_HEIGHT), buffer, NUMELMS(buffer));
    auto limitHeight = _tcstol(buffer, &endptr, 10);
    if (endptr == buffer) limitHeight = RegArgument::LimitHeightDefault;
    gRegArgument.setLimitHeight(limitHeight);

    Edit_GetText(GetDlgItem(m_Dlg, IDC_EDIT_MODEL), buffer, NUMELMS(buffer));
    gRegArgument.setModelName(buffer);

    return S_OK;
}

RegArgument::RegArgument()
{
    RegCreateKeyEx(HKEY_CURRENT_USER, "Software\\Anime4KCPP\\DSFilter", 0, nullptr,
        REG_OPTION_NON_VOLATILE, KEY_WRITE | KEY_READ, nullptr, &key, nullptr);
}
RegArgument::~RegArgument()
{
    RegCloseKey(key);
}
double RegArgument::getFactor()
{
    DWORD size = sizeof(factor);
    auto data = reinterpret_cast<LPBYTE>(&factor);
    if (ERROR_SUCCESS == RegQueryValueEx(key, FactorValueName, nullptr, nullptr, data, &size))
        return factor;
    else return FactorDefault;
}
int RegArgument::getProcessorId()
{
    DWORD size = sizeof(processorId);
    auto data = reinterpret_cast<LPBYTE>(&processorId);
    if (ERROR_SUCCESS == RegQueryValueEx(key, ProcessorIdValueName, nullptr, nullptr, data, &size))
        return processorId;
    else return ProcessorIdDefault;
}
int RegArgument::getDevice()
{
    DWORD size = sizeof(device);
    auto data = reinterpret_cast<LPBYTE>(&device);
    if (ERROR_SUCCESS == RegQueryValueEx(key, DeviceValueName, nullptr, nullptr, data, &size))
        return device;
    else return DeviceDefault;
}
int RegArgument::getLimitWidth()
{
    DWORD size = sizeof(limitWidth);
    auto data = reinterpret_cast<LPBYTE>(&limitWidth);
    if (ERROR_SUCCESS == RegQueryValueEx(key, LimitWidthValueName, nullptr, nullptr, data, &size))
        return limitWidth;
    else return LimitWidthDefault;
}
int RegArgument::getLimitHeight()
{
    DWORD size = sizeof(limitHeight);
    auto data = reinterpret_cast<LPBYTE>(&limitHeight);
    if (ERROR_SUCCESS == RegQueryValueEx(key, LimitHeightValueName, nullptr, nullptr, data, &size))
        return limitHeight;
    else return LimitHeightDefault;
}
const TCHAR* RegArgument::getModelName()
{
    DWORD size = ModelNameMaxSize;
    auto data = reinterpret_cast<LPBYTE>(modelName);
    if (ERROR_SUCCESS == RegQueryValueEx(key, ModelNameValueName, nullptr, nullptr, data, &size))
    {
        modelName[size - 1] = TEXT('\0');
        return modelName;
    }
    else return ModelNameDefault;
}
void RegArgument::setFactor(const double v) const
{
    DWORD size = sizeof(factor);
    auto data = reinterpret_cast<const BYTE*>(&v);
    RegSetValueEx(key, FactorValueName, 0, REG_BINARY, data, size);
}
void RegArgument::setProcessorId(const int v) const
{
    DWORD size = sizeof(processorId);
    auto data = reinterpret_cast<const BYTE*>(&v);
    RegSetValueEx(key, ProcessorIdValueName, 0, REG_DWORD, data, size);
}
void RegArgument::setDevice(const int v) const
{
    DWORD size = sizeof(device);
    auto data = reinterpret_cast<const BYTE*>(&v);
    RegSetValueEx(key, DeviceValueName, 0, REG_DWORD, data, size);
}
void RegArgument::setLimitWidth(const int v) const
{
    DWORD size = sizeof(limitWidth);
    auto data = reinterpret_cast<const BYTE*>(&v);
    RegSetValueEx(key, LimitWidthValueName, 0, REG_DWORD, data, size);
}
void RegArgument::setLimitHeight(const int v) const
{
    DWORD size = sizeof(limitHeight);
    auto data = reinterpret_cast<const BYTE*>(&v);
    RegSetValueEx(key, LimitHeightValueName, 0, REG_DWORD, data, size);
}
void RegArgument::setModelName(const TCHAR* const v) const
{
    DWORD size = ModelNameMaxSize;
    auto data = reinterpret_cast<const BYTE*>(v);
    RegSetValueEx(key, ModelNameValueName, 0, REG_SZ, data, size);
}

RegArgument& RegArgument::instance()
{
    static RegArgument object{};
    return object;
}
