#ifndef AC_CORE_IMAGE_HPP
#define AC_CORE_IMAGE_HPP

#include <cstdint>
#include <memory>

#include "ACExport.hpp" // Generated by CMake

namespace ac::core
{
    class Image;

    // Resize the `src` based on the size of the `dst` if `fx` or `fy` <= 0, otherwise, calculate the size using `fx` and `fy`.
    // if `dst` is empty and fx or fy <= 0, nothing will be done.
    // if target size is same as `src`(eg: `fx` and `fy` == 1), then just make `dst` = `src` and return, no data will be copied.
    // `src` and `dst` can be the same image.
    AC_EXPORT void resize(const Image& src, Image& dst, double fx, double fy) noexcept;
    // if `fx` or `fy` <= 0, return `src`, otherwise, calculate the size using `fx` and `fy`.
    AC_EXPORT Image resize(const Image& src, double fx, double fy) noexcept;

    // convert RGB or RGBA to YUV.
    // if `yuv` is not an empty image, ensure its shape is correct, which won't be checked.
    // `rgb` and `yuv` can be the same image, which will be converted in place.
    AC_EXPORT void rgb2yuv(const Image& rgb, Image& yuv);
    // convert RGB or RGBA to Y and UV.
    // if `y` and `uv` are not empty images, ensure their shapes are correct, which won't be checked.
    // `rgb`, `y` and `uv` **cannot** be the same image.
    AC_EXPORT void rgb2yuv(const Image& rgb, Image& y, Image& uv);
    // convert RGB or RGBA to Y, U and V.
    // if `y`, `u` and `v` are not empty images, ensure their shapes are correct, which won't be checked.
    // `rgb`, `y`, `u` and `v` **cannot** be the same image.
    AC_EXPORT void rgb2yuv(const Image& rgb, Image& y, Image& u, Image& v);

    // convert RGBA to YUVA.
    // if `yuva` is not an empty image, ensure its shape is correct, which won't be checked.
    // `rgba` and `yuva` can be the same image, which will be converted in place.
    AC_EXPORT void rgba2yuva(const Image& rgba, Image& yuva);
    // convert RGBA to Y and UVA.
    // if `y` and `uva` are not empty images, ensure their shapes are correct, which won't be checked.
    // `rgba`, `y` and `uva` **cannot** be the same image.
    AC_EXPORT void rgba2yuva(const Image& rgba, Image& y, Image& uva);
    // convert RGBA to Y, U, V and A.
    // if `y`, `u`, `v` and `a` are not empty images, ensure their shapes are correct, which won't be checked.
    // `rgba`, `y`, `u`, `v` and `a` **cannot** be the same image.
    AC_EXPORT void rgba2yuva(const Image& rgba, Image& y, Image& u, Image& v, Image& a);

    // convert YUV to RGB.
    // if `rgb` is not an empty image, ensure its shape is correct, which won't be checked.
    // `yuv` and `rgb` can be the same image, which will be converted in place.
    AC_EXPORT void yuv2rgb(const Image& yuv, Image& rgb);
    // convert Y and UV to RGB.
    // if `rgb` is not an empty image, ensure its shape is correct, which won't be checked.
    // `y`, `uv` and `rgb` **cannot** be the same image.
    AC_EXPORT void yuv2rgb(const Image& y, const Image& uv, Image& rgb);
    // convert Y, U and V to RGB.
    // if `rgb` is not an empty image, ensure its shape is correct, which won't be checked.
    // `y`, `u`, `v` and `rgb` **cannot** be the same image.
    AC_EXPORT void yuv2rgb(const Image& y, const Image& u, const Image& v, Image& rgb);

    // convert YUVA to RGBA.
    // if `rgba` is not an empty image, ensure its shape is correct, which won't be checked.
    // `yuva` and `rgba` can be the same image, which will be converted in place.
    AC_EXPORT void yuva2rgba(const Image& yuva, Image& rgba);
    // convert Y and UVA to RGBA.
    // if `rgba` is not an empty image, ensure its shape is correct, which won't be checked.
    // `y`, `uva` and `rgba` **cannot** be the same image.
    AC_EXPORT void yuva2rgba(const Image& y, const Image& uva, Image& rgba);
    // convert Y, U and V to RGBA.
    // if `rgba` is not an empty image, ensure its shape is correct, which won't be checked.
    // `y`, `u`, `v`, `a` and `rgba` **cannot** be the same image.
    AC_EXPORT void yuva2rgba(const Image& y, const Image& u, const Image& v, const Image& a, Image& rgba);
    // remove padding in the stride of `src`, if necessary.
    // `src` and `dst` can be same.
    AC_EXPORT void unpadding(const Image& src, Image& dst);

    enum
    {
        IMREAD_UNCHANGED = 0,
        IMREAD_GRAYSCALE = 1,
        IMREAD_COLOR     = 3,
        IMREAD_RGB       = 3,
        IMREAD_RGBA      = 4,
    };

    AC_EXPORT Image imdecode(const void* buffer, int size, int flag) noexcept;
#ifdef AC_CORE_ENABLE_IMAGE_IO
    AC_EXPORT Image imread(const char* filename, int flag) noexcept;
    AC_EXPORT bool imwrite(const char* filename, const Image& image) noexcept;
#endif
}

class ac::core::Image
{
private:
    struct ImageData;

public:
    // 2(float) or 1(int) or 0(uint) << 8 | sizeof (type)
    using ElementType = int;
    static constexpr ElementType UInt8   = 0 << 8 | 1;
    static constexpr ElementType UInt16  = 0 << 8 | 2;
    static constexpr ElementType Float32 = 2 << 8 | 4;

public:
    AC_EXPORT Image() noexcept;
    AC_EXPORT Image(int w, int h, int c, ElementType elementType, int stride = 0);
    AC_EXPORT Image(int w, int h, int c, ElementType elementType, void* data, int stride);
    AC_EXPORT Image(const Image&) noexcept;
    AC_EXPORT Image(Image&&) noexcept;
    AC_EXPORT ~Image() noexcept;
    AC_EXPORT Image& operator=(const Image&) noexcept;
    AC_EXPORT Image& operator=(Image&&) noexcept;

    AC_EXPORT void create(int w, int h, int c, ElementType elementType, int stride = 0);

public:
    int width() const noexcept { return w; }
    int height() const noexcept { return h; }
    int channels() const noexcept { return c; }
    int stride() const noexcept { return pitch; }
    int size() const noexcept { return pitch * h; }
    int elementSize() const noexcept { return elementType & 0xff; }
    int channelSize() const noexcept { return c * elementSize(); }
    ElementType type() const noexcept { return elementType; }
    std::uint8_t* data() const noexcept { return static_cast<std::uint8_t*>(pixels); }
    std::uint8_t* line(const int y) const noexcept { return data() + y * pitch; }
    std::uint8_t* pixel(const int x, const int y) const noexcept { return line(y) + x * channelSize(); }
    void* ptr() const noexcept { return pixels; }
    void* ptr(const int y) const noexcept { return line(y); }
    void* ptr(const int x, const int y) const noexcept { return pixel(x, y); }
    bool empty() const noexcept { return pixels == nullptr; }
    bool isUint() const noexcept { return (elementType >> 8) == 0; }
    bool isInt() const noexcept { return (elementType >> 8) == 1; }
    bool isFloat() const noexcept { return (elementType >> 8) == 2; }

    bool operator==(const Image& other) const noexcept { return this->pixels == other.pixels; }
    bool operator!=(const Image& other) const noexcept { return !operator==(other); }

private:
    int w, h, c;
    ElementType elementType;
    int pitch;
    void* pixels;
    std::shared_ptr<ImageData> dptr;
};

#endif
