#ifndef AC_BINDING_C_CORE_H
#define AC_BINDING_C_CORE_H

#include <stddef.h>
#include <stdint.h>

#include "CACExport.h" // Generated by CMake

#ifdef __cplusplus
#   define CAC_API extern "C" CAC_EXPORT
#else
#   define CAC_API extern CAC_EXPORT
#endif

typedef struct ac_image ac_image;
typedef struct ac_processor ac_processor;

enum ac_image_element_type {
    AC_IMAGE_UINT8   = 0 << 8 | 1,
    AC_IMAGE_UINT16  = 0 << 8 | 2,
    AC_IMAGE_FLOAT32 = 2 << 8 | 4
};

enum ac_processor_type {
    AC_PROCESSOR_CPU    = 0,
    AC_PROCESSOR_OPENCL = 1,
    AC_PROCESSOR_CUDA   = 2
};

enum ac_model_type {
    AC_MODEL_ACNET_HDN0,
    AC_MODEL_ACNET_HDN1,
    AC_MODEL_ACNET_HDN2,
    AC_MODEL_ACNET_HDN3
};

enum ac_imread_flag
{
    AC_IMREAD_UNCHANGED = 0,
    AC_IMREAD_GRAYSCALE = 1,
    AC_IMREAD_COLOR     = 3,
    AC_IMREAD_RGB       = 3,
    AC_IMREAD_RGBA      = 4,
};

CAC_API ac_image* ac_image_create(int w, int h, int c, int element_type, void* data, int stride);
CAC_API ac_image* ac_image_create_empty(void);
CAC_API void ac_image_destroy(ac_image* image);
CAC_API int ac_image_width(const ac_image* image);
CAC_API int ac_image_height(const ac_image* image);
CAC_API int ac_image_channels(const ac_image* image);
CAC_API int ac_image_stride(const ac_image* image);
CAC_API int ac_image_size(const ac_image* image);
CAC_API int ac_image_element_size(const ac_image* image);
CAC_API int ac_image_channel_size(const ac_image* image);
CAC_API int ac_image_type(const ac_image* image);
CAC_API uint8_t* ac_image_data(const ac_image* image);
CAC_API uint8_t* ac_image_line(const ac_image* image, int idx);
CAC_API uint8_t* ac_image_pixel(const ac_image* image, int x, int y);
CAC_API void* ac_image_ptr(const ac_image* image);
CAC_API void* ac_image_line_ptr(const ac_image* image, int idx);
CAC_API void* ac_image_pixel_ptr(const ac_image* image, int x, int y);
CAC_API int ac_image_empty(const ac_image* image);
CAC_API int ac_image_is_uint(const ac_image* image);
CAC_API int ac_image_is_int(const ac_image* image);
CAC_API int ac_image_is_float(const ac_image* image);
CAC_API int ac_image_same(const ac_image* a, const ac_image* b);

CAC_API ac_processor* ac_processor_create(int processor_type, int device, int model_type);
CAC_API void ac_processor_destroy(ac_processor* processor);
CAC_API void ac_processor_process(ac_processor* processor, const ac_image* src, ac_image* dst, double factor);
CAC_API int ac_processor_ok(const ac_processor* processor);
CAC_API const char* ac_processor_error(const ac_processor* processor);
CAC_API const char* ac_processor_name(const ac_processor* processor);
CAC_API const char* ac_processor_info(int processor_type);

CAC_API void ac_imread(const char* filename, int flag, ac_image* image);
CAC_API int ac_imwrite(const char* filename, const ac_image* image);

CAC_API void ac_resize(const ac_image* src, ac_image* dst, double fx, double fy);
CAC_API void ac_rgb2yuv(const ac_image* rgb, ac_image* yuv);
CAC_API void ac_rgba2yuva(const ac_image* rgba, ac_image* yuva);
CAC_API void ac_yuv2rgb(const ac_image* yuv, ac_image* rgb);
CAC_API void ac_yuva2rgba(const ac_image* yuva, ac_image* rgba);

#endif
