#ifndef _TQUIC_H_
#define _TQUIC_H_

/* Don't modify this file manually. It is autogenerated by cbindgen. */

#include <stdarg.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include "openssl/ssl.h"
#include "tquic_def.h"

/**
 * The current QUIC wire version.
 */
#define QUIC_VERSION QUIC_VERSION_V1

/**
 * The QUIC Version 1
 */
#define QUIC_VERSION_V1 1

/**
 * The Connection ID MUST NOT exceed 20 bytes in QUIC version 1.
 * See RFC 9000 Section 17.2
 */
#define MAX_CID_LEN 20

/**
 * Available congestion control algorithms.
 */
typedef enum quic_congestion_control_algorithm {
  /**
   * CUBIC uses a cubic function instead of a linear window increase function
   * of the current TCP standards to improve scalability and stability under
   * fast and long-distance networks.
   */
  QUIC_CONGESTION_CONTROL_ALGORITHM_CUBIC,
  /**
   * BBR uses recent measurements of a transport connection's delivery rate,
   * round-trip time, and packet loss rate to build an explicit model of the
   * network path. The model is then used to control data transmission speed
   * and the maximum volume of data allowed in flight in the network at any
   * time.
   */
  QUIC_CONGESTION_CONTROL_ALGORITHM_BBR,
  /**
   * BBRv3 is the latest version of BBR, including various fixes and
   * algorithm updates that reduce packet re-transmit rate and slightly
   * improve latency. (Experimental)
   */
  QUIC_CONGESTION_CONTROL_ALGORITHM_BBR3,
  /**
   * COPA is a tunable delay-based congestion control algorithm. COPA is
   * based on an objective function where the trade-off between throughput
   * and delay can be configured via a user-specified parameter.
   * (Experimental)
   */
  QUIC_CONGESTION_CONTROL_ALGORITHM_COPA,
  /**
   * Dummy is a simple congestion controller with a static congestion window.
   * It is intended to be used for testing and experiments.
   */
  QUIC_CONGESTION_CONTROL_ALGORITHM_DUMMY,
} quic_congestion_control_algorithm;

/**
 * Available multipath scheduling algorithms.
 */
typedef enum quic_multipath_algorithm {
  /**
   * The scheduler sends packets over the path with the lowest smoothed RTT
   * among all available paths. It aims to optimize throughput and achieve
   * load balancing, making it particularly advantageous for bulk transfer
   * applications in heterogeneous networks.
   */
  QUIC_MULTIPATH_ALGORITHM_MIN_RTT,
  /**
   * The scheduler sends all packets redundantly on all available paths. It
   * utilizes additional bandwidth to minimize latency, thereby reducing the
   * overall flow completion time for applications with bounded bandwidth
   * requirements that can be met by a single path.
   * In scenarios where two paths with varying available bandwidths are
   * present, it ensures a goodput at least equivalent to the best single
   * path.
   */
  QUIC_MULTIPATH_ALGORITHM_REDUNDANT,
  /**
   * The scheduler sends packets over available paths in a round robin
   * manner. It aims to fully utilize the capacity of each path as the
   * distribution across all path is equal. It is only used for testing
   * purposes.
   */
  QUIC_MULTIPATH_ALGORITHM_ROUND_ROBIN,
} quic_multipath_algorithm;

/**
 * The stream's side to shutdown.
 */
typedef enum quic_shutdown {
  /**
   * Stop receiving data on the stream.
   */
  QUIC_SHUTDOWN_READ = 0,
  /**
   * Stop sending data on the stream.
   */
  QUIC_SHUTDOWN_WRITE = 1,
} quic_shutdown;

/**
 * Configurations about QUIC endpoint.
 */
typedef struct quic_config_t quic_config_t;

/**
 * A QUIC connection.
 */
typedef struct quic_conn_t quic_conn_t;

/**
 * Endpoint is an entity that can participate in a QUIC connection by
 * generating, receiving, and processing QUIC packets.
 *
 * There are two types of endpoints in QUIC: client and server. Endpoint may
 * maintain one or more QUIC connections.  Endpoint provides a high level API
 * to use the QUIC library.
 */
typedef struct quic_endpoint_t quic_endpoint_t;

/**
 * An iterator over FourTuple.
 */
typedef struct quic_path_address_iter_t quic_path_address_iter_t;

/**
 * An HTTP/3 configuration.
 */
typedef struct http3_config_t http3_config_t;

/**
 * An HTTP/3 connection.
 */
typedef struct http3_conn_t http3_conn_t;

/**
 * An HTTP/3 header list.
 */
typedef struct http3_headers_t http3_headers_t;

typedef struct quic_tls_config_t quic_tls_config_t;

typedef struct quic_tls_config_select_methods_t {
  struct quic_tls_config_t *(*get_default)(void *ctx);
  struct quic_tls_config_t *(*select)(void *ctx, const uint8_t *server_name, size_t server_name_len);
} quic_tls_config_select_methods_t;

typedef void *quic_tls_config_select_context_t;

typedef struct quic_transport_methods_t {
  /**
   * Called when a new connection has been created. This callback is called
   * as soon as connection object is created inside the endpoint, but
   * before the handshake is done. This callback is optional.
   */
  void (*on_conn_created)(void *tctx, struct quic_conn_t *conn);
  /**
   * Called when the handshake is completed. This callback is optional.
   */
  void (*on_conn_established)(void *tctx, struct quic_conn_t *conn);
  /**
   * Called when the connection is closed. The connection is no longer
   * accessible after this callback returns. It is a good time to clean up
   * the connection context. This callback is optional.
   */
  void (*on_conn_closed)(void *tctx, struct quic_conn_t *conn);
  /**
   * Called when the stream is created. This callback is optional.
   */
  void (*on_stream_created)(void *tctx, struct quic_conn_t *conn, uint64_t stream_id);
  /**
   * Called when the stream is readable. This callback is called when either
   * there are bytes to be read or an error is ready to be collected. This
   * callback is optional.
   */
  void (*on_stream_readable)(void *tctx, struct quic_conn_t *conn, uint64_t stream_id);
  /**
   * Called when the stream is writable. This callback is optional.
   */
  void (*on_stream_writable)(void *tctx, struct quic_conn_t *conn, uint64_t stream_id);
  /**
   * Called when the stream is closed. The stream is no longer accessible
   * after this callback returns. It is a good time to clean up the stream
   * context. This callback is optional.
   */
  void (*on_stream_closed)(void *tctx, struct quic_conn_t *conn, uint64_t stream_id);
  /**
   * Called when client receives a token in NEW_TOKEN frame. This callback
   * is optional.
   */
  void (*on_new_token)(void *tctx, struct quic_conn_t *conn, const uint8_t *token, size_t token_len);
} quic_transport_methods_t;

typedef void *quic_transport_context_t;

/**
 * Data and meta information of an outgoing packet.
 */
typedef struct quic_packet_out_spec_t {
  const struct iovec *iov;
  size_t iovlen;
  const void *src_addr;
  socklen_t src_addr_len;
  const void *dst_addr;
  socklen_t dst_addr_len;
} quic_packet_out_spec_t;

typedef struct quic_packet_send_methods_t {
  /**
   * Called when the connection is sending packets out.
   * On success, `on_packets_send()` returns the number of messages sent. If
   * this is less than count, the connection will retry with a further
   * `on_packets_send()` call to send the remaining messages. This callback
   * is mandatory.
   */
  int (*on_packets_send)(void *psctx, struct quic_packet_out_spec_t *pkts, unsigned int count);
} quic_packet_send_methods_t;

typedef void *quic_packet_send_context_t;

/**
 * Connection Id is an identifier used to identify a QUIC connection
 * at an endpoint.
 */
typedef struct ConnectionId {
  /**
   * length of cid
   */
  uint8_t len;
  /**
   * octets of cid
   */
  uint8_t data[MAX_CID_LEN];
} ConnectionId;

typedef struct ConnectionIdGeneratorMethods {
  /**
   * Generate a new CID
   */
  struct ConnectionId (*generate)(void *gctx);
  /**
   * Return the length of a CID
   */
  uint8_t (*cid_len)(void *gctx);
} ConnectionIdGeneratorMethods;

typedef void *ConnectionIdGeneratorContext;

/**
 * Meta information of an incoming packet.
 */
typedef struct quic_packet_info_t {
  const struct sockaddr *src;
  socklen_t src_len;
  const struct sockaddr *dst;
  socklen_t dst_len;
} quic_packet_info_t;

typedef struct quic_path_address_t {
  struct sockaddr_storage local_addr;
  socklen_t local_addr_len;
  struct sockaddr_storage remote_addr;
  socklen_t remote_addr_len;
} quic_path_address_t;

/**
 * Statistics about path
 */
typedef struct quic_path_stats_t {
  /**
   * The number of QUIC packets received.
   */
  uint64_t recv_count;
  /**
   * The number of received bytes.
   */
  uint64_t recv_bytes;
  /**
   * The number of QUIC packets sent.
   */
  uint64_t sent_count;
  /**
   * The number of sent bytes.
   */
  uint64_t sent_bytes;
  /**
   * The number of QUIC packets lost.
   */
  uint64_t lost_count;
  /**
   * The number of lost bytes.
   */
  uint64_t lost_bytes;
  /**
   * Total number of packets acked.
   */
  uint64_t acked_count;
  /**
   * Total number of bytes acked.
   */
  uint64_t acked_bytes;
  /**
   * Initial congestion window in bytes.
   */
  uint64_t init_cwnd;
  /**
   * Final congestion window in bytes.
   */
  uint64_t final_cwnd;
  /**
   * Maximum congestion window in bytes.
   */
  uint64_t max_cwnd;
  /**
   * Minimum congestion window in bytes.
   */
  uint64_t min_cwnd;
  /**
   * Maximum inflight data in bytes.
   */
  uint64_t max_inflight;
  /**
   * Total loss events.
   */
  uint64_t loss_event_count;
  /**
   * Total congestion window limited events.
   */
  uint64_t cwnd_limited_count;
  /**
   * Total duration of congestion windowlimited events in microseconds.
   */
  uint64_t cwnd_limited_duration;
  /**
   * Minimum roundtrip time in microseconds.
   */
  uint64_t min_rtt;
  /**
   * Maximum roundtrip time in microseconds.
   */
  uint64_t max_rtt;
  /**
   * Smoothed roundtrip time in microseconds.
   */
  uint64_t srtt;
  /**
   * Roundtrip time variation in microseconds.
   */
  uint64_t rttvar;
  /**
   * Whether the congestion controller is in slow start status.
   */
  bool in_slow_start;
  /**
   * Pacing rate estimated by congestion control algorithm.
   */
  uint64_t pacing_rate;
} quic_path_stats_t;

/**
 * Statistics about a QUIC connection.
 */
typedef struct quic_conn_stats_t {
  /**
   * Total number of received packets.
   */
  uint64_t recv_count;
  /**
   * Total number of bytes received on the connection.
   */
  uint64_t recv_bytes;
  /**
   * Total number of sent packets.
   */
  uint64_t sent_count;
  /**
   * Total number of bytes sent on the connection.
   */
  uint64_t sent_bytes;
  /**
   * Total number of lost packets.
   */
  uint64_t lost_count;
  /**
   * Total number of bytes lost on the connection.
   */
  uint64_t lost_bytes;
} quic_conn_stats_t;

typedef struct http3_methods_t {
  /**
   * Called when the stream got headers.
   */
  void (*on_stream_headers)(void *ctx,
                            uint64_t stream_id,
                            const struct http3_headers_t *headers,
                            bool fin);
  /**
   * Called when the stream has buffered data to read.
   */
  void (*on_stream_data)(void *ctx, uint64_t stream_id);
  /**
   * Called when the stream is finished.
   */
  void (*on_stream_finished)(void *ctx, uint64_t stream_id);
  /**
   * Called when the stream receives a RESET_STREAM frame from the peer.
   */
  void (*on_stream_reset)(void *ctx, uint64_t stream_id, uint64_t error_code);
  /**
   * Called when the stream priority is updated.
   */
  void (*on_stream_priority_update)(void *ctx, uint64_t stream_id);
  /**
   * Called when the connection receives a GOAWAY frame from the peer.
   */
  void (*on_conn_goaway)(void *ctx, uint64_t stream_id);
} http3_methods_t;

typedef void *http3_context_t;

/**
 * An extensible HTTP/3 Priority Parameters
 */
typedef struct http3_priority_t {
  uint8_t urgency;
  bool incremental;
} http3_priority_t;

typedef struct http3_header_t {
  uint8_t *name;
  uintptr_t name_len;
  uint8_t *value;
  uintptr_t value_len;
} http3_header_t;

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

/**
 * Check whether the protocol version is supported.
 */
bool quic_version_is_supported(uint32_t version);

/**
 * Create default configuration.
 * The caller is responsible for the memory of the Config and should properly
 * destroy it by calling `quic_config_free`.
 */
struct quic_config_t *quic_config_new(void);

/**
 * Destroy a Config instance.
 */
void quic_config_free(struct quic_config_t *config);

/**
 * Set the `max_idle_timeout` transport parameter in milliseconds.
 */
void quic_config_set_max_idle_timeout(struct quic_config_t *config, uint64_t v);

/**
 * Set handshake timeout in milliseconds. Zero turns the timeout off.
 */
void quic_config_set_max_handshake_timeout(struct quic_config_t *config, uint64_t v);

/**
 * Set the `max_udp_payload_size` transport parameter in bytes. It limits
 * the size of UDP payloads that the endpoint is willing to receive.
 */
void quic_config_set_recv_udp_payload_size(struct quic_config_t *config, uint16_t v);

/**
 * Enable the Datagram Packetization Layer Path MTU Discovery
 * default value is true.
 */
void enable_dplpmtud(struct quic_config_t *config, bool v);

/**
 * Set the maximum outgoing UDP payload size in bytes.
 * It corresponds to the maximum datagram size that DPLPMTUD tries to discovery.
 * The default value is `1200` which means let DPLPMTUD choose a value.
 */
void quic_config_set_send_udp_payload_size(struct quic_config_t *config, uintptr_t v);

/**
 * Set the `initial_max_data` transport parameter. It means the initial
 * value for the maximum amount of data that can be sent on the connection.
 * The value is capped by the setting `max_connection_window`.
 * The default value is `10485760`.
 */
void quic_config_set_initial_max_data(struct quic_config_t *config, uint64_t v);

/**
 * Set the `initial_max_stream_data_bidi_local` transport parameter.
 * The value is capped by the setting `max_stream_window`.
 * The default value is `5242880`.
 */
void quic_config_set_initial_max_stream_data_bidi_local(struct quic_config_t *config, uint64_t v);

/**
 * Set the `initial_max_stream_data_bidi_remote` transport parameter.
 * The value is capped by the setting `max_stream_window`.
 * The default value is `2097152`.
 */
void quic_config_set_initial_max_stream_data_bidi_remote(struct quic_config_t *config, uint64_t v);

/**
 * Set the `initial_max_stream_data_uni` transport parameter.
 * The value is capped by the setting `max_stream_window`.
 * The default value is `1048576`.
 */
void quic_config_set_initial_max_stream_data_uni(struct quic_config_t *config, uint64_t v);

/**
 * Set the `initial_max_streams_bidi` transport parameter.
 */
void quic_config_set_initial_max_streams_bidi(struct quic_config_t *config, uint64_t v);

/**
 * Set the `initial_max_streams_uni` transport parameter.
 */
void quic_config_set_initial_max_streams_uni(struct quic_config_t *config, uint64_t v);

/**
 * Set the `ack_delay_exponent` transport parameter.
 */
void quic_config_set_ack_delay_exponent(struct quic_config_t *config, uint64_t v);

/**
 * Set the `max_ack_delay` transport parameter.
 */
void quic_config_set_max_ack_delay(struct quic_config_t *config, uint64_t v);

/**
 * Set congestion control algorithm that the connection would use.
 */
void quic_config_set_congestion_control_algorithm(struct quic_config_t *config,
                                                  enum quic_congestion_control_algorithm v);

/**
 * Set the initial congestion window in packets.
 * The default value is 10.
 */
void quic_config_set_initial_congestion_window(struct quic_config_t *config, uint64_t v);

/**
 * Set the minimal congestion window in packets.
 * The default value is 2.
 */
void quic_config_set_min_congestion_window(struct quic_config_t *config, uint64_t v);

/**
 * Set the threshold for slow start in packets.
 * The default value is the maximum value of u64.
 */
void quic_config_set_slow_start_thresh(struct quic_config_t *config, uint64_t v);

/**
 * Set the minimum duration for BBR ProbeRTT state in milliseconds.
 * The default value is 200 milliseconds.
 */
void quic_config_set_bbr_probe_rtt_duration(struct quic_config_t *config, uint64_t v);

/**
 * Enable using a cwnd based on bdp during ProbeRTT state.
 * The default value is false.
 */
void quic_config_enable_bbr_probe_rtt_based_on_bdp(struct quic_config_t *config, bool v);

/**
 * Set the cwnd gain for BBR ProbeRTT state.
 * The default value is 0.75
 */
void quic_config_set_bbr_probe_rtt_cwnd_gain(struct quic_config_t *config, double v);

/**
 * Set the length of the BBR RTProp min filter window in milliseconds.
 * The default value is 10000 milliseconds.
 */
void quic_config_set_bbr_rtprop_filter_len(struct quic_config_t *config, uint64_t v);

/**
 * Set the cwnd gain for BBR ProbeBW state.
 * The default value is 2.0
 */
void quic_config_set_bbr_probe_bw_cwnd_gain(struct quic_config_t *config, double v);

/**
 * Set the delta in copa slow start state.
 */
void quic_config_set_copa_slow_start_delta(struct quic_config_t *config, double v);

/**
 * Set the delta in coap steady state.
 */
void quic_config_set_copa_steady_delta(struct quic_config_t *config, double v);

/**
 * Enable Using the rtt standing instead of the latest rtt to calculate queueing delay.
 */
void quic_config_enable_copa_use_standing_rtt(struct quic_config_t *config, bool v);

/**
 * Set the initial RTT in milliseconds. The default value is 333ms.
 * The configuration should be changed with caution. Setting a value less than the default
 * will cause retransmission of handshake packets to be more aggressive.
 */
void quic_config_set_initial_rtt(struct quic_config_t *config, uint64_t v);

/**
 * Enable pacing to smooth the flow of packets sent onto the network.
 * The default value is true.
 */
void quic_config_enable_pacing(struct quic_config_t *config, bool v);

/**
 * Set clock granularity used by the pacer.
 * The default value is 10 milliseconds.
 */
void quic_config_set_pacing_granularity(struct quic_config_t *config, uint64_t v);

/**
 * Set the linear factor for calculating the probe timeout.
 * The endpoint do not backoff the first `v` consecutive probe timeouts.
 * The default value is `0`.
 * The configuration should be changed with caution. Setting a value greater than the default
 * will cause retransmission to be more aggressive.
 */
void quic_config_set_pto_linear_factor(struct quic_config_t *config, uint64_t v);

/**
 * Set the upper limit of probe timeout in milliseconds.
 * A Probe Timeout (PTO) triggers the sending of one or two probe datagrams and enables a
 * connection to recover from loss of tail packets or acknowledgments.
 * See RFC 9002 Section 6.2.
 */
void quic_config_set_max_pto(struct quic_config_t *config, uint64_t v);

/**
 * Set the `active_connection_id_limit` transport parameter.
 */
void quic_config_set_active_connection_id_limit(struct quic_config_t *config, uint64_t v);

/**
 * Set the `enable_multipath` transport parameter.
 * The default value is false. (Experimental)
 */
void quic_config_enable_multipath(struct quic_config_t *config, bool enabled);

/**
 * Set the multipath scheduling algorithm
 * The default value is MultipathAlgorithm::MinRtt
 */
void quic_config_set_multipath_algorithm(struct quic_config_t *config,
                                         enum quic_multipath_algorithm v);

/**
 * Set the maximum size of the connection flow control window.
 * The default value is MAX_CONNECTION_WINDOW (15 MB).
 */
void quic_config_set_max_connection_window(struct quic_config_t *config, uint64_t v);

/**
 * Set the maximum size of the stream flow control window.
 * The value should not be greater than the setting `max_connection_window`.
 * The default value is MAX_STREAM_WINDOW (6 MB).
 */
void quic_config_set_max_stream_window(struct quic_config_t *config, uint64_t v);

/**
 * Set the Maximum number of concurrent connections.
 */
void quic_config_set_max_concurrent_conns(struct quic_config_t *config, uint32_t v);

/**
 * Set the key for reset token generation. The token_key_len should be not less
 * than 64.
 * Applicable to Server only.
 */
int quic_config_set_reset_token_key(struct quic_config_t *config,
                                    const uint8_t *token_key,
                                    size_t token_key_len);

/**
 * Set the lifetime of address token.
 * Applicable to Server only.
 */
void quic_config_set_address_token_lifetime(struct quic_config_t *config, uint64_t seconds);

/**
 * Set the key for address token generation. It also enables retry.
 * The token_key_len should be a multiple of 16.
 * Applicable to Server only.
 */
int quic_config_set_address_token_key(struct quic_config_t *config,
                                      const uint8_t *token_keys,
                                      size_t token_keys_len);

/**
 * Set whether stateless retry is allowed. Default is not allowed.
 * Applicable to Server only.
 */
void quic_config_enable_retry(struct quic_config_t *config, bool enabled);

/**
 * Set whether stateless reset is allowed.
 * Applicable to Endpoint only.
 */
void quic_config_enable_stateless_reset(struct quic_config_t *config, bool enabled);

/**
 * Set the length of source cid. The length should not be greater than 20.
 * Applicable to Endpoint only.
 */
void quic_config_set_cid_len(struct quic_config_t *config, uint8_t v);

/**
 * Set the anti-amplification factor.
 *
 * The server limits the data sent to an unvalidated address to
 * `anti_amplification_factor` times the received data.
 */
void quic_config_set_anti_amplification_factor(struct quic_config_t *config, uint8_t v);

/**
 * Set the batch size for sending packets.
 * Applicable to Endpoint only.
 */
void quic_config_set_send_batch_size(struct quic_config_t *config, uint16_t v);

/**
 * Set the buffer size for disordered zerortt packets on the server.
 * The default value is `1000`. A value of 0 will be treated as default value.
 * Applicable to Server only.
 */
void quic_config_set_zerortt_buffer_size(struct quic_config_t *config, uint16_t v);

/**
 * Set the maximum number of undecryptable packets that can be stored by one connection.
 * The default value is `10`. A value of 0 will be treated as default value.
 */
void quic_config_set_max_undecryptable_packets(struct quic_config_t *config, uint16_t v);

/**
 * Enable or disable encryption on 1-RTT packets. (Experimental)
 * The default value is true.
 * WARN: The The disable_1rtt_encryption extension is not meant to be used
 * for any practical application protocol on the open internet.
 */
void quic_config_enable_encryption(struct quic_config_t *config, bool v);

/**
 * Create a new TlsConfig.
 * The caller is responsible for the memory of the TlsConfig and should properly
 * destroy it by calling `quic_tls_config_free`.
 */
struct quic_tls_config_t *quic_tls_config_new(void);

/**
 * Create a new TlsConfig with SSL_CTX.
 * When using raw SSL_CTX, TlsSession::session() and TlsSession::set_keylog() won't take effect.
 * The caller is responsible for the memory of TlsConfig and SSL_CTX when use this function.
 */
struct quic_tls_config_t *quic_tls_config_new_with_ssl_ctx(SSL_CTX *ssl_ctx);

/**
 * Create a new client side TlsConfig.
 * The caller is responsible for the memory of the TlsConfig and should properly
 * destroy it by calling `quic_tls_config_free`.
 * For more information about `protos`, please see `quic_tls_config_set_application_protos`.
 */
struct quic_tls_config_t *quic_tls_config_new_client_config(const char *const *protos,
                                                            intptr_t proto_num,
                                                            bool enable_early_data);

/**
 * Create a new server side TlsConfig.
 * The caller is responsible for the memory of the TlsConfig and should properly
 * destroy it by calling `quic_tls_config_free`.
 * For more information about `protos`, please see `quic_tls_config_set_application_protos`.
 */
struct quic_tls_config_t *quic_tls_config_new_server_config(const char *cert_file,
                                                            const char *key_file,
                                                            const char *const *protos,
                                                            intptr_t proto_num,
                                                            bool enable_early_data);

/**
 * Destroy a TlsConfig instance.
 */
void quic_tls_config_free(struct quic_tls_config_t *tls_config);

/**
 * Set whether early data is allowed.
 */
void quic_tls_config_set_early_data_enabled(struct quic_tls_config_t *tls_config, bool enable);

/**
 * Set the session lifetime in seconds
 */
void quic_tls_config_set_session_timeout(struct quic_tls_config_t *tls_config, uint32_t timeout);

/**
 * Set the list of supported application protocols.
 * The `protos` is a pointer that points to an array, where each element of the array is a string
 * pointer representing an application protocol identifier. For example, you can define it as
 * follows: const char* const protos[2] = {"h3", "http/0.9"}.
 */
int quic_tls_config_set_application_protos(struct quic_tls_config_t *tls_config,
                                           const char *const *protos,
                                           intptr_t proto_num);

/**
 * Set session ticket key for server.
 */
int quic_tls_config_set_ticket_key(struct quic_tls_config_t *tls_config,
                                   const uint8_t *ticket_key,
                                   size_t ticket_key_len);

/**
 * Set the certificate verification behavior.
 */
void quic_tls_config_set_verify(struct quic_tls_config_t *tls_config, bool verify);

/**
 * Set the PEM-encoded certificate file.
 */
int quic_tls_config_set_certificate_file(struct quic_tls_config_t *tls_config,
                                         const char *cert_file);

/**
 * Set the PEM-encoded private key file.
 */
int quic_tls_config_set_private_key_file(struct quic_tls_config_t *tls_config,
                                         const char *key_file);

/**
 * Set CA certificates.
 */
int quic_tls_config_set_ca_certs(struct quic_tls_config_t *tls_config, const char *ca_path);

/**
 * Set TLS config selector.
 */
void quic_config_set_tls_selector(struct quic_config_t *config,
                                  const struct quic_tls_config_select_methods_t *methods,
                                  quic_tls_config_select_context_t context);

/**
 * Set TLS config.
 *
 * Note: Config doesn't own the TlsConfig when using this function.
 * It is the responsibility of the caller to release it.
 */
void quic_config_set_tls_config(struct quic_config_t *config, struct quic_tls_config_t *tls_config);

/**
 * Create a QUIC endpoint.
 *
 * The caller is responsible for the memory of the Endpoint and properly
 * destroy it by calling `quic_endpoint_free`.
 *
 * Note: The endpoint doesn't own the underlying resources provided by the C
 * caller. It is the responsibility of the caller to ensure that these
 * resources outlive the endpoint and release them correctly.
 */
struct quic_endpoint_t *quic_endpoint_new(struct quic_config_t *config,
                                          bool is_server,
                                          const struct quic_transport_methods_t *handler_methods,
                                          quic_transport_context_t handler_ctx,
                                          const struct quic_packet_send_methods_t *sender_methods,
                                          quic_packet_send_context_t sender_ctx);

/**
 * Destroy a QUIC endpoint.
 */
void quic_endpoint_free(struct quic_endpoint_t *endpoint);

/**
 * Set the connection id generator for the endpoint.
 * By default, the random connection id generator is used.
 */
void quic_endpoint_set_cid_generator(struct quic_endpoint_t *endpoint,
                                     const struct ConnectionIdGeneratorMethods *cid_gen_methods,
                                     ConnectionIdGeneratorContext cid_gen_ctx);

/**
 * Create a client connection.
 * If success, the output parameter `index` carrys the index of the connection.
 * Note: The `config` specific to the endpoint or server is irrelevant and will be disregarded.
 */
int quic_endpoint_connect(struct quic_endpoint_t *endpoint,
                          const struct sockaddr *local,
                          socklen_t local_len,
                          const struct sockaddr *remote,
                          socklen_t remote_len,
                          const char *server_name,
                          const uint8_t *session,
                          size_t session_len,
                          const uint8_t *token,
                          size_t token_len,
                          const struct quic_config_t *config,
                          uint64_t *index);

/**
 * Process an incoming UDP datagram.
 */
int quic_endpoint_recv(struct quic_endpoint_t *endpoint,
                       uint8_t *buf,
                       size_t buf_len,
                       const struct quic_packet_info_t *info);

/**
 * Return the amount of time until the next timeout event.
 */
uint64_t quic_endpoint_timeout(const struct quic_endpoint_t *endpoint);

/**
 * Process timeout events on the endpoint.
 */
void quic_endpoint_on_timeout(struct quic_endpoint_t *endpoint);

/**
 * Process internal events of all tickable connections.
 */
int quic_endpoint_process_connections(struct quic_endpoint_t *endpoint);

/**
 * Check whether the given connection exists.
 */
bool quic_endpoint_exist_connection(struct quic_endpoint_t *endpoint,
                                    const uint8_t *cid,
                                    size_t cid_len);

/**
 * Get the connection by index
 */
struct quic_conn_t *quic_endpoint_get_connection(struct quic_endpoint_t *endpoint, uint64_t index);

/**
 * Gracefully or forcibly shutdown the endpoint.
 * If `force` is false, cease creating new connections and wait for all
 * active connections to close. Otherwise, forcibly close all the active
 * connections.
 */
void quic_endpoint_close(struct quic_endpoint_t *endpoint, bool force);

/**
 * Get index of the connection
 */
uint64_t quic_conn_index(struct quic_conn_t *conn);

/**
 * Check whether the connection is a server connection.
 */
bool quic_conn_is_server(struct quic_conn_t *conn);

/**
 * Check whether the connection handshake is complete.
 */
bool quic_conn_is_established(struct quic_conn_t *conn);

/**
 * Check whether the connection is created by a resumed handshake.
 */
bool quic_conn_is_resumed(struct quic_conn_t *conn);

/**
 * Check whether the connection has a pending handshake that has progressed
 * enough to send or receive early data.
 */
bool quic_conn_is_in_early_data(struct quic_conn_t *conn);

/**
 * Check whether the established connection works in multipath mode.
 */
bool quic_conn_is_multipath(struct quic_conn_t *conn);

/**
 * Return the negotiated application level protocol.
 */
void quic_conn_application_proto(struct quic_conn_t *conn, const uint8_t **out, size_t *out_len);

/**
 * Return the server name in the TLS SNI extension.
 */
void quic_conn_server_name(struct quic_conn_t *conn, const uint8_t **out, size_t *out_len);

/**
 * Return the session data used by resumption.
 */
void quic_conn_session(struct quic_conn_t *conn, const uint8_t **out, size_t *out_len);

/**
 * Return details why 0-RTT was accepted or rejected.
 */
int quic_conn_early_data_reason(struct quic_conn_t *conn, const uint8_t **out, size_t *out_len);

/**
 * Send a Ping frame on the active path(s) for keep-alive.
 */
int quic_conn_ping(struct quic_conn_t *conn);

/**
 * Send a Ping frame on the specified path for keep-alive.
 * The API is only applicable to multipath quic connections.
 */
int quic_conn_ping_path(struct quic_conn_t *conn,
                        const struct sockaddr *local,
                        socklen_t local_len,
                        const struct sockaddr *remote,
                        socklen_t remote_len);

/**
 * Add a new path on the client connection.
 */
int quic_conn_add_path(struct quic_conn_t *conn,
                       const struct sockaddr *local,
                       socklen_t local_len,
                       const struct sockaddr *remote,
                       socklen_t remote_len,
                       uint64_t *index);

/**
 * Remove a path on the client connection.
 */
int quic_conn_abandon_path(struct quic_conn_t *conn,
                           const struct sockaddr *local,
                           socklen_t local_len,
                           const struct sockaddr *remote,
                           socklen_t remote_len);

/**
 * Migrate the client connection to the specified path.
 */
int quic_conn_migrate_path(struct quic_conn_t *conn,
                           const struct sockaddr *local,
                           socklen_t local_len,
                           const struct sockaddr *remote,
                           socklen_t remote_len);

/**
 * Return an iterator over path addresses.
 * The caller should properly destroy it by calling `quic_four_tuple_iter_free`.
 */
struct quic_path_address_iter_t *quic_conn_paths(struct quic_conn_t *conn);

/**
 * Destroy the FourTupleIter
 */
void quic_conn_path_iter_free(struct quic_path_address_iter_t *iter);

/**
 * Return the address of the next path.
 */
bool quic_conn_path_iter_next(struct quic_path_address_iter_t *iter, struct quic_path_address_t *a);

/**
 * Return the address of the active path
 */
bool quic_conn_active_path(const struct quic_conn_t *conn, struct quic_path_address_t *a);

/**
 * Return the latest statistics about the specified path.
 */
const struct quic_path_stats_t *quic_conn_path_stats(struct quic_conn_t *conn,
                                                     const struct sockaddr *local,
                                                     socklen_t local_len,
                                                     const struct sockaddr *remote,
                                                     socklen_t remote_len);

/**
 * Return statistics about the connection.
 */
const struct quic_conn_stats_t *quic_conn_stats(struct quic_conn_t *conn);

/**
 * Return the trace id of the connection
 */
void quic_conn_trace_id(struct quic_conn_t *conn, const uint8_t **out, size_t *out_len);

/**
 * Check whether the connection is draining.
 */
bool quic_conn_is_draining(struct quic_conn_t *conn);

/**
 * Check whether the connection is closing.
 */
bool quic_conn_is_closing(struct quic_conn_t *conn);

/**
 * Check whether the connection is closed.
 */
bool quic_conn_is_closed(struct quic_conn_t *conn);

/**
 * Check whether the connection was closed due to handshake timeout.
 */
bool quic_conn_is_handshake_timeout(struct quic_conn_t *conn);

/**
 * Check whether the connection was closed due to idle timeout.
 */
bool quic_conn_is_idle_timeout(struct quic_conn_t *conn);

/**
 * Check whether the connection was closed due to stateless reset.
 */
bool quic_conn_is_reset(struct quic_conn_t *conn);

/**
 * Returns the error from the peer, if any.
 */
bool quic_conn_peer_error(struct quic_conn_t *conn,
                          bool *is_app,
                          uint64_t *error_code,
                          const uint8_t **reason,
                          size_t *reason_len);

/**
 * Returns the local error, if any.
 */
bool quic_conn_local_error(struct quic_conn_t *conn,
                           bool *is_app,
                           uint64_t *error_code,
                           const uint8_t **reason,
                           size_t *reason_len);

/**
 * Set user context for the connection.
 */
void quic_conn_set_context(struct quic_conn_t *conn, void *data);

/**
 * Get user context for the connection.
 */
void *quic_conn_context(struct quic_conn_t *conn);

/**
 * Set the callback of keylog output.
 * `cb` is a callback function that will be called for each keylog.
 * `data` is a keylog message and `argp` is user-defined data that will be passed to the callback.
 */
void quic_conn_set_keylog(struct quic_conn_t *conn, void (*cb)(const uint8_t *data,
                                                               size_t data_len,
                                                               void *argp), void *argp);

/**
 * Set keylog file.
 * Note: The API is not applicable for Windows.
 */
void quic_conn_set_keylog_fd(struct quic_conn_t *conn, int fd);

/**
 * Set the callback of qlog output.
 * `cb` is a callback function that will be called for each qlog.
 * `data` is a qlog message and `argp` is user-defined data that will be passed to the callback.
 * `title` and `desc` respectively refer to the "title" and "description" sections of qlog.
 */
void quic_conn_set_qlog(struct quic_conn_t *conn,
                        void (*cb)(const uint8_t *data, size_t data_len, void *argp),
                        void *argp,
                        const char *title,
                        const char *desc);

/**
 * Set qlog file.
 * Note: The API is not applicable for Windows.
 */
void quic_conn_set_qlog_fd(struct quic_conn_t *conn, int fd, const char *title, const char *desc);

/**
 * Close the connection.
 */
int quic_conn_close(struct quic_conn_t *conn,
                    bool app,
                    uint64_t err,
                    const uint8_t *reason,
                    size_t reason_len);

/**
 * Set want write flag for a stream.
 */
int quic_stream_wantwrite(struct quic_conn_t *conn, uint64_t stream_id, bool want);

/**
 * Set want read flag for a stream.
 */
int quic_stream_wantread(struct quic_conn_t *conn, uint64_t stream_id, bool want);

/**
 * Read data from a stream.
 */
ssize_t quic_stream_read(struct quic_conn_t *conn,
                         uint64_t stream_id,
                         uint8_t *out,
                         size_t out_len,
                         bool *fin);

/**
 * Write data to a stream.
 */
ssize_t quic_stream_write(struct quic_conn_t *conn,
                          uint64_t stream_id,
                          const uint8_t *buf,
                          size_t buf_len,
                          bool fin);

/**
 * Create a new quic stream with the given id and priority.
 * This is a low-level API for stream creation. It is recommended to use
 * `quic_stream_bidi_new` for bidirectional streams or `quic_stream_uni_new`
 * for undirectional streams.
 */
int quic_stream_new(struct quic_conn_t *conn,
                    uint64_t stream_id,
                    uint8_t urgency,
                    bool incremental);

/**
 * Create a new quic bidiectional stream with the given priority.
 * If success, the output parameter `stream_id` carrys the id of the created stream.
 */
int quic_stream_bidi_new(struct quic_conn_t *conn,
                         uint8_t urgency,
                         bool incremental,
                         uint64_t *stream_id);

/**
 * Create a new quic uniectional stream with the given priority.
 * If success, the output parameter `stream_id` carrys the id of the created stream.
 */
int quic_stream_uni_new(struct quic_conn_t *conn,
                        uint8_t urgency,
                        bool incremental,
                        uint64_t *stream_id);

/**
 * Shutdown stream reading or writing.
 */
int quic_stream_shutdown(struct quic_conn_t *conn,
                         uint64_t stream_id,
                         enum quic_shutdown direction,
                         uint64_t err);

/**
 * Set the priority for a stream.
 */
int quic_stream_set_priority(struct quic_conn_t *conn,
                             uint64_t stream_id,
                             uint8_t urgency,
                             bool incremental);

/**
 * Return the stream’s send capacity in bytes.
 */
ssize_t quic_stream_capacity(struct quic_conn_t *conn, uint64_t stream_id);

/**
 * Return true if all the data has been read from the stream.
 */
bool quic_stream_finished(struct quic_conn_t *conn, uint64_t stream_id);

/**
 * Set user context for a stream.
 */
int quic_stream_set_context(struct quic_conn_t *conn, uint64_t stream_id, void *data);

/**
 * Return the stream’s user context.
 */
void *quic_stream_context(struct quic_conn_t *conn, uint64_t stream_id);

/**
 * Extract the header form, version and destination connection id from the
 * QUIC packet.
 */
int quic_packet_header_info(uint8_t *buf,
                            size_t buf_len,
                            uint8_t dcid_len,
                            bool *long_header,
                            uint32_t *version,
                            struct ConnectionId *dcid);

/**
 * Create default config for HTTP3.
 */
struct http3_config_t *http3_config_new(void);

/**
 * Destroy the HTTP3 config.
 */
void http3_config_free(struct http3_config_t *config);

/**
 * Set the `SETTINGS_MAX_FIELD_SECTION_SIZE` setting.
 * By default no limit is enforced.
 */
void http3_config_set_max_field_section_size(struct http3_config_t *config, uint64_t v);

/**
 * Set the `SETTINGS_QPACK_MAX_TABLE_CAPACITY` setting.
 * The default value is `0`.
 */
void http3_config_set_qpack_max_table_capacity(struct http3_config_t *config, uint64_t v);

/**
 * Set the `SETTINGS_QPACK_BLOCKED_STREAMS` setting.
 * The default value is `0`.
 */
void http3_config_set_qpack_blocked_streams(struct http3_config_t *config, uint64_t v);

/**
 * Create an HTTP/3 connection using the given QUIC connection. It also
 * initiate the HTTP/3 handshake by opening all control streams and sending
 * the local settings.
 */
struct http3_conn_t *http3_conn_new(struct quic_conn_t *quic_conn, struct http3_config_t *config);

/**
 * Destroy the HTTP/3 connection.
 */
void http3_conn_free(struct http3_conn_t *conn);

/**
 * Send goaway with the given id.
 */
int64_t http3_send_goaway(struct http3_conn_t *conn, struct quic_conn_t *quic_conn, uint64_t id);

/**
 * Set HTTP/3 connection events handler.
 */
void http3_conn_set_events_handler(struct http3_conn_t *conn,
                                   const struct http3_methods_t *methods,
                                   http3_context_t context);

/**
 * Process HTTP/3 settings.
 */
int http3_for_each_setting(const struct http3_conn_t *conn, int (*cb)(uint64_t identifier,
                                                                      uint64_t value,
                                                                      void *argp), void *argp);

/**
 * Process internal events of all streams of the specified HTTP/3 connection.
 */
int http3_conn_process_streams(struct http3_conn_t *conn, struct quic_conn_t *quic_conn);

/**
 * Process HTTP/3 headers.
 */
int http3_for_each_header(const struct http3_headers_t *headers, int (*cb)(const uint8_t *name,
                                                                           size_t name_len,
                                                                           const uint8_t *value,
                                                                           size_t value_len,
                                                                           void *argp), void *argp);

/**
 * Return true if all the data has been read from the stream.
 */
bool http3_stream_read_finished(struct quic_conn_t *conn, uint64_t stream_id);

/**
 * Create a new HTTP/3 request stream.
 * On success the stream ID is returned.
 */
int64_t http3_stream_new(struct http3_conn_t *conn, struct quic_conn_t *quic_conn);

/**
 * Create a new HTTP/3 request stream with the given priority.
 * On success the stream ID is returned.
 */
int64_t http3_stream_new_with_priority(struct http3_conn_t *conn,
                                       struct quic_conn_t *quic_conn,
                                       const struct http3_priority_t *priority);

/**
 * Close the given HTTP/3 stream.
 */
int http3_stream_close(struct http3_conn_t *conn,
                       struct quic_conn_t *quic_conn,
                       uint64_t stream_id);

/**
 * Set priority for an HTTP/3 stream.
 */
int http3_stream_set_priority(struct http3_conn_t *conn,
                              struct quic_conn_t *quic_conn,
                              uint64_t stream_id,
                              const struct http3_priority_t *priority);

/**
 * Send HTTP/3 request or response headers on the given stream.
 */
int http3_send_headers(struct http3_conn_t *conn,
                       struct quic_conn_t *quic_conn,
                       uint64_t stream_id,
                       const struct http3_header_t *headers,
                       size_t headers_len,
                       bool fin);

/**
 * Send HTTP/3 request or response body on the given stream.
 */
ssize_t http3_send_body(struct http3_conn_t *conn,
                        struct quic_conn_t *quic_conn,
                        uint64_t stream_id,
                        const uint8_t *body,
                        size_t body_len,
                        bool fin);

/**
 * Read request/response body from the given stream.
 */
ssize_t http3_recv_body(struct http3_conn_t *conn,
                        struct quic_conn_t *quic_conn,
                        uint64_t stream_id,
                        uint8_t *out,
                        size_t out_len);

/**
 * Parse HTTP/3 priority data.
 */
int http3_parse_extensible_priority(const uint8_t *priority,
                                    size_t priority_len,
                                    struct http3_priority_t *parsed);

/**
 * Send a PRIORITY_UPDATE frame on the control stream with specified
 * request stream ID and priority.
 */
int http3_send_priority_update_for_request(struct http3_conn_t *conn,
                                           struct quic_conn_t *quic_conn,
                                           uint64_t stream_id,
                                           const struct http3_priority_t *priority);

/**
 * Take the last PRIORITY_UPDATE for the given stream.
 */
int http3_take_priority_update(struct http3_conn_t *conn,
                               uint64_t prioritized_element_id,
                               int (*cb)(const uint8_t *priority_field_value,
                                         size_t priority_field_value_len,
                                         void *argp),
                               void *argp);

/**
 * Set logger.
 * `cb` is a callback function that will be called for each log message.
 * `data` is a '\n' terminated log message and `argp` is user-defined data that
 * will be passed to the callback.
 * `level` is a case-insensitive string used for specifying the log level. Valid
 * values are "OFF", "ERROR", "WARN", "INFO", "DEBUG", and "TRACE". If its value
 * is NULL or invalid, the default log level is "OFF".
 */
void quic_set_logger(void (*cb)(const uint8_t *data, size_t data_len, void *argp),
                     void *argp,
                     const char *level);

#ifdef __cplusplus
} // extern "C"
#endif // __cplusplus

#endif /* _TQUIC_H_ */
