#!/usr/bin/env python3

import os
import sys

import json
import matplotlib.pyplot as plt
import numpy as np
from matplotlib.colors import ListedColormap

# QUIC implementations
CLIENT_IMPLS = ["tquic", "lsquic", "quiche", "picoquic", "ngtcp2", "msquic",
                "s2n-quic", "quinn", "neqo", "kwik", "aioquic", "chrome",
                "go-x-net", "quic-go", "mvfst"]

SERVER_IMPLS = ["lsquic", "quiche", "picoquic", "ngtcp2", "msquic", "s2n-quic",
                "quinn", "neqo", "kwik", "aioquic", "nginx", "haproxy",
                "go-x-net", "quic-go", "mvfst"]

# Interop test cases
INTEROP_TESTS = ["handshake", "retry", "resumption", "zerortt", "amplificationlimit", "http3",
                 "ipv6", "chacha20", "keyupdate", "transfer", "multiplexing", "longrtt", "blackhole",
                 "handshakeloss", "handshakecorruption", "transferloss","transfercorruption"]


# Read a interop file generated by interop testing
def read_data(data_dir, server, client):
    dirname = "%s-%s/%s-%s-logs" % (server, client, server, client)
    path = os.path.join(data_dir, dirname, "interop.json")
    try:
        with open(path) as f:
            data = json.load(f)
            return convert_data(data["results"][0])
    except:
        return [0] * len(INTEROP_TESTS)


# Convert interop results to a vector
def convert_data(result):
    data = [0] * len(INTEROP_TESTS)
    for i, name in enumerate(INTEROP_TESTS):
        for item in result:
            if item["name"] != name:
                continue
            if item["result"] == "succeeded":
                data[i] = 3
            elif item["result"] == "unsupported":
                data[i] = 2
            else:
                data[i] = 1
            break
    return data


# Convert test result to a letter
def convert_text(value):
    if value == 3:
       return "Y" # success
    elif value == 2:
       return "U" # unsupported
    elif value == 1:
       return "N" # failure
    else:
       return "-" # unknown


# Plot the interop graph
def plot(data_dir, is_tquic_server):
    impls = CLIENT_IMPLS if is_tquic_server else SERVER_IMPLS
    name = "server" if is_tquic_server else "client"
    if is_tquic_server:
        data = [read_data(data_dir, "tquic", impl) for impl in impls]
    else:
        data = [read_data(data_dir, impl, "tquic") for impl in impls]
    interop_result = np.array(data)
    print(interop_result)

    fig, ax = plt.subplots()
    im = ax.imshow(interop_result, cmap=ListedColormap(['gray', 'red', 'blue', 'green']),
                   interpolation='nearest')
    ax.set_xticks(np.arange(len(INTEROP_TESTS)), labels=INTEROP_TESTS)
    ax.set_yticks(np.arange(len(impls)), labels=impls)
    plt.setp(ax.get_xticklabels(), rotation=45, ha="right",
             rotation_mode="anchor")
    for i in range(len(INTEROP_TESTS)):
        for j in range(len(impls)):
            text = ax.text(i, j, convert_text(interop_result[j, i]),
                           ha="center", va="center", color="w")
    ax.set_title("TQUIC %s interop results" % (name))
    fig.tight_layout()

    filename = "interop-tquic-%s.png" % (name)
    plt.savefig(filename, dpi=300)


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: %s [data_dir]" % (sys.argv[0]))
        exit(1)

    data_dir= sys.argv[1]
    plot(data_dir, True)
    plot(data_dir, False)

