//
// This source file is part of the Stanford Spezi open source project
//
// SPDX-FileCopyrightText: 2022 Stanford University and the project authors (see CONTRIBUTORS.md)
//
// SPDX-License-Identifier: MIT
//

import Foundation

/// Represents the parameters of the LLM.
public struct LLMLocalParameters: Sendable {
    /// Defaults of possible LLMs parameter settings.
    public enum Defaults {
        /// Default system prompt for local LLMs.
        public static let defaultSystemPrompt: String = {
            String(localized: LocalizedStringResource("SPEZI_LLM_LOCAL_SYSTEM_PROMPT", bundle: .atURL(from: .module)))
        }()
    }
    
    
    /// The to-be-used system prompt of the LLM
    let systemPrompt: String?
    /// Indicates the maximum output length generated by the LLM.
    let maxOutputLength: Int
    
    let extraEOSTokens: Set<String>
    /// Interval for displaying output after every N tokens generated.
    let displayEveryNTokens: Int
    
    /// Creates the ``LLMLocalParameters`` which wrap the underlying llama.cpp `llama_model_params` C struct.
    /// Is passed to the underlying llama.cpp model in order to configure the LLM.
    ///
    /// - Parameters:
    ///   - systemPrompt: The to-be-used system prompt of the LLM enabling fine-tuning of the LLMs behaviour. Defaults to the regular default chat-based LLM system prompt.
    ///   - maxOutputLength: The maximum output length generated by the Spezi LLM, defaults to `512`.
    ///   - extraEOSTokens: Additional tokens to use for end of string
    ///   - displayEveryNTokens: Interval for displaying output after every N tokens generated, defaults to `4`.
    public init(
        systemPrompt: String? = Defaults.defaultSystemPrompt,
        maxOutputLength: Int = 512,
        extraEOSTokens: Set<String> = [],
        displayEveryNTokens: Int = 4
    ) {
        self.systemPrompt = systemPrompt
        self.maxOutputLength = maxOutputLength
        self.extraEOSTokens = extraEOSTokens
        self.displayEveryNTokens = displayEveryNTokens
    }
}
