# LinuxStat [![Ruby Gem](https://badge.fury.io/rb/linux_stat.svg)](https://rubygems.org/gems/linux_stat) ![Workflow Status](https://github.com/souravgoswami/linux_stat/workflows/LinuxStat%20Test/badge.svg)

---

![logo](https://raw.githubusercontent.com/Souravgoswami/linux_stat/master/images/logo.jpg)

For reading the eyecandy HTML version, visit [linux_stat](https://souravgoswami.github.io/linux_stat/).

LinuxStat lets you read status of a Linux system. It can show you cpu stats and usages, memory stats and usages, swap stats and usages, battery usage, bios info,  kernel details, local ip, os details and parse os-release + lsb-release, etc.

It only works on Linux, and detecting the OS is upto the user of this gem.

---

| Table of Contents |
|:---|
| [Runtime Dependencies](https://github.com/Souravgoswami/linux_stat#runtime-dependencies) |
| [Build Dependencies](https://github.com/Souravgoswami/linux_stat#build-dependencies) |
| [Optional Runtime Dependencies](https://github.com/Souravgoswami/linux_stat#optional-runtime-dependencies) |
| [Installation](https://github.com/Souravgoswami/linux_stat#installation) |
| [Usage](https://github.com/Souravgoswami/linux_stat#usage) |
| [Notes](https://github.com/Souravgoswami/linux_stat#notes) |
| [Return Types](https://github.com/Souravgoswami/linux_stat#return-types) |
| [Ruby On Rails](https://github.com/Souravgoswami/linux_stat#ruby-on-rails) |
| [Android](https://github.com/Souravgoswami/linux_stat#android) |
| [Cloning this Repo](https://github.com/Souravgoswami/linux_stat#cloning-this-repo) |
| [Development](https://github.com/Souravgoswami/linux_stat#development) |
| [Testing](https://github.com/Souravgoswami/linux_stat#testing) |
| [Contributing](https://github.com/Souravgoswami/linux_stat#contributing) |
| [License](https://github.com/Souravgoswami/linux_stat#license) |
| [Languages](https://github.com/Souravgoswami/linux_stat#languages) |

---

## Runtime Dependencies:

1. Kernel: Any Linux distribution with Kernel 3.14+.
2. Ruby: Ruby 2.3.0 and above.

## Build Dependencies:
#### You need to have the C compiler installed to be able to compile the C extensions.

+ On Arch Linux / Manjaro / Archlabs / Other Arch Based Distributions
```
# pacman -S gcc make ruby
```

+ On Debian / Ubuntu / Linux Mint / Pop!_OS / Raspberry Pi OS / Other Debian Based Distributions
```
# apt install gcc build-essential ruby ruby-dev build-essential
```

+ Gentoo / Gentoo Based Distributions
```
# emerge --ask dev-lang/ruby
```

+ Fedora / AmazonLinux* / CentOS* / Other RedHat Based Distributions
```
# yum install gcc ruby-devel ruby make
```

You can run linux_stat on *AmazonLinux and *CentOS if you have Ruby 2.3.0+.

+ OpenSUSE
```
# zypper install gcc ruby ruby-devel make
```

+ You can remove the above packages once the gem is installed.

---

## Optional Runtime Dependencies

You need hwdata to decode vendor and product ids if you use LinuxStat::USB and/or LinuxStat::PCI

You can install hwdata simply.

+ Arch:

```
# pacman -S hwids
```

+ Debian based systems:

```
# apt install hwdata
```

+ Gentoo / Gentoo Based Distributions
```
# emerge --ask sys-apps/hwids
```

+ Fedora / Amazon Linux / CentOS
```
# yum install hwdata
```

+ OpenSUSE
```
zypper install hwdata
```

But without hwdata, it won't show such information.

You can also point to a downloaded copy of hwdata (pci.ids / usb.ids) file.

Follow [Note 7](https://github.com/Souravgoswami/linux_stat#note-7-hwdata) below for more information on that.

---

## Installation

Add this line to your application's Gemfile:

```ruby
gem 'linux_stat'
```

And then execute:

```
$ bundle install
```

Or install it yourself as:

```
$ gem install linux_stat
```

## Usage

Following are the LinuxStat modules and module functions in a table.
Usages.md is generated by `linuxstat.rb -md` command, which is available after the installation.

For ease of use, LinuxStat is also assigned to the LS constant.
So LinuxStat::USB can be replaced with LS::USB for example.

| LinuxStat Module | Description |
|:---|:---|
| [LinuxStat::BIOS](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatbios) | System's BIOS related informaion |
| [LinuxStat::Battery](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatbattery) | System's first Battery realted information |
| [LinuxStat::CPU](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatcpu) | System's CPU usage and other related information |
| [LinuxStat::FS](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatfs) | System's file system related information. It's used by Filesystem module. |
| [LinuxStat::Filesystem](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatfilesystem) | System's file system usage and other related information |
| [LinuxStat::FTW](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatftw) | File Tree Walk: Walks through a file and gives you data related to it's own and sub files and directories |
| [LinuxStat::Kernel](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatkernel) | System's kernel related information |
| [LinuxStat::Memory](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatmemory) | System's memory (RAM) usage and other related information |
| [LinuxStat::Mounts](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatmounts) | System's mount point related information |
| [LinuxStat::Net](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatnet) | System's internet usage and other information |
| [LinuxStat::OS](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatos) | System's OS related information and parse release files |
| [LinuxStat::PCI](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatpci) | System's PCI device related information (optional parsing of hwdata) |
| [LinuxStat::PrettifyBytes](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatprettifybytes) | Convert bytes into human readable format (kB, kiB, etc.) |
| [LinuxStat::ProcFS](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatprocfs) | Read proc file system in C for faster access. Used accross |
| [LinuxStat::Process](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatprocess) | System's processes and related information |
| [LinuxStat::ProcessInfo](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatprocessinfo)   | System's single process CPU, memory usages, CPU time, etc. info |
| [LinuxStat::Swap](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatswap) | System's swap related information |
| [LinuxStat::Sysconf](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatsysconf) | Used by other LinuxStat modules, provides various system details like pagesize, max children, open files, etc |
| [LinuxStat::Sysinfo](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatsysinfo) | Used by other LinuxStat modules, faster access to information like totalram, freeram, totalhigh, load average, etc. |
| [LinuxStat::Thermal](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatthermal) | System's thermal status, fan related info, sensor count, fan count, etc. |
| [LinuxStat::USB](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatusb) | System's USB device related information and other counts (optional parsing of hwdata) |
| [LinuxStat::Uname](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatuname) | Used by other LinuxStat modules, System's Uname information in C, that's also provided by the `uname` command. |
| [LinuxStat::User](https://github.com/Souravgoswami/linux_stat/blob/master/Usages.md#linuxstatuser) | Systems' various user related information, provides current user name, home directory, etc. |

---

# Notes

## Note 1: CPU usage, and Net usage
To calculate the current usage, we need to get two usages at a given interval, and subtract the 2nd from the 1st.
For example, if the current download (`LinuxStat::Net.total_bytes_received`) is 1000 bytes, and if 0.1 seconds ago, it was 100 bytes, that means 900 bytes was received in 0.1 seconds.
That means the current speed is 9000 bytes/s or 9 kB/s.

Without the polling, it's not really possible to calculate the current usage. Although the total usage can be calculated.
A system monitor does that, too...

Thus these methods requires a polling interval:

1. LinuxStat::CPU.stat, usage, total_usage, usage.
2. LinuxStat::ProcessInfo.cpu_usage, cpu_stat.
3. LinuxStat::Net.usage, current_usage.

They sleep for a given interval and then differentiate between the data.

For more info look at the ri documentation for the above methods.

These methods can slow down your application a bit unless you implement them in a thread.

Other methods doesn't have the sleep implemented, and they just works under a millisecond.

For example:

```
LinuxStat::CPU.stat(0.1)
=> {0=>7.69, 1=>0.0, 2=>0.0, 3=>18.18, 4=>10.0}
```
This will sleep for 0.1 seconds. To be reliable, use a time like 0.05 seconds or so.

If you want to build a system monitor and don't want to wait, you have to do something like this:

```
#!/usr/bin/ruby
require 'linux_stat'

usages = []
thread = Thread.new { }
counter = 0

while true
	thread = Thread.new { usages = LinuxStat::CPU.usages(0.5).values } unless thread.alive?

	# clears the screen and prints the info
	puts "\e[2J\e[H\e[3J"\
	"#{counter += 1}\n"\
	"\e[1;33mTotal CPU Usage:\e[0m #{usages[0]}%\n"\
	"#{usages[1..-1].to_a.map.with_index { |x, i| "\e[1;33mCore #{i}\e[0m => #{x}%\n" }.join}"\
	"Total Download: #{LinuxStat::PrettifyBytes.convert_decimal LinuxStat::Net.total_bytes_received}\n"\
	"Total Upload: #{LinuxStat::PrettifyBytes.convert_decimal LinuxStat::Net.total_bytes_transmitted}"
end
```

This will not wait in every loop for 0.5 seconds, but it will not update the cpu usage in every loop either.
So what you will be seeing in the CPU usage in every 0.5 seconds interval.

You will also see the counter increases like crazy. Which means it's not getting waited for 0.5 seconds.

But the other methods doesn't have this delay, thus in this example,
you will be able see the "Total Download" and "Total Upload" in real time,
well as soon as the Linux kernel updates the data and ruby executes the loop.

Just run the linuxstat.rb command to test what method takes what time measured in microseconds.

## Note 2: Count CPU

There are confusingly 6 different methods to count the number of CPU. But they are here for a reason!

Well this section actually demystifies the methods.

1. The good old `LinuxStat::CPU.count()`:

It gets the configured CPU for the system. It doesn't count for hotplugged CPU.
If 3 out of 4 CPU are hotplugged out, it will still show 4. It calls `sysconf(_SC_NPROCESSORS_CONF)`

2. The mysterious `LinuxStat::ProcessInfo.nproc(pid = $$)`:

[ Also aliased to `LinuxStat::ProcessInfo.count_cpu()` ]

It returns the number of processors, like the other 3 methods.
Without any arguments, it's like running `require 'etc' ; puts Etc.nprocessors`

So there the caveat of checking what the number of processors are actually allocated to the pid.

For example:

```
$ ruby -r linux_stat -e "puts LS::ProcessInfo.nproc"
4

$ taskset -c 0 ruby -r linux_stat -e "puts LS::ProcessInfo.nproc"
1

$ taskset -c 0-1 ruby -r linux_stat -e "puts LS::ProcessInfo.nproc"
2

$ taskset -c 0-1,3 ruby -r linux_stat -e "puts LS::ProcessInfo.nproc"
3

$ taskset -c 0-1,3 ruby -r linux_stat -e "puts LS::ProcessInfo.nproc "
3
```

Or with argument:

```
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::ProcessInfo.command_name 4775
=> "electron"

irb(main):003:0> LinuxStat::ProcessInfo.nproc 4775
=> 4
```

3. The `LinuxStat::CPU.online()`:

This returns the number of online CPU as an Array. It doesn't get affected by taskset or anything.

For example:

```
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::CPU.online
=> [0, 1, 3]
```

By using `LinuxStat::CPU.online.count` you count the actual online CPU on your system.

Any n number of CPU can get hotplugged in and out, and this will report that correctly.

It just gets the info from /proc/stat; but if it fails it will read /sys/devices/system/cpu/online
and parse the output to get an array.

4. The `LinuxStat::CPU.count_online`
It's a more robust method that counts the online CPU. It shouldn't fail in most if not all cases!
But if it fails for some really spooky reasons, it will return nil.

5. The `LinuxStat::CPU.offline()`:

This returns the number of offline CPU as an Array. It doesn't get affected by taskset or anything.

For example:

```
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::CPU.offline
=> [2]
```

Any n number of CPU can get hotplugged in and out, and this will report that correctly.

It just gets the info from /sys/devices/system/cpu/offline, and parses the output.

6. The `LinuxStat::Sysconf.processor_configured()`:

Sounds repetitive! Actually yes, this is written in C, and it is called by `LinuxStat::CPU.count`.

The difference is that `LinuxStat::CPU.count` caches the return value, and this method doesn't.

7. The `LinuxStat::Sysconf.processor_online()`:

This may again sound repititive to LinuxStat::CPU.online, but it's actually not!

If you are using while loops, it might not report the correct number of CPU everytime.

Worst, it can take a long time to update the total number of CPU.

The benefit is, it's quite fast!

It's mostly here just for the sake of completeness to sysconf.

## Note 3: Filesystem

Filesystem can take arguments. By default it's '/' or the root of the system...

But for the sake of example, to get the free disk space of /, you do:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::Filesystem.free('/').fdiv(1024 ** 3).to_s << " GiB"
=> "35.666873931884766 GiB"
```

To see the free and total space of a thumbdrive:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::Mounts.mount_point('/dev/sdb1')
=> "/run/media/sourav/5c2b7af7-d4c3-4ab4-a035-06d18ffc8e6f"

irb(main):003:0> thumbdrive = _
=> "/run/media/sourav/5c2b7af7-d4c3-4ab4-a035-06d18ffc8e6f"

irb(main):004:0> LinuxStat::Filesystem.free(thumbdrive).fdiv(1024 ** 3).to_s << " GiB"
=> "2.504791259765625 GiB"

irb(main):005:0> LinuxStat::Filesystem.total(thumbdrive).fdiv(1024 ** 3).to_s << " GiB"
=> "29.305004119873047 GiB"
```

## Note 4: ProcessInfo

All the methods LinuxStat::ProcessInfo can take an argument containing the Process ID of a process.
By default it's $$ or the PID of the current process, ruby, itself.

Example:
Say you want to see how much CPU Firefox is consuming, for that you have to do the following (firefox can create a lot of child process though):

1. Get the PID of Firefox:
```
LinuxStat::Process.names.find { |x| x[1].include? 'firefox' }[0]
=> 770 # but this differs all the time
```

2. Get the CPU usage:
```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> pid = LinuxStat::Process.names.find { |x| x[1].include? 'firefox' }[0]
=> 770

irb(main):003:0> LinuxStat::ProcessInfo.cpu_usage(pid: pid)
=> 0.0

irb(main):004:0> LinuxStat::ProcessInfo.cpu_usage(pid: pid)
=> 15.0
```

To get the memory usage of Firefox (for example):

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::ProcessInfo.mem_stat(LinuxStat::Process.names.find { |x| x[1].include? 'firefox'.freeze }[0])
=> {:memory=>468472, :virtual_memory=>4754080, :resident_memory=>814388}
```

To get ONLY the memory usage in MiB:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::ProcessInfo.memory(LinuxStat::Process.names.find { |x| x[1].include? 'firefox'.freeze }[0]).fdiv(1024).round(2).to_s << " MiB"
=> "467.51 MiB"
```

## Note 5: FS

LinuxStat::FS module gives you the raw info in Hash collected from statvfs.

It's not documented above because it's not suggested to run this directly. But it shouldn't cause any issue. `LinuxStat::Filesystem.stat_raw(fs = '/')` does that automatically.

It always requires an argument, and it's very fast. It directly calls the C API without any intermediate Ruby code.

For example, to get the info about '/' or root:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::FS.stat('/')
=> {:block_size=>4096, :fragment_size=>4096, :blocks=>29292283, :block_free=>9349843, :block_avail_unpriv=>9349843, :inodes=>58612160, :free_inodes=>56708247, :filesystem_id=>2050, :mount_flags=>1024, :max_filename_length=>255}

irb(main):003:0> t = Time.now ; puts LinuxStat::FS.stat('/') ; Time.now - t
{:block_size=>4096, :fragment_size=>4096, :blocks=>29292283, :block_free=>9349843, :block_avail_unpriv=>9349843, :inodes=>58612160, :free_inodes=>56708247, :filesystem_id=>2050, :mount_flags=>1024, :max_filename_length=>255}
=> 5.0468e-05
```

To learn more about them, just run ri and the method name. To see all available methods.

## Note 6: User
Most of the LinuxStat::User supports arguments.

For example, to get a user's home by the username:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::User.home_by_username('root')
=> "/root"

irb(main):003:0> LinuxStat::User.home_by_username('ftp')
=> "/srv/ftp"

irb(main):004:0> LinuxStat::User.home_by_username('mail')
=> "/var/spool/mail"
```

Or to get the user's home by the GID/UID:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::User.homes_by_uid(1001)
=> ["/home/userx", "/home/userz"]

irb(main):003:0> LinuxStat::User.homes_by_uid(1000)
=> ["/home/sourav"]

irb(main):004:0> LinuxStat::User.home_by_gid(1001)
=> "/home/userx"

irb(main):005:0> LinuxStat::User.home_by_gid(1000)
=> "/home/sourav"

irb(main):006:0> LinuxStat::User.home_by_gid(0)
=> "/root"
```

Or to get the UID/GID by username:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::User.uid_by_username('root')
=> 0

irb(main):003:0> LinuxStat::User.uid_by_username('ftp')
=> 14

irb(main):004:0> LinuxStat::User.gid_by_username('ftp')
=> 11

irb(main):005:0> LinuxStat::User.gid_by_username('InvalidUser')
=> nil
```

Or to get the current user (in docker for example):

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::User.get_current_user
=> "x"

irb(main):003:0> LinuxStat::User.get_user
=> "x"

irb(main):004:0> LinuxStat::User.get_login
=> ""
```

Right, the get_login() can return an empty string. But LinuxStat::User.get_user also aliased as LinuxStat::User.get_current_user shouldn't return an empty string under most circumstances.

## Note 7: Hwdata
The PCI and USB modules actually rely on hwdata found in /usr/share/hwdata/.
The LS::USB.devices_stat and LS::PCI.devices_stat returns the information in a Hash:

```
$ ruby -r linux_stat -e "puts LS::USB.devices_stat.to_s[0..200]"
[{:path=>"/sys/bus/usb/devices/1-1.2/", :id=>"04d9:1203", :vendor_id=>"04d9", :product_id=>"1203", :bus_num=>1, :dev_num=>7, :hwdata=>{:vendor=>"Holtek Semiconductor, Inc.", :product=>"Keyboard"}, :aut
```

But if the files are not available, it won't return hwdata related information.

So it's suggested to install hwdata. But you might face issues with heroku and
other online PaaS where you can't install it. So Version 1.1.1+ comes with a module function called `hwdata_file = file`.

+ You can use any usb.ids or pci.ids files:

```
LS::PCI.hwdata_file = File.join(__dir__, 'hwdata', 'pci.ids')
LS::USB.hwdata_file = File.join(__dir__, 'hwdata', 'usb.ids')
```

Assuming that you have `pci.ids` and `usb.ids` under ./hwdata directory.

On rails, you can put this (replace `__dir__` with `Rails.root`) inside environment.rb.

But do note that the file can be set only once. It's suggested to do that in the beginning of your app.

+ There's one method to check if the hwdata file was already set:

```
irb(main):001:0' require 'linux_stat'
=> true

irb(main):002:0> LS::USB.hwdata_file_set?
=> false

irb(main):003:0> LS::USB.devices_stat ; ''
=> ""

irb(main):004:0> LS::USB.hwdata_file_set?
=> true
```

It works on USB and PCI modules.

Once the file is set, calling `LS::PCI.hwdata_file = file` is futile.

+ Initializing hwdata can take 0.1 to 0.2 seconds at the first, so there's a method to initialize_hwdata at first:

```
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LS::PCI.initialize_hwdata
=> true

irb(main):003:0> LS::PCI.initialize_hwdata
=> false
```

It will return true if it worked, else it will return false. It's intended to be done once.

If you don't initialize and call methods that utilizes hwdata, they will call it and the first
call may take 0.1 to 0.2 seconds, the consecutive calls will then take under a millisecond.

## Note 8: PrettifyBytes
Often times we need to work with KB, MB GB, TB, or KiB, MiB, GiB, TiB, etc.
And we need some work to convert bytes to those units.
Because LinuxStat provides a lot of data in bytes, and kilobytes, it's quite tedious to convert them all the time.
To avoid such duplication, it comes with a PrettifyBytes module.

For example, to convert bytes to decimal suffixes:

```
$irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::PrettifyBytes.convert_decimal(1000)
=> "1.00 kilobyte"

irb(main):003:0> LinuxStat::PrettifyBytes.convert_decimal(10000)
=> "10.00 kilobytes"

irb(main):004:0> LinuxStat::PrettifyBytes.convert_decimal(100000)
=> "100.00 kilobytes"

irb(main):005:0> LinuxStat::PrettifyBytes.convert_decimal(10 ** 13)
=> "10.00 terabytes"
```

To convert bytes to binary suffixes:

```
irb(main):006:0> LinuxStat::PrettifyBytes.convert_binary(1000)
=> "1000.00 bytes"

irb(main):007:0> LinuxStat::PrettifyBytes.convert_binary(10000)
=> "9.77 kibibytes"

irb(main):008:0> LinuxStat::PrettifyBytes.convert_binary(100000)
=> "97.66 kibibytes"

irb(main):009:0> LinuxStat::PrettifyBytes.convert_binary(10 ** 13)
=> "9.09 tebibytes"
```

To convert them to short Metric decimal suffixes:

```
irb(main):010:0> LinuxStat::PrettifyBytes.convert_short_decimal(1000)
=> "1.00 kB"

irb(main):011:0> LinuxStat::PrettifyBytes.convert_short_decimal(10000)
=> "10.00 kB"

irb(main):012:0> LinuxStat::PrettifyBytes.convert_short_decimal(100000)
=> "100.00 kB"

irb(main):013:0> LinuxStat::PrettifyBytes.convert_short_decimal(10 ** 13)
=> "10.00 TB"
```

To convert them to short IEC binary suffixes:

```
irb(main):014:0> LinuxStat::PrettifyBytes.convert_short_binary(1000)
=> "1000 B"

irb(main):015:0> LinuxStat::PrettifyBytes.convert_short_binary(10000)
=> "9.77 KiB"

irb(main):016:0> LinuxStat::PrettifyBytes.convert_short_binary(100000)
=> "97.66 KiB"

irb(main):017:0> LinuxStat::PrettifyBytes.convert_short_binary(10 ** 13)
=> "9.09 TiB"
```

It can support values upto hundreds of yottabytes and yobibytes, or yb and yib. You can also do stuff like:

```
$ irb
irb(main):001:0> require 'linux_stat'
=> true

irb(main):002:0> LinuxStat::PrettifyBytes.convert_short_decimal(LinuxStat::Mounts.device_stat('/dev/sdb1')[:total])
=> "31.47 GB"

irb(main):003:0> LinuxStat::PrettifyBytes.convert_short_binary(LinuxStat::Mounts.device_stat('/dev/sdb1')[:total])
=> "29.31 GiB"

irb(main):004:0> LinuxStat::PrettifyBytes.convert_short_binary(LinuxStat::Mounts.device_stat('/dev/sdb1')[:used])
=> "26.80 GiB"

irb(main):005:0> LinuxStat::PrettifyBytes.convert_short_binary(LinuxStat::Mounts.device_stat('/dev/sdb1')[:available])
=> "2.51 GiB"
```

Read the ri documentation for more info.

---

## Return Types
+ In general, if a method returns either a Float or a Integer or a Time, it will return a Float or Integer or Time in all cases. But if the status isn't available, it will return nil.

+ If the method returns a Hash / Array, it will return return Hash / Array in all cases. If the status isn't available, it will return an empty Hash / Array.

+ If the method returns a String, it will return return String in all cases. If the status isn't available, it will return an empty *frozen* String.

+ It doesn't have implementation of any Error that gets raised in runtime for the ease of use.

+ If you need to check some stat that returns an integer or float, and you get nil, you know it's not available, so you can work accordingly. But if you need the integer or float value in 0 to whatever format, you can use the .to_i or .to_f method on the object, nil will get converted to number then.

If some error is *raised* it should be reported as a bug.

---

## Ruby on Rails

1. Just add `gem linux_stat`:

```
$ bundle add linux_stat
```

You can use LinuxStat directly in rails.

![RailsApp](https://raw.githubusercontent.com/Souravgoswami/linux_stat/master/images/rails.gif)

---

## Android

LinuxStat does support Android OS. But it's not rigorously tested on all device like android apps.

But in Termux you can just run LinuxStat without facing issues.
Note that the CPU count can differ due to hotplugging feature. So if you see the CPU count changes, there's not really nothing to do about that.

![termux](https://raw.githubusercontent.com/Souravgoswami/linux_stat/master/images/termux.webp)

Issues regarding running LinuxStat on termux are also welcomed.

---

## Cloning this Repo
Users of this gem are requested to follow the above installation step to install this gem.

This repo is only for development purpose. It has C extensions that could do worst to your stable app.
Cloning, compiling code from this repo for a production app may seg fault and crash the whole app directly.

Gems on Rubygems are released after various tests.

So just install the gem, don't clone this repo just because the version is bumped here with new features, and the gem is not out on Rubygems.org.

## Development
After checking out the repo, compile and install this gem onto your local machine with `bundle exec rake install`

You can also run `bin/console` for an interactive prompt that will allow you to experiment.

To test all modules, run `rake install` and then `exe/linuxstat.rb`. Also check "Testing" below.

---

## Testing
Like other gems, this doesn't have a test like RSpec.

We suggest using the exe/linuxstat.rb file on various Linux systems to test.

First you need to execute `bundle exec rake install` to compile and install this gem.

If you need to test a specific module, say the CPU, just run it like this:

```
$ ruby exe/linuxstat.rb CPU
```

Or:
```
$ ruby exe/linuxstat.rb cpu
```

That is, the argument passed is not case-sensitive.
But if the argument passed isn't available and outright wrong, it will run all the module methods. For example, you can't do:

```
$ ruby exe/linuxstat.rb upc
```
This is not a valid module and can't be run.

---

## Contributing

Bug reports and pull requests are welcome on GitHub at https://github.com/Souravgoswami/linux_stat.

---

## License

The gem is available as open source under the terms of the [MIT License](https://opensource.org/licenses/MIT).

## Languages

<img src="https://linuxstatloc.herokuapp.com/svg" width="260px">
