/**
 * Builds a URL by combining various URL components together
 * @param components The list of components to join with a single '/' between them
 */
export function buildUrl(...components: string[]) {
    return components.map(part => (part || "").replace(/\/?(.*[^/])\/?/, "$1")).join("/");
}

/**
 * Builds a query string for a given object
 * @param q The map describing the values for each query parameter
 */
export function buildQuery(q?: { [key: string]: string; }) {
    if (!q) return "";

    return "?" + Object.keys(q).map(k => `${encodeURIComponent(k)}=${encodeURIComponent(q[k])}`).join("&")
}

/**
 * A generic handler which converts an API response into a useful
 * object if possible
 * @param res The response generated by a fetch method call
 */
export function apiHandleResponse<T>(res: Response): Promise<T | { text: string }>
/**
 * A generic handler which converts an API response into a useful
 * object if possible
 * @param res The response generated by a fetch method call
 * @param strict Whether an error should be thrown if the response is not a JSON object
 */
export function apiHandleResponse<T>(res: Response, strict: true): Promise<T>
/**
 * A generic handler which converts an API response into a useful
 * object if possible
 * @param res The response generated by a fetch method call
 * @param strict Whether an error should be thrown if the response is not a JSON object
 */
export function apiHandleResponse<T>(res: Response, strict: boolean = false): Promise<T | { text: string }> {
    if (res.status !== 200)
        return res.json().then(data => {
            const err = new Error(data.message || "Failed to make a request to the API.")
            err["code"] = data.code || 500
            err.name = data.error || "Server Error"

            return Promise.reject(err)
        })

    if (~res.headers.get("content-type").indexOf("application/json"))
        return res.json()

    if (strict) return Promise.reject(new Error(`Received a non-JSON response`))

    console.error("Received a non-JSON response")
    return res.text().then(text => ({
        text: text
    }))
}