<?php

/**
 * FileMimeTypeTest.
 *
 * PHP Version 7.2.11-3
 *
 * @package   Verum-PHP
 * @license   MIT https://github.com/SandroMiguel/verum-php/blob/master/LICENSE
 * @author    Sandro Miguel Marques <sandromiguel@sandromiguel.com>
 * @copyright 2020 Sandro
 * @since     Verum-PHP 1.0.0
 * @version   1.1.3 (2020/06/25)
 * @link      https://github.com/SandroMiguel/verum-php
 */

declare(strict_types=1);

namespace Verum\Tests;

use PHPUnit\Framework\TestCase;
use Verum\Rules\RuleFactory;
use Verum\Validator;
use Verum\ValidatorException;

/**
 * Class FileMimeTypeTest | tests/FileMimeTypeTest.php | Test for FileMimeType
 */
class FileMimeTypeTest extends TestCase
{
    /**
     * Validates the field value against the rule.
     *
     * @param mixed $fieldValue Field Value to validate.
     * @param array $ruleValues Rule values.
     *
     * @return bool Returns TRUE if it passes the validation, FALSE otherwise.
     */
    private function validate($fieldValue, array $ruleValues): bool
    {
        $fieldName = 'some_field_name';
        $fieldLabel = 'Some Field Name';
        $ruleName = 'file_mime_type';
        $validator = new Validator(
            [
                $fieldName => $fieldValue,
            ],
            [
                $fieldName => [
                    'label' => $fieldLabel,
                    'rules' => [$ruleName => $ruleValues],
                ],
            ]
        );
        $rule = RuleFactory::loadRule($validator, $fieldValue, $ruleValues, $fieldLabel, $ruleName, '');

        return $rule->validate();
    }

    /**
     * If there is no uploaded file, validation must pass.
     *
     * @return void
     */
    public function testValidateNoFile(): void
    {
        $this->assertTrue($this->validate(null, ['image/png']));
    }

    /**
     * If the Rule Values are not defined, an exception should be thrown.
     *
     * @return void
     */
    public function testValidateWithoutRuleValues(): void
    {
        $this->expectException(ValidatorException::class);
        $this->expectExceptionMessage(
            'Invalid argument; Argument name: $ruleValues; Argument value: null; Rule "file_mime_type": the rule value is mandatory'
        );
        $this->validate(['size' => '102400'], []);
    }

    /**
     * The String ('text/plain') value should not pass validation.
     *
     * @return void
     */
    public function testValidateInvalidType(): void
    {
        $this->assertFalse($this->validate(['type' => 'text/plain'], ['image/png', 'image/jpeg']));
    }

    /**
     * The String ('image/png') value should pass validation.
     *
     * @return void
     */
    public function testValidateValidType(): void
    {
        $this->assertTrue($this->validate(['type' => 'image/png'], ['image/png', 'image/jpeg']));
    }
}
