[//]: # (DO NOT EDIT THIS FILE DIRECTLY. Instead, edit the corresponding stub file and execute `npm run docs:api`.)

# <code class="doc-symbol doc-symbol-class"></code> `Table` {#safeds.data.tabular.containers.Table data-toc-label='[class] Table'}

A two-dimensional collection of data. It can either be seen as a list of rows or as a list of columns.

To create a `Table` call the constructor or use one of the following static methods:

| Method                        | Description                            |
| ----------------------------- | -------------------------------------- |
| [Table.fromCsvFile][safeds.data.tabular.containers.Table.fromCsvFile]     | Create a table from a CSV file.        |
| [Table.fromJsonFile][safeds.data.tabular.containers.Table.fromJsonFile]    | Create a table from a JSON file.       |
| [Table.fromParquetFile][safeds.data.tabular.containers.Table.fromParquetFile] | Create a table from a Parquet file.    |
| [Table.fromColumns][safeds.data.tabular.containers.Table.fromColumns]     | Create a table from a list of columns. |
| [Table.fromMap][safeds.data.tabular.containers.Table.fromMap]         | Create a table from a map.             |

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `data` | [`Map<String, List<Any?>>?`][safeds.lang.Map] | The data of the table. If null, an empty table is created. | `#!sds null` |

**Examples:**

```sds hl_lines="2"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="33"
    class Table(
        data: Map<String, List<Any?>>? = null
    ) {
        /**
         * The names of the columns in the table.
         */
        @PythonName("column_names") attr columnNames: List<String>
        /**
         * The number of columns in the table.
         */
        @PythonName("column_count") attr columnCount: Int
        /**
         * The number of rows in the table.
         *
         * **Note:** This operation must fully load the data into memory, which can be expensive.
         */
        @PythonName("row_count") attr rowCount: Int
        /**
         * The plotter for the table.
         */
        attr plot: TablePlotter
        /**
         * The schema of the table.
         */
        attr schema: Schema

        /**
         * Create a table from a list of columns.
         *
         * @param columns The columns.
         *
         * @result table The created table.
         *
         * @example
         * pipeline example {
         *     val a = Column("a", [1, 2, 3]);
         *     val b = Column("b", [4, 5, 6]);
         *     val result = Table.fromColumns([a, b]);
         * }
         */
        @Pure
        @PythonName("from_columns")
        @Category(DataScienceCategory.UtilitiesQConversion)
        static fun fromColumns(
            columns: union<Column, List<Column>>
        ) -> table: Table

        /**
         * Create a table from a CSV file.
         *
         * @param path The path to the CSV file. If the file extension is omitted, it is assumed to be ".csv".
         * @param separator The separator between the values in the CSV file.
         *
         * @result table The created table.
         *
         * @example
         * pipeline example {
         *     val result = Table.fromCsvFile("input.csv");
         * }
         */
        @Impure([ImpurityReason.FileReadFromParameterizedPath("path")])
        @PythonName("from_csv_file")
        @Category(DataScienceCategory.DataImport)
        static fun fromCsvFile(
            path: String,
            separator: String = ","
        ) -> table: Table

        /**
         * Create a table from a map that maps column names to column values.
         *
         * @param data The data.
         *
         * @result table The generated table.
         *
         * @example
         * pipeline example {
         *     val data = {"a'": [1, 2, 3], "b": [4, 5, 6]};
         *     val result = Table.fromMap(data);
         * }
         */
        @Pure
        @PythonName("from_dict")
        @Category(DataScienceCategory.DataImport)
        static fun fromMap(
            data: Map<String, List<Any>>
        ) -> table: Table

        /**
         * Create a table from a JSON file.
         *
         * @param path The path to the JSON file. If the file extension is omitted, it is assumed to be ".json".
         *
         * @result table The created table.
         *
         * @example
         * pipeline example {
         *     val result = Table.fromJsonFile("input.json");
         * }
         */
        @Impure([ImpurityReason.FileReadFromParameterizedPath("path")])
        @PythonName("from_json_file")
        @Category(DataScienceCategory.DataImport)
        static fun fromJsonFile(
            path: String
        ) -> table: Table

        /**
         * Create a table from a Parquet file.
         *
         * @param path The path to the Parquet file. If the file extension is omitted, it is assumed to be ".parquet".
         *
         * @result table The created table.
         *
         * @example
         * pipeline example {
         *     val result = Table.fromParquetFile("input.parquet");
         * }
         */
        @Impure([ImpurityReason.FileReadFromParameterizedPath("path")])
        @PythonName("from_parquet_file")
        @Category(DataScienceCategory.DataImport)
        static fun fromParquetFile(
            path: String
        ) -> table: Table

        /**
         * Return a new table with additional columns.
         *
         * **Notes:**
         *
         * - The original table is not modified.
         * - This operation must fully load the data into memory, which can be expensive.
         *
         * @param columns The columns to add.
         *
         * @result newTable The table with the additional columns.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3]});
         *     val newColumn = Column("b", [4, 5, 6]);
         *     val result = table.addColumns(newColumn);
         * }
         */
        @Pure
        @PythonName("add_columns")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun addColumns(
            columns: union<Column, List<Column>>
        ) -> newTable: Table

        /**
         * Return a new table with an additional computed column.
         *
         * **Note:** The original table is not modified.
         *
         * @param name The name of the new column.
         * @param computer The function that computes the values of the new column.
         *
         * @result newTable The table with the computed column.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.addComputedColumn("c", (row) -> row["a"] + row["b"]);
         * }
         */
        @Pure
        @PythonName("add_computed_column")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun addComputedColumn(
            name: String,
            computer: (row: Row) -> computedCell: Cell
        ) -> newTable: Table

        /**
         * Get a column from the table.
         *
         * @param name The name of the column.
         *
         * @result column The column.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.getColumn("a");
         *     // Column("a", [1, 2, 3])
         * }
         */
        @Pure
        @PythonName("get_column")
        @Category(DataScienceCategory.UtilitiesQTable)
        fun getColumn(
            name: String
        ) -> column: Column

        /**
         * Get the data type of a column.
         *
         * @param name The name of the column.
         *
         * @result type The data type of the column.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.getColumnType("a");
         * }
         */
        @Pure
        @PythonName("get_column_type")
        @Category(DataScienceCategory.UtilitiesQTable)
        fun getColumnType(
            name: String
        ) -> type: DataType

        /**
         * Check if the table has a column with a specific name.
         *
         * @param name The name of the column.
         *
         * @result hasColumn Whether the table has a column with the specified name.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.hasColumn("a"); // true
         * }
         */
        @Pure
        @PythonName("has_column")
        @Category(DataScienceCategory.UtilitiesQTable)
        fun hasColumn(
            name: String
        ) -> hasColumn: Boolean

        /**
         * Return a new table without the specified columns.
         *
         * **Note:** The original table is not modified.
         *
         * @param names The names of the columns to remove.
         * @param ignoreUnknownNames If set to True, columns that are not present in the table will be ignored. If set to
         * False, an error will be raised if any of the specified columns do not exist.
         *
         * @result newTable The table with the columns removed.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.removeColumns("a");
         *     // Table({"b": [4, 5, 6]})
         * }
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.removeColumns("c");
         *     // Table({"a": [1, 2, 3], "b": [4, 5, 6]})
         * }
         */
        @Pure
        @PythonName("remove_columns")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeColumns(
            names: union<List<String>, String>,
            @PythonName("ignore_unknown_names") ignoreUnknownNames: Boolean = false
        ) -> newTable: Table

        /**
         * Return a new table with only the specified columns.
         *
         * @param names The names of the columns to keep.
         *
         * @result newTable The table with only the specified columns.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.removeColumnsExcept("a");
         *     // Table({"a": [1, 2, 3]})
         * }
         */
        @Pure
        @PythonName("remove_columns_except")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeColumnsExcept(
            names: union<List<String>, String>
        ) -> newTable: Table

        /**
         * Return a new table without columns that contain missing values.
         *
         * **Notes:**
         *
         * - The original table is not modified.
         * - This operation must fully load the data into memory, which can be expensive.
         *
         * @result newTable The table without columns containing missing values.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, null]});
         *     val result = table.removeColumnsWithMissingValues();
         *     // Table({"a": [1, 2, 3]})
         * }
         */
        @Pure
        @PythonName("remove_columns_with_missing_values")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeColumnsWithMissingValues() -> newTable: Table

        /**
         * Return a new table without non-numeric columns.
         *
         * **Note:** The original table is not modified.
         *
         * @result newTable The table without non-numeric columns.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": ["4", "5", "6"]});
         *     val result = table.removeNonNumericColumns();
         *     // Table({"a": [1, 2, 3]})
         * }
         */
        @Pure
        @PythonName("remove_non_numeric_columns")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeNonNumericColumns() -> newTable: Table

        /**
         * Return a new table with a column renamed.
         *
         * **Note:** The original table is not modified.
         *
         * @param oldName The name of the column to rename.
         * @param newName The new name of the column.
         *
         * @result newTable The table with the column renamed.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.renameColumn("a", "c");
         *     // Table({"c": [1, 2, 3], "b": [4, 5, 6]})
         * }
         */
        @Pure
        @PythonName("rename_column")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun renameColumn(
            @PythonName("old_name") oldName: String,
            @PythonName("new_name") newName: String
        ) -> newTable: Table

        /**
         * Return a new table with a column replaced by zero or more columns.
         *
         * **Note:** The original table is not modified.
         *
         * @param oldName The name of the column to replace.
         * @param newColumns The new columns.
         *
         * @result newTable The table with the column replaced.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.replaceColumn("a", []);
         *     // Table({"b": [4, 5, 6]})
         * }
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val column1 = Column("c", [7, 8, 9]);
         *     val result = table.replaceColumn("a", column1);
         *     // Table({
         *     //     "c": [7, 8, 9],
         *     //     "b": [4, 5, 6]
         *     // })
         * }
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val column1 = Column("c", [7, 8, 9]);
         *     val column2 = Column("d", [10, 11, 12]);
         *     val result = table.replaceColumn("a", [column1, column2]);
         *     // Table({
         *     //     "c": [7, 8, 9],
         *     //     "d": [10, 11, 12],
         *     //     "b": [4, 5, 6]
         *     // })
         * }
         */
        @Pure
        @PythonName("replace_column")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun replaceColumn(
            @PythonName("old_name") oldName: String,
            @PythonName("new_columns") newColumns: union<Column, List<Column>, Table>
        ) -> newTable: Table

        /**
         * Return a new table with a column transformed.
         *
         * **Note:** The original table is not modified.
         *
         * @param name The name of the column to transform.
         * @param transformer The function that transforms the column.
         *
         * @result newTable The table with the transformed column.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.transformColumn("a", (cell) -> cell + 1);
         *     // Table({"a": [2, 3, 4], "b": [4, 5, 6]})
         * }
         */
        @Pure
        @PythonName("transform_column")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun transformColumn(
            name: String,
            transformer: (cell: Cell) -> transformedCell: Cell
        ) -> newTable: Table

        /**
         * Return how many rows in the table satisfy the predicate.
         *
         * The predicate can return one of three results:
         *
         * - true, if the row satisfies the predicate.
         * - false, if the row does not satisfy the predicate.
         * - null, if the truthiness of the predicate is unknown, e.g. due to missing values.
         *
         * By default, cases where the truthiness of the predicate is unknown are ignored and this method returns how often
         * the predicate returns true.
         *
         * You can instead enable Kleene logic by setting `ignore_unknown = False`. In this case, this method returns null
         * if the predicate returns null at least once. Otherwise, it still returns how often the predicate returns true.
         *
         * @param predicate The predicate to apply to each row.
         * @param ignoreUnknown Whether to ignore cases where the truthiness of the predicate is unknown.
         *
         * @result count The number of rows in the table that satisfy the predicate.
         *
         * @example
         * pipeline example {
         *     val table = Table({"col1": [1, 2, 3], "col2": [1, 3, 3]});
         *     val result = table.countRowIf((row) -> row["col1"] == row["col2"]); // 2
         * }
         *
         * @example
         * pipeline example {
         *     val table = Table({"col1": [1, 2, 3], "col2": [1, 3, 3]});
         *     val result = table.countRowIf((row) -> row["col1"] > row["col2"]); // 0
         * }
         */
        @Pure
        @PythonName("count_row_if")
        @Category(DataScienceCategory.DataExplorationQGeneral)
        fun countRowIf(
            predicate: (cell: Row) -> satisfiesPredicate: Cell<Boolean?>,
            @PythonName("ignore_unknown") ignoreUnknown: Boolean = true,
        ) -> count: Int?

        /**
         * Return a new table without duplicate rows.
         *
         * **Note:** The original table is not modified.
         *
         * @result newTable The table without duplicate rows.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 2], "b": [4, 5, 5]});
         *     val result = table.removeDuplicateRows();
         *     // Table({"a": [1, 2], "b": [4, 5]})
         * }
         */
        @Pure
        @PythonName("remove_duplicate_rows")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeDuplicateRows() -> newTable: Table

        /**
         * Return a new table without rows that satisfy a condition.
         *
         * **Note:** The original table is not modified.
         *
         * @param query The function that determines which rows to remove.
         *
         * @result newTable The table without the specified rows.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.removeRows((row) -> row["a"] == 2);
         *     // Table({"a": [1, 3], "b": [4, 6]})
         * }
         */
        @Pure
        @PythonName("remove_rows")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeRows(
            query: (row: Row) -> shouldRemoveRow: Cell<Boolean>
        ) -> newTable: Table

        /**
         * Return a new table without rows that satisfy a condition on a specific column.
         *
         * **Note:** The original table is not modified.
         *
         * @param name The name of the column.
         * @param query The function that determines which rows to remove.
         *
         * @result newTable The table without the specified rows.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.removeRowsByColumn("a", (cell) -> cell == 2);
         *     // Table({"a": [1, 3], "b": [4, 6]})
         * }
         */
        @Pure
        @PythonName("remove_rows_by_column")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeRowsByColumn(
            name: String,
            query: (cell: Cell) -> shouldRemoveRow: Cell<Boolean>
        ) -> newTable: Table

        /**
         * Return a new table without rows containing missing values in the specified columns.
         *
         * **Note:** The original table is not modified.
         *
         * @param columnNames Names of the columns to consider. If null, all columns are considered.
         *
         * @result newTable The table without rows containing missing values in the specified columns.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, null, 3], "b": [4, 5, null]});
         *     val result = table.removeRowsWithMissingValues();
         *     // Table({"a": [1], "b": [4]})
         * }
         */
        @Pure
        @PythonName("remove_rows_with_missing_values")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeRowsWithMissingValues(
            @PythonName("column_names") columnNames: List<String>? = null
        ) -> newTable: Table

        /**
         * Return a new table without rows containing outliers in the specified columns.
         *
         * Whether a data point is an outlier in a column is determined by its z-score. The z-score the distance of the
         * data point from the mean of the column divided by the standard deviation of the column. If the z-score is
         * greater than the given threshold, the data point is considered an outlier. Missing values are ignored during the
         * calculation of the z-score.
         *
         * The z-score is only defined for numeric columns. Non-numeric columns are ignored, even if they are specified in
         * `column_names`.
         *
         * **Notes:**
         *
         * - The original table is not modified.
         * - This operation must fully load the data into memory, which can be expensive.
         *
         * @param columnNames Names of the columns to consider. If null, all numeric columns are considered.
         * @param zScoreThreshold The z-score threshold for detecting outliers.
         *
         * @result newTable The table without rows containing outliers in the specified columns.
         *
         * @example
         * pipeline example {
         *     val table = Table(
         *         {
         *             "a": [1, 2, 3, 4, 5, 6, 1000, null],
         *             "b": [1, 2, 3, 4, 5, 6,    7,    8],
         *         }
         *     );
         *     val result = table.removeRowsWithOutliers(zScoreThreshold=2.0);
         *     // Table({"a": [1, 2, 3, 4, 5, 6, null], "b": [1, 2, 3, 4, 5, 6]})
         * }
         */
        @Pure
        @PythonName("remove_rows_with_outliers")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun removeRowsWithOutliers(
            @PythonName("column_names") columnNames: List<String>? = null,
            @PythonName("z_score_threshold") zScoreThreshold: Float = 3.0
        ) -> newTable: Table

        /**
         * Return a new table with the rows shuffled.
         *
         * **Note:** The original table is not modified.
         *
         * @result newTable The table with the rows shuffled.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.shuffleRows();
         *     // Table({"a": [3, 2, 1], "b": [6, 5, 4]})
         * }
         */
        @Pure
        @PythonName("shuffle_rows")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun shuffleRows() -> newTable: Table

        /**
         * Return a new table with a slice of rows.
         *
         * **Note:** The original table is not modified.
         *
         * @param start The start index of the slice.
         * @param length The length of the slice. If null, the slice contains all rows starting from `start`. Must greater than or
         * equal to 0.
         *
         * @result newTable The table with the slice of rows.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.sliceRows(start=1);
         *     // Table({"a": [2, 3], "b": [5, 6]})
         * }
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.sliceRows(start=1, length=1);
         *     // Table({"a": [2], "b": [5]})
         * }
         */
        @Pure
        @PythonName("slice_rows")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun sliceRows(
            start: Int = 0,
            length: Int? = null
        ) -> newTable: Table

        /**
         * Return a new table with the rows sorted.
         *
         * **Note:** The original table is not modified.
         *
         * @param keySelector The function that selects the key to sort by.
         * @param descending Whether to sort in descending order.
         *
         * @result newTable The table with the rows sorted.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [2, 1, 3], "b": [1, 1, 2]});
         *     val result = table.sortRows((row) -> row["a"] - row["b"]);
         *     // Table({"a": [1, 2, 3], "b": [1, 1, 2]})
         * }
         */
        @Pure
        @PythonName("sort_rows")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun sortRows(
            @PythonName("key_selector") keySelector: (row: Row) -> key: Cell,
            descending: Boolean = false
        ) -> newTable: Table

        /**
         * Return a new table with the rows sorted by a specific column.
         *
         * **Note:** The original table is not modified.
         *
         * @param name The name of the column to sort by.
         * @param descending Whether to sort in descending order.
         *
         * @result newTable The table with the rows sorted by the specified column.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [2, 1, 3], "b": [1, 1, 2]});
         *     val result = table.sortRowsByColumn("a");
         *     // Table({"a": [1, 2, 3], "b": [1, 1, 2]})
         * }
         */
        @Pure
        @PythonName("sort_rows_by_column")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun sortRowsByColumn(
            name: String,
            descending: Boolean = false
        ) -> newTable: Table

        /**
         * Create two tables by splitting the rows of the current table.
         *
         * The first table contains a percentage of the rows specified by `percentage_in_first`, and the second table
         * contains the remaining rows.
         *
         * **Notes:**
         *
         * - The original table is not modified.
         * - By default, the rows are shuffled before splitting. You can disable this by setting `shuffle` to false.
         *
         * @param percentageInFirst The percentage of rows to include in the first table. Must be between 0 and 1.
         * @param shuffle Whether to shuffle the rows before splitting.
         *
         * @result firstTable The first table.
         * @result secondTable The second table.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3, 4, 5], "b": [6, 7, 8, 9, 10]});
         *     val firstTable, val secondTable = table.splitRows(0.6);
         * }
         */
        @Pure
        @PythonName("split_rows")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun splitRows(
            @PythonName("percentage_in_first") percentageInFirst: Float,
            shuffle: Boolean = true
        ) -> (firstTable: Table, secondTable: Table)

        /**
         * Return a new table with the columns of another table added.
         *
         * **Notes:**
         *
         * - The original tables are not modified.
         * - This operation must fully load the data into memory, which can be expensive.
         *
         * @param other The table to add as columns.
         *
         * @result newTable The table with the columns added.
         *
         * @example
         * pipeline example {
         *     val table1 = Table({"a": [1, 2, 3]});
         *     val table2 = Table({"b": [4, 5, 6]});
         *     val result = table1.addTableAsColumns(table2);
         *     // Table({"a": [1, 2, 3], "b": [4, 5, 6]})
         * }
         */
        @Pure
        @PythonName("add_table_as_columns")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun addTableAsColumns(
            other: Table
        ) -> newTable: Table

        /**
         * Return a new table with the rows of another table added.
         *
         * **Notes:**
         *
         * - The original tables are not modified.
         * - This operation must fully load the data into memory, which can be expensive.
         *
         * @param other The table to add as rows.
         *
         * @result newTable The table with the rows added.
         *
         * @example
         * pipeline example {
         *     val table1 = Table({"a": [1, 2, 3]});
         *     val table2 = Table({"a": [4, 5, 6]});
         *     val result = table1.addTableAsRows(table2);
         *     // Table({"a": [1, 2, 3, 4, 5, 6]})
         * }
         */
        @Pure
        @PythonName("add_table_as_rows")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun addTableAsRows(
            other: Table
        ) -> newTable: Table

        /**
         * Return a new table inverse-transformed by a **fitted, invertible** transformer.
         *
         * **Notes:**
         *
         * - The original table is not modified.
         * - Depending on the transformer, this operation might fully load the data into memory, which can be expensive.
         *
         * @param fittedTransformer The fitted, invertible transformer to apply.
         *
         * @result newTable The inverse-transformed table.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3]});
         *     val transformer, val transformedTable = RangeScaler(min=0.0, max=1.0, columnNames="a").fitAndTransform(table);
         *     val result = transformedTable.inverseTransformTable(transformer);
         *     // Table({"a": [1, 2, 3]})
         * }
         */
        @Pure
        @PythonName("inverse_transform_table")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun inverseTransformTable(
            @PythonName("fitted_transformer") fittedTransformer: InvertibleTableTransformer
        ) -> newTable: Table

        /**
         * Join a table with the current table and return the result.
         *
         * @param rightTable The other table which is to be joined to the current table.
         * @param leftNames Name or list of names of columns from the current table on which to join right_table.
         * @param rightNames Name or list of names of columns from right_table on which to join the current table.
         * @param mode Specify which type of join you want to use. Options include 'inner', 'outer', 'left', 'right'.
         *
         * @result newTable The table with the joined table.
         *
         * @example
         * pipeline example {
         *     // from safeds.data.tabular.containers import Table
         *     // table1 = Table({"a": [1, 2], "b": [3, 4]})
         *     // table2 = Table({"d": [1, 5], "e": [5, 6]})
         *     // table1.join(table2, "a", "d", mode="left")
         * }
         */
        @Pure
        @Category(DataScienceCategory.DataProcessingQTable)
        fun join(
            @PythonName("right_table") rightTable: Table,
            @PythonName("left_names") leftNames: union<List<String>, String>,
            @PythonName("right_names") rightNames: union<List<String>, String>,
            mode: literal<"inner", "left", "outer"> = "inner"
        ) -> newTable: Table

        /**
         * Return a new table transformed by a **fitted** transformer.
         *
         * **Notes:**
         *
         * - The original table is not modified.
         * - Depending on the transformer, this operation might fully load the data into memory, which can be expensive.
         *
         * @param fittedTransformer The fitted transformer to apply.
         *
         * @result newTable The transformed table.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3]});
         *     val transformer = RangeScaler(min=0.0, max=1.0, columnNames="a").fit(table);
         *     val result = table.transformTable(transformer);
         *     // Table({"a": [0, 0.5, 1]})
         * }
         */
        @Pure
        @PythonName("transform_table")
        @Category(DataScienceCategory.DataProcessingQTable)
        fun transformTable(
            @PythonName("fitted_transformer") fittedTransformer: TableTransformer
        ) -> newTable: Table

        /**
         * Return a table with important statistics about this table.
         *
         * @result statistics The table with statistics.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 3]});
         *     val result = table.summarizeStatistics();
         * }
         */
        @Pure
        @PythonName("summarize_statistics")
        @Category(DataScienceCategory.DataExplorationQMetric)
        fun summarizeStatistics() -> statistics: Table

        /**
         * Return the data of the table as a list of columns.
         *
         * @result columns List of columns.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val columns = table.toColumns();
         * }
         */
        @Pure
        @PythonName("to_columns")
        @Category(DataScienceCategory.UtilitiesQConversion)
        fun toColumns() -> columns: List<Column>

        /**
         * Write the table to a CSV file.
         *
         * If the file and/or the parent directories do not exist, they will be created. If the file exists already, it
         * will be overwritten.
         *
         * @param path The path to the CSV file. If the file extension is omitted, it is assumed to be ".csv".
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     table.toCsvFile("output.csv");
         * }
         */
        @Impure([ImpurityReason.FileWriteToParameterizedPath("path")])
        @PythonName("to_csv_file")
        @Category(DataScienceCategory.DataExport)
        fun toCsvFile(
            path: String
        )

        /**
         * Return a map that maps column names to column values.
         *
         * @result map Map representation of the table.
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     val result = table.toMap();
         *     // {"a": [1, 2, 3], "b": [4, 5, 6]}
         * }
         */
        @Pure
        @PythonName("to_dict")
        @Category(DataScienceCategory.UtilitiesQConversion)
        fun toMap() -> map: Map<String, List<Any>>

        /**
         * Write the table to a JSON file.
         *
         * If the file and/or the parent directories do not exist, they will be created. If the file exists already, it
         * will be overwritten.
         *
         * **Note:** This operation must fully load the data into memory, which can be expensive.
         *
         * @param path The path to the JSON file. If the file extension is omitted, it is assumed to be ".json".
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     table.toJsonFile("output.json");
         * }
         */
        @Impure([ImpurityReason.FileWriteToParameterizedPath("path")])
        @PythonName("to_json_file")
        @Category(DataScienceCategory.DataExport)
        fun toJsonFile(
            path: String
        )

        /**
         * Write the table to a Parquet file.
         *
         * If the file and/or the parent directories do not exist, they will be created. If the file exists already, it
         * will be overwritten.
         *
         * @param path The path to the Parquet file. If the file extension is omitted, it is assumed to be ".parquet".
         *
         * @example
         * pipeline example {
         *     val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
         *     table.toParquetFile("output.parquet");
         * }
         */
        @Impure([ImpurityReason.FileWriteToParameterizedPath("path")])
        @PythonName("to_parquet_file")
        @Category(DataScienceCategory.DataExport)
        fun toParquetFile(
            path: String
        )

        /**
         * Return a new `TabularDataset` with columns marked as a target, feature, or extra.
         *
         * - The target column is the column that a model should predict.
         * - Feature columns are columns that a model should use to make predictions.
         * - Extra columns are columns that are neither feature nor target. They can be used to provide additional context,
         *   like an ID column.
         *
         * Feature columns are implicitly defined as all columns except the target and extra columns. If no extra columns
         * are specified, all columns except the target column are used as features.
         *
         * @param targetName The name of the target column.
         * @param extraNames Names of the columns that are neither feature nor target. If null, no extra columns are used, i.e. all but
         * the target column are used as features.
         *
         * @result dataset A new tabular dataset with the given target and feature names.
         *
         * @example
         * pipeline example {
         *     val table = Table(
         *         {
         *             "item": ["apple", "milk", "beer"],
         *             "price": [1.10, 1.19, 1.79],
         *             "amount_bought": [74, 72, 51],
         *         }
         *     );
         *     val dataset = table.toTabularDataset(targetName="amount_bought", extraNames=["item"]);
         * }
         */
        @Pure
        @PythonName("to_tabular_dataset")
        @Category(DataScienceCategory.UtilitiesQConversion)
        fun toTabularDataset(
            @PythonName("target_name") targetName: String,
            @PythonName("extra_names") extraNames: List<String>? = null
        ) -> dataset: TabularDataset

        /**
         * Return a new `TimeSeriesDataset` with columns marked as a target column, time or feature columns.
         *
         * The original table is not modified.
         *
         * @param targetName The name of the target column.
         * @param windowSize The number of consecutive sample to use as input for prediction.
         * @param extraNames Names of the columns that are neither features nor target. If None, no extra columns are used, i.e. all but
         * the target column are used as features.
         * @param forecastHorizon The number of time steps to predict into the future.
         * @param continuous Whether or not to continue the forecast in the steps before forecast horizon.
         *
         * @result dataset A new time series dataset with the given target and feature names.
         *
         * @example
         * pipeline example {
         *     val table = Table(
         *         {
         *             "day": [0, 1, 2],
         *             "price": [1.10, 1.19, 1.79],
         *             "amount_bought": [74, 72, 51],
         *         }
         *     );
         *     val dataset = table.toTimeSeriesDataset(targetName="amount_bought", windowSize=2);
         * }
         */
        @Pure
        @PythonName("to_time_series_dataset")
        @Category(DataScienceCategory.UtilitiesQConversion)
        fun toTimeSeriesDataset(
            @PythonName("target_name") targetName: String,
            @PythonName("window_size") windowSize: Int,
            @PythonName("extra_names") extraNames: List<String>? = null,
            @PythonName("forecast_horizon") forecastHorizon: Int = 1,
            continuous: Boolean = false
        ) -> dataset: TimeSeriesDataset
    }
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-attribute"></code> `columnCount` {#safeds.data.tabular.containers.Table.columnCount data-toc-label='[attribute] columnCount'}

The number of columns in the table.

**Type:** [`Int`][safeds.lang.Int]

## <code class="doc-symbol doc-symbol-attribute"></code> `columnNames` {#safeds.data.tabular.containers.Table.columnNames data-toc-label='[attribute] columnNames'}

The names of the columns in the table.

**Type:** [`List<String>`][safeds.lang.List]

## <code class="doc-symbol doc-symbol-attribute"></code> `plot` {#safeds.data.tabular.containers.Table.plot data-toc-label='[attribute] plot'}

The plotter for the table.

**Type:** [`TablePlotter`][safeds.data.tabular.plotting.TablePlotter]

## <code class="doc-symbol doc-symbol-attribute"></code> `rowCount` {#safeds.data.tabular.containers.Table.rowCount data-toc-label='[attribute] rowCount'}

The number of rows in the table.

**Note:** This operation must fully load the data into memory, which can be expensive.

**Type:** [`Int`][safeds.lang.Int]

## <code class="doc-symbol doc-symbol-attribute"></code> `schema` {#safeds.data.tabular.containers.Table.schema data-toc-label='[attribute] schema'}

The schema of the table.

**Type:** [`Schema`][safeds.data.tabular.typing.Schema]

## <code class="doc-symbol doc-symbol-function"></code> `addColumns` {#safeds.data.tabular.containers.Table.addColumns data-toc-label='[function] addColumns'}

Return a new table with additional columns.

**Notes:**

- The original table is not modified.
- This operation must fully load the data into memory, which can be expensive.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `columns` | `#!sds union<Column<Any?>, List<Column<Any?>>>` | The columns to add. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the additional columns. |

**Examples:**

```sds hl_lines="4"
pipeline example {
    val table = Table({"a": [1, 2, 3]});
    val newColumn = Column("b", [4, 5, 6]);
    val result = table.addColumns(newColumn);
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="178"
    @Pure
    @PythonName("add_columns")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun addColumns(
        columns: union<Column, List<Column>>
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `addComputedColumn` {#safeds.data.tabular.containers.Table.addComputedColumn data-toc-label='[function] addComputedColumn'}

Return a new table with an additional computed column.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `name` | [`String`][safeds.lang.String] | The name of the new column. | - |
| `computer` | `#!sds (row: Row) -> (computedCell: Cell<Any?>)` | The function that computes the values of the new column. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the computed column. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.addComputedColumn("c", (row) -> row["a"] + row["b"]);
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="201"
    @Pure
    @PythonName("add_computed_column")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun addComputedColumn(
        name: String,
        computer: (row: Row) -> computedCell: Cell
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `addTableAsColumns` {#safeds.data.tabular.containers.Table.addTableAsColumns data-toc-label='[function] addTableAsColumns'}

Return a new table with the columns of another table added.

**Notes:**

- The original tables are not modified.
- This operation must fully load the data into memory, which can be expensive.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `other` | [`Table`][safeds.data.tabular.containers.Table] | The table to add as columns. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the columns added. |

**Examples:**

```sds hl_lines="4"
pipeline example {
    val table1 = Table({"a": [1, 2, 3]});
    val table2 = Table({"b": [4, 5, 6]});
    val result = table1.addTableAsColumns(table2);
    // Table({"a": [1, 2, 3], "b": [4, 5, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="788"
    @Pure
    @PythonName("add_table_as_columns")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun addTableAsColumns(
        other: Table
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `addTableAsRows` {#safeds.data.tabular.containers.Table.addTableAsRows data-toc-label='[function] addTableAsRows'}

Return a new table with the rows of another table added.

**Notes:**

- The original tables are not modified.
- This operation must fully load the data into memory, which can be expensive.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `other` | [`Table`][safeds.data.tabular.containers.Table] | The table to add as rows. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the rows added. |

**Examples:**

```sds hl_lines="4"
pipeline example {
    val table1 = Table({"a": [1, 2, 3]});
    val table2 = Table({"a": [4, 5, 6]});
    val result = table1.addTableAsRows(table2);
    // Table({"a": [1, 2, 3, 4, 5, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="815"
    @Pure
    @PythonName("add_table_as_rows")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun addTableAsRows(
        other: Table
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `countRowIf` {#safeds.data.tabular.containers.Table.countRowIf data-toc-label='[function] countRowIf'}

Return how many rows in the table satisfy the predicate.

The predicate can return one of three results:

- true, if the row satisfies the predicate.
- false, if the row does not satisfy the predicate.
- null, if the truthiness of the predicate is unknown, e.g. due to missing values.

By default, cases where the truthiness of the predicate is unknown are ignored and this method returns how often
the predicate returns true.

You can instead enable Kleene logic by setting `ignore_unknown = False`. In this case, this method returns null
if the predicate returns null at least once. Otherwise, it still returns how often the predicate returns true.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `predicate` | `#!sds (cell: Row) -> (satisfiesPredicate: Cell<Boolean?>)` | The predicate to apply to each row. | - |
| `ignoreUnknown` | [`Boolean`][safeds.lang.Boolean] | Whether to ignore cases where the truthiness of the predicate is unknown. | `#!sds true` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `count` | [`Int?`][safeds.lang.Int] | The number of rows in the table that satisfy the predicate. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"col1": [1, 2, 3], "col2": [1, 3, 3]});
    val result = table.countRowIf((row) -> row["col1"] == row["col2"]); // 2
}
```
```sds hl_lines="3"
pipeline example {
    val table = Table({"col1": [1, 2, 3], "col2": [1, 3, 3]});
    val result = table.countRowIf((row) -> row["col1"] > row["col2"]); // 0
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="496"
    @Pure
    @PythonName("count_row_if")
    @Category(DataScienceCategory.DataExplorationQGeneral)
    fun countRowIf(
        predicate: (cell: Row) -> satisfiesPredicate: Cell<Boolean?>,
        @PythonName("ignore_unknown") ignoreUnknown: Boolean = true,
    ) -> count: Int?
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `getColumn` {#safeds.data.tabular.containers.Table.getColumn data-toc-label='[function] getColumn'}

Get a column from the table.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `name` | [`String`][safeds.lang.String] | The name of the column. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `column` | [`Column<Any?>`][safeds.data.tabular.containers.Column] | The column. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.getColumn("a");
    // Column("a", [1, 2, 3])
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="223"
    @Pure
    @PythonName("get_column")
    @Category(DataScienceCategory.UtilitiesQTable)
    fun getColumn(
        name: String
    ) -> column: Column
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `getColumnType` {#safeds.data.tabular.containers.Table.getColumnType data-toc-label='[function] getColumnType'}

Get the data type of a column.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `name` | [`String`][safeds.lang.String] | The name of the column. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `type` | [`DataType`][safeds.data.tabular.typing.DataType] | The data type of the column. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.getColumnType("a");
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="243"
    @Pure
    @PythonName("get_column_type")
    @Category(DataScienceCategory.UtilitiesQTable)
    fun getColumnType(
        name: String
    ) -> type: DataType
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `hasColumn` {#safeds.data.tabular.containers.Table.hasColumn data-toc-label='[function] hasColumn'}

Check if the table has a column with a specific name.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `name` | [`String`][safeds.lang.String] | The name of the column. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `hasColumn` | [`Boolean`][safeds.lang.Boolean] | Whether the table has a column with the specified name. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.hasColumn("a"); // true
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="263"
    @Pure
    @PythonName("has_column")
    @Category(DataScienceCategory.UtilitiesQTable)
    fun hasColumn(
        name: String
    ) -> hasColumn: Boolean
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `inverseTransformTable` {#safeds.data.tabular.containers.Table.inverseTransformTable data-toc-label='[function] inverseTransformTable'}

Return a new table inverse-transformed by a **fitted, invertible** transformer.

**Notes:**

- The original table is not modified.
- Depending on the transformer, this operation might fully load the data into memory, which can be expensive.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `fittedTransformer` | [`InvertibleTableTransformer`][safeds.data.tabular.transformation.InvertibleTableTransformer] | The fitted, invertible transformer to apply. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The inverse-transformed table. |

**Examples:**

```sds hl_lines="4"
pipeline example {
    val table = Table({"a": [1, 2, 3]});
    val transformer, val transformedTable = RangeScaler(min=0.0, max=1.0, columnNames="a").fitAndTransform(table);
    val result = transformedTable.inverseTransformTable(transformer);
    // Table({"a": [1, 2, 3]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="842"
    @Pure
    @PythonName("inverse_transform_table")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun inverseTransformTable(
        @PythonName("fitted_transformer") fittedTransformer: InvertibleTableTransformer
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `join` {#safeds.data.tabular.containers.Table.join data-toc-label='[function] join'}

Join a table with the current table and return the result.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `rightTable` | [`Table`][safeds.data.tabular.containers.Table] | The other table which is to be joined to the current table. | - |
| `leftNames` | `#!sds union<List<String>, String>` | Name or list of names of columns from the current table on which to join right_table. | - |
| `rightNames` | `#!sds union<List<String>, String>` | Name or list of names of columns from right_table on which to join the current table. | - |
| `mode` | `#!sds literal<"inner", "left", "outer">` | Specify which type of join you want to use. Options include 'inner', 'outer', 'left', 'right'. | `#!sds "inner"` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the joined table. |

**Examples:**

```sds hl_lines="5"
pipeline example {
    // from safeds.data.tabular.containers import Table
    // table1 = Table({"a": [1, 2], "b": [3, 4]})
    // table2 = Table({"d": [1, 5], "e": [5, 6]})
    // table1.join(table2, "a", "d", mode="left")
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="867"
    @Pure
    @Category(DataScienceCategory.DataProcessingQTable)
    fun join(
        @PythonName("right_table") rightTable: Table,
        @PythonName("left_names") leftNames: union<List<String>, String>,
        @PythonName("right_names") rightNames: union<List<String>, String>,
        mode: literal<"inner", "left", "outer"> = "inner"
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeColumns` {#safeds.data.tabular.containers.Table.removeColumns data-toc-label='[function] removeColumns'}

Return a new table without the specified columns.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `names` | `#!sds union<List<String>, String>` | The names of the columns to remove. | - |
| `ignoreUnknownNames` | [`Boolean`][safeds.lang.Boolean] | If set to True, columns that are not present in the table will be ignored. If set to False, an error will be raised if any of the specified columns do not exist. | `#!sds false` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the columns removed. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.removeColumns("a");
    // Table({"b": [4, 5, 6]})
}
```
```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.removeColumns("c");
    // Table({"a": [1, 2, 3], "b": [4, 5, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="295"
    @Pure
    @PythonName("remove_columns")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeColumns(
        names: union<List<String>, String>,
        @PythonName("ignore_unknown_names") ignoreUnknownNames: Boolean = false
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeColumnsExcept` {#safeds.data.tabular.containers.Table.removeColumnsExcept data-toc-label='[function] removeColumnsExcept'}

Return a new table with only the specified columns.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `names` | `#!sds union<List<String>, String>` | The names of the columns to keep. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with only the specified columns. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.removeColumnsExcept("a");
    // Table({"a": [1, 2, 3]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="317"
    @Pure
    @PythonName("remove_columns_except")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeColumnsExcept(
        names: union<List<String>, String>
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeColumnsWithMissingValues` {#safeds.data.tabular.containers.Table.removeColumnsWithMissingValues data-toc-label='[function] removeColumnsWithMissingValues'}

Return a new table without columns that contain missing values.

**Notes:**

- The original table is not modified.
- This operation must fully load the data into memory, which can be expensive.

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table without columns containing missing values. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, null]});
    val result = table.removeColumnsWithMissingValues();
    // Table({"a": [1, 2, 3]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="341"
    @Pure
    @PythonName("remove_columns_with_missing_values")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeColumnsWithMissingValues() -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeDuplicateRows` {#safeds.data.tabular.containers.Table.removeDuplicateRows data-toc-label='[function] removeDuplicateRows'}

Return a new table without duplicate rows.

**Note:** The original table is not modified.

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table without duplicate rows. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 2], "b": [4, 5, 5]});
    val result = table.removeDuplicateRows();
    // Table({"a": [1, 2], "b": [4, 5]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="518"
    @Pure
    @PythonName("remove_duplicate_rows")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeDuplicateRows() -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeNonNumericColumns` {#safeds.data.tabular.containers.Table.removeNonNumericColumns data-toc-label='[function] removeNonNumericColumns'}

Return a new table without non-numeric columns.

**Note:** The original table is not modified.

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table without non-numeric columns. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": ["4", "5", "6"]});
    val result = table.removeNonNumericColumns();
    // Table({"a": [1, 2, 3]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="360"
    @Pure
    @PythonName("remove_non_numeric_columns")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeNonNumericColumns() -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeRows` {#safeds.data.tabular.containers.Table.removeRows data-toc-label='[function] removeRows'}

Return a new table without rows that satisfy a condition.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `query` | `#!sds (row: Row) -> (shouldRemoveRow: Cell<Boolean>)` | The function that determines which rows to remove. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table without the specified rows. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.removeRows((row) -> row["a"] == 2);
    // Table({"a": [1, 3], "b": [4, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="539"
    @Pure
    @PythonName("remove_rows")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeRows(
        query: (row: Row) -> shouldRemoveRow: Cell<Boolean>
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeRowsByColumn` {#safeds.data.tabular.containers.Table.removeRowsByColumn data-toc-label='[function] removeRowsByColumn'}

Return a new table without rows that satisfy a condition on a specific column.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `name` | [`String`][safeds.lang.String] | The name of the column. | - |
| `query` | `#!sds (cell: Cell<Any?>) -> (shouldRemoveRow: Cell<Boolean>)` | The function that determines which rows to remove. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table without the specified rows. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.removeRowsByColumn("a", (cell) -> cell == 2);
    // Table({"a": [1, 3], "b": [4, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="563"
    @Pure
    @PythonName("remove_rows_by_column")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeRowsByColumn(
        name: String,
        query: (cell: Cell) -> shouldRemoveRow: Cell<Boolean>
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeRowsWithMissingValues` {#safeds.data.tabular.containers.Table.removeRowsWithMissingValues data-toc-label='[function] removeRowsWithMissingValues'}

Return a new table without rows containing missing values in the specified columns.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `columnNames` | [`List<String>?`][safeds.lang.List] | Names of the columns to consider. If null, all columns are considered. | `#!sds null` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table without rows containing missing values in the specified columns. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, null, 3], "b": [4, 5, null]});
    val result = table.removeRowsWithMissingValues();
    // Table({"a": [1], "b": [4]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="587"
    @Pure
    @PythonName("remove_rows_with_missing_values")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeRowsWithMissingValues(
        @PythonName("column_names") columnNames: List<String>? = null
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `removeRowsWithOutliers` {#safeds.data.tabular.containers.Table.removeRowsWithOutliers data-toc-label='[function] removeRowsWithOutliers'}

Return a new table without rows containing outliers in the specified columns.

Whether a data point is an outlier in a column is determined by its z-score. The z-score the distance of the
data point from the mean of the column divided by the standard deviation of the column. If the z-score is
greater than the given threshold, the data point is considered an outlier. Missing values are ignored during the
calculation of the z-score.

The z-score is only defined for numeric columns. Non-numeric columns are ignored, even if they are specified in
`column_names`.

**Notes:**

- The original table is not modified.
- This operation must fully load the data into memory, which can be expensive.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `columnNames` | [`List<String>?`][safeds.lang.List] | Names of the columns to consider. If null, all numeric columns are considered. | `#!sds null` |
| `zScoreThreshold` | [`Float`][safeds.lang.Float] | The z-score threshold for detecting outliers. | `#!sds 3.0` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table without rows containing outliers in the specified columns. |

**Examples:**

```sds hl_lines="8"
pipeline example {
    val table = Table(
        {
            "a": [1, 2, 3, 4, 5, 6, 1000, null],
            "b": [1, 2, 3, 4, 5, 6,    7,    8],
        }
    );
    val result = table.removeRowsWithOutliers(zScoreThreshold=2.0);
    // Table({"a": [1, 2, 3, 4, 5, 6, null], "b": [1, 2, 3, 4, 5, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="627"
    @Pure
    @PythonName("remove_rows_with_outliers")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun removeRowsWithOutliers(
        @PythonName("column_names") columnNames: List<String>? = null,
        @PythonName("z_score_threshold") zScoreThreshold: Float = 3.0
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `renameColumn` {#safeds.data.tabular.containers.Table.renameColumn data-toc-label='[function] renameColumn'}

Return a new table with a column renamed.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `oldName` | [`String`][safeds.lang.String] | The name of the column to rename. | - |
| `newName` | [`String`][safeds.lang.String] | The new name of the column. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the column renamed. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.renameColumn("a", "c");
    // Table({"c": [1, 2, 3], "b": [4, 5, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="382"
    @Pure
    @PythonName("rename_column")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun renameColumn(
        @PythonName("old_name") oldName: String,
        @PythonName("new_name") newName: String
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `replaceColumn` {#safeds.data.tabular.containers.Table.replaceColumn data-toc-label='[function] replaceColumn'}

Return a new table with a column replaced by zero or more columns.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `oldName` | [`String`][safeds.lang.String] | The name of the column to replace. | - |
| `newColumns` | `#!sds union<Column<Any?>, List<Column<Any?>>, Table>` | The new columns. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the column replaced. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.replaceColumn("a", []);
    // Table({"b": [4, 5, 6]})
}
```
```sds hl_lines="4"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val column1 = Column("c", [7, 8, 9]);
    val result = table.replaceColumn("a", column1);
    // Table({
    //     "c": [7, 8, 9],
    //     "b": [4, 5, 6]
    // })
}
```
```sds hl_lines="5"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val column1 = Column("c", [7, 8, 9]);
    val column2 = Column("d", [10, 11, 12]);
    val result = table.replaceColumn("a", [column1, column2]);
    // Table({
    //     "c": [7, 8, 9],
    //     "d": [10, 11, 12],
    //     "b": [4, 5, 6]
    // })
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="431"
    @Pure
    @PythonName("replace_column")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun replaceColumn(
        @PythonName("old_name") oldName: String,
        @PythonName("new_columns") newColumns: union<Column, List<Column>, Table>
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `shuffleRows` {#safeds.data.tabular.containers.Table.shuffleRows data-toc-label='[function] shuffleRows'}

Return a new table with the rows shuffled.

**Note:** The original table is not modified.

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the rows shuffled. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.shuffleRows();
    // Table({"a": [3, 2, 1], "b": [6, 5, 4]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="649"
    @Pure
    @PythonName("shuffle_rows")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun shuffleRows() -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `sliceRows` {#safeds.data.tabular.containers.Table.sliceRows data-toc-label='[function] sliceRows'}

Return a new table with a slice of rows.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `start` | [`Int`][safeds.lang.Int] | The start index of the slice. | `#!sds 0` |
| `length` | [`Int?`][safeds.lang.Int] | The length of the slice. If null, the slice contains all rows starting from `start`. Must greater than or equal to 0. | `#!sds null` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the slice of rows. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.sliceRows(start=1);
    // Table({"a": [2, 3], "b": [5, 6]})
}
```
```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.sliceRows(start=1, length=1);
    // Table({"a": [2], "b": [5]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="679"
    @Pure
    @PythonName("slice_rows")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun sliceRows(
        start: Int = 0,
        length: Int? = null
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `sortRows` {#safeds.data.tabular.containers.Table.sortRows data-toc-label='[function] sortRows'}

Return a new table with the rows sorted.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `keySelector` | `#!sds (row: Row) -> (key: Cell<Any?>)` | The function that selects the key to sort by. | - |
| `descending` | [`Boolean`][safeds.lang.Boolean] | Whether to sort in descending order. | `#!sds false` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the rows sorted. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [2, 1, 3], "b": [1, 1, 2]});
    val result = table.sortRows((row) -> row["a"] - row["b"]);
    // Table({"a": [1, 2, 3], "b": [1, 1, 2]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="704"
    @Pure
    @PythonName("sort_rows")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun sortRows(
        @PythonName("key_selector") keySelector: (row: Row) -> key: Cell,
        descending: Boolean = false
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `sortRowsByColumn` {#safeds.data.tabular.containers.Table.sortRowsByColumn data-toc-label='[function] sortRowsByColumn'}

Return a new table with the rows sorted by a specific column.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `name` | [`String`][safeds.lang.String] | The name of the column to sort by. | - |
| `descending` | [`Boolean`][safeds.lang.Boolean] | Whether to sort in descending order. | `#!sds false` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the rows sorted by the specified column. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [2, 1, 3], "b": [1, 1, 2]});
    val result = table.sortRowsByColumn("a");
    // Table({"a": [1, 2, 3], "b": [1, 1, 2]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="729"
    @Pure
    @PythonName("sort_rows_by_column")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun sortRowsByColumn(
        name: String,
        descending: Boolean = false
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `splitRows` {#safeds.data.tabular.containers.Table.splitRows data-toc-label='[function] splitRows'}

Create two tables by splitting the rows of the current table.

The first table contains a percentage of the rows specified by `percentage_in_first`, and the second table
contains the remaining rows.

**Notes:**

- The original table is not modified.
- By default, the rows are shuffled before splitting. You can disable this by setting `shuffle` to false.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `percentageInFirst` | [`Float`][safeds.lang.Float] | The percentage of rows to include in the first table. Must be between 0 and 1. | - |
| `shuffle` | [`Boolean`][safeds.lang.Boolean] | Whether to shuffle the rows before splitting. | `#!sds true` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `firstTable` | [`Table`][safeds.data.tabular.containers.Table] | The first table. |
| `secondTable` | [`Table`][safeds.data.tabular.containers.Table] | The second table. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3, 4, 5], "b": [6, 7, 8, 9, 10]});
    val firstTable, val secondTable = table.splitRows(0.6);
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="760"
    @Pure
    @PythonName("split_rows")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun splitRows(
        @PythonName("percentage_in_first") percentageInFirst: Float,
        shuffle: Boolean = true
    ) -> (firstTable: Table, secondTable: Table)
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `summarizeStatistics` {#safeds.data.tabular.containers.Table.summarizeStatistics data-toc-label='[function] summarizeStatistics'}

Return a table with important statistics about this table.

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `statistics` | [`Table`][safeds.data.tabular.containers.Table] | The table with statistics. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 3]});
    val result = table.summarizeStatistics();
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="914"
    @Pure
    @PythonName("summarize_statistics")
    @Category(DataScienceCategory.DataExplorationQMetric)
    fun summarizeStatistics() -> statistics: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `toColumns` {#safeds.data.tabular.containers.Table.toColumns data-toc-label='[function] toColumns'}

Return the data of the table as a list of columns.

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `columns` | [`List<Column<Any?>>`][safeds.lang.List] | List of columns. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val columns = table.toColumns();
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="930"
    @Pure
    @PythonName("to_columns")
    @Category(DataScienceCategory.UtilitiesQConversion)
    fun toColumns() -> columns: List<Column>
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `toCsvFile` {#safeds.data.tabular.containers.Table.toCsvFile data-toc-label='[function] toCsvFile'}

Write the table to a CSV file.

If the file and/or the parent directories do not exist, they will be created. If the file exists already, it
will be overwritten.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `path` | [`String`][safeds.lang.String] | The path to the CSV file. If the file extension is omitted, it is assumed to be ".csv". | - |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    table.toCsvFile("output.csv");
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="949"
    @Impure([ImpurityReason.FileWriteToParameterizedPath("path")])
    @PythonName("to_csv_file")
    @Category(DataScienceCategory.DataExport)
    fun toCsvFile(
        path: String
    )
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `toJsonFile` {#safeds.data.tabular.containers.Table.toJsonFile data-toc-label='[function] toJsonFile'}

Write the table to a JSON file.

If the file and/or the parent directories do not exist, they will be created. If the file exists already, it
will be overwritten.

**Note:** This operation must fully load the data into memory, which can be expensive.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `path` | [`String`][safeds.lang.String] | The path to the JSON file. If the file extension is omitted, it is assumed to be ".json". | - |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    table.toJsonFile("output.json");
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="989"
    @Impure([ImpurityReason.FileWriteToParameterizedPath("path")])
    @PythonName("to_json_file")
    @Category(DataScienceCategory.DataExport)
    fun toJsonFile(
        path: String
    )
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `toMap` {#safeds.data.tabular.containers.Table.toMap data-toc-label='[function] toMap'}

Return a map that maps column names to column values.

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `map` | [`Map<String, List<Any>>`][safeds.lang.Map] | Map representation of the table. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.toMap();
    // {"a": [1, 2, 3], "b": [4, 5, 6]}
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="968"
    @Pure
    @PythonName("to_dict")
    @Category(DataScienceCategory.UtilitiesQConversion)
    fun toMap() -> map: Map<String, List<Any>>
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `toParquetFile` {#safeds.data.tabular.containers.Table.toParquetFile data-toc-label='[function] toParquetFile'}

Write the table to a Parquet file.

If the file and/or the parent directories do not exist, they will be created. If the file exists already, it
will be overwritten.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `path` | [`String`][safeds.lang.String] | The path to the Parquet file. If the file extension is omitted, it is assumed to be ".parquet". | - |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    table.toParquetFile("output.parquet");
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="1010"
    @Impure([ImpurityReason.FileWriteToParameterizedPath("path")])
    @PythonName("to_parquet_file")
    @Category(DataScienceCategory.DataExport)
    fun toParquetFile(
        path: String
    )
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `toTabularDataset` {#safeds.data.tabular.containers.Table.toTabularDataset data-toc-label='[function] toTabularDataset'}

Return a new `TabularDataset` with columns marked as a target, feature, or extra.

- The target column is the column that a model should predict.
- Feature columns are columns that a model should use to make predictions.
- Extra columns are columns that are neither feature nor target. They can be used to provide additional context,
  like an ID column.

Feature columns are implicitly defined as all columns except the target and extra columns. If no extra columns
are specified, all columns except the target column are used as features.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `targetName` | [`String`][safeds.lang.String] | The name of the target column. | - |
| `extraNames` | [`List<String>?`][safeds.lang.List] | Names of the columns that are neither feature nor target. If null, no extra columns are used, i.e. all but the target column are used as features. | `#!sds null` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `dataset` | [`TabularDataset`][safeds.data.labeled.containers.TabularDataset] | A new tabular dataset with the given target and feature names. |

**Examples:**

```sds hl_lines="9"
pipeline example {
    val table = Table(
        {
            "item": ["apple", "milk", "beer"],
            "price": [1.10, 1.19, 1.79],
            "amount_bought": [74, 72, 51],
        }
    );
    val dataset = table.toTabularDataset(targetName="amount_bought", extraNames=["item"]);
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="1046"
    @Pure
    @PythonName("to_tabular_dataset")
    @Category(DataScienceCategory.UtilitiesQConversion)
    fun toTabularDataset(
        @PythonName("target_name") targetName: String,
        @PythonName("extra_names") extraNames: List<String>? = null
    ) -> dataset: TabularDataset
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `toTimeSeriesDataset` {#safeds.data.tabular.containers.Table.toTimeSeriesDataset data-toc-label='[function] toTimeSeriesDataset'}

Return a new `TimeSeriesDataset` with columns marked as a target column, time or feature columns.

The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `targetName` | [`String`][safeds.lang.String] | The name of the target column. | - |
| `windowSize` | [`Int`][safeds.lang.Int] | The number of consecutive sample to use as input for prediction. | - |
| `extraNames` | [`List<String>?`][safeds.lang.List] | Names of the columns that are neither features nor target. If None, no extra columns are used, i.e. all but the target column are used as features. | `#!sds null` |
| `forecastHorizon` | [`Int`][safeds.lang.Int] | The number of time steps to predict into the future. | `#!sds 1` |
| `continuous` | [`Boolean`][safeds.lang.Boolean] | Whether or not to continue the forecast in the steps before forecast horizon. | `#!sds false` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `dataset` | [`TimeSeriesDataset`][safeds.data.labeled.containers.TimeSeriesDataset] | A new time series dataset with the given target and feature names. |

**Examples:**

```sds hl_lines="9"
pipeline example {
    val table = Table(
        {
            "day": [0, 1, 2],
            "price": [1.10, 1.19, 1.79],
            "amount_bought": [74, 72, 51],
        }
    );
    val dataset = table.toTimeSeriesDataset(targetName="amount_bought", windowSize=2);
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="1080"
    @Pure
    @PythonName("to_time_series_dataset")
    @Category(DataScienceCategory.UtilitiesQConversion)
    fun toTimeSeriesDataset(
        @PythonName("target_name") targetName: String,
        @PythonName("window_size") windowSize: Int,
        @PythonName("extra_names") extraNames: List<String>? = null,
        @PythonName("forecast_horizon") forecastHorizon: Int = 1,
        continuous: Boolean = false
    ) -> dataset: TimeSeriesDataset
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `transformColumn` {#safeds.data.tabular.containers.Table.transformColumn data-toc-label='[function] transformColumn'}

Return a new table with a column transformed.

**Note:** The original table is not modified.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `name` | [`String`][safeds.lang.String] | The name of the column to transform. | - |
| `transformer` | `#!sds (cell: Cell<Any?>) -> (transformedCell: Cell<Any?>)` | The function that transforms the column. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The table with the transformed column. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val table = Table({"a": [1, 2, 3], "b": [4, 5, 6]});
    val result = table.transformColumn("a", (cell) -> cell + 1);
    // Table({"a": [2, 3, 4], "b": [4, 5, 6]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="456"
    @Pure
    @PythonName("transform_column")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun transformColumn(
        name: String,
        transformer: (cell: Cell) -> transformedCell: Cell
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-function"></code> `transformTable` {#safeds.data.tabular.containers.Table.transformTable data-toc-label='[function] transformTable'}

Return a new table transformed by a **fitted** transformer.

**Notes:**

- The original table is not modified.
- Depending on the transformer, this operation might fully load the data into memory, which can be expensive.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `fittedTransformer` | [`TableTransformer`][safeds.data.tabular.transformation.TableTransformer] | The fitted transformer to apply. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `newTable` | [`Table`][safeds.data.tabular.containers.Table] | The transformed table. |

**Examples:**

```sds hl_lines="4"
pipeline example {
    val table = Table({"a": [1, 2, 3]});
    val transformer = RangeScaler(min=0.0, max=1.0, columnNames="a").fit(table);
    val result = table.transformTable(transformer);
    // Table({"a": [0, 0.5, 1]})
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="896"
    @Pure
    @PythonName("transform_table")
    @Category(DataScienceCategory.DataProcessingQTable)
    fun transformTable(
        @PythonName("fitted_transformer") fittedTransformer: TableTransformer
    ) -> newTable: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-static-function"></code> `fromColumns` {#safeds.data.tabular.containers.Table.fromColumns data-toc-label='[static-function] fromColumns'}

Create a table from a list of columns.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `columns` | `#!sds union<Column<Any?>, List<Column<Any?>>>` | The columns. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `table` | [`Table`][safeds.data.tabular.containers.Table] | The created table. |

**Examples:**

```sds hl_lines="4"
pipeline example {
    val a = Column("a", [1, 2, 3]);
    val b = Column("b", [4, 5, 6]);
    val result = Table.fromColumns([a, b]);
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="73"
    @Pure
    @PythonName("from_columns")
    @Category(DataScienceCategory.UtilitiesQConversion)
    static fun fromColumns(
        columns: union<Column, List<Column>>
    ) -> table: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-static-function"></code> `fromCsvFile` {#safeds.data.tabular.containers.Table.fromCsvFile data-toc-label='[static-function] fromCsvFile'}

Create a table from a CSV file.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `path` | [`String`][safeds.lang.String] | The path to the CSV file. If the file extension is omitted, it is assumed to be ".csv". | - |
| `separator` | [`String`][safeds.lang.String] | The separator between the values in the CSV file. | `#!sds ","` |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `table` | [`Table`][safeds.data.tabular.containers.Table] | The created table. |

**Examples:**

```sds hl_lines="2"
pipeline example {
    val result = Table.fromCsvFile("input.csv");
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="93"
    @Impure([ImpurityReason.FileReadFromParameterizedPath("path")])
    @PythonName("from_csv_file")
    @Category(DataScienceCategory.DataImport)
    static fun fromCsvFile(
        path: String,
        separator: String = ","
    ) -> table: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-static-function"></code> `fromJsonFile` {#safeds.data.tabular.containers.Table.fromJsonFile data-toc-label='[static-function] fromJsonFile'}

Create a table from a JSON file.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `path` | [`String`][safeds.lang.String] | The path to the JSON file. If the file extension is omitted, it is assumed to be ".json". | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `table` | [`Table`][safeds.data.tabular.containers.Table] | The created table. |

**Examples:**

```sds hl_lines="2"
pipeline example {
    val result = Table.fromJsonFile("input.json");
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="133"
    @Impure([ImpurityReason.FileReadFromParameterizedPath("path")])
    @PythonName("from_json_file")
    @Category(DataScienceCategory.DataImport)
    static fun fromJsonFile(
        path: String
    ) -> table: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-static-function"></code> `fromMap` {#safeds.data.tabular.containers.Table.fromMap data-toc-label='[static-function] fromMap'}

Create a table from a map that maps column names to column values.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `data` | [`Map<String, List<Any>>`][safeds.lang.Map] | The data. | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `table` | [`Table`][safeds.data.tabular.containers.Table] | The generated table. |

**Examples:**

```sds hl_lines="3"
pipeline example {
    val data = {"a'": [1, 2, 3], "b": [4, 5, 6]};
    val result = Table.fromMap(data);
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="114"
    @Pure
    @PythonName("from_dict")
    @Category(DataScienceCategory.DataImport)
    static fun fromMap(
        data: Map<String, List<Any>>
    ) -> table: Table
    ```
    { data-search-exclude }

## <code class="doc-symbol doc-symbol-static-function"></code> `fromParquetFile` {#safeds.data.tabular.containers.Table.fromParquetFile data-toc-label='[static-function] fromParquetFile'}

Create a table from a Parquet file.

**Parameters:**

| Name | Type | Description | Default |
|------|------|-------------|---------|
| `path` | [`String`][safeds.lang.String] | The path to the Parquet file. If the file extension is omitted, it is assumed to be ".parquet". | - |

**Results:**

| Name | Type | Description |
|------|------|-------------|
| `table` | [`Table`][safeds.data.tabular.containers.Table] | The created table. |

**Examples:**

```sds hl_lines="2"
pipeline example {
    val result = Table.fromParquetFile("input.parquet");
}
```

??? quote "Stub code in `Table.sdsstub`"

    ```sds linenums="152"
    @Impure([ImpurityReason.FileReadFromParameterizedPath("path")])
    @PythonName("from_parquet_file")
    @Category(DataScienceCategory.DataImport)
    static fun fromParquetFile(
        path: String
    ) -> table: Table
    ```
    { data-search-exclude }
