using ARMeilleure.State;
using NUnit.Framework;

namespace Ryujinx.Tests.Cpu
{
    [Category("Thumb")]
    public sealed class CpuTestThumb : CpuTest32
    {
        private const int RndCnt = 2;

        public static uint RotateRight(uint value, int count)
        {
            return (value >> count) | (value << (32 - count));
        }

        [Test, Pairwise]
        public void ShiftImm([Range(0u, 2u)] uint shiftType, [Range(1u, 0x1fu)] uint shiftImm, [Random(RndCnt)] uint w1, [Random(RndCnt)] uint w2)
        {
            uint opcode = 0x0000; // MOVS <Rd>, <Rm>, <shift> #<amount>

            uint rd = 1;
            uint rm = 2;
            opcode |= ((rd & 7) << 0) | ((rm & 7) << 3) | ((shiftImm & 0x1f) << 6) | ((shiftType & 3) << 11);

            SingleThumbOpcode((ushort)opcode, r1: w1, r2: w2, runUnicorn: false);

            switch (shiftType)
            {
                case 0:
                    Assert.That(GetContext().GetX(1), Is.EqualTo((w2 << (int)shiftImm) & 0xffffffffu));
                    break;
                case 1:
                    Assert.That(GetContext().GetX(1), Is.EqualTo((w2 >> (int)shiftImm) & 0xffffffffu));
                    break;
                case 2:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(((int)w2 >> (int)shiftImm) & 0xffffffffu));
                    break;
            }
        }

        [Test, Pairwise]
        public void AddSubReg([Range(0u, 1u)] uint op, [Random(RndCnt)] uint w1, [Random(RndCnt)] uint w2)
        {
            uint opcode = 0x1800; // ADDS <Rd>, <Rn>, <Rm>

            uint rd = 0;
            uint rn = 1;
            uint rm = 2;
            opcode |= ((rd & 7) << 0) | ((rn & 7) << 3) | ((rm & 7) << 6) | ((op & 1) << 9);

            SingleThumbOpcode((ushort)opcode, r1: w1, r2: w2, runUnicorn: false);

            switch (op)
            {
                case 0:
                    Assert.That(GetContext().GetX(0), Is.EqualTo((w1 + w2) & 0xffffffffu));
                    break;
                case 1:
                    Assert.That(GetContext().GetX(0), Is.EqualTo((w1 - w2) & 0xffffffffu));
                    break;
            }
        }

        [Test, Pairwise]
        public void AddSubImm3([Range(0u, 1u)] uint op, [Range(0u, 7u)] uint imm, [Random(RndCnt)] uint w1)
        {
            uint opcode = 0x1c00; // ADDS <Rd>, <Rn>, #<imm3>

            uint rd = 0;
            uint rn = 1;
            opcode |= ((rd & 7) << 0) | ((rn & 7) << 3) | ((imm & 7) << 6) | ((op & 1) << 9);

            SingleThumbOpcode((ushort)opcode, r1: w1, runUnicorn: false);

            switch (op)
            {
                case 0:
                    Assert.That(GetContext().GetX(0), Is.EqualTo((w1 + imm) & 0xffffffffu));
                    break;
                case 1:
                    Assert.That(GetContext().GetX(0), Is.EqualTo((w1 - imm) & 0xffffffffu));
                    break;
            }
        }

        [Test, Pairwise]
        public void AluImm8([Range(0u, 3u)] uint op, [Random(RndCnt)] uint imm, [Random(RndCnt)] uint w1)
        {
            imm &= 0xff;

            uint opcode = 0x2000; // MOVS <Rdn>, #<imm8>

            uint rdn = 1;
            opcode |= ((imm & 0xff) << 0) | ((rdn & 7) << 8) | ((op & 3) << 11);

            SingleThumbOpcode((ushort)opcode, r1: w1, runUnicorn: false);

            switch (op)
            {
                case 0:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(imm));
                    break;
                case 1:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1));
                cmpFlags:
                    {
                        uint result = w1 - imm;
                        uint overflow = (result ^ w1) & (w1 ^ imm);
                        Assert.That(GetContext().GetPstateFlag(PState.NFlag), Is.EqualTo((result >> 31) != 0));
                        Assert.That(GetContext().GetPstateFlag(PState.ZFlag), Is.EqualTo(result == 0));
                        Assert.That(GetContext().GetPstateFlag(PState.CFlag), Is.EqualTo(w1 >= imm));
                        Assert.That(GetContext().GetPstateFlag(PState.VFlag), Is.EqualTo((overflow >> 31) != 0));
                    }
                    break;
                case 2:
                    Assert.That(GetContext().GetX(1), Is.EqualTo((w1 + imm) & 0xffffffffu));
                    break;
                case 3:
                    Assert.That(GetContext().GetX(1), Is.EqualTo((w1 - imm) & 0xffffffffu));
                    goto cmpFlags;
            }
        }

        [Test, Pairwise]
        public void AluRegLow([Range(0u, 0xfu)] uint op, [Random(RndCnt)] uint w1, [Random(RndCnt)] uint w2)
        {
            uint opcode = 0x4000; // ANDS <Rdn>, <Rm>

            uint rd = 1;
            uint rm = 2;
            opcode |= ((rd & 7) << 0) | ((rm & 7) << 3) | ((op & 0xf) << 6);

            SingleThumbOpcode((ushort)opcode, r1: w1, r2: w2, runUnicorn: false);

            uint shift = w2 & 0xff;
            switch (op)
            {
                case 0:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1 & w2));
                    break;
                case 1:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1 ^ w2));
                    break;
                case 2:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(shift >= 32 ? 0 : w1 << (int)shift));
                    break;
                case 3:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(shift >= 32 ? 0 : w1 >> (int)shift));
                    break;
                case 4:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(shift >= 32 ? (uint)((int)w1 >> 31) : (uint)((int)w1 >> (int)shift)));
                    break;
                case 5:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1 + w2));
                    break;
                case 6:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1 + ~w2));
                    break;
                case 7:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(RotateRight(w1, (int)shift & 31)));
                    break;
                case 8:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1));
                    {
                        uint result = w1 & w2;
                        Assert.That(GetContext().GetPstateFlag(PState.NFlag), Is.EqualTo((result >> 31) != 0));
                        Assert.That(GetContext().GetPstateFlag(PState.ZFlag), Is.EqualTo(result == 0));
                    }
                    break;
                case 9:
                    Assert.That(GetContext().GetX(1), Is.EqualTo((uint)-w2));
                    break;
                case 10:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1));
                    {
                        uint result = w1 - w2;
                        uint overflow = (result ^ w1) & (w1 ^ w2);
                        Assert.That(GetContext().GetPstateFlag(PState.NFlag), Is.EqualTo((result >> 31) != 0));
                        Assert.That(GetContext().GetPstateFlag(PState.ZFlag), Is.EqualTo(result == 0));
                        Assert.That(GetContext().GetPstateFlag(PState.CFlag), Is.EqualTo(w1 >= w2));
                        Assert.That(GetContext().GetPstateFlag(PState.VFlag), Is.EqualTo((overflow >> 31) != 0));
                    }
                    break;
                case 11:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1));
                    {
                        uint result = w1 + w2;
                        uint overflow = (result ^ w1) & ~(w1 ^ w2);
                        Assert.That(GetContext().GetPstateFlag(PState.NFlag), Is.EqualTo((result >> 31) != 0));
                        Assert.That(GetContext().GetPstateFlag(PState.ZFlag), Is.EqualTo(result == 0));
                        Assert.That(GetContext().GetPstateFlag(PState.CFlag), Is.EqualTo(result < w1));
                        Assert.That(GetContext().GetPstateFlag(PState.VFlag), Is.EqualTo((overflow >> 31) != 0));
                    }
                    break;
                case 12:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1 | w2));
                    break;
                case 13:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1 * w2));
                    break;
                case 14:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(w1 & ~w2));
                    break;
                case 15:
                    Assert.That(GetContext().GetX(1), Is.EqualTo(~w2));
                    break;
            }
        }

        [Test, Pairwise]
        public void AluRegHigh([Range(0u, 2u)] uint op, [Range(0u, 13u)] uint rd, [Range(0u, 13u)] uint rm, [Random(RndCnt)] uint w1, [Random(RndCnt)] uint w2)
        {
            if (rd == rm)
            {
                return;
            }

            uint opcode = 0x4400; // ADDS <Rdn>, <Rm>

            opcode |= ((rd & 7) << 0) | ((rm & 0xf) << 3) | ((rd & 8) << 4) | ((op & 3) << 8);

            ThumbOpcode((ushort)opcode);
            ThumbOpcode(0x4770); // BX LR

            GetContext().SetX((int)rd, w1);
            GetContext().SetX((int)rm, w2);
            GetContext().SetPstateFlag(PState.TFlag, true);

            ExecuteOpcodes(runUnicorn: false);

            switch (op)
            {
                case 0:
                    Assert.That(GetContext().GetX((int)rd), Is.EqualTo(w1 + w2));
                    break;
                case 1:
                    Assert.That(GetContext().GetX((int)rd), Is.EqualTo(w1));
                    Assert.That(GetContext().GetX((int)rm), Is.EqualTo(w2));
                    {
                        uint result = w1 - w2;
                        uint overflow = (result ^ w1) & (w1 ^ w2);
                        Assert.That(GetContext().GetPstateFlag(PState.NFlag), Is.EqualTo((result >> 31) != 0));
                        Assert.That(GetContext().GetPstateFlag(PState.ZFlag), Is.EqualTo(result == 0));
                        Assert.That(GetContext().GetPstateFlag(PState.CFlag), Is.EqualTo(w1 >= w2));
                        Assert.That(GetContext().GetPstateFlag(PState.VFlag), Is.EqualTo((overflow >> 31) != 0));
                    }
                    break;
                case 2:
                    Assert.That(GetContext().GetX((int)rd), Is.EqualTo(w2));
                    break;
            }
        }

        [Test]
        public void SubSpTest()
        {
            ThumbOpcode(0xb0fd); // SUB SP, #0x1f4
            ThumbOpcode(0x4770); // BX LR

            GetContext().SetX(13, 0x40079d98);
            GetContext().SetPstateFlag(PState.TFlag, true);

            ExecuteOpcodes(runUnicorn: false);

            Assert.That(GetContext().GetX(13), Is.EqualTo(0x40079ba4));
        }

        [Test]
        public void TestRandomTestCases([ValueSource(nameof(RandomTestCases))] PrecomputedThumbTestCase test)
        {
            if (Size != 0x1000)
            {
                // TODO: Change it to depend on DataBaseAddress instead.
                Assert.Ignore("This test currently only support 4KiB page size");
            }

            RunPrecomputedTestCase(test);
        }

        public static readonly PrecomputedThumbTestCase[] RandomTestCases =
        {
            new()
            {
                Instructions = new ushort[] { 0x42ab, 0xabed, 0x43fc, 0x4360, 0x40c1, 0x40b5, 0x4353, 0x43ac, 0xba74, 0x4613, 0xb03a, 0xbf2a, 0xa5bd, 0x187b, 0x410a, 0x4405, 0x15b3, 0x1472, 0xb248, 0x43a6, 0x41af, 0xbaf4, 0x1a6d, 0x4684, 0xbf04, 0x4068, 0xb26e, 0x014b, 0x4064, 0xba15, 0x3d2d, 0xb22d, 0x4378, 0x2501, 0x4279, 0xb299, 0x1586, 0x42e7, 0x2f55, 0xba55, 0x1f6f, 0x443d, 0x4194, 0xbfcb, 0xbacd, 0xb2a0, 0x406b, 0x1b65, 0xbfd0, 0x1bf7, 0x41e7, 0xba76, 0x436f, 0x46f8, 0x4042, 0x1ffe, 0x447c, 0xba50, 0x341e, 0x42fd, 0x409c, 0xbf09, 0xb2cf, 0x1c99, 0x41f7, 0x41a6, 0x4278, 0x4040, 0x3747, 0x301a, 0x01cc, 0x4304, 0x1ca3, 0xab1d, 0x43dd, 0x41e3, 0x35ad, 0x43f5, 0xb091, 0x3214, 0x44f1, 0x41d7, 0xb244, 0xb2a2, 0x41e1, 0x3d7a, 0xbf24, 0xbac1, 0x0059, 0xba3b, 0xabe5, 0x4550, 0x0a9e, 0xbfe2, 0x034e, 0xb0b7, 0x0aec, 0x1ee6, 0x3276, 0x0866, 0xa763, 0x1e45, 0x4275, 0xad25, 0x416d, 0x0d6d, 0x0b50, 0xb200, 0x1eba, 0x4378, 0x0547, 0x43bb, 0xb2f2, 0xb2af, 0x1ea0, 0x25b0, 0x4287, 0x1d55, 0x2225, 0xbf46, 0xb276, 0xba57, 0x4253, 0x19e0, 0xbf32, 0x1b06, 0x40ab, 0xb2a4, 0x1eda, 0x1dfd, 0xaabd, 0x1b08, 0xba0e, 0x4014, 0xa079, 0x119b, 0x1d17, 0x41b1, 0x005a, 0xb2ba, 0x419a, 0x4361, 0x2d2a, 0xba36, 0x1b49, 0xbf0c, 0x423c, 0xbae6, 0x2bc0, 0x409c, 0xbfa0, 0xba02, 0x2496, 0xb279, 0xbace, 0xb2e0, 0xbf07, 0x3b17, 0x40a9, 0x3154, 0x41ab, 0x284e, 0xb03c, 0xbf06, 0x41ff, 0xba45, 0x4144, 0xab32, 0x4000, 0xb2b2, 0x3729, 0x46fb, 0xb042, 0x32d0, 0x0557, 0x3583, 0xb249, 0x43f5, 0x0a54, 0x416f, 0x1fc6, 0x4560, 0xb2f6, 0xa114, 0x1dd7, 0x40b4, 0x4304, 0xbfc2, 0x41f2, 0xb2e2, 0xb0d4, 0xbaca, 0xbf65, 0x4065, 0xa1e5, 0xb043, 0x4004, 0xb237, 0x41e9, 0xba0f, 0xaa8e, 0x421d, 0x1d12, 0x435e, 0x434b, 0x4356, 0xacaf, 0x4551, 0x1935, 0x1c1a, 0xba67, 0x2441, 0xbf9a, 0x41a0, 0xb2d8, 0x1c00, 0xad68, 0x4185, 0xb272, 0x43ba, 0x436c, 0xb08b, 0xb2f4, 0x4379, 0x1d0f, 0x46f8, 0x4379, 0x1b8f, 0x4260, 0x40d8, 0xbfa7, 0x0730, 0xb27d, 0xb242, 0x43d3, 0x46fd, 0xbf92, 0xb067, 0xaa15, 0x428c, 0xb2e1, 0x43e5, 0xb2f6, 0x1da1, 0x023e, 0x1a86, 0xba01, 0xb203, 0x13ae, 0xa6a6, 0x45d8, 0xba4c, 0xb228, 0x438c, 0xbae8, 0x4378, 0x3b55, 0xbf5c, 0xb036, 0xb261, 0x1bea, 0x41b8, 0x43f5, 0xba70, 0xa959, 0xb2a9, 0x41fd, 0x19ad, 0x41cd, 0x4171, 0xb046, 0xbf37, 0x431e, 0x4302, 0xb04f, 0x396f, 0x44f0, 0x36d3, 0x01b0, 0x18f5, 0x424b, 0xb27a, 0x43a9, 0xba49, 0x4357, 0x41e5, 0x42d4, 0x0f27, 0x4325, 0x1f9b, 0x447a, 0x46b8, 0xbf60, 0x43c5, 0xa78b, 0x46b2, 0xb2c4, 0x4322, 0xbfad, 0xbf90, 0x4275, 0x40c4, 0x1e91, 0xba0c, 0xbfd0, 0xafd3, 0xbfa0, 0xb2ff, 0xb2dc, 0x43a1, 0xb0db, 0xbf1e, 0x269d, 0xb21a, 0x405d, 0x1822, 0x3dc1, 0xb247, 0x119e, 0xb2b9, 0x4056, 0xaedd, 0xba59, 0x40ed, 0x43c0, 0xbf5f, 0xba35, 0xad1b, 0xb0db, 0x274a, 0x4042, 0x404c, 0xb0a6, 0x2983, 0x41b9, 0x4226, 0x455b, 0xbad3, 0xba5c, 0x42e3, 0xbf8f, 0x3b28, 0xadf6, 0x0321, 0x41c0, 0x223b, 0x40ca, 0x1ea5, 0x4256, 0x409a, 0x4339, 0xba7c, 0x43cf, 0x40bb, 0xb217, 0x427a, 0x289f, 0xb28b, 0xb250, 0xbf2a, 0xb084, 0x410c, 0x416b, 0x4079, 0xb2c9, 0x4149, 0x4143, 0x0c11, 0x461f, 0x18ed, 0x41e1, 0x4651, 0xba1c, 0x422b, 0x44fb, 0xb2b4, 0xbfde, 0x45e1, 0x4601, 0x4483, 0xa182, 0x3311, 0x40bd, 0x446a, 0x431c, 0x2f1e, 0xbac8, 0x02c3, 0x43d5, 0x415d, 0x42f7, 0x1cad, 0x1597, 0xb2cd, 0x1cbd, 0x42df, 0x069e, 0x0986, 0x435d, 0xbad5, 0xbf39, 0x1044, 0x417c, 0x3afb, 0x3be4, 0xb2b8, 0xb0b6, 0x401b, 0x1e4f, 0x4270, 0x4278, 0xacb6, 0xbad1, 0xbaee, 0x1920, 0xba01, 0x4367, 0xb24d, 0xb2db, 0x05da, 0xa2c6, 0x4627, 0xba2b, 0xbfa3, 0x42aa, 0x3637, 0xbafa, 0x4364, 0x43f3, 0xaaea, 0x40a5, 0x424f, 0x45a3, 0x44e9, 0x4477, 0xb09e, 0x42ea, 0x44b9, 0x4694, 0x4629, 0xba4b, 0x42c8, 0x412e, 0xbf9e, 0xba0a, 0x420a, 0xb2d0, 0xbae8, 0xbfb3, 0x40f8, 0x36d0, 0x40bd, 0x4395, 0x0d0f, 0xbafa, 0xb20b, 0x41c6, 0x40ea, 0xbf62, 0x429b, 0x4142, 0x43a6, 0x1f1b, 0xa9b4, 0x4455, 0xb0b1, 0xbae8, 0xb24c, 0x0396, 0x4360, 0xb26d, 0x0693, 0x1842, 0xba1e, 0xb29b, 0xba44, 0x1c45, 0x456f, 0x1a6a, 0x439c, 0xb24d, 0xac7d, 0x410b, 0x419f, 0xbf7e, 0x4057, 0x4377, 0x1ea6, 0x2f69, 0xbfc0, 0x1a75, 0xb2cf, 0x3759, 0x4187, 0x429b, 0x40d7, 0x18e7, 0x2eaa, 0x2bc6, 0xbf0d, 0x028c, 0x40f2, 0x421c, 0xbad1, 0xbf70, 0x41c5, 0xb012, 0x42fe, 0xb2ef, 0xbf96, 0xba10, 0x4103, 0x4434, 0x449a, 0x0b03, 0x40d5, 0x4221, 0x345a, 0x1491, 0xb2fc, 0x46e1, 0x456c, 0xb0f0, 0xb2ba, 0x4254, 0x415f, 0x4269, 0x434f, 0xb04c, 0x4316, 0xb049, 0xba18, 0x1988, 0x4194, 0x4412, 0xbfb3, 0x43f8, 0xba4f, 0xb218, 0x4033, 0x38a4, 0x3d99, 0xa6b2, 0xb076, 0xbfc0, 0x42c6, 0x40ad, 0x432a, 0x4226, 0x1ee9, 0x1aa5, 0xbf13, 0x3370, 0x2820, 0x1191, 0x432c, 0x40b8, 0x464e, 0x1e58, 0x462f, 0xb022, 0x40ba, 0x464f, 0x4083, 0x4683, 0x16c7, 0xbf43, 0x42a0, 0xba56, 0x2f3d, 0x417b, 0x43ff, 0xac7d, 0xb285, 0x4041, 0xba51, 0xbf31, 0x4041, 0xb281, 0xb2d8, 0x1a99, 0x4342, 0x402f, 0x4185, 0x40f3, 0xbfd0, 0x4274, 0x40a3, 0x4064, 0x39a4, 0xba04, 0x408e, 0x3250, 0x3c3d, 0xa2bb, 0xba1c, 0x1a4b, 0x40e5, 0x3ca6, 0xbf67, 0x4285, 0xa9f3, 0x46bc, 0x420d, 0xba05, 0xb009, 0x417c, 0xb27e, 0x1c48, 0x41c8, 0x42e3, 0x1e8c, 0x42b4, 0xba08, 0x3598, 0xb263, 0xb2f2, 0x1053, 0x40da, 0x37c7, 0x370d, 0x1853, 0x41e9, 0xb061, 0x1af7, 0x4015, 0x44aa, 0x1db0, 0x4165, 0xbf27, 0x3d1f, 0x42f4, 0x432f, 0x407c, 0x1457, 0x43c4, 0xb226, 0x40d3, 0x4317, 0x409d, 0x4024, 0xb062, 0xaeba, 0x3c71, 0x43a1, 0xada2, 0x4115, 0xba3f, 0x4163, 0x4386, 0xb2c9, 0x1b5a, 0xba22, 0xbf1e, 0xbaea, 0xbadc, 0x0ff5, 0x1102, 0x41a5, 0x0122, 0x43b3, 0x30c2, 0x43cb, 0x41c8, 0x0365, 0xaa50, 0xb200, 0x4347, 0x4049, 0x4101, 0x0ba9, 0x325b, 0x296c, 0x4364, 0x43d7, 0xbaf1, 0xbfb4, 0x436e, 0xb28f, 0x0163, 0x42d8, 0x4016, 0x33c6, 0x23f4, 0x40e6, 0x25d3, 0x40cf, 0x3a7c, 0x445c, 0x3781, 0x4083, 0xbf16, 0x46a5, 0xae3c, 0x43d1, 0xbf2f, 0x41d9, 0x41b5, 0xbac7, 0x4381, 0x434b, 0x099c, 0x2c64, 0x4202, 0x40c8, 0x4194, 0x4625, 0xbf19, 0x423b, 0x4098, 0x40f0, 0x437f, 0x2643, 0xbaee, 0x40a0, 0x00b2, 0xb06f, 0x42d2, 0x4047, 0xa2b2, 0xb284, 0x4008, 0xaeed, 0x46a9, 0x40d4, 0x42fe, 0x4084, 0x41d0, 0x4304, 0x0437, 0xbad6, 0x4247, 0x4068, 0x43db, 0xbf8e, 0x1fcf, 0x278a, 0xb202, 0x4304, 0x0bf0, 0x4389, 0x3f6d, 0xb266, 0xb243, 0x152d, 0x4630, 0xaee7, 0x0104, 0x40c3, 0x40b9, 0xa96c, 0xb2e0, 0xa0f9, 0xb273, 0x0d05, 0x402b, 0x408f, 0x0b31, 0xa305, 0xbf65, 0x4200, 0xb2e0, 0x43e8, 0x43a2, 0xb206, 0x436f, 0xba61, 0x4037, 0xba4c, 0xbad3, 0x46ed, 0x3501, 0x4321, 0x4137, 0x41c3, 0x17ef, 0x42b7, 0xba13, 0x4246, 0xbf58, 0xb210, 0xb04f, 0x40e1, 0x42e3, 0xa428, 0x404e, 0xb2d1, 0x4043, 0x3f7b, 0x1d55, 0x35c0, 0x1b98, 0x1f9b, 0x44db, 0xbf17, 0xb240, 0x1e91, 0xb0a7, 0x402a, 0x4045, 0x414b, 0x415c, 0x08b1, 0xba10, 0x188b, 0x4219, 0x32e9, 0x427a, 0x2242, 0x1f51, 0x43f8, 0x409c, 0xbaff, 0x1f4a, 0x418f, 0x4566, 0x4151, 0xbaca, 0x4083, 0x409f, 0x44e4, 0xbf70, 0x341c, 0xbf0e, 0x400f, 0x40b5, 0x466d, 0x01a2, 0x4493, 0x43e4, 0x40e4, 0xba5b, 0xb289, 0xbad8, 0x0d68, 0x424f, 0xb20e, 0x418b, 0xb2c3, 0xade0, 0x4066, 0xbf95, 0x2228, 0x4110, 0x1db8, 0xa1e2, 0x0589, 0x4203, 0x4371, 0xbf77, 0x3288, 0x42f8, 0x4341, 0x413e, 0xb027, 0xbaf9, 0xbf70, 0x157d, 0x1d0d, 0xa3c2, 0x1e40, 0x448c, 0x405b, 0x14c2, 0x433f, 0x43b6, 0xba18, 0x1689, 0xbf82, 0x40aa, 0x4048, 0x320d, 0x4620, 0x42c9, 0x1670, 0x409f, 0xba26, 0xbad0, 0x43a5, 0x31b7, 0xbac8, 0x4405, 0xa60d, 0x1e58, 0x2989, 0xb22b, 0x3c62, 0x409f, 0x4295, 0xb287, 0xad38, 0xb025, 0x43eb, 0xbfbf, 0x0126, 0x4253, 0x42df, 0xa723, 0xa96e, 0x42f7, 0x3748, 0xac86, 0x408a, 0xae5c, 0x3778, 0xb2e5, 0xbfc1, 0x418b, 0xbac3, 0xb0fb, 0x45a6, 0x4293, 0x426b, 0xb07e, 0x43c3, 0x34f3, 0x08bc, 0x02c7, 0xbfbc, 0x42af, 0x421d, 0x2675, 0x428b, 0xba07, 0x4107, 0xb07a, 0x433d, 0x438e, 0x1195, 0x403d, 0x4545, 0xbad3, 0x1b5e, 0xbf8f, 0xb289, 0x23df, 0x402d, 0x413b, 0x2af8, 0x003b, 0x46ad, 0xbf9a, 0x4271, 0x433d, 0x43c0, 0x2f27, 0xa5e3, 0x179d, 0x1ced, 0x445d, 0x43c0, 0x3eb2, 0x40e8, 0x39ba, 0x4088, 0x4117, 0x3137, 0x189b, 0xb024, 0xa52f, 0x41b4, 0x1f27, 0x439d, 0x42e2, 0x4265, 0xbfe0, 0x252f, 0xbf7f, 0x3bd7, 0x40dd, 0x2cc1, 0x1c04, 0x41a6, 0x1403, 0x408c, 0x46bd, 0x2a99, 0x4230, 0x42f9, 0x04e8, 0xba5d, 0x284d, 0x43c7, 0x4332, 0x3921, 0x443c, 0xbf08, 0x4390, 0x1edf, 0xa980, 0x40a7, 0xb250, 0xb074, 0x41fd, 0xa12a, 0x1edd, 0xbad8, 0xb231, 0xbfd7, 0x1951, 0x4337, 0x2393, 0x4491, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xfb8e51a4, 0xf9f21a25, 0x2a13c422, 0x57b1ba81, 0xf344e297, 0x1706c70a, 0xb7e92cf4, 0x512917e4, 0x5a86f7df, 0x755562e5, 0xae714612, 0x2c5f02dc, 0x0dcdaeee, 0x75c86709, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x00000000, 0xffffff4b, 0xffffff4e, 0x00000093, 0xffffffff, 0xfffffffd, 0xffffff4e, 0x00000000, 0x00000000, 0x03f9e62a, 0x00000082, 0x000007de, 0xffff8cdd, 0x000042ac, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x18f0, 0x405d, 0x4152, 0x40a4, 0x426b, 0xb201, 0x1a63, 0x4623, 0x41a9, 0x1a48, 0x4165, 0xbf7d, 0xba5c, 0xbf00, 0xb06d, 0x43b7, 0x44ca, 0x41b2, 0x2b08, 0xb2c5, 0xb224, 0xb224, 0xb217, 0x2985, 0xba24, 0xb080, 0x43e8, 0x1d56, 0x45f2, 0x4226, 0x3d7c, 0x41bd, 0xbf9f, 0x4315, 0xbafe, 0x400f, 0x463e, 0x22db, 0x4143, 0x42c2, 0x43e0, 0x43e7, 0x40c5, 0xb221, 0xbfc2, 0xb0e4, 0xba4a, 0xbf90, 0x437a, 0x41e2, 0x403c, 0x2581, 0xa403, 0xa17c, 0x4322, 0x1bed, 0x01f8, 0xbf92, 0x3c28, 0x1ed9, 0x0c6a, 0xb219, 0xbf44, 0x416f, 0x18b5, 0xb2ce, 0x405e, 0x31f9, 0x1afe, 0xb068, 0xa488, 0xbaf0, 0x429e, 0xb265, 0x4154, 0x435c, 0x3f65, 0x4132, 0x400f, 0xbfcb, 0xb2b4, 0x4302, 0xa661, 0x429b, 0xba09, 0xb0bd, 0x1c5d, 0x4396, 0xb205, 0x4210, 0x4184, 0x4114, 0x30e9, 0x4694, 0x4281, 0xba3b, 0xb02c, 0x3c7c, 0xb068, 0x1f5f, 0xbad2, 0x2fc9, 0xb2fa, 0x422c, 0x40d1, 0x427a, 0xb00b, 0x1bd1, 0x1959, 0xbf94, 0x35f7, 0x415f, 0x4230, 0x4491, 0xbada, 0x4390, 0xa1d1, 0x4201, 0x4109, 0x3b9a, 0x4279, 0x4372, 0xbff0, 0xb2b4, 0x1c29, 0x0bba, 0x42ff, 0x0b07, 0x431a, 0x35b1, 0x2674, 0x1d4b, 0x43e4, 0xbfdc, 0x4385, 0x00d9, 0x33f4, 0x4010, 0xab8a, 0x42e7, 0xb093, 0xba39, 0x4140, 0x2120, 0xb2e8, 0x2254, 0x4354, 0x430e, 0xba16, 0x441b, 0x1b54, 0xaf2b, 0x4144, 0x424f, 0x3e6a, 0x0c2f, 0x4330, 0x44cc, 0x42a9, 0x433f, 0xbf93, 0x4367, 0x3de9, 0x4203, 0x4120, 0xa589, 0x1a1d, 0xaa95, 0x4068, 0x02ff, 0x4071, 0x44da, 0x181f, 0xbf8a, 0x2c8c, 0x183e, 0x40fd, 0x3711, 0xb2c1, 0x40d1, 0x0e30, 0x42e0, 0xba5f, 0xb00f, 0x0e5e, 0xb2b0, 0x419e, 0x42df, 0x4240, 0xa9a1, 0xaeba, 0x4319, 0x45f1, 0xbae7, 0x4390, 0x43ad, 0x40a0, 0x1e6e, 0x0c84, 0x2ddb, 0xbf45, 0xbfb0, 0x4405, 0x1ca9, 0x39cb, 0xb2e8, 0x0a6f, 0xba79, 0x4044, 0x4491, 0x3a70, 0xa45d, 0x425d, 0x3a30, 0x4107, 0xa248, 0x4055, 0x40f8, 0x43fb, 0x4078, 0x351f, 0x41a3, 0x439f, 0xba6a, 0x426a, 0x3f2f, 0xbf5d, 0x40ee, 0x43b9, 0x1b4a, 0xa56a, 0x2fec, 0xba54, 0x4613, 0x46c5, 0x4237, 0x3dd2, 0xb037, 0x3e1a, 0x04e2, 0x415d, 0x426c, 0xb2ed, 0x42c3, 0x416f, 0x1d8a, 0x42cd, 0x42c8, 0x422a, 0xbf7f, 0xa9b0, 0x4303, 0x43ea, 0x359a, 0x4058, 0x418f, 0x436e, 0xb289, 0x2731, 0x432b, 0x40c4, 0xb298, 0x401e, 0x011e, 0x4132, 0x43b6, 0x4148, 0x41a9, 0x413b, 0x4191, 0x1cb7, 0x0612, 0x4333, 0x3dfe, 0x4264, 0x4261, 0xaabe, 0xba60, 0x40d6, 0xbfe8, 0xa1fc, 0x025e, 0x4249, 0x33b2, 0x2d28, 0x428c, 0xb22d, 0xbae2, 0x449c, 0x40ac, 0x41b2, 0x193a, 0x435b, 0x4155, 0x4060, 0x42b8, 0x4363, 0x18d6, 0x4162, 0x145b, 0x439a, 0xbf2b, 0xba23, 0x41fa, 0x311d, 0xb2bf, 0x3045, 0xba6f, 0xbf06, 0x213a, 0x4565, 0xb0af, 0xb2cf, 0xa9db, 0x41b9, 0x19e3, 0x427d, 0xa296, 0x440c, 0x4363, 0x411f, 0x425b, 0x421a, 0x4207, 0x1840, 0x435e, 0x4285, 0xba48, 0xb2de, 0x420c, 0xba22, 0xbf5d, 0x41d3, 0x1db2, 0x400a, 0xb2b3, 0x42f1, 0x4014, 0x42a5, 0xae6b, 0x37bf, 0xa4b4, 0x437b, 0x4089, 0xbf23, 0x41ca, 0x43c7, 0x45b4, 0xb2d0, 0x418f, 0x4086, 0x03a8, 0x401c, 0x42b0, 0xb270, 0x41df, 0x0b28, 0x428e, 0x1cd2, 0x4666, 0x426a, 0x275d, 0x4189, 0x42a3, 0x0223, 0x40a6, 0xb041, 0x16a8, 0x2e46, 0xba19, 0xb2e6, 0x1b94, 0xbfc1, 0x40cd, 0x1074, 0xb29a, 0x411c, 0x41e6, 0xb052, 0x18f6, 0x4356, 0x43fc, 0x19b9, 0x4045, 0x1a37, 0x4582, 0x43ba, 0x462b, 0x2e1b, 0x4052, 0x0a1a, 0x43ae, 0x31a1, 0x41f2, 0x42dd, 0x17ea, 0x43ee, 0x42f3, 0x2cbf, 0xbf66, 0xb023, 0x40f6, 0x407e, 0x4499, 0x4234, 0x44f4, 0x404a, 0xb2ed, 0x40d4, 0xbaea, 0x0ea6, 0x439a, 0xae0e, 0x1334, 0x4558, 0x336a, 0xb03b, 0x0b94, 0x41cf, 0x4256, 0x4188, 0x31f8, 0x404b, 0x410e, 0xbf69, 0x2a6c, 0xbacb, 0x1e69, 0x011a, 0xb23e, 0x44d0, 0x3916, 0xb295, 0xa0d9, 0xbf84, 0x3b79, 0xa6c3, 0xb089, 0xbfb0, 0x4303, 0x331b, 0xb229, 0x4341, 0x0bbb, 0x1ad9, 0x4047, 0x1ae3, 0x431c, 0xa448, 0x4179, 0x432c, 0xb281, 0x348d, 0x4170, 0xbf47, 0x43e2, 0x41a1, 0x419f, 0xa0fb, 0xb2d7, 0xb2d1, 0x412f, 0x467f, 0xba40, 0xb2e0, 0xbad8, 0x423b, 0x4628, 0x43ea, 0x41b9, 0x41eb, 0xb2b3, 0xbf28, 0x4350, 0xa460, 0x433f, 0x4032, 0x4390, 0xb287, 0x1865, 0x4668, 0x448c, 0x41b5, 0xbf75, 0x41ce, 0xbacd, 0x18ba, 0x1c8d, 0x41fc, 0x42eb, 0xbf18, 0x401d, 0x3d5d, 0x4122, 0xb01a, 0x40c4, 0x4321, 0xa7b0, 0xba1f, 0x4035, 0x0f4e, 0x2829, 0x031f, 0x464a, 0x404f, 0xbf70, 0x4093, 0x4321, 0x42ad, 0x4230, 0x4124, 0x4218, 0x00a8, 0xa451, 0xb2b3, 0xbf7c, 0xb26f, 0x44d3, 0x423b, 0xbfd0, 0x04af, 0xb28f, 0x4335, 0x39ab, 0x42c8, 0xbf24, 0xbf70, 0x3665, 0xb24e, 0xb2d8, 0xa20d, 0x10a4, 0x407e, 0xb0c1, 0xbfc5, 0xb281, 0x400e, 0x4352, 0x46c8, 0xbfb8, 0x4074, 0x4410, 0x4249, 0x4335, 0x248a, 0x43aa, 0x4074, 0x4171, 0x43fd, 0x4060, 0x1fe9, 0xbfae, 0x4477, 0x2eb5, 0x410a, 0x187d, 0x41aa, 0x4137, 0x434c, 0x4353, 0x1ff7, 0xb215, 0x3c25, 0x183f, 0xb2ad, 0xbf4b, 0xb2c7, 0xb00b, 0x4339, 0x417b, 0x44f3, 0xaa2f, 0x4304, 0x43e5, 0x00a0, 0xba0a, 0x41b8, 0x4250, 0xbf22, 0x45bd, 0x41f0, 0xbaf1, 0x419a, 0x1d5c, 0x2eae, 0x42e7, 0x43e3, 0x4680, 0x433b, 0xb290, 0x26f8, 0x4398, 0xbff0, 0x41e6, 0x230e, 0x4173, 0xba5c, 0x077b, 0x4202, 0x40ff, 0xb2f9, 0x4137, 0xba2e, 0xbf90, 0xb08f, 0xb052, 0x42fe, 0xbf9a, 0xb046, 0xb023, 0x4348, 0x286e, 0x1c5b, 0x40af, 0x1c83, 0x41ae, 0xb2d7, 0x4198, 0xbafd, 0x42fd, 0x43ae, 0x3637, 0x30a2, 0x42e9, 0x34b2, 0x409b, 0x1fe7, 0x07b7, 0x19ea, 0xbf46, 0x2813, 0x1bb9, 0x42d9, 0x433b, 0xa211, 0xb24a, 0x24a3, 0xa199, 0x1fde, 0xb2c6, 0x4337, 0xbfce, 0x2ee1, 0x3710, 0xb20a, 0x41d4, 0x4026, 0x4361, 0x0eaf, 0x2d39, 0x1542, 0x04c4, 0x4361, 0x419b, 0x40e2, 0x417d, 0x4060, 0x1f09, 0x3473, 0x46ab, 0x430c, 0x400c, 0xb2d7, 0x4334, 0x4054, 0x0f0c, 0x188e, 0xbf92, 0xbaed, 0xb299, 0x4381, 0x0ac1, 0x0c56, 0x4267, 0x401f, 0x33e1, 0x4304, 0x3db4, 0xb0a2, 0x4635, 0x41f5, 0xbf92, 0x3fad, 0x4115, 0x1cf4, 0xb2bb, 0x4229, 0x437c, 0x4097, 0x412e, 0xa0b9, 0x4356, 0xb2ca, 0x43b0, 0x42f0, 0xba39, 0x4348, 0x4263, 0x4335, 0x42dc, 0x44f0, 0x42c6, 0xaa4f, 0x42bc, 0x43ed, 0xbf87, 0xb03d, 0x41cc, 0x1a30, 0x415e, 0xb257, 0x0f62, 0x4469, 0x4051, 0xa912, 0x4303, 0x4214, 0xb07d, 0x4067, 0xb264, 0x1dbe, 0xbf97, 0x41b8, 0x415c, 0x42d6, 0x414d, 0x13ac, 0x401c, 0x40ed, 0x25d9, 0x1883, 0x41e9, 0x410b, 0x1a38, 0xbfaa, 0x19e7, 0x19f6, 0x19db, 0x4173, 0x41a5, 0x17bc, 0x412d, 0x44d9, 0x4010, 0x1a2b, 0xbaf4, 0x393a, 0xba48, 0xb22d, 0xb0f5, 0xbaf2, 0xa232, 0xba3e, 0x41ce, 0x22dd, 0xbf8c, 0xb21b, 0x1ecd, 0x43a2, 0xba6b, 0x4044, 0x401e, 0x404b, 0x1854, 0xaafa, 0x4120, 0xa640, 0x43f7, 0x40c6, 0xb037, 0xa95e, 0xb2c5, 0xa3a2, 0x41f0, 0xb0e9, 0x4250, 0x4233, 0x438a, 0x401c, 0x46c5, 0x42ca, 0x4366, 0xbfd3, 0xba2b, 0x4029, 0x4060, 0x400f, 0x36eb, 0x404d, 0x400b, 0x171b, 0xb0c3, 0x4027, 0xb2eb, 0x0c68, 0xb04c, 0xa245, 0xb252, 0xb252, 0xb2e1, 0x414a, 0xbf0d, 0x426e, 0x1efb, 0xb2ba, 0x1865, 0xbafc, 0xb005, 0x41da, 0x40ce, 0x3cbd, 0x043e, 0x404c, 0x3d77, 0x4685, 0x1b3d, 0x2daa, 0xa0d8, 0xb2e3, 0x43b3, 0x4181, 0x2bba, 0x4316, 0x41dd, 0x4197, 0xbfc1, 0x0e0d, 0x430f, 0x11cd, 0x3a11, 0x413a, 0xb2b7, 0x4013, 0x409d, 0xb209, 0x281f, 0xba08, 0x41f6, 0xb024, 0x428b, 0xbafb, 0x42eb, 0x40c5, 0x11a0, 0x1876, 0xb009, 0x4004, 0xb224, 0x4395, 0xb04c, 0x3b7a, 0xbf01, 0x439b, 0xb247, 0x4162, 0x2a56, 0x1a27, 0x1d9a, 0x4065, 0xbf11, 0x05ff, 0xb25a, 0x1c63, 0xba4c, 0x3bdb, 0x40da, 0x33b3, 0x36fc, 0xbf7e, 0x41c5, 0xb0e1, 0x408f, 0x43a6, 0x187d, 0x4080, 0x009a, 0x41a1, 0x29d0, 0x41ad, 0x1c45, 0x37ad, 0x3de5, 0x1d62, 0x09f7, 0x402d, 0x43e5, 0xba78, 0xb221, 0xbad6, 0x1e4b, 0x289e, 0x43f0, 0x1bee, 0xb2dd, 0x43cc, 0xbfc2, 0x1d46, 0x40c2, 0xaaf5, 0x171e, 0xbfdf, 0x419c, 0xb2e8, 0xb20d, 0xbad1, 0xbadf, 0x4581, 0x27ba, 0x406b, 0x1a29, 0x436a, 0x466d, 0x121a, 0xb2e4, 0x4220, 0xba0f, 0x19d6, 0x1950, 0xb2ff, 0xbfb0, 0x1a2c, 0xb215, 0x4162, 0xbfcd, 0x05c0, 0x42cc, 0x40f1, 0xb2da, 0x05a2, 0x2929, 0xb0c1, 0xbf14, 0x4381, 0xbad8, 0x4480, 0xb0b6, 0x192e, 0x1f16, 0xba09, 0xbfc0, 0x4313, 0xb279, 0x40e9, 0x41af, 0x4233, 0x45e4, 0x4238, 0xb248, 0xb2a8, 0xb0f9, 0x3043, 0x4107, 0x070c, 0xba6a, 0x25de, 0x4212, 0x19d1, 0x41a4, 0xa17f, 0xbf36, 0x41fe, 0x08d3, 0x405f, 0xbff0, 0x41b2, 0x43e3, 0x4182, 0x08f4, 0x4293, 0x4267, 0x4030, 0x1822, 0xb2d6, 0x0575, 0x2c57, 0xb250, 0x166f, 0xaa81, 0x1c30, 0x4205, 0xab93, 0xbf60, 0xac46, 0x4328, 0xbf79, 0xb201, 0x33c5, 0x2c7f, 0x410c, 0x424e, 0x2676, 0xb0ce, 0x1f23, 0xbaf2, 0xb043, 0x392d, 0xb019, 0x04fb, 0x4227, 0xac6a, 0x27c0, 0x42a5, 0xba24, 0xbaf8, 0x4566, 0xb04f, 0x4299, 0x1cc3, 0x06e8, 0xb249, 0xbf00, 0xb0ba, 0x1be0, 0xbf96, 0x1251, 0xb2da, 0x3640, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x32016fcd, 0xf25a55e0, 0xc71b8a9f, 0x506131e5, 0xb5f08c7b, 0x8bec972b, 0x7ac97655, 0x7a33d75d, 0x0edf0fa3, 0x7cb7ebe2, 0x7905d5a9, 0x340efd29, 0x6029b34a, 0xfe1edb24, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0xef04ff40, 0x00000013, 0x00000003, 0xffffc003, 0xef050000, 0x08000000, 0x00000076, 0x000000c0, 0xe86c0001, 0x61d69c09, 0x29ccbeb4, 0xffffcd3f, 0x63b7d8f7, 0x0000049b, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1e27, 0x1948, 0x1be1, 0x43de, 0x0cb5, 0x25a1, 0x4599, 0x0a02, 0xb22d, 0xb224, 0xba71, 0x420e, 0x1e8e, 0xb09a, 0xb2db, 0x40d0, 0xbf5f, 0x3342, 0x1e2b, 0x42b0, 0x43a6, 0xb2ab, 0x43dc, 0xaad3, 0x186c, 0x1b0b, 0xa519, 0xa22e, 0x2d3c, 0x1bde, 0x138c, 0x42ff, 0x4225, 0xba3f, 0xa44f, 0x4009, 0x2ad5, 0x417b, 0xb262, 0x43db, 0x41c3, 0x436b, 0xba2c, 0x40c9, 0x42ee, 0xba09, 0xbf75, 0x408d, 0x2dc2, 0xb27b, 0xb09e, 0x4468, 0x40da, 0x334e, 0x42f9, 0x40c7, 0x414b, 0x42b9, 0x4476, 0x40b3, 0x41b8, 0x4267, 0x41ac, 0xb25c, 0x4095, 0xbfcc, 0xb2e8, 0xb249, 0x1942, 0x4156, 0x1fae, 0xb27f, 0xb2ca, 0xbf60, 0x14ca, 0x408b, 0x12db, 0xb202, 0x40cf, 0x4037, 0xb2a7, 0x43f7, 0x41c0, 0xbf8a, 0x4340, 0xba72, 0x0994, 0x1910, 0x4442, 0xb073, 0x4351, 0x41fe, 0xbac0, 0x4415, 0x1f23, 0xb0ba, 0x3850, 0x4334, 0xbf75, 0x40bb, 0x1e9f, 0x40f5, 0x415c, 0xb046, 0x0819, 0x4222, 0xaae3, 0x3564, 0xb2ca, 0xba1d, 0x0578, 0xb26f, 0x4392, 0x127c, 0xa13e, 0x4307, 0xbfc5, 0x38f9, 0x40ae, 0x4013, 0xb24a, 0x422f, 0x436e, 0xb23a, 0x403c, 0x1d4e, 0xad67, 0xb28e, 0xbaf9, 0xba44, 0xb25d, 0x43be, 0x1ba1, 0x402e, 0x43c7, 0xba0e, 0xba5e, 0xbfad, 0x43ec, 0x42c4, 0x41f0, 0x1e15, 0x1ada, 0x42e2, 0xa13f, 0x086d, 0x4427, 0x4249, 0x416e, 0xb29c, 0x4388, 0x42c6, 0x42a9, 0x43e5, 0x41f1, 0x4024, 0x44fb, 0x40ba, 0xba65, 0x4143, 0x2a73, 0x21c3, 0x401d, 0x0307, 0xbf52, 0x2b55, 0x4412, 0xaf31, 0xbf99, 0x4374, 0x41c6, 0x4194, 0x187d, 0x03ca, 0x42a7, 0xbaca, 0x1c5c, 0x44a9, 0xba03, 0x1951, 0xaa90, 0xba3d, 0x0779, 0x1c92, 0xbf64, 0x1f8a, 0x3e68, 0xb2e4, 0xb078, 0x0aac, 0x1c35, 0x4105, 0x1fd2, 0x4260, 0x4119, 0x3a45, 0x43e5, 0x42e8, 0xa547, 0xb277, 0x3210, 0x2663, 0x4271, 0xb228, 0x4242, 0x4671, 0x40b7, 0x4328, 0xba4a, 0xbf22, 0xba26, 0x3a73, 0xad7d, 0x1ef1, 0x4114, 0xbf00, 0x420a, 0x4287, 0x40c6, 0xa529, 0x1c0e, 0x43c5, 0x0179, 0x305b, 0x4323, 0x1e26, 0x4261, 0x0a63, 0x45ae, 0x404d, 0x2fd0, 0x4341, 0xb0fb, 0xbfd5, 0x2fc4, 0x1f1e, 0x42d8, 0xbac4, 0xbae0, 0xb2ea, 0x281e, 0x17d1, 0x18fe, 0x4243, 0x4265, 0xb278, 0x4193, 0xaccc, 0x41e2, 0x4129, 0x4111, 0x41e7, 0xb02b, 0x42b1, 0xb252, 0x405e, 0x2ea6, 0x4355, 0xb29d, 0x408d, 0x4272, 0xbfbe, 0x4301, 0x425c, 0x43ef, 0x1be2, 0x1c16, 0x2a79, 0x3f7d, 0x446c, 0x1c40, 0x1890, 0xb20f, 0x19bc, 0x3f0f, 0x4109, 0xba4a, 0x1f0f, 0xb2b4, 0xba7e, 0x418a, 0x43e7, 0x094e, 0x4367, 0xaaf4, 0xbade, 0xbac4, 0x4082, 0xbfa4, 0x3bab, 0x4172, 0xb27e, 0x1bb3, 0x16c0, 0xb217, 0x408d, 0xbad1, 0xb202, 0x0c8a, 0x41d6, 0x432f, 0x1f7a, 0x1ab8, 0x43aa, 0xbae5, 0xbf8a, 0x4342, 0x0d85, 0x2184, 0xb2d6, 0x24ec, 0x4281, 0x4393, 0xbfba, 0x4542, 0x4368, 0xb05f, 0xb22b, 0xa6d1, 0x2917, 0x4071, 0x427a, 0xb2d9, 0x0c5b, 0xb0e2, 0xba75, 0x410d, 0x3696, 0x4078, 0x4203, 0xb2da, 0xb225, 0x4073, 0x408e, 0x43ec, 0x4335, 0x10a4, 0x33f6, 0x402e, 0x4264, 0xb240, 0xbfcb, 0x4314, 0x1b15, 0x41d7, 0xb0f6, 0x0643, 0xba6d, 0x4073, 0xad70, 0xbad7, 0x4646, 0x4275, 0xae3f, 0x46d5, 0xbf35, 0x4390, 0x0321, 0x06c6, 0x4219, 0x02fe, 0x4243, 0x41dd, 0x1e18, 0xb226, 0xbf67, 0x432d, 0x15c5, 0xb273, 0x4043, 0x1fb3, 0xb28e, 0x1df2, 0x42f4, 0x187f, 0xbfc2, 0xba73, 0x4356, 0x3703, 0xbfb2, 0x423d, 0x2afd, 0x412e, 0x42ff, 0xba54, 0xba15, 0xba18, 0x186a, 0xbf70, 0xba3e, 0x43bb, 0x0d94, 0x0940, 0xb015, 0x2f56, 0x4196, 0x4227, 0x437d, 0x40da, 0xb261, 0x4312, 0x4061, 0xbf32, 0x0d8e, 0x4203, 0x4308, 0xbac4, 0x437a, 0x4074, 0xbf5d, 0x427b, 0xb2c7, 0x43e4, 0x39b8, 0xb272, 0x42d4, 0x4183, 0x423e, 0x44ba, 0x428a, 0x2b49, 0x463f, 0x410a, 0x0679, 0x4194, 0xb0dd, 0x41d8, 0x43c4, 0x0a84, 0xbfd7, 0x3b2c, 0xb285, 0x41db, 0x40e2, 0x0e78, 0x43df, 0x4212, 0xbf00, 0x4072, 0x433f, 0xb26b, 0x4389, 0x431a, 0x1890, 0x4264, 0x1f58, 0xbf8a, 0x46d8, 0xa682, 0x1b9e, 0x41d6, 0xbfa2, 0x438e, 0x42f7, 0x46b9, 0x416d, 0xba21, 0xb015, 0x428c, 0x1135, 0x4221, 0x42eb, 0x2cbd, 0x40a3, 0xb033, 0xb2de, 0x4562, 0x43c1, 0x433d, 0x406a, 0xb273, 0x0ddf, 0x104a, 0x440a, 0x2b0d, 0x40ff, 0x40af, 0x43d5, 0xbf42, 0xbacd, 0x143b, 0x43f2, 0x4267, 0xbf7f, 0xb009, 0x1925, 0xb054, 0x1c76, 0x0fb2, 0xb2bd, 0x42cd, 0x43f9, 0x40b1, 0x42af, 0x40df, 0x42fb, 0x421a, 0x4259, 0x311c, 0x4057, 0xba52, 0xbf9f, 0x1f33, 0xba70, 0x44c3, 0xbae6, 0x3643, 0xbaf1, 0x4242, 0x00ff, 0x43ba, 0xb01b, 0x40c8, 0x4551, 0xb26b, 0x14dd, 0x43cd, 0xbfa8, 0xb29f, 0x31ff, 0xb2cd, 0xb2b9, 0x43ce, 0xa0da, 0x46e8, 0x43cd, 0x42a5, 0x4346, 0x431b, 0xba29, 0xb229, 0xbf44, 0x1ed8, 0x42a0, 0x42ed, 0xb067, 0x4217, 0x418a, 0xb020, 0x0ba2, 0xb25e, 0x1498, 0x238f, 0x1f2c, 0x1c97, 0xbf53, 0x41b2, 0x19de, 0x41ae, 0xb050, 0xbf25, 0x4139, 0x41dd, 0x4378, 0xa7a1, 0x1a9e, 0x0941, 0xb22f, 0x2449, 0x4048, 0xba34, 0xb2dd, 0xb2b3, 0xbfa7, 0x438d, 0xba40, 0x03a0, 0x0a6d, 0x11d7, 0x2d4f, 0xb2b8, 0x43c7, 0xb010, 0xb04e, 0x4335, 0xb233, 0x0886, 0x184c, 0x45a9, 0x42a7, 0x42cf, 0x1a82, 0x0bf2, 0x043d, 0x40e7, 0x428b, 0xbf3f, 0x3742, 0x1ac4, 0x4112, 0xbfd0, 0x41db, 0x4128, 0xab3c, 0xbaee, 0x4108, 0xba1b, 0x4298, 0x296b, 0x405a, 0xacd9, 0x4439, 0x392f, 0xbad4, 0x46d5, 0x4349, 0xbfa5, 0x18bd, 0x15c3, 0x4124, 0x1e91, 0xba64, 0xb26b, 0x45a9, 0x1e72, 0xb2e1, 0xa04c, 0x0fa8, 0x418c, 0xabfa, 0xaafd, 0x420f, 0x243b, 0xb0d6, 0xbfa0, 0x4485, 0x4360, 0x41f9, 0x418e, 0x41b2, 0xbf7c, 0xb22a, 0x421e, 0xaa51, 0x40cd, 0x41b0, 0xbfbb, 0x1e7f, 0x4224, 0x43be, 0x420a, 0xbfe1, 0x46d3, 0x4364, 0x427e, 0x404a, 0x422b, 0x3a5f, 0x4280, 0xba77, 0xb0ee, 0x41e9, 0xbfca, 0x4230, 0x4264, 0x2b53, 0xba07, 0x4136, 0xba67, 0xbfd7, 0xb0d5, 0x059a, 0x406a, 0x4216, 0x43d9, 0x3325, 0xa91c, 0xb048, 0x415c, 0xb20e, 0xb2c7, 0xba6f, 0x18d9, 0xba46, 0x1abb, 0x433a, 0x41d0, 0xbf55, 0x4300, 0x4030, 0x412b, 0x37b3, 0xbfd0, 0xa03a, 0x40c0, 0x05ba, 0xb26d, 0x437e, 0x1049, 0x43c6, 0xb00f, 0x44d0, 0x441b, 0x31f0, 0x40a5, 0x199e, 0x40ec, 0xbfa5, 0x1983, 0x4377, 0x40de, 0x2838, 0x1b98, 0x1385, 0x409b, 0x417a, 0x460c, 0x436d, 0x3510, 0xba26, 0x3883, 0x4110, 0x4495, 0x409e, 0xbfae, 0xb297, 0x4678, 0x143a, 0x40a3, 0xb221, 0x333b, 0x1f02, 0xb031, 0x1d7a, 0x4061, 0x4138, 0x41dd, 0xb2dc, 0xb2ed, 0xba39, 0xbf28, 0xb2e8, 0x3fbf, 0x411c, 0x3271, 0x14a9, 0x414f, 0xa624, 0x1d2c, 0xbfa0, 0x4195, 0x4361, 0xb234, 0xb052, 0x4097, 0x43be, 0x4083, 0xba41, 0xbf34, 0xad4e, 0x4109, 0xb256, 0xb2e5, 0x39b3, 0x43ae, 0xb2e3, 0x2479, 0x40ba, 0xb229, 0x40d2, 0xbf7f, 0x3365, 0x41ba, 0xba78, 0x1b0c, 0x40db, 0x4358, 0xa8af, 0x42d7, 0x4573, 0xb29d, 0x15fb, 0xb00a, 0xb2ee, 0xb25c, 0xbfc1, 0x428b, 0x0d95, 0x4121, 0x43d1, 0x424c, 0x2cb8, 0xbf4c, 0xa326, 0x42b0, 0x14f3, 0x43eb, 0x379b, 0x41b8, 0xba3d, 0x1d51, 0x42c0, 0x1012, 0x414d, 0x221b, 0x434d, 0x38f4, 0x19d3, 0xb22e, 0xbfcb, 0x41d2, 0x4124, 0x3cb4, 0x1be3, 0x40ae, 0xba40, 0xb0a4, 0x1bec, 0xbac0, 0x41da, 0x43a7, 0xbf21, 0x41f5, 0x406f, 0xb090, 0x2a79, 0x4314, 0x1e36, 0x1a8e, 0x4163, 0x336b, 0xb0c0, 0x43e4, 0x4266, 0x4312, 0x1e45, 0x42ba, 0x4215, 0x0214, 0xa1c8, 0x4032, 0x43ea, 0xba10, 0x43c3, 0x41cd, 0x46e5, 0x3272, 0x21ec, 0x26d0, 0xbf2f, 0x4266, 0xba19, 0x419e, 0x3453, 0xb2b1, 0xb21b, 0xb2ad, 0x400a, 0x40f5, 0x1c18, 0x433e, 0x40a5, 0x1e4e, 0xb0ec, 0x4354, 0x3d65, 0xaac5, 0xba57, 0x4649, 0x42ab, 0x4123, 0x4434, 0x43eb, 0x430c, 0x2549, 0x0da8, 0x4309, 0x4119, 0xbfa3, 0x4380, 0x24cb, 0x410d, 0x43da, 0x431a, 0x404c, 0xb008, 0x41f2, 0x4588, 0x4112, 0x4572, 0xb231, 0x4196, 0x14a7, 0xa3f8, 0x407d, 0x42a1, 0x4074, 0xbf9a, 0x40ef, 0x4450, 0xb2a4, 0x463d, 0xb0cb, 0xbfa0, 0x4141, 0x316d, 0x19f4, 0x1273, 0x4433, 0x3c10, 0x4228, 0x441e, 0xbfd6, 0x42bc, 0x4088, 0x430e, 0xb01d, 0x1fb4, 0x4054, 0xb287, 0x40ce, 0x4012, 0xba62, 0xb26d, 0x3cfc, 0x43d4, 0x2128, 0xba0d, 0x1837, 0x460b, 0x1add, 0xba23, 0x1abf, 0x437f, 0xbf2d, 0x43bb, 0x407e, 0xb09e, 0x428d, 0x4263, 0x18d9, 0x2f7a, 0x405d, 0x406b, 0x4293, 0xb251, 0x4151, 0x4300, 0x4383, 0xb240, 0xb2f2, 0x2fc6, 0x2bc6, 0xb234, 0xba56, 0xbf17, 0xba48, 0x41e6, 0xba28, 0x4035, 0x4324, 0xb050, 0x43e3, 0x1fb1, 0x42ab, 0x42d4, 0x0549, 0x456b, 0x41d0, 0xb0c4, 0xb07a, 0x42b5, 0x2d57, 0xba52, 0x40c0, 0x43f1, 0xb288, 0xb2db, 0xb231, 0xba37, 0xb2fd, 0x11f8, 0x416d, 0xbfa4, 0x41de, 0x3da7, 0x4157, 0x3ddb, 0x4083, 0x40a4, 0x43ea, 0xb004, 0x295b, 0x416b, 0xbfb6, 0x4695, 0xb275, 0x41db, 0x19b8, 0xb2ad, 0x402e, 0xbf7b, 0x44a4, 0xbfa0, 0x4035, 0x404c, 0xb034, 0x41e9, 0x46f0, 0xbfb2, 0x0925, 0xb022, 0xb0df, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3ca5ddec, 0x567ac17b, 0x5090d856, 0x31dc27ca, 0xce87e6ed, 0x44ad92e1, 0xe67fdf6b, 0xfc9f1f92, 0x0b5c7af3, 0x2abf17e7, 0xcbcf4b4e, 0xcccdc713, 0x7e5f0f46, 0x63825539, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x00048400, 0x00001000, 0x00000181, 0xffffdebf, 0x00060440, 0x0000fe7e, 0x00008000, 0x00040400, 0x00000000, 0xfffffbe7, 0xcbcf4b30, 0xcbcf4b30, 0x7e650f86, 0x0000015d, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x3a7c, 0x0655, 0x41a9, 0x3921, 0x4147, 0x429a, 0x13d5, 0x415e, 0x43ad, 0x373d, 0xb2e7, 0xa82f, 0x45b2, 0xb299, 0x4174, 0xbf15, 0x1155, 0xba32, 0x3a9c, 0x4037, 0x437e, 0xb01c, 0x4245, 0xbf92, 0x4029, 0x4111, 0x3fd8, 0x4299, 0x1b95, 0x404a, 0x4057, 0x4315, 0x4047, 0x423f, 0xaf03, 0xb27d, 0x22a0, 0xb012, 0xa3cf, 0x40d9, 0x40c6, 0x0e90, 0x4694, 0x40fd, 0x418b, 0x03c9, 0x44e0, 0xa932, 0x4397, 0xba5b, 0xbf3d, 0x4223, 0x1ff6, 0xb296, 0x4322, 0x28dc, 0x32c9, 0x4108, 0xbfdb, 0x433f, 0xb249, 0xb00c, 0x1d27, 0x46cd, 0x1aa1, 0xba2b, 0x1c1d, 0xa28d, 0xbfd5, 0x0057, 0xbac6, 0x4308, 0x1a2a, 0x42d1, 0x18ff, 0x432f, 0xbae0, 0x415c, 0xb032, 0x3c6b, 0x43e3, 0xb265, 0x42a1, 0x4013, 0x430f, 0x1c8c, 0x1fa1, 0x33f7, 0x4168, 0xbf21, 0x2136, 0x431e, 0x422d, 0xbae0, 0x432d, 0xb0ed, 0xb2d8, 0x0a2c, 0x41e8, 0xbf80, 0xbf88, 0xba28, 0xb2aa, 0xb267, 0x430e, 0xbf47, 0x4091, 0x40a5, 0xb2b3, 0x1b9e, 0x41e0, 0x420f, 0x1bfd, 0x1a1f, 0x1e8e, 0xbf7e, 0xbaeb, 0xa08a, 0x437e, 0x4339, 0xb006, 0x221f, 0x42b5, 0xb2c8, 0xb202, 0xbaef, 0x1902, 0x4161, 0x413c, 0x42b3, 0x18ba, 0x2f5c, 0xb252, 0xba3d, 0x4166, 0x43ee, 0x438d, 0xbf8d, 0x46e2, 0x0540, 0x303a, 0x43f5, 0x42a2, 0x4081, 0x34fa, 0x024c, 0x11d5, 0x2ac3, 0xbf00, 0xbfcc, 0x45c3, 0x439d, 0x4019, 0x401d, 0x40b2, 0x0891, 0xa357, 0x0a3d, 0xbf00, 0x0e7b, 0xb29d, 0xae6d, 0x438f, 0x4364, 0x4311, 0x42bd, 0x19ad, 0x44a4, 0x4624, 0xb2e5, 0x4134, 0x43e5, 0xbf16, 0x402d, 0x4391, 0xba51, 0x438f, 0x18f7, 0x455a, 0xafda, 0xbf6f, 0x4125, 0xbf60, 0x2490, 0xba2a, 0x038d, 0x410b, 0x42e6, 0x4224, 0x40b0, 0x1a97, 0xa5ee, 0x4073, 0xbf90, 0x4362, 0x1fbf, 0x4314, 0x4051, 0xb234, 0x4336, 0xbf60, 0xb046, 0xba05, 0xa6cc, 0xb2e1, 0x3022, 0xbfe0, 0x4608, 0xb2b0, 0xbaf1, 0xbfc3, 0x2620, 0x45b5, 0xbad8, 0x408d, 0x24c8, 0xb242, 0xb0f6, 0x4284, 0x4251, 0xaed4, 0x4302, 0x4197, 0xba77, 0x1678, 0x45a8, 0xba65, 0x4333, 0x1e84, 0x1af4, 0x4212, 0xbaff, 0xa4d9, 0xac32, 0xbfd0, 0x2306, 0x42ac, 0xbfaf, 0xb04c, 0xb285, 0x21dc, 0x42a6, 0x4121, 0xbfc0, 0x432f, 0xbf99, 0x3be7, 0xbf90, 0x40e6, 0xb288, 0xb27f, 0x4328, 0x245d, 0x41ef, 0x43bd, 0xb291, 0x40bd, 0xb0b5, 0xbf90, 0xbfe4, 0xb209, 0x1860, 0x4270, 0x42f8, 0xbfb0, 0x40c2, 0xbaeb, 0x09fb, 0x44f1, 0x41fc, 0x406a, 0x4134, 0xbfb0, 0x332a, 0xb26e, 0x0619, 0xba45, 0xbf89, 0x432c, 0xb2c3, 0xac47, 0x1d59, 0x2714, 0xb098, 0x433a, 0xa38a, 0x4573, 0x40b7, 0x41d6, 0x4180, 0xb259, 0xad42, 0x46c1, 0xb076, 0x406b, 0x4131, 0x40fb, 0x243f, 0x4285, 0x1b92, 0xb256, 0xa70f, 0x4168, 0x40bd, 0xb290, 0x12c5, 0xbfb8, 0x3068, 0x42d4, 0x410b, 0x1ef1, 0x4258, 0x0368, 0x46b5, 0x406b, 0x40aa, 0x4321, 0x1f9d, 0x4173, 0x430c, 0x43e5, 0x4026, 0xb2d8, 0xbf01, 0x3733, 0xbac9, 0x4275, 0x4276, 0x43d1, 0x1b8c, 0x4169, 0x4146, 0xa9aa, 0x4302, 0x1316, 0x4416, 0x43ca, 0x02f5, 0x42bb, 0xa7f7, 0xb204, 0x4632, 0x416c, 0xb030, 0x409a, 0x404b, 0x4385, 0x41cd, 0xb2b5, 0xb287, 0x4390, 0xbfce, 0x1b41, 0xb279, 0x1df6, 0x4220, 0xb020, 0x40b1, 0x45d3, 0x1d7b, 0x4162, 0x4234, 0x4201, 0x41cb, 0x30ba, 0x4039, 0xba5e, 0x4402, 0x429d, 0xba4e, 0x4652, 0xb2e8, 0x19cc, 0x38fd, 0x41b3, 0xbfd0, 0x43d6, 0xbf01, 0x43ee, 0x42c8, 0x07a7, 0x1893, 0x43c5, 0x4013, 0x1288, 0xba4a, 0x1534, 0x4091, 0x429b, 0x426b, 0xb234, 0xbf3a, 0x2bf6, 0x1e2f, 0xb28a, 0xa71a, 0xba32, 0x41e8, 0xbf65, 0x43c9, 0xacce, 0x403b, 0xb250, 0xaa0b, 0xabc5, 0xa194, 0xb214, 0x421d, 0x40cc, 0xa0e9, 0xb2d7, 0x1eff, 0x43d1, 0x417e, 0x42f1, 0x4275, 0xba1b, 0x3012, 0x13d4, 0x1fdd, 0x0af4, 0x4255, 0x4264, 0xbf9b, 0x1f56, 0x43e9, 0x418b, 0xbad9, 0x410a, 0x083d, 0x30af, 0xba48, 0x2c36, 0xbfdb, 0xb0fa, 0xb216, 0x3970, 0xba5f, 0x0556, 0x435a, 0x41d2, 0xb2db, 0x4395, 0xb020, 0xa2df, 0x43f7, 0x1dca, 0xb017, 0xba46, 0xb2aa, 0xbfa0, 0x07b6, 0x4040, 0xbf60, 0xa3c8, 0xbf69, 0xb232, 0x3ac1, 0x42bc, 0x44a3, 0xbadf, 0x0651, 0xb27d, 0x1f97, 0xbf93, 0x1625, 0xb2f8, 0xb2b8, 0x1244, 0xbf00, 0x41ac, 0xba03, 0x4030, 0x3d8e, 0x08b1, 0x41c7, 0xb04e, 0x3332, 0x1f10, 0x2903, 0x2272, 0x462e, 0xb204, 0xbf3f, 0xb2d0, 0x4375, 0x4204, 0x4062, 0x180c, 0x4084, 0xb298, 0x0fc2, 0xb270, 0x4085, 0xbf9b, 0x243f, 0x4390, 0x4221, 0x4445, 0x43e3, 0x45a9, 0x43d4, 0xbfdc, 0x402e, 0xbaec, 0xa8b2, 0x40bd, 0x2eb6, 0x4281, 0xb206, 0xb268, 0x30ef, 0x4043, 0x434e, 0x1a1d, 0x14c5, 0x43a9, 0x1fa9, 0x4376, 0x41bd, 0x423c, 0x43ab, 0x1aa7, 0x42d6, 0x0513, 0x4033, 0x31c5, 0x410f, 0x1db1, 0x40d3, 0xbf0c, 0x422c, 0x427b, 0x4019, 0x4066, 0x419e, 0x1cc6, 0x3571, 0xa6f0, 0x1813, 0x40d3, 0xb099, 0x035f, 0x19c3, 0x42ad, 0x42fb, 0x1d9e, 0xb25f, 0x1a90, 0xb28f, 0xbf3b, 0x3c2e, 0xba0a, 0xb067, 0xb2d9, 0x401c, 0x406e, 0x1b33, 0x4320, 0x43f7, 0x0df9, 0xbf41, 0x4166, 0x42ca, 0x43e7, 0xba7c, 0xb077, 0x1792, 0x43d3, 0x4085, 0xb27a, 0x4293, 0x403d, 0x410c, 0xbf08, 0x11e3, 0x41c2, 0x4083, 0x4364, 0xa113, 0x4010, 0x207c, 0x42aa, 0x1b60, 0x25fe, 0x4097, 0xb202, 0x40fc, 0xb07c, 0xbfdb, 0x041e, 0x40e1, 0x4046, 0x43e8, 0x42a3, 0x424a, 0xba73, 0x4002, 0x4350, 0xb2ae, 0x05a7, 0xb244, 0xb2a4, 0x19b4, 0xbf81, 0x4146, 0x431b, 0x4137, 0x46f8, 0xbfb4, 0x423f, 0xba3c, 0x1c8a, 0x40b6, 0x416f, 0xbade, 0x0565, 0x4027, 0xb2e7, 0x42a7, 0x0c0a, 0x4213, 0x40e5, 0x4058, 0xba68, 0x41eb, 0x4234, 0xbf84, 0x4333, 0xba18, 0x4073, 0x39be, 0x4339, 0x1e78, 0x42e2, 0x1396, 0xba2e, 0x4051, 0x3e8c, 0x40a6, 0xbf5e, 0xbaef, 0x36c9, 0x4060, 0xbfe0, 0x19fb, 0xbf35, 0x40b3, 0xba2d, 0x417a, 0x425b, 0x0bc2, 0xa2c7, 0x43ef, 0x4550, 0x427f, 0x402d, 0x18b3, 0x4104, 0xba2e, 0x1a9a, 0x4243, 0xbacd, 0x4116, 0xb2dc, 0xba6c, 0x410d, 0xbfad, 0x4189, 0x40fb, 0x437f, 0xba6b, 0x1023, 0x4063, 0xb2af, 0xaa18, 0x2514, 0x41c0, 0x32e8, 0x01c7, 0x35a9, 0xbfad, 0x4029, 0xb083, 0xba10, 0xb2be, 0x3276, 0xaf71, 0x4086, 0xb277, 0xbf6b, 0xb2d6, 0xb2f3, 0xb2ae, 0x19c3, 0x4366, 0x443b, 0x4124, 0x43bf, 0xa919, 0x434e, 0x12bf, 0x009f, 0x4006, 0xbff0, 0x43c5, 0xbf1a, 0xb0f6, 0x416d, 0x4114, 0x4624, 0x4161, 0x4089, 0x43b1, 0x4155, 0xbad4, 0xb000, 0xb042, 0x191e, 0x41c1, 0x4013, 0xb24f, 0xbf6c, 0x1dee, 0xb294, 0x1ce2, 0x435d, 0x41b1, 0xad83, 0x11c0, 0x40ce, 0x317f, 0x43c6, 0xb21a, 0xb002, 0xb298, 0x41ae, 0xb20f, 0x05ef, 0xb2a6, 0xb2fe, 0x1ad7, 0x45a0, 0x030c, 0x4089, 0xb26b, 0x4088, 0x4079, 0x3dc9, 0xbf87, 0x42bb, 0x1efc, 0x4018, 0xb254, 0x40ce, 0x3abc, 0xb0be, 0x4017, 0x1890, 0x414e, 0x4670, 0x1c83, 0xbf70, 0xbf41, 0xb277, 0x4283, 0x42c1, 0x41b9, 0x4006, 0xb260, 0x40c6, 0x455f, 0x4169, 0xb09d, 0x19ba, 0x263d, 0x40f9, 0xb07c, 0x41f8, 0x41c3, 0xbf97, 0x4031, 0x40a4, 0x1d03, 0xb21e, 0x4051, 0x003a, 0x4191, 0x409f, 0xb240, 0xbfd0, 0x0eaa, 0x43c4, 0x4262, 0x112b, 0x4264, 0xba2c, 0x4239, 0xbfc1, 0x43d7, 0x422e, 0xb291, 0x323a, 0xbae2, 0xb22f, 0xb0da, 0xb26e, 0x1a25, 0x43e1, 0x4208, 0x407c, 0x1d1e, 0xb27e, 0x1c0c, 0x0edc, 0x43ae, 0x4064, 0x459c, 0xa866, 0xb06f, 0x437a, 0x411e, 0x0a15, 0x1aeb, 0xb228, 0xb21b, 0x426f, 0x42f6, 0xbf34, 0x44bc, 0x016d, 0xac00, 0xb2e4, 0xbafc, 0x41d6, 0x44eb, 0x4063, 0x41bd, 0x409d, 0x39be, 0x438f, 0x45d0, 0x172e, 0x45bb, 0xbf6e, 0xb28f, 0x4262, 0xb2b8, 0x1c01, 0x45c4, 0x188e, 0x432d, 0x3671, 0x4039, 0x4450, 0x0adc, 0x44eb, 0xbf2f, 0xbaec, 0xbfa0, 0x0c02, 0x4293, 0x43c3, 0x19b8, 0xb251, 0x22f4, 0x1f80, 0xba39, 0x42c6, 0x1caa, 0xa3ae, 0x3100, 0x41dc, 0x4252, 0xb00b, 0xba1e, 0x421b, 0x437e, 0x1770, 0x4094, 0xa2d9, 0x1b0d, 0x148d, 0x4328, 0x4569, 0xbf46, 0x4283, 0x43a3, 0x3399, 0x3025, 0x40ce, 0xb284, 0x432a, 0x03d0, 0xbf90, 0xbafd, 0xbfcb, 0x43ad, 0x4013, 0xba27, 0x3d2a, 0x466f, 0x1653, 0xb204, 0x225c, 0xb213, 0x21e4, 0x43d3, 0x0eb4, 0xbae8, 0x415e, 0x4244, 0x4239, 0x4692, 0x2014, 0x411f, 0x1dfb, 0x42a8, 0xbfb6, 0x42f3, 0x1138, 0x22d0, 0xbfbf, 0xbacc, 0x0049, 0xbfa0, 0x4315, 0xba7a, 0x4182, 0x142f, 0x42b7, 0xbaf4, 0x465e, 0x4030, 0xb21d, 0x1a68, 0xb240, 0x2a11, 0xa788, 0x4088, 0x373f, 0xbfc9, 0x4347, 0x4088, 0x40ad, 0xb24d, 0xba22, 0xba5b, 0xb2f6, 0x4294, 0x406d, 0x428c, 0x1e96, 0xba49, 0xb0bc, 0x41a1, 0x4298, 0xb099, 0x410b, 0xbf38, 0xb21e, 0x43a8, 0x41d9, 0x4316, 0x4308, 0xbf07, 0xba54, 0x402a, 0x045d, 0x4356, 0x4077, 0x1573, 0xb208, 0x4069, 0x4100, 0x42f4, 0xaffb, 0x284f, 0xae93, 0x4348, 0x33ec, 0xb2b1, 0x41ce, 0x40a1, 0x1683, 0x427f, 0x42c6, 0x18f3, 0xb29e, 0xb2c0, 0x3d0f, 0xba15, 0x4116, 0xbf12, 0x43bf, 0x1c9d, 0xbae6, 0x4167, 0x42e5, 0x40ff, 0x404f, 0x1a21, 0x4028, 0x4289, 0x438d, 0xbfa3, 0x182c, 0x1cd9, 0x4304, 0x45a0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x916cb869, 0x845ddb9e, 0xc71efbae, 0xb5b0ddd7, 0xc96a166f, 0x0896734c, 0xe975a539, 0xda33958c, 0xf0631e50, 0x552ee4b2, 0x94f77f64, 0x1bf67a8f, 0x4cd88fc1, 0xd2a84a20, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x0990000d, 0xffa3ffff, 0x0990000a, 0x00000000, 0x00000000, 0xffffffa3, 0x00000000, 0xf0631ef0, 0xf0631ef0, 0x0000005c, 0x1bf68b67, 0x000000a0, 0x00000744, 0x00000000, 0x600001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1e86, 0x39ce, 0xb015, 0x4241, 0xba7a, 0x0ca5, 0xba1c, 0xb2c7, 0x1ade, 0x40e0, 0xbfc0, 0x18d9, 0x4195, 0x43e7, 0x144f, 0x1aa8, 0x402c, 0x19e3, 0x0f08, 0xbf33, 0x0146, 0x0896, 0x4298, 0x4011, 0x40b9, 0x4225, 0xae23, 0xa371, 0xb2ea, 0xb293, 0xaa59, 0xbacb, 0xb203, 0x40e4, 0x42a3, 0x158d, 0x42d3, 0x4435, 0xa63f, 0x1cb7, 0xae3d, 0x1b9c, 0x1ae6, 0x43a5, 0x4682, 0x38c3, 0x4199, 0x40e3, 0x4462, 0xbfd6, 0x1737, 0x4034, 0x413d, 0x1491, 0x40bd, 0x40d4, 0x1b2a, 0xbf9f, 0xbafa, 0xb2e1, 0x435b, 0x4057, 0x414d, 0x43c2, 0x46b2, 0x4247, 0x41a7, 0x422e, 0x4082, 0xb207, 0x1a3a, 0x1515, 0x301f, 0x42a4, 0x391a, 0x41af, 0x4102, 0xb246, 0xbf1c, 0xb2a7, 0x1fcb, 0xbad6, 0x41e6, 0x290a, 0x42c7, 0x1a97, 0x4616, 0x42f6, 0x1668, 0x4144, 0xb229, 0x43de, 0x1abb, 0xaeb6, 0x4658, 0xb2fd, 0xbf80, 0xb027, 0x4068, 0xac56, 0x0b56, 0xba3e, 0xb0ae, 0x2699, 0x4175, 0xbf39, 0x4367, 0x39a9, 0x4377, 0x3b13, 0x0504, 0x41cb, 0x43fe, 0x43a9, 0xb037, 0x4226, 0x4090, 0xb219, 0x4236, 0x4043, 0x435b, 0x163b, 0x41b5, 0x42ff, 0x03d5, 0xbf4e, 0xbad9, 0x42d2, 0x4063, 0x18c2, 0x4441, 0x1dac, 0xb287, 0x40b8, 0x4007, 0xb28d, 0x41a5, 0x4230, 0xbfa0, 0x419f, 0x0084, 0x4042, 0x1fe7, 0x42e3, 0x03a8, 0x2f05, 0x4641, 0xba39, 0xb266, 0x42cd, 0xb0f9, 0xb012, 0x4066, 0xb276, 0xbf3c, 0xb2b6, 0x197f, 0x4091, 0x1871, 0xbf11, 0xba70, 0xa8b5, 0x404b, 0x1bce, 0x43bb, 0x3ac6, 0xb021, 0xb2f4, 0x41f0, 0x4402, 0x2c5a, 0x45cc, 0x437c, 0xb023, 0xb23c, 0x37b9, 0x412d, 0x42b2, 0x43a1, 0xb089, 0x417f, 0x43ad, 0x4220, 0x42b4, 0x4176, 0x2e55, 0xb228, 0xbf0d, 0x3038, 0x4315, 0x4302, 0xba33, 0x43e0, 0x43cd, 0x0765, 0xbfc3, 0x41b6, 0x1bbb, 0x4007, 0xafba, 0x1c38, 0xb25e, 0xb2f2, 0x1ad5, 0x4196, 0xba2c, 0x40b4, 0x1a04, 0xb01a, 0xba71, 0xb2a8, 0x4685, 0xb0b2, 0xb20c, 0xba54, 0x46c5, 0xbf86, 0xa737, 0x4309, 0xb230, 0x03e0, 0xb282, 0x45ae, 0xb293, 0x40d7, 0x40c5, 0x40da, 0xb2fc, 0x40fb, 0x42c4, 0xb217, 0x367a, 0x41bd, 0xba6f, 0xba59, 0xba7a, 0x08b7, 0xbf67, 0x13e5, 0x408c, 0x12cf, 0xa5fd, 0x44ad, 0x34f1, 0xb2eb, 0x407c, 0x2b7a, 0xbfe0, 0x4481, 0xb286, 0x18f7, 0x42c5, 0x1fa6, 0x41b2, 0x42de, 0x40a0, 0x2f21, 0x4090, 0x4309, 0x4071, 0x3c32, 0x4203, 0x42f0, 0xba32, 0x41d8, 0x1daa, 0xbfbf, 0x402f, 0xb2d2, 0x435b, 0x4158, 0x407e, 0x20ff, 0xba0a, 0x1a6a, 0xba33, 0x4093, 0x43f0, 0xbf57, 0x43f9, 0xb2a1, 0x432f, 0xb2f5, 0x41cc, 0x4244, 0x1dc6, 0xb2b1, 0xba27, 0xbfa7, 0x347c, 0xb042, 0xba06, 0x41b2, 0xbaeb, 0x199f, 0xbfa7, 0x432f, 0xb2dd, 0x1a37, 0xb2dc, 0x434d, 0xbac5, 0xbaf8, 0x420e, 0x3f3f, 0xb0b2, 0x4005, 0x2b8a, 0x1c8d, 0x46e8, 0xade4, 0x425b, 0x4152, 0x0240, 0x4175, 0x0d64, 0x449a, 0x4018, 0x4366, 0xb2d5, 0x226b, 0x40ae, 0xb20f, 0xbf57, 0xa2c7, 0x42ac, 0xaf3b, 0x461f, 0xba67, 0x2af4, 0x4230, 0x4087, 0x1543, 0x43a9, 0x4232, 0x432e, 0xba6b, 0x46da, 0xba31, 0xbacb, 0x431a, 0x4343, 0xbfa9, 0x40c7, 0x18fb, 0xb2c6, 0x4066, 0xbf13, 0x4399, 0x4480, 0x3d8f, 0x111f, 0xbf1f, 0xbad3, 0x42be, 0x22b8, 0x438a, 0x0dd7, 0xa490, 0x400b, 0x424b, 0x4285, 0x407a, 0xae75, 0x17ab, 0x41a9, 0x1d62, 0xb214, 0xbf29, 0x4084, 0xa46a, 0x23a2, 0x44d2, 0xb0e4, 0x4203, 0x45ac, 0x4266, 0x1e81, 0x1a45, 0x10c8, 0xb0e2, 0x0833, 0x40aa, 0xb2cf, 0x1866, 0x1948, 0x435b, 0x18e0, 0xb230, 0x3774, 0x41b7, 0x1d3c, 0xb2b5, 0xbf90, 0x4265, 0xbfba, 0x4008, 0xba33, 0x1e27, 0x42d4, 0x1eef, 0xaf74, 0x41b5, 0x4069, 0x0d28, 0x0e57, 0x4254, 0x406f, 0xaac3, 0x1f94, 0x14a1, 0x4290, 0x4141, 0x4216, 0xb01f, 0x081b, 0xbfa1, 0xbfb0, 0x412b, 0xb24b, 0xb035, 0xb21f, 0x4248, 0x3144, 0xba32, 0x41d9, 0x461c, 0x1bb8, 0x40c8, 0xb266, 0xb258, 0xb2d6, 0xbadf, 0x42d4, 0x1a11, 0xbae0, 0x4629, 0xbf80, 0x4268, 0x0cef, 0x410f, 0xba0f, 0xba5e, 0xbf16, 0x42b3, 0xa03d, 0xb2b4, 0xb030, 0x3899, 0x4541, 0xb2e3, 0x1803, 0x16d3, 0xbfc2, 0xb216, 0x41dd, 0x42fa, 0xbf80, 0xafdf, 0x42da, 0x40e8, 0x1c5b, 0xb291, 0x1afb, 0xbf31, 0x41e2, 0x3e9d, 0x3a13, 0x11f2, 0x40ae, 0x400e, 0xba28, 0x40b8, 0x409d, 0x417c, 0x1ee6, 0x4252, 0xaa67, 0x0485, 0x43c7, 0xb0f3, 0x410c, 0x28df, 0xba12, 0x2ed5, 0xbf9f, 0xba74, 0x1d89, 0x44ac, 0x437a, 0x1e57, 0xbff0, 0x40cf, 0x415b, 0xb0a7, 0x414b, 0x402c, 0x4495, 0x29be, 0xa885, 0x39bb, 0x43c7, 0xbfb8, 0xb233, 0x1b91, 0x438e, 0xb21c, 0x1b36, 0xbf61, 0x4378, 0x4393, 0xba2b, 0x34c6, 0x1fef, 0xaa51, 0xa22d, 0xb270, 0x0598, 0xbfb0, 0xbad6, 0x42d9, 0xba35, 0x1bb7, 0x45f3, 0x4115, 0x1cfd, 0x42e9, 0x402f, 0xb20c, 0xbfd0, 0xae0a, 0x40c1, 0xb09d, 0xba08, 0x1fad, 0x41a9, 0x4387, 0x461c, 0xbf53, 0x186f, 0x2e59, 0x42b8, 0x425d, 0xbf11, 0x436c, 0xba65, 0x43b9, 0x4626, 0x44c3, 0x41b6, 0x1884, 0xb057, 0x1e30, 0xbfdd, 0x43a6, 0x405e, 0xa060, 0x423e, 0xb014, 0xba47, 0xad06, 0x1f36, 0xbf75, 0x464a, 0x3179, 0xa268, 0x463e, 0xb086, 0x2610, 0x401f, 0x2f67, 0x4409, 0x42c7, 0x1f54, 0x409b, 0x4111, 0x424a, 0xbf5d, 0x43b2, 0x467c, 0x40dd, 0x4085, 0xa14a, 0xa7d5, 0xba37, 0x4378, 0x4152, 0x43f6, 0x022e, 0x22b8, 0xb219, 0xba5b, 0xbf27, 0x424a, 0x0809, 0x4413, 0xb046, 0xbf60, 0x10bf, 0x1d41, 0x41bd, 0x1ed3, 0xac2b, 0xb2e2, 0xbfa0, 0x0d1b, 0x1fa2, 0xbf0e, 0xba7c, 0x40e4, 0x404e, 0x4593, 0xb276, 0x4170, 0x3ab1, 0xaad4, 0x2013, 0x3e57, 0xb0b8, 0xba09, 0x463b, 0xa62e, 0x43ac, 0x4390, 0x1c3e, 0x41f4, 0x01b9, 0x43b4, 0xb20d, 0xbf69, 0x4246, 0x4183, 0x42f6, 0x1bae, 0x4081, 0xabe8, 0x4013, 0x44c1, 0xa15e, 0x4271, 0x41a5, 0x42e1, 0x1cc6, 0x4302, 0x1176, 0x0ade, 0x0697, 0x41ad, 0x42ab, 0x41c5, 0x4260, 0x454a, 0xba34, 0xbf02, 0xb035, 0xb04b, 0x430d, 0x4418, 0x1bdd, 0xbfba, 0xb26f, 0x1b87, 0x42f2, 0x403c, 0x420a, 0x40b0, 0x1c57, 0xb061, 0x438a, 0x4173, 0x2e49, 0xb2f9, 0xbfc6, 0x41db, 0x1ad3, 0xba5d, 0x456c, 0xba02, 0x4245, 0xb098, 0x4267, 0x4141, 0x4665, 0x44a4, 0x05b1, 0xb2c1, 0xb001, 0x4205, 0x43fb, 0x40df, 0x19be, 0x43a4, 0x405a, 0x463f, 0x4130, 0x41ee, 0xb2da, 0xbf97, 0xac39, 0x4601, 0x41e1, 0x4175, 0xa137, 0x43d0, 0x4182, 0x1bf3, 0xbfd9, 0x409d, 0xb2a1, 0xba51, 0x013e, 0xbac1, 0x12a3, 0x4069, 0xabd0, 0x4220, 0xb0f8, 0xa4f4, 0x198b, 0x2d94, 0x43ee, 0x1fce, 0xba06, 0x022e, 0x42ea, 0x4229, 0x1d1e, 0xb0e2, 0xbfa3, 0x0bed, 0x4095, 0xba7c, 0x0052, 0x403c, 0xbadb, 0x4180, 0xba66, 0x4028, 0x427c, 0x2473, 0xba2c, 0xbf9b, 0xb208, 0xba24, 0x41f2, 0x4377, 0x1dd5, 0x40c1, 0xbf90, 0xb2c3, 0x1b1d, 0x42ac, 0x45ca, 0xbf8a, 0x1edd, 0x276c, 0x46d5, 0x32fc, 0xa25c, 0xb2d8, 0x4063, 0xbfb0, 0x02b4, 0x1c84, 0x2bcf, 0x429e, 0xb0f0, 0x43c5, 0x0fe7, 0x4323, 0xb2af, 0x1c79, 0x42c9, 0xa5ef, 0x41e9, 0x039f, 0x46ec, 0x1cd3, 0x1747, 0xbf6e, 0x42ba, 0xbad7, 0x4048, 0xb25a, 0x1930, 0x4296, 0xb08c, 0x406a, 0xb066, 0x191b, 0x3bd6, 0xba7c, 0x416e, 0x0db1, 0x4273, 0xbff0, 0xbf19, 0xba02, 0xbaf5, 0x4297, 0x41cd, 0x42d0, 0x411a, 0x1adb, 0x3814, 0xbf68, 0xb0b9, 0x415a, 0x4658, 0xb215, 0x4198, 0x4082, 0x43f7, 0xb097, 0xba0e, 0x1ac2, 0x431a, 0xb075, 0x4253, 0xb2a6, 0xbf9a, 0x1816, 0x41d5, 0xb2ef, 0xb0fa, 0x428f, 0x1b75, 0xba57, 0x4241, 0x43b8, 0x41de, 0xb2c4, 0xbaf0, 0xa9d9, 0xbfd4, 0x3b4a, 0xbacd, 0x426c, 0x1fa9, 0x3d6f, 0xbf07, 0xa213, 0x42c7, 0x0aa3, 0x43a1, 0xb090, 0xba23, 0x42ff, 0xa8ce, 0x4313, 0x41f5, 0x4691, 0xa85a, 0xac75, 0xbfe0, 0x0459, 0xaabc, 0x42c0, 0xbf1a, 0x41e2, 0xb03b, 0x1b5e, 0x2210, 0x4263, 0xbfc0, 0x2f43, 0xa745, 0x1bcc, 0x403c, 0x443a, 0x44bd, 0xbfc0, 0xb25c, 0x400f, 0x42f4, 0xbfe4, 0x4446, 0xb23d, 0x4166, 0xba55, 0xb2b2, 0xb2aa, 0x19ad, 0xa192, 0xb0d9, 0xa1b6, 0x42d5, 0x429e, 0x402b, 0xbf37, 0xb2c3, 0x2823, 0x425c, 0x0c29, 0x42a0, 0x0885, 0x4077, 0x43db, 0xb2c9, 0x2cd3, 0x4249, 0xbf81, 0xb212, 0x400a, 0x133a, 0x03ef, 0x30dc, 0x2ebf, 0x1b95, 0x1abb, 0xbfb0, 0xb2e7, 0x41e0, 0x401b, 0x400f, 0x3cd1, 0x3f45, 0xbf5a, 0x43be, 0x44c1, 0x4662, 0xbfc0, 0x4019, 0x4357, 0xb298, 0x4278, 0x4338, 0x2492, 0x4279, 0xb0a0, 0x436e, 0x4378, 0x4272, 0x19be, 0x4243, 0x43dc, 0x41aa, 0x1938, 0xbf22, 0x4202, 0xb0b5, 0xbac3, 0x1989, 0xa4aa, 0xb21e, 0x43ad, 0x4495, 0xbfde, 0x2891, 0xb287, 0xb2bf, 0x182f, 0x436c, 0x0e88, 0xb2fd, 0x4207, 0xba23, 0xb0ac, 0xbf39, 0x3a14, 0xb2a6, 0x42f3, 0x13e5, 0xb097, 0x4008, 0x4286, 0x421c, 0xbf4a, 0x1dc9, 0x1a6a, 0x41bf, 0xba55, 0x2f4b, 0xaa67, 0x4175, 0x4003, 0x4189, 0xb279, 0xbf3e, 0x4102, 0x45dd, 0xbfd0, 0xbad2, 0x0cb3, 0xb06f, 0x43f0, 0x44ed, 0x1af4, 0x40e7, 0xbfa1, 0x1ea1, 0x461d, 0x42b7, 0xb0a6, 0x44c0, 0xa90f, 0x280c, 0x42ff, 0x39ac, 0x4334, 0x4395, 0xba55, 0x4119, 0x1c16, 0xbfa0, 0x40d1, 0xb24e, 0x41f0, 0x3292, 0xba52, 0x4016, 0x43ce, 0x0873, 0x3836, 0xb074, 0xb08c, 0xbacc, 0xbf8b, 0x4301, 0xb274, 0xbfa0, 0x41f3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5a5bb2e2, 0x70b44505, 0xf439f718, 0x7601686e, 0x506efe14, 0xe57877bf, 0xae23b7eb, 0x04b61cf8, 0x3e1feb19, 0x3ee9a3be, 0x31b2d471, 0x4f5928f4, 0x9e0f7156, 0x0e22cfed, 0x00000000, 0xc00001f0 },
                FinalRegs = new uint[] { 0xffffffc9, 0xffffffc9, 0x00009100, 0x7fffffff, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0x7c404452, 0x8d794b1c, 0x4f5928f4, 0x8d794b1d, 0xfb201937, 0xf1ddb58c, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbac1, 0x4158, 0x40ea, 0x242c, 0x4288, 0xb29e, 0x418a, 0xbf23, 0xb2fa, 0x4148, 0x3f4d, 0xba28, 0x43a5, 0xb0fb, 0x1ba0, 0x4000, 0x2c42, 0xbac7, 0x40bf, 0xa072, 0x43fc, 0x1202, 0x40a6, 0xba20, 0xbf96, 0xb0ff, 0x0d2b, 0xb221, 0xb274, 0xb237, 0x4048, 0x43ef, 0x45a1, 0x1d08, 0x1c73, 0xbf35, 0x44c0, 0xbacb, 0xb0bf, 0xa4a5, 0x06ac, 0xa673, 0x4076, 0x42de, 0x411c, 0x4316, 0xba5a, 0xb222, 0xbf60, 0x1f22, 0x4047, 0x4599, 0x43eb, 0x3d4e, 0x4684, 0x4159, 0x42d6, 0x4397, 0x4019, 0x37f0, 0x0c64, 0xba56, 0x404f, 0x419b, 0xbfab, 0x0a60, 0x1678, 0xb2e1, 0x4087, 0x4113, 0x1c75, 0x43e2, 0x4580, 0xb2f4, 0x1a65, 0x41bf, 0xa5c4, 0xbf00, 0x023e, 0xba5b, 0x40ec, 0x423f, 0x4265, 0x4159, 0xba17, 0x43d5, 0xbfa0, 0x419d, 0x0690, 0x4613, 0x4026, 0x4237, 0x419a, 0xbfbd, 0xaf55, 0xb248, 0x1543, 0x43c3, 0x1eac, 0x4208, 0xb224, 0xbacd, 0x4290, 0x40dd, 0xb2bb, 0x4551, 0x40a6, 0x4595, 0x4118, 0x1940, 0xbf41, 0x3427, 0x18e1, 0x405d, 0x4086, 0x42c3, 0xaaaf, 0x39b0, 0x40a6, 0x2091, 0xbf1f, 0xb29c, 0x0054, 0x3139, 0x40a2, 0x3fcc, 0x2890, 0x4073, 0x4378, 0x1fe5, 0x00f7, 0x1a7b, 0xbad5, 0x401d, 0x4229, 0x424e, 0x1e92, 0xa256, 0x40ed, 0xa108, 0xbfa8, 0x41b1, 0x40ad, 0x1f7c, 0xb2ce, 0xb27e, 0xbf7b, 0x4066, 0x305d, 0xba36, 0xbafb, 0x2265, 0xba43, 0xb213, 0x43fe, 0x40ba, 0xa9a2, 0xbfc5, 0x1c2e, 0x4254, 0xa6ce, 0x40c7, 0xb0d1, 0x416d, 0x425f, 0xa0b5, 0x4365, 0x1da1, 0xb204, 0x3eca, 0x430c, 0x4646, 0x43f9, 0x42c3, 0xbf68, 0x4058, 0x416d, 0x1e07, 0x1e89, 0x40ec, 0x4041, 0x40b4, 0xb01d, 0xb234, 0x4023, 0xb019, 0x4679, 0xb0ff, 0x4624, 0xad8f, 0x188f, 0x412e, 0xaf18, 0x1adc, 0xbf11, 0x1f44, 0x436d, 0x436b, 0xa7ad, 0x0ba2, 0xb28b, 0x4273, 0xb22d, 0x4097, 0x4105, 0x426c, 0xb25a, 0xbf90, 0x428c, 0x138c, 0x44e1, 0x418a, 0x1aff, 0x219f, 0xb269, 0xbf15, 0x0b01, 0x4201, 0x4008, 0xbafe, 0x3265, 0xb0b0, 0x4359, 0x45ba, 0x425f, 0xb214, 0x42e9, 0x4456, 0x404d, 0xbfd7, 0x40a2, 0xb0f8, 0x4032, 0x18b3, 0x3009, 0xb2ab, 0x41b7, 0x44c9, 0x4419, 0xb096, 0x429f, 0x433d, 0x38f2, 0xbfe0, 0x4083, 0x414b, 0x4055, 0xba74, 0xbf75, 0x4072, 0x41fb, 0xb04a, 0x43e3, 0x43f7, 0xad8d, 0xb251, 0xba67, 0x4300, 0x4259, 0x4206, 0xb0a2, 0xb242, 0x4699, 0x1c9e, 0xbf73, 0x434f, 0x0744, 0xa414, 0x4099, 0x4661, 0xba62, 0x2e9f, 0x431a, 0x05c6, 0x465d, 0xb2ee, 0xbae1, 0x093a, 0xbf90, 0x4399, 0xb21d, 0x40dd, 0xb2b5, 0x0057, 0xbf60, 0x1f1c, 0xbf18, 0xb28b, 0x41f3, 0xbadb, 0xbf9c, 0xb261, 0x129d, 0x42fb, 0x42b7, 0x4306, 0xb06f, 0xb2d6, 0x41b6, 0xa264, 0x4001, 0x2ef9, 0x2dc8, 0x40c6, 0x40f3, 0x456f, 0x414e, 0x26ae, 0x2625, 0x41bf, 0x40b0, 0x1a12, 0x31a8, 0xb000, 0x1a30, 0xbf31, 0x1198, 0x44fd, 0x043c, 0x4301, 0x1a9d, 0x10cc, 0xbfac, 0x401e, 0x135e, 0xb2df, 0x4356, 0xae2b, 0x40e0, 0x43f7, 0x322f, 0xba6b, 0xbfde, 0x1e91, 0x21a5, 0x413f, 0x4081, 0x134a, 0x43d9, 0x417b, 0xb089, 0xadcd, 0xb220, 0x43ab, 0x1aed, 0xa819, 0x1ee7, 0xb0de, 0xb2dd, 0x2249, 0x4122, 0x29d1, 0xa87d, 0xb0b2, 0x128c, 0x33a1, 0x0d9c, 0xbf16, 0x15ea, 0x4607, 0xbf70, 0x2f6c, 0x1900, 0x4060, 0x1abd, 0xb2f2, 0xba65, 0x4166, 0x3593, 0xbfcf, 0x1a90, 0x4308, 0x4447, 0xba58, 0x1bf2, 0x419c, 0x181c, 0x2f88, 0x4141, 0x418a, 0x4067, 0x12a7, 0xb0c5, 0x4197, 0x0a82, 0x0774, 0x34e5, 0x4456, 0x4109, 0x4209, 0x445d, 0x4029, 0x0d4e, 0x424b, 0x42ee, 0x41a0, 0x4673, 0xbf70, 0xbf3c, 0x18f1, 0x3e63, 0xa7f4, 0x097a, 0x4586, 0x4163, 0x40f1, 0xba57, 0x284b, 0x420e, 0x41cd, 0x4092, 0xba0d, 0x4230, 0x43ce, 0x445c, 0x44cd, 0xb23f, 0x40a6, 0x41ce, 0x1d86, 0xb235, 0xb251, 0xb289, 0x43ba, 0xbfc5, 0x408f, 0xba1b, 0xb0fe, 0x41c0, 0x4107, 0x36ad, 0x3817, 0x2df3, 0xab39, 0x223f, 0x4253, 0xbf67, 0x4148, 0x40d9, 0x42f0, 0xb0e5, 0x4314, 0x404d, 0xb2c4, 0x2d43, 0x41eb, 0x2b28, 0x4630, 0x21d0, 0x1f98, 0xb28a, 0x43c0, 0xbf80, 0x1d10, 0xacb5, 0x45db, 0xbf59, 0x1132, 0x1ef9, 0x429c, 0x1cce, 0x41af, 0xa433, 0x15e1, 0x42b5, 0x419c, 0x1b1f, 0xb2e5, 0x4012, 0xb28c, 0x1d9b, 0xa11e, 0x1a24, 0x423a, 0x40e8, 0xb257, 0x41f4, 0x4452, 0x24ad, 0xbf9a, 0x1809, 0xb2dd, 0xb2be, 0x4104, 0x19b0, 0x43c4, 0xb037, 0x43ea, 0xae35, 0x04ab, 0xb2f8, 0x2c34, 0x2827, 0x40bd, 0x466b, 0xbf59, 0xb029, 0xb219, 0x42fd, 0x434f, 0x1cfa, 0xa456, 0x44c4, 0x415b, 0x1e44, 0xbfbe, 0x4648, 0x4263, 0x41bc, 0x186f, 0xba0f, 0x1cda, 0x1a0c, 0xb0b6, 0x1c21, 0x143a, 0x4284, 0xbae2, 0x4083, 0x1fbb, 0x436d, 0x4286, 0x0262, 0x1fcd, 0x4368, 0x40b5, 0x4339, 0x360f, 0x420a, 0x40e0, 0xb29e, 0xbf61, 0x0a3d, 0x40d3, 0x433b, 0xb20b, 0x4588, 0x401e, 0xb2cf, 0x0ab7, 0x46ab, 0xbfcd, 0x42b0, 0x4199, 0x1deb, 0x16b0, 0x43bb, 0xa042, 0x0943, 0x42d8, 0x4298, 0x412d, 0xbf3c, 0xb01b, 0x43b0, 0xbafe, 0xb02c, 0x4086, 0xbfd1, 0x3145, 0x40aa, 0x1e27, 0xbaff, 0x2464, 0xbf60, 0x435d, 0x43cf, 0x41d7, 0x4408, 0xba3f, 0x40bb, 0x1900, 0x2295, 0xb2d8, 0x2325, 0x46d1, 0xb0f8, 0x438a, 0x43ea, 0x460f, 0x4205, 0x41a7, 0x0b21, 0x1c41, 0xbaeb, 0x3f2f, 0x4340, 0xbf08, 0x1d1a, 0x2ee5, 0xb0b8, 0x432e, 0xbf0f, 0x1edc, 0x4365, 0x1bda, 0xba31, 0x186c, 0xbf95, 0x4027, 0x4268, 0x4136, 0x41a1, 0x407d, 0x1d56, 0x420d, 0xbf11, 0x4246, 0x1bd6, 0x41eb, 0x41b7, 0xb073, 0x40bb, 0x1804, 0x4146, 0xb234, 0x1ca5, 0x40e5, 0xba68, 0xaa9b, 0x2f01, 0x4102, 0x40d0, 0x1c95, 0xb2be, 0x402f, 0xb264, 0xb29b, 0x46fb, 0xb288, 0x418b, 0xbf38, 0xa198, 0x0caa, 0x14ba, 0xadde, 0x033f, 0x40ae, 0x1b2b, 0xb0a9, 0x067e, 0x1fc9, 0x35ab, 0xbad2, 0xac87, 0x1001, 0xb2d3, 0x01a8, 0x407f, 0xbac0, 0xbf23, 0xb2d0, 0x45a6, 0x430b, 0x44e0, 0x4223, 0x42dd, 0xb2bf, 0xb209, 0x462b, 0xb2d5, 0x18a0, 0x4280, 0x1d75, 0x4261, 0xb2fc, 0xb2f7, 0x37a6, 0xba4c, 0x42ee, 0xbaf7, 0x2e93, 0xbf17, 0x4295, 0x43fc, 0xb06f, 0xb20b, 0x4065, 0x4068, 0xbfde, 0x40b8, 0x17c8, 0x43bd, 0x40c4, 0x4159, 0x118d, 0x412a, 0xaebf, 0xbf04, 0x2ba2, 0xb2d5, 0x0b54, 0x0295, 0x438a, 0xbaf1, 0x32b1, 0x2c88, 0x414a, 0x19c9, 0x4026, 0x434d, 0x2396, 0x1d91, 0xb24d, 0xad96, 0x418b, 0x37e2, 0x436c, 0x42ee, 0xbf8b, 0x4166, 0x433a, 0x40b5, 0xbae9, 0x41dd, 0x4303, 0x431a, 0x41f6, 0xbaef, 0x411a, 0xb2ab, 0x113c, 0x1a84, 0xb03e, 0x063d, 0xbf1e, 0xb22a, 0x3926, 0x40e6, 0xba22, 0x2aca, 0x2561, 0xba47, 0xba3b, 0x0a0a, 0xb05d, 0xb2fc, 0xbf95, 0x467e, 0x17a2, 0xba2d, 0xb031, 0x4227, 0x1c1d, 0x427c, 0x3211, 0xb086, 0x43af, 0x4041, 0xad6c, 0x1877, 0x202b, 0x405a, 0x446b, 0x34ef, 0x4084, 0xb252, 0x1ef7, 0x41c8, 0xbf6f, 0xb06f, 0x02fc, 0x409f, 0xba72, 0xb030, 0x4221, 0x4139, 0xbf87, 0x42d1, 0x19be, 0x42de, 0x46eb, 0xba3c, 0x4078, 0xbfb7, 0xba51, 0x43e8, 0x4194, 0x42c7, 0xbf04, 0x4085, 0x42de, 0x40aa, 0x303e, 0x4234, 0x1a9f, 0x1abc, 0x42bc, 0x3b66, 0x361d, 0x01fd, 0x41d2, 0x427e, 0x4332, 0xb21f, 0xb01b, 0xb2de, 0xba46, 0x40bb, 0x1f62, 0x419c, 0x43b1, 0xb2bf, 0xbf7a, 0x1f92, 0xa922, 0x018a, 0xbadf, 0xb229, 0xb02f, 0x195a, 0xbae1, 0x400f, 0x182c, 0xb252, 0x412a, 0x40d9, 0xbf7c, 0xbad9, 0x06da, 0x43e5, 0xb0ab, 0x431f, 0xba39, 0xb016, 0x1ac4, 0x406a, 0x414e, 0x42b8, 0x4237, 0xb28a, 0x44eb, 0x1aa1, 0x34f8, 0xba2a, 0x4568, 0xb210, 0x42b8, 0x1836, 0x434f, 0x4253, 0x21fd, 0x43ec, 0x4181, 0x4239, 0xbf79, 0x43eb, 0x4061, 0xb2c0, 0x29fd, 0x1ac6, 0x3e8c, 0xacfe, 0x43a7, 0xb257, 0xba3d, 0xb06b, 0x40ce, 0xbf61, 0x4205, 0x430d, 0x43ad, 0x02a9, 0x07c4, 0x41c3, 0x1c78, 0x4018, 0x4286, 0xbfd4, 0xbac1, 0x41e3, 0xba27, 0x1e65, 0xb20f, 0x4001, 0xbf68, 0x4642, 0x43af, 0x0ff7, 0x43c4, 0x0a6f, 0xbf19, 0x405d, 0x401c, 0xa368, 0x1c07, 0x06eb, 0x42d9, 0x1863, 0xb0ae, 0x42a9, 0x41ac, 0xb0b6, 0xbfa0, 0xbf90, 0xb2db, 0x4075, 0x4284, 0x431f, 0x1f38, 0xbf2d, 0x4385, 0xb00f, 0x40af, 0x1b7a, 0xaaa3, 0xa6de, 0xba15, 0x4159, 0x41a1, 0x3aee, 0x4072, 0xb0bb, 0x40fb, 0x442b, 0x3328, 0x1e7d, 0x1d86, 0xb232, 0x1b14, 0xb00d, 0x41b9, 0x40fe, 0x1a56, 0x1c3d, 0x2f3c, 0x09ad, 0x4111, 0xbf71, 0x4630, 0x030d, 0x4117, 0x4034, 0x3845, 0x0b31, 0xbae2, 0x462f, 0x0e01, 0xb225, 0x2df4, 0x4651, 0xbfa0, 0x4381, 0x1bcc, 0x432d, 0x4265, 0x42ad, 0x4379, 0x410a, 0xbf04, 0xb28c, 0x42a8, 0x414c, 0x4065, 0x1586, 0x43aa, 0x336d, 0x4211, 0x1bee, 0x42a7, 0x4026, 0x263a, 0x1fa1, 0x4354, 0xbfa9, 0x1d2b, 0x4337, 0x432e, 0xb2d6, 0x18a9, 0xb0a0, 0xba29, 0x4131, 0x4348, 0x403f, 0x0ab3, 0x4214, 0x4175, 0x4259, 0x43d7, 0x33cf, 0xba42, 0x11bb, 0xbf3a, 0xba77, 0x4098, 0x4232, 0xb2a7, 0x4375, 0x0786, 0x1f14, 0xa4fd, 0x01a8, 0xa0ad, 0x411e, 0x2763, 0xba0a, 0xb05f, 0xbac6, 0x4123, 0xbf46, 0x41b5, 0xb20d, 0xac2e, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4ccb59f7, 0x5e2a5ea7, 0x0598ceb9, 0xfde9560f, 0xaeee5f97, 0x81c7018e, 0x8dfd005d, 0xf0809e08, 0xf2345361, 0x7be970a4, 0xd526e1ae, 0xd985c732, 0x2702a40c, 0xf395aaaa, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x00001a78, 0x00000000, 0x00000000, 0xffffffff, 0x00001bb4, 0x00000000, 0x0000781a, 0x00000063, 0xc8d14d87, 0xd526e1ae, 0xd526e1ae, 0xf296cb30, 0xe468a6c5, 0xf296b6a4, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x30ba, 0x3e17, 0xb2b6, 0x0e7a, 0x1ad3, 0x16d7, 0x3d09, 0xb2d4, 0x4149, 0x4388, 0xbf70, 0x403e, 0x4081, 0x2f93, 0xb263, 0x421d, 0x4136, 0x1fbc, 0xbadd, 0x42e0, 0x41df, 0x42f2, 0xb2ac, 0x4188, 0x1f29, 0xbf0c, 0x46e4, 0x0f06, 0x4220, 0xb090, 0xb2be, 0xa4e6, 0x4047, 0xbfe0, 0x465b, 0x3199, 0x4216, 0x1b5d, 0x0645, 0xb04b, 0x1ae2, 0xaec8, 0x1942, 0x4225, 0x4214, 0x3aa9, 0xba12, 0x41b2, 0x30a1, 0xb27b, 0xba72, 0xba69, 0x15c8, 0xbfdd, 0xb204, 0x427c, 0x42e0, 0xb260, 0x4169, 0xa447, 0xb204, 0x424e, 0x465d, 0x3fce, 0xabb3, 0x43f3, 0xba44, 0x418d, 0x42b9, 0xb045, 0x20a4, 0xb285, 0xb2c1, 0x424d, 0x0575, 0xb081, 0x0475, 0xb273, 0x43db, 0x4297, 0xbf16, 0x41d7, 0x1b1c, 0x1a44, 0x4220, 0x29a3, 0x4069, 0x4143, 0x464e, 0xb2e9, 0x4120, 0xb28b, 0xacb2, 0xac86, 0x2c6e, 0x43c5, 0x42d4, 0x42e7, 0xbf15, 0x1c4d, 0x41a9, 0x4300, 0x401c, 0x32f0, 0x431e, 0x4117, 0x43f3, 0xb041, 0x02d0, 0x443d, 0xbfc9, 0xb292, 0x436c, 0xb236, 0xb022, 0xb023, 0x430b, 0xbfc0, 0x4318, 0x456a, 0x2809, 0x4006, 0x432a, 0xba3e, 0x4396, 0x406b, 0xbf16, 0xba55, 0x4163, 0x4279, 0x035d, 0x4380, 0x4078, 0x1ccc, 0xba46, 0x4064, 0xb21a, 0x0727, 0x1a5e, 0xba7f, 0xa4ed, 0xbac7, 0x0244, 0xb26b, 0x23f1, 0x3321, 0xa490, 0x4284, 0xba64, 0xa24a, 0x4057, 0xbaff, 0x1877, 0x417c, 0x0d9c, 0xbf9c, 0x1c58, 0x42c3, 0x1c7f, 0xba5f, 0x0daa, 0x42d8, 0x41be, 0x40d2, 0x4298, 0x4118, 0x4085, 0x4353, 0xbf06, 0x402f, 0xb0fc, 0x45ab, 0x41da, 0x11d6, 0xbf04, 0xb025, 0x40a2, 0xb26c, 0x40ae, 0x40c5, 0x08ba, 0xb227, 0x4356, 0x423b, 0xb20c, 0x230b, 0x40e1, 0xb0fa, 0x4013, 0x14bd, 0x43de, 0x3daf, 0x0368, 0x407f, 0xb298, 0x41f8, 0x43de, 0x4048, 0x18a4, 0x08f7, 0x42b9, 0x41a8, 0xbf33, 0x404f, 0x4174, 0x4261, 0x43f1, 0x1609, 0x4281, 0x00ee, 0x4291, 0x43b1, 0x359a, 0xb0d0, 0xb09f, 0x40fe, 0xb2b5, 0x4466, 0xb050, 0xb25c, 0xba0b, 0xb03a, 0x04a6, 0xb247, 0x4390, 0xb291, 0x4198, 0xb02a, 0x421f, 0x40b3, 0xbf7d, 0x41f6, 0x23ee, 0x329a, 0x43d2, 0x421c, 0x4199, 0xb253, 0xb0c6, 0x3180, 0xb0b7, 0x43a0, 0xbfbf, 0x43e6, 0x1e76, 0x157f, 0x3290, 0x1952, 0x428a, 0x1b3f, 0x05ce, 0x0e2e, 0xb0bf, 0xb264, 0x41a3, 0x0a74, 0x4181, 0x4112, 0x3e9f, 0x42c4, 0x4214, 0x46b9, 0x1a96, 0xbf0f, 0xa787, 0x42fb, 0xb008, 0x40a0, 0x2262, 0xb261, 0x26a5, 0xa01b, 0x01fa, 0x40a4, 0xba0f, 0xa676, 0xb26b, 0xb262, 0xba1f, 0x4075, 0xbfc2, 0x415d, 0xb095, 0x1ead, 0x410a, 0x4328, 0x0c46, 0xb2e9, 0x430f, 0xb274, 0x415d, 0x420f, 0x197b, 0x46fb, 0x125d, 0x43ca, 0xa547, 0x4064, 0x425e, 0xbf2f, 0xb036, 0xb292, 0x40e9, 0x3d9e, 0xb227, 0xa088, 0xb0e0, 0xb21c, 0xafc4, 0xbace, 0x40a2, 0x100d, 0x405d, 0x135d, 0x4045, 0x054f, 0x430c, 0x415c, 0x3b73, 0x1c6b, 0x0905, 0x4213, 0x4471, 0xa947, 0xb269, 0xbf77, 0xba2b, 0x42c8, 0x119f, 0x1d18, 0x4031, 0x4657, 0x4240, 0x4147, 0x4191, 0x44ad, 0x410b, 0xbad7, 0x40d8, 0xa692, 0x1bd6, 0x4137, 0xb05c, 0x4204, 0xbf72, 0x22bb, 0xa447, 0x1ec3, 0xb07f, 0xbfd0, 0x438d, 0x4220, 0xbf4a, 0x40eb, 0x42f2, 0x2d2f, 0x40e1, 0xa238, 0x41fa, 0xb24e, 0xb2e5, 0xbfa0, 0x441c, 0xb056, 0x4365, 0x1cca, 0x1a9a, 0x4346, 0x2fbc, 0x4222, 0xb2c2, 0xbfda, 0x416c, 0xbfe0, 0xba4f, 0x40e8, 0x4386, 0xbfc0, 0x06e0, 0x4370, 0x419e, 0x3b0e, 0xae3e, 0x413b, 0xb0b1, 0xb02f, 0x1f7c, 0x377d, 0xbac4, 0x45d2, 0xb25d, 0x0d8c, 0x46c4, 0x4304, 0xa6dd, 0x1a5d, 0x4050, 0x410c, 0x4269, 0xb284, 0xbf2d, 0x414f, 0x044d, 0xba0a, 0x19a1, 0x4107, 0x4273, 0xbafa, 0xb222, 0x42ba, 0x45c0, 0x408b, 0x4081, 0x4102, 0x070a, 0xb2eb, 0xb056, 0x385b, 0x1b53, 0x046b, 0xba7b, 0x41a3, 0xbaf9, 0x4553, 0x313a, 0xb028, 0xbf42, 0x418c, 0x05c5, 0x4161, 0xa924, 0x4026, 0xba41, 0x40a4, 0x4285, 0x3ff7, 0x1c6a, 0xb207, 0x42ae, 0x4371, 0x116e, 0xbf35, 0x1d23, 0x429f, 0x1a06, 0x44fb, 0xb2ac, 0x40d4, 0x46e5, 0x403e, 0x4387, 0xb099, 0x4109, 0x40d0, 0x2f0a, 0x38cd, 0x02a6, 0xbf38, 0x46ac, 0x30d9, 0x087d, 0xb27b, 0x41ff, 0x1537, 0x4101, 0xba15, 0xb294, 0x41d6, 0x2af8, 0xba0e, 0x41d9, 0x3f8b, 0xba63, 0xb238, 0x4003, 0x1c25, 0x41c7, 0x43a7, 0x0ad6, 0xbf2b, 0x1ee5, 0xb2c4, 0xa57f, 0xb06e, 0x1e2e, 0x4039, 0x409b, 0x4102, 0x416d, 0xa855, 0xb235, 0x3847, 0xa008, 0x4411, 0xb26a, 0xa9fe, 0xbfd7, 0x41d7, 0xbf70, 0xbae6, 0xba06, 0x35df, 0xb219, 0x2f49, 0xb0cd, 0x35d3, 0xb0ca, 0x4383, 0xb233, 0x4381, 0x3a8f, 0x460c, 0x1df7, 0x0edb, 0x43d4, 0x1a05, 0x1318, 0xbac2, 0xb2b9, 0xb2d1, 0x4230, 0xbf3e, 0x422d, 0xb213, 0xb061, 0xa09a, 0x4057, 0x4077, 0x426c, 0x0376, 0x42a5, 0x19c9, 0xbf1d, 0x422d, 0xb2f2, 0xb04b, 0x41a5, 0x410e, 0xb2ac, 0x0819, 0x415b, 0x3316, 0x4314, 0x1fd3, 0xb075, 0xbfb3, 0xb08b, 0xa994, 0xbf80, 0xbae1, 0x40b4, 0xbfb8, 0x409d, 0x01ee, 0x43ec, 0x09d0, 0x2384, 0x4183, 0x416b, 0x411d, 0xbf78, 0xba54, 0xba72, 0x4397, 0x1eb9, 0x1efa, 0xb2ec, 0x183e, 0x18e1, 0x0a6c, 0x42c6, 0x232a, 0xba57, 0xbff0, 0x41f5, 0x42c9, 0x3b22, 0x421f, 0xb294, 0xb277, 0xa937, 0xbf8e, 0x462f, 0x20b5, 0x2e40, 0x4306, 0x43f1, 0x036f, 0xa324, 0xbf23, 0x3670, 0xba5c, 0x4050, 0x0906, 0x4389, 0x4684, 0x40b0, 0x3eb0, 0x1f2e, 0x1aba, 0x1963, 0x4145, 0x122d, 0xb2c1, 0x4555, 0x403d, 0xb28b, 0x4252, 0x1a41, 0x4077, 0x434f, 0x4459, 0x4360, 0xbf22, 0x1b7f, 0xb24e, 0x1433, 0x40b1, 0x41a3, 0xb2cd, 0xb021, 0x4251, 0x42e3, 0xb0b5, 0xb2d0, 0xbf55, 0xa034, 0x41e3, 0x05fb, 0x4067, 0x41c2, 0x4059, 0x40ea, 0x3288, 0x4328, 0x4268, 0xbf2d, 0xb05b, 0xb234, 0x4220, 0xba68, 0xbfc0, 0x42e8, 0x406d, 0x4371, 0x12db, 0x4157, 0x448c, 0x162f, 0x3d79, 0x4190, 0x4050, 0x0c23, 0x2cef, 0x415b, 0xbaf1, 0x4005, 0xbf57, 0x404d, 0x214b, 0xa974, 0x404c, 0x40cb, 0x03ca, 0x1675, 0x36c7, 0xbfda, 0xaecc, 0x4576, 0x2c9c, 0xa7ec, 0x423e, 0x1d8c, 0x4172, 0x437a, 0x0b36, 0xa6b8, 0x4160, 0x4648, 0x45c4, 0x1d48, 0x1cfb, 0xa650, 0x4265, 0xb2b3, 0x414c, 0x443a, 0x4378, 0xba29, 0x464e, 0x1d73, 0xbfa7, 0x121c, 0x2eb2, 0xba30, 0xb0e6, 0x40d8, 0x4108, 0xbf70, 0xbad4, 0xb27e, 0xbaca, 0x4694, 0x320f, 0x1605, 0xba48, 0xafdb, 0x41e1, 0x4095, 0x3649, 0x00ef, 0xbf25, 0x4233, 0x4226, 0x4390, 0x46b8, 0x4231, 0x43d0, 0x45d2, 0x42d7, 0x3871, 0x1a1d, 0x1d3e, 0xbad1, 0x418b, 0xb072, 0x1843, 0xb215, 0x1ae0, 0x0898, 0x4106, 0x41a2, 0xba0a, 0xbf43, 0x41c9, 0x4022, 0xa59a, 0xb035, 0x4549, 0xbfa7, 0x441f, 0x4167, 0x3b3a, 0x1bd7, 0xbf75, 0x3d63, 0x1bba, 0xb2ae, 0x401b, 0x0e2e, 0x42cd, 0x1ccd, 0x4028, 0xba22, 0x4343, 0x3458, 0x2dd4, 0xa814, 0x1985, 0x469a, 0x43c1, 0x4562, 0x444c, 0x19b5, 0x42ab, 0xbf7a, 0xb2a1, 0x4394, 0x4008, 0x0f19, 0xbae8, 0xb266, 0x4141, 0xae9b, 0x4349, 0x45ad, 0x1af6, 0x40c0, 0x40a2, 0xba1a, 0x439b, 0x1fb8, 0x11da, 0xa672, 0xb2d0, 0x4223, 0x449d, 0x431d, 0x42e8, 0x45ec, 0x18d7, 0xbf5e, 0x41df, 0xba7f, 0x423b, 0xa7ae, 0x28f4, 0x414b, 0x43fa, 0xb216, 0x190b, 0xa562, 0x4391, 0xb252, 0x03f3, 0xbac0, 0x40fe, 0x2da8, 0x4449, 0x0adc, 0x4103, 0xb0dd, 0x403f, 0x34fa, 0xbf00, 0x45a6, 0x4357, 0xbf8a, 0xb2a2, 0x4109, 0xadc3, 0x407b, 0x40ac, 0xb227, 0x41fa, 0xaccd, 0x46a8, 0x40fa, 0x1f95, 0xb069, 0x444a, 0x4037, 0xb230, 0x419b, 0x1426, 0x424d, 0xb05f, 0x107b, 0x4055, 0x3be8, 0x0e2b, 0xb0f2, 0x1793, 0x46cb, 0x1fe7, 0xbfc6, 0x4300, 0x1e82, 0x43cc, 0x34b4, 0x173b, 0xb290, 0x36a7, 0x4348, 0x0f9f, 0xba7e, 0x0bef, 0xb22d, 0x42bf, 0x4541, 0xbf70, 0x4193, 0xbff0, 0xb0f2, 0x4420, 0xb01f, 0x41b7, 0x4251, 0x4301, 0xbf0e, 0xbad0, 0xbafe, 0x1f39, 0x2717, 0x4651, 0x1858, 0xb0e6, 0xbf88, 0x0dc3, 0x162d, 0x1947, 0xb24e, 0x24fa, 0xba59, 0x18a4, 0x42f6, 0x4018, 0x42a2, 0x407c, 0x42c6, 0x30d6, 0x4299, 0xb081, 0x4288, 0x41f8, 0x01b4, 0xb00a, 0xb26d, 0x436c, 0xbf0c, 0xb27d, 0xb2a6, 0x45ae, 0x04d9, 0xa8af, 0xb050, 0xbf61, 0x3332, 0x27cd, 0x0c39, 0xb28b, 0xb0e7, 0x2bfa, 0x4231, 0x43cc, 0x4387, 0x1d3e, 0x2fcc, 0x0863, 0x43eb, 0xba00, 0x41bd, 0x4139, 0xbf80, 0x1a3d, 0xba4b, 0x1a7f, 0x41d5, 0x4695, 0x41eb, 0x4003, 0x4457, 0x4312, 0x3a69, 0xb253, 0xbf68, 0x4143, 0xba2f, 0xbaf5, 0xb2e5, 0xbf00, 0x0f33, 0x418a, 0xb254, 0xbf3a, 0xba72, 0x404a, 0xb297, 0xbf46, 0x4645, 0x41e7, 0x4320, 0x1a06, 0x44e0, 0x3e22, 0x41d4, 0xbaf8, 0x4226, 0x41a5, 0x4291, 0xbae8, 0xbaf5, 0x4199, 0x45d1, 0xbf0e, 0x1f1d, 0x3bb4, 0xba40, 0x0793, 0x4645, 0x0312, 0xb262, 0x41b9, 0xb20f, 0x42e3, 0x4018, 0x4570, 0x10d0, 0x038a, 0x434f, 0xb2d2, 0x4430, 0xa2b1, 0x042c, 0x194b, 0x45db, 0x43b7, 0xbf5e, 0x1b6d, 0xb282, 0x40a5, 0x4076, 0x2065, 0x1c00, 0x462d, 0xaded, 0x3907, 0x1fe4, 0x429a, 0x40bd, 0xba32, 0x43b0, 0x1b1d, 0x43ef, 0x41c0, 0x427a, 0xba34, 0xba7b, 0x2307, 0x008d, 0x46c3, 0x4373, 0x40f3, 0xbfb9, 0x4169, 0x4135, 0x413f, 0x2474, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x7602de1d, 0x625d8c21, 0x6c16bc87, 0xecb33bf3, 0xf2e0d596, 0x9ed96ef4, 0xc82738a1, 0xf563157e, 0x31d024d5, 0x481b545e, 0xbb853a8f, 0x828b1a9a, 0x77119f7e, 0x41a530b7, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x28000003, 0xf00ebff8, 0xd843d7d2, 0x00000000, 0x00000074, 0xc03affe0, 0x00000000, 0x00000000, 0x000017cb, 0xffffffae, 0x001b0000, 0x000017cb, 0xffffffff, 0xfffffffe, 0x00000000, 0x600001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x183a, 0xb2ba, 0xbff0, 0x4236, 0x4329, 0x416b, 0xb0ab, 0xbfc0, 0x461c, 0x1cc6, 0x438a, 0x1dfb, 0x43ef, 0x1b4e, 0xbaee, 0x36a4, 0xb2e2, 0x4169, 0xb048, 0x1d98, 0xa51c, 0x40cc, 0x4332, 0x42e2, 0xbf01, 0xa8ff, 0xb2c5, 0x4322, 0xba13, 0xb017, 0x278e, 0xba57, 0x43ca, 0x4097, 0xb239, 0x41e7, 0x43df, 0x1fe8, 0x4152, 0x34e7, 0xba3e, 0xb211, 0x41ce, 0xb229, 0x4194, 0xbfc8, 0x1c04, 0x403a, 0x43d2, 0xbf41, 0x434d, 0xba36, 0xba4f, 0x40f1, 0x12a2, 0xb2a6, 0x0ba3, 0x1430, 0xb2cf, 0xaf47, 0x4380, 0x215d, 0x18bc, 0xb0ff, 0x4088, 0x41b3, 0x4131, 0xb0ae, 0x1cf1, 0xbfdd, 0x42f0, 0xb2a4, 0x4245, 0x43dc, 0x19b7, 0x4322, 0xba58, 0x1c79, 0x4387, 0x4596, 0x1ece, 0x41fc, 0xbf01, 0x092d, 0xb016, 0x40d1, 0xbad3, 0xb2bf, 0x40a6, 0x4247, 0x402f, 0xa9b7, 0xb244, 0x19fa, 0x4244, 0x1b06, 0x18ef, 0x4181, 0x144c, 0x18b1, 0x4197, 0x4123, 0x3a3e, 0xb204, 0xbfbe, 0xbaea, 0x1c38, 0x436a, 0x2039, 0x1ccf, 0xaff3, 0x1847, 0xaecc, 0xb292, 0xb20f, 0x1f5a, 0xb2f2, 0x4183, 0x4004, 0x43b8, 0xab48, 0x43f9, 0x4234, 0x0ca8, 0x3607, 0x4118, 0x4206, 0xbfbe, 0x4073, 0x4688, 0x391a, 0x40b5, 0xba2f, 0xbf00, 0x3d16, 0x1b41, 0x421c, 0x434a, 0x413c, 0x405c, 0x41d8, 0x4375, 0x415c, 0x101b, 0x4540, 0x42ce, 0xb0aa, 0xbf9e, 0x42d0, 0x4215, 0x2cc7, 0x43db, 0xb23e, 0x2a9f, 0x461e, 0xbacf, 0x4344, 0xb2dd, 0x41d1, 0x4233, 0x2398, 0xa583, 0xb09d, 0xa90c, 0x3974, 0x407f, 0x41fe, 0xb03d, 0xb08a, 0x4283, 0x429a, 0x1730, 0x2b83, 0x414d, 0x3df6, 0x41ed, 0xbf0e, 0x19fd, 0x178e, 0x4158, 0xba46, 0x418b, 0x1030, 0xb033, 0x405b, 0x1e2f, 0x4063, 0xbfca, 0xbaf3, 0xb21d, 0x43dc, 0x412f, 0x42f0, 0xba72, 0xb040, 0x426e, 0x3339, 0xba6f, 0x10b4, 0x466c, 0x426b, 0xb276, 0x26bb, 0x42ea, 0x1338, 0x41f2, 0x4252, 0x41a3, 0x3467, 0x445c, 0x1034, 0x4577, 0x4277, 0x446c, 0x1933, 0x4032, 0xbfc6, 0x469c, 0xba1b, 0x2658, 0x4049, 0xb24d, 0x43ae, 0x42c8, 0x4316, 0x40cc, 0xb22b, 0xbf60, 0x3670, 0xba72, 0xb0e2, 0x40b8, 0xba6b, 0xb214, 0xbfd1, 0x428b, 0x1876, 0x1f3a, 0xb06f, 0x435e, 0x447a, 0x427d, 0xba75, 0xa04f, 0xaaee, 0x4222, 0x194f, 0x2702, 0xb05d, 0x015b, 0x4129, 0x425b, 0x4275, 0xaced, 0xb228, 0x04a5, 0x0d2c, 0x4140, 0xa285, 0xbfb3, 0x4208, 0x42c9, 0x380a, 0xbad3, 0x41c1, 0x4290, 0x4294, 0xba2b, 0xba5a, 0xbaed, 0x1fef, 0x4548, 0xb209, 0x43fd, 0x4233, 0xb0dd, 0xbff0, 0x418f, 0x4132, 0x4081, 0xae38, 0xb0e6, 0x4343, 0x427b, 0x1b87, 0xbf3c, 0xb051, 0x296c, 0xba3f, 0xa2de, 0xbf89, 0x4031, 0x1e9c, 0x429e, 0xb085, 0xbf90, 0x11c4, 0x40dc, 0x1dc6, 0x1d74, 0x4331, 0x4014, 0xb29f, 0x3445, 0x14a4, 0x4216, 0xbf51, 0x40f2, 0x4303, 0x4592, 0x42e2, 0x2058, 0xb296, 0xbfbc, 0xbfa0, 0x42d4, 0xbac6, 0x0f05, 0x4584, 0xba26, 0x4059, 0x26b3, 0x419d, 0xb2ac, 0x4261, 0xb200, 0x4201, 0x41cf, 0x4148, 0x4101, 0x43b7, 0xbf6f, 0x1c26, 0xb2f8, 0x426c, 0x1cce, 0x4438, 0xbaf2, 0x193a, 0x44d3, 0x1bf6, 0xba76, 0x39d2, 0x1223, 0x4243, 0xae1b, 0x405d, 0x40a8, 0x187c, 0x420f, 0xb226, 0xb050, 0x41ea, 0x418d, 0x42c6, 0xbf54, 0xa0f9, 0x38d5, 0x185c, 0x432a, 0xbfb4, 0xb2c7, 0x11a8, 0xb277, 0x4201, 0x41ab, 0xbf27, 0xb2ac, 0x1ab3, 0x3b89, 0x40f5, 0x2c46, 0x42ea, 0x43f4, 0x16c3, 0x4264, 0x3e2e, 0x4258, 0x4111, 0x40af, 0x01fe, 0x40b5, 0x0924, 0x1cb3, 0xba0f, 0xa039, 0x32d7, 0xbf80, 0x0949, 0x1bbe, 0xba79, 0xba3e, 0x41da, 0x4395, 0x4408, 0x18e5, 0xbf9e, 0x1ef5, 0x2b17, 0xa1b6, 0x1cb7, 0x42bb, 0x3992, 0x4109, 0x419c, 0x4445, 0x41d9, 0xbafc, 0x412d, 0x01bc, 0x4433, 0x434c, 0xba5a, 0xb207, 0xa3d7, 0xb090, 0x43bb, 0x42b1, 0xbfd7, 0x42e9, 0x4025, 0x331b, 0xad80, 0x41be, 0x430b, 0xb0c8, 0xbacd, 0x3b83, 0x1eca, 0xb073, 0x417d, 0x421a, 0x2cbc, 0x460d, 0xb224, 0x0041, 0xbfd0, 0xba2c, 0xb0d7, 0x42aa, 0x4360, 0xb239, 0xbf13, 0x4232, 0x43dc, 0x1d4d, 0x4172, 0x432f, 0xb217, 0x4014, 0xaf57, 0xb269, 0x4214, 0x401f, 0x4615, 0xba6d, 0x45f5, 0x3366, 0xba10, 0x41a7, 0x4320, 0x433c, 0x41aa, 0x4447, 0x43c4, 0x06b7, 0xbf13, 0x421a, 0x431e, 0x4337, 0xba43, 0x403e, 0xb281, 0x0bbe, 0x41ee, 0xa920, 0x4234, 0xb2f2, 0x43f3, 0x35f2, 0x41aa, 0xb247, 0x378e, 0xb214, 0x3c80, 0xba55, 0xb2fc, 0xb2ea, 0x402b, 0xb2bd, 0x4004, 0x4434, 0x443b, 0x0d49, 0xbfc5, 0x2a68, 0x4333, 0x4616, 0x421e, 0xb2b9, 0x4359, 0x43f8, 0x421a, 0x41bd, 0x16fe, 0xa8e2, 0xb25a, 0x435a, 0xb0d4, 0x416b, 0x1ea0, 0x4035, 0xb0ff, 0x4280, 0x31ab, 0xbf0c, 0x4043, 0x1d5c, 0xb01e, 0x414d, 0x43d1, 0x4283, 0x4008, 0x35e1, 0xb2c5, 0x4080, 0x26fa, 0x35a4, 0x2fee, 0xae2d, 0xbf94, 0x332b, 0x1c4b, 0xb234, 0xba72, 0x1f99, 0x4375, 0xb09b, 0xbfb1, 0x23e8, 0x1a82, 0xbad3, 0x4229, 0x40fa, 0x4271, 0x4454, 0x3f9c, 0xbf7f, 0x43dd, 0x43cf, 0x4051, 0x3ddd, 0x201d, 0xb293, 0xb260, 0xba02, 0x0725, 0x434e, 0x46b5, 0x46e9, 0x1748, 0x428b, 0xbac6, 0xb0b7, 0x41c3, 0xbaf9, 0xbf96, 0xb26d, 0x40bf, 0x1c2a, 0x41ee, 0x3251, 0x402b, 0x436d, 0xb012, 0x28c6, 0x3e10, 0x4472, 0xb064, 0x4254, 0x4008, 0x4326, 0x4187, 0x04fd, 0xb0f8, 0x4656, 0x400f, 0x1277, 0x3d1b, 0xba3a, 0x42bf, 0x43f3, 0xbf69, 0x4153, 0x43ad, 0x4695, 0x0135, 0x436b, 0x4167, 0xba22, 0xa2a9, 0x42f8, 0xbf0e, 0x437a, 0x1d74, 0xb01f, 0x017c, 0x1991, 0x42bc, 0x1faa, 0x43d2, 0x401f, 0xbf35, 0x43fc, 0xa8a4, 0x1b70, 0xb016, 0x0929, 0x2e6b, 0x4325, 0x4558, 0xb05d, 0xbf2e, 0x414c, 0x42cd, 0x4391, 0x433b, 0x4301, 0xbaeb, 0x1b75, 0xbfbb, 0xae43, 0x44aa, 0x41b4, 0x42a7, 0x1b64, 0x18d4, 0x2cac, 0xb222, 0xb2c7, 0x42c9, 0xbf90, 0xa53a, 0xbad0, 0x41b7, 0xb099, 0x46e3, 0xba47, 0x438d, 0x4244, 0xb0b5, 0x20f6, 0x456b, 0x433d, 0x4648, 0xba3f, 0x3fc9, 0xbf31, 0x194d, 0xac62, 0x42bb, 0x4315, 0x40ce, 0xaedd, 0xad17, 0x4026, 0x1270, 0xb0b2, 0x17c4, 0x1fbc, 0x417c, 0x1a36, 0x3eb7, 0xb004, 0xb03e, 0x4163, 0xba06, 0x424b, 0x43b4, 0xa91d, 0x4127, 0xb016, 0x4041, 0xb2db, 0x1abd, 0x2772, 0xbf2b, 0x410c, 0x1bc0, 0xbae4, 0x407f, 0x412b, 0x1900, 0x410f, 0x4255, 0xba25, 0x464a, 0xbf02, 0xbad9, 0x418a, 0x431a, 0xbad0, 0xb0b3, 0x4379, 0x442d, 0x4104, 0x13b2, 0x1b63, 0xbfc0, 0x1a55, 0x4271, 0x1d04, 0xb009, 0x42bd, 0x405c, 0x4175, 0xb268, 0x437d, 0xbfa3, 0x3b47, 0xbaec, 0xbae7, 0x0eca, 0xb275, 0x42d0, 0xb24a, 0x4369, 0x404d, 0x086e, 0x409c, 0x418f, 0x45b8, 0x3e1e, 0xbf81, 0x1f68, 0x435e, 0xba1b, 0x43b7, 0x40b5, 0x4064, 0x14b1, 0x13ed, 0x1849, 0xbfc9, 0xb05d, 0x40c3, 0x3623, 0xb2a0, 0x184d, 0x411c, 0x417e, 0xbf7a, 0x46e8, 0x41dc, 0x4329, 0x1cd8, 0xa148, 0xb0f5, 0x2e2a, 0x414d, 0x19ce, 0x41ee, 0x44cd, 0xa6a0, 0x412f, 0x4374, 0x43b9, 0xb06e, 0x1b62, 0x3bb0, 0x4336, 0x2103, 0xbf1f, 0x4291, 0xbadc, 0x4282, 0x4122, 0xab33, 0x4368, 0xbfb0, 0xbaf9, 0xb25f, 0xb0b5, 0xba09, 0xb271, 0x23c2, 0x4565, 0xa12c, 0xb24f, 0xbf3b, 0x2a8a, 0xa3fe, 0x09ff, 0xb288, 0x42b8, 0x40c4, 0xbac9, 0xba34, 0x425a, 0xba04, 0x42be, 0x0208, 0xb003, 0x29cc, 0xb23d, 0xa865, 0x4086, 0x2faa, 0xb0ab, 0xb0a2, 0xbfbb, 0x1955, 0x43af, 0xb2ba, 0x4211, 0x2327, 0x1d02, 0x4347, 0xb28d, 0x4126, 0xba03, 0x42f0, 0x4386, 0x4322, 0xa906, 0x0326, 0xba39, 0xba27, 0x1cfb, 0x4270, 0x419b, 0x0e80, 0x4662, 0x4303, 0x2737, 0x427d, 0x1a84, 0xbf7d, 0x4603, 0x1c74, 0xb27a, 0x4170, 0x346b, 0xba00, 0x38f6, 0x2096, 0x41aa, 0x1c51, 0x2ce4, 0xb097, 0x401d, 0xbf97, 0x4132, 0x439f, 0xb2dc, 0x43fc, 0x41a6, 0x1dc0, 0x4377, 0xb2c3, 0x4326, 0x4240, 0x42bd, 0x185d, 0x1a08, 0xba26, 0x4360, 0x4307, 0x093f, 0x23f5, 0x402e, 0x4423, 0x36b8, 0xacdd, 0x16a5, 0x4210, 0x421d, 0x4317, 0x42cb, 0x281f, 0xbf00, 0xbf38, 0x429c, 0x43f5, 0x4342, 0x44e4, 0xb217, 0x1af3, 0x40b5, 0x423f, 0x4095, 0xb0f7, 0x4593, 0xbf82, 0xbae0, 0x4132, 0xae4e, 0x2512, 0x4088, 0x4355, 0x43b2, 0xb24d, 0x3d61, 0x41ed, 0x1134, 0xb252, 0xb26b, 0x2517, 0x431f, 0xaf24, 0x42da, 0xbf31, 0xa58f, 0x3660, 0x227e, 0x43ef, 0x42de, 0x4203, 0x40fc, 0x42b6, 0x4158, 0x37ff, 0x366d, 0x4002, 0x1805, 0x466d, 0xb230, 0x423a, 0xb21e, 0xba0f, 0x41b2, 0xbad7, 0xb25e, 0x1948, 0xb029, 0xbf4b, 0x423f, 0xb2c8, 0xb25f, 0xba2d, 0x46b8, 0xb220, 0xbf60, 0x421e, 0xb20f, 0xba65, 0x4475, 0xbae1, 0x3874, 0x4031, 0xbf84, 0xb2d8, 0xbfe0, 0x2aa6, 0xbfa0, 0x4287, 0x4372, 0xba1e, 0x4250, 0xba77, 0x4619, 0x4098, 0x408f, 0x432d, 0xba12, 0xbfa2, 0x2b04, 0xba75, 0xb04d, 0x4251, 0x40c4, 0x4034, 0xba30, 0xba2a, 0x4010, 0x4139, 0x43ca, 0xba41, 0x18d1, 0x43ed, 0xb2c1, 0x4167, 0x2288, 0xba05, 0xad26, 0x0d4d, 0x4376, 0xbf53, 0x4139, 0xb2e0, 0xba4f, 0x4254, 0xbf98, 0x4331, 0xbf16, 0xa567, 0x420c, 0x081b, 0x1c4e, 0x03c9, 0x4220, 0xb097, 0x0871, 0x43da, 0xb06c, 0x427e, 0xb006, 0x42e0, 0xbf70, 0x42db, 0x06dc, 0x420f, 0x40eb, 0x41a4, 0xbfbc, 0x42d6, 0xb24a, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd3cfc4df, 0x5f6feaeb, 0x1bacfb37, 0xe934848c, 0x07b4b459, 0x8beb3310, 0x3e769455, 0x1dc2cbe1, 0xaa106f96, 0xf071655c, 0xa0cd784c, 0xed40276a, 0xb836b593, 0x485f4321, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x54ca3e07, 0x419af099, 0x485f4374, 0x90be86e8, 0xa9947992, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4499, 0x4021, 0x1d2e, 0x1c30, 0xb2ac, 0x44b1, 0x4148, 0xba7c, 0x26b5, 0x42af, 0x4151, 0xa7ea, 0x1bc7, 0x10ed, 0x439b, 0xa19b, 0x4201, 0x3aff, 0x2dc8, 0xba14, 0xb244, 0xbf0c, 0xbff0, 0x45c9, 0x418e, 0xbfc0, 0x4300, 0x282d, 0x4432, 0x41fc, 0xa442, 0x400d, 0xbaff, 0x1ed1, 0x1faa, 0x1e48, 0x433d, 0xbf5b, 0x4661, 0xb2ef, 0x45b9, 0x420e, 0xb0f9, 0x24c6, 0xb2aa, 0x429a, 0x403c, 0xb24c, 0xb0a2, 0xb205, 0xabd2, 0x42a8, 0x4660, 0x4340, 0x44ac, 0x46d9, 0x407c, 0x1b8f, 0xbf18, 0x440e, 0x46e1, 0x4184, 0x4210, 0x421f, 0x1bb3, 0x437a, 0xb272, 0x4320, 0x4230, 0x15b4, 0x110f, 0x2278, 0x418f, 0x1069, 0x0cde, 0x3a07, 0x420d, 0xb2d3, 0x4073, 0x40ee, 0xb0db, 0x3342, 0x4073, 0xbf31, 0xbaee, 0x4359, 0x43c6, 0x42e6, 0x0503, 0xba7b, 0xa23c, 0x42c8, 0x1b6c, 0xbf98, 0x434f, 0x1caa, 0x43f3, 0x4143, 0x41ca, 0x19c6, 0xbff0, 0x105d, 0x42a1, 0x3045, 0x419e, 0xb2c4, 0x4395, 0x40df, 0xa984, 0x41a6, 0x18c1, 0x41cc, 0x43bc, 0x211d, 0x429f, 0x324a, 0x102b, 0x40f3, 0x404a, 0xbf17, 0x40d2, 0x0930, 0xafec, 0x25ca, 0x43f6, 0x1ddd, 0x418d, 0x41d7, 0x43de, 0x18e2, 0x43b2, 0x43a8, 0xa27b, 0xba5e, 0xba38, 0x42fa, 0x4176, 0x1e51, 0xb28a, 0xac01, 0x093b, 0x42a1, 0xbff0, 0xb0fe, 0x29c7, 0x0dae, 0x403b, 0x44c9, 0xbf4c, 0x3212, 0x0a25, 0x43a6, 0x14ea, 0xb095, 0x4293, 0x42bf, 0xba44, 0x2636, 0xbaf8, 0x446a, 0x1967, 0x428c, 0xb0e5, 0xba08, 0xb0d3, 0x4081, 0x4293, 0x4208, 0xbfb0, 0x038e, 0x3581, 0x4484, 0x40b4, 0xba43, 0x424e, 0x4000, 0xbf38, 0x4054, 0xb2ff, 0x44e5, 0x41d8, 0x4291, 0x45eb, 0x43c2, 0xba56, 0x1c4e, 0x44bc, 0x41d0, 0x3ee3, 0x00fb, 0x4038, 0xa75b, 0xbae2, 0xbf23, 0x44a4, 0x404f, 0xb277, 0xba23, 0x4178, 0x4620, 0x410e, 0x4028, 0x4040, 0xa3e9, 0x2281, 0x2238, 0xb211, 0xb285, 0x1adf, 0x4138, 0x240f, 0x4353, 0x428c, 0x4347, 0xbf24, 0x4360, 0x1ffc, 0xbf57, 0x412c, 0x40c9, 0x3611, 0x4062, 0xa2f2, 0x4332, 0xb079, 0x42e9, 0x15c2, 0x40af, 0x4409, 0xba47, 0x423b, 0x1b23, 0x1fc7, 0xbf3b, 0x23f9, 0x40f3, 0x4044, 0x41fa, 0x438d, 0x4081, 0x436f, 0x429a, 0x266b, 0x4277, 0x4544, 0x3a28, 0xba6e, 0x4131, 0x1d81, 0xbaf7, 0x011a, 0x415c, 0x09a7, 0x41de, 0x40cb, 0xbfb0, 0x40b7, 0xbfad, 0x42c9, 0x1993, 0x426b, 0x1201, 0x41d7, 0x42db, 0xb258, 0x41db, 0x435c, 0x447d, 0x461d, 0x43dc, 0x2410, 0x4287, 0xb08b, 0xb2c9, 0x438e, 0x433a, 0xb233, 0xad57, 0x44b8, 0x40fc, 0x4459, 0x1c30, 0x4118, 0x3dfe, 0xba53, 0x1ee0, 0x4130, 0xbfa6, 0xbac3, 0x0f62, 0xa018, 0x424b, 0x4109, 0x4351, 0x407e, 0xbf70, 0x1a92, 0x4289, 0x2f97, 0x41bb, 0x414d, 0x419a, 0xb0ab, 0x40a6, 0xb20d, 0x405e, 0x43fe, 0x422e, 0x4033, 0x41b9, 0x027e, 0x45e6, 0xbfe2, 0x4365, 0x4301, 0xa5e0, 0x430e, 0x4260, 0x40a4, 0x44cb, 0x0824, 0x2e66, 0x4490, 0x1efc, 0xbade, 0x228a, 0xbf60, 0x4048, 0x43b7, 0xbfce, 0x3516, 0x1e5d, 0xaabb, 0x2dbd, 0x36a6, 0xb2fe, 0x459a, 0xb0af, 0x2212, 0xb2f3, 0x42a2, 0x4346, 0x18fb, 0xbfda, 0xb2bb, 0xba39, 0xb261, 0xb231, 0x40f7, 0x4221, 0xb0d0, 0x427e, 0x4270, 0x463e, 0x4332, 0x40ff, 0xb086, 0x43a2, 0x41fc, 0x42d8, 0xbf33, 0xba11, 0x41f6, 0xb23d, 0xb004, 0x41ef, 0x2b68, 0x41a9, 0xbfc8, 0x41af, 0xb0bf, 0x2d51, 0xb08a, 0x41e4, 0xba33, 0x0fbb, 0x445b, 0x3c22, 0x4404, 0x42b7, 0x42da, 0x396d, 0xbfd9, 0x4069, 0x46d9, 0x40c9, 0x424b, 0xa8a1, 0x4399, 0x0771, 0x1ff4, 0xba29, 0x4232, 0x2361, 0xbf1c, 0xb248, 0x40b0, 0x45a2, 0x4171, 0x40b1, 0xb281, 0x4187, 0xb2ff, 0x1a83, 0x160c, 0x1f34, 0xb049, 0xbf2d, 0x430b, 0x4024, 0x2813, 0xbfb0, 0x4146, 0x2134, 0xb2a3, 0x42ca, 0xb0e6, 0x42c3, 0xb2aa, 0x462d, 0x41ee, 0x3025, 0xbf41, 0x152d, 0xb2ab, 0x22cc, 0x22c3, 0xb2ae, 0xba61, 0x430d, 0xbaf2, 0x42ef, 0x40f0, 0x2d16, 0x19da, 0xbf98, 0xba66, 0x1b36, 0xb202, 0x41ba, 0x43ca, 0x28ca, 0x406c, 0x458d, 0xbf5f, 0xba62, 0x42a5, 0xbf70, 0x412a, 0x42e1, 0xa8e9, 0xb000, 0xb0b3, 0x16fc, 0x45e4, 0x4345, 0xbf67, 0x4115, 0x4226, 0xaf4f, 0x41cf, 0x29e1, 0xa321, 0x42e8, 0x426e, 0x3646, 0x42ea, 0xb2f7, 0x42de, 0xac68, 0x341c, 0xb21f, 0x4352, 0xba35, 0xbfe0, 0xb0be, 0xbf24, 0x40f8, 0x1ace, 0xa6be, 0x4627, 0x445b, 0x4130, 0x30b1, 0x26c4, 0xbf00, 0xb08c, 0xb2ed, 0xba6f, 0xb219, 0xb2d1, 0x1ce1, 0x0ae6, 0xb25f, 0xbfd4, 0x05e1, 0x4648, 0xba06, 0xabfd, 0x4068, 0xb27f, 0x1a01, 0x0851, 0x42ff, 0x4691, 0x3367, 0x436c, 0xb20c, 0x4096, 0x43bf, 0x1f4b, 0x10e5, 0xb24f, 0x4325, 0xbfc0, 0x4113, 0xbf44, 0x41ee, 0x4045, 0x43d5, 0xbfbb, 0xb2cb, 0x42e1, 0x3336, 0x43e5, 0x0914, 0xb245, 0xba06, 0x198e, 0xadcb, 0x4079, 0xb23c, 0xbf3f, 0xa03a, 0x4155, 0x41d7, 0xba45, 0x37fe, 0x2b2e, 0x404e, 0xbaf3, 0x405a, 0x18c6, 0x42c6, 0x40f1, 0x402d, 0x184d, 0xa630, 0x42f2, 0xbfac, 0xb2ff, 0xb26a, 0xb2fe, 0xabc6, 0xb288, 0x4180, 0xbaf8, 0x4239, 0x1a7f, 0xb2c1, 0x40cc, 0x3a53, 0x34be, 0x4273, 0x435d, 0xb009, 0x1f2d, 0x4392, 0x424b, 0x459e, 0x43d3, 0x409d, 0x41d6, 0x3faf, 0xa208, 0x4107, 0xbf82, 0x4442, 0xa1e1, 0x1bc0, 0x42ed, 0x4068, 0x0b47, 0xb04b, 0x43fa, 0xba1a, 0xac7d, 0x1899, 0xa69d, 0x4327, 0xad06, 0x1f24, 0x1a75, 0xa2d1, 0x4257, 0x4106, 0x411f, 0x43b0, 0x4406, 0x45f1, 0xb2d5, 0x41a7, 0xb2f6, 0xa9a5, 0xbf25, 0x402e, 0x46a4, 0x41de, 0x4323, 0x1e92, 0x2b9b, 0xbfe0, 0xbfa0, 0xa5cb, 0x4276, 0x19cd, 0x0928, 0x40a9, 0xb2da, 0x4224, 0x42ca, 0xb095, 0x4399, 0x1917, 0xb260, 0xba50, 0xbf2d, 0xa243, 0xb29f, 0x42a1, 0x45aa, 0xa358, 0xb299, 0x4411, 0x4056, 0x4322, 0x1261, 0x40cc, 0x4153, 0x43d7, 0x4492, 0xba2d, 0x42fa, 0x437e, 0x40d1, 0xb0b6, 0xb28a, 0xb2c8, 0x07f6, 0x45ad, 0x4063, 0xb27b, 0x41b8, 0x1df7, 0xbf6e, 0xb225, 0xbf90, 0x1823, 0xb0a6, 0xbaf9, 0xba4b, 0x441d, 0x1bb0, 0x46bd, 0xa111, 0x43cc, 0x40bb, 0x41f1, 0xbf1d, 0x1c87, 0x409b, 0xba0a, 0x41ac, 0xaec5, 0xb24c, 0xbfab, 0x41f9, 0x45d3, 0x2e14, 0xb0fa, 0x156b, 0x4346, 0xb2b6, 0x19af, 0xbaf0, 0x418d, 0xb29e, 0xba47, 0x4342, 0x19aa, 0x3a55, 0x42b5, 0x4623, 0x42ac, 0xb0bb, 0xb246, 0xbfd0, 0x40af, 0x420e, 0x1f14, 0x250e, 0x1e9b, 0xbfca, 0x1d20, 0xbf70, 0xba73, 0x3e76, 0x1f68, 0x1b7b, 0x41d5, 0xbadb, 0x18ab, 0x42a0, 0x4024, 0x03ef, 0xbfd2, 0x1d77, 0xb298, 0x40b2, 0xb20f, 0x4102, 0x4046, 0x42a3, 0xbf8f, 0xb2b4, 0x3410, 0x1aca, 0x4032, 0x1bfc, 0x3a0b, 0xbf1f, 0x40b4, 0xb2d8, 0xb02c, 0x4288, 0x425d, 0x2c05, 0xab89, 0xbaec, 0x43d7, 0xb22b, 0xb20d, 0x4013, 0xb22d, 0x1819, 0xbf0c, 0xaba0, 0x43c0, 0xbac0, 0xba66, 0x4013, 0xb065, 0x18aa, 0x4139, 0x42bc, 0x45c5, 0x4128, 0x405e, 0x41eb, 0xba01, 0x4302, 0xaf34, 0x402e, 0x1181, 0xba70, 0x40c3, 0x191d, 0x4037, 0x40e4, 0xba4b, 0x09b6, 0x4561, 0xbfb9, 0x43ea, 0x02ee, 0x4311, 0x40ea, 0x4394, 0x4367, 0x4059, 0x4399, 0x43b5, 0xb077, 0x40af, 0x4394, 0x42d0, 0xb2b5, 0xb0f4, 0x3bf1, 0x44dc, 0x1887, 0x401f, 0x3da4, 0xbf7a, 0xb26f, 0x4421, 0x2540, 0x4107, 0x4229, 0x4389, 0xba20, 0x3378, 0xbae3, 0xba3e, 0x1c59, 0x40c5, 0x40a4, 0x1a2f, 0x1f4a, 0x432b, 0xa962, 0x41a1, 0x4631, 0xbf3b, 0x4245, 0xb295, 0xb28d, 0x42be, 0xbae8, 0x1b8a, 0x1a0c, 0x44fc, 0x469a, 0x42b7, 0x1ba4, 0x41b3, 0x4361, 0x4241, 0x4034, 0x4099, 0x433d, 0x438f, 0x41c4, 0xb2e4, 0xbf92, 0x426d, 0x4023, 0xba0b, 0xba3c, 0x40d4, 0x4152, 0xb23a, 0x425e, 0x43ac, 0x4146, 0xbfcd, 0x1617, 0x41a8, 0x08af, 0x4364, 0xbadc, 0x4348, 0xb295, 0xb2ee, 0xb21f, 0x400c, 0x4214, 0x4253, 0x4193, 0x4544, 0x1ff0, 0x43bc, 0x2489, 0x4145, 0x414c, 0x409c, 0xa728, 0xb09b, 0xbf19, 0x431f, 0xba0d, 0xba6c, 0x4201, 0x4063, 0x418e, 0x2b73, 0x0d77, 0x4574, 0x408f, 0x31b6, 0x077e, 0xbf48, 0xb2ac, 0xb2be, 0x1088, 0x411e, 0x42e5, 0x410d, 0x428d, 0x4319, 0xbf36, 0x354d, 0x0e3b, 0xb25f, 0x4316, 0xbfa3, 0x419a, 0x039b, 0x41b6, 0x04fd, 0x4583, 0x4284, 0x43c1, 0xbf24, 0x14e3, 0x4366, 0x438f, 0x4388, 0x4335, 0x423b, 0x4181, 0xb227, 0x0e84, 0x0fa9, 0x43a6, 0x1264, 0x0b9d, 0xb0e9, 0xb283, 0x46f2, 0x0ca3, 0x4364, 0x4311, 0xbac1, 0x4115, 0xb0d0, 0x2aa3, 0x1923, 0xbf5b, 0xb04e, 0x35af, 0xa9d4, 0x42c9, 0x4315, 0xbf60, 0x3bfd, 0x0a60, 0xbad1, 0x405a, 0x18b4, 0xb20a, 0xb075, 0x432b, 0x43ea, 0x4272, 0xb2ab, 0x4218, 0x430a, 0x13ee, 0xba22, 0xbf8e, 0x17e6, 0xb260, 0x43ed, 0x439a, 0xb29d, 0x1e98, 0x4355, 0xba54, 0xaffd, 0xb2d2, 0x4394, 0x404e, 0xb08f, 0x3bdc, 0x0f8b, 0x4243, 0x4382, 0xbf48, 0x4138, 0x44fd, 0x422b, 0xb08f, 0x1150, 0xbf73, 0x41d6, 0x1b03, 0xb0bf, 0x432e, 0x283d, 0x45a8, 0x0fa2, 0x0992, 0x40c4, 0x46d2, 0x43d8, 0x42a4, 0x4119, 0x4016, 0x427d, 0xb0c3, 0xb29e, 0xa0df, 0x4258, 0xba3a, 0x1df4, 0x0f25, 0x2f1c, 0x435a, 0x421c, 0x41d4, 0xbf07, 0xb2f7, 0x1a9e, 0x43a2, 0x43d8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5cc0f53c, 0x72fcd234, 0xde457376, 0x49d4e05a, 0x758c22b2, 0x79c1668a, 0x3635ed63, 0xeac80e34, 0x347d2ad7, 0xd42d60b3, 0xe8a20e81, 0x381745c5, 0x0ec66078, 0xe13b4962, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0x0000b3bc, 0x00000000, 0x8e4f0000, 0xffff4c43, 0x00004c4a, 0x00000000, 0x00004c43, 0x0000051b, 0x6c94709c, 0x00090000, 0x00000000, 0x55a4c7ed, 0x3cb8de89, 0x0000173b, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb2ea, 0x42ba, 0x46c5, 0x422b, 0x1be5, 0x02d1, 0x1caa, 0x41bc, 0x43bd, 0x1dfd, 0xa8ec, 0xb044, 0x0ec5, 0x0aa4, 0x4022, 0x45c5, 0xb00b, 0x42da, 0x04c3, 0x00d9, 0x1e50, 0xbf03, 0x4184, 0x4254, 0xb282, 0x00fc, 0xb041, 0xb27c, 0xbafd, 0x18fc, 0x417c, 0x41d6, 0x4313, 0xba21, 0xb237, 0xbfb4, 0xb29f, 0xb260, 0x1acf, 0x21b0, 0xb062, 0xbac1, 0xbacb, 0x41aa, 0x1399, 0xb255, 0xb28d, 0x1f91, 0x408a, 0x1391, 0xb200, 0x40ba, 0x4234, 0xbad2, 0x4031, 0x417a, 0x43fe, 0x40ff, 0x4131, 0x44c2, 0xbf6d, 0xba23, 0xb22a, 0x1a46, 0x432b, 0x1a26, 0x1525, 0x4148, 0x421b, 0x3ac6, 0xba73, 0xb007, 0x448c, 0xb211, 0x40e9, 0xba0c, 0x2a7e, 0x409b, 0x4478, 0x4495, 0xba61, 0x40e5, 0x465d, 0x4110, 0xb07e, 0xbf04, 0x4174, 0x1ce1, 0xb2b7, 0xb26a, 0x40e2, 0x11ea, 0x091b, 0x46d5, 0x1def, 0x4351, 0x4244, 0x405a, 0x4109, 0xa87b, 0xb249, 0x44b1, 0xba4f, 0xb038, 0xb083, 0xbff0, 0x4016, 0x43f0, 0xb092, 0x03f6, 0x19ad, 0x4287, 0xbf76, 0x43b0, 0x4356, 0x3e6a, 0x1aa0, 0x41c5, 0xb2d4, 0x1cb6, 0xba0c, 0x43ce, 0x45f2, 0xb066, 0x361b, 0x294f, 0x4133, 0xaa78, 0x4659, 0x4019, 0x4310, 0x4158, 0x412d, 0x4077, 0x41d9, 0xba7e, 0xbf96, 0x112e, 0x3932, 0xb0cf, 0x4395, 0x4237, 0x2cd3, 0x35bb, 0x03ff, 0x3c3f, 0xa63b, 0x4352, 0x4110, 0x04e4, 0xaad1, 0xb05a, 0xb2af, 0x4101, 0x4309, 0x43c7, 0xa7cc, 0x1b1d, 0xb2bc, 0x42b2, 0x446d, 0x42e2, 0x4044, 0xbfae, 0xbfd0, 0xa92b, 0x2ca8, 0xba7a, 0x0c15, 0x18af, 0x10c6, 0x4125, 0x411c, 0xb290, 0x40c7, 0x38e2, 0x425a, 0x26d4, 0x45ca, 0x46b5, 0xa9af, 0xbfab, 0x401a, 0x3480, 0x43e2, 0xb205, 0xb06b, 0xb0ce, 0x4213, 0xba35, 0x42d3, 0x1d9c, 0x40c8, 0x43dd, 0x42cc, 0x264c, 0x0f86, 0xbf8a, 0x18b3, 0xb0b6, 0xbff0, 0x08c3, 0x146a, 0x43a1, 0xbfad, 0xb024, 0x419b, 0xada4, 0xb207, 0xa205, 0xb2b0, 0x4013, 0x1ffc, 0xb2dc, 0xb2a6, 0x4189, 0xa033, 0x373b, 0xa54c, 0x4229, 0x1f08, 0xbf17, 0x426b, 0xbf90, 0x18ca, 0xa154, 0x41c6, 0x3c30, 0x426a, 0x431a, 0x1add, 0xb0d0, 0xad69, 0x1d25, 0xbf1f, 0xb2c7, 0x26fe, 0x1dc6, 0xacb6, 0xba38, 0x4554, 0x19cd, 0x4167, 0xa82c, 0xb2d0, 0x15fe, 0x2a17, 0x43cf, 0x45bc, 0x4002, 0x438b, 0xb24a, 0x1bca, 0xb00a, 0x40e5, 0x4367, 0xbfa5, 0x425c, 0x1a88, 0x0641, 0x41e3, 0xb22c, 0xb245, 0x42d5, 0xb23f, 0x4615, 0x43fc, 0xbaed, 0x4272, 0x414c, 0xbf31, 0x41c1, 0xb2df, 0xb2cb, 0x402b, 0x42a6, 0x4356, 0xbf3e, 0xb2e9, 0x2814, 0xbfa0, 0x4143, 0xbfe0, 0x0b79, 0x1b81, 0x38a3, 0x412f, 0x20d2, 0x4068, 0x42cb, 0xbae8, 0x2a4f, 0x1d57, 0xb09d, 0x4168, 0xb208, 0x41e9, 0x408c, 0x357e, 0x469d, 0x43eb, 0x4189, 0xbac2, 0x40d8, 0x4580, 0xbf7a, 0x0b11, 0x4070, 0x4471, 0x43f4, 0x43b0, 0x4586, 0xba4a, 0x346a, 0xa7e9, 0x43b2, 0xbac3, 0x0423, 0x40d6, 0x248d, 0x40a5, 0x420a, 0x438f, 0x427b, 0x4202, 0x31ec, 0x2e50, 0xba26, 0xbf5e, 0x41f2, 0xb08f, 0x42ab, 0x2f2e, 0x4111, 0xbfd0, 0x4324, 0x40de, 0x4236, 0xba1b, 0x02e1, 0x3f0c, 0x0a8d, 0xa1e9, 0x092b, 0x42d4, 0x45dd, 0x43f4, 0x3b18, 0xaff9, 0xba2f, 0xba68, 0xb0de, 0x19b1, 0xbf37, 0x43c5, 0x1642, 0x41d7, 0xbf00, 0x311f, 0xbfd0, 0x4254, 0x423a, 0x4124, 0x1eb4, 0x4364, 0x4365, 0x40e0, 0xbf3e, 0x3c12, 0x43c1, 0x2fbc, 0xb0bd, 0xbac3, 0x4014, 0x41f6, 0x40dc, 0x40ac, 0x4330, 0x1e86, 0x438b, 0xb0d0, 0xac1f, 0x055f, 0x378c, 0x1f34, 0x1f9c, 0x0795, 0xb0c8, 0x410b, 0x4050, 0xbfdb, 0x41eb, 0x3e25, 0x4171, 0x432d, 0xb2d0, 0xb25f, 0xb281, 0x41e1, 0x432f, 0x4073, 0xbf60, 0x40fc, 0xbaf7, 0xb0ba, 0xaac8, 0x4318, 0x410c, 0xa369, 0xb08a, 0x42f8, 0x1c9b, 0xbf78, 0x34a5, 0x3209, 0xadae, 0x4272, 0x0d71, 0xa43d, 0x1180, 0x4229, 0x0689, 0x4313, 0xb208, 0x444c, 0x3a48, 0xba19, 0x4007, 0x4107, 0x2a6a, 0x41be, 0xb2ee, 0xbf42, 0x4332, 0x1870, 0x10cb, 0xb078, 0x411f, 0x29f2, 0xbfe0, 0x20df, 0x41f2, 0x39f3, 0x3ecf, 0x42fc, 0x1623, 0x43a9, 0xbf73, 0x40e3, 0x43ff, 0x40d3, 0x0ef5, 0x43a4, 0xb2aa, 0x4187, 0x403a, 0x1b1c, 0xb0be, 0x354e, 0xb09c, 0x402c, 0x2b3a, 0x0e11, 0x1335, 0xbf60, 0x44f1, 0x2ecf, 0x1b37, 0xb235, 0xb244, 0x428b, 0xb0bb, 0xb2d6, 0x19a6, 0x4589, 0xbf04, 0x1a8c, 0xb277, 0x13d1, 0x4394, 0x1f6f, 0xab11, 0x4335, 0x40f2, 0xb0b6, 0x1fb6, 0x403c, 0x40b0, 0x4274, 0x4345, 0x4282, 0x2a39, 0xb2e2, 0x4327, 0x4278, 0x44ab, 0xb014, 0x2313, 0x425e, 0x41e4, 0xbf6f, 0x1cd2, 0x40c1, 0x3319, 0x40dc, 0x43a4, 0x1839, 0x0638, 0x42b7, 0xbf17, 0x42b9, 0x0c5b, 0xb0e4, 0xb29a, 0xba78, 0xb290, 0x1ed9, 0xbf05, 0xb2ff, 0x42dc, 0x4253, 0x417a, 0x4304, 0x16d8, 0x3184, 0xb257, 0x3bd3, 0x419f, 0x3bd5, 0x4068, 0x10b7, 0x0f9e, 0x40a9, 0x4263, 0x414f, 0x43cd, 0xbfd0, 0xb0d9, 0x411e, 0xb26b, 0x3f5f, 0xbad3, 0x148b, 0x187a, 0x42ce, 0xb296, 0x1d75, 0xbf15, 0x4637, 0x42d9, 0x44d8, 0x4089, 0x05a3, 0x43d8, 0x43ad, 0x4091, 0xb21a, 0xb2bd, 0x4027, 0x1a6a, 0x40a2, 0x41ac, 0xa8f3, 0x41f9, 0xbf31, 0x4345, 0x3a56, 0xa529, 0x4215, 0x2faa, 0xb2b4, 0x195b, 0xb275, 0x4181, 0xba12, 0x43a6, 0x43f5, 0x315d, 0x4629, 0x1d9f, 0xbf94, 0x4212, 0x417b, 0x417c, 0xbfd1, 0x4614, 0x427d, 0x4140, 0xb2ae, 0x41ad, 0x1f90, 0x4286, 0x42ff, 0xac6c, 0xa113, 0xb29a, 0xb2e0, 0x0304, 0x46ed, 0xb2b4, 0xbf2c, 0x1079, 0x4056, 0x424b, 0x40b0, 0x43c2, 0x400d, 0xb2ba, 0xa574, 0x4354, 0x1145, 0x4567, 0x46a9, 0xba63, 0x4684, 0x4259, 0x4396, 0xb231, 0x4388, 0x4047, 0x40c5, 0x42b9, 0xba19, 0x31fc, 0xb012, 0x4154, 0x1a7b, 0x435c, 0xbf3b, 0x4211, 0x4136, 0xafba, 0x1a25, 0x4348, 0xb049, 0xba0e, 0xba6d, 0x43bd, 0xb231, 0xb2b6, 0x464e, 0x17cd, 0x4050, 0x438a, 0x410a, 0x41f7, 0x0d50, 0x41f6, 0x433c, 0xb0a6, 0xbaca, 0x40c2, 0xbf48, 0x4210, 0xbfc6, 0xba0c, 0xba18, 0x4641, 0xb2b3, 0x404f, 0x44f0, 0x4317, 0x2b5f, 0x4077, 0x458e, 0x1005, 0xb2a4, 0xbf42, 0x1ce5, 0x10f8, 0x4340, 0x43ca, 0xbfe0, 0x182e, 0x38a8, 0xa2ad, 0x42ed, 0x4592, 0x2346, 0xbf70, 0x43a2, 0x4037, 0x1360, 0xbafe, 0xb0bb, 0x43b2, 0x19c0, 0xbfa1, 0x4336, 0x4397, 0x1f3f, 0xba20, 0xb0e4, 0x4134, 0x1e98, 0x1926, 0xba44, 0x434d, 0xb087, 0x4673, 0xb20e, 0xba0f, 0x262a, 0xa308, 0x2693, 0x1e06, 0x424b, 0xb096, 0x439f, 0x36fa, 0x42d6, 0xb06a, 0x40fb, 0xbf13, 0xaa9f, 0x1d6c, 0xb2e0, 0x4363, 0x10e8, 0x1c58, 0xb290, 0xb072, 0x2deb, 0x421e, 0x1897, 0x4154, 0xab61, 0xba24, 0xbf70, 0x4326, 0x406a, 0x4595, 0x4169, 0xb228, 0x1e7f, 0xb20d, 0x188c, 0xb2a2, 0x1263, 0x0cbc, 0x46b0, 0xbf37, 0x42b0, 0x4374, 0x4312, 0x2515, 0xaa4a, 0x4150, 0x41fb, 0x434e, 0x42e0, 0x42f6, 0x408c, 0x26be, 0x42ea, 0x1edf, 0x1c47, 0x402c, 0x435e, 0xbf59, 0x3faf, 0x4462, 0xba3f, 0xb27d, 0xbf9d, 0x4207, 0xaede, 0xb223, 0x437f, 0x1786, 0x42a8, 0x4262, 0xb294, 0x1a29, 0xb2d6, 0x40cf, 0x1804, 0x42cb, 0xb025, 0xb265, 0x4298, 0xba0b, 0x4627, 0xb2d2, 0x40b4, 0xbf9b, 0x407c, 0x42b8, 0x18d0, 0xb2d0, 0xa7c8, 0x3b60, 0xb259, 0x2e49, 0xb228, 0x42dd, 0xb2b1, 0x237d, 0x43cd, 0x4001, 0x1a2b, 0xba71, 0xbf3f, 0x15f9, 0xb095, 0x01dc, 0x365f, 0x142e, 0xab9e, 0x4242, 0x4629, 0x424d, 0xb229, 0x404c, 0x0a33, 0x41a0, 0x455f, 0x077d, 0x1eb8, 0x1ba4, 0x13d8, 0x3aba, 0x1dbd, 0x4289, 0xbfca, 0x4253, 0x31bd, 0xb025, 0x27c1, 0x42dc, 0x1909, 0x37cd, 0xb28f, 0x43f0, 0x42af, 0x4185, 0xba6f, 0x466a, 0x15c5, 0x42e2, 0x3d8e, 0xba2d, 0x27ef, 0x45e6, 0xbfa3, 0x4204, 0x41b6, 0xb2bc, 0x4025, 0xbf67, 0x0886, 0x3457, 0x1cc4, 0x276c, 0x42a4, 0x3f63, 0xbad3, 0xbf16, 0xba62, 0xb087, 0x1e84, 0x4015, 0xb068, 0x31ca, 0x411e, 0x1fd1, 0xb01f, 0x4607, 0x1d77, 0x461c, 0x43cc, 0x401f, 0xb2b5, 0xa43d, 0xba66, 0xb28b, 0xbf16, 0xb209, 0x4097, 0x417e, 0x40bb, 0xa1b2, 0x4443, 0x4042, 0x405c, 0x1bfc, 0xba06, 0xb098, 0xb014, 0x40ec, 0x4159, 0x417a, 0xb2a6, 0x2ff6, 0xb21b, 0x44f8, 0x420a, 0xb239, 0x4156, 0xbf12, 0x1d1f, 0x42b6, 0xba5f, 0xb0a9, 0xbfe2, 0x43bd, 0x4384, 0x44bd, 0x4085, 0x43e4, 0x199f, 0x41bc, 0x186b, 0x429b, 0xb285, 0x1bf3, 0xb0ec, 0x4155, 0x40ff, 0x4019, 0x439f, 0xb0ad, 0x35bd, 0x21d8, 0x417f, 0x434d, 0x180b, 0x1a8d, 0xab25, 0xba3e, 0x1832, 0xbf37, 0x40f9, 0x427a, 0x4184, 0xba66, 0xba18, 0x4022, 0xbf60, 0xb2cc, 0xb2f2, 0x4127, 0x433b, 0xb0bc, 0xba3c, 0x4189, 0x439d, 0xb0e3, 0x4685, 0x3f24, 0x1d66, 0xbf08, 0x43c5, 0xb2de, 0x40f3, 0x183e, 0x430c, 0x40ea, 0x19e2, 0x1cd2, 0x1cf6, 0x3a92, 0xb0d3, 0xb2c3, 0x4236, 0x4313, 0x3162, 0x406d, 0xbf90, 0x42d3, 0x46d4, 0x40e9, 0x41a1, 0xa0cc, 0x4071, 0xba15, 0xbfa3, 0x3d16, 0x4361, 0x4247, 0x04b4, 0xb0ee, 0x1f85, 0x3042, 0x41a7, 0xb242, 0x428b, 0x4322, 0x420e, 0x38df, 0x4083, 0x41f9, 0x06c0, 0x420b, 0xb2a2, 0xb20f, 0xbf63, 0x4148, 0x0544, 0x4066, 0xa5e5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x1ad1dd6f, 0x8a780aab, 0x92d86fe7, 0x3dcc1243, 0x715b8d62, 0xdd61381a, 0x5655e1f0, 0x38a9fd45, 0x5503834e, 0x5c5e12c5, 0xc6ca1316, 0xf112ee7a, 0x4c667450, 0xbace516f, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x78000000, 0xe27ff7b4, 0x00000000, 0x00000000, 0xff7c0000, 0x00001b68, 0xa713ffdf, 0xfffff7b4, 0x05001840, 0x00000000, 0x1bcd9664, 0xf112ee7a, 0x1bcd9664, 0xa713fcfc, 0x00000000, 0x600001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x43ea, 0xb039, 0xb0f0, 0x4163, 0x08a9, 0xba5c, 0x410d, 0x06ef, 0x42b8, 0xb095, 0x4045, 0x456a, 0xbf74, 0xabfb, 0x4084, 0x159e, 0xb230, 0x4372, 0xb005, 0x404f, 0x4017, 0x43f0, 0x43da, 0x40c7, 0xbf9f, 0x4207, 0x44c0, 0x0047, 0x0b05, 0xbf7b, 0xbfe0, 0x4057, 0x1d47, 0x1f0f, 0x40b2, 0x403c, 0xb09f, 0x03e8, 0x4246, 0xb26d, 0xb26c, 0xbf04, 0x42d6, 0xb040, 0x4631, 0x4092, 0x430e, 0xbf00, 0x40cc, 0xb243, 0x214e, 0x06a1, 0xbfc6, 0x40b0, 0x4631, 0x43cd, 0x4340, 0x13ca, 0x14d1, 0x4603, 0x4079, 0x2c83, 0x421d, 0xbacb, 0xafa3, 0x409c, 0xb2f9, 0xa17d, 0x30ee, 0xba61, 0x4207, 0x0a66, 0x420d, 0x4189, 0x41ea, 0x20f3, 0x42b9, 0xb20d, 0xa046, 0x4388, 0xb2bc, 0xbf7c, 0xb010, 0x43a3, 0xb21d, 0xb22f, 0xb2c8, 0x40bb, 0x19aa, 0xa4a9, 0xac73, 0x4296, 0x2be3, 0x0f78, 0x4107, 0x421a, 0x16f8, 0x427c, 0x444b, 0xbacb, 0x40b4, 0x0480, 0xb0b8, 0xba71, 0x402b, 0x40b5, 0x46a8, 0xbf34, 0xb25d, 0x2418, 0xaac1, 0xb251, 0x2ca1, 0xbf60, 0x4012, 0xbae5, 0xba4d, 0xba63, 0x4172, 0x41bd, 0xa74e, 0xbad9, 0x1f15, 0xb2d0, 0xb26e, 0x2e39, 0x413e, 0x4026, 0x4136, 0xb0eb, 0xb2fe, 0x4211, 0xbf76, 0x44fb, 0x43da, 0x3f14, 0x01c4, 0x41d7, 0x41ee, 0x0326, 0x237a, 0xb026, 0xbfbf, 0xb286, 0x40d8, 0x183c, 0xb2fc, 0x43cd, 0x41d4, 0x1b4c, 0x42c7, 0x03da, 0x4156, 0x40ce, 0x1888, 0x02d2, 0xba79, 0x40ef, 0x4081, 0xb23b, 0x4099, 0x417b, 0xbfa0, 0x4152, 0x4158, 0xbff0, 0xad95, 0x4477, 0x4176, 0x4132, 0xbadb, 0xbf0c, 0xba3a, 0x43e2, 0x1e39, 0x438f, 0x2393, 0x413b, 0x423d, 0xba05, 0x30e8, 0x4383, 0x4081, 0x1dc3, 0xba17, 0xbf9b, 0x294d, 0xac43, 0x18c2, 0x4362, 0xb2d4, 0x4349, 0x420d, 0x42a7, 0x42df, 0x2363, 0x40c8, 0x226c, 0x417a, 0xbfd0, 0x435c, 0x411d, 0xb07f, 0xba54, 0x1da1, 0xa883, 0x1b72, 0x4022, 0x43cf, 0x129b, 0x0952, 0x1a40, 0x1aaf, 0xbf3a, 0x406e, 0xb274, 0xb0a9, 0xbfe0, 0xba4e, 0x4004, 0x38de, 0x41cc, 0x4167, 0xb2a4, 0x413b, 0x1b27, 0xb072, 0x1ed4, 0x3b32, 0xb237, 0xb0fe, 0x1821, 0x414d, 0x41f1, 0x42b0, 0xbf1b, 0x426e, 0x460d, 0x4246, 0x41b1, 0x1c38, 0x4473, 0x413c, 0x4143, 0x24b8, 0x43bf, 0x43dc, 0x3a1b, 0x41fd, 0x4386, 0xba70, 0x41e7, 0x2256, 0x4086, 0x4341, 0x1af4, 0xb21b, 0xbfa0, 0x1f37, 0x4598, 0x431d, 0xbf63, 0x2db5, 0x416c, 0x428d, 0xb2a5, 0x4558, 0xb2f6, 0x41a8, 0x4014, 0x43f5, 0x33d6, 0x41cb, 0xb268, 0x4275, 0xb023, 0x438e, 0xbfe4, 0x415e, 0x19b9, 0x468a, 0xb252, 0x3626, 0x43ea, 0x2d17, 0x40aa, 0x42aa, 0xbfe0, 0x43a3, 0x1b9b, 0xba61, 0x4330, 0x0dbd, 0xa126, 0x40c7, 0x41de, 0xbfc6, 0x404d, 0x46a4, 0x4647, 0x4340, 0x40c8, 0xbf80, 0x42c4, 0x1df3, 0xbf90, 0xa7ac, 0xba4e, 0x4308, 0x0dac, 0xba47, 0x1d7a, 0xb06e, 0x3e91, 0x2e8b, 0x4008, 0x1c78, 0xb219, 0x4161, 0x410d, 0x4076, 0xbf9d, 0xa660, 0x422a, 0x42f6, 0x19e9, 0xbf9a, 0x43fb, 0x421e, 0xb28a, 0x40a0, 0xa3a0, 0x4081, 0x1990, 0xba24, 0xbf02, 0x41d5, 0x4078, 0x4176, 0xba52, 0x1f86, 0xb042, 0x430f, 0x2816, 0x428f, 0x4084, 0xa29f, 0xb019, 0xbfc8, 0x432c, 0x40a9, 0x40a3, 0x1b29, 0x43f5, 0x3867, 0x40cd, 0x4330, 0xb277, 0x3830, 0x41d4, 0xb083, 0x4164, 0x36a6, 0x4082, 0x4141, 0x1167, 0xb2fa, 0x1a54, 0x4284, 0x43e4, 0xa27e, 0xa4e3, 0x413d, 0xbf9a, 0x08da, 0x1b35, 0x4385, 0xb232, 0x4166, 0x422c, 0xbf90, 0x412f, 0xbf2f, 0x41b9, 0x1b51, 0xba0c, 0xbfb0, 0xba34, 0xbf8f, 0x1d3d, 0x30c1, 0x439c, 0x4543, 0xb2d0, 0x1fef, 0x4359, 0x4167, 0xb29f, 0x4331, 0xb2f5, 0x4308, 0xa8a7, 0xba06, 0x400f, 0x43e8, 0x246f, 0x4206, 0x416f, 0xb22c, 0x4043, 0x42b3, 0xbacd, 0x41ac, 0xb280, 0x27e3, 0xbf44, 0x402b, 0x4009, 0xb22b, 0x423d, 0xb28a, 0xb234, 0x4114, 0x42c3, 0x2c01, 0x4344, 0x40a0, 0x41e8, 0x180b, 0x259d, 0x4296, 0x0859, 0x4004, 0xbf6e, 0x43a3, 0x42ca, 0x409f, 0xbaca, 0x1c37, 0x4085, 0x40c7, 0x4113, 0xbac5, 0xb02d, 0x1c91, 0x3cb2, 0x0819, 0x4352, 0x432f, 0x3f57, 0x3c39, 0x43a0, 0xb063, 0x430f, 0x4374, 0x4131, 0xbf52, 0x43d4, 0x432d, 0x07d6, 0xa457, 0x431b, 0xad5a, 0xb24f, 0x16dd, 0xb026, 0x22aa, 0x432c, 0xa6c9, 0x41f6, 0xac8c, 0xb07d, 0xa40a, 0x4137, 0xbf95, 0x3196, 0x41e4, 0xb2c0, 0x4362, 0xb24b, 0x407a, 0x4093, 0xbfdf, 0x0cd1, 0x3b7c, 0x427f, 0x4312, 0xa0a2, 0x20fd, 0x41cc, 0xba31, 0x424e, 0x1aa4, 0xbacb, 0xb26a, 0x4048, 0x4416, 0x43e5, 0x42d4, 0xba20, 0xbf8f, 0x4079, 0xb27e, 0x4271, 0x40ee, 0x466d, 0x2d6c, 0x1843, 0x4083, 0x4327, 0x0cc7, 0x4582, 0xbf6f, 0x424c, 0xafb3, 0x41a0, 0x4239, 0x4404, 0xbf80, 0x418d, 0x43e0, 0x1ab2, 0xbfaf, 0x43ab, 0x4357, 0x42cb, 0xb0bf, 0xba12, 0x1396, 0x41de, 0x4116, 0x2112, 0x1b76, 0x4007, 0x44e9, 0xb224, 0x14d0, 0x402d, 0x412e, 0xb2d1, 0x2afc, 0x42f2, 0x1eca, 0x3412, 0x4472, 0x4252, 0x40cd, 0xb0fa, 0x0898, 0xbf79, 0xbaca, 0x3752, 0x0b3b, 0x1f37, 0x3758, 0x42b8, 0x43bd, 0x01e7, 0x42d1, 0x400a, 0xb044, 0xb2c6, 0x40bb, 0x4384, 0x3f19, 0x43ae, 0xba2c, 0x05ef, 0xb211, 0x1e3a, 0x437d, 0xbfa0, 0xbf7b, 0xb0a2, 0x4326, 0x1c73, 0x425b, 0x412d, 0x44d0, 0x11c1, 0x1b69, 0xb2c3, 0x1f89, 0xa6b4, 0x2ed9, 0x418d, 0x1ccb, 0xaa53, 0x402d, 0x1a6b, 0x3c2d, 0x4622, 0xb2c7, 0xa4b8, 0x4066, 0x45cb, 0x10a2, 0x4010, 0x41bb, 0x4300, 0xbf15, 0xb0b0, 0xba4e, 0xb29f, 0x1d46, 0xba36, 0x43b8, 0x0c37, 0x4499, 0x1bcf, 0xae12, 0xb26a, 0x42d5, 0x4362, 0x25db, 0xba02, 0xb29e, 0x3650, 0x4564, 0x4158, 0x1d0e, 0x4132, 0x4237, 0xb06f, 0x4377, 0xbf92, 0xbfb0, 0x4151, 0x443e, 0x439f, 0x43af, 0xaf28, 0x4151, 0x40d7, 0x40c7, 0x070e, 0x467f, 0xb226, 0x4418, 0xb016, 0x4288, 0xb009, 0x3e9f, 0x2434, 0x45f5, 0xb0a5, 0x1b5f, 0xb270, 0xbf39, 0xb2b4, 0x4093, 0x169a, 0x43c0, 0x29c4, 0xbf61, 0x4206, 0x2bcc, 0x401c, 0x43b0, 0x432e, 0x07bf, 0x40a7, 0x1b14, 0xb09b, 0x1a2b, 0xa498, 0xbaf3, 0x4204, 0x415b, 0xb02f, 0x42c9, 0x4264, 0x40eb, 0x40b9, 0x2f8a, 0x162c, 0x1b93, 0x18c7, 0x40d6, 0x4484, 0xbf63, 0x407b, 0x40cc, 0x19bd, 0x4106, 0xbaec, 0x4076, 0x4271, 0xadcc, 0x42b6, 0xbf12, 0xb2fc, 0xb267, 0x40ea, 0x40c4, 0x34ed, 0xba7e, 0x0e23, 0xae59, 0xbf4a, 0x0743, 0x43d7, 0x42dd, 0x4097, 0x40ed, 0x40a8, 0x45ed, 0x1e93, 0xb213, 0x1c3b, 0x0ea9, 0x420d, 0xb038, 0xba5a, 0x42ed, 0x41e0, 0x41b2, 0x4327, 0x19c5, 0x42d8, 0x26fb, 0x43fd, 0x0f88, 0x43a7, 0x1c9b, 0xb207, 0xbfc9, 0xb2c5, 0x430a, 0x2512, 0x4356, 0x41ed, 0x4078, 0x40c0, 0xbf99, 0x4141, 0xba72, 0x4226, 0x42d0, 0xb2b2, 0x4150, 0x42d1, 0xb04c, 0xba33, 0x2f36, 0x45b0, 0x18b0, 0x1d78, 0xbf43, 0x426a, 0x4350, 0x4287, 0x406a, 0x1911, 0x4199, 0x4074, 0xbf9a, 0x43f2, 0x4344, 0x41f6, 0x0efc, 0x30a0, 0x2e3e, 0x4359, 0xb261, 0x46b5, 0x4164, 0x25dd, 0x0839, 0xa6d1, 0x4181, 0x4627, 0x419a, 0xba3a, 0xbf16, 0x3ee3, 0x158a, 0x407d, 0x424f, 0xba32, 0xb212, 0x41f9, 0xb019, 0xbf80, 0xbfa0, 0xb2af, 0xb2bb, 0xb21e, 0x427c, 0x441b, 0x40e7, 0x3e04, 0xba75, 0xbfde, 0x40dc, 0x4278, 0x2b7d, 0x3966, 0xb06e, 0x1733, 0x465b, 0x4185, 0xb2f9, 0x42dd, 0xb246, 0xbff0, 0xb019, 0x42e4, 0xb259, 0x42a0, 0xba79, 0x18d5, 0x2f05, 0x4093, 0xbf7d, 0x2891, 0x423a, 0xb201, 0xb001, 0x1ce7, 0x1de1, 0xbafd, 0xb2b0, 0x42d6, 0x41f4, 0x206e, 0xb0b6, 0x3d9b, 0x3156, 0xbff0, 0x4483, 0x46d2, 0x41bb, 0x22a6, 0x413e, 0x4180, 0x3d93, 0x46f3, 0xb2e1, 0x4064, 0x42b4, 0xb279, 0xbf6c, 0x41fd, 0x42a1, 0x396b, 0x1383, 0x42ec, 0x1e32, 0x4564, 0x312c, 0x416f, 0x43be, 0x2633, 0x1e58, 0xb29b, 0x415a, 0x384f, 0x43ec, 0x428a, 0xbad6, 0x40bb, 0xba2d, 0xbf3b, 0x41fb, 0xb26f, 0x4208, 0x407e, 0xbacd, 0x1bb6, 0xbf6c, 0x435d, 0x41db, 0x2381, 0x402e, 0x4278, 0x40cb, 0xba40, 0xbac3, 0x3c20, 0x3b4a, 0x1a38, 0x437d, 0x4100, 0x40d5, 0xb0fd, 0x46c4, 0xbfdf, 0x4251, 0xba51, 0x42c7, 0x0106, 0x0ab7, 0x1cf3, 0xa039, 0x1e55, 0x42b5, 0x43c6, 0x4614, 0x409c, 0x4383, 0xbf76, 0x45f3, 0x01b8, 0xb045, 0xba1d, 0x4000, 0xbfe0, 0x41ac, 0x42e8, 0xb055, 0xa8af, 0x4413, 0x3917, 0xba4b, 0x42a6, 0xb097, 0x3a05, 0x410d, 0x43a3, 0xa1f5, 0xba69, 0x4164, 0x295b, 0xb0d1, 0x43d5, 0xbf6d, 0x0b74, 0x41c4, 0xba32, 0x1c4d, 0x40cf, 0x41c7, 0x4591, 0xa4d4, 0x4376, 0x131b, 0x42f1, 0x4308, 0x42a9, 0x1d4d, 0x41bc, 0x425f, 0x42b1, 0xacd1, 0xaa4f, 0x45de, 0x0d5f, 0x270c, 0x1e62, 0x4353, 0x4113, 0x2cc0, 0xa57c, 0x1eba, 0x4153, 0xbfa8, 0x413a, 0x41be, 0xbf2f, 0x22f9, 0xada0, 0x4171, 0xb2fc, 0x458e, 0x405a, 0x4119, 0x42a9, 0xae09, 0x435e, 0x1c93, 0xbf32, 0x1d79, 0x401d, 0x4098, 0xbfa0, 0x3409, 0x41d9, 0x46a1, 0xbfa3, 0x2470, 0x4583, 0x41be, 0xb295, 0x1aab, 0x319a, 0xbf97, 0x1b2d, 0x0205, 0x424c, 0x41a8, 0x433f, 0xbae9, 0xba25, 0xbfd0, 0x43ab, 0x401c, 0x2ddb, 0x3ade, 0x404a, 0x4203, 0xb2b9, 0x4559, 0xba40, 0xaa0b, 0x3a83, 0xa48e, 0x4288, 0x10f8, 0x40fd, 0xbfd2, 0xb09f, 0x4280, 0x40bf, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3dfcb14d, 0x03113440, 0x43d4feab, 0xf4773a5f, 0x012ea932, 0x9dad9d54, 0xf24b822f, 0x4f0bd97c, 0x95955220, 0xe3991301, 0xb50130a1, 0xe8c03445, 0x19b5cca3, 0x0afb31b9, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0x00000001, 0x0000000c, 0x3bae2320, 0x00000000, 0x00001a18, 0x0006677f, 0x54cd640e, 0x0000000c, 0x001bcc0d, 0x3bae26c4, 0x001c3ffd, 0x00000000, 0x001bcc0d, 0x3bae2377, 0x00000000, 0x600001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1ba6, 0x4015, 0xb0f7, 0xba63, 0x1f03, 0xb220, 0x42c4, 0x1831, 0x2745, 0x42db, 0xb2af, 0x455f, 0x0c95, 0x2125, 0x454b, 0x3a14, 0x17d0, 0xb250, 0x4237, 0x1fb6, 0xbf79, 0x43dd, 0x4336, 0x407f, 0x1b4f, 0x305a, 0x4094, 0x3a18, 0xa357, 0xadef, 0xbaee, 0x3be9, 0x438a, 0xbf90, 0x4316, 0xba54, 0x40a1, 0x46e8, 0x1b13, 0xba2d, 0x4083, 0xbad4, 0xbf13, 0x1852, 0xb064, 0x1935, 0x4494, 0x4159, 0x4433, 0x2b74, 0xba35, 0x212c, 0x41a2, 0x41f0, 0x4078, 0xb24f, 0xb213, 0x4049, 0x439b, 0x1e45, 0x417c, 0xbff0, 0x17c1, 0x4304, 0x4363, 0xbf71, 0xb267, 0x41ef, 0x4213, 0x46f9, 0x1c2a, 0xba31, 0x40b6, 0xbfce, 0x43c3, 0x4341, 0x42f4, 0x4159, 0x42ea, 0x4233, 0x427f, 0xb2fa, 0x439b, 0x43b9, 0xba25, 0x1f55, 0x41bd, 0xbf8d, 0x3fdf, 0x40fd, 0x3e6e, 0xb219, 0x04ff, 0x2c3e, 0x44e2, 0x409c, 0x403c, 0x0d36, 0x441e, 0x01ff, 0x0059, 0x41db, 0x1af3, 0x0cd0, 0x1db2, 0x4384, 0x42bf, 0x1c2b, 0x2068, 0x4648, 0xb201, 0x3ab3, 0x380e, 0xba27, 0x4160, 0x0100, 0x1adc, 0xbf1c, 0x10ff, 0x0ce6, 0x4100, 0x4211, 0x4242, 0x417a, 0x419c, 0x4026, 0x4247, 0xb26e, 0x0b32, 0xb022, 0xbf82, 0xa679, 0x4307, 0xba4f, 0xbfce, 0x3af9, 0x1cf8, 0x0647, 0xbf59, 0xb225, 0x423b, 0xbff0, 0xba38, 0x3523, 0x3a16, 0x15c0, 0xb290, 0xa1f9, 0x4369, 0x422a, 0x42c8, 0x1897, 0xabe5, 0xbfc1, 0xb205, 0x22e3, 0x42cf, 0xb091, 0x30d5, 0xb244, 0x266b, 0x402b, 0x4076, 0x1cfb, 0x466d, 0xa4e3, 0xa5a0, 0x40f9, 0xb229, 0x433f, 0xb075, 0xb2b0, 0x4216, 0x4295, 0xbf4d, 0xb04e, 0x1ca6, 0x44e8, 0x4136, 0x42a2, 0x3f2b, 0x42a7, 0x425f, 0x4240, 0x46d2, 0x416b, 0x0590, 0x4118, 0xbae6, 0xb058, 0xae7a, 0xbfab, 0x19a7, 0xba3a, 0x15dc, 0x0445, 0x1dbf, 0x14eb, 0xbfb9, 0x4159, 0x0825, 0x40e2, 0x43a5, 0xb2dc, 0xb2c5, 0x4382, 0x4389, 0xb0c7, 0x4196, 0x2221, 0x4133, 0xba0e, 0xbfb0, 0xbf49, 0x1baa, 0x1239, 0xb2b2, 0x4078, 0x1b33, 0xba6b, 0x3c77, 0x412e, 0x397b, 0x1e15, 0xbfd2, 0xa6f7, 0xb23f, 0x3196, 0x43bf, 0x40a7, 0xab47, 0x43a0, 0xb028, 0xba0f, 0x4557, 0xa960, 0xb05a, 0x4004, 0x41b9, 0x41be, 0xbad8, 0xbf17, 0x42fb, 0x467f, 0x42fd, 0xba23, 0x1a1f, 0xb291, 0x1c6a, 0x4190, 0x23a3, 0x448b, 0x4047, 0x4284, 0xb2ab, 0xb052, 0x429a, 0x4108, 0x1114, 0x4498, 0x401c, 0xb24b, 0x415b, 0xb2e8, 0xbfb0, 0x45d9, 0xbadf, 0xbfb2, 0xbff0, 0x41d4, 0xb09a, 0x431a, 0x405b, 0x1d7d, 0x1d8c, 0x4388, 0x060c, 0x1f0b, 0xba03, 0x4167, 0xb2af, 0xb2e8, 0x4095, 0x1b2a, 0x4142, 0xb038, 0x1e28, 0x2d5a, 0xb007, 0x37ae, 0x432f, 0x43f9, 0x41f5, 0xbf72, 0x4377, 0x0585, 0xba7a, 0x2893, 0x3546, 0x1b6b, 0x4315, 0x3494, 0x4000, 0x4305, 0x3758, 0xbf05, 0x4298, 0x43ca, 0x4299, 0x0b40, 0x433c, 0xba6f, 0x41e3, 0x29c3, 0xa073, 0x176f, 0x41fa, 0x465e, 0x28ab, 0xbf04, 0x40f8, 0x4347, 0x4357, 0x4269, 0x43f7, 0x42f1, 0x41b9, 0x4373, 0x41ac, 0x419f, 0xb03a, 0x0ff1, 0xba12, 0x469d, 0x233d, 0x440a, 0xbac6, 0x43ca, 0x2379, 0x1332, 0x43d0, 0xb2cc, 0x4281, 0xbf0e, 0xb24d, 0xbf90, 0xb291, 0x4043, 0x2e1b, 0x02b8, 0xbf74, 0xb2ea, 0x4399, 0x408e, 0x42ed, 0x456a, 0x414d, 0xba4d, 0x1929, 0x039e, 0xb22d, 0x38ee, 0x408e, 0x40b8, 0xb20f, 0x410a, 0x4553, 0x189f, 0x110e, 0x17e1, 0x414a, 0x1ec4, 0x4499, 0xba68, 0xb060, 0x42c3, 0x4369, 0xbf0f, 0x3aec, 0x4157, 0x3771, 0x46cc, 0xb230, 0x4169, 0xb054, 0x10b7, 0x14f3, 0x33ae, 0x40cc, 0x1f4e, 0xba61, 0x4356, 0x422e, 0x4047, 0x409d, 0x41b5, 0x4193, 0x2017, 0xb013, 0x4559, 0xb20d, 0xba6c, 0x41db, 0xba06, 0x4167, 0x4252, 0xbf09, 0x4360, 0x188d, 0xb239, 0x4385, 0x413a, 0xb2e8, 0x4303, 0x0c1d, 0xbff0, 0xb2ea, 0xb06b, 0xba28, 0xb217, 0x4330, 0xa300, 0x16ab, 0x1053, 0x0c0c, 0x414b, 0x43dd, 0x0a6d, 0x428d, 0x4378, 0x1b7a, 0xbf80, 0xb265, 0xa0df, 0x4079, 0xbfbb, 0x3808, 0x1e91, 0xa711, 0xbaf8, 0x4665, 0x2c9b, 0x03fb, 0x2a72, 0x4186, 0x4052, 0x43ff, 0x407b, 0xa0ed, 0x4207, 0xbf70, 0x43d7, 0x1367, 0xb2b4, 0xb207, 0xa2b7, 0x0b98, 0xba72, 0xbf44, 0xb05e, 0x41db, 0x4231, 0xba15, 0xba3d, 0xbfe0, 0x3b3c, 0x434f, 0x1a8a, 0x4373, 0x437e, 0x19db, 0x45f0, 0xb291, 0xbfa8, 0x40c4, 0xbada, 0xb2a6, 0x410c, 0x2f9d, 0x4499, 0xb254, 0xb251, 0x4688, 0x36c7, 0xb28b, 0x3ca1, 0x2de1, 0x401d, 0xb233, 0xba5a, 0xb03a, 0xb2c0, 0x429c, 0x43b8, 0x40e6, 0xbfdc, 0x415f, 0x40c8, 0xb295, 0x46d0, 0x449a, 0xa9e5, 0x17e0, 0x4127, 0x4165, 0x4213, 0x4324, 0x3d81, 0x413b, 0x41c3, 0x1fb5, 0x4041, 0x1dce, 0xb2aa, 0xbaee, 0x2b6b, 0xa9ff, 0xbfaa, 0x1e69, 0xb056, 0xba20, 0xbaf9, 0x3584, 0x1ce5, 0x4170, 0xbfd6, 0x4029, 0x3811, 0x43c3, 0xba36, 0x4136, 0x0fcd, 0xa452, 0xa18a, 0x42f8, 0xb236, 0x46d1, 0x180f, 0x42b5, 0x43ac, 0xb2c3, 0xba0f, 0xb004, 0x4638, 0xb2bb, 0xba5f, 0xba19, 0xbf11, 0x4640, 0x1f23, 0xba63, 0xb2d6, 0x4159, 0xbf6d, 0x4242, 0x3570, 0xb2b6, 0x410c, 0x4243, 0x4340, 0x4347, 0x23c7, 0xb257, 0x1e21, 0xb2b1, 0x402c, 0x429e, 0xb2fc, 0x3a4c, 0xbfa0, 0x41bc, 0x4659, 0x02de, 0x40c0, 0xb21e, 0xbfd8, 0x40fb, 0x37f9, 0x17e4, 0xb0ab, 0x4144, 0x4158, 0x4182, 0xaaae, 0x400a, 0x41ac, 0x40ff, 0x4207, 0x414f, 0x41e8, 0xbf09, 0xb2c8, 0xb0a2, 0xb229, 0x41a3, 0x4210, 0xb20a, 0x11b2, 0x2fdf, 0xb242, 0x34cc, 0x430a, 0x4462, 0xb24f, 0x42ed, 0x3e95, 0x4214, 0xa49d, 0x1a98, 0x42ae, 0x407f, 0x1c46, 0xb260, 0x46d2, 0xbf38, 0x45a2, 0x4105, 0xb25a, 0x406b, 0x0e70, 0x1c2e, 0xbf6c, 0x42c7, 0x1e6b, 0x40f0, 0x1edc, 0x418f, 0x1c63, 0x4403, 0xaeec, 0x1d5e, 0xb289, 0xb20d, 0x0fbd, 0x40f0, 0xb0a7, 0xba3d, 0x43e8, 0xbfa4, 0x12ff, 0x437b, 0x43dc, 0xba54, 0x416d, 0x42b7, 0xb2e8, 0x1d0a, 0xbf60, 0x1ac0, 0x1972, 0xb07d, 0xba21, 0xbf5f, 0x45b1, 0xa40b, 0x42b3, 0x4008, 0x20cf, 0x184a, 0x41fe, 0x2f74, 0x4129, 0xba6e, 0x43f7, 0x438e, 0x459a, 0xba5f, 0x1d12, 0x1c06, 0xba3b, 0x4338, 0x4319, 0x412e, 0x14df, 0xbac7, 0xbf06, 0x0b43, 0xa015, 0x1998, 0x1ae9, 0xb251, 0x1b33, 0x29d5, 0x1def, 0x4306, 0x4170, 0x4674, 0x4036, 0xb29a, 0x415e, 0x4367, 0xbf36, 0xba3a, 0x400f, 0x1aab, 0x41d9, 0x0d1c, 0x4598, 0x44b5, 0xa285, 0x406f, 0xbf07, 0xba2b, 0xba21, 0x41ae, 0x4488, 0xbf81, 0x24c3, 0xbf00, 0x080d, 0x3f1d, 0x45c8, 0x01cb, 0xa168, 0xbf88, 0x1d8a, 0x4469, 0x0868, 0x1fb2, 0xb28d, 0x40c5, 0xb23b, 0xba1b, 0x4075, 0x43fd, 0x30ab, 0xb2d8, 0x3f10, 0x1c49, 0xab69, 0xbf27, 0x1415, 0x0643, 0x46f1, 0x1c17, 0xbad5, 0x4405, 0xa383, 0xa1ed, 0x420c, 0x1866, 0x41e3, 0x43c0, 0xba28, 0x428f, 0xb08f, 0xba7b, 0x4576, 0x0987, 0x4266, 0x42a8, 0xb278, 0xba0b, 0xbfdb, 0x1ae6, 0x19bc, 0x191d, 0x4019, 0x4358, 0x2895, 0x10c4, 0x33fe, 0x1eb2, 0x41bd, 0x42fd, 0x4230, 0x1c37, 0x445d, 0x1970, 0x420a, 0x4325, 0x4179, 0x4053, 0x0bb2, 0x4373, 0x3cd8, 0x40c0, 0x4194, 0xad38, 0xbf9e, 0x144c, 0x215a, 0x4100, 0xb29f, 0xbfe8, 0x146d, 0x1a76, 0x37cd, 0x2bc4, 0xba24, 0x4359, 0x0417, 0xbfcd, 0x0ffa, 0x4100, 0xbac8, 0x41c8, 0x350d, 0xb2d9, 0x41b3, 0xbf5f, 0xba5a, 0x1f92, 0x442f, 0x4342, 0xb2cc, 0x42f7, 0x42ec, 0xbfa0, 0xa018, 0x3c0a, 0x4650, 0x464a, 0x1d7b, 0xbf11, 0xba23, 0x421c, 0x42ea, 0xb23c, 0xb2cf, 0x131a, 0xbf8f, 0x00af, 0xb29d, 0xba50, 0xbaee, 0x4607, 0x4645, 0xb2a8, 0xa65e, 0xb0ae, 0x42c2, 0x4153, 0xbfb5, 0x1f5c, 0x2a64, 0xb04d, 0xb221, 0x2bcf, 0x4102, 0x4115, 0x4301, 0x1a80, 0x38b0, 0x41da, 0xbaf9, 0x42a2, 0xba22, 0x42dc, 0xba39, 0x41c0, 0x46bb, 0x3fbd, 0x1f57, 0x412f, 0x4150, 0x4319, 0xbf01, 0x4162, 0x4211, 0xba0b, 0x2ec4, 0x1345, 0x4336, 0x4064, 0x4622, 0x4372, 0x422b, 0x43b3, 0x427e, 0xb0f6, 0xbf46, 0x333c, 0x43d0, 0xb2fd, 0x4219, 0xb06a, 0x114c, 0xb287, 0x44e3, 0x41a7, 0xb089, 0x422e, 0x4059, 0xb066, 0xabef, 0x190a, 0x4263, 0x1dd3, 0x4209, 0xbf67, 0x42ca, 0x409b, 0x4286, 0xba45, 0x42a3, 0xbae1, 0x2947, 0x414f, 0x43b7, 0x0fc6, 0x4350, 0xbfab, 0x40e6, 0x45b6, 0x1dab, 0x4132, 0x4289, 0xb060, 0xbf97, 0x4212, 0xba0b, 0x068c, 0x4390, 0x4181, 0x03ac, 0x406a, 0x4044, 0xaa37, 0xb247, 0x46ab, 0x41dc, 0xa6bd, 0xbf6d, 0x4007, 0xb2fe, 0x40ee, 0x1047, 0x41d3, 0xb29b, 0x428d, 0x4275, 0x4392, 0xb2da, 0x40fa, 0xb0e0, 0xbf4c, 0xb0b4, 0x05a7, 0x414a, 0x1c39, 0xb2e9, 0x4080, 0x430f, 0x3d9b, 0x1ef4, 0x4167, 0xbf19, 0xb01b, 0x42dc, 0xb04b, 0x44cd, 0xb284, 0xb046, 0x3574, 0xb2c5, 0x41b5, 0xae2e, 0xb216, 0x1fe5, 0x10e6, 0xb2b8, 0x18a6, 0x4600, 0x4050, 0x406f, 0x420a, 0x41da, 0x43c4, 0xbf14, 0x202f, 0x43ea, 0x41e7, 0x4044, 0x3472, 0xb2f3, 0x42f8, 0x1937, 0x1ffd, 0xbf07, 0x420a, 0x0351, 0xbad2, 0xba49, 0x43be, 0x41eb, 0xbf70, 0x40b7, 0x4479, 0x0ed9, 0x4261, 0xb066, 0xb22d, 0x1b22, 0x4225, 0x1edd, 0x4180, 0x4090, 0x409c, 0x46ab, 0x413c, 0x3858, 0x4127, 0x42c6, 0xbfb0, 0xa6c1, 0xbfd4, 0x46e0, 0x423f, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xcb6e3399, 0x779d0a27, 0xb05aed8d, 0x68a1a047, 0x971f3c1b, 0xd982dff6, 0x911b2b51, 0x3af672e5, 0xc8b54557, 0xb853ba8e, 0x82d6c3bf, 0x67250d2b, 0x982aa4b0, 0x611a482f, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0xffffffa8, 0x845c5046, 0x00000000, 0x00006500, 0x7ba3afba, 0x000064fd, 0x00001ae0, 0x00000000, 0x982aa4b0, 0x00000000, 0x1b016935, 0x000064fd, 0x982aa4b0, 0xffd0f718, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1604, 0x0a21, 0x4319, 0x4564, 0xb217, 0x459a, 0x4428, 0xb20d, 0x1b2c, 0x2a6d, 0x403e, 0x410b, 0xba30, 0x1e1d, 0xbad8, 0x309a, 0xbf63, 0xbfa0, 0xb283, 0x1430, 0x468c, 0xb071, 0xb082, 0xb0c2, 0xbfa0, 0x1caf, 0xb278, 0x4270, 0x307d, 0xbacc, 0xb0d7, 0x40ec, 0xbacf, 0x33cb, 0xa1ac, 0x1b5a, 0xbf80, 0xb2e6, 0xba4b, 0x1e68, 0xbfe0, 0xb2dc, 0x4085, 0xbf7f, 0x2943, 0x1edf, 0x4346, 0x32fb, 0x410d, 0x44f4, 0x1e4f, 0xa16f, 0xb25d, 0x4348, 0xbf18, 0xbac6, 0x0b17, 0xbac6, 0xba0c, 0xb232, 0x3f9e, 0xb22f, 0xa529, 0xbfe0, 0x43b4, 0x2768, 0x427a, 0xb2fd, 0x408f, 0x4269, 0x4388, 0x42c0, 0x3a6c, 0xbf25, 0x4219, 0x4311, 0x1de4, 0x2564, 0x41c2, 0x1963, 0xb239, 0x42c3, 0x421f, 0xb0a4, 0x403f, 0x1c19, 0x1f37, 0x40f1, 0xbfc8, 0x40ac, 0x43aa, 0x0762, 0x4672, 0xb226, 0x3a12, 0x1aa7, 0xbad5, 0xb21a, 0x42ff, 0xbfdb, 0x4302, 0x1a52, 0x16ad, 0x42cd, 0x43fa, 0xb2f5, 0x413b, 0x411c, 0xb044, 0xb056, 0xa3cf, 0x40a9, 0x11c8, 0x455b, 0x41f6, 0xb04e, 0x3cea, 0x4257, 0x4251, 0x42e9, 0x2352, 0x3e76, 0x1aa5, 0x4086, 0x4355, 0x40e8, 0x3a84, 0x103c, 0xbfb5, 0x4227, 0xb07d, 0x1056, 0x4078, 0x4166, 0xba60, 0x1dd5, 0xbfdc, 0xb294, 0x4020, 0xa37a, 0x433d, 0xb2ab, 0xb284, 0x4223, 0xbf22, 0x43c2, 0x18f8, 0x4181, 0x40f5, 0xb2fb, 0x38ea, 0x4151, 0xac89, 0xbff0, 0x40a4, 0xa65b, 0x098e, 0x4020, 0x415c, 0x244e, 0x4034, 0x27e0, 0xbf94, 0xb2ec, 0xb0d0, 0x4255, 0xbf41, 0x22ee, 0x42c3, 0x139b, 0xa16a, 0xbfc0, 0x18b8, 0x4020, 0xb05e, 0xa1b8, 0xbad3, 0xbf2e, 0x460e, 0xbaf4, 0x23aa, 0x1c22, 0x34b3, 0xba2c, 0xa046, 0x223a, 0x4236, 0x1f81, 0x40e4, 0x43ea, 0x065e, 0x4251, 0x4266, 0x409b, 0x2271, 0x43be, 0x4026, 0x4394, 0x2c3b, 0x1dfa, 0xb287, 0xa1f2, 0xaed8, 0x4044, 0x42d9, 0xbf33, 0xb2db, 0xaaec, 0x1a26, 0xa731, 0x41be, 0x4203, 0xa536, 0x4663, 0x09c1, 0x4236, 0x1bcd, 0x234a, 0x3426, 0x420c, 0x43fa, 0x42e2, 0x1fcf, 0x43f1, 0x43ef, 0x4122, 0xab1c, 0x40cb, 0x41ac, 0x40ba, 0xb2f7, 0x46b0, 0xa3fb, 0xb037, 0x4583, 0xbf1e, 0x40a3, 0x4005, 0xafe0, 0xba7e, 0x42a9, 0x38ab, 0xb2b3, 0x43bd, 0x43c4, 0x1ca1, 0x4208, 0x410d, 0x4319, 0xba00, 0x43a2, 0xbae2, 0x194f, 0x316c, 0xba00, 0xbfc1, 0x41af, 0x40eb, 0x2d1e, 0x0f59, 0xba37, 0x40ec, 0x454f, 0x23a9, 0x0eb7, 0x4130, 0x410c, 0x4273, 0xba56, 0xa4e0, 0x42d8, 0x439b, 0xa5a8, 0x43e4, 0x41ab, 0x463e, 0x4097, 0x407f, 0xbf83, 0x4203, 0x0ecb, 0x41d5, 0x423b, 0x413e, 0x41aa, 0xbf5f, 0x14be, 0x40f8, 0xbada, 0xb294, 0x1a4d, 0x41c1, 0x120e, 0x424b, 0x1ac7, 0x42e8, 0x2300, 0x4386, 0x46f2, 0x41f1, 0x413f, 0x4078, 0x4193, 0x42b0, 0x07d0, 0x194c, 0x4647, 0x4216, 0x3906, 0x1e78, 0x4038, 0x045d, 0xb2be, 0x2603, 0x416d, 0xbf4b, 0xb26e, 0x403f, 0x45c3, 0x0349, 0xba7a, 0x4186, 0x4438, 0xb078, 0xbfe0, 0x4314, 0xb236, 0xb0fc, 0x4467, 0x420b, 0x407c, 0x43d2, 0x4252, 0xba6b, 0xbadc, 0xbf4c, 0x4021, 0xba6d, 0xb2c7, 0x1f82, 0xb07b, 0x1706, 0xbf1c, 0xbfd0, 0x1d27, 0x4176, 0xb284, 0x380c, 0x4642, 0x4607, 0xb0cc, 0x44d9, 0xb2a7, 0x40b5, 0x42c2, 0xb2d6, 0xba22, 0x1ae8, 0x28fa, 0x4030, 0x0012, 0x19af, 0x4545, 0x407f, 0x43fd, 0x4603, 0xbfd6, 0x41de, 0x1ede, 0xba4c, 0x41a1, 0x41c3, 0x4371, 0x408e, 0x41bc, 0x42cd, 0x42be, 0x4172, 0xab14, 0x4068, 0xa34b, 0x1bfc, 0x1929, 0xb2a6, 0xbf76, 0xb2eb, 0x410c, 0x1fa5, 0x4241, 0x469a, 0x1eec, 0x408b, 0xbf15, 0x4304, 0x464f, 0x1424, 0x431e, 0xbf66, 0xa1af, 0xa8a3, 0xba6d, 0xb2c3, 0xba52, 0x0e61, 0x40a3, 0x4348, 0xbaf0, 0x412c, 0x1aae, 0x42a3, 0x1da6, 0x4404, 0x424c, 0x336d, 0x4044, 0x40bc, 0xb2cb, 0x3f79, 0x416e, 0xbf5f, 0x2954, 0xba01, 0xba05, 0xaef8, 0x1e85, 0x4035, 0x41c5, 0xb062, 0x1500, 0x4008, 0xbf85, 0x0e11, 0x279e, 0x4125, 0x42c2, 0xbac4, 0x438a, 0x46f2, 0x0210, 0xba56, 0xba5b, 0x11f8, 0xba09, 0xbf5f, 0xb2bd, 0x1e57, 0x3541, 0xba26, 0x1e26, 0x10de, 0x20bb, 0xbf90, 0xbf3c, 0x4182, 0x43cc, 0x40cb, 0x4275, 0x4150, 0x4164, 0x41c0, 0x42d8, 0x40c8, 0x3c64, 0xa413, 0xa6e0, 0x18fa, 0xa514, 0x464f, 0x40f1, 0x43a0, 0x1131, 0x4067, 0x4017, 0x0294, 0xb2f3, 0x4159, 0xba10, 0x3e05, 0xbf8b, 0x042f, 0x431c, 0x40c7, 0xba5a, 0x1cfe, 0x40d2, 0x4178, 0xb2ad, 0x2175, 0x41df, 0x226c, 0x426e, 0xb0a0, 0x4081, 0x42ec, 0x4035, 0x40b2, 0xbfd7, 0x4165, 0x4419, 0x43cd, 0x46ba, 0xbfc0, 0x4384, 0x0cc7, 0x1e7a, 0x466a, 0x04ed, 0x0fdc, 0x42e2, 0x42a3, 0x2874, 0x42df, 0xb2e6, 0x223a, 0x3d2a, 0x349c, 0x41b0, 0x1fb5, 0x2ef3, 0xbfa3, 0x14e3, 0xbac0, 0x4357, 0xb2a5, 0xba2a, 0x4161, 0x4218, 0xb0b3, 0x4110, 0x15d7, 0x412e, 0xbfaf, 0xb2bb, 0x1930, 0xba71, 0x1dc0, 0xbf53, 0x4363, 0x4237, 0x206b, 0x1a83, 0x1f94, 0x41fb, 0x4071, 0x4632, 0x41ca, 0xb000, 0x2298, 0x4191, 0xa626, 0xb215, 0x24be, 0x40b3, 0x02a0, 0xb209, 0x2ff4, 0x0d5d, 0x43ef, 0x21e6, 0x1915, 0x401f, 0xbf84, 0x461f, 0x4244, 0x421c, 0xba26, 0x43f2, 0xba11, 0x42b4, 0x17d4, 0x43e7, 0x072e, 0x41a1, 0xaee1, 0x4090, 0xb20e, 0x40e9, 0xb207, 0x1dd0, 0x1cc6, 0xbfb1, 0xb045, 0x2731, 0x4271, 0xbade, 0xb217, 0x403f, 0x35ea, 0x1eea, 0xaa77, 0x2632, 0x19c7, 0xba1a, 0x433f, 0x439e, 0xbf15, 0x1a25, 0x40b9, 0x14f0, 0x3dd4, 0x3c2a, 0x3be5, 0x43c9, 0x40e1, 0x435d, 0x41eb, 0xb236, 0x46ec, 0x1701, 0x403b, 0xbaee, 0xb030, 0x43d4, 0xa1a0, 0x0433, 0xb0ed, 0xbaee, 0xbf47, 0xb25d, 0x410c, 0x41e2, 0x1d4a, 0x1fa0, 0x3d98, 0x1f8f, 0x3a41, 0x41c6, 0x456d, 0x04a1, 0xb2ca, 0x1d5e, 0x40ca, 0xa917, 0x40d1, 0xa2c5, 0x428f, 0x225d, 0x11fe, 0x42d3, 0xbfd9, 0xb2d2, 0x1ca9, 0xafda, 0xbf80, 0xb252, 0x2a1a, 0xbaf3, 0x430a, 0xb244, 0x1c06, 0xa3c7, 0x1c7b, 0x407f, 0xba7a, 0xb2c9, 0x43af, 0x4266, 0xba74, 0xbfbd, 0x1822, 0xba00, 0xb218, 0xb23a, 0x42da, 0x1ec2, 0x412b, 0xb257, 0x2422, 0xbf68, 0xbac7, 0x43fa, 0x1072, 0x4102, 0xb2f7, 0xba43, 0x4303, 0xbac6, 0xb296, 0x1cd6, 0x4379, 0x180e, 0x3654, 0x432f, 0xbf80, 0x228d, 0x03ca, 0x43d0, 0x43b1, 0x2ddd, 0x41fa, 0x35b8, 0x35fe, 0x4212, 0x3dc5, 0xbf59, 0x446f, 0xba34, 0xb234, 0x45c0, 0x461c, 0x1c69, 0xa91c, 0xbfab, 0x43e0, 0x098b, 0x4052, 0x4389, 0xb2b5, 0x1ed5, 0x4129, 0x0b1b, 0x2abf, 0x1b81, 0x32cd, 0x431f, 0xbf60, 0x1adb, 0x42fb, 0x40f0, 0x43d0, 0xb272, 0x4041, 0x421b, 0x43f7, 0xbf2c, 0x46a2, 0x43d8, 0x43bd, 0x439c, 0xba01, 0x3e66, 0x19dc, 0xbf28, 0xb04a, 0x4286, 0x429d, 0x25ec, 0x4381, 0xba19, 0xb2f8, 0x4042, 0x1f44, 0x1ea0, 0x42dc, 0x438d, 0x40f1, 0x1f2f, 0xba50, 0x4311, 0xa700, 0xb0ed, 0x1c9b, 0xb0c5, 0x2742, 0x431c, 0xbf1e, 0x409e, 0xaf81, 0xba68, 0x4364, 0x4078, 0x0ac1, 0xab14, 0xbaec, 0x4499, 0x0a77, 0x0594, 0x2e04, 0xba67, 0x2c60, 0x41c8, 0x443e, 0x4273, 0xb00f, 0x41b1, 0x43b1, 0x1f25, 0xbf34, 0x4282, 0xb2fc, 0xbf44, 0x400d, 0xb275, 0xaf07, 0xb06d, 0x44db, 0x12f2, 0x4082, 0x0b28, 0xb0ac, 0xbfc0, 0xb2ca, 0xb2cb, 0x4462, 0x45b6, 0x4328, 0xb24a, 0x2a77, 0x425e, 0xbf78, 0xb24c, 0x43f5, 0x17b8, 0x2ab7, 0xb285, 0x40b9, 0xa7d7, 0x45e4, 0xb0e1, 0x4395, 0x4020, 0x2b72, 0x4200, 0xb069, 0x40db, 0x41af, 0x42dc, 0x41c6, 0x16a0, 0xb268, 0xbf68, 0xb0e3, 0x421b, 0x4208, 0xa3e3, 0x3cd0, 0xbf54, 0x3d55, 0xba7c, 0x4399, 0xb213, 0x43d0, 0x160a, 0xbfe4, 0x4659, 0x4327, 0x107b, 0xb2f7, 0x42ea, 0xbf00, 0x43d4, 0x2fa0, 0x41ad, 0x195b, 0xb2c5, 0x05dc, 0x43c7, 0x4074, 0x426e, 0xb084, 0x4354, 0x1993, 0xbf44, 0x4226, 0x118b, 0x0a67, 0xbaca, 0x43ea, 0x2159, 0x416e, 0x190e, 0x0629, 0xbf68, 0x4155, 0x41f9, 0x2408, 0xbf2c, 0x1a7a, 0x4137, 0x17b6, 0x4092, 0xb039, 0x404c, 0x4324, 0x40de, 0x40df, 0xbf35, 0x400e, 0x1853, 0x4246, 0x3887, 0xb2b8, 0x44b3, 0x0cc0, 0x294e, 0x0158, 0xb24b, 0xba1c, 0x1f56, 0x0aff, 0xb298, 0xbf69, 0x32c4, 0x44ad, 0x1804, 0x4189, 0x40ff, 0x1993, 0xada4, 0x1f03, 0xbaec, 0x43ed, 0x40e9, 0x1957, 0x40af, 0x46dc, 0xad97, 0x4368, 0x4271, 0x4156, 0xbf52, 0x41ba, 0x4348, 0x4550, 0x0bf9, 0x4039, 0x4692, 0x2bed, 0x40d5, 0xba49, 0x0c78, 0x1ba4, 0xba44, 0x0087, 0x4045, 0x42fe, 0xb299, 0xba7b, 0xbf86, 0x46e9, 0x1a43, 0x0677, 0x3bf7, 0x445e, 0x1c2d, 0x4208, 0x43e0, 0x18ff, 0x1077, 0x2711, 0x425a, 0xba76, 0xb04f, 0xb2a9, 0x1a42, 0x449a, 0xb2e8, 0x4019, 0x1195, 0x42c2, 0xb208, 0x449c, 0xbacc, 0xba2f, 0xbf90, 0xbf44, 0xbacb, 0xbaeb, 0x1d0c, 0x4119, 0xb2e4, 0xb209, 0x45c3, 0x4580, 0xba39, 0x41f0, 0xbf7b, 0xb239, 0x42ed, 0x412f, 0x3276, 0xb0e3, 0xb285, 0x146e, 0xb064, 0xa6c5, 0xbf12, 0x4037, 0xbaf9, 0xb2ba, 0x4345, 0x156c, 0x404b, 0x407a, 0x1c20, 0x44f2, 0xba73, 0x404e, 0x4338, 0x2e25, 0xba5e, 0x1b5b, 0xb2f5, 0x4247, 0xbf5a, 0xa67b, 0xb257, 0x1c8e, 0x2226, 0x43d4, 0xb045, 0x4197, 0x41f6, 0x08df, 0xbf5c, 0x42b0, 0x4170, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x51ade2cc, 0xec524c8c, 0x38eff947, 0x36d7c854, 0xe5629cca, 0xa8d08a62, 0x7a884da5, 0xfbdda384, 0xdb38fbe9, 0xf6976dfd, 0x005fd422, 0x93c6813c, 0xb8f2dcc9, 0x6cfd2716, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0x40ffffff, 0xffffffff, 0x00000026, 0xff767516, 0xffffffd9, 0x000000b4, 0x80000000, 0x1feecea2, 0x6cfd1de1, 0xf75b1a7b, 0xffffff09, 0x278d022c, 0x278d0135, 0x6cfd2fc6, 0x00000000, 0x900001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xa6c2, 0x4488, 0xb2d1, 0xb06a, 0x4083, 0x44ca, 0xba42, 0x1d14, 0x420d, 0xbade, 0x4103, 0x40f9, 0x4003, 0xb0d9, 0x4552, 0xbfcc, 0xa2db, 0xa663, 0xba1e, 0x2e6c, 0x43d1, 0x407f, 0x401a, 0xb2c1, 0x4631, 0x0681, 0xa7bb, 0x4649, 0xb05b, 0xbfe1, 0x28a0, 0x1af0, 0x16fa, 0x45c8, 0xab5a, 0x4399, 0x4423, 0x1bdd, 0xb044, 0x2499, 0x4440, 0x3ee0, 0xb02f, 0x0ddd, 0x4127, 0xbf08, 0xba20, 0xba1c, 0x03ac, 0xb0f7, 0x40e5, 0x4116, 0x4116, 0x468a, 0x422d, 0xb2d9, 0x43c9, 0x41ea, 0x4054, 0xbace, 0x437c, 0xbf91, 0x40a4, 0xb247, 0x18af, 0x16e1, 0xb255, 0xbade, 0xb0a3, 0x1f17, 0xbf47, 0xb0be, 0x1037, 0xb2b8, 0x175f, 0x18fc, 0xb286, 0x41bd, 0x46f3, 0x42b4, 0x0b47, 0x4359, 0x18c2, 0xadd5, 0xbf2c, 0x1c76, 0xba76, 0x4132, 0xba48, 0x43f4, 0x168c, 0x416e, 0xbfab, 0x4194, 0x1bc0, 0x28da, 0x42c5, 0x4423, 0x41ff, 0xb2ff, 0xb008, 0x20d3, 0x437e, 0x4034, 0xba40, 0x425e, 0xa06c, 0xba68, 0xba72, 0x00eb, 0x41b2, 0x411e, 0x1b3d, 0x2f08, 0xb089, 0x411a, 0x14ff, 0xb2be, 0x4633, 0xbf7d, 0x4150, 0x40c2, 0xb28a, 0x1e72, 0xba3d, 0x0aae, 0x435e, 0x205c, 0xb067, 0x4353, 0x21c0, 0x43e1, 0x4141, 0x0572, 0x43ae, 0x427c, 0x42ae, 0x4026, 0x2df2, 0x405c, 0x37ec, 0x4074, 0x1559, 0xb2e7, 0xbfe0, 0xb2b4, 0x41cb, 0x0d6c, 0x433a, 0xbf64, 0x1d3e, 0xb2f3, 0xba2e, 0x4020, 0xba4f, 0xb2b6, 0x43ff, 0xa82c, 0x1929, 0x421e, 0x18af, 0x4415, 0xbf33, 0x4698, 0x1b4a, 0x27e0, 0x40f2, 0xb25e, 0xba7b, 0xb26a, 0x4309, 0x4064, 0x402b, 0x4098, 0xb245, 0x43bd, 0x0c9f, 0x42de, 0x015f, 0x4321, 0x1a23, 0xb28e, 0xba29, 0x406b, 0x1998, 0x4265, 0xbf38, 0x4314, 0xb2b6, 0x165c, 0xbafc, 0xbf70, 0xb048, 0xba43, 0x46bb, 0x4001, 0x0cb8, 0x43fb, 0x270f, 0xb2fc, 0x0ae9, 0x4223, 0x423f, 0x26fd, 0x419e, 0xa390, 0xb0ff, 0xb2cf, 0x0bc0, 0xbf4b, 0x4135, 0x4559, 0xb2bf, 0xba2e, 0xa3e9, 0x4310, 0xba44, 0x43ff, 0x4394, 0x401c, 0x41f6, 0x42d0, 0xbf84, 0xa7d9, 0x42c0, 0x0a4b, 0x4285, 0x42a0, 0xb040, 0x1ba8, 0xa9fa, 0xba04, 0x358a, 0x4003, 0x46d4, 0x41fe, 0xba48, 0x2c12, 0x41a3, 0x4161, 0x4372, 0x448b, 0xbf98, 0x0627, 0xb23c, 0x4170, 0x0a61, 0xb279, 0x4260, 0x06b6, 0x3f4f, 0x4319, 0x40a8, 0x24ea, 0xbfe8, 0x412c, 0x3af2, 0x415d, 0x42f8, 0x27fc, 0xb04c, 0x2615, 0x432f, 0x447e, 0xb0b6, 0xb0e8, 0xbf15, 0x4034, 0xbadf, 0xba1e, 0x43d6, 0x00f1, 0x2101, 0xb229, 0x270d, 0x0d8f, 0x1c5a, 0x3665, 0x4364, 0xbad0, 0x435f, 0x1ae3, 0x20f4, 0x42f6, 0x4606, 0x38db, 0x1391, 0x42ee, 0x4550, 0xb233, 0xa680, 0x4310, 0x4005, 0x41a2, 0x42dc, 0x43e8, 0xbf0e, 0xb2ad, 0xa2f4, 0x1508, 0xae86, 0x4320, 0xbf55, 0x2d19, 0x43ef, 0x42db, 0xb0a9, 0x3556, 0xb01e, 0x4176, 0x43b8, 0x4462, 0x077f, 0xba0b, 0x42d6, 0xb2c2, 0x4336, 0x4384, 0xb2bf, 0x4002, 0x4262, 0x1f32, 0x46db, 0x4239, 0xb200, 0x1f3c, 0x4167, 0xb23b, 0xb0d4, 0x43cd, 0x0623, 0xbfb5, 0x2020, 0x1c56, 0x1cf6, 0x433f, 0x43b3, 0x429e, 0x41e3, 0xb2aa, 0x43fd, 0x1cac, 0x3d18, 0x4077, 0x411f, 0x4058, 0x43dd, 0xb244, 0x41ee, 0x4268, 0x2fa0, 0x243a, 0xb00a, 0xb2e4, 0x3b24, 0xa3df, 0xbfa6, 0x42c4, 0xb072, 0x1cbb, 0x4090, 0xbf9f, 0xba2a, 0x3835, 0xb27c, 0x4217, 0x4283, 0xb232, 0xbf1c, 0x43c9, 0xb2a2, 0xb2a3, 0x4194, 0xb25c, 0xbf92, 0x4085, 0x4278, 0x4073, 0x40b4, 0x423c, 0x1d1a, 0xbac1, 0x43b0, 0x4330, 0x1c88, 0xb210, 0x2e6a, 0x4018, 0x41ac, 0x0ac5, 0xb2ea, 0x4569, 0x141c, 0xbf8b, 0xbf70, 0x42ab, 0x4031, 0xbacb, 0x41ec, 0x4019, 0x3670, 0x2cf6, 0x401f, 0x26a1, 0x41eb, 0xb23f, 0xb221, 0x4671, 0xb2bb, 0xbf60, 0x363e, 0x420b, 0xab50, 0xbf77, 0x1e0f, 0xae4e, 0xb025, 0xba19, 0x1f73, 0x37a5, 0x1c37, 0xbf90, 0xbf93, 0xb058, 0x1c46, 0x1c5d, 0x19a0, 0x0b41, 0xbfa0, 0xb2ae, 0x1f46, 0xb237, 0x14c9, 0x405c, 0x2f26, 0x46e8, 0x41a5, 0x46b0, 0xb22d, 0xb03f, 0x1fc2, 0x3b4f, 0xbfa6, 0x1faa, 0x432c, 0x19ea, 0x40f5, 0x4246, 0x1f94, 0x4385, 0xa1df, 0x38f5, 0x049a, 0x455f, 0xa132, 0x08ee, 0x1c37, 0xbfa0, 0x1e5e, 0x401a, 0x43ac, 0x402c, 0x4222, 0x3931, 0x2124, 0x44aa, 0x42a8, 0xb0f0, 0xa8c8, 0x1e5e, 0xa92c, 0xbf65, 0x4045, 0xbaeb, 0xba57, 0x44ad, 0xac8a, 0x420b, 0x353f, 0xb23e, 0xa0ff, 0x1872, 0x41cb, 0xb22e, 0xb0cc, 0x4213, 0x40e3, 0x0aef, 0x411c, 0xb0bc, 0x1d52, 0xa778, 0xb0d0, 0xbf56, 0x43d1, 0x42f4, 0x1d5b, 0x1dc2, 0x0f45, 0x4381, 0xb214, 0xb09f, 0x4175, 0x4680, 0x24ea, 0xb231, 0xb264, 0xa2ed, 0xbace, 0x181b, 0xbf08, 0x42a5, 0xa7c2, 0x4018, 0xbfbf, 0xb2b2, 0x4309, 0x4264, 0x02c3, 0xa5f0, 0x1d34, 0xb209, 0x4090, 0x3005, 0x431d, 0xb223, 0x434e, 0x0cec, 0x0970, 0x43d2, 0x402e, 0x403d, 0x1385, 0x1eac, 0x1fe2, 0x427b, 0x0159, 0x0702, 0x0390, 0xbf6b, 0x40e8, 0x403d, 0xa17c, 0x2fb8, 0x41ec, 0x4112, 0xba4c, 0x2c3a, 0xaecc, 0x458a, 0x435a, 0x1950, 0x429d, 0x1d8e, 0xaf4e, 0x36e9, 0xb205, 0xb0d3, 0x4394, 0x4489, 0x425e, 0x01cf, 0xbae3, 0x4131, 0xbae9, 0x4001, 0x0f9b, 0x431c, 0xbf91, 0xabc7, 0x40b6, 0xba5c, 0xa210, 0xbaf8, 0x0f46, 0x43d3, 0x3404, 0xaaea, 0x430e, 0x4248, 0x1d7e, 0x42e1, 0x41ff, 0x011a, 0xb230, 0x4037, 0x0e79, 0x3a4a, 0xb0d9, 0xbf97, 0xb21f, 0xafa8, 0xa731, 0x4185, 0x1b7d, 0xb095, 0x4156, 0xb2cb, 0x14d5, 0x41e0, 0x40b7, 0x31e8, 0x42ee, 0x2060, 0x3a39, 0x419b, 0x40a7, 0x1e06, 0x39bc, 0x1930, 0x4238, 0x412b, 0x43e6, 0xbfcb, 0xbaf1, 0x3518, 0x27a0, 0x42d2, 0xb2a4, 0xbfdc, 0x4293, 0x4337, 0xae16, 0x4429, 0x4050, 0xb079, 0x4031, 0x1638, 0x420b, 0x1f17, 0x43a4, 0x1bf4, 0x433b, 0x178f, 0x4060, 0xbf7b, 0x4339, 0x43d1, 0x40c6, 0xbaea, 0x154c, 0x4247, 0x412a, 0x41a0, 0x41d1, 0x4260, 0xbacf, 0x20af, 0x4285, 0x126a, 0xbf44, 0xbfb0, 0x4425, 0xb22f, 0x1a6f, 0x427c, 0x4196, 0x4367, 0x4213, 0x1620, 0xbf7b, 0x4565, 0x32f9, 0x40b2, 0x395c, 0x4105, 0xbf70, 0xbfb5, 0x412d, 0x409e, 0x180e, 0x1a80, 0x4136, 0x27e3, 0xbae3, 0x1c3a, 0xb213, 0x1301, 0xb2fb, 0xb098, 0xbfd0, 0x324c, 0x2876, 0xab33, 0x1c23, 0x248e, 0xb256, 0x1dd9, 0x0d3e, 0x01e8, 0x403b, 0x40fa, 0x41a4, 0x41f6, 0xbf69, 0x4371, 0x40dd, 0x46a1, 0x402c, 0x41e8, 0xba49, 0x40d7, 0x4605, 0x1aa1, 0x37d2, 0xba62, 0xbf61, 0xb28b, 0x4136, 0x36ee, 0x40fc, 0x3ead, 0x412c, 0x43aa, 0x233d, 0x3ac3, 0x421a, 0x4117, 0x091d, 0xb2c9, 0x163a, 0x408b, 0x4273, 0x141e, 0xa5b2, 0x40da, 0x4082, 0xbfd8, 0x4080, 0x4347, 0x1830, 0x414c, 0xbafe, 0xbf07, 0x21d3, 0x435b, 0x415c, 0xbf00, 0xb285, 0x0453, 0x1b3f, 0x383a, 0x40db, 0x36f2, 0xba4d, 0x4128, 0xbf72, 0x428f, 0xa6c7, 0x19c1, 0x2878, 0x1f51, 0xba30, 0xbae9, 0x4346, 0x43c2, 0x0c82, 0x1d23, 0x466b, 0x40e0, 0xba10, 0xac8c, 0x1ff2, 0x1adf, 0x4356, 0x401c, 0xbaf8, 0xac9d, 0xb000, 0x41e9, 0xb297, 0x4155, 0xbfdb, 0xb04e, 0xb281, 0x43a2, 0x428c, 0xb2d8, 0x42bb, 0x2cfb, 0xb2de, 0x1fc8, 0x4301, 0x45f2, 0x1ee3, 0x04c6, 0xba61, 0x4002, 0x0798, 0xb29a, 0x433d, 0x1e61, 0x42fb, 0xbf56, 0x411f, 0xb0ed, 0xbf90, 0x456d, 0x1437, 0x413c, 0x4126, 0x42ca, 0xbad4, 0x1917, 0x4350, 0xbae2, 0x4135, 0x0247, 0x418b, 0x151d, 0x173d, 0x4054, 0xb284, 0xb22f, 0x4296, 0xb28f, 0x2e6c, 0xb2c0, 0x1f1c, 0xbfa9, 0x0906, 0xba7e, 0x1b04, 0xba58, 0xbaee, 0x4000, 0xb200, 0xb0d5, 0x414f, 0x14ff, 0xb0a0, 0x4220, 0x4249, 0xba6a, 0x4187, 0xbf53, 0x43bd, 0x1de3, 0x467d, 0x3a73, 0x18c3, 0x4263, 0x42cd, 0x42ec, 0x44b5, 0x089b, 0x1ce1, 0x18d2, 0x41d3, 0x3636, 0x1de6, 0xba4c, 0xb08d, 0x1b88, 0x1d9e, 0x3a85, 0xbfd3, 0xb051, 0xb2cd, 0x1c2b, 0xb07a, 0xb261, 0x07f4, 0x3ffa, 0xb0a1, 0x42a9, 0x40f3, 0xba2c, 0x40d5, 0xb026, 0x08bd, 0x415d, 0x1e67, 0x4377, 0x43c9, 0x1d0c, 0xb278, 0x437e, 0x4159, 0x456c, 0xbf90, 0x4191, 0x428e, 0xbfa9, 0x2e5f, 0x43af, 0x41d7, 0xb259, 0x42e0, 0x4221, 0xb2d4, 0xb07e, 0x4186, 0xbf80, 0x40a3, 0x45bd, 0x416a, 0x4324, 0x4068, 0x3029, 0x43dc, 0x41d1, 0xa79d, 0xbf1d, 0x2443, 0xba2b, 0x4111, 0x1a70, 0xa102, 0x43a3, 0xb0c8, 0xba62, 0x41b6, 0x41b6, 0x09a5, 0x41a6, 0x427a, 0x3e4d, 0x419e, 0x062f, 0xb24e, 0x1110, 0x41c9, 0x41ad, 0x2a40, 0xbf6b, 0x4279, 0x4347, 0xb242, 0xb2b0, 0x097c, 0x460e, 0x1e67, 0x4144, 0xa11e, 0x1ee4, 0x351d, 0xb247, 0x43f5, 0x1a26, 0x4357, 0x41fa, 0x40cb, 0x41ac, 0xba63, 0x1d28, 0xba76, 0x28e1, 0x38e7, 0x4098, 0x42b3, 0xbfbd, 0xb2bc, 0xbadc, 0xbad0, 0xba1a, 0x40fd, 0x42ca, 0x40ad, 0x42d4, 0xb251, 0xb2f2, 0xb2fd, 0x4138, 0x1b82, 0xba23, 0x1efb, 0x11d5, 0x413c, 0xb273, 0x1ce6, 0xa2a1, 0xbf79, 0x4197, 0x43e3, 0xbae9, 0x4015, 0x43e9, 0xa21c, 0xb0d0, 0x4059, 0x408f, 0x4067, 0x4387, 0xb2f2, 0x461b, 0x4019, 0x40a4, 0x4619, 0x41f1, 0xb04c, 0x46ac, 0x43d7, 0xa438, 0x4108, 0x0e03, 0xba59, 0x41e4, 0x43db, 0xb2e0, 0xb2b8, 0x1f88, 0xbf55, 0x432d, 0xb223, 0x09e7, 0xaea6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x76fb2e83, 0x2c9a97ce, 0x4afe2dc9, 0x2a8a6d29, 0xf34a1d2a, 0xe92c22a4, 0xcd859d8b, 0xab9291ff, 0xdca83fbb, 0x24a63f86, 0xf4d326a0, 0xc55ab8ca, 0x50c313fb, 0x47bdd172, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0xfffffffa, 0x00000000, 0x00000003, 0x0000018a, 0x4000018a, 0xff91f802, 0x47bdd252, 0xfffffffc, 0x000017c0, 0x24a35e06, 0x20022b04, 0x47bdd732, 0xff91f802, 0x47bdcfba, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb247, 0xbac6, 0xb2bc, 0xba00, 0x4236, 0x4542, 0xbf84, 0x4353, 0x454b, 0x2d7d, 0xb214, 0xbf73, 0x2346, 0x1da1, 0xa600, 0x41de, 0xb0cf, 0x45c5, 0xb057, 0xbf11, 0x43f5, 0x4171, 0x026e, 0x2a0c, 0xafe4, 0x4215, 0xba19, 0x3837, 0x4677, 0x428a, 0xb281, 0xb2e1, 0x43af, 0x427f, 0xbf45, 0x41ef, 0xb289, 0x2c54, 0x4128, 0x4076, 0xb2d4, 0x41c2, 0x191b, 0x438c, 0xa81f, 0xb075, 0xb2e1, 0xbfb9, 0x4396, 0x2ac3, 0x4052, 0x4453, 0xb2c7, 0x40b6, 0x4273, 0x40b6, 0x424e, 0x449d, 0x0849, 0x42c1, 0xa3b4, 0x2413, 0x1e42, 0x40c1, 0x41f8, 0x1e05, 0xbf6e, 0xbae1, 0x1e62, 0x445c, 0x3418, 0x4363, 0x4094, 0x445c, 0x4283, 0x4611, 0xa556, 0x42fb, 0x1f16, 0x43f5, 0x1a7a, 0x020f, 0xba79, 0xbf24, 0x44cb, 0x0902, 0x4376, 0x19ed, 0x4181, 0x41b4, 0xb222, 0x4391, 0xbfcd, 0x05b5, 0x1fa3, 0x422c, 0xb210, 0x18d5, 0xbf70, 0x40c1, 0x42b9, 0x107f, 0xb281, 0xa90c, 0xbfba, 0x4301, 0x4282, 0xb254, 0x2d6a, 0x42d5, 0x42c1, 0x3ff4, 0xb09d, 0x0856, 0x41a4, 0x4357, 0x42d8, 0xbfd9, 0x3dcf, 0xba42, 0x206a, 0x4168, 0x1f5e, 0x460a, 0x0f10, 0xa577, 0x456c, 0x3967, 0x4643, 0xbf4e, 0x4183, 0x36fb, 0x41ea, 0xb0dc, 0x32d3, 0x43ee, 0x31b0, 0x4030, 0x407c, 0xb22d, 0x4455, 0x3342, 0xb0d7, 0xbf83, 0x1c18, 0x35e6, 0xb063, 0x4315, 0x45ec, 0x1f25, 0x18e9, 0xb264, 0xa7c2, 0x3e41, 0x43b8, 0x3363, 0x0c12, 0x0d27, 0x3e41, 0xbfb0, 0xba45, 0xa29a, 0xabaa, 0x3d44, 0xb2f5, 0xa599, 0x3ed6, 0x423f, 0x45da, 0x28ca, 0x40aa, 0x4288, 0x2acc, 0xbfc6, 0x21e5, 0x41a9, 0x4140, 0x4236, 0x0189, 0xb21a, 0x197d, 0xbf93, 0x13b7, 0x44a5, 0x3404, 0x443b, 0x421b, 0xba46, 0xba41, 0xb2d1, 0x022f, 0xb276, 0xba18, 0xb26a, 0xb20e, 0xb294, 0x42c5, 0x42f5, 0x1b96, 0x11dc, 0x1c7e, 0xbfb5, 0x46db, 0xab1b, 0x416e, 0xb0a2, 0xb2ab, 0x43dc, 0x3076, 0xba40, 0x2e15, 0xba09, 0xb2df, 0x4100, 0x4139, 0x004d, 0x40fe, 0xbf38, 0xa914, 0x42b4, 0x43cb, 0x43e8, 0x407c, 0x21cf, 0x2d7f, 0xb2e4, 0xb26f, 0xb2e2, 0x18d5, 0xb03e, 0x45a8, 0x433a, 0xb235, 0x44b4, 0xb000, 0x43ea, 0x313a, 0x4641, 0xb268, 0x3771, 0x40e5, 0xbff0, 0xbf9c, 0x4051, 0xba56, 0x07db, 0xb2b3, 0xb0e7, 0x40ed, 0xb264, 0x4193, 0xb08b, 0x422e, 0x4079, 0x26bc, 0x42e8, 0xba16, 0xa565, 0xba1c, 0xb06c, 0x409b, 0x43f5, 0x41fb, 0x42fa, 0xb2ea, 0x4184, 0x23d2, 0x431b, 0x4125, 0x1ef6, 0xbfa3, 0xaecf, 0x44b8, 0xb2a2, 0x1b75, 0x4145, 0x46f4, 0x42d4, 0x11f0, 0xb255, 0x411d, 0x121a, 0x42d1, 0x1a78, 0xbfae, 0x1d7b, 0xb0e1, 0x40e4, 0x422b, 0x428a, 0xba68, 0x18d6, 0x2f33, 0x4093, 0x433d, 0xbf32, 0x45c9, 0x4227, 0x423b, 0x41a1, 0x0f2d, 0x2021, 0x4679, 0x1f9b, 0x418c, 0x436b, 0xbf87, 0x41e4, 0x41bd, 0x407b, 0xa9d5, 0x1e4b, 0x00d3, 0x4389, 0x42b6, 0x46bd, 0x401e, 0x4067, 0x41d4, 0x4341, 0x4146, 0x2898, 0x463d, 0x1ece, 0x324f, 0xbaed, 0x4186, 0xbf31, 0x1dc1, 0xb275, 0x3cd7, 0xbf00, 0x4162, 0x4202, 0x431b, 0xba19, 0xb281, 0x40f0, 0xb2d9, 0x1ebc, 0xb2ae, 0x40ec, 0x42c1, 0xb2b9, 0x4241, 0x1eec, 0x081f, 0x4225, 0x3497, 0x4371, 0xac55, 0xad9c, 0xbf7e, 0xa740, 0x1a53, 0xb2ee, 0x0406, 0x403a, 0x40de, 0x1ea4, 0x4252, 0xa544, 0x41b5, 0xba14, 0x4225, 0x42c2, 0x430d, 0xbf43, 0x40db, 0xa456, 0x1b59, 0x4347, 0xb2b6, 0xbff0, 0xbae4, 0x4363, 0xb200, 0x40dd, 0xba74, 0x420a, 0xbf2e, 0xa9e7, 0x3c0c, 0x4285, 0x13d7, 0xb28b, 0xb05e, 0x1826, 0x42bd, 0x2d51, 0x4242, 0x4227, 0xbfd3, 0x43fb, 0x41ce, 0x4171, 0x416e, 0x4313, 0xa531, 0x4302, 0x010c, 0x201b, 0x2549, 0x45b4, 0x1366, 0x41a2, 0xba6f, 0xb2f8, 0x4049, 0x1e75, 0x4471, 0x45be, 0x1814, 0x4430, 0x415e, 0xbf4f, 0x187a, 0x418d, 0x2a18, 0xb278, 0x4002, 0x2b61, 0xbf6f, 0x06cc, 0x4265, 0x41fb, 0x408d, 0xbf56, 0x3963, 0x1e83, 0x45c0, 0xa06b, 0xa6e9, 0xb252, 0x1e7d, 0xb000, 0x4024, 0xb0ab, 0x4394, 0x418b, 0x1873, 0xaeb9, 0xbfaf, 0x4331, 0x1e39, 0x4405, 0x1ccb, 0x274d, 0x4344, 0xb0be, 0x41ca, 0x4268, 0x431f, 0x44c9, 0x41bf, 0x40d0, 0x42f3, 0xb09a, 0x1c08, 0x2223, 0xbf24, 0x0599, 0x46a3, 0x4207, 0x4114, 0xbf48, 0x1e64, 0xb0fe, 0x2c19, 0x4074, 0xab7a, 0x41d7, 0x4262, 0x46b0, 0x15eb, 0x42ae, 0xb23b, 0x35c8, 0x4567, 0x4177, 0x42a7, 0x4219, 0xb018, 0xb231, 0x42f7, 0xa4cc, 0x4414, 0x2023, 0x1c10, 0xbf3d, 0x1242, 0x4258, 0xb046, 0x4159, 0xba45, 0x4585, 0x4276, 0x4123, 0xb243, 0x443d, 0x409b, 0xa4f1, 0x426d, 0x42e9, 0x3228, 0x4645, 0x4543, 0x2747, 0x42ed, 0x4316, 0xbf7e, 0x4120, 0x41ec, 0x41eb, 0x4174, 0x424f, 0xbfb1, 0x03c4, 0x2fe1, 0xb25d, 0x4068, 0x1b8f, 0xb270, 0xb058, 0x1c88, 0xbf6f, 0x1fce, 0x1bb9, 0xb23a, 0x4300, 0x4480, 0xaf8f, 0xb21d, 0xba5e, 0xba13, 0x1e5e, 0xbad9, 0x41a4, 0xb2c8, 0x40dd, 0xb27c, 0x382e, 0x089a, 0x429a, 0x4076, 0xae8a, 0xbf72, 0x1e10, 0x433d, 0xa7c9, 0xb0b4, 0xb034, 0xb0d4, 0x41b6, 0x1858, 0xbf7e, 0x1ec5, 0x40e7, 0x02cc, 0x41ce, 0x06fb, 0x42e7, 0xa6f3, 0x1a85, 0x40e9, 0x427f, 0x0051, 0xb0ca, 0xb28e, 0x1888, 0x4682, 0x4344, 0x1f2e, 0x429e, 0x426a, 0x324f, 0x0eae, 0xbf9d, 0xb00a, 0x4081, 0xab91, 0x465e, 0xbf2b, 0xaf28, 0x4091, 0x428f, 0x45c4, 0x191d, 0x191b, 0x42bb, 0x0d5a, 0x428a, 0x463e, 0xbf44, 0x400a, 0x40b3, 0x0aa6, 0x4085, 0xba42, 0xbf1f, 0x425e, 0xbfa0, 0xb008, 0x4315, 0xb232, 0xba77, 0x1412, 0x41dc, 0x43c2, 0x0566, 0xbf6e, 0x34aa, 0xb272, 0x1b03, 0x4171, 0x1b82, 0x0772, 0x2458, 0xab4d, 0x1c39, 0x43f9, 0x429b, 0x40ad, 0x42a2, 0xba19, 0xad15, 0xb078, 0x42f2, 0x1e62, 0x43a8, 0x3a59, 0x425a, 0xb00c, 0x0ce3, 0x29fc, 0xb291, 0x3847, 0x1bc4, 0xbf86, 0x431b, 0x1b36, 0xb26f, 0x40ee, 0x43aa, 0x124f, 0xbac5, 0x0f90, 0x402d, 0xb2d0, 0xb2b6, 0x46b1, 0x4327, 0xa182, 0xbf9a, 0x406e, 0x43a9, 0x430a, 0x05fc, 0x41d8, 0x4072, 0xbf12, 0x4287, 0x43d0, 0x2863, 0x4392, 0x1bf0, 0xa512, 0xbf92, 0x442c, 0x187c, 0x4429, 0xbad3, 0x43a2, 0x42cb, 0xa016, 0x422c, 0xbfb1, 0xb245, 0x1b85, 0x42a3, 0xba1d, 0xb023, 0x4185, 0x40ce, 0x43e7, 0x1c32, 0x4222, 0xa312, 0x34a2, 0xbfc0, 0x42a0, 0xb26c, 0x1e0d, 0x19fd, 0x434a, 0xba0c, 0x41be, 0x40a1, 0x1009, 0x410c, 0xbf9c, 0x441c, 0x41ca, 0xbf0b, 0x088d, 0xbf60, 0x2873, 0x1968, 0x41b9, 0x411f, 0x06ff, 0x463f, 0x40f2, 0x4220, 0x4278, 0x41bb, 0x06d0, 0x42d7, 0x42e2, 0x43c5, 0x1ece, 0x3ef3, 0x4218, 0xb2ae, 0x41bc, 0xa77e, 0x1068, 0xbf28, 0xbf00, 0xb296, 0x11b9, 0x4309, 0xbafd, 0xb286, 0x40e8, 0xbfd1, 0x437a, 0x35d5, 0x42d1, 0x424e, 0x4548, 0x40a7, 0x4056, 0x40ff, 0x44c9, 0xb26a, 0xba07, 0x43ca, 0x4310, 0xb2b3, 0x40ce, 0x1810, 0x462f, 0xb28f, 0xb276, 0x4266, 0x4669, 0x1e7a, 0xb21a, 0x21ce, 0x4611, 0xbf53, 0xbac4, 0x4477, 0xb2ff, 0x1eda, 0xb2de, 0x0819, 0xb28b, 0x4029, 0xbaeb, 0x050d, 0x43bb, 0x4327, 0xb23b, 0x27fd, 0xbf98, 0x1d5c, 0x4275, 0xaa35, 0x186d, 0x2879, 0xbaf5, 0x4011, 0x401c, 0x43a0, 0x3fd9, 0xba7a, 0x1b93, 0xb288, 0xaad0, 0xbf22, 0x3c38, 0xb26b, 0x2763, 0x3f20, 0x4003, 0x1a28, 0xb26d, 0xabec, 0x0d07, 0x0bf5, 0x4117, 0xbac5, 0x434a, 0xb0ee, 0xb028, 0x433d, 0x4695, 0xba77, 0x3ea1, 0x4177, 0x446f, 0xbad3, 0xa161, 0x11de, 0x1a2b, 0xb23f, 0xb2cd, 0x3450, 0xbf61, 0x3f26, 0xb291, 0x34e5, 0x46c1, 0x1fce, 0x4215, 0x423b, 0xab8c, 0x1627, 0x1011, 0x40e9, 0xb0d3, 0x1f42, 0x4164, 0xbfb3, 0x1886, 0x4018, 0x2218, 0xbadc, 0x4245, 0x42ab, 0x1f0c, 0xb2ec, 0xaa74, 0xad5c, 0x413a, 0xbf79, 0xb07a, 0x4213, 0x43ed, 0x41ac, 0x423d, 0x06e5, 0x0ad8, 0x1aaf, 0x4388, 0xb280, 0xbae9, 0x403a, 0xa331, 0xb2d9, 0xbfdf, 0x421e, 0xb0ad, 0x424e, 0x4065, 0x422c, 0x1001, 0x1a77, 0x4017, 0xbf65, 0x14cb, 0x4573, 0xb26b, 0x4371, 0x4026, 0x46fc, 0x42e2, 0xa06e, 0xae37, 0x0f59, 0xb045, 0x42d9, 0x4136, 0xba1b, 0xb2cf, 0xb2da, 0x3ce5, 0x3b2e, 0x44f9, 0x46b9, 0x3256, 0x4265, 0xbff0, 0x410d, 0xbff0, 0x2599, 0x4287, 0xbf74, 0x42ae, 0x4009, 0x2b2f, 0x4167, 0x4321, 0x459b, 0x42da, 0x416f, 0x406c, 0x42c4, 0xb2c2, 0x4630, 0x40db, 0x39ab, 0x0d75, 0xbf55, 0xb2df, 0x3c88, 0xb084, 0xb220, 0x2296, 0xbae0, 0x3e6b, 0xb002, 0x1d1f, 0x2309, 0xb26d, 0xba77, 0x2e13, 0x4084, 0xb29b, 0x10cb, 0x4397, 0xbf3b, 0xb0da, 0x42ee, 0xb2e6, 0x42c4, 0x4603, 0x1c1b, 0xbfd0, 0xbfc0, 0x4297, 0x1f20, 0x3f0e, 0x4246, 0x27a5, 0x42ca, 0xb2d0, 0x02d7, 0xaf51, 0x1d35, 0x41c5, 0x43cc, 0xb251, 0xb24f, 0x428f, 0x4101, 0x3bfb, 0x1f9d, 0xbf46, 0x4314, 0x469c, 0x1932, 0x40bf, 0xafff, 0x1cb1, 0x4141, 0xa9be, 0x43ed, 0x41a7, 0x4035, 0xba21, 0x1f1b, 0x1f56, 0x4054, 0x1554, 0x456b, 0x4249, 0xbf41, 0x4327, 0xba1f, 0x41ab, 0x407a, 0xb050, 0xbaf9, 0xb272, 0xabcc, 0x40c8, 0x4078, 0x1bcf, 0x42ae, 0xba1c, 0x42fc, 0x40b0, 0xb2ef, 0xb0c6, 0x401d, 0x46fd, 0x29e9, 0xb2a4, 0x1805, 0x42b5, 0xb01d, 0xbf47, 0x40e8, 0xb0d8, 0x056d, 0x4271, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x1a5fd34d, 0x3103aa3c, 0x3e4f9ef4, 0xf82df4a2, 0x3ba27697, 0x32d28d47, 0x5158d643, 0x5cf29a73, 0x8ce84a47, 0x4e93790b, 0xc7f42d1c, 0x6b5dd957, 0x35ca5f7d, 0xde453ad4, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x00000000, 0xffff9c02, 0xffffff91, 0x000004b0, 0x00000000, 0x00000000, 0x00000091, 0x00000000, 0x00000844, 0x00000000, 0x19860000, 0x6b5dd957, 0xffff8204, 0x000016e4, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xa14c, 0x4132, 0x41bc, 0x4020, 0x0b92, 0x157b, 0x41d2, 0x4033, 0x0ac1, 0x1a9f, 0x4029, 0x27ea, 0xb2ec, 0xbff0, 0x1c03, 0xb249, 0xbfba, 0x2af5, 0x063a, 0xb2f7, 0x1fde, 0x4650, 0x419d, 0x4254, 0x423c, 0x45c5, 0x43f5, 0xbfb7, 0xb22c, 0x439d, 0x41ae, 0xa021, 0x405d, 0xb2f5, 0xa199, 0x4328, 0x43c1, 0xa29f, 0x3ace, 0x40cd, 0xb29a, 0x4192, 0x4165, 0xbaf5, 0x281e, 0x2f02, 0xa053, 0xb2a9, 0x414c, 0x2319, 0x40bb, 0xbfdb, 0xbf80, 0xa7d7, 0xbff0, 0x424b, 0xb031, 0x4063, 0xbf29, 0x443d, 0xba16, 0x2f1f, 0x2cae, 0x4108, 0xb0e1, 0x4362, 0x3931, 0xbac3, 0xbf8d, 0xb286, 0xb2bd, 0x43b0, 0xb01c, 0x42f2, 0xbfb5, 0xba58, 0x4066, 0x42a6, 0x40f4, 0x25d9, 0xba15, 0xba76, 0x4173, 0x32e9, 0x4184, 0x45f3, 0xb0e2, 0xb2e3, 0x32b6, 0xbf81, 0x4136, 0x448a, 0xba7b, 0xba55, 0xb23c, 0x4153, 0x4323, 0xb2cb, 0x18e5, 0x4161, 0xba7b, 0x363d, 0x408c, 0xbf39, 0x400f, 0x406c, 0x41c0, 0x420d, 0x32e5, 0xb24a, 0x2bb8, 0xaa8c, 0xb053, 0xb276, 0xbf4d, 0x4109, 0xae28, 0xba33, 0x429e, 0xb055, 0x2c5a, 0x43c2, 0xb283, 0x05b1, 0xbfd3, 0x19f9, 0x4081, 0xbace, 0x43c3, 0x455e, 0x42aa, 0xbfcf, 0x42d3, 0x4000, 0xb23a, 0x42ee, 0x41b4, 0xba39, 0x2cd1, 0x303c, 0x43a4, 0xbfbc, 0xb2db, 0x2de9, 0xbaf6, 0xafed, 0xba77, 0x42f4, 0x4163, 0x41b5, 0x41b5, 0x3a75, 0x4298, 0x4215, 0xba5c, 0x42bc, 0xb2c8, 0x41ec, 0x2f90, 0x42b1, 0xb065, 0xbfbc, 0xba6f, 0x456e, 0xb234, 0xbf42, 0x44b4, 0x46f9, 0x1b3f, 0x1d00, 0x4201, 0xb223, 0xa1f4, 0x27e7, 0x41c0, 0x1e5c, 0x46b1, 0x412b, 0xa523, 0xbf00, 0x4033, 0x151c, 0x3328, 0xba10, 0x420f, 0x45b4, 0x1217, 0xbf6f, 0xa77d, 0x2c63, 0x1985, 0x411a, 0xb262, 0xbf1f, 0xb260, 0x1a16, 0x43ac, 0x41d4, 0x440c, 0x41f6, 0x3409, 0xb2a6, 0xbfe1, 0x2013, 0x43e1, 0x43fd, 0x46d1, 0x4160, 0xbf73, 0x4132, 0x429b, 0x23b3, 0xbae9, 0x10bd, 0xba7f, 0xba1b, 0x1886, 0x109c, 0x41a6, 0xb201, 0x1be2, 0xbf95, 0x410e, 0x46a2, 0x430a, 0x02f1, 0xb2d5, 0xb206, 0xbae9, 0x1b5c, 0x40f3, 0x44da, 0xbf43, 0xabc8, 0x0f66, 0x1883, 0x458b, 0x426d, 0xa283, 0x4033, 0x416e, 0x4173, 0x4461, 0xb2f6, 0x1139, 0xb0b0, 0xbfe0, 0x437a, 0x41f9, 0xb226, 0x4359, 0x4160, 0x4618, 0xb27e, 0x41ab, 0xbfb4, 0xb0f4, 0x4603, 0x401e, 0x2dd7, 0x1b5c, 0xb2ec, 0x16ab, 0xb00d, 0x41b9, 0xa2eb, 0xb00d, 0x40bf, 0x2e03, 0x46b2, 0xbf13, 0x38c1, 0x1ed1, 0xbac9, 0x4036, 0xb0bf, 0x41bc, 0xba14, 0x02dc, 0x41c1, 0x4383, 0xba42, 0xb223, 0xba53, 0x4037, 0xba00, 0x43d0, 0x4362, 0x42ea, 0xb03f, 0x41a3, 0x4111, 0x001b, 0x0080, 0xbf7f, 0xb299, 0x40e7, 0x4158, 0x4196, 0x1a5d, 0x07f6, 0x41ee, 0x0c49, 0xb2c8, 0xb243, 0x409a, 0x392d, 0xb2bf, 0xa325, 0x4629, 0xbfd4, 0x4334, 0x4305, 0x2dd2, 0xbfe0, 0xb01d, 0x36a2, 0x4211, 0xbfe4, 0x4151, 0xbfb0, 0xb2da, 0xbf70, 0xb2ee, 0x43a8, 0xbf85, 0x448d, 0x43e6, 0x1b34, 0x18e6, 0x43f7, 0x1d9f, 0xb251, 0x40b6, 0x0822, 0x4048, 0xb086, 0x0161, 0xbaca, 0xb211, 0x0500, 0xb076, 0x400d, 0xbf9c, 0x40bd, 0x41bd, 0x22d5, 0x421d, 0x3876, 0x1a86, 0x408e, 0x357b, 0x4567, 0x41f6, 0xb29f, 0x43f4, 0x439b, 0x431b, 0x4115, 0x3962, 0xb0cc, 0x4198, 0x0afa, 0x445d, 0xbac5, 0xaa18, 0x41e1, 0x43e4, 0xbf59, 0x43b8, 0x431e, 0x1849, 0x4233, 0x4393, 0x42dd, 0x4275, 0x430f, 0x420b, 0xbf90, 0x4116, 0x43b5, 0xb226, 0xad7b, 0x45e8, 0x4202, 0x43c7, 0x2c0c, 0x4070, 0x4185, 0x43ce, 0xbad1, 0x1a16, 0x1c88, 0x413b, 0x4163, 0x2d64, 0xb03b, 0xbf63, 0x1d22, 0x40c5, 0x285b, 0x1f82, 0x41ba, 0x417d, 0xba1b, 0x4219, 0x40e2, 0x4297, 0x4307, 0x1873, 0x4050, 0xbfa0, 0x229d, 0x4055, 0x436a, 0x3545, 0x1a18, 0x42ef, 0xb280, 0x40a1, 0x1c13, 0xb05e, 0x408e, 0x42d4, 0x2688, 0x40d3, 0xbfc3, 0xa2eb, 0x421d, 0x440e, 0x090b, 0xb264, 0x4160, 0xb2b3, 0xbad4, 0xba36, 0x40f8, 0xb2f5, 0x3be0, 0xb096, 0x1da7, 0x1f4e, 0xba52, 0x4155, 0x4273, 0xbf8e, 0x42f4, 0x4247, 0x4148, 0x428e, 0x4481, 0x1cd6, 0x436c, 0xb22a, 0x4189, 0x27ea, 0x42c1, 0x435f, 0x41ca, 0x24f5, 0x40a8, 0x4304, 0x4575, 0x416a, 0x42d4, 0x41d8, 0x1cf2, 0x430a, 0x4249, 0xa6d5, 0x4208, 0x4361, 0xbf77, 0xb2bf, 0x4061, 0x42f5, 0x1df1, 0xb2fb, 0x2eb2, 0x1f36, 0x0636, 0x4233, 0xb051, 0x0d22, 0x437a, 0x187c, 0x431d, 0x2642, 0x4329, 0xba5e, 0x402e, 0x1a7d, 0xb27c, 0xbfc0, 0x4382, 0x430d, 0xaad4, 0x4033, 0x423d, 0x4266, 0xbfc1, 0x1a59, 0x4349, 0x416c, 0xb236, 0x4337, 0x2973, 0x43cb, 0x41d8, 0xb031, 0x431b, 0x1bce, 0x1553, 0x40ea, 0x0c9e, 0xba3b, 0x1192, 0x41a9, 0x4261, 0x4086, 0x1582, 0x4637, 0x4056, 0xbf8c, 0x4600, 0x429b, 0xbfac, 0x2ba6, 0x2e02, 0xbf2d, 0x463a, 0x41d0, 0x1d05, 0x43f9, 0x2c9c, 0x40af, 0xb00f, 0x2d8b, 0x4201, 0x4354, 0xba21, 0x1e4b, 0x41ae, 0xb2ba, 0xb28c, 0xa6c2, 0xbafd, 0x27e8, 0xa499, 0xbf8f, 0x4272, 0x076d, 0x4342, 0x4337, 0x1f5e, 0x41a3, 0x37d5, 0x417e, 0x4215, 0x4678, 0x143e, 0xbafc, 0xab29, 0x4095, 0xb28d, 0x0ed2, 0xbf80, 0xabb9, 0x433d, 0xba51, 0x0bba, 0xaa69, 0xa93d, 0x4117, 0xbfc7, 0x312f, 0xb099, 0x119d, 0x4393, 0x40e9, 0xbf94, 0x1cb6, 0x4294, 0x43f5, 0x42b8, 0xb26f, 0x2772, 0xb0a6, 0x413b, 0xadeb, 0x408e, 0xbfbc, 0x4542, 0xba61, 0xb066, 0x40f9, 0x45d5, 0xabc9, 0xba51, 0x19c0, 0x4202, 0x4245, 0x4162, 0xb256, 0x0fff, 0x1ca8, 0xbaf9, 0x3ce8, 0x45ee, 0x1bd9, 0xbfa0, 0xa474, 0xbfb3, 0x42c1, 0x39dc, 0xb08f, 0x303f, 0xba67, 0x1ec2, 0x43bd, 0xb2b5, 0x42df, 0x0cbd, 0x240e, 0x402b, 0x411d, 0xba53, 0xb20e, 0x41b8, 0xbf78, 0x413f, 0xbff0, 0x0612, 0xbf36, 0x4685, 0x3a69, 0x4318, 0x4137, 0xba21, 0x42ba, 0x4259, 0x4207, 0x41d7, 0xb2bc, 0x3b7e, 0x40d0, 0x423b, 0xab19, 0xb2ae, 0xb2b8, 0x43a4, 0xba50, 0xba2f, 0x21c8, 0x24c1, 0x4110, 0x422d, 0xbf4d, 0xa1c2, 0x1745, 0x18d8, 0x095c, 0xb260, 0x4070, 0xb23f, 0xa815, 0x346e, 0xb21f, 0x418c, 0x407f, 0xb259, 0xba4e, 0x3d84, 0xb2ff, 0x40d6, 0x44b3, 0x1bfb, 0xa6b7, 0x41f2, 0x412c, 0x4262, 0xbafd, 0x4460, 0xbfb0, 0x414c, 0xbfa9, 0x129a, 0x43d9, 0x09f1, 0x4262, 0x2b67, 0x4369, 0x40b7, 0xbf4c, 0x4473, 0xba7f, 0xba3f, 0xba30, 0xb039, 0x4378, 0x431a, 0xbf71, 0xb20b, 0xba76, 0x410a, 0x4284, 0xa2fb, 0xb283, 0xb2c3, 0xa762, 0x3e09, 0xba6c, 0xbf65, 0x4387, 0x218b, 0x2a1c, 0x4184, 0xb2b6, 0x41c0, 0x4298, 0x4249, 0x4397, 0x1ed0, 0xbafe, 0x4659, 0x1fc5, 0xa570, 0x4684, 0x40d0, 0xb012, 0x2c6b, 0xb22b, 0x105a, 0x4229, 0x409a, 0x43ce, 0xb06d, 0x4085, 0x4280, 0x12a1, 0xbf4d, 0xb09d, 0x436c, 0x281c, 0x40ff, 0x0cc9, 0xbfad, 0x1195, 0x42b4, 0x43f7, 0x415b, 0x40ad, 0x43e6, 0xb016, 0x4086, 0x4037, 0x0149, 0xbfd0, 0x41f5, 0x3cc4, 0xad52, 0x2422, 0xa9f5, 0x41c6, 0x26b2, 0x4226, 0xb27e, 0xbf3e, 0x02a3, 0xbac8, 0xb225, 0xbf31, 0x0663, 0xaea1, 0x4010, 0x00b1, 0x4201, 0xb0ce, 0xba4c, 0x400b, 0x4442, 0x436d, 0x0722, 0x42f6, 0x4321, 0x4001, 0x1f25, 0x411e, 0x401d, 0xba3a, 0x0ba3, 0xb2e1, 0x0141, 0xb01f, 0xb26c, 0x41a7, 0x435a, 0xaa86, 0xbfbb, 0x459c, 0x1caf, 0x441e, 0x43e5, 0x3643, 0x4240, 0xb292, 0x3097, 0xb2c2, 0x1b2f, 0x0db8, 0x40ab, 0xba0c, 0xb21c, 0x415e, 0x42ba, 0x4182, 0x25a2, 0x42c0, 0xb08f, 0x2808, 0xbf97, 0x400b, 0x43af, 0xb22e, 0x416e, 0xba40, 0xbf80, 0x43fe, 0x4349, 0xbada, 0x4003, 0xbff0, 0x0a40, 0x3d0b, 0x412c, 0xbf2f, 0x430f, 0x101c, 0x1a47, 0x41ce, 0xb2de, 0x4359, 0x4297, 0x41ad, 0x1d4a, 0x44eb, 0xbf80, 0x436f, 0xa5d7, 0x1cb2, 0xbf1f, 0x2907, 0x1b41, 0x43c7, 0x42a7, 0x41d0, 0xb2d9, 0x429f, 0x43d2, 0xa0eb, 0x4106, 0xb2d9, 0xa5a9, 0xa43e, 0x43d7, 0x12d3, 0x1e49, 0x41e3, 0x4589, 0x433d, 0x430f, 0xab65, 0xb22d, 0xbaee, 0xb22a, 0xaec2, 0xb265, 0xba6c, 0x1a12, 0xbfb2, 0x4137, 0xa12b, 0x4115, 0x4306, 0x43e9, 0x1c8b, 0x1542, 0x2360, 0x45ea, 0x4387, 0xba7a, 0x1fab, 0xb245, 0xbfd8, 0x2725, 0xbf78, 0x4146, 0x428a, 0xb21b, 0x19f9, 0xb282, 0xb2da, 0x4622, 0x40ae, 0x46ac, 0x41ac, 0xbf33, 0xb213, 0xb22d, 0xba56, 0x40b7, 0xab93, 0x18e7, 0x4372, 0x40ce, 0xb23d, 0x1a8c, 0xb281, 0x42a2, 0x41ce, 0xb22e, 0x43ee, 0xb221, 0xbf2f, 0xb2c3, 0x4081, 0x23a1, 0x4162, 0x42b6, 0xb004, 0x1e59, 0xb2d0, 0x4121, 0x435d, 0xb2af, 0xb2f0, 0x42ab, 0xbfe0, 0x1cb8, 0x44f3, 0x4605, 0x1e0e, 0x4625, 0x31e0, 0x4570, 0x40ec, 0xbf7a, 0x4173, 0xa0f7, 0xb2f3, 0xba19, 0x4149, 0x1f51, 0xba44, 0x29ac, 0x402d, 0x1808, 0x01f2, 0x41bf, 0x4186, 0x42e4, 0x10e4, 0x1f7a, 0x416a, 0xb0b8, 0x4130, 0x4467, 0xbf4a, 0xba17, 0x4237, 0xb2bd, 0x1cb4, 0x1955, 0x3010, 0xbf6c, 0x419f, 0xbf90, 0x4326, 0x423d, 0x40b3, 0xb24d, 0x413a, 0x414f, 0x4317, 0x4018, 0xb254, 0x0dd2, 0x4281, 0x4175, 0x4590, 0x19ad, 0x405f, 0x1d33, 0xbf2b, 0x3132, 0x1412, 0x1bbc, 0x1bb8, 0x4259, 0x426e, 0x414f, 0x4358, 0x43df, 0xb056, 0x4677, 0xa879, 0xade8, 0xba78, 0x436a, 0xb211, 0x4013, 0x40e9, 0x4280, 0xbff0, 0x41e8, 0x068f, 0x185e, 0x43c5, 0x43c6, 0xb021, 0xbfa7, 0x42da, 0x1aa9, 0x4214, 0xa017, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x371597f2, 0x76cf74c6, 0xf6f07bf9, 0x14bc3ff2, 0x44d46dd4, 0x85b9d3ca, 0x44827883, 0x4ba02ac3, 0x45d4fe43, 0x55355bd7, 0xa1571ee2, 0x4065eef1, 0x6a318c25, 0x4295c0bf, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0x00000000, 0x45d4fe43, 0xa1571ee2, 0x00000000, 0x82fbba0c, 0x00000064, 0x4295cc27, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x200c, 0xbf4b, 0x3eb3, 0x4389, 0x4279, 0xb2c4, 0x432f, 0xb2ad, 0x1a23, 0xa8ec, 0x4396, 0x424d, 0x4177, 0xb0e4, 0x30b3, 0x4175, 0x42f2, 0x43e7, 0x40d1, 0x1e00, 0xb20a, 0x1f46, 0x42a6, 0x401c, 0xbf26, 0x4342, 0x4082, 0x38b3, 0x01dc, 0xbfe0, 0x429c, 0x4192, 0xb294, 0x1c89, 0xbf8a, 0x4290, 0xba28, 0xba38, 0x4255, 0xa885, 0x41bc, 0x1e2c, 0x4188, 0xb25d, 0xb245, 0x0f0d, 0x1af7, 0x1a13, 0x00b2, 0x23e6, 0x05cc, 0x350b, 0x43da, 0xb2ee, 0xa17e, 0x4290, 0x11a2, 0x4151, 0x1629, 0xbfaa, 0x44a8, 0x3802, 0x2ddb, 0x3c35, 0x4328, 0x1c04, 0x41cc, 0x4010, 0x4031, 0x41b9, 0xbfcd, 0x420c, 0x43fe, 0xb209, 0x4070, 0x4009, 0xb04f, 0x1b1a, 0x1ad0, 0xba64, 0xab75, 0xbaf7, 0xb01e, 0x41f5, 0x423f, 0x46d9, 0x0581, 0x40aa, 0x2fd3, 0x19d7, 0x1d89, 0x357c, 0xb0e6, 0x1ef7, 0xbf29, 0x44e1, 0x066d, 0x3b50, 0xac4d, 0x19a8, 0x449d, 0x0215, 0x0662, 0x4348, 0x4168, 0x4392, 0xac1f, 0xb02d, 0xbf88, 0x1835, 0xa8cb, 0xb06d, 0x426f, 0x3a1f, 0x1aeb, 0xb01d, 0xb014, 0x4221, 0x1e94, 0xbf53, 0xb287, 0xadbc, 0xb275, 0x42b6, 0x46c4, 0xab25, 0x394f, 0x1a27, 0x1dc2, 0xbff0, 0x43bc, 0xb062, 0xb2bd, 0x4109, 0x4305, 0xbf37, 0xb09e, 0x40a7, 0xb2ca, 0x42dd, 0xb248, 0xb0c0, 0x311c, 0x40cd, 0x40e7, 0x428f, 0x422c, 0x4213, 0x4101, 0x1838, 0x1f0a, 0x1b05, 0xbad3, 0xb28d, 0xbf70, 0x40bc, 0x4383, 0x43f6, 0x0bb7, 0x4036, 0x4582, 0x3552, 0xbf0f, 0x4329, 0xb244, 0xb2cf, 0x4095, 0xb261, 0x41f2, 0x083b, 0xb278, 0x0ce4, 0x415c, 0xbf3f, 0x4001, 0xb0ec, 0x0cbe, 0x1347, 0xbf1d, 0x4254, 0x199b, 0x1f28, 0xb2c1, 0x071c, 0x4234, 0xba53, 0xb074, 0xba5d, 0x4391, 0x3240, 0xb2e7, 0x4320, 0x19b5, 0x020e, 0x439b, 0x2782, 0xba6a, 0x094a, 0xb044, 0x40f5, 0xb21a, 0x40cd, 0xb274, 0xbf9a, 0x424f, 0x4624, 0x4009, 0xb23e, 0x42ef, 0x4284, 0x4548, 0x1da1, 0x19f5, 0x4100, 0xb2b2, 0xad48, 0x40a6, 0x1f3d, 0x45e1, 0xb22d, 0xbf12, 0x4008, 0x08ae, 0x420c, 0xbae7, 0x4273, 0x4225, 0x1c10, 0xbf68, 0x4070, 0x4113, 0xb08a, 0x0a0a, 0xa398, 0x08f0, 0xbfe0, 0x0f55, 0x41c3, 0x1c12, 0x385a, 0xa0f9, 0x1e68, 0x13ce, 0x1ffe, 0x4032, 0x435f, 0xba10, 0xb231, 0xb27f, 0xbf13, 0x42e5, 0x16d2, 0xb2e2, 0x4379, 0xba75, 0x428e, 0x4239, 0x4311, 0x42df, 0x43ed, 0x29dd, 0xbfa4, 0xb05e, 0xaf16, 0x1b1d, 0x4101, 0x415a, 0xb2a8, 0x40ed, 0x3eb0, 0x4346, 0x412b, 0xb28f, 0x415a, 0xbfd9, 0x425f, 0x4182, 0xb249, 0x43e6, 0xb2b9, 0x1400, 0x4073, 0x1272, 0x42d9, 0x4391, 0x407b, 0x43ba, 0xb07c, 0x41b6, 0xba09, 0x4419, 0xbfba, 0x40ae, 0x44f2, 0x1e03, 0x3d31, 0x4313, 0x417c, 0x1f4a, 0x40e1, 0x1b00, 0x425e, 0x46d2, 0x406c, 0xbfc3, 0x1ef0, 0x4266, 0x407f, 0x2f1e, 0x2462, 0x43ae, 0xb077, 0x4187, 0x4624, 0x4366, 0xba60, 0x43ed, 0x4429, 0xb20c, 0x2bc8, 0x181e, 0xbfb0, 0xb02b, 0xb26f, 0xa896, 0x42e8, 0x1cc9, 0x404d, 0x2159, 0xb27e, 0xbf2e, 0xbff0, 0x0378, 0x4598, 0x45ec, 0xbfda, 0xba4e, 0xb250, 0xb2b4, 0xb0f9, 0x43d1, 0xb027, 0x41e3, 0x4076, 0x43df, 0xbafd, 0x4397, 0x4046, 0x4567, 0xb251, 0x454f, 0x44e5, 0xbf8d, 0xba25, 0x461d, 0x415d, 0xae84, 0x4577, 0xbaeb, 0x0cd3, 0x4201, 0x025f, 0x4126, 0x2412, 0x1896, 0x424a, 0x1daf, 0xb22e, 0x4311, 0xba23, 0xbaf4, 0x4243, 0x3d58, 0xb0c3, 0xb0dc, 0xb230, 0x424f, 0x401f, 0xbf32, 0xa999, 0x42d4, 0x4306, 0xbfd2, 0x45f4, 0xaab3, 0xb0d5, 0x34d6, 0xb280, 0xb229, 0x4068, 0x1e55, 0x4368, 0x40bd, 0xba7f, 0xba63, 0x4108, 0x1fd9, 0xb000, 0xb295, 0x40b9, 0x2f6d, 0x447c, 0x43b9, 0xbf7c, 0xb059, 0xba47, 0x40d6, 0x1e41, 0x378b, 0xbfa0, 0x25c8, 0x2b2f, 0x014f, 0xbf03, 0x4139, 0xb238, 0x01a8, 0x3707, 0x1878, 0x4301, 0x19e6, 0xab18, 0xba2e, 0x4126, 0x43eb, 0xb29f, 0x42a6, 0x41bb, 0x0eca, 0x08f5, 0x414a, 0x41c5, 0x4305, 0x43bd, 0x40ed, 0x0b88, 0x4387, 0x4380, 0xb0fd, 0x31e5, 0xba59, 0xbf90, 0xbf57, 0x40f1, 0x435f, 0x385f, 0x403b, 0xbadc, 0x301d, 0xb265, 0x440c, 0xbac3, 0xb2a3, 0x402c, 0x45bb, 0x43e4, 0x2833, 0x4109, 0xae7c, 0xb0d2, 0x458e, 0x40ec, 0x14c5, 0x405a, 0x2480, 0x0070, 0xbf24, 0x41b2, 0xba46, 0x401c, 0xbf27, 0x2583, 0xb24d, 0x41ff, 0x32fc, 0xb0d4, 0x4045, 0x4145, 0x00ca, 0x4067, 0x42ac, 0x4094, 0x3145, 0x4120, 0x1e2a, 0xba20, 0x4156, 0x3e8c, 0x40e0, 0x1c02, 0xba0d, 0xb061, 0x46fd, 0x42e6, 0xbf6b, 0x411b, 0x1fbf, 0xb070, 0x1846, 0xb22b, 0x4695, 0x407e, 0xba6e, 0x10bc, 0xbf94, 0x42fd, 0x43b9, 0xac56, 0x414a, 0x0183, 0x41a6, 0x42f3, 0x433d, 0x19c1, 0x4053, 0x4204, 0xb282, 0xb241, 0x1e8c, 0xb287, 0x421b, 0xb270, 0xb036, 0xb285, 0x4220, 0xbaca, 0xb289, 0xb275, 0x2849, 0x410d, 0xa4a3, 0xb231, 0xbfb4, 0x1e9b, 0x413e, 0xa953, 0xb289, 0x408c, 0x4237, 0x3547, 0xb0fe, 0xbf4c, 0x45d8, 0x424e, 0x40a3, 0x1e5c, 0x41c3, 0xbafe, 0x32ef, 0x4191, 0x41f2, 0x1d7d, 0xaacc, 0x43f2, 0x418f, 0xb203, 0xb0cb, 0x3ac6, 0xa494, 0x41fe, 0x4015, 0xbace, 0x34f5, 0x4385, 0xbf1c, 0x323a, 0x414f, 0x4312, 0x402a, 0x426f, 0x41b7, 0x4257, 0x2fda, 0xb2d1, 0x4251, 0xbf34, 0x405a, 0x44c0, 0xbfbc, 0x45e2, 0x4285, 0x40b7, 0x4293, 0xb0fa, 0xba14, 0x15bf, 0x4350, 0x420a, 0x418c, 0x33de, 0xba58, 0xa868, 0x1eb6, 0xb202, 0x43a4, 0x439c, 0x42a7, 0x4323, 0x45e6, 0x1e1b, 0xa2d3, 0x1833, 0x1896, 0x4174, 0x411b, 0xbf6d, 0x13be, 0x2d69, 0x41eb, 0x429c, 0xb2d2, 0xb2a0, 0x2ec9, 0x41ee, 0x1b0a, 0x1fbf, 0x423a, 0x1db5, 0xba10, 0x40df, 0x41dc, 0xb25f, 0xb2b9, 0xb2dd, 0x45dd, 0x403d, 0x31a2, 0x4190, 0x402b, 0xbf5c, 0x43ce, 0xa551, 0x386b, 0xbae4, 0x09fd, 0x4318, 0xbf46, 0x413d, 0x1c10, 0x41e9, 0x42b9, 0x1792, 0x419e, 0x1f8d, 0xbfb2, 0xb2be, 0x426b, 0x12da, 0xa7d4, 0x19a3, 0x40c7, 0xbad2, 0x43ee, 0x45cb, 0x0e6b, 0xabd7, 0x094f, 0xbac9, 0x40a0, 0x43aa, 0xbfe2, 0x4375, 0x2acd, 0x41de, 0x43e1, 0x1035, 0x067c, 0x40e4, 0x43cd, 0x1499, 0x19fd, 0xb09a, 0x43eb, 0x3b91, 0x0573, 0x4043, 0x1edc, 0xba7a, 0x4341, 0x37ca, 0x4030, 0x1037, 0xbae7, 0xb2b5, 0x4339, 0xbf03, 0x1731, 0x461f, 0xb040, 0x417d, 0x41dc, 0xb013, 0x1e18, 0x05e4, 0xbfb8, 0x41d7, 0x1c58, 0x0e05, 0x41ed, 0x431f, 0x406f, 0x43f3, 0x19df, 0x4058, 0x412a, 0x1e46, 0x4248, 0x4561, 0xbf6d, 0x4073, 0x429a, 0x4280, 0x1bd4, 0xb0fe, 0x4024, 0xbfde, 0x41ca, 0x4143, 0x424d, 0x4137, 0x434d, 0x4206, 0x2331, 0x434b, 0x41d2, 0x412d, 0x3037, 0x423c, 0x4582, 0xbaed, 0x41f8, 0x4113, 0x4584, 0xbf2a, 0xb234, 0x1de1, 0xb2cd, 0x4099, 0xa0ca, 0x4167, 0xbf00, 0x42e3, 0x2ee1, 0x1d91, 0x0fa0, 0x400d, 0x3248, 0xbfde, 0x25d6, 0x4332, 0x0ca8, 0x18fd, 0xb25f, 0xba37, 0x4121, 0x4402, 0x42c8, 0xbfc0, 0x3ff9, 0x41c3, 0xb2a8, 0x184f, 0x37a4, 0x439d, 0x17f7, 0x4273, 0x0b72, 0x428d, 0x4293, 0xad65, 0xbf60, 0x1be0, 0xba20, 0x43e8, 0x4033, 0x219f, 0xbf27, 0x425b, 0x1cab, 0x42c0, 0xa753, 0x411b, 0x42ad, 0x14cd, 0xba71, 0x4221, 0xba79, 0xbfbf, 0x423f, 0x437b, 0x418d, 0x43c5, 0x4062, 0x0c46, 0x3c2f, 0x442a, 0xbf26, 0x1a90, 0xb2b7, 0x07b4, 0xbf52, 0x04df, 0x42e1, 0x1d76, 0x418b, 0x333f, 0x1f08, 0x4007, 0x1246, 0xb259, 0x4084, 0x4223, 0xb257, 0xb26b, 0x2ba1, 0xb232, 0x11ce, 0xb25e, 0xb213, 0xbf5e, 0xb26f, 0x1bdd, 0x426c, 0x4375, 0x18e4, 0x403c, 0x42b8, 0xb02b, 0x4197, 0x1861, 0x2f7a, 0xbfb8, 0xa15b, 0x43d1, 0x3df1, 0xb2e7, 0x43dd, 0x425d, 0xba27, 0x4281, 0xbaf7, 0x4644, 0xbf4d, 0xa7c4, 0x4208, 0x411c, 0x43f3, 0xbacf, 0xb04f, 0x4091, 0x41d0, 0x2fd4, 0x4332, 0xb205, 0x45ee, 0xb2b7, 0xb0a1, 0xb235, 0x214e, 0x4232, 0x41e1, 0xbada, 0x2e04, 0xb2ba, 0x40f8, 0x3e46, 0xbf05, 0x3862, 0x0057, 0x426a, 0x401e, 0xb014, 0xba19, 0x0d65, 0xba7d, 0x41cc, 0x34c8, 0x4201, 0x4202, 0x43a6, 0xb004, 0xbf2c, 0xa057, 0x0d9f, 0x431a, 0xba38, 0x40b7, 0xb277, 0x4275, 0x27a1, 0x4222, 0xb098, 0xb02e, 0xb2de, 0x40e5, 0xbf27, 0x42a3, 0xad53, 0x4627, 0xbfe0, 0xba06, 0x4098, 0xb24c, 0x40e0, 0x0ec3, 0x411a, 0x403f, 0xb256, 0x42eb, 0xba5f, 0xbfb6, 0x42ac, 0x41cd, 0xb2ec, 0xbf85, 0x0e6f, 0x164c, 0xb207, 0x434c, 0xbfe0, 0x40f4, 0xbaed, 0x4056, 0xb20a, 0x1834, 0x4659, 0x1d2e, 0x0587, 0xbfb6, 0xaaf8, 0x45c6, 0x401b, 0xba6a, 0xba64, 0x40b0, 0xba31, 0x0731, 0x41ae, 0x427d, 0x4021, 0x272c, 0xba2e, 0x409f, 0xbae6, 0x4048, 0x4065, 0xbf06, 0x40e6, 0x430c, 0xb20a, 0xba68, 0xba40, 0xbf95, 0x4171, 0x43f2, 0xb25a, 0x43cb, 0x42d0, 0x46db, 0xa3e7, 0x43e8, 0x402c, 0x053e, 0x422a, 0x4261, 0x429e, 0x417d, 0xb234, 0x41f5, 0x2f6f, 0x44d2, 0xbf6a, 0x1fff, 0x1a5b, 0xba78, 0x29e0, 0x43ad, 0x4200, 0xb2dd, 0x0096, 0x3a21, 0xb247, 0x382e, 0xba2e, 0x4336, 0x4021, 0x4221, 0xbf59, 0x43cd, 0x1810, 0xa198, 0x2e75, 0xba5d, 0x42dd, 0x42a6, 0x422b, 0xbf07, 0xa0f8, 0x4206, 0x41d6, 0x3238, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3aa91aa4, 0x4c5784a8, 0x59e0657d, 0x51ef61ac, 0x760b00f1, 0x0da5a30b, 0x69a772e3, 0x96f320f6, 0x01c12c2c, 0xe5449e38, 0x74e0fd57, 0xcf8b32f0, 0x484f3e23, 0x911d3cda, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0xffffffb0, 0x00000000, 0x00000017, 0x00001b1c, 0x00000000, 0x00001c1b, 0x38000000, 0xffffffff, 0x0382586e, 0x17da7113, 0xe9c1faae, 0xcf8b32f0, 0x01c12c37, 0xfffffbd4, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb251, 0x317d, 0xb2ae, 0x2093, 0xb04a, 0xb0b8, 0x1e16, 0x405a, 0xbfb8, 0xb0cf, 0x4099, 0x431e, 0x41e5, 0xbf70, 0xb0e1, 0xa18f, 0x1f25, 0xba65, 0x1d38, 0x1119, 0xba41, 0xb23a, 0x03ae, 0x01a6, 0x1dfc, 0x1e5c, 0x4181, 0xba71, 0x401c, 0x3c40, 0x4154, 0x1895, 0xbfd0, 0xbf7c, 0x1af1, 0x429c, 0x1148, 0x41b5, 0x432d, 0x43f4, 0x461b, 0x41d6, 0x1ef7, 0x1d2d, 0x41c8, 0x3021, 0xb25c, 0x431b, 0xa59d, 0x4355, 0xbae7, 0xbff0, 0xb2fa, 0x39aa, 0x0c26, 0x4144, 0x0954, 0x419b, 0x03a0, 0xbfd1, 0x45e9, 0x439a, 0x4127, 0xb23d, 0x4132, 0x1ae1, 0x4671, 0x4613, 0x45e2, 0xbac3, 0x421c, 0x1379, 0x1ea9, 0x16a8, 0x406d, 0x3d2b, 0x4267, 0xb205, 0x4042, 0x4232, 0x45e4, 0xb2ba, 0x1a6c, 0xb2a8, 0xb0fe, 0x2dc3, 0xb0d7, 0xbf4a, 0x407c, 0x40f2, 0xb21b, 0x0c04, 0x2884, 0x1b10, 0x42fb, 0x41b3, 0x0235, 0x2120, 0xb062, 0x438b, 0x444c, 0x43d0, 0x41d9, 0x1dc3, 0xb02b, 0x1b8e, 0x4231, 0xba4a, 0x4372, 0x42d1, 0xb246, 0x1cfa, 0x2420, 0x41ec, 0x41a2, 0x43cf, 0xbf6c, 0x0d46, 0xba57, 0x4266, 0xaca4, 0xb220, 0x33fb, 0xb2da, 0x40ac, 0x4207, 0xb2ca, 0x40fa, 0xb2ce, 0x2f83, 0x426d, 0x1f13, 0x42f0, 0x1902, 0x4374, 0x1fff, 0xa5f3, 0x40e8, 0x42ed, 0x43c6, 0x40de, 0x4244, 0x0f5d, 0x40f3, 0xbfb6, 0x36a3, 0x418e, 0xb0f3, 0x46c5, 0xbac8, 0x1cc6, 0x1495, 0x4121, 0xb286, 0x43d1, 0x431b, 0xad3e, 0xb240, 0xbaed, 0xba30, 0x41af, 0x400c, 0xb216, 0x08b2, 0x4426, 0x4321, 0xb28e, 0x439c, 0xbfa4, 0x2883, 0xba2e, 0x094c, 0x4382, 0x4118, 0x19a5, 0x05b0, 0x4000, 0x4064, 0x19b5, 0x435f, 0xb2e0, 0x4399, 0x1ec1, 0xb09a, 0x28c9, 0x0895, 0xba6b, 0x41c7, 0x4481, 0x42f6, 0x1dc4, 0xbfb8, 0xb249, 0xba32, 0xa293, 0xba7e, 0x40c7, 0x4369, 0x4211, 0x4198, 0x4294, 0x4256, 0x1be2, 0x1f8a, 0xb20f, 0x40c6, 0x409c, 0xbfd4, 0x4197, 0x42ab, 0x4149, 0x4284, 0x4610, 0xb2fd, 0x41f9, 0xb260, 0x427d, 0x18bf, 0x45f4, 0xa24d, 0xba4e, 0xba66, 0xb2dd, 0xbaed, 0xbf25, 0x1a95, 0x1d6a, 0xb2c6, 0x4005, 0x0738, 0x1930, 0xba7b, 0xb289, 0x14bb, 0x13be, 0x4368, 0x4005, 0xba63, 0x18f7, 0x078c, 0x464a, 0xbf84, 0x0f5a, 0x1e55, 0x4262, 0x26de, 0x4132, 0x421e, 0xb04b, 0x0632, 0xb2a5, 0x43a4, 0xb03d, 0x18a6, 0x41f7, 0xb2dc, 0xb226, 0xbf34, 0x4427, 0x4019, 0x4448, 0xbaeb, 0x43de, 0x42d8, 0x4345, 0x407f, 0x4691, 0x4245, 0x4200, 0x1c1d, 0x4130, 0xa717, 0x417e, 0x42eb, 0x3c1a, 0xa401, 0xbf24, 0x0fd2, 0x424e, 0xb26c, 0x4315, 0x4064, 0xb025, 0x4431, 0xb2d9, 0x0876, 0xb2df, 0x0bb1, 0xbf48, 0x259e, 0x3f2b, 0x38c9, 0x44fb, 0x4216, 0xb0ab, 0x4138, 0x4203, 0xbfab, 0xa3bb, 0xba73, 0x3abe, 0x4136, 0x4212, 0x129e, 0xbfb0, 0xbf67, 0xb2f7, 0xbafe, 0x4066, 0x4329, 0xb0dc, 0xb050, 0x42d8, 0xb05c, 0x421d, 0x43c2, 0x42c6, 0x2ed9, 0x4088, 0xbf7e, 0x43c3, 0xba22, 0x40b5, 0x18ed, 0x4064, 0xbfd1, 0xaf28, 0xb0d5, 0x03d5, 0xb2d3, 0x2eb3, 0x10a5, 0x4298, 0xba28, 0x1329, 0x3160, 0xb01f, 0xaea8, 0x41fa, 0xb24a, 0x44c4, 0x05f4, 0x1eaf, 0xba36, 0xbf60, 0x1f18, 0x2c2c, 0x408b, 0x3269, 0x436e, 0xb2c0, 0x1392, 0xbfa1, 0x4220, 0x41f7, 0xbacc, 0xbad9, 0x0d2a, 0xb0bb, 0x4175, 0xb20e, 0x4007, 0xb225, 0x4152, 0xbf62, 0xb0f3, 0x4127, 0x1e7d, 0x4615, 0x375f, 0x4545, 0x4064, 0x40ed, 0x4015, 0x4082, 0x404a, 0xb211, 0x4073, 0x41f6, 0x42ae, 0x409d, 0x4348, 0x068b, 0x1364, 0x42af, 0x45c6, 0x1fdc, 0x25a7, 0xb24c, 0xa940, 0x1cc5, 0x40d4, 0x4006, 0xbfca, 0x4352, 0xb288, 0x2129, 0x438b, 0xb2f9, 0xa900, 0x40a9, 0xbf03, 0x456d, 0x2efc, 0xbad4, 0x1f74, 0x4204, 0x421f, 0xba1f, 0x42a2, 0x404d, 0x2d74, 0x406b, 0x2fb9, 0x4334, 0xbfd6, 0x40c5, 0x4105, 0x42a6, 0xbf00, 0x4024, 0x406a, 0x43c7, 0x4456, 0xb26d, 0x4034, 0x4230, 0x0f59, 0x4065, 0x4037, 0x0ee0, 0xadf8, 0xbf7c, 0xb2e3, 0x43b7, 0x429c, 0x40e0, 0x4338, 0xba25, 0x424e, 0x005a, 0x400e, 0x407e, 0x43e8, 0xbf0d, 0x4270, 0x1dd0, 0x4021, 0x1eb2, 0x4381, 0xae1b, 0x4122, 0x0dc4, 0xba5e, 0xbf79, 0x46da, 0x1bb1, 0x1117, 0xb03a, 0xa9c5, 0xb0de, 0x1e18, 0x42c2, 0x4043, 0x42f9, 0x4151, 0x013b, 0xbf00, 0xbf76, 0xb2c1, 0x4263, 0x40fd, 0xb02f, 0x41e1, 0x4484, 0xbfd8, 0x27ae, 0xba27, 0x430f, 0x283f, 0x4331, 0xb254, 0x4183, 0x1f9f, 0x1a56, 0xb264, 0xbf70, 0x2af2, 0x4390, 0x032a, 0x4383, 0x31cd, 0x18b8, 0x32ac, 0x41de, 0x43b5, 0xb2a0, 0x434f, 0xbf68, 0x466f, 0x3cd0, 0x4398, 0xb2a7, 0x4225, 0xb204, 0x4445, 0x4446, 0xba17, 0x4102, 0x11e8, 0x1f7d, 0xbf29, 0x438c, 0xb242, 0xb23c, 0x4016, 0x41ee, 0xbf15, 0x0f9b, 0x2845, 0x431f, 0x4012, 0xa905, 0x43bd, 0x40a7, 0xb21c, 0xbfa0, 0xb218, 0x41fb, 0x21c7, 0x1e5c, 0x2eba, 0x26a0, 0xba1e, 0x42a2, 0xbf23, 0x1c9b, 0x3ddb, 0x1ade, 0x421e, 0xb2ae, 0xa855, 0xbacc, 0x4067, 0x4670, 0xb299, 0x4633, 0x4009, 0xbfe2, 0x4005, 0x37ee, 0xbfc0, 0x0bf7, 0x439b, 0x4099, 0x425a, 0x1ebc, 0x445b, 0x4373, 0xba1c, 0x41b7, 0xb2d8, 0xb2b6, 0xbf34, 0x4273, 0x4121, 0x435f, 0x4158, 0x4021, 0x4231, 0x0c75, 0x4369, 0x3a94, 0xb247, 0x43c7, 0x4556, 0x4186, 0xb22d, 0xb25c, 0x42f0, 0x31d2, 0xbf46, 0x4062, 0x2ee4, 0x0296, 0x41f3, 0x4173, 0x4295, 0xb2db, 0x1d70, 0xbf84, 0x29a9, 0x2a9d, 0x41a0, 0x400d, 0xbf80, 0xb2a8, 0x1d70, 0x430d, 0xb22b, 0x4299, 0x0a23, 0x402a, 0xb275, 0xbf8c, 0x4405, 0x41bc, 0x43dd, 0xbf80, 0xb258, 0x4231, 0x1842, 0xa26e, 0x1e27, 0xb2ad, 0xb064, 0x4057, 0x46a1, 0x43e5, 0x439b, 0x4011, 0x41ce, 0x194e, 0xba1f, 0xb004, 0xb26b, 0x43ff, 0xb253, 0x4287, 0x402a, 0x40b4, 0x3a6d, 0xbfd4, 0xba49, 0x4172, 0x4043, 0x1e30, 0xb24c, 0xbad8, 0x4061, 0x418f, 0x1940, 0x4180, 0xbf7d, 0x23b5, 0x42a2, 0x40f9, 0x1e06, 0xbf99, 0xba15, 0x4164, 0xbac4, 0xb246, 0xa06a, 0x1923, 0x4553, 0xba1e, 0xbfa0, 0xab04, 0x4290, 0x4298, 0x3dc9, 0x4031, 0x40f5, 0x43aa, 0x329f, 0x45ae, 0xb209, 0xb2cf, 0xbf9d, 0x4123, 0x400e, 0xbae6, 0xb0c4, 0x05a3, 0x4353, 0x4057, 0xb060, 0x424c, 0x4189, 0xb2ec, 0x401f, 0xb2be, 0x1e8c, 0x4656, 0xb05d, 0x0fb9, 0x4069, 0x42b0, 0xb21f, 0xb277, 0x43f0, 0x4071, 0x419c, 0x22b1, 0xb02f, 0xa8c1, 0xb2ae, 0xbf84, 0x2d2c, 0x1df6, 0x4617, 0xae87, 0xb20f, 0x06bb, 0x420e, 0x0ba4, 0x4440, 0x422d, 0x4207, 0xb0ac, 0x431e, 0x1829, 0xb23b, 0xb23a, 0x41ed, 0x4016, 0xb023, 0x19db, 0x4351, 0xb2b8, 0x4151, 0xbf73, 0x4607, 0xaa70, 0x465a, 0x4242, 0xba3e, 0xb27f, 0x43ce, 0x0c36, 0xa72f, 0xb29d, 0x415d, 0xbfb3, 0xb0a3, 0xb2da, 0x429d, 0xb2df, 0x408c, 0x445a, 0x43fc, 0x4022, 0x4262, 0x1782, 0x4424, 0x40ef, 0x39b4, 0x1cf4, 0xb22d, 0x41a7, 0xba1e, 0x40be, 0x40ac, 0x36a1, 0xbae3, 0xbfb1, 0xac8c, 0xb23c, 0x40ca, 0xb203, 0x1d66, 0x117b, 0xb2a0, 0xba12, 0x401b, 0xaafd, 0xbaed, 0x1f75, 0x1c02, 0x42b4, 0x1f7f, 0x1d3c, 0xbad4, 0xbf24, 0x020b, 0x4226, 0x425f, 0xb27e, 0x3ed4, 0xba40, 0x2724, 0x4006, 0x107b, 0x2d4b, 0x464d, 0xb215, 0x41b4, 0x4172, 0x42dd, 0xa957, 0xba2e, 0x3f84, 0x1fdb, 0x290d, 0x43e3, 0xbf99, 0xb231, 0x422c, 0x02c8, 0xbacb, 0x432c, 0x1b38, 0xa296, 0x424f, 0x4640, 0x4175, 0xbf5b, 0x436b, 0x428b, 0x40a1, 0x4629, 0x40cd, 0xbfa4, 0x097e, 0x42e0, 0x4566, 0x4063, 0x3ef3, 0x464e, 0xb25c, 0x04b0, 0x4272, 0x41b1, 0x422d, 0xb070, 0x4254, 0xb034, 0x40b5, 0xb2a4, 0x41f0, 0x3546, 0xbf36, 0x40be, 0xba51, 0x437e, 0xba32, 0xbf60, 0x04b2, 0x16eb, 0x4092, 0x444f, 0x4634, 0x405a, 0x420b, 0x4342, 0x46b0, 0x1e8d, 0x4206, 0xba4c, 0xba29, 0x45d9, 0x1cd4, 0xbf8f, 0x42e7, 0xb0c7, 0x047c, 0x1f41, 0x4132, 0x4125, 0x23aa, 0x26ae, 0x294f, 0xae1f, 0x408d, 0x1aaa, 0xb052, 0xb2d6, 0xb2bf, 0x437e, 0xaf3c, 0x42ab, 0x2fe4, 0xb0d7, 0xbfc0, 0x38cb, 0x428e, 0xb2c3, 0xb081, 0x429f, 0xb220, 0xbf03, 0x4018, 0x1e76, 0x4167, 0x14fc, 0x42a0, 0x4642, 0x4543, 0xba6d, 0x1e81, 0x42fa, 0xbaeb, 0xbae0, 0xbac6, 0x0364, 0x0be5, 0xb2f1, 0x456b, 0xb225, 0xb2ba, 0x426a, 0x4359, 0x40db, 0xbf05, 0x4255, 0x147a, 0x4072, 0x0e8c, 0x0c55, 0x4178, 0xb28d, 0x4272, 0xba3b, 0x38c8, 0x41af, 0x46b8, 0x4075, 0x435c, 0x1b06, 0xbaeb, 0x3f2b, 0x0271, 0x310e, 0x424e, 0x013d, 0x4038, 0x2416, 0x35e0, 0x428d, 0xb0b8, 0x424a, 0x4380, 0x43e2, 0xbf3e, 0x4115, 0xb079, 0x4134, 0x3588, 0x20b7, 0xaa55, 0x43d3, 0x1ff2, 0x1ce9, 0x420e, 0x1ba2, 0x4007, 0x4327, 0xb0f2, 0xb082, 0xbfd4, 0x4390, 0xb233, 0xb2b2, 0xaa54, 0xb2d9, 0x4142, 0x4065, 0x41a9, 0x43fc, 0x42c6, 0x0daa, 0xb29e, 0xa5ae, 0xb288, 0x4241, 0xb237, 0x4303, 0x189b, 0xbf1e, 0x0a78, 0xbace, 0x0a87, 0x4154, 0x31e7, 0xbf3b, 0xba28, 0x419c, 0xba17, 0x3f48, 0x3ef8, 0x42f1, 0x4104, 0x4605, 0x413c, 0x4558, 0xb22a, 0xb287, 0xba47, 0x33a1, 0x430d, 0x42aa, 0xbf32, 0xb270, 0x42b3, 0x4392, 0x0bbe, 0x43bf, 0x41fc, 0x19ee, 0x2639, 0x42ca, 0x4219, 0xb2ec, 0x4600, 0x436c, 0xb011, 0xbaea, 0xb218, 0x412a, 0x41d8, 0x2cb9, 0x41b7, 0x4001, 0x0d3f, 0x1bc5, 0xbf03, 0x42d0, 0x4091, 0x40bd, 0x1ecd, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xaf3cc1f1, 0x05fac305, 0x3db327e1, 0x36ea8c1c, 0x589f9f57, 0xa4ce47b1, 0x8ee4ab13, 0xd31f4de3, 0xa8944d30, 0xd6eab546, 0xbc8af831, 0x69b223cd, 0x24711ddc, 0x33c795e8, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0x00059360, 0x00000060, 0x00000000, 0x00002c9b, 0x00003d09, 0x0000005d, 0x00000039, 0x00000fff, 0xa89457bc, 0xffffffdb, 0x69b23619, 0x69b23619, 0xcd056b3d, 0xa89454e4, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x2427, 0x3f90, 0x2091, 0x2633, 0x40e5, 0x429e, 0xb272, 0x4572, 0xbacb, 0x358e, 0xb054, 0x426b, 0xb2ad, 0x17b5, 0xbae7, 0x403b, 0xb2af, 0x199f, 0xa0e1, 0x148a, 0xb068, 0x40ac, 0x1c34, 0x41cd, 0x4387, 0xbfa8, 0x42c7, 0xa9ac, 0x2d10, 0xbae3, 0x419c, 0xba60, 0x004e, 0xb0c7, 0x18c0, 0x42fd, 0x4021, 0x4592, 0x4365, 0x4591, 0xba50, 0x4440, 0xa24c, 0x438b, 0x1aea, 0x404f, 0x4171, 0xbf8f, 0x27e3, 0x4170, 0xb024, 0x4156, 0xb2c5, 0x4361, 0x2b01, 0x42af, 0x43a7, 0x1c67, 0xb088, 0xaf29, 0xba76, 0x45a6, 0x1daf, 0x06d5, 0x40b0, 0x40d8, 0xba5c, 0x1cb8, 0x185a, 0xa518, 0x43ce, 0xbf76, 0x4378, 0x4216, 0x43cc, 0x3321, 0x40af, 0xbae8, 0x41a2, 0x40ac, 0x4602, 0xbf00, 0x1919, 0x40d8, 0x1df2, 0x1eac, 0xbf07, 0x18df, 0x41ea, 0xba54, 0x4211, 0x2835, 0xb230, 0x2c47, 0xb2aa, 0x4118, 0x4013, 0x4327, 0x3e32, 0xbfd0, 0x4351, 0x41ba, 0x40c9, 0x4174, 0x181a, 0x423a, 0xb2ad, 0x424d, 0x4296, 0x4232, 0xbaf5, 0xba6d, 0xb0ef, 0xbf48, 0xbafe, 0x428b, 0xba5f, 0x3c65, 0xba32, 0x0379, 0x3701, 0x40bb, 0xbae9, 0x4164, 0x42f8, 0x435a, 0x042a, 0xb2d2, 0x4159, 0x425d, 0x46cb, 0x1db3, 0x4463, 0x4299, 0xbf0c, 0x37c3, 0x0331, 0x19be, 0x406b, 0x41ad, 0xb22c, 0x43bc, 0x4195, 0x4351, 0x4227, 0xb097, 0x42b8, 0xb2a9, 0x408e, 0xb0c5, 0x411b, 0xbfcd, 0xb200, 0x4113, 0x40d2, 0x35f4, 0x1bb3, 0xbf97, 0x1d53, 0x41ad, 0x1c7d, 0xb283, 0x43c9, 0x419b, 0x4263, 0xb0a4, 0x1f68, 0xa4bb, 0x4005, 0xbac2, 0x3f7c, 0x2f53, 0x4192, 0xb05d, 0x03fb, 0x4619, 0xba2e, 0xb2ad, 0xb038, 0x0315, 0xbf1c, 0x43ba, 0xb0da, 0xbfb0, 0x1836, 0xb08d, 0xb066, 0xb0a2, 0x4132, 0x1883, 0x4688, 0xac7b, 0x35bc, 0x4016, 0x40b4, 0x40fe, 0x09dd, 0xb2b4, 0x41b9, 0x4331, 0x46d8, 0xbfd4, 0x1b67, 0x427f, 0x0d12, 0x4141, 0x3ca3, 0x1bf4, 0x406f, 0xbf04, 0x425f, 0xa73c, 0x1fd5, 0x441b, 0xb04a, 0x0fa5, 0xbf23, 0x4407, 0xbfb0, 0x24d0, 0xb073, 0x4283, 0xbf16, 0x061a, 0x41d7, 0x2266, 0x41ef, 0xb2ef, 0x4324, 0x40ba, 0x0c29, 0xb08c, 0x4160, 0x43e0, 0x439a, 0x3f33, 0xb03e, 0xb264, 0x4275, 0x45f2, 0x070e, 0x41b4, 0x43d4, 0x4127, 0x2d76, 0x0c90, 0x4207, 0xbf97, 0x4250, 0x4221, 0x4017, 0xbac8, 0xaf4c, 0xbac6, 0x1d9f, 0x4148, 0xa3e9, 0xa244, 0x4013, 0x1ffe, 0xbf6a, 0x4031, 0x41d6, 0xaa58, 0x40cc, 0x328a, 0x437e, 0xb06f, 0xbaeb, 0xbf15, 0x4106, 0x06a9, 0x41fd, 0x0967, 0x3300, 0x22da, 0x41c8, 0x45d4, 0x4030, 0xba6e, 0xb042, 0x1fc5, 0xba63, 0xa4e4, 0x2c03, 0xbfa0, 0x3858, 0xb2c5, 0xb0d9, 0x4067, 0x42c8, 0x4225, 0x0778, 0xbf07, 0x42a6, 0xb273, 0x4048, 0x42a1, 0x1125, 0x41ea, 0x051b, 0x19e4, 0x42c7, 0x09cd, 0x1a48, 0x0be5, 0xb271, 0xbf9e, 0x4167, 0x4471, 0xb2ed, 0x4058, 0x423d, 0x219d, 0x43bc, 0x1bbb, 0x22c4, 0x1e06, 0x4153, 0xb267, 0x2bde, 0x1d6b, 0x40db, 0x41cf, 0x437e, 0xba1a, 0x166c, 0xb068, 0xb23a, 0xbf77, 0xb238, 0x1fde, 0x4335, 0x4640, 0x409e, 0x42b4, 0x434f, 0xbad5, 0x2036, 0x4314, 0xb2e2, 0xb2fa, 0xb020, 0x41f7, 0xba26, 0xb002, 0x1200, 0x4326, 0x32db, 0x156e, 0xb2e7, 0x424f, 0x4079, 0xbfab, 0x3d18, 0x447a, 0x10c9, 0x454b, 0xa020, 0xbf70, 0xbfa9, 0x4336, 0x3cc4, 0x2dc9, 0x423d, 0x46ab, 0x4150, 0x1847, 0x40ec, 0x0a6d, 0xbf19, 0x36e4, 0x436a, 0x436c, 0x1efc, 0x4200, 0xbfc5, 0xa281, 0xbac7, 0xb0e5, 0x41bf, 0x4119, 0x13e8, 0x4142, 0x43e0, 0x40bd, 0x0769, 0x41d6, 0xa91c, 0xb2db, 0x4078, 0x41e3, 0x40e5, 0x1b36, 0xb247, 0x42cb, 0xbf75, 0x4175, 0x4258, 0x4276, 0x4572, 0x4237, 0x1eac, 0xb2ca, 0x4171, 0x40b8, 0x42d1, 0x4293, 0x41bd, 0xbfe0, 0x34d8, 0x4024, 0x4128, 0xa7bb, 0x43c5, 0x4342, 0x42a0, 0x0306, 0x42cb, 0x4082, 0xbf70, 0x0fa2, 0x43ef, 0x45d6, 0x4300, 0xbf94, 0x077c, 0xbaf3, 0xa980, 0x33d4, 0x3b98, 0x18ab, 0x435f, 0x4310, 0xb0a1, 0x40e1, 0x149a, 0xaab9, 0x422a, 0xb2df, 0xbf80, 0x4592, 0x086f, 0x337d, 0x1dd5, 0x09b8, 0x197f, 0x45a0, 0x422d, 0xb084, 0xbf19, 0xbaf5, 0x04e9, 0xba29, 0x194a, 0x1a57, 0x32bd, 0xa80e, 0xb2c8, 0x3542, 0x2086, 0x422c, 0xba36, 0x46ab, 0x14bc, 0x20b4, 0x402f, 0x40ff, 0x46e1, 0x40a0, 0x43fc, 0x41ce, 0xbf4f, 0xb26f, 0x415b, 0x1523, 0x061f, 0xb090, 0x06ec, 0xb2eb, 0xb292, 0x426f, 0x42db, 0x167a, 0x43a3, 0x424f, 0xb247, 0x1b40, 0x1e25, 0x4087, 0xbad4, 0x431f, 0x435d, 0x006f, 0xbfb1, 0x40b3, 0x39d4, 0x4291, 0x165f, 0x42b8, 0x4200, 0x42aa, 0x43f5, 0x154e, 0xa5de, 0xba50, 0x3da7, 0x0d4e, 0xa9bf, 0xba1c, 0x402b, 0x418c, 0x4360, 0xb056, 0x3ef6, 0xbfcd, 0x4038, 0x1a87, 0x2473, 0x4317, 0x4285, 0x40c7, 0x40f6, 0xbf8b, 0x4077, 0x431a, 0x05ad, 0xba42, 0x4040, 0x16d8, 0x1913, 0xb265, 0x4436, 0x4102, 0x40fd, 0xa3f4, 0xba27, 0x4109, 0x40b2, 0xb035, 0x437e, 0xbf84, 0xba37, 0x189e, 0x0bee, 0xba76, 0xb2d0, 0xb204, 0xa150, 0x41a7, 0x41bb, 0xb295, 0x33f3, 0xbf80, 0x2f8a, 0x459b, 0xabe4, 0x11a0, 0x426b, 0xba78, 0xb272, 0x3718, 0x43c8, 0xbfb7, 0xb224, 0x1ec8, 0x06d7, 0x4327, 0x273c, 0xb2fe, 0x4294, 0x179f, 0x42a5, 0x4131, 0x43f0, 0xb25d, 0x4298, 0x42b2, 0x40b1, 0xbff0, 0x0bbe, 0xbf22, 0x2944, 0x4147, 0x0969, 0x433c, 0x1b91, 0x419b, 0xa0b4, 0x429a, 0xaa98, 0xa7ae, 0xbfad, 0x430e, 0xb2b8, 0x40dc, 0x403f, 0x19b8, 0x41f4, 0x1da0, 0x406c, 0x40aa, 0x1e72, 0xbfc0, 0x41e6, 0x43a2, 0x40d6, 0x1a1a, 0x191d, 0x1b1b, 0x2f8f, 0xb0d0, 0x1ce2, 0x4375, 0xb04b, 0xba0c, 0x4146, 0x4305, 0x40b4, 0xbf1b, 0x42d1, 0xb2ae, 0x4347, 0xb22e, 0x42ad, 0x41df, 0x1d6d, 0xb2b8, 0x409b, 0x38ce, 0xba78, 0x2add, 0x4377, 0x3fea, 0x42d5, 0xb2a0, 0xac6f, 0x3f00, 0x42e0, 0x42e4, 0x121d, 0x411a, 0xba59, 0x43b2, 0xb069, 0x4310, 0xb297, 0xbf88, 0x41ac, 0xb033, 0x1884, 0xb030, 0x42fa, 0x2a9b, 0x419a, 0xb270, 0x014f, 0x4100, 0x414a, 0x42b2, 0xb264, 0x43a7, 0x2e7d, 0x417d, 0x30eb, 0x4367, 0x429d, 0x3747, 0x3eda, 0x3129, 0xbfb4, 0x1f57, 0x0b56, 0x4039, 0xb26f, 0x462f, 0xb26f, 0x4181, 0x445f, 0x46c3, 0xb033, 0xbfb0, 0xb201, 0x422b, 0x4017, 0x431d, 0x4274, 0x40b4, 0x4481, 0x44a2, 0xaeb3, 0x42ad, 0x41ec, 0x40f8, 0xb2a7, 0x294e, 0xbf64, 0x41b6, 0xbac6, 0x423e, 0xb0b6, 0xba73, 0x45cc, 0x1d55, 0xb2f8, 0x3cee, 0x187b, 0x0e9e, 0x41b7, 0x42ce, 0xa773, 0xb2d4, 0x18dc, 0x207b, 0x4050, 0x4198, 0x3ee7, 0x40ea, 0xb243, 0x18a8, 0xbfe2, 0xba17, 0xbf00, 0x2883, 0xb06f, 0x40ce, 0x42fc, 0xba28, 0x4201, 0x1c4e, 0xbf80, 0xbf19, 0xae7d, 0x406a, 0x0bf7, 0xb0a2, 0xbf70, 0x4683, 0x4319, 0xbad2, 0xa731, 0x43ae, 0xba75, 0x426d, 0x2679, 0x2eeb, 0x149d, 0x45c5, 0x2848, 0xb229, 0xada6, 0x3a2c, 0x4002, 0x4098, 0xbf58, 0x4004, 0x43dc, 0x4585, 0x28bf, 0x4291, 0x4025, 0xba3d, 0xb242, 0x310b, 0xba76, 0xb2da, 0x4275, 0x41a4, 0x44b2, 0xb283, 0xb251, 0x3af7, 0x4188, 0x424d, 0xbf47, 0x38f5, 0x4602, 0x402d, 0x15f2, 0xb0e8, 0x4364, 0x2c36, 0x14be, 0x40a0, 0xbf4b, 0x40c8, 0x4150, 0x3428, 0x3375, 0xa475, 0x43c8, 0x4129, 0xbf00, 0x40de, 0x415d, 0x4235, 0x43fb, 0xb0f8, 0x4120, 0xbad0, 0x43b6, 0x4220, 0xb0d3, 0xba72, 0x0b42, 0xbf00, 0x421d, 0xb2d2, 0x0ccc, 0xb2ee, 0x2fb3, 0xaa42, 0xb0b1, 0xbf54, 0x4182, 0x40ce, 0x1fc6, 0xb291, 0x133c, 0x40ac, 0x405f, 0xbfca, 0xac9d, 0x4024, 0x0355, 0x43d6, 0x46b4, 0xb254, 0x1e21, 0x1824, 0x41f8, 0xbf56, 0xb0f4, 0x4396, 0x3777, 0xb01f, 0x42a3, 0xb2c6, 0x41ef, 0x0db1, 0xb0fb, 0xba3f, 0xa7bf, 0x4217, 0x0bf8, 0x4327, 0x4368, 0xbfc0, 0x42e1, 0xb0fe, 0xba62, 0x427f, 0xbf80, 0x3777, 0xb2f7, 0xbf16, 0x42a0, 0xaa78, 0x42d0, 0x41d5, 0xa44e, 0xbadd, 0xb0be, 0xb2e0, 0xba54, 0x4121, 0xbfd0, 0x2647, 0x4574, 0xba1c, 0x20a2, 0x1ae6, 0x2e30, 0x43e5, 0xb27d, 0xb0ee, 0x4053, 0xb2b6, 0x1c72, 0xbf70, 0x1562, 0xbfa9, 0x1f12, 0x4198, 0x4297, 0xb09b, 0x440e, 0x1eb9, 0x2ec0, 0xbfa4, 0x1c9d, 0xba34, 0x428e, 0x22cb, 0x4313, 0xb29f, 0x418d, 0xb27b, 0x4664, 0x1daa, 0xba05, 0xb230, 0xbf07, 0x314a, 0x42c9, 0x03a2, 0x4198, 0x1c5f, 0xb0aa, 0xb078, 0x404d, 0xb2b7, 0x4204, 0x4378, 0xbf09, 0xbfe0, 0x4307, 0x41fa, 0xba3f, 0xba74, 0xb042, 0x0f95, 0xb251, 0x44d8, 0x0fa5, 0x42a6, 0x0927, 0x1e0f, 0xbfd9, 0xb0ac, 0xa99d, 0x3a4b, 0x3435, 0x4154, 0x4046, 0xb231, 0xb201, 0x19c1, 0x418b, 0x0d1a, 0x4000, 0xb032, 0xbfc1, 0xb257, 0x2c04, 0xb029, 0xba42, 0x42ab, 0x2783, 0x1c91, 0x2f6d, 0x40cb, 0x1ac3, 0x4093, 0x45da, 0x434d, 0x4429, 0xbf55, 0x419c, 0x41c1, 0x43f2, 0x3a64, 0x4041, 0x183f, 0x4103, 0x4057, 0x41ec, 0x4268, 0x20b0, 0xb294, 0xba16, 0xb238, 0xbf0a, 0x4622, 0x432e, 0x3ed4, 0x46a2, 0xb239, 0xb2ca, 0xb2bb, 0xbfa3, 0x41fb, 0x400b, 0x42f9, 0x4296, 0xb215, 0x11b5, 0xb292, 0xbfcb, 0xab70, 0x4088, 0x18fc, 0x40fa, 0x31cf, 0x44c0, 0x1a3e, 0x20a8, 0x36f0, 0x421a, 0x4547, 0x4101, 0xb259, 0x4094, 0xa222, 0x1d0f, 0xbfc0, 0x0582, 0xb080, 0xb214, 0x4605, 0x407d, 0x1a4a, 0x400d, 0x4168, 0xb091, 0x1ef9, 0x424b, 0xba65, 0xbfc3, 0x4033, 0x4258, 0xb043, 0x4095, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x9f18068a, 0x03643f9c, 0x4d4c0930, 0xb5ba15e8, 0xff9da78d, 0x8731d796, 0x094b9e5e, 0x291d98f7, 0x1392fdc3, 0x4fe11ddf, 0x2f0cc7b5, 0x2d5bda6f, 0x2d85b32d, 0x081e2031, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0xffffffa0, 0xffffff96, 0x00000000, 0x00000060, 0x00000000, 0x00000000, 0x000000f0, 0xffffff99, 0x2fc23bbc, 0x2d85b418, 0x0000d42f, 0xc7ffffff, 0xf7e1d472, 0x081e259d, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb235, 0x3f52, 0xb2ed, 0xa9b5, 0xbf53, 0x2bef, 0x1829, 0x2f6e, 0xba3a, 0xba63, 0x42c9, 0x41fa, 0xbf22, 0x415e, 0xa64f, 0xb072, 0xb2fe, 0x2efd, 0x4080, 0x0976, 0xb299, 0x4673, 0xb22c, 0x436d, 0xb2f3, 0x41f3, 0x02e9, 0x1955, 0x43fa, 0x411d, 0x4197, 0x41b0, 0x40cd, 0xbae5, 0xbadf, 0xbf42, 0x41ed, 0x41ea, 0xb240, 0xba2a, 0xb0eb, 0x4049, 0x40e4, 0xb223, 0x4361, 0x430c, 0x4144, 0x17db, 0x4053, 0xb284, 0xb0f5, 0x4074, 0x401b, 0x213c, 0x40eb, 0x4331, 0xb27f, 0xb2ba, 0xb2c6, 0x428e, 0x40c1, 0xbf8b, 0xb239, 0x41c2, 0x43a2, 0xb2f8, 0x4193, 0x4005, 0x432e, 0x4596, 0x44b9, 0x43a5, 0xb288, 0x3e1e, 0x449d, 0xb2b7, 0x1b4f, 0x2bc5, 0x43f4, 0xba10, 0x409c, 0x4090, 0x414c, 0x0672, 0xbfd4, 0x4356, 0x44b3, 0x424e, 0x1fb8, 0x4434, 0xb00d, 0x4045, 0x2fcd, 0x40fb, 0xba20, 0x41ae, 0x4266, 0xba12, 0x407b, 0x420a, 0x1a0f, 0x2281, 0x31c6, 0x1abf, 0x1d76, 0x41b8, 0x4055, 0x45c4, 0x43cc, 0x40c8, 0x4165, 0x163e, 0xbf1f, 0xba6c, 0xacaf, 0xba04, 0x412c, 0xb2a6, 0x42fb, 0xbaea, 0xb0ac, 0x446b, 0x4381, 0xa37d, 0xbf27, 0x4163, 0x42f8, 0xb24c, 0x41d5, 0x182b, 0x426e, 0x40e5, 0x3fa2, 0xbf9f, 0xba3c, 0x2e88, 0x4362, 0x3a89, 0xb2f3, 0xbacc, 0xb218, 0x3a16, 0x2c8f, 0x42f4, 0x43a4, 0xbf2c, 0x2016, 0x0105, 0x2c60, 0x425d, 0xbf17, 0x443d, 0x19cd, 0x21e8, 0x2803, 0x402d, 0x40e0, 0x1330, 0x1a90, 0x42e5, 0x151c, 0xb022, 0xbf47, 0x1f76, 0xba71, 0xb207, 0x40f2, 0x4172, 0x0965, 0x42af, 0x407f, 0x4162, 0x4355, 0x13d1, 0xbfa0, 0x0910, 0x4574, 0x42f4, 0x206f, 0x431d, 0x437c, 0x42a0, 0x4108, 0x4469, 0x21dc, 0x42d4, 0x3b95, 0xbad0, 0x420f, 0x1295, 0x428b, 0xbfd6, 0x05f7, 0xbf00, 0x4685, 0x4193, 0x463c, 0x4190, 0xbf5e, 0xba3e, 0xb2b8, 0xbafb, 0x07a5, 0x4320, 0x4161, 0x43d8, 0x004d, 0x40f4, 0x4040, 0xb253, 0x412b, 0xb230, 0x408f, 0xbf00, 0x021f, 0xbfa0, 0x42aa, 0x3978, 0x3bc6, 0x1e67, 0x406f, 0x4161, 0x4628, 0xad0c, 0x4232, 0xba05, 0xb2ea, 0xbf45, 0x2bbd, 0x224d, 0x43ce, 0x4262, 0xbfe1, 0x0ac9, 0x42d2, 0x4592, 0x2030, 0x3981, 0xbf7d, 0x3d39, 0xae51, 0xbfd0, 0x4161, 0x41e6, 0x414d, 0xba32, 0x4680, 0x311a, 0x18e4, 0x1f85, 0xbf64, 0x1a28, 0xbadf, 0x46e4, 0xa5c9, 0x4199, 0x3d07, 0x0825, 0xbf2d, 0x40e5, 0x21ec, 0x0b46, 0x430a, 0x1cd7, 0xa4e2, 0x4396, 0x4370, 0x2c8a, 0xbf04, 0xb273, 0xba2e, 0x423f, 0x428d, 0x1f54, 0xbfc6, 0x41cc, 0x1aa1, 0x402c, 0x4397, 0x4175, 0xb287, 0x4222, 0x46c3, 0x305a, 0x421f, 0x41c1, 0xb283, 0x43a0, 0x431f, 0x1ec0, 0x43fa, 0x0821, 0xba45, 0x405d, 0xb0e4, 0x407e, 0xbfbf, 0x00ab, 0xb22b, 0x309d, 0x4269, 0x4021, 0xb09e, 0xbf0d, 0xb251, 0xbae7, 0x42de, 0x41d1, 0x4132, 0xb08a, 0xa1ec, 0x1fc5, 0xb09a, 0x4332, 0xbfd0, 0x26f4, 0x4069, 0x42e6, 0xb275, 0x1ae3, 0x4350, 0xbf31, 0x3b04, 0x1f7a, 0x4152, 0xbaca, 0x1f2f, 0xb032, 0x1d51, 0x44cc, 0x4016, 0x18ce, 0x41fc, 0x1d3e, 0x409d, 0xbf4b, 0x4241, 0x2393, 0x4139, 0x4374, 0x43c6, 0x40fb, 0x41ba, 0x370d, 0xb205, 0x42d5, 0x4231, 0xb2c5, 0x4253, 0x42c6, 0x40ea, 0x38fd, 0xb2f0, 0x1002, 0x1676, 0x42bc, 0xbfdb, 0x1c6c, 0x0b4f, 0xb230, 0x41ee, 0x4233, 0x29f8, 0x191e, 0xba4a, 0x408e, 0x45bd, 0xb20f, 0x3a1d, 0x2891, 0x3659, 0x127e, 0x43a2, 0x167c, 0xb2b5, 0xb298, 0x43b5, 0xb2ba, 0x4344, 0x426e, 0xb259, 0x1293, 0xbf25, 0x4134, 0x2a59, 0x40d7, 0xafc9, 0x2b8d, 0x42da, 0xb03d, 0xb000, 0x4656, 0x420a, 0x2c8d, 0x065e, 0x4252, 0x43c6, 0x458b, 0xa5fd, 0x4111, 0x42fd, 0xbf85, 0xbaff, 0x4032, 0xb2b7, 0xb236, 0xb044, 0xb205, 0xbae5, 0x416e, 0x42e0, 0x412d, 0x1d40, 0x0fc4, 0x42e7, 0x428e, 0x4376, 0x1b9e, 0x263a, 0x42c3, 0x43a4, 0x08bf, 0x409a, 0x2124, 0x40f9, 0xb29c, 0xbf9d, 0x415e, 0x4003, 0x275f, 0xaf43, 0xba41, 0xa013, 0xb02a, 0x4398, 0x424f, 0x411f, 0x42ee, 0xa2da, 0x45f3, 0x440d, 0xb2ed, 0x429c, 0x4263, 0x1f66, 0x0f8d, 0x3500, 0xbf68, 0x40a6, 0xb2de, 0x43df, 0x40a1, 0xba52, 0xb0fb, 0x145d, 0x19ce, 0x43a1, 0x427d, 0x4493, 0x40cf, 0x4103, 0x463b, 0xb0cf, 0x4113, 0x458c, 0x1ac8, 0x030e, 0x42ef, 0x4381, 0x19ae, 0xa923, 0x4209, 0xbf91, 0xb01d, 0x199e, 0x1a59, 0x41d7, 0x4390, 0x414f, 0x4144, 0x19df, 0xa99a, 0x43d8, 0x4059, 0x400b, 0xba42, 0x4327, 0xb20f, 0x2ad1, 0xbfa6, 0x4335, 0xb2ef, 0x10be, 0x1845, 0xbfd8, 0xb2a8, 0x40f2, 0x405d, 0x42bd, 0x43c2, 0xabd0, 0x40f0, 0x4361, 0x413b, 0x1dcf, 0xb20d, 0x3168, 0xba5b, 0x43e1, 0x43a3, 0xb053, 0xba1f, 0xa340, 0xba35, 0x27b0, 0x45c6, 0x4093, 0x40d7, 0x43fd, 0x1f45, 0xbfb9, 0x424e, 0xaf65, 0x1215, 0xb247, 0xb220, 0x44d0, 0xb27c, 0xbf82, 0x232d, 0xa8e7, 0x42cb, 0x3cd3, 0x1aea, 0x4110, 0xbaee, 0x2c77, 0x4028, 0xb2f8, 0x2376, 0xb262, 0x0ccd, 0x4059, 0x40bc, 0xba54, 0x4367, 0xb2dc, 0x4012, 0x4012, 0x4221, 0x182a, 0x195a, 0xbf02, 0x4361, 0x04db, 0x427f, 0x44a2, 0x43c3, 0x41d8, 0x427f, 0x017f, 0xaffb, 0xbfc9, 0x426e, 0xb09d, 0x11ce, 0xb21b, 0xba66, 0x0ab4, 0x420b, 0x075f, 0xb25e, 0xbf95, 0x42bc, 0xb23a, 0x4070, 0x41b8, 0x43c6, 0xa5d7, 0x4271, 0x438a, 0xbf05, 0xac31, 0xbadf, 0x43e3, 0x1eb1, 0xb225, 0x4237, 0x4015, 0x425d, 0x07fc, 0xa809, 0x43b6, 0x43f8, 0x4330, 0x4028, 0xbf5a, 0x42c3, 0x1ce7, 0xb2eb, 0x42ef, 0x34ec, 0x4334, 0xba3d, 0x1d32, 0xb232, 0xbaca, 0xb23c, 0x4039, 0xadc4, 0xb0a5, 0xb2f2, 0x199a, 0x41f9, 0xb26c, 0x404a, 0xba1e, 0x40ac, 0xb044, 0xba2a, 0xbf35, 0xb015, 0xba51, 0xa8ac, 0x1963, 0x20b1, 0xbf39, 0x40ad, 0x40ed, 0x4151, 0x2d49, 0xb20b, 0xba2d, 0x1edd, 0xb211, 0x1f04, 0x4339, 0x4349, 0xb218, 0x4286, 0x43d1, 0xbfd3, 0xba2d, 0x41ce, 0x18d1, 0x4543, 0x1d0b, 0xa053, 0xbfe0, 0x15da, 0x2d45, 0x3d82, 0x4115, 0x04a6, 0x43b4, 0xb065, 0x282f, 0x0267, 0xa36f, 0x422f, 0x011c, 0xbf0e, 0x029e, 0xba71, 0xad75, 0x43c6, 0x4288, 0x42f7, 0x4616, 0xbf2b, 0x404e, 0x1a44, 0x30f1, 0xb09e, 0x4354, 0xb2c7, 0x4000, 0x3fd3, 0x18f4, 0x2e42, 0xb279, 0x4257, 0xb0f8, 0x42e9, 0x4065, 0x4021, 0xba0f, 0x4121, 0x419d, 0x40e4, 0x41c7, 0x259c, 0x4023, 0x435a, 0x1afd, 0x1f12, 0xba21, 0xbf4e, 0x4313, 0x254b, 0xb27e, 0x41f6, 0x4557, 0x4287, 0xb270, 0x42cf, 0xb07e, 0xbf94, 0xb2c9, 0x432f, 0x4397, 0xafbd, 0x3d40, 0xa8ae, 0x3690, 0x40b8, 0xafdc, 0xb064, 0x26fe, 0x43a3, 0x466f, 0x39e6, 0xb2ee, 0x0008, 0x4273, 0xba2e, 0x1d7b, 0x39d9, 0xbf48, 0xba57, 0x1c17, 0x0edb, 0x415e, 0xbfc6, 0x14c0, 0xbacf, 0xbacc, 0xacb0, 0x4359, 0xb21a, 0xbf5d, 0x415e, 0x41cf, 0x4423, 0x42f2, 0xb27b, 0x41ac, 0x4571, 0xbf00, 0xba7e, 0xb01c, 0x42db, 0x2818, 0x3d11, 0xbf90, 0x4186, 0xb0f6, 0xb0c3, 0xba49, 0x1c79, 0xbac8, 0x19b8, 0x407a, 0xa439, 0x1af9, 0x414e, 0xb227, 0xa3ab, 0xbf83, 0x4107, 0xba3e, 0x1e2f, 0xbfe0, 0x0b3b, 0x437d, 0x40b5, 0xba44, 0x45f2, 0x447f, 0xb2f9, 0xb2f6, 0x44a0, 0x4470, 0x4634, 0x42df, 0x1b5d, 0x426f, 0xbfb0, 0x4056, 0x1e3d, 0x1ad9, 0xa59d, 0xbfc0, 0x1404, 0xbf48, 0xb2f1, 0xbf61, 0x099c, 0x43c9, 0xba5a, 0xb278, 0x4002, 0x4083, 0x08c7, 0x42d0, 0x44eb, 0xa734, 0x462e, 0x4121, 0x4218, 0x4060, 0x402f, 0x415b, 0x464b, 0x441c, 0xabc9, 0xb255, 0xbaf4, 0xb2c5, 0x43c2, 0xbf7b, 0x1f61, 0x1dda, 0x42f6, 0xb22d, 0xbf72, 0x4604, 0x4384, 0x1ef3, 0x1d46, 0xbf05, 0xbafd, 0xb207, 0xaf8f, 0x4561, 0xba57, 0x406d, 0x421e, 0x420f, 0x41c5, 0x45e5, 0x1dbd, 0x1fc6, 0x41f5, 0x4203, 0x422f, 0x42c3, 0xb247, 0xb069, 0xbf5c, 0x414b, 0x41c3, 0x41fa, 0xbf00, 0xafae, 0x4215, 0xbf4a, 0x4000, 0x41b2, 0x1e3b, 0x2842, 0x133b, 0x42d9, 0x4165, 0x4262, 0x43b9, 0x2626, 0x41b8, 0x4295, 0xacbc, 0x2fcf, 0xbfe2, 0x2b38, 0x4020, 0x09e3, 0xbfae, 0x40bb, 0x4289, 0xb226, 0xbf17, 0xa215, 0x4326, 0x2a3b, 0x0daa, 0xbf98, 0xb045, 0x42b7, 0x462d, 0x43d2, 0xa357, 0xb20e, 0x401b, 0x0d93, 0xba3d, 0xba27, 0x3a82, 0xbf60, 0x1844, 0x4211, 0x4004, 0x42ac, 0xba4b, 0x282a, 0xb272, 0x42f8, 0xbf90, 0xa34e, 0xbf0f, 0x43f0, 0x2f72, 0xb06f, 0x461e, 0x1d80, 0x19fb, 0xada0, 0xa8f2, 0x3b75, 0xbf02, 0xb294, 0x0e2f, 0x0a11, 0xb21e, 0x1af3, 0x43b1, 0xa521, 0xb264, 0xbad0, 0x43fc, 0x42c9, 0x1877, 0x403d, 0x43fc, 0x0217, 0xbf0c, 0xb0cd, 0xb2a4, 0x4196, 0xb2ee, 0x2b9c, 0xb007, 0x41a1, 0xbf2c, 0xba44, 0xb06d, 0xba5d, 0x4063, 0x1c68, 0x45c8, 0xbf71, 0x4100, 0xb274, 0xb0c0, 0x4421, 0x3a36, 0x0655, 0x204b, 0x1fc2, 0xb2b6, 0x1c68, 0xb050, 0x1e86, 0x3b90, 0x3b9e, 0x2bb7, 0xa64d, 0xaf8e, 0xbf14, 0x4219, 0x074a, 0x43c7, 0x4179, 0x42ae, 0xb026, 0x4551, 0xbfc9, 0x427e, 0xa12c, 0x465b, 0x42c1, 0x419c, 0x2a70, 0x4200, 0x4217, 0x4437, 0x1842, 0xbf70, 0xb03c, 0xa47b, 0x40fd, 0x4623, 0x1ee8, 0xbf23, 0xb23b, 0x41a1, 0x40ff, 0x42be, 0x1b0b, 0x43a1, 0x4193, 0x4008, 0xba0d, 0x4392, 0x0f39, 0x46bc, 0x2b84, 0x462e, 0xbf0d, 0x426a, 0xb093, 0x3ef0, 0x2b53, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x284dbe99, 0x058f67cb, 0xb9dae013, 0xa1c1f55b, 0x52670afd, 0x96fcee8f, 0x8a2497af, 0xbd5ee36d, 0x01bd25ee, 0x57094880, 0x25509dd6, 0xeb081cb3, 0x37cda5c3, 0x8213e3ef, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x01ffe449, 0x00000000, 0x00000000, 0x65ffccb7, 0x000019a4, 0x49e4ff65, 0x49e4fe75, 0x00000000, 0x25508003, 0x57094880, 0x25509e4c, 0x000070de, 0x00000000, 0xffffa158, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1d00, 0x407e, 0x3525, 0x2982, 0x43ea, 0x4352, 0x4151, 0xbf5e, 0x09a3, 0x4196, 0xa5c3, 0x3298, 0xb061, 0x1d31, 0x4086, 0xbf05, 0xb040, 0x1336, 0x1c1e, 0xa99b, 0x1174, 0x3dc2, 0x4117, 0x40bb, 0x43a0, 0xbaec, 0xb0de, 0xb229, 0x4450, 0xbf05, 0x4142, 0x1f56, 0x4160, 0xb258, 0x438b, 0x4160, 0x2c59, 0x433f, 0x13c8, 0x35ae, 0x4398, 0x1eb1, 0x236d, 0x406a, 0x41cf, 0xbfd4, 0x417a, 0x41a3, 0x42d5, 0xb2f5, 0x410e, 0x41b9, 0x45bb, 0x4083, 0x4189, 0x41b7, 0x01e2, 0x1f77, 0xba53, 0x2388, 0x4561, 0x40c8, 0xbf1f, 0x401d, 0x007d, 0x26a6, 0x24e5, 0xb209, 0x4623, 0x4346, 0x403a, 0xb0b1, 0xba2d, 0x41d9, 0x1af2, 0x444e, 0xb0ea, 0x40f1, 0x01d5, 0xa4bd, 0xbf91, 0x41d9, 0x425c, 0x414e, 0x40d0, 0xba56, 0xb2ce, 0x0d2b, 0x1984, 0xb0bb, 0x3257, 0x2ba1, 0x43d1, 0x1f95, 0x43d6, 0x40d2, 0x1fa9, 0xba3f, 0xbaca, 0x4294, 0x174e, 0x40ef, 0x434e, 0x4089, 0x445b, 0x19e7, 0x42a7, 0xbfb9, 0x4056, 0xac80, 0x2696, 0x43f6, 0x465f, 0x425d, 0xba41, 0x4092, 0x251d, 0x1e2e, 0x40af, 0x4397, 0xbfb0, 0xbf9f, 0x4166, 0x444d, 0x41c4, 0xbaf0, 0x316d, 0x43db, 0x414b, 0x32c1, 0x325c, 0x41ab, 0xbaf1, 0xbf60, 0x08d1, 0x313c, 0x4387, 0x4613, 0xbf0b, 0x19f0, 0x411d, 0x4141, 0x11c6, 0xb2c3, 0x1f54, 0x41d1, 0xb030, 0xb227, 0x1d06, 0x355b, 0xbf21, 0x0f72, 0x41b6, 0x1f75, 0xb048, 0x4382, 0x43ff, 0x411f, 0x1bbf, 0xb263, 0x434d, 0x40c1, 0x1f58, 0x2d9e, 0x403d, 0x4341, 0x4379, 0xb2a8, 0x30d6, 0xbf08, 0x4027, 0x4312, 0xbacf, 0x4021, 0x414c, 0x4304, 0x1878, 0x2a19, 0x4688, 0x40a8, 0x146a, 0x410c, 0xb02d, 0xb2bc, 0x2063, 0x3346, 0x438f, 0x43fd, 0x4160, 0x1a4f, 0x4373, 0x4128, 0x4132, 0xb2c2, 0xb0ce, 0xa0e0, 0xbf3f, 0x4282, 0x423e, 0x41fc, 0xac39, 0xbfb8, 0xbfc0, 0x1996, 0x3e39, 0x0f2b, 0x1ece, 0xb069, 0x40fc, 0xbaf6, 0xbfc2, 0x0f93, 0xb0f4, 0x4346, 0x08d7, 0xba06, 0xb200, 0xba43, 0xba72, 0x0fc7, 0x1b92, 0xba08, 0x41f0, 0x19fe, 0xb256, 0x1bac, 0x4222, 0xb062, 0xbade, 0xb0aa, 0x436d, 0x1faa, 0x0663, 0xbf80, 0xbf53, 0x4195, 0x408b, 0x4034, 0xb2c6, 0x1a0f, 0xb2e8, 0xb291, 0x40b8, 0xbaec, 0x117c, 0x27ef, 0x4084, 0x405a, 0x40e7, 0xafea, 0x40d1, 0x1157, 0x4121, 0xb06f, 0x4135, 0x2090, 0x16e4, 0x4318, 0x1c80, 0x310f, 0xb0e2, 0x40cf, 0xbf8e, 0x407a, 0xb09c, 0x388f, 0x4299, 0x1fdd, 0xb23f, 0xbf15, 0x1f0a, 0x40b2, 0x42de, 0x2ca0, 0x424f, 0xa934, 0x40be, 0x1dda, 0xb04d, 0x0c79, 0xbac7, 0x18ec, 0x4300, 0x1a08, 0xb219, 0x2eab, 0xbf8e, 0x1a3c, 0xb228, 0xad72, 0x403c, 0xb000, 0x44b2, 0x23a8, 0x41a6, 0x43d7, 0x404d, 0x0805, 0x43e8, 0x459b, 0xa1b1, 0x45b3, 0x4159, 0x44b4, 0x09a2, 0x42a1, 0xbf7e, 0xb2d5, 0xb007, 0x391a, 0x1bb2, 0x45d1, 0x4179, 0x420f, 0xba1f, 0x1ad5, 0xb22f, 0xb05f, 0xb0a6, 0xb269, 0x279d, 0x41c9, 0x4012, 0x0e08, 0xbf11, 0xba49, 0x417d, 0x3e9d, 0x436e, 0x4189, 0x42fb, 0x43e7, 0x4229, 0x40f3, 0x1a1e, 0x1ef2, 0x1061, 0x133f, 0xbac4, 0xb211, 0x42f1, 0x46ab, 0xb25e, 0x4387, 0x404f, 0x400e, 0xb038, 0x4167, 0x4564, 0xb201, 0x4073, 0xb03c, 0x234a, 0xbfae, 0x46e0, 0xb20d, 0x1862, 0xbf88, 0x42b6, 0xb029, 0xa541, 0x406d, 0x4236, 0x413e, 0x45e5, 0x1b98, 0x4224, 0x40ad, 0x41d4, 0x4197, 0x0ec7, 0xbac8, 0x425b, 0x0254, 0xb28b, 0x328c, 0xba1a, 0x42db, 0xba4f, 0x418b, 0x4104, 0xb265, 0x40d9, 0xbf39, 0x41df, 0xb23c, 0x43bd, 0x43b3, 0x4312, 0x41de, 0xbf2b, 0x365d, 0x195c, 0x40ac, 0x0fea, 0x1dc7, 0x2d8d, 0x420f, 0x43bc, 0x41ac, 0xb0d0, 0xb279, 0x25fb, 0x4386, 0xb095, 0xa6c5, 0x4106, 0x3591, 0x41c0, 0xb06c, 0x43f2, 0x4037, 0x4388, 0x0cab, 0x4065, 0xa28e, 0xbf02, 0x4209, 0x1874, 0xb0c1, 0x4085, 0xb2c9, 0xb099, 0xb045, 0x4183, 0x13ed, 0x1dd4, 0xa5f9, 0x42d7, 0x1691, 0x436c, 0xa84e, 0x40e4, 0x439e, 0xb2cc, 0x0e65, 0xb238, 0x42b6, 0x3c33, 0xbf73, 0x1e26, 0xb2f2, 0x4376, 0x43dc, 0xbacf, 0x40dd, 0x431d, 0xba58, 0x2b96, 0x463a, 0x3767, 0x42cb, 0xb0ec, 0x40d7, 0xb007, 0x454f, 0xb28f, 0x1ee1, 0xbf1c, 0xb20e, 0x454f, 0x46dc, 0x4380, 0xa1b6, 0xbfd2, 0x07c6, 0x0f93, 0x4294, 0x4159, 0xbf39, 0xb27e, 0x4396, 0x40b8, 0x27c1, 0x410a, 0xb05f, 0x4364, 0x0507, 0x41a0, 0xb2ea, 0x1c8d, 0x007f, 0x214d, 0x4191, 0x11b4, 0x4096, 0x01bb, 0xbf2a, 0xaff8, 0x2725, 0xb29d, 0x1c3a, 0x41a2, 0x0bea, 0x4337, 0x413b, 0x3f7a, 0x4493, 0x4058, 0x1f5f, 0x29c8, 0x1b7d, 0xba6b, 0x41a1, 0x1142, 0x1b05, 0xba5a, 0x422a, 0xb23e, 0xbfd3, 0x4171, 0x43c0, 0xb04b, 0x4417, 0x2814, 0x4068, 0x40b9, 0x4123, 0xa999, 0x44ba, 0x4375, 0x3e79, 0x42d2, 0x4104, 0x1b76, 0x3c21, 0x4150, 0x431e, 0xb2fe, 0xba47, 0xb279, 0xbf85, 0x415b, 0x1c20, 0x4477, 0xac9a, 0x28ba, 0x433f, 0xbfd0, 0xb0fb, 0x409b, 0x4299, 0xba02, 0x418e, 0xa2a0, 0x3aae, 0xb0ad, 0xa584, 0x1a84, 0xa29f, 0x3a28, 0xa99b, 0xba7b, 0xb0d9, 0xba18, 0x32f2, 0x460b, 0x441a, 0x3bb7, 0xbfd5, 0x4324, 0xb2bb, 0x4117, 0x4563, 0x1fd6, 0xba34, 0x429e, 0x4078, 0xaba3, 0xa4bb, 0x0e12, 0xb259, 0xb2f0, 0x443a, 0x4215, 0x42f5, 0xa264, 0xbaed, 0xba2b, 0xbf1b, 0xba59, 0x0098, 0xb205, 0x414e, 0x1e4e, 0x40a9, 0x42f0, 0x4052, 0x4485, 0xbf93, 0x15d8, 0x3ddc, 0x407d, 0x4381, 0x418a, 0xa2dd, 0xb29f, 0xb27d, 0xb0f2, 0x304b, 0x1d56, 0x46e5, 0xa925, 0x2fb8, 0x419e, 0x45ec, 0x41b4, 0x406d, 0x13e1, 0x426b, 0x4095, 0x4262, 0x15d2, 0x443f, 0x41d1, 0xbfcb, 0x2b00, 0x4319, 0xb229, 0xb21e, 0x4277, 0xbf35, 0xb28c, 0xb22d, 0x4558, 0xbad2, 0x43b9, 0xbac8, 0x404d, 0xba27, 0x4236, 0xb217, 0xacb6, 0x3991, 0xb271, 0xac00, 0xb2ba, 0x3bf7, 0x438b, 0xb2b9, 0x26a3, 0x431d, 0x2cf2, 0x421f, 0x23ac, 0x41db, 0x40c1, 0xb2c7, 0x1678, 0x427c, 0xbf8b, 0x3bd6, 0x43b7, 0x041a, 0x42ab, 0x21b7, 0x3236, 0x1892, 0x149e, 0xb27f, 0xb28e, 0x0325, 0xba7c, 0x43a8, 0x4242, 0x441c, 0xbaf4, 0x4210, 0x417f, 0x401e, 0x4252, 0xb2d2, 0x4321, 0x4062, 0xb2d4, 0x3dcc, 0xbfac, 0x40ab, 0xb00f, 0xb08f, 0x4148, 0x41e1, 0xb0a5, 0x41b9, 0x0630, 0x0865, 0x43e9, 0xb0c2, 0x4551, 0x4131, 0x41c5, 0x4014, 0x2bfc, 0x42a0, 0xbf4e, 0x407f, 0x41f6, 0x1b7a, 0x0220, 0x41ac, 0x099e, 0x46b4, 0x42d8, 0xb223, 0x423d, 0x42d1, 0x1d10, 0x41bd, 0x4188, 0x4165, 0x12c7, 0x4174, 0x4018, 0x41e2, 0x414e, 0xbae3, 0x4250, 0x43f9, 0x1ed9, 0x28a1, 0x0e3f, 0x408e, 0x43db, 0xbf9b, 0x4009, 0xb295, 0x4131, 0xb275, 0x4099, 0xb2b9, 0x0ea5, 0x456c, 0x1eaa, 0xa4ed, 0xb085, 0x4194, 0x4197, 0x0d72, 0xaffc, 0x4031, 0x0194, 0xadd6, 0xb270, 0x1eac, 0x1981, 0x2bdd, 0xba1f, 0xbf70, 0xbfbf, 0xbad7, 0x4073, 0x4454, 0xba1d, 0x446b, 0x419e, 0x4021, 0xbfd0, 0x40a7, 0x1e5e, 0xbf63, 0xb2cf, 0x462a, 0xba68, 0x0c00, 0xb02a, 0xb070, 0x1318, 0x2880, 0xb061, 0x0995, 0x4273, 0xb283, 0xbad8, 0x1b22, 0x40a1, 0x15d5, 0xbf84, 0x419e, 0xb274, 0xba71, 0x46d4, 0x45b1, 0x432e, 0x1fb3, 0x381a, 0x42d1, 0x4183, 0x4588, 0x408b, 0x4599, 0x43f9, 0x3324, 0x4338, 0xbf60, 0xbff0, 0xbff0, 0x1fb8, 0xbafc, 0x421c, 0x40d7, 0x40fb, 0xbf79, 0xbade, 0xba55, 0x2335, 0xb047, 0xbfb8, 0x403b, 0xb0a0, 0xbad2, 0x1c8b, 0xba65, 0x3c31, 0x407d, 0x1992, 0x2c0f, 0x0e6d, 0x2f2a, 0x4119, 0xb218, 0x0128, 0x4216, 0xa76e, 0x4390, 0x4045, 0x466f, 0xba55, 0x43e0, 0x3c13, 0x41f9, 0xbf91, 0x3368, 0x1acc, 0x4040, 0x4349, 0x406d, 0x408e, 0xbaf5, 0x42f7, 0x400c, 0xb21a, 0x4306, 0xb253, 0x41e1, 0x32fa, 0x3fdf, 0xa3d0, 0xb207, 0x402c, 0xb25a, 0xba45, 0xbfcf, 0xa252, 0x443a, 0x40c1, 0x0341, 0xbf05, 0xabce, 0x4354, 0x4559, 0xb000, 0x40c2, 0x02ff, 0xb288, 0x425d, 0x40ec, 0x4599, 0xba76, 0x4198, 0x1188, 0x385c, 0x193e, 0x411e, 0xba18, 0xbf47, 0x41ff, 0x1b0c, 0x4298, 0x40c5, 0xb235, 0xba44, 0x3560, 0x4346, 0x415a, 0x199e, 0x1b6e, 0xbad1, 0x45b6, 0x42a7, 0x4253, 0xa5a4, 0x4384, 0x420a, 0xb0aa, 0xbf11, 0x3731, 0x12c0, 0xadb1, 0x426f, 0x4310, 0x2243, 0x25ec, 0xb2e4, 0x0845, 0xbf97, 0x426f, 0x425f, 0x40d3, 0xb2c2, 0x3293, 0x4610, 0xa6f4, 0x4376, 0x1838, 0x4143, 0x179e, 0x4381, 0x1922, 0x15ff, 0x41b3, 0xb2fb, 0x188a, 0xbf3e, 0xb028, 0x3da2, 0x4227, 0x43ef, 0x4079, 0xb055, 0x429d, 0xba59, 0x2586, 0xb0ad, 0x19bf, 0x41fd, 0x4023, 0xb2a1, 0x43e8, 0xa823, 0x4266, 0x41f9, 0x42a1, 0xb228, 0x2c34, 0x41da, 0xb211, 0x16da, 0x2895, 0x4392, 0x2eea, 0xbf24, 0x4255, 0x1922, 0x40c6, 0xbaf3, 0xb2ff, 0x40dc, 0x4141, 0x437f, 0x43c4, 0xb283, 0x1b59, 0x4198, 0x3aba, 0x1f81, 0x1f5b, 0x429a, 0x4204, 0x4037, 0xaefc, 0x183d, 0x1309, 0x4563, 0x09a5, 0xb207, 0xba1f, 0xbfa0, 0x1cd7, 0xbfa3, 0x4258, 0xb205, 0x427a, 0x43b2, 0x41cc, 0x1797, 0x43de, 0xb227, 0x1a3c, 0x4038, 0x4355, 0x4033, 0xbf6b, 0x42f5, 0xb083, 0x4396, 0x3d30, 0x0b59, 0x38cb, 0x4319, 0x37ab, 0x425a, 0x251a, 0x41fd, 0xb2c8, 0x454b, 0xbaea, 0x40ec, 0x43fd, 0x4286, 0x445e, 0x4689, 0xb270, 0x4011, 0x2d71, 0xb0df, 0x4026, 0xa569, 0xbf2e, 0x4121, 0x4393, 0x40e1, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x0542461d, 0xf5f04e12, 0x8bc7b1fe, 0xbd178d21, 0x26456cae, 0xca50d85d, 0xf3453942, 0x72f63230, 0x00bbda65, 0xae6250a2, 0xb5896c69, 0xd8e9e883, 0x00e97798, 0xcb5e9574, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0xffffffc9, 0x00000000, 0x00000000, 0x00000000, 0xfffff79f, 0x00001998, 0xfffff389, 0xfffff84a, 0x00e9e597, 0x00000000, 0xb58955e6, 0xfffffff5, 0xb58955e6, 0x000000c1, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x0039, 0x3a23, 0x427e, 0x40e9, 0xb280, 0x0a50, 0xbfc9, 0x4114, 0x0bf0, 0x1254, 0xbf90, 0x2ad8, 0x440a, 0x414d, 0xbae8, 0xb243, 0xb01e, 0x2da6, 0xae3f, 0x4280, 0x445e, 0x4080, 0x4080, 0x3da8, 0x29dc, 0x2d10, 0x4373, 0x426e, 0xbf22, 0xb24b, 0x424a, 0x4332, 0x40a5, 0x46e3, 0xb28a, 0x42af, 0x4574, 0x1219, 0x1a9c, 0x43df, 0x438a, 0x43dc, 0xbfa9, 0x4318, 0xb28d, 0xb278, 0x0f00, 0xa9a1, 0x4359, 0xb0d5, 0xbf9a, 0x4283, 0xb2bf, 0x1f9a, 0xbfd0, 0x4054, 0x439c, 0xbfd0, 0x4243, 0xb026, 0x0245, 0xbf3b, 0x1b0d, 0x21ca, 0x1fff, 0x4199, 0x1a16, 0x425a, 0xb05a, 0x43cb, 0xbfd2, 0xba68, 0x434e, 0xb237, 0x438c, 0xa6ac, 0xbfcb, 0xbad2, 0x3246, 0x4255, 0xb263, 0x0abb, 0xaec5, 0x2008, 0xb049, 0xa1ea, 0xaa8d, 0xb2d9, 0x2152, 0xb27a, 0x4121, 0xb05a, 0xac70, 0x4138, 0xae5b, 0x2ee8, 0x18cd, 0xb253, 0x4143, 0xba11, 0x4483, 0x41b1, 0xbfcd, 0x1bac, 0x4222, 0x41e5, 0xa4ae, 0x1904, 0x4070, 0x1ff3, 0xb290, 0x40b3, 0xb031, 0x4098, 0x4624, 0x1a74, 0x4284, 0x05c1, 0x401d, 0x416b, 0xb238, 0xbf25, 0x434d, 0x1fa2, 0x40b2, 0x11e3, 0xb2cc, 0x413a, 0x41b7, 0xbf94, 0x43d4, 0x189b, 0xbacf, 0x4571, 0x430f, 0x1056, 0xba57, 0xb202, 0x1be7, 0x42eb, 0xb20e, 0x4544, 0x3f1d, 0xbfbf, 0x18ff, 0xbafa, 0x1a37, 0x4352, 0x0bfe, 0x1eec, 0x42e9, 0xbfd0, 0x419b, 0xbf6c, 0x3b65, 0x4115, 0x4051, 0xbfe0, 0x0eb4, 0x40b3, 0xa454, 0x4399, 0xb05f, 0x39c5, 0x4061, 0x1cb2, 0xbf0a, 0x369f, 0x210e, 0xb0b8, 0xbaf7, 0xa0a5, 0x4197, 0x45ce, 0x4073, 0x1918, 0xb2b8, 0x02a5, 0xb2e8, 0xbf79, 0xb24a, 0x43c6, 0x4034, 0xb08e, 0x46cc, 0x42be, 0x050b, 0x414a, 0x432e, 0x422b, 0xba14, 0x18f5, 0x372a, 0x239e, 0xb0e7, 0x1843, 0x4148, 0x465c, 0xb0b9, 0x3b44, 0x420e, 0xbf9a, 0x4299, 0x39a6, 0x2008, 0x2573, 0x4154, 0xb022, 0xae69, 0x417c, 0x3cbf, 0xb29d, 0x2b2d, 0xba06, 0x4175, 0xb222, 0xbfe4, 0xb205, 0x430a, 0xb211, 0x43ff, 0x0a2c, 0x41e5, 0x1575, 0x0b8d, 0xaf72, 0x1b99, 0x1e09, 0x4149, 0x40b8, 0x1d9c, 0x0c47, 0xba36, 0xb279, 0x4001, 0xb0d3, 0x1b1d, 0xbf13, 0xb219, 0x25db, 0x2745, 0x40c1, 0x1830, 0x4117, 0x428b, 0xb297, 0xb0b8, 0x406e, 0x40ec, 0x41f2, 0x42a1, 0x40da, 0x1df0, 0xaad8, 0x0ad3, 0x40f2, 0x42ae, 0xbf04, 0xba01, 0xae3e, 0xb26a, 0x1dbf, 0x442a, 0xb01f, 0x4051, 0x4147, 0xadb7, 0x420a, 0x1b8a, 0x18b9, 0xb0ab, 0xab57, 0xb072, 0x425d, 0x434a, 0xb2e0, 0xb280, 0xba33, 0x2c1d, 0x430f, 0x1080, 0xbf34, 0x41af, 0xba02, 0xb09a, 0x43d2, 0x1971, 0x3a78, 0x42e3, 0x4300, 0x4482, 0x1bbf, 0x02dd, 0x422a, 0x41a2, 0xbfdd, 0x4592, 0x41af, 0xba74, 0x15b3, 0x42cf, 0xba5b, 0xb2e1, 0xba39, 0x1e81, 0x461f, 0x415f, 0x4173, 0x4253, 0x4041, 0x405a, 0x1f00, 0xbf92, 0xa1f5, 0xb260, 0x456a, 0xb2ff, 0x3029, 0x45a5, 0x418f, 0xb01b, 0xba4d, 0x2658, 0x4384, 0xba5c, 0x3e1b, 0x35d4, 0x174a, 0x09ac, 0xb009, 0xb2ee, 0xb255, 0xbfda, 0x2f7e, 0xa637, 0x41d5, 0x1d37, 0xb0cc, 0x100a, 0x288d, 0x1d1f, 0x4129, 0xba7f, 0x19c8, 0xbf5f, 0x43ee, 0xb0d3, 0x4111, 0x4142, 0xb283, 0x1fd6, 0x4498, 0x43b5, 0x194a, 0x4124, 0x3b97, 0x4670, 0x427c, 0x45dd, 0x410c, 0x4377, 0xbadd, 0x4184, 0x424c, 0x40f7, 0xb25c, 0x4052, 0x2cdd, 0xbfa6, 0xbfe0, 0x416f, 0xb048, 0x4126, 0x40f3, 0xb0f5, 0x4153, 0x13cf, 0xbfd0, 0x41fd, 0x4072, 0x0639, 0x22bf, 0x3c4b, 0x0edb, 0x401d, 0x1f77, 0x4126, 0x4580, 0xbfe8, 0x1bd0, 0xb0fd, 0x2d03, 0x43e4, 0x4418, 0x405a, 0x41cd, 0x28aa, 0x425c, 0x4107, 0xa420, 0x4678, 0xba4f, 0xa155, 0x41e3, 0x4430, 0x400f, 0xba18, 0xba08, 0xb219, 0xbad8, 0xb2e6, 0xbfa2, 0x0925, 0x40f6, 0xba5f, 0xb0e6, 0xbf70, 0xb253, 0x40fd, 0x430e, 0x3042, 0x40ad, 0x4016, 0xba0f, 0xbf5e, 0x4544, 0x3faf, 0x464f, 0x1cb3, 0xbf90, 0x432c, 0x277c, 0x41e9, 0x42d3, 0x4181, 0x4149, 0xb27c, 0x19f6, 0x1e2f, 0x4118, 0x426f, 0xb214, 0xba78, 0x418b, 0xa945, 0x15e6, 0xba23, 0xb242, 0xbfd0, 0x4322, 0xbfab, 0x41cb, 0x4194, 0xba49, 0x4199, 0x42c0, 0xb2b0, 0x43d2, 0xb21d, 0x3ad9, 0x3b7a, 0xba1c, 0xbff0, 0x1b38, 0xbfc6, 0x42a1, 0xba7b, 0x4051, 0xba14, 0x413e, 0xbf85, 0xbfc0, 0xba1a, 0x43cc, 0x44f5, 0xb013, 0xba6c, 0x3c27, 0x4057, 0xb2f1, 0x1bb7, 0x1d41, 0xb28f, 0x42a1, 0xbfda, 0x1d46, 0xb0da, 0xb244, 0xa4aa, 0xb2b6, 0x1b67, 0x2f35, 0x1f03, 0x41cb, 0xb271, 0xbac4, 0x059f, 0x423a, 0xba22, 0x40e9, 0xb0e1, 0xb01b, 0xa845, 0xb063, 0xbf8c, 0x43a3, 0x40b4, 0x39d6, 0x4175, 0x4282, 0xb0de, 0x4037, 0x41e0, 0x42aa, 0xba1c, 0x41ae, 0x1d11, 0xbfa4, 0x186a, 0x4310, 0x432e, 0x1856, 0xb229, 0x4079, 0xa1d2, 0xbf80, 0x4665, 0x4225, 0xbfe2, 0x4372, 0x42a3, 0x448c, 0xbf7c, 0x4293, 0xba76, 0x2941, 0x42ff, 0x15ef, 0xbaee, 0x1735, 0x407c, 0x40c1, 0x1fab, 0xbf44, 0x43cd, 0xb2a0, 0xb2ad, 0x405e, 0x43d4, 0x40a5, 0x4128, 0x2340, 0x43c0, 0x027b, 0x434d, 0x1e76, 0xba2d, 0x4119, 0xbf0d, 0x1c53, 0x0600, 0x40dd, 0x4183, 0x1a96, 0x4167, 0x1c55, 0x4333, 0x46ad, 0xb22e, 0xb244, 0x41ee, 0x40da, 0x4063, 0x022b, 0xb26b, 0x43f3, 0x435a, 0x186e, 0xba0f, 0x1894, 0x4212, 0x4073, 0x41ca, 0x4368, 0xbfb9, 0x2971, 0xbf60, 0xb2f6, 0x43bc, 0xbf9e, 0x4651, 0xb224, 0xa186, 0x40ba, 0x415f, 0x42b0, 0xb044, 0x410c, 0x4035, 0x46c3, 0x3fa3, 0x400a, 0xbf60, 0xbf07, 0xb00a, 0xb263, 0x1cd7, 0x4119, 0x463a, 0x42c9, 0x4307, 0x1f80, 0x030c, 0xb2f0, 0x427c, 0x4619, 0x438a, 0xbfb1, 0x4178, 0x4197, 0x425b, 0x41fd, 0xb26f, 0x4101, 0xb08d, 0x2e09, 0xb072, 0x413a, 0xb281, 0x1eee, 0xaac3, 0x4175, 0x405d, 0x41d8, 0x4665, 0x418b, 0x458a, 0x4059, 0xac9b, 0x3661, 0x4332, 0xbf3a, 0xba5d, 0x4332, 0x4458, 0xb290, 0x1aa8, 0x42db, 0xa756, 0xb2df, 0x284d, 0x1cf5, 0x4148, 0x416b, 0x4445, 0x4114, 0x44aa, 0x4455, 0x40b5, 0xb01a, 0xb249, 0xbad0, 0x1fec, 0x4025, 0x19c2, 0x2014, 0x40c4, 0x42b8, 0xbf93, 0x415b, 0xb08a, 0x4363, 0x43ee, 0xba7e, 0xb2ec, 0xb28d, 0xb0e9, 0xb069, 0x43f1, 0xb2d9, 0xb254, 0x436c, 0x4479, 0x4345, 0xb0d7, 0xba17, 0x4213, 0x46d1, 0x30a5, 0x2e55, 0xbf4f, 0xba63, 0x4426, 0x363e, 0xa9e7, 0x4294, 0xb257, 0x20d4, 0xb22a, 0x1b17, 0x43e6, 0x419e, 0x40cc, 0xaf14, 0xba0b, 0xa0e6, 0xb086, 0x42a3, 0x28ed, 0x43a1, 0x427c, 0xb2f0, 0x1ccc, 0xba6c, 0x41b9, 0xb022, 0xba4f, 0x19a4, 0xbf12, 0x40a4, 0x412b, 0x412d, 0x1e6a, 0x4122, 0x412c, 0x4365, 0x0f73, 0xb2b8, 0x40ca, 0x07ee, 0x3940, 0x200e, 0x377a, 0x4167, 0x40d2, 0x4311, 0x4351, 0x433c, 0x1749, 0x1c33, 0x4137, 0xbf90, 0x1fc8, 0xbfc7, 0x43d6, 0x4294, 0x4338, 0xb27b, 0x4268, 0x2cd4, 0xbf1f, 0x41af, 0xbac7, 0x413b, 0x4099, 0x411f, 0x4270, 0x43e7, 0x40bc, 0xbfdc, 0x425a, 0x4489, 0x4004, 0x4162, 0x2867, 0x454d, 0x1607, 0xba54, 0xb2e4, 0x40c2, 0xbff0, 0x1641, 0xb2b1, 0xbf1d, 0x4560, 0x4371, 0xb2ca, 0x2dc0, 0x41ae, 0x4290, 0x43a2, 0x02d6, 0x1e8f, 0xa801, 0xb054, 0xb03f, 0x4204, 0xba5e, 0x1d4b, 0x42ba, 0x11a0, 0x1f7f, 0xba40, 0x421d, 0xba0d, 0xbacc, 0x434c, 0x4369, 0xbfb5, 0x0d27, 0x2c32, 0x4057, 0x4215, 0x438a, 0xb26c, 0xb028, 0x438a, 0x432d, 0x40a6, 0x0e5c, 0xb224, 0x410c, 0x40f8, 0xa486, 0x421f, 0x0fd8, 0xbf51, 0xab22, 0x135b, 0x1bad, 0x1afa, 0x43de, 0x43cc, 0x2f57, 0x41ff, 0xb27f, 0x15b7, 0x4362, 0xbf03, 0x4084, 0x444e, 0xb21c, 0x4285, 0x41d1, 0x199f, 0x4199, 0x1b47, 0xb2ee, 0x4150, 0xbfd5, 0x432e, 0xa445, 0x40b2, 0x41f0, 0xbac4, 0x45a9, 0x1ee8, 0x0504, 0x410d, 0x34f6, 0x4096, 0x34d1, 0xb26e, 0x439d, 0x24b8, 0x4299, 0x1226, 0x4009, 0xbf46, 0xbff0, 0x42a0, 0x40f7, 0x42b7, 0x43f7, 0x4164, 0xaa14, 0x424a, 0x45d8, 0x414c, 0xbf9d, 0xb218, 0x4223, 0x4430, 0x4036, 0x4163, 0xb0ac, 0xbfaa, 0x409d, 0xb2db, 0x0ae2, 0xb23b, 0xb218, 0x1a2c, 0x441b, 0xa0d2, 0x0e6a, 0xbfb9, 0x3fec, 0x22c2, 0x3012, 0x43ec, 0x1c2c, 0x4264, 0x42f9, 0xb2a2, 0xba13, 0xb270, 0x41dd, 0xb28f, 0x3315, 0x4267, 0x404a, 0x40b8, 0x4077, 0xbf8a, 0x4287, 0x0ce2, 0x406e, 0xbf2b, 0x1ab6, 0xb0aa, 0xbf90, 0xb2c4, 0x4666, 0x1af4, 0xba17, 0x4049, 0x4179, 0xb01b, 0x4154, 0x42e8, 0xb2dc, 0x431f, 0x4029, 0x4442, 0x19ff, 0x3b2a, 0x1a44, 0x1ac3, 0x160c, 0xb06d, 0x31fd, 0xb073, 0x4313, 0x3f1b, 0x43a2, 0xb22f, 0xa562, 0xbfb1, 0xba65, 0xb0d2, 0x1c51, 0x4158, 0xb27e, 0x40b8, 0x43fd, 0x4294, 0xb042, 0xba17, 0xb2ff, 0xbfdf, 0x400f, 0x3598, 0x404c, 0x4101, 0xb202, 0x2e65, 0x417a, 0xb05b, 0x1cd6, 0xab8e, 0x0d74, 0xbff0, 0x3f0d, 0xbadb, 0x0dce, 0x435f, 0xb269, 0xb291, 0x42e0, 0x46bd, 0x41c9, 0xbf63, 0xac82, 0x194d, 0x1cb3, 0x42db, 0xba49, 0x41dd, 0x40b3, 0x1c57, 0x43c3, 0xbf9c, 0x1d84, 0xb2bd, 0xb2c2, 0xbfe2, 0x1df6, 0x405c, 0x4022, 0x4173, 0x423f, 0x40ce, 0x4198, 0x42d3, 0x1cac, 0xbf65, 0x42bf, 0xba1c, 0x1889, 0x42c2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xaa459540, 0xb7417607, 0x90b1ad8e, 0x16822de0, 0x187003f9, 0x872131ae, 0x3224d4e1, 0x42eca858, 0x011e7d28, 0x6dc708e1, 0xb65556a7, 0xb63bce08, 0xc50698a6, 0x40eb158e, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x023ec6b9, 0x0080b091, 0x00000060, 0xfee09ca6, 0x00006364, 0x00006362, 0x00000000, 0x00006362, 0x011f635a, 0xc060e558, 0xc060e558, 0x011f635a, 0x6dc72045, 0x00028578, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbac5, 0x1cc4, 0x3c3f, 0x43b2, 0x408c, 0x4133, 0x4354, 0x4190, 0xbf13, 0x4393, 0xba2e, 0x41f1, 0xb2db, 0xaaa2, 0x4338, 0xbf80, 0xba55, 0x211b, 0x37fa, 0x407b, 0x4103, 0x3bfe, 0xb08f, 0xbf09, 0x413b, 0xa520, 0x4223, 0x1e36, 0xb047, 0x4388, 0x44ed, 0x43b3, 0x180a, 0x41fd, 0x1940, 0x4037, 0x4136, 0x4075, 0xa15f, 0x3b50, 0xb2e2, 0x0a6b, 0x343e, 0xb0b8, 0xae5e, 0x1666, 0xa2f2, 0x42fa, 0x4045, 0xbf3b, 0x4490, 0xb250, 0x4197, 0x406f, 0x1dd6, 0xa912, 0x41b6, 0xa73c, 0x41a3, 0xbf70, 0x40be, 0x1c04, 0x42b0, 0x2e98, 0x27d6, 0xba6a, 0x1eed, 0xb21d, 0x05ea, 0xbfa8, 0xa64d, 0x41e5, 0x42f8, 0x4241, 0xbf90, 0x40b6, 0x4301, 0x3ef6, 0x2451, 0xb076, 0xbf6e, 0x45a5, 0x1fc7, 0x2e04, 0x2934, 0x4098, 0x42e1, 0x102b, 0xbfcb, 0x2cb4, 0xb2ca, 0x1679, 0x2d5a, 0xba39, 0xb280, 0xa5d9, 0x1f1e, 0x3bd4, 0xb20d, 0x4089, 0xba3f, 0x0ef3, 0x392f, 0x40b4, 0xba51, 0x44c8, 0x4340, 0x402b, 0x4400, 0xbf3c, 0x4655, 0x4011, 0xa8bd, 0x057f, 0x432a, 0x1827, 0x434b, 0xbfab, 0xb25c, 0x1467, 0x4101, 0x40b2, 0x43e0, 0x43e1, 0x404c, 0x46bc, 0xb23b, 0x3c33, 0x081d, 0xb006, 0xbaca, 0xb2dc, 0xb202, 0xa346, 0x43a2, 0xb072, 0x4092, 0x1e9d, 0xa2b5, 0x4111, 0xbfac, 0xaf78, 0x2c58, 0x1f71, 0xa7c8, 0xb2ac, 0x3fd4, 0x45cb, 0x44b2, 0x196d, 0x4007, 0x42b5, 0x1953, 0x4049, 0xb0d6, 0xb2e3, 0x4384, 0x1f22, 0x163d, 0xb288, 0xba2f, 0x4226, 0xb292, 0xbf66, 0xba2b, 0xbac8, 0xb0fa, 0xb079, 0x1205, 0xb2f9, 0x43de, 0x3cff, 0x424e, 0x40e0, 0x0cf1, 0x4073, 0x4390, 0x4082, 0x4000, 0x4084, 0xabf0, 0xabd0, 0x0886, 0x466a, 0x41ca, 0xb0ed, 0xb2ec, 0x41fc, 0xbf4b, 0x4478, 0xba51, 0xb2df, 0x419b, 0x1e34, 0xb2f8, 0xbf49, 0xaa3d, 0x41c6, 0x41a3, 0x1aba, 0x40b2, 0x44bd, 0x3d3a, 0xb2bf, 0x28a6, 0x4012, 0x438e, 0x43bc, 0x0ad5, 0xbaca, 0xbf5e, 0x4445, 0x42dc, 0xba1a, 0xa19c, 0x42c9, 0xbfd5, 0x40a1, 0x4048, 0x1893, 0xabd3, 0x404f, 0x4244, 0x4187, 0x4621, 0x4330, 0xb01f, 0x46d3, 0x0861, 0xb2e6, 0x415f, 0x3a61, 0xb24d, 0x1e93, 0x1bac, 0x42d8, 0xb280, 0x40af, 0x40b7, 0x4045, 0xbfbf, 0x40ec, 0xba4c, 0x4494, 0x1454, 0x3538, 0xb2c8, 0xba51, 0x0e63, 0xb276, 0x1d0e, 0x20ee, 0x4337, 0x4204, 0x401e, 0x43b4, 0xb23f, 0x12a9, 0x229b, 0x44ed, 0x444d, 0x407a, 0x1ec5, 0x45c4, 0xba1a, 0xbfc3, 0x3dad, 0x40f6, 0x4085, 0x2377, 0x4418, 0x43a7, 0x2f7d, 0x4071, 0xbfaa, 0x129b, 0xbae5, 0x4691, 0x1be4, 0x1d41, 0x43ce, 0x35f0, 0x1bda, 0x411e, 0xba1a, 0xb2dc, 0x40f8, 0xa3d4, 0x431c, 0x4368, 0xba34, 0x1c65, 0xba13, 0x439f, 0x1b26, 0x4477, 0x40e3, 0x43a2, 0xbf00, 0xaab3, 0xba61, 0x43ab, 0xbf19, 0x4145, 0x40af, 0x182b, 0xb2b4, 0xba74, 0xb0a9, 0x4111, 0x448c, 0x4107, 0x4069, 0x0494, 0x41ef, 0x1af8, 0x42c0, 0x424d, 0x40a4, 0x43da, 0x4246, 0xb260, 0x41cb, 0x403a, 0x424c, 0xba33, 0x43a4, 0x37d0, 0x4390, 0x189b, 0xb2a0, 0xb052, 0xbf83, 0xa25c, 0x24d7, 0xb01e, 0x34d1, 0x3654, 0xb239, 0x1874, 0xba5e, 0xb27a, 0xb24e, 0x406b, 0x43a9, 0x45e2, 0xbafa, 0x4258, 0x393e, 0xbf7b, 0xb2d2, 0xa24e, 0xb293, 0x4027, 0x4268, 0xbf60, 0x4100, 0xb08b, 0x4062, 0x16d0, 0xbf71, 0x4132, 0x4398, 0x41d6, 0x4074, 0x059c, 0x437d, 0xbaf1, 0x400f, 0x4239, 0x0f8a, 0x4387, 0x1e07, 0x40b2, 0xb0a9, 0x4168, 0x43f8, 0xbf84, 0x0189, 0x1606, 0xb0fe, 0x41fa, 0xbfc5, 0x418e, 0x1a73, 0x42bf, 0x4269, 0x41b7, 0xbf95, 0x1abf, 0xac8e, 0x41b7, 0x4249, 0x41e2, 0xb06d, 0x186f, 0x4097, 0x4333, 0x4223, 0x424e, 0x46c4, 0x435b, 0x1c82, 0x416c, 0x40f2, 0xb000, 0xb09b, 0x434a, 0xbac7, 0x4255, 0x365a, 0xb27e, 0x4223, 0xbf2a, 0x4113, 0x411f, 0x4234, 0x3feb, 0xb2bf, 0x436a, 0xb0cd, 0x18e6, 0xa251, 0xbf72, 0x196c, 0xbac9, 0x190f, 0xb065, 0x4222, 0xbf9f, 0x0e34, 0xb234, 0x04ae, 0x401a, 0x19e3, 0xa8be, 0xba60, 0xb0b5, 0xbfd6, 0x4288, 0x375c, 0xb2d6, 0x4269, 0x419e, 0x46c5, 0x228a, 0x467e, 0xba01, 0x2f41, 0x415e, 0xb0d3, 0xb0de, 0x306a, 0x41b2, 0xb251, 0x432f, 0x430c, 0x189f, 0xbf69, 0x4061, 0x4423, 0x0dcf, 0xb272, 0x43e9, 0xa0e9, 0x03b4, 0x4606, 0xba5c, 0x42e5, 0x442d, 0x428f, 0x4044, 0x2c64, 0x4188, 0x4250, 0xa133, 0xbfe1, 0x4467, 0x4379, 0xb209, 0x43ef, 0x10af, 0x42b5, 0x43cd, 0xbad7, 0x4385, 0x1434, 0x4371, 0x44a9, 0xbaff, 0xb2c5, 0xb028, 0x46b1, 0x249e, 0x1a7c, 0x3608, 0x1561, 0x19da, 0x420f, 0x38a2, 0xa417, 0xbf8f, 0x1805, 0x18f3, 0x2abc, 0x44d3, 0xb210, 0x422d, 0xb29e, 0xb2d8, 0x4463, 0x4148, 0xa528, 0x407c, 0x40a0, 0x433e, 0xbfc1, 0x43dc, 0x38b7, 0xb232, 0x43f0, 0x3de3, 0x3077, 0xbf9d, 0x3ccd, 0xb0be, 0xb2c7, 0xba66, 0x420c, 0xb21e, 0x22d8, 0x2e67, 0xba0c, 0x4293, 0x2f4a, 0xba64, 0x413f, 0x1a2d, 0x4177, 0x424a, 0x4258, 0x4280, 0xa986, 0x3839, 0x4171, 0xb055, 0x4330, 0x4233, 0x41a5, 0x1f63, 0xbf78, 0x43c6, 0xb24c, 0x4199, 0xb2bd, 0x42a6, 0x0c98, 0x1f23, 0xb0ce, 0x4159, 0x1e07, 0xbad3, 0xbae3, 0x42db, 0x41bc, 0xb227, 0xba6d, 0x42f3, 0xbf52, 0xb0f4, 0x45c9, 0x4358, 0x4012, 0x411e, 0x41e5, 0x0b04, 0x41af, 0x43d1, 0xbf1e, 0x2603, 0x466d, 0xba17, 0x1fb1, 0x4363, 0xb21e, 0x4221, 0xb093, 0xbadd, 0xba2c, 0xb2a4, 0x43a0, 0xb2f0, 0xb221, 0x4601, 0xbf2b, 0x4629, 0xb07b, 0x0cb6, 0xb262, 0x439e, 0x40d0, 0xb02b, 0x337d, 0xbf03, 0x4683, 0xba7e, 0x18b9, 0x425b, 0xb0e9, 0x467f, 0x22c2, 0xbfba, 0x400a, 0x426b, 0x43b5, 0x3e16, 0x417e, 0xa4af, 0x40c7, 0x4379, 0x4079, 0x2d9d, 0x4374, 0x4333, 0x412e, 0x2fb4, 0xbf70, 0x4295, 0xae48, 0x1be3, 0x41f9, 0xbfd8, 0xb074, 0x4275, 0x45b5, 0x4133, 0x40a1, 0x402a, 0x405f, 0xbfbd, 0x423b, 0x463c, 0xba19, 0x439b, 0x3b6f, 0xb23f, 0x0f1d, 0xba56, 0xb261, 0xba54, 0xa2ae, 0x46bd, 0x43f6, 0x4017, 0xb2b0, 0x4435, 0x40bd, 0x1e76, 0x413e, 0x410b, 0x411b, 0x1a0e, 0x12dc, 0x1d50, 0x42d7, 0x420f, 0x40e3, 0xb2d6, 0xbfa6, 0xb242, 0xb2eb, 0x40e3, 0x4368, 0x24cc, 0x455d, 0x06b0, 0xbf49, 0xb0a0, 0x3dec, 0xba24, 0x432a, 0x4296, 0x40f9, 0xbff0, 0xa816, 0xb27b, 0x4053, 0xb095, 0x42a4, 0x46e2, 0x19c6, 0x421d, 0x406d, 0xbfc1, 0x4226, 0x4101, 0xbafc, 0x427e, 0x29ba, 0x2f83, 0x41b6, 0x4315, 0x4252, 0xbfa0, 0x45f0, 0x0cbd, 0x40dd, 0x1a3c, 0x43c3, 0x4020, 0xbacd, 0x414d, 0x41e7, 0x4323, 0xb25a, 0x1983, 0x17c3, 0xbf8a, 0x40ab, 0x4601, 0x44f9, 0x4080, 0x3d43, 0x09f7, 0x43de, 0x4637, 0x4052, 0x40dc, 0xbaeb, 0x446a, 0x1e28, 0x4007, 0x359a, 0x45ae, 0xab99, 0x23d8, 0x1cb7, 0xb29f, 0x41c3, 0x4463, 0xbf4f, 0x1f1a, 0x42b4, 0xbacd, 0xb27c, 0xb25b, 0x1aa8, 0xadce, 0xbfd0, 0x352d, 0x4383, 0xb26b, 0x43c0, 0xb201, 0x425c, 0x4356, 0x40a4, 0x43f7, 0x40f8, 0xbf6a, 0x4149, 0x3093, 0xba79, 0xba61, 0x431d, 0x1bb2, 0x405d, 0x09e3, 0x43c3, 0xb0da, 0x4116, 0x4561, 0x4094, 0xbfe0, 0xba18, 0xbaf1, 0xbf8c, 0xb225, 0x12db, 0x1aa6, 0x40f3, 0x327f, 0xb2c3, 0x15eb, 0xb22c, 0x27c4, 0xb229, 0x4015, 0xa837, 0xa0aa, 0x4045, 0x44c9, 0xb2cd, 0x416d, 0xbf58, 0x441a, 0xba16, 0x44aa, 0x1be5, 0xb2dd, 0x456e, 0x2da2, 0x4485, 0x3a61, 0x43d4, 0x127d, 0x0f03, 0x4046, 0xa7c2, 0x43ba, 0x422d, 0xb2f8, 0xa357, 0x40bc, 0x1f75, 0xa5db, 0xbf90, 0x42a1, 0x41bc, 0xbf87, 0x1bc0, 0xbf60, 0x3130, 0x40a2, 0x416d, 0x4171, 0x43d6, 0x180c, 0x46c4, 0x1efb, 0xbfad, 0x118b, 0xb211, 0x41cd, 0xba1f, 0xa45a, 0x400a, 0xbfc1, 0xba49, 0x4103, 0x4212, 0xb20b, 0x42d7, 0x42e0, 0x413a, 0xba68, 0x42d4, 0x4241, 0x41b2, 0x427f, 0x42e9, 0xbfa7, 0xb09c, 0x20ae, 0x3704, 0xb2ce, 0x1f73, 0xa18a, 0xbf80, 0x4266, 0xb0d8, 0x419a, 0xbf41, 0x4270, 0x1810, 0x4264, 0x39da, 0x1d65, 0x41a7, 0xa97c, 0x1288, 0xba61, 0x0e71, 0x425c, 0x41e5, 0x4542, 0x42fe, 0x4183, 0xb283, 0x401d, 0x00d1, 0x4232, 0x4246, 0x226e, 0x1918, 0x4299, 0x15ac, 0xbfad, 0x1ce1, 0x2a0a, 0x40a6, 0xb2fd, 0x3635, 0x1933, 0x412c, 0x1439, 0xa558, 0x4220, 0xbf33, 0x43c6, 0x416d, 0x26aa, 0xb213, 0x44e4, 0xbfe4, 0x29dc, 0x1d40, 0x42a2, 0x1ea9, 0x3f4b, 0x2e37, 0xb0e2, 0x0098, 0x427a, 0x4347, 0xbfe0, 0xb0a6, 0x053b, 0xb236, 0xba79, 0x444d, 0x429b, 0xbf94, 0xba27, 0xb2a0, 0x42aa, 0xb2e1, 0x43d0, 0xbf90, 0xba6b, 0xba4d, 0xa3ce, 0xb266, 0x43e6, 0x23cf, 0x422e, 0x4102, 0xaaf3, 0xa539, 0x419a, 0x137f, 0x45d8, 0x4067, 0xbfc2, 0x40f6, 0x416f, 0x08f8, 0x1f01, 0xb22b, 0x40a3, 0x40f9, 0x3a0e, 0x4370, 0xb2ad, 0x437a, 0x16d6, 0x0960, 0x1f6d, 0xa78d, 0x0385, 0xb2c2, 0x41cf, 0xb2f5, 0x1f83, 0xbaef, 0x4077, 0x1b8c, 0xba12, 0x4386, 0x40d0, 0xbf12, 0x40a8, 0xb237, 0x4237, 0xba4f, 0x4296, 0x45c8, 0x1f18, 0x1d76, 0x43ab, 0x3e32, 0x43f0, 0xbf72, 0x1a78, 0x1b64, 0x4028, 0x40cf, 0x43bb, 0x1f2c, 0x438f, 0x1fb8, 0x4282, 0x4669, 0x41d5, 0x1db5, 0xbafd, 0x41d4, 0x11c4, 0x405f, 0x1fb3, 0xb24b, 0x034f, 0x4345, 0x0970, 0x44fd, 0xb06b, 0xbf87, 0xa48d, 0x4492, 0xb095, 0x40b4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xec6fcb87, 0x37fe5171, 0x476101c9, 0xb4b6dc30, 0x961e2dee, 0xdd91c1a4, 0xedf2ecd8, 0xdc6fc622, 0x228a66a9, 0x3398789a, 0xa6476515, 0xda1e950b, 0x49ed8832, 0xb877df50, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x07fffffe, 0x0000285e, 0x00000000, 0x0000005e, 0x00001a14, 0x00000000, 0xffffffd3, 0x050bc000, 0x5622f36b, 0x00002e90, 0x5622f36b, 0xa6476511, 0xac45e6d6, 0x000041e6, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x42f8, 0x07a3, 0xba44, 0x42a4, 0xbff0, 0x43f8, 0x2211, 0xb240, 0x40ef, 0x3564, 0xbad8, 0x45a3, 0x41ca, 0xba31, 0x4163, 0xbf47, 0x0b13, 0x42a0, 0xb288, 0x4498, 0x40a7, 0x4388, 0x3a67, 0xba1f, 0x4109, 0x4267, 0xb250, 0x437a, 0x46d4, 0x1fc3, 0x44cb, 0x421f, 0x36f5, 0xba5b, 0x1805, 0x46f8, 0x39bb, 0x170b, 0xb245, 0x4671, 0x40a1, 0x4175, 0xbf8e, 0xaafd, 0x4046, 0x415e, 0x41ad, 0x4342, 0x43b2, 0x182a, 0x1fea, 0xbfce, 0xa26c, 0x4232, 0x4184, 0xbfcb, 0x421c, 0x4350, 0xbf00, 0x420a, 0x4073, 0x434b, 0x43c6, 0xba76, 0xba57, 0x43fa, 0x422a, 0xb0d1, 0x42ff, 0xb2fe, 0x4327, 0xbf52, 0x41ef, 0x0772, 0xb293, 0x4371, 0xb0d4, 0x3aea, 0x3a19, 0xbac1, 0x408c, 0x4177, 0xbf00, 0xa92b, 0xba79, 0x4317, 0x4231, 0x4139, 0xbae8, 0x4107, 0xba54, 0x43c4, 0x4066, 0x1a61, 0xba23, 0xb0b9, 0x41a5, 0x410a, 0x43f4, 0xb027, 0xbfbe, 0xb02e, 0xaa85, 0xa3ba, 0x1c41, 0x3ab9, 0x1700, 0x44db, 0x1e0f, 0x40fc, 0xbaeb, 0xb015, 0xbf5c, 0x433d, 0x3457, 0x408e, 0x443e, 0xa97f, 0x19f0, 0xbf6a, 0x2099, 0x41a3, 0x400d, 0x41b8, 0x42d7, 0x4049, 0x416f, 0x428f, 0xbad9, 0xbf65, 0xb066, 0xa894, 0x18db, 0x11e7, 0x4351, 0x4113, 0x4182, 0xbae8, 0x1ea4, 0x3b34, 0xbf8d, 0xba5c, 0x4198, 0x436b, 0x421d, 0xbaf3, 0x435b, 0x12bd, 0x40cd, 0x2ba7, 0x180e, 0x43a7, 0x409b, 0xba76, 0x20b6, 0xba6e, 0x22f8, 0x184a, 0x41c9, 0x2f18, 0x43bb, 0xb28a, 0x3e03, 0x42bd, 0x1873, 0x4076, 0xbf42, 0xbaf0, 0x40b4, 0x185e, 0xbfaa, 0xbf00, 0x14e1, 0x27ae, 0x16e9, 0x43d5, 0x412c, 0xbf8f, 0x1bc8, 0x4048, 0x403e, 0x176a, 0x4126, 0x469d, 0xba7f, 0x0dc0, 0x4363, 0x436d, 0x1848, 0x1944, 0x422e, 0xb273, 0xbf00, 0x1c50, 0xbfc2, 0xb2b1, 0x421b, 0xbfd0, 0x320b, 0x300e, 0x1cd6, 0xb2ae, 0x433e, 0xb2f0, 0x1f18, 0x2045, 0x3f7d, 0xb2ac, 0x31e0, 0x1882, 0x1c64, 0xb2e2, 0x1d30, 0xbf90, 0x4316, 0x4429, 0x3b53, 0xba3c, 0x1c65, 0x21a2, 0x4246, 0xb070, 0xbfc6, 0xa834, 0xb2c4, 0xa89e, 0xb24a, 0xb292, 0x1341, 0xb20c, 0xbad2, 0x40a1, 0x425e, 0x0227, 0x4294, 0x2458, 0x4278, 0x4130, 0xad67, 0xb2dc, 0x4112, 0x4329, 0xb037, 0xa54d, 0x031f, 0x4105, 0xbfca, 0x400f, 0x233b, 0x4108, 0x1a02, 0xbf44, 0xb0a7, 0x416d, 0xbfdb, 0xb2d8, 0x2ed4, 0x4046, 0x1bd0, 0x1e0e, 0x426a, 0xa1db, 0x1fb1, 0x40b9, 0x4304, 0x424a, 0x02d1, 0xb245, 0x4153, 0x41cb, 0xbf22, 0x43f8, 0x41c8, 0x18fc, 0xa2a2, 0x42cf, 0x411e, 0xbfc6, 0x41db, 0xb2ed, 0xae81, 0x43c9, 0xbaef, 0x40fa, 0x0fd7, 0x430f, 0x072e, 0x3c69, 0xa74f, 0xbfa0, 0xba11, 0x410c, 0x000b, 0x4028, 0x1c4d, 0xba4c, 0x0ca2, 0x0b17, 0x43bd, 0x0149, 0xbfa7, 0x1c43, 0xbae0, 0x4245, 0x14f5, 0xba01, 0x4231, 0x421e, 0x4390, 0xa298, 0x4200, 0x2a55, 0x416f, 0x1a4f, 0x42b7, 0x097a, 0x1d13, 0x2dc8, 0xb2f7, 0x3cbf, 0x4270, 0x1f05, 0xa2c0, 0x057e, 0x40bb, 0xbac7, 0x1fef, 0xbfc6, 0x2651, 0x1ef8, 0x41fb, 0x382d, 0x40b0, 0xa7ea, 0xb034, 0x439a, 0xba7d, 0x4387, 0xb2f0, 0x4319, 0xbad5, 0x4427, 0x19c9, 0xb27d, 0xbf70, 0x2c31, 0x4260, 0x098c, 0x40de, 0x4220, 0x418c, 0x435b, 0x4589, 0xb27d, 0xb290, 0x1d46, 0xbf13, 0xb21d, 0xb2b3, 0x42cc, 0xba42, 0x40c0, 0x02a9, 0x4345, 0x40f4, 0xbfe4, 0x436f, 0xb25e, 0x1f19, 0xba1a, 0x2f2c, 0xb23a, 0xb238, 0xb239, 0x436b, 0x4287, 0xbac9, 0xbaf9, 0x1b47, 0xb245, 0x4338, 0x0077, 0x4179, 0xbae7, 0x431d, 0x4108, 0xb06f, 0x418a, 0x4253, 0xb02a, 0xa84c, 0xbfc2, 0xba77, 0x1a6c, 0x4018, 0x2ed1, 0xabb2, 0x409b, 0xa523, 0x4335, 0x4319, 0x4250, 0xb256, 0xb228, 0xa371, 0x1b22, 0x4350, 0x4273, 0xb09d, 0x18a4, 0x4297, 0x0814, 0xbf58, 0x27b9, 0x22c1, 0xb053, 0xa1b1, 0x414c, 0x1e1a, 0xbfd2, 0x4246, 0x42af, 0x4340, 0x4069, 0x4236, 0x4334, 0x425c, 0x41e5, 0x43ff, 0x4349, 0x3950, 0x237f, 0xba38, 0x42f8, 0x40f8, 0x034e, 0xb2c9, 0xb255, 0xba46, 0x426d, 0xa40b, 0x1ed1, 0xbf19, 0xb26f, 0x158c, 0x43a4, 0x402a, 0xb205, 0xbaf6, 0x45db, 0x4167, 0xb28a, 0x218e, 0x189d, 0x1d70, 0x458c, 0x072f, 0x424e, 0x1eff, 0xbacd, 0xb2ca, 0x45c1, 0x0ef0, 0xac77, 0xbfb2, 0x4099, 0x3d38, 0xad0b, 0xbfa4, 0x07ea, 0x41cb, 0x4262, 0x428f, 0x117b, 0xbfc0, 0x33fb, 0xbfc0, 0x413d, 0x4414, 0x45c1, 0x15b1, 0x01b3, 0x419c, 0x4257, 0xbfc6, 0x1c88, 0x406f, 0xa0e7, 0x4068, 0x4292, 0xb295, 0x1e8d, 0x41e8, 0x3710, 0x42da, 0x4234, 0xb0b0, 0xb06b, 0xbf77, 0x0b97, 0x4060, 0xba38, 0x1cb0, 0xba36, 0x427b, 0x4281, 0x411f, 0xb04b, 0x4026, 0xa71a, 0xbf9f, 0x41a0, 0xba1c, 0xb289, 0xadd3, 0xba4b, 0x43d7, 0xb0c9, 0xbafb, 0xba0a, 0x414e, 0x401b, 0x2a82, 0x4300, 0x43a9, 0xbfdd, 0x40e4, 0x4327, 0x1c8d, 0x1c33, 0x4260, 0x43de, 0x4669, 0x433f, 0xb253, 0x1d24, 0x107c, 0xba13, 0x40f4, 0x43f5, 0xbf90, 0x438e, 0x1c5e, 0x41e1, 0x3f39, 0x1bd4, 0xbfd9, 0xb2c4, 0x418e, 0xb2e7, 0x15b8, 0xac3d, 0x2430, 0xb291, 0x42c9, 0x4003, 0xb2fd, 0x34aa, 0x4680, 0x2a30, 0x42f9, 0x43eb, 0xbf41, 0x41eb, 0x2f26, 0xaa5f, 0x42b7, 0x42e0, 0x3e1d, 0xba1c, 0x4261, 0x0676, 0xbfe8, 0x423c, 0x3bd6, 0xba24, 0xbf70, 0x4152, 0x228a, 0x1848, 0x120b, 0xb262, 0x4309, 0x402e, 0x4106, 0xa354, 0x200b, 0x415e, 0xbfd6, 0x407f, 0x4563, 0x41b5, 0x212e, 0xb0e2, 0xbfb6, 0xae17, 0xb021, 0x2a99, 0xb0cd, 0xa019, 0xb090, 0x42ad, 0x42ad, 0x15ca, 0x43d3, 0x43cf, 0xb2e3, 0x43dd, 0xb2be, 0xac16, 0x41bb, 0x249a, 0x42b0, 0x083b, 0x4435, 0x4349, 0x40b3, 0x463a, 0x1a4e, 0xba55, 0xbf19, 0x40fe, 0x4300, 0xba69, 0x4299, 0x1f04, 0x013a, 0xb276, 0x09e9, 0x0926, 0x402a, 0x46c9, 0x1243, 0x4347, 0x412e, 0xb26f, 0xb2d5, 0x4388, 0x41cb, 0xba1e, 0xbfa2, 0xbac4, 0x4254, 0xb2e2, 0x0ffb, 0x20b7, 0x40b7, 0xb2b1, 0x432e, 0x466a, 0x40b0, 0xb2a7, 0x43c7, 0x18b4, 0x4279, 0xb272, 0x41c0, 0x4561, 0x40bb, 0xb047, 0x1fcc, 0xa0e0, 0x437a, 0xbf36, 0x2ab2, 0xb202, 0xad4d, 0xbad7, 0x0427, 0x1995, 0x31b3, 0x1235, 0x1f89, 0x340f, 0x45b5, 0x4556, 0x1941, 0x4203, 0x4593, 0x4071, 0x40a0, 0x387d, 0x3c40, 0x15de, 0x4215, 0xb219, 0xbf5a, 0x16e9, 0x44ed, 0xa529, 0x1cd4, 0x428b, 0x414a, 0xb2d6, 0x4572, 0xbf41, 0x414d, 0x40e1, 0x1862, 0x44d9, 0x414a, 0x4292, 0x425e, 0xb061, 0x436c, 0x3ba1, 0xb247, 0x40cb, 0xbaed, 0x412e, 0xba4b, 0xb298, 0x431c, 0xbf36, 0x0bfe, 0x3111, 0xba26, 0x1fb8, 0x4215, 0x1841, 0x40b7, 0xb0f5, 0xbf80, 0x42f8, 0x08a3, 0x45ee, 0x40d2, 0xb2f2, 0x33d3, 0xbf0d, 0x4200, 0x4083, 0x42af, 0x1a3b, 0xbfd1, 0xb060, 0xb0a4, 0x3ab1, 0xb23d, 0xba06, 0x4296, 0xb29f, 0x421c, 0xb2bd, 0xbae4, 0x46fa, 0x467e, 0xb019, 0x40f0, 0xb216, 0xbf35, 0x107c, 0xb296, 0x42c3, 0x4418, 0xba0c, 0xb22f, 0x0196, 0x42bc, 0x27fa, 0xa5b8, 0xbf02, 0x407c, 0x2e4e, 0xb28c, 0x091b, 0x1c4e, 0x1a4e, 0x4320, 0x40f6, 0x419b, 0xbac2, 0x43ab, 0xba4e, 0x4331, 0xb2ba, 0x43aa, 0x437f, 0x3ace, 0x27c8, 0xb246, 0x265d, 0x1ceb, 0x2cc8, 0xbff0, 0x2612, 0xbf1b, 0x1fd7, 0x43b0, 0xa909, 0x418b, 0xae30, 0xbfa2, 0x4563, 0x37cc, 0x4396, 0x4303, 0x4129, 0x1d9b, 0xbfcd, 0xb228, 0x43d6, 0x417e, 0xb00c, 0x4142, 0xba08, 0x41b2, 0x4260, 0x4066, 0x4362, 0x4210, 0xbf81, 0x40a6, 0x3a50, 0x40ac, 0x305e, 0xba75, 0x41e3, 0xb09c, 0xbf5a, 0x42e9, 0xbae5, 0x1a1b, 0x418d, 0xa430, 0x41ae, 0x4265, 0x2f35, 0x0f5f, 0x41b7, 0xbf71, 0xb056, 0x3e3a, 0xb2b8, 0x4227, 0x16be, 0xb213, 0x3ea4, 0xbf0c, 0xa864, 0xb25c, 0x42e2, 0x4083, 0x23de, 0x07bc, 0x42ca, 0x42cb, 0x1a5f, 0x3d6f, 0xb266, 0x3a16, 0x43ba, 0x434f, 0x41db, 0xb0b8, 0x427d, 0xb0b7, 0xbfe1, 0x1e76, 0x400e, 0xba71, 0xa62e, 0xb23b, 0x4358, 0x18ae, 0xb292, 0x4254, 0xbfbb, 0xba59, 0xb28e, 0x33cb, 0x46aa, 0x4219, 0xa5a6, 0x1ed8, 0x42d0, 0x150b, 0x4214, 0x42ed, 0x2da7, 0xb22b, 0x0e19, 0x3532, 0x43f0, 0x42a1, 0x439e, 0x40b3, 0x4361, 0x2193, 0x4451, 0x3894, 0x2d7e, 0x43c2, 0x1f33, 0xb0d9, 0x4440, 0xbf39, 0x1af5, 0x3c98, 0x42b7, 0x2c63, 0x1bb5, 0x06ac, 0x415c, 0x4012, 0x0d2f, 0x43cf, 0x43d1, 0x4272, 0x411d, 0x1dbc, 0xa93d, 0x1f13, 0x4222, 0x1904, 0xb24e, 0x44b1, 0xa1e6, 0x4074, 0x1a21, 0x4046, 0x4095, 0x4047, 0xbf8a, 0xba15, 0xb0ae, 0xa64e, 0xb26f, 0x4387, 0x05f8, 0x4548, 0xba5d, 0xb227, 0x407e, 0x43e8, 0xb238, 0x1f65, 0x45a2, 0x40f5, 0xba4a, 0x35d1, 0x403f, 0x4637, 0xbf49, 0x1dc2, 0x128b, 0xa29c, 0x41c0, 0xba48, 0x1adb, 0xb286, 0xb250, 0xb0af, 0x4132, 0x43e3, 0x1895, 0xba7b, 0x169e, 0x113d, 0x4589, 0x1fa1, 0xbfb7, 0x412d, 0x43fe, 0xb232, 0x3c71, 0x2e4a, 0xa936, 0x417e, 0x410c, 0xa245, 0x0a3e, 0x16c3, 0x0a0b, 0x0823, 0x3518, 0xba3e, 0x4135, 0x45f0, 0x412c, 0x4217, 0x4068, 0x4376, 0xb232, 0x416d, 0x42a2, 0x46db, 0x40a5, 0xb269, 0xb070, 0xbfbf, 0x2ff5, 0x2a6b, 0xb2a0, 0x45be, 0x1ab5, 0x1e69, 0x0d07, 0x4018, 0x1fcb, 0x414a, 0xb065, 0x08e8, 0xbaff, 0xbad4, 0x4332, 0xa6fd, 0x441e, 0xba61, 0xbfc2, 0xba53, 0xa9e9, 0x4260, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x618c22bd, 0x5bef2a39, 0xee6c28df, 0x55cc73d5, 0xfa7df287, 0x5074f3f0, 0x2ce650c2, 0xead87231, 0x32e2b3c0, 0xe82f84ef, 0x182094bd, 0x98581b15, 0x94fdcdeb, 0xe3d0b1ff, 0x00000000, 0x800001f0 },
                FinalRegs = new uint[] { 0x00006f62, 0x6525d096, 0x68d99e90, 0xd968909e, 0xffff909e, 0x68da0000, 0x68da1bcc, 0xffffff0f, 0xb3350301, 0xe82f84f5, 0x000000df, 0x010f4008, 0x182094bd, 0x6525ccf2, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x40a3, 0x4088, 0x431e, 0x40a5, 0xb02c, 0xba79, 0xb24d, 0x40f5, 0x43df, 0x43c1, 0xaef2, 0x4008, 0x414b, 0x41fc, 0xb235, 0xbfc0, 0xafbb, 0x1c05, 0x196a, 0x40b4, 0xb072, 0x38d1, 0xbf43, 0x46b1, 0xa0e6, 0x45a4, 0x4043, 0xbf90, 0x4626, 0xb2fc, 0xbf7e, 0xba61, 0xa1ca, 0x2175, 0xaf95, 0x4312, 0x1d8a, 0x190e, 0xb2df, 0x1698, 0x2d52, 0x1a9a, 0xb0ed, 0x407f, 0xa460, 0x4254, 0xb066, 0xbf89, 0xb0a9, 0xb2b8, 0x1a5d, 0x4194, 0x4211, 0x41a0, 0x40e0, 0x4139, 0x4362, 0x4336, 0x42a4, 0x447b, 0xabbd, 0x443a, 0x4459, 0xaf35, 0xbfb2, 0x45cc, 0x1e31, 0x1b94, 0xb07a, 0xb29e, 0x4212, 0xbae6, 0x427e, 0xb034, 0x2b28, 0x400a, 0x3ca6, 0x43a4, 0xb04a, 0xb248, 0x409a, 0xab1f, 0x4278, 0xb04b, 0x407f, 0x4291, 0xbf75, 0x415f, 0x4087, 0xb298, 0x41d4, 0x423e, 0x1b3f, 0x4206, 0xba6d, 0xb2d7, 0x458c, 0x319d, 0x4222, 0x40c1, 0x43ac, 0x4340, 0x1308, 0x4011, 0x4118, 0x40b3, 0x4379, 0xba4b, 0xb23e, 0xbf74, 0x40b5, 0x3319, 0xb07b, 0x41cb, 0xb2f9, 0x431d, 0x02c4, 0x1996, 0x419a, 0x435b, 0x255c, 0xb2f5, 0x33ad, 0xb04a, 0x4408, 0x0974, 0x42f7, 0x4339, 0xb280, 0x1b82, 0xb247, 0x464c, 0x4211, 0xbf0b, 0x4030, 0x43e6, 0x4087, 0x3611, 0x0981, 0x416e, 0x4315, 0x43c1, 0x2147, 0x4090, 0x4136, 0xbf66, 0x2933, 0x2ecf, 0x43eb, 0xb283, 0x42af, 0x1c96, 0xb09f, 0x41f2, 0xb21c, 0xbf09, 0xb087, 0x4463, 0xbaf1, 0x419d, 0xa7b0, 0xb211, 0x41a6, 0x3f86, 0xba54, 0x3385, 0x03b0, 0xb211, 0x4331, 0xaf96, 0x4119, 0x0c9e, 0x419b, 0xb2dc, 0x43d0, 0xb24c, 0x4042, 0xbf00, 0x4162, 0x1fc3, 0x4189, 0xb25a, 0x06f9, 0xb274, 0xbf6c, 0xbacf, 0x407e, 0x4204, 0x140f, 0x4309, 0x4176, 0x407e, 0x4130, 0x1c8a, 0x4354, 0x2411, 0x2605, 0xb057, 0xbf70, 0x406b, 0x3cd9, 0x437f, 0xa2e1, 0xba57, 0x069e, 0xbf76, 0x46ec, 0xbaf9, 0xb2ae, 0x436a, 0xadd9, 0xb09d, 0x40db, 0x4244, 0x1a39, 0xac08, 0x408d, 0x4021, 0x4263, 0xb281, 0x4418, 0x4116, 0x42f6, 0xa919, 0x1470, 0x1f33, 0x4245, 0x434f, 0x43ff, 0x43a1, 0x0d92, 0x4051, 0x43af, 0x30c5, 0xbf46, 0x425e, 0x4008, 0x4678, 0x42db, 0x1c2c, 0x4410, 0x1c4e, 0x4282, 0xba0d, 0x4075, 0x194f, 0x41ab, 0xb065, 0x1871, 0x412f, 0x0399, 0xa0db, 0xb239, 0x4559, 0x43d6, 0x1faf, 0x411f, 0xb0b7, 0x4606, 0x43c2, 0xbf81, 0x0823, 0x2a43, 0x18c5, 0x1a77, 0xad27, 0x43ff, 0x4212, 0x0e3c, 0x411a, 0x43a6, 0x1916, 0x4394, 0xb225, 0x1eb5, 0x449b, 0x438e, 0x40ca, 0x4623, 0xb2e3, 0x41ef, 0x4493, 0x3e02, 0x4069, 0x1c2b, 0xb234, 0x1e69, 0x41df, 0xb2b8, 0x41d8, 0xbf56, 0x05ad, 0xba1c, 0x42c2, 0x4051, 0x42b2, 0x40f4, 0x43ce, 0x44f0, 0x43f3, 0x4229, 0x213e, 0xb2c2, 0x3200, 0x4214, 0x42b6, 0x005f, 0x4075, 0xb272, 0xbaf8, 0xb224, 0xb29f, 0xbfd5, 0x40c1, 0x438e, 0x43c4, 0x0aa3, 0xa689, 0xb22f, 0x2428, 0x45bd, 0x4004, 0x43f9, 0xb29e, 0x4679, 0xb21b, 0x40bb, 0x4027, 0x40f4, 0x4074, 0xbf7b, 0x45c2, 0x43f6, 0xb2a0, 0x416c, 0xba07, 0x1da5, 0x416e, 0xba1c, 0x406a, 0x0523, 0xa9b7, 0x352b, 0x35c0, 0xbf97, 0xb26c, 0xbf60, 0x40f8, 0xba7a, 0x423f, 0x02f2, 0x4068, 0x3e52, 0x42c7, 0x4050, 0x1d10, 0x4211, 0x42d9, 0x4565, 0xb2a5, 0x416d, 0xba27, 0x420a, 0xb228, 0xbf80, 0xa3ea, 0x437b, 0xb2cf, 0xb2cb, 0x1473, 0xbf7c, 0x4285, 0x41b6, 0x4280, 0x3a9f, 0xba06, 0xba22, 0x3f7e, 0x1822, 0x1663, 0xbf80, 0x182b, 0x1d47, 0x37c4, 0xaa51, 0x1efc, 0xbfb7, 0x43b7, 0xba12, 0xbaf0, 0x40e9, 0xbf80, 0xb09e, 0x4448, 0x1f27, 0x4083, 0xac6a, 0x400e, 0x1c32, 0xba54, 0xba14, 0x249e, 0x3f80, 0x4610, 0x4293, 0x4396, 0x1e57, 0xb23f, 0xb093, 0x4161, 0x42c0, 0x3494, 0x40d5, 0xbf56, 0x1ee5, 0x2143, 0x18b0, 0x4360, 0x46a4, 0x1930, 0xba5a, 0x44d0, 0x1ad6, 0x371e, 0xb2ce, 0x1534, 0xb09c, 0x1b0b, 0x095d, 0x2f9f, 0x1608, 0x40f7, 0xbada, 0x05c1, 0xbf3e, 0xa521, 0xb29a, 0x42af, 0x404a, 0x0f8c, 0x41c9, 0xb089, 0x1a99, 0xba0a, 0x1fc9, 0xb25b, 0x41a2, 0x4109, 0x40b7, 0x364f, 0x2648, 0x011f, 0x4157, 0xbf47, 0xb037, 0x1b4a, 0x4096, 0x41a8, 0x4246, 0xba0f, 0x40f0, 0xb250, 0xbf59, 0x42b7, 0xb251, 0xba5b, 0xb0d1, 0x4208, 0x4103, 0x40a7, 0x4355, 0xb216, 0x4476, 0x1962, 0xb03d, 0xba1a, 0x1928, 0x1f0b, 0xbf71, 0x2d76, 0x40f6, 0x4369, 0x29d5, 0x42f6, 0x1dd5, 0xb25e, 0xaf1a, 0x42fd, 0x40b9, 0x41bb, 0xa259, 0x40b7, 0x45f6, 0x4283, 0xbf3e, 0xba6c, 0xbae8, 0x41cb, 0xbf08, 0x42f3, 0x15e3, 0x1eb6, 0x386c, 0xb090, 0x1d6c, 0x3160, 0x43ca, 0x1a41, 0xb260, 0x4252, 0x3a39, 0xae9f, 0x41cf, 0xb2ae, 0x40ed, 0x4347, 0xb2ae, 0xba31, 0xbfda, 0x43a6, 0x4125, 0xb036, 0x408f, 0xba00, 0x1ddf, 0x4062, 0xb0ba, 0x4176, 0xba23, 0x2c17, 0x42e6, 0x3bc2, 0x43fb, 0x11b5, 0x400e, 0x439d, 0x3079, 0xb0a7, 0xbf7b, 0xb0e1, 0x4271, 0x44d1, 0xbfb0, 0xb028, 0xb015, 0x43d1, 0x4322, 0x1dcf, 0x42cc, 0xbac0, 0x1847, 0xbfd1, 0xb036, 0xb060, 0x2bea, 0x4020, 0x4193, 0x43d5, 0xaa64, 0x41c7, 0x427c, 0x41ec, 0xbf0b, 0x4398, 0x4200, 0x4283, 0x4000, 0x1f81, 0x44d1, 0x42d6, 0x402c, 0x0e29, 0x4606, 0xb0b9, 0x4093, 0x40fa, 0x42b6, 0xba48, 0x43a9, 0xbf72, 0x1aba, 0xbf00, 0x369e, 0xbf00, 0x4109, 0x251e, 0xba6f, 0x40bc, 0x4464, 0x40d4, 0x3776, 0xb2de, 0xb01a, 0xb05d, 0x27ed, 0xb252, 0x368f, 0xb044, 0x45e0, 0xba62, 0x4135, 0xbf23, 0x4188, 0x41f4, 0xb223, 0x418e, 0x40e4, 0x4256, 0xbf01, 0x411b, 0x1f16, 0x4253, 0xbadd, 0x43e0, 0xaeab, 0x4431, 0x24cb, 0x30b8, 0x406e, 0xb0e9, 0x41aa, 0x43a6, 0x438e, 0x4342, 0x4307, 0x419c, 0x4053, 0x4311, 0x41c7, 0x4152, 0x1ba5, 0x41db, 0x42a0, 0x40b8, 0xbf2e, 0xb01c, 0x423b, 0x1d13, 0xb262, 0xba11, 0x1803, 0x433e, 0x287b, 0x1977, 0xb0ba, 0x297d, 0xbf01, 0xb294, 0x40bf, 0x4391, 0x43e8, 0x232d, 0x431a, 0x4228, 0x375a, 0x36d7, 0x424d, 0xbf60, 0x4071, 0x40e8, 0x3242, 0xb0fe, 0x4214, 0x41a3, 0xbf62, 0x2756, 0x40f0, 0xba17, 0x3696, 0x427f, 0xaaa1, 0xbfa9, 0x4285, 0x427c, 0xa783, 0x4369, 0x274e, 0xb299, 0x411c, 0x40f0, 0x073b, 0x40d4, 0x43f7, 0xba1b, 0xb22d, 0x1c83, 0x1f11, 0x4277, 0x32b1, 0xbf52, 0x424b, 0x438f, 0xbfd0, 0x4118, 0x39e7, 0xba37, 0x3e6d, 0x41f3, 0x4239, 0x2daa, 0xbfa6, 0x432e, 0xb015, 0xb2f6, 0xbaf2, 0x4184, 0x4324, 0x1c78, 0xbaf1, 0xba29, 0xbfca, 0x1b4d, 0xb28b, 0x4577, 0x3ade, 0x43b3, 0xbf64, 0x401f, 0x41d7, 0x4060, 0x1be9, 0x4453, 0xba44, 0x37d4, 0x424a, 0xb006, 0x1882, 0xbac4, 0xba5b, 0x3c59, 0x0c92, 0x42a0, 0xba4e, 0x2d9b, 0xbfe4, 0x441b, 0x2199, 0x05fd, 0x43b2, 0xba42, 0x4301, 0x1ec4, 0x4656, 0xb0e4, 0xb06e, 0xbf82, 0x40a6, 0xaebe, 0x30ca, 0x4181, 0x42a4, 0xb24d, 0x4080, 0x43ca, 0x3592, 0x1943, 0x4217, 0x43ee, 0x08c3, 0xb003, 0x2cbd, 0x4412, 0x2e2f, 0x26db, 0x4179, 0x42a4, 0x36d3, 0xbfb2, 0x4386, 0x431c, 0x44ba, 0x4464, 0xb281, 0x40b3, 0x4152, 0xb25b, 0x41f2, 0xb059, 0xb278, 0x0dfe, 0x425f, 0x4215, 0x3ca0, 0x4234, 0xbad2, 0x41f0, 0x3f2d, 0x43ce, 0x4218, 0xbf12, 0x45e9, 0x1a54, 0x1fff, 0xacd6, 0x410e, 0xbf8f, 0xb0e1, 0x352d, 0x1fed, 0x1af4, 0x428e, 0x3c91, 0xb039, 0x4079, 0x4347, 0x3c04, 0x4424, 0x4260, 0xa798, 0x41b9, 0x4033, 0x4238, 0x2419, 0x4227, 0x0d53, 0xb2c3, 0x42d7, 0xbf14, 0x46a5, 0x0852, 0xa387, 0x13d9, 0x41c7, 0x1f38, 0x43e8, 0x1dcc, 0x43e6, 0x465a, 0x419a, 0x4387, 0x4338, 0x43a0, 0x033e, 0xba5d, 0x4176, 0x3d63, 0x3285, 0x1d0b, 0xacee, 0x4105, 0x44e8, 0x3c90, 0xa4a5, 0xbf17, 0x432e, 0x4020, 0x4367, 0x43ee, 0x43f8, 0x3f58, 0x1f04, 0x08fa, 0x43cf, 0xb097, 0xba07, 0xb228, 0x45c4, 0x09ae, 0xbf31, 0x31a7, 0x3a09, 0x42cf, 0x4019, 0x2390, 0x4145, 0x4179, 0x40bd, 0xb243, 0xbf5d, 0x3f0c, 0x3406, 0x1f8a, 0xba3b, 0xbfb5, 0x43b0, 0x3120, 0xb26c, 0xa3b0, 0x4272, 0x465d, 0x3214, 0xba0e, 0x3a60, 0x4230, 0x2d1c, 0x43ea, 0xbf63, 0x43ae, 0x14fa, 0x38d6, 0x423d, 0xac84, 0xb210, 0x402e, 0xb273, 0x441e, 0x43b0, 0xab4e, 0xb275, 0x42c8, 0x41f1, 0xb2ba, 0x42fd, 0x40b9, 0x4244, 0x42ed, 0x420d, 0x4190, 0xba13, 0xba1f, 0x2758, 0x40d7, 0x1012, 0x41f7, 0xbff0, 0xbfaf, 0x1aee, 0x425b, 0x0e1a, 0x3ddb, 0xb2e9, 0xba50, 0x3fd8, 0x1ba9, 0xb002, 0x437d, 0x4279, 0x34fb, 0xba3b, 0x43e3, 0x4185, 0xba1f, 0xa0db, 0x4218, 0x3d58, 0x0f00, 0xb2b0, 0x429f, 0xb063, 0xbf0d, 0xba52, 0xba07, 0xa5c8, 0x409c, 0x431f, 0xb2f9, 0x1ff7, 0xb0e6, 0x463f, 0x3ac2, 0xba14, 0xba7c, 0xbf43, 0xbff0, 0x40b3, 0x2e64, 0xa3a0, 0x1bac, 0xb2c7, 0x1f9a, 0x31d1, 0x387c, 0x1993, 0x4224, 0xba20, 0x4044, 0xb2f2, 0x1983, 0x428f, 0x11fc, 0x00a5, 0xb071, 0x12ae, 0x4007, 0x15cc, 0xbfcf, 0x416e, 0x0667, 0x1b46, 0xb250, 0x4285, 0x41ea, 0xa38d, 0x43dd, 0x43a7, 0xa77b, 0x0711, 0xba66, 0x445c, 0x1b40, 0x4095, 0x41b8, 0x400b, 0x4079, 0x3567, 0x43a6, 0x461d, 0x435d, 0x4036, 0x42fa, 0xa9a6, 0x43b7, 0xb295, 0x427e, 0x4300, 0xbf54, 0xb0f4, 0x2b0b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x36179bbd, 0x4c84b624, 0xf2f6b902, 0xcc71b138, 0x0afbc809, 0xac7ce4df, 0xb01ce320, 0xe9d87f98, 0xc61c700a, 0x6f44ba87, 0x76d17376, 0x6c662875, 0x88e99f9f, 0x5ec4aa7f, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0xffffffeb, 0x00000415, 0x6000000a, 0x00000000, 0xf16324f3, 0x0000000a, 0xffffe660, 0x000019a0, 0x3cede399, 0xd596226d, 0xe4d0754a, 0xf16324f3, 0x00000132, 0x0000017d, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4194, 0x41e1, 0x426d, 0xabea, 0xa7e7, 0x181f, 0x4209, 0x38b4, 0x4330, 0x42fd, 0x40db, 0xbfb4, 0x1903, 0x4570, 0xba65, 0xa18b, 0x1868, 0x41bc, 0xb269, 0x3a0f, 0xb2ef, 0x43d0, 0x301c, 0x1b59, 0x4185, 0xa33d, 0x43c3, 0x448a, 0xa735, 0x1c22, 0x4273, 0x4584, 0xb215, 0xbfb5, 0xb059, 0x43ed, 0x412b, 0x1c26, 0xb0cb, 0xbae5, 0x40ca, 0xb28a, 0x19c5, 0x4120, 0x4274, 0x4594, 0xba53, 0x42c7, 0xbf71, 0x4182, 0x43e9, 0x412a, 0x28e9, 0x4057, 0x4310, 0x41a1, 0x40cd, 0xbfac, 0xba50, 0xaa8a, 0x26a9, 0x432c, 0x46b1, 0x454c, 0x1542, 0x4222, 0x02f8, 0xb2f9, 0x4014, 0xb23d, 0xbf55, 0xbfd0, 0x404a, 0x3680, 0x40a4, 0x4275, 0x42b9, 0x4249, 0xbfa7, 0x2da6, 0xa070, 0x4377, 0x10a2, 0x411c, 0x1f50, 0x466a, 0xa9fd, 0x237a, 0x43e4, 0xb2b7, 0x4457, 0x4157, 0x2ec4, 0x4606, 0x41f1, 0x4432, 0x1376, 0x4143, 0x4663, 0x43ae, 0xbf8a, 0x426f, 0xb299, 0x422f, 0x1c31, 0x415c, 0x39de, 0xbf60, 0x25cc, 0x4097, 0xb24e, 0xbac7, 0xb2c8, 0x4030, 0x43f2, 0x41af, 0x4012, 0xb2d2, 0x4415, 0xbf9e, 0x4208, 0xba30, 0x1d45, 0x0615, 0xb2d8, 0xba11, 0x40f9, 0x4011, 0x42c7, 0xb262, 0x41df, 0x429f, 0x0b94, 0x1fbf, 0xbf94, 0xb2c3, 0x4073, 0x40e2, 0x4166, 0xa2e0, 0x4475, 0x4026, 0xb0a2, 0xb20a, 0x1c3b, 0xa334, 0xa04c, 0xaae3, 0xbf85, 0xba51, 0x42c5, 0xa2df, 0x43c6, 0x42f2, 0x4252, 0x43bb, 0xbf96, 0xa5a2, 0x41ae, 0xa663, 0x11de, 0xbaeb, 0x1b6f, 0x42f3, 0xba22, 0xbfe1, 0x438c, 0xba76, 0xb2e7, 0xa84c, 0x4389, 0x14e8, 0x316f, 0x1e56, 0xa1ea, 0x4270, 0x0962, 0x1f5d, 0x40f4, 0xbf69, 0xbae7, 0x4019, 0x1c1e, 0x43fe, 0x4245, 0x1a32, 0x418f, 0x10f5, 0x1a47, 0x1d6f, 0xa720, 0xb0e0, 0x43ae, 0x0ac4, 0x4232, 0x4044, 0x42fe, 0xbaf8, 0xba6b, 0x19ef, 0x408b, 0x4287, 0x41a5, 0xbfc6, 0x1d24, 0xb2f7, 0xb291, 0xaf34, 0x40f3, 0x1a78, 0xba54, 0x08f7, 0x19b6, 0x3b08, 0xba23, 0x318f, 0x1fe9, 0x4646, 0xbf41, 0x1b4a, 0x4224, 0xb201, 0x221a, 0x4027, 0xb09c, 0xbada, 0x438d, 0xa29c, 0xba71, 0x43ed, 0x40ee, 0x408a, 0x4366, 0xbf02, 0xb2f8, 0xb09a, 0x35f4, 0xb2db, 0x0d06, 0xba58, 0x42c8, 0x4374, 0x40f9, 0x42c4, 0xbaee, 0x4014, 0xba77, 0x4330, 0xb0f9, 0x43b5, 0x3ee4, 0xba5c, 0xa37c, 0x18f9, 0xba01, 0x43e4, 0x45ad, 0x3b98, 0x431c, 0xae45, 0xb0e9, 0xba6a, 0xbf3f, 0xb093, 0x40cf, 0x31f1, 0xbae3, 0x18e3, 0xafbb, 0x46fb, 0x4467, 0x4377, 0x4439, 0x46fd, 0x01e6, 0x43ca, 0x423d, 0xbf80, 0xb2f8, 0x439c, 0x4158, 0xbf9e, 0x4343, 0x433d, 0x41a2, 0x2ced, 0xbae8, 0x1b42, 0x3e7d, 0x41ba, 0x4022, 0x43a7, 0x424b, 0x4253, 0xb0e9, 0x42b3, 0x1d29, 0x2454, 0x412b, 0x4253, 0x4156, 0xbfd3, 0x423a, 0x400b, 0xba35, 0x1cf9, 0x00db, 0x04bb, 0x1b4c, 0xbafc, 0x1fb0, 0xb034, 0xb2a7, 0x4328, 0x4248, 0xbf67, 0x1bc5, 0x42c7, 0x157d, 0xb206, 0x4040, 0xa039, 0xaf24, 0x41b8, 0xa314, 0x2db8, 0x1a50, 0xb2df, 0x35ca, 0x423d, 0xb23f, 0x41ee, 0x456c, 0x4365, 0xb2d6, 0x421a, 0xba7d, 0x43be, 0x43a1, 0xbf39, 0x2460, 0xb271, 0x2223, 0x3e58, 0x1ac1, 0x4225, 0x422b, 0xb0af, 0xb2df, 0xba76, 0x4085, 0x4226, 0x1f02, 0xb2a0, 0x41a6, 0xa49b, 0xbfb0, 0x4097, 0x4145, 0xbf82, 0x414e, 0xb240, 0x246d, 0xba4b, 0xaca3, 0x35f0, 0x42c9, 0x3d4b, 0x46db, 0x43fe, 0xba60, 0x1736, 0x100c, 0xbaf4, 0xa17b, 0x42ae, 0xbfdd, 0x0867, 0xb0b5, 0x4236, 0x420b, 0x4256, 0x4166, 0x4653, 0x4372, 0x1a16, 0xa649, 0x43e3, 0x42a6, 0xbf5f, 0x3fd3, 0x31db, 0x41fa, 0x2e10, 0x3186, 0xa1ae, 0xb012, 0xb2f5, 0x3260, 0x44a2, 0x43dc, 0x05d4, 0x41a8, 0xb030, 0x443b, 0x2d52, 0xba2f, 0x44f2, 0x4200, 0xba05, 0x42f4, 0x431b, 0x1ae3, 0xb2db, 0x4405, 0xbaeb, 0x3772, 0x423d, 0xbf1f, 0x20f7, 0x40be, 0x1b66, 0x1b9f, 0x439e, 0x0d71, 0xa52b, 0x1fd0, 0x2322, 0xbff0, 0xac82, 0xbfa0, 0x274e, 0x4079, 0xafe0, 0x1f38, 0x4235, 0x1de4, 0x403d, 0x434b, 0x343f, 0xba6d, 0xba07, 0xbf02, 0x439f, 0x46c1, 0xba20, 0xb206, 0x4127, 0xba1f, 0x4226, 0xb295, 0x436b, 0xb2aa, 0xa2b9, 0xba5e, 0x423f, 0xb008, 0xb00f, 0xb081, 0x0a56, 0xbfa3, 0x3f44, 0xb0a9, 0x4022, 0x244d, 0xb07d, 0xba2b, 0xbf90, 0xbfa0, 0x41a1, 0xba26, 0x2997, 0xa317, 0xb2ce, 0x410e, 0xba51, 0x40a8, 0x4285, 0x20bd, 0x411d, 0x1eaa, 0xbfcd, 0x4027, 0x4032, 0x42f8, 0x286a, 0xb212, 0x4191, 0x184a, 0x43a3, 0x4097, 0xba16, 0x369b, 0x4336, 0x46e2, 0xa6ed, 0x41f2, 0x0322, 0xb2ab, 0xbf48, 0x4139, 0x4077, 0x4096, 0x1da1, 0x4266, 0x4365, 0xb051, 0x413b, 0x45cb, 0x1533, 0x425c, 0x423e, 0xba34, 0x4679, 0x3ad5, 0x439b, 0x420d, 0x4228, 0x4406, 0xbf42, 0x4211, 0x4272, 0xb2a1, 0xb272, 0xb2d0, 0xa24b, 0xb0b4, 0x36f9, 0x1c6f, 0x405c, 0x36ad, 0x411f, 0x41d2, 0x426f, 0xb2f9, 0x29f9, 0x4572, 0x43cf, 0x4016, 0x4068, 0x4262, 0x151a, 0xbfa5, 0xafd2, 0x4382, 0x4308, 0x46e4, 0x43cb, 0x18af, 0x428e, 0xac0f, 0x1f7c, 0xbfc1, 0x425a, 0x424e, 0x120e, 0x43fa, 0x0332, 0x19c4, 0x4044, 0xbf0a, 0xbac9, 0x435b, 0x4255, 0xbf92, 0x41b7, 0x42a9, 0x08e3, 0x4114, 0x42ad, 0xb27b, 0x1ab9, 0x0adf, 0xbaed, 0x40b7, 0x307f, 0x4457, 0x40f1, 0x4296, 0xb218, 0xb21d, 0xb2d0, 0x4054, 0x40b4, 0x41ee, 0x45d1, 0x1f57, 0x011c, 0x4174, 0xbf02, 0x417a, 0x4124, 0x2931, 0x4021, 0xbac1, 0xb2ac, 0x4034, 0x4138, 0x44a1, 0x41a5, 0x4034, 0x30f3, 0x1952, 0xb003, 0x3fb2, 0x4624, 0xb04f, 0x3f0f, 0xbf54, 0xa9c0, 0x43af, 0xb2b0, 0xb0f0, 0xb2d8, 0x4288, 0x17b6, 0x2e75, 0x416f, 0x1f51, 0x1a93, 0xbfe4, 0x410c, 0x4465, 0x407b, 0x426d, 0x46a2, 0x37d7, 0xbfc7, 0xb2ee, 0x09c2, 0x41b0, 0x41a5, 0xba50, 0x4081, 0x459d, 0x4437, 0x433e, 0x1473, 0x435d, 0xbf00, 0x43fa, 0x1863, 0xb021, 0x4585, 0x4096, 0x1b2a, 0xbfb2, 0x067e, 0x18e2, 0xb20f, 0x43af, 0x4080, 0x1b32, 0xb29e, 0x0a11, 0x434a, 0x4392, 0x424e, 0xb2e3, 0x40a6, 0xab7a, 0x09d9, 0x1fed, 0x43c1, 0x19d4, 0x4089, 0xb236, 0x1f21, 0xb2e0, 0x4289, 0x461d, 0xaaf9, 0xbf3c, 0x1183, 0x0293, 0xa59b, 0x43bc, 0xb047, 0x403e, 0xb26a, 0xbfd2, 0x11e9, 0x4672, 0xbfa0, 0x414f, 0x39c9, 0xb2e5, 0x43a6, 0x43f4, 0xa04b, 0xa4bb, 0x4359, 0x4297, 0x41cb, 0x44e3, 0x2f11, 0xb237, 0x1971, 0x1c47, 0xbf3a, 0x424d, 0x448d, 0xb211, 0x412c, 0x41c1, 0x41e1, 0x41cc, 0x4265, 0xb08b, 0x4224, 0xba4b, 0xb240, 0xba27, 0xba7b, 0xbaca, 0xbfac, 0xb27c, 0x1d89, 0x43be, 0x1f77, 0xbf00, 0x43ea, 0x4326, 0xba22, 0x0315, 0xbf02, 0x42a2, 0x405a, 0xba05, 0x4152, 0xbfb0, 0xba4c, 0x42c6, 0x4334, 0x18b9, 0x4250, 0x40c8, 0xbf77, 0x33a6, 0xa4d7, 0xba55, 0x4361, 0x274f, 0xba03, 0x1ae7, 0x4038, 0xbf8d, 0x40fe, 0x1b4b, 0x4063, 0x40b7, 0xb2c8, 0x40ce, 0x38c1, 0xba40, 0x410f, 0x4445, 0x33b7, 0xba40, 0x4378, 0x3c3e, 0x4286, 0x41b1, 0xb219, 0xb23e, 0x437b, 0x2d6d, 0xbf9f, 0x1a09, 0xb00d, 0x436b, 0x41a5, 0x2188, 0xb29e, 0xb02f, 0x4255, 0x40ff, 0x4095, 0x4226, 0x1baf, 0x46ad, 0x262a, 0xa6a5, 0x1bf9, 0x41f0, 0x42fb, 0x43b9, 0xbf4a, 0x1d63, 0x1a48, 0x3668, 0x2ad8, 0xa836, 0xb0ad, 0xb205, 0xbf96, 0x1e1d, 0x27ba, 0x4677, 0xbac0, 0xa376, 0x40be, 0xbf05, 0xb2e9, 0x4164, 0x3e9d, 0x43b9, 0x4330, 0x1136, 0xba46, 0x41ae, 0x41c7, 0xb20f, 0xbafd, 0x42a2, 0x4007, 0xa269, 0x434e, 0xac40, 0x42da, 0x4553, 0xb2f3, 0xbfce, 0x433f, 0xa7f6, 0x4294, 0xb2e6, 0x41f3, 0x4355, 0x10e0, 0xb226, 0xb239, 0x1a8c, 0xb014, 0x431c, 0x1391, 0xba2d, 0x41ba, 0x402f, 0xb025, 0xb2bb, 0x35c5, 0xa0c6, 0x4042, 0xb27b, 0x4120, 0x250f, 0x4272, 0xbf32, 0x1a7c, 0x1e48, 0xbaf6, 0x1f12, 0x432a, 0x4608, 0x40f1, 0x1b8e, 0x1e05, 0x4021, 0x4053, 0xbf48, 0x41aa, 0xb2e4, 0xb230, 0x43ea, 0x1f4c, 0xa125, 0x1ef8, 0x4139, 0x4376, 0xbf6d, 0x42db, 0x32da, 0x4222, 0x2930, 0x2189, 0x4116, 0xb077, 0x429a, 0x40db, 0x45ac, 0x1a86, 0x1546, 0xb21d, 0x463a, 0x4062, 0xbf3e, 0x1a52, 0x407c, 0xba38, 0x454e, 0xa28b, 0x42a2, 0x3760, 0x1e45, 0x19ab, 0x292e, 0x0f80, 0x1d7e, 0x4572, 0x19be, 0x4318, 0xb0e2, 0x2ead, 0xbf15, 0x43d3, 0xb297, 0x4156, 0x4274, 0x3215, 0x43d5, 0x4101, 0x4009, 0x14bb, 0x4217, 0xb228, 0xba72, 0x1a16, 0x41af, 0x40c8, 0x4314, 0xb22a, 0x43a4, 0xb212, 0x4139, 0xb23d, 0x1b80, 0x1b4f, 0xb064, 0xbfb1, 0xb0ea, 0x419e, 0xa853, 0x406b, 0x41ba, 0xbfbd, 0x4353, 0x4018, 0x45e1, 0xba2b, 0xb264, 0x42e9, 0xb2e9, 0x40c2, 0x395e, 0x437f, 0x43f4, 0x406e, 0x0807, 0xba3e, 0xbf88, 0x430a, 0x43af, 0x4417, 0x426d, 0x410c, 0xbff0, 0x4683, 0xb25f, 0xbf00, 0xbf5c, 0xba47, 0x4033, 0x1942, 0x43e0, 0x12e6, 0x3c46, 0x436c, 0xba3c, 0x0145, 0x43cc, 0xb2d6, 0x4477, 0xb2d5, 0xb0d1, 0x4189, 0x43eb, 0x1ac6, 0xb258, 0xb0b6, 0xbf57, 0xbf60, 0x4285, 0x4364, 0x416f, 0x4111, 0x0d56, 0xb286, 0x4066, 0xb2dd, 0x4156, 0x4217, 0xa9c0, 0x43c6, 0xaf85, 0xb258, 0x41cb, 0xbfc4, 0x4220, 0x43bc, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5163634f, 0x69953bc6, 0xc471194b, 0xd797bd76, 0xca82e30c, 0xd0e3d0b1, 0x315d5946, 0x74209518, 0xbfc706cf, 0x907cff0b, 0x9e3f1e57, 0x892343bb, 0x37dc0fed, 0xbc49e0f0, 0x00000000, 0xa00001f0 },
                FinalRegs = new uint[] { 0x00000025, 0x9eb0014e, 0xfffec9da, 0xfc97ffff, 0x00000e10, 0x00000025, 0xffffffda, 0x9eb00062, 0xbfc706cf, 0x000000ad, 0x00000000, 0xfffee373, 0x37dc0fed, 0x9eaffe4e, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbacf, 0xbfa6, 0xb202, 0x40b6, 0x4165, 0x42b2, 0xb299, 0x40d5, 0x4183, 0xb09b, 0x4018, 0xb2a2, 0x4123, 0x0034, 0x45b6, 0x05e6, 0xaadf, 0x400e, 0x44da, 0xb220, 0xbfc8, 0x4147, 0x089b, 0x40e1, 0xa64f, 0xac9d, 0xb2cc, 0x39d6, 0xbf46, 0x42c9, 0x4314, 0x2b9a, 0x42cc, 0x19d2, 0x187f, 0x3b09, 0xba5f, 0xb2ef, 0x4281, 0x4298, 0x4253, 0xb284, 0x40ef, 0x4148, 0xaa53, 0x2d5e, 0x4301, 0xbf49, 0x4365, 0x3271, 0xb278, 0x10ca, 0x3100, 0x4298, 0xbfaa, 0xb27e, 0x46b9, 0x1972, 0xb08a, 0x428c, 0x4398, 0x1610, 0x4160, 0x40e9, 0x403b, 0x0964, 0xbf0d, 0x40e6, 0x40b1, 0x4183, 0x43af, 0x4195, 0x3dda, 0xa5d0, 0x2ba7, 0xb0d0, 0x1486, 0x4312, 0x42c5, 0xa382, 0x016e, 0xbf5c, 0x428b, 0xb2f2, 0x4367, 0xbf27, 0xba7b, 0x4676, 0x4229, 0x41f9, 0xb0b1, 0x4371, 0x40e3, 0x41f3, 0x1d96, 0x4019, 0xbad5, 0x0714, 0xbaf8, 0x3df6, 0x1b08, 0xb2ad, 0x32ce, 0xb26b, 0x120d, 0x438c, 0x41a6, 0x1e12, 0x421c, 0x416b, 0xb2ca, 0x4298, 0x31dc, 0xbf0c, 0x27c0, 0x41ea, 0x18b6, 0x4097, 0x42ed, 0x059a, 0x42fd, 0x428a, 0x4216, 0x1846, 0x43ec, 0x4194, 0x247c, 0x4347, 0xbf8f, 0x45b4, 0xb2cd, 0x41e7, 0xb2b8, 0xaf7f, 0xbf6c, 0xbae3, 0x4117, 0x43ae, 0x41b9, 0xbfe0, 0x144c, 0xbf13, 0x4292, 0x0fcd, 0x4329, 0x43f7, 0x1aba, 0x2109, 0x41e3, 0x4426, 0x401c, 0x40ea, 0xae86, 0x42c2, 0xbf02, 0x43c1, 0x2e2f, 0x42c3, 0x40a1, 0x4161, 0x43ff, 0x1a3e, 0x4334, 0xb234, 0xb256, 0x4037, 0xbaf8, 0x40e5, 0xba12, 0x2f78, 0x4019, 0x4144, 0x4305, 0x21a3, 0xbfcf, 0x0a76, 0x412b, 0xad51, 0x01b7, 0x4276, 0x1aa1, 0x42eb, 0x405b, 0xa6b4, 0x40d1, 0x410a, 0x41f4, 0x1a37, 0x1bd6, 0x38ea, 0xb27b, 0x4120, 0x4238, 0xb250, 0x4099, 0x437a, 0xbfce, 0xba3b, 0xbfb0, 0xb211, 0xa07d, 0x1deb, 0xbad9, 0x448d, 0x1528, 0xbf00, 0xb2d4, 0x44b3, 0x1efb, 0x408d, 0x41c8, 0x43f3, 0x4289, 0x43fa, 0x2672, 0xb2cc, 0x4253, 0x4234, 0xbadc, 0x4049, 0x430a, 0xb280, 0x42ef, 0xbfb7, 0xb276, 0xbac8, 0x416f, 0xb2aa, 0x1bc1, 0x0b1a, 0xb0ee, 0x4022, 0xb273, 0x2575, 0x1867, 0x1e00, 0xb245, 0x41d9, 0x446e, 0xba5f, 0xb07a, 0xb263, 0x2274, 0x135c, 0x4045, 0xb29a, 0x19c5, 0xbf56, 0x4361, 0x422a, 0x4193, 0x4138, 0x0c7c, 0x40b9, 0x2b3b, 0xb2e2, 0xb075, 0x40b4, 0x434e, 0x4363, 0x43d6, 0x10cf, 0x400d, 0x1fc8, 0xbf3d, 0x41ee, 0x4244, 0x2a6c, 0x1e41, 0x3e24, 0x4675, 0x43b5, 0x4032, 0xb080, 0xbf95, 0x431d, 0x4159, 0x43d6, 0xba46, 0xb025, 0xbfae, 0x4024, 0x14c7, 0x2339, 0x41f7, 0x445f, 0x43e8, 0x12db, 0xba11, 0x42bb, 0x42d0, 0x1b2e, 0x1d0b, 0xba72, 0x1c00, 0x1672, 0xbf5a, 0x40bc, 0x1094, 0xb068, 0xb053, 0xb027, 0xb22a, 0x213b, 0x4133, 0xb0e5, 0x1c94, 0xb244, 0x4228, 0x41ec, 0xb2a8, 0x440a, 0x3ea0, 0x3cd2, 0xb28b, 0x431a, 0x4372, 0x4084, 0xb2b5, 0x4135, 0xbf3b, 0x0936, 0x4385, 0x3455, 0x1f02, 0xb2b5, 0x00b2, 0x0a84, 0x43b7, 0x41a9, 0x42ca, 0x2e91, 0xbf18, 0xb242, 0xbfb0, 0x4220, 0xb2bf, 0x44ba, 0x41e7, 0x4181, 0x1eaa, 0x18c0, 0x1a7a, 0x4123, 0x3ffe, 0xa60e, 0x434f, 0x082b, 0x428f, 0xbf4b, 0x3482, 0x18c0, 0x4158, 0x4454, 0xbf5e, 0xbae3, 0x18cf, 0x4221, 0x42e8, 0x2748, 0xbfbd, 0x424f, 0x41f9, 0x41d8, 0x40e2, 0x285a, 0x426a, 0x43e0, 0x4170, 0x4387, 0xaf96, 0x3f64, 0x4066, 0xbf28, 0x4089, 0x2a20, 0x43c9, 0x43fd, 0x4096, 0x460b, 0xbf7b, 0xbadd, 0x1a8d, 0x1c50, 0x401c, 0x41ea, 0xa29b, 0xba76, 0xbae3, 0xb2f9, 0x4240, 0x4193, 0x0dd9, 0xbf9a, 0xba6e, 0x4362, 0x2f82, 0x41b2, 0x4111, 0x442b, 0xa3ab, 0x4069, 0xba77, 0x3e1a, 0xbfa0, 0xaac3, 0x1fa1, 0x4122, 0x4601, 0x43f2, 0x4031, 0x43b3, 0x1305, 0xb2ff, 0x1da3, 0x40a3, 0x4019, 0xbfba, 0x40a3, 0xba43, 0x41f9, 0xb2a0, 0xae39, 0x41d2, 0x0268, 0xb2b4, 0x4082, 0x4197, 0x1ed8, 0x4323, 0x43a4, 0xbaef, 0x082d, 0x4065, 0x4188, 0x42c9, 0xa7c3, 0xbf45, 0x4269, 0x401e, 0x45b1, 0x427b, 0x43cf, 0x4242, 0xba2b, 0xbf0f, 0x4392, 0xb09c, 0xb284, 0x4117, 0x17e3, 0x414a, 0xb0ac, 0xbfd7, 0xba34, 0xbf70, 0xb0ba, 0x4243, 0xb20e, 0x431e, 0x441b, 0x43e7, 0x4005, 0xacd8, 0xba47, 0xbf85, 0xa91c, 0x42c8, 0x0b69, 0xa1d4, 0x1f9d, 0x4100, 0x41c3, 0x40a5, 0xb2a1, 0x1f71, 0xb2d3, 0x4424, 0x4176, 0x439e, 0xb2e0, 0x42f8, 0x40d1, 0xb204, 0x4617, 0x462b, 0x40e6, 0x416f, 0x4197, 0xb2ea, 0x42cf, 0x0d0d, 0xbfa7, 0xb2b5, 0x13da, 0xbaec, 0x4351, 0x42af, 0xbaf1, 0xb2e7, 0x0ff2, 0x45d2, 0xbf9a, 0x414d, 0x1ddb, 0x4047, 0x4106, 0xa497, 0x40b7, 0x2043, 0xbfe1, 0x40e4, 0x41e4, 0xb207, 0xb255, 0x4134, 0x285e, 0xb24d, 0xb28c, 0x40ab, 0x4359, 0xbfc9, 0xb25f, 0x3c0a, 0x40d3, 0x1b61, 0x41fe, 0x4443, 0x45cc, 0x4242, 0xb275, 0xb24b, 0x228c, 0x42c1, 0x437a, 0x40d9, 0x27fc, 0x38fc, 0x43e4, 0x418f, 0xbf3c, 0xb043, 0x18c6, 0x4246, 0xbfab, 0xb248, 0x08df, 0xba5e, 0x419f, 0x41d2, 0x41c1, 0x4084, 0x42a7, 0x46e4, 0x40d0, 0x43e2, 0xaef6, 0x40f3, 0xb2f6, 0x422b, 0xa7b8, 0x4596, 0x405a, 0xb203, 0xad67, 0xb023, 0xbf0b, 0x3f5c, 0x46d9, 0x29a6, 0x42c9, 0x41c7, 0xbaf7, 0x1173, 0x41cf, 0x4388, 0xbf5b, 0x45c3, 0xba74, 0xbad9, 0xb279, 0x40ad, 0x4591, 0x4332, 0x40ae, 0x1963, 0x1ced, 0x0f02, 0xba62, 0x42b6, 0x4260, 0xbaed, 0x42e7, 0xb2f9, 0x312b, 0x43d8, 0x1dfa, 0x1440, 0xa31d, 0x44c2, 0xba08, 0xb01a, 0xbac7, 0xbf9d, 0x45dc, 0x43ca, 0xaad2, 0x0c3a, 0xaa18, 0x1024, 0x3f34, 0x46fb, 0xb0e3, 0xb2e8, 0x331a, 0xbf80, 0x2092, 0x4257, 0x4328, 0xb0e0, 0xb07d, 0x4268, 0xbf76, 0x0232, 0x42df, 0xb2a0, 0xb2ab, 0x4022, 0x402f, 0x42d3, 0x1ec6, 0xb2f6, 0x18a4, 0xbaed, 0x41a9, 0xbad1, 0x4175, 0xb06b, 0x405f, 0x4189, 0xaa10, 0x42c4, 0xba1a, 0xa362, 0x2a89, 0x4022, 0xb0df, 0x0583, 0xba04, 0xbf98, 0x46b4, 0x0d9a, 0xbaec, 0xba1c, 0x2507, 0x43be, 0x43d0, 0x407c, 0xbf80, 0x0db8, 0x4114, 0x4329, 0xbf4a, 0x442b, 0xb0d9, 0xb02a, 0x4063, 0x068e, 0x40a1, 0xbfcb, 0x4398, 0x1aab, 0x42ec, 0x41b6, 0xba5a, 0x411e, 0x456b, 0x42a1, 0xba39, 0x4183, 0x4222, 0xbfb1, 0x43ca, 0xb05a, 0x4474, 0x4546, 0x1140, 0x18cc, 0x43e2, 0xba0c, 0x2c60, 0x428a, 0x426c, 0x1909, 0x42b3, 0xb049, 0x425a, 0x4234, 0x406a, 0x02de, 0x0a53, 0x4038, 0x4079, 0xbfe0, 0x40d4, 0xba20, 0x4010, 0xbaeb, 0xb2b8, 0x3c73, 0x384d, 0xbf5f, 0xb22c, 0x3e1f, 0xb2f5, 0xa11e, 0x1d16, 0x1cd4, 0xbfaa, 0x0d49, 0x4112, 0x09c1, 0x4249, 0xb211, 0x4393, 0xb210, 0xb295, 0xbf43, 0x08a4, 0x40ad, 0x4635, 0xbf70, 0x462d, 0x07ae, 0x4357, 0xb07c, 0x433c, 0x1d49, 0x394d, 0x4340, 0x4310, 0x3c5b, 0xae19, 0xb044, 0xb0c1, 0x4675, 0x3437, 0xb225, 0x4013, 0xba48, 0x42e5, 0x382d, 0x42f8, 0x01ac, 0xbf28, 0x426a, 0x424b, 0x45ce, 0xb0f4, 0x40fc, 0xb28a, 0xb2aa, 0xb098, 0xb2ec, 0x4193, 0x1b94, 0x40c9, 0x4145, 0x22d4, 0x14ab, 0xbf32, 0x3629, 0x46b5, 0x454b, 0x1c0c, 0xbaf7, 0x1078, 0xb20f, 0x434d, 0xbaff, 0x4327, 0x4282, 0x4251, 0x3560, 0x419b, 0x4694, 0x1f84, 0x42e6, 0xa35c, 0x4194, 0xa25f, 0x2f49, 0xbace, 0x1cdf, 0xb062, 0x2c47, 0x41b0, 0xbf77, 0x429e, 0xb280, 0x42e6, 0x40e5, 0xb282, 0x4002, 0x08db, 0xba3f, 0x46fc, 0xb00b, 0xba3b, 0x41dd, 0xb286, 0x1677, 0x1eff, 0xb254, 0xb2c2, 0x0100, 0x1699, 0x1c80, 0x00fd, 0x0604, 0x3bfd, 0x15b4, 0xba0c, 0x25fc, 0x42db, 0x4354, 0xbf0b, 0x1c85, 0x40cc, 0xb2f8, 0xb01c, 0x43f9, 0xb2e3, 0x438f, 0x45cb, 0x1d66, 0x430c, 0x46c0, 0x42bc, 0x41d6, 0xbfaf, 0xba63, 0x078d, 0x42c2, 0x4266, 0x4591, 0x2b98, 0x4478, 0x1fa1, 0x3b41, 0xb0ea, 0xb032, 0x4033, 0xb093, 0x0e51, 0x4357, 0xba31, 0xba29, 0xaa85, 0x40e3, 0x1e97, 0x1f3a, 0x067b, 0x428d, 0x4266, 0x430c, 0xb2ee, 0xbfcb, 0xac4b, 0xba04, 0xb009, 0x4290, 0x12a4, 0x40a2, 0x1d68, 0x4140, 0x07d3, 0x1fcb, 0xb226, 0x03c1, 0x459d, 0x42a3, 0x40a1, 0x409f, 0x1a49, 0x4040, 0x39e8, 0x4214, 0x40c5, 0xb066, 0xbf4e, 0x43ec, 0x01da, 0x420f, 0x43d5, 0xba21, 0x0941, 0xba04, 0x401a, 0xba44, 0x421c, 0xbf98, 0xbac2, 0xa729, 0xbf80, 0x0966, 0x1970, 0x402f, 0x22fc, 0x432b, 0x4026, 0x434b, 0x43e7, 0xb06f, 0xbf08, 0x403d, 0x1454, 0x44e8, 0xbf4c, 0xbafd, 0x42cd, 0x104f, 0x4361, 0x4253, 0x34fe, 0x1d4d, 0xa421, 0x21da, 0x4488, 0xbafb, 0x418f, 0x41ec, 0xb203, 0x4190, 0x45d6, 0x3a4f, 0xbac0, 0xbf9c, 0x45a4, 0x43c1, 0xba5c, 0x0eaa, 0x4241, 0xb230, 0xb21a, 0xb070, 0x43f9, 0x40a0, 0x42ec, 0x423c, 0xba04, 0x41b0, 0xb25d, 0xb09e, 0x462e, 0x41ae, 0x40f5, 0x4338, 0x28dc, 0xb004, 0xaf65, 0xb29b, 0x40d0, 0x2689, 0x4302, 0xbf37, 0x1467, 0x40ff, 0xab16, 0x40cf, 0x45a8, 0x403c, 0x4340, 0x05f5, 0x2cca, 0x42d3, 0x1583, 0x1a0e, 0x40ae, 0x3d53, 0x412e, 0x4354, 0x4328, 0x461a, 0x4208, 0x413c, 0x366a, 0xbad5, 0x41be, 0x1b57, 0xbf3f, 0x43d2, 0x425f, 0x324a, 0xb0b4, 0x3dbb, 0x4395, 0x4168, 0x4381, 0xb0af, 0x1b4b, 0x43b9, 0xb272, 0xb290, 0x3ca5, 0xba31, 0xba02, 0x43a2, 0xba37, 0xbf00, 0x1d08, 0x4442, 0x4261, 0x3c8b, 0x09bd, 0x4051, 0x294a, 0x435e, 0xbf9a, 0x426b, 0xb0de, 0xba13, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xb365a6c0, 0x01723518, 0x0009419f, 0x745fc87d, 0x5ed42207, 0x1f49801a, 0x138d5de7, 0xd2498a44, 0x0115d59c, 0x1cc12bbe, 0x5e000be4, 0xe36b4a94, 0x3974216e, 0x5afb38c7, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x69000004, 0x5c11201c, 0x5c1120b9, 0xb920115c, 0xfffffed0, 0x01a40000, 0x00004ffb, 0x69000000, 0x5c1120b9, 0xf9076780, 0x42812c14, 0x000014ca, 0x00001650, 0x5afb4adf, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x3e9f, 0x423a, 0xb038, 0x1a66, 0x2ab4, 0xbf88, 0x41f2, 0x43ed, 0x46e2, 0xbafa, 0xb277, 0xacc3, 0x3d1f, 0xb2f5, 0x008e, 0x4288, 0x298f, 0x4058, 0x1fc4, 0x4234, 0x4372, 0x422f, 0xb260, 0x41ca, 0x4030, 0x4004, 0x40a9, 0x40c8, 0xb2e9, 0xb24e, 0x1d5a, 0xbf9b, 0xb062, 0x4160, 0x3cb1, 0xba5f, 0x43a1, 0x4362, 0x4553, 0x4260, 0x4329, 0x1061, 0xba58, 0xb2f4, 0x40ad, 0x41be, 0x436a, 0x43a3, 0xbfc0, 0xb0f6, 0x401a, 0x0c62, 0x1513, 0xb205, 0xb215, 0x090b, 0x1c21, 0x28af, 0x4092, 0xb294, 0xbfbd, 0x2cb1, 0x1b1e, 0x0984, 0x4175, 0xa7e8, 0xb273, 0x2636, 0x410b, 0x45c0, 0x41ec, 0xb2eb, 0xb0d0, 0x42ca, 0xba52, 0x4185, 0x1455, 0x4600, 0xb215, 0x1947, 0x24af, 0xba32, 0xbfab, 0x40ac, 0xba0b, 0xba5b, 0x07fa, 0x0723, 0xb2c4, 0x2163, 0x42d5, 0x4004, 0x40a7, 0x45d8, 0x43af, 0x42b4, 0x465d, 0xba27, 0x4395, 0x2d9c, 0x192f, 0x1a6a, 0xb28d, 0xbfc7, 0x458b, 0x4281, 0xadc4, 0x4092, 0x125e, 0xbfac, 0x43c8, 0xadde, 0x2713, 0xb265, 0xbf62, 0x3095, 0xbff0, 0x1bd1, 0x40a8, 0x43e3, 0x18c1, 0x41b0, 0x3fb9, 0x18ae, 0x4314, 0x347e, 0x4258, 0x45b5, 0x4660, 0x401d, 0x1dcc, 0x4667, 0x420b, 0x4172, 0xac70, 0x39f3, 0x4097, 0x19b6, 0x41fe, 0xb0c1, 0x4623, 0x1394, 0xbf25, 0xba65, 0x4101, 0x439d, 0x42e1, 0x4333, 0xb0e5, 0x41dd, 0xb268, 0x22c7, 0x2674, 0xba71, 0x421f, 0x4450, 0x0cd7, 0x40e4, 0x1b43, 0x3ea7, 0x1311, 0xbf4e, 0x411b, 0x4112, 0xa8fa, 0xbf19, 0x4015, 0x3844, 0x40d4, 0x4087, 0xbfa7, 0x1f9b, 0x0c01, 0x213b, 0x3a56, 0x4242, 0x40bf, 0x4233, 0xb046, 0x42b5, 0x4556, 0x3058, 0x42b8, 0x36be, 0x437b, 0x1eb6, 0xbacd, 0x4173, 0x433f, 0x42a2, 0x41f3, 0xba23, 0x412f, 0xacfb, 0x4221, 0xbf25, 0x40b0, 0x40b4, 0xba72, 0xb0f2, 0x461f, 0x3123, 0x037e, 0x4580, 0xa9da, 0x2938, 0x42a6, 0x3aaa, 0x4125, 0x2721, 0xbfa4, 0x2a82, 0xb215, 0x41c0, 0xa1c3, 0xb27f, 0x4223, 0xba05, 0x1e73, 0x4054, 0x4629, 0x4415, 0x1cae, 0xb222, 0x1ac2, 0x424f, 0x206a, 0x4080, 0xb2c3, 0x412a, 0xa37e, 0x4037, 0xb26d, 0xb09a, 0xb211, 0x442d, 0xaca9, 0xbf0d, 0x34d4, 0xbfa0, 0x196a, 0x408f, 0x4313, 0x41ea, 0x1560, 0x46ea, 0x449c, 0x3a47, 0x40a6, 0x440a, 0xb267, 0x33b8, 0xb22c, 0x46eb, 0x1824, 0x3843, 0xbfca, 0xbfd0, 0x0716, 0x1031, 0xba00, 0x0daf, 0xba7e, 0xba72, 0xba11, 0x4375, 0xad64, 0xa1db, 0xb263, 0x42e7, 0x46f3, 0xa375, 0x4097, 0xa116, 0x4302, 0x19f5, 0x40ce, 0xbaf9, 0x269e, 0xbf0b, 0x407e, 0xb0f0, 0x45f4, 0x40fc, 0x1e2e, 0xba56, 0x4136, 0xba41, 0x4222, 0xbf1c, 0xb00c, 0x433f, 0xba7a, 0x41a5, 0xbfd0, 0x4260, 0xaf36, 0x42f3, 0x4376, 0xb0e8, 0x42e7, 0x0b6f, 0x4315, 0x1e04, 0xa4ef, 0x135b, 0xa6cd, 0x29d1, 0x40b6, 0x42d0, 0x4596, 0x4383, 0x466d, 0xbf7d, 0x46e4, 0x3014, 0xba02, 0x2ad7, 0xb2fe, 0x3de9, 0x1aa7, 0x2d8c, 0x150c, 0x4233, 0x40a0, 0xba40, 0x3a7a, 0x4361, 0x4447, 0xae8e, 0x4268, 0x4391, 0x41c8, 0x29dc, 0x4174, 0xb23c, 0x1bfc, 0x2c34, 0xbfc0, 0x194b, 0xbf9c, 0x43a6, 0xba78, 0x1b4b, 0x43ce, 0xbad7, 0xbfad, 0x067f, 0x4117, 0x4298, 0x4380, 0xb206, 0xba22, 0x4553, 0x3d55, 0x42f4, 0x1c7d, 0x419e, 0x206f, 0x1af1, 0xbff0, 0x3531, 0x1d92, 0x4150, 0x3fb8, 0x1e5f, 0x4250, 0x1fa1, 0x4202, 0xb0a1, 0xbf11, 0x45d9, 0x4676, 0x40fa, 0x421e, 0x1f6a, 0x0bf8, 0x29cc, 0xbf9d, 0x412e, 0xbf60, 0x3489, 0x4007, 0xba3b, 0x1836, 0x297c, 0x4284, 0x42e1, 0x4409, 0x429f, 0xba1a, 0xbac9, 0xb2b9, 0x3794, 0xa128, 0xa5c4, 0x42fb, 0x4593, 0x4184, 0xb0de, 0x0384, 0xbf3a, 0x3416, 0x3ea6, 0x411c, 0x42cd, 0xbfc0, 0xbaf3, 0xaa2e, 0x4317, 0xb2b2, 0x42d2, 0xb025, 0xb2d1, 0x091b, 0x423a, 0xb094, 0x468d, 0x4426, 0x31f9, 0xbf2e, 0x1e6b, 0x4388, 0x1c68, 0xba48, 0x3089, 0x4607, 0xba01, 0x416d, 0x1d85, 0x4424, 0x40d4, 0x41fa, 0xb28e, 0x1d5f, 0x4177, 0xb0d1, 0x1c5e, 0x4203, 0x0cf5, 0xb24b, 0x1cde, 0x1d65, 0x4281, 0xbfa6, 0xba36, 0x42a4, 0x4243, 0x44fa, 0x43b6, 0xa162, 0x41be, 0xb05d, 0x001b, 0x4283, 0xb2fd, 0x0395, 0xb28d, 0x40fb, 0xba1f, 0x43b1, 0x41af, 0x4157, 0xb245, 0x42f4, 0xb06b, 0xb27c, 0x1d2f, 0x45ee, 0x4435, 0xbfe2, 0x190a, 0x43a7, 0x4565, 0x41c3, 0xba46, 0x42d3, 0xb014, 0x23e7, 0x43a1, 0x436d, 0x4386, 0x1b31, 0x44e9, 0x427e, 0x1b44, 0xba66, 0xb02b, 0x404f, 0x42df, 0x438c, 0xba6f, 0x4083, 0xb02c, 0xba59, 0xaf47, 0x456d, 0xbfbd, 0xb2bd, 0x1e66, 0xbaee, 0x41af, 0xb05e, 0xbf70, 0x41eb, 0x4099, 0x42f1, 0x41d3, 0x40ed, 0xaefe, 0x4223, 0xa7ab, 0x02cb, 0x4393, 0xb258, 0xbfa3, 0x4311, 0x43f3, 0xb2f5, 0x43f4, 0xbf5e, 0x1854, 0x2a65, 0x1c8d, 0x4363, 0x4087, 0x40b2, 0x410a, 0xa11a, 0xb2ca, 0x1efa, 0x20d8, 0x419c, 0x41fd, 0x4283, 0x43f4, 0x4264, 0x420a, 0x469c, 0x1961, 0xb2fc, 0x429a, 0xb2b7, 0x4293, 0xa373, 0xbf98, 0xab6d, 0xb266, 0x01e3, 0x4176, 0xa2e0, 0x0265, 0x42ba, 0x40ce, 0x0695, 0x42a8, 0x3b85, 0x1ea1, 0x428b, 0x442a, 0x4452, 0xbf3c, 0x1bcb, 0xba76, 0x466b, 0x41bb, 0x44b8, 0x401d, 0xba78, 0x4020, 0x46a2, 0xb2ce, 0xb24f, 0x4202, 0xba02, 0x46d1, 0xbf3b, 0x4329, 0x0e1d, 0x40f7, 0x0daa, 0x40b6, 0xb2f0, 0xb208, 0x45a8, 0xbf0a, 0x43d7, 0xb238, 0x43e5, 0x43f3, 0x1dee, 0x3982, 0xbaf5, 0xbf70, 0x022f, 0x42d8, 0x4270, 0xae68, 0x1b50, 0xb2fa, 0x43ce, 0xa22b, 0x4103, 0xb0fe, 0xbf0a, 0xba03, 0xba58, 0xb21c, 0x4242, 0x428b, 0xb21b, 0x46fc, 0xbad7, 0x44f9, 0x409d, 0x1fa0, 0x422f, 0x4045, 0x0cb9, 0xbf37, 0x4168, 0x42e2, 0x1d17, 0x4146, 0x404b, 0x4397, 0x4023, 0x4218, 0x1be7, 0xb20c, 0xbfd0, 0x4166, 0x4677, 0x442e, 0x0bc1, 0x462e, 0x4103, 0x1b0b, 0xb0b5, 0xbf39, 0x3c04, 0xb2a6, 0x4132, 0x031d, 0xbac8, 0xb0d9, 0x436b, 0xb208, 0x4127, 0xbad3, 0x0705, 0x116f, 0xb23b, 0xba75, 0xaccc, 0xba6c, 0x19b5, 0x42c1, 0xb296, 0xbf98, 0x28ab, 0x1165, 0x4195, 0xb06b, 0x4356, 0xbf21, 0x2e2c, 0xba6e, 0xbf70, 0x1b50, 0x1d56, 0x4299, 0x41d0, 0xb2b1, 0x4097, 0x436b, 0x442b, 0xba7f, 0x409b, 0x2918, 0x40a4, 0x2155, 0x444c, 0x3bce, 0x4417, 0x1d3d, 0xb0b1, 0x4414, 0xb28b, 0xbf9e, 0x26d9, 0xba2e, 0x1b11, 0xbf60, 0x40a3, 0xb244, 0xb2b3, 0x4357, 0x40fd, 0x4253, 0x1b39, 0x408f, 0x43e5, 0x2a5f, 0x467b, 0x0fc0, 0x43fa, 0x1ad5, 0x0dca, 0x414e, 0x4401, 0x1cb1, 0xb229, 0x0cdd, 0x19a8, 0x29c7, 0xb2d4, 0x45b6, 0xbf6a, 0x085b, 0x0aa6, 0x195f, 0x4178, 0x0eae, 0x42af, 0xbfa2, 0x287d, 0x40e7, 0x45cb, 0x4233, 0xa5e9, 0x4005, 0xba4a, 0x42c7, 0xbf99, 0xba16, 0x402b, 0xb0a5, 0xb286, 0x40cb, 0x436b, 0x4326, 0x42da, 0x1762, 0x41f8, 0xb2c4, 0x4172, 0x3324, 0x1be2, 0xb210, 0x0a29, 0x2800, 0x225c, 0x02b0, 0xa200, 0x4125, 0x43c4, 0x4055, 0xb22c, 0x2b85, 0x42bc, 0x3465, 0xa4e5, 0xbfba, 0x4268, 0x33b2, 0x1ef1, 0x1d70, 0x4314, 0x4335, 0x4112, 0x2e63, 0x1fc5, 0xb091, 0x0ce5, 0x42a8, 0x17b0, 0x407a, 0xbf5d, 0x407a, 0x3b90, 0x0908, 0xb282, 0x1327, 0xbf00, 0x4376, 0x4278, 0x4075, 0x1de1, 0x4209, 0xb2c6, 0x42fd, 0x0018, 0x1d05, 0x46a3, 0x4290, 0xba7b, 0x0fc7, 0x1652, 0xbac1, 0x4309, 0x41b7, 0x36d6, 0xbf34, 0x43d6, 0x449c, 0xb2a3, 0xb269, 0x41d7, 0x24c0, 0xba4e, 0x40a0, 0xb005, 0x4454, 0x1542, 0x4311, 0x40ef, 0xb28f, 0x42c9, 0x2a56, 0x44f9, 0x407f, 0x454b, 0x2eff, 0x43e9, 0xba66, 0x4332, 0x4344, 0xb0a0, 0xbf31, 0x3bdd, 0x1e29, 0x1289, 0x3d48, 0x4077, 0xace3, 0x43ee, 0xb085, 0x1d32, 0x43b9, 0x411b, 0xb2ee, 0xb06f, 0xb26a, 0xb0ed, 0x0de4, 0x2e64, 0x36cd, 0x44d9, 0xbf26, 0x2866, 0xb271, 0x4132, 0x435c, 0xab9f, 0xa5f0, 0x40e8, 0x1dc4, 0x44e5, 0x409e, 0x15a3, 0x2a03, 0x4032, 0x43c2, 0xb29d, 0x3611, 0xbf02, 0x1a71, 0xb2bd, 0x44a5, 0x44dd, 0x2f63, 0x1176, 0x4017, 0x1c57, 0x42c5, 0xb257, 0xb01a, 0xb210, 0xba3e, 0x1033, 0xbaea, 0x2913, 0xb216, 0xbfa0, 0x414a, 0xbf01, 0x45dc, 0xba34, 0x43a4, 0x09ca, 0x44cc, 0x1fe2, 0xba21, 0x1c5d, 0x1a65, 0xba3f, 0xbfa4, 0x4344, 0xb24f, 0x1d4a, 0x4191, 0x4444, 0xba06, 0x413b, 0xbf58, 0x2509, 0xba2b, 0xb2a8, 0x1f68, 0x3a63, 0x467a, 0x05de, 0x4186, 0xba34, 0x42db, 0x1e2d, 0xbaef, 0xbfd0, 0x1c2d, 0x4176, 0x26bd, 0x4214, 0xa8fb, 0x4187, 0xba36, 0x42aa, 0x4073, 0x46d8, 0xb2f4, 0x40b5, 0x43fa, 0xbf87, 0x4225, 0x4336, 0x4376, 0x1c37, 0x4142, 0xb2e3, 0xbfe0, 0x3578, 0x41e3, 0x41cb, 0xb246, 0x402f, 0x2890, 0x422c, 0x40e9, 0x1892, 0x422d, 0xb209, 0xb2d6, 0x4045, 0xa2a6, 0x43bc, 0xbf37, 0x43ee, 0x4239, 0x326a, 0x3538, 0xb240, 0x43e8, 0xaae3, 0x3b5c, 0xa277, 0xb26f, 0xbf6d, 0xaf28, 0x4372, 0x1e45, 0x40f4, 0xb09f, 0x46e9, 0xb272, 0x4275, 0x2cb5, 0x42df, 0x03a3, 0x413f, 0xbac0, 0x40fa, 0xbad1, 0x0fba, 0xb2ca, 0x42dc, 0x4119, 0xb299, 0x41b9, 0x415f, 0xb240, 0xbf9c, 0xb224, 0x41de, 0x42f1, 0xbace, 0x34a5, 0xb073, 0xb2e6, 0x4073, 0x4185, 0xba32, 0x429b, 0x037d, 0xbfb0, 0x1855, 0xba71, 0x4199, 0x425a, 0x2faf, 0x410d, 0x41f6, 0x4349, 0x1eb6, 0x4333, 0x40dd, 0x40e9, 0xbf8b, 0x431c, 0x1eb9, 0x40c9, 0xb2ca, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x034bf2c4, 0xca960aa8, 0x9456a116, 0xb988fc86, 0x50cab48a, 0xa5759d59, 0x0dfb191d, 0x2b41a1c1, 0x1b2df3ef, 0x1463f435, 0x89ed0b15, 0x9d5b9a9c, 0x39b4b3b7, 0x23d2d921, 0x00000000, 0xc00001f0 },
                FinalRegs = new uint[] { 0xffffffc7, 0xfffffffd, 0x000000fd, 0x280000a7, 0x000000a5, 0x00000000, 0x28000003, 0xffffffff, 0x00001dec, 0x00003494, 0x000000ac, 0x00001dec, 0x00005e9c, 0x00003660, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x404a, 0x1e38, 0x41ce, 0x41f6, 0xa61e, 0x43e1, 0xba33, 0x0e8f, 0x4366, 0x1fcf, 0x1f02, 0x417c, 0xb24c, 0xbf09, 0xadc6, 0x1cfb, 0x2652, 0x3b78, 0x43dd, 0x4058, 0x435c, 0x418d, 0x41e9, 0xba1e, 0x461a, 0x417c, 0x4070, 0x2e53, 0xbf46, 0xb213, 0x0a50, 0x43ce, 0x403c, 0xb0fb, 0xba32, 0x43c9, 0x408b, 0x4240, 0xba6c, 0x4345, 0x43c8, 0x422f, 0x1b87, 0xb288, 0x1b5c, 0xba0a, 0xbf8c, 0x4371, 0xba11, 0xbf45, 0xa04e, 0xb290, 0x430b, 0x43e5, 0x1ae7, 0x41e3, 0x4260, 0x4631, 0x3e11, 0x46e1, 0xbae9, 0x4207, 0x40ac, 0x404c, 0x2f8f, 0xbf2e, 0x43db, 0x0ef2, 0x41bd, 0x0426, 0xb2e5, 0x3c47, 0xb092, 0x4331, 0xae0c, 0xa2f5, 0x420d, 0x4245, 0x436a, 0x1e7b, 0xac86, 0xb221, 0xb0d1, 0x38f0, 0xbfe0, 0x420b, 0x428d, 0x46c3, 0xa967, 0xb236, 0xbac2, 0x41fc, 0x4026, 0xbf4a, 0x1a93, 0xa7fd, 0x43e8, 0x0d91, 0x42f0, 0xb013, 0x1cb9, 0xb2bd, 0x039a, 0x4307, 0xbfb2, 0x049f, 0x1b17, 0x0a41, 0x18b0, 0xb23e, 0x4126, 0x14d5, 0x3efd, 0x42cd, 0xa767, 0x431e, 0x41f3, 0xb29c, 0x06c1, 0x29be, 0x363e, 0xb26a, 0xa0f2, 0x1e42, 0xbad5, 0xbaca, 0x1ab0, 0xba62, 0xb29d, 0xba6a, 0xbf27, 0x4385, 0x4135, 0x4637, 0x429e, 0xb04e, 0xbf90, 0x432d, 0x4199, 0x40ff, 0x4009, 0x4229, 0xb250, 0x055e, 0x2803, 0xb27c, 0x40f2, 0x4060, 0x41de, 0xb087, 0x2667, 0x439e, 0x466d, 0x4060, 0x1c10, 0x4003, 0xa2e9, 0xb0fc, 0xbfca, 0x4230, 0x42db, 0x1a8c, 0xb2dc, 0x42d5, 0x40d2, 0xba72, 0xbad8, 0x2197, 0x4029, 0x434b, 0x1b03, 0x20bf, 0xb2d0, 0x45e6, 0x4248, 0xac8e, 0x45d8, 0xb23f, 0x431c, 0x2e97, 0x4041, 0x413f, 0xbf92, 0x42b5, 0x4351, 0x46c5, 0x4098, 0x40d4, 0xb2cd, 0x43af, 0x41a1, 0xb035, 0x3e50, 0xbf45, 0x419c, 0x3771, 0xb270, 0x0cda, 0x1f3d, 0xb2a1, 0x18d4, 0x3595, 0xa300, 0x2257, 0x4230, 0xb267, 0xbf63, 0x22b5, 0x334c, 0x4008, 0xb2fb, 0xba14, 0x435b, 0x4253, 0x2bbd, 0x4375, 0x114d, 0xa458, 0x2764, 0x4073, 0x437d, 0x32d1, 0xb034, 0xba67, 0x40b2, 0xa45b, 0xa541, 0x417e, 0xbf93, 0x449d, 0x19af, 0xbada, 0x1c8f, 0xb288, 0x44b9, 0x35bc, 0x407f, 0x4389, 0x099d, 0x1daf, 0x433e, 0x07cc, 0x31b7, 0xb0e3, 0xa4fb, 0x4095, 0xb207, 0x40ec, 0xb2e8, 0x0ec7, 0x0c90, 0xb218, 0xbf6d, 0xa111, 0x1cb3, 0x2234, 0x1fa6, 0x4242, 0x2c82, 0x41d4, 0x2053, 0xbac3, 0x0cd9, 0x2ead, 0x425d, 0xb0c0, 0x44b0, 0xb2b5, 0xb01f, 0x1ce8, 0x46b1, 0x4156, 0xb2e2, 0x41b7, 0xae46, 0x1ee3, 0x444a, 0x415f, 0x4375, 0x1251, 0xbaee, 0xbf13, 0xb22f, 0x401b, 0x19ba, 0x4290, 0xa67b, 0xbac7, 0x1b71, 0x4156, 0x241a, 0x4577, 0x3cbe, 0x1b08, 0x40b2, 0x2f7d, 0x1a27, 0x4335, 0xbaf1, 0x2f52, 0x1850, 0x2414, 0x4129, 0xb24d, 0xbfbb, 0xb283, 0x07e8, 0x399e, 0xbac4, 0xb0af, 0x374d, 0x1bc1, 0x0f91, 0xb09d, 0x22a6, 0x419f, 0x3131, 0x1bdc, 0x42f7, 0x43d7, 0x29ac, 0xbf4c, 0xa2c6, 0xac1c, 0x1d92, 0x4127, 0xaa2a, 0xba5b, 0x4159, 0x41b8, 0x4320, 0x1a6f, 0x249a, 0x3fec, 0xbf5c, 0x41c0, 0x42ed, 0x1a95, 0x1ce2, 0xa1a3, 0x41e9, 0x4014, 0x05f4, 0xb03d, 0x4177, 0x3a2b, 0x4230, 0xabc9, 0x115d, 0x4308, 0x41c6, 0xb273, 0xb2cf, 0xa77c, 0xac87, 0x417e, 0x300c, 0x28a6, 0x42a9, 0xbf39, 0xaeb8, 0x1d9b, 0x1aed, 0x4346, 0xbaf9, 0x462a, 0x4199, 0x0418, 0x45cc, 0x4371, 0x04e7, 0x43f4, 0xb204, 0x4398, 0x438c, 0x43eb, 0xb201, 0x4155, 0x0a6e, 0x2d49, 0x42a3, 0xbac0, 0x43f4, 0x4245, 0xbfa3, 0x1edc, 0x446c, 0x3da5, 0x42e3, 0x42ae, 0xbf2c, 0xb25a, 0x40e7, 0xbfae, 0xb2ec, 0xba4a, 0x4301, 0x43c0, 0x0f46, 0x4182, 0x4233, 0x4084, 0xa36e, 0xbf9f, 0x410e, 0x42a0, 0x1879, 0x43d8, 0x44ea, 0xba40, 0xbfc5, 0x43fe, 0x42b9, 0x42b1, 0x4172, 0x4151, 0x42d4, 0xb25a, 0xbff0, 0xb026, 0x4116, 0x1ac9, 0x435b, 0xbfa2, 0x4482, 0x188e, 0x46f0, 0x26c3, 0x4103, 0x4491, 0xbf80, 0x2b1b, 0x41a2, 0x4022, 0x4132, 0x4235, 0x4214, 0x43d6, 0x4025, 0xa55b, 0x4040, 0x2d18, 0xac95, 0x4316, 0xbadd, 0x41c9, 0xbf9c, 0x2e9b, 0x34ac, 0x40d6, 0x1f75, 0xbfc0, 0x43a8, 0x3e7b, 0x20b6, 0xb0d7, 0xb22f, 0x41d0, 0x1c3a, 0x23a0, 0x1a3a, 0x426e, 0x4178, 0x45c8, 0x426a, 0xbf18, 0x42d7, 0x42e9, 0x40c1, 0x40da, 0xb273, 0xba11, 0xb209, 0xb07b, 0x18af, 0x40bf, 0x4253, 0x0e6f, 0xbf9d, 0x42a7, 0xa767, 0xba35, 0xad49, 0x1d61, 0x1dca, 0x436a, 0x2f36, 0x40e9, 0x1cf2, 0x4061, 0xba3e, 0x099e, 0xac24, 0x4559, 0x467b, 0x1a05, 0x08ce, 0xb295, 0xbfb3, 0x4057, 0x4384, 0x1f5a, 0x13e2, 0x4561, 0x442b, 0xbf02, 0x1563, 0xba37, 0xb230, 0xb0be, 0x41a4, 0x4235, 0xbf3c, 0x4008, 0xb04a, 0x4297, 0xb20d, 0x2cca, 0xb0d8, 0x4386, 0x4054, 0x402c, 0xbf60, 0xbf82, 0x41d9, 0x18bf, 0x1a14, 0x1236, 0x2e49, 0x429d, 0xa669, 0x4120, 0xbfdb, 0x3c29, 0x04af, 0x3169, 0x425f, 0xa558, 0xbf0d, 0xaf86, 0xb0ef, 0x0707, 0x41ba, 0xb263, 0xbfb9, 0x42eb, 0x3db5, 0x3ac5, 0xb214, 0x40ab, 0x42c6, 0x1d21, 0x439c, 0x2200, 0xb0fb, 0x4695, 0x41ef, 0x4178, 0x42ea, 0x43ec, 0xb297, 0x419d, 0x4091, 0x405b, 0x43a6, 0xb2de, 0x21ab, 0xba6d, 0x4002, 0xabb4, 0x400f, 0x40a2, 0xbf2f, 0xb243, 0xa1a2, 0x460c, 0xb2e0, 0xbf9f, 0x1ad6, 0x41d4, 0xb0a0, 0x1b7f, 0x43d7, 0xb228, 0xb2ca, 0x2655, 0x3efc, 0xb249, 0x27dc, 0x428c, 0x40f9, 0x4354, 0x429d, 0xb266, 0x4595, 0xb0cf, 0xb2c9, 0x401e, 0x4264, 0x42b4, 0x0c35, 0x0e87, 0x407f, 0x4336, 0x403d, 0x1df2, 0x41c1, 0xbf39, 0x43bd, 0x4174, 0x4329, 0x44a2, 0x1932, 0xbf4b, 0xb282, 0x1ded, 0xbacf, 0x404a, 0xb246, 0x4630, 0x1ac0, 0x43d7, 0x40a2, 0x4270, 0xb2e1, 0x0cf0, 0x1ecc, 0xbac3, 0x408b, 0x4358, 0x1ca6, 0xbf6c, 0x416b, 0x2d6e, 0xb017, 0x4385, 0xa144, 0xb2b5, 0x41fd, 0xbf28, 0x1aef, 0xb239, 0xb2ba, 0x430e, 0xb0c0, 0x42cb, 0xbfdd, 0xb020, 0x44d9, 0x414d, 0x402b, 0xb033, 0xba4c, 0x1ace, 0x401b, 0xb200, 0x40ae, 0xa013, 0x1fb4, 0x42a2, 0xba53, 0x4006, 0xbfc5, 0xba13, 0xba0a, 0xb21c, 0x406c, 0x0f32, 0x0d61, 0xb28a, 0xadd9, 0x2943, 0x42d8, 0xb212, 0x1e8f, 0xb273, 0x426f, 0xb08b, 0xa7ca, 0xbac8, 0x43b2, 0xa2d0, 0x0940, 0xbaee, 0x42b3, 0x42fe, 0x0239, 0xbf6c, 0x4082, 0xb272, 0x1219, 0x4101, 0x43a8, 0x403e, 0xb251, 0xb251, 0xb035, 0x1abe, 0x1853, 0x4373, 0x4086, 0x4330, 0xb22d, 0x4491, 0x42cb, 0x442d, 0x073a, 0x1abd, 0xb23c, 0xbf6e, 0x43b5, 0x1cd5, 0x1cac, 0x1a26, 0x42c9, 0x4312, 0x41f6, 0x034a, 0xbf1b, 0xba22, 0x43c1, 0x408b, 0x3efd, 0xbf8e, 0x45b1, 0x1ff9, 0xb0b4, 0x42a4, 0x40ff, 0xbae4, 0xa80c, 0xac90, 0x42fd, 0x41e9, 0x434a, 0xba62, 0x4263, 0xafea, 0x42a8, 0x1ebd, 0x0ace, 0x41e2, 0x448b, 0xb231, 0x4186, 0x40cf, 0xbf93, 0xba41, 0x4355, 0xbaca, 0xba6e, 0xb2c3, 0xa584, 0x421d, 0x42c3, 0x0b33, 0xb24a, 0x459b, 0x406a, 0x44c3, 0x1c47, 0x1dbe, 0x1b02, 0xa03c, 0x43af, 0xba58, 0x41b4, 0x43f2, 0x1e2f, 0xbfb8, 0x2fcb, 0x43d3, 0x425e, 0x2f90, 0xb2eb, 0xbfaf, 0x4446, 0x42d0, 0xb0e1, 0x3d22, 0x40fe, 0x425a, 0x439f, 0x4190, 0x417d, 0xb23f, 0x0e6c, 0xbf5d, 0xbfc0, 0xbfb0, 0x4136, 0x3dbb, 0x4548, 0x1ed0, 0x2e7c, 0x427e, 0x420b, 0x1a4a, 0x1b24, 0x405e, 0x42c8, 0x1e86, 0xb2b6, 0x4369, 0x02c5, 0x41c0, 0x4586, 0x417c, 0xb23c, 0x13d1, 0x2eb6, 0xbf9a, 0x4406, 0xbae8, 0x0fc1, 0x4302, 0x37a4, 0x3676, 0x16fc, 0x4292, 0xbf8b, 0x4022, 0x4122, 0xb2d8, 0x3706, 0x4258, 0xbf91, 0x4310, 0x4631, 0x408d, 0x19e5, 0xb01d, 0x4087, 0xa7ab, 0x0ffe, 0xba00, 0x37e6, 0x41ec, 0x407c, 0x406d, 0x1896, 0x43c4, 0x1666, 0x4023, 0x26e4, 0xbf7d, 0xad97, 0x43cd, 0x41cd, 0x215e, 0x41f7, 0x40e6, 0xb270, 0x414c, 0x18d9, 0x3998, 0xb078, 0xb2f5, 0x42bd, 0x0eef, 0x42c4, 0x0f53, 0xaf6b, 0xb29b, 0x11f7, 0x1f08, 0x4030, 0x4384, 0x3b0a, 0x3012, 0x4004, 0x43f4, 0x4636, 0x403f, 0x4484, 0xbfd4, 0xba40, 0xb049, 0x4608, 0x40c4, 0xbf92, 0x1898, 0x36dd, 0x403d, 0x1b5e, 0x4136, 0xbac7, 0x0f85, 0x1f13, 0x4337, 0xbae7, 0x416d, 0xa007, 0xbfcf, 0xaa8b, 0x2329, 0xb28f, 0x4380, 0x431c, 0xa918, 0xadc2, 0xad91, 0x250b, 0x4127, 0x11a2, 0x4395, 0xba07, 0x2356, 0xb21a, 0x4367, 0xb25b, 0x32db, 0x435e, 0x0285, 0xa8d6, 0x4164, 0x045d, 0x3a1c, 0x4191, 0xb0f5, 0x192a, 0x43c2, 0xb278, 0xbf1f, 0xb2e9, 0x456c, 0x4308, 0xb2f4, 0x4261, 0xbf2c, 0xbfe0, 0x4592, 0x431b, 0x402c, 0x1e54, 0x427f, 0x1f79, 0x4052, 0xa524, 0x4132, 0x2bc5, 0x0dfc, 0x1b3d, 0x4053, 0x42e1, 0xbaea, 0x3163, 0x42bc, 0xbfb1, 0x3240, 0x34fb, 0x42b0, 0xb034, 0xadf0, 0x4069, 0x413f, 0xbad6, 0x4335, 0xba39, 0x428e, 0x3b25, 0x41f9, 0x1816, 0x2ec7, 0x1e92, 0x4307, 0xb2d1, 0x405c, 0x41a0, 0x0efc, 0xa329, 0xbfc3, 0x435e, 0x438f, 0x4352, 0xb0c0, 0xb259, 0xae76, 0x40bf, 0x18f7, 0xbae0, 0xb2e7, 0xa6fe, 0x46c3, 0xbf69, 0x4613, 0x406b, 0xb2eb, 0x43ce, 0xa9e3, 0x10eb, 0x46ec, 0xba10, 0x427a, 0x412b, 0xbf9e, 0x41c1, 0x4353, 0xb2c7, 0x42f8, 0xafae, 0x1ab7, 0xb062, 0x407d, 0x4330, 0xb2af, 0x3801, 0xbfab, 0x43c8, 0x423d, 0x4159, 0x41e0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x7059abe5, 0xb415f906, 0x8490eae9, 0x0ff183c6, 0xf39f6a2c, 0xb4413795, 0xee05d88d, 0x50185d92, 0xf621a094, 0x8902e42e, 0x9ecca830, 0x152bbe7f, 0x038e4ec7, 0xeb5d3b32, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x89283ba8, 0x000001bb, 0xfffffff1, 0x00000000, 0x0000000f, 0x0000240e, 0x00001ba8, 0x0000240e, 0x00000000, 0x000013e8, 0x94ee481b, 0x00000000, 0xffffffe8, 0x00000170, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xba51, 0x414a, 0x4030, 0x4380, 0x4328, 0xb07b, 0x425a, 0xb214, 0x42fd, 0x2b6a, 0xaf72, 0x0da2, 0xb260, 0xb0a2, 0x4017, 0x4242, 0x38f4, 0x40d8, 0xb2b5, 0xbf43, 0x40f1, 0xb29b, 0x4240, 0x412b, 0x1c5d, 0xb27f, 0x2483, 0x4322, 0xba3d, 0x380c, 0xba4e, 0x1a5d, 0x4387, 0xb21d, 0xb2b6, 0xbf29, 0x2514, 0x43b9, 0x424b, 0x3cb5, 0xbf90, 0x40b2, 0x2a94, 0x42ce, 0x42df, 0xb23a, 0x12c2, 0xb232, 0x3230, 0x194f, 0xbf9e, 0xac89, 0x40d9, 0x4023, 0x424a, 0x2736, 0x0a55, 0x401a, 0x1835, 0x402e, 0xb20b, 0xbf6f, 0x2155, 0xbaf4, 0x39c6, 0x0161, 0x4217, 0x33de, 0x379d, 0x1dbc, 0x0b38, 0xb2ff, 0x23eb, 0x4072, 0xbf8a, 0x4201, 0x4240, 0x41b1, 0x1916, 0x40c4, 0xb26a, 0x4345, 0x0e81, 0xbf17, 0x4359, 0x4396, 0x4185, 0xa8f1, 0x43ce, 0x0472, 0x424d, 0x438d, 0x4018, 0x3684, 0xb244, 0x4349, 0x40a9, 0x400c, 0x41ac, 0x42ff, 0xaea0, 0xa7e6, 0xb2ec, 0x31eb, 0xb211, 0x42c8, 0x2649, 0x4064, 0xbad9, 0xbf05, 0x4169, 0x4664, 0x4679, 0x4576, 0x415b, 0xb200, 0x1f31, 0x2abf, 0xb0f4, 0x46d5, 0x4387, 0x40fa, 0x42dd, 0x18c0, 0xb021, 0x3d89, 0x403f, 0x4257, 0x1f9a, 0x43ed, 0x3c5c, 0x1c96, 0x41d4, 0xb201, 0x2ff5, 0x43ab, 0x0f0f, 0xbf69, 0x4637, 0x1895, 0x1d62, 0xa75b, 0xb0c3, 0x45d2, 0x432b, 0xb217, 0x18c7, 0x41dd, 0xa705, 0x4219, 0x433d, 0xa9c2, 0xb29d, 0x40f1, 0xbaf1, 0x420c, 0xb256, 0x4112, 0xbfbd, 0x1ab8, 0xba34, 0xbfe0, 0xbadf, 0xa4ed, 0x4323, 0x41a6, 0x40d6, 0x40c3, 0x4361, 0x0974, 0x1990, 0x42dc, 0x4377, 0xb2d8, 0x44d4, 0xba4e, 0x46e9, 0x188c, 0xbfd5, 0x429b, 0x4374, 0x4551, 0xb259, 0x418c, 0x4588, 0x43fe, 0xbfd0, 0x4175, 0x4227, 0x44b8, 0xbfe0, 0x2e9a, 0x406a, 0x2535, 0x4168, 0x0ad7, 0x1d42, 0x4264, 0xb086, 0xbada, 0xb00f, 0xb26f, 0x1b8d, 0x1c5c, 0x44c1, 0x3f77, 0x0f15, 0xb24c, 0xbf0e, 0x42d1, 0x44eb, 0x4253, 0x4233, 0x4203, 0xbfcd, 0x427a, 0x18a6, 0x4011, 0x29a1, 0x40cc, 0x448c, 0x1d4f, 0x0d2b, 0xb2dc, 0x4026, 0x12ff, 0x434c, 0x06a7, 0x415f, 0x41e4, 0x1842, 0xbf3a, 0xba5a, 0x4082, 0x43d2, 0x42a7, 0xa3c3, 0x41f5, 0x4341, 0x35dd, 0x425e, 0xb2c6, 0x4396, 0x420d, 0xb298, 0xbfd0, 0xa02a, 0x060e, 0x009f, 0x0988, 0x22b2, 0x4168, 0xbfbd, 0x437e, 0x38d0, 0x08f1, 0x41eb, 0x41c6, 0x21e9, 0x40d5, 0x1495, 0x436d, 0x41a6, 0x43f7, 0x21f7, 0xb04a, 0x4349, 0xb2d4, 0x0baa, 0xb034, 0x4087, 0x41f3, 0x40ff, 0xbf48, 0x41e7, 0xa4cb, 0x3b1b, 0xa68b, 0x18b1, 0xbaeb, 0xb045, 0xb231, 0x4194, 0x4425, 0x40f8, 0x1fda, 0xa62a, 0x4302, 0xb2ee, 0x40f9, 0x1bb8, 0x4365, 0x42ff, 0x434d, 0xba42, 0xb2ad, 0xbfb3, 0xba16, 0xb2bd, 0x41df, 0x42c9, 0x4367, 0x0235, 0x34d6, 0x4176, 0x40cb, 0x4295, 0x42b9, 0x0de1, 0x41c8, 0x41be, 0xb2ea, 0x1312, 0x4230, 0x1907, 0xb2f6, 0x4215, 0xba0a, 0xba32, 0x42a1, 0xbf07, 0x397f, 0xbfc0, 0x4206, 0x124e, 0x40cb, 0x440d, 0xb0a0, 0xb27c, 0x3960, 0x411d, 0x4217, 0x463a, 0x3a91, 0x3950, 0x438c, 0x1c16, 0xbfcb, 0xb207, 0x435b, 0x40ee, 0xbfb0, 0x2ebf, 0x3df7, 0xb21e, 0xba74, 0x1b05, 0x46fb, 0x428d, 0x43c1, 0x4196, 0x4379, 0x4239, 0x19e7, 0x411d, 0xbfd2, 0x3b6c, 0x42a7, 0x1ab9, 0xbafe, 0x40f8, 0x0e6a, 0x4422, 0xba1f, 0xbfa5, 0xba17, 0x43cb, 0x3d92, 0x43f7, 0x18a0, 0xbf21, 0x2196, 0x1fb2, 0x2491, 0x402b, 0xbf6a, 0x244e, 0x43b9, 0x03b7, 0x4228, 0x40e1, 0x42e3, 0x4353, 0x00d9, 0x41a9, 0xbac9, 0xbf74, 0x1b6a, 0x4381, 0x412b, 0x43d6, 0xbf26, 0x4194, 0x19f9, 0xb28a, 0x4000, 0xba61, 0x41a6, 0x1f6d, 0x19da, 0xb211, 0xbfa5, 0x1bf6, 0x4070, 0x1f03, 0xb275, 0x413d, 0x40f0, 0x45c8, 0x4036, 0x1844, 0x400a, 0x408b, 0x1cf3, 0xb2f6, 0x4302, 0xbf23, 0x16bb, 0x40c6, 0xba47, 0xba62, 0x42b8, 0xb24d, 0x4088, 0xba35, 0x43f5, 0xbade, 0x434f, 0x1ae1, 0x1df1, 0xb2da, 0xbfcc, 0x42ab, 0x1ece, 0x4252, 0x1e4c, 0x3aed, 0x4175, 0xb2c9, 0x422d, 0x4306, 0x3dd4, 0x4247, 0x413f, 0x3a6f, 0xb0c0, 0xb052, 0x41eb, 0xb222, 0xb05d, 0xbf16, 0x4332, 0x4142, 0xb29f, 0xb26d, 0x43b0, 0x2c2f, 0xa515, 0x43e4, 0x3d7a, 0xbf80, 0x2e76, 0x45e0, 0x1b4e, 0xbfd2, 0x45e2, 0x4033, 0x4305, 0xbf47, 0x1ad6, 0xbff0, 0x150c, 0x3e9a, 0x1ea8, 0x4120, 0x19ce, 0xbf79, 0x43ba, 0xbfe0, 0x3779, 0x42db, 0x39fe, 0xac2d, 0x1c17, 0x43b7, 0x4338, 0x29d6, 0x4069, 0xb0ba, 0xbf90, 0xa22d, 0x423c, 0x1df9, 0x40ba, 0xb24a, 0xbf87, 0x1999, 0xb277, 0x465f, 0x41da, 0xba31, 0xb2b8, 0xba07, 0x1950, 0x2ad8, 0x03a6, 0x4598, 0x1e2e, 0x4291, 0x439e, 0x43b4, 0xb0dd, 0x407d, 0x1859, 0xbf36, 0x196c, 0xb2f4, 0x19c2, 0x426f, 0xb2ed, 0xba27, 0xb209, 0x44e1, 0x2f28, 0x412a, 0x1f41, 0x26d7, 0xba26, 0x2bcb, 0x43ce, 0xbfa0, 0x1ee7, 0x307b, 0x1f27, 0x1438, 0x3bae, 0xb20a, 0x2af0, 0x1747, 0xbf03, 0xb272, 0x42c1, 0x445d, 0x1032, 0x43ac, 0x4048, 0x40c7, 0x43eb, 0x20f9, 0x4167, 0x4257, 0x419d, 0xb254, 0x020a, 0x0824, 0xbf60, 0x1bc9, 0xaa64, 0x4306, 0x41a8, 0x4368, 0x2461, 0x4412, 0x0a06, 0x426b, 0x380f, 0x429d, 0xb2b6, 0xb06a, 0xbf2c, 0x44a2, 0x4216, 0x42c3, 0xb24d, 0x455a, 0x417b, 0x4237, 0x1a76, 0xba20, 0xb08b, 0xba73, 0x178c, 0x144c, 0x1477, 0x3fff, 0xbac6, 0x40c6, 0x416b, 0xb221, 0x2221, 0x14a6, 0x3de1, 0xbf8f, 0x426c, 0xb28f, 0x4094, 0xb21a, 0x371b, 0x4654, 0x46ec, 0xba7f, 0x086c, 0x159f, 0xbaf8, 0x4094, 0xb2d1, 0x412e, 0x4087, 0xb246, 0x4190, 0x4318, 0xb2f2, 0x0736, 0xb269, 0xbf60, 0x4569, 0x466c, 0x4271, 0x4591, 0xbf91, 0x1b68, 0x4287, 0xba47, 0x4338, 0x42c7, 0xb27a, 0x4046, 0x4280, 0xbf77, 0x42e7, 0x456a, 0x117c, 0x4216, 0xbf51, 0x443e, 0x4362, 0x4036, 0x4336, 0x1e6b, 0xb25e, 0x0281, 0x4258, 0x456f, 0x4247, 0xb253, 0x358c, 0x4225, 0xb209, 0x4292, 0x2ac6, 0xb2c5, 0x2643, 0xad8c, 0x432d, 0xba18, 0x4155, 0x2239, 0x2027, 0xb27b, 0x4161, 0xb224, 0x0e38, 0xbf6f, 0x316c, 0xba5f, 0xad65, 0x4059, 0x1cd7, 0x4174, 0x325d, 0x1c2f, 0x41f3, 0xbaeb, 0x4065, 0xbf18, 0x4107, 0xb04b, 0xb2e6, 0x4060, 0x4058, 0x43e0, 0x13fe, 0x3d85, 0x4240, 0xbf58, 0x41ac, 0xbacd, 0xb218, 0x422f, 0xb228, 0x04cc, 0x1ed0, 0x43c1, 0x414a, 0x21dd, 0x1de5, 0x40d8, 0x4174, 0x4607, 0x430e, 0x4303, 0x40fd, 0xbf80, 0xbf96, 0x0173, 0xb21f, 0x44bb, 0xba35, 0x1e73, 0x40cf, 0xb0a8, 0x412d, 0x43ee, 0x42de, 0xb258, 0x411f, 0x4139, 0xb2aa, 0xae46, 0x0246, 0x421c, 0x18d7, 0x4258, 0xba6e, 0xb09e, 0x4024, 0xbaec, 0xba26, 0x41aa, 0x1e8d, 0x2a8c, 0xbf2e, 0x43e9, 0x46a9, 0x42ef, 0x43a1, 0x409c, 0x4176, 0x4133, 0xba70, 0x4649, 0xb0c6, 0x41a3, 0x4285, 0xb2e9, 0x41d2, 0x3f71, 0x403e, 0x4322, 0x4627, 0xbfae, 0xaa92, 0x43c6, 0x42d2, 0xac63, 0x0a31, 0xb258, 0xbfc6, 0x417a, 0xbaf1, 0xba1f, 0x43b8, 0x41fb, 0x4253, 0x1b4b, 0x4419, 0xbf4e, 0x4012, 0xb245, 0x3e4e, 0x44f8, 0x368f, 0x4324, 0x416d, 0x409b, 0x4254, 0xbf80, 0x27fa, 0xa534, 0xbf90, 0x1ab2, 0xb2ec, 0x4335, 0x1eb9, 0xba45, 0x4333, 0xbfb9, 0x1c4d, 0x2c8a, 0x2e8a, 0x423a, 0xb0e3, 0xb21c, 0x422d, 0xbf8e, 0x2e30, 0x42a7, 0xb0bf, 0x43b5, 0x1af2, 0x43c7, 0x1aef, 0x466c, 0xbf13, 0x3802, 0xba0b, 0x4043, 0xb089, 0xba08, 0x4153, 0x1a77, 0xb2fd, 0xa20a, 0x43b8, 0x402e, 0x4031, 0x18ce, 0x400a, 0xbf60, 0x32d5, 0x2169, 0xba6d, 0x41f3, 0x1722, 0x4396, 0xb266, 0x4165, 0xb2ad, 0x402a, 0x40ca, 0xb254, 0xbf15, 0x1b9e, 0x32eb, 0x4148, 0xba7a, 0x3889, 0x1cf3, 0x4117, 0xbad5, 0x40aa, 0x41e2, 0xbf67, 0x30d2, 0x1428, 0x42ab, 0x3d03, 0xa881, 0xbfb0, 0xb2bd, 0x1759, 0x4550, 0x0672, 0x1807, 0x300d, 0x40ad, 0x2e4a, 0xb237, 0x290f, 0xb228, 0x433e, 0x4273, 0x0bf7, 0xb0ac, 0xbaf3, 0xb00c, 0x0408, 0x4009, 0xbf3d, 0x411e, 0x433e, 0x4079, 0x41fb, 0x4375, 0x43b5, 0x4365, 0xb02c, 0x4654, 0x44e5, 0x4271, 0xb216, 0x41c5, 0x42c0, 0x4630, 0x3d54, 0xbfde, 0xba79, 0x1dba, 0xb0b7, 0x402a, 0x4279, 0x4120, 0x1e83, 0x4383, 0x18e9, 0x462c, 0x4178, 0x40a4, 0x4285, 0xb058, 0x1e53, 0xbf27, 0xb025, 0x29c0, 0x40d3, 0xb2e7, 0xba77, 0xbae1, 0x2061, 0x425a, 0x42e6, 0xbf71, 0x4018, 0x4337, 0x4287, 0xbafd, 0x42eb, 0x418e, 0x4177, 0xb01f, 0x4274, 0x1e4b, 0xad9b, 0xb2d2, 0xbad4, 0x40aa, 0x4302, 0x18be, 0x4214, 0x427c, 0x41ed, 0x18e8, 0xbfb5, 0x41ee, 0x41c8, 0x43f5, 0x401a, 0xaacd, 0x402a, 0x0dae, 0xb05e, 0x40af, 0x4019, 0x1c2f, 0x4273, 0x28ca, 0x3e2e, 0x4636, 0xba00, 0x41cf, 0xbfcd, 0x4393, 0x4225, 0x42c6, 0x08dd, 0x4085, 0x41df, 0x43e0, 0x1ef3, 0x2fc5, 0x1893, 0x407c, 0x4263, 0xa557, 0xb0ad, 0xbfbd, 0x2c87, 0xb254, 0xba35, 0x404b, 0x1f02, 0x4128, 0xba23, 0x21ae, 0x04df, 0x4097, 0xb06f, 0x4630, 0xb2d9, 0x4076, 0x4118, 0x34d4, 0x402c, 0x4014, 0x4660, 0x1b57, 0x1ed9, 0x1d79, 0x0656, 0x1a34, 0x3ed4, 0xbfde, 0x1cbc, 0x4318, 0xba0c, 0x4485, 0x3cb7, 0x1be9, 0x411e, 0x0c28, 0x4320, 0x4048, 0xba28, 0xba50, 0xb0ce, 0x43d5, 0x409f, 0x454c, 0xa404, 0x41ce, 0xbf25, 0x402a, 0x4621, 0x4193, 0x41d2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc53d8d03, 0xff04be91, 0xd7eb41e1, 0xf292368c, 0xc17640e7, 0x57b5fdc3, 0x3ca7f7e5, 0x438f9293, 0xbf87ea5c, 0x7ed4e28f, 0xf51a6864, 0x539c9c7c, 0x62d8cca2, 0xc0701359, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0xfffffcff, 0x000017f4, 0x00000000, 0x00000800, 0x000017f4, 0x00000003, 0xf2cf7fff, 0xffffe718, 0xc0464ce0, 0x0d53d3ba, 0xf51a6864, 0x000012ba, 0xf51a6b68, 0xdf4f4290, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x2af4, 0x4229, 0x413f, 0xb264, 0x4327, 0x4106, 0x402a, 0xb0a1, 0x45b0, 0xb2fa, 0x1ebf, 0x441e, 0x43f6, 0xbf8f, 0x4628, 0x3760, 0x4360, 0x42af, 0x4179, 0xbf1f, 0x41d3, 0x1991, 0x4181, 0xb293, 0x4235, 0x1f56, 0x4650, 0xaf8a, 0xadfe, 0xb20e, 0x1abd, 0xba65, 0x4210, 0xb2ea, 0xa27b, 0xbafe, 0xbf00, 0x41ca, 0x1a78, 0x1668, 0xb2a2, 0xb061, 0x400f, 0x43a0, 0xbf58, 0x41af, 0x413f, 0x183a, 0x1e39, 0xbf15, 0x41e3, 0x4322, 0x41ed, 0xb0d3, 0x37ae, 0x120e, 0xb229, 0x417e, 0xb2dd, 0x4181, 0x08cd, 0xb2e2, 0x42b3, 0x43f4, 0x0bb2, 0x40ca, 0x412b, 0xb066, 0xbad4, 0x1c95, 0x4128, 0xa2c0, 0xbfa1, 0x4165, 0x40a1, 0xb2fa, 0x4054, 0x439d, 0xb29b, 0xb219, 0xb0b8, 0xbae4, 0x10ce, 0x43c6, 0xba78, 0x4163, 0xb2b7, 0x4151, 0x416f, 0x4323, 0xb205, 0x4443, 0x421b, 0x4378, 0x439d, 0x4306, 0x11f7, 0x41ce, 0xb270, 0x4040, 0x3828, 0xbf4a, 0xa7f5, 0x4036, 0xbfb0, 0x46fb, 0x4638, 0x41ce, 0x4018, 0xb2f7, 0x3777, 0x4297, 0x4372, 0x1b90, 0x215f, 0x4096, 0x436b, 0x0217, 0xb26a, 0xb200, 0xa480, 0x4406, 0x2fbe, 0x0701, 0x426c, 0xb2f6, 0x41b3, 0x431f, 0xb2aa, 0x406a, 0xbf3a, 0x427b, 0x064a, 0x1ad3, 0xb265, 0xac12, 0xb0a2, 0x4100, 0xbf7e, 0x1d85, 0x43a1, 0x4009, 0x1338, 0x4315, 0xba41, 0x3f3c, 0x4211, 0x40a4, 0x055d, 0x455a, 0xba61, 0x439e, 0xbf80, 0x43ab, 0x45ed, 0x3ab9, 0xb2e2, 0x3618, 0xbf76, 0x2654, 0x4555, 0xa5a8, 0x4349, 0xaa9c, 0x41c5, 0x4356, 0x3c41, 0x40c2, 0x085c, 0x40a7, 0xbfba, 0xbae7, 0x42b5, 0x42a0, 0x21d7, 0x420f, 0x1a68, 0x41f7, 0xbfa0, 0x43ab, 0xbf9a, 0x41b4, 0x397b, 0xb23b, 0x1b68, 0x4260, 0xb256, 0x432a, 0xb2b4, 0x44c8, 0xb2cc, 0x409c, 0xbff0, 0x416e, 0xbf13, 0x4312, 0xb0bb, 0x342f, 0x43c9, 0x4107, 0x1a19, 0x223f, 0xabb8, 0x43f4, 0xbf83, 0x2d30, 0x4155, 0x424b, 0xb2b6, 0x4296, 0xb21c, 0x3cca, 0x41fd, 0xba12, 0x424c, 0xbf2e, 0xba38, 0xbacd, 0xa7e7, 0x00be, 0x43f9, 0x1a9f, 0x2f15, 0x1eb0, 0x42c1, 0x4576, 0x418c, 0x4203, 0x1428, 0x1b1a, 0xbaff, 0x430a, 0x41d1, 0x42ee, 0x42d1, 0xb0dc, 0x437a, 0xbf6e, 0x4009, 0xb2f8, 0x43e9, 0x4188, 0x3d63, 0x1f9c, 0xa47d, 0xba6a, 0xbadd, 0x45d4, 0xbfb1, 0xb2bf, 0x1fdf, 0x41c9, 0xbf90, 0xba05, 0xb2e1, 0x4012, 0x4353, 0x438d, 0x462c, 0xb019, 0x41c6, 0x4092, 0xb2dd, 0xb21e, 0xba72, 0x2b0f, 0x401e, 0xba33, 0x0636, 0xb213, 0x3216, 0x41a6, 0xbf99, 0x3934, 0x4290, 0xbff0, 0xba7f, 0x0aa5, 0x1b9d, 0x1b5e, 0xba5d, 0xbfd8, 0x39cd, 0x41ea, 0x427e, 0x42ea, 0xb292, 0xb2c5, 0x422d, 0x413d, 0xba0d, 0xafa1, 0x324a, 0xbf26, 0xb243, 0x1cb1, 0x4017, 0x4430, 0x42ce, 0xbf51, 0xb23e, 0xba00, 0xba27, 0x18b7, 0x4084, 0xb0d7, 0xb229, 0x423c, 0x429e, 0xbad8, 0xba28, 0x430c, 0xb070, 0x4091, 0x4191, 0x4038, 0x436d, 0x4285, 0x41de, 0x4037, 0xbfbb, 0x45a5, 0x1c02, 0x2189, 0x0fae, 0xb25d, 0x1d7c, 0x43ef, 0xa694, 0x10c2, 0xbfc8, 0xba12, 0xad26, 0x1501, 0x41c6, 0x4212, 0x197b, 0x4350, 0x403b, 0xbfe0, 0xb29d, 0x4558, 0x4205, 0x439c, 0x444f, 0xb01c, 0x1e76, 0xb2d5, 0x44f9, 0x46d3, 0x4036, 0x41de, 0xba6d, 0x2a32, 0xbf38, 0x4160, 0xbf37, 0xb0ec, 0xba41, 0xb28f, 0x41f1, 0xbf86, 0x2b4a, 0xb209, 0x4338, 0x1eb8, 0xbf04, 0x4336, 0xb210, 0xb0d2, 0x41de, 0xaa40, 0xb00f, 0xb025, 0x424e, 0x4454, 0x43c9, 0x42c6, 0x4158, 0x42e4, 0xbf22, 0x40d0, 0x40b8, 0xb2cf, 0x1b2e, 0xba36, 0xb2df, 0xbf4e, 0xbad7, 0x4172, 0x41ab, 0x46e0, 0x048d, 0x40d0, 0x42d6, 0x4302, 0xbacf, 0x42b1, 0x30ee, 0xb2ce, 0x43e0, 0x3e75, 0x402b, 0x0cb9, 0xba34, 0x0d88, 0xbfce, 0x0bf1, 0x1510, 0x4316, 0x463d, 0x458e, 0x1925, 0xaec3, 0xbf4c, 0x408b, 0xb08c, 0xb268, 0x4283, 0x410e, 0xbfc8, 0x261d, 0x1efb, 0x4309, 0x43ca, 0x4080, 0x423b, 0x4099, 0x4566, 0x4000, 0x10e8, 0xb20e, 0xba40, 0xbad2, 0x21b3, 0x4310, 0x4023, 0x1c67, 0x418e, 0xa05b, 0x4275, 0xb00a, 0xbf7c, 0x1ae8, 0xba49, 0x1039, 0x438d, 0x435f, 0x40ad, 0xb287, 0x282a, 0xaa2d, 0x427f, 0x42c7, 0x4354, 0x435d, 0x41b9, 0xb06f, 0x46e9, 0xba23, 0x40e8, 0x408d, 0x4013, 0xbf0a, 0x435c, 0x2384, 0x4542, 0xa98c, 0xb20b, 0xb27c, 0xbafa, 0x4303, 0x4097, 0x41f9, 0x42ea, 0x407e, 0x3c9a, 0x4097, 0x4359, 0x1e96, 0x0020, 0x3c97, 0xba3c, 0xb2ff, 0x42ca, 0x4471, 0x4076, 0xb250, 0xbf7e, 0xbfd0, 0xb23f, 0x4365, 0x4638, 0x1d42, 0x21a7, 0x1325, 0x4367, 0xb038, 0x4379, 0x4304, 0x4574, 0x410c, 0xbf8e, 0x23da, 0x40ad, 0xb202, 0x42a5, 0x4321, 0x44f4, 0x1a22, 0xba30, 0x40f6, 0x40a1, 0xa786, 0xb0b2, 0x414a, 0x43cd, 0x42ef, 0x034a, 0xbf3d, 0x4029, 0x463e, 0x0ac8, 0x43f0, 0x1ac0, 0x401e, 0xbaea, 0x0d5e, 0xbfd3, 0xb2a2, 0x42b9, 0xb2ff, 0x4151, 0x42e3, 0x4083, 0x1864, 0x42f9, 0x19ba, 0xb058, 0x1a60, 0xa486, 0x2355, 0x2060, 0xbf0b, 0x42bd, 0x418e, 0x1ceb, 0x4290, 0x1f52, 0x1c67, 0xa739, 0x437c, 0x189e, 0x43e6, 0xbf0f, 0x415d, 0xb208, 0xb267, 0xa399, 0x411f, 0x1954, 0x346b, 0x19f7, 0x43e8, 0x433b, 0x3630, 0xba34, 0xbad2, 0xbf2c, 0xb291, 0x42b5, 0x4102, 0x1bbe, 0x4033, 0x4632, 0x1ffe, 0x436a, 0x4364, 0xbaf1, 0xbf94, 0xb289, 0xb0fc, 0x432a, 0x1ced, 0xba74, 0x2483, 0xbf5b, 0x426c, 0xb207, 0x41d8, 0x4157, 0x42e7, 0x42df, 0xb26e, 0x13cd, 0x4120, 0x1d31, 0xa70b, 0xbff0, 0xb03f, 0xb272, 0x4388, 0x370d, 0x1d74, 0x1ce9, 0x40ee, 0x403b, 0x1c1b, 0xbac0, 0x1007, 0x4008, 0x1996, 0xbf32, 0x1b8b, 0x23d1, 0x4556, 0x1cac, 0xa867, 0x3b4b, 0xb22c, 0x4419, 0xba1d, 0x4384, 0x4038, 0xba49, 0x4173, 0x3e55, 0xba7a, 0x4041, 0x421a, 0x4475, 0x409a, 0xbfd0, 0xb023, 0xb2be, 0xb2b7, 0x46ad, 0x4027, 0xa0cf, 0xbfe1, 0x3b58, 0x4156, 0xb213, 0x4388, 0x4443, 0xb21a, 0x42a6, 0xb2bd, 0xbace, 0x415c, 0x19b0, 0xbf97, 0x0d21, 0x1f02, 0x4133, 0x0a3b, 0xb027, 0x39b7, 0x429d, 0xb2b5, 0x4243, 0x00bf, 0x42a1, 0x45eb, 0xbf99, 0x4124, 0x2790, 0x4250, 0xbae9, 0x1d8c, 0x041e, 0x4222, 0x4243, 0x421a, 0x41e6, 0x1efc, 0x4611, 0xbfd6, 0x1e2d, 0x41a8, 0x4220, 0x42ee, 0x40ae, 0xba77, 0x4177, 0xae2a, 0x4190, 0x43f0, 0x1dc7, 0xb036, 0x464f, 0x4654, 0x4100, 0xbad3, 0x420e, 0x2f13, 0x42c7, 0xb00b, 0x4387, 0x12cc, 0x406e, 0x419b, 0x420e, 0x1e6c, 0x2d42, 0xb24d, 0xbf11, 0x421e, 0x43a3, 0x18d2, 0x3178, 0x4093, 0x42cd, 0x1c0a, 0x4173, 0x1821, 0x1c38, 0x43d8, 0x44e5, 0x4239, 0x13f2, 0xbf11, 0x42df, 0xba20, 0x4020, 0x4190, 0x40d4, 0x438a, 0x403e, 0xb22b, 0x400a, 0x2101, 0x223e, 0x0d56, 0xbf9b, 0x4328, 0xb276, 0xb09a, 0xb229, 0x1c8c, 0x09ad, 0x1cc6, 0x0e1f, 0x43f6, 0x411c, 0x41b2, 0x265b, 0xba5c, 0x0f83, 0x41be, 0x2897, 0xba67, 0x40eb, 0x413c, 0x1c6c, 0x4010, 0x1829, 0x43ce, 0x4292, 0xbf06, 0x4156, 0x43d6, 0x431b, 0xb2e6, 0x4167, 0x0318, 0xbfe0, 0x4224, 0x13e7, 0x4165, 0x4402, 0x41be, 0xa3ad, 0xb22e, 0xba0d, 0x1fae, 0xaf2b, 0x13bb, 0xb299, 0x422d, 0x4113, 0x3fef, 0x41ea, 0x1ebd, 0x12b6, 0x1b80, 0xbf4e, 0xb2f6, 0x33be, 0x4075, 0xbf35, 0xb24b, 0x42fc, 0x4039, 0x18d0, 0x2d6f, 0x1e21, 0x1cd0, 0xba62, 0xab70, 0xa93f, 0xba33, 0xbf92, 0xb23f, 0x41fc, 0x176e, 0x436f, 0x4246, 0x4119, 0x445f, 0xba22, 0xbf60, 0x4075, 0x45da, 0xbf70, 0x4011, 0xb29b, 0x1d67, 0x413f, 0x0a8d, 0x464c, 0xac75, 0x432d, 0xa2cf, 0x0872, 0x18ae, 0x4315, 0xbf66, 0xb249, 0xba19, 0x4074, 0xa674, 0x4009, 0xbf90, 0x4373, 0x4072, 0x1f85, 0x1968, 0x3a9a, 0x1e06, 0x44cd, 0x4152, 0x433c, 0xba71, 0x1835, 0x40bf, 0xbf80, 0x4150, 0xbf05, 0xb241, 0xba5b, 0x41a7, 0x4445, 0x4232, 0xb2c9, 0x418b, 0xb20c, 0x436c, 0x46b9, 0xba16, 0x411b, 0x40d0, 0x1af2, 0xb06c, 0xbf3f, 0x429d, 0x4296, 0xac36, 0x40b1, 0x19f1, 0x1a06, 0x131c, 0x403f, 0x19e2, 0xa234, 0xbf62, 0xba62, 0x2ad5, 0x1b63, 0x4048, 0xb00f, 0xb265, 0xb03b, 0xb2da, 0x17c2, 0xb0c6, 0x1c3e, 0x41c4, 0x42af, 0xbfae, 0x2515, 0xb2e3, 0x42f6, 0x4065, 0x21bc, 0x2a1c, 0x2d3e, 0x43ba, 0x40e0, 0x435e, 0xb24d, 0x0e36, 0x406f, 0x404b, 0x1c48, 0xbf97, 0xbf70, 0x4153, 0xb20f, 0xb25e, 0x43c9, 0x4226, 0x40dd, 0x405c, 0x43cb, 0x0d09, 0x19fa, 0x1d04, 0x430c, 0x4029, 0xb2fa, 0xbf1f, 0x0aa8, 0xb223, 0xb2ca, 0x413f, 0x1b91, 0x36e0, 0x19dc, 0x005f, 0x1420, 0x40fc, 0x439f, 0xbfb3, 0x0279, 0x1824, 0x1879, 0xa189, 0xba14, 0x33bb, 0x4266, 0x0602, 0x41c4, 0xba73, 0x40eb, 0x419b, 0xbfc3, 0x4179, 0x183d, 0x4339, 0x41bb, 0xa7fd, 0xb257, 0x42bf, 0x4551, 0x425d, 0xba0a, 0x1e0f, 0x200a, 0x4204, 0x42ca, 0xa1de, 0x0124, 0x4282, 0x1d1d, 0x411c, 0x18bd, 0x406d, 0xba1f, 0xbad9, 0x0714, 0x421f, 0xbf5f, 0x409f, 0x4317, 0x419c, 0xba34, 0x40e1, 0x1b14, 0xbf21, 0x38e6, 0x1016, 0x4387, 0x42c9, 0x10ca, 0x101a, 0x43e1, 0xb2a6, 0x1c7d, 0x4318, 0xbf49, 0x4573, 0x1e86, 0xb256, 0x40df, 0x43fb, 0xbfd3, 0xa2c5, 0x4175, 0xb22e, 0x18cb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xff38a7fb, 0x4577c13c, 0xbafd4bd9, 0x608edce2, 0x0b12ce77, 0xdd0d292c, 0xd713d1ca, 0x0199fdd1, 0xee4843e5, 0x33f7a879, 0xf40bded8, 0x113e3e39, 0xfa00e8ce, 0xeb295429, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0xffffffbe, 0x3ffeffff, 0x00001ae0, 0x3ffefffe, 0xc0010000, 0x000000dc, 0xffffffff, 0x00000000, 0xfa00e8ce, 0x5b129380, 0xf40bded8, 0xf40bded8, 0xfa00e8ce, 0x9b2a428a, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbae5, 0xb227, 0x44d8, 0x43d3, 0x41d7, 0x40a2, 0x41d7, 0x42f3, 0x42eb, 0xb065, 0x42f8, 0xbfd8, 0x1e0b, 0x4085, 0x4121, 0x4130, 0xbf3d, 0x4042, 0x43f3, 0xa770, 0x4317, 0xb214, 0x4005, 0x2688, 0xb26b, 0x4044, 0x41b2, 0x2c6e, 0x1b18, 0x1d98, 0x1f7c, 0xb0fc, 0x427e, 0xbfa4, 0x0b8c, 0xb2a3, 0x1991, 0x3cc3, 0xa055, 0x1bcb, 0x42a4, 0xaa13, 0xbf8c, 0x400e, 0x1fcd, 0x420e, 0x437f, 0xb009, 0x44e0, 0x40d3, 0x45ad, 0x4242, 0x435b, 0x24d1, 0xa139, 0x3049, 0xba6c, 0x1689, 0x2538, 0x009e, 0x12e4, 0x4226, 0x1db2, 0xa704, 0x41ff, 0xb252, 0xbf25, 0xb201, 0x45a3, 0xb2f1, 0xb06a, 0x10c5, 0x24cf, 0x42ed, 0x4420, 0x40d0, 0xb29b, 0xbf0a, 0x2ff2, 0x46d3, 0xba69, 0x44da, 0x0515, 0xba24, 0xb263, 0x1e73, 0x43ce, 0x4269, 0xbfb0, 0x4322, 0x41e8, 0xb229, 0x1d20, 0x46aa, 0xb2c3, 0x4613, 0x41d9, 0x0ac4, 0x4352, 0x4031, 0x40dd, 0xa769, 0xb008, 0xb09e, 0x29dd, 0xbf87, 0x437b, 0x0284, 0x4337, 0x4224, 0x417f, 0x3e53, 0x44a4, 0x400d, 0x1cf0, 0x4388, 0xbaf0, 0x4181, 0x4399, 0x2de9, 0x1200, 0xbf97, 0x4200, 0x41ea, 0xb0e3, 0x40f6, 0x4436, 0x4077, 0xa2c4, 0xbf6c, 0x1095, 0x1f06, 0x05cd, 0xb2e0, 0x40a7, 0xb0e6, 0x4090, 0xa9ac, 0x1fff, 0x4303, 0xb08d, 0xbad3, 0x413b, 0xbf2a, 0xbae2, 0xbfd0, 0x41d7, 0x437e, 0x42f2, 0xa025, 0x42d1, 0x3ebf, 0x1acc, 0xb28a, 0xa56f, 0x407b, 0x1ffc, 0x4124, 0x400b, 0x41fb, 0x1d4e, 0x40a0, 0xba67, 0x4383, 0x4239, 0x418c, 0x43e7, 0xba5c, 0xb03e, 0x4260, 0x404d, 0xbf8d, 0x1f0c, 0x1e74, 0x4132, 0x41ce, 0x4263, 0xb204, 0x1be7, 0x426d, 0x40ca, 0x408d, 0x4298, 0x183a, 0x1b8f, 0x4063, 0xb207, 0xbf7d, 0x4259, 0x2680, 0x44e4, 0x414f, 0x4390, 0x4218, 0x4111, 0x4477, 0xa2f6, 0xaac4, 0x40f7, 0xb24e, 0x026d, 0x1dae, 0x4296, 0x4151, 0xb0b1, 0x439b, 0x4264, 0x0f33, 0x1cee, 0xba12, 0x4026, 0x32d6, 0xbf26, 0xba1d, 0x4310, 0x1da4, 0xbfa5, 0xa234, 0x4551, 0xba63, 0xad72, 0x38dd, 0x421f, 0x418b, 0x4198, 0xba0c, 0xb063, 0xba67, 0xbf00, 0xa1f0, 0x419d, 0x459e, 0x2726, 0x40c7, 0xbad0, 0x4105, 0x1835, 0x4245, 0x4088, 0x411d, 0x0545, 0x42fa, 0x4549, 0xbf68, 0x0932, 0x3b74, 0x27a2, 0xbad1, 0x403e, 0x1edf, 0x125f, 0x4020, 0xba01, 0xb243, 0x40cc, 0x40c3, 0x431a, 0xba6a, 0x1e8f, 0xb297, 0x3805, 0x4385, 0x3068, 0x1ffa, 0x4568, 0x4240, 0x41c8, 0x449d, 0xbfda, 0x423a, 0x1853, 0xb217, 0xa4c3, 0x4260, 0x41e0, 0x405f, 0x4258, 0x4348, 0xb0ad, 0x407e, 0x42e6, 0x1c88, 0xbafc, 0x4112, 0x42f5, 0xba7c, 0xb200, 0xb243, 0x4050, 0xbf8c, 0x409c, 0x16eb, 0x4324, 0xb26a, 0xb0e0, 0xbf11, 0xba41, 0x4276, 0x3fbd, 0xb227, 0x46f9, 0x4096, 0x4115, 0x428f, 0x4341, 0xba68, 0x430d, 0x439c, 0x0258, 0xb0ac, 0x3247, 0xb2fd, 0x18dc, 0x4196, 0x1bc4, 0xbf69, 0x1fc3, 0x17db, 0x4028, 0x426b, 0xb045, 0x3fc7, 0x403b, 0x42b2, 0x4155, 0x4027, 0x4055, 0x4397, 0x40ca, 0xb2e4, 0xbfe0, 0x4067, 0x4213, 0xb019, 0x42f1, 0x18c4, 0xb26e, 0x2a83, 0x40d4, 0x43a9, 0x3df6, 0xb2d0, 0xbfd0, 0x1aa2, 0xbf38, 0x4283, 0x41a3, 0x4261, 0x07ee, 0x406e, 0x4323, 0x24cd, 0x413d, 0xb2d9, 0x4217, 0x41a7, 0x0450, 0x40e0, 0xb0f8, 0xb2cd, 0xbf6f, 0xa598, 0x36ad, 0xbafa, 0x4247, 0x428c, 0x419b, 0x2ddd, 0x2ff4, 0x42a3, 0x420b, 0x0665, 0xb09f, 0x25aa, 0xb222, 0x424b, 0x41d5, 0x084b, 0xbfdc, 0x448c, 0x1ee0, 0x427a, 0x42ec, 0xb24b, 0xb203, 0x4277, 0x40d6, 0x43b7, 0xbfce, 0x40fd, 0xb05a, 0x443e, 0x4319, 0xaf57, 0x4372, 0xbfb3, 0x0ea7, 0x1c4d, 0xa3d4, 0x4445, 0x424c, 0x0597, 0xb2e2, 0x44bb, 0xba22, 0x448a, 0x40b2, 0x0495, 0x1d64, 0x4227, 0xbaf8, 0xb2eb, 0x40e2, 0x350e, 0x38b4, 0x19de, 0x4351, 0x43d9, 0x4078, 0x0edb, 0x42b0, 0xbf08, 0x07cf, 0x41f1, 0xbff0, 0xbad4, 0x45a0, 0xbae6, 0xb001, 0xbad4, 0x430d, 0x425b, 0xbf6d, 0x0b2e, 0xa8b3, 0xb282, 0xbaff, 0xaf30, 0x0047, 0x18af, 0x2845, 0xbf0a, 0xba43, 0xb2e9, 0x445d, 0x42b1, 0x415c, 0x428b, 0x4305, 0x4298, 0x432a, 0x43ed, 0x142c, 0xab21, 0x4167, 0x211c, 0x4346, 0xbf7f, 0xb0a2, 0xb2a2, 0x23cf, 0xb22b, 0x41d2, 0x4187, 0x3931, 0x1ad7, 0x402b, 0x41d2, 0x4128, 0x0ad0, 0x4170, 0x1dac, 0xba34, 0x402d, 0x40e6, 0x41d2, 0x0a0b, 0x45f2, 0xa738, 0xa730, 0x4076, 0xbf74, 0x44f0, 0x2744, 0x1bf9, 0xbafa, 0xb258, 0xb215, 0xbf9f, 0x46b2, 0xb083, 0xb2d0, 0x18ba, 0xba65, 0x43ee, 0xbfdc, 0x0348, 0xb2a8, 0xbf72, 0x4345, 0x1fa2, 0xb08b, 0x433a, 0xbf14, 0xba2d, 0xbacf, 0x400b, 0xbac3, 0x4203, 0x443c, 0x40e6, 0x1f06, 0x43ab, 0xb2c5, 0xba4f, 0x417d, 0x43b4, 0x4044, 0x1f8a, 0x4299, 0xaa10, 0x183b, 0xbfd4, 0x454a, 0x3c00, 0xb2c6, 0x0d40, 0x411f, 0x4292, 0x11dc, 0xb2e2, 0x422f, 0xb2cc, 0x41f2, 0x438b, 0xbac1, 0x423e, 0x4329, 0x4390, 0xb019, 0x40e1, 0x458c, 0x4305, 0x09ce, 0x4666, 0x4459, 0x46dd, 0x0581, 0x4368, 0x34e4, 0xbf5c, 0xb22a, 0x4199, 0x4330, 0x42e2, 0x46ec, 0x3b8e, 0x2edc, 0x4173, 0x4356, 0x41e0, 0x43ab, 0x0b77, 0x4222, 0x1f4c, 0xba39, 0x407d, 0xb298, 0xb26d, 0x4088, 0xb0b2, 0x4055, 0xbf59, 0x4689, 0xb004, 0x4380, 0x406b, 0x19ab, 0x2018, 0x447c, 0xbf1f, 0x41e0, 0xb24f, 0x406b, 0x2d16, 0x05be, 0x4252, 0x008b, 0x1d45, 0x424b, 0x41fe, 0xb282, 0x4089, 0xbf25, 0xb252, 0x430c, 0xba72, 0x42e9, 0x3df2, 0x4351, 0x40af, 0xbf79, 0x4375, 0x1be2, 0x43b9, 0x28e9, 0xbf08, 0xb203, 0x4201, 0x40b3, 0x0265, 0xb060, 0x4236, 0x41dc, 0xbfe0, 0x1d87, 0x41bf, 0x4040, 0xb27e, 0x0ad0, 0x3812, 0x4141, 0x1d57, 0x43fc, 0x1e44, 0xba47, 0xa9bb, 0xbfbc, 0x414d, 0x4386, 0x1cd0, 0x4445, 0x4014, 0xba62, 0x19ed, 0x41d1, 0xbfbf, 0x41bc, 0xb0d9, 0x42b1, 0xb281, 0x3f21, 0xbf65, 0x409f, 0x4051, 0x004b, 0xba2b, 0x463d, 0xaa52, 0xa84c, 0xb0ec, 0x06ee, 0xa560, 0x463c, 0xb007, 0x2991, 0xb22c, 0xbf80, 0x3632, 0xb296, 0x3c62, 0x43cc, 0x120c, 0x42e2, 0x4030, 0xbfd0, 0x43a0, 0x0c17, 0x3416, 0x4624, 0xaae3, 0xbfd2, 0x454e, 0x430d, 0xba74, 0x41b8, 0x37e3, 0x3b51, 0x411f, 0x4200, 0x437a, 0x1e3e, 0x4098, 0xbf4b, 0x19a0, 0x4544, 0xa8a3, 0x41ec, 0xb0de, 0xbf80, 0x3899, 0x0994, 0x1c6c, 0x42d6, 0xba74, 0x43c6, 0xba20, 0x429d, 0x4010, 0x118b, 0x402c, 0xbf62, 0x42b4, 0x4638, 0x4269, 0x2880, 0xb015, 0x36fd, 0x4179, 0x1d14, 0x46da, 0x1a37, 0x21a6, 0xb0b8, 0xb2a7, 0x1f43, 0xba10, 0x1ea2, 0xba07, 0xb09e, 0x28a5, 0x426e, 0x3bc9, 0x41b8, 0x465e, 0x1cfe, 0xbf72, 0x41e6, 0x1bc3, 0xb04a, 0xbae4, 0x43df, 0x4281, 0x2e57, 0x408f, 0xb08e, 0x4069, 0xbfb0, 0x440a, 0xbfb0, 0xb230, 0xb0f2, 0x45b4, 0x402b, 0xb2e8, 0x4207, 0xbac1, 0x4186, 0xba45, 0x43d8, 0x43b2, 0x42cf, 0x41ad, 0x432d, 0xbf66, 0x401f, 0x43f5, 0x44f8, 0x42af, 0xa978, 0x439a, 0xb2fd, 0xb259, 0xb24a, 0x42b1, 0xba6d, 0x398d, 0x4190, 0x43cf, 0xb2a3, 0xba25, 0x26d9, 0x43db, 0xba1f, 0x1d49, 0x197a, 0xb0bc, 0x40ae, 0xa9b4, 0x3109, 0xbf03, 0x4125, 0x31de, 0x459b, 0xb2e0, 0xba14, 0x41bb, 0x437a, 0x425a, 0x1a58, 0x288b, 0x1d8f, 0xb084, 0x4235, 0x4349, 0x1631, 0x4043, 0x2199, 0x075f, 0xb071, 0x0a46, 0xba55, 0x40a5, 0x26e7, 0xbf00, 0x3d20, 0xa8f9, 0x4025, 0x44c4, 0xbf2a, 0x40eb, 0xba2a, 0xba0b, 0x4115, 0x42c2, 0x1d34, 0xba10, 0xba7c, 0x4079, 0x1b5d, 0xba50, 0x4369, 0x3eba, 0x400b, 0xb0b8, 0xbae1, 0x41f2, 0x43ef, 0x42a4, 0x4559, 0x242c, 0xacb2, 0xad93, 0xbf72, 0x4297, 0x42fc, 0x4269, 0x43b5, 0x1e97, 0x40dc, 0x35e6, 0xbacf, 0x1a73, 0x411a, 0xbae3, 0xbf33, 0xb2b2, 0x34a3, 0x425a, 0x405e, 0x3399, 0x21cc, 0x195d, 0x435e, 0x4099, 0xbfa0, 0xae2f, 0x43d9, 0x0109, 0x4025, 0x4155, 0x4462, 0x1602, 0x43bd, 0x4248, 0x4194, 0x434a, 0xabb9, 0xb2c8, 0xbf4d, 0x413d, 0xba58, 0x03e3, 0x40f5, 0x403c, 0x1598, 0x4265, 0x1f54, 0x4188, 0xb0aa, 0x18e2, 0x4223, 0xbfb0, 0x410b, 0x40f5, 0x1ab4, 0x23b7, 0x19ba, 0xbf7f, 0x18fe, 0x199f, 0x4011, 0x410d, 0x27b0, 0x4339, 0x1d01, 0x2979, 0x31af, 0xa4a3, 0x1dfb, 0x401b, 0x40b7, 0x3d6e, 0xb22d, 0xbf64, 0xb040, 0x3846, 0xbfbf, 0xb24b, 0x41fd, 0x2226, 0x43cf, 0x3af2, 0x4365, 0xb003, 0x4333, 0xaa10, 0x1ca0, 0xb0f2, 0xbf9f, 0x1fed, 0xb09e, 0x41b3, 0x468b, 0x40f3, 0x4398, 0x413c, 0x40b8, 0xb017, 0xb226, 0x43f4, 0x4340, 0x41d7, 0x01f2, 0x4374, 0x410a, 0x28e5, 0xb07f, 0x0cf8, 0x40c0, 0x422f, 0xbfdd, 0x4023, 0x085c, 0xb22c, 0x41c8, 0x4433, 0x45c9, 0x43d7, 0x4378, 0xbf06, 0x4570, 0x40e0, 0xb004, 0xb292, 0x449d, 0x1d73, 0xb24e, 0x4347, 0x4071, 0xb2d4, 0x1aed, 0x43ec, 0x1f32, 0x43e5, 0xa386, 0x3bc4, 0x4418, 0x40d4, 0x4031, 0xbfd5, 0x41f7, 0x401c, 0x41cd, 0x2c18, 0xb04b, 0x43da, 0xbafd, 0x41de, 0x4169, 0x406e, 0x43ee, 0x204a, 0xbf41, 0xbaff, 0x42ab, 0xbac3, 0x4691, 0xb269, 0x188e, 0x4277, 0xbf81, 0x43c8, 0xb223, 0x1dfd, 0x2342, 0x12d6, 0xb254, 0x19ce, 0x359d, 0x4343, 0x1cc8, 0x00c8, 0x26e1, 0xa58f, 0xb2c4, 0x4047, 0x3780, 0x4157, 0x3829, 0xbf48, 0x447c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf96275eb, 0x02fe41e7, 0xeefa2e22, 0x4b46bc43, 0x003dddf6, 0xa6272337, 0xb656ca8c, 0x7ccb9d65, 0xf8d3063c, 0x09bff66f, 0x1d231647, 0xe7c3d8f9, 0x0fbd2fe9, 0x8a2c763d, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0xffffffd7, 0x00000000, 0xffffe71f, 0x000730c0, 0x000017ea, 0x00001a18, 0x000000e1, 0x00000080, 0xf054250c, 0x00000000, 0x1d231647, 0x00000a54, 0x0d773b53, 0x1d23118b, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x409f, 0x4371, 0x4362, 0xb217, 0xbf00, 0x43fa, 0xbf8c, 0x18bb, 0x42a0, 0xba1a, 0xada9, 0x403c, 0x00d4, 0xbf02, 0xb01d, 0x43d0, 0xb264, 0xb2aa, 0xb0dd, 0x425d, 0x1dc1, 0x0378, 0xb072, 0x1d92, 0x2fa8, 0x4140, 0x40ac, 0x4620, 0x0a08, 0x42a6, 0xba07, 0xbafe, 0x4171, 0x40c2, 0x4376, 0x41db, 0x40e1, 0x4305, 0xbf97, 0x14b7, 0xba36, 0x4393, 0x1f2f, 0x4483, 0xbf46, 0x2407, 0xba63, 0x4096, 0x4103, 0x4650, 0x2684, 0x41aa, 0x4377, 0x43fc, 0x11ca, 0x4595, 0x3398, 0x43ef, 0x4634, 0x436d, 0xb060, 0x0ea8, 0xb2b5, 0x2045, 0x43ef, 0x46a4, 0x40b6, 0x40bd, 0x127f, 0x290e, 0x4087, 0x422c, 0x1856, 0xbf8c, 0x4065, 0x3e07, 0xb004, 0x44ec, 0xbf4c, 0x193a, 0xbaf4, 0xafa0, 0xb2c6, 0x406a, 0x1865, 0xbfc0, 0x19b2, 0x43d6, 0x1b78, 0xb039, 0x407a, 0x4163, 0x411a, 0x1d8c, 0x04e2, 0x1988, 0xb215, 0x40a0, 0x13bd, 0x38a5, 0xb2ce, 0x403b, 0x4291, 0x0562, 0x4680, 0x402c, 0xbf2e, 0x409c, 0xbaf0, 0x433f, 0xa329, 0x4031, 0x3824, 0x403f, 0xb2c0, 0x428b, 0x0991, 0x4153, 0x3e46, 0x4097, 0x2e7d, 0x05f3, 0xb2ee, 0xbade, 0x2267, 0xba3a, 0x198c, 0xbf6d, 0x461a, 0x337e, 0x43f0, 0xb225, 0x42a7, 0xb2ab, 0x45f3, 0x1f4b, 0x421a, 0xbac9, 0xbf47, 0x42c3, 0x44b4, 0x1c39, 0x0574, 0x4051, 0x4010, 0xb20e, 0xb2b3, 0x4575, 0x2bae, 0x1821, 0x366d, 0xb273, 0xb237, 0xbf4e, 0x425c, 0x4340, 0x191e, 0xbaf4, 0xbadf, 0x1a8b, 0x118f, 0xbfb0, 0x1ee0, 0x0041, 0xbfd9, 0xa940, 0xba2a, 0x4389, 0xba3a, 0x410e, 0xb03c, 0xba6a, 0x0b96, 0xb2d5, 0x1cb5, 0x026c, 0x4176, 0x1c59, 0x43d8, 0xb291, 0x3bf6, 0x1cff, 0x0650, 0xb2f6, 0x4348, 0x460d, 0x411d, 0x413c, 0x42fe, 0x0242, 0x415c, 0x088e, 0xbfb4, 0x136d, 0x2a54, 0x4183, 0xb2ae, 0x4457, 0xb0ff, 0x2d0f, 0x1f2b, 0x437a, 0x2ba1, 0xbfb4, 0x439e, 0x4196, 0x43e3, 0xb23e, 0x1be7, 0x4181, 0x4250, 0xbfd0, 0x413d, 0xbf80, 0x4289, 0xb058, 0xbac1, 0x432b, 0x40b4, 0xbf8a, 0x400d, 0x236d, 0x42a4, 0xa46c, 0xb053, 0x2598, 0x015b, 0x0966, 0x4335, 0xb29d, 0xbf51, 0x4185, 0xb2f6, 0xac2b, 0xb2fa, 0x3432, 0x4628, 0x2d72, 0xb053, 0xa1e1, 0x059e, 0xb225, 0x42e2, 0x4383, 0x400f, 0x42bc, 0x190e, 0x4094, 0x43f7, 0xb245, 0x403e, 0x41b3, 0x419d, 0xbf65, 0x4045, 0xb013, 0x417b, 0x0ea0, 0xaf84, 0x42d0, 0xb228, 0x40f7, 0xba01, 0xbfc0, 0xb2d9, 0x1dc9, 0xbfc6, 0x1aca, 0x4375, 0x1b51, 0x1f9c, 0x1b0a, 0x0661, 0x447e, 0x436a, 0x4051, 0xad93, 0x41b5, 0x4213, 0x006d, 0x40da, 0x4619, 0x42d5, 0xbaf4, 0xaabe, 0xbf70, 0x38e2, 0xbf8b, 0x40ce, 0xb21f, 0xbfd0, 0xb24e, 0xab0e, 0xbae3, 0xb25d, 0xb2c7, 0x443c, 0x42f8, 0x4173, 0x420a, 0xb2d7, 0x42dc, 0xb283, 0xa7ba, 0xbfb0, 0x4074, 0xb2d4, 0xbfb2, 0x40b8, 0x40bb, 0x1964, 0x40ff, 0x43e7, 0x4290, 0xb26c, 0xb226, 0x240c, 0xb090, 0x4321, 0xb26b, 0x4234, 0xba39, 0xba20, 0x40e5, 0x4153, 0x0199, 0xb262, 0x0485, 0x40f7, 0x2228, 0xbfe0, 0x4241, 0x4287, 0x4352, 0xb233, 0x1f2a, 0xbf07, 0xb0fb, 0x19b1, 0x3a72, 0x41c3, 0x1d5c, 0x0cfd, 0xbf90, 0xb293, 0xbaca, 0xbf6d, 0xba19, 0xb06f, 0x2275, 0xb246, 0xae73, 0x405f, 0xb247, 0x40c7, 0x1cb3, 0xb2ef, 0x4125, 0x413d, 0x1e18, 0x28fe, 0xb0a7, 0xb0af, 0xbfb5, 0xba0e, 0x40ee, 0x41f1, 0x424b, 0x4308, 0x17c4, 0xb00a, 0x42ff, 0xbf49, 0x21f1, 0x409f, 0x43b7, 0x1990, 0xbfa6, 0x4602, 0x1a37, 0xb28e, 0x4223, 0x401b, 0x0ee5, 0x4382, 0x4017, 0xba24, 0x402e, 0x4220, 0x41a7, 0x411b, 0xb2e5, 0x42f2, 0x41f2, 0xb247, 0xbf90, 0x430e, 0x2206, 0x3b7b, 0x4253, 0x4376, 0x4017, 0xba4d, 0x4257, 0xbf28, 0xb2a0, 0x4194, 0x42b6, 0x1919, 0x40c2, 0x467a, 0x4374, 0xb2cd, 0x1d2c, 0x4462, 0x40d7, 0x402a, 0xba75, 0xb0f9, 0x44c0, 0x428a, 0x3166, 0x0ecd, 0xbf07, 0x0e7f, 0xb03c, 0xb023, 0x416d, 0x4106, 0xba15, 0x072f, 0x1f48, 0x1d5c, 0x45b6, 0x3b35, 0x4003, 0xbff0, 0x3519, 0x4268, 0x40a9, 0x4364, 0x40a1, 0x412e, 0x4393, 0x462b, 0xa299, 0x2497, 0xb286, 0xacdc, 0xb027, 0xbf70, 0xbaee, 0xb213, 0xbf5c, 0xb00c, 0x4376, 0x4541, 0x33d7, 0x41d9, 0x43a6, 0x2c34, 0x44fc, 0xb001, 0x05ac, 0x40d4, 0x4201, 0x2dc1, 0xbaf3, 0xb201, 0xba73, 0x1f51, 0xb20a, 0x43ab, 0x4100, 0x4389, 0xba77, 0xbf3f, 0xa8c2, 0xba34, 0x41ce, 0x43fa, 0xb059, 0x46db, 0xbf7f, 0xb2e4, 0x43d5, 0xba10, 0x41c4, 0xbad6, 0x41df, 0x1918, 0x4117, 0x117b, 0xbf00, 0x42ed, 0x442e, 0x4229, 0x4393, 0x422e, 0xba58, 0x4058, 0xb270, 0xba2f, 0x40a7, 0x36a0, 0x405b, 0x2f9e, 0xbfb9, 0x41df, 0x4201, 0x3389, 0x405f, 0xa198, 0x43f0, 0x4151, 0x410c, 0x2c84, 0x2f92, 0x4206, 0xba04, 0x462b, 0xbf1a, 0x41eb, 0x2a6c, 0x4245, 0xa394, 0xb28e, 0xb24e, 0x04f0, 0x46e1, 0x42b3, 0xb03f, 0x30ef, 0x1d17, 0xbaec, 0xb2f8, 0x4168, 0x1b83, 0xbfac, 0x2cce, 0x4159, 0xb227, 0x45a2, 0x43d6, 0x43ca, 0x4124, 0xbf07, 0x0220, 0x40f0, 0x4201, 0xb224, 0x32db, 0x4340, 0x4145, 0x42c3, 0xbfa0, 0xb2e1, 0x42ee, 0xb0ed, 0x40e9, 0x414b, 0x42bd, 0x4555, 0x09ca, 0xb076, 0x1fcf, 0x09e5, 0xaf80, 0xb209, 0x40f0, 0xb04a, 0x4015, 0x41b1, 0x0408, 0xbfa8, 0x177c, 0xbf60, 0x41f9, 0x4251, 0xbf1f, 0x464a, 0xbf70, 0xba33, 0x425c, 0x4233, 0x454e, 0x4374, 0xbadb, 0x4105, 0x431b, 0x32b3, 0xb042, 0xb242, 0x4357, 0x28d3, 0x20b8, 0x2d3a, 0xba3a, 0xba75, 0x409b, 0xb2e3, 0x2cf2, 0x4078, 0xb257, 0xbf55, 0x427d, 0xb29a, 0xbad7, 0xbff0, 0x4207, 0x3766, 0x1fff, 0x1e14, 0x3528, 0x41c9, 0x27a9, 0x4114, 0x44f8, 0x18b7, 0x1c48, 0x4278, 0x2785, 0x1db7, 0x4473, 0x41f3, 0x18f7, 0x2d3c, 0xbfd8, 0x1c68, 0x402d, 0x43db, 0x4017, 0xb049, 0x401a, 0xba2f, 0xb090, 0x4558, 0x42e2, 0xba3b, 0x444c, 0xb01f, 0xad32, 0x4391, 0xbafc, 0xbf18, 0x418c, 0xb224, 0xb2bd, 0x1d32, 0x4076, 0xac73, 0x3419, 0x4128, 0x435b, 0x44bb, 0x42ed, 0xba33, 0xaaf8, 0xba11, 0xbf80, 0xb04f, 0x4253, 0x3fd6, 0x43df, 0xba71, 0xb255, 0xbf23, 0xbad5, 0xb01d, 0x4054, 0x4178, 0x4213, 0xbad9, 0xbf5f, 0x43e8, 0xb204, 0x45bd, 0x311b, 0x4286, 0xb261, 0x2ba0, 0x1a16, 0xb042, 0x4417, 0x1915, 0x4220, 0x429f, 0x4223, 0x41b3, 0xbac5, 0xb06a, 0x1b2c, 0x1452, 0xbf74, 0xb251, 0x423d, 0x1d7d, 0x43f3, 0xba14, 0x230c, 0x41e8, 0x436a, 0xb2c1, 0x40fd, 0x413f, 0x420a, 0xb2dd, 0xb28c, 0x46a5, 0xb291, 0xb2ee, 0x1f83, 0xb2e8, 0xb2cf, 0xa1bd, 0xb0c7, 0x1ba4, 0xbae4, 0x425a, 0x41f3, 0xb04b, 0xbfdc, 0x0f78, 0xba29, 0xb0e8, 0x42c9, 0xb291, 0xba18, 0x432b, 0xbfac, 0x4109, 0xab55, 0x436f, 0x4349, 0x4132, 0x41d9, 0xb240, 0x4227, 0x43c5, 0x403c, 0x1d71, 0x4182, 0x403c, 0x40e0, 0x4077, 0xb27b, 0x45bc, 0xa153, 0xb017, 0x429e, 0x42ee, 0x4453, 0x4064, 0xb0c1, 0xbf43, 0x0c27, 0x0f14, 0x3796, 0x433a, 0xad1c, 0xb27f, 0x431e, 0x1f68, 0xbf00, 0x3852, 0x43b8, 0x32f1, 0x4380, 0x1ec3, 0x1500, 0xa820, 0xa8b8, 0xbf73, 0xb28d, 0xac37, 0x1ab0, 0x0759, 0xaf00, 0xb28d, 0x40e7, 0x4314, 0xbaf9, 0xb05d, 0x33b3, 0x372a, 0x430c, 0x421d, 0x432d, 0xbfc7, 0x14a2, 0xba0b, 0x2f72, 0x45d5, 0x40da, 0xbf94, 0x4398, 0xb2c2, 0x10af, 0xba46, 0xaedb, 0x41ed, 0xba66, 0xba70, 0xb28d, 0x4130, 0x4106, 0x225a, 0x42d1, 0xba7f, 0xb07b, 0x41f3, 0x18bc, 0xa31f, 0x1c7e, 0x2979, 0xb267, 0x430c, 0xbfb3, 0x4005, 0x45b6, 0x1d0a, 0xb29c, 0xbfe8, 0x413f, 0x4212, 0x1a6c, 0xb2cd, 0x41a3, 0x1b19, 0xad5e, 0xba4a, 0x155b, 0x0e84, 0x4041, 0x43fc, 0x41ac, 0xba0c, 0x41ab, 0x38f2, 0x43c9, 0x1f49, 0xb006, 0xb218, 0x0657, 0x0076, 0xb071, 0xb292, 0xbf33, 0x42d9, 0x419a, 0x430a, 0x40d0, 0x43ed, 0xae7a, 0xba09, 0x1dee, 0x4172, 0xbac3, 0x40b6, 0x418f, 0x4102, 0xba3a, 0xb255, 0xa8b8, 0x28b4, 0xbf2a, 0xbac3, 0x46b3, 0xb029, 0x402b, 0x18ac, 0x14fb, 0xbf26, 0x42ff, 0x4545, 0x413c, 0x43f9, 0xb099, 0xb2ea, 0xa112, 0x4037, 0xac7a, 0x42d3, 0xb2d3, 0x4136, 0xba7d, 0x4081, 0x40fb, 0x4421, 0x41ea, 0x46eb, 0x43d5, 0x1956, 0x3dd2, 0x1e1c, 0x4323, 0x1833, 0xb0c3, 0x4267, 0x428e, 0xbad9, 0xbfb5, 0x1cb6, 0x4248, 0x42b6, 0x19fb, 0xb0a6, 0x1a7f, 0xba5a, 0x1a28, 0x4446, 0x4207, 0x3fd4, 0x4119, 0x4069, 0xb215, 0xa19b, 0x4392, 0xbaef, 0xb2de, 0x1c2a, 0x46f5, 0xbf73, 0x4286, 0xbff0, 0x4006, 0xb279, 0xb27f, 0x428e, 0x416f, 0x411a, 0xbfd8, 0x0c5e, 0x444e, 0x43c3, 0x156d, 0x4175, 0xb2fb, 0x3877, 0xb2d7, 0x400a, 0x4429, 0x38fe, 0xb27a, 0x1868, 0x429f, 0xb0d5, 0x1515, 0x42ce, 0x45a1, 0x43e7, 0xbf55, 0xba1e, 0x4365, 0x1d96, 0x42fc, 0x43f5, 0x41ed, 0x4020, 0x4390, 0x1fa2, 0xbfd6, 0xb25f, 0x4128, 0x43ad, 0x1f47, 0x4373, 0x41e2, 0x4040, 0x4102, 0xad91, 0xb2ea, 0x4064, 0x44f9, 0x1cc9, 0x42d0, 0x2e1d, 0xb094, 0xbad1, 0xb0bd, 0x4144, 0x4074, 0xbfa0, 0x00b7, 0x4556, 0xb21d, 0xba47, 0x42e0, 0xb0fb, 0xbfcd, 0x193b, 0x4266, 0xb230, 0x41ec, 0x4271, 0x416e, 0xa323, 0x4225, 0xa208, 0x4057, 0x356c, 0x2a9d, 0x0db2, 0xbfc1, 0x19d8, 0x4242, 0x4240, 0x41eb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x1ecc668c, 0x2e9c60ff, 0x9990f0ee, 0xf1a38886, 0x3f294482, 0xc54ff2ce, 0x36359d5d, 0x6e0e101b, 0xc58ac456, 0x564992fe, 0xcc932f92, 0x05cbf818, 0x7c022c30, 0xc34238ff, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0xffffcfb8, 0xc342cf05, 0xffffcfb8, 0x80000185, 0xc342cf05, 0x000013a4, 0x3cbd4433, 0x000017f0, 0x0000138c, 0xc34266a3, 0xcc932f92, 0x00000371, 0xc3424f05, 0xfffffb7c, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x430e, 0x1b60, 0x2a48, 0xbadf, 0xb061, 0x420b, 0x41bf, 0xba6c, 0xb2dd, 0x3070, 0xb2ae, 0xb21f, 0x4226, 0x4260, 0xbaef, 0xb233, 0x4645, 0x439f, 0x2b19, 0x394a, 0xba1e, 0xba60, 0x40fe, 0xba31, 0xbf48, 0x4296, 0x433d, 0x4353, 0x26f9, 0x42c3, 0x4286, 0x2142, 0xba78, 0xa6be, 0x2ec6, 0xba58, 0xb2e3, 0x43c8, 0x43ce, 0x3eb2, 0x1c58, 0xbf61, 0xb24d, 0x1a44, 0xb2f5, 0xa49d, 0xba3d, 0xba1c, 0x420f, 0x0ec6, 0xbf1f, 0x4636, 0xad48, 0x4181, 0x4155, 0x3050, 0x4001, 0xba03, 0x422b, 0x0266, 0xb0c4, 0xba50, 0xa8b4, 0x1e73, 0x41f5, 0xa645, 0x436a, 0x277c, 0x432a, 0x1b94, 0x1d44, 0xb294, 0x4080, 0x42a6, 0xb256, 0xbf2d, 0x1926, 0x4000, 0x4396, 0xb221, 0xb071, 0x0912, 0xba74, 0xb2d9, 0x3bd3, 0x45f5, 0x35e1, 0xb2a0, 0x42f2, 0x1e6b, 0x1595, 0x0e57, 0x42a0, 0x45d6, 0xba44, 0x41a2, 0xb26d, 0x43d3, 0xa79e, 0x434c, 0xbfd9, 0x40a3, 0x418c, 0xb230, 0x2774, 0x414a, 0x4385, 0xba77, 0x4019, 0x4202, 0x4251, 0xbafe, 0x4243, 0xb0e9, 0x1fc2, 0x4196, 0x2115, 0x4570, 0x420b, 0x4344, 0x40fc, 0x1820, 0xbf13, 0x40f3, 0xbaf1, 0x4280, 0x4110, 0x43ba, 0x25f9, 0x4175, 0xbfa0, 0x09ba, 0x35e7, 0x41ec, 0x12ba, 0xb249, 0xbade, 0x3bf7, 0x3c20, 0xba5f, 0x4094, 0x3b99, 0x2bb1, 0xb2e3, 0x4166, 0x402f, 0x414c, 0x425d, 0x415d, 0x43ba, 0x40da, 0xbfdf, 0x45e1, 0x412c, 0xb2da, 0x44d5, 0x4180, 0x3b39, 0x4221, 0xbf62, 0xba33, 0x42c8, 0xadf9, 0xb2af, 0xb28f, 0x4254, 0x4121, 0x393f, 0xa074, 0x4320, 0xb0ef, 0x400e, 0xbf37, 0x41a4, 0xbfb0, 0x39e8, 0x4226, 0xafd2, 0x400d, 0x435e, 0x42f0, 0xb223, 0x403e, 0xba0c, 0xb09d, 0x42a0, 0x2b3a, 0x4016, 0x4308, 0xae2c, 0x3e99, 0x06a6, 0x45cb, 0x0741, 0xba55, 0xbf93, 0x1c12, 0x40e2, 0x24c4, 0x1fb7, 0x41ca, 0x2258, 0xbf5d, 0x3ccc, 0x420f, 0x41d4, 0xba60, 0x402d, 0x40e3, 0xb261, 0xbfcd, 0x415d, 0xb01f, 0x1901, 0x420b, 0x1a54, 0x3fa6, 0x4021, 0x1dcc, 0xb260, 0x425f, 0xb038, 0xba79, 0xba11, 0x103d, 0x1f3c, 0xba7c, 0xb283, 0xbfba, 0xb24d, 0xb273, 0x3b72, 0x3c0a, 0x40ca, 0xb2c3, 0x3882, 0xb282, 0xba42, 0x402a, 0x4655, 0x43bf, 0x2362, 0x43c6, 0x4148, 0xb08d, 0x4072, 0xa1f3, 0x3784, 0xb2a2, 0x438a, 0xbf3c, 0x22a4, 0x401c, 0x4201, 0xa508, 0xbae9, 0x4490, 0xb2ea, 0x1de4, 0xbae9, 0x43ab, 0x3fcb, 0x4028, 0xa9ca, 0x1a68, 0x2cfc, 0xb01d, 0xbfb8, 0x406b, 0x3db1, 0x41ff, 0x2386, 0x3663, 0xb206, 0x421c, 0xab7f, 0x41c8, 0xba13, 0x2a54, 0x41db, 0x407d, 0xb066, 0xbf1f, 0x1a65, 0x1d6f, 0x066a, 0x428d, 0x4065, 0x404f, 0x1ac2, 0x44db, 0x0b90, 0x4488, 0x4255, 0x4550, 0xb0c3, 0x4291, 0x414c, 0xb2e4, 0x4231, 0x430e, 0xb240, 0x4003, 0x1210, 0x43b1, 0xb246, 0x4344, 0xbf4f, 0x44f9, 0xb218, 0xba07, 0x4053, 0xbf60, 0x4248, 0x1432, 0x45c3, 0x4083, 0x4403, 0x2f70, 0xbfb9, 0xaecd, 0xb050, 0x40e7, 0x409f, 0xaeeb, 0x40d6, 0x4315, 0x125b, 0x400b, 0xb0a3, 0x44db, 0xbf37, 0x4202, 0x4179, 0x1d0e, 0x2d2d, 0xa705, 0x43ee, 0x3206, 0x1586, 0xb25f, 0x37da, 0x423c, 0x4151, 0x401f, 0x04c3, 0x219b, 0x22dd, 0x402d, 0x1c87, 0x4201, 0x41bb, 0xbfa6, 0xb040, 0xaf1a, 0x4372, 0xbadd, 0xbf02, 0x0c72, 0xb20c, 0x41cd, 0xb2e9, 0x4115, 0x1f3c, 0x4000, 0x44e1, 0xb2c9, 0x31aa, 0xbac6, 0xb083, 0x2dc0, 0x432d, 0x2a5c, 0xb2d3, 0x4019, 0x46ed, 0xbf6f, 0x4322, 0x4466, 0xb249, 0x2cf1, 0x4387, 0x4170, 0x43e0, 0x40a7, 0x16ff, 0x1793, 0x29d4, 0xba1d, 0x2029, 0x4173, 0xb217, 0x434f, 0xb231, 0xa7fa, 0xbf5e, 0x43ce, 0xb0ae, 0x3cf2, 0x3a2e, 0x43c6, 0x2f25, 0x4304, 0x0394, 0x43de, 0x4312, 0x429c, 0x1fd2, 0xb2c4, 0x3066, 0x4159, 0x4192, 0x46f8, 0xbae6, 0x412d, 0x400d, 0x4272, 0x3c14, 0x427f, 0x447c, 0xb294, 0x4609, 0xbf88, 0x43c4, 0x35ea, 0x4308, 0xbacc, 0x42b6, 0x433f, 0xa5b2, 0x43f6, 0xb212, 0x0a9d, 0xbae2, 0x118e, 0x4214, 0x4344, 0x1ff4, 0x18ec, 0xb025, 0x1865, 0x04b5, 0xbf63, 0x4102, 0xbae5, 0x41a5, 0x40fc, 0x1fee, 0x42f4, 0x438e, 0x1d08, 0xb27c, 0xba61, 0x40b7, 0x03e2, 0x1bcf, 0x34ad, 0xb2b0, 0x007d, 0x3c4f, 0x41d8, 0x45d1, 0x41ac, 0x442b, 0x424d, 0x4279, 0xa121, 0x44a9, 0x41f3, 0x01c8, 0x40bd, 0x414f, 0xbfcf, 0xba19, 0x42ea, 0x2383, 0x00d8, 0xba6a, 0xba41, 0xb01c, 0x317c, 0x40a8, 0x401d, 0xb054, 0x400b, 0x1d1a, 0x1fd3, 0x1b51, 0xbfa8, 0x403c, 0xb003, 0x4336, 0x3dd6, 0x40d2, 0x410b, 0xafc3, 0xabd6, 0xa7d5, 0x02c0, 0x4201, 0x4242, 0xa5f6, 0xb061, 0xba2d, 0x1be8, 0x0262, 0x434c, 0xa56e, 0xbf95, 0xba08, 0x22fd, 0xb22d, 0xa383, 0x1876, 0x024b, 0x4134, 0x2c98, 0x4232, 0xa829, 0x41cc, 0x4237, 0x4287, 0x44db, 0x46fa, 0x415c, 0x432b, 0x416d, 0x4231, 0xaba6, 0xb208, 0x402f, 0xba7d, 0xa377, 0x4005, 0xacdb, 0xbfc2, 0x1ce5, 0xbf70, 0xb26c, 0x3bb5, 0x05cf, 0xb0f3, 0xb222, 0x4398, 0x4632, 0x4132, 0x4371, 0x1bd3, 0xb212, 0xbfac, 0xbf80, 0x41e2, 0xbf3e, 0x431e, 0x0ade, 0x4225, 0x26de, 0x4062, 0x4660, 0x4054, 0xbf5a, 0x43b4, 0x4282, 0x4255, 0x1cb3, 0x286c, 0xba6f, 0xabbd, 0xb0f6, 0x4206, 0x4154, 0x40f2, 0x42e7, 0x30f7, 0x4005, 0x45b6, 0xbf9f, 0x40a2, 0xb2b8, 0xba26, 0x2711, 0xad55, 0x4382, 0xb0b4, 0x1990, 0xba12, 0xb2e0, 0xba00, 0xa4f2, 0x4356, 0xb211, 0x435a, 0x4052, 0xaaa8, 0x404c, 0x41d0, 0x1cb9, 0xb28e, 0x41dc, 0x2275, 0x4492, 0x0359, 0xb2e5, 0xbf4e, 0x1301, 0x44d5, 0x4122, 0x39cf, 0x347b, 0x43cd, 0x28be, 0x1851, 0x1a41, 0x40e7, 0xba0e, 0x18bd, 0xb095, 0xb269, 0x4029, 0x1879, 0x1b1b, 0x4156, 0x4210, 0xbf3b, 0x44ac, 0x405e, 0x42b1, 0x4306, 0xb297, 0x4198, 0xb002, 0x4232, 0x3704, 0x3643, 0x4220, 0xbf9a, 0x3986, 0xb2a7, 0x412d, 0x416f, 0x3c6c, 0xba06, 0x1bda, 0x435d, 0x421d, 0x41cf, 0xa2e4, 0x1cf3, 0x442b, 0x40ec, 0xb281, 0x1ebc, 0x4373, 0x40c9, 0xac98, 0xbff0, 0xa100, 0x409f, 0xb290, 0x41f9, 0x1733, 0xb255, 0x1efe, 0x4004, 0xbf3e, 0xb0c3, 0x4023, 0x09b0, 0x0995, 0x1ec1, 0x2998, 0xba06, 0x1e93, 0x43cc, 0xb2de, 0x42d3, 0x41ac, 0x4354, 0x4013, 0xa9a8, 0xbfd4, 0xb241, 0x4022, 0x1dcf, 0xb0ee, 0x43cc, 0x1a03, 0xbac9, 0x1b7e, 0x4311, 0x439e, 0xbf8a, 0x025e, 0x43b7, 0xa7ed, 0x4276, 0x4200, 0x42fa, 0xb04a, 0xa648, 0x418e, 0x40bd, 0x09bd, 0xbaec, 0x42d1, 0x2ede, 0x1b6e, 0x06b5, 0x08c6, 0xb2ba, 0xb2f6, 0x1d2f, 0x1aac, 0xba61, 0xbf2d, 0xb283, 0x4311, 0xb203, 0x4193, 0xa79a, 0x1eb0, 0x436e, 0xba53, 0x4612, 0x42d8, 0xb202, 0x3e0c, 0xaf8a, 0xba34, 0x4084, 0xabe0, 0x19fe, 0x4331, 0x4337, 0xb247, 0x4647, 0x407b, 0x31ff, 0xbf1c, 0x4148, 0x385b, 0x3fa5, 0xb2e5, 0x1e35, 0x087e, 0xaef2, 0xb093, 0x424c, 0xb227, 0x43dc, 0x102d, 0x4612, 0xb2c3, 0xb221, 0xb2ac, 0xbf24, 0x4359, 0x42b6, 0xb25d, 0x3d87, 0x42e8, 0xbaea, 0x436d, 0x1a91, 0xbafc, 0x42db, 0xa79c, 0x1946, 0x4140, 0x40af, 0x4095, 0x1ce6, 0x12fc, 0x2a6a, 0x40fd, 0x1e44, 0xbf75, 0x1ecd, 0x4114, 0x1fb8, 0xb07d, 0x41e1, 0x4306, 0x4040, 0x4007, 0xba64, 0xbf70, 0x4011, 0xbf39, 0x40be, 0xb257, 0xb05a, 0xb000, 0x2cf7, 0x0fae, 0xaa47, 0xbf1f, 0x4095, 0x43a6, 0xbfc0, 0x34a2, 0x3dc5, 0x42fe, 0x4329, 0x4135, 0xbae2, 0x46b9, 0xbfdb, 0x1c14, 0x4482, 0x432a, 0xb29a, 0x1fb4, 0x1869, 0x199f, 0x43b6, 0x3cd5, 0x437e, 0x2331, 0xb201, 0x4348, 0x2663, 0x41c9, 0x421e, 0xb2a6, 0x41a1, 0xbfc4, 0x3897, 0x4458, 0x41a9, 0xb2c6, 0xbfbf, 0xbf70, 0x1c52, 0x4132, 0x1e34, 0x0d87, 0xbae6, 0xb2b4, 0x2385, 0x41c1, 0x42ab, 0x1a6a, 0x07b7, 0x40aa, 0xbf21, 0x403b, 0xbaee, 0x43d9, 0x43ef, 0x41ab, 0x426c, 0x4280, 0x4019, 0xbf60, 0x4245, 0x1f77, 0x4061, 0xba66, 0x4086, 0x337a, 0x1942, 0x2580, 0x0667, 0x40dd, 0x4135, 0xbf56, 0x408c, 0xb2b1, 0xba18, 0xb2ac, 0xbf1b, 0x42d0, 0xb0c8, 0x418c, 0xb283, 0xaf4c, 0x1bab, 0xb047, 0xb092, 0x16b1, 0xbae1, 0xa058, 0xa49b, 0x4007, 0xaca0, 0x1e8e, 0x41dc, 0x4065, 0x3fbb, 0x4192, 0x191a, 0xb263, 0xbfa9, 0x4384, 0x438c, 0x46da, 0x1e3b, 0x1cc2, 0x43a3, 0xba39, 0xbfa5, 0xb0bf, 0x1e42, 0x1a0a, 0x1cad, 0x401c, 0x439f, 0xa4ab, 0x4302, 0x4484, 0x40d3, 0xbfd0, 0x41cd, 0x3bc6, 0x1fc1, 0x41b9, 0xb255, 0x41c4, 0xba13, 0xb298, 0x2459, 0x054c, 0xbfd4, 0x41d4, 0x1b7d, 0x1d2c, 0x2806, 0x1bec, 0xb2f3, 0x268c, 0x4131, 0x42cb, 0x4146, 0xbf8a, 0x42c8, 0x4057, 0xa7d7, 0x36f5, 0xb082, 0x439c, 0x42cf, 0x454e, 0x403b, 0xb225, 0x1e75, 0x4357, 0x2f15, 0x4096, 0x314f, 0xa555, 0xba10, 0x4003, 0x3197, 0xba21, 0x40a5, 0xbf9f, 0x41cd, 0x24ad, 0xaf2c, 0x42d0, 0x38f1, 0x407c, 0x429e, 0xb03e, 0xbf1f, 0x40c7, 0xb294, 0xa6a8, 0x387d, 0x1236, 0x4148, 0x3c48, 0xbf76, 0x00f5, 0xb2bf, 0x12a8, 0x423b, 0x41c2, 0x4206, 0xb0a4, 0x0c83, 0x40ff, 0x4138, 0xbfb0, 0x1b14, 0x4228, 0x3697, 0x3a11, 0x4107, 0xbafd, 0x447c, 0x43a3, 0xb28f, 0x4246, 0x454f, 0x432e, 0xbfdc, 0xb2d1, 0x437e, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xcc2ef87d, 0xae95a3fd, 0x5e6e0ca5, 0x1ef09cca, 0x00ea6c00, 0x963e3b79, 0xc2107f27, 0x7b8c8e2a, 0xa52501a4, 0xdd47e07c, 0x58514dc3, 0xb401ec82, 0x88632001, 0xabb2221b, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x0000003e, 0x0000183e, 0x00000000, 0x544dfc0a, 0x00000000, 0x00000000, 0x0000ffff, 0x0000133c, 0xffffffff, 0x0000148f, 0xa00f6410, 0x88633849, 0xabb2349a, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1a9a, 0xaba5, 0x42a3, 0x4339, 0x43b9, 0xb263, 0xb077, 0xbf80, 0x409b, 0x4178, 0xba22, 0x439a, 0x428e, 0xb0b6, 0x404e, 0x1ae0, 0xba1e, 0x0e81, 0x41c2, 0xbfb8, 0x40e2, 0x18d2, 0x462c, 0x43c9, 0x1e7d, 0xb21b, 0x43ce, 0x45b5, 0x42d0, 0xb038, 0x42c9, 0x4238, 0x1e37, 0x42b5, 0x41cc, 0x42f9, 0xb072, 0x4154, 0x030b, 0x4384, 0x42d6, 0x46a8, 0xbf6c, 0x0ce2, 0xb24b, 0x4305, 0x4418, 0xb006, 0x404c, 0x438f, 0x438f, 0x0532, 0x41e4, 0x4290, 0x4019, 0xba4e, 0xbf54, 0x412d, 0x4202, 0x4208, 0x4614, 0x193b, 0xbf57, 0x3ef2, 0x4692, 0xbaf4, 0xad07, 0xba72, 0x4549, 0xb22f, 0x40a5, 0x4184, 0x1cf9, 0x416c, 0x421a, 0x4371, 0xba66, 0xba38, 0x1c55, 0xaa65, 0xaa8c, 0x4070, 0x0cf2, 0x1ff4, 0x1a0b, 0xbfad, 0xb219, 0x400b, 0x2178, 0x1d28, 0x43f7, 0x419f, 0x421b, 0x1f66, 0x40a0, 0x4224, 0x4370, 0x4325, 0x37c4, 0xa62d, 0xb080, 0x43e0, 0xb2a1, 0x00db, 0xab3b, 0xbf15, 0x40ae, 0x109e, 0x3d73, 0x1fc1, 0x19fb, 0x3e2e, 0x42d5, 0x1946, 0x419c, 0x4135, 0x02c7, 0x46fb, 0xb22c, 0x40b6, 0x43a1, 0x4229, 0x4382, 0x42c3, 0x422d, 0x3ecb, 0x4289, 0x407d, 0xb090, 0xbf12, 0x0c06, 0xba06, 0x4324, 0x439b, 0x440d, 0x4177, 0x41a8, 0x41d5, 0x4282, 0xbaff, 0xbfc0, 0x3148, 0x4294, 0xbfb0, 0x1b88, 0x1f7c, 0xb2ba, 0xbf9f, 0x4085, 0x421a, 0xb2d2, 0x1e81, 0x40e0, 0x41c4, 0xbade, 0x19d2, 0x4289, 0xb237, 0xba1c, 0xba3c, 0x41e1, 0xba5e, 0xb2b8, 0x41f3, 0x423a, 0xb272, 0x40d7, 0x41d2, 0x4046, 0xbf88, 0x40d6, 0x4045, 0x43ff, 0x18df, 0xa909, 0x40cf, 0x42bc, 0x4198, 0x0130, 0x071f, 0x427e, 0x45a3, 0x1b5c, 0xbf0a, 0xb222, 0x3b72, 0x4039, 0x43af, 0x1dfe, 0x4338, 0xbfc2, 0x43f5, 0x41f6, 0x4012, 0x416e, 0x1d8b, 0x1ff4, 0xb2e2, 0x29ae, 0xa123, 0xb2ef, 0xbf26, 0x1f7f, 0x347b, 0x3277, 0x41e4, 0x43b0, 0x41b1, 0x4193, 0xbf05, 0xba7f, 0xb26d, 0x2ee4, 0x1eb0, 0x43dd, 0x413e, 0x4072, 0x1b4b, 0x4047, 0xb2ae, 0xbaf6, 0x41d7, 0xa6ec, 0xbfdb, 0xb250, 0x4057, 0x2618, 0xb282, 0x4321, 0x4288, 0x400d, 0x4138, 0xa678, 0xa9a6, 0xba47, 0xb2e9, 0x1995, 0x43d3, 0x4237, 0xb03d, 0x4144, 0x46a2, 0xb206, 0x1d19, 0x1b25, 0x4389, 0x4104, 0xb066, 0x1d21, 0x1f7e, 0x2adb, 0xbfd2, 0xa0ea, 0xa87a, 0x4382, 0xb018, 0x44ba, 0x425f, 0x3c44, 0x40eb, 0x43b6, 0xb0a6, 0xba7c, 0xba0c, 0xaeec, 0x4592, 0xb036, 0xbf15, 0xba48, 0xb076, 0x4042, 0xac7c, 0x4560, 0x4003, 0xb03b, 0x1bc1, 0x0ad5, 0x4293, 0xb262, 0x1c32, 0x199d, 0x435d, 0xbf80, 0x2273, 0xb2b2, 0x427b, 0x23cc, 0xb251, 0xb09b, 0x42c1, 0x37da, 0x437a, 0xb022, 0x4384, 0x4280, 0x2b09, 0x4396, 0xbfb8, 0xb09d, 0xaebd, 0xbf34, 0x4346, 0x169e, 0x425d, 0xb036, 0x1e3d, 0x1384, 0xb00e, 0x1cce, 0x135c, 0x4266, 0x025b, 0x46cd, 0x3cf3, 0xbf9d, 0x43d8, 0x41be, 0x2ba8, 0x1ba7, 0x43b3, 0x1ae9, 0xbad7, 0x4019, 0x1deb, 0x292d, 0x413c, 0x4096, 0xb203, 0x4033, 0x1e9e, 0x40c4, 0x2339, 0xb228, 0xb090, 0xbf2a, 0x4172, 0xb22f, 0x4418, 0x162b, 0x4061, 0xa58e, 0xba40, 0x42e7, 0x41e1, 0xb27b, 0x1c28, 0x45ae, 0x41a1, 0x4383, 0x4149, 0x4382, 0x1e4c, 0xbf14, 0x40bd, 0x4209, 0x401c, 0x0482, 0x42f0, 0x4681, 0x43d3, 0x4226, 0xadee, 0x1e44, 0xb0ad, 0x417e, 0x1efc, 0xbf49, 0xb22f, 0x2bad, 0x42a4, 0x42b2, 0xba1f, 0x4562, 0x40d3, 0x4619, 0xbae8, 0xba17, 0x4154, 0xbf6a, 0x407f, 0x40e1, 0x4359, 0x42f3, 0x137f, 0xb2b2, 0x419e, 0xbf0a, 0x4277, 0x41ca, 0x4025, 0x41f8, 0x11b6, 0xb015, 0x42bc, 0xb098, 0xbf26, 0x4382, 0xba65, 0x2399, 0x44e3, 0x1b58, 0x4395, 0x30cc, 0x41c1, 0x41fa, 0x4632, 0xa687, 0x192f, 0x3f96, 0x45d8, 0x41b2, 0x4075, 0x4080, 0xbf9f, 0x43b9, 0x1902, 0x1a58, 0xb003, 0xa201, 0xb28f, 0x405c, 0x41d8, 0xb0cb, 0x42ba, 0x43c2, 0x40fa, 0x4331, 0xb294, 0xb205, 0xb0a7, 0xbfc2, 0x4268, 0x403f, 0x40bf, 0x41e1, 0x407b, 0x4013, 0x2de8, 0xbf75, 0xb039, 0x1f48, 0x428a, 0x4226, 0xba11, 0x0c28, 0x338a, 0x1d9a, 0xb075, 0x3f72, 0x3f5e, 0x4165, 0x4585, 0x3a3d, 0x427e, 0xb0b8, 0xa4b0, 0x436a, 0x4385, 0xbf02, 0x45ba, 0xb07c, 0x42b1, 0xb26b, 0xb22a, 0x4297, 0x1bda, 0x4634, 0x0a08, 0x43e7, 0x169b, 0xa98d, 0x43e2, 0x42f6, 0x42c1, 0x41c2, 0x1cfd, 0xbf48, 0x40b7, 0x40de, 0x1f98, 0xb246, 0xb03d, 0x1e78, 0x3ac4, 0x407d, 0x1862, 0xb238, 0x2451, 0xba4a, 0x445e, 0x1a72, 0x4252, 0x1d66, 0xbf41, 0x42df, 0x31d5, 0xb095, 0x41d9, 0x434b, 0x4303, 0x2470, 0x1dff, 0x469a, 0x45b8, 0x418e, 0x456e, 0x4417, 0xba49, 0xb2b7, 0x4228, 0xa06c, 0x41e7, 0xa99b, 0x36cf, 0x46c4, 0x1852, 0x142a, 0xbfce, 0x06af, 0x4135, 0xb260, 0x3bfc, 0x410a, 0xbf90, 0xb0ca, 0x4165, 0xbf2d, 0x4001, 0x42d4, 0x2d2d, 0x46e0, 0x190d, 0xbfba, 0x18a6, 0xb240, 0x4309, 0x1378, 0x466f, 0x41e6, 0xb079, 0x188f, 0x4012, 0xba77, 0x082f, 0x4018, 0xba48, 0x1786, 0xbac1, 0x1734, 0xbf73, 0x1a6e, 0x4295, 0x1bc2, 0x1956, 0x4196, 0x0623, 0x3a43, 0x4197, 0xbf91, 0xb283, 0x4234, 0x0324, 0x4663, 0x4050, 0x4364, 0xb237, 0xb22e, 0x362c, 0xb21f, 0x2ab6, 0x3262, 0x1b32, 0xbf49, 0xaf0d, 0xb2d7, 0xba6d, 0x43e4, 0xb2a6, 0xb20d, 0x390c, 0xbfcf, 0x1a4a, 0xb2df, 0x447c, 0x4555, 0x437d, 0x412f, 0xbf90, 0xb0fe, 0x43bc, 0xb210, 0xb22b, 0xba63, 0xba2e, 0x4293, 0xb015, 0x0ba7, 0xbf92, 0x1956, 0x413e, 0x40db, 0xa728, 0x4360, 0x04c0, 0xb060, 0x41fa, 0x460d, 0x4168, 0xb24f, 0xaff6, 0x4284, 0x4018, 0x002e, 0x43ea, 0xbfb4, 0x3528, 0x40d5, 0xba2d, 0x4296, 0x1e62, 0x44d9, 0x189c, 0xb0c5, 0x4485, 0xb2b2, 0x42e6, 0x432f, 0x26d4, 0xa09d, 0x0530, 0xb0df, 0x42af, 0x4367, 0x1aff, 0x40c6, 0x2ceb, 0xba26, 0x4337, 0x4013, 0x42a1, 0xbf47, 0x1a0a, 0x435d, 0x4602, 0x26ee, 0x4313, 0x4128, 0x4135, 0x1ada, 0x1f63, 0x40be, 0x1a3d, 0xac4c, 0x40dd, 0x2cc1, 0x206b, 0x4224, 0xbf4e, 0xb2b9, 0x2ace, 0x1fba, 0xbad6, 0x42b6, 0xbf3e, 0xba6f, 0x41a1, 0xb0c5, 0x20d2, 0xb243, 0x1113, 0xba49, 0x04a8, 0xb253, 0xb0e5, 0xb27b, 0xba5e, 0x1f73, 0x4165, 0xb2e0, 0xb239, 0x4044, 0x431f, 0xb2ed, 0x047d, 0x2407, 0x464b, 0x1298, 0xbfc3, 0x4216, 0x1e84, 0x40c9, 0x4383, 0xb0de, 0x410b, 0xb03b, 0x41a5, 0x409b, 0x1186, 0xbfd2, 0x442c, 0xb0ed, 0x1b84, 0x06a9, 0x43d7, 0xb2ee, 0x4370, 0x420f, 0x41a9, 0x4281, 0xbf00, 0xbac5, 0xad27, 0x1f67, 0x4304, 0xbfe0, 0x4079, 0xba16, 0x425e, 0xbf2a, 0xb00d, 0xa749, 0x3c0b, 0xb00e, 0x19d6, 0xb264, 0x433f, 0x2049, 0x406c, 0xbf61, 0x420e, 0x4223, 0x41b5, 0x418e, 0x4032, 0x4012, 0x4379, 0x4323, 0x3185, 0x43ba, 0x4190, 0xb093, 0xafab, 0x4592, 0xb0b0, 0xbfae, 0x173a, 0x412d, 0x4273, 0x2c58, 0x183c, 0x4389, 0x420c, 0xb211, 0x4304, 0x43e2, 0x41eb, 0x335a, 0x431e, 0x1a7d, 0x4119, 0x12ff, 0xbf3a, 0x42c1, 0x1ea5, 0xb060, 0x4222, 0xb0d4, 0xa12d, 0x4217, 0x0a9a, 0xaa17, 0x4206, 0x445f, 0x4348, 0x20b4, 0xb209, 0xbf11, 0x4072, 0x4023, 0xbadf, 0x41f3, 0x43bf, 0xa351, 0xb266, 0x43a0, 0x1e38, 0x43a9, 0xb24b, 0x2ade, 0x4167, 0x3e49, 0x41f3, 0x4103, 0x0576, 0x433e, 0x42d4, 0xbf91, 0xb2f7, 0x45c9, 0x41b5, 0x4040, 0x3aff, 0xbac7, 0xb213, 0x4192, 0xbf00, 0xb205, 0x43e9, 0x4264, 0x1c65, 0x41e6, 0x45f4, 0xbf7b, 0x4290, 0x42dd, 0x1e56, 0xbadf, 0xb03a, 0xadad, 0x42f2, 0x404b, 0x4635, 0x1eaf, 0x46a5, 0xb25c, 0x2d7c, 0x4053, 0x4005, 0xb068, 0x42b6, 0x1a03, 0x4346, 0xa2f9, 0x402f, 0x40ed, 0xbf51, 0x4694, 0x23a6, 0x1a50, 0x29c4, 0x404b, 0x1283, 0x425c, 0x1de7, 0x40ad, 0xb205, 0x402f, 0x4399, 0x1cb5, 0x430e, 0xb2ba, 0x19d9, 0xbf1f, 0x43f4, 0x41df, 0x1cc0, 0x407d, 0x430f, 0xbf06, 0x43e0, 0xad43, 0x065c, 0x42ed, 0x260a, 0xba07, 0x4415, 0xb0c3, 0x419b, 0x180a, 0xb049, 0x38a3, 0xbf93, 0xb0f3, 0x1c0b, 0xb241, 0x417f, 0xbf9f, 0x428f, 0x39fb, 0x45e5, 0x407d, 0x4664, 0x4202, 0xbaf6, 0x43c1, 0x42a5, 0x409e, 0x4049, 0x0925, 0xadf3, 0x0581, 0x4043, 0x1a2c, 0x42ca, 0x1b73, 0xbfaa, 0xb248, 0x42bb, 0xb253, 0xb254, 0x4432, 0x43ac, 0xbaf8, 0xba1e, 0x27ea, 0xbf55, 0x17bc, 0x4193, 0x0a07, 0x198a, 0x41f9, 0xbf95, 0x4370, 0x2713, 0x3500, 0x4093, 0x41df, 0xba7c, 0x19ec, 0xb21f, 0x357a, 0x0226, 0x3727, 0x43db, 0xb0a4, 0x4017, 0xa52a, 0x4381, 0x40ae, 0xb258, 0x2d00, 0x2206, 0x4286, 0xa01c, 0x4452, 0x43b3, 0xaf31, 0x4208, 0x4342, 0x1cc8, 0x41e0, 0xbf88, 0x02a7, 0xb002, 0x394d, 0xb0f4, 0x43c2, 0x42f8, 0x413a, 0x411c, 0xbace, 0x3d27, 0x42d9, 0x1410, 0x3a6d, 0x4106, 0x027a, 0x4346, 0x408a, 0x43bb, 0x43c6, 0x41b2, 0xbf08, 0x1c77, 0x421b, 0x1322, 0x4425, 0xbace, 0x2ecb, 0x43ad, 0x4415, 0x414a, 0x420e, 0xba66, 0x41a3, 0x40eb, 0x3809, 0x2732, 0xb0b1, 0xb2dc, 0x3668, 0xbf89, 0x413a, 0xb03e, 0x3a5e, 0x41cd, 0xbf37, 0x4372, 0xb2db, 0xacdb, 0x086c, 0x42be, 0x4275, 0xbafd, 0x422b, 0x1bcc, 0x4272, 0x41c1, 0x0e46, 0x433d, 0xbfc1, 0x0ce0, 0xaaad, 0x1982, 0x42cb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x46ce3787, 0x91b9dd82, 0x5c30a10f, 0x98b32b04, 0x61635373, 0xdfb71ffc, 0x105be440, 0x6bd3228d, 0x9979b985, 0xe4135b49, 0x78e8170d, 0x00193fd1, 0xf7f4c7a5, 0x0e660f37, 0x00000000, 0xd00001f0 },
                FinalRegs = new uint[] { 0x00001fff, 0xfffecfff, 0x0000207e, 0xe413586f, 0xffffff81, 0x00003232, 0x0000007f, 0x00000032, 0x6bd3228c, 0xf7f4ed77, 0x00000000, 0xf7f4d893, 0x00001a68, 0x1beca537, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x40de, 0xbf27, 0x1dbc, 0x42ee, 0x414f, 0xab56, 0x4170, 0x4225, 0x1b8e, 0xa89e, 0x00b6, 0x3e79, 0x46d8, 0xb2dc, 0x410d, 0x4058, 0x4395, 0x08c6, 0x43ce, 0xb000, 0xb07f, 0xa5ad, 0xbf14, 0x1da0, 0xb283, 0xbf2e, 0xb0a5, 0x425c, 0xb0ff, 0x4097, 0x3cc7, 0xb2ea, 0x406a, 0x401a, 0x4189, 0x1d12, 0x4265, 0x4167, 0x409d, 0x19c6, 0x40ef, 0xba26, 0x43e0, 0xbf94, 0xbf80, 0x1afd, 0xb067, 0xbfa0, 0xb227, 0x18c1, 0x4584, 0x249c, 0x4037, 0x1d6e, 0x462f, 0x42c7, 0xbf0d, 0xaf08, 0x1c64, 0xba1d, 0x40b0, 0x41a4, 0xa66a, 0x41b1, 0x4164, 0x1108, 0xbfb0, 0x096a, 0xb25f, 0x41aa, 0x1af7, 0x28de, 0x44a1, 0xb261, 0x4084, 0x43bd, 0xbfd3, 0x423d, 0x2de7, 0x1d14, 0x4079, 0xa83f, 0x41fb, 0xb216, 0x0c50, 0x1b4f, 0x42ec, 0xb212, 0x4085, 0x4211, 0x42a1, 0x1fdd, 0x4288, 0xba4e, 0x1029, 0x1d39, 0x3e7c, 0x4175, 0x4065, 0xb045, 0x40a2, 0xa278, 0xba5b, 0xbfd0, 0xad9f, 0xafe7, 0xbfdd, 0x1c5c, 0xb055, 0x3b51, 0x420c, 0xa911, 0x45b2, 0x43f3, 0xbf14, 0x403f, 0x3b14, 0x406f, 0x43c3, 0x41f5, 0x0da8, 0x4083, 0x09b3, 0xb20d, 0xaa12, 0xb022, 0x4693, 0xbf87, 0xba59, 0x165c, 0xb299, 0x19e1, 0x410a, 0x072b, 0xb236, 0x427d, 0x4310, 0xb2b1, 0x43c9, 0xaf02, 0x1ff3, 0xbf27, 0xadf3, 0xaec5, 0x42ee, 0x426e, 0xbfb6, 0x4153, 0x143e, 0x4354, 0x3bde, 0x2571, 0xaaa9, 0xbfb2, 0x401a, 0x4014, 0x186d, 0x1b1d, 0x1d49, 0xb0ed, 0xbfe2, 0x0f4b, 0xb28e, 0x4353, 0x2d51, 0xb219, 0x2806, 0x4614, 0x4046, 0x0727, 0x439d, 0xb2d1, 0xa936, 0xb21c, 0x2643, 0x4338, 0x4115, 0x2663, 0x1fab, 0xb234, 0x4484, 0x46ad, 0x1de8, 0xb291, 0x4379, 0x3751, 0x4672, 0xb23e, 0xbfa3, 0x411f, 0xb217, 0xb0af, 0xba5f, 0xb287, 0xba1a, 0x40b7, 0x428f, 0x4179, 0xbaf8, 0x42ba, 0xb01d, 0xb216, 0x432e, 0x411c, 0x405c, 0x10dd, 0x4358, 0x4277, 0x1b1d, 0x1afd, 0x442f, 0x408e, 0xbf1a, 0xb2d1, 0x20f4, 0x42e0, 0x1865, 0x4360, 0x419b, 0xad85, 0xbf47, 0x42dc, 0x1c28, 0x099a, 0xb021, 0x42c6, 0x3001, 0x4204, 0xb04c, 0x43cc, 0x4011, 0x4383, 0x42c0, 0x42b8, 0x426c, 0x431e, 0x4073, 0xb0de, 0xbf7d, 0xbadf, 0x1b97, 0x44dd, 0xba7a, 0x43a2, 0x2527, 0x1e82, 0x17ea, 0x1ae3, 0x1f1f, 0x40ad, 0xaf9c, 0x44f8, 0xa34e, 0x19ef, 0x3993, 0xbf9a, 0xb246, 0xba3a, 0x2ae0, 0x40b1, 0x230a, 0x41c9, 0x41d8, 0x4612, 0x4217, 0xb2af, 0x0e46, 0x469b, 0x06c2, 0xbfbd, 0xb2d8, 0xb20c, 0xb278, 0xa4ee, 0x4220, 0x2bee, 0xbf03, 0x0713, 0x46a4, 0xb296, 0x45c9, 0x381e, 0x26f9, 0x2596, 0xb20d, 0x1855, 0x31ca, 0xabe7, 0x1e1f, 0xbf95, 0x42ec, 0x4335, 0xb261, 0xba75, 0xba1e, 0xb2e6, 0xb277, 0x41ff, 0x408f, 0x1968, 0x401a, 0xb03f, 0x02b7, 0xb000, 0x3dd6, 0x4165, 0x42c7, 0x4159, 0xbfa8, 0x433e, 0x0a1e, 0x42bf, 0x145b, 0xbaf9, 0x4327, 0x4355, 0xafcf, 0x2661, 0x3e86, 0x415c, 0xa1d9, 0xbae1, 0x1dba, 0xa6de, 0x3bcb, 0xbfd0, 0x1fe8, 0x0997, 0x4268, 0x42d9, 0x432c, 0x3238, 0x1bbb, 0x4076, 0xbfc7, 0x0e40, 0x078c, 0x42ad, 0x2739, 0x3255, 0xbf88, 0x4096, 0x1d9a, 0xb228, 0xbfc1, 0x4025, 0x4021, 0xb21a, 0x41ac, 0x410a, 0x40bb, 0x1b99, 0x22c4, 0xaf4c, 0x21f4, 0x40fc, 0xb0d0, 0xbf18, 0x1ce1, 0xba1c, 0x1d77, 0xb08c, 0xb2ca, 0x160e, 0x1660, 0x0400, 0x41c4, 0x4038, 0x40d4, 0x19d5, 0x40de, 0x4216, 0xb0f3, 0x05c6, 0xba08, 0x2094, 0x18df, 0x43a2, 0x0d66, 0x42b6, 0xbf5d, 0x4088, 0x4150, 0x4207, 0x1fc0, 0x434d, 0xba10, 0x404c, 0x43da, 0xbfd4, 0xa24f, 0xba3a, 0xba01, 0x0721, 0xba0d, 0x41eb, 0xb2d4, 0xb2f9, 0x40d5, 0x423a, 0x430f, 0x41d1, 0x19f0, 0x179f, 0x42a6, 0x4325, 0x37cd, 0x1943, 0x43a7, 0xbf38, 0x3469, 0xbf55, 0xb024, 0x415e, 0xb24e, 0xb042, 0xb29f, 0xb257, 0x43d5, 0x421c, 0x017e, 0x33bd, 0x3636, 0x4102, 0x40b7, 0x4562, 0xb092, 0x42dd, 0xbac5, 0x4352, 0xbf64, 0x4608, 0xa1f5, 0x1810, 0xb080, 0xa0f4, 0x4373, 0x40bb, 0x43f3, 0x42bb, 0x4372, 0x0653, 0x41c3, 0x4205, 0xbafa, 0x0977, 0xbaff, 0xb2c9, 0x43e7, 0xba3e, 0x443d, 0x4390, 0xbf80, 0x409d, 0x42ae, 0x1d3a, 0xb0dd, 0xbfa7, 0xba14, 0x43c7, 0x4064, 0xab20, 0x41a3, 0xb22f, 0x41f3, 0x4338, 0x4088, 0xa77f, 0x3c19, 0x4212, 0x41a2, 0x407b, 0x0a0e, 0x41df, 0x1fba, 0xba60, 0x1432, 0x1fdf, 0x4082, 0x1b87, 0x2588, 0x430b, 0xaf0a, 0xbf44, 0xba6c, 0xbace, 0x16c6, 0x1fb1, 0xa6d9, 0x1b20, 0x4246, 0x2523, 0x00c0, 0x43b2, 0x4612, 0xb0a6, 0x4180, 0x46bb, 0x42da, 0x430f, 0xb2ac, 0xbafe, 0x4655, 0xb2d3, 0xba32, 0xbfb8, 0x4259, 0x1f8c, 0x40d1, 0x415f, 0x404d, 0x4418, 0x40db, 0x43d8, 0x08aa, 0x1e2b, 0x1ef5, 0x41fe, 0x4346, 0x4269, 0xadc3, 0xb2f5, 0x42ab, 0xb0c9, 0xb034, 0x28d6, 0x4276, 0xba16, 0x4294, 0xad99, 0x418f, 0x06b5, 0xbfba, 0x427c, 0x4542, 0xb2fa, 0xa6ac, 0x418d, 0x4564, 0xb04e, 0xbf80, 0x3973, 0x3f72, 0x1830, 0xbfd2, 0xacbf, 0xbaf1, 0x4546, 0x203c, 0x1a47, 0x28e1, 0xbf2e, 0xb235, 0x1eae, 0x4195, 0x1476, 0x43da, 0x4450, 0x428b, 0xb22d, 0xbfd0, 0x1805, 0xba04, 0x1a83, 0x297b, 0x403a, 0x1fca, 0xbaf8, 0x40b6, 0x43c5, 0x3560, 0x43f8, 0x416d, 0xba13, 0x4329, 0xa727, 0x159f, 0x4212, 0xbad7, 0x4085, 0xbfb5, 0x33d8, 0x4388, 0xb280, 0xba44, 0x41cd, 0x1f18, 0x40ae, 0x437a, 0xb081, 0x4340, 0x4163, 0x1a20, 0xbad1, 0xbf03, 0x4390, 0x41d0, 0x40c1, 0x1ff4, 0xb299, 0xbfd0, 0x1bc9, 0xbad6, 0x16cb, 0x4062, 0x1c0c, 0x41df, 0x401b, 0x40e3, 0x4103, 0xa549, 0xbfb0, 0x3b0b, 0x4405, 0x40ca, 0xb290, 0x4100, 0x433c, 0x424b, 0x2d35, 0x46eb, 0xbf70, 0x40ae, 0xbf9d, 0x425c, 0x41c5, 0x4168, 0x434a, 0xbfd6, 0x25f5, 0x43b9, 0x3278, 0x432d, 0x3711, 0x181a, 0x1b7e, 0xba70, 0x437d, 0xb2ed, 0x259f, 0x19f0, 0x1c30, 0x1b3b, 0x43df, 0x443a, 0xb29a, 0x4091, 0xad24, 0xbfaa, 0x419b, 0x25e8, 0x20d2, 0xb2f9, 0xba50, 0x42ef, 0x42b5, 0x368e, 0x3a76, 0xb240, 0x1e37, 0xbafb, 0x42ac, 0x1a73, 0x2848, 0xb0a9, 0x4656, 0x4138, 0xbad8, 0x4115, 0x42ba, 0x425b, 0x414f, 0x430f, 0xb2f1, 0x1ec6, 0xbf37, 0x4171, 0x41a4, 0x41bd, 0x1911, 0xb268, 0x44e9, 0xac0d, 0x3861, 0xb2cf, 0x425a, 0x0c98, 0x1799, 0x3eca, 0x1076, 0xba36, 0x35c9, 0x1a07, 0x4131, 0x4312, 0xbf14, 0x41df, 0x40fc, 0x1d3f, 0x424b, 0x4332, 0x4186, 0xbfc2, 0x42cb, 0x4202, 0xb202, 0xb2d1, 0x1331, 0x2071, 0xb2a9, 0x41ea, 0x4129, 0xab06, 0x4396, 0x41aa, 0x1d8b, 0xba52, 0xb23f, 0x07dd, 0xa183, 0x19f1, 0x05cf, 0x1809, 0x434c, 0xbf84, 0x4008, 0x4286, 0x4289, 0x0abd, 0x20ef, 0xb2ed, 0x1a86, 0x43f6, 0xbadc, 0x4257, 0x3cd6, 0xb207, 0x43ab, 0xbf90, 0xb2d2, 0x4208, 0x405f, 0x1bad, 0x4315, 0x0b32, 0xbf74, 0xbfd0, 0x3ed1, 0x4074, 0x423c, 0x40fd, 0x441c, 0xa2af, 0x2616, 0x2ca8, 0x11e7, 0x438b, 0x43d2, 0x046c, 0xbfbd, 0x18f0, 0x1c25, 0x42fc, 0x43c8, 0x03e7, 0x409d, 0x4277, 0x46f3, 0x4162, 0xbf14, 0xa46b, 0xaf5f, 0x16cb, 0x422b, 0xabe0, 0x0653, 0x0cb8, 0xb2ca, 0xb286, 0x1b6d, 0x4134, 0x215e, 0x437f, 0x4049, 0x41c9, 0x1b35, 0x4214, 0x4048, 0xbf42, 0x43ca, 0xb2ac, 0x43de, 0xa5f1, 0x42b0, 0xb075, 0x413f, 0xba3c, 0x41f2, 0x1d5b, 0x1b0f, 0x3cad, 0x4175, 0x43df, 0xbfd0, 0xb081, 0x0b73, 0x4330, 0xb079, 0x41ec, 0xb232, 0x4621, 0xb06e, 0x400f, 0xa68f, 0x2c56, 0x42f3, 0x1a96, 0xbfab, 0x418c, 0x43b6, 0x4360, 0x4310, 0x1012, 0x4315, 0x42c4, 0x1df4, 0x41ed, 0x2de9, 0x4277, 0x46c8, 0x411e, 0xbf9a, 0xba26, 0xbfd0, 0x4229, 0x1deb, 0x42b0, 0xb245, 0x40bc, 0x426e, 0x42af, 0xb2a0, 0x427c, 0x4262, 0x43ae, 0x4648, 0x4346, 0x40a4, 0x4135, 0xba30, 0x1a4b, 0x437f, 0x402b, 0x4064, 0x2f49, 0x4699, 0xa05f, 0xbaeb, 0x08a5, 0xbf32, 0xba5a, 0x19b5, 0x1807, 0x43d6, 0x4243, 0x4548, 0x015d, 0x09fa, 0x4161, 0x40c3, 0x1528, 0x22e7, 0x43e1, 0x40c1, 0x42f5, 0x4219, 0x1719, 0x4391, 0xba42, 0x1aa8, 0x42de, 0xba05, 0x40e1, 0x42b8, 0x46e2, 0xb07e, 0xbf16, 0x435c, 0x4255, 0x42a5, 0x414d, 0x0353, 0x423e, 0xb0ac, 0xaf6a, 0xbf6d, 0x423a, 0x4303, 0x42b4, 0x42ff, 0xbf09, 0x4043, 0x4052, 0x42cb, 0x439d, 0x3cae, 0x1752, 0xbff0, 0x4120, 0x1b16, 0x42cd, 0xb0bd, 0x4109, 0xbf2b, 0xba63, 0x415b, 0x1f93, 0xbaff, 0xb25c, 0x4327, 0xb0ea, 0x456f, 0xb280, 0x41da, 0x2a24, 0x184d, 0x420f, 0x400b, 0x42d8, 0xb200, 0xb270, 0xb050, 0xba5b, 0xb20b, 0x1eb7, 0x432e, 0xbfd1, 0x4367, 0x2e30, 0xb225, 0x43d0, 0x4211, 0xb2e7, 0x40b4, 0x41b3, 0xba16, 0x409e, 0x1452, 0x28cd, 0x4342, 0x335d, 0x25d5, 0x1fcb, 0x0ab1, 0x4179, 0x40a2, 0xb213, 0x4113, 0x298a, 0x415c, 0x4209, 0x42dd, 0x41c1, 0xb0f2, 0x423a, 0x4021, 0xbf8c, 0x0529, 0xa789, 0xbae0, 0xb217, 0x4195, 0x42dd, 0x424a, 0x40dc, 0x429b, 0x18b2, 0xb228, 0x4288, 0x171a, 0x438a, 0xb2a5, 0x4453, 0xb04f, 0x0047, 0xb27d, 0xbfbf, 0x3cd5, 0xb294, 0x1550, 0x402f, 0xb027, 0x1e0f, 0x43b7, 0x42bb, 0xbf9c, 0xa295, 0x43c2, 0xb2b3, 0xaec7, 0x4168, 0x43bd, 0xb09b, 0xbae4, 0x40dc, 0x4351, 0x4266, 0x0b06, 0xb24c, 0x18ee, 0x41b7, 0xbf7c, 0x43fc, 0x420c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x1c45d6ba, 0x53e984f2, 0x3a0df554, 0xdcee8efa, 0x8d61a15b, 0x42a11c1a, 0xe46650af, 0x77ca9117, 0x8878f892, 0x1c01e917, 0x9521508e, 0x1cb29f47, 0xca2a8d7c, 0xe5798906, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x0000007d, 0x00000000, 0x00000000, 0x00000000, 0xffffffa8, 0xffffffa8, 0xffffffa8, 0x00000057, 0x017b7293, 0x00000000, 0xca2a8d7b, 0x00000000, 0xca2a8d7b, 0xe5798e79, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1ed2, 0xb283, 0x434a, 0x3937, 0x4179, 0x4253, 0x4017, 0x023f, 0x42bc, 0x408d, 0xb22a, 0x4268, 0xba5b, 0x1a31, 0x40a2, 0x1dd9, 0x2161, 0x4449, 0x4000, 0x402e, 0x41ce, 0xbf83, 0xba1e, 0x44b0, 0xba18, 0x425e, 0xa764, 0xbfc0, 0x406f, 0x40bd, 0xb0b7, 0xa99c, 0x4005, 0xa0c2, 0x41a4, 0x3ff8, 0x402c, 0x036d, 0xbf57, 0x3e68, 0x4153, 0xb0e0, 0x4689, 0x42e0, 0x406f, 0x40c8, 0xb0b1, 0x4593, 0x40f8, 0xb2d1, 0xa877, 0x4379, 0x434e, 0xba1e, 0xbf71, 0xba71, 0xb2b1, 0xb0c1, 0xb232, 0x41fc, 0x463f, 0x4069, 0xb25b, 0x4241, 0xb2c0, 0x405a, 0x4048, 0xba32, 0x4258, 0x41ef, 0xbf78, 0x0754, 0xb277, 0xbf60, 0x4089, 0xba4a, 0x4074, 0xb252, 0xb2cb, 0x4134, 0x434c, 0x43e1, 0x1e9c, 0x1c02, 0x40a7, 0xb24e, 0x1e02, 0xbf17, 0x3511, 0xb2a3, 0xba19, 0xaccd, 0x4091, 0x1fc1, 0xb057, 0x43b8, 0xb2c8, 0x4180, 0x377e, 0x43d3, 0x09e2, 0x4266, 0xb26c, 0xbf1f, 0x4260, 0xb24e, 0xb24a, 0xb2ac, 0xbfd0, 0xba7a, 0x45b2, 0x4038, 0x418f, 0x15f9, 0xba63, 0xba2d, 0x42f3, 0x403e, 0x4038, 0x41c9, 0x4152, 0x2ee7, 0x4259, 0x0226, 0xbf41, 0x21d2, 0x4430, 0x4007, 0x17ed, 0xbfbd, 0x4058, 0x3e16, 0x407a, 0x40e3, 0x41a0, 0x43d8, 0x404a, 0x40f2, 0x1605, 0x42a3, 0x4631, 0x44f0, 0xbfe0, 0xa21c, 0x290b, 0xaade, 0x40aa, 0x43ae, 0x4126, 0x29c9, 0x420d, 0xbfa4, 0xba6b, 0x41cc, 0x2a49, 0xa316, 0x4308, 0xb04c, 0x4261, 0x0a3b, 0x27f8, 0x422b, 0x02f7, 0x401f, 0x4084, 0xa4c4, 0x1d89, 0xaf46, 0x4159, 0xbad0, 0xb287, 0x41f4, 0x0ae8, 0x423b, 0x3f2b, 0xb210, 0x41bf, 0xbfa5, 0x4333, 0x447e, 0x4403, 0xb24d, 0xbf80, 0x4201, 0x40ff, 0xb2da, 0x41df, 0xb2fa, 0x4257, 0x4366, 0x43e5, 0xbfba, 0x431d, 0x4558, 0xb296, 0xba51, 0xbacf, 0x4313, 0xbadd, 0x40e6, 0x25fe, 0x4223, 0xb2e4, 0x400a, 0x362d, 0x40bd, 0x1e5c, 0x4010, 0xa3e8, 0x4695, 0x22de, 0x3070, 0x1c9a, 0x2d98, 0xb0b1, 0x1f41, 0xb2e6, 0x415c, 0xbac2, 0xbf69, 0x3816, 0x1f3a, 0x1f46, 0xb06c, 0x10f7, 0x1ee5, 0x400e, 0xbf08, 0x2a02, 0x179e, 0xadb5, 0x42f2, 0x410d, 0x4258, 0xb24d, 0x4424, 0xbaf2, 0x4143, 0x40e8, 0x4339, 0xb231, 0x437a, 0xb2f1, 0x42fd, 0x1dff, 0xbf91, 0x43ed, 0x468c, 0x427d, 0xb211, 0x4388, 0x4101, 0xba6d, 0x3708, 0x19fc, 0xb2fe, 0x410f, 0x1fdc, 0x2d26, 0x0b81, 0xb23e, 0x46b2, 0x4194, 0xacb8, 0x40ec, 0x3012, 0xbfb0, 0x45b1, 0xb2b7, 0x1ea0, 0xb246, 0x4007, 0x23e5, 0xbf92, 0x426f, 0x1eab, 0x4371, 0x0c6a, 0xb21d, 0x4442, 0x1954, 0x1957, 0xba05, 0xb27f, 0xb00c, 0x437f, 0xb022, 0x4223, 0x1d0e, 0xbf97, 0xbaf4, 0x1c45, 0xba0e, 0x469d, 0x4049, 0x2a26, 0xb2f0, 0x4272, 0x441a, 0x465b, 0xb2b7, 0x1885, 0x45ca, 0x43b9, 0x1b66, 0x42fe, 0x467e, 0x3529, 0x36da, 0x43fb, 0x1fea, 0x4249, 0x281b, 0xbf81, 0x3ce3, 0xb25a, 0x4119, 0x41d7, 0x1ba6, 0x426c, 0x41ad, 0x4041, 0x41af, 0x44b2, 0x41c6, 0xbf86, 0x41b6, 0xac9f, 0x2b28, 0x4001, 0xba03, 0xbfb0, 0x40b8, 0x3572, 0xba6e, 0x19d2, 0xbf22, 0xacf7, 0xab27, 0x2775, 0x40d1, 0x438f, 0x4214, 0x17c4, 0x40f8, 0x42ce, 0xa942, 0x3854, 0x44cd, 0x1cf3, 0xbfb8, 0x1a6f, 0x1817, 0x2f34, 0x42f3, 0xaa11, 0xba6c, 0x4560, 0x2208, 0x43ac, 0x45bb, 0x45f5, 0x09eb, 0xbf06, 0xb251, 0x1def, 0xbff0, 0x0c9f, 0x40e4, 0x4453, 0x1aab, 0x4365, 0xb2e1, 0x46a4, 0x4000, 0x12f4, 0x3204, 0x4360, 0x4350, 0x4185, 0x41ba, 0x3fca, 0x31d6, 0xa211, 0x42eb, 0x42a0, 0xadb7, 0xbaec, 0x41d4, 0x409c, 0xbaf5, 0xbfa1, 0x15b0, 0x4234, 0x4345, 0x4215, 0x43fc, 0xba60, 0xb034, 0x25ec, 0x4108, 0xb250, 0x2343, 0x44aa, 0xb2ed, 0x4234, 0x46ba, 0x420a, 0x0fcf, 0xbf2e, 0xb0aa, 0xa5a5, 0xb0e2, 0x41d3, 0xb292, 0x1d9c, 0xbf75, 0x190f, 0x2337, 0xb245, 0x080c, 0xb2cf, 0xa08b, 0x40ce, 0x40c3, 0x4121, 0x192b, 0x327a, 0x4341, 0x0fb8, 0xbfe0, 0xb2a8, 0x26ea, 0x3b79, 0x41eb, 0x18ea, 0xb28a, 0x3412, 0xbff0, 0xbf2d, 0x43a0, 0xaba3, 0x1b78, 0x43f6, 0x406c, 0x43ad, 0x3e65, 0xbac1, 0x45ae, 0x3dce, 0x42da, 0x43c0, 0x42d3, 0x412a, 0xbfc7, 0x0784, 0x41db, 0xb288, 0x209d, 0x117f, 0x404f, 0x2dca, 0x2fa2, 0x0b73, 0x414d, 0x406e, 0xbad2, 0x431b, 0xbafd, 0x466b, 0x429e, 0x4198, 0xb2cb, 0xb2e6, 0xbf9e, 0x404a, 0xb2ba, 0xb096, 0xbac1, 0xba3b, 0x037e, 0x40fa, 0xb26c, 0x416c, 0xb2cb, 0xb014, 0x14dc, 0x412c, 0x2d78, 0x1d2e, 0x199d, 0xb2e8, 0x2adc, 0x426c, 0x1e46, 0x4305, 0x1c4a, 0x43a9, 0x1c02, 0x43e3, 0x466d, 0xbf6a, 0x3376, 0x11a5, 0x37df, 0x416c, 0x1bb3, 0xb27b, 0x4385, 0x0a41, 0x1dd5, 0xb2c3, 0x409e, 0x4112, 0x1cb0, 0x428d, 0x0dfe, 0x4391, 0xbf90, 0xba2b, 0xb2a9, 0x4159, 0x426e, 0x2534, 0x4260, 0xbafc, 0xb294, 0x4332, 0x3fed, 0xbfa9, 0x42a7, 0x422b, 0x1f0f, 0xbaf5, 0x21f3, 0xba2f, 0x15fb, 0xb2fa, 0xb025, 0x4024, 0x4014, 0x3c14, 0xbfac, 0x3734, 0xba6e, 0x435b, 0xba12, 0x2012, 0x4058, 0x423e, 0xabd2, 0x3645, 0x40c3, 0x20d0, 0x1d9d, 0x43c2, 0x41b3, 0x4693, 0x40b4, 0x2296, 0x409b, 0xb291, 0x437a, 0x424b, 0xbaf7, 0x4104, 0x437f, 0xbf96, 0x4102, 0x1886, 0x0fb9, 0x426e, 0x40b8, 0xb20e, 0x431d, 0x4291, 0x4291, 0x0c05, 0xb2d4, 0x199c, 0x428f, 0x44db, 0x413d, 0xbf31, 0xb289, 0x43f0, 0x332d, 0x4569, 0x4299, 0xb014, 0xba7d, 0xb0f3, 0x402e, 0x4045, 0x06a4, 0x43ca, 0x4094, 0x31bd, 0x4081, 0x427b, 0xb20d, 0xb2e6, 0x4632, 0x162d, 0x41c4, 0x2895, 0x40e7, 0x4653, 0xa661, 0xbf66, 0xbf90, 0x42b2, 0xaab5, 0xb090, 0x1583, 0x4639, 0xb28d, 0xb270, 0x40de, 0xbf82, 0xb222, 0x4270, 0x435a, 0x0de4, 0xb092, 0x40a8, 0x429e, 0x3997, 0x414e, 0x4399, 0x21a7, 0xb0e8, 0x4137, 0x46c0, 0x2806, 0x4172, 0x421d, 0x4322, 0x412e, 0x423d, 0x4127, 0x135d, 0x40da, 0x422f, 0x1a75, 0x40f4, 0xbaf0, 0xbf18, 0xba57, 0x0d36, 0xb2bb, 0x0ca6, 0x42fd, 0x425b, 0xbfe2, 0x4373, 0xbfc0, 0xa74d, 0xbfb6, 0xb262, 0xba79, 0x1e98, 0xa253, 0x0db2, 0x4160, 0x29eb, 0x0771, 0x4360, 0x1f61, 0xb2ac, 0x4196, 0x433c, 0x412b, 0x4054, 0x4197, 0xb04b, 0x240f, 0xb2fb, 0x4277, 0x4292, 0xb0a4, 0x1b9f, 0x4007, 0xb01d, 0xbf3d, 0xbfe0, 0x45e0, 0x4297, 0x420b, 0x4659, 0x4051, 0x4012, 0xbf78, 0x468a, 0x4083, 0x2755, 0x42be, 0x4699, 0x40fc, 0xb0e4, 0xaead, 0x458e, 0xa47e, 0x43cd, 0xac1b, 0x4401, 0x4182, 0x45f1, 0xbad0, 0x277d, 0x4580, 0x43f0, 0x1e04, 0x4001, 0x41f1, 0x42a9, 0xbf31, 0x2018, 0x435d, 0xb097, 0x0bdd, 0x449c, 0x4553, 0x1052, 0x41e1, 0x42ef, 0x43a3, 0x04ac, 0xae49, 0x4346, 0x1de8, 0x1ce6, 0x44a3, 0x432c, 0x4646, 0xb240, 0xb29f, 0x1d48, 0x436d, 0xbf36, 0x06c9, 0xbfa0, 0xb231, 0xbf52, 0x1d57, 0x40df, 0x40de, 0xb23c, 0x39bb, 0x2829, 0x4313, 0xa2f6, 0xb003, 0x446e, 0x1c7c, 0x3a4f, 0x007b, 0x427e, 0x083a, 0x1d84, 0xb088, 0x4176, 0x433b, 0x439c, 0xbf05, 0x44c2, 0x42ae, 0x4211, 0xa9ac, 0x1a23, 0xb20a, 0x46e4, 0xa4cc, 0xba2b, 0x4655, 0x2b20, 0x4070, 0xbfc0, 0x40ff, 0x4071, 0x1e59, 0x43a5, 0xbfdf, 0x1c22, 0x4315, 0x462d, 0x4183, 0x4560, 0x43b1, 0x1dc0, 0x4089, 0x17b2, 0x416c, 0x4345, 0x4302, 0xbf1d, 0x42a9, 0xbaf4, 0x40ea, 0x418d, 0xbad4, 0x1f28, 0x4035, 0x4255, 0x2404, 0x43a6, 0x4167, 0xba28, 0x2c13, 0x1a6a, 0x1bb3, 0x4186, 0x430e, 0x4556, 0x1a01, 0xbacc, 0xba0c, 0x410c, 0xb2e3, 0xbf1a, 0x1fc7, 0x4229, 0x462a, 0x4295, 0x1a63, 0xbf00, 0xbf70, 0xb26c, 0x1360, 0xbf95, 0x4151, 0x1132, 0x432b, 0x42d3, 0x0fb4, 0xba25, 0x1505, 0xbfde, 0x4338, 0x41dc, 0x4223, 0x3524, 0x422f, 0x4104, 0x414f, 0x3667, 0x1993, 0x4174, 0x40c2, 0xba13, 0xba2b, 0x289e, 0xba20, 0x406e, 0x20ae, 0xbacb, 0xa364, 0x40d6, 0xbfb8, 0x404f, 0x4398, 0xae33, 0x4023, 0x4333, 0x40c9, 0x43e8, 0x1b35, 0xb0c3, 0xbafb, 0x08c2, 0x4139, 0x42b3, 0xb089, 0xb242, 0x4129, 0xb008, 0xbf47, 0x2a1c, 0xb092, 0x2c58, 0x43c8, 0x41ca, 0x40c9, 0x32e6, 0x4182, 0x1769, 0x363c, 0x1639, 0x4231, 0xb286, 0x4075, 0xbf7f, 0x424c, 0x0094, 0x1c3d, 0x43a3, 0x40b6, 0x41ca, 0x435a, 0x3614, 0x378c, 0xb213, 0xbf85, 0x1839, 0xbae4, 0x1dd2, 0x414c, 0x4040, 0x41d3, 0x4040, 0xb254, 0x4110, 0x1e0a, 0x42e6, 0x095a, 0x0a7b, 0xbf46, 0x1375, 0xba1e, 0x4224, 0x41ee, 0xa734, 0x40dd, 0x42ee, 0x2906, 0x397f, 0xba5b, 0x41eb, 0xa422, 0x3dd1, 0x1dfc, 0x4444, 0x1c7b, 0x4068, 0x4258, 0x1350, 0xb2a6, 0x4155, 0x4213, 0x43ab, 0xbf37, 0xa368, 0x0c7a, 0x4175, 0x1cf7, 0x4260, 0x04fe, 0x18dc, 0xba1c, 0xa450, 0x2a7c, 0x351b, 0x416f, 0xbf55, 0x3b8b, 0x401f, 0x4282, 0xa41b, 0x41e1, 0xa299, 0xa6f3, 0xaca9, 0x1c37, 0x0145, 0x4285, 0xb269, 0x438b, 0x138c, 0x1eb7, 0x1d6e, 0xa51c, 0xb2d2, 0xae4b, 0xb2fc, 0xac6e, 0x4176, 0x1937, 0x42f2, 0x4293, 0xbf48, 0x4280, 0x4685, 0x421f, 0x4394, 0x050c, 0xbfc4, 0x372a, 0x401d, 0xbae7, 0x42f6, 0x40b6, 0xa1e8, 0x44aa, 0x3def, 0x429c, 0x3117, 0x1dba, 0x1a58, 0x415a, 0xa67a, 0x43b3, 0xbf21, 0xbf90, 0xa973, 0xba48, 0xba58, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2222d05c, 0xcfc689d4, 0xc2140172, 0x47506221, 0xe5f77550, 0x38359c85, 0x4e95a963, 0xe20bac00, 0x9f1922b1, 0x5e07b7dc, 0x32b31e2b, 0x9b89a1e5, 0x70e8b6e8, 0xae4936fc, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0xffffe4fa, 0x00001b77, 0x00000077, 0x00000041, 0xf8000000, 0x00001719, 0x000019b8, 0x00000000, 0x9f1922b1, 0x00000057, 0x00001666, 0xfffffe5e, 0x00007157, 0x60e6c544, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x439f, 0x404c, 0x454a, 0x41ed, 0xbf7a, 0x1535, 0x428d, 0x1c71, 0x4125, 0xb281, 0x40a4, 0xb25a, 0x40f7, 0xaab2, 0xb256, 0xb236, 0xb08c, 0x4180, 0x1249, 0x3e60, 0x235c, 0x40f0, 0xba68, 0xbaf7, 0x45e3, 0x4314, 0x411a, 0xb2f1, 0xbf35, 0x4689, 0xb26e, 0xba71, 0x4072, 0x1cd9, 0x4451, 0x4343, 0x4275, 0x1acd, 0x3026, 0x43af, 0x4279, 0xb2fb, 0xb24f, 0x41c1, 0x4101, 0x40d8, 0x4171, 0x436e, 0xb092, 0xbf9b, 0x4190, 0x40ab, 0x1bbc, 0x1ad7, 0x45a3, 0x412f, 0x1374, 0x45c8, 0x1e94, 0x40c4, 0xb0d8, 0x41c8, 0x4044, 0x1d6f, 0x4134, 0x45d8, 0x426b, 0x404c, 0x1986, 0x401c, 0xb2aa, 0x4375, 0x423e, 0x1ac8, 0xbfb3, 0xb296, 0xba27, 0x3e43, 0x4315, 0xbfb0, 0x1914, 0x43f8, 0x42a1, 0x426a, 0x0cfa, 0x4027, 0x4270, 0xadfb, 0x45dc, 0x309c, 0x407d, 0xa010, 0x1d78, 0x34b3, 0x1b29, 0xb2c0, 0x463a, 0x196a, 0x41b1, 0x42c5, 0x40a5, 0x454d, 0xb28f, 0xbf37, 0xba22, 0x1c18, 0x2832, 0x41fe, 0x1a1b, 0x1dd0, 0x4634, 0x4058, 0x2da1, 0x427b, 0xb2c7, 0x4280, 0x1e11, 0x40a9, 0xba2d, 0xba19, 0xbf44, 0xba11, 0x4068, 0x180e, 0x4178, 0x41ee, 0x4651, 0x407e, 0x4353, 0x4245, 0xbae4, 0xa4d0, 0x0dd5, 0xb27b, 0x1be0, 0x4062, 0xb220, 0x4352, 0x40e3, 0x41c9, 0xb21f, 0xbf00, 0xbf03, 0x29da, 0x40c0, 0x4179, 0x1f95, 0x187d, 0x43d8, 0x1ad8, 0xb223, 0x4648, 0x3fb2, 0x2ec5, 0xba50, 0x42c2, 0x3b0b, 0x1fbf, 0xb240, 0x41e0, 0x3a20, 0x032c, 0xb21a, 0xa161, 0x42bb, 0xbf26, 0xbf80, 0x446c, 0x4305, 0xb068, 0xba5f, 0x130a, 0x2736, 0xabf8, 0x1967, 0x2038, 0xb260, 0xba40, 0x4241, 0xaf27, 0x4381, 0x40af, 0x438c, 0x437c, 0x4672, 0xbae3, 0x3e4c, 0xb228, 0x1cc4, 0x432e, 0xbf3a, 0x1c2b, 0x1678, 0xa23f, 0x1dd4, 0x42a9, 0xbff0, 0x397e, 0x42a1, 0xbf54, 0x41a6, 0x4124, 0xb2f5, 0x1d50, 0x42a9, 0xba79, 0xba33, 0xa80d, 0xbf60, 0x4329, 0x4161, 0x3a12, 0x40ed, 0x4562, 0x4303, 0x1853, 0xbf6b, 0x4379, 0x4307, 0x436e, 0x41b0, 0xba44, 0x4178, 0xbf90, 0xb2db, 0xa99c, 0x4671, 0x46a9, 0x44b3, 0x40fc, 0xab9e, 0xb21d, 0x42a7, 0x40f3, 0xb2cd, 0x24b3, 0x41b7, 0x1db8, 0xbf4c, 0x4207, 0xba4b, 0x4227, 0xb064, 0x405a, 0x416a, 0x1222, 0xb0e1, 0x0ff4, 0xba13, 0xbf8a, 0x423f, 0x440b, 0x4007, 0xbadf, 0xb25b, 0xae38, 0x229d, 0xbfcb, 0x3f66, 0x259e, 0xb092, 0x4248, 0x422f, 0x18b3, 0xbfd2, 0x147a, 0x43ce, 0xac74, 0xa9c0, 0x454c, 0xba1f, 0x0865, 0x40bf, 0xb291, 0xb241, 0x4311, 0x1e25, 0xb2e5, 0x3038, 0x4647, 0x4254, 0x4007, 0x4276, 0x42cb, 0x4024, 0x1ee1, 0x206a, 0x0c99, 0x3aea, 0xbfc8, 0xbf60, 0x1eea, 0x410e, 0x4365, 0x03ee, 0x1d17, 0xb2e5, 0x4244, 0x0fc2, 0x1bf2, 0x1b74, 0xbac0, 0x1770, 0xbfcd, 0x02d1, 0x4251, 0x4200, 0xb2c5, 0xbafb, 0x43b8, 0xbf87, 0x425a, 0xbfd0, 0x18bb, 0x4186, 0xb0de, 0xba16, 0x199f, 0x444c, 0x4265, 0xa487, 0x41cf, 0x40e4, 0xb093, 0xbfa0, 0x4164, 0x0082, 0x433b, 0x40a3, 0x1957, 0x0653, 0xba10, 0x1011, 0x43cc, 0xb08c, 0xb24a, 0x2f88, 0xbfe4, 0xb291, 0x4251, 0xb0ec, 0xba67, 0x3fec, 0x401c, 0x2df3, 0xb225, 0x1ed0, 0x22cf, 0x41c4, 0xbfdc, 0xaa7d, 0x1e25, 0x40a5, 0x43a0, 0x41bf, 0xba39, 0x40cc, 0x4117, 0x46b9, 0x407b, 0xb0d5, 0x1cfb, 0x43ed, 0xb2ce, 0x4229, 0x2a4a, 0x1317, 0x41f3, 0x1a08, 0x1aae, 0x1737, 0xbf8a, 0x1ee8, 0x430e, 0x401b, 0x419c, 0x4121, 0xba1d, 0x109a, 0x18d9, 0x1dc3, 0x4331, 0xbac3, 0x00f7, 0x4178, 0x41c8, 0xa2e7, 0xb2be, 0x18bc, 0xb288, 0x1527, 0x43e5, 0x1c45, 0x2f37, 0x423d, 0xba43, 0x4238, 0xbf75, 0x22e5, 0xbfb0, 0x4073, 0x34ad, 0xa802, 0xb084, 0x4154, 0x4333, 0xa7e3, 0x4170, 0x43d9, 0xbfa0, 0x43c6, 0x43d0, 0xb292, 0xa2aa, 0x42e5, 0xbace, 0x42a1, 0x4361, 0x42f9, 0xbf4c, 0x423e, 0x0e80, 0xb233, 0xb2c7, 0xa44f, 0x1a00, 0xbf00, 0x1d47, 0xb265, 0xb203, 0x4330, 0x4268, 0x40c9, 0xb299, 0xba4d, 0xb23a, 0x2d3b, 0xba5e, 0xbf00, 0x43cb, 0xbf28, 0x1595, 0x43e5, 0x0dd2, 0xb287, 0x429c, 0x4683, 0xbfe0, 0x410c, 0x4363, 0x117d, 0x0706, 0x377b, 0xbfa4, 0xb2e7, 0x42e5, 0x45e1, 0x441b, 0x42bf, 0xb08c, 0x427e, 0xb05a, 0x09e5, 0xbf69, 0x438c, 0xbafe, 0x45f6, 0xaad2, 0x0016, 0xbf0b, 0x419d, 0x4024, 0x4146, 0xb0b5, 0x461d, 0x41f8, 0xb044, 0x1cc0, 0xbae5, 0xb086, 0x44cc, 0x420f, 0xb0b0, 0x4679, 0x4009, 0x1a55, 0x404c, 0x41cb, 0x4035, 0x4313, 0x4085, 0x404f, 0x000e, 0x1544, 0xada3, 0xbf09, 0x2288, 0xb215, 0x40ff, 0x426a, 0x40a6, 0xb26e, 0x305e, 0xb206, 0x3b04, 0xa30a, 0xba3a, 0x459e, 0xba24, 0x418b, 0x40ad, 0x440a, 0x435d, 0xb262, 0x42ad, 0xb0d9, 0xb08c, 0xbf5c, 0x4351, 0xb2d0, 0xbf12, 0x1b96, 0xb27a, 0x4154, 0xa009, 0x1c94, 0xa32e, 0x1c08, 0x38a7, 0x43af, 0x418b, 0x43c7, 0x296c, 0x273c, 0x4276, 0x25d3, 0x42c3, 0xba0c, 0xb284, 0xbf19, 0xb2ea, 0xba0e, 0x4069, 0xb003, 0x469c, 0xb0ee, 0xb219, 0x43cb, 0x424b, 0x41f6, 0x3973, 0xb271, 0x401c, 0x23a6, 0x40ea, 0xba16, 0x194a, 0xa210, 0x1c5b, 0xb07a, 0xbfa1, 0x4271, 0x190b, 0xba0c, 0x0783, 0xa0dc, 0x46d0, 0xbf54, 0x42fc, 0x0ea4, 0x1cde, 0x0a0a, 0x4115, 0xb2b7, 0xb2fa, 0xbf2f, 0x436e, 0x4374, 0xba73, 0x43cd, 0x4460, 0xad56, 0xb2cd, 0x4157, 0x40c1, 0x1a23, 0x4011, 0x42de, 0x42ab, 0x41f5, 0x4299, 0x43d1, 0x36ef, 0x4135, 0xade3, 0xbfcc, 0x1d9e, 0x425d, 0x43d0, 0xb26e, 0xb26d, 0xb0fe, 0xbaf7, 0x4106, 0x43d0, 0x4311, 0xbfdc, 0x2ac1, 0xb048, 0x434f, 0xb2b4, 0x12bc, 0x1ff4, 0x1aa2, 0xbad4, 0x428d, 0xb28c, 0x44e3, 0x4323, 0x407e, 0x350d, 0x4600, 0x4304, 0x40b5, 0x4253, 0xb2d9, 0xbfd7, 0xb062, 0xb0bd, 0x4567, 0x465d, 0xba1f, 0x0580, 0x46da, 0xb073, 0x3249, 0xb2fc, 0x41a4, 0x4235, 0x2daa, 0xb097, 0xbf5a, 0x41aa, 0x44c3, 0x42c0, 0x0536, 0xb239, 0x42c5, 0xbf6e, 0x19da, 0x296c, 0x324a, 0x395f, 0x40c6, 0x40fe, 0x4023, 0x00f5, 0x281c, 0x2646, 0x3352, 0xb2b7, 0xb2fb, 0xb27b, 0x44a0, 0x41ee, 0xbf81, 0xa1c3, 0x43e2, 0x4047, 0x43c2, 0xba7c, 0x2eae, 0xb281, 0x1807, 0x23c0, 0x1b92, 0x4116, 0xa4a6, 0x43f3, 0x4333, 0x1ae6, 0x25ec, 0x373d, 0xbfc3, 0xafa6, 0x409b, 0x1ffa, 0x183a, 0x46d5, 0x42e4, 0x418e, 0x422e, 0x43ad, 0xad75, 0x18e8, 0x4087, 0x0fb4, 0x0dc6, 0x1913, 0xabed, 0x4331, 0x40fa, 0xbf95, 0x18ab, 0x1aa0, 0x33e6, 0x4354, 0x4338, 0xb2cf, 0x4683, 0x4289, 0x42a1, 0x1ed3, 0x43c5, 0x434e, 0xb2bc, 0xa1bd, 0x4098, 0x418a, 0x25fe, 0x4348, 0x27b1, 0xa5ce, 0x121a, 0x4229, 0xbfa4, 0xb2d8, 0xb290, 0xbfdd, 0xa03c, 0x4217, 0x41b7, 0x41c0, 0xb0e8, 0x19ee, 0x41ea, 0x4294, 0x418d, 0x4075, 0xb04c, 0xbfce, 0xb083, 0x3289, 0x40a7, 0x429d, 0x4277, 0x454a, 0x43a4, 0x1f9a, 0xb2cf, 0x40a6, 0x42b7, 0xb277, 0x27de, 0x43cd, 0xba20, 0xb255, 0x4301, 0x43de, 0x41f0, 0xba42, 0x40b3, 0x05ae, 0xbafb, 0xbaf1, 0x443b, 0x4022, 0xbf68, 0xb286, 0xb2f3, 0xac85, 0x4129, 0x435d, 0x4344, 0x42c3, 0x41d8, 0xb247, 0xbf65, 0x42b9, 0x0ff6, 0x412c, 0xb2e8, 0x4406, 0x42e1, 0x3032, 0x416c, 0xb298, 0xab81, 0xb090, 0x434d, 0xbf87, 0x1cb0, 0x404c, 0xaf0e, 0x33b1, 0x3a35, 0x1c47, 0x43e5, 0xbacc, 0x323d, 0x336a, 0x4125, 0xb062, 0xba38, 0x429b, 0x4173, 0xbf95, 0x40bd, 0x164b, 0xb232, 0xb292, 0x4283, 0xa0d8, 0xb0a5, 0x197f, 0x43d2, 0x1b4f, 0x092b, 0x40d5, 0x1abc, 0x468d, 0xb0ac, 0xb289, 0xb231, 0x1f75, 0xba2c, 0x3a4e, 0x429d, 0x3a08, 0x414a, 0x400f, 0xbfc5, 0x42db, 0x4041, 0x4204, 0xb225, 0x022c, 0x41a9, 0x1def, 0x1cf5, 0x4217, 0x4047, 0x0cb3, 0x3896, 0x4383, 0x28ec, 0x4036, 0x434c, 0x424d, 0x392c, 0x0f8f, 0x2d25, 0x1ca0, 0x40d1, 0x4273, 0x4365, 0x41ce, 0xbf27, 0xb22d, 0x1762, 0x430b, 0xb23a, 0x419f, 0x40f7, 0x421c, 0x44f9, 0x41fa, 0x43d0, 0x428a, 0xb010, 0xb0b8, 0x402a, 0x40ba, 0xbf65, 0x3a3b, 0x4205, 0x41d3, 0xb2df, 0xb019, 0x422b, 0x4207, 0xba0a, 0x430c, 0x18fd, 0xbaf8, 0x4162, 0x400e, 0x4099, 0x40bd, 0x268f, 0xbaee, 0x42ed, 0x1a55, 0x42a5, 0x19bf, 0xb229, 0xb2de, 0x4169, 0xba79, 0x0f16, 0xb2d5, 0x4193, 0xbf2f, 0x0861, 0x4315, 0x05f0, 0xba76, 0xb2e6, 0xbf75, 0x0d63, 0x4580, 0x41dc, 0x4196, 0xbad8, 0x042b, 0x1a1c, 0x1e88, 0xba71, 0x19b1, 0x08a3, 0x314e, 0xba39, 0x40f9, 0x23e3, 0xb0e9, 0x11cb, 0x41b8, 0xbf1c, 0x1a4f, 0xa122, 0x416d, 0xb045, 0xb05f, 0x456c, 0x2fbd, 0x1eae, 0x43ef, 0x1264, 0x41be, 0x2531, 0x4240, 0xbf8c, 0x40fa, 0xb2ef, 0xaa96, 0x4398, 0x40fe, 0xb2e5, 0x4331, 0x419d, 0xba00, 0x4618, 0x4219, 0x408f, 0x438c, 0xb2a7, 0x41f6, 0xb272, 0xab3a, 0x1bd0, 0x2ece, 0x40ff, 0x1f10, 0xbf34, 0x2253, 0x40a3, 0x426e, 0x4206, 0x4262, 0x4354, 0x2793, 0xae78, 0x427a, 0x4370, 0x1911, 0x387b, 0x46e4, 0x18dc, 0x4259, 0x4013, 0xad55, 0x443c, 0x4231, 0xb015, 0x42d5, 0xa0d6, 0x41b8, 0xbf92, 0x4689, 0x4305, 0x3f97, 0x4356, 0xba74, 0x40a2, 0x43b8, 0x46bc, 0x4002, 0x0f62, 0xbacf, 0xbadb, 0x4562, 0xbf3c, 0xbfd0, 0x1eb6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xb6e12d7d, 0x2bc8f2a0, 0xe074d4cb, 0x76b33514, 0x92975e84, 0xc4f82ea7, 0xeb560f6d, 0x9507ee5e, 0xd48a048b, 0x1aa04244, 0xe9639443, 0xe7dafbbf, 0x050a37ec, 0x91c1b356, 0x00000000, 0xd00001f0 },
                FinalRegs = new uint[] { 0x00000001, 0xffffff18, 0x00000007, 0x00006800, 0xfeff60ec, 0x00001b7d, 0xfffeec5e, 0x000018ff, 0xe9639443, 0x000016c5, 0x00001547, 0xf400171a, 0xfffffffc, 0x00000054, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb29c, 0xba51, 0x43ec, 0x4103, 0x372e, 0x2b03, 0x42b6, 0x41ad, 0xbac4, 0x463c, 0x2969, 0x24f2, 0x2801, 0x4274, 0xb048, 0x438e, 0x3129, 0x41e5, 0xb223, 0x40f8, 0xb004, 0xbac2, 0xba3e, 0x1aa2, 0xbf3b, 0x427c, 0x458c, 0x4062, 0x1d1d, 0x4184, 0x16db, 0x4429, 0x1dae, 0x4326, 0xbacf, 0x404a, 0xb2cc, 0xbf0a, 0xac02, 0x2e4d, 0x29e9, 0x298b, 0x30b9, 0xb0c0, 0xbf43, 0x1d40, 0x0fd8, 0xb21a, 0x2ed1, 0xb23d, 0x1fc5, 0x4639, 0x4308, 0x41f1, 0xba0c, 0x0fb2, 0xbff0, 0xbfd4, 0x40c3, 0xba36, 0x42d3, 0xba79, 0x42f6, 0x44c1, 0x414d, 0x430a, 0x4163, 0xbf0d, 0x3df5, 0x4390, 0xb280, 0xa4a3, 0x1918, 0x4075, 0x40f7, 0x4254, 0x431b, 0x40f2, 0x4272, 0xb252, 0x43c5, 0xb21f, 0xba24, 0xbfaa, 0x4096, 0xbac4, 0x4469, 0xbf28, 0xb098, 0xb24b, 0x42b8, 0x3c87, 0xb0bf, 0xba48, 0xa742, 0x41ed, 0xba57, 0xbff0, 0x42dd, 0x43f2, 0xb2e8, 0x1f6e, 0x2f4d, 0xbfc8, 0xafdb, 0xbaf9, 0x4313, 0x2895, 0xb06b, 0x180c, 0xb247, 0x1a0e, 0x4002, 0xb2be, 0x3206, 0x43e3, 0x40ec, 0x4609, 0xbf00, 0x410e, 0x4328, 0x45c5, 0xbf21, 0x169f, 0x1a33, 0xa416, 0xb2af, 0x42d6, 0xbfa4, 0x439f, 0xacea, 0x40c3, 0x43d7, 0x4184, 0x1ea5, 0xbade, 0x41b2, 0xbf96, 0xbac9, 0x1703, 0x18d6, 0xbf60, 0x1cb9, 0x2c88, 0x3e09, 0xba5f, 0x3f42, 0xb2d1, 0xb2a6, 0x42cf, 0xb2be, 0x2c25, 0x4315, 0xba76, 0xb029, 0x41fe, 0x437c, 0x4186, 0x41ce, 0x1adf, 0xba71, 0xb0d3, 0xbf76, 0x4399, 0x42b3, 0x418d, 0x4258, 0x1fe3, 0x4150, 0x4595, 0x1b75, 0x4255, 0x40d4, 0x42ee, 0x4207, 0xba40, 0x41c3, 0x030d, 0x2a47, 0x1f8f, 0x40a9, 0xb0c1, 0xbfae, 0x4080, 0x4358, 0x4005, 0xbfe2, 0x1724, 0xa1be, 0x183c, 0x408e, 0x3b25, 0xb255, 0x4222, 0xb2e2, 0x1244, 0xb0bd, 0xa6fa, 0x1fe0, 0xb05f, 0xb237, 0x07aa, 0xbaef, 0xa553, 0x3698, 0x41c0, 0x41a0, 0x4354, 0xbfcc, 0x46d0, 0xb0fd, 0x1374, 0x42ec, 0x4077, 0xa062, 0x1e82, 0x40fe, 0xb227, 0x172b, 0xbfa3, 0x0541, 0xb093, 0xba2e, 0x4359, 0x3053, 0x4163, 0x4190, 0x01d8, 0x434b, 0x420a, 0x0cdf, 0x1b4d, 0x4045, 0x09df, 0xb268, 0x1d2c, 0xb008, 0xbfe2, 0x4342, 0x4138, 0x2a50, 0x4197, 0x43d7, 0xbafc, 0x443a, 0xa855, 0xb00a, 0x43b9, 0xbaf2, 0xbf4d, 0x1f75, 0x42d8, 0x431a, 0x407c, 0xbac1, 0x42b1, 0x08e3, 0x4217, 0x2206, 0x436d, 0x429d, 0x43d8, 0xb061, 0x4280, 0x4164, 0x422c, 0xbfd8, 0x405c, 0x1ae9, 0x1cce, 0x4346, 0xb079, 0x2ad0, 0xafb0, 0x10f5, 0x430d, 0x41ad, 0xbfe2, 0x0a34, 0x0894, 0x1b3e, 0x429b, 0x46a8, 0xb2e6, 0x40cb, 0x3e07, 0x40ea, 0x2248, 0x4096, 0xba45, 0x43b5, 0x40c4, 0x40c0, 0x1a05, 0x4044, 0x4303, 0x42ab, 0x15f7, 0x4679, 0x4321, 0x447e, 0x429b, 0xbfc2, 0x1d66, 0x42f6, 0xb079, 0x24c9, 0x189d, 0xbad0, 0x20cb, 0x43d4, 0x1572, 0x4352, 0x2b09, 0xb2b0, 0x0800, 0xba78, 0x1e22, 0x1973, 0xb27e, 0xb2d0, 0xb21c, 0x3d33, 0x2ea5, 0x0470, 0x419e, 0x066e, 0xbadb, 0x43a1, 0x412f, 0xbf52, 0xba60, 0xb2a5, 0xba7f, 0xb019, 0x4117, 0x43cf, 0x1f1f, 0x42ca, 0x40ab, 0xa704, 0x18b4, 0x3160, 0x435a, 0xb2da, 0xbaee, 0x252b, 0xbf94, 0x374f, 0xb0e2, 0x3612, 0xbf90, 0x1c26, 0xa209, 0x4173, 0x4119, 0x4375, 0xb09a, 0x2b4a, 0x41d0, 0x404e, 0x4337, 0xa05b, 0xbad3, 0x3b56, 0x1b76, 0xba1d, 0xb217, 0x41de, 0xbf1a, 0x1e6f, 0xb234, 0xbae1, 0x45c5, 0x2c47, 0x41e8, 0x1fe5, 0x42bb, 0xbacb, 0xb007, 0x43e8, 0x423a, 0x3f88, 0x0af3, 0x1cbb, 0x408f, 0x427d, 0x403e, 0xb253, 0x43b4, 0x436a, 0xb252, 0x2ae0, 0x3ae8, 0x435e, 0xb06f, 0xbf84, 0x34ff, 0x1acd, 0xba06, 0x22dc, 0x4022, 0x4343, 0x45f4, 0x4361, 0x18d9, 0x40fc, 0xb024, 0x4208, 0x1820, 0x4391, 0x41ea, 0x37a3, 0x46c8, 0xbf60, 0xb04f, 0x41dd, 0x0cfa, 0x0e15, 0xa479, 0xbac6, 0xbf54, 0x46c2, 0x288a, 0x182e, 0xba13, 0x1f09, 0x40dd, 0xa3e4, 0x426a, 0xb27d, 0x2af6, 0xbafe, 0x41e7, 0xba35, 0xb07c, 0x050a, 0xb2e1, 0xabb5, 0x1423, 0x3493, 0x40be, 0x2644, 0x1da5, 0x1bae, 0x1e6c, 0x41c0, 0x44c9, 0xbfc3, 0x40c4, 0x21e1, 0xaeb3, 0x1fdc, 0xb254, 0x418a, 0xbf0b, 0xb224, 0xae41, 0x42e5, 0x42f6, 0x2d3f, 0x4137, 0xb093, 0x4105, 0x420d, 0x40ac, 0xb210, 0x42dd, 0x2bf7, 0x43bc, 0x417d, 0xb2f3, 0xac37, 0x4031, 0xb2cc, 0x405b, 0x415f, 0x23b1, 0xb215, 0xba5e, 0xa2d0, 0x4415, 0x2582, 0xbf1c, 0x408a, 0x42a5, 0x4118, 0xb292, 0x413e, 0x4472, 0x406b, 0xb2c3, 0x3639, 0x40e3, 0xb25e, 0xbfd0, 0x40f2, 0x1fd9, 0xbfb0, 0x1e92, 0x1e5f, 0x40df, 0x4665, 0xbf37, 0x4479, 0x1602, 0x1a33, 0x437a, 0x2900, 0x4411, 0xbf00, 0xbf60, 0x4108, 0x41ef, 0xb2d3, 0x42f9, 0x38a6, 0x43f4, 0xb2d5, 0x292a, 0xba42, 0x2eb4, 0xbae0, 0xa21b, 0x1a60, 0xbf8a, 0xb239, 0x1dea, 0x1be6, 0x4042, 0x18b4, 0x4217, 0x0c3d, 0x1826, 0x41d1, 0xba1c, 0x36b3, 0xa241, 0x43f6, 0x4435, 0x425e, 0x3c53, 0x2281, 0x25fe, 0x409d, 0x42c6, 0xba3b, 0x43ba, 0xbf86, 0xba54, 0x2522, 0x43df, 0xbfd7, 0x18f1, 0x4308, 0x1fb2, 0x429d, 0xb010, 0x4445, 0x4169, 0x4054, 0x2abe, 0xa0b6, 0x3317, 0x1f53, 0x42d4, 0x4272, 0x46ab, 0x2008, 0x1d36, 0xbf73, 0x42d6, 0xb238, 0x425f, 0x411c, 0x4098, 0xb044, 0xb23a, 0x420d, 0x1f38, 0xb20e, 0x0203, 0xacf3, 0xb2e9, 0x4055, 0x1667, 0x42a1, 0x4073, 0x3006, 0x4229, 0x41e2, 0x4630, 0xb2e0, 0xa326, 0x4146, 0x1d00, 0xbf35, 0xb2d1, 0x14a1, 0xb2e9, 0x31e8, 0x0f16, 0x41ec, 0x4245, 0x1a7a, 0x421a, 0x2429, 0x4017, 0x0882, 0x3395, 0x00f1, 0xbad4, 0x42b0, 0x43d9, 0x293c, 0xbf98, 0x1938, 0x40a9, 0xb26c, 0x1bf5, 0x1a28, 0x4016, 0x432b, 0xb21e, 0x43e9, 0x4457, 0x417a, 0x41b0, 0x4052, 0xbae8, 0xa883, 0xb266, 0xba3a, 0x431e, 0x1b96, 0x4238, 0x43ac, 0xbfc0, 0x4157, 0xbf04, 0xaa02, 0x42a0, 0xba3b, 0x4114, 0x4145, 0x3e46, 0x0cc6, 0x1885, 0x4102, 0x43ba, 0x430e, 0xb2d1, 0xb286, 0x3084, 0x01e7, 0xa8bb, 0x42d1, 0xba6c, 0x402b, 0x427c, 0x4279, 0xbf61, 0x4177, 0x40f1, 0x424e, 0x01c2, 0x11e7, 0x369e, 0x413e, 0x1a54, 0xb090, 0x1844, 0x43c1, 0x42dc, 0x4038, 0xbadb, 0xbf1c, 0x4345, 0xbaf7, 0x4103, 0xb297, 0xb030, 0xba42, 0xac35, 0x4078, 0xba2a, 0xbfb9, 0x416f, 0xba16, 0xa940, 0x41f8, 0xbf65, 0x43c9, 0x45a4, 0xb2fe, 0x4020, 0xba2b, 0x400e, 0xb282, 0x40ec, 0x1afb, 0x4584, 0xbf70, 0x3323, 0x41c6, 0x1bfb, 0xb288, 0xbfac, 0xba01, 0xa344, 0x3658, 0xba58, 0x408e, 0x1efe, 0xb0bf, 0x119d, 0x0af0, 0x4194, 0x4162, 0x419b, 0x4311, 0xb2f7, 0x4312, 0x411a, 0xbf70, 0x405d, 0x405d, 0x1b4b, 0xbad8, 0x416e, 0x4327, 0xbf63, 0x4311, 0x1fcc, 0x41f8, 0x2cae, 0x41d4, 0x422b, 0xbf6b, 0x4260, 0x43cb, 0x1cd2, 0xa5a3, 0xbfb0, 0xb030, 0xbfac, 0xbfc0, 0x4148, 0x4264, 0x3326, 0x41cb, 0x4132, 0x2736, 0xb012, 0x11c0, 0x3453, 0xbfa9, 0x1e76, 0xb0f2, 0xb2b5, 0xba0e, 0x09f7, 0x1b71, 0x46a5, 0x415a, 0xb29f, 0x4493, 0xb0f9, 0x3943, 0x317b, 0x44e4, 0x438c, 0xbf0b, 0x4144, 0x07a1, 0x4392, 0xadf3, 0xaf15, 0x1422, 0x212f, 0xa165, 0x44db, 0x070c, 0x4381, 0xb0e2, 0x1f2b, 0x40a3, 0x4647, 0x3bf5, 0x1b02, 0x42c9, 0x2996, 0x43a7, 0x0d6b, 0x46eb, 0xb0e6, 0x4160, 0x41be, 0xb053, 0xb083, 0x1e27, 0xbf2a, 0x22b5, 0xbaf4, 0x08fb, 0x4556, 0x464b, 0x4333, 0x1337, 0x1a48, 0xb29e, 0x4167, 0x41ee, 0xb2d8, 0xbfa9, 0x1cc6, 0x4351, 0xb28b, 0x4308, 0x43f7, 0x432e, 0x1f58, 0x43b0, 0x4057, 0x41e8, 0x411c, 0x22d8, 0x3135, 0xba7c, 0x43a2, 0x2df5, 0x42cb, 0xb2d6, 0x441d, 0xb2ef, 0x2d14, 0x436b, 0xaed5, 0x322f, 0x1da6, 0xb08e, 0xbf2e, 0xb271, 0x33f0, 0x33be, 0x424c, 0x1be6, 0x4319, 0x2dc9, 0xb06b, 0x1f95, 0x3107, 0xb2ca, 0x439b, 0x1965, 0x2ed1, 0x45ba, 0x2066, 0xbf64, 0x1aca, 0x411b, 0xae5e, 0xadca, 0xb0cf, 0xba28, 0xbf85, 0xb278, 0x41c5, 0xb287, 0xb26e, 0xa4a7, 0x1aeb, 0xba16, 0x41b8, 0x3a44, 0xbf80, 0x42e4, 0x0580, 0x3e82, 0xb2f4, 0x1afe, 0xb244, 0xb25b, 0xbafb, 0x1961, 0x41f9, 0xbfb0, 0xba5f, 0x0ba4, 0x40d9, 0xb21b, 0xbf29, 0x01dd, 0xa878, 0xbfd0, 0xb2ad, 0x1ba2, 0x1c18, 0xaf6a, 0x39a8, 0xbfc0, 0x3400, 0x40b3, 0x192a, 0x4128, 0x3f37, 0x42c0, 0x14c8, 0x42d4, 0x402d, 0x21a6, 0xb25b, 0x4045, 0x184f, 0x406f, 0xb0de, 0x4046, 0xbf8a, 0xba50, 0xaae3, 0x2719, 0xb2d4, 0xb22b, 0x35ad, 0x40fb, 0x0e62, 0x43ac, 0x439a, 0x0f5d, 0x4063, 0x1ca1, 0x40a4, 0x406a, 0x18fb, 0x27b5, 0x409a, 0xbf6b, 0x43c7, 0x293d, 0x074b, 0x336f, 0x456f, 0x41e1, 0x40a7, 0x2b00, 0x3c88, 0x4616, 0x1f8d, 0xb2b5, 0x129a, 0x4224, 0x407e, 0xbf53, 0x3fae, 0xbacf, 0x4313, 0x46ec, 0x1a1c, 0x4268, 0xb07b, 0x433e, 0x4407, 0x4189, 0x45ce, 0x428c, 0xb068, 0x413f, 0xbfac, 0xb200, 0xbae2, 0x1f79, 0x4236, 0x417c, 0xa561, 0xbaf0, 0xb2f6, 0x1290, 0xb243, 0xba63, 0x40c8, 0xbfa0, 0xafcf, 0x43f2, 0x1bdd, 0x42bf, 0xbfa3, 0x42c0, 0xba51, 0x1c8c, 0xba37, 0xb239, 0x4008, 0xb2e7, 0xb290, 0xbaee, 0x431d, 0x421f, 0x4364, 0x437c, 0x01b9, 0x43ad, 0x4544, 0x2c61, 0x4640, 0xa354, 0x2ad2, 0x2d4b, 0x4052, 0xbf2f, 0xaff9, 0xaa46, 0xae6b, 0x0c27, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x60ab798e, 0xeb64755b, 0x32fad41e, 0x906517a9, 0x6b2d648d, 0x9c5a9d43, 0xefba7776, 0x297a6da6, 0x3750e535, 0xf070ec03, 0x5dff78cd, 0x69746ba1, 0xc40e9ad7, 0xa94af44d, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x27c1d138, 0x00000040, 0x00000374, 0x0000192c, 0x82cf9201, 0x00000000, 0x00000408, 0x000082cf, 0x27c1d138, 0x4f83a270, 0x27c1d138, 0x00000068, 0xfffffed0, 0x0000025c, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb234, 0x4215, 0x2ab0, 0x43d9, 0x42a0, 0x432b, 0x0bb3, 0x191b, 0x086d, 0x4251, 0x42d3, 0xba6f, 0xa2e4, 0x4069, 0x42ec, 0xae3e, 0xb293, 0xbf16, 0xa0fb, 0x433a, 0x4366, 0x45a5, 0x41d4, 0xb2ca, 0x1ed4, 0xb06d, 0x2a1c, 0x1fbf, 0xbaf6, 0xb266, 0xb293, 0x1c35, 0x3693, 0x2df7, 0xb031, 0x1c23, 0xad6c, 0x36e1, 0x4116, 0xbfc0, 0x32a2, 0xba37, 0x455d, 0x40e1, 0x4047, 0xbf01, 0xbaf8, 0xbad3, 0x19a9, 0x4255, 0xb074, 0x4360, 0xba2b, 0x1e38, 0x18c5, 0x44ed, 0x44e5, 0x187b, 0xb22a, 0x40a3, 0x1852, 0xb276, 0x40dc, 0x42dc, 0x2cf9, 0xb041, 0x18c1, 0xba29, 0xbff0, 0xa748, 0x42ab, 0xbf7d, 0x253d, 0x4352, 0x0c85, 0x438f, 0x4274, 0x3cf4, 0xafce, 0x1974, 0xb074, 0xbfa6, 0x4257, 0xba5b, 0x4037, 0xa515, 0xb0bd, 0x04fe, 0x1cf8, 0x420c, 0x4119, 0x4184, 0x4312, 0x45d3, 0x41ef, 0x4011, 0xb00a, 0xb027, 0xb0f9, 0x41bc, 0x4087, 0x41bd, 0xa595, 0x4126, 0x45d8, 0xbfbd, 0x2aaa, 0xb22b, 0x4361, 0xb2ad, 0x02e9, 0x43f5, 0xb21b, 0x41c2, 0x0d25, 0x03aa, 0x3663, 0x4404, 0x45ce, 0xbf8a, 0x21db, 0x4362, 0x1af0, 0xbf68, 0x415b, 0x4223, 0x2828, 0x4075, 0xbf42, 0xb262, 0xacd1, 0xb011, 0x1da5, 0x43fc, 0x408c, 0x43b3, 0x43c5, 0xb0c7, 0xa5b2, 0xbf17, 0x409d, 0xbac7, 0x4440, 0x0ba9, 0x4025, 0x2022, 0xa67c, 0x4214, 0x40e4, 0xba71, 0xb28c, 0x40ea, 0x41cf, 0x41e2, 0xbfbf, 0xb2bc, 0x3e8c, 0x40eb, 0xb2d4, 0xa13f, 0x4311, 0xbf52, 0x19f8, 0xbf70, 0xb2f4, 0xb06c, 0x0ab2, 0x0a8e, 0x40ae, 0xba23, 0x1c9c, 0x4211, 0x17ac, 0x1c16, 0x1235, 0xab09, 0x1a59, 0x41f7, 0x46a3, 0x1b42, 0x416e, 0x0f01, 0x0b26, 0x407f, 0x4228, 0x0d9f, 0x43c2, 0x435d, 0xbf22, 0x4270, 0x1893, 0x4246, 0xbf06, 0x30d0, 0xba13, 0xae81, 0x0378, 0x4088, 0x463e, 0x4101, 0x4243, 0x1ddd, 0x41bf, 0xba17, 0xae45, 0x1c8e, 0xb014, 0xa8ef, 0x4167, 0x4207, 0x40c6, 0xb018, 0xa950, 0x4073, 0x049b, 0x41a2, 0x45c5, 0x424b, 0x439a, 0x1a0a, 0x3231, 0xbf32, 0xb20b, 0x433b, 0xba2b, 0xbf5d, 0x430c, 0xb2f5, 0x4131, 0x43e4, 0x1e5f, 0x42e8, 0x1f98, 0x115d, 0x4268, 0x41eb, 0x05f8, 0x28e7, 0x06ea, 0xb002, 0xb2d0, 0x1cac, 0x43ca, 0x0c9b, 0x425b, 0xbfe0, 0x40d1, 0x42a1, 0x3faf, 0x403d, 0xb011, 0xb204, 0x4204, 0x4274, 0xbf7f, 0x0588, 0xa465, 0xbadf, 0x1860, 0x0720, 0x4115, 0xbaf0, 0x417e, 0x4569, 0xb211, 0x40a2, 0x43d8, 0x38ea, 0x4061, 0x42df, 0xb2d6, 0x415e, 0x432c, 0xba38, 0x4353, 0x4352, 0x45a5, 0xbf05, 0x4078, 0x3f2a, 0xba41, 0x44e8, 0xb209, 0x1da4, 0xb2c8, 0x41e8, 0xbf29, 0xbafe, 0x1a67, 0x1829, 0x1bf3, 0x0647, 0xb2c6, 0x0c9f, 0x41f5, 0x11dc, 0xbf87, 0x4124, 0xbacc, 0x424a, 0x2a70, 0x403e, 0xb25a, 0x1d42, 0xa62a, 0x4097, 0x43d5, 0xbfa0, 0x3791, 0xba00, 0x1644, 0xba57, 0xb215, 0x4251, 0x07ef, 0xbf4a, 0x2bd0, 0x2e96, 0x4355, 0x4393, 0x4376, 0xbf59, 0x411a, 0x400e, 0xba18, 0x43f5, 0xbf65, 0x4054, 0xae5d, 0xb2e8, 0x4230, 0xa533, 0x4134, 0xb2d9, 0x4149, 0x3cee, 0xbf80, 0x40a1, 0x1745, 0x1b9c, 0x1b90, 0x41be, 0x4468, 0xab8c, 0x4618, 0x1d05, 0x1c91, 0x259f, 0x4228, 0x414b, 0x4028, 0xbfd9, 0x4298, 0x4139, 0x1f8a, 0x4355, 0x4302, 0x42f9, 0x1951, 0xbf9c, 0xba05, 0x400c, 0xacba, 0x44b1, 0xb216, 0x4151, 0x2013, 0x40df, 0x3de1, 0x1f4a, 0x4036, 0x401e, 0x42a0, 0x42cd, 0x40dd, 0x428c, 0xbfd2, 0x4012, 0x1b19, 0x4099, 0x437f, 0x0289, 0x40ef, 0x1f98, 0x418f, 0x19e0, 0x40c5, 0xb2ea, 0x4366, 0x46ea, 0x427a, 0x4464, 0x4262, 0x4280, 0xbafa, 0xbf6f, 0x21d0, 0x1e60, 0x0a92, 0x0f02, 0x2122, 0x434c, 0x40c6, 0xba0c, 0xbf64, 0xa3fa, 0x10b4, 0xbac4, 0x41f5, 0x4079, 0x4197, 0x226a, 0xb2bf, 0x209e, 0x4085, 0xb07d, 0x437c, 0x3671, 0x2d5b, 0x4090, 0x44d0, 0xbf3b, 0x4259, 0xbaf9, 0x4329, 0x418d, 0x4292, 0x4485, 0x4694, 0x4059, 0xb21e, 0x41e6, 0xad2c, 0x3caf, 0xbf0b, 0xb29f, 0x4133, 0x4366, 0x04e2, 0xba2f, 0x1f7a, 0x41a6, 0x0e62, 0xb2f1, 0x43a3, 0xb09c, 0x443d, 0x025f, 0x4128, 0x1fb1, 0x1dea, 0x3733, 0xbfc5, 0x4263, 0xbad3, 0x42f0, 0xb238, 0xba0d, 0x3c1f, 0x4206, 0x41bb, 0x430a, 0x43ad, 0x1984, 0x4303, 0x0cb0, 0xba11, 0x1c1f, 0xb2df, 0xba43, 0x415b, 0xbfa4, 0xbf90, 0x1c9d, 0x1d4a, 0xab5e, 0x46f0, 0xb22e, 0xba16, 0x09f6, 0xba6a, 0x408e, 0x4380, 0xac19, 0x439f, 0x43d2, 0xbfa0, 0x4258, 0xa366, 0x4051, 0xbf36, 0x438a, 0x429f, 0x43f3, 0xb2c5, 0x3bd3, 0x1fa1, 0xb23c, 0x443d, 0xba27, 0xb089, 0x4354, 0x0f0f, 0x1da0, 0x4384, 0x4228, 0xb201, 0xbfe2, 0x4227, 0x395f, 0x447c, 0x43b0, 0x3a46, 0x1ba2, 0xbf80, 0x1c7c, 0xb200, 0x42e2, 0xba2b, 0xb279, 0x43e7, 0xb275, 0x3d99, 0x23a0, 0x41c6, 0x020a, 0x456b, 0x41ea, 0xb0dc, 0x466d, 0x3dec, 0xa2f3, 0x0a1d, 0xbfcc, 0x2a39, 0x215f, 0xbade, 0xb2a5, 0xb22a, 0x406c, 0x4093, 0x33ea, 0x117d, 0xabbf, 0x4291, 0x4055, 0x425e, 0x43e6, 0x4202, 0x4383, 0xbf3e, 0xae2d, 0x4165, 0x404b, 0x45e2, 0xbf60, 0xb0c6, 0x405b, 0xbff0, 0x0ed3, 0xb256, 0x4305, 0x02d6, 0x10a9, 0xac5f, 0xba55, 0xad0d, 0x1b53, 0x42c5, 0x40b6, 0xb211, 0x40cc, 0x1d30, 0x4098, 0x1df1, 0xbf66, 0x4553, 0x41af, 0x0cff, 0x4307, 0x40fa, 0x318f, 0x1b3e, 0xb2f6, 0x442e, 0xb229, 0x0b45, 0xb2de, 0x1948, 0xba36, 0x1a5b, 0x40ae, 0x0dc0, 0x4316, 0x1e5a, 0x258e, 0x4299, 0x1e58, 0x42b1, 0x4322, 0xb058, 0xbf0b, 0x080d, 0xba66, 0x4326, 0x096f, 0x4366, 0x42eb, 0x1b82, 0x43af, 0x4032, 0xb20e, 0x42a5, 0x1ed5, 0xa3f5, 0x410f, 0xba5d, 0x4120, 0xbf5f, 0x2aa9, 0xba1b, 0x0051, 0x40ad, 0x04a1, 0x420e, 0x3c42, 0x42e0, 0x42bb, 0x43d1, 0x447f, 0xbf14, 0x0ca2, 0xba56, 0x42ed, 0x4372, 0x2ffe, 0x42bd, 0x411e, 0x400a, 0x4285, 0x4652, 0xb079, 0x40ee, 0x0a3a, 0x11c7, 0x403f, 0x14df, 0x40eb, 0xb236, 0x323f, 0x43d7, 0x05c3, 0xbfca, 0xb0cc, 0xba51, 0x1d2c, 0xbf72, 0xba3a, 0x2c3a, 0x4154, 0xbf31, 0xba27, 0x19d1, 0x43b4, 0x4167, 0xba36, 0x42bb, 0x0e3a, 0x4011, 0x4363, 0xbae0, 0x186e, 0x40d1, 0x3dff, 0x38c3, 0x12b7, 0x0292, 0x4561, 0xb037, 0x1dda, 0x401e, 0x43d8, 0xacf4, 0x3e49, 0xbf81, 0x1a75, 0x1d50, 0xb2e7, 0x4080, 0x108c, 0x0fbc, 0x41c5, 0x4189, 0xbad6, 0x1d8f, 0x3b0a, 0xa7a7, 0xbfe0, 0x4072, 0x43a6, 0x19e2, 0x4376, 0x09a4, 0x42fe, 0x0780, 0x4158, 0xbfc9, 0x435d, 0xa29f, 0x4111, 0x01c5, 0x435f, 0x41b2, 0xba73, 0x46cd, 0xbf1b, 0xba2a, 0x161c, 0xb0f2, 0xbff0, 0x00c3, 0xa332, 0x2528, 0x19e9, 0x26a9, 0x1d5c, 0xb2ff, 0x42d8, 0xbf70, 0x175e, 0xbf3c, 0x40db, 0x43af, 0x2e2b, 0xb207, 0x421b, 0x4172, 0x0976, 0x410d, 0x439f, 0x1933, 0x1b15, 0x0093, 0x43df, 0x41f1, 0x40c6, 0x1c52, 0x1e4d, 0x4600, 0x0be1, 0xbf7d, 0x2de0, 0x43e4, 0x4319, 0x428c, 0x41bd, 0x4155, 0x46ab, 0x2730, 0x3a48, 0x4365, 0x403d, 0x1ad8, 0x0537, 0x41a0, 0x1994, 0x401b, 0x4088, 0xb2a0, 0x4050, 0x1dc2, 0x435d, 0x437d, 0xb0b7, 0x4060, 0x4109, 0x44aa, 0x41c1, 0xbf8b, 0x43f8, 0xb2bc, 0xba6e, 0xb023, 0x215b, 0x43c9, 0x43c4, 0x1feb, 0x00f7, 0x40e7, 0xb267, 0xbfc5, 0xb229, 0x42e8, 0xbad3, 0x4171, 0x400c, 0xbadf, 0x42ac, 0x431d, 0x4393, 0x4380, 0xa188, 0xbf81, 0xa3d8, 0x442b, 0x42d3, 0x41ce, 0x1e19, 0x4143, 0x4005, 0xb0c1, 0x4211, 0x455c, 0x25ba, 0x0780, 0x422e, 0x400f, 0x45d4, 0x2e74, 0x41cb, 0x1c7e, 0xbfcb, 0xb26a, 0x46f5, 0x3ee4, 0xa482, 0x18c4, 0x44b2, 0x4204, 0x43db, 0x3545, 0x421f, 0xb2f0, 0x463e, 0x2001, 0x43eb, 0x42d4, 0xb28a, 0x3886, 0x4191, 0xadd9, 0x415b, 0x4138, 0x4103, 0x45c9, 0x111b, 0xb02b, 0xb0e6, 0xbf3d, 0x23cd, 0x40ad, 0xb2d2, 0xb0cb, 0xb2e1, 0x21e4, 0xba28, 0xba05, 0xbf48, 0xbfb0, 0x060e, 0x1fa5, 0xba0d, 0x45dd, 0x4017, 0xbf1b, 0x30fd, 0x210c, 0x42b3, 0x17b5, 0x401f, 0x0b91, 0x264c, 0x40af, 0x40c2, 0xbf7d, 0x0302, 0xa7b3, 0x43e0, 0xba6c, 0xbf18, 0x0add, 0x1fe5, 0x0f9f, 0xb073, 0x381f, 0x1b07, 0xb0fa, 0x40d8, 0x1852, 0x4211, 0x42a6, 0x4036, 0xbac4, 0xa27b, 0x435d, 0x18eb, 0x4171, 0x4318, 0xb08e, 0x4296, 0xb2f4, 0xb2ea, 0x3e89, 0xbf32, 0x43b9, 0x1a0b, 0xb2d2, 0x429c, 0x2aad, 0xbf90, 0x18a6, 0x4360, 0x1a4a, 0x4435, 0xba4e, 0x4071, 0x1b83, 0x465d, 0x2ed8, 0x1a91, 0xbada, 0xbfb1, 0x41a8, 0x278e, 0x4182, 0x42c1, 0x1821, 0xbfa0, 0xbad2, 0xb0fe, 0x42e1, 0x1964, 0x46b9, 0xb2a2, 0x40f9, 0xbacc, 0x4221, 0x425a, 0x4464, 0x418e, 0xb0e6, 0x4392, 0x3a7b, 0x4206, 0x1970, 0xbf67, 0x4241, 0xb2f2, 0xbad7, 0x429e, 0x4332, 0x42e6, 0xa4a7, 0x42f6, 0x4011, 0xa32e, 0x40bb, 0x4651, 0x4458, 0x4309, 0x43f5, 0x438c, 0x2ca1, 0xba00, 0x438e, 0xba55, 0x1c48, 0x43f6, 0x41d3, 0xb2a4, 0x4034, 0xba2f, 0xbf0f, 0xbf60, 0xbaf7, 0x42fc, 0xbacb, 0x4217, 0x421e, 0x4113, 0x1863, 0x4103, 0xb052, 0x43b2, 0xb2ab, 0x440f, 0xb22a, 0x4207, 0x40df, 0x449b, 0x185d, 0xbf33, 0x0307, 0x1f5c, 0xb22f, 0x365e, 0x415a, 0xb218, 0x44fc, 0x3db0, 0x41bf, 0x41e6, 0x21d8, 0x412b, 0x4065, 0xbf4d, 0x1013, 0xa00f, 0x444e, 0x4341, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x81c55787, 0xada11b20, 0x3e10df97, 0x7b862743, 0x1f0517ba, 0x4d0933c2, 0x10df7fe1, 0x2dec0c53, 0xcbcf889f, 0x3e17e3e5, 0x8eacf129, 0xd8f1b147, 0xb8b631ab, 0x63405cb4, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x0000181c, 0x000000d8, 0x0000fffe, 0x00000000, 0x00000a00, 0x7f37fe5f, 0xfffffffc, 0x00000000, 0x00000000, 0x0000008e, 0x7f36f510, 0xfb5e1489, 0x0000183a, 0xbf82a5e0, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x43e9, 0xb2ae, 0x2b63, 0x4354, 0xbf91, 0xb23c, 0x4231, 0xb075, 0xb2fc, 0xba3a, 0x4162, 0x1cd2, 0x42af, 0x426e, 0x43fa, 0x0099, 0x1acc, 0x4434, 0x411d, 0x4624, 0xb25e, 0xbfc4, 0x1ef8, 0x45ac, 0x0b2d, 0xb2b0, 0x424d, 0x365f, 0x4336, 0xb0ac, 0x2a8b, 0xaa5c, 0x1bc1, 0x40ca, 0x18db, 0x3c16, 0x4181, 0x4348, 0x1cdf, 0x0413, 0xb2ad, 0xbfce, 0x1e96, 0x4304, 0x405b, 0x42ed, 0x1cea, 0x4159, 0x4057, 0x1c6b, 0x4256, 0x372b, 0x41d8, 0x4268, 0xa6ea, 0x430f, 0x1e36, 0x42bc, 0x40f5, 0xa28f, 0x1ae4, 0xb2a0, 0x4282, 0xa134, 0x4355, 0x42ba, 0x4395, 0x1faa, 0x154b, 0x434a, 0xbf6e, 0x43d1, 0xb2cd, 0x40b8, 0xbfe0, 0x0c97, 0x2541, 0x350e, 0x41c6, 0x40b5, 0x1e4d, 0x4048, 0x4657, 0x428a, 0x07dc, 0x42bb, 0x3923, 0x430d, 0x41a0, 0xbaed, 0xbf4e, 0x40ee, 0x0acb, 0x1f73, 0xbae9, 0xa18e, 0x41a8, 0x1947, 0x181b, 0x4248, 0x4349, 0xbf11, 0x1e48, 0x41d8, 0x18dc, 0x2073, 0xb20f, 0x41df, 0x41bf, 0x1ba3, 0xbf90, 0x4144, 0x428d, 0x41f9, 0x43b8, 0xb239, 0x4180, 0x432a, 0x4119, 0x46f8, 0x4329, 0x4091, 0x4302, 0xb228, 0x4094, 0x40b1, 0xbfcd, 0x2d26, 0x454f, 0x18a4, 0x42fb, 0x1b0e, 0x0d5a, 0x1ddb, 0x14c4, 0x43c2, 0xa374, 0x426b, 0xb0c9, 0x4262, 0xbf80, 0xbf8f, 0x46d3, 0x40cb, 0x4377, 0xb2b3, 0x3a47, 0xba60, 0xbfd9, 0xb267, 0x429a, 0x1806, 0x43d3, 0x3142, 0xb2be, 0x1487, 0x18ab, 0x1cd7, 0xba36, 0x1027, 0xb028, 0x1742, 0x435b, 0x4147, 0x2501, 0x41f0, 0xbadb, 0x40af, 0x444b, 0xb24c, 0xbae8, 0x40aa, 0xb2fe, 0xa75f, 0xbfcb, 0x4162, 0x4146, 0x4328, 0x2a69, 0x4247, 0x286e, 0x0b98, 0x24fb, 0xb061, 0x43a8, 0x409b, 0x1a0b, 0x41b6, 0x435a, 0xb244, 0x242e, 0xbad4, 0x0f4e, 0xbf0b, 0x4069, 0x273a, 0x4422, 0x4208, 0x4248, 0x43fa, 0x4406, 0xb2c1, 0x4046, 0x40bb, 0x4267, 0xb2b6, 0x03b8, 0xb2fd, 0x3dd7, 0xba6c, 0xb29b, 0xba17, 0x4109, 0x1b66, 0xb2df, 0xb005, 0x3496, 0x4598, 0xbf48, 0x40b2, 0x2979, 0x40d4, 0x4269, 0xb2ba, 0x3b3b, 0x408f, 0x336d, 0x1b00, 0x42c9, 0x2a70, 0xbf00, 0xaf87, 0x4386, 0x40c3, 0x4601, 0x461a, 0xbf90, 0xb217, 0x1d8e, 0x1c52, 0x42eb, 0x403f, 0xa25c, 0xa437, 0xbfcd, 0x19b1, 0x413f, 0x404a, 0x19f3, 0xbae3, 0x20a3, 0xbf4e, 0x3dd9, 0x4157, 0xb2ee, 0x29e2, 0xbad3, 0x42ab, 0x40af, 0xbf0d, 0xbaea, 0x43c8, 0xbae8, 0x0451, 0x424a, 0x1c6b, 0x400e, 0x4144, 0x3a3a, 0x412f, 0xbf04, 0x42df, 0x3821, 0x40ec, 0x433e, 0x4196, 0x43fd, 0x42e0, 0xb00f, 0x44dc, 0x2af8, 0x157c, 0x4316, 0xb095, 0xbf4b, 0x1a70, 0x1aaf, 0xb263, 0x41a3, 0x42ae, 0x44ba, 0x289a, 0xb293, 0x40d8, 0x3bc3, 0x42a1, 0x41e8, 0xbaf7, 0x403d, 0x418d, 0x42cf, 0xb023, 0xb05c, 0x4029, 0x40e4, 0x4209, 0x40fc, 0x108c, 0x40d5, 0xbf14, 0xb0e2, 0x425c, 0x4220, 0x244c, 0x0736, 0x1d18, 0x401d, 0x40ac, 0x4045, 0x4051, 0xaf3a, 0xbaef, 0x4451, 0x439a, 0x43ba, 0x406b, 0x1daf, 0x409e, 0x44ad, 0xb09c, 0x4087, 0x205e, 0x4540, 0x40fc, 0xb27d, 0xbf7a, 0x28c4, 0x42c5, 0x4178, 0x42eb, 0x40d3, 0xb273, 0x45f3, 0x401b, 0x43ca, 0x43f4, 0x43d1, 0xaed9, 0xba70, 0x41c3, 0x11e8, 0x41bb, 0x40e2, 0x41c0, 0xa5a2, 0x09c4, 0xbfc0, 0x43aa, 0x388c, 0x411e, 0x189f, 0xbf81, 0x4339, 0x1f88, 0x1b87, 0xa26a, 0xbf0a, 0x4366, 0x415a, 0x42ac, 0x41bc, 0xbf60, 0xba4b, 0xba12, 0x41b7, 0x413d, 0xb02f, 0x4235, 0x0548, 0x0509, 0x41de, 0x42c6, 0x16c5, 0x3d8d, 0xbf0d, 0x2de3, 0x40b7, 0x417f, 0x40f6, 0xa413, 0x4232, 0xba60, 0x4126, 0x4044, 0x1bd0, 0x2d4b, 0x20bd, 0xb215, 0x40d9, 0x415c, 0xbf08, 0x1d75, 0x1e01, 0x411c, 0x0b49, 0x410e, 0xbfcf, 0x1455, 0x0ee7, 0x05cd, 0x4394, 0xb2b1, 0x432c, 0x25fb, 0x4671, 0x4053, 0x3e48, 0x0be7, 0x405f, 0x4276, 0x41f6, 0x440e, 0x40f6, 0x435a, 0x41a5, 0xb2ef, 0x3d3f, 0x443a, 0x332a, 0xb00d, 0x4432, 0x1d1c, 0x428e, 0x4296, 0x42ef, 0xbfaf, 0x4062, 0x1918, 0x08bd, 0x4241, 0xba2f, 0x40b9, 0x1e32, 0x42be, 0x2983, 0xbf99, 0x402b, 0x423e, 0x41e5, 0x403a, 0xbf70, 0x41d1, 0x42cf, 0x434f, 0x41ea, 0x1dd7, 0x418c, 0x418f, 0xb248, 0x1e0f, 0x2ec4, 0x42f5, 0x40e2, 0x4330, 0x4118, 0xad76, 0xb089, 0x41b9, 0x40de, 0x42fa, 0xba54, 0xb2c6, 0xbf09, 0x4102, 0x41e7, 0xb08b, 0x41bd, 0x1dbf, 0x4140, 0x4009, 0x42fc, 0x46db, 0xbad8, 0x45ae, 0x40e0, 0x4212, 0x43fd, 0x43cb, 0x435f, 0x4276, 0x15a5, 0xb222, 0x409b, 0x40cb, 0xbf48, 0xad26, 0x40e7, 0x3491, 0x36b5, 0xbfa6, 0x4192, 0x43e1, 0x430d, 0xb28b, 0xaa14, 0x42eb, 0x4492, 0x41a8, 0xa8d2, 0x436e, 0x41c9, 0x4211, 0x42cf, 0x0d26, 0x41ea, 0x40bd, 0x0f79, 0x40d2, 0x394c, 0x4072, 0x40c9, 0xbf84, 0xb278, 0x2591, 0x43a9, 0x13e4, 0xb287, 0x02ad, 0x3290, 0x43ec, 0x4141, 0x431a, 0x46a9, 0x4629, 0x2e09, 0x1a14, 0x41f5, 0xba19, 0x35d8, 0x2c4a, 0x43dc, 0xb28e, 0x40b7, 0x41db, 0x41fe, 0x2f4d, 0xbfe8, 0x4459, 0x46d8, 0x42c5, 0x1d4f, 0x1a9e, 0x4203, 0x42b2, 0x4299, 0x428d, 0xbf29, 0x4129, 0xb223, 0x427a, 0x41c5, 0x089e, 0xad9a, 0xb24c, 0x1f2a, 0xb023, 0xb291, 0x42b3, 0xb07a, 0x1e3c, 0xa3d6, 0x4194, 0xbaf7, 0x370e, 0x1faa, 0x197f, 0x409e, 0x19c8, 0x4616, 0xbf7a, 0x42a2, 0x437e, 0xba3e, 0x4312, 0x428a, 0x418a, 0x433b, 0x03d8, 0x0250, 0xb29a, 0x1dd0, 0x33c0, 0x4334, 0xb0fd, 0xbfa0, 0xb279, 0x4067, 0x405b, 0xb23e, 0x2224, 0x4375, 0x2e15, 0x406f, 0x43b4, 0xbf94, 0xba6d, 0x04eb, 0x42f4, 0x42d7, 0x4184, 0x40cb, 0x40c5, 0x422d, 0x19c7, 0x415e, 0x426f, 0x4124, 0xb2e7, 0xa9a4, 0x40ed, 0x412f, 0x43c9, 0x3ca8, 0x1b05, 0x32fc, 0x4276, 0x4037, 0x447f, 0x40c3, 0xb267, 0xbf81, 0x43bb, 0xb243, 0x4570, 0x2b22, 0x0e9c, 0x4063, 0x1bea, 0x4055, 0x2367, 0xba35, 0x1e69, 0x0926, 0x44f5, 0xb00b, 0x18f8, 0x4007, 0xb255, 0x4350, 0xb0a5, 0x4095, 0xba26, 0xa26b, 0x14b9, 0x4346, 0xbfae, 0x4312, 0x4446, 0xbf80, 0xa9c8, 0xba7d, 0x410a, 0xbfa2, 0x2319, 0xb282, 0x4185, 0x425f, 0x41c1, 0x409c, 0x1a10, 0x1003, 0x284b, 0xbae6, 0xb221, 0xb242, 0x4262, 0xbad8, 0xbafe, 0xaa78, 0x4223, 0x41d8, 0xb0f9, 0x4091, 0x445c, 0x321e, 0x42c1, 0x4023, 0x411a, 0x438f, 0x43fd, 0xbf29, 0x434f, 0xa7a0, 0x1d7a, 0x40f0, 0xba4f, 0x2879, 0x4093, 0xa50b, 0x42b5, 0xb289, 0x43dd, 0xb014, 0xba26, 0x0450, 0x413b, 0x1828, 0xba63, 0x4156, 0x292f, 0x428f, 0xb053, 0xbf68, 0x42c4, 0x059b, 0x14ec, 0x4207, 0x43c8, 0x43b7, 0xba72, 0xb297, 0x1ab9, 0x1525, 0x416d, 0x420c, 0x446b, 0xb205, 0x466b, 0x416f, 0x19c5, 0x4351, 0x22d5, 0x4670, 0xb06b, 0x0374, 0x066b, 0xbfb9, 0x0c25, 0xabee, 0x4143, 0x41f2, 0x181b, 0x42f7, 0x40ab, 0x4130, 0xba4a, 0x1a6d, 0xba02, 0x4165, 0xbf60, 0x417e, 0x340c, 0xba72, 0xbaff, 0x3ddc, 0xbf35, 0x415a, 0x4011, 0x42cb, 0x437c, 0x3d03, 0x41d4, 0x1855, 0x0b24, 0xa86b, 0x41ef, 0xb262, 0x41a4, 0xb272, 0x43a4, 0x030c, 0x4365, 0xbf52, 0x46b5, 0x464d, 0x42b3, 0x1b6e, 0xba3d, 0x20f7, 0x35dc, 0x2ee9, 0xbf4e, 0x4378, 0xaf6a, 0x142e, 0xb290, 0x42d9, 0x40d8, 0xb069, 0x1fb9, 0x430a, 0x0594, 0x4231, 0xbafd, 0x3a53, 0xb2b1, 0xb0e0, 0x001d, 0xa85d, 0x17cd, 0x42b5, 0x4330, 0xb012, 0xbacd, 0x0ed9, 0xbf86, 0xba22, 0x4068, 0x263e, 0x0048, 0xb2a4, 0xba4a, 0xbf3a, 0x4224, 0x426e, 0x0a8d, 0xa5f2, 0x14f4, 0x4164, 0x1e8e, 0x324b, 0xba1c, 0x40ed, 0x4108, 0x170d, 0x30b8, 0x04a7, 0xb073, 0x2d88, 0x4171, 0x1b68, 0xbfa4, 0x40fe, 0xbac1, 0xba57, 0xb2ce, 0x330f, 0x2614, 0x070e, 0xb0c2, 0x4244, 0xad7d, 0x06ac, 0xa5ef, 0x0d63, 0x42aa, 0xb2df, 0xb2ce, 0x42d1, 0x2ea4, 0xbf4d, 0x4185, 0x41bb, 0x45e1, 0x326c, 0x42ea, 0x443c, 0xb219, 0x41a7, 0xbf31, 0x42af, 0x43f5, 0x4137, 0x1efb, 0x4175, 0x40fa, 0x4340, 0x40f0, 0x4342, 0x43f6, 0xbafe, 0x17c7, 0xbfcc, 0x4485, 0x43e3, 0x4085, 0x4003, 0x4187, 0x07dd, 0x3c2d, 0x1acf, 0x40a2, 0xbae2, 0x417c, 0xbfe0, 0x198c, 0xbf8c, 0x1a73, 0xbafc, 0x2c1d, 0x0cf6, 0x43a9, 0x403d, 0xbf34, 0x4297, 0xb000, 0xb27b, 0xab7d, 0x05c9, 0xb245, 0x4326, 0x426d, 0x4226, 0x42df, 0xb0d8, 0x46f2, 0x43ca, 0x4096, 0x283c, 0x4406, 0x41bc, 0xbf6d, 0x433b, 0x458b, 0x41be, 0xb019, 0x4029, 0x4167, 0x42a1, 0x4024, 0xbaf1, 0xb26c, 0x1e3d, 0xb292, 0x41ea, 0x241b, 0x4024, 0x1aec, 0x431e, 0x4606, 0x4209, 0x356f, 0x4308, 0xa65e, 0xbf7d, 0xbf70, 0x1a6b, 0xbaf8, 0x42f3, 0x1dac, 0xb2bf, 0x26b6, 0xba4a, 0x460b, 0x4390, 0x422d, 0x0a84, 0xb0d1, 0x43d7, 0x4022, 0xb0ee, 0x409e, 0xba72, 0x2a10, 0x429f, 0xbf70, 0x304b, 0xa8bc, 0xbf2c, 0x4288, 0x41cc, 0xbf28, 0xba26, 0x419b, 0x42af, 0x4357, 0xab5e, 0x4307, 0xbae7, 0x4449, 0x2dca, 0x46d9, 0xa4d7, 0x4384, 0xb0ae, 0x39ca, 0x4153, 0xba66, 0x43ae, 0xba2b, 0xbfe1, 0x189b, 0x4062, 0xbaf2, 0xba5c, 0xb041, 0x431f, 0x0f58, 0x4474, 0xbf19, 0x421c, 0x09c9, 0x0947, 0xb013, 0xba56, 0x3cdb, 0x43d0, 0xbf7a, 0x40d6, 0x079a, 0x030a, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x69b72ece, 0x9ea70faf, 0x0bef8eae, 0x8b8f6dd5, 0x4fde440e, 0x8c63dba4, 0xee0baa0f, 0xb2812116, 0x764ead9f, 0xda04bbb5, 0xc49fd907, 0xf371f841, 0xb64410f1, 0x71a52826, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0xffffef27, 0x00002035, 0x00000000, 0x4e010000, 0x014dff25, 0x0000014e, 0x00000000, 0x00000000, 0xf371f841, 0xf371f841, 0x00000000, 0xf371f841, 0xa9b60932, 0x41f962cf, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x40c2, 0x4026, 0x4391, 0x4114, 0x43fc, 0x4369, 0x4099, 0x0707, 0xbfe0, 0xbf8f, 0x3fa9, 0x4220, 0x1d09, 0x4261, 0xba25, 0xb018, 0x2131, 0x41ab, 0x404f, 0x4031, 0xbaf3, 0xb2b6, 0x1e5f, 0x4202, 0x1b23, 0x42e2, 0x2214, 0x4247, 0xba0a, 0x4146, 0xbf5f, 0x0974, 0xa6cd, 0x1ecc, 0x41a9, 0x403b, 0x4312, 0xb08f, 0x4074, 0x4191, 0x4381, 0x404f, 0x45a0, 0xb2e0, 0x0258, 0x1eb3, 0x057e, 0x444f, 0x259e, 0xbfc7, 0x1e86, 0xb080, 0x43cd, 0x1c81, 0x41cb, 0xba3f, 0x423a, 0x4627, 0x4040, 0x32a8, 0x3811, 0x463d, 0x284f, 0x0180, 0xb29e, 0x2fcb, 0x18ee, 0xba4b, 0x4239, 0xba51, 0x42b7, 0xbad0, 0x429c, 0xbf2c, 0x438a, 0x41aa, 0x0bf1, 0x23b2, 0x4393, 0xb27f, 0xafd2, 0xb233, 0x4434, 0x2e78, 0x41f2, 0xbfac, 0xb27e, 0x1ace, 0x40b0, 0x45d2, 0x400e, 0x1fed, 0xb071, 0x239f, 0x4171, 0x18f1, 0xa24f, 0xb2b1, 0x421b, 0x0b5c, 0x464b, 0xba55, 0x424a, 0x4541, 0xbf90, 0x440d, 0xb2de, 0x42a9, 0xb05b, 0xbfce, 0x431a, 0x465b, 0x1108, 0x411b, 0xbafd, 0x4141, 0x40fd, 0x174d, 0x46a8, 0xba6f, 0x2829, 0x0e03, 0x19ed, 0x26d3, 0x4379, 0x40a2, 0xb21b, 0x40bc, 0x185d, 0x4352, 0xb09a, 0xb280, 0x40ae, 0x42a9, 0x4218, 0xba0a, 0x1aa9, 0xbf86, 0xaf41, 0x40df, 0xb2f9, 0x429a, 0xa93c, 0x26b5, 0x22a0, 0xa0ec, 0x1def, 0xb2ea, 0xb231, 0x437b, 0xab30, 0x41ee, 0x1f12, 0x423f, 0x43b6, 0x4218, 0xb0e1, 0x40b9, 0x4175, 0x41aa, 0xbf9f, 0xb04a, 0xb28d, 0xba0e, 0x40f5, 0x1bda, 0x4460, 0x45ce, 0xb253, 0x1fbc, 0x2b3b, 0xb054, 0x216b, 0x1ef5, 0xb239, 0x4117, 0x3e6a, 0x1c59, 0xb212, 0x1e71, 0x435d, 0xb0f6, 0x40af, 0x426f, 0xb267, 0xbfac, 0xb29a, 0x4231, 0x1064, 0xb2bf, 0x42ee, 0xbf90, 0x4194, 0xb2cf, 0x4029, 0x1651, 0x43f7, 0x4091, 0xa737, 0x1c29, 0x36ec, 0x2620, 0x4664, 0xb210, 0x4630, 0xaa7e, 0x4254, 0x4571, 0xbf04, 0x41ad, 0x40b9, 0x407d, 0x06a0, 0xb29d, 0x43af, 0xb2b2, 0xbafc, 0xb2bc, 0xb278, 0xb235, 0x40d5, 0x1e08, 0x3d80, 0x42f6, 0xb01d, 0x4383, 0x08db, 0x428b, 0x43d1, 0x1835, 0x1efa, 0x221b, 0xb057, 0x4051, 0x0216, 0xbfb5, 0xad04, 0x424a, 0x42b2, 0x427a, 0xa99f, 0x2987, 0xb2a1, 0x0a0e, 0xb2d1, 0xbf97, 0x41b9, 0x41d7, 0x18b3, 0x1e5f, 0x3969, 0x323f, 0x1a8e, 0x4643, 0x46e1, 0x4005, 0x435e, 0x4353, 0x35c5, 0x41af, 0x43c6, 0xbaf6, 0x05c4, 0xba7f, 0x41dc, 0x464f, 0x454e, 0xbfe1, 0x3f23, 0x46e4, 0x42e1, 0xb26b, 0x1fe7, 0x44a3, 0x22da, 0x02a8, 0x4366, 0x0239, 0xb2ba, 0x431e, 0x4267, 0xba59, 0x2155, 0x41ff, 0x43b6, 0x4112, 0xb289, 0xb28d, 0xa0b5, 0x415d, 0x4234, 0xbf81, 0x442b, 0x3b05, 0x4359, 0xb2f1, 0xad7d, 0xb2c7, 0x411f, 0x414b, 0x1ce1, 0x41dc, 0xb2ed, 0x4580, 0xba56, 0xb259, 0x407a, 0x1c86, 0xba38, 0x422a, 0x40c4, 0xbf81, 0x411b, 0x1dd0, 0x1eba, 0x3194, 0x3435, 0xb225, 0x43a8, 0x1b6d, 0x231b, 0xbf7d, 0x46ab, 0xb2ed, 0xb0f3, 0xb254, 0xb0e0, 0x41f3, 0x2773, 0x1869, 0x40c3, 0xb25f, 0x1820, 0xbf97, 0x3435, 0x41d2, 0xb2f2, 0x1c37, 0x176a, 0x425f, 0x43f8, 0x400b, 0x23a7, 0xb09a, 0xbff0, 0x414d, 0x4221, 0x283e, 0x43bd, 0x41b9, 0xbaeb, 0x3c41, 0x2790, 0x0b0a, 0xb2e4, 0x1fca, 0xbf80, 0xb0f0, 0xbac8, 0xbfc0, 0xbf7c, 0xba70, 0x41d4, 0x4655, 0x22f4, 0x40bc, 0x4223, 0x42e7, 0x39e7, 0x437c, 0xa249, 0x0634, 0x1fac, 0x413f, 0x2aa4, 0xa70a, 0x4651, 0x424e, 0x4074, 0xbfca, 0x405f, 0x1201, 0xba3f, 0x40ee, 0x40d4, 0x46ac, 0xa9d0, 0x1dc7, 0x41b3, 0x1d14, 0x41c8, 0x332d, 0xb2be, 0x4290, 0x4625, 0xa891, 0xbf70, 0x4335, 0xbf38, 0x42bc, 0xac43, 0x4180, 0x4357, 0x41c2, 0x3ce3, 0x368c, 0x414b, 0x40d4, 0x40a4, 0x4072, 0xbadd, 0x436c, 0x4102, 0x34c9, 0x4430, 0x43e2, 0x407d, 0x018e, 0xa86d, 0x431e, 0x4072, 0xbf74, 0x4271, 0xba6f, 0x4333, 0x180b, 0xb2c8, 0x18ca, 0x40a4, 0x2445, 0xaf56, 0x43a6, 0x438d, 0x41b5, 0xb20a, 0xb0ec, 0x410c, 0x012a, 0x40d4, 0x43e2, 0x40c9, 0x41ce, 0xb2c3, 0xba5e, 0xbf95, 0x4135, 0x1f74, 0x4107, 0x46e4, 0xbfd8, 0x42ed, 0x0c9a, 0x0919, 0x43f9, 0xbfc4, 0x4035, 0x0f3f, 0xbac2, 0xab07, 0x2db8, 0x0c01, 0x4012, 0x1ef4, 0xaae2, 0xb02a, 0xbaf4, 0xbfbc, 0x41d3, 0x4416, 0x40c7, 0x44e4, 0xb2c9, 0x40fa, 0x38e7, 0x42da, 0xba4a, 0x4145, 0x3e87, 0xb018, 0x42d3, 0x07f6, 0x42ec, 0xb2b7, 0x4379, 0xb060, 0xba0d, 0x40ff, 0x2f57, 0xba5b, 0x4263, 0xbf8b, 0x426a, 0x42e1, 0x41b0, 0x461c, 0x438c, 0x33c7, 0x43ef, 0x46b8, 0x1dd3, 0xb233, 0xbf1f, 0x439e, 0x4459, 0x4411, 0xbfd0, 0xab1b, 0xb067, 0xb277, 0x01d0, 0x415b, 0x416e, 0xb20e, 0x4309, 0x423b, 0x0763, 0xbf7e, 0xb04d, 0xb2b9, 0x1c91, 0x4399, 0x4341, 0xba55, 0x40ae, 0xb288, 0xb06e, 0xb2a7, 0x4095, 0x44e8, 0x19a4, 0x42d0, 0x4318, 0xb063, 0x41c6, 0xb284, 0x415f, 0xba44, 0x3c51, 0x4388, 0xb0d2, 0x433a, 0x418e, 0x430b, 0x40bb, 0x4075, 0xbf96, 0x4191, 0xba78, 0x4380, 0x402f, 0xbaec, 0x45da, 0xa7ba, 0xb280, 0x42cd, 0x0e15, 0xbfc0, 0x4292, 0x46c9, 0x44ad, 0xb280, 0x4246, 0xb262, 0xb247, 0x4082, 0x402b, 0xb230, 0xba1b, 0xb077, 0x41f3, 0xbf0c, 0x1901, 0xb016, 0x1edf, 0x3519, 0x4058, 0x418b, 0x42bb, 0xbfd0, 0x014d, 0x43dd, 0xbf52, 0x1b22, 0x434a, 0x41f1, 0x4181, 0xa9a5, 0xb20d, 0xb0cb, 0xbf42, 0x3727, 0xb271, 0x4088, 0xabfb, 0xbf61, 0x1ab9, 0x0344, 0x40cd, 0x41d4, 0x1f44, 0x43b7, 0x298e, 0xbf72, 0x1c94, 0xb0ed, 0x2b19, 0x4427, 0x4337, 0x40c3, 0x3601, 0x19a2, 0x414d, 0x18c7, 0x4006, 0x4313, 0x4253, 0x1d49, 0xbf94, 0xba13, 0xa1dc, 0xb2d7, 0x0b0b, 0x41c8, 0x41b5, 0x1b21, 0x3242, 0x1d53, 0x3ac5, 0xb0ec, 0xba22, 0xa9c9, 0x40ef, 0x4027, 0xb0bf, 0xb00f, 0x211f, 0xb018, 0x1fb8, 0x14c1, 0x410c, 0x4081, 0x4604, 0x3ab7, 0xb20f, 0x4301, 0xbf54, 0x42dc, 0x1861, 0x41bf, 0x405e, 0x1dbd, 0x17ce, 0x46e1, 0x4287, 0xa83c, 0x3e4d, 0xb03a, 0xb2bb, 0x41cd, 0x40a4, 0x430e, 0xb288, 0xb25d, 0x4312, 0xb0de, 0x18b3, 0xb2c9, 0x08aa, 0xba4f, 0xbf61, 0x42aa, 0xb249, 0x2856, 0xa13d, 0x4092, 0x424b, 0x2284, 0xb206, 0x40c5, 0x2801, 0x443b, 0x43df, 0x0c18, 0xbf35, 0x42f2, 0x4031, 0x4048, 0xb225, 0x4215, 0x449c, 0x1ecf, 0x43fd, 0x421a, 0x4584, 0xb03e, 0x2473, 0x16d8, 0x42c8, 0x44db, 0x43bb, 0xba55, 0xba18, 0xb06e, 0x1a1d, 0xb281, 0x23c5, 0xbf72, 0x4288, 0x176a, 0x431a, 0x4392, 0xb0d3, 0x43d6, 0x0263, 0x43e7, 0xbfde, 0x0aab, 0x22c9, 0x4203, 0xba25, 0xb2ed, 0x0d84, 0x3b27, 0x17d9, 0x1c5e, 0xb027, 0x19d8, 0xb22d, 0x4284, 0xada7, 0x44c3, 0x431d, 0x1549, 0xb293, 0xb20b, 0xbaf3, 0x419b, 0x2a68, 0x3d99, 0xbf3f, 0x411d, 0xb2c7, 0x4346, 0x1ea1, 0xbf22, 0x1c0c, 0x16ab, 0x1a2e, 0xb0c8, 0x4253, 0x432b, 0xa1c3, 0xbadb, 0x408b, 0x4339, 0x064c, 0x43be, 0x421d, 0x439c, 0x4042, 0x4197, 0x11d2, 0xbf18, 0xbad3, 0x1e4a, 0x4239, 0x2672, 0xb267, 0xa0f4, 0x4240, 0xba29, 0x42a2, 0x4225, 0xbf73, 0x4263, 0x0e6d, 0x43a7, 0x1be0, 0x4460, 0xba4b, 0x42ad, 0x406a, 0x4002, 0x40c9, 0x41e8, 0xb237, 0x4281, 0x4378, 0x4473, 0xb297, 0x3829, 0xbf60, 0x4110, 0x426d, 0x1b20, 0x1ade, 0xbac1, 0x4227, 0xb216, 0x41ed, 0x430d, 0xbf46, 0x4200, 0xb22b, 0x46ed, 0x1496, 0x180b, 0x3e53, 0x4601, 0x4404, 0xbf37, 0x42a3, 0x24ec, 0x3704, 0x2ead, 0xb041, 0xb239, 0xb0f5, 0x418b, 0xbaef, 0xba37, 0x413c, 0x4076, 0xbada, 0x35ef, 0x40cf, 0xbf8b, 0x4379, 0x3548, 0x1edb, 0x41ea, 0x1d56, 0xba6a, 0xba46, 0xb0ac, 0x33d6, 0x40e1, 0x21ce, 0xbf8f, 0xba17, 0xb257, 0xba54, 0xba5f, 0x26c7, 0x40fc, 0x4283, 0xbf04, 0xb280, 0x43de, 0x1eed, 0x3a39, 0x401d, 0x4310, 0xba5b, 0x4656, 0x404c, 0xbfd9, 0x465b, 0xba4b, 0x4656, 0x438a, 0x4347, 0xa967, 0x4049, 0x41c0, 0x1930, 0xb040, 0x433f, 0x1c50, 0xbf54, 0x41bf, 0x4027, 0xbf03, 0xb0dd, 0x42b3, 0x435a, 0xb292, 0x404c, 0xb290, 0xba28, 0x400b, 0x4092, 0x406e, 0x4318, 0x442c, 0x0c48, 0xa5ad, 0x4148, 0x424d, 0x4279, 0x40b1, 0x4331, 0xb27b, 0xb049, 0x1cae, 0x45ab, 0x4319, 0xb06e, 0x40b7, 0xbf1c, 0x4484, 0x1ce2, 0x439b, 0x41d4, 0xb28f, 0x4038, 0x45c1, 0xaf03, 0xa786, 0x409e, 0x0b05, 0x1295, 0x3ba6, 0xa9df, 0x42df, 0xba11, 0xb2d0, 0xba4e, 0xbf38, 0x43a1, 0x3a98, 0x4163, 0x41e4, 0x40e0, 0x4619, 0x4259, 0x1b6a, 0xbf81, 0x396c, 0x3ebc, 0xb2e6, 0xad08, 0x4350, 0x13f0, 0x1898, 0x1f15, 0xbfaa, 0x4335, 0x321b, 0x4189, 0x40b1, 0x0a31, 0xbfa3, 0x3fc0, 0x2d58, 0x43ad, 0xa85a, 0x40af, 0x423d, 0x02b8, 0x1dfe, 0xbac5, 0x4342, 0x40e1, 0x41fb, 0xab7c, 0x28a1, 0x4219, 0x098d, 0x4085, 0xb20e, 0x2e27, 0x0b99, 0x4403, 0xb2c0, 0x405d, 0xb0bc, 0xb0cb, 0xbf98, 0x42c0, 0xb207, 0xb250, 0x40d5, 0x19ff, 0xb2bb, 0x42cc, 0x42d6, 0xae29, 0xba31, 0xbfdd, 0x3003, 0x0f3d, 0xb091, 0xba54, 0x41ed, 0x40e3, 0x20e4, 0xb012, 0xb2c2, 0x409d, 0xb28e, 0x080e, 0xaedd, 0x06d2, 0xb226, 0x4160, 0x4059, 0xb229, 0x040c, 0x4223, 0x422f, 0x3772, 0xb2b6, 0x417e, 0xb21b, 0xbacb, 0xb2d4, 0x2fb1, 0x45e2, 0xbfc9, 0x41f2, 0x42c9, 0x0d4f, 0x4300, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x82916b79, 0x515ecb9e, 0x9eeb80a7, 0x4448bdb6, 0x9fbc4379, 0x7f419a33, 0x42be67d1, 0x712a0dae, 0x4382a2e5, 0x7d2ed79f, 0xe8c9993b, 0x9fd1abb1, 0xed4e4991, 0xf7cf8b5c, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0xa000b5cd, 0x00000000, 0x00000008, 0x00000000, 0x00000000, 0x00000000, 0x0000b55a, 0x00000000, 0xf7cf8df7, 0xd1933276, 0xe8c9993b, 0xf7cf8df7, 0xd1942582, 0xf7cf8df0, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x45f6, 0xa7e3, 0x4240, 0x4552, 0x19ea, 0x1ee9, 0x4223, 0xb029, 0x03fc, 0x41ff, 0x4699, 0x4543, 0x40eb, 0x1189, 0xb2eb, 0x2ed0, 0x46c5, 0x152e, 0xbf8b, 0x42ee, 0x454e, 0x2a1e, 0x3168, 0xa0a7, 0x1fa6, 0x415b, 0xb286, 0x3cff, 0x35a2, 0x405c, 0x4228, 0x40e1, 0x43e2, 0x40c6, 0x37cc, 0xbfda, 0x4320, 0xba70, 0xb067, 0x43f0, 0xb2b7, 0x4213, 0x4247, 0xb229, 0xb2f5, 0x42e7, 0x40cd, 0x1bdf, 0x43f8, 0x3887, 0xa60a, 0xba2e, 0x1b0d, 0x1e22, 0xba51, 0x437e, 0x45e4, 0x466e, 0xb017, 0x29df, 0x4057, 0x19e9, 0xb0cd, 0xbf34, 0x4466, 0x1283, 0xb0de, 0x42ff, 0x1c85, 0x4430, 0x42b0, 0xba35, 0x4274, 0x41e6, 0x4331, 0x42dd, 0x40ec, 0xbf43, 0xb23f, 0x405d, 0xba5f, 0x4563, 0x432c, 0xba0d, 0x43a2, 0x2a71, 0x439d, 0xb26e, 0x4685, 0xb086, 0xb217, 0xb264, 0x4412, 0xbff0, 0x4388, 0x18ff, 0x43c7, 0x407f, 0x4174, 0x410b, 0x4324, 0x469c, 0x410a, 0x4302, 0xbaff, 0xbfa5, 0x4371, 0xb26b, 0xb079, 0x42f6, 0xbf8f, 0xb251, 0x411a, 0x42e3, 0x21bb, 0xba73, 0xb2c3, 0x3374, 0x0cd7, 0x4128, 0x1ca9, 0x40fd, 0xae9f, 0x40fc, 0x1df9, 0x44a8, 0x1d1d, 0x4061, 0xb08f, 0x40f0, 0x4375, 0xb244, 0x45f3, 0x404a, 0x406e, 0xbf5f, 0x4146, 0x41ff, 0x4065, 0x4071, 0x4391, 0x0932, 0x42c2, 0xba62, 0x4146, 0x402b, 0x424f, 0x4263, 0xba47, 0xb0df, 0xbf79, 0x419b, 0x40d5, 0xbfa0, 0x4157, 0xb2e6, 0x42f3, 0x4216, 0x4174, 0x40b0, 0x3c8e, 0xa763, 0x0bb7, 0xba04, 0x45de, 0x4598, 0x1c37, 0xb250, 0x414b, 0xb299, 0xba17, 0xb0c3, 0xa7f5, 0x1e74, 0xbfc7, 0xb23b, 0x42ad, 0x4235, 0xb270, 0x11c2, 0x436f, 0x42ac, 0x45f4, 0x1221, 0xb26e, 0x46a4, 0xb283, 0x2610, 0xb05b, 0x098e, 0x1946, 0x439c, 0xa1ce, 0x4057, 0x16ac, 0x4357, 0xb2d1, 0x4231, 0x4300, 0xae54, 0xbf70, 0x40f3, 0xb2e8, 0xbf2d, 0x1b23, 0xba0a, 0xb0fc, 0x280d, 0x423d, 0xba50, 0xb29c, 0x416e, 0x41f3, 0x42f3, 0xb06a, 0xabca, 0x24ba, 0x1ba6, 0xb2ef, 0x41ec, 0xb22a, 0x1b17, 0xaa10, 0x41b2, 0xba10, 0x1fa7, 0x3414, 0xbacb, 0x4439, 0xbf90, 0xae5e, 0x41a1, 0xbfd4, 0xba4d, 0xb271, 0x0305, 0x422a, 0x411a, 0x4194, 0xb20e, 0xb29b, 0x41b3, 0x46a1, 0xb2c4, 0x3014, 0x40dc, 0x36e7, 0x432d, 0x09b3, 0x430c, 0xb01f, 0x12d7, 0x40aa, 0xb2d9, 0x055b, 0xbaee, 0x42d2, 0x43b4, 0x1875, 0xbf51, 0xa093, 0x226a, 0x284d, 0xb29f, 0x44a4, 0x3de8, 0x35aa, 0x465e, 0xb266, 0xb0a1, 0xb2f5, 0x437f, 0x4250, 0xa47f, 0xbfb4, 0x273b, 0x1d3e, 0x15c1, 0x1c60, 0x0830, 0x0bcd, 0x4255, 0x44db, 0xbf1b, 0x434d, 0x2582, 0xb094, 0x1fa4, 0xa542, 0x05e2, 0xb29f, 0xba52, 0xb00f, 0xb2f3, 0x4285, 0x1a85, 0x4636, 0xb09e, 0x2a91, 0xbf90, 0xb28f, 0xbf80, 0xb234, 0x18fc, 0xbf2d, 0xb07c, 0x41bf, 0x425d, 0x419e, 0x4327, 0x4026, 0x4241, 0xa452, 0x0cbc, 0x41f0, 0x46f4, 0x42d6, 0x4464, 0x1f9f, 0xba60, 0xbfe4, 0x42a5, 0x40a2, 0xb2bf, 0x405f, 0xb2f5, 0x4333, 0x43af, 0x435d, 0x191d, 0x4002, 0xbf68, 0x42a3, 0xb0ee, 0x26e9, 0xbfcc, 0x3226, 0x4342, 0x4280, 0x1715, 0x19cc, 0xb2ed, 0x4407, 0x2653, 0xbaf6, 0xb2c4, 0x43c4, 0xb2b1, 0x4131, 0x1090, 0x41e3, 0x4217, 0xbf90, 0xba7f, 0x401d, 0xb268, 0x408f, 0x4287, 0x4257, 0xbfce, 0x4185, 0x1e28, 0x4265, 0x45ec, 0x43da, 0x422e, 0x2405, 0x00ae, 0xb269, 0xb2c5, 0xbfb9, 0x3700, 0xb067, 0x41ab, 0x408c, 0x4053, 0x4023, 0xbfc6, 0x1c83, 0x4078, 0xace2, 0xbf00, 0x390d, 0x4247, 0x4343, 0x4291, 0x44e4, 0x18eb, 0x4214, 0x4044, 0xb0ae, 0xbacd, 0x4171, 0xb0b7, 0x42f1, 0x4045, 0xb277, 0x40dc, 0x19a8, 0xbf91, 0x09fd, 0xb21c, 0xba3e, 0x41a5, 0xbad5, 0x1abb, 0xbfbd, 0x3179, 0x22a8, 0x1555, 0x31a9, 0xbfe0, 0x420d, 0xbf7d, 0x428d, 0xbaea, 0x1e6b, 0x2b66, 0xb05d, 0x1ac3, 0x4176, 0x41c6, 0x41c5, 0x421b, 0xb2ec, 0x41e9, 0x2cf0, 0x4386, 0x4108, 0xbf1f, 0x4066, 0x181d, 0xb2a1, 0x43f5, 0x43a1, 0x00a7, 0x42ff, 0x1f65, 0x43aa, 0xb24d, 0x0ab5, 0x42ee, 0x2d96, 0x456b, 0xbaec, 0x3863, 0xbaf2, 0xbafd, 0x1f60, 0xb2f5, 0x4144, 0x2be1, 0xaca3, 0xb2a9, 0xbf96, 0xaed3, 0xac10, 0x404d, 0x4040, 0xbac2, 0x228d, 0x4144, 0x43d7, 0x45e5, 0x419f, 0x1d89, 0x4463, 0x311c, 0xbf4a, 0x1cd6, 0x4618, 0xb2db, 0x43fa, 0x40ae, 0xb0af, 0x32a0, 0xb2fe, 0xaef6, 0x1de3, 0x4302, 0x42a0, 0xabb6, 0x411e, 0x439b, 0x464b, 0x406f, 0x2a8b, 0x38cd, 0xbfc2, 0x422d, 0xb068, 0x4327, 0x4328, 0x4109, 0xbf4b, 0x0a21, 0x44b3, 0x4089, 0x40bd, 0x18ca, 0x4112, 0xba1f, 0x0bf6, 0x31e8, 0x41bf, 0x45c1, 0x4354, 0x2f9f, 0x0993, 0xba25, 0x4010, 0x1a4b, 0x43ab, 0xbf8e, 0xbacb, 0xa49a, 0x42f5, 0xb029, 0xbfc8, 0x4376, 0x40af, 0x464f, 0x01c2, 0x43da, 0x4251, 0x40a8, 0x422c, 0x438e, 0xbacf, 0x0b64, 0x437d, 0xa181, 0x40ca, 0x411f, 0x4677, 0x4353, 0x2f02, 0x43f2, 0x05fe, 0xb2e2, 0xb083, 0xbf3a, 0x303c, 0xb007, 0x24d0, 0xb27d, 0xb216, 0x3049, 0xba34, 0x4399, 0xb27a, 0x2f9c, 0xb006, 0x223e, 0x0844, 0x4044, 0x436a, 0xb276, 0xb2b6, 0xba1e, 0xbf52, 0x425c, 0xbaee, 0x42d5, 0x417d, 0x4187, 0xbaf2, 0x4238, 0x402b, 0x4206, 0x41cd, 0x05b2, 0x402d, 0x408b, 0x4681, 0x419d, 0xb0b5, 0xb28d, 0xbfe8, 0x27bb, 0x25c4, 0x02ee, 0x40e2, 0x435b, 0x40ad, 0x43ea, 0x1fd2, 0x4065, 0x1d5a, 0x4347, 0xb2d5, 0x436f, 0xae55, 0xba71, 0x0f2c, 0x4187, 0xbf73, 0xb20a, 0x44eb, 0xa5dd, 0x40ed, 0xba53, 0xba13, 0x4191, 0x42e5, 0x4193, 0x4168, 0x40ab, 0x4476, 0x4135, 0x41c3, 0x4145, 0x353e, 0xbfd0, 0xb220, 0xa3e1, 0x42a1, 0xb2d3, 0xb2af, 0x433d, 0x412a, 0x28d7, 0x411d, 0x42bc, 0x4155, 0xbfd5, 0xac98, 0x4353, 0x4142, 0x4435, 0x03bd, 0x402c, 0xb056, 0xb08b, 0x460f, 0xb284, 0x45e0, 0x4240, 0x265c, 0xba4a, 0xa052, 0xa53d, 0xbae5, 0xb2a0, 0xbf31, 0x413b, 0x1ecb, 0x4234, 0xb27a, 0xb204, 0xac49, 0x4387, 0xb064, 0xbf80, 0xbfd1, 0xb229, 0x4342, 0x41b5, 0x43f9, 0x43a7, 0x405e, 0x449d, 0x43c1, 0xba4e, 0x4181, 0x40a6, 0x238d, 0x4280, 0x422f, 0x401a, 0xbf70, 0x09df, 0xa4e7, 0x4040, 0x37cc, 0xbfa2, 0x300b, 0x013f, 0x423e, 0x42ab, 0x412a, 0x45b3, 0xb2f0, 0x405b, 0x12ca, 0x40a4, 0xabdf, 0x43ab, 0x4383, 0x43c4, 0xba14, 0x4105, 0xbf05, 0x14da, 0x4294, 0x4065, 0x40f4, 0xb246, 0x4179, 0x1d11, 0xb240, 0x283b, 0xb2df, 0xbf75, 0x1b5a, 0x435a, 0x05db, 0xbaf6, 0x4399, 0x4394, 0x42b3, 0x1fdc, 0x40e4, 0x41ba, 0x432e, 0x438f, 0xb2e0, 0x3f84, 0xb003, 0x137f, 0xb051, 0x419f, 0xba3e, 0xbad0, 0xb208, 0x417b, 0x431f, 0x2355, 0xb25c, 0x266c, 0x43f6, 0x4478, 0xbfa8, 0x42af, 0x4167, 0x2ef4, 0x42b2, 0x411a, 0x4151, 0xba77, 0x4391, 0x27b8, 0xba78, 0x0a0c, 0xb2d7, 0x4595, 0x41e3, 0x40dc, 0x41e5, 0x41ec, 0x1c70, 0xbfe4, 0x41bc, 0x4109, 0xb2fd, 0xb292, 0x0159, 0x240f, 0x4363, 0x4398, 0x4371, 0x1ba9, 0x4088, 0x3e3e, 0x4016, 0x4363, 0xb2c2, 0x05b5, 0xbfb0, 0xba63, 0x24f5, 0xb23c, 0xbf03, 0x4146, 0xb2a1, 0x380d, 0xb267, 0x3bde, 0x4092, 0xb274, 0x251e, 0x421c, 0x0687, 0xba79, 0x059e, 0x429b, 0x45b9, 0x0d1c, 0x45a6, 0x1185, 0x46a3, 0x0230, 0x4128, 0x1b96, 0x3e30, 0xbaca, 0x418f, 0x43d4, 0xbfda, 0xb0e8, 0xbf80, 0x3cf6, 0x4264, 0x4005, 0xa0bc, 0x3a93, 0x1777, 0x454d, 0xbad4, 0x4066, 0x414f, 0x181b, 0x1b26, 0x40d9, 0x42bd, 0xba42, 0x1930, 0x0023, 0x405b, 0xbf3b, 0xad73, 0xb278, 0x4111, 0xbadc, 0x4276, 0xaa86, 0x4310, 0x441a, 0x0ee1, 0x2eb8, 0x4379, 0x4268, 0x43ed, 0xb268, 0x187e, 0x4221, 0x2188, 0xb05a, 0x43e6, 0x43ed, 0x4148, 0xb2cf, 0x4491, 0xbf3f, 0x0b27, 0x1cc0, 0xb236, 0x1a4a, 0x43fe, 0x43c3, 0xba58, 0x4326, 0xb0d9, 0x1a6c, 0xa71b, 0x1461, 0x40cc, 0x41ba, 0x1d96, 0x42aa, 0x19d1, 0x1413, 0xba22, 0xb2a4, 0xba5b, 0x441f, 0x0936, 0x3137, 0x4491, 0xbf76, 0xb0dd, 0xa93b, 0xb0a4, 0xb09d, 0x1677, 0x1dd3, 0x40fa, 0xa6a6, 0x4054, 0x424f, 0xb008, 0x4650, 0x401f, 0xab43, 0x43a7, 0x1a59, 0x4034, 0xbacd, 0xaff8, 0xb062, 0xbfc3, 0x4081, 0xba07, 0x459e, 0x4037, 0x1f32, 0xba28, 0x43b7, 0x0c12, 0x4133, 0x1b17, 0xbf94, 0x4286, 0x4212, 0x41cd, 0x1953, 0x4049, 0xba1c, 0x43cc, 0x1c4e, 0x43ac, 0x432d, 0x4616, 0x45c3, 0x3f22, 0x0c24, 0x2bbe, 0x414e, 0xbf51, 0x41ba, 0x3fed, 0x42b0, 0xb2eb, 0x1a7a, 0x1d81, 0x43d0, 0x4358, 0x1f1a, 0x4362, 0xba60, 0x42a1, 0x40e8, 0x43f5, 0x1c51, 0x4249, 0xba7b, 0xbf27, 0x4169, 0x4296, 0xbf70, 0x4344, 0x4356, 0xbaff, 0x436f, 0x42d8, 0x0bc9, 0xba0d, 0x428e, 0x4038, 0x40f8, 0x1ba6, 0x428e, 0x417a, 0x44b1, 0xb0cc, 0x4328, 0x4187, 0x4090, 0x41d3, 0x458c, 0xba28, 0xbfba, 0x42e5, 0x459b, 0xb27c, 0x42ec, 0x2a3e, 0x408c, 0x3be5, 0x2aec, 0xbf0d, 0xb282, 0xba27, 0x43fa, 0x419b, 0x43c8, 0x1934, 0x445c, 0xbf86, 0xb0e9, 0x4240, 0x4282, 0x044a, 0xb240, 0xaa3d, 0x424a, 0x39f5, 0x0ee1, 0x41ab, 0x1a6e, 0x4324, 0x0e64, 0x41e5, 0xbf8c, 0x4170, 0x19eb, 0x3ff4, 0x4404, 0xb021, 0x435a, 0x4380, 0x46b0, 0xb2ea, 0x24d2, 0x40ed, 0x42d8, 0x40e1, 0x4245, 0x4108, 0x19c7, 0x428d, 0x422e, 0x4244, 0x260e, 0xb045, 0xb060, 0x1ab1, 0x40b1, 0xb0b9, 0x42eb, 0xbf9e, 0x1a35, 0xb24a, 0x4411, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xdb27f4a8, 0x745a69a1, 0xc76c58a7, 0xcb93d578, 0x59551d1e, 0xdd275e29, 0xa41a07ab, 0xcd4584a9, 0x8cd3e811, 0x92d54b8c, 0x7a714058, 0xd6ca1b2d, 0xbb97bd53, 0x4fb916ee, 0x00000000, 0xc00001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00034000, 0x00000000, 0x0382fe19, 0x00000000, 0x0000000e, 0x0000000e, 0xffffff0c, 0xf47d00e9, 0x1f5a553b, 0x7a714058, 0x00000000, 0x00000000, 0x605fe8c0, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4283, 0xa90f, 0x4210, 0x1dab, 0x4172, 0x40c4, 0x40b1, 0x44d5, 0x2a9b, 0xba65, 0x4002, 0x1842, 0x308f, 0xb226, 0x42c5, 0x425c, 0x40fd, 0xb279, 0xba34, 0x4284, 0xbf60, 0x3d5d, 0x420c, 0x401a, 0x0e39, 0xbf79, 0x4395, 0x1ccb, 0x41ae, 0x1ddc, 0x4062, 0x44db, 0x2e06, 0x4389, 0x1d42, 0x1c39, 0x4347, 0xb298, 0xb007, 0x43a7, 0xb22a, 0xba22, 0xb22c, 0xba54, 0x4304, 0xa8d3, 0x42a8, 0x2c0a, 0x40a3, 0x2b03, 0x1b34, 0xb2ef, 0x06d3, 0xbf3a, 0x1d56, 0x1ef0, 0x0fa3, 0xb2a8, 0x1939, 0xbff0, 0x422c, 0x019c, 0x1f8e, 0x42cb, 0xbae4, 0x17be, 0x404e, 0xb2d3, 0x1fa5, 0xb209, 0x42ae, 0x2bb5, 0x424e, 0xb273, 0x4202, 0xba33, 0xb2af, 0x1d35, 0x1a2f, 0x1cf0, 0xb266, 0x2255, 0xbf3f, 0xac6b, 0x42f7, 0xb092, 0x1263, 0x46a9, 0x4099, 0x105a, 0x18b5, 0x34aa, 0x1b91, 0x4129, 0x1449, 0x4096, 0xb27a, 0x437a, 0x40f5, 0xb27d, 0x1628, 0xb242, 0x42da, 0x404b, 0xbfbb, 0xb077, 0x4297, 0x0570, 0x4152, 0x2525, 0x1be3, 0xbfe0, 0x4633, 0x12e2, 0x4547, 0xba67, 0x06a6, 0x40c9, 0x371b, 0xbfb0, 0x4201, 0x0101, 0xba0d, 0xbfd9, 0x4192, 0x2123, 0x4117, 0x428e, 0x4084, 0xba04, 0x417c, 0x4390, 0x41e5, 0x41ad, 0x462e, 0x46da, 0x4268, 0x40b4, 0xb279, 0xa5df, 0xb26a, 0xbf4d, 0x356c, 0x09bf, 0xb06e, 0xb292, 0xb2f2, 0x1897, 0x0d15, 0xb261, 0xb29b, 0x40e6, 0x3e3a, 0x46ba, 0x3342, 0x1d7b, 0x401c, 0xb0ce, 0x177f, 0x430d, 0x0189, 0x43ba, 0x4309, 0xb20f, 0xb228, 0xac9c, 0x086d, 0x13e5, 0xbf61, 0x1fc8, 0x41b5, 0x2273, 0x40ab, 0x4334, 0x43fd, 0xbf6d, 0x4304, 0x4286, 0x4372, 0xb213, 0x4426, 0x02e4, 0x40da, 0xb2bd, 0x441a, 0xbf79, 0x40c6, 0x1e05, 0x1b15, 0x420d, 0x4248, 0xbf60, 0xb093, 0x39aa, 0x40c8, 0x085d, 0x37e8, 0x400a, 0x15dc, 0x218d, 0xb223, 0x1910, 0xb248, 0xb00d, 0x4168, 0x44c2, 0x418d, 0xb0e5, 0xb0b2, 0x33e7, 0xa4e7, 0x43e3, 0x0066, 0x165b, 0x4231, 0xbfa7, 0x2e38, 0xb249, 0x431a, 0x4089, 0xba16, 0xa64d, 0x4181, 0xbfd0, 0x3a3f, 0xb258, 0x44cd, 0xaa78, 0x41d7, 0xa0cb, 0x4279, 0x4061, 0x41de, 0xb2bc, 0x4125, 0xae63, 0x41bc, 0x418f, 0xba71, 0xa347, 0xbf4f, 0x4666, 0xb204, 0x4202, 0x42db, 0xb006, 0xbf7b, 0x0978, 0x1c50, 0x1bc7, 0x1818, 0x053b, 0xbfb5, 0xa485, 0x4134, 0x27f9, 0x2199, 0xae03, 0x4126, 0x46e5, 0x1a12, 0x1ee6, 0xb22d, 0x2692, 0x4361, 0x411d, 0x4064, 0x4091, 0x3d71, 0x3936, 0xb2a0, 0x34de, 0xbfdf, 0x4552, 0x1bb7, 0x1a4a, 0x1264, 0x40d4, 0x1d83, 0x4227, 0xbfe0, 0x0a4d, 0x429b, 0x3aa5, 0x4055, 0x43f3, 0x1c83, 0xbf5a, 0x4567, 0x43f6, 0xa01f, 0x1cdc, 0x40bd, 0x41c3, 0x4223, 0x3fea, 0x422b, 0x3d0a, 0x18f7, 0x432d, 0x1e00, 0xbf04, 0xaf5d, 0x1b42, 0xb0e2, 0x46e4, 0xbf09, 0x463a, 0x4355, 0xb0dd, 0x44d2, 0xb24c, 0xba2b, 0xa14c, 0xbfc0, 0xb2b7, 0xba12, 0x43f1, 0xb294, 0x41b6, 0xb247, 0x0688, 0x43e8, 0x42c7, 0x4155, 0x43c4, 0x44fd, 0xa8d2, 0x19a7, 0xb2d5, 0xb035, 0xb2a8, 0x43fd, 0x455d, 0x4314, 0x1f85, 0xbf2c, 0x4287, 0x4154, 0x4204, 0xbf06, 0x4201, 0x408b, 0x40f5, 0x43f3, 0x43b4, 0x4329, 0x42ab, 0x45d0, 0xb093, 0x434c, 0x0f4f, 0x1d8c, 0x402f, 0xba58, 0x444c, 0xb294, 0x41b1, 0x42a7, 0x4177, 0xbfc7, 0x4560, 0xba10, 0xb28a, 0x41f7, 0x40c1, 0xb21b, 0x4068, 0xacc3, 0x41b5, 0x0829, 0x42eb, 0x408f, 0x46f9, 0x1c2f, 0xa7ff, 0xb27e, 0xb2a2, 0x194b, 0x4182, 0x3acc, 0x1527, 0x4143, 0x40f4, 0x3a3c, 0xb2ea, 0x44b5, 0x4076, 0xbf69, 0x34c1, 0x43ce, 0xb299, 0x18d5, 0xbfa0, 0xb24d, 0xbf15, 0x427e, 0xb003, 0x4289, 0xba4b, 0x356d, 0xb2a7, 0xa497, 0xba5a, 0x405e, 0xb249, 0xb2a7, 0x4176, 0x426b, 0xb204, 0xaea9, 0x4110, 0x2479, 0x3502, 0x46ca, 0xbfbc, 0x4321, 0x41f4, 0x4170, 0xba41, 0x1d2c, 0xba4a, 0x11a4, 0x009a, 0x28c5, 0x43d3, 0xbfcf, 0x4034, 0xabeb, 0xaac6, 0xa608, 0x4056, 0x338f, 0x0394, 0xb02f, 0x4274, 0xa5c6, 0x1be8, 0x40fc, 0x4029, 0xb2b5, 0x0227, 0x090b, 0xbacc, 0x1116, 0x2c45, 0x08ee, 0x14fa, 0xbf82, 0x405a, 0x423f, 0xba2a, 0x4052, 0x4200, 0xa671, 0x0efd, 0x42de, 0x1e89, 0xb049, 0xb264, 0x3e26, 0x41d5, 0x4445, 0x32af, 0xb0e8, 0xbafb, 0xb0f7, 0x4093, 0xb034, 0xb2e8, 0x2e69, 0xb259, 0x2479, 0x439d, 0xb2de, 0x1b5b, 0xbf8c, 0x437b, 0xbad1, 0x217d, 0x4093, 0xb2e6, 0x1972, 0xb2e5, 0x1d51, 0x4210, 0x4310, 0xb20c, 0xad06, 0x1879, 0x40b7, 0x4077, 0xbfbd, 0x1fa9, 0xb00b, 0x40bf, 0xbaf2, 0xa690, 0x4298, 0xa00d, 0x2c47, 0x1e51, 0x41c2, 0x42dc, 0xbf6e, 0xb0c0, 0x2ddd, 0x43b9, 0xba69, 0xa0f1, 0x1c40, 0xbfe0, 0xba04, 0x4263, 0xba26, 0xa397, 0x1731, 0x0de3, 0x40bc, 0xa1e7, 0x411c, 0x4005, 0x417a, 0xb043, 0x43d3, 0x42a1, 0x4053, 0x42e8, 0xb0eb, 0xbf16, 0x1c21, 0x4442, 0xbf80, 0x4164, 0x401e, 0x41e5, 0x18d4, 0xb241, 0x45ed, 0x4037, 0xba05, 0x415a, 0xbf25, 0xa2a9, 0x0e62, 0x26af, 0x182b, 0x4134, 0xa7ee, 0x4212, 0x42bd, 0x4243, 0x402a, 0x4182, 0xbf2e, 0x16a0, 0xb213, 0x42b4, 0xb2fe, 0xb20f, 0xb23c, 0x0bb8, 0x43da, 0x4265, 0x42cb, 0xb02a, 0xb226, 0x43db, 0x421e, 0x41e7, 0x1f20, 0x22ca, 0x41d9, 0x1e79, 0xb00b, 0x41cf, 0x401b, 0x3174, 0xb0f4, 0x404a, 0xbf03, 0x4674, 0xb243, 0x43cb, 0x431a, 0x43cd, 0xb0af, 0x0481, 0xaca0, 0x419a, 0x3433, 0x43dc, 0x4197, 0x3d98, 0x19f0, 0x21f1, 0x42d2, 0xba2f, 0xa60c, 0x4337, 0x4102, 0x2851, 0xba4b, 0x44db, 0x4432, 0x422c, 0x4339, 0x4002, 0x1e4d, 0xbf7d, 0x434f, 0xb20d, 0xb00e, 0x3545, 0x41a7, 0x40bb, 0x03ab, 0xba58, 0x2faa, 0x45eb, 0x0643, 0x4098, 0xb2ff, 0xb038, 0xa185, 0x3d65, 0x062b, 0xb010, 0xb20e, 0x1fab, 0x417d, 0x46bc, 0x080c, 0x1913, 0x4146, 0xbfd3, 0x42af, 0x064d, 0xbaeb, 0xba5c, 0x2be1, 0xbf90, 0x404a, 0x0e2c, 0xba56, 0x434e, 0x4320, 0x4180, 0x4658, 0x42d3, 0x4281, 0x1986, 0x424a, 0x4281, 0xb28f, 0x42db, 0xbf6b, 0x41da, 0x42d3, 0xbaed, 0x43de, 0x1d8d, 0x04df, 0x1e09, 0x421f, 0xb21a, 0xb2eb, 0x41cf, 0xbff0, 0x44bb, 0x1ce0, 0x185f, 0x2ceb, 0x42a9, 0x1d0f, 0x4000, 0x1c0c, 0x4127, 0xba21, 0xbac3, 0x3822, 0x15bb, 0x3d53, 0xbf47, 0xb064, 0xb075, 0xb258, 0xba4e, 0x4314, 0x466c, 0x41d2, 0x1a9c, 0xb056, 0x413e, 0x4089, 0xb0d0, 0x40c2, 0x4202, 0x421f, 0x1ca7, 0xb2bc, 0x3348, 0xb280, 0xbf6a, 0xb0ae, 0x016e, 0xb0a7, 0x4672, 0x4147, 0x3098, 0xa236, 0xb00f, 0xbf1a, 0x4330, 0x42ec, 0xbf00, 0xba37, 0xb2b5, 0x4182, 0x4075, 0xb038, 0x413e, 0xba59, 0x45eb, 0xba2b, 0xbaf7, 0x44b1, 0x1dd1, 0x417b, 0x1a2c, 0x40d5, 0x438a, 0xb03a, 0x1d73, 0xb21e, 0xb25d, 0xb27f, 0xbf1e, 0x430a, 0xb0ec, 0x4159, 0x2733, 0x4247, 0x424a, 0xbf4a, 0xb263, 0x1b5f, 0x40ae, 0xb2fe, 0xbf76, 0x407d, 0xb28a, 0x1f72, 0x1e7c, 0x463c, 0x42c2, 0x40e7, 0x4166, 0x4380, 0x1603, 0x1b69, 0xb212, 0x1739, 0x4377, 0x3103, 0xbf49, 0x40ac, 0x278e, 0xb07d, 0x45c1, 0x1ef5, 0xbff0, 0xb2a0, 0x4379, 0x1956, 0xb233, 0xb26d, 0xbfdf, 0x463b, 0x4096, 0xaeb4, 0x17d4, 0x4663, 0xba6b, 0x4640, 0x3004, 0x404c, 0x46aa, 0xbaed, 0x3318, 0x44e5, 0x1d45, 0x1c9c, 0x1883, 0x425e, 0x0c46, 0xba40, 0xa97f, 0xbff0, 0x0b6c, 0x425e, 0x4207, 0x420c, 0x4257, 0xbf86, 0xabef, 0x40d0, 0x4050, 0x43bf, 0x38f6, 0xb240, 0x3692, 0xbf9c, 0x4288, 0x0c04, 0x41a9, 0xac55, 0xb0b2, 0x414c, 0x1b78, 0xb003, 0x3531, 0xb280, 0x0064, 0x4437, 0x1c72, 0x1f4c, 0xbf1c, 0x435e, 0x40fd, 0x3a0e, 0xbfa0, 0xba1b, 0x4060, 0xba2e, 0x3a97, 0xbaf7, 0x39bd, 0x4264, 0x4472, 0x4328, 0xbf49, 0x1acf, 0x42f6, 0x42b4, 0xb294, 0xb272, 0xb090, 0x4654, 0x4017, 0x467f, 0x468a, 0xace4, 0x44ed, 0x4557, 0x466a, 0x1257, 0x4153, 0xb05e, 0x4121, 0x41b0, 0x4393, 0x410b, 0xbaf3, 0x4556, 0x4242, 0x1f43, 0xbf1d, 0x42c3, 0xb240, 0x4038, 0x404b, 0x43e7, 0xb0fa, 0x420a, 0xbf9d, 0x4323, 0xb0a3, 0x4340, 0x42e9, 0x0052, 0x1cc4, 0x4305, 0x020b, 0xb09d, 0x411d, 0x2c14, 0xb0ee, 0x416c, 0x38f9, 0x4357, 0xb2b9, 0xb21f, 0x4152, 0xbfa4, 0xa537, 0x1f40, 0x0f0b, 0xa47e, 0x4040, 0xbade, 0x401b, 0xb006, 0x0394, 0x2d32, 0xbfba, 0x438b, 0x42b3, 0xb2b0, 0xa573, 0x02fc, 0x1ac5, 0x28b6, 0x43c8, 0x2d0f, 0x3b61, 0x423d, 0x262f, 0x411e, 0x43c5, 0x435d, 0x03e2, 0xae57, 0x1b6e, 0x4238, 0x4110, 0xbf31, 0x2052, 0x38bb, 0x454c, 0x2b69, 0x4009, 0x0ac4, 0x0dad, 0x41f9, 0x1f6e, 0x414d, 0x30e6, 0xb2ea, 0xbfa0, 0x0fbf, 0x42e4, 0xbfe0, 0x400e, 0x061f, 0x00c8, 0xba25, 0x427c, 0x38db, 0xbf5a, 0x44bd, 0x42c9, 0x417e, 0x032e, 0x4205, 0x2143, 0xaf50, 0xbfb2, 0x36cb, 0xb2f9, 0x402c, 0x427a, 0x4350, 0x4315, 0xb0a9, 0x402c, 0x0802, 0x3632, 0x4495, 0x4179, 0x1c57, 0x1bd3, 0xbfd8, 0xa483, 0x3213, 0x4113, 0x4341, 0xb2de, 0xba5c, 0x4246, 0xba0f, 0x0363, 0x3661, 0x41b0, 0xb0ae, 0x4138, 0x4143, 0xba45, 0xb2aa, 0xba01, 0x1824, 0xb049, 0x0a3a, 0x4096, 0xb0ea, 0x274a, 0x43eb, 0xb274, 0xbfde, 0x1aea, 0x417b, 0x0b2f, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd2fc5a9a, 0xcbe6f391, 0xc5271121, 0x66c4d03a, 0xce34ff10, 0xe796f79c, 0x37623fb3, 0x4d059abb, 0xf938c94f, 0x8d0659ac, 0xc525e62e, 0xad6bb1ea, 0xe561afeb, 0x8226b8c9, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0xfffec5fe, 0xfec5feff, 0x00807dfa, 0x0100013a, 0x00000000, 0xfefffec5, 0x00000000, 0x0000004a, 0xf938c94f, 0x0002e84e, 0xec28f749, 0xb5aec7a8, 0x000000c3, 0x69c37868, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1d4f, 0xbfc5, 0xb068, 0x40ba, 0x3da9, 0xaeb3, 0x40b6, 0xa066, 0x13be, 0x4028, 0x4049, 0xba69, 0x4602, 0xbfaf, 0x21d5, 0x1879, 0xa68c, 0xa2f5, 0x41f6, 0x0953, 0xbfd0, 0x2aeb, 0x1d85, 0x3904, 0x38c6, 0xb07e, 0x4079, 0x04e6, 0x1634, 0xbf3c, 0x40cd, 0x4237, 0x41ca, 0xbf72, 0x41ee, 0x1cc9, 0xac16, 0x435a, 0xa938, 0x1f53, 0x3c3e, 0xbf9b, 0xb0b9, 0xb0fa, 0x2372, 0x031a, 0xb011, 0xa5c9, 0x2022, 0xb062, 0x40d8, 0xb223, 0x2f62, 0x46a8, 0xb0f8, 0x4201, 0x4398, 0x19ce, 0x404a, 0xa2b4, 0x2542, 0x41c1, 0xbfc1, 0x42af, 0xb065, 0x20ad, 0x1db5, 0x405a, 0x4008, 0x42bd, 0x0eb3, 0x19c3, 0x414f, 0x4288, 0xb2fe, 0x423d, 0x41ff, 0x4285, 0x4249, 0x42a4, 0xbfe2, 0x4226, 0x1c2a, 0x1323, 0xaf72, 0xb008, 0x42a1, 0x14d3, 0x12cd, 0x4655, 0xb2b7, 0xadf9, 0xba1c, 0x40b0, 0xb21d, 0x4045, 0x1ba6, 0x41d3, 0x41ee, 0xbf6a, 0x416b, 0x3ed9, 0x424d, 0xb238, 0x420e, 0xb22f, 0x4013, 0x3404, 0x41bc, 0x08d8, 0xba7f, 0x407c, 0x1924, 0x424b, 0xb2cc, 0xb232, 0x0896, 0x2a78, 0x43b1, 0x437c, 0xa967, 0x4546, 0xbf8f, 0x4039, 0xa773, 0x40f6, 0xb236, 0xb2e8, 0x43fa, 0x1b34, 0x420e, 0xa787, 0x43a8, 0x2f7a, 0x40b1, 0x46d1, 0x1c90, 0x41ad, 0x4200, 0xba78, 0xbf9f, 0x406d, 0x23c7, 0x0b7f, 0xb2ed, 0x3622, 0x1dd2, 0x425a, 0x43bc, 0xb0ea, 0x4544, 0xa6be, 0x460e, 0x1722, 0x42e9, 0x273e, 0x4366, 0x4691, 0xa936, 0x41a2, 0xbf2e, 0x3231, 0x42a6, 0x43da, 0x2f37, 0x1d1a, 0xb26c, 0xbaff, 0x404c, 0x4132, 0x4076, 0x43fc, 0xb218, 0xba40, 0x4337, 0x385b, 0xb265, 0xb018, 0x462a, 0xba63, 0x41e6, 0xbf59, 0x4289, 0x1fa7, 0x40d7, 0xba59, 0xba5a, 0x0422, 0x07af, 0x428d, 0xb211, 0x4355, 0x431d, 0xbf51, 0x39c4, 0x1ee0, 0x0119, 0x414d, 0xa186, 0x1809, 0x42b4, 0x4065, 0xb2db, 0x43e6, 0xb26c, 0x45b6, 0x1be2, 0xb2ac, 0x42a9, 0x4067, 0xaf6c, 0xb2df, 0xbf1b, 0x4299, 0xb015, 0xb064, 0x44c9, 0x43f8, 0x2914, 0x1c11, 0x2d02, 0x1ebc, 0x43bb, 0x400e, 0x1cb3, 0x4246, 0xb00a, 0xb003, 0xbfb0, 0x1909, 0x3166, 0x30fd, 0xb25f, 0x1c45, 0x1e5e, 0xba68, 0xbf5a, 0x43a4, 0x1a5e, 0x4080, 0x050c, 0xbfc0, 0x14c5, 0x41c0, 0x438e, 0x416e, 0xba10, 0x1a22, 0xb0c6, 0x3aaf, 0x43a3, 0x0345, 0xbf72, 0xb248, 0x44c3, 0x42a4, 0xba19, 0x1e6b, 0x4475, 0x4347, 0xb29a, 0xb285, 0xb2dc, 0x40e7, 0x18bd, 0x4022, 0xb26f, 0xb01e, 0x4329, 0x4592, 0xa298, 0x433b, 0x2c82, 0x413d, 0xbf6e, 0x4554, 0x418b, 0xb0ab, 0x1a76, 0x4015, 0x4050, 0x4613, 0x42a6, 0x435b, 0x467c, 0x0354, 0x3662, 0x093c, 0xb2a8, 0xb2f2, 0x42db, 0x41ff, 0x4030, 0x420c, 0xb063, 0x4580, 0xbf1f, 0x07e2, 0x408b, 0xb0af, 0xb096, 0xba7c, 0xb273, 0x4336, 0x4628, 0x2ede, 0x4249, 0x41fa, 0x3d4c, 0x43b4, 0x40b6, 0x4132, 0xb2c7, 0x138e, 0x2c2a, 0xafa8, 0xb284, 0x1fcc, 0xbad1, 0xba35, 0x40d2, 0xae8d, 0x40b7, 0xb0ef, 0xb245, 0xbf88, 0x1aa3, 0x36d9, 0x4217, 0x0d64, 0x4247, 0xb07b, 0x4093, 0x4398, 0xbf95, 0xbaeb, 0x41db, 0x009d, 0x2ac3, 0xb298, 0x3a13, 0x4361, 0x072c, 0x01ba, 0x1b20, 0x1bc5, 0x404b, 0x0f91, 0x4129, 0xb01a, 0x0794, 0x4001, 0x406a, 0x273d, 0x40a3, 0x4021, 0x441a, 0x418d, 0x4161, 0xbf2f, 0x40d9, 0xbfe0, 0x0fdc, 0x1ae0, 0x37ff, 0x42ed, 0xb0a6, 0x2e0f, 0xa6c5, 0xb0ba, 0x19cc, 0xb21f, 0x220f, 0xb2d3, 0x1dbf, 0x433e, 0x44f4, 0x36f5, 0x422f, 0x1a45, 0x052c, 0x40fd, 0x4285, 0x4294, 0xbae3, 0xba73, 0x42e7, 0xbfab, 0x40f3, 0xba5f, 0x168e, 0x405c, 0x4200, 0x4267, 0xba7d, 0x42ca, 0xb294, 0x42a6, 0x43b9, 0xb2d1, 0x40b4, 0xa04c, 0x18b5, 0x4627, 0x3b0d, 0x4292, 0x460c, 0x4133, 0xb2c5, 0x40f7, 0x42cd, 0xbf21, 0x1c7e, 0x2cb5, 0x2027, 0x4689, 0xb20f, 0xbfd4, 0x1f88, 0xa0aa, 0x1a08, 0xa41d, 0xb2dc, 0x1cae, 0x423e, 0x40e5, 0x43ea, 0x182c, 0xbadd, 0x4349, 0x41b3, 0x1a46, 0x4545, 0xbff0, 0x19b3, 0x3943, 0x0759, 0x36e9, 0x42ff, 0x27f6, 0x0bda, 0x22f4, 0xbf9e, 0x405a, 0x2a35, 0xb2bf, 0x432c, 0x0307, 0xab18, 0x4128, 0x4350, 0x1eae, 0xac20, 0x411d, 0xb2bc, 0x39f4, 0x43b9, 0x4065, 0x1f99, 0x1fc1, 0x4245, 0xa9e3, 0x456e, 0x41ae, 0x4251, 0x4292, 0x4277, 0xbf3a, 0xba5e, 0xb205, 0xb27b, 0x40ca, 0xb233, 0xb037, 0x411f, 0x2fdc, 0xba7a, 0xb0d7, 0x42fa, 0x2a0d, 0x29bf, 0x46ad, 0xb062, 0xa1f5, 0x1f2f, 0x434b, 0x41f3, 0x2f98, 0xbf72, 0x41c9, 0x40da, 0xa73f, 0x4388, 0x42bf, 0xb201, 0x342d, 0x41d8, 0xb2e2, 0xb2a8, 0x2e8d, 0x4096, 0xb280, 0xba42, 0x40fd, 0x418e, 0x4616, 0x1902, 0x4162, 0x1a4b, 0xbfbc, 0xb22c, 0x4108, 0x432f, 0x46bb, 0x46c9, 0xba32, 0x1a44, 0xae82, 0x45c4, 0xba4d, 0x12e4, 0x4273, 0xbaf9, 0xbfd0, 0x43ab, 0x193e, 0x46e9, 0xb09f, 0x116a, 0x4062, 0xbf4b, 0x3063, 0x4186, 0x4022, 0x4413, 0x1e60, 0x4221, 0xb067, 0x42cc, 0x43b5, 0xbaec, 0x2f0a, 0xba21, 0x40f1, 0xb021, 0x42c5, 0x41b5, 0x4408, 0x019a, 0x1bce, 0x4445, 0xbade, 0x3ba5, 0x1b4c, 0x3db9, 0x407c, 0xba29, 0x438a, 0x28d3, 0xbf36, 0x464d, 0x426d, 0x3ef8, 0xb2d1, 0x41b4, 0x1b02, 0xb2b2, 0x18f4, 0x436a, 0x1c88, 0x3013, 0xb2e5, 0xb0fb, 0xb20a, 0x1f07, 0xaaf3, 0xb09b, 0x4150, 0xbf44, 0x425f, 0xb26c, 0xba7d, 0xbfda, 0x1a16, 0x2eb2, 0x4127, 0x4358, 0xbf25, 0x1e43, 0xb26e, 0x46b5, 0xb2f1, 0xb20f, 0x4190, 0xba77, 0xb2d0, 0xb256, 0x192d, 0x1c18, 0x0155, 0x4019, 0xbaf7, 0x424e, 0xba74, 0xb28c, 0xba1e, 0x1a15, 0xb084, 0x3be4, 0xbf44, 0x4244, 0xba70, 0x4109, 0x0237, 0xb2fb, 0x43fb, 0x4113, 0xbae0, 0xa7a3, 0x4552, 0x422f, 0xb291, 0xb263, 0xbfd4, 0xb22b, 0x2765, 0xbaf4, 0xb049, 0x1a38, 0x405e, 0x1b6f, 0xb25e, 0xba65, 0x0639, 0x4581, 0xbaca, 0x41a7, 0x1c50, 0x4030, 0x4272, 0x435e, 0xbf43, 0x4225, 0x180c, 0x42f2, 0x416f, 0x429f, 0xbad2, 0x40bd, 0x448d, 0x417e, 0xb220, 0x10d2, 0xbf06, 0x4068, 0x3614, 0x1307, 0x432a, 0x46d0, 0x4233, 0x41c4, 0x46f3, 0x1c04, 0x40a0, 0xaa36, 0xbaf7, 0xb2e8, 0x425c, 0x42ba, 0x3767, 0xb20e, 0x400b, 0xb087, 0x1886, 0x2c3e, 0x2840, 0xb06a, 0xbf05, 0x43a1, 0xb250, 0x4543, 0x412c, 0x22a1, 0x41c1, 0x4133, 0x1976, 0x404f, 0x425a, 0xaad6, 0x41ed, 0x40a4, 0x4333, 0x4278, 0xb04c, 0x41c5, 0x42c3, 0x336a, 0x4004, 0x411b, 0xb24b, 0x4399, 0x43b7, 0x4098, 0x4180, 0xbf7b, 0x41d0, 0x423c, 0x4301, 0x416c, 0x432b, 0xb2a3, 0xb2a2, 0xb0d9, 0x42df, 0x1237, 0x43d1, 0x437d, 0x41fd, 0x4352, 0xba00, 0xbf6b, 0x1c45, 0x0c59, 0x21e6, 0x1d8a, 0x1a4b, 0x43cf, 0x098f, 0x2f7b, 0x4307, 0xba68, 0x4093, 0xb2cf, 0x4106, 0x19a1, 0x0fd2, 0xbac5, 0x1a26, 0x43da, 0x4279, 0x40dd, 0xbf01, 0xb0b5, 0x41e6, 0x3dc6, 0x406f, 0xb258, 0x19fe, 0x40e1, 0xba16, 0x4267, 0x4093, 0xb2e8, 0x42c0, 0xb282, 0x4083, 0xb269, 0x40f8, 0x3805, 0xbfc3, 0x437a, 0x425d, 0xb0f4, 0x2503, 0x4074, 0xb021, 0xbfaa, 0x42f4, 0x413d, 0x42c7, 0x0bf2, 0xb06c, 0x1949, 0x40e4, 0x40b4, 0x1a5d, 0xbf13, 0xb2a1, 0x230b, 0xb0e9, 0xb294, 0x46b8, 0x2508, 0xb09d, 0xb24f, 0xb0d1, 0xad5e, 0xbfb2, 0x4037, 0xba4b, 0x1f76, 0x1ed9, 0x412a, 0xbf90, 0x4694, 0xb0aa, 0x39bf, 0x4070, 0x4492, 0x0b7e, 0xa95a, 0xb08c, 0x0ebd, 0x4381, 0xa285, 0x3e4b, 0xbf07, 0x4159, 0xae4d, 0x4269, 0xb277, 0x46d3, 0x29f9, 0x45be, 0x4362, 0x1925, 0xba2a, 0x4573, 0x0196, 0xb2ea, 0x434c, 0xb26d, 0xbae2, 0x1e42, 0x411c, 0x416e, 0x4259, 0xb28b, 0xbfe2, 0xbac6, 0x437e, 0x076e, 0x41ba, 0x407a, 0x45f1, 0x4205, 0xb289, 0x1840, 0x4027, 0x4190, 0x408f, 0xb23c, 0x419a, 0x001e, 0xb0b6, 0x4081, 0xba25, 0xa216, 0xb2ff, 0x405c, 0x18ad, 0x426a, 0x4179, 0xb21a, 0xbf8a, 0x4312, 0x44d3, 0xba1d, 0x42a8, 0x283f, 0x4169, 0x13bc, 0xb00d, 0xaac5, 0x403a, 0x439d, 0xaaca, 0x4369, 0x1c07, 0x4256, 0x434e, 0xb054, 0x1634, 0xbfdf, 0x1ab7, 0x3f94, 0x1b06, 0x42f4, 0x407e, 0x058b, 0x409a, 0xba4b, 0x180e, 0xaf82, 0x43f8, 0x161d, 0x44e0, 0xbf14, 0x4215, 0x1acc, 0x401a, 0x402e, 0x41e4, 0x4678, 0x1d6b, 0x1d43, 0x427b, 0x015f, 0xaf31, 0x41de, 0x0fa6, 0xbf45, 0xb2c8, 0x417c, 0x438b, 0x45be, 0x4334, 0xba6f, 0x429a, 0xbf00, 0xb2ad, 0x4342, 0x4154, 0xbf26, 0xb2ad, 0x4347, 0x16db, 0x4423, 0x40ac, 0x4384, 0xb270, 0xb005, 0x42b8, 0x4305, 0xba31, 0x1d89, 0x2ca0, 0xb21f, 0xba55, 0xba38, 0xb26a, 0x30c9, 0xb20e, 0xb224, 0x4069, 0x431a, 0x43fa, 0xa8e2, 0x41a8, 0x40f1, 0x4253, 0x1eae, 0xbf0b, 0x405d, 0x3e93, 0x43ec, 0x45a3, 0xb267, 0x1c8a, 0x19fc, 0x417c, 0xb214, 0x4350, 0x4083, 0x2458, 0x42cc, 0x40f4, 0x40f0, 0x31dd, 0x43b0, 0xb2bf, 0x419f, 0xb2bb, 0x41f0, 0xbfbd, 0x42bd, 0x1c0f, 0xb06e, 0x4085, 0x43ea, 0xb2c7, 0xb0c4, 0x428a, 0x4294, 0xb2ea, 0x44d1, 0x4374, 0x4361, 0x41bf, 0xa12e, 0x02fa, 0xbac8, 0xbf15, 0x4004, 0x3b3a, 0x05be, 0x41c9, 0x40ab, 0xbadf, 0x3b7c, 0xb075, 0x2c0e, 0x02fb, 0x18b4, 0x19f6, 0x43eb, 0xb254, 0x431d, 0x40e1, 0x40f1, 0x0f2a, 0x4248, 0xbf95, 0xa360, 0x4095, 0x1a15, 0x4387, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x14152dbc, 0xac42cb3d, 0x80d39deb, 0xf842bf41, 0xc80af0cb, 0xb2924a15, 0xe20c7a41, 0xd0c9f409, 0xfd5e5b81, 0x4d1dced7, 0x6694566b, 0x954b09bd, 0x823d8b85, 0xd32e765b, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0xffffe7a0, 0x00001860, 0x0000000f, 0x00001958, 0x00000000, 0x0000186f, 0xffc00000, 0x00000000, 0x00000000, 0x669456ff, 0x6694566b, 0xcd28acd6, 0x00000000, 0x000060e9, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4379, 0x2480, 0x4389, 0x4462, 0xb2b1, 0xb229, 0x4064, 0xb2ce, 0xbf8a, 0xb233, 0x4052, 0xbafd, 0x40a1, 0x047d, 0xb2a8, 0xbac2, 0x43fc, 0x41d7, 0xb0c2, 0xba50, 0x41ac, 0x24ec, 0x1cde, 0x4678, 0xb0e8, 0x46f0, 0x438c, 0x4034, 0x3f2b, 0xa9c5, 0xa5d8, 0x1472, 0xbfc6, 0xb2f0, 0x223e, 0x0dc1, 0x408e, 0x4220, 0x417f, 0xa4bb, 0x4347, 0xa2fa, 0xbf8a, 0x1a45, 0x0d2e, 0xb2ed, 0xbaf2, 0x41cf, 0x437f, 0xb077, 0xb271, 0xb256, 0x4138, 0x0a13, 0x439d, 0x275a, 0xb0ad, 0x4094, 0x4133, 0xbfbc, 0xb0d7, 0x290e, 0x43ec, 0x41df, 0x259f, 0x402a, 0x42c6, 0x4148, 0xa7d3, 0x2e2c, 0xb2c3, 0xb223, 0x439b, 0x4209, 0x1e5b, 0xb213, 0x421f, 0x0902, 0xb2b4, 0xba54, 0x4340, 0x1988, 0xbfcb, 0x43b0, 0xba07, 0x436e, 0x41b2, 0xb0ff, 0x43f6, 0x42a5, 0x29d5, 0x4325, 0x22f7, 0x403b, 0x4363, 0x1a03, 0x08bc, 0x1a8e, 0x458a, 0x1328, 0xba5c, 0x315f, 0x40af, 0xbadc, 0x1b66, 0x1bfa, 0x1a5d, 0xb06a, 0x3d85, 0xb077, 0x434c, 0x40c6, 0xbf59, 0x40ec, 0x40a9, 0x4491, 0x4274, 0xba68, 0x1b59, 0x2148, 0xba32, 0x288d, 0x41ba, 0x2e3b, 0x4561, 0xbf5a, 0x42e1, 0x445d, 0x4181, 0x1f97, 0x436c, 0xb005, 0xba1d, 0xab7e, 0x38a1, 0x3f8a, 0x4139, 0x463c, 0xb047, 0xbafd, 0x43c1, 0x40a5, 0x429c, 0xba1c, 0xad94, 0x420a, 0xb275, 0xba3e, 0x447c, 0x1b00, 0xba5e, 0xb282, 0xbf78, 0xb226, 0xb293, 0x1e34, 0x1cf0, 0xae4a, 0xbfa7, 0x40d6, 0xba23, 0x46a0, 0x2172, 0x44eb, 0xbf2f, 0x4095, 0x42cb, 0x2e23, 0x312c, 0xb2e8, 0x1a06, 0x431b, 0xb2b4, 0x42ae, 0x424d, 0xb2ec, 0x1bbd, 0x45f2, 0x137e, 0x204b, 0x18e0, 0x40be, 0xb2c0, 0x25a8, 0x4319, 0xb277, 0x2318, 0x4181, 0xb29e, 0x43a4, 0x1b2b, 0xa73b, 0xbfc7, 0x074b, 0x2a98, 0xb07d, 0xba4b, 0xb256, 0x4088, 0xa8c4, 0xa053, 0x4448, 0x41ca, 0x42f1, 0xb239, 0xbf2d, 0x40a8, 0x3a69, 0x2ffe, 0x3bec, 0x23ee, 0x43f1, 0x2309, 0x43c7, 0x431c, 0x42a9, 0xb29d, 0x41dc, 0x39ff, 0x412f, 0xb2c0, 0xba55, 0xa7bf, 0x41cc, 0xbad7, 0x4104, 0x018a, 0xb2bf, 0x43c6, 0x4170, 0xb28f, 0x41f2, 0x409b, 0x4372, 0xb007, 0xbf2e, 0x01f0, 0xba38, 0x409d, 0xb0ad, 0x4363, 0xb042, 0x2a86, 0x4140, 0x415f, 0xbfb8, 0x41e5, 0x28f2, 0x4035, 0x42ce, 0xb0ae, 0x43d6, 0x4357, 0x44a9, 0x40a8, 0x4117, 0x433b, 0x4213, 0x3385, 0x09e7, 0xbf80, 0xb063, 0xaa00, 0xb013, 0xbf90, 0x409a, 0x4007, 0xbf04, 0x4060, 0x4377, 0xb0f3, 0xb0ab, 0x4279, 0x1a56, 0xba2b, 0x1931, 0x434c, 0xb090, 0x0500, 0x1b0f, 0x1a85, 0x4681, 0x3e3a, 0x43ce, 0xb2fa, 0x430b, 0x4043, 0xbfb0, 0x3a7d, 0xbf76, 0xb009, 0xbace, 0xbf70, 0x403a, 0x438c, 0x41fd, 0x433c, 0x4202, 0x2d3d, 0x4277, 0x428e, 0x4156, 0xb2d2, 0xbf68, 0x3ab5, 0x409c, 0x1f03, 0x1d6d, 0x42f8, 0x0aaf, 0x2d64, 0x2022, 0xbf12, 0x137d, 0xba6f, 0x45a2, 0xa7a7, 0x40f5, 0x45d3, 0xb2e4, 0x3f2e, 0xba5e, 0x1df2, 0xafc9, 0x42be, 0x4683, 0x4001, 0xba00, 0xb2ae, 0xba1c, 0x089d, 0xb052, 0x1bb8, 0x198b, 0x2162, 0x1f49, 0x3d1b, 0x46fa, 0xbf68, 0x4018, 0xa1e0, 0x40fa, 0x1bdd, 0xbf7c, 0x1a6b, 0x435a, 0xb240, 0x434e, 0x2af4, 0x438b, 0x40f3, 0x4049, 0x43df, 0xb231, 0x0e09, 0x434a, 0x1a40, 0x4079, 0x34eb, 0x41b5, 0x4033, 0x431b, 0x28b1, 0x4575, 0xb022, 0x4221, 0x41b5, 0xbf05, 0x1e57, 0x41d7, 0xb26b, 0x40e3, 0x18d7, 0x40c4, 0x4204, 0xb249, 0x1c5a, 0xbfa4, 0x4063, 0x36b9, 0x4414, 0x42ca, 0x1c8b, 0x42d0, 0x458c, 0xb04f, 0xbf79, 0x4369, 0x1d45, 0x383b, 0x41af, 0x132e, 0x1a3c, 0x0552, 0x467b, 0xba03, 0xb09d, 0xb0ed, 0x43a8, 0x2abd, 0x181a, 0x43c3, 0xb2f9, 0xaa2e, 0x38d6, 0xba77, 0x427b, 0x41f9, 0xbf21, 0x42e7, 0xb282, 0x4224, 0xb2e2, 0x1fab, 0x40f1, 0xbfd0, 0xbf84, 0x01fc, 0x42d2, 0x4379, 0xb000, 0x4246, 0xbfbd, 0xba44, 0x42ec, 0x43e8, 0x0d71, 0x238f, 0xbfd8, 0x46d4, 0x4068, 0x430e, 0xa79c, 0xa11e, 0x4375, 0x0698, 0xb241, 0x4343, 0xbaf7, 0x1af4, 0x4399, 0x4227, 0x43c3, 0xb2e3, 0xb08b, 0x39ed, 0xb07b, 0xb2d1, 0x4162, 0x425e, 0x0066, 0x4046, 0x4025, 0x0499, 0xbf48, 0x430d, 0xba16, 0x215d, 0x2992, 0x4078, 0xb226, 0x126c, 0x406e, 0x43cc, 0xb23f, 0x43db, 0xbf37, 0x2e15, 0xbaec, 0x4356, 0x3c9d, 0x4393, 0x245d, 0x411b, 0xbf00, 0x2629, 0xb21e, 0x423e, 0x27ca, 0x4001, 0x46e2, 0x4191, 0x439b, 0xba57, 0x465a, 0x1d3a, 0x1cb4, 0x4377, 0xbfc5, 0x40f1, 0x43b1, 0x42ea, 0x4249, 0xba57, 0x439e, 0x4263, 0x42cf, 0x4204, 0xb296, 0x4022, 0xa2ce, 0x4155, 0x07a1, 0x4005, 0x41bc, 0x430f, 0xbf0c, 0x4388, 0x1383, 0xb0da, 0x2956, 0x44e3, 0x432a, 0x4061, 0x421b, 0x4215, 0x455a, 0x41fa, 0x1d49, 0x4165, 0x40b8, 0x1258, 0x208a, 0xbf12, 0xa283, 0x1d11, 0x41bf, 0xbf55, 0xba7b, 0x4204, 0xba24, 0xbff0, 0xbf61, 0x40e1, 0x1ed0, 0x43c5, 0x42c8, 0x4215, 0xbad5, 0x4220, 0xb029, 0x439a, 0xb07b, 0xa83b, 0xb2ef, 0x1fb5, 0x4298, 0xb071, 0x40ef, 0x42e1, 0x2792, 0x1a2b, 0x1fb9, 0x32df, 0x209b, 0x3ecf, 0xba10, 0x409e, 0xb253, 0x4041, 0x196f, 0x443c, 0xbfd9, 0x189c, 0x441c, 0xbac0, 0xaa98, 0x0b73, 0xb229, 0x42d2, 0x4225, 0xbff0, 0x4226, 0x404c, 0x43b7, 0x4039, 0x4348, 0x411d, 0x40d0, 0x401b, 0x4187, 0xa008, 0xb2fa, 0xa858, 0xbf35, 0x420b, 0xb2c3, 0xb2fb, 0xb03c, 0x4063, 0xb0e0, 0xbfb0, 0x1b44, 0xbf5d, 0x4034, 0x4427, 0x43a2, 0xba5a, 0x46d8, 0x46d1, 0x3045, 0x33da, 0x43bb, 0x4018, 0x0618, 0x4192, 0x4357, 0x430e, 0x414b, 0x3656, 0xb2b3, 0xb03f, 0xbf90, 0x460b, 0x43c5, 0xbf1c, 0x4368, 0xba32, 0xbfc0, 0x415b, 0x4113, 0x1e66, 0x1eef, 0x4217, 0xba3a, 0xb280, 0x4355, 0x44a0, 0x4340, 0xb097, 0x423c, 0x43f6, 0x2cf6, 0xb250, 0x4627, 0x42cc, 0x2090, 0xba59, 0x4277, 0x24e1, 0x1f43, 0x14a1, 0xa852, 0xbfb6, 0x45f4, 0x46ec, 0x2c9b, 0x3866, 0xb2a8, 0x1d24, 0x4311, 0xb051, 0x4082, 0x2e93, 0x40f7, 0x40e3, 0xb010, 0xac6c, 0x1928, 0xbfdd, 0x0c95, 0xba78, 0x4409, 0x41af, 0x3367, 0x4204, 0x436a, 0x1d3c, 0x4043, 0xb0f9, 0x417f, 0x42ea, 0x42c7, 0x1cd8, 0x1db7, 0xa9fc, 0x1d0c, 0xbf60, 0x4589, 0x41e8, 0x25ec, 0x422c, 0xb239, 0x3b17, 0x45f5, 0xbf7b, 0x0725, 0x40b4, 0x4005, 0xb2f4, 0xbff0, 0x40a6, 0x412e, 0x4167, 0x30c3, 0x403b, 0xbfb8, 0x43c5, 0x402b, 0x1fa2, 0x3aca, 0xbfc5, 0xbafa, 0x4232, 0x4632, 0xb2b6, 0x4639, 0x4061, 0x4256, 0xb2e8, 0x17be, 0x4346, 0x4313, 0xbfb6, 0xba6f, 0x4103, 0xb030, 0x4196, 0xb03a, 0x0a90, 0xbf90, 0x4610, 0x40ae, 0xba66, 0x1d15, 0xa4b5, 0x4210, 0x42a7, 0xbac1, 0xbfb4, 0xaa8b, 0xb2e7, 0x405f, 0x4013, 0xa7f6, 0x1a48, 0x38c2, 0x420c, 0x210c, 0x40ad, 0x4309, 0x2e0a, 0xba29, 0xbad8, 0x0cb0, 0x4333, 0xba55, 0x1e96, 0xb21b, 0x2b22, 0x3f84, 0xbfa1, 0xa0a2, 0x419a, 0xb2c8, 0x40c3, 0x42ef, 0x4132, 0x41f4, 0xbfc6, 0x4650, 0x4322, 0x341e, 0xbf8b, 0x426b, 0x46b2, 0xb271, 0x422c, 0x4205, 0x4329, 0x419d, 0xa66e, 0xb28b, 0x409d, 0x3f5c, 0x37f3, 0x4135, 0xb25e, 0x4032, 0x4368, 0x44ad, 0x4089, 0x468a, 0x4309, 0x19f0, 0xb2ec, 0x419a, 0x4642, 0xaade, 0x221c, 0xbf12, 0xa8f7, 0xb251, 0x43cd, 0x4189, 0xbadc, 0x0a21, 0xb02b, 0x43a2, 0x4136, 0xbf07, 0x005c, 0x1ab6, 0x4192, 0x4084, 0x36b5, 0xb239, 0x19f8, 0xb236, 0x1a91, 0x40f4, 0x1c8b, 0x4142, 0x015e, 0x0894, 0x4229, 0xb090, 0xb051, 0x4073, 0x42f0, 0x1f44, 0x0dc9, 0x447f, 0xba48, 0x4150, 0x1af6, 0xba5e, 0x2fa7, 0xb088, 0x421c, 0xbf92, 0x43b1, 0x4007, 0x1992, 0x411c, 0x4021, 0x4004, 0x26a4, 0x0e50, 0x43ad, 0x3588, 0x1d58, 0x426c, 0xb0c6, 0x2222, 0xb2ef, 0x4440, 0xbf59, 0xb05b, 0x42fb, 0xb277, 0x4314, 0xbacf, 0x17b8, 0x4192, 0x40c9, 0x1d08, 0x43a8, 0xba2a, 0xb069, 0x43b9, 0x44d5, 0x10b0, 0xb2cd, 0x46e9, 0x4054, 0x4417, 0x2d2c, 0x42fb, 0x4094, 0xbf9c, 0x4598, 0x41d7, 0x4257, 0x4284, 0x4277, 0xaf28, 0xba78, 0xb0d1, 0x40f2, 0x43d2, 0x41e7, 0xb25d, 0x420c, 0x40e6, 0x1fa1, 0xb00c, 0x4066, 0x40ee, 0xbf08, 0x4387, 0xbf70, 0xb048, 0x42af, 0xbac4, 0x44cb, 0x40dd, 0x2044, 0x4247, 0x1dda, 0x250b, 0xbadd, 0x45a8, 0x4195, 0xbf38, 0x42d6, 0x4613, 0xb26d, 0x4581, 0x2f5e, 0xb2c5, 0x464e, 0xba66, 0x41c2, 0x2431, 0x4301, 0x4063, 0x2637, 0x46a0, 0xb059, 0x1d7b, 0x1f71, 0xb089, 0xbf15, 0x4111, 0x4372, 0xba3f, 0x1869, 0x13c5, 0x42cd, 0xb216, 0xb2b9, 0x4324, 0xbf70, 0x418d, 0x405a, 0x130b, 0xbfd7, 0x43e3, 0x131f, 0x42d2, 0x42b0, 0x0eb5, 0xba07, 0xb261, 0x4173, 0x4124, 0x3d11, 0x410c, 0xb268, 0xb23b, 0xb219, 0x4044, 0x011f, 0x33bd, 0x1f80, 0xb2c8, 0x4133, 0x4352, 0x1aad, 0xbfc2, 0x3647, 0xb2ef, 0x1964, 0x41f0, 0x2fd2, 0x4291, 0x438f, 0x1518, 0x4184, 0x4138, 0x1c5b, 0xa1de, 0xae02, 0x40f0, 0xbfdc, 0xba47, 0x40eb, 0x42f7, 0x2577, 0x1ba5, 0x431d, 0x4227, 0x1e88, 0xbf28, 0x1cfb, 0x409d, 0xbf5d, 0x4039, 0x4328, 0xb23c, 0x0f79, 0xb03e, 0x4177, 0x44c3, 0x4334, 0xbf4d, 0x40bc, 0x4343, 0x402a, 0x40c6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2facc260, 0xdcf5a5c6, 0x996babf4, 0xcf26a6f8, 0x2df7e7e5, 0xee4ea13b, 0xf856da3e, 0x6c32216c, 0x0f115238, 0x504a66b5, 0xb0cc8c19, 0x7ac767c7, 0x4550fd6f, 0x45a84035, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x00001b12, 0x00000000, 0x60000d99, 0x00000003, 0xfffc0000, 0xd2bdaee8, 0x0000116a, 0x45a84a12, 0x00000031, 0x45a848bd, 0x00000000, 0x8af9467f, 0x45a844ad, 0x45a84b01, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb2fe, 0x3eee, 0x4080, 0x4136, 0xb2e1, 0xb2b8, 0xba6e, 0x4298, 0x4164, 0x427d, 0x42cc, 0x416f, 0xbf4b, 0xba4a, 0x1e12, 0x43ca, 0x1c32, 0x41dd, 0x40be, 0x4374, 0x401a, 0xb0ad, 0x1506, 0x19bc, 0x11fe, 0xb298, 0x44c2, 0x1c97, 0xbf29, 0x4123, 0x4052, 0x4092, 0x4160, 0x209c, 0x42d7, 0x413b, 0x4333, 0xad04, 0x1147, 0x4054, 0x437a, 0xbaf8, 0x427a, 0x4197, 0x2636, 0xbfa2, 0x1958, 0x41b0, 0x40a3, 0x1ee7, 0x4132, 0x4306, 0x404f, 0xba70, 0x432f, 0x4551, 0xb0d9, 0xa5fd, 0x2460, 0x06fc, 0x3f2e, 0x29f1, 0x4233, 0x4318, 0x1044, 0xb277, 0xbf09, 0xb28f, 0x439f, 0x43ba, 0x06a4, 0x43b8, 0xba39, 0x1ebb, 0xb21f, 0x41e4, 0x4280, 0xbfb9, 0x3ebe, 0x11aa, 0x1b41, 0x1f7b, 0x4407, 0x2a7a, 0x4211, 0x41bb, 0xb0ae, 0x418b, 0x1665, 0x4177, 0xbfc0, 0xb280, 0x1905, 0x4267, 0xbfc6, 0x43a0, 0x42e5, 0x45b4, 0xba2e, 0x43a3, 0x2486, 0x40eb, 0x4297, 0x23fa, 0xbf71, 0x41e5, 0x4449, 0xb051, 0x0d5b, 0x4027, 0xba23, 0x3126, 0xbf57, 0x40b8, 0x41cf, 0x464f, 0x4082, 0x4075, 0x416a, 0x18df, 0xb0e9, 0xba0a, 0x4172, 0x14b5, 0x4394, 0x4232, 0x0ff1, 0xbf96, 0x406f, 0x43d7, 0x42e3, 0x0dc2, 0x4025, 0x46a0, 0x1d27, 0xb2f3, 0x42e4, 0x4099, 0x1a00, 0xb0b2, 0x1b48, 0x40a8, 0x420f, 0x43cd, 0xb270, 0xbf92, 0x412d, 0x4178, 0x4400, 0xb2f4, 0x40f1, 0x409f, 0x124a, 0x4584, 0x4312, 0x40f4, 0x1b70, 0x441e, 0x1866, 0x40ba, 0x4337, 0xbf1f, 0xb235, 0x41a4, 0xbf80, 0xb250, 0x40f9, 0xb2f5, 0x4237, 0xba2e, 0xbf1a, 0x433c, 0xb2b5, 0xbad6, 0xb0c9, 0xbf41, 0x43b3, 0x43c4, 0x3c17, 0x414f, 0x4194, 0xaf3b, 0x42cb, 0x1b27, 0x41bf, 0x418c, 0x050e, 0x44e5, 0x41ac, 0x45db, 0x431e, 0x43e8, 0x421e, 0x41f6, 0x18a1, 0x417f, 0xbf7f, 0x412b, 0x1dc5, 0xbacb, 0x46e1, 0x1321, 0x1e34, 0x42c6, 0x413f, 0x3725, 0x3245, 0x43cd, 0x1de1, 0x40d4, 0x35bb, 0x42c1, 0x4100, 0x4438, 0x028c, 0xb208, 0x4449, 0xbae6, 0xb08b, 0x0936, 0x433c, 0x412e, 0xb095, 0x41b8, 0x1d8c, 0x41b3, 0xbfa8, 0x4616, 0x4043, 0x45b3, 0x4214, 0x1ceb, 0x4133, 0x410a, 0x1c15, 0xbf16, 0xb2fc, 0xba2d, 0xab9f, 0x4037, 0x1f6d, 0x2f24, 0x4309, 0x4431, 0xbace, 0xba3d, 0x43cc, 0xb2e1, 0x137d, 0x43d5, 0x3f6c, 0x41e8, 0x3089, 0x42bf, 0x45d3, 0x4049, 0xbf89, 0x0e07, 0x2da2, 0x427e, 0x43c9, 0xb2cd, 0x1f71, 0xadba, 0xb2e3, 0xba77, 0x4250, 0x46fc, 0x24c9, 0x43ce, 0xb02f, 0x468d, 0xba17, 0x2ee2, 0x448b, 0xbfbd, 0x195a, 0x41a3, 0x33fb, 0x437a, 0xbf65, 0x4004, 0x43ab, 0x42c3, 0xaa5c, 0x2944, 0x1baf, 0x2a27, 0x4266, 0xbf91, 0x41c6, 0x42b8, 0xba74, 0x41b7, 0x355c, 0xbae4, 0x405f, 0x40be, 0xbaf5, 0x437e, 0xbac0, 0x4204, 0x436f, 0x46f3, 0x1f70, 0x4277, 0x4053, 0xbf80, 0xb2ee, 0xbfd5, 0xb2b0, 0x41e9, 0x424b, 0x4135, 0x4561, 0xbae0, 0xb200, 0xba21, 0x4360, 0xb0c1, 0xb207, 0x40a1, 0x4373, 0xb09e, 0x0929, 0x3cad, 0x1997, 0x411a, 0xb020, 0xbfa2, 0x3fca, 0xb23d, 0x087e, 0x055d, 0x4149, 0xb289, 0x2152, 0x1be1, 0x43b1, 0x40a1, 0xbf0d, 0x431d, 0xa7b2, 0x3d9a, 0x40a1, 0x42cf, 0x43e4, 0xbfc2, 0xaf46, 0xb23e, 0xb2e6, 0xb279, 0xa8af, 0xb076, 0xb0e7, 0x184a, 0x41bf, 0x42c8, 0x3d7b, 0xb02c, 0x1bd0, 0x1866, 0x14fe, 0x19c0, 0x409b, 0x435c, 0x4169, 0x405d, 0x4556, 0x4052, 0x435b, 0x412b, 0xbf73, 0x1ed8, 0x419f, 0x198b, 0x43da, 0x42f3, 0xba35, 0x1db6, 0xb012, 0x43d6, 0x0534, 0xabbe, 0x2ce2, 0x427f, 0x435f, 0x4169, 0x4265, 0xbf58, 0xb280, 0xbf90, 0x404e, 0xba75, 0x4162, 0x4105, 0xba5d, 0x40c4, 0x4044, 0x3603, 0x43c0, 0xb2ba, 0xb0b5, 0x247f, 0xba45, 0x4330, 0xb21a, 0x40ba, 0xba1d, 0xb24f, 0x43e3, 0xb09c, 0xbf5e, 0xb07d, 0xba5a, 0xb093, 0xb244, 0xba35, 0x419d, 0x4303, 0x09ce, 0x1af4, 0x4584, 0xb215, 0x4385, 0xb2ef, 0x46a4, 0x413d, 0x4389, 0x1d06, 0x07b8, 0x4558, 0x4494, 0x460a, 0xbf3c, 0x4158, 0xb225, 0xb2f1, 0x41ce, 0xbf2e, 0xb054, 0x4249, 0x45c6, 0x433e, 0x4685, 0x0ade, 0x4172, 0x42dc, 0x0fbd, 0x24d2, 0x4101, 0xbfd0, 0xb2a9, 0x4155, 0x416a, 0x4186, 0xba42, 0xb269, 0x1b7d, 0x4132, 0xa644, 0x1fc0, 0x43a1, 0xbf23, 0x02c3, 0x280b, 0x40b4, 0x4149, 0x4029, 0x217e, 0x10d4, 0x422a, 0x43ba, 0xba4b, 0x4291, 0x04ca, 0x1b38, 0x439a, 0x2d58, 0x42ca, 0x1e61, 0x2260, 0x4639, 0x46ca, 0x45be, 0xbf26, 0xb26d, 0x422e, 0x28d7, 0x4055, 0x3584, 0x41ff, 0x4322, 0xb27a, 0x40ee, 0x42f5, 0xba76, 0x40cc, 0xba68, 0x41b2, 0x1053, 0xbf7a, 0x4351, 0xb2a9, 0x4209, 0x2895, 0xb013, 0xbfd0, 0xa94c, 0x32bd, 0x41d8, 0xb283, 0x405c, 0x195d, 0xb25d, 0x1f07, 0xb249, 0x1a63, 0xb079, 0xaa92, 0x140c, 0xa9f0, 0x2d54, 0xa645, 0x37b2, 0x0ae8, 0xbf57, 0xbaf4, 0xba7c, 0x0abd, 0x42b9, 0x405c, 0x3f47, 0x43df, 0xb2db, 0x4329, 0x4134, 0x298d, 0xbaf3, 0xbfd5, 0xb2e3, 0x193e, 0xb232, 0xb276, 0xa3a0, 0x1d92, 0x41f6, 0x420f, 0x424d, 0xba32, 0x426a, 0xa208, 0x37fd, 0xba7e, 0x4363, 0x405f, 0x2b88, 0x1094, 0xbf1b, 0x1f17, 0xa26c, 0xbf70, 0xb27a, 0xb03a, 0x415b, 0x436d, 0xacf7, 0x42b2, 0xa2a5, 0x42f9, 0x414b, 0x4236, 0x1c8d, 0xbfa9, 0x41d8, 0x45a8, 0x43f5, 0x3226, 0xb0ec, 0x4248, 0x1e80, 0x4192, 0x1c5b, 0xb287, 0x4107, 0x467b, 0x4391, 0x406b, 0x4489, 0x18c6, 0x44dc, 0x4126, 0x1c8d, 0xb029, 0x42ac, 0x4055, 0x413e, 0x408e, 0xbf76, 0x052a, 0xbf90, 0xb261, 0xb2ae, 0x42b4, 0x26af, 0x43d5, 0x4084, 0xba6b, 0xb002, 0x09f3, 0x1f9b, 0x4300, 0xbf92, 0xa86c, 0x4187, 0x46f3, 0xb0cb, 0x41c1, 0x400e, 0x04b6, 0xbaf0, 0x10bc, 0xb0ff, 0xbfc1, 0xb244, 0x3bf7, 0x45a2, 0x4173, 0x415b, 0xba34, 0x3976, 0x1837, 0x40ad, 0x197e, 0x284b, 0x459a, 0x17e0, 0x41bf, 0x408f, 0x1ae0, 0x13a0, 0xbacd, 0x40c1, 0x420b, 0xba22, 0x44e8, 0x41a9, 0xa4b1, 0xb2ba, 0x41b2, 0x2d95, 0x4320, 0x4663, 0xbf37, 0xa9ae, 0xb2c5, 0x2310, 0x4329, 0x4239, 0xb0ef, 0x2d06, 0x4418, 0x42f3, 0xba76, 0xa639, 0x45d8, 0xb263, 0x2dc3, 0x1ce6, 0xbf4b, 0x43e6, 0xa2b7, 0x1c0c, 0x0252, 0xbf9b, 0x4151, 0x4062, 0x4148, 0x4360, 0x43bf, 0x07ca, 0x15e8, 0x40e2, 0x34b3, 0x433a, 0x42e4, 0x38ce, 0x34cc, 0x1796, 0xb04d, 0x3fc2, 0x46d3, 0x422e, 0xb09d, 0xbaf0, 0x4319, 0x4057, 0x435e, 0xbfe0, 0x4393, 0xbf64, 0x1a89, 0x422e, 0xb01e, 0x400c, 0xb07d, 0x1d17, 0x426c, 0x46ab, 0x454f, 0x44e5, 0x4085, 0xb209, 0x194d, 0x1f97, 0x411d, 0x4051, 0x1fd4, 0x3a6e, 0xbf4a, 0xb27a, 0x0cda, 0xae68, 0x4026, 0xb2cc, 0xba30, 0xb24b, 0x1e97, 0xbf77, 0x4231, 0xa0c7, 0xb268, 0x268b, 0xb0ad, 0xb209, 0x1b9f, 0x4035, 0x4237, 0x0231, 0x1aaf, 0xb0cb, 0x1d8e, 0x42e4, 0x42e4, 0x21cf, 0xb2d2, 0x420d, 0x4041, 0xac5b, 0xa18a, 0x43f6, 0xbaf7, 0x388e, 0x1c7f, 0xa574, 0xbf89, 0x43bd, 0x1ab6, 0xba61, 0x4218, 0x1ea0, 0x42c9, 0x432d, 0x4376, 0x1933, 0x4134, 0x1d9b, 0x27df, 0x43c4, 0xbfd5, 0x4203, 0xa8dc, 0x0ae8, 0x440a, 0xba0c, 0x3294, 0x4174, 0x42c9, 0xa2d7, 0x42ed, 0xbad3, 0x1bfe, 0x4326, 0x4035, 0x4348, 0xb292, 0xbf9e, 0x18f3, 0x43f1, 0x1367, 0x411b, 0x4477, 0x1e0b, 0x1d22, 0x1c39, 0x40bd, 0x435a, 0x1be5, 0x13c2, 0x1b7c, 0x433c, 0xa9d8, 0xa165, 0xbf69, 0xb057, 0x413d, 0x40f9, 0xb202, 0x3e95, 0x4185, 0xbf29, 0x1f39, 0x19fd, 0x42f1, 0xa2f2, 0xb2d2, 0xb070, 0x415c, 0xbf24, 0x43ba, 0x3aca, 0x1a2f, 0x4150, 0x43de, 0xb2df, 0x436d, 0xbfba, 0xad69, 0x1080, 0xb0f5, 0x4307, 0x0ab6, 0x4002, 0x41f4, 0xba63, 0x4220, 0xb28f, 0xba7e, 0xa3b6, 0xa952, 0xba6a, 0x447f, 0x1cd0, 0xb055, 0x4392, 0xb231, 0x1af3, 0x4354, 0x41cf, 0xa231, 0xbfde, 0x35ca, 0xab3b, 0xb0e7, 0x43ea, 0xbae7, 0x4375, 0x43ce, 0x31e5, 0x393c, 0x42f9, 0x4135, 0xb260, 0x409b, 0xba67, 0xb26d, 0x3fac, 0xb26c, 0x425d, 0x43de, 0xbf6d, 0x435b, 0x2efd, 0x1ae3, 0x429f, 0x187d, 0xb0a5, 0x41d7, 0x40b9, 0xbad6, 0x15d3, 0x4666, 0x46ad, 0xbf6f, 0x2348, 0x4006, 0x1221, 0xba18, 0x4271, 0x4057, 0x1829, 0x1e42, 0xa497, 0x1dc7, 0x41eb, 0x4225, 0x43b5, 0x1603, 0xb04f, 0xbf84, 0x3451, 0xb055, 0x4499, 0xb2e1, 0x41cd, 0xb25a, 0x41d8, 0x4305, 0x40ee, 0x411e, 0xba6f, 0x0452, 0x4160, 0x413d, 0x408c, 0x4232, 0x4047, 0x4452, 0xba4c, 0xa758, 0xbf21, 0x4616, 0x1990, 0x422a, 0x1dae, 0x1f74, 0x0f3f, 0x1f61, 0xbac5, 0x1d9e, 0xbaf1, 0xbf12, 0x1bd8, 0x2b26, 0xb05d, 0x09a5, 0x408e, 0x1ca7, 0x0a35, 0xa904, 0xaed0, 0x16fb, 0xbf80, 0xa941, 0xae2a, 0x1e3e, 0xb256, 0xa51e, 0xb2c4, 0x435e, 0xb288, 0x403f, 0xb2a9, 0xa590, 0x40a3, 0x4112, 0xbf99, 0x1c5f, 0x426e, 0x40c1, 0xb26c, 0xbfae, 0xb2a3, 0x43a7, 0xb213, 0x420f, 0x429f, 0x4080, 0x38a0, 0xba39, 0x1afa, 0xb027, 0x36e3, 0x37ef, 0x42c4, 0x40c9, 0x42de, 0xb0ec, 0xbf0b, 0xbac5, 0x241d, 0x25fb, 0x4182, 0xb251, 0xaa91, 0x4383, 0xbfb3, 0xba70, 0xb0a0, 0xba77, 0x4068, 0x1c8f, 0x12d3, 0xb27f, 0xb2c8, 0xbada, 0x3b8f, 0xbf32, 0xbaec, 0xb092, 0xbae3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x989fe979, 0xeb6d1dc2, 0x0199a793, 0xfd6ddabf, 0x2337d8f4, 0x10d088ef, 0x6abc19fa, 0x03f7a6b3, 0x8efc124d, 0x26bfcdfd, 0x162448d8, 0x1f47e67e, 0xbfa7bf3d, 0xf5019635, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x000000ea, 0xffffffea, 0xfffffbff, 0x00001d00, 0x0000001d, 0x000019b4, 0x000000a6, 0xffffffec, 0x3fffff72, 0xffd83f36, 0xbfa7bf3d, 0x000000d6, 0x020003a0, 0xffffdcc8, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x432c, 0x0c6d, 0xb075, 0x4339, 0x43c2, 0xb043, 0x4305, 0x42c6, 0x0cb5, 0x3d42, 0x3249, 0x40c7, 0xbf9f, 0x0213, 0x4268, 0x4192, 0x342f, 0x20a6, 0xb2bf, 0x0700, 0x41ae, 0x17af, 0x1c55, 0xbfd1, 0x437d, 0x3f9d, 0x40a9, 0x41e4, 0xba3d, 0x415d, 0xb2bf, 0x4211, 0x2b93, 0x4161, 0x412a, 0x41e7, 0x2540, 0x41a9, 0xbacf, 0x34a4, 0x1847, 0x448b, 0x448a, 0x4187, 0x40cc, 0xa4d4, 0xb2fd, 0xb2d5, 0x1e75, 0x0762, 0x43ac, 0x4020, 0xbf8f, 0xb207, 0x4351, 0xb296, 0xb230, 0xb0e4, 0xbf4c, 0x44e2, 0x4043, 0xbac6, 0xb26b, 0xb2c3, 0x3a43, 0x4182, 0x46d9, 0x312b, 0x32a1, 0x3341, 0x4136, 0x4385, 0xbf77, 0x30bb, 0x4083, 0x4621, 0xba7a, 0x46c0, 0xb2eb, 0x42c5, 0xb27f, 0x1e4f, 0x416f, 0x199e, 0x4463, 0xbf8b, 0x4117, 0xbaf4, 0x45f2, 0x3e2f, 0xb248, 0x41d0, 0xb068, 0x467f, 0xa251, 0xb26e, 0x29aa, 0x2bbf, 0xb2cf, 0x439e, 0x1e50, 0x41f9, 0xbf80, 0x456d, 0xbf48, 0x1ff7, 0x46f9, 0x28ba, 0xbae5, 0x455a, 0xb23a, 0x4234, 0x4224, 0x3582, 0x41d8, 0x4038, 0x420d, 0x42a4, 0x2823, 0x2019, 0xbf98, 0x289f, 0xb027, 0x3f10, 0x2356, 0x43d4, 0x4327, 0x2b77, 0xab61, 0x43dc, 0x41fe, 0x1560, 0x01d5, 0x43cc, 0x4310, 0xbf04, 0x4053, 0x093b, 0x43b5, 0x4311, 0xb033, 0x30ba, 0xae34, 0xbf78, 0x4051, 0xba31, 0x4113, 0x4364, 0xaa6d, 0x2548, 0xb254, 0x1979, 0x4399, 0xb2d1, 0x2a2e, 0xb22e, 0x40a2, 0x1026, 0x428d, 0xb29f, 0x18b9, 0xb0b5, 0x422a, 0x4138, 0x2e00, 0x402b, 0xbf2c, 0xb06a, 0x187c, 0x402f, 0xba21, 0x407d, 0x008f, 0x4119, 0xb2ef, 0x4260, 0x4185, 0x43f8, 0x4028, 0xbae3, 0x4005, 0x1f3a, 0xad44, 0xb28e, 0xa699, 0xa201, 0x4243, 0xb2b5, 0x1978, 0x40d2, 0xbf48, 0x430a, 0x1965, 0xb222, 0x123c, 0x40ab, 0x0239, 0xb2b0, 0xbf9d, 0x36b8, 0x2273, 0xafdb, 0xb2dd, 0x1b98, 0x3bd9, 0x2680, 0xba1b, 0x1485, 0x4387, 0x42e2, 0x3bc3, 0x43c9, 0xb005, 0x363c, 0xbf07, 0xba48, 0x3851, 0xba45, 0x4288, 0x41ea, 0x19f6, 0xb2e6, 0x1a33, 0x413c, 0x4002, 0x4304, 0x42de, 0xbadf, 0x1b8a, 0x4032, 0xb219, 0x4423, 0x4634, 0x4089, 0x4640, 0x1b8d, 0x4207, 0x3e42, 0x41f1, 0x42fa, 0xbfa7, 0x4601, 0x1f59, 0xb2cf, 0x40bf, 0x4559, 0x1eb2, 0x1bdb, 0x42af, 0xb226, 0x44f4, 0x43a2, 0x1aea, 0x3d70, 0x190d, 0x436a, 0xb2b9, 0x17f6, 0xba64, 0x44d3, 0x400c, 0xb033, 0x4015, 0x0c75, 0xba2c, 0xbae8, 0xbf64, 0x1a67, 0x411e, 0xb2dc, 0x41ca, 0x45ec, 0x08d1, 0x4039, 0x42b9, 0x40f1, 0xb076, 0xae19, 0xb256, 0x3711, 0x4067, 0x410b, 0xa4c0, 0x1cd5, 0x1c2e, 0x1f01, 0x426c, 0x4274, 0x075d, 0xbf66, 0x2825, 0x1fca, 0xb266, 0xbaf2, 0x424a, 0x42f6, 0x236d, 0xb27f, 0xb2fa, 0x45eb, 0xbf0e, 0x1cdd, 0xaf37, 0x43a7, 0xb08e, 0x4258, 0x41f3, 0x211e, 0x40b6, 0xba7a, 0xbf76, 0x467c, 0x417d, 0x4173, 0x4623, 0x4058, 0xbad5, 0x423c, 0xbf80, 0x42e7, 0x3860, 0x2a24, 0x0276, 0xb033, 0xbfdc, 0xb2c3, 0xb224, 0xb27e, 0x2369, 0xba38, 0x4277, 0xb07a, 0x42c2, 0x41c2, 0x4668, 0xa9f2, 0xbae0, 0xbfc2, 0x41c2, 0x414e, 0x4322, 0x42d9, 0xbaf8, 0x169e, 0x4229, 0x2d30, 0x402b, 0x42ef, 0xb2e0, 0x42c4, 0x3c0f, 0x33ac, 0x117b, 0xbacd, 0x1961, 0x41af, 0x4041, 0xba1d, 0xbfa0, 0xbf7c, 0x403d, 0x3710, 0x429d, 0x41e8, 0xba30, 0xbf80, 0xacb2, 0x196a, 0x40b1, 0x416c, 0x1a81, 0xbf90, 0x41ef, 0xb079, 0x412c, 0xb00a, 0xbf80, 0x03d7, 0xb28b, 0x44ed, 0x4153, 0xbf7f, 0x413a, 0x4553, 0x416b, 0x4030, 0x408a, 0x1fc8, 0xb0d4, 0x3ad1, 0x4153, 0x4188, 0x409e, 0x45a0, 0x390b, 0x0957, 0x105d, 0x2a7c, 0x437c, 0xbf4f, 0xba1b, 0x1847, 0xb22a, 0x4574, 0xba75, 0x4162, 0x17f6, 0x4357, 0x42e1, 0x4186, 0x4378, 0x4035, 0xa34c, 0x43e1, 0x1d89, 0x46a1, 0x3132, 0x443e, 0x4173, 0x45c2, 0x1fd2, 0x41f9, 0xbfb4, 0x460c, 0x1f6e, 0x3143, 0x435a, 0xb2bc, 0x4272, 0x1719, 0xbacc, 0xbf46, 0x427d, 0x40d9, 0x437e, 0x03aa, 0x41fd, 0x4187, 0x3d10, 0x40b6, 0xb2d2, 0xbfd0, 0x427d, 0x421c, 0x430a, 0xba14, 0x18cb, 0xba2c, 0xb087, 0xbfb3, 0x3e14, 0x4305, 0x4134, 0xb291, 0x165c, 0x406b, 0xbf1e, 0x40cd, 0xb2dc, 0x403c, 0x029a, 0x0386, 0x406e, 0x429d, 0x1e23, 0x441c, 0x0b65, 0x460e, 0xb224, 0x4158, 0x3c93, 0xa9dc, 0xbfbd, 0x0dfc, 0x46ba, 0xb07f, 0xba01, 0x414d, 0x1744, 0x1cce, 0x40fa, 0x4120, 0x2b23, 0x4465, 0x1c51, 0x0aef, 0x4234, 0xb04a, 0x4392, 0x2c8e, 0xb20e, 0xbf61, 0xbae5, 0xba31, 0x4055, 0x42b4, 0xb28b, 0x02b5, 0x02a8, 0xa873, 0x417e, 0x41fe, 0x42c0, 0x4259, 0x41bc, 0xbff0, 0xbfc1, 0x1987, 0x40d2, 0x1b97, 0x41fc, 0x0019, 0x4139, 0xb212, 0x151d, 0x4231, 0xbfb0, 0xbafb, 0x06e7, 0x1817, 0x4270, 0x006c, 0x409c, 0x3797, 0x1801, 0x4116, 0xb242, 0x4028, 0xb2e8, 0x0e08, 0x426b, 0xbf8c, 0x1eb2, 0xb27e, 0xb2e9, 0x3814, 0x0b2a, 0xb2e7, 0xbf87, 0x2e03, 0x1bd0, 0x43f5, 0x4356, 0x42b1, 0xa47c, 0xb0f4, 0x427d, 0xb26a, 0xb287, 0x1b09, 0x0484, 0xbf85, 0x4112, 0xb057, 0xbfc0, 0x2044, 0x435f, 0x41a3, 0x3ba9, 0x4141, 0xb2a4, 0x4060, 0x40af, 0xb0fd, 0x4243, 0x4170, 0x422a, 0xba39, 0x416d, 0xbf81, 0xb2ef, 0xb061, 0xb015, 0x437e, 0xbfe2, 0x4168, 0x13c9, 0xb02d, 0x296a, 0x42d2, 0x3994, 0x426b, 0x46c3, 0x1a5d, 0x4336, 0x4133, 0x43a4, 0xb202, 0x42dd, 0x3226, 0xb2a9, 0xb0e7, 0x23db, 0xb072, 0x1bf5, 0x43f7, 0x44c5, 0xb223, 0x4312, 0xbf47, 0xba56, 0x1089, 0x27e8, 0xb208, 0xabdd, 0x3972, 0x41ce, 0x4371, 0x41c1, 0x4082, 0x2c57, 0x1ef7, 0x428c, 0x4053, 0x21b6, 0x447d, 0xb015, 0x40d2, 0x4110, 0x0df3, 0xbf8d, 0xb28d, 0x4350, 0xbada, 0x4058, 0xbaf3, 0x1e42, 0xbad4, 0xabca, 0x2d58, 0x41b7, 0x421f, 0x40a4, 0x0c09, 0xb022, 0x4170, 0x1811, 0x3af8, 0x30c1, 0xba50, 0x41e6, 0x4477, 0x425d, 0x4076, 0x42ca, 0x43a5, 0x1834, 0xbf7a, 0x31ad, 0xbafe, 0x1bc6, 0xb247, 0x1db0, 0x418f, 0x4163, 0x4008, 0x4674, 0xb2f5, 0xb269, 0xb215, 0x066e, 0x19dc, 0x24a4, 0xba0e, 0x4100, 0xbac7, 0xa0f0, 0x46ec, 0xa3c8, 0x311e, 0x0490, 0xb04b, 0x4016, 0x4085, 0x4024, 0xbf5b, 0x4028, 0xb2e5, 0xb054, 0x4189, 0xb267, 0xa751, 0x42b0, 0x40ca, 0xaf76, 0xbace, 0xbad2, 0xbf99, 0x1ef3, 0x435f, 0x4384, 0x3383, 0x4247, 0x4338, 0x1a84, 0x416a, 0xbaea, 0x41a6, 0x4002, 0xbf90, 0xb05a, 0xa518, 0x3010, 0x43eb, 0x4350, 0x352f, 0x43c9, 0xba2b, 0x3361, 0x421e, 0xb268, 0x407d, 0xbfb4, 0xba4e, 0x44b5, 0x41a2, 0x419f, 0x410f, 0x46c9, 0x19d8, 0x0eec, 0xbad3, 0xa0a0, 0xbf60, 0xbfda, 0x188a, 0x129a, 0x400d, 0xba64, 0x4310, 0x41af, 0xba79, 0xba55, 0xb063, 0xbf80, 0xbf0e, 0x442d, 0x1826, 0x42d1, 0x1bf8, 0x435f, 0xba15, 0x2f8f, 0x1945, 0xa28c, 0x41f4, 0xbf93, 0x1eab, 0xaaf6, 0xba0b, 0x2fda, 0xbfb0, 0x4228, 0xa802, 0xb2f6, 0x425d, 0xb2b8, 0xbf33, 0x418e, 0x435b, 0x1feb, 0x4252, 0x4359, 0x462b, 0xba17, 0x40bf, 0x43f9, 0x43d5, 0x03c1, 0x40bb, 0x40f7, 0x4391, 0xaa2c, 0x45b8, 0x4678, 0x42ad, 0x4088, 0x03c6, 0x4573, 0x40e9, 0xb282, 0x41cd, 0xb2f6, 0x1913, 0x40cd, 0x41e9, 0xbfa5, 0x262c, 0x42c9, 0x28d4, 0x3fec, 0x3b30, 0x42a4, 0x41ab, 0xbff0, 0x43f8, 0xb225, 0x41dc, 0xb2ff, 0x1866, 0x2870, 0x4286, 0xab63, 0x3889, 0x407d, 0x40fa, 0x25a1, 0x4488, 0x416c, 0x433a, 0xbfbf, 0x0155, 0x43f6, 0xb225, 0xb22e, 0x403f, 0x1355, 0x428f, 0xba2e, 0x438b, 0xb074, 0x445a, 0xb2de, 0xb0f1, 0x4241, 0x387f, 0xa5ab, 0x401c, 0xb02a, 0xb23a, 0xb22c, 0x2cba, 0x43c6, 0x1755, 0x4313, 0xbfb1, 0xa2de, 0x085c, 0xb0d9, 0x2705, 0xbf81, 0xb2af, 0xba09, 0x4314, 0xb2fa, 0xbad6, 0xbfb0, 0xbaf6, 0x1ba9, 0x4699, 0xb058, 0x43de, 0x32d1, 0x1c10, 0x41cb, 0xb281, 0x41be, 0x4360, 0xb296, 0x01d1, 0xb227, 0xafe4, 0x3965, 0x19d5, 0x40b6, 0xb09d, 0xb2e4, 0x42c3, 0x426a, 0xab02, 0xbf48, 0x4143, 0xbf1e, 0xb091, 0x0169, 0x09c9, 0xba62, 0x1f2b, 0xb060, 0xb0c4, 0x4353, 0x4377, 0x1c67, 0x4083, 0x1ba9, 0x43cb, 0x435f, 0x1b26, 0xb2ed, 0x3409, 0x2f3c, 0x41d0, 0x4306, 0x19f8, 0xb035, 0x4401, 0x076b, 0x4474, 0x1add, 0x41a4, 0xbf6a, 0x2302, 0x401a, 0x44f1, 0xbf6c, 0x426f, 0xb214, 0xb25d, 0xb226, 0x010f, 0x4261, 0xbacc, 0xbf25, 0x434e, 0x2220, 0xa694, 0x0089, 0x421d, 0x4083, 0x4092, 0x4333, 0xb22f, 0x4217, 0x432b, 0xb0d5, 0x4637, 0xab70, 0x42b4, 0x3cef, 0xb21a, 0x09ba, 0xb2fb, 0x4193, 0x41eb, 0xba2a, 0xba50, 0xa277, 0x4198, 0x4075, 0xbfbc, 0xaba3, 0x1814, 0x1dbd, 0xb260, 0x4305, 0xbfc0, 0x4334, 0x43cb, 0x2690, 0x1dd1, 0x42f4, 0xbf3f, 0x4294, 0x034c, 0x31f5, 0x4616, 0x4340, 0x224f, 0xba03, 0x4063, 0xb2ef, 0x4026, 0x4182, 0x4321, 0xaa72, 0x4242, 0x4265, 0xb085, 0xbaf2, 0xbfc9, 0xb2dc, 0x432c, 0x0a3b, 0xba19, 0x40fa, 0x4580, 0x1ec1, 0xba1b, 0x1a32, 0xa367, 0xbfe0, 0xacd3, 0x4179, 0x13e8, 0x2574, 0xb25c, 0xb228, 0x1852, 0xbfcb, 0xb2b6, 0x415c, 0x42bc, 0xba53, 0x439d, 0xbf2a, 0x15c3, 0xb2ac, 0x0d9c, 0xbf1c, 0xba32, 0xbaed, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x73229dce, 0x0f640776, 0x9c25ec11, 0x66a54f6d, 0xca4c2372, 0x617f5393, 0xfc67eb08, 0x99c5589c, 0x6805463b, 0x1f6ada41, 0x7ed9c21c, 0x50147fab, 0x6e783256, 0x5296ab63, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x00000074, 0x00000135, 0x10000000, 0x00000000, 0x00000000, 0x00003000, 0x00000010, 0x00000017, 0x6805463b, 0x0d36c137, 0x0a3e95fc, 0x6805463b, 0x0d32b38d, 0x0d36bf7f, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x42c2, 0x2d74, 0x4116, 0x4185, 0x40fd, 0x428a, 0x1707, 0x406c, 0xb298, 0xb252, 0x41ac, 0x1a53, 0xba5c, 0x411c, 0x4253, 0x3541, 0xbf27, 0x015a, 0xba76, 0x1226, 0x4251, 0xba16, 0xbfd0, 0x352c, 0x417b, 0xb009, 0x34c2, 0x41a9, 0x1d34, 0x438e, 0xb083, 0xb24a, 0x41d0, 0x1f5b, 0x1c0b, 0x4322, 0x1ae0, 0x234a, 0xbfd2, 0x43c9, 0x4212, 0x1556, 0x456d, 0x07bd, 0x4336, 0x43ab, 0xba1f, 0x3fd4, 0x4210, 0x41e2, 0x18d9, 0x1d4d, 0x4572, 0x4199, 0xb25e, 0xb207, 0x43fd, 0x3dee, 0x4382, 0x42fb, 0xb0cc, 0xba09, 0x2beb, 0x00e1, 0xba2a, 0xb068, 0xbf31, 0x40f0, 0x3c42, 0xb236, 0xb274, 0x44da, 0xa2d8, 0xb211, 0xba22, 0x43f8, 0x4102, 0x42e7, 0x46ca, 0x24ab, 0x4423, 0xb0f0, 0xb221, 0x421f, 0xb20d, 0x1dbe, 0x460c, 0x4040, 0xb026, 0xba47, 0x404d, 0xb22b, 0xbfb3, 0x43a8, 0x107b, 0xbf60, 0xb2d0, 0x430a, 0x3e97, 0xb272, 0x1c2b, 0x42b5, 0x4053, 0xb286, 0x42e0, 0xba44, 0x4114, 0x421c, 0xb2bf, 0x1eb5, 0x41de, 0x4097, 0xb273, 0x4121, 0x198c, 0xb281, 0xb03c, 0x409c, 0xbf46, 0xb2db, 0x443a, 0x41bf, 0xb244, 0x43a6, 0xaa45, 0xb03b, 0x43c8, 0x1d75, 0x400b, 0x1c22, 0xbae4, 0x0c72, 0x4130, 0xbae0, 0x46cc, 0xbfb1, 0x43ad, 0xba49, 0x41c2, 0x0327, 0x4027, 0x41ff, 0x4085, 0xbfcb, 0x412a, 0x1157, 0x43ab, 0xb23c, 0xb278, 0x421c, 0x416c, 0x4173, 0x43b1, 0x41c4, 0x40e9, 0x420e, 0x1bdf, 0xbfd0, 0x4672, 0xba6d, 0x07d1, 0x43c0, 0xb2aa, 0x40d1, 0x41d4, 0xa179, 0x44b2, 0xb052, 0x464a, 0xb24e, 0x4476, 0xa5a9, 0xbf0d, 0xbf90, 0x40c4, 0x44d9, 0x43cc, 0xbf00, 0x42bd, 0x4123, 0xb0ef, 0xba5f, 0x3212, 0x26f0, 0x4357, 0xb227, 0x43ef, 0x28d3, 0x1a18, 0x1e20, 0xade4, 0x42b0, 0xae3b, 0xa4cb, 0x1e56, 0x41c0, 0x434c, 0xbf51, 0x411e, 0xbfe0, 0xba57, 0xb06f, 0x4278, 0xbadd, 0xa0b7, 0x43b7, 0x41fe, 0x4284, 0x41a9, 0x40d7, 0xb0e4, 0x4418, 0x2466, 0x401f, 0xbfa4, 0xb258, 0x1aa7, 0x427b, 0xb2ec, 0x40e4, 0x40f3, 0x1f6b, 0x4462, 0xb29d, 0xb09d, 0x3a5d, 0x43d9, 0xb253, 0x40a7, 0xbac5, 0x4193, 0x0e8b, 0x43c0, 0x4214, 0x2268, 0xb2fc, 0x24b2, 0xb20c, 0x431e, 0x31a0, 0x43de, 0xbfc7, 0xad2f, 0xb0f3, 0x1fc6, 0xb272, 0x40fe, 0xb24f, 0x4115, 0x456e, 0x3ea3, 0x4122, 0x4323, 0x41b7, 0x414b, 0x4624, 0x33c0, 0x1eb4, 0x4333, 0x419e, 0x1a49, 0x1cc2, 0x35b5, 0x423e, 0xbf7c, 0x1f87, 0xb2cd, 0x43a1, 0xbf1c, 0x4215, 0xba79, 0xbfd7, 0xbad0, 0xba62, 0x4213, 0x429a, 0xbfbb, 0x43d6, 0x4197, 0x426d, 0x43f5, 0x4267, 0x40cd, 0xb2ad, 0x4071, 0xa27f, 0x3837, 0x408f, 0x41f2, 0x4167, 0x1d86, 0xaa44, 0xbae5, 0x1f57, 0x456a, 0x45c6, 0x42fb, 0xb2bc, 0xbf21, 0x199d, 0x1f8b, 0xba71, 0x443f, 0x1192, 0x4240, 0x43c2, 0xba2b, 0x4196, 0xb242, 0x3996, 0x0844, 0x4340, 0x4691, 0xbfe4, 0x1911, 0xb0fd, 0x402d, 0x419f, 0xad1e, 0x4038, 0x3877, 0x2e52, 0xb2a3, 0x2181, 0x35e8, 0xa034, 0x4115, 0x1b67, 0x43c1, 0x1c10, 0x2cac, 0x2a71, 0xb2f6, 0xa5dc, 0x4184, 0x4013, 0x19f5, 0x40c7, 0x4085, 0xbfc8, 0x456d, 0x4235, 0xba27, 0xbac8, 0xb298, 0xb29c, 0xbf00, 0xb051, 0x42d1, 0xb220, 0xbf03, 0xa71c, 0x44c0, 0x4327, 0x195d, 0x4092, 0x3e66, 0x40d0, 0x4345, 0x42fe, 0x41f3, 0x11cf, 0x4043, 0x1c92, 0xbacc, 0xbf04, 0x406f, 0x1da6, 0xba6a, 0x4059, 0xb01b, 0x43b6, 0x3fce, 0x43ff, 0x403e, 0x43da, 0xafac, 0x409e, 0x2481, 0x4148, 0x439e, 0x4220, 0x44a2, 0x437c, 0xbf3f, 0xb24d, 0x0870, 0x1cbd, 0x41bc, 0x4035, 0xb27f, 0xbf0a, 0x411c, 0x40f0, 0x1740, 0x4449, 0x4254, 0x430e, 0xb209, 0x41b6, 0x43f3, 0xb244, 0xbf70, 0x19bf, 0x412d, 0xa727, 0x1c54, 0x41e4, 0x43ee, 0xb093, 0xbf2b, 0xba16, 0x4063, 0x1e36, 0x35fc, 0xbae6, 0x4198, 0xbfb5, 0x01d4, 0x3182, 0x1fe1, 0x40f2, 0xbfc9, 0xb248, 0x2caa, 0x36f3, 0x40ac, 0x18b3, 0xb2a0, 0x43bc, 0xbfc0, 0xbf81, 0xb044, 0x43e6, 0x1fc6, 0xa96e, 0xb2fd, 0xa63b, 0xb002, 0x1aa8, 0xba59, 0x4307, 0x401a, 0x3bc9, 0xb2b3, 0x0cde, 0x4183, 0x4163, 0x4493, 0x430f, 0x4302, 0x42a0, 0x1d82, 0xbfd0, 0xb21e, 0x432c, 0xbfd9, 0x30c9, 0x406e, 0xa56b, 0xb05a, 0x3184, 0xba02, 0x43a5, 0xb204, 0xb229, 0xa861, 0x0096, 0x2731, 0xbf48, 0x13cf, 0x424a, 0x4564, 0xb2e8, 0x1c0d, 0x03f4, 0x434c, 0xb051, 0x0c36, 0x4286, 0xb2a2, 0xb082, 0x11ca, 0x195d, 0x35c8, 0xa2bb, 0x4649, 0x41a1, 0x423c, 0x458e, 0xb28a, 0xb0e0, 0x1dea, 0x365e, 0xbfdc, 0x33a6, 0xba77, 0x4318, 0x09c2, 0xbf9e, 0x1d19, 0x1660, 0x4213, 0x3a84, 0x1c5a, 0x1e48, 0x4174, 0x4585, 0x1cb4, 0xbf3f, 0x1280, 0x4215, 0x4036, 0x43f3, 0x4361, 0x4119, 0xb26a, 0x40a1, 0xb0e1, 0x4609, 0x3d49, 0x3257, 0xb2c3, 0xbfc0, 0x41c0, 0x41ff, 0x40ff, 0x3cec, 0x1c4c, 0x4217, 0xb20e, 0x411c, 0xb07c, 0x41ce, 0xbf79, 0x4349, 0x4392, 0x1b61, 0xba4f, 0x415e, 0x3dc1, 0x1623, 0xb0d5, 0x4099, 0x41c4, 0xbf0d, 0x1e73, 0x43e9, 0x4461, 0xb2ab, 0xba3a, 0x43a1, 0x4083, 0x428c, 0x4277, 0x0587, 0xb29d, 0xae40, 0xbad2, 0x42ad, 0x45e1, 0x407a, 0x4290, 0x409c, 0x42b5, 0x4001, 0xb2af, 0x25ef, 0x32ea, 0x1ad7, 0xbfc7, 0x42a0, 0x1cf5, 0x3c74, 0xbf00, 0x41f7, 0xb270, 0x4009, 0xaf02, 0x4331, 0x4627, 0x42ac, 0x4020, 0xbfb3, 0x42c2, 0xa284, 0xab3b, 0x40b2, 0x4319, 0xb051, 0x42cc, 0xb0ae, 0x206b, 0x413b, 0xb241, 0x2001, 0xb2b6, 0xbfa2, 0x408d, 0x34d3, 0x43db, 0xba01, 0x1a1f, 0x4396, 0x42b9, 0x4335, 0xa054, 0xa317, 0x1c89, 0x4186, 0xb01f, 0xba56, 0x4616, 0xb204, 0xb24e, 0x028d, 0x41d0, 0xb217, 0x2943, 0x402d, 0x1de0, 0x1988, 0xba51, 0xbfdd, 0xa8d1, 0x41d9, 0x40e2, 0x441a, 0x41fc, 0x4018, 0x4354, 0x430c, 0x404a, 0xb2a3, 0x42dc, 0xb230, 0xa56c, 0xbf05, 0x4121, 0x434e, 0xbae3, 0x401c, 0x2b71, 0xb2bb, 0x43b2, 0x084a, 0x19e1, 0x1f23, 0x444a, 0x40cf, 0xbf21, 0x41ef, 0xb243, 0x43ab, 0x404d, 0xb229, 0x00d5, 0xb2ce, 0x0a30, 0x41f0, 0x1c86, 0x419b, 0x1f9f, 0xa91e, 0x4164, 0xb0a2, 0x318d, 0x414a, 0xa788, 0xbf60, 0x42aa, 0x432f, 0x1754, 0xba1d, 0x4148, 0xbf0e, 0x44bc, 0xba0e, 0xabb9, 0x18ad, 0x0df7, 0x46e3, 0xbad6, 0x432d, 0x42c3, 0xb21e, 0x1ee1, 0x43be, 0x437d, 0xb23d, 0x43a1, 0x42cb, 0x0e87, 0xb226, 0xba14, 0x1cd1, 0x0971, 0x1a7d, 0x2824, 0x40bb, 0xb2bb, 0x0c39, 0xbf9f, 0xb2a2, 0x43b1, 0x0463, 0x426a, 0x18bb, 0xba13, 0x4311, 0xabcc, 0xba1e, 0x41d1, 0x3d05, 0x43f5, 0xafc8, 0x3d69, 0x405e, 0x38ae, 0xa74e, 0x401b, 0x417b, 0x378f, 0x4204, 0xbf60, 0x1c69, 0x378e, 0x0c89, 0xbfb4, 0xbfc0, 0xb235, 0x4032, 0xb218, 0x4011, 0x4188, 0x3b23, 0x4277, 0x186e, 0x3f07, 0x3860, 0x44f5, 0x46ac, 0xb019, 0x42fc, 0x41ef, 0x2835, 0x45e4, 0xbf77, 0x4038, 0x2bca, 0x42fc, 0x4303, 0xbad5, 0x00fc, 0x197c, 0x2b44, 0x39e8, 0xb2aa, 0x4674, 0xa0f7, 0xb2b3, 0xa01d, 0x1b99, 0x429e, 0x101a, 0x41c5, 0xb085, 0x4320, 0x3520, 0x1c04, 0x4283, 0x420c, 0x3cf3, 0x42f2, 0x414f, 0xb2c6, 0xbfdf, 0x41db, 0x4139, 0x18e3, 0xba4c, 0x43b7, 0xb0c2, 0x051f, 0x42cc, 0x4300, 0x1caf, 0x1ef3, 0x417a, 0x1aef, 0xbfa1, 0x43d4, 0x18cd, 0xb24c, 0x2610, 0xb22b, 0x4355, 0xb213, 0x1c9b, 0xb266, 0xbf82, 0x422b, 0xb048, 0x03e0, 0x4312, 0x4325, 0xa9c3, 0x43a1, 0xb083, 0x295f, 0x4257, 0xb221, 0x1ab7, 0xba30, 0x43b0, 0x27d6, 0x03ce, 0x403e, 0x4247, 0x1a77, 0x179e, 0x1ecd, 0x42b2, 0x1fcb, 0x3acb, 0xbf93, 0x1fdb, 0xb06b, 0xb2ec, 0x4116, 0xbad3, 0x0437, 0xb209, 0x2cc6, 0x1355, 0x402d, 0x4424, 0xbff0, 0xb0df, 0xb2ae, 0x4373, 0xbf75, 0x448c, 0x1c21, 0x1ebf, 0x40f4, 0x427d, 0x417f, 0xb205, 0x40fb, 0x406a, 0xba49, 0x40b8, 0xba45, 0x45d0, 0x419a, 0x41d2, 0x43c6, 0x1ada, 0x42bb, 0x44dd, 0xbf87, 0x1c48, 0x1116, 0x02ee, 0x41db, 0xb225, 0x3137, 0x0f02, 0xaf6e, 0x1ba4, 0x4085, 0xb2fd, 0x4699, 0xbf70, 0x4401, 0x03f6, 0xb0d6, 0xa069, 0x42f2, 0x1a31, 0x1c36, 0x421c, 0x403f, 0x3545, 0x1f3f, 0x4255, 0x1cae, 0xb0ef, 0x3f75, 0x41bc, 0xbfde, 0x40f2, 0x42cc, 0x1dbe, 0xb240, 0xb00b, 0x0f1b, 0xba61, 0x41da, 0x424a, 0xba4b, 0xbf0d, 0x2f8d, 0x418f, 0x189f, 0xb045, 0xa835, 0x2001, 0x28d8, 0xb2e9, 0x4018, 0x441d, 0xb0ff, 0x156f, 0xb240, 0x1d6c, 0xb209, 0xb067, 0xbf60, 0x4280, 0xbaee, 0xbafb, 0x41dd, 0xb040, 0xa92d, 0xbfac, 0x00f2, 0x1d85, 0x320b, 0xb2f5, 0x415e, 0x42d2, 0xb23b, 0x0f07, 0x436d, 0x4142, 0xb2fa, 0xb218, 0xba0a, 0x4088, 0xbf80, 0x19da, 0x1356, 0x43c8, 0x2171, 0x43e1, 0xbfda, 0x4215, 0x42bd, 0x4364, 0x424e, 0xb228, 0x4009, 0xbaee, 0x1bce, 0x0502, 0xb2ae, 0xb217, 0xa6c0, 0x2900, 0x4033, 0x40e8, 0xba45, 0x2caf, 0xb21f, 0xbae6, 0x37ae, 0x2684, 0x43c8, 0x31de, 0xb26f, 0x40eb, 0x4293, 0xb209, 0xbf0e, 0x41ea, 0x4290, 0x40c1, 0xb282, 0x1aa9, 0x28c0, 0x1b96, 0x1d5d, 0x13c2, 0x437f, 0xafd5, 0x41fa, 0xba21, 0x0efd, 0x412a, 0xb08c, 0x4011, 0x46ed, 0xbaee, 0x0121, 0xb250, 0x4075, 0x4386, 0x41a3, 0xbfbe, 0x21d1, 0x34ae, 0x2fce, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xebd6df8c, 0x38bf55a7, 0xab137ca7, 0x96f9989a, 0x94c70e1c, 0x27865596, 0x517c3a55, 0x121297cb, 0xb921216d, 0x142c6285, 0xbbeafbb7, 0x8e13fca7, 0x38c24dd1, 0x08c88e6c, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x00000061, 0x000000d1, 0x0ffffc61, 0xbd9eb746, 0x42616367, 0x00000303, 0x00000300, 0x1cf4f061, 0xb921216d, 0x00000000, 0x142c6306, 0x142c6285, 0x0000589c, 0x1cf4ecdd, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4338, 0x4299, 0x454c, 0x40a5, 0x4298, 0xbfa8, 0x3d94, 0x19e4, 0x345c, 0x45c0, 0x41ff, 0x1ab0, 0x19a5, 0x420a, 0x41df, 0xba39, 0x3566, 0x410a, 0x4337, 0x444a, 0x4290, 0x0663, 0xabc7, 0xa261, 0x324d, 0xbf41, 0xb222, 0x31c6, 0x10f2, 0xaee3, 0x43e0, 0x297b, 0xb284, 0x42f3, 0xbf60, 0x419e, 0x1577, 0xb2a4, 0x43a9, 0x1fcf, 0x44f2, 0x4404, 0xbaf0, 0x1f57, 0x4047, 0x1e49, 0x4187, 0xbf56, 0x40f3, 0x1380, 0x1fe3, 0x41a4, 0x4320, 0x407d, 0x428a, 0x1547, 0x439d, 0x40b0, 0x3a5b, 0x430b, 0x43f6, 0x4256, 0x195f, 0x423f, 0xb225, 0x4616, 0x4041, 0xb0c4, 0x43e0, 0xbfa6, 0xa0f5, 0xb077, 0x419f, 0xb2da, 0x1d2d, 0x40c0, 0xbfd0, 0xb243, 0x41f6, 0x41ff, 0x4390, 0x435b, 0x416e, 0x3f68, 0x0b87, 0x435d, 0xba48, 0x1af2, 0x410f, 0xa9b6, 0xbfbc, 0x4608, 0x1e08, 0x0178, 0xbfa8, 0x4019, 0xbae4, 0xb2f8, 0xa945, 0x1037, 0x421e, 0xb083, 0x422d, 0xa323, 0x34de, 0x439b, 0x407a, 0x4076, 0x4343, 0x4150, 0xb242, 0xba63, 0x302d, 0xbfd6, 0x06a7, 0x2697, 0x41d0, 0x415d, 0x4205, 0xbf70, 0x40c4, 0x1aa6, 0x42c8, 0x43b3, 0x4044, 0x4250, 0x3f43, 0x25e4, 0xbfc5, 0x19c7, 0xbf60, 0x43f1, 0xa2d9, 0xba5c, 0x4112, 0x461e, 0x1fc7, 0x1ad9, 0x425a, 0x4100, 0x2df6, 0xb284, 0x4295, 0x01fe, 0x0e8d, 0x42c2, 0x401f, 0xb277, 0xba3c, 0x4231, 0xbf43, 0x1e0b, 0x2c65, 0x12e5, 0x4119, 0x463a, 0x408e, 0x4358, 0x4356, 0x40d5, 0x4329, 0xbf71, 0x11c0, 0x0e12, 0x4022, 0x41e7, 0x467d, 0x460e, 0xb221, 0x41a2, 0x40bf, 0xbf27, 0xaeb8, 0xb2fc, 0xb0f1, 0x42ad, 0x1a47, 0x0111, 0x1a9e, 0xb04f, 0x40de, 0x4146, 0x3ad4, 0xa618, 0x408d, 0x1a2f, 0x1b24, 0x1dcd, 0xbac0, 0x3909, 0x3e34, 0x43e0, 0x4328, 0x434d, 0x4268, 0x404b, 0x1493, 0x435b, 0x45b9, 0xbfc9, 0x4439, 0x412a, 0x1de7, 0x0b4e, 0x436b, 0x4173, 0x304b, 0x447e, 0xa9c9, 0x403b, 0x42d6, 0x42a2, 0x43da, 0x1d54, 0x42eb, 0x4378, 0x464b, 0x1a7b, 0x292c, 0x3a9a, 0xbf75, 0x1ab2, 0x2b8f, 0x1db2, 0xb2d8, 0xb288, 0xbaed, 0x4202, 0x1df2, 0xb282, 0x0d5f, 0xbf05, 0x2cb7, 0xb293, 0x09e9, 0x4016, 0x403d, 0x43aa, 0x42bb, 0x4218, 0x44a0, 0x40e0, 0x4079, 0x1b6b, 0x41b3, 0x429e, 0x469c, 0x42fe, 0x1a84, 0x4481, 0xb2c7, 0xa3da, 0x42e6, 0xbfb9, 0x1f14, 0xb2ef, 0xba2b, 0x38bf, 0xadd9, 0x42fd, 0x1918, 0x1bc9, 0x0495, 0x08f9, 0x40a9, 0x3a59, 0x1821, 0x45f5, 0x4050, 0x427c, 0xb234, 0x45d0, 0xbaef, 0x43cc, 0xbfaf, 0xbad3, 0xb20a, 0xba2a, 0x43fb, 0xb26d, 0x1913, 0xb074, 0x1e82, 0x427c, 0x1bb6, 0x40d6, 0xbf04, 0xb202, 0x1ad3, 0x4041, 0xb26d, 0xb284, 0x405f, 0x1e47, 0x421f, 0x4036, 0x40e8, 0x4117, 0x414b, 0x421c, 0x40bb, 0xb2c4, 0x189b, 0xb20e, 0x4262, 0xabcd, 0xbf8c, 0xba15, 0x40ec, 0x406d, 0xb032, 0xb288, 0x2202, 0x1574, 0xb280, 0xbfa2, 0x4229, 0x0a10, 0x25cd, 0xb0bb, 0x442d, 0x41d1, 0x400e, 0x40d7, 0x41da, 0xb0f5, 0x439e, 0x1c27, 0xbfa5, 0x1037, 0xa2f5, 0x1d46, 0xbfe0, 0x4118, 0x2e5b, 0x1d76, 0xb2eb, 0x413e, 0x422e, 0x2f5d, 0xb277, 0x4255, 0x293d, 0x42ba, 0xa3c3, 0x3e34, 0xbfa9, 0x4372, 0x459b, 0xb2a6, 0xbfb0, 0x418f, 0x4039, 0x420b, 0xa411, 0x3f81, 0xbf71, 0x463b, 0x0320, 0x4136, 0xba7e, 0xa4fa, 0x41cd, 0xba36, 0xbf36, 0xb0f5, 0x415e, 0x1e0c, 0x44cb, 0xba29, 0x1e27, 0x04f0, 0x1d30, 0xbfe0, 0x41c2, 0x412d, 0xb270, 0x1d49, 0x1c0f, 0xb04c, 0x434d, 0x3c92, 0x1f4e, 0x30be, 0xade7, 0x4344, 0xba1f, 0xa02e, 0x4113, 0x1e0d, 0xb25b, 0xbfa4, 0xb001, 0x3134, 0x409f, 0x4150, 0x1b5c, 0x4007, 0xa6fa, 0x2ffd, 0x42d5, 0xba53, 0xb23f, 0x40e3, 0xb0ca, 0xa7f4, 0x18ef, 0x0936, 0x43c7, 0xbfc2, 0xab87, 0x4378, 0x3710, 0x40f6, 0x4622, 0x456a, 0x46ca, 0x46a4, 0x41b8, 0x412b, 0xba2f, 0x1e94, 0xba4e, 0xb2e4, 0x43e0, 0x1a71, 0x411d, 0x4164, 0x0b2b, 0x40bb, 0x43c1, 0x4267, 0xba1d, 0xb2ab, 0xbfc0, 0x0624, 0xbfbd, 0xb02d, 0x43e9, 0x18ad, 0x3146, 0xa882, 0x227f, 0x4382, 0x4314, 0x19cc, 0xb25d, 0x465b, 0x43c9, 0x1800, 0x4169, 0x426f, 0x427a, 0x182c, 0x41ed, 0x2dfb, 0x1239, 0x41a2, 0x1aae, 0x3887, 0x42c2, 0x4352, 0x46d2, 0x4362, 0xb27a, 0xbf5b, 0x3437, 0xb09b, 0x41fa, 0x1fa2, 0xb2ee, 0x3c06, 0x437b, 0x2315, 0x410e, 0x404e, 0xbf78, 0x4231, 0x4387, 0x4258, 0x32ba, 0xb264, 0x43c0, 0xbfc0, 0x4244, 0x46b2, 0x408a, 0xba45, 0xbf07, 0x1a65, 0x19c0, 0x448b, 0x3b5f, 0x419d, 0x4653, 0xbaf2, 0x423f, 0x4118, 0x4138, 0xbaf1, 0xb2ed, 0xb26c, 0x2947, 0x41d4, 0xb265, 0xa04c, 0x2b65, 0xba2d, 0xbad6, 0x418c, 0x3bc0, 0xb26b, 0x1ebd, 0x43e5, 0xbfd4, 0x43ae, 0xb271, 0x4226, 0xb03c, 0x409a, 0xb21f, 0x42e8, 0xbf1b, 0x449b, 0x4310, 0xb2a4, 0x245b, 0x4322, 0xa98a, 0x41eb, 0xb284, 0xb2ac, 0x410f, 0xb2cc, 0x439f, 0x437f, 0x4120, 0xba47, 0x190f, 0x4313, 0x424d, 0x36a7, 0x432a, 0x45e3, 0xb263, 0xbf13, 0xb0c7, 0x423a, 0x437b, 0x41a1, 0x42b1, 0xba59, 0x43e4, 0x40a7, 0x4679, 0x438a, 0xad17, 0x43b4, 0x1420, 0xb229, 0x42b8, 0xb2cf, 0x1a2e, 0x3665, 0x08c0, 0xbf9d, 0x32d9, 0x0937, 0xb0be, 0x43c4, 0x4220, 0x1d47, 0x40f6, 0x4373, 0x4257, 0xafdc, 0x23cf, 0x41f4, 0x3b0e, 0x40ce, 0xb230, 0xb255, 0x4664, 0x1dcb, 0xb291, 0x42fa, 0xbf91, 0x1d7c, 0xb020, 0xba72, 0x4007, 0x3978, 0x4054, 0xbaed, 0x1bf6, 0x31ff, 0x46cc, 0xa2f6, 0x3bd0, 0xba7e, 0x193e, 0xb287, 0xb031, 0xbae0, 0x40d3, 0xb0d4, 0xb0ae, 0x02d5, 0x434e, 0x403d, 0xafe6, 0x3faa, 0x411c, 0xbfbd, 0x0cd3, 0x119f, 0x4394, 0xbacc, 0xb2e1, 0x4180, 0x4351, 0xbaeb, 0x43a4, 0x43f9, 0x4138, 0x2b8b, 0x2caf, 0x413e, 0x41a8, 0xbf48, 0x42c4, 0x012d, 0x4172, 0x422d, 0xb08b, 0x4295, 0x19b4, 0x1fd9, 0x40d1, 0x45e2, 0x4228, 0x424b, 0xbad8, 0x40ee, 0x40e9, 0x439d, 0xbaef, 0xbadf, 0xbf2a, 0x40c8, 0x192c, 0x4408, 0x401c, 0x2766, 0x30bd, 0x40f5, 0x4387, 0x3c4a, 0xbf0f, 0xba2e, 0xb242, 0x4624, 0x1a7b, 0xb27e, 0xba6c, 0x4169, 0x419f, 0x406b, 0x4427, 0x0415, 0x3065, 0x4015, 0xb2eb, 0x1cdd, 0x4222, 0xb225, 0xa0ed, 0x1826, 0xba16, 0x437d, 0x0dae, 0xb0fe, 0xbfb6, 0xb256, 0x1666, 0x42df, 0x45b4, 0x0c1b, 0xba18, 0xb29b, 0x4586, 0xbf32, 0x1bfe, 0xbad5, 0x422c, 0xbf2d, 0x1cc6, 0x1e0d, 0xb0c9, 0x437c, 0x342f, 0x429d, 0x4103, 0xbf7c, 0x41a5, 0xb02f, 0x4035, 0x0a5d, 0xbac1, 0xac84, 0x4071, 0x0d2f, 0x4235, 0x4203, 0x44b8, 0x0677, 0xba51, 0xb0d1, 0xbae3, 0xae14, 0xbf49, 0xb29a, 0x41ff, 0x1298, 0x1e6a, 0xb286, 0x4156, 0x3c1e, 0x4088, 0x075d, 0x414c, 0x4343, 0x4197, 0xa990, 0xb085, 0xacb8, 0x1984, 0xbfe0, 0xa3f4, 0x423f, 0xb2e3, 0x41d6, 0x1d5d, 0xbf65, 0x407c, 0x4269, 0x348b, 0xb2af, 0x43e5, 0x4169, 0x42e6, 0x4258, 0x434b, 0xbfc0, 0x4385, 0x3551, 0x4452, 0x41fd, 0x461c, 0xb2c5, 0x1e89, 0xba0b, 0xbf00, 0xacdf, 0xb2c0, 0x4241, 0x4247, 0x421b, 0x436f, 0x43d2, 0x14e5, 0x45a5, 0xbf1b, 0x1e6b, 0x4195, 0x45f2, 0xb27b, 0x4125, 0xb29f, 0xb26e, 0x4010, 0x422a, 0xa1a0, 0xbafc, 0xaa80, 0x1c8f, 0xb06f, 0x43a8, 0x42ac, 0xbae1, 0x415f, 0x117d, 0x4339, 0xb250, 0xbfe2, 0x4263, 0xb2bd, 0x409c, 0xb269, 0x19a4, 0xbfc0, 0x0fdf, 0x41f6, 0x4621, 0x4044, 0x40f5, 0x2e6e, 0x4221, 0x46d3, 0xbf9a, 0xba60, 0x41bf, 0x424a, 0x43bf, 0xbf9c, 0x26c7, 0x4242, 0x1d65, 0xb050, 0x4612, 0xa5c5, 0x1da4, 0x43a3, 0xb0fb, 0x444d, 0xba09, 0x3649, 0x4062, 0x1293, 0xba5b, 0x2db9, 0x43dc, 0x4333, 0x1a29, 0xbf0d, 0x0c2d, 0x4306, 0x4001, 0xbf70, 0xb04a, 0x43b5, 0x32dc, 0xba2b, 0x4367, 0x412e, 0x191f, 0x1a9a, 0x4027, 0x4139, 0x1951, 0xbf00, 0x4239, 0x40e2, 0x42c2, 0x40f2, 0x0880, 0x43f8, 0x1da1, 0xbf99, 0xb245, 0x464e, 0xbaf4, 0x4220, 0x4653, 0x43b9, 0x3e09, 0xbfc5, 0xba26, 0xb287, 0x1747, 0xb231, 0xb2ff, 0x21a4, 0xba1e, 0x44b4, 0x44a3, 0x4281, 0x4113, 0x40ab, 0x02cd, 0xb0f6, 0x40c1, 0xba59, 0x42a6, 0xb2e1, 0x1ed0, 0x3728, 0xba7f, 0x425d, 0xbf82, 0x2b31, 0xb03e, 0x4133, 0x1ce6, 0x413d, 0x4027, 0xb2c5, 0x1e77, 0x424d, 0x4051, 0xa275, 0xb00b, 0x4090, 0x408c, 0xbf79, 0x0e9e, 0x0767, 0x196e, 0x424a, 0x46fa, 0x2756, 0x24c8, 0x2108, 0x09cd, 0x1b81, 0x421a, 0xb256, 0xb007, 0x0ad0, 0x40cd, 0x4114, 0x404a, 0x400e, 0x40d8, 0xbf70, 0x4476, 0x01e4, 0x40e0, 0x2e5d, 0xbf9a, 0x2d03, 0x45c2, 0xbf00, 0xb0c5, 0x4585, 0x40e6, 0x42dd, 0xbac3, 0x41c0, 0x1a33, 0x1f44, 0x4347, 0x41ec, 0x20fe, 0xbf69, 0x3e36, 0x41e0, 0xadd6, 0x1aee, 0x2879, 0xb258, 0x432a, 0x2c20, 0x1ba8, 0x0ba8, 0x4361, 0x42bc, 0x408c, 0xbf9a, 0x4127, 0x42e9, 0xba00, 0x4127, 0x0bd8, 0x421a, 0x1d0b, 0xb2c6, 0xb0b8, 0xbf65, 0x2eb1, 0x43ee, 0x43e4, 0x4103, 0xb041, 0x40e0, 0xa3f4, 0x421a, 0xb284, 0xba0c, 0xae98, 0x42b7, 0x2272, 0x1c87, 0x3cf3, 0x41c5, 0x423b, 0xb202, 0x1f4a, 0x43a6, 0x124a, 0xb20e, 0xb0b4, 0x40e4, 0x40e9, 0x4135, 0xa232, 0x18a6, 0xbf48, 0x461c, 0xba4a, 0xba16, 0xa76b, 0x4106, 0x4289, 0x43fe, 0xba65, 0x4127, 0x0a75, 0xaddb, 0x2441, 0x1904, 0x1705, 0xb2c6, 0x1503, 0xbff0, 0x43c1, 0xbf52, 0xbac9, 0xb241, 0xbf80, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2fdaaeb9, 0xb79696f8, 0x5b75e8b3, 0xb8bc6fba, 0xcb255eac, 0xe7fd35fd, 0xddb0f0b1, 0x6a6b7524, 0x61dac898, 0x5adfbd3a, 0xe9b18555, 0xf3de49cb, 0x123c8a0c, 0x745a0959, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x00001400, 0x00000000, 0x00000000, 0x00000000, 0x00001441, 0x00000000, 0x00000000, 0x00000000, 0x61dac89a, 0x5adfbff0, 0x0000171e, 0x00000900, 0x5adfbff0, 0x745a0295, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4177, 0x1819, 0x2414, 0x1b72, 0x43dc, 0xbf80, 0xba41, 0xa052, 0xb20b, 0xb083, 0xa090, 0x4329, 0xa61b, 0x43d1, 0x46cc, 0x1a14, 0xbf68, 0x46fa, 0xb233, 0x1a2a, 0xb2b8, 0xa819, 0xb037, 0xba10, 0x4042, 0xb298, 0x41fb, 0x409a, 0xa96a, 0xbfbd, 0x42cf, 0x4253, 0x408a, 0x405b, 0x42f3, 0x4059, 0xba0f, 0x4567, 0x1cdc, 0x46fc, 0x408c, 0x4422, 0x35df, 0x1b9c, 0x2518, 0x40fd, 0xbfc0, 0x427e, 0x088c, 0x12b3, 0x420e, 0x1f21, 0xb2db, 0x4082, 0xba04, 0x43d5, 0x417d, 0x40cc, 0xbfe1, 0xb2e4, 0x41cb, 0xb26a, 0x0fc6, 0x41fd, 0xa87b, 0x19ba, 0xb082, 0x4007, 0x403f, 0xb051, 0xb2ae, 0xbfb4, 0x41ae, 0x30da, 0x3846, 0x428b, 0x42d0, 0x4346, 0x42c7, 0x432c, 0x4130, 0x1e5c, 0x456b, 0x40d8, 0x4220, 0x40aa, 0x4158, 0x431d, 0x44bb, 0x1bcc, 0x04bf, 0x1cfa, 0x4025, 0x119d, 0xbfbb, 0x44ed, 0xa46d, 0x405d, 0x444c, 0x1c30, 0x4203, 0x45c5, 0x41d8, 0xb032, 0xbaf2, 0x0ea6, 0x45da, 0x4201, 0xb08f, 0xba40, 0x46fd, 0xb036, 0x4299, 0xb204, 0x41b6, 0x2913, 0x3bba, 0xba0b, 0x0540, 0x417a, 0xbafd, 0xbf73, 0x4048, 0xb25b, 0x1f4c, 0x429e, 0x3765, 0x1070, 0x1cd1, 0xa19b, 0x462d, 0xae77, 0x4138, 0x4417, 0xb26e, 0xade1, 0x1fc2, 0xb066, 0x43e2, 0x45e4, 0xba67, 0xb2dc, 0x1ce1, 0x4618, 0x41f8, 0x41cc, 0xbfd9, 0x21de, 0xba18, 0xbad6, 0x1ced, 0x4268, 0x408e, 0xbf42, 0xb203, 0xb22f, 0x41a0, 0xb01e, 0x148e, 0xb213, 0x1f18, 0xb066, 0x4220, 0x402f, 0x2024, 0xb0d0, 0xb029, 0x4013, 0xb2e6, 0xaad2, 0x4412, 0x2186, 0x08b7, 0xb017, 0x2c68, 0xac81, 0x03a5, 0x410d, 0x1ad9, 0x4385, 0xbf8f, 0x423e, 0x43a9, 0x4540, 0xb206, 0x1913, 0x45f4, 0x410a, 0xba27, 0xb03a, 0xa1a3, 0x1bab, 0x40a9, 0x1e67, 0x1d41, 0x45a1, 0x43a2, 0x433c, 0xbf3e, 0x447b, 0x40fa, 0x42f1, 0xbfba, 0x3ad7, 0x1a64, 0x436a, 0x4236, 0xb244, 0x423e, 0x46ca, 0x20e3, 0xb2e8, 0x433e, 0xb2e5, 0x2bee, 0x42bb, 0xa98d, 0xad9c, 0x4386, 0x2257, 0xbff0, 0x42c6, 0xbf33, 0x4493, 0xb283, 0x42fe, 0x2141, 0x4036, 0x45ee, 0x4021, 0x4094, 0x1804, 0x415d, 0xbf83, 0x1901, 0xba69, 0x40b4, 0x2756, 0xa4e3, 0x46f5, 0x44e4, 0xbf08, 0x409c, 0xb28d, 0x432e, 0x40b3, 0x2d20, 0x3e1f, 0xbfb6, 0xada3, 0xba6b, 0x1a51, 0xb230, 0x417d, 0xb08f, 0x4147, 0xb204, 0x1c74, 0xbfc0, 0x40d6, 0x297f, 0x42da, 0x4647, 0x0785, 0x41ed, 0x43e8, 0xbfc3, 0x1814, 0x40f2, 0x44d1, 0x4336, 0x40cd, 0x4310, 0x0796, 0x0448, 0x0afd, 0x2b10, 0x454c, 0x409c, 0x4071, 0x4190, 0x022b, 0x42aa, 0x4077, 0x3044, 0x43c2, 0xb201, 0xbf8d, 0xba41, 0x4057, 0xb057, 0x1b40, 0xb24a, 0xa1a7, 0x4307, 0x0539, 0x416a, 0xba50, 0xbf87, 0xba7b, 0x43a4, 0x2500, 0xba5b, 0x43bb, 0x42f3, 0x4611, 0x4280, 0xbac7, 0x41ae, 0x1ca3, 0x43b2, 0xa44d, 0xba30, 0x0ab4, 0x1dcd, 0x461a, 0x435a, 0x025a, 0x432a, 0x20e9, 0xa954, 0x4392, 0xba42, 0x424c, 0x41b8, 0xb235, 0xbf51, 0x4185, 0xb006, 0x4049, 0xb26a, 0x417b, 0xb0e2, 0x19d5, 0x3021, 0xbf00, 0x19dd, 0x4005, 0x4153, 0x1c3e, 0x411b, 0xb272, 0x4222, 0xb249, 0x45e0, 0xb258, 0xb0fc, 0x193e, 0x42eb, 0xb2b2, 0x3edb, 0x4186, 0xb0cb, 0x302a, 0xbf36, 0xb289, 0x4139, 0x4193, 0xb08c, 0x42b8, 0xb254, 0x4421, 0x411c, 0x1ef1, 0xb041, 0x46e3, 0xbfb8, 0x43c0, 0x4015, 0xa898, 0xb20e, 0x42ce, 0x42ac, 0x4067, 0x43d6, 0x42a2, 0xba33, 0xbafe, 0x4208, 0x4142, 0x4159, 0x328e, 0x31d0, 0x40f7, 0x318d, 0x191f, 0x4393, 0x21e0, 0xbf15, 0x4350, 0x19a5, 0x44b5, 0x1a52, 0x18d9, 0x4298, 0xb2f1, 0xb283, 0x1d3c, 0x184a, 0x1919, 0x4185, 0x389d, 0x1fcf, 0x4075, 0x4354, 0x426d, 0x412b, 0x08c8, 0xaa20, 0xb0fc, 0x430a, 0x4371, 0xb036, 0xba55, 0x401d, 0x0a8a, 0x4216, 0xbf76, 0x418a, 0x438a, 0x4239, 0x41f2, 0x43bf, 0x1183, 0x163f, 0x1d04, 0x44d4, 0x4142, 0x40ec, 0x2292, 0x4216, 0x114a, 0x402c, 0x4182, 0x4274, 0x4175, 0xbf34, 0x0866, 0xba17, 0x425d, 0x408c, 0x455f, 0x4226, 0x352f, 0xbfd0, 0x41aa, 0xba23, 0xb22e, 0x4332, 0xb2dc, 0xb0f0, 0x41c1, 0x4244, 0x1935, 0xba0a, 0x434a, 0x4102, 0xbf3b, 0x4006, 0x40df, 0xb0f8, 0x4056, 0x400f, 0x43f9, 0x43ce, 0x42fe, 0x40d4, 0x28b0, 0x429b, 0xbf74, 0x182a, 0x41cb, 0xb286, 0x43e0, 0xbaf2, 0x4048, 0x4290, 0xb01d, 0x42a0, 0x4607, 0xb05c, 0x4350, 0x1c44, 0x332b, 0x4472, 0x41e1, 0x4367, 0x4326, 0x246e, 0x424d, 0xacbc, 0xb25d, 0x42fc, 0x1cfd, 0xbfba, 0x1862, 0x4434, 0x4642, 0x1a03, 0xac03, 0xb011, 0xb268, 0x0d38, 0x4269, 0x447e, 0x40d4, 0xb02c, 0x08bf, 0xb2b2, 0xbfa0, 0x410b, 0xb284, 0x1c4a, 0x414b, 0xbaf7, 0x4177, 0x3a36, 0xbf4a, 0x1dd8, 0x41c6, 0x4094, 0x4586, 0xbfb9, 0xb0e7, 0x43c0, 0x45dc, 0x0f5d, 0x164f, 0x0ba3, 0x1ddc, 0x0097, 0xb29b, 0xba38, 0xba74, 0x4381, 0xb0da, 0x4250, 0xb2a1, 0x41d4, 0x4081, 0x43bb, 0x4406, 0x345b, 0x3396, 0x1de6, 0x1962, 0x1b0e, 0xbf09, 0xb297, 0x44a0, 0x2a36, 0x3650, 0x427f, 0xb26b, 0xb271, 0x438f, 0x4339, 0x4129, 0x376e, 0x4296, 0x1ea5, 0xbfd8, 0x4225, 0x41e9, 0xbadc, 0x4033, 0x06d5, 0xb204, 0x4004, 0x43db, 0xb056, 0x1a4d, 0x168e, 0xbf13, 0xb2c3, 0x40b7, 0x4099, 0x0e40, 0x04b2, 0x2491, 0xbfb0, 0xba29, 0x40ab, 0x45c5, 0xb235, 0x4682, 0x44cd, 0x4336, 0xa9f5, 0x3173, 0xb2ba, 0xb2ae, 0x410b, 0x2490, 0xba68, 0xb05a, 0x4243, 0x4046, 0xb29e, 0xb26d, 0x412d, 0xbfc5, 0x3c36, 0x43ad, 0x4233, 0xb276, 0x1bbe, 0x4130, 0xad7d, 0x43e8, 0xbf86, 0x42b4, 0xb201, 0x39a2, 0xa298, 0x422d, 0x420e, 0xbae0, 0x42ba, 0x419b, 0x4373, 0x3c8a, 0x1b84, 0x2f19, 0x4215, 0xbae3, 0x4301, 0xbfd4, 0xb271, 0x4071, 0x184c, 0xbaf9, 0x1a13, 0x46d8, 0xbfc7, 0xb230, 0x46e9, 0x3397, 0x44d2, 0x2603, 0xba1b, 0x217b, 0xb0fd, 0x42c2, 0x0680, 0x1e31, 0xb203, 0xbac1, 0x2ed6, 0x41cc, 0xb232, 0x423d, 0x24a3, 0xbf43, 0x3980, 0x4273, 0x4234, 0xa0f4, 0xb270, 0x401e, 0xad39, 0xb045, 0x1bbb, 0x1f36, 0xb0ac, 0xba3f, 0xb227, 0x4433, 0xb001, 0xb060, 0x1679, 0x287b, 0x4134, 0xb2b5, 0xabaa, 0x1a26, 0x44a2, 0x42fe, 0xb293, 0xbf8d, 0x4198, 0x1f37, 0xb2c9, 0x425c, 0xb2e3, 0xaaa2, 0xbf26, 0x4306, 0xb013, 0x1e28, 0x1668, 0x411a, 0x4370, 0x0257, 0x1efc, 0x4060, 0x4243, 0x4419, 0xb211, 0x1ade, 0x413e, 0xbfd6, 0x4329, 0x4305, 0x423c, 0x1a58, 0x4075, 0x4377, 0xb24c, 0x4055, 0x1b29, 0xb212, 0xba6d, 0x4011, 0x265d, 0xaa17, 0xa9c1, 0xb2ed, 0x4011, 0x31b0, 0x421d, 0x4399, 0xb0da, 0x4040, 0x4160, 0xbfd0, 0xbad9, 0xbfb4, 0x45f3, 0x4329, 0x434a, 0x40fe, 0x432e, 0xb290, 0x4043, 0x4077, 0xba4d, 0x238f, 0x40ea, 0x46da, 0x3def, 0xbf0e, 0x4565, 0x180b, 0x25be, 0xba29, 0xb0cf, 0x421a, 0xba15, 0x0c57, 0x1ca8, 0x4202, 0x4369, 0x3400, 0xb230, 0xb010, 0x1e9e, 0x4185, 0xaa1b, 0x4069, 0x41f7, 0xba0a, 0x1bbf, 0x401d, 0x1bff, 0xbf2a, 0xb24b, 0xb06f, 0x24ba, 0xa781, 0x42f6, 0xb26b, 0x2a8a, 0xba6c, 0x2c03, 0xb2e8, 0xb24a, 0x42aa, 0x45e3, 0x40a0, 0x40e2, 0x4044, 0x406e, 0x4011, 0xb2b0, 0xbfa7, 0x4155, 0xb20c, 0x421a, 0xb011, 0x06e8, 0x1c54, 0xb20d, 0xafeb, 0xbf90, 0xbfa1, 0x1d96, 0xb2c9, 0xbae8, 0x4123, 0xb29f, 0xac33, 0xbfc0, 0xb2d6, 0x119e, 0x41fa, 0xba79, 0x436a, 0x41f6, 0xbadc, 0x406d, 0x08c6, 0xbafc, 0x4247, 0x419d, 0x40e4, 0x4032, 0xbf9d, 0xb2b5, 0x4301, 0x1fd1, 0x1878, 0x190b, 0x41b6, 0x1a06, 0x1884, 0x413e, 0x1e69, 0xb26e, 0x400f, 0xba25, 0xacaa, 0xaa90, 0xb2f7, 0x41be, 0x414d, 0x2fe7, 0xb01b, 0xbf6b, 0xbf80, 0x4424, 0x1ac3, 0x2b49, 0x42d4, 0xba6d, 0x00b4, 0xaccd, 0x2bba, 0x42b0, 0x4317, 0x408d, 0x4088, 0x1c63, 0x41f2, 0x340f, 0x41be, 0xbfbc, 0xba04, 0xb2a4, 0xb03e, 0x41f3, 0x43d5, 0x3fb9, 0x4130, 0x40b7, 0x1a94, 0xba2c, 0xb24d, 0x40ac, 0x4397, 0x1e2d, 0x0f84, 0x1ac9, 0x4094, 0xba79, 0x4014, 0x345f, 0x4369, 0xa53c, 0xb236, 0x417e, 0xba2d, 0x0ddf, 0x41eb, 0xbfd8, 0xb236, 0x4430, 0x1c1f, 0x408b, 0xb260, 0xab37, 0xb252, 0x40e5, 0x41fe, 0xb289, 0x4157, 0x416e, 0x1cc9, 0xbfb0, 0x40ba, 0x4358, 0x1f8c, 0x40fb, 0xb009, 0xb296, 0x3130, 0x4042, 0x43ca, 0x433c, 0xbf12, 0x4349, 0xbfa0, 0x4092, 0xb0f5, 0xb2a1, 0x4112, 0x2f71, 0xba43, 0x4480, 0x4075, 0x4066, 0x09fb, 0x404b, 0xb263, 0xbfa9, 0xb224, 0x41dd, 0xb03f, 0x19ba, 0x45e5, 0x1a02, 0x40ef, 0x3fa2, 0x1ee5, 0x0f21, 0x4175, 0x4009, 0x4364, 0x0066, 0x4243, 0x1b6e, 0x3b34, 0x40fd, 0x058f, 0x4021, 0xa4f2, 0xb2c8, 0xbf3a, 0xbfe0, 0x402b, 0x4633, 0x2f79, 0xb027, 0xb066, 0x4046, 0x186b, 0x4288, 0x41ef, 0x28df, 0x3cb6, 0x0efe, 0xba01, 0x42a7, 0x3d58, 0xbae9, 0x41eb, 0x1c69, 0xbae0, 0x434d, 0x44f4, 0x41fc, 0x32c2, 0xb028, 0xbf71, 0x40a0, 0x2fd0, 0x1eea, 0xb02c, 0x4393, 0x3037, 0x2793, 0x405e, 0x25a9, 0x2f3c, 0x3e1b, 0xb2fe, 0x41ba, 0xbfd7, 0xba39, 0xb249, 0x4150, 0x43c2, 0x43e1, 0xba14, 0x412c, 0xb003, 0x0b24, 0x4243, 0x306c, 0x436a, 0x1a6a, 0x1cff, 0x1599, 0x43b2, 0x4306, 0x4264, 0x424f, 0x22b4, 0xb2bb, 0x418e, 0x4021, 0xbf0f, 0xb243, 0x1971, 0x43d7, 0x435d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xa631da32, 0x8b25e0fa, 0xc036f7d7, 0x14e1473e, 0x0d42c45b, 0x1272500d, 0xe6b2d752, 0xafd08c78, 0xdf628cd7, 0xcc7f4ab5, 0xc596f8ca, 0x035ba176, 0x9004931a, 0xcc587fb1, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x000000a3, 0x00000000, 0x000000b4, 0xffffffa3, 0x00000000, 0x000000a9, 0x000000b3, 0x00000001, 0xe32471b4, 0xcc7f08d4, 0x000020a4, 0x000020a4, 0xcc7f6b59, 0xcc7f0a40, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1eba, 0xbf6c, 0x40f9, 0x42e2, 0xbf00, 0x4352, 0xba71, 0x415f, 0x4121, 0x4370, 0x18bf, 0xb228, 0xb21f, 0x437a, 0x410e, 0xa9be, 0x4094, 0xbf01, 0x1bfc, 0x417c, 0x1d9a, 0x342e, 0xb058, 0xa3e9, 0x15d8, 0x42da, 0xbfd6, 0x414f, 0x43b2, 0x46dd, 0x41c9, 0x46d0, 0x46f2, 0xba5f, 0x1558, 0x4221, 0x4156, 0x42f2, 0x38cc, 0x1fa0, 0x436a, 0xb062, 0x4070, 0x19c3, 0xb205, 0x4206, 0xba6a, 0x42be, 0x4388, 0x42ae, 0xb29c, 0xbf1b, 0xbae8, 0x00da, 0xa33e, 0x1ca7, 0x40fd, 0xbf81, 0xb2d7, 0xba69, 0x1ea1, 0xb0bb, 0x0645, 0x1cb6, 0xb27a, 0xbfb0, 0x28b8, 0xbad7, 0x40e0, 0x4253, 0xbaf6, 0x1466, 0x318a, 0xa4ad, 0x1f98, 0xbf4a, 0xbaf1, 0x4350, 0x08cd, 0xbfb6, 0xb047, 0x0b95, 0xb250, 0x405b, 0x4125, 0x42f5, 0x1da1, 0xa969, 0xbfc4, 0x400d, 0xb2b6, 0x4321, 0x43ec, 0xb279, 0x41e6, 0x066a, 0x1e01, 0xbf6c, 0xb260, 0xb099, 0x4237, 0x4192, 0x4453, 0xb258, 0x40c4, 0x1f86, 0x1f54, 0x1da2, 0xb0cf, 0x460b, 0x40e3, 0xbfc0, 0xb27b, 0xba0a, 0x424d, 0xb099, 0x1e52, 0xb213, 0x4073, 0xb091, 0x4456, 0x4549, 0xbfc0, 0xbfb5, 0x435a, 0x1eb5, 0xb2fc, 0xb2b8, 0x1ccd, 0x0313, 0xbacd, 0x1835, 0x4048, 0xba3e, 0x423e, 0xb0ef, 0x33e7, 0x406b, 0xb294, 0x4176, 0xaaa3, 0x43e9, 0xba3e, 0x4592, 0xb2b8, 0x41ee, 0x4166, 0x40d4, 0x407a, 0x41b4, 0xb08c, 0x466c, 0x466b, 0xbf84, 0xb256, 0x05dd, 0x402d, 0x400d, 0xae81, 0x448a, 0x409e, 0xb2bb, 0x4330, 0xb2a2, 0x3a1d, 0x155f, 0x4375, 0x42a3, 0x4042, 0x1e6c, 0x4043, 0x437f, 0xbf5d, 0x0a83, 0x0278, 0xbaef, 0xb225, 0xb2c6, 0xa084, 0x42c4, 0xb044, 0xb276, 0x4254, 0x1ff5, 0x3ecd, 0x42d5, 0x40ab, 0x1d0b, 0xbf5b, 0xa84c, 0x42fa, 0x4153, 0xb20f, 0x43f5, 0xb0ae, 0x40a6, 0xba43, 0x40d1, 0x4031, 0xb201, 0x40a2, 0xa821, 0x41ba, 0x057c, 0x4158, 0xbfdc, 0x4058, 0x0e46, 0x41b1, 0x1ede, 0x430b, 0x427a, 0x2579, 0xb210, 0x10b9, 0x40ac, 0x144d, 0x277e, 0x4361, 0x407b, 0x4292, 0x438e, 0x1c52, 0xbf29, 0x43da, 0x4175, 0x40cd, 0x422e, 0x1bce, 0xbafe, 0x4048, 0x4392, 0x1f58, 0xb06f, 0x4172, 0x140a, 0x42c1, 0x4129, 0x3dc7, 0x43c9, 0xbf62, 0x1cfc, 0x1c76, 0xb258, 0xb2a7, 0x0c27, 0x428d, 0x419d, 0x4247, 0xbfa0, 0x4003, 0x462a, 0xb20e, 0xba7f, 0x410b, 0xbf49, 0xb063, 0x406f, 0x1d81, 0x0426, 0x1ecc, 0x37da, 0xba5c, 0x265f, 0xba45, 0x4623, 0xb2ce, 0x437a, 0x3263, 0x4216, 0x46d2, 0xbf60, 0xb2b3, 0x438b, 0x43a6, 0x127c, 0x2a4a, 0xb087, 0x418d, 0xbfdc, 0xbaf9, 0x2d49, 0xbad8, 0x4555, 0x4034, 0xbf3c, 0x4163, 0x4195, 0x1af2, 0x436b, 0xba0c, 0x4645, 0xb006, 0xba20, 0x2774, 0x0b44, 0x4191, 0x40c4, 0x3255, 0x4316, 0xad9e, 0xb281, 0x2148, 0x19bf, 0x2cb2, 0x41da, 0x4245, 0x4304, 0xbf1c, 0x4266, 0x40d7, 0x1d42, 0x420e, 0x0277, 0x4072, 0x1f21, 0x4117, 0xbfcb, 0x4020, 0xbaf0, 0x3750, 0x438c, 0xb249, 0xb232, 0x42bf, 0x087b, 0xbf60, 0x20a3, 0x2159, 0x4376, 0x401f, 0xac09, 0x41b6, 0xb2eb, 0x44e0, 0x44f5, 0x4180, 0xbf70, 0x4027, 0x4376, 0x4249, 0xbf76, 0x4013, 0xa685, 0x4178, 0x1b24, 0x3bf1, 0x43fc, 0x4564, 0x4359, 0xbf60, 0x3001, 0xbf7a, 0x42b3, 0x431f, 0x0fb3, 0xba60, 0xab2c, 0x4084, 0x4382, 0x1b93, 0x445d, 0x0fa1, 0x42c8, 0xaf44, 0x326c, 0x2fce, 0x41fa, 0x46ed, 0x435a, 0x43e7, 0x3b92, 0x41ad, 0x4387, 0x4148, 0xb243, 0xba0f, 0x468d, 0x41d3, 0x4283, 0x3a35, 0xbf74, 0xb280, 0xb26e, 0x4288, 0x1b27, 0x3713, 0x460d, 0x4608, 0xa4ad, 0x40af, 0x2de0, 0x111b, 0xbafa, 0x421f, 0x432d, 0xba3b, 0x4353, 0x418e, 0xbfa8, 0x41f0, 0xba53, 0xb0e1, 0xb0b1, 0x13bc, 0xba4a, 0x42d6, 0xb26d, 0x468c, 0x4064, 0xb048, 0xa308, 0x4171, 0xbf7a, 0xa025, 0x4079, 0x287b, 0xb2d9, 0xafc6, 0x1287, 0x3e24, 0xbfc0, 0x4383, 0x0175, 0xb229, 0xb098, 0xba56, 0x1cdf, 0xb216, 0x3b28, 0x4083, 0x41cd, 0x4233, 0xb0fb, 0xba6a, 0x40f7, 0xb092, 0x4151, 0x4332, 0xbf01, 0x431e, 0x4064, 0x3e5b, 0xbae3, 0x41a0, 0xbad8, 0x4396, 0x16ef, 0xb2ac, 0x43db, 0xba05, 0x42a4, 0x287a, 0xbfe4, 0x423c, 0x4093, 0x4363, 0x428c, 0x4245, 0xbf1d, 0xb288, 0x4170, 0x44fd, 0xb20d, 0x1ba5, 0x41df, 0x404e, 0x1bf4, 0x363f, 0x4235, 0x2879, 0x3929, 0x1c19, 0x462f, 0x40ea, 0xb291, 0x1ba1, 0x40fa, 0x1144, 0xb291, 0xbf16, 0x4355, 0x410a, 0x4359, 0x2f7f, 0xbf4a, 0x4315, 0x18ae, 0xb28b, 0x013c, 0x2be1, 0x423a, 0x426b, 0x2686, 0x0bda, 0xb070, 0x1a04, 0x433b, 0x4000, 0x4374, 0x122a, 0xab72, 0x240a, 0x43cd, 0x1fcc, 0xb24a, 0x3817, 0xb28a, 0x43a1, 0xbfbf, 0x41ad, 0x4353, 0x1ca2, 0x0f5f, 0xb2cc, 0x335c, 0xbfdc, 0x4564, 0x42f9, 0xb27b, 0x4334, 0x4164, 0x3681, 0x42f9, 0xbf4e, 0x4638, 0x4449, 0x43ed, 0x19bb, 0xb215, 0xbfe0, 0x423f, 0x2108, 0x4383, 0xb228, 0x4652, 0x1dc3, 0xba3f, 0xbfc4, 0x438b, 0x40e8, 0x4567, 0x418c, 0x40fc, 0xba28, 0x40b1, 0x1e4c, 0xbfba, 0x3458, 0xb0ab, 0x2ad2, 0x45e9, 0x1a0c, 0x43b5, 0x4198, 0xab8d, 0x41f9, 0x1d52, 0x4436, 0x2e46, 0x436d, 0xba14, 0x449b, 0x1c84, 0xb2bc, 0xba6a, 0xbf28, 0x382e, 0x41c1, 0x4630, 0x40fb, 0xb25a, 0xa5ff, 0xbf66, 0x407f, 0x1ba9, 0x40f7, 0xba4d, 0xad6a, 0x42d4, 0x416a, 0x41ec, 0x058b, 0xb03f, 0x400b, 0x41de, 0x42f8, 0xb239, 0x43a1, 0xb0bc, 0xb278, 0x3723, 0x2aba, 0x08ce, 0xbac4, 0x4117, 0xba71, 0x43e7, 0x4160, 0x40f9, 0xbfb5, 0xb204, 0x46aa, 0x2031, 0x427d, 0x434d, 0xbac1, 0x41c2, 0xba36, 0x3418, 0x4154, 0x0077, 0xbf8f, 0x1ce1, 0xb299, 0x403b, 0x4327, 0x417e, 0x2a98, 0x41f9, 0x44fb, 0x4284, 0x3e49, 0x4338, 0xb20a, 0x0cc0, 0xb24c, 0x4379, 0x0318, 0x4361, 0xaf1c, 0x4374, 0x4050, 0x18ef, 0xb060, 0x2d77, 0x1924, 0x41bd, 0xbfa8, 0xb0e8, 0x4190, 0x42dc, 0x415d, 0x40ea, 0xb23b, 0x400a, 0x22d5, 0xb25d, 0x1dd6, 0x1d96, 0x2ce9, 0xbf91, 0x4038, 0xb066, 0x43e7, 0x1822, 0xb269, 0x4593, 0xb2f5, 0x19bd, 0xa3f6, 0x0e7e, 0xb24b, 0xba7d, 0x42ca, 0x4663, 0xbfad, 0x4299, 0x1e65, 0x4225, 0x2788, 0x43e8, 0xb0d3, 0xba16, 0x1e5d, 0x404a, 0x1f58, 0x4365, 0xaa39, 0xb2bb, 0xb220, 0x4582, 0x43a7, 0xb256, 0xb209, 0xb2ab, 0xb0d2, 0x43e4, 0x42c8, 0x413f, 0xb243, 0xbf53, 0x2e4e, 0x466d, 0x4037, 0x41b3, 0x4440, 0xbf49, 0x42be, 0xb2cd, 0x4600, 0x32a9, 0x4148, 0x446c, 0x4220, 0xb0de, 0x0ce2, 0xb0b4, 0xabd7, 0x42ff, 0xb026, 0x419a, 0x40b3, 0x0b25, 0xb2cd, 0x3c35, 0x19ab, 0x1f78, 0xbfb0, 0x0e94, 0x1e9d, 0x258f, 0xa68b, 0xbf65, 0xb08c, 0x4419, 0x079e, 0xb268, 0x0e32, 0xba48, 0xb263, 0x404f, 0x0184, 0x4363, 0x40ac, 0x2bea, 0x4333, 0x4212, 0x1b18, 0x2632, 0x419e, 0xaddd, 0x26d7, 0xab35, 0x43ae, 0x44a1, 0x4553, 0xa3ae, 0xb035, 0x43eb, 0x431a, 0x3b78, 0xbf5f, 0x44f2, 0xa8a4, 0x2b62, 0x41b5, 0xb2a0, 0xb26b, 0xbf60, 0x408a, 0x42f5, 0xb248, 0x4186, 0x0be3, 0xba08, 0xba09, 0x43b1, 0x32eb, 0x435e, 0x1a5f, 0x46a1, 0xa636, 0x40a4, 0x1cc4, 0xb205, 0x4008, 0x1ab0, 0x43dd, 0x44ba, 0x1a0a, 0x41e8, 0xbf0b, 0x4107, 0x43fb, 0xb278, 0x0d1b, 0xb24f, 0x4072, 0xb0fe, 0x1f22, 0x41d6, 0xa4b7, 0xa2e7, 0xbace, 0x1986, 0x41ea, 0xbf8f, 0xb2bd, 0xa889, 0x2c1e, 0x4377, 0x43fe, 0x4208, 0x0d22, 0x1fb7, 0x13a5, 0x1dc7, 0xb2a7, 0x4322, 0xb0e3, 0x20ce, 0x2c78, 0xb02e, 0x1dd1, 0x4260, 0x409c, 0x4072, 0x41c0, 0x43b5, 0xb229, 0x42ae, 0xb2fb, 0x41c4, 0x0949, 0xbf46, 0x4240, 0x0cde, 0x1cd9, 0x3d63, 0xba31, 0x4182, 0x46c9, 0x207c, 0xbfd0, 0xba2e, 0xb03a, 0xa2cf, 0x430a, 0xbf00, 0xbfa3, 0x2012, 0x1aef, 0xbae5, 0x429b, 0x42a3, 0x40d8, 0xb27e, 0x43e3, 0x00d0, 0xbfd0, 0xb0f6, 0x235a, 0x4206, 0x28e5, 0x12df, 0x44c8, 0x29d4, 0x07dc, 0xb0ee, 0x2a46, 0x4684, 0x41c8, 0x4256, 0x419f, 0x1840, 0xbfb6, 0x2d6c, 0xbf00, 0x4111, 0xb2a4, 0x2ac4, 0x4357, 0x2969, 0xbfdb, 0x06e7, 0x1834, 0x4303, 0x1301, 0x4349, 0x4359, 0x4199, 0x42d7, 0x43df, 0x1c72, 0xbf29, 0x42e9, 0x0dc8, 0x444f, 0xb2b3, 0xbade, 0x4096, 0xbf39, 0x4222, 0x4132, 0xbaf8, 0x423a, 0x411a, 0x3d0e, 0x27cd, 0xa5b1, 0xba60, 0x402c, 0x1c06, 0x1e7f, 0x4125, 0x437c, 0x2d39, 0x41e8, 0x43b0, 0x1aec, 0x410f, 0xbf70, 0x4151, 0xbfd7, 0xb24b, 0x434a, 0x420a, 0xb0bb, 0x40b5, 0x2a48, 0x42f7, 0x4357, 0xa542, 0x4377, 0x456a, 0xba32, 0x256c, 0x4293, 0x420d, 0x41b2, 0x4116, 0x0db1, 0xb2fd, 0x2b59, 0xbf5e, 0x4386, 0x46a9, 0x44f0, 0x403d, 0x42a2, 0xbfad, 0x4332, 0xb278, 0x33f7, 0x4154, 0x42d1, 0x45a4, 0xbf2c, 0x43d3, 0x4328, 0x1b3b, 0x2fb0, 0xb2fc, 0xba1f, 0x4081, 0xbac6, 0x43a8, 0xaf84, 0x41f1, 0x4242, 0x417b, 0xb210, 0xbf2e, 0x40d4, 0x43d6, 0x1ccf, 0x187b, 0x308a, 0x4377, 0x42ba, 0x31d7, 0xaffe, 0x1fcd, 0x44e8, 0xb28b, 0x1bb6, 0x441a, 0xa875, 0x40b0, 0x40b8, 0xa3ff, 0xbaea, 0xbf0f, 0xb025, 0xb00c, 0x400b, 0xb242, 0xba32, 0x46f3, 0x4637, 0xba20, 0xa645, 0x42df, 0x40a8, 0x42ed, 0xbf5c, 0x1a0c, 0x1899, 0xbaf9, 0xb259, 0x430a, 0xb2f9, 0x026c, 0x4245, 0xb2d4, 0x2e6f, 0x4339, 0xa74d, 0xafb6, 0x410a, 0xb20b, 0x431a, 0x41cc, 0xaa3b, 0x4138, 0xb0c4, 0x4481, 0xb22d, 0x4048, 0xbf81, 0x29ef, 0x44eb, 0x4173, 0xbff0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x229975ae, 0x62ce2e1e, 0xd2564bd4, 0x39d45247, 0xd49d4418, 0x1318e716, 0x729bdcf6, 0x3efcaaa5, 0xa00a7b25, 0x4ae3af8e, 0x4525a887, 0x9586bb86, 0x0c982ef9, 0x7b92ae66, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0xfffff740, 0x00000000, 0x000000a8, 0x00000000, 0x000018d8, 0xfffff92c, 0x51bdcd40, 0x00000000, 0x00000005, 0x00000000, 0xfffffff8, 0xfffff544, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4140, 0xb2b3, 0x4323, 0xba15, 0x4344, 0xbff0, 0xbfbf, 0x43a6, 0x4559, 0x4223, 0x44dc, 0x4127, 0xb0a9, 0xbf8a, 0xaaf8, 0x46c3, 0x4581, 0x40c3, 0x34c1, 0x0454, 0x4391, 0x1926, 0x0506, 0x409b, 0xb240, 0xb2d2, 0xb2bc, 0x41a3, 0x401e, 0x467e, 0x461e, 0x418a, 0xbfa2, 0x4106, 0xb2d4, 0xba41, 0x41e2, 0xbf60, 0xa5bd, 0xb0f2, 0x0ae4, 0xaf92, 0x4178, 0x4288, 0xaafb, 0x2c49, 0x4297, 0x42aa, 0xb279, 0x417e, 0x4030, 0xb092, 0x43af, 0xb2dd, 0x40f2, 0x4248, 0x4185, 0x424d, 0x1bde, 0x42be, 0xbfa3, 0xb2fb, 0x4202, 0xb284, 0x1e62, 0x4435, 0x1a75, 0xad47, 0x1f63, 0xbfc9, 0x4248, 0x42a4, 0x4043, 0x406f, 0x4235, 0x2e18, 0xba30, 0x0eb5, 0xb010, 0x4284, 0x1c3d, 0xba47, 0x1ab5, 0x40c5, 0x42bb, 0x42b3, 0xb2bd, 0x40cb, 0xbf66, 0x42b1, 0xba69, 0xa9e3, 0xba74, 0xba6b, 0x362a, 0xbaf3, 0x41fa, 0x3157, 0x43e1, 0x197e, 0x2357, 0x2121, 0x4399, 0x40e8, 0xba4e, 0x46a4, 0xb24a, 0x1e4c, 0xbf14, 0x434f, 0xba7e, 0x0794, 0x41a9, 0x4271, 0x0540, 0xb245, 0x4034, 0xa808, 0xa761, 0x1e1b, 0x432a, 0xbfd0, 0x4056, 0x0f35, 0x42dc, 0x41e8, 0x28b7, 0x41e4, 0x42d3, 0x4003, 0xb266, 0x23b0, 0x40f3, 0xbfc2, 0xba05, 0x37dc, 0xa115, 0x2d82, 0x427d, 0x4267, 0xb231, 0x425b, 0x4333, 0x15c9, 0x4365, 0x33cc, 0x4245, 0xb258, 0x4456, 0x418d, 0xbfda, 0xa73d, 0x4151, 0xb06a, 0x113e, 0x40a9, 0x0bd6, 0x4011, 0x0e0d, 0x428b, 0xb085, 0x1887, 0x4306, 0x1e58, 0xba34, 0x0264, 0x1425, 0xbf75, 0xbf80, 0x063a, 0x4075, 0xb0ad, 0x4301, 0x41a9, 0xb2de, 0xb06d, 0xbf4e, 0x2c08, 0x195b, 0x1f78, 0x4665, 0x18a8, 0x427d, 0x1e10, 0x1902, 0x4184, 0xb26c, 0x4202, 0xabcc, 0xa09e, 0xb240, 0xbf97, 0xb2f9, 0x0e5d, 0x4016, 0x41e6, 0x0fa0, 0xb03e, 0xba09, 0xba2d, 0x1d8d, 0x4236, 0x4169, 0x42e1, 0xbfa2, 0x1c7a, 0x4209, 0x46d8, 0x40ba, 0x131e, 0x43e9, 0x4322, 0x2c4c, 0xb0f2, 0x420d, 0x43cf, 0xbfe0, 0xbaf7, 0x11fd, 0x3c3c, 0xb074, 0xa603, 0x40c0, 0xba24, 0xbfdf, 0xba1f, 0x4355, 0x1c3e, 0x3c1f, 0x1771, 0x435e, 0x43f0, 0xbfe0, 0xbf1e, 0x2909, 0x4388, 0xb283, 0xbfb0, 0x4345, 0xb037, 0x4422, 0xba30, 0xb299, 0xb27c, 0xba4e, 0x439c, 0x402a, 0xbfb7, 0x42b5, 0x4306, 0x12ed, 0x4036, 0xa7ea, 0x3301, 0x00dd, 0x43ff, 0xbf80, 0x43a5, 0xa662, 0xb2cc, 0x1fd1, 0xb2f3, 0xb2e5, 0xba3b, 0x41d9, 0x19bb, 0xbf53, 0xb2d3, 0x4060, 0x40a0, 0x41e4, 0x430e, 0x1ace, 0xb292, 0x4126, 0x14a7, 0x4213, 0xba11, 0xba68, 0xbaea, 0x40f4, 0x1efb, 0xad4e, 0xb219, 0x403a, 0xa6a1, 0xbac6, 0x400e, 0x4179, 0x27cf, 0xbf2c, 0xb00d, 0x41da, 0xba16, 0x43f4, 0x419b, 0xae62, 0x40e8, 0xba76, 0xa541, 0xba4f, 0x42c2, 0x41aa, 0x446e, 0x423c, 0x400e, 0xbf45, 0x411e, 0xba27, 0x4150, 0xbff0, 0xbfb0, 0x1d0a, 0x4110, 0x0264, 0xb08d, 0x4094, 0xb237, 0x433d, 0xb2df, 0x445a, 0x18bd, 0x232b, 0x42aa, 0x4049, 0x41e8, 0x4241, 0x43c0, 0xabef, 0xbf27, 0x466a, 0xb235, 0xb210, 0x40c1, 0x1b03, 0x34b9, 0x330b, 0x0d21, 0x1c91, 0x411e, 0x4309, 0x24b1, 0x4694, 0x4352, 0xb06b, 0x43fe, 0x4335, 0x4062, 0xb262, 0xb2cf, 0x41ba, 0x1f6a, 0xbf19, 0xaa47, 0x41d2, 0x3d1a, 0x4214, 0xb041, 0xba7d, 0x4256, 0x44cc, 0x443b, 0x0975, 0x40f5, 0xb218, 0x3e53, 0xb20d, 0x3524, 0xb266, 0x3cde, 0xb268, 0x4488, 0x4242, 0x0d85, 0x4359, 0xbfc0, 0xb2c5, 0xbf42, 0xb2ac, 0x30e2, 0x40f1, 0x0e8c, 0x4253, 0x41f5, 0x4338, 0x23ee, 0x4452, 0x435b, 0xbfe0, 0x4322, 0xba03, 0x4434, 0x425e, 0x4138, 0xb0f5, 0xb027, 0x1244, 0x4199, 0x069e, 0x1cd5, 0x43e7, 0xb251, 0x2463, 0x1b4f, 0x369f, 0xbf07, 0x3825, 0xb2a9, 0x425d, 0x01f8, 0x459d, 0x41f8, 0x1c1a, 0xbfb0, 0x4036, 0x400b, 0x4020, 0x42a5, 0x1c96, 0x4282, 0x1e80, 0x1a86, 0xa400, 0xbfd0, 0x437f, 0x42de, 0x4071, 0x43a6, 0x403d, 0xb2c8, 0x36f0, 0xbf7d, 0xa9ca, 0x4321, 0x4397, 0x3f2d, 0xb0ff, 0x1db9, 0x4327, 0x42ee, 0x4102, 0x1929, 0x1a60, 0x42ae, 0x43fa, 0x4238, 0x335e, 0x4195, 0x41a0, 0x434f, 0x07df, 0x0b88, 0x415c, 0x1bed, 0x1239, 0x2484, 0xbfe0, 0x429a, 0x43f1, 0xbf00, 0xbf81, 0x1d1a, 0x03ca, 0xb001, 0x1144, 0xba4c, 0x4588, 0x4298, 0x1adc, 0x0bf3, 0x425c, 0x40bc, 0x46d0, 0x4196, 0x3c97, 0x1a0e, 0x40d8, 0x4689, 0x4211, 0xb2c7, 0x3858, 0xbf94, 0x1f8b, 0x1dd1, 0x43fc, 0x4221, 0x4134, 0x4091, 0xbf45, 0xb216, 0xa369, 0x425f, 0x41f1, 0x1d77, 0x4351, 0x4052, 0x43b8, 0xaab0, 0x41bd, 0x436a, 0x10b8, 0x1be7, 0xb0a0, 0x4408, 0xba5c, 0x46ed, 0x4280, 0x43e5, 0xb24f, 0x413e, 0x40de, 0x4072, 0x0af6, 0xb0fc, 0x182b, 0xbf18, 0x40a2, 0x4274, 0xbf7b, 0x40e1, 0x4374, 0x1980, 0x446c, 0x3816, 0xb011, 0x1a4f, 0x432b, 0x3edd, 0x443b, 0x4100, 0x4089, 0x1aab, 0xadf1, 0xb27b, 0x41b2, 0x4654, 0x14fc, 0x42be, 0x0988, 0xbade, 0x01d9, 0xbfb2, 0x4495, 0xa451, 0xb236, 0x40a0, 0xa32b, 0x160f, 0x158c, 0x1b5a, 0x1bf9, 0x415f, 0xa9d3, 0x2d3b, 0x0186, 0x2ee4, 0xb2ee, 0x4203, 0x4282, 0xbf25, 0x42fb, 0x41cd, 0x4005, 0x412e, 0x4166, 0xb060, 0x419a, 0xb096, 0x421a, 0xbf9e, 0xbaea, 0xb24b, 0xb266, 0x04cc, 0x4346, 0x1cf3, 0x3752, 0x4191, 0x40df, 0x2bc9, 0xba3f, 0x418a, 0xa35c, 0x40bf, 0xb202, 0xbf75, 0xb022, 0xb2af, 0x4494, 0xb2b3, 0x42ff, 0x4234, 0xba3d, 0x4294, 0xafd7, 0xbf5e, 0x42dd, 0xb06e, 0xb289, 0x16e1, 0x4296, 0xba26, 0xb23d, 0x3187, 0x4257, 0x43b4, 0xb055, 0x43f0, 0xba3b, 0xba4f, 0x4264, 0x1d45, 0x4139, 0x2e94, 0x43c5, 0xbf3a, 0xa887, 0x1db3, 0xb2ed, 0x4155, 0x403a, 0x459c, 0xbf16, 0x4185, 0x435d, 0x4233, 0x4040, 0x195b, 0x3b21, 0x3ac0, 0x41d6, 0xb0d6, 0xbaf9, 0x32b7, 0xb25a, 0xa09f, 0x403a, 0x36e7, 0x01a5, 0x0cbb, 0x19d8, 0x432c, 0x4144, 0xb210, 0x40a9, 0x369e, 0xb208, 0xbfe0, 0xbf74, 0xba77, 0x43fd, 0x430e, 0x431d, 0x0f9e, 0x4093, 0x41ca, 0xac30, 0x1d21, 0x435a, 0xba6a, 0x4070, 0xa76e, 0x40a2, 0x3100, 0x433f, 0x1c9a, 0x447d, 0x0792, 0x28d8, 0xb2ae, 0x309b, 0xa509, 0xb2c1, 0x43d3, 0x1ab7, 0xbf39, 0xb261, 0xa83d, 0x0040, 0x43f9, 0x43aa, 0x0d1a, 0xb2f5, 0x1f4f, 0xa00a, 0x1b3a, 0xbae6, 0xbfba, 0x1dc5, 0x4244, 0x41c2, 0xb2ab, 0x4202, 0xb29c, 0xb257, 0xa703, 0x427e, 0xbf89, 0xb275, 0x4019, 0x46ca, 0x1d3d, 0x1a9b, 0x1f9c, 0x06a0, 0x2760, 0xb258, 0x42d9, 0xbfe0, 0xb284, 0x4693, 0xb028, 0xba10, 0x4070, 0xb0d6, 0x4117, 0x4144, 0x1cb9, 0xbafd, 0x414d, 0x428b, 0x404c, 0x40a9, 0xbfc1, 0xbad9, 0x1cd4, 0x4193, 0x2175, 0x4424, 0x4353, 0x41fb, 0x1085, 0xbfd9, 0x40d9, 0xb226, 0xba45, 0x4397, 0xba74, 0xb04a, 0x0585, 0x0166, 0x43fc, 0x0cb9, 0x4163, 0xa2c8, 0x4167, 0x4298, 0xb2f4, 0x4178, 0x25dd, 0xbf78, 0x4219, 0xba57, 0x2cec, 0x41eb, 0xb2b4, 0x42e3, 0x41d4, 0x4028, 0x19fc, 0x25f6, 0x42c6, 0x411a, 0xb2fc, 0x405f, 0x2db0, 0xba12, 0xba0d, 0x4086, 0x190a, 0xb0d4, 0xb0c7, 0x1a0e, 0x1865, 0x43d6, 0x40f3, 0xbf53, 0x4346, 0x2cce, 0x1b64, 0x4251, 0x4132, 0x3b90, 0x4106, 0x4068, 0x42c8, 0xba69, 0x435b, 0x409c, 0xb23c, 0x40d8, 0xba3a, 0x41f8, 0x0255, 0xba6c, 0x4057, 0x154c, 0x423d, 0xb295, 0xbf5e, 0x39c1, 0xb2c6, 0xb2aa, 0x42ba, 0x41e1, 0x2188, 0xba24, 0xba62, 0xb07c, 0x21ef, 0xb0f3, 0x43ec, 0x4167, 0x16d4, 0x03a1, 0x042a, 0x41bd, 0x41ef, 0x0ce7, 0x40b1, 0x1945, 0x4412, 0x416d, 0xbf73, 0x0cc6, 0x43be, 0x406c, 0x4650, 0x434e, 0xb09f, 0x0557, 0x13bd, 0x1647, 0x1a01, 0x40eb, 0x4257, 0x165b, 0x39b5, 0xba40, 0xbf8d, 0xbf00, 0x45cc, 0xacf1, 0xb21a, 0xb045, 0xba06, 0x42c3, 0x434c, 0x22f5, 0x4361, 0x4168, 0x2d5a, 0x4197, 0x42c2, 0x429f, 0x420f, 0x41ce, 0x4072, 0xb2e6, 0x42b7, 0x0373, 0x0dd4, 0xb2b9, 0x426e, 0x4398, 0x41d1, 0x4391, 0x4388, 0xbfbb, 0x1116, 0x41d9, 0x43a3, 0xb01c, 0x423c, 0xbf23, 0x31dc, 0x4363, 0x1dda, 0x41cb, 0xbf57, 0xba39, 0xa456, 0xb210, 0xb2c3, 0xbacd, 0xb2ae, 0x19df, 0x0c65, 0xbf3c, 0x42c9, 0x4073, 0x46a9, 0x41b0, 0xa506, 0xafdf, 0x4279, 0x416e, 0xa894, 0x15c3, 0x4182, 0x17b8, 0x41c0, 0x2e9c, 0x40bf, 0x4297, 0x42c9, 0x221d, 0xbfe0, 0x27c7, 0xb072, 0xbf13, 0x42e0, 0x1610, 0x43b9, 0x388e, 0x0521, 0xb29e, 0x38ac, 0x1a33, 0x4163, 0x40ae, 0x0262, 0x19ad, 0x1987, 0xbf04, 0x4029, 0x1f7d, 0x4358, 0x046c, 0x1f40, 0xbf00, 0x4052, 0x4228, 0xb0a8, 0xb094, 0xbfdc, 0x4183, 0x1e1d, 0xb006, 0xb244, 0x46b2, 0x40e0, 0x438a, 0x1cf2, 0x09e7, 0xb077, 0x3b0c, 0x42c6, 0x411d, 0x40f4, 0x1d92, 0xb230, 0x3ef5, 0x25ff, 0x40d3, 0xb2fa, 0x0906, 0x33f4, 0x4616, 0xbf93, 0x3405, 0x401a, 0xafb7, 0x4653, 0xb2f2, 0xba5f, 0x43cd, 0x433e, 0xaa58, 0xba2f, 0xb21f, 0x44f2, 0xb2c9, 0xb27f, 0xb219, 0x1f6e, 0x4234, 0xb241, 0x41fc, 0xb299, 0x4262, 0x4332, 0x43c9, 0x1ef1, 0xb209, 0xbf31, 0x4169, 0x4561, 0x439f, 0x43bb, 0x416e, 0x3af5, 0x435e, 0x2c99, 0x407a, 0x41eb, 0xa890, 0x4362, 0xb278, 0xba0b, 0x3322, 0x40ee, 0x4381, 0x3041, 0x434d, 0xbf33, 0xb0f7, 0x40a3, 0x46bd, 0xac4b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3b6339cb, 0xac4b9998, 0x998fa8a0, 0x40e73eaf, 0xf51f8ebf, 0x6f72ff20, 0xf89a82e1, 0x902e336b, 0xebc6f449, 0x4b10b8cd, 0x4c95ab80, 0xffe2804d, 0xe91ff5c2, 0x37fd5281, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0x00000041, 0xfffffff7, 0x0c803106, 0xf8000021, 0x37fd52a9, 0x02400009, 0x00000000, 0x00000000, 0x4c95ab80, 0x00000000, 0x06f00000, 0xc802acfb, 0x4af3391b, 0x37fd517d, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xa5fa, 0x42f7, 0xb2c5, 0x40ad, 0x440d, 0x430e, 0x1a77, 0xbadb, 0x4199, 0x40e9, 0xa24d, 0xb221, 0x1a13, 0x403c, 0x4128, 0xba46, 0xbf7c, 0x43e8, 0x1a3e, 0xba00, 0x1aaf, 0x42f7, 0x433a, 0xbacb, 0x46c1, 0x4312, 0x43a5, 0x2370, 0x40f9, 0x195f, 0x4133, 0xb06b, 0xba1a, 0xba11, 0x09a0, 0x42e4, 0x4269, 0x42f2, 0xb28d, 0x0ba3, 0x05ab, 0xb298, 0xbfcf, 0xbaff, 0x4302, 0x438e, 0x4286, 0xb06a, 0x42ae, 0xbae0, 0x049b, 0x39b9, 0x4570, 0x4222, 0x0878, 0xbf66, 0x3af7, 0xbf90, 0x45e6, 0x4321, 0x400b, 0x0fdf, 0x42ff, 0xba07, 0x4424, 0x427d, 0x2f2b, 0xb23e, 0x4233, 0xbad7, 0xba58, 0xb293, 0x4430, 0xa78b, 0x424f, 0xb2f0, 0x1cab, 0xbf4f, 0x431c, 0x4199, 0xb0fa, 0x29cf, 0x405a, 0x150b, 0xb2fb, 0x43a7, 0x41c5, 0x31b3, 0xb258, 0x40dc, 0xbfb0, 0xb0b5, 0xa486, 0x4338, 0xba40, 0xbfe0, 0xba52, 0x4157, 0xa730, 0x431c, 0x188f, 0xbf92, 0x4306, 0x415b, 0x251b, 0x407f, 0xaef4, 0x4127, 0x0478, 0x0bcd, 0xb2b8, 0xa743, 0x4220, 0x420f, 0x1bfb, 0x4340, 0xa62a, 0x4329, 0xbf70, 0x2233, 0x43d6, 0x46da, 0x06d3, 0x1a7b, 0xb2fb, 0x08eb, 0x4372, 0xb2cc, 0x20d8, 0x0227, 0xbfc2, 0x1f6f, 0x2906, 0xb2c8, 0x4676, 0xb086, 0xb21f, 0x43e1, 0x4094, 0xbf0b, 0x4269, 0x4153, 0x322b, 0x05ec, 0xb00f, 0x41e5, 0xab98, 0xb03e, 0xb2f7, 0x44dd, 0xb29b, 0x1b5f, 0xb223, 0x1136, 0xb222, 0xb0d3, 0xb2c3, 0x413d, 0xba0c, 0x40c4, 0xbaf2, 0x355b, 0x4352, 0xbaec, 0x428a, 0x3477, 0xbf07, 0x427f, 0x193c, 0x4101, 0x4047, 0x42bd, 0x316c, 0x1da0, 0x460e, 0x18ac, 0x442d, 0x4489, 0xb261, 0xbf6a, 0xb265, 0x4134, 0x430b, 0xb280, 0xb007, 0x146b, 0x0779, 0xb07d, 0x4478, 0x42b5, 0xb25c, 0xbfaa, 0xa1d2, 0x4139, 0xb206, 0xb212, 0xba5e, 0xb261, 0xb237, 0xba11, 0xba18, 0x4357, 0x4178, 0x43cd, 0x4234, 0x4134, 0xb2cc, 0x4388, 0x4374, 0xa39f, 0x43e8, 0x411e, 0xb274, 0xbf5f, 0x1e46, 0x1ddd, 0x404b, 0xb279, 0x37e1, 0x44c4, 0xb093, 0xbf6c, 0xbf70, 0xb068, 0x1584, 0x4018, 0xb256, 0x00cb, 0x4205, 0xa715, 0xbf70, 0x4253, 0x44c5, 0x1f85, 0x189e, 0xbfa0, 0x400c, 0x3a61, 0x424c, 0x46dc, 0xab38, 0xbfb7, 0x0388, 0xba10, 0x1f7f, 0xb052, 0x43f1, 0xbf80, 0xb26b, 0xb2c1, 0x4249, 0x40d6, 0x26e0, 0xbae6, 0x4132, 0x404f, 0x461b, 0x4062, 0x1a87, 0x23fe, 0xb216, 0xba38, 0x4141, 0xa221, 0x40ab, 0x19c9, 0x1f3c, 0xbf65, 0x40fe, 0xb268, 0x413a, 0xb0b3, 0x44e4, 0xa369, 0x4211, 0x2341, 0xbfb0, 0xbfb2, 0xb2aa, 0x2549, 0x4034, 0xb269, 0xa110, 0x4140, 0x4089, 0x4028, 0x2609, 0x4218, 0x43e4, 0xbf00, 0xb2e3, 0xba33, 0x42db, 0x40d0, 0x18fb, 0x3271, 0xbf19, 0x3067, 0x3c16, 0x4130, 0x3563, 0x072b, 0xb043, 0xb0df, 0xb077, 0x4326, 0x42cc, 0x24bd, 0x37cb, 0x41c6, 0x4350, 0x18da, 0xb267, 0x44da, 0x0379, 0xaf9b, 0x42c5, 0xb04f, 0x4012, 0x401e, 0x433c, 0x23d8, 0x4130, 0x1b4a, 0xb264, 0x416a, 0xbf18, 0xb2e9, 0xb056, 0x1944, 0xba2c, 0x4110, 0x2e24, 0x3a58, 0xb291, 0xba4a, 0x322e, 0x45d4, 0x43a9, 0xba4d, 0xbfe1, 0xba67, 0xa1ba, 0x0aab, 0x41d7, 0x1393, 0xaff4, 0xbfc0, 0xaa25, 0x4004, 0x4646, 0x044a, 0x42f6, 0x407f, 0x432c, 0x4394, 0xbf60, 0x409a, 0x4174, 0x0bde, 0x41b7, 0x42e1, 0x42e0, 0x19fd, 0x072c, 0xb005, 0x23da, 0xba36, 0x013b, 0x02b8, 0xbf83, 0x1896, 0x38ab, 0x1802, 0xa075, 0xba64, 0xb23b, 0x40e7, 0x4245, 0x42f4, 0xa000, 0xb27a, 0xbaef, 0x4052, 0x06e6, 0x3e0f, 0xb02a, 0x29cd, 0x0a60, 0x42ad, 0x25d3, 0x4220, 0xb24c, 0x1ad2, 0x4077, 0xb218, 0x4030, 0x4273, 0xbfc1, 0x4288, 0xb0e5, 0x42d9, 0xb219, 0x4303, 0xb2a7, 0x1845, 0x43d0, 0x4378, 0x3c9c, 0x33b3, 0x4156, 0xb0ec, 0xbaf1, 0x3ee5, 0xbada, 0xb2ff, 0x4571, 0x41dc, 0xbfc9, 0xb043, 0x0309, 0xa14d, 0x411d, 0x0eb5, 0x41da, 0x406f, 0x43a7, 0xba70, 0xba57, 0x1bbf, 0x42bb, 0x3d25, 0x1b41, 0xb0f2, 0xba0c, 0x4127, 0xbfcf, 0x4157, 0x1e08, 0xb2a9, 0x4615, 0x41bd, 0x4279, 0xab6d, 0xbaeb, 0x05c6, 0x41de, 0xb24b, 0x0bf1, 0x42ba, 0x42be, 0x392a, 0x4251, 0xba79, 0x0623, 0xbac2, 0x43c4, 0x4279, 0x4222, 0x1d6e, 0x43a2, 0x10dc, 0xb0d9, 0x18c8, 0xb25c, 0xbfa8, 0x402c, 0x1b45, 0x42bd, 0x4367, 0xb0c8, 0x4375, 0x4341, 0xb2d6, 0x42ee, 0xa002, 0x4279, 0xb21b, 0x4397, 0x4057, 0x40c4, 0x2547, 0xa55a, 0x41ed, 0xb06b, 0x43c9, 0x45d2, 0x41b8, 0x4338, 0xbfc9, 0xb249, 0x066a, 0xa8d4, 0xbfa0, 0x41e7, 0x013f, 0x40e1, 0xb27d, 0xba48, 0x19f8, 0xb0d6, 0xb02a, 0xb022, 0x41e3, 0x4363, 0xb200, 0xbf46, 0x431d, 0x0a44, 0x419c, 0x42ce, 0x2bd3, 0x42c1, 0xb0d5, 0x0d26, 0x44d8, 0x1cfa, 0x4054, 0xba30, 0xb29a, 0xb241, 0xbfd0, 0xb297, 0xb2b9, 0x43f8, 0xb23e, 0xb2ad, 0xba6f, 0xb23d, 0xb013, 0xba77, 0x3e95, 0xba62, 0xb0cc, 0xba22, 0xbf79, 0xba6e, 0x1ac2, 0x339f, 0xb041, 0x415b, 0xb289, 0x42b5, 0xb246, 0x0107, 0x406c, 0x4219, 0x41ae, 0x1dfa, 0xba38, 0xb217, 0xb217, 0x432f, 0x43fe, 0x1e95, 0xbfcc, 0x43df, 0x06a8, 0x4158, 0xad6e, 0x43d8, 0x1902, 0x43ba, 0x4289, 0xbf55, 0xba62, 0xbafa, 0x41dc, 0x3893, 0x41a6, 0x3bd9, 0x0066, 0x4433, 0xbae0, 0x4373, 0x4367, 0x3924, 0x4193, 0xb295, 0xb23e, 0x19cf, 0xb24b, 0x42a2, 0x3c87, 0xb015, 0xbf55, 0x1b93, 0x46bd, 0x423e, 0x43f0, 0xb0b4, 0x4036, 0xb2ef, 0x0fc6, 0xb225, 0x45c6, 0xa8dd, 0xb00a, 0x4091, 0xb265, 0x4211, 0xbacb, 0xbf68, 0xb21d, 0x1058, 0x4385, 0xaa47, 0x32ae, 0x46f0, 0xbf29, 0x1858, 0xab7c, 0xba78, 0x44d3, 0x464d, 0x42dd, 0x43d2, 0x17b7, 0xb0f1, 0x4138, 0x408d, 0xbf6f, 0x41b8, 0xb25c, 0x1b45, 0x40b6, 0x4084, 0x1a56, 0x1c02, 0xb2c7, 0x41df, 0xb0f9, 0x3625, 0x0a77, 0xbf80, 0x1997, 0x441f, 0x410a, 0xb209, 0x4323, 0x2e9c, 0xade2, 0xb211, 0x40aa, 0x4618, 0x4160, 0x18d1, 0x4398, 0xb25a, 0x409d, 0x4311, 0xbf05, 0x29a5, 0x438b, 0xb003, 0xa5db, 0x42b5, 0xbae1, 0x4011, 0x4415, 0x416c, 0xbfce, 0x08cd, 0xb29b, 0xb26c, 0x40c6, 0xbf8f, 0x41de, 0xb09d, 0x3a51, 0x2fba, 0x4341, 0xb27d, 0xb250, 0x438d, 0x43a3, 0x417a, 0xb2b8, 0xba3b, 0x40c8, 0x4349, 0x4323, 0xb2b1, 0x45c1, 0x41a5, 0xba36, 0x4138, 0x401f, 0xbfd0, 0xb20b, 0xb27d, 0x4267, 0x42e2, 0x45a8, 0xbf9e, 0xbadd, 0x41e8, 0x1c5e, 0x4133, 0xbaef, 0xbf6e, 0xa20d, 0xba19, 0x4194, 0x283b, 0x1d04, 0xb2bf, 0x42aa, 0x2109, 0x433c, 0x45ca, 0xa3e1, 0xbff0, 0xb0cc, 0x3304, 0x40af, 0xa1b6, 0x38e2, 0xba5f, 0x3c6d, 0xb2c9, 0x44f3, 0xb23c, 0x3f80, 0xbf60, 0x419c, 0x427c, 0x4207, 0xbfc5, 0x1e21, 0xa5e6, 0x40b7, 0x4020, 0xbf70, 0xa166, 0xbf2b, 0xb065, 0x10e9, 0x4355, 0xb262, 0x43ad, 0x40aa, 0x0291, 0x1958, 0x1c0b, 0x02ce, 0x4046, 0x3833, 0x1b88, 0xbfd0, 0xbf60, 0xbf66, 0xb09b, 0x1abf, 0x4381, 0x40fe, 0x4175, 0x3d96, 0x1bcb, 0x0a61, 0x43ee, 0x2678, 0xa7a5, 0x40d8, 0xb283, 0x4419, 0x40df, 0x1f45, 0x436f, 0x4085, 0x40c5, 0x413d, 0xb251, 0xbf15, 0x0cf4, 0x4205, 0xba59, 0xb239, 0x4394, 0x41ba, 0x4162, 0x17bb, 0xb2c0, 0x404c, 0x4155, 0x41e9, 0xbf2c, 0x42f5, 0xb254, 0x43a2, 0xbad4, 0xb266, 0xbaf9, 0x1a34, 0x2cef, 0xba72, 0x411d, 0xb0d2, 0x423e, 0x4437, 0x460a, 0xbfb2, 0x38b1, 0x28e2, 0xb285, 0xbfd0, 0x0d57, 0xb0a5, 0x439c, 0xb21f, 0x0547, 0x2a77, 0x43a2, 0x3c57, 0xb223, 0x1a5c, 0x12e5, 0xaa5a, 0xbf97, 0xba6b, 0x1f64, 0xba18, 0x431a, 0xb2fc, 0x4245, 0x438c, 0x4248, 0xbf60, 0x403c, 0x436e, 0x42c7, 0xb218, 0x4169, 0xbfc0, 0xba72, 0x4285, 0x4493, 0x4296, 0x4255, 0xb0e8, 0x45c9, 0x4083, 0xbfaf, 0xbf60, 0x4323, 0xbfd0, 0xb260, 0x45e4, 0x40e6, 0x43b3, 0xbf73, 0x440f, 0x1b4e, 0x430c, 0x43aa, 0xba3a, 0xb27b, 0xb251, 0x2bbe, 0xb2c2, 0xae3d, 0xb2d6, 0xba1b, 0x3599, 0x40ce, 0xb060, 0x40bc, 0xb237, 0x4087, 0x0927, 0x41a4, 0xb03b, 0x2497, 0x4485, 0x41c7, 0x426c, 0x4097, 0xbfaf, 0x41ed, 0xba17, 0x43fb, 0x40a4, 0xbf4e, 0xa776, 0x206d, 0x288a, 0x4300, 0x4111, 0x40cf, 0x4287, 0x404e, 0x426f, 0x2cfd, 0xb26a, 0xabd6, 0x43c3, 0xbfa9, 0x259b, 0x2856, 0x41ef, 0x196c, 0xb242, 0x45e4, 0x41ad, 0xa510, 0xb2ba, 0x16fe, 0x44e8, 0x42f3, 0x40a6, 0x1d3a, 0x2313, 0x4167, 0x44c5, 0x43e2, 0xa74d, 0x42a4, 0x257a, 0x1d63, 0xb202, 0x1ccf, 0x46b2, 0x2a5f, 0x4354, 0x39a8, 0xbf65, 0xb09e, 0x43e6, 0xa4e6, 0x1cb6, 0x1f46, 0x1ff3, 0x348a, 0xb220, 0xb2d8, 0xbfa0, 0xbf5d, 0x415d, 0xb265, 0x4299, 0x413f, 0xba62, 0x031b, 0x4473, 0x0ba1, 0x1acf, 0x41ce, 0x4041, 0x403b, 0xb276, 0xba7e, 0x0190, 0x0272, 0x408e, 0xbf13, 0xb2ea, 0x4216, 0x43a0, 0x4216, 0x4034, 0x401f, 0x43c0, 0x4147, 0xba0b, 0x42c1, 0xba30, 0xbaf4, 0x416a, 0x4178, 0xbad5, 0xbf90, 0xa8a6, 0x4192, 0xbfc3, 0x40c7, 0x4377, 0x2630, 0xb0eb, 0x45a1, 0x43dd, 0x1a55, 0xb229, 0xb2dc, 0x406a, 0x40f7, 0xb075, 0x43d3, 0xb2aa, 0x0a60, 0x07d8, 0x42ab, 0x4266, 0xbf69, 0x436f, 0xbfe0, 0x45d9, 0x1de6, 0x4003, 0x435b, 0xa29c, 0xb255, 0x2ad4, 0x41c4, 0x458d, 0xbf04, 0xba6b, 0x1aad, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xa3e9d705, 0xe9aa48da, 0xc172d1f9, 0x9b3d01b0, 0x01afea06, 0xeb3a9b8a, 0x12cc8d93, 0x01c88d27, 0x58e8c5c1, 0xdc241641, 0x7eb66cf7, 0x4ab2dd86, 0x10ad69fd, 0x3b68308b, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x00000000, 0xffffff9e, 0x00001a40, 0x00000000, 0x00000000, 0x00000040, 0x00000000, 0xffe68b7f, 0x008fee88, 0x58e8c62c, 0x00000000, 0x22b65886, 0x9565bb0c, 0x011fdd38, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x43cc, 0x407a, 0x415b, 0x42d2, 0xb20d, 0x41de, 0x1b41, 0x19b6, 0xb051, 0x1dcf, 0x36bc, 0x29fd, 0xbf00, 0x15eb, 0x415d, 0x2be9, 0x1bd3, 0xba76, 0x41a0, 0x45b0, 0xb26e, 0x0c41, 0x3dc0, 0xa863, 0x08ac, 0xbf87, 0x01a2, 0xba51, 0x430e, 0x1090, 0x425e, 0x428c, 0xba79, 0xb0b7, 0x1c66, 0x1c0e, 0x1bf3, 0x40df, 0x4663, 0x4346, 0x1b2f, 0x19fe, 0xbfdd, 0x0720, 0xa14b, 0x1a70, 0xba79, 0xba54, 0xba53, 0x42ef, 0x42c0, 0x409f, 0xbfd6, 0x4276, 0x41df, 0x2a4b, 0x43d6, 0x4094, 0x403d, 0x456a, 0x4027, 0x40b1, 0xac6c, 0x4201, 0x426d, 0x40f2, 0x3d25, 0xb26c, 0x2c76, 0x4193, 0xb0ae, 0x4048, 0x46f4, 0x0f9b, 0xbf00, 0x435b, 0xbfd0, 0x4166, 0x42da, 0xbf78, 0x42a6, 0x455c, 0x428d, 0xbafc, 0x140b, 0xbf8d, 0xb201, 0x4348, 0x4198, 0x1ad2, 0x41fb, 0xba39, 0xb295, 0xbf07, 0x43cb, 0x1d7c, 0xb285, 0x4223, 0x41a0, 0x2f9b, 0xbfa0, 0xb0ab, 0xb00d, 0x411f, 0x463a, 0x43d7, 0xb240, 0xbac6, 0x4067, 0xaa78, 0xb25f, 0x41f2, 0x438d, 0x42ed, 0x4217, 0xbf36, 0x4110, 0x4127, 0xb2dd, 0x4033, 0xae52, 0x4633, 0x401e, 0x1060, 0xb026, 0x3426, 0x4328, 0x4223, 0x3866, 0x4665, 0x1c0d, 0x05a2, 0xbf5c, 0xb2e6, 0xa188, 0xac49, 0x424b, 0x261f, 0x41c5, 0x41e9, 0xad0a, 0x2738, 0xba47, 0xa121, 0xb095, 0x45ee, 0x18ca, 0xbf80, 0x030f, 0x4279, 0x46e2, 0x418d, 0x44c5, 0xb250, 0x4248, 0xb026, 0xbfa8, 0x41d3, 0xa696, 0x426f, 0x45da, 0xb294, 0x40e2, 0x1ccd, 0xa595, 0x420e, 0xaf3b, 0x10ee, 0xb0ae, 0x3c78, 0x413a, 0xb0b4, 0x23f4, 0xbf5d, 0x18be, 0xb077, 0x4070, 0x022e, 0x41be, 0xb2a9, 0x42c3, 0xbf1f, 0x4194, 0x037b, 0xbfd0, 0x42ef, 0x025e, 0xb21c, 0xbf91, 0x4016, 0x43e7, 0x4280, 0x409e, 0xb25b, 0xb023, 0xb224, 0xaa0f, 0xba4b, 0x1b56, 0xba31, 0xa177, 0x4118, 0x08be, 0x433b, 0xb023, 0x40cb, 0x2d55, 0xb0a3, 0x40ef, 0x44d9, 0x420f, 0xb0a7, 0x39b0, 0x412b, 0xbfa8, 0x1e44, 0x4602, 0xbfa3, 0xba1e, 0xb2c6, 0x429a, 0xb205, 0x047a, 0xb0fe, 0x4373, 0xbad8, 0x434c, 0x30e0, 0x1ee5, 0xbfb0, 0x410e, 0xb21e, 0xa97d, 0x42b1, 0x4671, 0x433f, 0xbf0d, 0x0ee6, 0x42ac, 0x19e3, 0x41a3, 0xbfa0, 0xba44, 0xa51c, 0xb241, 0xb2af, 0x415a, 0x0a3c, 0xbf1c, 0x28a0, 0x43d5, 0x4200, 0x43f3, 0x4240, 0x419c, 0x3d0b, 0xb09a, 0xa807, 0x42fa, 0x41e0, 0xbad5, 0x3438, 0xba09, 0xb0f9, 0x4628, 0x4167, 0x1b7b, 0xba0a, 0xa17a, 0xb276, 0x40b8, 0x121a, 0x133c, 0xbf34, 0x3613, 0xb259, 0x4105, 0x41de, 0x4155, 0xa2c3, 0x189d, 0xbf65, 0xb017, 0xa019, 0xb082, 0x40c1, 0x4319, 0x1806, 0x1131, 0xbf87, 0x402b, 0x2960, 0x1969, 0x4232, 0x465a, 0xa10c, 0x464a, 0x37eb, 0x45a3, 0x401d, 0x1f11, 0xb0f1, 0x4210, 0x439a, 0x4048, 0x44a1, 0x28fb, 0x44f0, 0xb0b6, 0x41bf, 0x199b, 0xb268, 0x421a, 0x0b58, 0x40ad, 0x18b6, 0xbf09, 0x402a, 0x137c, 0x4046, 0x175f, 0xb0cf, 0x419b, 0xbf52, 0x419d, 0x4213, 0x04a6, 0xba01, 0xbff0, 0x01d4, 0x1932, 0x42d7, 0x4123, 0x4160, 0x0e55, 0x40c5, 0x195a, 0x43bc, 0x413d, 0x1d89, 0x40e8, 0xb21b, 0xb2a3, 0x43e1, 0x417e, 0x423c, 0x24d1, 0xbf6d, 0xa57c, 0x2ff1, 0x05fa, 0x4200, 0xb26f, 0x05f6, 0x064f, 0xbfcd, 0xb2dd, 0x40ed, 0x360f, 0xbafa, 0xb0e6, 0x2f81, 0x09cc, 0x4072, 0x408c, 0x0899, 0xb26c, 0x1ee4, 0x465f, 0x4175, 0xa5b6, 0xba63, 0x1ffc, 0x4094, 0x1a4f, 0xba23, 0xbf2b, 0x25dc, 0xba1f, 0x413b, 0x4105, 0x4180, 0x4186, 0xb2f9, 0x4194, 0xbfae, 0x1da7, 0x1afc, 0x4330, 0x3083, 0x43e5, 0x1f44, 0xba0a, 0x4469, 0x4062, 0xbaeb, 0x16ab, 0x363c, 0x1bdd, 0x40b0, 0xbfa8, 0x4346, 0x4129, 0xb080, 0x4450, 0x40d1, 0x12e2, 0x40a6, 0xae28, 0xbf74, 0xb08f, 0xb280, 0x0fb5, 0xb287, 0xb28e, 0x4238, 0x296f, 0x4398, 0x43b3, 0xb0b7, 0xba78, 0xbf73, 0x41be, 0x40ec, 0x1602, 0xb227, 0xb2c4, 0xbf28, 0x4298, 0x2f81, 0xbadc, 0x1dd0, 0xb255, 0x2521, 0xbac6, 0x03e6, 0xa5ac, 0xba11, 0x401e, 0x4172, 0x1912, 0x43a1, 0xbfa0, 0x41da, 0xb283, 0x2be3, 0xbaff, 0x24bf, 0x415a, 0x1da5, 0x4212, 0x436f, 0x41b8, 0xb29e, 0xbf28, 0xb2e8, 0x1ff8, 0xabd8, 0xb064, 0xba4e, 0x4136, 0x425c, 0x41b5, 0xba40, 0x19bb, 0x26a6, 0xbf26, 0xbaf6, 0xba55, 0x4169, 0x40c2, 0xb2c0, 0x0da2, 0x41fc, 0x3d98, 0x43d1, 0x19af, 0x4023, 0x0a9d, 0xb003, 0xb046, 0x1866, 0x40b2, 0x41af, 0xbafa, 0x1bd8, 0xbfa8, 0x4450, 0x427a, 0x4190, 0x1aa2, 0xa784, 0x1069, 0xb095, 0x181a, 0x4450, 0x41b6, 0x1bf7, 0x1d12, 0x347a, 0x41a3, 0xb263, 0x4215, 0x431c, 0x433c, 0xbf61, 0xb200, 0xb2db, 0x1c1d, 0xb2e5, 0xbf60, 0x12e2, 0xa6a8, 0x45aa, 0xb2bc, 0x41ca, 0xb2d7, 0xab7e, 0x0d22, 0xbf9e, 0x40b8, 0x2c08, 0x1db6, 0x3d47, 0x42ee, 0x4081, 0x412b, 0xa807, 0x350a, 0x43c6, 0xaea7, 0xbf5d, 0xa806, 0xb2cc, 0x4143, 0x30fb, 0x0043, 0x46b9, 0x1c7f, 0x1e13, 0x0da5, 0xb214, 0x3da5, 0x437f, 0x3532, 0x458a, 0x4131, 0x424b, 0x144f, 0x433a, 0x3f9b, 0x0dfc, 0x1faf, 0xb253, 0x4137, 0xbfd0, 0xbf49, 0x4107, 0x24b5, 0x4324, 0x40fb, 0x30c9, 0x1ea2, 0x324f, 0x1a34, 0x0dc7, 0x40e7, 0x1d48, 0x41dd, 0xae7c, 0x43a3, 0x299f, 0xbf01, 0xb077, 0x3770, 0x4140, 0xbfe0, 0x4265, 0x4019, 0x40ed, 0x0afe, 0x1b29, 0x1afc, 0xb23c, 0x04db, 0xb214, 0x439d, 0x2938, 0x0092, 0x40b0, 0x43c0, 0x43d5, 0xb0c1, 0xa37e, 0x1e61, 0xb021, 0x4022, 0xbf6f, 0x4114, 0x407d, 0x428c, 0x415e, 0x4256, 0xb0ea, 0x41e6, 0xbaf7, 0x4060, 0x1ee1, 0x1100, 0x42d5, 0xb25f, 0xb063, 0xb253, 0x1945, 0x30ee, 0x4210, 0x3373, 0x42a7, 0xbf72, 0x42d1, 0xb038, 0xbae9, 0x43ea, 0x40c7, 0xbfa0, 0x299d, 0xb294, 0x404e, 0x085d, 0x4341, 0x4172, 0x1c2a, 0x4183, 0xbf2b, 0xb2b7, 0x40ab, 0xb279, 0x43a9, 0x1cc5, 0x4208, 0x402f, 0x43b8, 0x43ee, 0x41e4, 0x3928, 0xba55, 0x4235, 0x2c5d, 0xbf90, 0x4278, 0xab33, 0x4338, 0x1631, 0x43b3, 0xbfc0, 0x058f, 0xba1c, 0x424d, 0x40e6, 0x3bc1, 0x43a9, 0xba00, 0xbf3e, 0x41fc, 0xa8a3, 0xba2b, 0x1e86, 0x23a6, 0x41f5, 0x40e4, 0x40c8, 0x430d, 0x46b2, 0xba5d, 0x415b, 0x32b1, 0x43ec, 0x2faf, 0xbfda, 0x43ac, 0xa3e2, 0xb283, 0x1c45, 0x4341, 0x4133, 0x40b4, 0x36da, 0x1c1e, 0x4395, 0xba7e, 0x4070, 0x1aeb, 0x467a, 0xb29e, 0x345f, 0x4364, 0xbf00, 0x412e, 0xb2e8, 0x1b7e, 0x2b8a, 0x42c7, 0x412b, 0x4097, 0xb2bf, 0xbf0c, 0x1914, 0x443e, 0x3c97, 0x310b, 0x4091, 0x4367, 0x409f, 0xba38, 0xbf33, 0x403d, 0x4024, 0x1d51, 0x43ed, 0xba4c, 0x4057, 0x46fc, 0x4356, 0x4095, 0x4119, 0x4019, 0xb232, 0xba07, 0x22e0, 0x4082, 0xb038, 0x19e5, 0x1171, 0xbfc3, 0x1cf0, 0x1e78, 0xb2bd, 0x250a, 0x4128, 0x42bf, 0x43c4, 0xb0cf, 0xb269, 0xba09, 0x42dd, 0x1ad6, 0x41c3, 0x2cab, 0x411d, 0xbf95, 0xaf53, 0x45b5, 0x4255, 0x41d7, 0x43c1, 0x4447, 0xb2c7, 0xb284, 0x1c1e, 0xba76, 0xaada, 0x1344, 0x40ac, 0x426f, 0xaa92, 0x41d1, 0x261f, 0x189e, 0x1904, 0x41fe, 0xb283, 0xa695, 0xbfa3, 0xb0f1, 0x41dd, 0xb2c2, 0xb2fd, 0x431b, 0x438d, 0xb2a1, 0x4240, 0x42c9, 0xa2b2, 0x345e, 0x435e, 0xbf1f, 0xba7f, 0x41d4, 0x1e3f, 0x0eee, 0x1d12, 0x42d2, 0xbfb0, 0x41b0, 0xa79a, 0xbf80, 0x32a0, 0xb20d, 0x3e30, 0xb2ad, 0xb273, 0x41a4, 0xb2c3, 0x404c, 0xba3e, 0x4377, 0x4197, 0xbfd9, 0xba3a, 0xb23e, 0x41c3, 0xb042, 0xb2ee, 0xb24c, 0x43ce, 0xbade, 0xa3fb, 0x04f6, 0xb293, 0xa149, 0xb25e, 0x4319, 0x19e5, 0xbf1c, 0x1878, 0x1bb1, 0x4061, 0x4657, 0x414d, 0x1f05, 0xba55, 0x1a6a, 0x4065, 0x0674, 0xab5f, 0xa70a, 0xbf87, 0x4095, 0xb089, 0x4121, 0x4300, 0x4012, 0xbfbf, 0x40a1, 0x436c, 0x4032, 0xbad1, 0xa20b, 0x1d95, 0xa4be, 0x1deb, 0x419f, 0x4324, 0x4418, 0x4363, 0x42ce, 0x403c, 0xbfb3, 0xb07a, 0x4179, 0xbaf5, 0x1daa, 0xb22a, 0x4671, 0xba28, 0x188a, 0x41cd, 0xb298, 0x19e9, 0x428b, 0x1e5c, 0x413a, 0x3a1f, 0x41d5, 0xa080, 0x43d2, 0x28e1, 0x436e, 0x40ce, 0xb27c, 0x26cf, 0x407e, 0x456a, 0x43c7, 0xb253, 0x4667, 0xbf34, 0x411e, 0x43f8, 0x4152, 0xbaff, 0x42ce, 0x4147, 0x128c, 0x4095, 0x1a0a, 0x4392, 0x1c9b, 0x41ed, 0x2a7a, 0x01f2, 0xbf00, 0x004d, 0xb2b3, 0x1d3b, 0xb0e9, 0xb2f3, 0xbf93, 0x10bb, 0x41bf, 0x1f9e, 0x1d2d, 0xb2d4, 0x08c2, 0x4190, 0xbadc, 0xba46, 0x42cc, 0x446d, 0x424e, 0x4146, 0x40ef, 0xa359, 0x1a54, 0xb066, 0xba55, 0xbacf, 0x41f3, 0x4689, 0xbf0f, 0x1492, 0xbae7, 0xb2b4, 0xb2a0, 0x413f, 0x00d2, 0x406e, 0x40dd, 0xb2c9, 0x0345, 0xb20e, 0x432d, 0x4227, 0x34a0, 0x429c, 0x416b, 0xb08d, 0xbf4f, 0x19d9, 0x42bd, 0xa315, 0x442a, 0x1a15, 0x445e, 0xaea8, 0x0a9d, 0x35e8, 0xb29f, 0x1a1e, 0x3022, 0xb050, 0xa914, 0x0e5a, 0x4267, 0x4445, 0x4367, 0x1ff7, 0x40f8, 0x23dd, 0x4283, 0x454d, 0x415e, 0x4199, 0xbfe4, 0xbfc0, 0x4185, 0x4564, 0xb059, 0x409f, 0xbf55, 0x446b, 0x2c78, 0x43b4, 0x435a, 0xa547, 0x411d, 0x2c11, 0xb0a8, 0x0e79, 0xb23c, 0x4111, 0x42e8, 0x1a3e, 0x1e15, 0xbf06, 0x0cce, 0x1e92, 0x0485, 0xba74, 0x0a52, 0xa544, 0x4120, 0xb087, 0x41e8, 0xba64, 0xbf25, 0x3059, 0xb2f3, 0x4301, 0xb2e3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5b8d957d, 0xda740f74, 0xf45748f5, 0x13975c8e, 0xd433eb4e, 0xa749743f, 0x6c798277, 0xd9c1885d, 0xc3623606, 0xea7f747e, 0x8bf0542c, 0xfe1fe0b1, 0xd0d3fbf2, 0xed311e85, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0x007fffff, 0xb0934dd8, 0x00000000, 0x000018e4, 0x00000000, 0x00000000, 0xc3623606, 0xffff8cd5, 0xb0934e5d, 0xfe1fe0b1, 0x000015a6, 0xb0934c3f, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x2722, 0x4544, 0x4378, 0x437b, 0x33fe, 0x41f0, 0x1814, 0x1583, 0x0945, 0xa03e, 0xb24c, 0x1ba2, 0x1e8e, 0x4291, 0xb2d6, 0x1b2d, 0x1637, 0xbfca, 0x3736, 0xb2a1, 0x41d7, 0x42e1, 0xb2b3, 0x4007, 0xb073, 0x1d95, 0xb219, 0x3d08, 0x415a, 0x4282, 0x4245, 0x43c9, 0x1a86, 0xb2bf, 0x463e, 0x40af, 0x1cd7, 0x43b1, 0x4108, 0x0da6, 0x430f, 0x418b, 0xbf2b, 0x4608, 0x441c, 0x41b7, 0xb254, 0xba40, 0x42b7, 0x1de0, 0xbf70, 0x1d62, 0xba76, 0xb045, 0x4138, 0x4067, 0x0654, 0x1e1a, 0xb015, 0x447a, 0x430b, 0x4154, 0x2823, 0x4685, 0xbf75, 0x4080, 0x426a, 0x43a2, 0x40eb, 0x437c, 0x4048, 0x4285, 0xba0d, 0x1eea, 0x4364, 0xbf29, 0x41b9, 0x4351, 0x4204, 0x1c9d, 0xa86b, 0x1174, 0x4064, 0x00d4, 0x4021, 0xb047, 0xba13, 0x4113, 0xb2a0, 0x12fd, 0x4171, 0x4178, 0x410e, 0xba24, 0x432b, 0xb2b0, 0x3ef8, 0xb2ea, 0x410e, 0x1db0, 0x18de, 0x1ffa, 0x1bc9, 0xbf27, 0x43af, 0x4021, 0xb2f6, 0x4307, 0x400e, 0xba4a, 0x44f4, 0x43e4, 0x22a9, 0xba03, 0x1c54, 0x4356, 0x1c4c, 0xbf8a, 0x426e, 0x4136, 0xac9b, 0xb011, 0xba49, 0x4458, 0x1230, 0x4551, 0x4003, 0xbf41, 0x1ff7, 0xb2a2, 0x19dc, 0xabbe, 0xbf28, 0xa7a7, 0x407c, 0x4019, 0xa9c5, 0x4479, 0xb07b, 0x00e2, 0xbf16, 0xb2a7, 0x323b, 0xbfe0, 0x42fd, 0xb2b7, 0x40fd, 0x4140, 0x421e, 0x41dc, 0x179a, 0x2e91, 0xb25a, 0xb0af, 0x4384, 0x4157, 0x45ad, 0x280d, 0xbfa2, 0x467b, 0xba07, 0x4262, 0x36bd, 0xb0d2, 0x289b, 0x4211, 0x4152, 0x41ef, 0x426e, 0xb2f2, 0x4356, 0x4614, 0xbaf0, 0x4229, 0x314a, 0x4182, 0xbf1e, 0x1fd7, 0xba5d, 0xad77, 0x43aa, 0xbae9, 0xbfd0, 0x0a05, 0xa8b2, 0x419a, 0xb23c, 0x4367, 0x4090, 0xbf70, 0xbf1c, 0xbf00, 0xb0f9, 0x4274, 0x4299, 0xb210, 0xb215, 0x1e80, 0x08e8, 0x43ee, 0x412f, 0x379d, 0x402e, 0x1919, 0xaff5, 0x4106, 0xbae6, 0x42e4, 0x45aa, 0x40e4, 0x1b18, 0x43b9, 0xbf1d, 0xbaca, 0x3c0c, 0xb090, 0x463c, 0x1fe5, 0xb21c, 0x4129, 0x3dd7, 0xb24f, 0xaeb3, 0x4651, 0xb0f0, 0x0629, 0xba16, 0x42be, 0x4147, 0x1d46, 0x2698, 0x436f, 0xb080, 0xbff0, 0x4337, 0x3d59, 0x465e, 0x2c6d, 0x4262, 0xb088, 0x417d, 0x41b8, 0xbf12, 0xbac4, 0x40c5, 0x1406, 0x3e5f, 0x3f1c, 0x04b0, 0xa681, 0x469a, 0x433a, 0x443c, 0x26c9, 0x41eb, 0x4198, 0xbfd3, 0xb2d5, 0x4380, 0xb24d, 0xba72, 0x4366, 0x42af, 0x1ae5, 0x43ff, 0x43ed, 0x412e, 0x1d52, 0x405a, 0x41ed, 0x45cd, 0x41ae, 0xb2b8, 0xbfda, 0x2331, 0x4430, 0x0f27, 0x4036, 0x1f45, 0x40b2, 0xb0a7, 0x4230, 0x0a6e, 0x44f8, 0x460d, 0xb29f, 0xa318, 0x430b, 0x3556, 0xae3e, 0xb034, 0xbf57, 0xb2d5, 0x3b8c, 0x44ea, 0xb05a, 0x42f4, 0xa5f5, 0x2209, 0x416c, 0x4466, 0x422b, 0x4263, 0x43d2, 0x1468, 0xb265, 0xbfcb, 0xbad4, 0xb28f, 0xbadb, 0xa163, 0x443b, 0x3b15, 0x197c, 0x261a, 0x43a1, 0x4028, 0xb24b, 0x4182, 0x1e7a, 0xb2d8, 0x4108, 0xba42, 0xbfe0, 0x42ba, 0x4212, 0x1849, 0x1a51, 0x1a9f, 0x15e7, 0xba28, 0xb27f, 0xbf23, 0x41c9, 0x3442, 0xad53, 0x406f, 0x0712, 0x1cd0, 0x467c, 0x1da8, 0x42fe, 0x43e1, 0xba5f, 0xb2fa, 0x41be, 0x3f24, 0xbfb0, 0x426b, 0xbf05, 0x2ee5, 0xbade, 0x43f9, 0x43ed, 0x4195, 0x4176, 0x4294, 0x4286, 0x00e1, 0xacdb, 0x1e04, 0x1f46, 0xbfc4, 0xa533, 0x4165, 0x41a6, 0x4344, 0xbf97, 0xba71, 0xa713, 0xbafc, 0x40a1, 0x00da, 0xb0c8, 0xba61, 0x4161, 0x1da3, 0x4053, 0x19d0, 0x40dc, 0x4036, 0x4075, 0x40a6, 0xab72, 0x08d1, 0xb21b, 0xaddc, 0x416e, 0x1c32, 0x41b8, 0x0b12, 0x1434, 0xb042, 0x408a, 0xb24d, 0x1743, 0xb287, 0xbf37, 0x40cd, 0x417f, 0x34ae, 0x427e, 0x42d5, 0x4039, 0x360f, 0x413f, 0x3864, 0x27c9, 0x40f8, 0x4318, 0x422e, 0x3c63, 0x1bd9, 0x424c, 0x1384, 0x1c1e, 0xa82e, 0xbfc8, 0xa049, 0x079e, 0xa832, 0x46e8, 0xba13, 0xbfc0, 0x1af3, 0xb2a3, 0xb074, 0x19b9, 0x2c75, 0xbacf, 0x3dca, 0x0f0b, 0x2979, 0x4007, 0x43be, 0xba62, 0xbadc, 0xbf87, 0x125a, 0x435f, 0x1c1a, 0x13ab, 0x42b7, 0xbf70, 0x42cc, 0x1978, 0x41b2, 0xba50, 0x42be, 0x4346, 0x4336, 0xb2a5, 0x45bd, 0xa15a, 0x2038, 0xba4c, 0x1a84, 0x430b, 0xb2b4, 0xbfae, 0x41c1, 0x42d3, 0x1ef4, 0xb01a, 0x1c63, 0x210b, 0xb272, 0xa49d, 0x1be2, 0x18b4, 0x1c08, 0xa4d3, 0x1f6b, 0x113c, 0x4663, 0x41c6, 0x4247, 0x402f, 0x4379, 0x0949, 0x4008, 0x42c6, 0x41fc, 0x0b89, 0xbf7c, 0x4370, 0x422d, 0x4362, 0x0b78, 0x040c, 0x43da, 0x462d, 0xb251, 0x1e56, 0x4123, 0xaaef, 0x43a6, 0x0408, 0x2a06, 0xb276, 0xb091, 0x42a4, 0x4319, 0x3ce5, 0xb297, 0x40df, 0xba2d, 0x3233, 0x29dc, 0x407f, 0xbf4a, 0x4685, 0xba15, 0x1404, 0xba06, 0x42a9, 0x262a, 0x3e76, 0x0ba0, 0xb2e8, 0x4482, 0x4047, 0xb037, 0xb06b, 0x05a4, 0x40ad, 0x435d, 0x4057, 0x4280, 0x4049, 0x42b3, 0xbf6d, 0xba7a, 0x42f6, 0x0984, 0x2d9f, 0x44d5, 0x4374, 0x4047, 0x1a23, 0x435b, 0x1f11, 0xb213, 0xba1a, 0x4030, 0x43ca, 0x0f9c, 0x4682, 0xaed1, 0x45be, 0x42eb, 0x41ef, 0xa19a, 0xa571, 0x43e4, 0xbf4c, 0x4255, 0x43bc, 0x40d9, 0x0a28, 0x2786, 0xbf33, 0x42c9, 0x1b40, 0xb2c6, 0xb0a8, 0xafac, 0x40a8, 0x1c68, 0x4184, 0x1420, 0xb2d5, 0xbf34, 0x433a, 0xb24b, 0x45e4, 0x0183, 0xb22d, 0xb091, 0xb2cf, 0x4038, 0x41a3, 0xbf60, 0x40ef, 0x4360, 0x45c4, 0xb080, 0x339d, 0xba16, 0x4115, 0xbf96, 0x406e, 0xaf42, 0xb28f, 0x413d, 0xbfbb, 0x4146, 0xb038, 0x4285, 0x216e, 0xb264, 0x431e, 0x42b9, 0x2f56, 0x0e0c, 0x4217, 0xa12d, 0xac5c, 0x4328, 0x40a2, 0x1a29, 0xb0e9, 0x442f, 0x41bc, 0x37c6, 0xb0cc, 0x18ce, 0xb042, 0x1b4e, 0xbf90, 0xbf9d, 0x419f, 0x414f, 0x1eb1, 0x1f9e, 0x4294, 0xa197, 0x1896, 0xbaf5, 0x4272, 0x45c8, 0x0da9, 0x424c, 0x2c62, 0x4022, 0x1bfa, 0x4091, 0x0417, 0xb099, 0xba6b, 0x40c9, 0x1d35, 0x43c1, 0xb0e2, 0x443b, 0x4137, 0x0c33, 0x418c, 0xbf7e, 0x4196, 0xba14, 0x4585, 0xbfca, 0x41f0, 0x3bb0, 0xb044, 0xb2e7, 0x2b49, 0x4428, 0xbfca, 0xb092, 0x4051, 0x4201, 0x1a8d, 0xbf90, 0xba5c, 0xb0bf, 0x4151, 0x19c8, 0xb293, 0x437a, 0xb2a9, 0xbace, 0xbf71, 0x40da, 0xb24e, 0x0bb8, 0x1879, 0xba3b, 0x402b, 0x1e0d, 0xbad2, 0xb204, 0xba07, 0x3165, 0xb08b, 0x3c16, 0x1e6d, 0x40f9, 0x1379, 0x17f9, 0xaf70, 0x4287, 0xbf2b, 0xbaf2, 0xba11, 0x41e7, 0x0360, 0xb237, 0x43d9, 0x1db8, 0xba48, 0xba5d, 0xbac7, 0xbfdd, 0x0be8, 0x1c5b, 0x20b0, 0x4099, 0x3a34, 0x4254, 0xb236, 0x419a, 0x1b31, 0xbf2d, 0x42a2, 0x4271, 0xbad5, 0xb254, 0x1bb2, 0x43b2, 0xb2b2, 0xbae9, 0xb22b, 0xa066, 0xba09, 0x4028, 0x400a, 0xb212, 0x42a3, 0x4047, 0xbf4a, 0x2bc0, 0xa346, 0x40a3, 0x42ee, 0x138f, 0x1b40, 0xba12, 0x1b0e, 0xa916, 0xb241, 0x400b, 0xbfdb, 0xba28, 0x41a1, 0x4390, 0x3d2b, 0x443e, 0x4673, 0x1d4d, 0x4120, 0x4269, 0x419d, 0x46f2, 0x4682, 0x41c5, 0xb2d2, 0xb234, 0x4036, 0xbf7e, 0xba39, 0xb2e8, 0x466f, 0x441e, 0x1b36, 0x3f5d, 0xba69, 0x20de, 0x1fdf, 0x18d0, 0x095f, 0x416b, 0x437d, 0x4331, 0x40de, 0x0e6a, 0x353b, 0x13c7, 0x4221, 0x437c, 0x1f97, 0x3a6d, 0x435d, 0x42cc, 0x43c4, 0x310b, 0x428c, 0x2383, 0xbfc7, 0xb0c1, 0x1fd1, 0xa1ce, 0x41d6, 0x45a6, 0xb275, 0x4257, 0xb2b3, 0x43ab, 0x4278, 0x1ecd, 0x359e, 0xb2e9, 0xba0c, 0xbad3, 0x1dc2, 0x419e, 0x424a, 0x41aa, 0xb2c1, 0x44e5, 0x40f0, 0x1778, 0x4088, 0xbfd6, 0x4246, 0x4197, 0x1c5c, 0x1b2e, 0xb28b, 0x0f3d, 0x1e81, 0x4122, 0x2e73, 0x1db5, 0x42fa, 0x1d43, 0x43f2, 0xba18, 0x0068, 0x42ff, 0x0dd0, 0x4247, 0x45ce, 0xb2ba, 0xa73d, 0x428f, 0x4329, 0xb225, 0xbf62, 0xba24, 0x43a0, 0x204c, 0x1466, 0xb2dc, 0x3f46, 0x42a7, 0x43a4, 0xba61, 0xb0f1, 0x40bb, 0x40c9, 0x4192, 0x4113, 0x1d47, 0x448d, 0xaa2a, 0xbaf6, 0xb2db, 0x4294, 0x4206, 0x0429, 0xb2b2, 0x41ee, 0x4081, 0x44b3, 0xbfd1, 0xbadd, 0x424e, 0x2a0e, 0x334e, 0x41f0, 0x1b77, 0xbae8, 0x4067, 0x42eb, 0x00b1, 0x423e, 0x4073, 0x4106, 0x40a9, 0x416b, 0x128f, 0x0281, 0xba3c, 0xbac5, 0x0c2f, 0xb012, 0x1c56, 0x40a6, 0x4096, 0x40e4, 0x42b5, 0xb090, 0x42ea, 0xbf49, 0x4287, 0x4051, 0x43d7, 0xba1d, 0x41dc, 0x4161, 0x2296, 0x4251, 0x4166, 0x43ad, 0xba47, 0x1b9f, 0xb026, 0xb2d0, 0xb2ec, 0x420d, 0x4355, 0x434c, 0xb08f, 0xb252, 0x1b04, 0xa364, 0xbafc, 0x19a4, 0x3c98, 0xbf9f, 0x41c0, 0x34d7, 0x456e, 0x42d5, 0xb264, 0x43df, 0x4542, 0xba15, 0x40de, 0x42d1, 0x1a20, 0xbf2f, 0x4103, 0xacdc, 0x40cc, 0xb099, 0x43eb, 0x43df, 0x317c, 0xba70, 0x1e2c, 0xbf84, 0x4158, 0x1c12, 0x40cb, 0x46d1, 0x408a, 0x037c, 0xb071, 0x4347, 0x4320, 0x25fa, 0x4078, 0xb25d, 0x1e50, 0xa645, 0x2831, 0xbf51, 0x4092, 0x19b4, 0xa3f1, 0xbada, 0x4036, 0x42a7, 0x09ac, 0x1f55, 0xb247, 0x2753, 0x426b, 0x40a5, 0x4223, 0x40f3, 0xbfae, 0x292e, 0x1f9b, 0x46cc, 0x181d, 0x40ca, 0x413f, 0x0b7e, 0x423a, 0xb264, 0x43b4, 0x214a, 0xad2e, 0x215a, 0xba7f, 0x409b, 0x4568, 0xaf04, 0x2c49, 0xa536, 0x425e, 0x42fe, 0xa0ba, 0x4107, 0xbf77, 0xb019, 0x1640, 0x4336, 0x2631, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x7f732ac7, 0xd652158c, 0xc810dc10, 0x451cfb90, 0x698e290f, 0x67d2fe9c, 0xfa98fc70, 0xc792f08a, 0x938d63b5, 0xf0a64fe6, 0xd6444518, 0x197a89a8, 0x8b76fcf7, 0x660046dd, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x00001ab4, 0x0000005a, 0x00000000, 0x00000000, 0x00000000, 0x0000189c, 0x00000031, 0xffffffff, 0x000000ac, 0x00000000, 0x00000000, 0x197a89a7, 0x00000000, 0x8b76fcc7, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4143, 0x4476, 0x3a03, 0x419e, 0x43b2, 0x0275, 0x1d6f, 0x4247, 0x28b0, 0x412b, 0x405c, 0x4317, 0xb2f8, 0xb0d0, 0x4669, 0xbf17, 0xb2cf, 0x2c9b, 0x0e98, 0xba5c, 0x4230, 0x2d3b, 0x28af, 0x46bd, 0x423d, 0x429e, 0xbae4, 0x4683, 0xba7e, 0x4359, 0xb0cb, 0xbfcd, 0x0cd3, 0xa0ac, 0x41a5, 0x1bcb, 0xba6f, 0x44f1, 0x4058, 0x0d71, 0x46cb, 0xbfb0, 0x4310, 0x0701, 0x4026, 0x4320, 0xb2fe, 0x4317, 0xb28d, 0xb269, 0xb2d7, 0x43f5, 0xbaed, 0x40d1, 0x42cd, 0xbfdc, 0x075b, 0x4183, 0x12da, 0x4646, 0xb258, 0x2fa1, 0xba16, 0xba7f, 0xb2f9, 0xa1b7, 0x4198, 0x3436, 0x428a, 0x41da, 0x0f61, 0x43c0, 0xb2a7, 0xba6e, 0xb210, 0x1820, 0x1923, 0xb278, 0x4136, 0x2035, 0x429e, 0xb2c2, 0xbf77, 0xb260, 0x4616, 0x1d26, 0x4176, 0x42fc, 0x43c9, 0xa2cf, 0x454d, 0x4342, 0x2e4c, 0xbf16, 0xb09b, 0xbaea, 0x438f, 0xbf65, 0xb0a5, 0xa6ba, 0x1f3d, 0x415f, 0xb296, 0x4113, 0x03e8, 0x2fa2, 0x19e9, 0x2348, 0x44e3, 0xb0d0, 0x43b4, 0x28e6, 0x1b96, 0x41d4, 0x0f21, 0x439d, 0xb07b, 0xbf74, 0x3b03, 0xba3f, 0x1942, 0x459e, 0x194f, 0xb0a7, 0x174e, 0xbf92, 0xb00d, 0x3171, 0x1f1d, 0x436c, 0xbf86, 0x15ba, 0x3e2f, 0x1711, 0x4351, 0x2b18, 0x418b, 0x0a50, 0x40a0, 0xba20, 0x4154, 0x4174, 0x1d34, 0x3da5, 0x41f4, 0x403a, 0x4201, 0xba51, 0x33d0, 0x1823, 0xb0b7, 0x4396, 0x40c1, 0xbf96, 0xb212, 0x23d4, 0xa987, 0x4344, 0x3df9, 0x32bf, 0x4082, 0xb297, 0x4174, 0xac14, 0x4297, 0xb2e2, 0xb2e8, 0xbfaa, 0x429e, 0xb0b8, 0x465f, 0xba46, 0x433d, 0xa49c, 0x432d, 0x3e9f, 0x1e23, 0xb2a2, 0xa5aa, 0xb0cf, 0xba2f, 0x41f9, 0xbfce, 0x41a6, 0x41af, 0x29a7, 0xbf59, 0xb004, 0x4003, 0xbf00, 0x3449, 0x40b1, 0x116e, 0xb027, 0xb25c, 0xabd8, 0x1bad, 0x0806, 0x0827, 0x46e1, 0x41b0, 0x419d, 0x1e88, 0xb213, 0xbad2, 0x341a, 0xafb2, 0x4416, 0x429e, 0x2857, 0x40a7, 0xba0c, 0x4205, 0x0eeb, 0x43b9, 0xbf2d, 0x41e5, 0x1a16, 0x45a0, 0x423c, 0xb00d, 0x1f94, 0xbae0, 0xbf1b, 0x42d0, 0xb071, 0x4345, 0x4340, 0x41ab, 0xb067, 0x3432, 0xbf1d, 0xb2ef, 0xb061, 0xba7f, 0x3047, 0xa46e, 0x429e, 0xbaf6, 0x41b3, 0x45ec, 0xb20a, 0xb2d5, 0x4360, 0x2af9, 0xb061, 0x2ed2, 0x1329, 0x1c60, 0x4151, 0x1351, 0x465d, 0x40f1, 0x4388, 0x408b, 0xba1e, 0xbfd0, 0xbfb6, 0x4132, 0x074b, 0x18de, 0x420a, 0x43ec, 0xb2e0, 0x4319, 0x41a9, 0xb0e2, 0x4190, 0x400d, 0x4159, 0xba5f, 0xa852, 0x4346, 0xb0b5, 0xbf44, 0xa4e4, 0x1b56, 0x1a5d, 0xba77, 0xbae3, 0xb2be, 0x405a, 0x41e3, 0x3cc6, 0x1ebf, 0x19c2, 0x41ed, 0x4498, 0x1c91, 0x3ae6, 0x42b5, 0xb2e3, 0x18f6, 0xaa8b, 0x45d8, 0x414a, 0xbf6a, 0x4172, 0x4017, 0x1a6f, 0x45d8, 0xa35e, 0x4259, 0xba00, 0x1ede, 0x4125, 0xba04, 0xb2bc, 0x419b, 0x18ef, 0xb20b, 0x18bc, 0x449a, 0xbad5, 0x40b3, 0x42d1, 0x2138, 0x1f04, 0xb266, 0x410f, 0xb04f, 0xba03, 0xbf44, 0x1a63, 0xb276, 0xb2ff, 0x408e, 0x2b75, 0x468d, 0x402e, 0x4265, 0x1e0a, 0x1e17, 0xba0d, 0x0383, 0x44d5, 0x1333, 0x1bf7, 0x408e, 0x0079, 0x4227, 0x43ba, 0x41f2, 0xa9a9, 0x4072, 0x42ca, 0x124f, 0xba40, 0xba14, 0x4146, 0xbf03, 0x426f, 0xba4f, 0xba49, 0xb28b, 0x4368, 0xbf47, 0x4315, 0x3f70, 0x43ed, 0x310d, 0xb013, 0xbacc, 0xbfc0, 0xbf90, 0x4272, 0xbf85, 0xb06d, 0x4162, 0x4155, 0x1d1a, 0xa425, 0x08d3, 0x1f77, 0xb0a4, 0x1e2b, 0xbf00, 0x40f9, 0x2f9d, 0x3eea, 0xb035, 0x42a1, 0xbaf2, 0x2410, 0xa6e0, 0xbf6f, 0x4335, 0x4319, 0x43bc, 0x1f9b, 0x41e0, 0x1b46, 0xbfe0, 0x4297, 0x4177, 0x19ef, 0x42bb, 0x4075, 0x40d4, 0x42cb, 0x1db4, 0xb251, 0x400a, 0xbfa3, 0x441c, 0xb2ce, 0x4187, 0x4365, 0xba30, 0x43a9, 0x42b6, 0x4330, 0x41eb, 0x1341, 0xb016, 0xba5b, 0x4208, 0xbff0, 0x42ba, 0x4001, 0x42b4, 0xb243, 0xb2bd, 0x1cca, 0x420a, 0xa277, 0x4129, 0x40c6, 0x187b, 0x3b6a, 0xbfc9, 0x4083, 0xb2a0, 0x07bb, 0x413d, 0x4116, 0xae4f, 0xb097, 0x39d7, 0xba11, 0x43d3, 0x42b9, 0x2daf, 0x43cd, 0xb21e, 0x4185, 0x4203, 0xba2e, 0xb21a, 0x4095, 0xbf2d, 0x4230, 0x42c8, 0x4277, 0x1e62, 0x19ae, 0x4309, 0xba43, 0xba09, 0x415b, 0x40b8, 0x4271, 0xaf78, 0xb298, 0x434f, 0x2fe1, 0x414a, 0x43b1, 0xbf72, 0x4043, 0xbff0, 0x0c11, 0x4161, 0x40e6, 0x191d, 0xb222, 0xa0d6, 0x4247, 0x2b58, 0x1941, 0x4157, 0x4223, 0xac08, 0xb0a0, 0x1aa4, 0x42fa, 0x18bb, 0x4080, 0x1a11, 0xb26c, 0x3e43, 0x4089, 0x41ac, 0xba1c, 0x4326, 0x42b3, 0xbf9c, 0x21eb, 0xba4a, 0xbf3e, 0x0c75, 0x2d1e, 0x43a7, 0xb200, 0x4005, 0x42a7, 0x1848, 0xb2db, 0x05fb, 0xbae2, 0xbf65, 0x407e, 0xb2f1, 0x418a, 0x43db, 0xb2a6, 0x4244, 0xbf2a, 0xa9d3, 0x0194, 0x42dd, 0x25d1, 0xb2d1, 0x41f3, 0x42da, 0x3808, 0x412a, 0xb287, 0x44d3, 0x1977, 0x41c2, 0x1091, 0xbf28, 0xbae4, 0x4144, 0xba62, 0x4643, 0x40f8, 0x4194, 0x1bec, 0x407a, 0xb0f3, 0xb234, 0xba45, 0xb0d3, 0xba4e, 0x404e, 0xbf76, 0xb2e2, 0xbaec, 0x4665, 0xaa4c, 0xb20e, 0xba08, 0xa8b3, 0xb072, 0x1b10, 0xbacb, 0xb064, 0x41ef, 0x44f2, 0xaaa2, 0x4362, 0x1eb0, 0xa948, 0x4236, 0x46c4, 0x1b32, 0x40f1, 0x2641, 0xab44, 0x4222, 0x42be, 0x1bc0, 0x4639, 0x4135, 0xbfe4, 0x434a, 0xba33, 0x1ee0, 0xb229, 0xb06d, 0x44e9, 0x04af, 0x40d1, 0x4456, 0x46e9, 0x4112, 0x18ce, 0x2066, 0xbfe8, 0x4240, 0xbaf0, 0xbfcf, 0x4691, 0x079d, 0x438f, 0x406e, 0x4075, 0x2821, 0xa06e, 0x4167, 0x4385, 0x3a13, 0xbac8, 0x45e4, 0x429d, 0x42b0, 0xb20b, 0xbfb1, 0xa2ee, 0x35d9, 0x1d65, 0x152c, 0xb04e, 0x0f48, 0x254e, 0x1ce5, 0x2e0f, 0x3304, 0xb0bf, 0xb2de, 0x1c11, 0x1e33, 0x1957, 0xbf60, 0x19b6, 0x400b, 0x401a, 0xbfd9, 0x4621, 0x19a6, 0x2a35, 0x26e2, 0xb294, 0x412b, 0x4165, 0x1b4f, 0xad1b, 0x2f76, 0x1a42, 0xbf65, 0x3406, 0xb20a, 0x43f5, 0xba32, 0x45db, 0x23fe, 0xbad4, 0x1ab0, 0x222a, 0xb22a, 0xb206, 0x43cd, 0x13e5, 0x1c19, 0x2ac8, 0x268a, 0x4319, 0x1c7b, 0x1f02, 0x45f3, 0xb25c, 0x400e, 0xa29c, 0xb041, 0xbfd2, 0x4261, 0xb0dc, 0xa298, 0x447b, 0xbfc3, 0x2c44, 0xba23, 0x41f3, 0x43a1, 0x41ec, 0x43ad, 0x41e2, 0x401d, 0xb096, 0x40dd, 0x3336, 0x415c, 0x4037, 0x42e5, 0x4323, 0xb257, 0xb2a7, 0x1cc7, 0xba28, 0x40de, 0x40dd, 0x4069, 0x1bda, 0x46e4, 0xbf4d, 0x010c, 0x1533, 0x41f8, 0x42c3, 0x41ff, 0x14cd, 0x4291, 0x1da6, 0x16e2, 0x3752, 0xafe8, 0x4192, 0x46c4, 0x4204, 0x40b8, 0x222c, 0x23e3, 0xaca3, 0xbf5c, 0x02c8, 0x4219, 0xb078, 0x205d, 0x4267, 0x41c0, 0xba36, 0x1c8d, 0x35be, 0xb2d4, 0xb2fe, 0x40e7, 0x404f, 0x43a1, 0x3017, 0x0f83, 0xb08b, 0x40ff, 0xbf68, 0x40f7, 0x2d84, 0x4072, 0xb2a9, 0x089e, 0x1c2f, 0x1ccf, 0x412c, 0xba2b, 0x1cda, 0x19d2, 0xb04a, 0xb053, 0x2419, 0x422f, 0x4269, 0x1997, 0x0701, 0x1326, 0x0e29, 0x4105, 0x425b, 0x4111, 0x4677, 0x4373, 0xbf6c, 0x1ad8, 0x4305, 0xba7c, 0x064b, 0x34a4, 0x4241, 0xb2b5, 0xbf0e, 0x4282, 0x187e, 0xbaf4, 0x460d, 0x417c, 0xba19, 0xbf90, 0x1e52, 0x42f2, 0x41c9, 0x23d2, 0xb221, 0xb2cc, 0x1f32, 0xba0d, 0x43b3, 0x408e, 0x4040, 0x43d5, 0xb05f, 0x40bb, 0x44f8, 0x179c, 0xa95e, 0x18de, 0xb229, 0x42ce, 0x4310, 0xbf45, 0x426a, 0xab45, 0x4339, 0x40a8, 0xabbe, 0x43d3, 0x3ba2, 0x1a98, 0x43f4, 0x4364, 0xaa92, 0x1878, 0xaf21, 0x4422, 0xa6c1, 0xa127, 0xba16, 0x4032, 0x1ed2, 0x4354, 0x46c2, 0x4275, 0x3beb, 0xb240, 0x4029, 0x417c, 0x41a4, 0x42c1, 0xbf5b, 0x411a, 0x4134, 0x3a11, 0x39e1, 0x2cc5, 0xb0f1, 0x35af, 0xb2b3, 0xb214, 0x4312, 0xbfc3, 0x4151, 0xb227, 0x4190, 0xa889, 0x402e, 0xaa1c, 0x4236, 0xbf72, 0x1ab4, 0x40f2, 0xab93, 0x33d2, 0xb26d, 0xa602, 0xb090, 0xb0af, 0x4333, 0xba5e, 0x4036, 0x411a, 0x3860, 0xb2dd, 0x4322, 0x41f9, 0xbff0, 0x031c, 0x4093, 0xb05a, 0x4284, 0x40fe, 0xb031, 0x31d3, 0xb060, 0xbf7f, 0xb02b, 0xb254, 0x42e4, 0xb247, 0xbf00, 0xb221, 0xb040, 0xb2e4, 0x40b1, 0x1dff, 0x400c, 0xba6e, 0x4457, 0x1e3f, 0xb2bf, 0x40de, 0xbfc3, 0x1e09, 0x42ed, 0xb2a3, 0xba5d, 0xbfe0, 0xba5f, 0xba77, 0x1976, 0x42c1, 0xb278, 0x46e5, 0x40b7, 0x4495, 0x43b4, 0xb00e, 0x40f8, 0x27fa, 0x4170, 0x422a, 0x411e, 0x1b39, 0x41e0, 0x447f, 0xbf94, 0xa87e, 0x41a6, 0x45f5, 0xb204, 0x43bb, 0xb270, 0xb0f6, 0xb0f3, 0x0fcf, 0x2554, 0xb202, 0x4009, 0x4056, 0x37e1, 0x1d32, 0x41e1, 0x1988, 0xba4e, 0x468b, 0x4186, 0x42d5, 0x42a4, 0xbf21, 0x3a5e, 0xbff0, 0x13c6, 0x2352, 0x1def, 0xb26e, 0x41ed, 0x429b, 0x4253, 0xb2d1, 0x4180, 0xbf6f, 0x427b, 0x0ff2, 0xa388, 0x42e5, 0xb2aa, 0x1c71, 0x429b, 0xb072, 0xb22d, 0xb211, 0xb066, 0x173e, 0x1d80, 0x4253, 0xa2d6, 0x4033, 0x404a, 0xb2a5, 0x413b, 0xbade, 0x417c, 0x40c2, 0xb257, 0x1fd5, 0x0bb8, 0x42aa, 0x1d98, 0x43e6, 0xbfba, 0x1e34, 0x020a, 0xb238, 0x3c80, 0x4023, 0x4000, 0x437c, 0xbf80, 0x15e7, 0x1366, 0x42d7, 0x44bd, 0xbf2b, 0x4274, 0x28cf, 0x4376, 0x46f5, 0x0319, 0x465b, 0x4087, 0x4106, 0x4135, 0x0ef5, 0xb2e9, 0xb07d, 0x415a, 0xba1f, 0x4368, 0x40af, 0x469c, 0x41bf, 0x2d0c, 0x4305, 0x4222, 0x41e3, 0xbf98, 0x4649, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xbb7d36f6, 0x99698622, 0x49d98852, 0x36cb9669, 0x824878bf, 0xc93e95e5, 0x63323f2d, 0xc8f7f318, 0x96c967ec, 0x5a2d76e7, 0x8dfbec30, 0x6fa00b5f, 0x2d920f82, 0x5d28d234, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0x004003e8, 0xfa000000, 0x000022ea, 0x00000000, 0x00000000, 0xffffffff, 0x9681941b, 0x00000000, 0x9681941b, 0x000003e8, 0x000003e8, 0x000001f4, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x43de, 0x4327, 0xb263, 0xa931, 0x4228, 0xb061, 0xbac6, 0x434d, 0xa868, 0xbf90, 0xba1c, 0x43bf, 0xbf70, 0x4229, 0x42b0, 0x40f0, 0x32dd, 0xba5d, 0xbf85, 0x43a4, 0xb2af, 0xb2b1, 0x4021, 0x2d77, 0xa666, 0x4176, 0x4089, 0x426e, 0x41de, 0x4345, 0x400b, 0x4056, 0x1bee, 0x19d7, 0xa0e7, 0x40b1, 0x2328, 0xbf06, 0xa385, 0x465d, 0xb277, 0x4664, 0x1a4b, 0xbad8, 0x0428, 0x0685, 0x4028, 0x300a, 0x4349, 0xba0f, 0xb2e3, 0x4662, 0x40d7, 0x435e, 0x0b39, 0x421f, 0x4670, 0xb2d6, 0x17aa, 0x4652, 0x4688, 0x1d08, 0x0544, 0x1fd9, 0xbfb3, 0x4166, 0x40ef, 0xbff0, 0xb0c9, 0x047f, 0x4063, 0x1011, 0x1f38, 0x4475, 0x427b, 0x4149, 0xba6c, 0x428f, 0x43da, 0xb2c2, 0x1f1a, 0x40d0, 0x4283, 0x1b46, 0x2aad, 0xb26d, 0x42b6, 0xa275, 0xba02, 0x40bf, 0x047c, 0xb2de, 0xbf60, 0x41a9, 0xbf28, 0x43ae, 0xba2b, 0x1dbc, 0x0735, 0x4234, 0x408b, 0xba25, 0x1086, 0x420d, 0xb074, 0xb290, 0x41e9, 0x410d, 0x4256, 0x42a2, 0x40bd, 0xabce, 0x4380, 0x4015, 0x4169, 0x405d, 0x3fbd, 0xb253, 0x433a, 0xbfd7, 0x41b2, 0x15be, 0x188a, 0x0fc1, 0x2952, 0x35a4, 0x40f1, 0x2a4c, 0xbfbd, 0xb2e4, 0xb23a, 0x1f28, 0x4630, 0x388c, 0x4147, 0x40f0, 0x4567, 0xbac7, 0xbfa0, 0x436c, 0x3e60, 0xb2d1, 0xb0e4, 0xb26a, 0x1830, 0xba33, 0xb26f, 0x2b61, 0x400b, 0x4346, 0xb069, 0x4039, 0x0aaa, 0xba75, 0x43d5, 0x417b, 0x42ec, 0xae57, 0xbf6e, 0x41cf, 0x3075, 0xb29b, 0x15ef, 0x1b99, 0x435d, 0xb215, 0x42fd, 0x4234, 0x2e88, 0xba7d, 0xbade, 0xba31, 0xbad9, 0xa2a3, 0x42f1, 0x1bb3, 0xbafa, 0xa940, 0x40a6, 0x3689, 0xb2ba, 0xbfaf, 0x4052, 0x1a92, 0xb2be, 0x462e, 0x41d1, 0x4145, 0x1619, 0x46c2, 0x3167, 0x409e, 0xba38, 0xb20f, 0xbacc, 0xb238, 0x4297, 0xbfa1, 0xbaf4, 0x38a8, 0xb293, 0xa847, 0x4138, 0x4175, 0x40fa, 0x259a, 0x4244, 0x4029, 0x3497, 0x15f3, 0xb07c, 0xbacb, 0x433f, 0x42e5, 0x3baf, 0x4029, 0x1aab, 0x20de, 0xa582, 0xbf87, 0xb2ff, 0x4098, 0xbace, 0x415c, 0x40cf, 0xb272, 0xbfdf, 0x40e0, 0x42dd, 0x4122, 0x41ba, 0xb2e2, 0x2632, 0x4153, 0xbf7d, 0x3131, 0x3b4d, 0x43ba, 0x41ab, 0x3b6c, 0xb2b3, 0x31e5, 0xba05, 0xb268, 0x1431, 0xba04, 0x4556, 0x43ab, 0x421c, 0xb2a4, 0xb23c, 0x410e, 0x42c9, 0x4063, 0x1da5, 0x402f, 0xb288, 0x0b26, 0x0570, 0xb03b, 0x4260, 0x4096, 0xbf8e, 0x10ad, 0x3f3a, 0x418e, 0x0b8b, 0x2f4f, 0x405a, 0xb00a, 0xb07e, 0x04ce, 0x41fa, 0x4271, 0x30c8, 0x40e5, 0x43f5, 0x3887, 0xa453, 0x4608, 0xbf71, 0x0220, 0x4540, 0x4188, 0xbff0, 0x434c, 0x3c0b, 0x41eb, 0x2d26, 0x4675, 0x4149, 0x4614, 0xb238, 0x308c, 0x4005, 0xbf5f, 0x4041, 0x0a0d, 0xb258, 0x43f4, 0x4316, 0x4170, 0xb2dc, 0xbfb6, 0x2875, 0xb052, 0x40dd, 0xbfb4, 0x4208, 0xba45, 0x409b, 0x4385, 0xa8f3, 0xb213, 0x42cb, 0x3f92, 0x3ae4, 0xb2f3, 0x3366, 0xb236, 0x4399, 0x400d, 0x4128, 0x4419, 0x34de, 0xa079, 0x43d7, 0xbaf8, 0x4277, 0x0594, 0x42fd, 0x4413, 0x43fe, 0xbf78, 0xb29b, 0x41ee, 0x185b, 0x19f6, 0xba51, 0x4012, 0x44d8, 0x05fc, 0xa2ea, 0xb2f6, 0x415a, 0xb2a7, 0x310d, 0x19e3, 0x408f, 0x19c7, 0xa5d4, 0xb05a, 0x1485, 0x4162, 0x41f0, 0x4200, 0x402a, 0xbf49, 0x0fdf, 0x185a, 0x1e02, 0xb2ab, 0xba13, 0xbaca, 0x46ed, 0x1e91, 0x42f5, 0xbf77, 0x459b, 0x42ae, 0x40eb, 0x19c3, 0x4213, 0x2acb, 0x1080, 0x41c1, 0x42fb, 0xb211, 0xab6c, 0x309e, 0xba6e, 0xb03e, 0x43fa, 0x42c5, 0xa06c, 0x1777, 0x3986, 0xbf51, 0x40d2, 0x4691, 0x22f9, 0x43fe, 0x40c2, 0x4215, 0xb028, 0x43a4, 0x42ae, 0xb2bd, 0x429e, 0x433d, 0x42e7, 0x42f5, 0xa94c, 0xbfaa, 0x34b2, 0xba3d, 0x424d, 0x4291, 0x4120, 0x1bb9, 0x43c2, 0x1ff7, 0x40fa, 0x342f, 0x43fe, 0x42bf, 0x433e, 0xbfb5, 0x4076, 0x3aaa, 0x4158, 0xb2bf, 0x40d7, 0x4068, 0xbf6f, 0xb036, 0x40ed, 0xbaec, 0x18fe, 0x00f2, 0xbadd, 0x439b, 0xa23d, 0x42a5, 0x411a, 0x41d8, 0x16cf, 0xba28, 0xbfe2, 0xba20, 0x43cf, 0xb2ae, 0xb2bc, 0xb2f7, 0x448c, 0x43dc, 0x43ae, 0xb2a8, 0xa5a6, 0x4216, 0x4295, 0x344b, 0x1da4, 0x4273, 0x408f, 0xb21d, 0xa9a9, 0x43db, 0x42b0, 0xbf5a, 0x4640, 0xb20a, 0xa7bb, 0x417c, 0xb26b, 0xbfd7, 0x1e2c, 0x4250, 0xb053, 0x4071, 0x424f, 0xb0d4, 0x4373, 0x429a, 0xba12, 0x4078, 0xba4e, 0x2762, 0xa5a4, 0xb200, 0x44b3, 0x2257, 0x0339, 0x42b5, 0x1a8f, 0x45b2, 0x4033, 0xbaed, 0x1f10, 0x1b45, 0xbae3, 0x3bae, 0x40d9, 0x03ff, 0x4175, 0xbf45, 0x32f9, 0x4053, 0x3ad1, 0x27bf, 0x40c4, 0xbf60, 0x40d2, 0x4202, 0x41f2, 0xb2be, 0x4341, 0xb25c, 0x1b18, 0x40bd, 0xb089, 0xbae2, 0xbf5f, 0x4403, 0x4607, 0xa684, 0xa2f3, 0x40ae, 0xba3b, 0x4358, 0x422c, 0x4167, 0x40a7, 0x4215, 0x4440, 0xb219, 0xbfd0, 0x40d5, 0x4038, 0xb2ca, 0x42e9, 0x43af, 0x42cb, 0x413c, 0x4158, 0x44a3, 0x422f, 0xbadf, 0xb279, 0xb286, 0x1f1e, 0xbf09, 0x4319, 0xba13, 0xba31, 0xa86a, 0x4367, 0xbf00, 0xb0d8, 0x423f, 0x413a, 0x44d5, 0xbac4, 0xad13, 0x4093, 0x4149, 0x41fb, 0x422b, 0x1e34, 0x3f5e, 0xb246, 0x41b8, 0xbf70, 0x435d, 0xbfdd, 0x01b2, 0x420b, 0xbfc0, 0x464f, 0xb034, 0x1c2d, 0x1a10, 0xbfd5, 0xa298, 0x412d, 0x41a3, 0x42de, 0x42b2, 0xba1e, 0xb298, 0x4170, 0x1827, 0x4093, 0xb015, 0xba4a, 0x2cf0, 0xb2d9, 0xb067, 0x0c34, 0x35ed, 0x3b0a, 0x085a, 0x46ab, 0x158d, 0x3889, 0x42ab, 0x1c3a, 0xbfc6, 0x372b, 0x17bc, 0x3899, 0x4611, 0x1a8a, 0xbf5d, 0x44d8, 0xb042, 0x1ab4, 0xb0ac, 0x438a, 0x41af, 0xba06, 0xb25e, 0x164a, 0x4067, 0xb01c, 0x43eb, 0xbf2e, 0x1b94, 0xba3c, 0x1494, 0xbf77, 0x41da, 0x09e3, 0xbadc, 0xa22f, 0x3fcc, 0x1f3d, 0xba4a, 0x400a, 0x432e, 0xb289, 0x1b35, 0xb07e, 0x3cc2, 0x428c, 0x2025, 0xafee, 0x251a, 0xbfb3, 0x44cb, 0x45a8, 0xb294, 0x4328, 0xb08c, 0xb0b4, 0xba26, 0x4097, 0xaa3d, 0x40a6, 0x434b, 0x4560, 0x4660, 0x1f5f, 0x0839, 0x42f0, 0xb2b9, 0x465b, 0x41b9, 0x194f, 0x405a, 0x400d, 0xa719, 0x2c50, 0x1e4e, 0xb242, 0xba30, 0x42ae, 0xb24f, 0xbf43, 0x4681, 0x1a48, 0x1d26, 0xb0fd, 0x4004, 0x423e, 0xa9f0, 0xba40, 0xbaff, 0x43e1, 0xa99b, 0x4432, 0x41f4, 0xbf26, 0x0a13, 0x1821, 0x2ba4, 0x44b9, 0xba22, 0x322b, 0x1c9a, 0x432c, 0xba70, 0xb234, 0x3ca7, 0x4547, 0x4078, 0x4144, 0x40ec, 0xad9a, 0x04d0, 0x4327, 0x37e6, 0x09ea, 0x3f50, 0x1526, 0x407d, 0x1835, 0x4402, 0x1e38, 0x0896, 0xbf04, 0x1909, 0xb2db, 0xb297, 0xb2a2, 0x012d, 0x389d, 0x0b0f, 0x0c37, 0xb0af, 0x4460, 0x4057, 0x44cb, 0x41c7, 0xbaef, 0x429c, 0xaf1a, 0x46dd, 0x4305, 0xb00e, 0x4071, 0x1817, 0xbf34, 0x4349, 0x01c4, 0xb0f7, 0xb209, 0xbfc0, 0x4362, 0xa7c2, 0x43bc, 0x45cd, 0x2edb, 0xb28f, 0xbf53, 0x4102, 0x44cc, 0x431a, 0x1c7c, 0xb207, 0x445e, 0x432c, 0xb099, 0x1de3, 0x46b2, 0x2163, 0x429a, 0x4108, 0x1361, 0xb09d, 0xbac4, 0xb285, 0x0eaa, 0xbf65, 0x437f, 0x41b1, 0x4197, 0x1ff3, 0x2b09, 0x403b, 0x41b8, 0xb231, 0x42c8, 0x42cf, 0x43e6, 0x0f35, 0x1989, 0x1ff2, 0x41c6, 0xbfd0, 0x1e9b, 0x434a, 0x42e3, 0xa90c, 0x40a4, 0x4208, 0x4200, 0xbf8a, 0x1b32, 0x410f, 0x2ee4, 0x33ed, 0xb240, 0xb0f4, 0x40c3, 0x4250, 0x2be7, 0xb019, 0xb2dd, 0xbfd7, 0xb0d4, 0xba08, 0xb2fb, 0x41b7, 0x1308, 0x4267, 0x41c0, 0xbfd0, 0x434c, 0xb06c, 0x4540, 0xb204, 0x1a34, 0x4140, 0xbad7, 0x1cc9, 0x3e9e, 0x390a, 0x40fa, 0x404e, 0xbfaa, 0x08cb, 0x4261, 0x4583, 0x339d, 0x1e37, 0x42fc, 0x437e, 0x435b, 0x3be3, 0x1891, 0xb222, 0x4245, 0x4130, 0x08b8, 0x447a, 0xb226, 0x034a, 0x4392, 0x4379, 0x40ec, 0xa38a, 0xba18, 0x42d9, 0x02f5, 0x4337, 0x41da, 0xbfc6, 0x4491, 0x4071, 0xb215, 0x1b51, 0x41dd, 0x41cd, 0x4223, 0x00af, 0xb2a7, 0x42e5, 0xb020, 0x43e6, 0xb27b, 0x439a, 0x4213, 0xbac3, 0x426a, 0x3d19, 0xa5b8, 0xb0f3, 0xbfc0, 0xba34, 0xba5e, 0xbfc6, 0x42f6, 0x103e, 0x45c1, 0x4482, 0x42ab, 0x2d50, 0x1f4e, 0x411b, 0x4316, 0xb01d, 0x45ee, 0x1db2, 0xb0a5, 0xbf6e, 0x0cf1, 0x1ac8, 0xba74, 0x43ca, 0xba28, 0xa979, 0x18a9, 0xbf01, 0xa761, 0x41fb, 0xb23a, 0x4242, 0x4358, 0x4163, 0xaabf, 0x13cb, 0xbf08, 0x189a, 0xb257, 0x43ee, 0x4085, 0x46f2, 0xb281, 0x45e3, 0x1a1f, 0xa8aa, 0x45e8, 0xba58, 0xb24c, 0x1e6b, 0xb252, 0x41b6, 0x43f7, 0xbf0c, 0xbad8, 0xb276, 0x43b7, 0x4205, 0x283f, 0x4208, 0x2afa, 0x46f1, 0x387c, 0x4269, 0x433a, 0x201c, 0x4072, 0x45bb, 0xbf55, 0x27d4, 0xb0b8, 0x4049, 0x426f, 0x400d, 0x4031, 0x41da, 0x41f1, 0x42c2, 0x4336, 0x432c, 0x4187, 0x445d, 0x423d, 0xb0cc, 0x408d, 0x43d5, 0x401b, 0x4397, 0xb26d, 0xba34, 0x41e3, 0x4326, 0x4355, 0xb01e, 0xbad7, 0xbfdd, 0xba3c, 0xba46, 0xba70, 0xa0a6, 0x1e83, 0x41cd, 0x41f0, 0x4256, 0xbf00, 0xb2df, 0x41c4, 0x45d3, 0xb2c6, 0xa98f, 0xbfde, 0x3ce1, 0x4263, 0x40b3, 0x45b6, 0x445f, 0xbf0b, 0x1d50, 0xb238, 0x2de2, 0x2b7f, 0xb0b8, 0x1629, 0x411b, 0xaeab, 0x4306, 0x2c95, 0x02e2, 0x4208, 0xba37, 0x25ef, 0xb08d, 0x4352, 0x2571, 0x425e, 0xba43, 0xb21d, 0xbfd2, 0x412b, 0x4644, 0x260f, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd8508733, 0x37686f1d, 0x97ff025a, 0x7e27c2d9, 0xe771ba18, 0xaba51d16, 0x78d8fc78, 0x633a3766, 0xf31e6a39, 0xaa1cca97, 0x3f7fc03c, 0x49e22f13, 0xd6f3fcdf, 0x6bb27080, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0xffffcb9d, 0x00000000, 0xe1000000, 0xffffffff, 0xffffff1e, 0xffff9dcb, 0xe0000000, 0xdfcfffff, 0x49e23000, 0x00000000, 0x00000000, 0xaa1ccb83, 0xd6f3fcdf, 0xaa1cc3e3, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x46f3, 0xbfa0, 0xba72, 0x422f, 0x4194, 0x403f, 0x418a, 0x1b2f, 0x1a10, 0xb282, 0xb004, 0xad90, 0xa257, 0x43cc, 0x1e2d, 0x44a9, 0x4093, 0xba76, 0x4158, 0xbfe0, 0x2347, 0x434f, 0x0059, 0x45eb, 0xbf72, 0x40e7, 0xb086, 0x1feb, 0x4339, 0x40e6, 0x42bd, 0x409b, 0x4000, 0x4420, 0xba49, 0x4276, 0xb210, 0x2b38, 0xba5c, 0x1a1e, 0xab1f, 0x2962, 0x20bd, 0x42ae, 0xb0ab, 0x2234, 0x3607, 0x0b88, 0x29e7, 0x4378, 0xb06b, 0x40ec, 0xbf28, 0x43f2, 0x4135, 0xb00e, 0x1c37, 0x4025, 0x3570, 0xbf80, 0x42db, 0x249a, 0x4328, 0x4211, 0x41c7, 0x4239, 0x2924, 0xa5ef, 0x0387, 0x1a9f, 0x4352, 0x1f8f, 0xb031, 0x41bd, 0x1cdd, 0xb2ed, 0x4002, 0x4182, 0x07c7, 0xbf77, 0x406f, 0x431f, 0x42d9, 0x3e9c, 0xbf60, 0x4377, 0x4277, 0x4314, 0xb001, 0xbf6b, 0x4070, 0x41f7, 0xbadf, 0x0139, 0xb246, 0xba0a, 0x439e, 0x433b, 0x18e3, 0x320f, 0x0f77, 0x45a2, 0x1d6b, 0x1a0b, 0x42eb, 0xbf01, 0x1a8f, 0x4286, 0xa09a, 0xbae4, 0x431c, 0x4307, 0x3e46, 0x430d, 0x42ce, 0x433f, 0x109e, 0xa573, 0xba79, 0x1ef6, 0x41cf, 0x29c8, 0x0dca, 0x0e56, 0xbae7, 0x3d0d, 0x4374, 0x1e44, 0x0b81, 0xba51, 0x4087, 0x4425, 0x4283, 0x420e, 0xbf94, 0x409f, 0x42be, 0x4353, 0x436d, 0x4291, 0xbfc4, 0x4365, 0x40b3, 0x418f, 0xbf3e, 0x1e3d, 0x46ba, 0x1f8c, 0x42ea, 0x1c25, 0x4135, 0xb21a, 0xb29b, 0x16e9, 0x401d, 0x4653, 0xa607, 0x43e2, 0x0ec5, 0x4201, 0xbae0, 0xba26, 0xa713, 0x43c5, 0x23cf, 0x10a8, 0xb08e, 0x4232, 0x43cb, 0x43d9, 0x0774, 0xb03b, 0xbf62, 0x4205, 0xbfa0, 0x4661, 0x403b, 0xbf17, 0x40bb, 0xbaee, 0x42a1, 0x42b3, 0x463e, 0xafdf, 0x3543, 0x3528, 0x1eea, 0x4038, 0x1dd7, 0x38ba, 0x4092, 0x41b0, 0xba32, 0x4416, 0x44fd, 0xbfc8, 0x439f, 0x4319, 0xa47d, 0xb275, 0xb244, 0xbf95, 0x1453, 0xb04f, 0x1c40, 0x3477, 0xbf76, 0xaeb7, 0x0406, 0xbad8, 0xbf5e, 0x1a0d, 0xb2d6, 0xba5e, 0x437b, 0x0b3c, 0x1d54, 0x3c58, 0x404e, 0x427b, 0x25f8, 0xbfa0, 0x4053, 0x127f, 0x42e6, 0x438b, 0xbf23, 0xb2d9, 0x4389, 0x4350, 0xade8, 0x439e, 0x4485, 0x3ec6, 0x19a8, 0x0fe2, 0x42fc, 0x4262, 0xada2, 0x40e6, 0xb0e5, 0x4251, 0xb221, 0x1f4c, 0x127c, 0xbf56, 0x25d6, 0x20f9, 0x4329, 0x4123, 0x413d, 0xb2df, 0xb23b, 0x4653, 0xb2e4, 0xb0a9, 0x42ea, 0xb215, 0x4130, 0x4249, 0x4391, 0xb28f, 0x4127, 0x420e, 0x16ab, 0xa580, 0xb26c, 0x29b2, 0xa29e, 0xb09c, 0xb20f, 0xbfd2, 0x42e1, 0x42cf, 0x0a23, 0x4049, 0x433c, 0x45e0, 0xbf46, 0x407f, 0x408f, 0x4165, 0x4148, 0x19e5, 0x4594, 0xb0f1, 0x438d, 0x4000, 0x42b7, 0x4136, 0x31af, 0x40ea, 0xb0b9, 0x4323, 0x0710, 0xbfa2, 0x38a1, 0x40d2, 0x41a9, 0x4149, 0x0c49, 0x4168, 0x3f89, 0x433c, 0xb2fd, 0x42e3, 0x4353, 0x40a7, 0x43e8, 0xba3a, 0xb266, 0x4397, 0x0a3a, 0x40ab, 0x1fc5, 0x1f5f, 0x4117, 0x4573, 0xb01a, 0x418c, 0x44a9, 0xaa7b, 0x13a2, 0xbfe4, 0xb213, 0x41c3, 0xb28f, 0x2ff1, 0x2600, 0x2452, 0x438c, 0xa2cc, 0x1b55, 0x42ad, 0xb2fc, 0x1067, 0xb2b8, 0x41f0, 0x4625, 0x0ac8, 0xb25e, 0x441f, 0x40a8, 0x43d6, 0x18c7, 0xbadb, 0xb292, 0x4453, 0xbf8b, 0x19d8, 0x1f32, 0x18f9, 0x46b5, 0xb22a, 0x088b, 0xba5a, 0xba7f, 0xb068, 0x03a6, 0x444b, 0xb21c, 0x3014, 0x41cb, 0xbf26, 0x4015, 0xb2c5, 0x1a29, 0x1a7f, 0x0b53, 0x4219, 0x3afd, 0xba1c, 0x0def, 0x3b5b, 0x423c, 0x4003, 0x42a3, 0xb0f8, 0x42ce, 0x4357, 0x41ed, 0x4070, 0x42c4, 0x1c75, 0xa517, 0xb0ab, 0xba3f, 0x4341, 0x2c35, 0x1bd9, 0xbf76, 0x07be, 0x4173, 0x4145, 0xb0a5, 0x11a1, 0x4068, 0x43d8, 0x42e2, 0x40f6, 0xbafd, 0x1fc1, 0x407e, 0xba38, 0x4109, 0xb016, 0x40a6, 0x43d5, 0x4274, 0xbf1e, 0x2433, 0x289d, 0x43ca, 0xb25d, 0xba5d, 0x0a85, 0x43d7, 0x3e24, 0x0e1d, 0x1723, 0x444a, 0x18e3, 0xb2ba, 0xba2c, 0xba55, 0x075d, 0x43e8, 0x0247, 0x40fc, 0x1e94, 0xbfa6, 0x1c10, 0x3a69, 0x4023, 0x4581, 0x1925, 0xb0d7, 0x1f3f, 0xb0a9, 0x45e6, 0x2ec7, 0x41c3, 0x046b, 0x1f29, 0xbf4e, 0xb289, 0xb250, 0x425b, 0x40df, 0xb2b8, 0xab56, 0x1cb4, 0xbac1, 0x4271, 0xbf64, 0x4373, 0x3c6a, 0xbf5a, 0x43bb, 0x429f, 0x1f9e, 0xbac2, 0x404a, 0x0103, 0x4213, 0x46f1, 0x287a, 0x4088, 0xbacf, 0x4331, 0xba4b, 0x32fc, 0x43bb, 0xb02b, 0x4176, 0xa3dc, 0x44e1, 0x1f3a, 0xbfce, 0x3dc6, 0x4227, 0xba11, 0x4148, 0x09b4, 0xbfd3, 0xb245, 0x4394, 0x40ce, 0x07e5, 0x1956, 0x0db9, 0xb238, 0x40f2, 0x412b, 0x1af5, 0xa1f7, 0x466b, 0xba34, 0x4038, 0xa130, 0x404e, 0x19ca, 0x3a04, 0x1d1c, 0x4077, 0x439c, 0xbadb, 0x1f1d, 0x36ff, 0xbf4e, 0x42d1, 0xba5d, 0x4020, 0x0dd2, 0x42a9, 0x1943, 0x405d, 0xba7f, 0x41a1, 0x447d, 0x4181, 0x0963, 0xb038, 0x436e, 0x412f, 0x404f, 0xae5c, 0x41e7, 0x40e5, 0x434b, 0x20c5, 0x1eb2, 0x4010, 0x3a97, 0xbfb1, 0xb047, 0x0ba6, 0x3abe, 0xb252, 0x4104, 0x4346, 0xb206, 0xad63, 0xbf46, 0x4144, 0x41e0, 0x422b, 0xb0ed, 0x4389, 0x423b, 0xba65, 0xbaf3, 0xaaed, 0xb219, 0xba43, 0x243c, 0x197c, 0x421c, 0x18fd, 0x4313, 0x1cae, 0x140a, 0x42a0, 0xbf4e, 0x4015, 0x18f5, 0x42c6, 0x4115, 0x42c0, 0xba0a, 0x437a, 0xbf5f, 0xb0ae, 0x1d9d, 0x4085, 0x4350, 0x1a00, 0x369b, 0xb22b, 0x24f9, 0x4253, 0x360c, 0x41cd, 0x0fb1, 0xb0a6, 0x025d, 0x18c4, 0xb251, 0x40c6, 0x4349, 0xb2cc, 0xa0be, 0x4377, 0x43da, 0x16b2, 0x413d, 0xbf80, 0x44ac, 0xbfdb, 0xb266, 0x0d95, 0x420e, 0xb083, 0x1b6c, 0x4305, 0xbae4, 0xb28f, 0xb236, 0xba09, 0x2da7, 0xb252, 0xba09, 0xa115, 0x447d, 0xb272, 0xbf75, 0x40f6, 0x0b5a, 0xb086, 0x466b, 0x4632, 0xbf0a, 0x43e0, 0x1f0f, 0xb2e7, 0xba3d, 0x36d4, 0x07bc, 0x4172, 0x3697, 0xb221, 0x0e04, 0x28f3, 0x42e6, 0x198a, 0x4089, 0xbf90, 0x3694, 0x2837, 0x4082, 0x4263, 0x4142, 0x4074, 0x2fc6, 0x1e01, 0xab23, 0x0cb5, 0x4033, 0x0067, 0xbfca, 0x111a, 0x4261, 0x29e3, 0x462d, 0xb299, 0x4072, 0xbfe0, 0x4342, 0xbf3d, 0x0830, 0x1afe, 0x419c, 0x1ce9, 0x42e6, 0x40d5, 0x41cc, 0xb072, 0x3bd5, 0xbfdf, 0x1807, 0x44c2, 0xba38, 0x4693, 0x41a2, 0x4072, 0x42e7, 0xbaef, 0x01b3, 0x25c5, 0x4006, 0x096e, 0x4115, 0x0622, 0x43ac, 0xb00c, 0x1f03, 0xbf48, 0x41c4, 0x43dc, 0x415a, 0x20b3, 0x4064, 0xba24, 0x40f8, 0xb23f, 0x0b4b, 0x42ec, 0xb014, 0x4119, 0xbf78, 0x4393, 0x3aa5, 0x410a, 0x0b23, 0x442f, 0xba39, 0x4298, 0xba42, 0x42c5, 0xa365, 0xbfba, 0x40a8, 0x17e3, 0x4057, 0x417c, 0xaa59, 0xafc2, 0x33ad, 0xa425, 0xbfb9, 0x2e0c, 0x0192, 0xaa5d, 0x0448, 0x421e, 0x4357, 0xb2f0, 0x2a8b, 0x378b, 0xb219, 0x416c, 0xbf8e, 0x43ff, 0x4348, 0x4262, 0xba30, 0x202d, 0xa784, 0xadcc, 0xba34, 0x121b, 0x41da, 0xbfd1, 0xb293, 0x435f, 0x31bc, 0xb263, 0x40a0, 0xbf1c, 0x46cd, 0x22d6, 0x3198, 0x250f, 0x2fc5, 0x4100, 0x4090, 0xbfd2, 0xb207, 0xba6f, 0x436b, 0xba5f, 0x4618, 0xb2d4, 0x1123, 0xba54, 0x33e5, 0x43eb, 0x1e8d, 0xb21b, 0xaf42, 0xab0f, 0xb2cb, 0x4101, 0x4026, 0xb2b9, 0xaaf3, 0x1f93, 0x4208, 0x1d01, 0x400d, 0xbf99, 0x3677, 0x182f, 0x3f84, 0x363f, 0x185c, 0xb2d0, 0x0c54, 0x43dd, 0xa014, 0xb23f, 0x41f8, 0xba0b, 0xba39, 0x255a, 0x4313, 0x4449, 0x1934, 0x4317, 0x4102, 0x156d, 0x01fd, 0x41ec, 0x41e1, 0xb023, 0xbf62, 0x41b9, 0x3f1c, 0x3b9c, 0x1930, 0x1e90, 0x429a, 0x1a6c, 0x1dda, 0x1834, 0x440c, 0xb284, 0xb225, 0xae4e, 0x4080, 0xbfca, 0x1cfd, 0x264f, 0x4298, 0x185e, 0x1dc1, 0xbf17, 0x244c, 0x4546, 0x1b6d, 0xbaf7, 0xb286, 0x2054, 0x3c0d, 0x3570, 0x25a7, 0xb261, 0xb21e, 0xa108, 0x13c0, 0x4274, 0x4310, 0xbf42, 0x4212, 0x40c5, 0x438f, 0x0e16, 0x424f, 0xbf90, 0x438e, 0x37c7, 0xbf70, 0x1c3a, 0x437c, 0x06c9, 0x3d63, 0xba3c, 0x1fc0, 0xb223, 0xb245, 0x4003, 0x43aa, 0x0028, 0xba31, 0x038a, 0xba46, 0x1c2d, 0x4435, 0x0726, 0x41d5, 0xbfa6, 0xb2a3, 0xad24, 0x4662, 0x1fdc, 0xb2f7, 0x4012, 0x37dc, 0x418c, 0x1bf3, 0x42b6, 0x44ca, 0x03ed, 0xba5a, 0x431e, 0xbfdf, 0x4315, 0x437b, 0x42ea, 0xbad9, 0xad16, 0xb24d, 0x0b0b, 0xb24e, 0xbfc0, 0xb05b, 0x435f, 0xb255, 0x424e, 0xbae9, 0x4148, 0xb05f, 0xb243, 0x4182, 0x0e5a, 0xb205, 0xbad7, 0x2fd5, 0xbf21, 0x40ac, 0xb2a3, 0xba5c, 0xb2e0, 0x4061, 0x402e, 0xb0a7, 0x3380, 0x42b8, 0xa8cf, 0x2a06, 0x42ab, 0xb296, 0x42c7, 0x41fd, 0xb25b, 0x4333, 0x419f, 0x437b, 0x2ab1, 0xbaf1, 0x17dd, 0x43a2, 0x36ea, 0x4153, 0xbf91, 0x4023, 0x410e, 0xb2b9, 0xb0e9, 0xace2, 0x00d9, 0x2367, 0xb2f9, 0x41df, 0x2087, 0x3147, 0x41f5, 0xb086, 0x40e6, 0x43ef, 0xb21a, 0x41fa, 0xbf6b, 0x311c, 0xb297, 0xba54, 0xa3cd, 0x4189, 0xb252, 0x4186, 0x43d4, 0x4312, 0x4096, 0xb20e, 0x45f6, 0xba7c, 0x3889, 0x4085, 0x4067, 0x4254, 0x1829, 0x414f, 0x4447, 0x2373, 0x4184, 0xb24e, 0x408b, 0x404b, 0xa39a, 0xa843, 0x4007, 0x401e, 0xbf68, 0xba09, 0xb0f7, 0x46a1, 0xb068, 0x427a, 0x42c9, 0x43f4, 0x41e2, 0x40fa, 0xbac6, 0x4377, 0xbf70, 0x3cb1, 0x41b3, 0x43a8, 0xbf74, 0x1f66, 0xb25c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x320c594d, 0xb070758d, 0x7506c560, 0x186ad0f5, 0x6d69bf75, 0xb7cad7d5, 0x7a31742f, 0xa62baad6, 0x3f33b7a4, 0x3237b374, 0x612a830d, 0xd1ce3d3c, 0xdf52c51d, 0x8c1be78a, 0x00000000, 0xa00001f0 },
                FinalRegs = new uint[] { 0xdf52c8e9, 0xfffffffe, 0x0077f7e0, 0x00003048, 0xffffe53e, 0x00000000, 0xffffe539, 0xd3cd7808, 0x3f33b7a4, 0xffffff9b, 0x407d482a, 0x00000000, 0xdf52c51d, 0xdf52c7a1, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x43ce, 0xba1b, 0xba2e, 0x02ec, 0x4241, 0xb244, 0x0d54, 0x40ed, 0x2352, 0xb025, 0x4172, 0x3f8e, 0x1fc3, 0x423a, 0x43ee, 0x41e7, 0x4155, 0xa91e, 0xbaee, 0x41fb, 0x35a0, 0x1a31, 0x414f, 0x4075, 0xbf8f, 0x39ec, 0x4032, 0x29b2, 0x2093, 0x4196, 0x1b62, 0x4411, 0x3f4e, 0xb242, 0x226a, 0xb2b0, 0x3fc2, 0xba70, 0x43b8, 0xbf7b, 0xa6be, 0x41ab, 0xbae6, 0x401e, 0x46a4, 0xbf2e, 0xb09d, 0x435c, 0x423a, 0xb2a8, 0xac17, 0x4382, 0xb2e8, 0x33df, 0x45a1, 0x11b9, 0x4435, 0xbacf, 0xbf23, 0x2fe4, 0x4137, 0x1be5, 0x1528, 0x41c1, 0xb241, 0xb2e2, 0x2601, 0xb2c8, 0xbace, 0x3e99, 0x417c, 0x4362, 0x426b, 0xb2cb, 0xac16, 0x42dd, 0xab93, 0xbad6, 0x44fb, 0x1c86, 0x445b, 0xa008, 0x41ca, 0xbfe0, 0x45d3, 0xb20a, 0xb06e, 0x443d, 0xbfce, 0x43cf, 0x4278, 0x406a, 0xb23d, 0xac1d, 0x4317, 0x14b6, 0x1aba, 0x4437, 0x409d, 0x40d8, 0x1c08, 0xad81, 0x2e13, 0x43da, 0x354d, 0x418a, 0x1911, 0x3024, 0x40cb, 0xbf4c, 0x190c, 0x400b, 0x4090, 0xbfcf, 0xb09d, 0x3f1b, 0xb235, 0x1f6d, 0x1a67, 0x4078, 0x4045, 0x066d, 0x404d, 0x4148, 0xba75, 0x1c4a, 0xb219, 0x2277, 0x41f9, 0xb0c6, 0xb2a5, 0x0907, 0xbf4b, 0xb244, 0x0cdc, 0xb2ea, 0x4093, 0x42a8, 0x40e0, 0x3381, 0x0521, 0x41dd, 0xb290, 0x4064, 0xb08d, 0x400b, 0x43fa, 0x4349, 0xbf2b, 0xab6c, 0xb0ed, 0x43df, 0x0ae3, 0x24ae, 0xbf67, 0x407d, 0x1041, 0xbae3, 0x43fe, 0xbf63, 0x4185, 0xb033, 0x40df, 0x4362, 0x30bd, 0x455d, 0xb090, 0x411a, 0x44b3, 0x40da, 0xbf80, 0x41d1, 0xb2eb, 0x3e13, 0x4112, 0x40b0, 0x1f92, 0x407e, 0x36db, 0x41cf, 0x4397, 0x1982, 0xbad6, 0xb0be, 0x240e, 0x1c50, 0xba3f, 0x4224, 0xbfdf, 0xa4fd, 0x42c3, 0x4022, 0x4021, 0x1e68, 0xb23d, 0x43c1, 0x46fc, 0x4667, 0xba1c, 0x381b, 0x4189, 0x4186, 0x42da, 0xba72, 0x4230, 0x2b23, 0x40b4, 0x431c, 0x414c, 0xa472, 0x419e, 0xba03, 0x429b, 0x0c07, 0x1985, 0x412a, 0x3faa, 0x4689, 0xbf8f, 0x41e3, 0x427b, 0xb28d, 0xa220, 0xa8e1, 0x42ad, 0x4302, 0x41d7, 0xb224, 0x4240, 0x4063, 0x418d, 0xb2f7, 0x422d, 0x2633, 0x46f8, 0x13db, 0xbf84, 0x3ef1, 0x434c, 0x115c, 0xbf98, 0xba27, 0x1d98, 0xbf95, 0x4273, 0x1cdc, 0x428c, 0xb215, 0x3d32, 0xb219, 0xba4a, 0x4640, 0x4349, 0x4123, 0xba50, 0x447f, 0x3398, 0x4017, 0x325e, 0x1afd, 0xb20b, 0x1f41, 0xb200, 0x422e, 0x16a1, 0x43b4, 0x1f01, 0xbf58, 0x42f1, 0xb2c7, 0xb269, 0x41c9, 0x4445, 0x42fd, 0x4228, 0x1796, 0x1d3a, 0x40d8, 0xbf23, 0x44ac, 0x433c, 0x41d0, 0x280a, 0x4555, 0x428d, 0xbf48, 0xb219, 0x40e5, 0x4178, 0x4072, 0x438f, 0x4351, 0x45a5, 0x42ed, 0xa654, 0xb27b, 0xbfab, 0x1c9b, 0x42b1, 0xb03b, 0xba2d, 0x19b2, 0x4279, 0x429a, 0x4025, 0x41d4, 0xb228, 0x426f, 0xb2dd, 0xb26b, 0x4377, 0x18e1, 0x1a2d, 0x1e4f, 0x40b3, 0x42e2, 0x4249, 0x1b7c, 0xbaec, 0x40c5, 0x193f, 0x414c, 0xba00, 0x437b, 0x40d7, 0xba26, 0xbf4c, 0x43b5, 0x3b21, 0xab9e, 0x44ac, 0x18b3, 0x40c8, 0xbaed, 0x17c8, 0x3a04, 0x1751, 0x41c0, 0x41b2, 0x41c4, 0xb039, 0xbf01, 0x43de, 0x41f5, 0x1b0f, 0x4336, 0xba08, 0xb03b, 0x41d8, 0x366c, 0x0cf1, 0x4615, 0x412e, 0x4541, 0x030a, 0x28f4, 0x419c, 0x45e3, 0x3ee7, 0x3be2, 0xb031, 0x4301, 0x4327, 0x403a, 0xbad8, 0x425e, 0x42f5, 0xbfaa, 0x413e, 0x448c, 0xafe4, 0x40de, 0x417c, 0xbfb0, 0x22f0, 0xb2de, 0x4197, 0x421f, 0x410f, 0xb2d4, 0x40e4, 0x4117, 0x40ef, 0xbf54, 0x415e, 0xa3a1, 0x42bb, 0x40b0, 0x43b0, 0x4176, 0xb0b5, 0xb2e8, 0x3f9f, 0xba16, 0x4130, 0x412a, 0xbf5b, 0x1a0c, 0x42ba, 0x43e9, 0xa069, 0xb287, 0x41fd, 0xb0f8, 0x4275, 0x4443, 0x2643, 0x41ee, 0xba6b, 0xb25d, 0x32bc, 0xacb5, 0x45e2, 0x1049, 0xbf01, 0x4093, 0x4349, 0xaf09, 0xba72, 0xbadf, 0xbf00, 0xbf3c, 0xb0ff, 0x4190, 0xbf70, 0x4246, 0x1fb4, 0x439b, 0x0a5c, 0x265f, 0x409e, 0x1bae, 0xbae4, 0x20cd, 0x1f9b, 0x1ec0, 0x3cf8, 0xbae2, 0xb0d1, 0xaeaf, 0xbac7, 0x44f2, 0x416f, 0xbf1a, 0x415a, 0x2deb, 0x42f6, 0xa78e, 0x26f7, 0x429c, 0x30db, 0xb0d6, 0x4348, 0x4161, 0x409a, 0x432f, 0xbae7, 0x4105, 0x41b1, 0x4103, 0x1f6d, 0x12bd, 0x4362, 0x42d4, 0xbfa2, 0xbaf3, 0x4287, 0x1c0e, 0xb27a, 0x4628, 0x1861, 0x0bdf, 0x4139, 0x3f0a, 0xbfe0, 0x431b, 0x40bf, 0x403c, 0x24d3, 0x11e4, 0x1802, 0x2b5b, 0xa8ce, 0x15cd, 0x0a73, 0x1f3d, 0x1aa4, 0x41e8, 0xbf7c, 0xba36, 0x1a63, 0x1d63, 0x255d, 0x437d, 0x0969, 0xbac2, 0xba01, 0x4053, 0x40a7, 0x46ec, 0xb2f4, 0x1eab, 0xb243, 0x1a9f, 0x40a3, 0x4488, 0xb217, 0x434b, 0xb22e, 0x36e8, 0xb0a1, 0xb2b3, 0xbf25, 0x33da, 0x37e0, 0x43ad, 0x4418, 0x38ba, 0xa987, 0x4139, 0xbf2a, 0xa07a, 0x39a8, 0xb282, 0xba45, 0x43af, 0xba0b, 0x4275, 0x4273, 0xbad6, 0x435a, 0xb088, 0x402a, 0x40a6, 0xb029, 0x4304, 0xaa79, 0xb263, 0x4094, 0x08e2, 0x42cb, 0x42e7, 0x40dc, 0x1604, 0x43ca, 0x2b37, 0xbfc2, 0x4398, 0x300a, 0x41f9, 0xbf9d, 0x4485, 0xa8ec, 0x3bfb, 0xb23f, 0x0380, 0x3972, 0x40d9, 0x4019, 0x1c09, 0x1091, 0xbadf, 0x40c8, 0x43d0, 0x45aa, 0xbf90, 0xb224, 0x43fb, 0xb25c, 0xbf99, 0xbaf7, 0xb08e, 0x1b57, 0xb202, 0xb26b, 0x4160, 0xb03b, 0x1b50, 0x1f3c, 0x4107, 0x4027, 0x446f, 0x4183, 0x323e, 0xa035, 0x41ff, 0x415d, 0xbaff, 0x4634, 0x430a, 0x434f, 0x414d, 0xba3f, 0x1c05, 0xb0e7, 0xbf6b, 0x43e5, 0x3391, 0x1fed, 0x4341, 0x46f3, 0x4280, 0x4103, 0x401e, 0x1e8c, 0xba73, 0x4110, 0x404e, 0x2e56, 0x1b29, 0x1fc5, 0x4562, 0x418f, 0xbfa1, 0x414a, 0xb2fd, 0x42e9, 0xb2c3, 0x4170, 0xb242, 0x4371, 0x160c, 0xb0da, 0x3438, 0xba7e, 0xba02, 0x40ac, 0xbf92, 0x438b, 0x4009, 0x41b3, 0xbae1, 0xb26c, 0x060c, 0x4023, 0x4377, 0x35b8, 0x413d, 0x411a, 0x43b2, 0x0ad3, 0x42a6, 0xbf00, 0x4279, 0x1fe4, 0xb228, 0x42f0, 0x0ab4, 0x41bf, 0x41f0, 0x24fe, 0xbf67, 0x2bbb, 0x0f1e, 0x4101, 0x4402, 0xb2f6, 0x435e, 0xba00, 0xbfb0, 0x41bd, 0x4540, 0xbfb8, 0x1ba4, 0xb25c, 0x1478, 0x415b, 0xb01a, 0x4067, 0xa8f1, 0x429b, 0x1211, 0x42d8, 0xbf6b, 0x42b0, 0xba0f, 0x4175, 0x42a3, 0x3eba, 0x418b, 0x3558, 0x4044, 0x1c92, 0x40ee, 0x4345, 0x429a, 0x42f0, 0xbac9, 0x46b4, 0x429d, 0xbf0a, 0x4338, 0x418b, 0x43c0, 0x4224, 0xbf0f, 0xaa79, 0xba2e, 0x2cfb, 0xa092, 0x4003, 0x2293, 0xb2fb, 0x1a02, 0x4298, 0x4204, 0x41f3, 0x255f, 0x1884, 0xbfc0, 0x403a, 0x407b, 0xb044, 0xbf5d, 0x12b1, 0x3384, 0x4336, 0xb05d, 0x42be, 0x419e, 0xb2ad, 0xba4a, 0x4117, 0xba5b, 0xb292, 0xbafd, 0x428e, 0x43f5, 0x40f8, 0x4601, 0x406b, 0xae07, 0x42e5, 0x4053, 0x4068, 0xb0b4, 0x3961, 0x4340, 0x37c8, 0xbfcb, 0x423c, 0xa03f, 0x4342, 0x423c, 0xbf88, 0x0fdd, 0x4193, 0xb2d9, 0x43da, 0xbfa0, 0x1396, 0x40af, 0x4317, 0xa51a, 0x211a, 0xadb9, 0x433e, 0xba22, 0x41f9, 0x180e, 0x411c, 0xba61, 0x4001, 0x4268, 0x03fc, 0xbf00, 0xb281, 0xb23f, 0xbae2, 0xbf80, 0xbf65, 0x43e1, 0x394f, 0x35a0, 0x329b, 0x422f, 0x4164, 0x0d8f, 0x4237, 0x465a, 0x43b6, 0xbaf0, 0xbfe0, 0xbf11, 0x4267, 0xba2f, 0x41a6, 0xb210, 0x4221, 0x4319, 0x1482, 0x3205, 0xb256, 0x2dfd, 0xba01, 0x402a, 0xb2ca, 0x163f, 0x432d, 0x41fd, 0x4161, 0xb0b4, 0x42f8, 0x40c2, 0x1aa5, 0x4167, 0xb051, 0x0978, 0x4149, 0xbf1f, 0x4175, 0x43fa, 0xb252, 0x4140, 0x4215, 0xba55, 0xb2d2, 0x4210, 0x406a, 0xaf57, 0xacee, 0xa935, 0xba5d, 0x4259, 0x05bf, 0xb015, 0x43e6, 0x4192, 0xa8db, 0x44e2, 0x1588, 0x0d4d, 0x437f, 0x0618, 0xbfdb, 0x41e3, 0x4108, 0xa17e, 0x416f, 0x1b33, 0x42df, 0x438e, 0x41d0, 0xbfb1, 0x0d0f, 0xb28f, 0xba4b, 0x3c20, 0x42e3, 0x0f9b, 0x41af, 0x40fd, 0x415c, 0x40f5, 0x436d, 0xa5a7, 0xa82e, 0xba46, 0x42ac, 0xb204, 0x4067, 0x438d, 0x40e0, 0x09d3, 0xaaf7, 0xbf79, 0x4386, 0x1b23, 0x41ea, 0x4010, 0xbff0, 0x4491, 0xa083, 0xb227, 0xb2d3, 0x4196, 0x1c87, 0xb22a, 0xb09b, 0x01cd, 0x1c6f, 0xafc3, 0x43be, 0xa9d2, 0x42c6, 0x0ad5, 0x4341, 0x432b, 0xba5c, 0xb0bc, 0x4120, 0x412e, 0x4475, 0x4284, 0xb01b, 0xbfda, 0xa893, 0x1afa, 0xa310, 0x40f9, 0xb25d, 0x42b4, 0xb00e, 0xb06b, 0xb26e, 0x43eb, 0x4237, 0x40a9, 0x4369, 0x40cc, 0x4233, 0x40fc, 0x3d00, 0x43aa, 0xb2d7, 0x40bc, 0xbf00, 0xb296, 0xb223, 0x0c49, 0xaaff, 0x4180, 0x18bc, 0xbf70, 0xbf6c, 0x430a, 0xb208, 0x0bd5, 0x1bcd, 0x421c, 0xb289, 0xb250, 0xbf1e, 0x41b7, 0x44fa, 0xb0f3, 0x2760, 0x1c87, 0x189c, 0xb217, 0x41a3, 0xaf06, 0xbfb4, 0x422b, 0x2caf, 0x1b6f, 0x429c, 0x406b, 0x417d, 0x41c5, 0xa01c, 0xb258, 0x43ee, 0x1c6b, 0x4171, 0xb0ab, 0xa87d, 0x40e3, 0x1b3f, 0xba4a, 0xbf46, 0x3418, 0x19da, 0xa67e, 0x4223, 0x3401, 0x3b45, 0x2a8f, 0x45ab, 0xb249, 0xb2d7, 0x416d, 0xbf80, 0x1b06, 0xba61, 0x070c, 0xb2e8, 0x4394, 0x3fb6, 0x41b7, 0xbf56, 0x4087, 0x42fa, 0xb01d, 0x438c, 0x1fb3, 0xbae0, 0xb01a, 0xa6da, 0x4396, 0xb009, 0x3260, 0x4347, 0x4097, 0xb06f, 0x42b9, 0x2e15, 0xb033, 0x4253, 0x43a3, 0x0261, 0x1e9b, 0x4657, 0xb2fb, 0xb288, 0xbf9e, 0x42ed, 0x43d9, 0xba33, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3e63c940, 0xd2f40362, 0xbb77737e, 0x677685e4, 0x97ccf408, 0x3dfa0433, 0x43332c01, 0xa5fc83ef, 0x5b5a10e7, 0x93fd7fe6, 0xe271cf59, 0xac70a2fd, 0xc4be1954, 0x93de22cc, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0xfffff05f, 0x00000059, 0x00000000, 0x00000020, 0x00001000, 0xe271cf59, 0x49f6f315, 0x93de1f28, 0xe271cf59, 0x00000000, 0x00000000, 0x93de1f1c, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4010, 0x421c, 0x3d78, 0x4344, 0x43d7, 0x401a, 0x1ce0, 0x4051, 0xb2b6, 0x0a48, 0xbad9, 0x4206, 0x4163, 0x1efa, 0x402d, 0x1e69, 0x4221, 0xbfba, 0x4154, 0xb2b8, 0x19e2, 0xb243, 0x15ea, 0x406b, 0x46d3, 0x4546, 0x417a, 0x41b7, 0xba61, 0x41fd, 0x41d5, 0x45bd, 0xb0fa, 0x0951, 0x454f, 0x0377, 0x4262, 0xb097, 0xb04b, 0xb277, 0x37da, 0xbfcc, 0x15dc, 0x4385, 0x18c8, 0xb22d, 0x412b, 0xb029, 0xba75, 0x411d, 0x14c5, 0x456f, 0xbaf2, 0xba3d, 0x45f3, 0xb0da, 0xb087, 0x4253, 0x411a, 0xb2fa, 0x469d, 0x44cd, 0x0459, 0x4410, 0x1378, 0x1239, 0xbf2d, 0x423d, 0x070b, 0x4353, 0x43a9, 0xb015, 0x4301, 0xbae3, 0x0c4b, 0x46bd, 0x4013, 0x3765, 0x4147, 0xbf59, 0x1d79, 0x2918, 0xb014, 0xb223, 0x4202, 0x1ceb, 0x18bb, 0x4274, 0xaa53, 0x4201, 0xbacc, 0xa37f, 0xb0af, 0x411f, 0x41cf, 0x186b, 0x40cd, 0x14ec, 0x4018, 0x435d, 0x327d, 0x43eb, 0x4305, 0xbf64, 0x13ef, 0x40d6, 0x4095, 0x369f, 0x4065, 0x41b3, 0x1cc7, 0xbf8b, 0xb297, 0x432e, 0x4268, 0x4002, 0x430a, 0x4479, 0xb048, 0x421e, 0x1ca3, 0xa979, 0xb01d, 0x4459, 0x07f7, 0xb262, 0x1862, 0xb289, 0xb211, 0xbac9, 0x43cd, 0xbfdf, 0xb072, 0x421a, 0x412d, 0x2962, 0x46ec, 0xb09e, 0x1e20, 0x44c1, 0xbfd1, 0xb00c, 0x437b, 0x18ee, 0x41e2, 0xba0c, 0xba2c, 0x04b5, 0xb05b, 0x4596, 0xa87e, 0x422a, 0x40cc, 0x4002, 0x40a3, 0xb041, 0xbf8b, 0x40d4, 0xb0c0, 0x41b8, 0x45e3, 0x3460, 0xb24b, 0xb285, 0xbf04, 0x4225, 0x4064, 0x420a, 0xbf34, 0x43ed, 0x0cab, 0x4194, 0x0ffb, 0x41b8, 0x43e0, 0x4173, 0xbf9a, 0x3fab, 0x210e, 0xa820, 0x426a, 0x4395, 0x43ee, 0x3657, 0xbfc0, 0x430b, 0x438b, 0x1b6d, 0x1c7b, 0xb231, 0x42f2, 0x439e, 0x4389, 0x41b5, 0x4082, 0x1de1, 0x1ca6, 0xbfd0, 0x34cc, 0x1dda, 0xae7c, 0x4181, 0xbf2f, 0x410f, 0xbaf2, 0x40aa, 0x409e, 0x13c8, 0x4076, 0x1bd4, 0x4020, 0x4128, 0xba1f, 0xb24b, 0xa578, 0xaabf, 0x4049, 0xa063, 0x418b, 0xbafd, 0x407c, 0xb234, 0x4380, 0x4288, 0x43e2, 0xbf38, 0x422b, 0xbf80, 0x43b2, 0x43fc, 0x3c13, 0x4282, 0xab63, 0x41db, 0x298d, 0xba1e, 0x1f37, 0x41e6, 0x464d, 0xb031, 0x4380, 0x467c, 0x1b44, 0xbf07, 0x1891, 0x0bcc, 0x420c, 0x4330, 0x4038, 0x4155, 0x4162, 0x4421, 0x4071, 0xbafd, 0x1b7a, 0x42b8, 0x37f8, 0x46aa, 0x0159, 0xb28a, 0x43cf, 0x43ba, 0xb056, 0x43b4, 0xb067, 0xbf35, 0x46fa, 0x40e7, 0x45c4, 0x42dd, 0xbfa3, 0x1995, 0x407b, 0x3862, 0x43bd, 0x0793, 0xbaf8, 0x1f40, 0x40fd, 0x1a53, 0xbf7f, 0x384d, 0x261a, 0x416d, 0x4200, 0x204f, 0x40e1, 0xb0eb, 0x0a31, 0x43ac, 0xb266, 0x1b0e, 0x4127, 0x27e7, 0x4570, 0x2786, 0x40e4, 0xb23c, 0x44dd, 0x4125, 0xbf6a, 0x402a, 0x40f1, 0x4161, 0x45f1, 0x18cf, 0x1e84, 0xba1e, 0x42dd, 0x406e, 0x09d5, 0xbac2, 0xaa6d, 0x4128, 0x226e, 0xb26e, 0xaf3c, 0x0e65, 0xbf75, 0x45d1, 0x18a2, 0x433e, 0x2402, 0x41e3, 0x43cb, 0x427f, 0xbac3, 0x4180, 0x1df8, 0x41c7, 0x1bf7, 0x439f, 0xab0c, 0x4463, 0xbf4a, 0x43d8, 0x44ba, 0x415a, 0x40d1, 0x4167, 0x4026, 0x0966, 0x423e, 0x1de0, 0x41f5, 0xb07c, 0x4628, 0x4288, 0x460a, 0x1376, 0xbf94, 0x4364, 0x41f7, 0x43de, 0xba74, 0x4027, 0x4453, 0x1dd1, 0x1a65, 0xa7ce, 0xb2c5, 0x4658, 0xb29b, 0xb2b8, 0x18be, 0xb2bc, 0xb022, 0x402c, 0xba7a, 0x0b2a, 0x40d6, 0xba03, 0xbf27, 0x4613, 0x1739, 0x1c8d, 0xb258, 0x1f23, 0xba74, 0x45e6, 0x1b3c, 0x4090, 0x0567, 0xbf4c, 0x1e05, 0xba36, 0xb2a9, 0xbfa5, 0xbae7, 0xafde, 0xba56, 0xb2c6, 0x08f6, 0xb0e6, 0x42dd, 0xbf74, 0xb0a8, 0x2a22, 0x1c73, 0x0bf3, 0xa2ab, 0x4359, 0x07b8, 0x40ea, 0x4207, 0xba17, 0x3f61, 0x1f21, 0x4040, 0x41ed, 0xbae2, 0xba23, 0x42d9, 0xb2e5, 0xa1b9, 0xb0a8, 0xa31a, 0xbf02, 0x1e10, 0xba2a, 0xb20d, 0x4261, 0x4013, 0x1b84, 0x40ce, 0xb218, 0xabc9, 0x1aa3, 0xba7f, 0x42e1, 0x4322, 0x41a1, 0x2a78, 0x4245, 0x46f0, 0x4189, 0x4164, 0xb080, 0x4003, 0x145d, 0xbf67, 0xba42, 0xa206, 0x1c5f, 0x4353, 0x441d, 0xb220, 0xbf3f, 0xba17, 0x416a, 0xb04e, 0x40e8, 0x40f1, 0x41ca, 0xbfb5, 0xba7b, 0xb2e2, 0x1300, 0xb0d2, 0xb03d, 0x435e, 0xb297, 0x1980, 0x4074, 0xbfcc, 0xa9ef, 0x0790, 0x464b, 0x3062, 0x4238, 0x3c74, 0x1fad, 0x34e8, 0x467f, 0xaa62, 0xa96d, 0xbf13, 0xbad9, 0x1c8e, 0xb21f, 0x4236, 0x4249, 0x40d1, 0x4371, 0x4130, 0x4066, 0x274a, 0x4073, 0xb2ee, 0x2f2d, 0x1a18, 0x26db, 0xb206, 0xb26b, 0x44d0, 0xb21e, 0x0cc4, 0xba08, 0x401b, 0xb0a5, 0xbf3b, 0x4362, 0xbf70, 0x4205, 0x432f, 0xba36, 0x4015, 0xb264, 0x4021, 0xb2e6, 0x40ff, 0x4323, 0xbadb, 0x22db, 0x0a58, 0xbf90, 0xb2a7, 0x378b, 0xbfc0, 0x4334, 0x41ff, 0x41a6, 0xab67, 0x4149, 0x42bf, 0x4295, 0xbf00, 0xbf82, 0xa510, 0xb2ac, 0x429a, 0x1e48, 0x2779, 0x4316, 0x1d29, 0x41b7, 0x408a, 0x4213, 0xba3c, 0x1423, 0xbfd2, 0x4647, 0x41ed, 0x219f, 0xbf8f, 0xba7d, 0x1ada, 0x1e7f, 0x3e40, 0x4325, 0x4166, 0x4206, 0x4335, 0x33d2, 0xb22b, 0x1f30, 0x42da, 0x402e, 0x43c6, 0xbfd6, 0x4312, 0xb2d3, 0x41c1, 0xb22a, 0x438d, 0x02af, 0xbf4a, 0x41d6, 0x0ac5, 0x40eb, 0xb00d, 0x41c8, 0x09ce, 0xb276, 0xbfd3, 0x24d8, 0x44b9, 0x4581, 0x4156, 0x071a, 0xba20, 0xb2ee, 0x41cd, 0xa9ca, 0xbf80, 0x2ad3, 0x4242, 0x4040, 0x1ba8, 0xb2c0, 0x41df, 0xbfab, 0xbad7, 0x1efd, 0x418e, 0xba0d, 0x2484, 0x41d0, 0x1b17, 0x402f, 0xb2c3, 0xb062, 0x02be, 0xbfbe, 0x0c82, 0x41c5, 0xba53, 0xa095, 0xa516, 0x1939, 0x43bf, 0x1e52, 0x408a, 0x2dff, 0x4001, 0x4146, 0x439e, 0x4397, 0xbf00, 0x40db, 0xba6a, 0x4445, 0x429d, 0xb2f1, 0x1d3d, 0x43a0, 0xbf74, 0x4246, 0xb096, 0xa063, 0xb269, 0x3cbc, 0x4552, 0xb221, 0xba52, 0xbadc, 0x4211, 0x3825, 0xb2db, 0xbfa0, 0xba0c, 0x3ad0, 0xb261, 0x4170, 0x430c, 0x41ec, 0x1dd5, 0x0d4d, 0x4140, 0x1a8b, 0x1904, 0xbfa5, 0x41f9, 0xba0c, 0x436d, 0x1d2a, 0x3b51, 0xba39, 0x4014, 0x40f5, 0x40a8, 0x415d, 0x2e7c, 0x3625, 0xb283, 0x421b, 0x2b20, 0x205e, 0x2a8e, 0x43be, 0xba39, 0xbf33, 0x4173, 0x2675, 0x10d5, 0x284b, 0x4227, 0xaaa0, 0x1bcb, 0x1c0e, 0x2a8d, 0x43fa, 0x417e, 0x4379, 0x4096, 0x1d43, 0x409a, 0x42b3, 0xba76, 0x0be7, 0x4316, 0xaed0, 0xbf1d, 0x407c, 0x42da, 0x18c3, 0x1f4d, 0x420e, 0x2a5f, 0xbfa9, 0x40a8, 0x1eb0, 0x4273, 0x40ea, 0x43d0, 0x07e3, 0xb210, 0xba4c, 0x41a8, 0xb294, 0xbf5d, 0x1a32, 0xbafd, 0x4031, 0xa0dc, 0xb2d0, 0x40bb, 0x431e, 0xbf73, 0x4186, 0x4054, 0x36b8, 0x40f2, 0x18fa, 0x1dc6, 0x431a, 0x4298, 0x43a4, 0x25c7, 0x1976, 0x1aa5, 0x41e8, 0x43cb, 0xbfb0, 0xbf58, 0x4328, 0xbf02, 0x43ad, 0x4118, 0xb0b3, 0x40cc, 0x24f8, 0x40e8, 0x4127, 0xa3a1, 0x30df, 0xb238, 0x3b0a, 0xb274, 0x4210, 0xb296, 0x4131, 0xb2dc, 0x37ca, 0x4243, 0x405c, 0x45ab, 0xbf2b, 0x270b, 0xbae4, 0x1bcd, 0x4396, 0x438b, 0x1ed0, 0xbf80, 0x2b6e, 0x436a, 0xb2ea, 0x437f, 0x4606, 0x1f3f, 0x4269, 0x0f94, 0xb223, 0x0628, 0xbf98, 0xbf60, 0xb234, 0xb23d, 0x43be, 0x4137, 0x40fe, 0x41f4, 0x40f5, 0xa3f3, 0xb2ba, 0xba6a, 0x4126, 0x4364, 0xb2f1, 0x42e2, 0xba6b, 0xb0b8, 0x199e, 0xb22a, 0x41e9, 0x461f, 0x431f, 0x44cd, 0x1bdb, 0x1b8e, 0x1deb, 0xbf3c, 0x253b, 0x4355, 0x41ad, 0x4340, 0x40ab, 0x1952, 0x40d0, 0x1cee, 0x43cd, 0xbad6, 0x2461, 0xbf28, 0x28e7, 0xbad3, 0x4420, 0xb29b, 0x436d, 0xb01c, 0xa3bc, 0x4173, 0x4194, 0xb2f1, 0x4298, 0xb0d3, 0xbfb0, 0xb200, 0x415d, 0x406e, 0xba4e, 0x4182, 0x41a4, 0x4207, 0x40dc, 0xbaec, 0x193c, 0x415a, 0x4317, 0xbf83, 0x1e0b, 0x3d29, 0x40a1, 0x4333, 0xb046, 0xaf27, 0x1988, 0xb28a, 0xb204, 0x16d6, 0xac6b, 0x425b, 0x1f0d, 0x4251, 0x408e, 0x42de, 0x464c, 0xbfae, 0x40e7, 0x44b8, 0x43fc, 0x4065, 0x4333, 0x4479, 0x0559, 0xbaeb, 0xb252, 0x41f3, 0x4380, 0xb230, 0x46c0, 0x44a0, 0xbf8e, 0xb023, 0x4288, 0x4017, 0x406b, 0x4288, 0x4398, 0x18cc, 0xabdd, 0x429a, 0x42ab, 0xbf95, 0x43f2, 0x4057, 0x4442, 0x40ee, 0x4111, 0x4358, 0xa518, 0x34fa, 0x4180, 0x3ecd, 0x4223, 0xb265, 0x42a4, 0x1f7c, 0xb296, 0x1d3c, 0x430f, 0x442d, 0x4307, 0x437a, 0xbf27, 0x1d82, 0x391c, 0xb2a8, 0xbae1, 0x40a0, 0x41b3, 0x40e2, 0xb27e, 0x43a4, 0xafdf, 0x43ad, 0xb024, 0x40e0, 0x4048, 0x3c42, 0x1d9d, 0x1954, 0x0cd9, 0xbf4b, 0x4236, 0xa2ef, 0x424e, 0xbac2, 0xba67, 0x4103, 0x43eb, 0x1d60, 0x459e, 0x4059, 0x41b5, 0xbf83, 0x4310, 0xba42, 0x3684, 0xb299, 0x43ca, 0x1f66, 0x43e1, 0x1358, 0xb08d, 0x41ee, 0x1b3a, 0x4386, 0x40ad, 0x43fd, 0x4653, 0xb0bc, 0xb2ac, 0x405b, 0x037e, 0x418c, 0x1887, 0x42bd, 0x05ee, 0x46ed, 0x4238, 0x4062, 0x43f9, 0xbfad, 0x461f, 0xb0fa, 0x3e21, 0x411c, 0x403b, 0x2342, 0x3259, 0x076d, 0xbf81, 0x4009, 0x4152, 0xba08, 0x1811, 0x4312, 0x32f2, 0x1b6f, 0x408c, 0x43d8, 0x437e, 0x40c1, 0x40a1, 0x40cd, 0x43af, 0x2042, 0x4342, 0x4082, 0x4029, 0xa300, 0xbf49, 0xb076, 0x421e, 0x45a2, 0x185f, 0xba07, 0x4283, 0x4306, 0x3c84, 0x454f, 0x42d5, 0x4071, 0xae63, 0xb2b0, 0xab50, 0x21b6, 0x4024, 0x4485, 0xb292, 0xb266, 0x424d, 0x1559, 0x0623, 0x42e7, 0xb2e3, 0x4094, 0xbf38, 0x4378, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xfbee3e06, 0xf7bc3994, 0x7ccee582, 0x65c6dfd8, 0x7d765d73, 0x2669bec2, 0x4d20d1a8, 0x414bdcb0, 0x53f02dc6, 0xad07dc25, 0xbd4dd249, 0x070bd4e7, 0x1404f4cb, 0x80985f65, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0x000061d2, 0xfffffdef, 0x00000000, 0x0000007c, 0xffffff7c, 0xffffff4a, 0x0000007c, 0x42000000, 0x0b273264, 0x00a885eb, 0x0b273265, 0xbd4dd249, 0x00000322, 0xbdf6c218, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xba5b, 0x07d6, 0x17ae, 0x42a8, 0x465e, 0x4335, 0x121d, 0xa28d, 0x4605, 0x4319, 0x4115, 0x1eed, 0x40b7, 0x43ba, 0x4680, 0x4079, 0x4604, 0xb239, 0x41f3, 0x40fa, 0xba5e, 0x4247, 0xba6d, 0x42ce, 0xbfb7, 0xba11, 0x4025, 0xbfa0, 0x4034, 0xb096, 0x1948, 0x4194, 0x42a7, 0x43b5, 0x416e, 0x40a9, 0x42db, 0x4601, 0xbf38, 0xb2e6, 0x426f, 0x444a, 0x40ee, 0x44ed, 0x44d1, 0x4000, 0xb215, 0x30a0, 0xba5a, 0xb261, 0x41bb, 0xb00b, 0xbf6c, 0xbae1, 0x0ef2, 0x431c, 0x00d4, 0x446c, 0x4357, 0x42d7, 0x2fd0, 0xb224, 0xb099, 0x424c, 0x1b9f, 0x01d3, 0x40e7, 0x3e10, 0x4648, 0xbfcb, 0x0411, 0xba0b, 0x40bb, 0xbac8, 0x24c9, 0x42cf, 0xba3d, 0x01fa, 0x1a3f, 0x03a8, 0x28e0, 0x42e1, 0x40b4, 0xb24b, 0x1cd5, 0xb2c8, 0x43ef, 0x2f59, 0x18d0, 0xa4b2, 0xba71, 0x2bb4, 0xb088, 0x21d1, 0xb2db, 0xb2b4, 0xbfb7, 0x1d84, 0x1c34, 0x438f, 0x40d3, 0x4542, 0xaf8e, 0xbad0, 0x4106, 0x28c9, 0x4102, 0x40e9, 0x42d2, 0xbf2c, 0xb203, 0xb295, 0x4262, 0x42ab, 0xbfca, 0x4477, 0x43df, 0x425f, 0xb2b8, 0x425d, 0xb2f7, 0x1d27, 0x40e0, 0x41d8, 0x40e2, 0x3bf3, 0x40f4, 0x416c, 0x17a2, 0xa33f, 0xb254, 0xba60, 0xb056, 0xbfd7, 0x4208, 0xb025, 0x4590, 0x18f9, 0xb20c, 0xba78, 0xb059, 0xba72, 0x2cea, 0x0a1e, 0xbf00, 0xaaee, 0x4651, 0x4315, 0xbaf9, 0x42c3, 0x4176, 0xa362, 0x0638, 0x2acd, 0x1a26, 0xb251, 0x40c7, 0x4469, 0x410c, 0xbfd7, 0x43a1, 0x419c, 0x4093, 0x43d5, 0x3b5f, 0x33cf, 0xa83a, 0x4160, 0xb222, 0x41bf, 0xb2c9, 0x4324, 0x1fd8, 0x1cd4, 0xb0ab, 0x427b, 0xbaf5, 0x407f, 0xbf63, 0x19af, 0xa6ef, 0x438b, 0x43cb, 0xbfc0, 0x4367, 0x428f, 0x4088, 0x4219, 0xb273, 0x0619, 0x4385, 0x420f, 0x46e3, 0x4466, 0xb019, 0x4366, 0x402f, 0xbf36, 0xb2b2, 0x2705, 0x1202, 0x0143, 0xbfa7, 0xbfc0, 0xad30, 0x42ee, 0x4279, 0x4273, 0x1d83, 0x432e, 0xb2fd, 0x42af, 0xbfd7, 0x4304, 0xba66, 0x1f02, 0x04b7, 0x4547, 0x0537, 0x4478, 0x4396, 0x421a, 0xbf64, 0x421e, 0x2be3, 0x41cd, 0x43fd, 0x435c, 0x4205, 0xa4ee, 0x439a, 0x4270, 0x42cd, 0x435b, 0x4057, 0x2f2c, 0xb22c, 0x27ee, 0x43df, 0xb2bd, 0x40ce, 0x17f1, 0xbf58, 0x4179, 0xba3d, 0x4036, 0x4150, 0xa831, 0xba54, 0x4278, 0x4270, 0x1454, 0x42e3, 0x4099, 0xb253, 0x41b7, 0x1c6d, 0x4148, 0x4158, 0xb096, 0x4495, 0xb07e, 0xbaca, 0xb2b8, 0x1b78, 0xbf53, 0xb2a8, 0xb2a0, 0x3743, 0xba20, 0x41cc, 0x1435, 0x438c, 0x4074, 0xb09f, 0xb291, 0x1fb0, 0x4040, 0xbafc, 0x1ebc, 0x14da, 0xbfa0, 0x4235, 0xb22c, 0x41b4, 0x3a86, 0x433a, 0xbfa8, 0xb047, 0x4639, 0xbadf, 0x4225, 0x43f8, 0x0dab, 0xa751, 0xbff0, 0x4347, 0xbfb6, 0xba06, 0xbadf, 0xb084, 0x06c0, 0x2b46, 0x42b3, 0x014f, 0x0b86, 0xba4f, 0x44ab, 0xb2c4, 0x154e, 0xb24e, 0xbaf5, 0xa94c, 0x30d0, 0x435b, 0xbaeb, 0xbfa3, 0x41fc, 0xba0e, 0x4180, 0xba0b, 0x1810, 0x0d4e, 0xba04, 0x0f96, 0xb0db, 0x4117, 0x4335, 0x4314, 0x3157, 0xb26d, 0x4554, 0xb23a, 0xb2ad, 0x3ee7, 0x3bcf, 0xba46, 0x42cc, 0x415a, 0x421b, 0x43bf, 0xbf80, 0x1b5d, 0xba10, 0x099c, 0x421d, 0xbf32, 0x20d0, 0xb21d, 0xacf5, 0xba23, 0x4120, 0x4102, 0xbaf4, 0xbfa8, 0x2f4b, 0x1913, 0x2795, 0x43fe, 0x4472, 0x4419, 0x4224, 0x423b, 0xba6a, 0x33f4, 0x466a, 0x0602, 0x42e6, 0x45c5, 0x0a3f, 0xb279, 0xb294, 0x42cb, 0x4013, 0xb2f7, 0x41ce, 0x1fbe, 0x4229, 0x41eb, 0xbf5d, 0x0f87, 0x4120, 0x4233, 0x410a, 0x3a7d, 0x1365, 0x4633, 0x42f0, 0xba0c, 0x415a, 0x438f, 0xb062, 0x42f1, 0x4205, 0xba13, 0xbfa5, 0xb221, 0xb2eb, 0x436a, 0xb2cf, 0xb241, 0x4120, 0x4233, 0x446e, 0x40c9, 0x3943, 0xb2c5, 0x0b33, 0x13f7, 0xb236, 0xa84f, 0x415a, 0xb09b, 0x43e2, 0x4105, 0xbfc0, 0x438e, 0x1901, 0x4146, 0xbf52, 0xa619, 0xba67, 0x4354, 0x3f22, 0x430e, 0x44d5, 0x1fbf, 0x413b, 0xa872, 0x4304, 0xba65, 0xb27e, 0xb20f, 0x41a5, 0xbade, 0x4157, 0xaeb4, 0x423d, 0x42fa, 0xbfc1, 0x4416, 0x434d, 0x44ea, 0xb239, 0x4594, 0x0037, 0xbf1b, 0x1864, 0x42d5, 0xba2f, 0x017e, 0x4065, 0xb0bc, 0x13a1, 0xbfa0, 0x43b2, 0x41a4, 0xb269, 0x46d1, 0xb27b, 0x410e, 0xb24b, 0xbf06, 0xb061, 0x0c0c, 0xb294, 0x4010, 0x1e48, 0x435f, 0xa4ab, 0xb2db, 0x248f, 0xa37f, 0x403a, 0xb0df, 0x4023, 0x42ba, 0x42dd, 0x1e2d, 0x4299, 0x41a2, 0x2d1b, 0x2b1b, 0x41f5, 0x194d, 0x422c, 0xb23b, 0x427d, 0xb2a0, 0xbf1a, 0xab55, 0x43ca, 0xa750, 0xa102, 0xbaed, 0xbff0, 0xba5f, 0x4274, 0xb28e, 0xa5be, 0x40e5, 0xb20e, 0x4173, 0x2778, 0xb26b, 0xa8ff, 0x18f0, 0x4036, 0xbf65, 0x1b9b, 0x46ec, 0xb0bf, 0x4017, 0x4126, 0x2f2d, 0x3504, 0xb05a, 0x45e5, 0x4053, 0x41f7, 0x43d1, 0x417c, 0x421c, 0xbfe2, 0x4377, 0x1723, 0xbff0, 0x46c2, 0xa232, 0xaec0, 0xba6c, 0xb251, 0xb2f6, 0x4060, 0x404f, 0x1ce5, 0xbf88, 0x4655, 0x1a5a, 0x403d, 0xba12, 0x19c7, 0xba50, 0x4169, 0x4106, 0x4250, 0xb291, 0x42c7, 0x4347, 0x27bd, 0x2c35, 0x0626, 0x1ec8, 0x0280, 0x400e, 0x2bf8, 0xb263, 0xb2af, 0xb01d, 0xba2b, 0x1cf0, 0xbfdf, 0x431f, 0x4604, 0xb295, 0xba0c, 0xb218, 0x4325, 0x213d, 0x415a, 0xb0aa, 0x410d, 0x41d0, 0x18ec, 0x445d, 0x4031, 0x1a49, 0xbaf3, 0x4089, 0x4001, 0x42c8, 0x40d9, 0x1fd4, 0xbf3f, 0x41ab, 0xa638, 0x42fa, 0x1c8f, 0xa077, 0x1b97, 0xb204, 0x4210, 0x42f2, 0x1b51, 0xb0ed, 0x1bc5, 0xb25f, 0xba2d, 0x335d, 0x3031, 0x15d4, 0x403d, 0xa32c, 0xbadb, 0x41be, 0xbf33, 0xba5e, 0xba5f, 0x45ba, 0xb01a, 0x4214, 0xbad1, 0xbf82, 0x429c, 0x171a, 0x1728, 0xbfb2, 0x413e, 0xac11, 0x42bc, 0x4149, 0x41b5, 0x433a, 0xb26f, 0x1d40, 0x461b, 0x4046, 0xb243, 0xb2fc, 0x4259, 0x1b06, 0x4253, 0x4179, 0x4369, 0xba48, 0x403d, 0xb2b1, 0xb06f, 0xb21c, 0x4122, 0x2dc7, 0xbf9d, 0x43c5, 0xb2cd, 0xa4b1, 0x417e, 0xb232, 0x42e0, 0xbfdc, 0xb073, 0xb2a0, 0x1caa, 0xb2d4, 0x4037, 0x16f9, 0x05b2, 0xbf00, 0x43f5, 0x4135, 0x43d8, 0x4115, 0x42e9, 0xb248, 0x4037, 0xb274, 0xb249, 0x4284, 0x3542, 0x3c24, 0xa2e5, 0x1a2f, 0x40d1, 0xb2e3, 0x41da, 0x4255, 0xbf0f, 0x2d1a, 0x46d8, 0x422c, 0x28a3, 0x1c9e, 0x1f60, 0xb0a9, 0xb08f, 0x4056, 0x19b4, 0xb20b, 0xba7b, 0x4308, 0x40de, 0xb204, 0x4684, 0x4179, 0x0f4b, 0xb2e6, 0x45f0, 0x43f6, 0x43f6, 0x4181, 0xb2ab, 0xbf82, 0xb242, 0x2355, 0xb227, 0x1e77, 0x403d, 0x4381, 0x404c, 0x427e, 0x4220, 0xbfd0, 0x3630, 0xb282, 0xbf75, 0x1fe1, 0x1f1f, 0x288b, 0x40f3, 0x46f9, 0x39bb, 0x4328, 0x407f, 0x4245, 0x434a, 0xae95, 0x41c9, 0xba43, 0x09d6, 0x4179, 0xa589, 0x1d18, 0x1cdd, 0x42dc, 0xbf02, 0x35d5, 0x43c8, 0x4270, 0x403c, 0xba6d, 0x42ea, 0xbf65, 0x1ade, 0x2288, 0x4660, 0x437f, 0x4281, 0xba01, 0xb2e8, 0x4248, 0x46d1, 0xba30, 0x41b0, 0xb21e, 0x4017, 0x024f, 0x4305, 0x40cb, 0xb29c, 0x40d0, 0x0b3f, 0x0661, 0xb2fc, 0x4083, 0x196d, 0xb2aa, 0x3149, 0xbfc9, 0x41ae, 0x40ea, 0x4126, 0x462a, 0x42e4, 0x434f, 0x4242, 0xb2e1, 0x43de, 0x43e2, 0x41cb, 0x4112, 0xba58, 0x42c6, 0x426d, 0xb082, 0x43e7, 0x419e, 0x43ad, 0x4576, 0x42a1, 0x40a6, 0xbf54, 0x40a9, 0xbac0, 0xb254, 0x1bc9, 0x3ef4, 0xba5a, 0xa96a, 0x40a2, 0x4688, 0xb2a4, 0xba68, 0xb21e, 0xb2a2, 0xbf38, 0x0405, 0xb219, 0x2f28, 0x40fb, 0xb23c, 0xb29e, 0x4035, 0xb0cc, 0x4653, 0x4006, 0x460b, 0xbf3c, 0x427b, 0x4182, 0x4274, 0xbff0, 0x4252, 0x1272, 0xbfa4, 0xae65, 0x0ca1, 0x14f9, 0x00b0, 0x40b8, 0xba23, 0x4349, 0xbfa3, 0x4461, 0x063b, 0xb229, 0x41eb, 0xb224, 0x4124, 0xb0cc, 0x4342, 0x3167, 0x42c9, 0x1d8e, 0x4626, 0x3374, 0x4167, 0x4060, 0x45dd, 0x1ad6, 0x3446, 0x4237, 0x442f, 0x4214, 0x1d7c, 0x46c5, 0x42c7, 0x4491, 0x3e38, 0x1d56, 0x42cc, 0xaec2, 0xbf8e, 0x4090, 0xba77, 0x45d4, 0x40e8, 0x190e, 0x1b87, 0x0e51, 0xba73, 0x0806, 0xb206, 0xbf46, 0x4326, 0x111b, 0xba4a, 0x1b79, 0xa71d, 0x42a8, 0x4294, 0x423c, 0x4667, 0xba77, 0x4110, 0x43a9, 0x1d4d, 0x42c5, 0xb2f7, 0xb216, 0x40d9, 0xb06f, 0x43ba, 0x41b9, 0x467f, 0xa135, 0xbaee, 0x40e7, 0xbfc6, 0xb241, 0xa4a6, 0x423b, 0x408e, 0x364b, 0xa098, 0x4005, 0xb275, 0xb0f2, 0xba79, 0xbfa0, 0x4387, 0x41d9, 0x0761, 0xb251, 0x418c, 0xba33, 0xbf51, 0x4007, 0xb256, 0x44c3, 0xb212, 0x434f, 0x40b4, 0x0edf, 0x4546, 0x4343, 0x45b3, 0xb29c, 0x40fe, 0x341e, 0x1af1, 0x42dc, 0x1c07, 0xba79, 0x08bb, 0xbfc7, 0x16a1, 0x4655, 0x1fe7, 0x4235, 0xbf3d, 0x338e, 0x43a2, 0x36ad, 0x2e82, 0xa991, 0x423e, 0xba76, 0xba17, 0x1a36, 0xbad3, 0x42e4, 0x41d8, 0xb206, 0x41fd, 0x3bb8, 0x42a0, 0xb2ec, 0xbacd, 0x43c3, 0x1bde, 0xba65, 0x4188, 0x41a4, 0x40fb, 0x4225, 0x25ce, 0xbf97, 0xb22b, 0x19e4, 0x43cc, 0x4101, 0x27f0, 0x1f7e, 0x1d8d, 0xb23d, 0x4280, 0x1d99, 0x41ca, 0x4170, 0x440d, 0xbae8, 0x4114, 0x0519, 0x41db, 0x214d, 0x4255, 0xb25f, 0x1b50, 0x14bd, 0xba46, 0xbf0f, 0x43b1, 0x20c0, 0xa4da, 0xb273, 0xbf65, 0x0636, 0x43e3, 0xb006, 0x410e, 0x161c, 0xba02, 0x4471, 0x2f5a, 0x40e6, 0xbf55, 0x4030, 0xba21, 0x412c, 0xb2a7, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd10f75bd, 0x744765d0, 0x4700bcdf, 0x41a4f9ee, 0x0164f011, 0x5d046666, 0xcc6ba3ee, 0xcaed7f40, 0xc754ff08, 0xa741fc6a, 0x1e4ce65a, 0x41d5f9d4, 0xf0c8c8a5, 0x9b837a0d, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x0000004d, 0x00000000, 0x03380000, 0x00000003, 0x00000000, 0x00000000, 0x00000000, 0x5553dda0, 0xd10f75bd, 0xd10f75bd, 0xf0c8c8a5, 0xffffff86, 0x5553ddac, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xa199, 0x43c5, 0x4111, 0x1dd7, 0x4284, 0x4103, 0x41a4, 0xbf24, 0x0c40, 0x44e5, 0x448a, 0x1c19, 0x4275, 0x1e0c, 0xa7fd, 0x1c09, 0x009a, 0x1ee8, 0x459c, 0x43d0, 0x40b2, 0x426f, 0x402e, 0x4421, 0xbae9, 0x2640, 0x402a, 0x1fcd, 0x2ff9, 0x3926, 0xbf1b, 0xb281, 0xaacb, 0x45b0, 0x1c61, 0xba01, 0xb22d, 0x277d, 0x4042, 0x4374, 0x13b1, 0x146d, 0x410f, 0xb09b, 0x179d, 0x421b, 0x1ae6, 0x4085, 0x2920, 0x425a, 0x43df, 0xb244, 0x4080, 0x42dd, 0x4178, 0x43a1, 0x4057, 0xa843, 0xb2ea, 0x4195, 0xbf3f, 0x4058, 0x4042, 0xa1e7, 0x4263, 0x4060, 0x45ea, 0xba42, 0x40c5, 0xbf63, 0x0953, 0x4287, 0x11ec, 0xba1b, 0x2900, 0x1d63, 0x425b, 0xb209, 0x427d, 0x2ffd, 0x40a6, 0xba7b, 0xbfa1, 0xb227, 0x1390, 0xb2ae, 0x413f, 0xb230, 0x1ea6, 0xb063, 0x44d9, 0x43ed, 0x38e6, 0xb266, 0x11c6, 0x4353, 0x4395, 0x423d, 0x4038, 0xb0dd, 0x418d, 0x405b, 0x4165, 0xbf39, 0x1373, 0x4200, 0x458b, 0x42d4, 0x4307, 0x1f3f, 0xbf95, 0x405d, 0x43cd, 0xbaf2, 0xa1f9, 0x40f1, 0x40cf, 0x40ec, 0xb203, 0xbfcc, 0x420f, 0x1ebf, 0x2b1b, 0x4186, 0x1e1f, 0x464a, 0xba6a, 0x423d, 0x4453, 0x1f67, 0x434e, 0x27b3, 0x15d6, 0xb2d8, 0xbf53, 0xba22, 0x413b, 0x187f, 0x42fd, 0xa8c3, 0xba09, 0x41fd, 0x4442, 0x420c, 0xb296, 0xbac5, 0x4215, 0x40f2, 0x4082, 0x1a04, 0x11e2, 0x1cbf, 0x43d9, 0xb216, 0x4026, 0x3608, 0x43ea, 0x4022, 0x4092, 0x0e1f, 0x429b, 0xb2e8, 0x42df, 0xbfad, 0x349f, 0xb2ff, 0x4212, 0x4168, 0xbad1, 0x390c, 0x41e0, 0xa34a, 0x40cf, 0x419f, 0x40db, 0x2fbd, 0xb0a9, 0x4329, 0x4351, 0x4129, 0x4104, 0x4268, 0xb2c4, 0xbf13, 0x1afa, 0x2de8, 0xb0e4, 0x4191, 0xb234, 0x4041, 0xb0b4, 0xb05c, 0xba7f, 0x2b96, 0xbfc6, 0xba54, 0x434b, 0x447f, 0xbfc2, 0x4113, 0xb2ea, 0xae3d, 0x417a, 0x42f8, 0x1884, 0x1f4a, 0xba51, 0xa2aa, 0xbf15, 0x4386, 0xba75, 0x40e8, 0x1f84, 0x4202, 0x42d7, 0x19e7, 0xbf80, 0x4168, 0x45e6, 0xbff0, 0x4152, 0xbf0c, 0xabb2, 0xa8e2, 0x42f0, 0x43d5, 0x405c, 0x4229, 0x42b1, 0xb0d1, 0xbf98, 0xba2d, 0x43c0, 0x024c, 0x3d62, 0xb22e, 0x400d, 0xba54, 0x4002, 0x2696, 0x1c64, 0x1d10, 0xb2f5, 0x40b8, 0xb04f, 0x24dd, 0x43fb, 0x4042, 0x182b, 0xbafd, 0x46d2, 0xbf78, 0x4317, 0x2901, 0xadeb, 0xa020, 0x43de, 0x352d, 0x4025, 0x4067, 0x4184, 0x42af, 0x4197, 0x410b, 0x4240, 0x42ef, 0xbfcc, 0x3013, 0x454d, 0xbf58, 0x4317, 0x0df1, 0x1877, 0x422f, 0xb01b, 0xb083, 0x402d, 0xb24c, 0x3837, 0xb0ad, 0x1a40, 0x0cad, 0xa457, 0x1c71, 0xb0c5, 0x40c7, 0xbf24, 0x4115, 0x4613, 0x41da, 0x1e59, 0x1099, 0xa19d, 0x25c9, 0xba1f, 0x0da3, 0xa9c4, 0x4263, 0x4128, 0xaae7, 0xb051, 0x427c, 0xba0b, 0x41fc, 0xb234, 0x1cec, 0x40be, 0xb221, 0xba4c, 0x1fcd, 0xb21e, 0xbf4c, 0xb0fd, 0x4172, 0x1d00, 0x1b78, 0x424e, 0xbf1e, 0x1f40, 0x0324, 0x4271, 0x41f0, 0xa81e, 0x4692, 0xb21e, 0x4114, 0xbf39, 0x46aa, 0x19a7, 0x2315, 0xbadf, 0xab7d, 0x1edf, 0xba3a, 0x1a7a, 0x1ede, 0xba56, 0xb074, 0xb207, 0x460f, 0xb20f, 0xba15, 0x40de, 0x40d3, 0x4029, 0x3388, 0xb265, 0x4274, 0xba28, 0xbfd4, 0xbac1, 0x403a, 0xb011, 0xb282, 0xb212, 0xbf37, 0x4134, 0x401e, 0x42b3, 0xbaf5, 0xbfbe, 0x417e, 0x4009, 0x14f9, 0xba76, 0xb2d3, 0x40de, 0x43ee, 0x446f, 0x4238, 0xbf90, 0x4302, 0x1f23, 0x0475, 0xb256, 0x40a3, 0x4008, 0xb0d8, 0x4147, 0x3b2a, 0x00cf, 0x4381, 0xb233, 0xbff0, 0xba1c, 0x43ff, 0x41a2, 0x4376, 0xb20f, 0xbf49, 0xb2b4, 0x0dbc, 0x3189, 0x418e, 0xba32, 0x1bd8, 0xbf6f, 0xba48, 0xb207, 0xa2ad, 0x411d, 0x1a87, 0x2df4, 0x03c2, 0x2dfd, 0x4046, 0xbf90, 0x3240, 0xb015, 0x4003, 0xb29c, 0x44a2, 0x43b7, 0x4378, 0x0fd8, 0x42fe, 0x466c, 0x41a6, 0x1940, 0x4087, 0xbfba, 0x4106, 0x3ee3, 0x4219, 0x409b, 0x1976, 0x18e7, 0x24f6, 0x3f8d, 0xbad5, 0x40d9, 0x1c9a, 0x2916, 0x1cd9, 0x41e2, 0x4069, 0x1c0c, 0x45c3, 0xb0e6, 0xbfb9, 0x40dc, 0xb055, 0x0946, 0x349c, 0x4004, 0x42d3, 0x1e78, 0x43f0, 0x427c, 0xb20b, 0xb2da, 0x4206, 0x42aa, 0x1d43, 0x428a, 0x19a6, 0xb04f, 0x1f39, 0x2fdc, 0x0fa5, 0x067d, 0x2bbc, 0x405a, 0x0080, 0xbfdd, 0x1ab5, 0xba61, 0x46ec, 0x2e41, 0x43aa, 0x4080, 0x437a, 0x4112, 0x428a, 0x1efc, 0xbaf3, 0xbfc3, 0x4203, 0x45ed, 0x42d6, 0x4552, 0x0588, 0x2ab8, 0x19d9, 0x4391, 0x1c52, 0xbf4f, 0x422e, 0x43c2, 0x43be, 0x13f3, 0x40da, 0xb2ba, 0x3a04, 0xb2cf, 0x0bb2, 0x40ab, 0x42fd, 0x3dd3, 0x417d, 0xb08a, 0x4494, 0xa6be, 0xb229, 0x426d, 0x341a, 0x4312, 0x4212, 0x4258, 0x4195, 0xb2ff, 0x4011, 0x410a, 0x42c5, 0xbfc3, 0xadc5, 0x4367, 0xb29b, 0x4272, 0x416b, 0x42d9, 0xb0bb, 0x458d, 0x4290, 0xa66c, 0x1b4c, 0xac1d, 0x4092, 0xbfd5, 0xb081, 0x438b, 0x4029, 0x00df, 0xb009, 0x449c, 0x432c, 0x425f, 0x08fe, 0xb2dd, 0xba3c, 0x4053, 0x4111, 0x43e5, 0x42fe, 0xba44, 0xb26b, 0x3581, 0x439c, 0x354a, 0x0240, 0x3da2, 0x1bad, 0x4143, 0x407e, 0xbf76, 0x40ac, 0x403e, 0x402b, 0x1fdf, 0xb2c3, 0x40cf, 0x33fb, 0x42cb, 0xb2de, 0x4075, 0x1f1c, 0x43f7, 0x4081, 0x4386, 0xb248, 0x1b37, 0xb237, 0xb251, 0xbae9, 0xb2fb, 0x43da, 0x41d1, 0xba24, 0xbfc3, 0xad4b, 0x40ad, 0x0935, 0xb2b5, 0x43f4, 0xa290, 0xb24c, 0xaea0, 0xbaf1, 0xba23, 0xab45, 0x1e3d, 0x4219, 0x4350, 0x12e3, 0x0d50, 0x4545, 0xbf70, 0x1c23, 0xb28c, 0xb2df, 0x4375, 0x44d4, 0x42b2, 0x43dd, 0xbaf0, 0x41ef, 0xbf91, 0xb24b, 0xab23, 0x1a7b, 0x285e, 0xb09a, 0x40d1, 0x4149, 0x4084, 0x1cc5, 0xba75, 0x4348, 0xba27, 0xa8ac, 0xa7d2, 0x42b9, 0x334a, 0x4594, 0x40c1, 0x42e5, 0x41a1, 0x43f8, 0x4378, 0xa603, 0xb264, 0xba4d, 0xbf53, 0xb073, 0x1f6a, 0x1d2f, 0x1c93, 0x46b3, 0x4289, 0xa678, 0x1899, 0x434b, 0x438d, 0x43e2, 0xa5b7, 0x1f65, 0x3ea4, 0x44dd, 0x43ba, 0xb2b5, 0x2490, 0x4316, 0x18d2, 0xb257, 0x4472, 0xbfb4, 0x403c, 0x4036, 0x1989, 0xa6ae, 0xb211, 0x1d91, 0x43ea, 0x404a, 0x30c3, 0x43ac, 0x124b, 0x164d, 0x40e4, 0x4271, 0xb2c4, 0x2d41, 0x417b, 0xbf14, 0x40cc, 0xba78, 0x426e, 0x31c8, 0x44c3, 0xafc8, 0x43d0, 0x1d72, 0x0d4e, 0x1c65, 0xa4d5, 0xbf7d, 0x2ea8, 0xb0c3, 0x4424, 0x466c, 0x16b6, 0x43c5, 0x181f, 0x1f13, 0x398f, 0x42e2, 0x4354, 0x18b0, 0x2c22, 0x43c8, 0xae35, 0x44d4, 0x4113, 0x1834, 0x4590, 0xbf27, 0x400e, 0x42e3, 0x4289, 0xb28d, 0x0cf4, 0x32b2, 0x0f02, 0x414b, 0x40d5, 0x43c8, 0x4565, 0x376a, 0x330d, 0x1a0e, 0xbf53, 0xb0bd, 0x42b4, 0x422e, 0xb05a, 0x41bb, 0xa3cb, 0x0947, 0xb064, 0x42c4, 0x4156, 0x1bb2, 0x0104, 0x42c6, 0x4181, 0x4096, 0xba72, 0xb0da, 0x42e1, 0xa882, 0x427e, 0x4078, 0x4174, 0x4165, 0x406c, 0xbf01, 0xb0e3, 0x40ec, 0x29ba, 0xba6a, 0x183d, 0xb22c, 0x1b41, 0xb20d, 0xbf18, 0x027e, 0x42de, 0x429a, 0x4326, 0xbace, 0x412b, 0x4394, 0x43cf, 0xbfa1, 0xaeee, 0x4398, 0x18da, 0x42d5, 0x1c5e, 0x3817, 0x312b, 0x145c, 0xb294, 0x42c0, 0x40b9, 0x42fb, 0x3eb7, 0x44e1, 0x43ef, 0x4137, 0x0b6e, 0x4248, 0x4260, 0x4130, 0x43ea, 0x413f, 0x1ae0, 0x41b9, 0x43dd, 0x40dc, 0xbf8f, 0x42bb, 0x425a, 0x40cf, 0xbaca, 0x1f8a, 0xb243, 0x1802, 0x1e4f, 0x035e, 0x411a, 0x4160, 0xae4e, 0x1d60, 0xbfdb, 0x32ef, 0x42b2, 0x46d4, 0xbf70, 0x1ffa, 0x40bc, 0xb05a, 0xbaef, 0xb2ef, 0xb2aa, 0xbf8c, 0x194e, 0x42e9, 0x46d3, 0x4388, 0x40d0, 0x200f, 0xba17, 0xba0d, 0x443e, 0x2066, 0x410d, 0x40ca, 0xa869, 0xb07f, 0x425e, 0x432d, 0xb2b5, 0x43db, 0x1d95, 0x435a, 0xaf6a, 0xb2b0, 0x1b4d, 0x41de, 0xba5f, 0x1217, 0xbf81, 0xb217, 0xb224, 0x1db7, 0x1a1b, 0x2695, 0x406a, 0x3710, 0xb291, 0x3615, 0x3e59, 0x4181, 0x188d, 0x41fd, 0xbace, 0x46b2, 0x431a, 0x4121, 0x43bf, 0x186f, 0x4178, 0x00f5, 0xbf39, 0xa8c3, 0x0415, 0xba21, 0xbad8, 0x3fc6, 0x4116, 0x41f6, 0x443e, 0x0825, 0x43cf, 0x4388, 0x3e82, 0x171a, 0xb0e0, 0xbfa7, 0x42eb, 0x4006, 0x4029, 0x4128, 0x422a, 0x4279, 0x44a1, 0x45cc, 0x1913, 0x4174, 0xbf54, 0xb2a7, 0x4123, 0x4236, 0xba06, 0x425d, 0x1d1c, 0xbf12, 0x12b2, 0xba57, 0x19cf, 0xba6a, 0xb29e, 0xbfb6, 0x2718, 0xaf18, 0xba0b, 0xb2d7, 0x405e, 0x3f04, 0x268f, 0x426e, 0x1a97, 0xa68b, 0x1dab, 0x4104, 0x4661, 0x280a, 0x4550, 0x29ff, 0x41c9, 0xbf70, 0x432b, 0x1fb6, 0x2eb9, 0x419c, 0xb2cc, 0x2b88, 0x0da0, 0xb2b1, 0x42a6, 0x1ad6, 0xbfc1, 0x4599, 0x40b2, 0xb291, 0x43d2, 0x421a, 0x4209, 0x432a, 0xb04f, 0xbaf6, 0x16c6, 0x46d3, 0xbac0, 0x1dbe, 0xbff0, 0x40e7, 0x2de7, 0xbf21, 0x1d42, 0x0f07, 0x4274, 0x4151, 0x46e4, 0xb00a, 0xbad8, 0xbf34, 0x40a1, 0x2ef9, 0x4396, 0xbf53, 0x40e9, 0x4239, 0x42da, 0x405f, 0xba0c, 0x4368, 0xb2de, 0x4449, 0x4278, 0xb2e9, 0x4317, 0x405f, 0xb063, 0x4100, 0x4698, 0x40b0, 0x45c5, 0xb2df, 0x454f, 0x2aca, 0x42ea, 0x43b0, 0x3976, 0x4327, 0x2cbd, 0xbf7d, 0x1c79, 0x2d95, 0x41cd, 0x1af1, 0x408a, 0x2eae, 0x419d, 0x1859, 0x42bf, 0x4272, 0x412a, 0x42a9, 0x0694, 0x1f68, 0x4653, 0x3a31, 0x4277, 0xa6d1, 0x413c, 0xb23c, 0xbfc0, 0x42f6, 0x4377, 0x418f, 0x464c, 0x4133, 0x4129, 0xb294, 0xbfa9, 0x247d, 0x1b16, 0x21cc, 0xbf70, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x181236f9, 0x459e9e0f, 0xd961342f, 0x122ddd5e, 0x38b78514, 0x9508b2ee, 0x97896aed, 0xad92e120, 0xfd37fa46, 0x2e0a81f8, 0x8cf97fb4, 0x8d4b1243, 0x5e1eda21, 0x54da9be4, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0x00fffff3, 0x000000cc, 0xffffffce, 0xffffffff, 0x0000007d, 0x00fffff8, 0x00001b20, 0xffff4210, 0x00000007, 0x6e4f1c33, 0xfffffaff, 0xfffffaff, 0x000000c5, 0x54dab658, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4239, 0xba45, 0xa273, 0xbacd, 0xb2e4, 0x0b51, 0x406c, 0x43a9, 0x1ae2, 0xbad9, 0x1eff, 0xbfc2, 0x42b3, 0xb0ef, 0xb0ab, 0x2572, 0x4344, 0x4165, 0xb279, 0x127a, 0xb2e2, 0xac76, 0x35b9, 0xb2d5, 0x41d2, 0x3d37, 0x43f2, 0xbf9a, 0xb207, 0x40b9, 0xba37, 0x21f3, 0x03e4, 0x430f, 0x444d, 0x1d73, 0x4354, 0x07a6, 0x1b8f, 0xb284, 0xb239, 0xbaed, 0xb2b2, 0xb2a6, 0x45da, 0xb20a, 0x1f8f, 0xb21e, 0x40fa, 0xa7cd, 0xb237, 0xbfd7, 0x444e, 0x4113, 0xbad8, 0x34c6, 0xbf43, 0xa315, 0x14bc, 0x416b, 0x4246, 0x3ae5, 0x46a3, 0xac12, 0xba29, 0xad08, 0x11c5, 0xb28e, 0x1443, 0xbf60, 0x41cb, 0xba32, 0x40a7, 0x40a0, 0x4206, 0xb250, 0xb268, 0x435b, 0xbf8f, 0x429d, 0xbfc0, 0x422b, 0x43e4, 0xb070, 0x1d90, 0x405b, 0x40f8, 0x435b, 0xb232, 0x4251, 0xbfac, 0x424d, 0x39e5, 0xb2b8, 0x46e3, 0x0171, 0x46bd, 0xbf90, 0x40c3, 0x36af, 0x429a, 0x41e9, 0xb25d, 0xb000, 0x3c9d, 0x096e, 0x4030, 0xbfd1, 0x1f7e, 0x27fe, 0xb073, 0x41e0, 0xa276, 0xb0ae, 0xb062, 0x1d63, 0xb203, 0xbfaf, 0x4323, 0x4376, 0xbac1, 0xb0e5, 0x43b8, 0x26f1, 0x4120, 0x37ee, 0xbf7a, 0xb232, 0x43ca, 0x43b9, 0xb2c2, 0x46c0, 0x3ea3, 0x152b, 0x4202, 0x4232, 0xb2b0, 0xbf15, 0xb299, 0xba6a, 0x4199, 0x102a, 0x4023, 0x401a, 0xaa93, 0x4190, 0xba4c, 0xa1ab, 0xb2bd, 0xb017, 0x4375, 0x40b6, 0x43b8, 0x18cf, 0x18c8, 0x406c, 0xac65, 0x2956, 0x0fe6, 0x4001, 0x1c31, 0x423a, 0x42c6, 0x41fa, 0x1bf9, 0x40b7, 0x21bc, 0xbf43, 0x405e, 0x1ce7, 0x41a4, 0x415e, 0x41eb, 0x40a9, 0x41fc, 0xb004, 0x42e1, 0x4370, 0x42db, 0xbaee, 0x42df, 0x0d78, 0xb0f8, 0x4394, 0xb2a4, 0x3923, 0x3c97, 0x4206, 0xb0fb, 0xb21c, 0x10cb, 0x4050, 0x3634, 0xbf03, 0xb0ad, 0x46ed, 0x38c3, 0x1945, 0x4236, 0x17c2, 0x282d, 0xbae2, 0x40b6, 0x22e8, 0x42ac, 0x1b6d, 0xba53, 0x414e, 0xb013, 0xb0c2, 0x41ff, 0xb281, 0x1cd3, 0x41da, 0x4141, 0x420f, 0xa91b, 0x35d5, 0xa6d8, 0xba09, 0x4156, 0x278e, 0xbf11, 0x45be, 0xa4ee, 0xb28b, 0x3d9d, 0xb2b3, 0x0dea, 0xa044, 0x43d1, 0x2843, 0x425b, 0xa5dc, 0x26b9, 0x1ef7, 0xba61, 0x028d, 0xa302, 0xb061, 0x4306, 0x2b7d, 0x1b33, 0x4215, 0x1dbe, 0x40cc, 0x41af, 0x1fef, 0xbf77, 0x4308, 0xba38, 0x4306, 0x4099, 0xb275, 0xbf6d, 0xbae5, 0x1f9f, 0x1cc6, 0x1101, 0x42ca, 0x2e58, 0x41a7, 0xb235, 0x29fa, 0x43dd, 0x41d8, 0x402c, 0x4138, 0x41e8, 0x3d38, 0xba10, 0x41f6, 0xb26f, 0x1e35, 0xba03, 0xb2b3, 0x1c33, 0x40f7, 0x3fbf, 0x4359, 0x45ae, 0xbf1f, 0x405e, 0x43f4, 0x419a, 0x207e, 0xa253, 0xb084, 0xac7c, 0xb232, 0xbf00, 0xb01e, 0x1b16, 0x318e, 0xae09, 0x447c, 0x1a2c, 0xaf1f, 0x0953, 0xaf93, 0xb23f, 0xb286, 0x43b0, 0x1cdb, 0xba74, 0x2e38, 0xbf3d, 0x3640, 0x1eba, 0x3f29, 0x4339, 0x43b4, 0x257a, 0xba48, 0xb20d, 0x188c, 0x412b, 0x1232, 0xba24, 0x464e, 0xb2f5, 0x42fe, 0x40d8, 0xa5b3, 0xb210, 0x43ad, 0xaf54, 0xb273, 0x0770, 0x4261, 0x415a, 0x42cf, 0x3104, 0x1c29, 0x368d, 0xba20, 0xbfd2, 0x416b, 0xb2ed, 0x40a7, 0x1ff4, 0xbaf4, 0x1d6a, 0x1d89, 0x4398, 0x1839, 0xb0d7, 0x458e, 0xba38, 0x4019, 0x432c, 0x028b, 0x4323, 0xb09c, 0x1444, 0x423d, 0xb286, 0xbfe0, 0xba19, 0x1439, 0x346c, 0xbfa5, 0xb0ab, 0x2da0, 0x403d, 0x40b0, 0x404a, 0x105c, 0x1c9c, 0x408b, 0x4239, 0x2701, 0x226e, 0x41ba, 0xb246, 0x4123, 0xb0ba, 0x40b6, 0xbfcf, 0xb065, 0x4220, 0x4200, 0x410e, 0x0e39, 0x415f, 0xba6e, 0x43ef, 0x424c, 0x3ba0, 0x3e8d, 0x45b6, 0x4037, 0x4179, 0x4241, 0xb0dd, 0xbf2f, 0x40ff, 0x434a, 0x4336, 0xbada, 0xb279, 0xb0f4, 0x11c7, 0x41c2, 0x1e64, 0x4301, 0x1e52, 0x1892, 0xbfd2, 0x409a, 0xba57, 0x4197, 0x432d, 0xbf25, 0x40a2, 0xac2d, 0xaaed, 0xb257, 0x34e1, 0xba04, 0x4258, 0x18b0, 0x4384, 0x1397, 0xa3ae, 0xb0d7, 0xb28f, 0x425e, 0x422b, 0x4314, 0x222d, 0x3b0e, 0xbf97, 0x41c3, 0x4209, 0x43b1, 0x2442, 0xa92c, 0xb02c, 0xbf35, 0xb293, 0x2e27, 0x41f5, 0x4322, 0x0a11, 0x40ee, 0x0966, 0xbad5, 0xb28e, 0x19b3, 0x4015, 0x45be, 0xb25d, 0x415d, 0x41e0, 0x1bc8, 0xbfd9, 0xb08a, 0x1c28, 0x4281, 0x4309, 0x430a, 0x402c, 0xb2c2, 0x40e4, 0xba22, 0xb00d, 0x418c, 0x32ee, 0x40a5, 0xb2bb, 0x43fb, 0xb2f9, 0x43bc, 0x41b8, 0x4098, 0x1a85, 0x4162, 0x41d2, 0x40be, 0x4613, 0x2b71, 0x0ee4, 0xb0ef, 0xbae0, 0xbf17, 0x43cb, 0x4082, 0xb20f, 0x41a0, 0x198a, 0x2530, 0x2c43, 0x3981, 0x1f52, 0x423e, 0xbf04, 0x43a2, 0x4043, 0x4079, 0x1bb9, 0x4268, 0x2138, 0x1aee, 0xbf88, 0x419c, 0xb2cd, 0x1cf4, 0x03df, 0x1acf, 0x4589, 0xba06, 0x4207, 0x43d8, 0xba09, 0xbfa0, 0x4606, 0x4243, 0x417c, 0x3043, 0xbf6e, 0x4365, 0x4488, 0x405d, 0x2d15, 0x41c4, 0x4111, 0xbf62, 0x4321, 0x4135, 0x3c6e, 0x0893, 0x260f, 0x400c, 0x4091, 0x4135, 0x40a7, 0x116e, 0x467d, 0xbf2e, 0x0fae, 0x4091, 0xa2a7, 0x0bed, 0x4550, 0x4268, 0x4305, 0xbf46, 0xb2e8, 0x3fe3, 0x1942, 0x439f, 0xa9d4, 0xbf9b, 0x4236, 0x1d7a, 0x28a1, 0xb0d2, 0x3a43, 0x40f6, 0x42c5, 0x4275, 0xb245, 0x43c2, 0x3019, 0x1eff, 0xa434, 0x4360, 0x3af7, 0x42bb, 0xbf80, 0x44ab, 0xbf4d, 0xab4f, 0xb068, 0xaf79, 0x4291, 0xb281, 0x4621, 0x42fa, 0xb2fe, 0x10f6, 0xbf80, 0x4018, 0x41ed, 0xb24f, 0xb02e, 0x456f, 0x41da, 0xbf1d, 0xbf90, 0x4118, 0x4307, 0x422d, 0x430e, 0x41c3, 0x0fe8, 0x315d, 0xb2b8, 0x1d2f, 0x1e41, 0x0a83, 0x41ac, 0x1d8c, 0x43b9, 0x4691, 0x45c2, 0xba25, 0x1cf0, 0xba01, 0x0098, 0xbfb2, 0x05fd, 0x404f, 0x4268, 0x0194, 0xabd2, 0xbfd0, 0xbf93, 0x1489, 0xb274, 0x2bec, 0x4225, 0x2a29, 0xba47, 0x23e9, 0xa627, 0xba14, 0x40ea, 0xa9ef, 0x27b9, 0xba57, 0xbf5d, 0x1a24, 0x465d, 0x4319, 0xba31, 0x4170, 0x04ed, 0xb295, 0xbaca, 0x417b, 0x4130, 0xb28a, 0x406b, 0xb252, 0x4090, 0x371e, 0x43fd, 0x1b99, 0x222a, 0x40f2, 0x4607, 0xbf91, 0xa406, 0x0d68, 0x410b, 0x41cb, 0xbfca, 0x43b5, 0x1cc7, 0x3cf1, 0xb275, 0x438c, 0x409d, 0xa26c, 0x433d, 0x1e60, 0xb24f, 0xbf63, 0x1b7e, 0x423d, 0x129a, 0xb0d1, 0x40b2, 0x1f17, 0x1fa4, 0x4095, 0xb068, 0xbf70, 0x40aa, 0x2399, 0x012b, 0x00cb, 0x410b, 0x2251, 0x1eed, 0x1f84, 0xaab2, 0xb064, 0x4090, 0xb043, 0x4180, 0x46a0, 0x2617, 0x4203, 0x4122, 0x136f, 0xbf46, 0x195a, 0x43d9, 0xb025, 0xb02b, 0x42db, 0x41e2, 0xa62b, 0x404e, 0xba7f, 0xbfa9, 0x31f0, 0x43db, 0x46b1, 0x404a, 0x1862, 0x425a, 0xb21e, 0x41a0, 0x4573, 0x40f0, 0xb228, 0x423b, 0x1f8f, 0x410f, 0x0667, 0x44a3, 0x02bf, 0xb280, 0x4008, 0x44d0, 0x4216, 0xbf61, 0x3b93, 0x420f, 0x0b94, 0xa28c, 0x432b, 0x1fa5, 0xbf5e, 0x4305, 0x42dc, 0x42ce, 0x4335, 0x1c24, 0x4615, 0x4295, 0x11a7, 0xba38, 0x42e6, 0x46aa, 0x1faa, 0xb2b0, 0xb222, 0x1112, 0x094a, 0x43d4, 0x4034, 0x1b1b, 0xbf66, 0x1fa4, 0x4082, 0x4228, 0x1b74, 0xb252, 0x3875, 0xb2f6, 0x23ec, 0x420b, 0x40b2, 0x46a8, 0x06f2, 0x1954, 0xb206, 0xbfac, 0x421d, 0xbac3, 0xbf80, 0xb0e5, 0x4274, 0xbfaf, 0xbf70, 0x36bc, 0xb2ae, 0xba44, 0x405c, 0xbf70, 0xb0be, 0xbadf, 0x304b, 0x4231, 0xb250, 0x18ca, 0xba6b, 0x4144, 0xb248, 0x2d0a, 0xbf90, 0x4474, 0x43e4, 0x4307, 0xa8c6, 0x4153, 0x41c6, 0xbf5c, 0x4161, 0x0875, 0x432f, 0xb234, 0xb2f0, 0x417e, 0xb2b2, 0x3ea9, 0x42d7, 0xba5e, 0xb04b, 0x1d7c, 0x4227, 0xbfbd, 0x422f, 0x43db, 0x3b1d, 0x41a3, 0x431b, 0xb29f, 0x433a, 0xb275, 0xa6ed, 0x4035, 0x07e4, 0x1978, 0x4096, 0x1f30, 0x386d, 0x4132, 0x41e5, 0xb254, 0x04e9, 0x1f81, 0xb2c5, 0x1507, 0xbf49, 0x4464, 0x431d, 0xa8cc, 0x41ab, 0x42df, 0x4158, 0x43e5, 0x4243, 0xb230, 0xb24b, 0xb262, 0x0f27, 0x42c5, 0x4321, 0xbad1, 0x0b26, 0x24b1, 0x18e4, 0xb285, 0xbf8d, 0x434a, 0x42f9, 0x4007, 0x4061, 0xbfc3, 0x3a04, 0x45ed, 0xa055, 0x44c8, 0xa20f, 0x423d, 0x1ee2, 0x44b5, 0x1e41, 0xae96, 0x0d96, 0x411b, 0x4384, 0x45ab, 0xbf15, 0x1f8a, 0x0b2d, 0x42e4, 0x437d, 0xb23c, 0x3ca6, 0x1d27, 0x429f, 0xbad6, 0x4056, 0x429a, 0xbf02, 0xb0c7, 0xbfb0, 0x4300, 0x413c, 0x40ca, 0x42e3, 0x426b, 0x08df, 0xa9a2, 0x44ec, 0x42e1, 0x4295, 0x0436, 0xb2e8, 0xbf53, 0x4499, 0xa2c7, 0x4000, 0x4274, 0x4299, 0xb211, 0x40c7, 0x31f0, 0xb25e, 0xb2f4, 0x413b, 0x1d10, 0x3f32, 0xbf22, 0x4381, 0x2c17, 0xb2a3, 0x4075, 0x3d15, 0x4455, 0x1a65, 0x40c1, 0x2389, 0x41eb, 0x1238, 0xb2ab, 0x1fea, 0x4022, 0x1f25, 0x4072, 0x403e, 0xb021, 0x45c5, 0x1c07, 0x421a, 0x4557, 0xbf60, 0x416b, 0xb2f3, 0xbf7d, 0x4005, 0x3e21, 0xbf60, 0xb225, 0x409b, 0x4340, 0x00db, 0x413d, 0x4275, 0x4574, 0x44b1, 0x412f, 0x4049, 0x2295, 0x20e6, 0xbf0c, 0x4021, 0x185e, 0x4317, 0x41a6, 0xb24e, 0x41c4, 0x2d7b, 0xb07b, 0x0b1e, 0x2317, 0x4254, 0x295f, 0x42f9, 0x4644, 0x1a5e, 0xb208, 0xba1e, 0x400d, 0x1bc5, 0xa2fc, 0x194f, 0xbfb5, 0xb26e, 0x40fd, 0x37c1, 0x4314, 0x432e, 0xaa7f, 0x17da, 0x43e1, 0x1ad4, 0xb06b, 0x4001, 0x071f, 0xba54, 0x1eeb, 0x1ddb, 0x2da8, 0x432c, 0x307f, 0x41c2, 0x2eea, 0xbad1, 0x43d7, 0xbf5c, 0x1edb, 0xbaed, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x7050d002, 0x2ad824dc, 0xcda44a5a, 0xbe440822, 0x6fa909fc, 0x5324049c, 0xb3f8f834, 0x42ed399f, 0x207b9830, 0xa4b5347a, 0x1f876040, 0xfb4d4a81, 0x16ce792a, 0x219b1d90, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x0000007f, 0x00000000, 0x00000000, 0x00000004, 0x00000000, 0x00000000, 0x17000000, 0xffffffff, 0x00001631, 0x0000160f, 0x00000001, 0x16ce8d4b, 0x16cfdbf9, 0x000166eb, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1d91, 0xbae9, 0x3565, 0xb2f7, 0x4627, 0x41f1, 0xbfe8, 0x4294, 0x0fb8, 0x2445, 0x0e33, 0x29fa, 0x116e, 0x345c, 0x44b5, 0xb2d2, 0x4013, 0x4246, 0x2612, 0x4133, 0x42bc, 0xb28a, 0xbf14, 0x4370, 0x194f, 0x4319, 0x4264, 0xb0d1, 0x1c87, 0x4329, 0xbfc4, 0x441e, 0x4212, 0x1b80, 0x4261, 0x0436, 0x46e1, 0x444e, 0x1a0b, 0x40d3, 0xa4e6, 0x407b, 0x4201, 0x426d, 0x43a3, 0x436d, 0x1a3c, 0x1b83, 0xad83, 0xbadf, 0x4016, 0xb058, 0xacc4, 0xbf90, 0x43ea, 0x4445, 0xb2fe, 0x08da, 0xbf29, 0x41a9, 0x0835, 0x0f8d, 0x1e32, 0x0a48, 0x41f4, 0xbaca, 0x41e4, 0x41ac, 0xba02, 0x1db6, 0xb099, 0x1f86, 0xbfd0, 0xa3b6, 0x463a, 0x4108, 0x2cb1, 0xb0ce, 0xb261, 0xba1b, 0x1edc, 0xba7a, 0x4285, 0xb206, 0x4078, 0x4252, 0xa63c, 0x107d, 0xbf9b, 0x4331, 0x40c1, 0x415c, 0x4062, 0x186b, 0x1e9d, 0x4014, 0x32c3, 0x43e7, 0x40e6, 0x4382, 0xba6e, 0x4332, 0x1cea, 0x0bb1, 0x1de5, 0x429c, 0x41f1, 0x4243, 0x0059, 0x41db, 0xa558, 0xbf7c, 0x4203, 0x43a7, 0x43d8, 0x4460, 0x413e, 0x439e, 0x424d, 0x1ae1, 0xbadd, 0x1cb2, 0x415a, 0xb240, 0x311d, 0x41de, 0x155b, 0xa4c4, 0xb25d, 0xb2b4, 0x4329, 0x445c, 0x40be, 0xa150, 0x4284, 0x44eb, 0x456d, 0x40a2, 0x439d, 0xbf6d, 0x4073, 0xa3d2, 0xb036, 0x41c9, 0xbad3, 0x41a2, 0x432a, 0xba1c, 0xbafe, 0x42a1, 0x43e8, 0x4120, 0x414b, 0xb202, 0x251f, 0xba64, 0xbac8, 0x1977, 0x4291, 0x19a1, 0x1459, 0x1af3, 0xb0ca, 0x3165, 0x1a2f, 0xb224, 0xa3d5, 0xbfd5, 0xb20e, 0xb2df, 0x4689, 0x4143, 0xbae5, 0x4169, 0x40e7, 0xa294, 0xaa9c, 0xba66, 0x2969, 0x35a9, 0x4363, 0xbf69, 0x40e8, 0xa271, 0x41f2, 0x4391, 0x426b, 0x40c5, 0x411c, 0x46ed, 0x1839, 0x4345, 0x4202, 0x4056, 0xb26c, 0x406d, 0x16ec, 0x4276, 0x1249, 0xba23, 0x4301, 0x409a, 0xbfa4, 0x408e, 0xb2d2, 0x0054, 0x4092, 0x421f, 0x4313, 0xb2f2, 0x4115, 0x1c95, 0x1ba7, 0xba42, 0x0497, 0x0d14, 0x1c8d, 0x41ca, 0x400e, 0x436d, 0x17de, 0x1cc0, 0x417c, 0x1e15, 0x1250, 0x4288, 0x0add, 0x2783, 0x40d7, 0xbfce, 0xbfa0, 0x41e4, 0x3706, 0x33b9, 0xba5f, 0xb063, 0xb215, 0x421c, 0x04a2, 0xbf0b, 0xa42c, 0xba4a, 0x425f, 0xb2f4, 0x0891, 0xbf60, 0x1796, 0x43d7, 0x1520, 0xb26d, 0xbf9c, 0xb28d, 0x1b2e, 0x4137, 0x41be, 0xa19f, 0x40eb, 0x4247, 0x2c5d, 0x40fd, 0x41e9, 0xb260, 0x4069, 0x2417, 0x407b, 0xbf62, 0x2e6e, 0x42e9, 0x1f8b, 0xbae8, 0x1d37, 0x3234, 0x4287, 0xba75, 0x4232, 0x245c, 0x1176, 0xb025, 0x3f5c, 0xbfd9, 0xba5c, 0x40ad, 0x43d1, 0x189d, 0xba3c, 0x41c0, 0x4240, 0x41f1, 0x4276, 0x42ed, 0x4242, 0x4398, 0x40de, 0x444b, 0xb295, 0x411c, 0x0bf0, 0x3c69, 0x40d3, 0xbf1d, 0x41e0, 0x320b, 0x43bd, 0x02b9, 0x41da, 0xbaf3, 0x1052, 0xb27b, 0x4116, 0x1de7, 0xb2d3, 0x41b7, 0x1b37, 0x15a2, 0xb074, 0x4246, 0x428b, 0x1dbe, 0x1dd3, 0x1a49, 0xba20, 0x04d8, 0xbfe1, 0xb0fc, 0x400a, 0xbaf7, 0xbae1, 0xb043, 0x2790, 0xb273, 0xabe4, 0x405b, 0xa232, 0xbf33, 0x2d1d, 0x40f3, 0x4129, 0x133f, 0x41a7, 0xa789, 0x4264, 0x292e, 0x4056, 0x2f67, 0x40c9, 0x0a78, 0x41b6, 0x43cc, 0x0601, 0x4177, 0x3fd1, 0x42bf, 0x1e48, 0x4256, 0x426d, 0x1342, 0x4279, 0x4341, 0xab7e, 0x43f5, 0xb293, 0x2783, 0x2ef7, 0xbfdb, 0x43e2, 0x4098, 0x41eb, 0x4115, 0x1b25, 0x1f61, 0x43b5, 0x0cf3, 0x3f68, 0x0ce3, 0x004b, 0xb2de, 0x41e2, 0xb07e, 0x1ed0, 0x2e59, 0xba46, 0xba44, 0x3742, 0x17bf, 0x3e63, 0xbf07, 0xa4a5, 0xba1f, 0x4249, 0x3421, 0x435b, 0x434f, 0xba74, 0x0c5d, 0x4121, 0x3700, 0xb226, 0x3869, 0xb279, 0x46dd, 0x441c, 0x43da, 0x40ab, 0xbf2d, 0x1e81, 0x422c, 0x1c93, 0x4090, 0x424c, 0x4108, 0xb213, 0xa996, 0x3006, 0x438e, 0x41f8, 0x1c12, 0x20d7, 0x41d7, 0x0620, 0x4216, 0x22be, 0xba02, 0xbf60, 0x18b1, 0x431f, 0xbad4, 0x0f22, 0x4607, 0xb257, 0x38ed, 0xbfca, 0xb25c, 0xb245, 0x401b, 0x4009, 0x4303, 0x1615, 0xbf23, 0x42e0, 0x42db, 0x428f, 0x42f0, 0xb205, 0x1d0b, 0x404b, 0x2453, 0xba3d, 0x0eb6, 0x4319, 0xb073, 0x0b49, 0x43e2, 0xba38, 0xbf41, 0x1dc4, 0x1ea7, 0x1ffc, 0x408f, 0x3064, 0x189c, 0xa006, 0xbaef, 0x405e, 0x0d23, 0x2a88, 0x43a7, 0x401c, 0x092a, 0x4023, 0x440f, 0x41aa, 0x407f, 0x353c, 0x1de4, 0x4250, 0x4431, 0x4606, 0x4160, 0x42c3, 0xbf5e, 0x1f23, 0x3650, 0x4652, 0xa61a, 0x44db, 0x446f, 0x418b, 0xba3a, 0xb217, 0x03a3, 0xb03c, 0xb00e, 0x42bf, 0x4582, 0x41af, 0xb22c, 0x1c40, 0x42d8, 0xa1af, 0x42a8, 0x40d6, 0x40c3, 0xb27b, 0xbf88, 0xa181, 0x18bd, 0x1d56, 0xb2c3, 0x3309, 0xbfd9, 0x0e90, 0x1b15, 0xb2cf, 0xbafd, 0x446c, 0x4166, 0x427f, 0x2cd4, 0xbff0, 0xb20b, 0xba7f, 0x4374, 0xbf44, 0xb0fa, 0xb2a7, 0x4595, 0x4240, 0x4405, 0xb253, 0xba4d, 0x1879, 0xb231, 0x2937, 0x2c54, 0x1b7c, 0x0fca, 0xb0be, 0xbf00, 0x428c, 0x2f47, 0x4248, 0x4075, 0x121d, 0x40bc, 0xbad3, 0xba4e, 0x44f8, 0xbf7e, 0x42f5, 0x4319, 0xb2d6, 0xa214, 0x41ca, 0x41b6, 0xb001, 0xbf88, 0xa731, 0x4452, 0x1ee4, 0xb2ab, 0x39e6, 0xb092, 0x4097, 0x407b, 0xba6b, 0x464d, 0x44c1, 0x40d8, 0x100e, 0x4010, 0x46eb, 0xbafb, 0x07e0, 0xb2d3, 0x4332, 0x41ab, 0x4022, 0x46b9, 0x3214, 0xbf8d, 0xba7d, 0x407a, 0x42c2, 0x195d, 0x2988, 0xb23d, 0x4382, 0xbf46, 0xb236, 0x44c2, 0x1f13, 0x0599, 0x1edf, 0x1984, 0xba34, 0x25fe, 0xb202, 0x2128, 0xb227, 0xbf5e, 0x1bd5, 0x4265, 0x454a, 0x1485, 0xa79d, 0x1484, 0x409e, 0x431e, 0x4028, 0x4272, 0xba7d, 0xb2d3, 0xb228, 0x4093, 0x3c65, 0xba5d, 0x3bb9, 0xb2cf, 0xbf76, 0xb053, 0x4272, 0x41fd, 0xb052, 0x3908, 0xb042, 0x407f, 0xb29e, 0x43cd, 0x2c83, 0x4310, 0x4680, 0xb210, 0x0762, 0x4350, 0xbf77, 0xb243, 0x1b98, 0xb0f5, 0x4041, 0xb287, 0x4038, 0xbad5, 0x408d, 0xabd0, 0xba34, 0xbf71, 0xb2cd, 0x4168, 0x1b2e, 0x42e3, 0x1824, 0xb21f, 0x3ab2, 0x025b, 0xb23f, 0x45f3, 0xbaf0, 0x1d87, 0x40e1, 0x4177, 0xb21c, 0x4362, 0x43c5, 0x3708, 0x0554, 0x1d4d, 0xb073, 0xbfe4, 0xac1d, 0x4172, 0x0cf2, 0x4048, 0xbaf8, 0x442b, 0x1a0b, 0xbf67, 0x426e, 0x1c1c, 0x40a9, 0xbf60, 0x434d, 0xbae5, 0x4426, 0xb280, 0x150b, 0xb2ac, 0x4421, 0xbf2d, 0x1ede, 0x42c3, 0x426f, 0xbf90, 0x40aa, 0x4175, 0xb205, 0xb2f0, 0xaf19, 0xbfa7, 0x43b3, 0xba7f, 0xada2, 0x40ff, 0x4263, 0x0093, 0x42a8, 0xb2d3, 0x4114, 0x381b, 0x417b, 0xb2e3, 0xb206, 0xa446, 0xb204, 0x4192, 0x2850, 0xbfc0, 0x4078, 0x0d01, 0xbf53, 0x1bf6, 0x4218, 0x4065, 0x435e, 0x016b, 0x3bcd, 0x4326, 0x4335, 0xadc9, 0x1c41, 0x2724, 0xbad3, 0x4125, 0x42f7, 0xae86, 0xb2ac, 0xaced, 0x42fb, 0x4572, 0xba46, 0x4178, 0xbfc0, 0x0a8a, 0x02c0, 0xa6fb, 0x4117, 0x430b, 0xbf73, 0x1ee6, 0x1b3a, 0x42e0, 0x4279, 0x40d8, 0x45a4, 0xb204, 0x40e3, 0x3114, 0x4274, 0xa303, 0x421b, 0xbac2, 0x42c5, 0x4284, 0xbfca, 0x4142, 0x1e76, 0x41f2, 0xb272, 0x1ec0, 0xb2fa, 0x4494, 0x1ec1, 0x1ec0, 0x414a, 0xba7d, 0x4647, 0x439b, 0xbfa0, 0x40fa, 0xbaef, 0x0dd9, 0xb26c, 0xa054, 0x4032, 0x40aa, 0xba37, 0x456f, 0xbfb3, 0x40a4, 0x4027, 0x12fe, 0x25e9, 0x259e, 0x2bbc, 0x4223, 0xba2c, 0x2553, 0xbf24, 0xa784, 0xb2db, 0xb23c, 0x4362, 0x40c8, 0x24fc, 0xa161, 0x08af, 0xb26a, 0x43cc, 0xb2f7, 0x41ad, 0xbfb6, 0x44a3, 0xbfb0, 0x1c2a, 0xb2cc, 0x456f, 0x040a, 0x400e, 0xb270, 0x288c, 0x4056, 0x29c3, 0x4330, 0x1244, 0x128a, 0xbfa6, 0x1f86, 0x3fb5, 0x40ec, 0x40ab, 0x4078, 0xba5d, 0x43a9, 0xbad0, 0x4052, 0x40e3, 0x428e, 0x4603, 0xbf03, 0x416b, 0x0d03, 0xa3b5, 0x1f7e, 0x3e32, 0xb04e, 0x4085, 0xb2d4, 0xb211, 0x4394, 0x3b55, 0xb2d7, 0xbf3b, 0x1379, 0x41d9, 0x4121, 0xb250, 0x43e4, 0xb231, 0x14c6, 0x4245, 0x40ce, 0xa90a, 0x45f4, 0x43bd, 0xb28c, 0xb058, 0x41ac, 0xb276, 0x322f, 0x42b0, 0xa84f, 0x42f0, 0x43da, 0x299e, 0xbfc1, 0x13ef, 0x4298, 0x4304, 0x1aea, 0xb006, 0x4352, 0x41cc, 0x1c38, 0x2f37, 0x439e, 0x4329, 0x45f6, 0x41d2, 0x4253, 0xb209, 0x1129, 0xb27f, 0xbaec, 0x40fd, 0xbf74, 0xa497, 0x372a, 0xbaea, 0xb06a, 0xb24c, 0xba25, 0x45d2, 0x1638, 0xb0b8, 0xba4e, 0x406b, 0xbaf5, 0xbf35, 0x4340, 0x1ffe, 0x4075, 0x1ebb, 0x4029, 0xba36, 0x3d21, 0x0bf9, 0x1dd1, 0x43f4, 0x4316, 0xb241, 0xb212, 0xb2d7, 0x42cc, 0x4379, 0x1e94, 0xb03b, 0x3bc7, 0xbfd0, 0xba35, 0xa8cd, 0x009f, 0x1f3f, 0x436e, 0xbaf2, 0x4369, 0x1f29, 0x4164, 0xbf39, 0x433f, 0x403d, 0x316c, 0x4252, 0x4204, 0xb2b4, 0xba22, 0x19f7, 0x4120, 0x412c, 0x05e0, 0xb2c5, 0x3aa5, 0xbf67, 0x46a0, 0x4076, 0xb2aa, 0x436f, 0x43a9, 0x4183, 0x1c21, 0x12f9, 0x408a, 0x41bb, 0x42a6, 0x220c, 0xb2cf, 0x3de8, 0xbfb5, 0xbf00, 0x2586, 0xb2ce, 0x40ba, 0xa486, 0xa26e, 0x1b02, 0xb063, 0x40d3, 0x07ab, 0x26fd, 0xba10, 0x43ac, 0xbf24, 0x3285, 0x3eb9, 0x1f75, 0xb202, 0x2771, 0xacd9, 0xba3d, 0x4326, 0xbf60, 0xb263, 0x4013, 0x3cb8, 0xbaef, 0x4074, 0x42f5, 0x43f8, 0x40d9, 0x20ec, 0x42f6, 0xbf2b, 0x4336, 0x4321, 0xb06c, 0x2f28, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x45444af8, 0x3713ac06, 0xc0dfa961, 0xa4318add, 0x64c2ab43, 0xd5a78f68, 0x13e65e07, 0x035cff70, 0xf5623146, 0xb8ad178f, 0xade3820c, 0x0224fa96, 0x9850b729, 0xe11a6537, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x000000ec, 0x00000000, 0xffffffff, 0x0000006b, 0x0000014c, 0x71000000, 0xe1eca9ff, 0x00000000, 0x00003ff7, 0x00000000, 0xade3820c, 0xe1ec9c83, 0x9850b74d, 0xe1eca7b7, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xa520, 0xb223, 0x4254, 0x04d2, 0xb2bb, 0x0b78, 0x4241, 0x3e94, 0x4367, 0xbf06, 0x40db, 0x409c, 0xbfb0, 0x40bd, 0xb297, 0x44fa, 0xb2b3, 0x406c, 0x4348, 0xb2ac, 0x4158, 0x40b0, 0x2122, 0x3cd3, 0x433b, 0xbf01, 0x42bb, 0xb2b2, 0x3a8d, 0x43f1, 0x4200, 0x1c8e, 0xada7, 0x40f6, 0x403d, 0x40db, 0x422b, 0xbad1, 0xa3da, 0x4319, 0xa9a2, 0xb21f, 0x4231, 0x41b5, 0x4104, 0x043a, 0xba2f, 0xbfb6, 0x41b2, 0x05c3, 0xb0be, 0x1de0, 0xa730, 0xba13, 0x05fe, 0x41ea, 0xbf4a, 0xa3f1, 0x182e, 0xb217, 0xad55, 0x44d9, 0xb0c0, 0x4389, 0x401f, 0xba25, 0x18ab, 0x02e1, 0x46dd, 0xba26, 0xba21, 0xb2bc, 0x42a2, 0x124d, 0xbfd0, 0xb271, 0x01b0, 0x42e0, 0x4011, 0xa094, 0xb02f, 0x43dd, 0xbf74, 0x41b0, 0x41bc, 0x4133, 0x105d, 0xb28f, 0xb0d0, 0xaf65, 0xaf83, 0xbf71, 0x4196, 0xb06b, 0x4122, 0x400d, 0x41ac, 0x1863, 0xb2e0, 0xb218, 0x41b8, 0x4588, 0xb2a0, 0xbfdb, 0x436c, 0x0f45, 0x0994, 0x43ab, 0x4106, 0x2c4b, 0x01db, 0xb048, 0x4495, 0x42a5, 0x40ae, 0x419c, 0x41ca, 0x40be, 0xa8ed, 0x3fe9, 0x42ea, 0x4127, 0x425f, 0x4499, 0x1c46, 0xb258, 0x3f68, 0xa832, 0xbf0c, 0x42cf, 0xa86c, 0x42ce, 0x4113, 0xb24f, 0xb0ec, 0xb0e0, 0x3d7f, 0xa273, 0x40cd, 0xb03f, 0x41ef, 0x4266, 0x413e, 0xb20c, 0x43b9, 0xb019, 0x4097, 0x05f4, 0x3445, 0x408b, 0x409a, 0x2673, 0xbf4f, 0x417b, 0xbaee, 0x443c, 0xbae3, 0x426c, 0xbf2d, 0x4185, 0x4366, 0x431c, 0x3f62, 0x42bd, 0x46c5, 0x416f, 0xb205, 0x4011, 0x07d0, 0x06d4, 0x413a, 0xbf5b, 0xb2b5, 0x1ae2, 0xa9a9, 0xba0e, 0x401c, 0xb22e, 0x44f0, 0x17c2, 0xb036, 0x1cf0, 0x4096, 0x4299, 0xba11, 0xa27d, 0x1e0a, 0xbf7f, 0xb285, 0x4342, 0x24a7, 0x1a9e, 0x2cbc, 0x4218, 0x0e6d, 0xb21d, 0x437c, 0x4392, 0x41e9, 0xb05d, 0x4101, 0xa9d4, 0x4371, 0x41ab, 0x1bc2, 0x4241, 0x4002, 0xb2d4, 0xb016, 0xbac0, 0xb201, 0x4398, 0x40a2, 0xbf49, 0x430c, 0x1c44, 0x3eed, 0xab02, 0xb20f, 0xbf3f, 0x17a7, 0x418f, 0x4151, 0xba01, 0xbf9b, 0x41a4, 0x0136, 0xa343, 0xb29b, 0x2019, 0xab1a, 0x43d2, 0xbf5e, 0x42b3, 0x297f, 0xb2ce, 0x2057, 0x1d80, 0x1633, 0x4310, 0x337b, 0x40bc, 0x4077, 0xb224, 0xb298, 0x2a78, 0x4233, 0x0d0e, 0x32a6, 0x42cb, 0x4396, 0x40a2, 0xb00f, 0x4050, 0xb271, 0x40ee, 0x410a, 0xbfc8, 0xb230, 0x1a49, 0x4360, 0xb03d, 0x0d57, 0x4276, 0x406d, 0xb223, 0x40b9, 0x40b7, 0x1e40, 0xb257, 0x07e4, 0xa579, 0xb217, 0xbf7c, 0xbae9, 0x4333, 0xbae6, 0xb23c, 0xbae4, 0x423d, 0x4121, 0xbacc, 0x41eb, 0xaaae, 0x1c89, 0x1960, 0x42af, 0x2b33, 0x4287, 0x4000, 0xb27b, 0x436d, 0x458d, 0x42aa, 0x43fd, 0x423a, 0x43fd, 0xbfc3, 0x1e10, 0x4317, 0xb272, 0xbaec, 0x4152, 0x4173, 0x30c9, 0x404e, 0x2173, 0x46c8, 0xb260, 0x4084, 0xb234, 0x4217, 0x1fbc, 0xbf13, 0x46b1, 0x41fd, 0xa309, 0x1f35, 0x418e, 0xb21a, 0x43a6, 0x41f1, 0x4219, 0x45d4, 0x45c6, 0x1ac2, 0x1c3e, 0xbad6, 0xbfc0, 0xba74, 0x4342, 0x41e9, 0x43c2, 0x4051, 0xbf88, 0xb2ce, 0xbfb0, 0xb03b, 0x248a, 0x42c8, 0xa7f9, 0xb2dc, 0x4284, 0xb27d, 0x4207, 0x43d4, 0x3a09, 0x43b6, 0xbfd2, 0x4636, 0x431f, 0x359a, 0x4035, 0xa58b, 0xb293, 0xba72, 0x40b4, 0x42fa, 0x08f7, 0x415c, 0x16ac, 0x438c, 0x42d5, 0x4069, 0x18ce, 0xa2a3, 0x4259, 0x245a, 0x2c8c, 0x4104, 0xbf52, 0xa8d4, 0x1a52, 0xae71, 0xb0dc, 0xa6c6, 0xb240, 0x33b4, 0x3314, 0x4183, 0x183d, 0x1fe5, 0xba4f, 0x1b3b, 0x41b3, 0x3897, 0xb209, 0x4380, 0x417f, 0xbaef, 0xb2e6, 0x42a0, 0x4177, 0xbfb2, 0x1e3c, 0x4250, 0xba34, 0x402c, 0x2e43, 0x435b, 0xbf8f, 0x3463, 0x40a0, 0x4174, 0x44dd, 0x40cf, 0x400b, 0x2fca, 0x3ccd, 0x2e2e, 0x3764, 0xa626, 0x40bc, 0xa3ba, 0x44ca, 0xbfa7, 0x43ed, 0xb278, 0x412a, 0xba1a, 0x01d4, 0x185c, 0x4365, 0x389a, 0x37b6, 0x4111, 0x40f7, 0x4114, 0x06a0, 0x4167, 0x3736, 0x42dc, 0x40c3, 0x32cb, 0x41b0, 0x412b, 0x2c08, 0xbacd, 0x1806, 0xb247, 0x3a1f, 0xbf70, 0x4137, 0xbf9e, 0x42a3, 0x3af2, 0x424f, 0x2f4b, 0xba6c, 0x3b5b, 0xbff0, 0xb2bc, 0x414b, 0x4570, 0xb216, 0xb2a1, 0xbad1, 0x40ae, 0x40d0, 0x41ca, 0x4059, 0xacfd, 0x43ae, 0x43de, 0x428c, 0x19ac, 0x2ed4, 0xba26, 0xbfc5, 0x1e66, 0x403a, 0x41a7, 0x3900, 0x432b, 0x4386, 0x3670, 0x1674, 0x0a49, 0x41b9, 0x438a, 0x42b9, 0x41aa, 0x40a8, 0x43c9, 0xba1c, 0x4167, 0xb0ed, 0x4359, 0xa917, 0xbf59, 0xb040, 0x4205, 0x43eb, 0x1811, 0x1e8e, 0x41cd, 0x41f2, 0x4190, 0x41b0, 0x368c, 0x387e, 0x1545, 0x18d3, 0x40d8, 0x4279, 0x438b, 0x0c33, 0xbf28, 0x4218, 0x41c7, 0xa71d, 0x4373, 0x229d, 0x4605, 0x2a5d, 0xb265, 0xbf15, 0x1c05, 0xb006, 0x193f, 0xba38, 0x4338, 0xbf4e, 0x4177, 0x4138, 0x22ae, 0xb268, 0x40e2, 0xbfa0, 0x42a9, 0x4612, 0x267c, 0x41c9, 0xb291, 0xbad0, 0xba3a, 0x45c2, 0xbac5, 0x43df, 0xb00c, 0x43e4, 0x1faf, 0x1c18, 0x1f38, 0x42f0, 0x1ac4, 0xbf2d, 0x4158, 0x40dc, 0x4329, 0xba24, 0x4009, 0xbace, 0x4206, 0x0e36, 0x3b42, 0x023c, 0x0ab2, 0x42f3, 0xb2d2, 0x4165, 0x236b, 0x4125, 0xb239, 0x45db, 0xbf5e, 0x4198, 0x444c, 0x1601, 0x19a8, 0xbaf6, 0xb251, 0x4035, 0x4169, 0x4265, 0x406f, 0x403c, 0x4319, 0xb29f, 0x4052, 0xb273, 0x1bfe, 0x4588, 0x40fe, 0x41cc, 0x40bb, 0x46f3, 0x42ff, 0xa67e, 0x1fff, 0x10c0, 0xbf33, 0x4286, 0x412d, 0xb26d, 0x41bd, 0x436e, 0x439d, 0x417e, 0x0746, 0x42c6, 0x04ce, 0x406d, 0x4249, 0xb22a, 0x43ff, 0x27d4, 0xb2a7, 0xbf95, 0xb287, 0x366b, 0x1cc0, 0x1802, 0xba5c, 0x43e7, 0x4061, 0x4153, 0x150c, 0x4681, 0x44a1, 0x45cd, 0xbad6, 0xbfc9, 0x42f2, 0x1bfc, 0x4180, 0x25a3, 0x42d3, 0x409b, 0xb0e1, 0x4676, 0xbfa1, 0x436a, 0x42b6, 0xba61, 0x424c, 0x4354, 0x340e, 0x43b7, 0x0ce6, 0x45bd, 0xa0d6, 0x42e8, 0xba77, 0x4037, 0x42b2, 0x446d, 0xbae6, 0x439f, 0xb0a1, 0xba6e, 0x4231, 0xa05f, 0x46ca, 0xb081, 0xba3e, 0x1d40, 0x435a, 0x196f, 0x4168, 0x405e, 0xbf9f, 0x4056, 0x30ab, 0x43be, 0x1956, 0xb01d, 0x41d5, 0xba3b, 0xbf48, 0x08f2, 0x42d2, 0x42a7, 0x4291, 0x423d, 0x405e, 0xb2a9, 0x2cb9, 0x43bd, 0x4270, 0xb2d2, 0x420d, 0x1494, 0x4098, 0xa21a, 0x43d1, 0x442e, 0x4559, 0x3192, 0x43d2, 0xbad2, 0x4379, 0x4298, 0xb2d6, 0xbafd, 0xbfc5, 0xbaca, 0xbaf2, 0x1ba9, 0x422c, 0xb2b0, 0x096c, 0x0443, 0xbafe, 0x44e4, 0xba0a, 0x1d8f, 0x15f2, 0xba7b, 0xbad0, 0xb295, 0x41f3, 0xb212, 0xb0dc, 0x24d5, 0x40c8, 0xba3a, 0x4023, 0xb2cd, 0xa317, 0xa11d, 0x4374, 0x1e3e, 0x414d, 0x401b, 0xbf85, 0x418b, 0xae75, 0xb226, 0x37f7, 0x4392, 0x432b, 0x426c, 0x4312, 0x0f46, 0xbfd0, 0x406c, 0xa729, 0x4136, 0x0f4f, 0x40dd, 0x1d25, 0x1a81, 0x41e9, 0x36ab, 0x4694, 0x4558, 0x1868, 0xb220, 0xbfa3, 0x1ecd, 0x4274, 0x1b66, 0x2988, 0x41e0, 0x08f1, 0x4214, 0xb28a, 0xb243, 0x4071, 0x2b56, 0x4038, 0x438f, 0x2441, 0x41bd, 0x41ba, 0x31cf, 0x408b, 0xb036, 0x1db4, 0x19c7, 0xb2c0, 0xbf55, 0x0c41, 0xbaf0, 0xb2ae, 0x4007, 0x1b06, 0xb08f, 0x1d3d, 0x430e, 0x2937, 0x41a0, 0x4209, 0x4448, 0x402a, 0xba03, 0x3597, 0xb20b, 0x45b3, 0x4376, 0x4663, 0x437c, 0x417e, 0xb088, 0xbf9c, 0x14f2, 0xb29a, 0x2554, 0x18c1, 0x180a, 0x417f, 0x0c42, 0xba14, 0xb05c, 0x41ba, 0xbf5e, 0x2562, 0x4243, 0x420f, 0x0a72, 0x438c, 0xb0ab, 0x4210, 0x1dca, 0xb2d9, 0x434a, 0xbf60, 0xad83, 0xbf35, 0xbadb, 0x407a, 0xb2a5, 0xa130, 0x429e, 0xa4e3, 0x142b, 0x35b8, 0xbfc0, 0x40a6, 0xbaf6, 0x3089, 0x4280, 0xba19, 0x43a6, 0x0fef, 0x442e, 0x1cf6, 0xb2ac, 0x18b2, 0xb0c3, 0xbad1, 0x42e0, 0x391a, 0x435c, 0xbf87, 0xb272, 0x05bc, 0x4226, 0xb2f2, 0x4313, 0x459e, 0x4326, 0x4495, 0x4236, 0x1239, 0xbf55, 0x43f8, 0xb289, 0xbf70, 0xb249, 0xb2cd, 0xb224, 0x4017, 0x1d5d, 0x4026, 0x423e, 0x1214, 0x431a, 0x351a, 0x40d3, 0xbf88, 0xba4e, 0x434f, 0x3815, 0x15ac, 0x1693, 0x40ad, 0xbf90, 0x43ac, 0x42c3, 0x060d, 0x4013, 0x3896, 0x428c, 0x2d3e, 0x435b, 0xaeac, 0x42b7, 0xbfbc, 0x42c5, 0x31b5, 0x1b10, 0x0f85, 0xbaf0, 0xbfa5, 0x09a7, 0x0762, 0xb2cb, 0xab8a, 0xba7c, 0x3c28, 0x40a2, 0x173c, 0xa6f8, 0x0788, 0x4623, 0x4324, 0x4310, 0xb208, 0x3297, 0x4667, 0x41c1, 0xbfc7, 0xbf00, 0xb222, 0x0499, 0x422f, 0x46f8, 0xa72c, 0x2e9e, 0xa41b, 0x4370, 0xbf08, 0xb073, 0xbfc0, 0x04bb, 0x2e56, 0xba6a, 0xba56, 0x4206, 0x4306, 0x427b, 0xb060, 0x4260, 0x4298, 0x4360, 0x434f, 0xbf70, 0xa2d2, 0xbae6, 0xbf80, 0xac39, 0x441b, 0xb2e7, 0x40d7, 0x42a2, 0x42b5, 0x08fe, 0xbf05, 0x195f, 0x414a, 0x40a5, 0x4344, 0xbafe, 0x4212, 0xb225, 0x429d, 0x428b, 0x22bc, 0x43e8, 0xb27d, 0xb2ac, 0x4108, 0xb2d2, 0x1d5f, 0x23c3, 0x4257, 0xb060, 0x4105, 0xb208, 0x441f, 0x2f98, 0x1f2b, 0xbf17, 0x43d2, 0x413e, 0xba24, 0x40ca, 0x4101, 0xbf90, 0x43dc, 0x42b6, 0x431d, 0x42db, 0x43e6, 0xac1f, 0x42c9, 0xba0b, 0x46a4, 0xba41, 0x42f3, 0x45a2, 0xbf2e, 0xb088, 0x40fd, 0x0d9d, 0x1a17, 0x2ee0, 0x1b8c, 0x41cc, 0x42b4, 0xb294, 0x436d, 0xabb1, 0xa7ed, 0xba2c, 0x4002, 0x4214, 0x417c, 0x03a3, 0x1b3a, 0x4645, 0xa99a, 0x4661, 0x41b7, 0xbacd, 0x2cb1, 0x1dca, 0x1d24, 0x42c5, 0xba4c, 0xbf9c, 0x1e2b, 0xb24b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6858fdc0, 0x2c117adb, 0x4e91c10b, 0x490c37fa, 0xdc3cc5ba, 0x19d843cc, 0x38054938, 0xa1f0a710, 0x540c538f, 0x8e7e0cd6, 0x8458328c, 0x917b025c, 0x75c492d1, 0x64488841, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0xffffa800, 0x540c5724, 0x540c572b, 0x00000024, 0x0c542457, 0x00002457, 0xfffffffc, 0x00001b93, 0x00001730, 0xffffff42, 0xffffff42, 0x00000000, 0x540c5724, 0x540c5688, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xab86, 0x436f, 0xb06c, 0x401b, 0x4278, 0x4283, 0x4243, 0x1d91, 0x427d, 0x312e, 0xad98, 0x4070, 0x4267, 0x4481, 0x438f, 0xba4e, 0xb280, 0xb2d0, 0x4139, 0x40e3, 0x42f9, 0xbf7f, 0x242c, 0x43f9, 0x40cf, 0xb0cc, 0x410e, 0xb250, 0xb083, 0x2759, 0x434c, 0xbad5, 0x0875, 0xb036, 0x426c, 0x3721, 0x412f, 0xbf4c, 0x1bd7, 0x4137, 0xb04a, 0xb2d2, 0xb017, 0xbae5, 0xb258, 0x41ec, 0x1aa9, 0x1e1b, 0x42d2, 0x42a6, 0xb2a0, 0xbf36, 0xbae1, 0x426e, 0x4011, 0x41fc, 0x3afd, 0x4289, 0x355c, 0xb24f, 0x0e2c, 0x4391, 0xb2b5, 0x4059, 0x4030, 0xb299, 0xb281, 0xb04f, 0xb0d6, 0xbf33, 0x0a54, 0xbfe0, 0x41a6, 0x1b05, 0xbae0, 0xac71, 0xbf00, 0xb042, 0x18f7, 0x4576, 0x4056, 0x0b0d, 0xb224, 0xbf62, 0xba1e, 0x4629, 0xad90, 0x44e5, 0x1a37, 0x41c5, 0xbae3, 0xba68, 0x4436, 0x45b4, 0x4203, 0xba1c, 0x026e, 0x4649, 0x42c6, 0x4166, 0xba1d, 0x1d0b, 0x4006, 0x12f0, 0x0f55, 0x425a, 0xbfe1, 0xb264, 0x4672, 0xb031, 0xbac4, 0x2a1f, 0x42ec, 0x4147, 0x437f, 0x4165, 0x407e, 0x1e5e, 0x429e, 0xb03d, 0x262b, 0x4181, 0xa9bc, 0x1b17, 0xbf60, 0x1a3b, 0xa1c0, 0x18d3, 0xb03d, 0xbf25, 0xbad5, 0x421b, 0xbaeb, 0x18d4, 0x3660, 0x42ad, 0xbad0, 0xbfc0, 0x42b4, 0x0065, 0xba05, 0xb242, 0x41d4, 0x43f1, 0x427a, 0xbaf9, 0x4594, 0x41ba, 0xb004, 0xbf11, 0x3700, 0xb272, 0x4242, 0x1bab, 0xa2be, 0x43af, 0x418a, 0xb2c7, 0x0201, 0xb2c8, 0x425c, 0x405e, 0x4119, 0xb20b, 0x22a8, 0xb0bf, 0x1fd3, 0x0867, 0x4324, 0xbf89, 0xb01e, 0xb0a3, 0x3674, 0x29de, 0x4349, 0xb070, 0x00b9, 0xb27a, 0x4396, 0x45f2, 0x280b, 0x4288, 0xbf9e, 0x42fb, 0x0703, 0x40e5, 0x44e5, 0x1bd1, 0x46a2, 0xb2ca, 0x18e5, 0x010b, 0x16dc, 0x4389, 0x440c, 0x4134, 0x0684, 0x3025, 0x400b, 0x4272, 0xba24, 0x2c2f, 0x43c7, 0x2148, 0x151a, 0x433b, 0x41a2, 0xa6d0, 0xbf57, 0x085a, 0xb2c6, 0x44ba, 0x40a1, 0x43cd, 0xb2bc, 0xba3e, 0xb2d1, 0xba61, 0x0c3e, 0x1cfa, 0xbf84, 0x1b40, 0x372a, 0x3b42, 0xb06c, 0x43b7, 0x1d1f, 0x41fc, 0x41f6, 0x4552, 0xb29d, 0x43dc, 0x1a4e, 0x2de6, 0x4204, 0x407e, 0x16a1, 0xb276, 0x40ba, 0xb223, 0x4377, 0x4073, 0xb007, 0xbfdc, 0x41e3, 0x3889, 0x4117, 0x43bb, 0xba05, 0xb2ac, 0x274e, 0xb0fe, 0x01ee, 0x0c90, 0x46ab, 0x1775, 0x426c, 0xbaf1, 0x1030, 0x2822, 0xbad0, 0x089f, 0x4036, 0xbaf4, 0x1e1b, 0xba3b, 0x428e, 0x420c, 0xbf4e, 0x4228, 0x1bf3, 0x40dd, 0x40c8, 0x4010, 0xa54e, 0x4646, 0xba62, 0xbf84, 0x1d90, 0xb25f, 0xbf0f, 0xbacd, 0x410b, 0xb25c, 0x43ec, 0x25fc, 0xa882, 0x29ac, 0x4339, 0x2a4f, 0x431a, 0x032a, 0x4384, 0x3a9f, 0xb2e9, 0x401d, 0x1d5e, 0xbf64, 0x1da6, 0x062e, 0x4150, 0x0935, 0xa143, 0x2840, 0xbac7, 0xbac8, 0x44dc, 0x1af0, 0x434a, 0xba37, 0x432e, 0x4223, 0xbf00, 0x40be, 0x41ac, 0xb2d9, 0x4373, 0x19dc, 0x402c, 0x3f03, 0x43f9, 0xbf44, 0x4057, 0x07ef, 0xb2e1, 0x4308, 0xba47, 0x4033, 0x4069, 0x1590, 0xb2d7, 0x2ff9, 0xb281, 0x07a2, 0x0caa, 0x4022, 0x42fa, 0x425b, 0xba05, 0x2f80, 0xbadf, 0xa0c2, 0xbf14, 0x41ca, 0xbafb, 0x3b0a, 0x084d, 0x41c2, 0x43d6, 0x016f, 0xba73, 0x40ff, 0x1e2f, 0x4634, 0xa32a, 0x4212, 0x4234, 0x193e, 0xb036, 0xb2d3, 0xb2b5, 0x40f7, 0x40a5, 0x43f4, 0xbf84, 0x4322, 0x431a, 0x40e3, 0x3a14, 0xb2ef, 0xba6a, 0x439a, 0x43e2, 0x40db, 0xba35, 0x279e, 0x41f9, 0xb053, 0x400d, 0x4442, 0x45d4, 0xa582, 0xbf6a, 0x4249, 0x05f6, 0x439d, 0xb29f, 0x41d6, 0xa1a1, 0x1849, 0xbf8e, 0x4215, 0x130c, 0x43c9, 0xbfe0, 0xb2ff, 0x1d87, 0x4378, 0xb062, 0x42de, 0x4335, 0xb0b6, 0x416d, 0x4241, 0x08b9, 0x04fb, 0xba2d, 0x44f2, 0x0af9, 0xba57, 0x17c9, 0x4188, 0x1f59, 0x02cc, 0x42fb, 0xbf8f, 0xb289, 0x4148, 0x0ee2, 0x43ef, 0xba1c, 0x40df, 0x456e, 0xb235, 0x1d34, 0x43b0, 0x44ed, 0x4010, 0x43a1, 0x4096, 0x1a2d, 0xbf9d, 0xb01c, 0x417a, 0x429d, 0xb226, 0x19eb, 0xb20e, 0x434a, 0x46ba, 0x4374, 0xbf48, 0x41ea, 0x23aa, 0x41f7, 0xaef1, 0x4331, 0x41c3, 0x1dff, 0x349b, 0x220e, 0x43ec, 0xaf93, 0xbf7d, 0xba45, 0x33db, 0x41e2, 0x465f, 0x42d7, 0xb268, 0xba51, 0x42bf, 0xba33, 0xb03c, 0x42b2, 0x101c, 0x422b, 0xbf05, 0xb2a6, 0x348a, 0xba46, 0x1691, 0xb055, 0xbf60, 0xb058, 0x40f3, 0x404b, 0x3a1b, 0x44b1, 0x18d2, 0x165e, 0x439a, 0x401c, 0x1860, 0x44ca, 0xba4f, 0xbf7a, 0x1f7f, 0x410a, 0x44e0, 0xa367, 0xa9b9, 0x43e5, 0x41a3, 0x112b, 0xba5f, 0xbf01, 0xbf80, 0x4288, 0x4613, 0x1f10, 0x2223, 0xb2ab, 0xb231, 0xba03, 0x415d, 0x38a5, 0xba64, 0x4287, 0x42c7, 0x3c02, 0x1b36, 0x444e, 0xbf0c, 0x42ec, 0x419b, 0xb08b, 0x0e22, 0xbf58, 0x45a0, 0x3d48, 0xa1d0, 0xba09, 0x2ae7, 0x1240, 0xa3da, 0x430b, 0x41d5, 0xb267, 0x4173, 0xab95, 0x4323, 0xb24f, 0x436b, 0x12fc, 0xbf06, 0xbac0, 0x184d, 0xb2c8, 0x1e2e, 0x1b5c, 0x1f35, 0x414f, 0xb2f4, 0x41c8, 0x1bf7, 0x1c02, 0x4179, 0x054d, 0xa9e5, 0x19de, 0x40d4, 0xba00, 0xbae8, 0x406d, 0x4090, 0x16e9, 0xba19, 0xbf17, 0x4135, 0x2fdd, 0xa4f3, 0x4007, 0x43be, 0x4103, 0x46a4, 0xb201, 0x4061, 0xba27, 0xb056, 0x1c97, 0xb294, 0xba6c, 0x1ed4, 0xbacf, 0x35d1, 0x3094, 0x1c4d, 0xba09, 0x28a4, 0xb0a4, 0x2f5f, 0x4263, 0xb03c, 0x0667, 0x31f0, 0xbfc7, 0x259c, 0x41a9, 0x3f6d, 0x4345, 0x4240, 0x1b3e, 0x40d3, 0xba05, 0x4066, 0xbad7, 0xb0e7, 0x1f93, 0xbff0, 0x1c32, 0x41c5, 0x36e4, 0xb086, 0x3d60, 0x1927, 0x4339, 0x41fc, 0xb27d, 0x4253, 0xa158, 0xba4f, 0x0102, 0xbf2a, 0x353f, 0x1eb5, 0x400c, 0xab8b, 0x402c, 0x44fc, 0x384f, 0xb253, 0xacde, 0xaec7, 0x446b, 0xb2dc, 0x413e, 0x1d6c, 0x4206, 0xb298, 0x4297, 0xb2cf, 0x4008, 0x41b4, 0x2866, 0x4115, 0x4369, 0x46e5, 0xbf70, 0x4006, 0xa904, 0x40c4, 0xbf56, 0xa695, 0x42a2, 0x40c0, 0x4152, 0x409a, 0x411b, 0x408e, 0xbf80, 0xb074, 0x43be, 0x072c, 0x40a7, 0x41e3, 0xb0d3, 0x455e, 0xb2a5, 0x1be1, 0x4178, 0x405a, 0x4066, 0xb242, 0x42da, 0xb205, 0xba4a, 0xbfde, 0xba32, 0x45ce, 0x1dc4, 0x42be, 0x4257, 0xb22d, 0xaad1, 0x41f9, 0xb298, 0x4033, 0xb26c, 0xb08a, 0x4210, 0x4094, 0x40d3, 0x4105, 0x41be, 0x02d9, 0x405f, 0xba3a, 0xbac2, 0xbfde, 0x436d, 0x41f2, 0x42cb, 0x4225, 0x40d2, 0x44e5, 0x42a0, 0x41f1, 0x41b7, 0x1e7e, 0x4385, 0x114c, 0x1e14, 0xbf26, 0x37d0, 0xb00b, 0x2124, 0xb276, 0xbac4, 0x41ad, 0x4077, 0x44d0, 0xba27, 0xbf13, 0x2327, 0x21a0, 0x22e5, 0xb069, 0xba4a, 0xb22a, 0xad1e, 0x4207, 0x2a4e, 0x1a9e, 0xb0a8, 0x4095, 0xb0b5, 0x419e, 0xb2fd, 0x422c, 0x33c7, 0xbfb1, 0x4050, 0xb2ed, 0x29d5, 0x191a, 0x1c19, 0xbfbe, 0x1823, 0x447c, 0x41ee, 0x0ba8, 0x46d3, 0xbfb0, 0xaf6e, 0xbfa3, 0x42cb, 0xb043, 0xaf56, 0x4122, 0x405e, 0x1e9e, 0xb2f6, 0x4171, 0x18c9, 0xba2d, 0x0c08, 0x4200, 0x402a, 0x401b, 0xadfe, 0x4365, 0x1fd0, 0x411b, 0xb03b, 0x43d2, 0x42d3, 0xba5c, 0xbfcf, 0x420b, 0x253c, 0xaf62, 0xbafc, 0x4223, 0x4484, 0x41bd, 0x424a, 0xba4b, 0x41be, 0x2525, 0x4262, 0x432b, 0x3897, 0x29b2, 0xbf42, 0xa188, 0x4004, 0x449b, 0x46e5, 0x42ec, 0x45be, 0x3f6a, 0xbf60, 0x436d, 0x412f, 0xbf60, 0x4284, 0xb227, 0xba46, 0x4329, 0xa61f, 0xb21d, 0xbfb7, 0x43aa, 0x1d94, 0xafd1, 0x4102, 0xa3bc, 0xb240, 0x42d0, 0x1b46, 0x4140, 0x1d73, 0x19e9, 0xb0b2, 0x1ac2, 0xb2be, 0xbf57, 0x42e1, 0x455c, 0xb27d, 0x42fa, 0x3cb0, 0xb218, 0x41fc, 0x3d3a, 0x2b79, 0xb2c3, 0x1853, 0xbfa0, 0x4556, 0x4035, 0x1d83, 0x33c4, 0x131e, 0xbad3, 0x428c, 0x41ff, 0x456a, 0xbf56, 0x41f6, 0x4115, 0x4654, 0xa539, 0x43ae, 0x41d6, 0xba1c, 0x3b02, 0x2859, 0xba41, 0xa4f5, 0x43de, 0x41d6, 0xb251, 0x401d, 0xbfe4, 0x42dc, 0xa953, 0x4129, 0x406e, 0x09c1, 0x3105, 0x2ce2, 0xa65c, 0x42e6, 0x42a8, 0x34ac, 0x4270, 0x42ef, 0x3dc8, 0xb2dd, 0x189c, 0xb2bd, 0x1d7d, 0x44d8, 0xb224, 0xbf6a, 0xba40, 0x4194, 0x0198, 0x436f, 0x423c, 0x429f, 0x4077, 0xbad1, 0x1ac3, 0xb0be, 0x2bdf, 0xb051, 0x435d, 0x1573, 0x1d5f, 0x0b67, 0xb0e6, 0x430f, 0x4025, 0xbad2, 0xbfa7, 0x1896, 0x4386, 0x1e61, 0xb250, 0x2733, 0x13f6, 0xba6c, 0x4246, 0x1d98, 0x41f0, 0x4362, 0xb213, 0x41f1, 0xb074, 0xb050, 0x4139, 0xbac9, 0x420e, 0x416e, 0x0a81, 0xbf61, 0xba0f, 0x41c5, 0x40f4, 0xb2e3, 0x2f84, 0xb278, 0xbf67, 0x46c5, 0xba2e, 0x40f1, 0x1036, 0xbff0, 0x3a3b, 0xbaee, 0x3276, 0xb093, 0x1b72, 0x400d, 0x3cda, 0xbf24, 0x197a, 0x42e5, 0x4623, 0xbfbb, 0x1cc0, 0x42cd, 0x41dd, 0x2127, 0x1817, 0x404e, 0x4207, 0x4331, 0x1bf1, 0xbfd7, 0xa8e1, 0x46eb, 0x4372, 0x406b, 0x4670, 0xbf70, 0x401c, 0x403d, 0x4087, 0x4039, 0x43ce, 0x266c, 0x42c2, 0x1b06, 0x2d2d, 0xb0ad, 0x4613, 0x4029, 0x1cbb, 0x1878, 0x4294, 0x4154, 0xb253, 0xbf9b, 0xba0f, 0x4400, 0x422a, 0x34f1, 0xbfd3, 0xb20e, 0x41bd, 0x1c19, 0x418d, 0x3be1, 0xb056, 0x43cc, 0x40e0, 0x133b, 0x43fa, 0x1a07, 0x4086, 0xb26c, 0xbf46, 0x1caf, 0xbae1, 0xb0bf, 0xb2cc, 0xb024, 0x4345, 0x4474, 0xb031, 0x4553, 0x1c60, 0x2731, 0xb2be, 0x297d, 0x4367, 0x40ff, 0x401c, 0xba33, 0x2258, 0xbfce, 0xb280, 0x40c7, 0x0b15, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x38d6f90d, 0xcd984b26, 0xc8b29dc9, 0x67c34614, 0x14951b86, 0xae2631e0, 0x418035e3, 0x147814c2, 0xb4096d8b, 0xc30ae4ae, 0xb5c6061f, 0x1b3e3fa0, 0x0b9ebe6d, 0x746bc3e8, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x00000100, 0xffffffff, 0x00000058, 0x31000000, 0x00000000, 0x00000000, 0x00000031, 0x00000000, 0x00fc342b, 0x2eb36651, 0x26796350, 0x00002db3, 0x00002d03, 0x00002fab, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1971, 0x439c, 0x401d, 0x1924, 0x1fc0, 0x3352, 0xba14, 0x4306, 0x432b, 0xb268, 0x1c6f, 0x4158, 0x4633, 0x37a0, 0xbfbe, 0x4286, 0x410c, 0x41c5, 0xb209, 0xb2f5, 0xa3dd, 0x40f7, 0xb087, 0xae09, 0xb2de, 0xba4c, 0x4349, 0xa6c5, 0x436c, 0xb05c, 0x4575, 0x2190, 0xbf3c, 0x1ed6, 0xa383, 0x4192, 0xbf60, 0xba2a, 0x36db, 0x4116, 0x2df4, 0x41a7, 0x299f, 0x406c, 0xa653, 0xbaca, 0xb282, 0xbac8, 0x40f7, 0x4669, 0xb274, 0xb022, 0xb2eb, 0x0ccf, 0x41b7, 0x20ed, 0x047a, 0xb083, 0xbf33, 0xb2bf, 0xa78d, 0x3ae6, 0x3882, 0xba1b, 0x455f, 0xbaeb, 0xaa73, 0x435d, 0xb201, 0x42d4, 0x4183, 0x415e, 0xba41, 0x41a3, 0xbfa5, 0x43ac, 0x4255, 0x1b2d, 0x422c, 0x4217, 0xbafe, 0x420c, 0x43e3, 0x40e3, 0x4199, 0x42fc, 0xbf4c, 0x41c6, 0x1001, 0x4674, 0x41d0, 0x404c, 0x4193, 0x4301, 0x16cb, 0x2399, 0x3509, 0x419b, 0x43d0, 0x18c3, 0x43d3, 0x40b9, 0x43ed, 0x137b, 0x40be, 0x2372, 0x3e64, 0x0ccb, 0xbf9b, 0x41a1, 0x26ed, 0x4359, 0x1326, 0xa32a, 0x4364, 0x3046, 0x410d, 0x416d, 0x3711, 0x41db, 0x40b5, 0x429f, 0x43a8, 0x42d0, 0xb2f9, 0x0758, 0xadfe, 0x421e, 0xb247, 0x2f6a, 0xbf0e, 0xb2e4, 0x2aad, 0x43a4, 0xb2d4, 0x439a, 0x4206, 0xbad9, 0x4153, 0x0dc3, 0xb217, 0xa3ea, 0x1a60, 0x1768, 0x4555, 0x40f9, 0x4205, 0x463d, 0x3e44, 0x4016, 0x3638, 0xba58, 0xb2c2, 0xba3a, 0xba04, 0x4224, 0x4359, 0x0001, 0x422a, 0xbf3c, 0x4342, 0x1b5e, 0xb22b, 0xb2ef, 0x36ae, 0xb204, 0xbfb0, 0x1858, 0x41da, 0x241b, 0xb04f, 0xb28d, 0xba31, 0x42f0, 0x42e7, 0x4028, 0x1dde, 0x45cd, 0xba2a, 0x40c0, 0x18b0, 0x0b4c, 0xbfbd, 0x1b99, 0x0418, 0x4239, 0x3b1d, 0x23c6, 0xba4f, 0x1f5d, 0xb2b0, 0xbac5, 0x1c2a, 0x45f1, 0x422b, 0x43bd, 0xa17c, 0xbf33, 0xba74, 0xb235, 0x0c06, 0x43b2, 0x40b3, 0xb0d2, 0xbf39, 0x4182, 0x46d9, 0xb2f1, 0x4271, 0x43e5, 0xb2b0, 0x41de, 0xbf00, 0x317f, 0x41e1, 0xa9e4, 0x18b0, 0x2a87, 0x4151, 0x43a1, 0x4179, 0x4204, 0x42c1, 0x44f8, 0x4389, 0x40f0, 0x2442, 0x43af, 0xbfa8, 0x3cac, 0x4365, 0x2149, 0x41ff, 0x443f, 0x4190, 0xb0ba, 0x216f, 0xb225, 0xb248, 0xb0f5, 0xba3e, 0x0b19, 0x1bc6, 0x41a7, 0xb0d7, 0xb2dc, 0xb20b, 0x1da9, 0x432d, 0x4293, 0x4035, 0x01ab, 0x2725, 0xbf93, 0xb264, 0x42b6, 0xb074, 0xb2db, 0xb2b6, 0x1dd3, 0x40fc, 0xba52, 0x4301, 0x1e30, 0xba78, 0x413c, 0xbfab, 0xa981, 0x414a, 0x439f, 0x4015, 0x0769, 0xb23d, 0x43de, 0x2bb2, 0xba4a, 0x4261, 0x412c, 0xbf9c, 0x4135, 0xb2ce, 0x4224, 0x1fcf, 0x42bc, 0x18cd, 0x1b98, 0x2bdb, 0x3605, 0x43df, 0xa64c, 0xbf9a, 0x32a3, 0x43ec, 0xb2d2, 0x43a1, 0xba7d, 0x43ee, 0x0e94, 0x44c9, 0x01d6, 0x4140, 0x2921, 0x424d, 0xb0a4, 0x0caa, 0xbf85, 0x41d4, 0x19a8, 0x0dc9, 0x4161, 0x1b35, 0x43ee, 0xa171, 0x4477, 0x4385, 0xba16, 0x420f, 0x3013, 0x0573, 0x4281, 0x3bf9, 0x181a, 0xb0bd, 0xb046, 0x3254, 0x4654, 0x43fd, 0xba34, 0x42b0, 0x4215, 0xb022, 0x42cf, 0x4225, 0x4281, 0xbf0e, 0x4021, 0x4217, 0x4351, 0x4283, 0xb2d9, 0xb223, 0xbfe0, 0x4209, 0xbfb6, 0xbaeb, 0x078c, 0xb21e, 0x1bc3, 0x2e6e, 0x43d9, 0x41c8, 0x40e0, 0x4361, 0x4225, 0x4078, 0x37fa, 0xb065, 0x4390, 0x40bd, 0x418c, 0xbfcf, 0xba18, 0x40e6, 0x4039, 0xbf00, 0x1bf0, 0x441a, 0x4345, 0x42d6, 0x2c06, 0x238c, 0x4284, 0x413b, 0x4295, 0xb291, 0x21de, 0xaabe, 0xba14, 0x4253, 0x3c60, 0xb29b, 0xbf00, 0xbfbc, 0x4081, 0xb2d5, 0xb2f1, 0x4253, 0x04c1, 0xb055, 0x1f26, 0x1d48, 0xbf88, 0x401d, 0x2150, 0x40ff, 0x08dd, 0xbf45, 0x18ce, 0x4341, 0x4217, 0x40b9, 0x09c6, 0x43a1, 0xbac8, 0xb28f, 0x420c, 0x41ea, 0xb2f7, 0x0e48, 0xa96a, 0xb29e, 0x0631, 0x3cc9, 0x43cb, 0x3771, 0xb23c, 0x46f9, 0xa882, 0xb2d3, 0xba10, 0x4375, 0x17f2, 0x03ab, 0xb2da, 0x43b0, 0x43ef, 0xbfbb, 0x2f20, 0xb051, 0x1ffa, 0x42c9, 0x2d15, 0x2157, 0xa778, 0x432c, 0x421a, 0x4430, 0x43f6, 0x4271, 0x436f, 0xbf62, 0xbac5, 0x4288, 0x24a8, 0x4361, 0xad27, 0x433f, 0x3a4e, 0xba0e, 0x41ed, 0xb2b6, 0xb27a, 0xbf53, 0xa105, 0x1f58, 0xa2f2, 0x43ff, 0xb074, 0x1ac5, 0x421e, 0x4241, 0xb2b7, 0x2ad2, 0x40d9, 0xa5fd, 0x25d8, 0xb2b6, 0xa3c8, 0xb2bf, 0xb226, 0xba6d, 0x07ed, 0x43db, 0x4261, 0x40e7, 0x41a5, 0x4619, 0x3e15, 0xb25a, 0x427f, 0xbff0, 0x1a2f, 0xbfd6, 0xbac4, 0x42ba, 0x0922, 0x11a7, 0x439f, 0x44d0, 0x43ff, 0x3c72, 0xba33, 0x4386, 0x1f3a, 0x40c0, 0x418f, 0x411a, 0x4212, 0x4110, 0x428f, 0x447c, 0xb097, 0xbf77, 0x42b7, 0x0621, 0xbacb, 0x46ca, 0x43af, 0x01f4, 0xba26, 0x1815, 0x41e1, 0x4150, 0x4171, 0x43b4, 0xa937, 0x412f, 0x42df, 0x429a, 0x415e, 0x431c, 0xad19, 0x4282, 0x4073, 0xbf87, 0x43d0, 0xba54, 0x2db6, 0x4350, 0x4237, 0x4164, 0xbae5, 0xb22b, 0x0a33, 0x43f8, 0xbf33, 0x40f2, 0xb2bc, 0xb2a4, 0x40e6, 0x41c9, 0x326c, 0x41c8, 0x42e9, 0x42af, 0xba3b, 0x359d, 0x4109, 0x4028, 0xb269, 0xba16, 0xb2fc, 0x3acb, 0x4063, 0xbf16, 0x09a3, 0xb2d5, 0xba22, 0x4142, 0xbf8b, 0x1c3c, 0x3e0e, 0xb29c, 0x1164, 0x331c, 0x1ce8, 0xbff0, 0x43d7, 0x4137, 0x429b, 0x18a9, 0x4193, 0xb0cc, 0xba49, 0xba38, 0x411f, 0xb205, 0xbfa6, 0x4173, 0x4142, 0x43e0, 0x4089, 0x182e, 0xba01, 0x41ea, 0x3a41, 0x2930, 0x4025, 0x0594, 0x1556, 0x0b43, 0x46f2, 0x248c, 0x4340, 0xbf7b, 0x4257, 0x4328, 0xba28, 0x04e8, 0x42c7, 0x4207, 0x1d28, 0x40d1, 0x2167, 0x43a3, 0x284d, 0xbf05, 0xb2a2, 0x1605, 0xb276, 0x412d, 0x40f3, 0x378b, 0xba41, 0xbaf7, 0x42a3, 0x425d, 0x45b4, 0xbf9b, 0xa2c2, 0xba13, 0x410c, 0x45da, 0x4372, 0x4228, 0x0a0b, 0x4550, 0x4295, 0x06c1, 0xb2cd, 0x1487, 0x31fb, 0xbac1, 0xb200, 0xba13, 0x2c7f, 0x4449, 0x42f8, 0xbaf4, 0xb2b4, 0x24b1, 0x4166, 0xba16, 0xbf7c, 0x4171, 0x4311, 0x32bc, 0x4572, 0x3a43, 0x4690, 0xb2b3, 0xb037, 0x41c0, 0x4198, 0x4459, 0x41e3, 0x42e3, 0x2d4e, 0x05da, 0xb20a, 0x1cfd, 0xb28f, 0xb080, 0xa6d7, 0x4268, 0x243a, 0x43b9, 0xbf27, 0x14b0, 0x2410, 0xba40, 0x1f3a, 0x4008, 0xbfd9, 0x1b4b, 0x40a4, 0x2a6f, 0x14aa, 0x407b, 0xba5c, 0x41ec, 0x4245, 0x1b03, 0x4315, 0x4139, 0x2c74, 0xbfc0, 0x1806, 0x0ae6, 0xbae8, 0x3d2f, 0xb244, 0x0bac, 0xb245, 0x4304, 0xbf90, 0xbf9a, 0xb218, 0x408a, 0x42e1, 0x0fa0, 0xb06d, 0x43d5, 0x41b2, 0xb20e, 0x222f, 0xbfd3, 0x3289, 0xb2d2, 0x45e1, 0xb0ab, 0xbf8b, 0xba5c, 0x41a9, 0x41e2, 0x42ee, 0x446d, 0x41d9, 0x4362, 0x4265, 0xba7d, 0x43c5, 0x2cce, 0xb284, 0xb2fb, 0x40fa, 0x3ddf, 0x4188, 0xb251, 0xba1d, 0xbf26, 0x424b, 0x4566, 0x14d4, 0x3784, 0xbf70, 0x4225, 0xb22b, 0x1f12, 0x4022, 0xb061, 0x4145, 0x42d3, 0xb2c9, 0x2199, 0x42e5, 0x1a4d, 0x25c8, 0xbf56, 0x401e, 0x1c37, 0xba38, 0x1498, 0x4052, 0x4181, 0xba67, 0x41ab, 0x1bc7, 0x43d8, 0x4379, 0x42d0, 0x4211, 0x3a14, 0x00c0, 0x4068, 0x41dd, 0x4138, 0x40cb, 0xb21f, 0xb0c9, 0xba5c, 0x4004, 0xbfd5, 0x43bf, 0x4344, 0x06d0, 0x40ea, 0x2d6a, 0xbfe4, 0x42bc, 0x40a0, 0x40c6, 0xaa16, 0x42bd, 0x421a, 0xb2d7, 0x41ca, 0xb016, 0x43ef, 0xae2d, 0x4341, 0xb0c0, 0xba1b, 0x40c7, 0xb0cd, 0x3c18, 0x2ab4, 0x41d2, 0x0506, 0x2144, 0x183c, 0xba30, 0xb266, 0xb2e5, 0xbf90, 0xbf4a, 0xa979, 0x4300, 0xb26e, 0xbfd0, 0xba66, 0x13dc, 0xb28e, 0x1a28, 0x4158, 0x43ab, 0x4075, 0x40f2, 0x4593, 0x198f, 0x38c5, 0x4061, 0xbf23, 0x3e56, 0xa0ef, 0xba50, 0x4433, 0xba0b, 0xad1b, 0x1df6, 0xbad9, 0xb253, 0x40cc, 0x424c, 0x1a18, 0x1b36, 0x186b, 0xbfdf, 0x4167, 0x0075, 0x431c, 0x400c, 0x3e22, 0x434d, 0xba3b, 0x4388, 0x2f78, 0x407f, 0x4254, 0x1610, 0xba4b, 0x43f8, 0x4089, 0xb231, 0x4050, 0xb29d, 0x0c5a, 0x3afd, 0x42e8, 0x1c7a, 0xb241, 0x0cad, 0xb268, 0xbfc0, 0xbf6a, 0xb2cc, 0x37b0, 0x46eb, 0x4129, 0x1e5b, 0xb216, 0x0748, 0x41d1, 0x4308, 0x45ac, 0xb2a5, 0x43d5, 0x4049, 0x4076, 0xb28d, 0x2611, 0xb26d, 0xba2e, 0x34d4, 0xbafb, 0x3761, 0x417a, 0x4156, 0x4575, 0xb210, 0xbf4d, 0x42ba, 0xba26, 0xa039, 0x422c, 0x4357, 0xaee4, 0x41ab, 0x43eb, 0x1d12, 0x434c, 0x1818, 0x0a20, 0xb208, 0xb2a2, 0x4187, 0xba69, 0x43de, 0xa5ce, 0xbfac, 0xb0db, 0xb274, 0x4347, 0x4151, 0x1f64, 0x1c6b, 0x1db7, 0xbad5, 0x1e0a, 0x198e, 0x416f, 0xbacf, 0x42fe, 0x4173, 0x183b, 0x44f4, 0x4075, 0x43a0, 0xbaf0, 0xbf41, 0x1e71, 0x39a2, 0x1dc5, 0xb212, 0x0159, 0xbaea, 0x1fdd, 0x42d7, 0x4453, 0xb2dc, 0x0817, 0x4698, 0xba22, 0xbf58, 0x04db, 0xb216, 0x1b55, 0x42e8, 0xba7c, 0xbf88, 0xa1f0, 0xba12, 0xb21f, 0x2628, 0x4091, 0x4554, 0x0c86, 0x4098, 0xb0fb, 0x43e1, 0xbaed, 0xb2c1, 0x1f65, 0x4019, 0x4365, 0x0c81, 0x40dd, 0xaf49, 0x40f2, 0x40a3, 0xba56, 0x4378, 0xa9c0, 0xbf81, 0xba7f, 0x404b, 0xaec3, 0x3eec, 0xbadf, 0xacb7, 0x434a, 0xba10, 0xbaeb, 0x1ea3, 0xbf19, 0x413d, 0xb07a, 0x1d45, 0x42ff, 0xb08d, 0x1a6b, 0xa7c4, 0xa3d1, 0x40bb, 0xba7f, 0x460a, 0x45cb, 0x4490, 0xb097, 0x40f8, 0x00c0, 0x3d1a, 0x42bb, 0x30ae, 0x1c4f, 0xad13, 0xbafd, 0x40c3, 0x40e1, 0x4046, 0x434e, 0xbf3a, 0x185c, 0x409a, 0x26fc, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6e009d47, 0x4404ee01, 0xc475426e, 0xc7651e4f, 0x99397e2c, 0xe2311d1e, 0x13b3b0d5, 0x94ee1dcc, 0x8a96e390, 0xfb8d253a, 0x3cd58f7e, 0xf938ad79, 0x61646a13, 0x14d0ae7f, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x000000ae, 0x00000000, 0x14d0af83, 0x00000000, 0x00000000, 0xffff84af, 0x00000000, 0x14d0af84, 0x14d0b083, 0x00001344, 0x00000000, 0xf938ad79, 0x61646a13, 0x14d0addb, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb08d, 0xba58, 0x1eea, 0x4241, 0x439f, 0x27f1, 0x40eb, 0xb27f, 0x1bed, 0x407c, 0x4327, 0x41ea, 0x27fd, 0x410a, 0xbf6d, 0x4038, 0xa806, 0xba0f, 0xb265, 0xb22d, 0x1b44, 0x1c14, 0x431d, 0xaad5, 0x3ee7, 0x4034, 0x2d0b, 0x2dcc, 0x4334, 0x3e1c, 0xbf00, 0x11eb, 0xb2db, 0xa20c, 0xb058, 0xbfab, 0x4390, 0xb042, 0xb051, 0x2fcb, 0x415c, 0x0a57, 0x408c, 0x1d6e, 0xa072, 0xba55, 0x2581, 0x447b, 0x43a5, 0x440b, 0x1e63, 0xb2cc, 0xa334, 0xba69, 0xabd9, 0x1d2d, 0xbf62, 0x4140, 0x024e, 0xb2ac, 0x40ac, 0x4567, 0xba10, 0x1eb6, 0x3178, 0x0a5f, 0xb2bc, 0x40b0, 0xbfd4, 0x4285, 0x0c35, 0xb21e, 0xbacd, 0x1a41, 0x3ab5, 0x117a, 0x4469, 0xb2a3, 0x2dc3, 0x416d, 0x4541, 0xba7f, 0x1a93, 0x4389, 0xbf57, 0x301f, 0x4320, 0x28ae, 0x4020, 0xbac7, 0x1192, 0xbf45, 0x42ba, 0x43c3, 0x42b2, 0x0233, 0x1fa2, 0xb0b2, 0x41f2, 0xbaf6, 0xbf92, 0x0ec4, 0xb28d, 0x406b, 0xbff0, 0x403f, 0xba61, 0xb204, 0xbaeb, 0xba0b, 0x402a, 0x1ee1, 0x431b, 0x4331, 0x2513, 0x2108, 0xbf62, 0x4257, 0xa533, 0x3fd8, 0x42c7, 0x2195, 0xb2df, 0x42ab, 0x2830, 0x4373, 0xb286, 0x3afb, 0x42a2, 0x2e11, 0x4188, 0xb0c4, 0xba0e, 0x1d2e, 0x4158, 0xa422, 0x1dec, 0x417a, 0xb239, 0x43c5, 0x4074, 0xbf1c, 0x459e, 0x417c, 0x1b2b, 0xb2d6, 0xb296, 0x43ca, 0xbfd6, 0x449d, 0x23aa, 0x4198, 0xb267, 0xa248, 0xbf00, 0x4368, 0xb04a, 0x0734, 0x1ae6, 0x42d9, 0x414f, 0x1ed1, 0x4291, 0x1cde, 0xba3d, 0x4103, 0xa7df, 0x4586, 0xb257, 0x2302, 0x46a2, 0x43e6, 0xba41, 0x4128, 0xb2dc, 0x268d, 0xb235, 0xbfb7, 0x183a, 0x432d, 0xb0be, 0x40c4, 0x46c2, 0x42ae, 0x1cef, 0x16a2, 0x432a, 0x034c, 0xa8cc, 0x1aa9, 0x4635, 0x33d5, 0x42be, 0xb225, 0x4054, 0x1856, 0xb02b, 0x4683, 0xbfd6, 0x3dcb, 0x45ad, 0x4149, 0x42ec, 0xa5a3, 0xbfb0, 0xb04d, 0xb266, 0x405c, 0x40ef, 0xb23d, 0xb2fe, 0x432d, 0x41fe, 0x43ad, 0xb268, 0xa0d6, 0x41d9, 0x023a, 0xb291, 0x4397, 0xb2fb, 0x035e, 0xbf3f, 0x43aa, 0x4371, 0x3540, 0x4013, 0x42bc, 0xba4d, 0xb04e, 0x4156, 0x43be, 0x40b7, 0x40a9, 0x409c, 0x4288, 0x415e, 0x0f3e, 0xb009, 0xbfd7, 0xb0d2, 0x4263, 0x3228, 0x26dd, 0x1b78, 0x4170, 0x408e, 0xbfc7, 0x1ed8, 0x459d, 0x4199, 0x41af, 0x4340, 0x434a, 0x43a6, 0xbfc1, 0x2986, 0xbadc, 0x1d21, 0x00cc, 0x42e0, 0xae51, 0xa7dc, 0xb24e, 0x42fb, 0x1ba5, 0x1a40, 0x1ce0, 0x1e2b, 0x355f, 0x4357, 0x40c2, 0xba61, 0x4182, 0x4383, 0x430e, 0x1593, 0x41d6, 0xb0fa, 0x410f, 0xb21c, 0x4075, 0xbf8f, 0xb2c4, 0x4285, 0x4003, 0x1281, 0x414a, 0x408b, 0x403e, 0x2363, 0x412a, 0x430e, 0xbf45, 0x2282, 0xb24a, 0x330a, 0x4145, 0x4285, 0x43ed, 0x4031, 0x18ea, 0x4227, 0xbf80, 0x2468, 0xba52, 0x41ce, 0x4659, 0x4095, 0xb28c, 0xa9d4, 0x4439, 0x1b5b, 0x1349, 0x41d5, 0xbac3, 0xb09a, 0xba48, 0x4216, 0xbfa1, 0x0eec, 0x205a, 0x1cee, 0x44f9, 0x43c9, 0x0ee8, 0x0e50, 0x4258, 0x4029, 0x403f, 0xb226, 0x42bc, 0x1ba6, 0xbf48, 0x42a7, 0x40f2, 0x4260, 0x4499, 0xbfd3, 0x38e7, 0x40f2, 0x2087, 0x4189, 0x2243, 0x4111, 0x3fa8, 0x40bb, 0xb0be, 0x4026, 0x1b74, 0xbf0e, 0xb047, 0xb20e, 0x1961, 0x4241, 0x4063, 0xba4f, 0x40dd, 0x411b, 0xafbd, 0x39a3, 0x4254, 0x4262, 0x41e0, 0xbacd, 0x2264, 0x40bc, 0xa651, 0xb20c, 0xba0d, 0xb261, 0x4484, 0x432a, 0xbfdf, 0xb01a, 0xb20e, 0x2256, 0xbac1, 0x1ac5, 0xb2eb, 0xb2ee, 0x2927, 0x009a, 0x4291, 0x3575, 0xb21a, 0xb2d1, 0x1a67, 0xbaff, 0x4115, 0x42ef, 0xbf26, 0x0fa4, 0x415d, 0xb253, 0x010c, 0xb0aa, 0x2d7b, 0xb020, 0xb0da, 0xae00, 0x2f87, 0xb095, 0x0ca0, 0x42f8, 0x4037, 0x1c6f, 0x436b, 0x42b4, 0x1d3d, 0x3251, 0x43ff, 0x411a, 0x4394, 0xbf71, 0x1ac3, 0x42b2, 0x37b3, 0x3db9, 0x4388, 0xb080, 0x4314, 0x406c, 0x40ad, 0xb0ac, 0xb047, 0xbaff, 0x4091, 0x4372, 0xb29b, 0x4119, 0x45ab, 0x405d, 0xbf95, 0x3f6d, 0x1d73, 0x1896, 0x4074, 0xbad7, 0x211b, 0xa48f, 0x436d, 0x4181, 0x1e91, 0x1d3a, 0x42d0, 0xb25b, 0x43c6, 0xba43, 0xb284, 0xbfe0, 0x1eed, 0x0218, 0x40a2, 0x42a8, 0x40a7, 0x1c26, 0xbf00, 0xbf38, 0x447c, 0xbae0, 0x40d9, 0x1c9a, 0x42ec, 0x43f7, 0x386f, 0xbf59, 0x189b, 0x4037, 0x4323, 0xb281, 0xba6c, 0x1a55, 0xb2a2, 0x4218, 0x41d1, 0x4215, 0x383f, 0x429f, 0xbf89, 0x2d17, 0x46b9, 0xb0d1, 0xbf70, 0xba01, 0xbf2e, 0x1f89, 0x45a3, 0xb07c, 0xadd1, 0x42dd, 0x400c, 0x1adc, 0x43b9, 0x4072, 0x418c, 0x0a53, 0xad90, 0x1f17, 0x418f, 0xbacf, 0x1fc4, 0xa559, 0xba37, 0x1b9c, 0x461c, 0x42c8, 0x401e, 0x1971, 0x43b5, 0xba29, 0xa499, 0x4366, 0xbf7d, 0x1b05, 0xbadb, 0xbad4, 0x1b87, 0x43f7, 0x45b4, 0xa2ab, 0x43a8, 0xb08b, 0x18c0, 0xb2fe, 0x435d, 0x0129, 0x4200, 0x43fa, 0x33ac, 0xb2f5, 0x411c, 0x423b, 0x418e, 0x3876, 0xb22f, 0xbad0, 0x4464, 0xbf0e, 0x24f1, 0xb21c, 0x42dc, 0x42c1, 0x40f5, 0xa9ad, 0x4283, 0xb2a1, 0x432e, 0xbfb0, 0x428b, 0x42d3, 0x08fd, 0x0f6c, 0x437c, 0x0f32, 0x43ea, 0x0768, 0x4192, 0x4001, 0x01af, 0x22ab, 0x43df, 0x41d1, 0xa87c, 0x1cc1, 0x0e0e, 0xbf55, 0x1b17, 0x4265, 0xb0e6, 0x4359, 0x43ca, 0x43bd, 0x40ff, 0x4110, 0x1cf1, 0x4037, 0xb254, 0x1957, 0xbf41, 0x0fec, 0x45b3, 0xb21a, 0x41e8, 0xb21e, 0x417f, 0xaf50, 0x0c9d, 0x4395, 0x4122, 0x43ef, 0xb01c, 0x42fa, 0x41da, 0x43a3, 0x41c8, 0x3d6b, 0xb06d, 0x41c4, 0xba07, 0x391d, 0x4214, 0xb2f7, 0x44dc, 0xa011, 0x42fa, 0x4483, 0x1ddc, 0xbf7a, 0xb24d, 0x4270, 0x269c, 0xba2a, 0x435a, 0xb29d, 0x1ce6, 0x1852, 0x093e, 0xb0ed, 0xbaf1, 0x1d00, 0x02b5, 0x17ec, 0x42d8, 0xba16, 0x4624, 0x26af, 0x41b1, 0x1631, 0xb2fe, 0x4361, 0x1058, 0x41db, 0x1992, 0xbf1e, 0x41d8, 0x40c2, 0x43a3, 0x42ba, 0x41b7, 0x1d22, 0x43d5, 0xb213, 0x0005, 0x4004, 0x4171, 0x4038, 0x4364, 0x3420, 0x4135, 0x445c, 0x42f6, 0x4282, 0x4110, 0xbf5a, 0x3781, 0xb2f5, 0x40f8, 0xb2bc, 0xb264, 0x4591, 0x1812, 0xb06d, 0xba2f, 0xbf43, 0xa219, 0x46c0, 0xb2f3, 0x40f0, 0xbf5d, 0xbae7, 0xa3f0, 0x423e, 0xb246, 0x36e6, 0x402e, 0x400e, 0xb2af, 0xbfde, 0xba53, 0x085b, 0xab40, 0x43ac, 0x431a, 0x2778, 0x1da4, 0xb029, 0x14e4, 0x439f, 0x15ca, 0xa275, 0x1f63, 0x41ad, 0x3dee, 0xbf97, 0xbff0, 0x40e4, 0x448d, 0xb2d7, 0xbf8a, 0x41e5, 0xba2e, 0x2351, 0x1b2d, 0xbfe0, 0x4261, 0x414d, 0xbf55, 0xb28e, 0x1bb7, 0x439d, 0x41f4, 0x04a2, 0x040e, 0xb2a2, 0x4313, 0x1df1, 0xbac7, 0x41cc, 0x41d6, 0xbfb5, 0x432e, 0x42c0, 0x1cdd, 0x1ccb, 0xbf07, 0xb052, 0x1ead, 0x4281, 0x431d, 0x4365, 0x4206, 0x1b51, 0x43b9, 0x16d4, 0x0878, 0xbf89, 0x40d5, 0xa72b, 0xbad8, 0x3d59, 0x43b9, 0x42ff, 0x454d, 0xbae8, 0x11ea, 0xb2d7, 0x1455, 0xb2bc, 0xb29d, 0xba69, 0x461c, 0x19b1, 0xb282, 0xb054, 0x08a5, 0x45cc, 0x40ec, 0xbf9b, 0x0e9d, 0x1b02, 0x4099, 0x4080, 0x4322, 0x1313, 0x236c, 0x19cb, 0xa05e, 0x4350, 0x4299, 0xba45, 0x1afa, 0x1a45, 0x41f1, 0x058f, 0x18aa, 0x4190, 0xb0b6, 0xbfc8, 0xba6c, 0xb019, 0x4260, 0x40cb, 0x40e2, 0x1c47, 0xae93, 0x340d, 0x0e55, 0xba34, 0xaacd, 0xbfae, 0x416d, 0xb0ef, 0x4056, 0xba07, 0x41eb, 0x4490, 0xba61, 0x402c, 0x469a, 0x3dbe, 0x4015, 0x4560, 0x439c, 0x2497, 0x3ceb, 0x4196, 0xbf2d, 0x3df9, 0xb029, 0x1aa2, 0x1f04, 0xbfb3, 0xb006, 0xb001, 0x4067, 0x405f, 0xbaf8, 0xba17, 0xbf6f, 0xb2a3, 0x32e4, 0x27be, 0x4297, 0x3a02, 0x438b, 0x422b, 0xb0bc, 0x437e, 0x1f3a, 0xba0e, 0x40fa, 0xbf00, 0xa05d, 0x4052, 0x4302, 0x3ae8, 0xb207, 0xb01a, 0x4317, 0x440d, 0x44ca, 0xbad5, 0xa59a, 0x462c, 0xb22a, 0xa1e3, 0x46c4, 0x417a, 0xbf14, 0xbaec, 0xb296, 0x43c8, 0xb213, 0x2bbb, 0xb20d, 0x0e03, 0x43a6, 0x139c, 0xba1a, 0xacf0, 0x406e, 0x0198, 0xbf60, 0x400f, 0x428d, 0x40dc, 0x1881, 0xba24, 0x4222, 0xb0f8, 0x4380, 0x4093, 0x40a8, 0x3f1c, 0xbf49, 0xb2ae, 0xb238, 0x42d6, 0xb20c, 0x24d6, 0x1e47, 0xbf1b, 0x396a, 0x1a86, 0x4184, 0x402e, 0x3053, 0xb26c, 0x0b64, 0x4109, 0x46c2, 0x424b, 0xba29, 0x1c0e, 0x41bd, 0x4459, 0xb21f, 0xb20a, 0x43af, 0x424a, 0x432c, 0xb0db, 0x41c0, 0x4454, 0xbf88, 0x37f0, 0x4129, 0x4137, 0x436b, 0x44b8, 0x4043, 0x310e, 0xba09, 0x258e, 0x40a9, 0x4205, 0x4269, 0xba39, 0xb011, 0xbf00, 0x1e11, 0xba64, 0x1c52, 0x4113, 0x2228, 0xb2bf, 0xb058, 0x2bfe, 0x430f, 0x1d24, 0xbf0e, 0x43f1, 0x4113, 0xb268, 0x189e, 0x43a1, 0xb21a, 0x4193, 0xb287, 0x346a, 0x419b, 0x42a4, 0xad64, 0x41ac, 0x459d, 0x30a2, 0x2363, 0x40ad, 0x462f, 0x459c, 0x3579, 0x298b, 0x4105, 0xbfab, 0x3d00, 0x42dd, 0x4390, 0x1c75, 0x4156, 0x1cb5, 0x4349, 0x43f2, 0x45d6, 0x40f5, 0x4328, 0x41d2, 0x4165, 0xb00f, 0x4023, 0xba7b, 0xb0d4, 0x41f4, 0xbff0, 0x4085, 0x40a0, 0xbf78, 0x1892, 0xb067, 0x4055, 0x4094, 0x42d6, 0xba35, 0x42b0, 0x41e8, 0xac04, 0x42d6, 0x41aa, 0xba6c, 0xb273, 0x1c21, 0x40cf, 0x4492, 0x46d5, 0x1965, 0x4221, 0xb2c4, 0x43e7, 0x415a, 0x432f, 0x407f, 0xbf8c, 0xb283, 0x46a5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd25485a3, 0x65f9ab76, 0x3fac21bc, 0xb38c4c4b, 0x9ae6ab78, 0xff30a4a6, 0x01b8bb18, 0x9f4c9ead, 0xa09d6abf, 0x3ea9cd0d, 0x60d7d350, 0x04406247, 0x923282a2, 0xf465cdc9, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00280000, 0xd7ff6025, 0x00000028, 0x00000000, 0x28280000, 0x00000028, 0x00000000, 0x55f83e89, 0x3eaa3cd7, 0x2df79e85, 0xb55ae7e5, 0x55f83e88, 0x00000000, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbad5, 0x1d59, 0xa4c4, 0x1a8d, 0x1b22, 0xa8c9, 0x45cc, 0x422b, 0x3e2e, 0x4264, 0x4080, 0x3475, 0x4121, 0x41b2, 0x4344, 0xa434, 0x45f0, 0x1e00, 0xbf34, 0x4409, 0x4310, 0x43ce, 0x43ed, 0x4044, 0x4019, 0x41dc, 0x1ea8, 0x4076, 0xbaf0, 0x42b2, 0x41ce, 0x406e, 0xb2e7, 0xbfc0, 0xb202, 0x1d55, 0xbf74, 0x42b3, 0xa20f, 0x41e6, 0x1551, 0x29e3, 0x417b, 0x4037, 0x2314, 0x4311, 0x4163, 0xb2d6, 0x416c, 0xadaa, 0xb258, 0x3667, 0xbfcd, 0x42b1, 0x42ab, 0xbfb0, 0x4591, 0xa1d1, 0xb05f, 0x442c, 0x1b90, 0x42c0, 0xb29a, 0x43e5, 0xb283, 0xba53, 0xb228, 0xbad6, 0x4031, 0x185e, 0xba00, 0xb2d1, 0x4298, 0xb2c9, 0xbfbd, 0x4229, 0xb297, 0xb016, 0x42a1, 0x4022, 0x120d, 0xba0f, 0x1dce, 0x435a, 0x4039, 0x183a, 0xbf70, 0x3a82, 0xa4d5, 0x250f, 0x4341, 0x413b, 0x4209, 0x467a, 0xbfa0, 0x1dc9, 0xbad6, 0x4106, 0x43dd, 0xbf8b, 0x45a0, 0x4253, 0x44d5, 0x1d62, 0x4128, 0xaaf0, 0x40be, 0x4373, 0xba76, 0x4314, 0xbf35, 0x400c, 0x019c, 0x43aa, 0x42dd, 0xba6c, 0xb23f, 0x4437, 0xbfd8, 0x4419, 0x2e1a, 0x4270, 0x429f, 0xba18, 0x2478, 0x3fa1, 0x40b6, 0x4130, 0x43c6, 0xbf9f, 0x401f, 0x01ae, 0x4264, 0xb264, 0x41db, 0xbac5, 0xbfd6, 0x4678, 0xb20e, 0x4244, 0x44c5, 0x1729, 0xba6d, 0x418b, 0x46a2, 0xb0bf, 0x38e1, 0x4386, 0xba49, 0x4387, 0x1dd4, 0x1e09, 0x40dd, 0x45e6, 0x40c6, 0xbf81, 0x05a7, 0xb2a8, 0xba43, 0x0104, 0x0c94, 0x42d5, 0x4153, 0x407a, 0xbfb6, 0x406f, 0x4146, 0x1c98, 0x43cf, 0x40c8, 0x40bf, 0x432b, 0x1ab7, 0xbf0b, 0x0f94, 0x42bb, 0xba05, 0xba0f, 0x43dd, 0x40d1, 0xb28d, 0x1d03, 0xb02f, 0x43f1, 0x430f, 0xbf19, 0xb05c, 0xba78, 0x423d, 0x41d4, 0x44d0, 0xa90f, 0x120b, 0x4649, 0x40b5, 0x3754, 0x42d1, 0x40a8, 0x2a90, 0xb20b, 0x41e6, 0xb066, 0xb290, 0x4188, 0x40c2, 0xb273, 0xbf41, 0x00f2, 0x427b, 0x1c07, 0x4654, 0xab5e, 0x03e3, 0x441a, 0x40c2, 0x4273, 0xb0a5, 0xba56, 0x45bd, 0x42a5, 0x4369, 0x049c, 0x23de, 0xbf76, 0x1a8a, 0x424c, 0xbff0, 0xbfc0, 0x1e0a, 0xb2c1, 0x4238, 0x4388, 0x4485, 0x34b5, 0xa687, 0x4235, 0xb295, 0xb0a0, 0x4175, 0x4280, 0xadc3, 0xa75e, 0x467c, 0x4169, 0xb20c, 0x42cb, 0xbf12, 0x1ad9, 0x1acf, 0x46e8, 0xba62, 0x4180, 0x1e87, 0x4028, 0x093f, 0x1810, 0xb015, 0x42d2, 0x1ba6, 0xbf04, 0xb0d5, 0x4156, 0x43e8, 0x4246, 0xbac6, 0x4204, 0xb272, 0xba2f, 0xb24e, 0x422c, 0x40e9, 0x43ec, 0x4019, 0x41a1, 0x1cad, 0x007a, 0x41b7, 0x4231, 0xbfc1, 0x3779, 0x1010, 0xbafe, 0x4311, 0x45b9, 0x469a, 0x41fe, 0xb205, 0x17e7, 0x0096, 0x166f, 0x4090, 0xb260, 0x4241, 0xb0ad, 0x1c8e, 0xbf4f, 0xb2a7, 0x432d, 0xba6a, 0x4358, 0xb26d, 0x429c, 0x40c8, 0x1de3, 0x244f, 0xa9e5, 0x412a, 0x1bff, 0x40b1, 0xb2c6, 0x469b, 0x0d3c, 0x1ef1, 0xbfb6, 0xa11f, 0x4011, 0xb298, 0x0847, 0x400c, 0xb255, 0xb0a0, 0xbae9, 0xba33, 0x413e, 0xb2a9, 0x40c4, 0x199a, 0xb2b8, 0xbf8b, 0x437d, 0x18dc, 0x19c1, 0x43ee, 0x0ce7, 0x0e99, 0x287c, 0x2f2a, 0x1e51, 0xbf90, 0xb217, 0x41a9, 0xb230, 0xb02b, 0xbf7b, 0x4319, 0x12d0, 0x4279, 0xba78, 0x1950, 0xbf90, 0x0410, 0xb08d, 0x260b, 0xbfa8, 0x3d18, 0x4338, 0x4185, 0x4061, 0x4042, 0x41ad, 0xbaef, 0x419e, 0x194f, 0x249b, 0xb2c1, 0xbf19, 0x1f07, 0x4370, 0x1cd7, 0x439a, 0xab32, 0x4256, 0xbf2f, 0x43cb, 0x412b, 0x41a7, 0x1b51, 0xb20f, 0xb03a, 0x4000, 0x3858, 0x3170, 0x2475, 0xae13, 0xb231, 0x4498, 0xbf9a, 0x45ad, 0x404e, 0x2df8, 0xb25b, 0x3db3, 0xa424, 0xb211, 0xb208, 0x43fb, 0x4241, 0x1937, 0x1c73, 0x40f4, 0x0729, 0x41cd, 0x2eff, 0x4288, 0xb0cc, 0x1be4, 0xbafc, 0xbf21, 0x1150, 0xba5b, 0x41ec, 0x1e5d, 0x4324, 0x430d, 0x189e, 0x416b, 0xb0ed, 0x1958, 0x43c4, 0x439c, 0x40aa, 0x42e3, 0xba03, 0x4255, 0x4014, 0x4583, 0xb2b2, 0xb278, 0xb233, 0x2f6f, 0x31e0, 0x41bb, 0xbfce, 0x439f, 0xa48e, 0x42cf, 0x434c, 0xb25d, 0x4317, 0xb281, 0x439b, 0x4098, 0x4201, 0xae51, 0x4281, 0x43c0, 0x417b, 0xb014, 0x4561, 0xba03, 0xb26b, 0x1e97, 0xb2f8, 0x4131, 0x4191, 0x40fe, 0xbf78, 0x227e, 0x4692, 0xba49, 0x4175, 0x4495, 0xb252, 0xba21, 0x43d7, 0x4317, 0x214c, 0xba01, 0x46d9, 0x41a8, 0x4023, 0xbf4b, 0x28eb, 0xb055, 0x431e, 0x40cb, 0x40d2, 0x1bce, 0x1a62, 0x4004, 0x438d, 0x1f3c, 0xb20b, 0x4025, 0x410f, 0xa075, 0xb262, 0xbf37, 0xbade, 0xb264, 0x1a2d, 0x19a6, 0x1811, 0xbfdd, 0xbad4, 0x4331, 0x2bc6, 0x42d3, 0x42ff, 0x23d1, 0x442b, 0x28f2, 0x1597, 0xbfa5, 0xb0ad, 0x41fc, 0x446c, 0xba71, 0x45d1, 0x24a8, 0x1f6f, 0xba30, 0x4228, 0x40dc, 0x402e, 0x43d7, 0xae6c, 0x403b, 0x4027, 0x4391, 0xbf69, 0xb2c4, 0x45ec, 0xba6a, 0x406c, 0x46a5, 0xba6a, 0xb2dd, 0xbad4, 0x41e5, 0x40a9, 0x41e5, 0x1abf, 0xbf74, 0x1c83, 0x42d6, 0x43d1, 0xba0f, 0x43f6, 0x042d, 0x4341, 0x42b2, 0xbf14, 0x1fe1, 0x4275, 0x4128, 0x401c, 0xb20d, 0x02b4, 0x4493, 0x41ac, 0xb21a, 0x417f, 0x4283, 0x2bd4, 0x40a7, 0xb293, 0x4028, 0xb2d5, 0xbac8, 0xb250, 0xbf2f, 0x426d, 0x424e, 0x433a, 0x1e87, 0x4352, 0xb295, 0xbadd, 0x4369, 0x05ca, 0x415d, 0xb215, 0x4247, 0x4220, 0x3a43, 0x0025, 0x4166, 0xbac9, 0xba2a, 0x4084, 0x0d4f, 0xb07c, 0x43b7, 0x45e5, 0x42de, 0x401b, 0x4382, 0xbf6b, 0x45ae, 0x279c, 0x4117, 0x4138, 0x0998, 0x425f, 0xbf24, 0x08b8, 0xb016, 0x1f7d, 0x2512, 0x19cd, 0xba6d, 0x42d8, 0xba0c, 0x30ef, 0x4407, 0x2342, 0x0c6e, 0x432c, 0x41d3, 0x13e8, 0x1350, 0x363a, 0xb072, 0x3631, 0xbf6f, 0x1a7d, 0x2ed0, 0x43d7, 0x28be, 0x3dc7, 0x15f5, 0x4118, 0xba6c, 0x4604, 0xa826, 0x423a, 0xb2bc, 0xb201, 0xbf0d, 0x41bc, 0x1afd, 0xb061, 0x3a1d, 0x4126, 0x42fe, 0x2aa7, 0x4165, 0xba5b, 0xb078, 0xbfc0, 0x4351, 0x42e4, 0x4183, 0x4211, 0x4651, 0x40b3, 0x4328, 0x43d0, 0x207b, 0xb24b, 0x1018, 0xae51, 0x18f3, 0x4037, 0xba1d, 0x1529, 0x3c02, 0xbf14, 0x0859, 0x40a1, 0x1daf, 0xb296, 0xb0b8, 0x2b0a, 0x4221, 0xbacc, 0x185d, 0x402d, 0x18ff, 0x42a7, 0xb28b, 0xbf6e, 0x43af, 0x419c, 0x3a08, 0x4230, 0x2623, 0x1903, 0x0462, 0x0149, 0xba1a, 0x1eba, 0x436d, 0xb0b8, 0x40df, 0xb279, 0xba60, 0x44f1, 0xba16, 0x4085, 0xb2c5, 0x1e6e, 0x40e0, 0xbf8c, 0x422a, 0x42ea, 0x42fe, 0x4544, 0xbf53, 0x430b, 0x3121, 0x4012, 0x4291, 0xb007, 0xb223, 0xb2a1, 0xb2ce, 0x09bd, 0x1d76, 0x4171, 0x40d3, 0x08b5, 0xacc9, 0x2e88, 0xb2dd, 0xb2e7, 0x42f5, 0x408d, 0x41a0, 0x41ef, 0x1c3f, 0x464e, 0x2766, 0xbf4a, 0x4087, 0xafb7, 0x461c, 0x43d1, 0xa153, 0x34a3, 0xbfa0, 0xba05, 0x0f87, 0x4095, 0xbfbf, 0x4252, 0x4160, 0x41d3, 0xb071, 0x09c2, 0xb05b, 0x4368, 0x40ab, 0x07d6, 0x45dd, 0xb047, 0x43b8, 0x42f8, 0x23c1, 0x19d1, 0x3c53, 0x42a3, 0x2e39, 0x4208, 0x04ed, 0x411c, 0x29ec, 0x43ac, 0xa6b8, 0x403b, 0xba30, 0xbf11, 0xb2fb, 0x1810, 0x05a6, 0x40b2, 0xa664, 0xbfaa, 0x434c, 0xb280, 0x2bd7, 0x22b9, 0x42ca, 0x0fe6, 0x2a39, 0xa493, 0x40ce, 0x424a, 0xba10, 0x42e2, 0xa50a, 0x43b9, 0x456b, 0xb2a1, 0x432b, 0x43ee, 0x05a5, 0x4142, 0x4417, 0x1ccf, 0x40a5, 0x43d8, 0xbfa1, 0xb268, 0x4092, 0xb20f, 0x401f, 0x40ea, 0x1b3d, 0xbaed, 0x0264, 0x0b78, 0x42fb, 0x0836, 0xb2ed, 0x424a, 0x40d1, 0x2cfd, 0x101f, 0x4293, 0x46a9, 0x46cb, 0xbfa3, 0x4058, 0x41d6, 0x40c1, 0xba3a, 0x45dc, 0x11ed, 0x3aa0, 0x468d, 0xbf80, 0xa4dc, 0x1936, 0x1766, 0x280e, 0x1ae8, 0xbfde, 0x33a9, 0x17d0, 0xa82c, 0xba1a, 0x43d1, 0x4310, 0x41b3, 0xb21a, 0x4270, 0xb285, 0x4055, 0xba60, 0x1c37, 0x224c, 0x43ce, 0x43c3, 0x234e, 0x43ca, 0xb213, 0xb29c, 0x4294, 0xba5c, 0xb2b3, 0x2d4b, 0xa6db, 0xbf1e, 0x4240, 0xa2f2, 0x138e, 0x1d24, 0x4235, 0x4569, 0x432e, 0x41ef, 0xa704, 0x25b2, 0x13bc, 0xb0eb, 0x41e4, 0x401e, 0xa704, 0x41ad, 0x41dd, 0x4298, 0x1b03, 0x423c, 0xb248, 0x405c, 0x0b80, 0x40c7, 0xba75, 0x43b0, 0xbfb6, 0x10b0, 0x41ec, 0xa1b4, 0xb26b, 0xb295, 0x412c, 0x44e5, 0x2796, 0x2eb7, 0x4165, 0xbfc9, 0x42b4, 0xac49, 0x40da, 0xba7f, 0x40c1, 0xbf00, 0x4376, 0x0fd1, 0x1a0f, 0x4356, 0x41ac, 0xb00e, 0x45f3, 0x2458, 0x333c, 0x0171, 0xb273, 0x4304, 0xbfe2, 0x45cb, 0xa2d9, 0x40d8, 0x4348, 0x1760, 0x412f, 0x42d4, 0x40a9, 0x424f, 0xbf18, 0xb07a, 0x40b1, 0x00cf, 0x43fa, 0x4299, 0x4367, 0x222f, 0x1212, 0x4401, 0x4388, 0x2451, 0xb29b, 0x4319, 0xb210, 0x420f, 0x40fd, 0x45b0, 0x32f6, 0x43b9, 0x1ca3, 0x1863, 0xbfac, 0xb083, 0x4385, 0x42d0, 0x422c, 0x1fe8, 0x2eaa, 0x1075, 0xb2a0, 0xbf0b, 0x42d9, 0xa9de, 0xb093, 0x42ce, 0x41eb, 0x40a4, 0x4242, 0x43df, 0xb254, 0xa2a0, 0x421b, 0x4305, 0xbf55, 0x10f1, 0xabe9, 0x41fb, 0x4333, 0x2b35, 0x405e, 0x1e82, 0x2a61, 0xbf46, 0x4234, 0xb26c, 0x1935, 0xbad9, 0x4164, 0xb241, 0x43fe, 0xb25b, 0x1df3, 0x4304, 0x1a0c, 0xba35, 0xbf0e, 0x336a, 0x3265, 0x068a, 0xba61, 0x06fd, 0x1c50, 0xb2c0, 0x4124, 0x4418, 0x41c6, 0xabd2, 0x429d, 0xba40, 0x4225, 0xb0e6, 0xb2c8, 0x4148, 0xb2b2, 0x400d, 0x1bc1, 0x24ff, 0xbf6c, 0x43f2, 0xba30, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xb95fc095, 0x46c9af8c, 0x44f999c1, 0xf75c725c, 0xeef9897b, 0x7583419d, 0xd7d5f784, 0xf2f60e6f, 0xa9e0afa9, 0x25abe41c, 0x5498383c, 0x7f430b81, 0xac6b7408, 0xf3f13b36, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x00401400, 0x00000052, 0x00004000, 0xac6b75d0, 0x000000ff, 0x00000000, 0x00144000, 0xffffffae, 0x2206c8a0, 0x00000000, 0x0000007e, 0x00000000, 0xac6b7408, 0xac6b70f0, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x185d, 0x4287, 0x405b, 0xb0b0, 0x192e, 0xb012, 0x414b, 0x436f, 0xa9b7, 0x438f, 0x108e, 0xb067, 0x418e, 0x4013, 0x3736, 0x4549, 0x4259, 0xb2c8, 0x42a8, 0x415c, 0x43a9, 0xb28b, 0xbfb3, 0xb2f8, 0x43c9, 0x0a66, 0x4216, 0xbf64, 0x1ee1, 0x42e3, 0xb2bd, 0x00ef, 0xbaf4, 0x40be, 0x4353, 0x44a9, 0x4104, 0x1b6b, 0x403d, 0xb266, 0x078d, 0xba6b, 0x3c0d, 0x05b8, 0xbfbd, 0xbaee, 0x4277, 0x42c0, 0x0958, 0x4254, 0x40a3, 0x4250, 0xa6f7, 0x4050, 0x4541, 0xb211, 0x14d3, 0x41ae, 0x46c8, 0xba25, 0x3f96, 0x0fc8, 0xb246, 0xbf45, 0x45ce, 0x130d, 0xbf00, 0x2176, 0x404f, 0x435a, 0xafd3, 0x4093, 0xb2b3, 0x4202, 0x41df, 0x4089, 0x43d4, 0x0739, 0xaa0d, 0x3f8f, 0xba47, 0x1a4c, 0xb224, 0xbf5d, 0xb20e, 0x1e5c, 0x46a1, 0xba65, 0x40af, 0x41e6, 0x4076, 0xb23a, 0x3fa5, 0x438d, 0xba57, 0xae3f, 0x4141, 0xb2f9, 0xbfe0, 0x43d0, 0x1ec1, 0xb2b6, 0xb28b, 0x1ded, 0x055e, 0x41f0, 0x40f8, 0x0692, 0x4051, 0xbf97, 0x4388, 0xacb2, 0xb04b, 0x19ce, 0x32d5, 0x0b66, 0x4368, 0x1627, 0xb288, 0xb241, 0x4042, 0x2698, 0xb0b7, 0x429f, 0x262d, 0x40b3, 0xb065, 0x42f4, 0xbf14, 0x4009, 0x1661, 0x4304, 0x430d, 0xbfb5, 0x417a, 0x428d, 0x1d15, 0x0175, 0x4282, 0x1183, 0xb22c, 0x1fbc, 0x4352, 0x0e63, 0xa412, 0xba64, 0xa285, 0x41f2, 0x41de, 0x4372, 0xbafe, 0x00ef, 0x40d8, 0xb29d, 0x41a1, 0x44b5, 0xbfaa, 0x4668, 0xbaf8, 0x4180, 0x4462, 0x1dbb, 0x4492, 0x4022, 0xba5c, 0x4070, 0xb2b6, 0x4321, 0x4314, 0x4330, 0xa81d, 0x38c2, 0x427a, 0x417c, 0x4691, 0x4111, 0x4065, 0x1c3d, 0xb02f, 0x424a, 0x4047, 0x4349, 0x411a, 0x40fa, 0xbac1, 0xbf7e, 0x36e7, 0x4189, 0x233b, 0xb032, 0x4071, 0x2ae4, 0x2b36, 0x4290, 0x310b, 0xb22c, 0x430a, 0x4204, 0xbf7f, 0x172e, 0xba71, 0x08ee, 0x43bf, 0x1b31, 0xb068, 0x42b5, 0xbf00, 0x4545, 0xb08e, 0xa37d, 0xb234, 0xb011, 0xbfe4, 0xb208, 0xa6ee, 0x0e43, 0x107e, 0x405d, 0x1c5f, 0x0f55, 0x0342, 0x44cd, 0x2b29, 0xb086, 0xa40a, 0xb21d, 0x4028, 0x426b, 0x2a20, 0x4267, 0x406f, 0x419f, 0x4274, 0x4006, 0xa492, 0x18a2, 0x4177, 0x4140, 0x42a6, 0xbf83, 0x2b7b, 0x4242, 0x40a5, 0x079e, 0x46e0, 0x1907, 0x44f5, 0x094d, 0xb295, 0x438d, 0x409c, 0x465c, 0x43db, 0x36e9, 0xba2d, 0xbf95, 0x430f, 0x4129, 0xb2d0, 0x46dc, 0xa2f1, 0xbf16, 0x433f, 0xb27b, 0x4348, 0xaf01, 0xbf35, 0xba7d, 0x1b43, 0xbaf0, 0x4006, 0x2e62, 0x1e1d, 0xae4d, 0xa032, 0x40fd, 0x42bb, 0x4395, 0x423b, 0x420c, 0x41c1, 0x42fc, 0x466f, 0x4604, 0x41be, 0x4283, 0x426c, 0xbfd3, 0xb2d1, 0x4025, 0x42e2, 0x1d04, 0x228c, 0xb052, 0x37a0, 0x1c67, 0x403a, 0xbfa6, 0x0a6e, 0xb06e, 0xb0b7, 0xaeb4, 0x30fd, 0x423a, 0x41ed, 0x4040, 0x4116, 0x3637, 0x4302, 0xb2f8, 0x1e49, 0xb2e1, 0x437a, 0x05e7, 0x408c, 0xb02a, 0xba09, 0x402c, 0x43db, 0x4062, 0x4664, 0x38d1, 0xbfcb, 0xb012, 0x4595, 0xb2d4, 0xb224, 0xba4c, 0xbf06, 0x088c, 0x1077, 0x414a, 0xafd7, 0x1496, 0xbfa0, 0x45b0, 0x1c0c, 0x3354, 0x43e0, 0x1eab, 0x4281, 0x45c1, 0x406b, 0x075b, 0x3898, 0x4242, 0x430f, 0x4612, 0x40fb, 0xb03f, 0xb0f6, 0x400e, 0x4316, 0x418c, 0x270c, 0xbf93, 0xbac5, 0x18ed, 0xa70b, 0x42f6, 0x4238, 0xbf76, 0x3431, 0x4200, 0x46b8, 0x411c, 0x43a6, 0x439b, 0xbaff, 0x1a05, 0x404b, 0xb238, 0xb2f6, 0xba00, 0x0df6, 0x4021, 0x409b, 0xba4f, 0x1fb3, 0x252f, 0x4018, 0xa721, 0x0f4b, 0x40b9, 0x4211, 0x40d7, 0xba5c, 0xb0f1, 0xbfdf, 0x2384, 0xba0a, 0x435a, 0x42d7, 0x4653, 0x401e, 0xac83, 0x4073, 0xba4b, 0xb0df, 0x0b60, 0x4316, 0x4624, 0xa092, 0x0bf5, 0xbfb0, 0xb057, 0x42d6, 0x42fe, 0x156c, 0x41d6, 0xb0bb, 0x13d3, 0x4248, 0xbfbb, 0x437d, 0x0a8b, 0x4317, 0x40db, 0x42b3, 0x40cd, 0x4095, 0x4362, 0x42d2, 0x4475, 0x4137, 0xae4f, 0x400e, 0x0014, 0xbf7c, 0x420f, 0x1b5a, 0xbf70, 0x18ef, 0xb07e, 0x44fb, 0x43cf, 0x404b, 0x4171, 0xb0e2, 0xba20, 0xa7cc, 0x4580, 0xbf0e, 0x385a, 0x43d1, 0xac36, 0x434c, 0xb0d6, 0x4284, 0x12a5, 0x43bd, 0xb26e, 0x438a, 0xb059, 0x4370, 0x18bd, 0x362f, 0x4309, 0xb0fc, 0xb2d3, 0xbf13, 0xb23e, 0x4285, 0xb0fb, 0xba0b, 0x41b6, 0x04d1, 0xb249, 0x1ee9, 0xb2aa, 0x4340, 0x4285, 0x43e9, 0xbac6, 0xba62, 0x40d2, 0xb23d, 0xb0f0, 0x1cca, 0x083a, 0xb0b2, 0x43c6, 0x18d1, 0x2883, 0xbf66, 0x4121, 0xbaeb, 0x402e, 0x40c5, 0xbaf2, 0x41a9, 0x1839, 0xba61, 0xb0cf, 0xb2ae, 0x40ef, 0x405a, 0xb207, 0x1bb5, 0xab4a, 0x4572, 0x41f5, 0x33d6, 0x40a5, 0x1a06, 0xb2c3, 0x403c, 0x4341, 0xa0bd, 0x42a7, 0xb276, 0x1daf, 0x37ad, 0xbf5e, 0x3150, 0x43eb, 0x4213, 0xbf66, 0xb21d, 0x429b, 0x4004, 0x4012, 0x41ab, 0x467e, 0xb2d5, 0x4213, 0x4229, 0x0fbc, 0xa2a8, 0xbfd8, 0x4042, 0x32a7, 0x3304, 0xbf8a, 0x125d, 0x3c49, 0x4596, 0x4060, 0xa368, 0x448b, 0x409c, 0x2f24, 0xb0ca, 0x42ff, 0xb22a, 0x0971, 0x469a, 0xb260, 0x4243, 0x2673, 0x0f84, 0xbf36, 0xa7f5, 0xa907, 0xb081, 0xb2b0, 0x4157, 0x1ad4, 0x4360, 0x1c8f, 0xba6a, 0x40ac, 0x429a, 0x425f, 0xb062, 0xb2cd, 0x4228, 0x1806, 0x1d41, 0xbf0d, 0xba56, 0xbf60, 0xb255, 0x1952, 0x19ca, 0x2d41, 0xb0a0, 0x45e8, 0x3a49, 0xba58, 0x41cc, 0xbf00, 0xb283, 0x462e, 0x18ad, 0xbaf6, 0x26f5, 0x1680, 0x41bb, 0xb2d7, 0x40a9, 0xbf59, 0x40a2, 0xba67, 0x0b0a, 0xa067, 0x1d7b, 0x42ad, 0x2159, 0xbf2a, 0xa8ac, 0x421b, 0x1c35, 0x4381, 0x424c, 0x4216, 0xba6b, 0x4262, 0x432d, 0xb2f3, 0x402b, 0xb29f, 0xb032, 0x0bae, 0x40ba, 0x41a6, 0x4063, 0x405b, 0xb051, 0xbf9e, 0x4242, 0xb2cd, 0x2966, 0x4388, 0xb281, 0xb24f, 0x442a, 0xbae4, 0xb0d7, 0x25c2, 0x4013, 0x308b, 0x4231, 0x3ca8, 0x1fc9, 0x4563, 0x4420, 0xbaf2, 0x0a8b, 0xa349, 0xbada, 0x4188, 0x04c9, 0xb0c5, 0x1db0, 0xb079, 0x4204, 0x4305, 0xbf53, 0xa0e7, 0x434e, 0x1fb1, 0xb079, 0x1c64, 0x43e0, 0x1c51, 0x3ea3, 0x428e, 0x409d, 0x4323, 0x43b9, 0x42fd, 0x22fb, 0xb228, 0x41bb, 0x4320, 0x40f4, 0x2bec, 0x45cc, 0x419d, 0xb2d8, 0x325d, 0x44c3, 0x201f, 0x41d5, 0xbf59, 0xbf70, 0x2ef4, 0xb27b, 0xb03b, 0xa979, 0x24f4, 0xba21, 0x44a8, 0x028d, 0xb207, 0xa84c, 0x409d, 0xbf80, 0xbf26, 0xb2ef, 0xb2c9, 0x2659, 0x430f, 0xb2eb, 0x41d1, 0x10fa, 0x2baa, 0x09a6, 0xb02f, 0xb2c9, 0xba2f, 0xba67, 0x4211, 0xbf57, 0x10f3, 0x185c, 0x419a, 0x4271, 0xad46, 0xb077, 0x4627, 0xb2e2, 0x1a5e, 0x42a8, 0x422f, 0xb27a, 0x40f5, 0x4085, 0x0255, 0xa249, 0x4270, 0x4392, 0xb0ff, 0xb2e3, 0x46b2, 0xb2d1, 0x4001, 0x4639, 0x1fa7, 0x42f2, 0xa470, 0xbaf5, 0xbfdd, 0x1d21, 0x3f09, 0x2d80, 0x1aa9, 0xbf1a, 0x4061, 0x400e, 0xae29, 0xa3a0, 0x4010, 0x1f99, 0x121b, 0x41a0, 0xb2af, 0xbad1, 0xbf1c, 0xb298, 0x3db6, 0xb05b, 0xba33, 0xa3d9, 0xa2e3, 0x4162, 0x4188, 0x413b, 0x2805, 0x40a0, 0x4151, 0x2a6d, 0xbae7, 0x444f, 0x430e, 0x413a, 0x1767, 0xb05b, 0x303f, 0x001c, 0x43b9, 0x28e3, 0x3dad, 0x420e, 0x4205, 0xbf03, 0xb21b, 0x022e, 0x44d1, 0xb283, 0xb2c7, 0x0c1a, 0x4107, 0x198e, 0x43e8, 0x1c24, 0x3d5c, 0xa5ee, 0xbfd0, 0xbf43, 0x2a0c, 0x40ac, 0x4414, 0xa9ce, 0xbf2c, 0xbac9, 0x396d, 0x42fa, 0xbf0b, 0x4292, 0x43ba, 0x424a, 0xb25f, 0x42d6, 0x4188, 0x2f30, 0xb081, 0x38a0, 0x4213, 0x4130, 0x415f, 0x2c4c, 0xadab, 0xba23, 0xadc1, 0xb25f, 0xabf8, 0x45cd, 0x416d, 0x40d1, 0x403b, 0x1dc9, 0x4150, 0xba03, 0xb076, 0xbaf1, 0xbf2d, 0x40dc, 0x0d13, 0x18a9, 0x43a4, 0xb076, 0xba07, 0xb27d, 0x0b40, 0x3d88, 0x40a7, 0x0533, 0x2c71, 0x41cc, 0x42f9, 0x1ef9, 0x4583, 0x41e2, 0xba3b, 0xba44, 0x41b1, 0x42f2, 0xbfdf, 0xb23d, 0xbfc0, 0x071e, 0x19b6, 0x1a1f, 0x417f, 0xb268, 0x4132, 0xbaf6, 0x3af3, 0x1f37, 0x4352, 0x437d, 0xbfa2, 0x05b2, 0x43cb, 0x42a5, 0x43e5, 0x42ac, 0xa525, 0x18ad, 0xbace, 0x2d15, 0x1f1e, 0x01c6, 0xa4ad, 0x433a, 0x436a, 0x1e2a, 0x4389, 0x1898, 0xb2a5, 0x1a20, 0x42ec, 0xa6ed, 0xb038, 0xbacd, 0xbf8e, 0x41b9, 0x43ea, 0x42ba, 0x30cd, 0xaff4, 0x42fb, 0x460d, 0xbfa0, 0x1f87, 0x122b, 0x1b3f, 0x1cd8, 0x4109, 0x1cf9, 0x4146, 0x27b9, 0x4637, 0x43f7, 0xba68, 0x459b, 0xbf8d, 0xbf60, 0x1be6, 0x4691, 0xb267, 0x4062, 0xb2b4, 0xb2ca, 0x4198, 0x4331, 0x2b7d, 0xbf60, 0xbadc, 0xb221, 0x1ca0, 0xb200, 0x4007, 0x1e11, 0x43f6, 0x43af, 0x2b4b, 0x41d0, 0x21a9, 0x43d0, 0x43d1, 0xbfd2, 0x410a, 0xb0a3, 0xb266, 0x4153, 0x42fa, 0x371e, 0x4037, 0x1c33, 0x43de, 0xb0ef, 0xb059, 0x43fe, 0x410f, 0x0ed6, 0xb2ef, 0xbf33, 0xb080, 0xb023, 0x416c, 0x4390, 0x4484, 0x059f, 0x03c7, 0x4176, 0x425b, 0x42b5, 0xbf0b, 0x4680, 0x4202, 0x1ee9, 0x456a, 0xaae9, 0x4333, 0x1c96, 0x20f2, 0x3d1d, 0x434e, 0x39dd, 0x40c9, 0x433b, 0xbf21, 0x444b, 0xb295, 0x3462, 0x3275, 0x4110, 0xb214, 0x1edc, 0x4136, 0xbad1, 0x43a5, 0xb2ae, 0xadf4, 0x40ac, 0xa037, 0x4199, 0x41d7, 0x426e, 0xb2de, 0x43eb, 0xba38, 0xb253, 0x4092, 0xbfba, 0x413c, 0x4238, 0x41a0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd919f6a3, 0x9e176573, 0x3795cb44, 0x666e776e, 0x370d6712, 0xb9f52030, 0x59d84426, 0xcda6b649, 0x3c751555, 0x13a5d9a9, 0x68b254f2, 0x4c783b43, 0xb4ebd830, 0x01180a73, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x0080fb0e, 0x005002e4, 0x00000000, 0x00000024, 0x00000000, 0x011021db, 0x0000003d, 0x0ffb8000, 0x0001470c, 0xfff80698, 0x00000003, 0x4c784f07, 0xb4ebd7a0, 0x01101e0b, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4139, 0x0239, 0x4470, 0x43b0, 0xa91a, 0xb068, 0x1c4a, 0x2666, 0x43c6, 0xa8cd, 0x4184, 0xbafc, 0xb29e, 0x42dd, 0x1781, 0x4033, 0x35f4, 0x31c2, 0xa3f5, 0xbf3a, 0x1884, 0xbafc, 0x424b, 0x42ab, 0x0955, 0x432a, 0x43f5, 0x410f, 0x4694, 0x401b, 0x41c7, 0xb058, 0x419f, 0x41f2, 0xb2fc, 0x1fd1, 0x464b, 0xbfb0, 0x41c5, 0x2b5f, 0x4330, 0x4152, 0xbfbf, 0xbf70, 0x43fb, 0x0114, 0x439b, 0x4376, 0x35bb, 0xa42b, 0x4130, 0x25df, 0x2ee3, 0x42a8, 0x4128, 0x42df, 0x4220, 0x43ec, 0x2ecd, 0xb291, 0xbf4f, 0xab48, 0x41da, 0x4198, 0x4214, 0x40c7, 0x3fb9, 0x424e, 0x414e, 0x01e3, 0x379f, 0x44b2, 0x4166, 0x42e8, 0x4208, 0xb2aa, 0xbf6f, 0xb05e, 0x40da, 0x463a, 0x193e, 0x42ef, 0x40ed, 0x1cc4, 0x4272, 0x1550, 0x42b1, 0x3bbb, 0x227a, 0x449c, 0x42b0, 0x41bf, 0xa310, 0xb0a6, 0x439e, 0x42a2, 0x4392, 0xbf1d, 0x2af5, 0xb2b1, 0x426b, 0x43dd, 0x4204, 0x466b, 0xb060, 0xbac4, 0xb0d2, 0x4320, 0x4006, 0x4303, 0xbf83, 0xa242, 0x42e1, 0xba65, 0xa4c1, 0xbfe0, 0xb26a, 0x3dc0, 0x1e3f, 0x423d, 0x439e, 0xba4e, 0xab3b, 0xba29, 0x4606, 0x41f2, 0xbac9, 0xb23d, 0xaac3, 0x19e2, 0xbf78, 0x1ad9, 0x4089, 0xba78, 0x46a3, 0xa084, 0xb27c, 0xbf72, 0x1439, 0x4000, 0xb275, 0x40cc, 0x4252, 0xbf7b, 0xbad4, 0x43d2, 0x1ce1, 0x4562, 0xb20b, 0x2fbd, 0xbadd, 0xba61, 0x2a48, 0x405a, 0x4343, 0xbf42, 0xb233, 0x4261, 0x423a, 0x42d4, 0x4330, 0x46f2, 0xbaf3, 0xb240, 0xba69, 0xba78, 0x091c, 0xb296, 0xbae4, 0x2223, 0x40f9, 0x42db, 0x1838, 0xb21f, 0x03b3, 0x43a0, 0xb2eb, 0xbf0f, 0x42ac, 0xb22e, 0xb013, 0xa332, 0xb2d0, 0xb271, 0x110f, 0x2fc7, 0x4668, 0x43dc, 0x41ef, 0x3a07, 0xbacf, 0x4158, 0x4129, 0x00ac, 0xb254, 0x416a, 0x4312, 0xad39, 0x4070, 0x1d12, 0x43e3, 0xaf2e, 0xb2fb, 0x250c, 0xa0c5, 0xbf96, 0x3811, 0x43a2, 0xa7d3, 0x1a37, 0x41b5, 0x414d, 0x43b6, 0xbacc, 0xa587, 0xbf35, 0x1ea8, 0x3706, 0xb0d5, 0x42f9, 0x405f, 0x426e, 0x1c33, 0x4369, 0xb01e, 0x4611, 0xa7cc, 0xbf9d, 0x1e4f, 0x1939, 0xbfe0, 0x43e7, 0x445d, 0xbfbc, 0x4146, 0x4093, 0xb255, 0x0325, 0x42bf, 0xbfe1, 0x45cb, 0x1a8f, 0xb2ca, 0x1912, 0x007c, 0xba33, 0x3771, 0x4355, 0xaa43, 0xbf00, 0xbad0, 0xbf93, 0x1e05, 0x1bb5, 0xb28c, 0x296d, 0xbf9c, 0xafa4, 0xb2e5, 0x4672, 0xbf27, 0xba7a, 0x419e, 0x4047, 0x4025, 0xb238, 0x4046, 0xb211, 0x3e6a, 0x4058, 0xbf32, 0x45c6, 0x3629, 0x42f1, 0x41da, 0xb2c1, 0x4441, 0x412d, 0xb09f, 0x43d8, 0x41c0, 0x40cb, 0x25d4, 0xa604, 0xa1e9, 0xb2b3, 0x42a5, 0x42c7, 0xb223, 0xb2f4, 0x403d, 0x41f9, 0x417e, 0xbf90, 0xbf3c, 0x1adc, 0x41c3, 0x0b57, 0xa98b, 0xace1, 0x19d8, 0x42bc, 0x141e, 0xbfb6, 0xb0bf, 0xba07, 0x425b, 0x3db1, 0x4186, 0x4052, 0x42c9, 0x114a, 0xb2c7, 0x4461, 0xbac5, 0x40f0, 0xa656, 0xa678, 0xb2a0, 0x1fdb, 0x186e, 0x4549, 0xb0c9, 0x2198, 0xb048, 0x4316, 0xa29b, 0xb2df, 0xb269, 0x1cf6, 0xbf8a, 0x418e, 0x012f, 0x1ff4, 0xb060, 0xad7c, 0x42a8, 0x42dd, 0x420e, 0x4153, 0xae8b, 0x03f1, 0x439f, 0x41f3, 0xbf83, 0x1395, 0xb281, 0x41f3, 0x19e3, 0x4002, 0x2ac4, 0x3a17, 0x4225, 0x4007, 0x4201, 0x45b4, 0xb29c, 0x410f, 0xa3ad, 0x43d4, 0xb2b9, 0x1d09, 0xb082, 0x4335, 0xbf58, 0xb297, 0xb0c3, 0x4240, 0xba79, 0xb230, 0xba2a, 0xbf57, 0xb0a6, 0xa975, 0x1f5b, 0x42e8, 0x190f, 0xbfc3, 0x420a, 0xb21a, 0x2c87, 0x417e, 0xbfe0, 0x40b1, 0x339a, 0x2f4c, 0x4353, 0x1b7c, 0x4349, 0x4335, 0x41c6, 0x4283, 0xa0bc, 0xa2da, 0x413b, 0x42c5, 0xbf8b, 0x4572, 0x061e, 0xbf90, 0x4159, 0xb280, 0xb0ef, 0x204a, 0x1dfe, 0x068c, 0x4122, 0x43be, 0x4286, 0xba1e, 0x4333, 0xb272, 0xa671, 0x4002, 0x42ed, 0x14b2, 0x32d9, 0xbf2c, 0x358d, 0x4615, 0x410b, 0x3ecd, 0xb223, 0x2f09, 0x1b6c, 0xbf2c, 0xbae5, 0x1cee, 0xb2fe, 0x42ea, 0xb045, 0x41ca, 0x4016, 0x10c6, 0x0828, 0xabe4, 0x123d, 0xa82a, 0xb04d, 0x45b0, 0xbfb0, 0x4118, 0xba15, 0x442d, 0x4098, 0x2145, 0x1a40, 0x1935, 0x394e, 0x45c2, 0xbf19, 0x402e, 0xb2ba, 0x42cb, 0x40d9, 0x4193, 0x41e8, 0x4261, 0x167e, 0xbad3, 0xbf1a, 0xb24d, 0x09c9, 0xb27f, 0x0295, 0x417d, 0x4362, 0x40a0, 0x4636, 0xba62, 0x2a28, 0x405d, 0x1ac1, 0x2e7e, 0x4027, 0x4222, 0xba7d, 0x429c, 0x40d6, 0x424b, 0x4430, 0xb042, 0xa73d, 0x406e, 0x3722, 0xb2a9, 0x4014, 0x44b1, 0xbfac, 0x4035, 0x4326, 0x0c54, 0x4007, 0xba2c, 0x43a7, 0x4333, 0x4092, 0xa7c2, 0xbf01, 0x27cb, 0xb047, 0x1a86, 0xbade, 0xb2b7, 0x2147, 0x0e71, 0xb295, 0xba38, 0x4388, 0x159c, 0x416b, 0xba1a, 0xb0d2, 0x0162, 0xba33, 0x40cd, 0x465c, 0x420f, 0x368f, 0x42bf, 0xbf60, 0x40f3, 0x4002, 0x0fed, 0x4165, 0x415c, 0xba29, 0xbfcd, 0x42b2, 0x40d4, 0xb0ac, 0x43f7, 0x12b0, 0x416d, 0x2be8, 0x4103, 0x408e, 0x43e6, 0x40d5, 0x427a, 0xba11, 0x1c74, 0x40b9, 0x4066, 0x4254, 0x4081, 0x2821, 0x4003, 0x44ab, 0x431a, 0x4282, 0x4569, 0x2f84, 0xb2ef, 0x4340, 0xb218, 0xbf8f, 0xb024, 0x1c87, 0x289f, 0x43d9, 0xba0a, 0x421d, 0x44f0, 0x42a2, 0x41b4, 0xa51b, 0x1c28, 0xa535, 0x40cb, 0x44b9, 0x40ed, 0x0bbe, 0x413e, 0x1eb8, 0x41ad, 0x1dac, 0x3f25, 0xb0ec, 0x12e7, 0xab9a, 0xbf77, 0xbae9, 0x1b49, 0x40a5, 0x40d7, 0x4392, 0x43e6, 0x190e, 0x40b0, 0xb2dc, 0x425f, 0x41ad, 0x10b1, 0x40f4, 0x43b7, 0x4037, 0xb03c, 0x1a72, 0x407b, 0xba50, 0x466e, 0xbf11, 0xbad4, 0xa393, 0xaa42, 0x41c5, 0x413c, 0x410a, 0xba1f, 0xba17, 0x41f4, 0x2138, 0xbfe0, 0x1ace, 0xba65, 0x40cc, 0x44e1, 0x1646, 0x4228, 0x43b7, 0xa7cf, 0x4160, 0xbfa8, 0xba18, 0x43f1, 0x1ed8, 0x4355, 0x2fce, 0xbacb, 0x41c5, 0x42c2, 0x44f8, 0x1f13, 0xbf5f, 0xaa2a, 0xb2fd, 0x1405, 0x25d1, 0xa644, 0x4134, 0x4280, 0x19f2, 0x0a34, 0x1b5e, 0x4137, 0xba1e, 0xba4c, 0x424c, 0x43d3, 0xb062, 0xb2b4, 0x43c3, 0x42a4, 0xb200, 0x4219, 0x4646, 0xa445, 0x4248, 0xbfe0, 0xbf05, 0x415f, 0x1832, 0x439d, 0xaaa6, 0xb234, 0x43b7, 0xb01a, 0xba02, 0x1f57, 0x43c9, 0x46c0, 0x4558, 0x4116, 0x1707, 0xb0b8, 0xbfc0, 0x41c0, 0x4032, 0x112c, 0x44e4, 0x4290, 0x04f6, 0xa289, 0x4226, 0x43bb, 0x1ee1, 0x434e, 0xbf66, 0x4210, 0x425f, 0xa705, 0x1aff, 0x31fd, 0xbf0e, 0x4295, 0x4107, 0xb283, 0x4067, 0x1dec, 0xb0b8, 0xb089, 0x2a04, 0x4172, 0xbfc0, 0x4217, 0xb2ea, 0x41a8, 0x41a2, 0x1610, 0x4294, 0xb016, 0x406e, 0x4281, 0x42a7, 0x1f7a, 0x1866, 0x4047, 0x4672, 0x20bc, 0x420b, 0x198a, 0xbf8a, 0x43f3, 0x3aed, 0xbfb0, 0x4612, 0x408c, 0x43e6, 0x0f02, 0x4214, 0xbfb5, 0xae8e, 0x408a, 0x41d9, 0x02f0, 0x341b, 0xb07d, 0x413b, 0xb0d9, 0xb2c5, 0x42f0, 0xb2f4, 0xba45, 0xa723, 0x42f1, 0x19aa, 0x4005, 0xb00b, 0x42ec, 0x4239, 0x43c4, 0x40ed, 0xaaa1, 0x1ff0, 0x4215, 0xb005, 0x414d, 0xb037, 0x43d1, 0xbf5b, 0xb24a, 0xb2cc, 0xb0e1, 0xb25e, 0x4298, 0x427a, 0xb0e5, 0x42be, 0x1cf1, 0x43b3, 0xb0b9, 0x4242, 0xbf0b, 0x435b, 0x14c6, 0x43d8, 0xb2e2, 0x414c, 0x4165, 0xad8d, 0xb2f3, 0xb053, 0x42a1, 0x4301, 0xa3c4, 0x43fe, 0x2e37, 0xb019, 0x4112, 0xba58, 0x40c9, 0x4204, 0x1870, 0x4592, 0x4360, 0x24a0, 0x42ab, 0x4066, 0x404b, 0x4144, 0x4252, 0x422c, 0xbfa5, 0x1caa, 0x02c7, 0x09fa, 0x4364, 0x43fe, 0x3b92, 0x4616, 0x4376, 0x1be2, 0x40ee, 0x34a4, 0x2c90, 0xba42, 0x42de, 0x435b, 0xb272, 0x413d, 0x1f64, 0x4099, 0x42b2, 0x43bb, 0x43fb, 0xbfd1, 0xba53, 0xb067, 0x42a2, 0x0f18, 0x43cc, 0x1f8e, 0x1dfe, 0xb28b, 0xbad5, 0x188f, 0x4363, 0x4299, 0x12b3, 0x0831, 0x41fb, 0x40fd, 0xbfa9, 0xba30, 0x4183, 0x42ad, 0x40b2, 0x418b, 0x42ea, 0xb0fb, 0x42aa, 0x42df, 0x1f34, 0x0bd2, 0x3df7, 0xb215, 0xba60, 0xb28d, 0x30e7, 0xb2a0, 0x4567, 0x42a8, 0xbfc3, 0xb0a0, 0x41c7, 0x0f8a, 0x16a1, 0x4168, 0x445e, 0xb236, 0x4271, 0xbf39, 0x4312, 0xbad2, 0x42e4, 0x1cb2, 0xb2c7, 0xbaee, 0x431c, 0x42dd, 0x42f2, 0xb262, 0x403a, 0x1dfb, 0x4330, 0x36a0, 0xba46, 0xb224, 0xbf56, 0x1b15, 0x3160, 0x40e6, 0xb025, 0x1390, 0x42a7, 0x078b, 0x4263, 0x141e, 0x0633, 0x4281, 0x40f5, 0x422b, 0x45f1, 0xba2b, 0x1a8a, 0x41a8, 0xb275, 0xb2a3, 0xb22a, 0xbf7a, 0x4540, 0x43da, 0x42da, 0xaa06, 0xb284, 0x3f4f, 0x42ae, 0x40a2, 0xbfb1, 0x4382, 0xbaf0, 0x1db0, 0x43af, 0x0b87, 0x433d, 0x461a, 0x2502, 0xb234, 0x3130, 0xb2b7, 0x43d2, 0xba2f, 0x37df, 0x3ab5, 0xafcb, 0x0f62, 0x46f8, 0x404f, 0x4560, 0x43fd, 0xbf59, 0x4237, 0x414e, 0x3b8d, 0x43e4, 0x4370, 0x2d12, 0xbf2b, 0x42cd, 0xba1a, 0xb2f9, 0xb2b0, 0xb0ea, 0xbac2, 0xb284, 0xaa89, 0x4041, 0x22ca, 0xb229, 0xafec, 0x438d, 0xbae3, 0x4314, 0x0739, 0xb229, 0xb2f4, 0xbf45, 0x189c, 0x43db, 0x4116, 0x409d, 0xb2b0, 0x4227, 0xb297, 0x469b, 0x31d0, 0x2360, 0x462f, 0xbacf, 0x4277, 0x305e, 0x2d0f, 0x381c, 0xbfcf, 0xb2f4, 0x42db, 0x435f, 0x3aaa, 0xb0ee, 0x44f4, 0x1b64, 0x4372, 0xb284, 0x40cd, 0x46d4, 0x13bf, 0xba5d, 0x4092, 0xba52, 0x41e2, 0x1dc6, 0xbf7d, 0x1eb2, 0x2606, 0x401f, 0x1d2f, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xb856fb9b, 0x192a6db9, 0x8500700e, 0x51377f3b, 0xcf8d1a8e, 0xeb924976, 0xf20bcc21, 0xd216b99a, 0xee51a075, 0x8bdc3aa5, 0x4c0f0144, 0xdedd3f79, 0xfe5590cd, 0x20762c18, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0x00010041, 0x000000d0, 0x00010046, 0x00000060, 0x00000041, 0x00006000, 0x00000006, 0x00000000, 0x00001756, 0xad5388b8, 0x00000000, 0x000058ca, 0x00000000, 0x20762a7c, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x404c, 0x424a, 0x2986, 0xb245, 0x4373, 0x4073, 0x4006, 0x41df, 0x02dc, 0x3691, 0x42c5, 0xb2bf, 0x4020, 0x422c, 0x396a, 0xa360, 0xbf6f, 0x1d2b, 0x2d8e, 0x1815, 0x42f8, 0x41e0, 0x0084, 0x42c0, 0x43a4, 0x3d65, 0x11d0, 0xb2ee, 0x21d7, 0xbf8b, 0x3e83, 0x406a, 0x42ba, 0x4370, 0x446d, 0x3c88, 0x0c4e, 0x3ac9, 0x4284, 0xb2a9, 0x3c89, 0xb21a, 0x42df, 0x43e9, 0x4079, 0x1879, 0x40ea, 0xbafd, 0x435f, 0x40c2, 0x2c52, 0xba2c, 0xba4d, 0x45ba, 0xb288, 0x4228, 0x40cd, 0xbf67, 0x4092, 0x41bf, 0xba76, 0xb25f, 0x1c56, 0xb23c, 0xba28, 0x4063, 0x4083, 0xba20, 0xbad0, 0x40c5, 0xb245, 0xba73, 0x37df, 0x405d, 0x3e0c, 0xba79, 0xb241, 0x425a, 0x40c9, 0x4031, 0xb2b4, 0x4288, 0x4333, 0xbf95, 0xb20d, 0x42d8, 0xb0e6, 0x41c3, 0xbad8, 0x417b, 0x1c10, 0x030e, 0x403c, 0xb0e2, 0x12a0, 0x4451, 0x420b, 0xb20c, 0x3174, 0xbf60, 0xbada, 0xb2fd, 0xbf4a, 0xb22e, 0xb0ea, 0x19bf, 0x41cd, 0xbfbc, 0x412a, 0xba0d, 0xba38, 0x4092, 0x42af, 0x1db0, 0x41b5, 0xb281, 0x4059, 0xba6f, 0x410d, 0x4421, 0x410a, 0xa37d, 0xb289, 0x2b11, 0xad56, 0x383f, 0xb20c, 0xbfb3, 0x428f, 0x1f82, 0xba34, 0xb0ef, 0x1c73, 0x401e, 0xba6a, 0x40c8, 0x4554, 0xb0b6, 0x1d62, 0xb092, 0xb000, 0xbf7c, 0x40c0, 0x3aca, 0x1db0, 0x4124, 0x461c, 0xba2f, 0x1a4d, 0x1f44, 0xbfd9, 0x33dc, 0xb2c9, 0x41fe, 0x1921, 0xba75, 0x40ea, 0x424e, 0x43ca, 0xa25a, 0x423e, 0x4175, 0x410a, 0x41e1, 0x41c2, 0xbfca, 0x1f2c, 0xb2ce, 0xae78, 0xb262, 0x4045, 0x4287, 0xb24e, 0x1d5e, 0x4092, 0x2e56, 0xbacd, 0x400e, 0xba42, 0x40f4, 0xbafd, 0x4482, 0x40fc, 0x2f63, 0xb01c, 0x408d, 0x4008, 0x4250, 0xa846, 0xb2a7, 0x376b, 0xb012, 0xbad0, 0xbf7c, 0x4382, 0xb23e, 0x414c, 0x4091, 0xb0ab, 0x40fe, 0x4076, 0x458c, 0xa397, 0x445a, 0xb267, 0x43d7, 0x42c5, 0xa924, 0x41c9, 0x431d, 0x4207, 0x4182, 0x4012, 0x0c79, 0xb26a, 0x1b23, 0xabb2, 0x40df, 0xbf4a, 0x27b0, 0x4467, 0xa236, 0xba10, 0x4386, 0x42c2, 0x41d5, 0x4665, 0x41bb, 0x41d0, 0xba31, 0x4388, 0x1d9c, 0x2d56, 0x420d, 0x432e, 0xba31, 0x1d22, 0x4428, 0x419b, 0x1cd7, 0xb081, 0x4327, 0x4162, 0x42d8, 0x1947, 0xbf2c, 0x43cb, 0xb250, 0x4012, 0x3734, 0x4214, 0xa5bb, 0x4611, 0xba60, 0x2aed, 0xba05, 0x43a1, 0x431d, 0x0608, 0xbade, 0xa596, 0x4312, 0xb0b8, 0xb249, 0xbf6d, 0x42ac, 0xbfe0, 0x42e8, 0xb284, 0x42cd, 0x380e, 0x405b, 0x42e8, 0xb07d, 0xbad9, 0x43ca, 0x4282, 0xb283, 0x4353, 0x1ca4, 0xba3e, 0x41f3, 0x18a3, 0xb2da, 0x41d5, 0x416a, 0x2f3e, 0x46ec, 0xbf9c, 0x430f, 0x434e, 0xb2dd, 0xb27d, 0x1064, 0x43f0, 0xb2c9, 0x15a6, 0x28ab, 0xa5f9, 0x08db, 0xbfbe, 0x41ed, 0x41f9, 0x4385, 0x4197, 0x0b37, 0x418b, 0x133a, 0xb0f9, 0xba40, 0x434b, 0xb2b2, 0xa303, 0x464d, 0x1c7a, 0xbfe0, 0x46ac, 0x028d, 0x445c, 0xb034, 0xbf4f, 0x1af3, 0x1e0d, 0x4148, 0xaf26, 0xb01e, 0x1575, 0xba61, 0x29a4, 0xb238, 0x40ca, 0xbf5e, 0xab8b, 0x351a, 0x4121, 0xba67, 0xbfd2, 0xacfe, 0xba11, 0x4381, 0x186f, 0x36ef, 0xbae7, 0x1389, 0x1a97, 0x42be, 0x4123, 0x21ca, 0xb2ee, 0xb2b4, 0x463f, 0x2825, 0x449c, 0x4225, 0x4271, 0x1f0c, 0xba51, 0x43c4, 0x4148, 0x017d, 0xbf7b, 0x43e6, 0x41ee, 0xb255, 0x43e6, 0xb048, 0x4033, 0x4282, 0x412e, 0xbae8, 0xaede, 0x4125, 0x41c2, 0x4229, 0xb014, 0x403a, 0xb001, 0xba7c, 0xbada, 0x1dc4, 0x25c1, 0xa5f1, 0x43c5, 0xb293, 0x4011, 0x2dea, 0x1912, 0x43cc, 0xa25f, 0xba41, 0xbfaa, 0x41db, 0x4060, 0x1be4, 0x09de, 0x3dd2, 0x1ca8, 0xb200, 0x460d, 0x4066, 0x02af, 0x432e, 0x1dce, 0xb20f, 0x4468, 0x4058, 0xb037, 0x4137, 0x4158, 0xbf96, 0x39fe, 0x40a2, 0x417e, 0xa421, 0x3a98, 0x12e7, 0x4106, 0xb0fc, 0x2564, 0x40e0, 0xb0a6, 0x0a30, 0x42eb, 0x0ad5, 0x42ec, 0xbacc, 0xbff0, 0x13cc, 0x1bdc, 0x33ae, 0xbf90, 0xba18, 0x42f5, 0xb084, 0x410c, 0x4571, 0xbf94, 0x422b, 0x424b, 0x43a2, 0x433b, 0xba17, 0x1b36, 0x4221, 0x40e2, 0x4247, 0x1e2f, 0x3746, 0x43ef, 0xbf26, 0x1a41, 0x40ae, 0xb2f1, 0x4348, 0x43ce, 0x4128, 0xb27e, 0x400f, 0x41c4, 0xb023, 0x4054, 0x40cb, 0xb260, 0x41c9, 0x3d51, 0x42bf, 0x36f6, 0x1d00, 0x2cc0, 0x412b, 0xbf08, 0x03d1, 0xb280, 0x432e, 0xb0a1, 0xb2d1, 0xa60e, 0xb229, 0x418b, 0xbfe2, 0x438a, 0xba58, 0xa4a8, 0x4393, 0xb2ce, 0xbfd0, 0x447d, 0xb0fc, 0x42c8, 0xba09, 0x43eb, 0xbf42, 0xb24c, 0x19f5, 0x426b, 0xb2a7, 0x1baf, 0x1b6b, 0x4276, 0x0857, 0x22f0, 0xb238, 0x41fc, 0x440b, 0xbf88, 0x2db9, 0x1b79, 0x4117, 0xba00, 0x04c4, 0xbafa, 0x42fa, 0x0f86, 0x439c, 0xa156, 0xb274, 0x4223, 0xbfcf, 0x10c5, 0xb28b, 0x43df, 0x41f3, 0xbf8d, 0x43f8, 0x42b4, 0xb031, 0x0817, 0xb261, 0xbfe0, 0x4137, 0x1da3, 0x404b, 0xbae6, 0xb254, 0x412f, 0xbad5, 0xb295, 0x444e, 0xb28e, 0xb279, 0x406f, 0x45ea, 0xb2db, 0x1d07, 0xbfb3, 0x3b0e, 0x4138, 0x2a53, 0x421f, 0x40bd, 0x43cb, 0x430f, 0x21b6, 0x18d6, 0x43a4, 0xb04e, 0x43a3, 0x420e, 0x467e, 0x4134, 0xb060, 0x44fc, 0x41ed, 0xbff0, 0x43a6, 0x217f, 0xbad4, 0x417d, 0x42e7, 0x3065, 0x1ce3, 0x4016, 0xb218, 0x4219, 0xbf1d, 0x3fb0, 0x2c2e, 0x089c, 0xbaf3, 0x42ef, 0xb26b, 0xbf3d, 0x39ec, 0x4375, 0xb2e2, 0x191e, 0x433f, 0x38b8, 0x4217, 0xb074, 0xb272, 0xbfa4, 0x1bb0, 0x223f, 0xbaf3, 0x4192, 0xba64, 0x4278, 0xbace, 0x01b2, 0x431a, 0xa920, 0x40bf, 0x4498, 0x4331, 0x43fa, 0x4212, 0x41d4, 0x0257, 0x4136, 0x3ca6, 0x4229, 0xbad2, 0x199e, 0x08c5, 0x406e, 0x4480, 0xb2c0, 0xbf70, 0xbf23, 0x4295, 0x46d2, 0x159c, 0x40ca, 0x281b, 0x4333, 0xbfab, 0x1fdd, 0xba6f, 0x3ac7, 0xba26, 0x1da2, 0x4216, 0x1ca9, 0xb263, 0x408e, 0x4491, 0xb230, 0xac9f, 0x4098, 0x4302, 0x41ea, 0x181d, 0xb289, 0xb2fa, 0x1e16, 0x424c, 0x414f, 0x443e, 0x1c8d, 0x428f, 0x2079, 0x44d4, 0x437d, 0xbae2, 0x0ba5, 0xbf43, 0x435f, 0x4446, 0x4383, 0x1b2e, 0xb2c1, 0x1ac6, 0x45d5, 0xaa8d, 0xa9ff, 0x4280, 0xab67, 0xb222, 0xba68, 0x435d, 0x43a2, 0x1957, 0x425a, 0xb03f, 0x42dc, 0xba2c, 0x423f, 0x4071, 0xa66f, 0x06e5, 0xbf51, 0x40e0, 0x30d9, 0x4263, 0xba77, 0x18c4, 0x424f, 0x2054, 0x4136, 0xb09d, 0x3276, 0xb0c2, 0x0886, 0xb20e, 0x3afb, 0x41db, 0x31d9, 0x4062, 0x41cb, 0x1813, 0x40fc, 0x4389, 0xb0b2, 0xbaf4, 0xba68, 0xbf8f, 0x40c3, 0x4068, 0x4017, 0x131f, 0x4310, 0x113a, 0x1dad, 0x43ae, 0xb230, 0x2a6e, 0xb221, 0xb274, 0xbadf, 0xbfc9, 0x23d7, 0xba0a, 0xba0c, 0x40fa, 0xb2b1, 0xbfd0, 0xbf0c, 0x06f7, 0x18c9, 0x46ba, 0xba47, 0x422f, 0x04b2, 0xb2f1, 0x0069, 0x4037, 0xb243, 0x41a8, 0xb253, 0x1a27, 0xb069, 0x42e3, 0x007f, 0xa0c1, 0xaafc, 0xbf87, 0xaa98, 0xb2b1, 0x421a, 0xbaf5, 0x3e9b, 0xbfa0, 0xba22, 0x16cb, 0x408a, 0x1721, 0xb2a9, 0xb2c6, 0x158e, 0x40dd, 0x3ac2, 0xbf5a, 0xafd9, 0x4334, 0xba57, 0x3f09, 0xb0e9, 0xb01e, 0x1bb9, 0xbfc7, 0xb2d0, 0xbac9, 0xb042, 0x4138, 0xbfaa, 0x4429, 0x0e51, 0x424d, 0x423e, 0x3928, 0x415a, 0xa9ee, 0x4049, 0x1bfd, 0xb030, 0x417f, 0x1335, 0x4151, 0x41ad, 0xb2e4, 0x1b22, 0x42cb, 0x34dd, 0x4104, 0x186f, 0x3a3b, 0x437a, 0x42c9, 0x43c3, 0xb275, 0xb281, 0xbfb7, 0x1a57, 0x4257, 0xb06e, 0x215e, 0xa0cf, 0xa019, 0x4438, 0xb2ad, 0x446b, 0xba48, 0x4595, 0x44e3, 0xbf33, 0x4078, 0x4366, 0x1a43, 0x4078, 0x43e9, 0xb0c3, 0x44e9, 0xb008, 0x4000, 0x4015, 0x40f3, 0xbf7c, 0x4063, 0x41e7, 0xbf7d, 0xb0ee, 0xb27e, 0xa702, 0xb2b0, 0x414e, 0xbf00, 0xb2d7, 0x4033, 0x1fb8, 0xb06b, 0x2c22, 0xbfb0, 0x42ca, 0xb220, 0xb019, 0x4274, 0x125e, 0x426d, 0xab1b, 0x4046, 0x465a, 0xb2ca, 0xba26, 0xba09, 0xb07a, 0x4565, 0x4253, 0xbaf0, 0x1b4f, 0xbf67, 0x1ace, 0x403a, 0x1e60, 0x3a08, 0xb030, 0xba66, 0x4176, 0x4003, 0x4227, 0xb000, 0xba2d, 0xbf4b, 0x46a2, 0x4343, 0xba26, 0x438b, 0x4217, 0xb23c, 0x41c3, 0x44a0, 0x40dc, 0x40f1, 0xb29a, 0x1be5, 0xb085, 0x421e, 0x10ec, 0x41da, 0xbfd0, 0x423f, 0x40c7, 0x1689, 0xb2bd, 0x4005, 0xbf7e, 0xba14, 0xb234, 0x41a4, 0x2b94, 0x43f5, 0xb048, 0xb059, 0x0639, 0x4255, 0x14a6, 0x400e, 0x1f55, 0x41ad, 0x3822, 0x4218, 0x1cb8, 0xb22a, 0x4410, 0x40c0, 0xa6c5, 0xb03b, 0xbf46, 0x2d61, 0x43f2, 0x15b7, 0x4347, 0xacc9, 0x422d, 0xb2f4, 0xbfd0, 0xbfe4, 0x42fd, 0xb28c, 0x1d63, 0xba3f, 0x432e, 0x40bc, 0x4337, 0xbfa0, 0x05a9, 0x4160, 0xb030, 0x0da5, 0x0cad, 0x42e0, 0xb2fa, 0x3352, 0x4184, 0x4309, 0x4262, 0x1cfc, 0x43b2, 0xbf25, 0xbaf4, 0x1eb8, 0xba24, 0x3854, 0xb2af, 0x4074, 0x16ab, 0xb28c, 0xa7af, 0x348f, 0x415d, 0x2104, 0xa0dc, 0xba00, 0x129c, 0x413e, 0x40a6, 0x439a, 0x407c, 0x1c34, 0x1b66, 0xb256, 0x425d, 0x423c, 0xbf71, 0xa64f, 0x412c, 0xb257, 0x4035, 0x420d, 0x45da, 0x11e7, 0x403b, 0xb0a8, 0xb2c9, 0x420c, 0xba22, 0xb0f6, 0x211b, 0x41f9, 0xb2b1, 0xbf00, 0xba40, 0xbf45, 0xbada, 0xbade, 0xb24b, 0x424c, 0x4050, 0x40a3, 0x4669, 0xaaff, 0xb089, 0x4335, 0x4104, 0x401d, 0x418d, 0x4000, 0x42f2, 0xa8da, 0x4151, 0xbf38, 0x45a4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x10ecc80c, 0x68d834e9, 0x195770d2, 0xf92fff26, 0x0d6457be, 0xa5d13c2e, 0x614b9061, 0xbd8cee14, 0x4cf5750a, 0x29eac697, 0xd29e8d3c, 0x1f3a1fb0, 0xc8c54efa, 0x28709a39, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x2870a341, 0x50e143f6, 0x2870a3f9, 0x00000000, 0xffffffff, 0xd78f6003, 0x000018dc, 0xffffffff, 0x4cf575b9, 0x525b6130, 0xffffcfff, 0x1bc387ef, 0xfc89683f, 0x28709fd9, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x155d, 0x212d, 0x4196, 0x4176, 0x42cb, 0x4436, 0xba13, 0x41cd, 0x4049, 0x42e3, 0x435c, 0xa81a, 0xba0c, 0xba6d, 0xbfa0, 0xb2d5, 0x42bc, 0xbfc0, 0xbf00, 0x45a9, 0xbf19, 0xba0c, 0xb256, 0xba51, 0x393b, 0xb048, 0xbf0f, 0xb2fc, 0x406e, 0x2068, 0x4007, 0xbf15, 0x4022, 0x4100, 0x4268, 0x408b, 0x42f9, 0xba0d, 0x41f9, 0x1b65, 0x4226, 0xb24a, 0x4677, 0x43ab, 0x3cba, 0x43ad, 0x3eae, 0x42df, 0xbf2a, 0x07ab, 0xb27e, 0x1e70, 0x440d, 0xb280, 0xba74, 0xb0a6, 0x1992, 0x275c, 0xb25e, 0xb027, 0x43c7, 0x4274, 0x4399, 0x40cb, 0x420e, 0x40be, 0x411c, 0x3e42, 0x1e52, 0x44ca, 0xb0b2, 0x4499, 0x418b, 0x3ecd, 0xac43, 0xbf76, 0xa566, 0xa33d, 0x4011, 0xbf22, 0x46ba, 0x45a1, 0x431d, 0x43a3, 0x4288, 0xbf02, 0x4157, 0x25bb, 0x24da, 0xbaf0, 0x4260, 0x12c9, 0x215f, 0xbf32, 0x432a, 0x419b, 0x413e, 0x1b77, 0xb0d4, 0x41a6, 0x0aae, 0xbff0, 0xb00f, 0x4091, 0x400e, 0x4036, 0xba1d, 0x1f20, 0x4194, 0xba73, 0x1a36, 0xbfc0, 0x405c, 0xbf93, 0x381b, 0x42d2, 0xba26, 0x41c8, 0xba43, 0xb03a, 0x410a, 0x4142, 0xb2b6, 0x46e1, 0x4227, 0xab57, 0xb29d, 0xbf3d, 0x195a, 0xba11, 0x42b7, 0x418a, 0x40b3, 0x420f, 0xb285, 0xbf41, 0x1aec, 0x10d6, 0x41eb, 0x2252, 0xbf80, 0xb0db, 0x1e20, 0x43c7, 0x41f8, 0xb24a, 0x4605, 0xbadb, 0x41b6, 0x4267, 0x3473, 0x42c0, 0x42e1, 0x42d2, 0x45aa, 0x4240, 0xbf90, 0x43d7, 0xb03a, 0x411f, 0x42ee, 0x4370, 0x4238, 0x4646, 0x1ce2, 0xbf65, 0x2fa6, 0x41fc, 0xba40, 0x40e0, 0x10fd, 0xb0b7, 0x1851, 0x4300, 0xbaf7, 0xb030, 0xb031, 0xa1cf, 0x135c, 0xb09e, 0x4131, 0xb039, 0x4548, 0x4022, 0xbfb2, 0x35c4, 0xb2a0, 0xb05a, 0xbfc3, 0x40bb, 0x4694, 0x419f, 0xbf80, 0x436f, 0x19c2, 0xb210, 0x1efd, 0x4104, 0x0fff, 0xbfc3, 0x4271, 0x4282, 0x2373, 0xb22e, 0x1d4c, 0x41a8, 0x41e0, 0xb264, 0xb0f2, 0xb202, 0x4087, 0xba73, 0x4351, 0x43cb, 0xae08, 0x40d4, 0x2529, 0xb22b, 0x412e, 0xa5ec, 0x42ce, 0x0a48, 0x3422, 0x186a, 0x2588, 0xba79, 0x40dc, 0x041b, 0xbf65, 0x434f, 0x1ca9, 0xb262, 0x299b, 0x4371, 0x43a0, 0x433d, 0x0586, 0x2f06, 0xb267, 0x4695, 0x4287, 0xbf7d, 0x1d88, 0x4665, 0xb282, 0x1912, 0x401e, 0x40cf, 0x1883, 0x4195, 0xba42, 0x45b4, 0xbf53, 0xb22f, 0xa9fe, 0xba1a, 0x3a48, 0xbaea, 0x4000, 0x07fe, 0x405f, 0x4351, 0xbf1d, 0x40f7, 0xb098, 0xb0a6, 0x3be8, 0x1980, 0xb014, 0x405d, 0x42bb, 0xac85, 0x06b7, 0x42a8, 0x42c8, 0x421e, 0xbf19, 0xb2ba, 0xb0b2, 0x4154, 0x42bc, 0x25fb, 0x4612, 0x4112, 0xb290, 0x40a0, 0x4195, 0x1e8b, 0x3ac1, 0x438d, 0x4340, 0x1a9e, 0x426a, 0x402a, 0x2d5c, 0xba5d, 0xb281, 0xb2ca, 0x4365, 0x191e, 0x2171, 0x3a69, 0x409a, 0x4304, 0xbfe1, 0x4316, 0xb2d2, 0xa9b9, 0x42e7, 0x4684, 0x2f78, 0xbaed, 0xbf81, 0x166e, 0x465e, 0xbadd, 0x4662, 0xbacb, 0x4003, 0x428b, 0x3668, 0x42ce, 0x42ab, 0xb22a, 0x40a0, 0x1cd8, 0x4029, 0xb2a7, 0xb061, 0x160d, 0x424f, 0xba7f, 0x4446, 0x3ea4, 0x0dac, 0xbfca, 0x0a49, 0xbae0, 0xb2d8, 0xabe7, 0xba1f, 0x40b2, 0xbafc, 0x4679, 0x37aa, 0x40e8, 0x12a2, 0xb0fa, 0x3fe1, 0x42a3, 0x1866, 0xba6b, 0x00a0, 0x3304, 0xbf5b, 0x4431, 0x40d8, 0x0a5a, 0x42e0, 0x41c7, 0x1980, 0xb096, 0xbfc3, 0xba10, 0x1efd, 0x2704, 0x419c, 0xb235, 0xbf97, 0x4391, 0x0740, 0x4057, 0x0d91, 0x4122, 0x1f06, 0x42d4, 0x42ce, 0x4211, 0x439a, 0x1d59, 0xbf98, 0xba68, 0x4275, 0x413f, 0x1d15, 0x4180, 0x414a, 0x421e, 0x438a, 0xa29c, 0x3fb4, 0xbf0b, 0x4025, 0x41ab, 0x428d, 0xb2b5, 0xb067, 0xbf90, 0xb258, 0xbf18, 0x2023, 0x43e4, 0x44d5, 0x37d4, 0xb09e, 0x427d, 0x29dc, 0x4171, 0x041a, 0xbfba, 0xb29f, 0xb26a, 0x427f, 0x4238, 0x4381, 0x15bc, 0x430b, 0x2760, 0x1aab, 0xb2ff, 0x4282, 0x103e, 0xa10f, 0x0891, 0x4056, 0x408e, 0x0276, 0xbfb6, 0x094c, 0x2c6a, 0xba4e, 0x403b, 0xb24c, 0xabc2, 0xb0f6, 0x4366, 0x1ded, 0x408a, 0x40bf, 0x4352, 0x10a3, 0xb0bc, 0x42e6, 0x40a5, 0x4551, 0x37ad, 0xbf02, 0xb0f1, 0x43e0, 0xb213, 0x4105, 0xb2bc, 0x40c4, 0xb20e, 0x1117, 0x40b2, 0x412b, 0x1dd0, 0x43a6, 0x43f9, 0x4391, 0x26fd, 0x07df, 0xbf13, 0x4152, 0x22a3, 0x1936, 0x1915, 0x40bb, 0x21c5, 0x2a65, 0x2572, 0x4117, 0x19ba, 0x43c3, 0x41ad, 0xb20a, 0x1fd6, 0x37bd, 0xb00b, 0x41b3, 0xa0b2, 0xb2f5, 0xb209, 0x40f3, 0xb244, 0x4380, 0xb20a, 0x41c7, 0xbf9c, 0xba2b, 0x4064, 0x4048, 0xb2f9, 0xbff0, 0x4399, 0x42bb, 0x1c85, 0x31be, 0x322d, 0x102a, 0x4614, 0x1860, 0x42fb, 0x1fd4, 0x1ba5, 0x4395, 0x3c83, 0xa80f, 0x439b, 0xba4a, 0xbf98, 0x40a3, 0x4065, 0x04dc, 0xbf38, 0x4260, 0xaa25, 0x220a, 0x05e2, 0x40ec, 0xbf60, 0x41b3, 0x127f, 0x40de, 0xb0a8, 0x43ba, 0x41be, 0xbf93, 0x2f9a, 0xb059, 0xb0a5, 0x4196, 0x4343, 0x400e, 0xbfb4, 0x4324, 0xb0fd, 0x09ed, 0xbad8, 0xb02d, 0x150b, 0x4016, 0xbfa6, 0xba3e, 0x439a, 0x3d0b, 0x1b7a, 0xba18, 0x241e, 0xa47d, 0xb22f, 0xb281, 0x40be, 0x4096, 0xb046, 0x43c2, 0x0c15, 0xb079, 0xb21e, 0xbf54, 0x42ab, 0xb234, 0x46b0, 0x1c7b, 0x4297, 0x43f4, 0x029f, 0x3405, 0x42d0, 0x42d2, 0x41b3, 0x40fe, 0x45c4, 0xb0db, 0x43ba, 0x4219, 0x4269, 0x1612, 0x41ac, 0x40ce, 0x3511, 0x4339, 0xb023, 0x130e, 0xb281, 0x1d1d, 0xbfcd, 0x419a, 0xba2b, 0x4011, 0xb295, 0x43f1, 0x40f7, 0xb264, 0xba37, 0xb2d0, 0xbf46, 0x18b3, 0x34c0, 0x040f, 0x401a, 0x138a, 0xa805, 0xb259, 0x440a, 0x1bee, 0xa5a7, 0x43f8, 0x407e, 0x424a, 0xb2f0, 0x433a, 0x39aa, 0x19d8, 0x42d2, 0xa041, 0x41c5, 0x214f, 0xb08f, 0x4231, 0x40b5, 0x05df, 0xbfa6, 0xa527, 0x455f, 0x2dc3, 0x1f02, 0xbaf8, 0x45e2, 0x466d, 0xa17d, 0x39d3, 0x1cdb, 0x4353, 0x43d0, 0x1b82, 0x0fc7, 0x4367, 0xba0b, 0xa21c, 0xbf4d, 0xba44, 0xa48c, 0x4185, 0x4168, 0x27d1, 0x21e1, 0x4316, 0x309b, 0x1859, 0x3d8d, 0x41fa, 0xba63, 0x2168, 0x0966, 0x43f5, 0x35ce, 0x3894, 0x412f, 0xbae5, 0x4557, 0x45b2, 0x4601, 0xb240, 0x4301, 0x4237, 0x4608, 0x1707, 0x08e5, 0xbf57, 0x3831, 0x41fe, 0x1722, 0x1f29, 0x45dd, 0x4233, 0x4550, 0xba2f, 0x3c89, 0xbfc0, 0xbf0c, 0xb2ad, 0x4042, 0x3c09, 0x1f76, 0xb29c, 0x43ce, 0x28fc, 0x4661, 0xb28c, 0x432b, 0x4277, 0xa32e, 0x42ae, 0x41c3, 0x1cad, 0x4203, 0x40bc, 0x43c6, 0xbf78, 0xbae9, 0x4222, 0x0094, 0x4350, 0x41d9, 0x415e, 0xb2b7, 0x4088, 0xb060, 0x40bd, 0xbf6b, 0x1a14, 0xa78c, 0xb2cf, 0x4021, 0x1adf, 0xba10, 0x1bfb, 0x41fe, 0x1b0e, 0x4369, 0x40e6, 0xbfab, 0x4077, 0x4673, 0x2c2d, 0x1df1, 0xb2bd, 0x42b0, 0xbf90, 0x03a8, 0xae03, 0x40ea, 0xbf15, 0x388f, 0x43ef, 0x2e69, 0x4130, 0x40b8, 0x3597, 0x4688, 0x1a86, 0x08aa, 0x4479, 0x45ea, 0x433c, 0xbf60, 0x1e2d, 0x1a2d, 0xb001, 0x1932, 0xaf95, 0x41aa, 0x4477, 0xb050, 0x42f6, 0xb219, 0xb03e, 0xb27e, 0xb0e7, 0xbae9, 0x4189, 0x4077, 0xbfb1, 0x407f, 0x454d, 0x044d, 0x410f, 0x46fa, 0x4265, 0x43aa, 0x41e0, 0x19da, 0x3196, 0x403d, 0x435c, 0x40f5, 0x4067, 0x3122, 0x1ea4, 0x40d7, 0x40dd, 0x1cb3, 0xb241, 0xb20d, 0xae94, 0xb20c, 0x42bb, 0x41a6, 0x4084, 0xa88d, 0xbfb8, 0x42c2, 0xb260, 0xba2d, 0x4119, 0xba63, 0xb2cf, 0x2aaa, 0xa2c2, 0x32e4, 0x42eb, 0xb24c, 0x04eb, 0x43fc, 0xb2c5, 0x4158, 0xb26f, 0x4664, 0x2f2a, 0xb0f7, 0x0290, 0x40d7, 0x1955, 0x1cc8, 0x405c, 0x1d87, 0x40b5, 0xbf89, 0xa488, 0xb2db, 0x401e, 0xb25d, 0xb2c8, 0xba28, 0xb09b, 0x0813, 0x06b2, 0xba07, 0x416e, 0x2476, 0x40c1, 0x1902, 0xb0c8, 0x3d59, 0x26cd, 0xbad0, 0x3338, 0xb2d5, 0x41c1, 0x19e0, 0xbfdf, 0x1e0b, 0x4301, 0x00f0, 0xb252, 0xb2be, 0x4614, 0x1c4b, 0xb00a, 0x44b3, 0xba75, 0x416c, 0x4283, 0x0e46, 0xae89, 0xba0e, 0x4287, 0x3290, 0x4296, 0x43fe, 0xa47a, 0x4243, 0xba58, 0x4076, 0xb2d3, 0xb2ee, 0x42c8, 0xbf8a, 0x4087, 0x465d, 0x0812, 0xb0d0, 0x1b04, 0x41c3, 0xba2c, 0x461b, 0x43c2, 0x1d5b, 0x439d, 0x41d2, 0xbf62, 0x4283, 0x41d3, 0x2cd1, 0x19da, 0xba70, 0xba7d, 0xbf4c, 0x4024, 0xa8a8, 0x3533, 0x4356, 0x4360, 0x40e8, 0x1a0c, 0x4217, 0x46ad, 0x2bcf, 0x4136, 0x41ba, 0xa279, 0x25e8, 0xbfe0, 0xb2a5, 0xa2f0, 0xbf15, 0x41e7, 0xbac9, 0x1f73, 0x410f, 0x0d8a, 0x40d3, 0xab54, 0x4219, 0x1984, 0xbada, 0x4384, 0x1b68, 0x44a8, 0x11d9, 0x4048, 0xbfa6, 0xb2f9, 0x42a0, 0x4237, 0x1298, 0xb0ce, 0x42e5, 0xb2c3, 0x40ce, 0x43eb, 0x29fe, 0x437a, 0xb238, 0xb2e5, 0xa899, 0xb05c, 0xbf91, 0x1947, 0x446b, 0x2586, 0x407f, 0xb275, 0x4193, 0x40eb, 0x439a, 0x2b6e, 0x4378, 0xb2b0, 0xba74, 0x438a, 0x1949, 0xab0c, 0xb208, 0xbae5, 0x4631, 0xbfcd, 0x4165, 0xba67, 0x4038, 0xb007, 0x177e, 0x4275, 0x1b62, 0x2fff, 0x45c6, 0x31d9, 0x429d, 0x18c1, 0x46dc, 0xb2c2, 0x41a9, 0x4390, 0x4362, 0xb0a3, 0xbae3, 0x102a, 0xb08c, 0x27c5, 0xbf0d, 0x1b86, 0xadce, 0xb0bb, 0x1d65, 0x412e, 0xb0d9, 0x2bf0, 0x320b, 0x433e, 0xadb6, 0x406a, 0x20e1, 0x1c81, 0x40ba, 0x13e1, 0x19cd, 0xb2a4, 0xbf5a, 0x46aa, 0x43b8, 0xb211, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x35faf52d, 0x178d83c4, 0xd9eed1e5, 0x2d403c23, 0x142e3ba1, 0xfdb96242, 0xb8568fae, 0xd9e9fcd8, 0xdb303d2f, 0xb29babc4, 0x02ba22df, 0x57781ed2, 0x79ab341b, 0x84723a31, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x00000020, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x000000c5, 0x000000c5, 0x000000c5, 0x0fa51006, 0x79ab341b, 0x000000c5, 0x57781ed2, 0x57781ed2, 0xfffffd5f, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x0968, 0x41a2, 0x1b65, 0xb07e, 0xba6a, 0x403d, 0x25a7, 0xba28, 0x0ab4, 0x3baf, 0xbf63, 0x300c, 0x16a4, 0x3997, 0xb26e, 0xb094, 0xb27b, 0x4184, 0x26c7, 0x4019, 0xbf84, 0x437f, 0x1beb, 0x40ff, 0x4103, 0x4642, 0x41df, 0x4310, 0xa2cb, 0xbfbf, 0x33f8, 0x4387, 0x3837, 0x41fb, 0x26da, 0xb076, 0x459d, 0x044e, 0xbf19, 0x4041, 0x021c, 0x1051, 0x17ba, 0xba72, 0xb26d, 0x4418, 0x0d16, 0x4235, 0xb299, 0x2c02, 0xb27c, 0xbaf1, 0xac4f, 0x4434, 0xb20c, 0x0a23, 0x41dc, 0xb00f, 0xbf14, 0x3ed6, 0x404c, 0x45c2, 0x3528, 0x2bdb, 0xa5c7, 0x40b6, 0xbf60, 0x4042, 0x43ae, 0xa0f5, 0xbf62, 0x1fdf, 0xba2b, 0xb2eb, 0xa7ef, 0x1c7a, 0x02de, 0x43c3, 0xbfde, 0xb247, 0xba79, 0x4010, 0x43d3, 0x41eb, 0x4211, 0x43a9, 0x238e, 0x44ec, 0x3f4b, 0xb0e7, 0x4338, 0xb2b1, 0x40d2, 0x438c, 0x409e, 0xb285, 0x191c, 0xb2f6, 0x4047, 0x1d63, 0x213a, 0x400d, 0xb2c8, 0xa8b9, 0x04ea, 0x40ca, 0x407b, 0xbfdc, 0xb295, 0x265d, 0x404a, 0x427d, 0x42d7, 0xb03e, 0x1647, 0x4325, 0x183d, 0x0b07, 0x1c1a, 0xa8e1, 0x4147, 0xae6f, 0x2492, 0x10cf, 0x1af3, 0x411e, 0xa991, 0x12a1, 0xbfa1, 0x40df, 0xa2c7, 0x4275, 0x17de, 0x0093, 0x151f, 0x427d, 0x40d9, 0x4171, 0x429b, 0x3fac, 0x4170, 0x0994, 0x4140, 0xb04b, 0xad99, 0xb257, 0x3c21, 0x4301, 0xba22, 0xbae2, 0x3a3d, 0x4290, 0xbf5a, 0x41e3, 0x424d, 0xba3a, 0xb264, 0xb2e8, 0xbfe1, 0x42b7, 0xb217, 0x42a0, 0x404d, 0xb200, 0x42fd, 0x40c3, 0x410d, 0xb231, 0x434c, 0x41a2, 0x2c01, 0x02ff, 0x432f, 0x4335, 0x42a8, 0x426c, 0x4180, 0x446b, 0xbf80, 0xa9b8, 0x4223, 0xbf45, 0x02ae, 0xb2c5, 0xb291, 0xb0a2, 0x1e0f, 0xbaca, 0x44f9, 0x2594, 0xb02c, 0x1697, 0x1a0f, 0xba47, 0xa098, 0x411b, 0xbae1, 0x45eb, 0x42c2, 0xba6b, 0x409f, 0xb02c, 0x4382, 0xba24, 0xba79, 0x2dda, 0x4141, 0x41fc, 0x462d, 0xb0bd, 0xbfa8, 0x43d8, 0x2cda, 0x43dc, 0x401c, 0xba12, 0xb228, 0x436f, 0x423b, 0xb20a, 0x2089, 0xa5eb, 0x421a, 0xb2dc, 0x406c, 0xbf71, 0x439c, 0x43d2, 0xb27d, 0x43a6, 0xaa82, 0x4195, 0x403d, 0x1811, 0x41c9, 0xbafe, 0x41a8, 0x3d5e, 0x414b, 0xbf61, 0x1955, 0x374f, 0x2d82, 0xb23b, 0xb080, 0x0c12, 0x14c7, 0xba50, 0xb25c, 0x40ab, 0x4132, 0x4270, 0x4375, 0x184b, 0xbf3e, 0xbaec, 0x3645, 0x4045, 0x40e8, 0x1939, 0x40de, 0xbf03, 0x406e, 0x4201, 0x4544, 0x40fc, 0x427d, 0x2550, 0x0046, 0xa85e, 0x39dc, 0x419f, 0x401f, 0xb0a7, 0x4299, 0x1635, 0x430f, 0x40e4, 0x4163, 0x40b5, 0x4304, 0xbf80, 0x07b0, 0x1b13, 0xb207, 0x4218, 0x437c, 0xbf86, 0x400f, 0x4334, 0xb26a, 0x288d, 0x4235, 0x4462, 0xb000, 0xa2a9, 0xba54, 0x42e0, 0xb29e, 0x4273, 0xb09b, 0x341f, 0xb281, 0x33e0, 0x38e9, 0x40da, 0x464c, 0xbf9e, 0xa1f8, 0x40a4, 0x3a31, 0x4190, 0x459d, 0x40ec, 0x2a92, 0x463e, 0xba46, 0x014a, 0x4001, 0x1903, 0x428b, 0x38c6, 0x40d0, 0xb27c, 0x4361, 0x4467, 0x411d, 0x4044, 0x4205, 0x196f, 0xbf2b, 0xb2d9, 0x4351, 0xb29b, 0xbaef, 0x15fc, 0x410a, 0x1ac1, 0xb2c9, 0x150f, 0x419a, 0xb266, 0x0f11, 0x46a2, 0x412d, 0xb2da, 0xb259, 0x42ab, 0x0bad, 0x40a5, 0x42a6, 0x42a8, 0x1d2c, 0x42d0, 0xb2a3, 0x1ac2, 0xbf0a, 0x1c92, 0xb0a7, 0xaaa2, 0xbf5a, 0x4210, 0x428a, 0x1d36, 0xbfd4, 0x42d9, 0xb25f, 0x426e, 0x4198, 0x4200, 0x01dd, 0x1a2b, 0x4216, 0x4482, 0xbadf, 0x406f, 0x1839, 0x41f5, 0x43df, 0xb28f, 0xba7f, 0x402b, 0x412b, 0xb031, 0xbf1f, 0x419c, 0x43dc, 0xa31c, 0x43de, 0xb21c, 0x07e8, 0xa414, 0x41e8, 0x190a, 0x1974, 0x43a7, 0x43ec, 0xba51, 0x42a3, 0x4174, 0x1c77, 0x1cce, 0xb217, 0x0fd5, 0xbf4a, 0x07cb, 0x403c, 0x4011, 0xb20f, 0x2d7e, 0x4049, 0x4131, 0x1a6a, 0x187a, 0x420b, 0x4395, 0x24f5, 0x4009, 0xbf44, 0x0822, 0x0457, 0x137e, 0x1dd1, 0xb093, 0x1543, 0xb2df, 0x4022, 0xb0ca, 0xbf0a, 0xba0d, 0x44f5, 0x433e, 0xb274, 0x41ce, 0xb26d, 0x413e, 0x442b, 0xb0b6, 0x405b, 0x45b9, 0xb271, 0x454f, 0x3b98, 0xb250, 0x42ea, 0xbf41, 0x3ab9, 0x18ac, 0x42c8, 0x1f9c, 0x4342, 0x456d, 0x40d6, 0x32e8, 0x1e55, 0x40af, 0x40f6, 0x406d, 0xb2b0, 0x4081, 0xba3f, 0x45c3, 0xbfc9, 0x1197, 0x46e8, 0xb25f, 0x336f, 0x0655, 0xbf47, 0x460b, 0x467f, 0xaad1, 0xb0d9, 0x3dc1, 0xba3c, 0x1986, 0xbf18, 0x410f, 0x42de, 0x18db, 0xb284, 0xb2cb, 0x433c, 0x2e9e, 0x1c28, 0x4316, 0xbf0c, 0xa396, 0x44e4, 0x043d, 0x421d, 0x24f3, 0x46d8, 0xbfdb, 0x43ba, 0xbfe0, 0x3e5d, 0x42ba, 0x4141, 0x42d3, 0xba66, 0x4113, 0xbae7, 0xbfd2, 0x209b, 0xba1f, 0x430b, 0x40da, 0x25ba, 0x4183, 0x1e9b, 0x4160, 0x4321, 0x41ab, 0x1ddb, 0x1c06, 0x41c9, 0x44e5, 0xba67, 0x2f30, 0x42ad, 0x43be, 0xa2cc, 0x41d7, 0x434e, 0x41a1, 0x3017, 0x34dd, 0xbf42, 0xb253, 0x3bb3, 0x42a0, 0x40f5, 0x1c73, 0x42d9, 0xba2a, 0x42aa, 0xb2f7, 0x424f, 0xbf61, 0x427f, 0x319f, 0x4281, 0x0e25, 0x0bf2, 0x40df, 0x4367, 0xbf16, 0x128a, 0x42b4, 0x43af, 0x1d3c, 0x20d8, 0x42d0, 0x4374, 0xb291, 0x3c17, 0x413e, 0xbf67, 0xba0e, 0xb09e, 0x432c, 0x4355, 0x422b, 0x468a, 0xba1f, 0x414c, 0x4092, 0xb24a, 0x4085, 0x0c34, 0xb0e7, 0x055e, 0x4052, 0x350c, 0x00c1, 0x1b27, 0x42ff, 0xb258, 0x46db, 0xb204, 0xbf9a, 0x204b, 0xb228, 0x4438, 0x42aa, 0x1acc, 0x43c5, 0x40ea, 0x4028, 0x1ff9, 0x0634, 0xba3c, 0x41fe, 0x4205, 0x4310, 0x18b9, 0x1c8a, 0x4390, 0x23ab, 0x4131, 0x4632, 0xbf4e, 0x4110, 0x4337, 0x3ef8, 0xa221, 0xa445, 0x3505, 0x4016, 0xb0be, 0x41ca, 0xbff0, 0x40e0, 0xaf82, 0x43d3, 0x4004, 0xbaf2, 0x44fa, 0x43d3, 0xbadb, 0x1a50, 0x42ba, 0x07db, 0xba13, 0xb230, 0xbadf, 0x43a0, 0x18c3, 0xb2a0, 0xb283, 0xbf41, 0x1d95, 0x1070, 0x2958, 0xb047, 0x42fb, 0xb21f, 0x04f0, 0x43b4, 0xba7e, 0x4200, 0x41f2, 0x42ea, 0x42d1, 0x4270, 0x1f13, 0x434b, 0xb24d, 0x2b14, 0x1577, 0x1eae, 0x4207, 0x46d4, 0x41a0, 0x41d7, 0x0088, 0x4069, 0x1ee5, 0x40a1, 0xbf44, 0x4053, 0x440d, 0x1fe8, 0x427b, 0x4048, 0x40fb, 0x4340, 0xb270, 0x25d9, 0xba7a, 0x1859, 0x393f, 0xb2b3, 0x150b, 0x4332, 0x0152, 0x0589, 0xb2e5, 0x4088, 0x20a7, 0xba4d, 0xa406, 0x37d6, 0x40b6, 0xbadd, 0xbfd7, 0x409b, 0x4116, 0x031c, 0x417b, 0x438a, 0xbff0, 0x40a8, 0x1d5e, 0x426d, 0xb2ee, 0x4380, 0xbf0d, 0xb212, 0x1b52, 0x2108, 0x1b29, 0x1b85, 0x43cb, 0x379e, 0x43b6, 0xbfb0, 0xb0ce, 0x0e2c, 0x02eb, 0x223d, 0x4185, 0x400d, 0xb200, 0x435e, 0x4205, 0xba20, 0xbf5f, 0xb06d, 0x18b9, 0x4235, 0x42e5, 0xbaf4, 0xb0fe, 0x0de1, 0x458c, 0x463b, 0x1a59, 0x1efc, 0xb2de, 0x41df, 0x1f03, 0x3cb2, 0x1c76, 0xa923, 0x462f, 0x1abc, 0xbf0d, 0x41e6, 0x42ad, 0x4625, 0x0d75, 0x45bc, 0xba5a, 0x42b3, 0xba0b, 0x4250, 0x4123, 0x1956, 0x275c, 0xbaf5, 0x3f77, 0xba2f, 0x1b93, 0x4209, 0x4370, 0xb2d5, 0x4164, 0xbfe8, 0x40c2, 0x4399, 0x40ee, 0xbacf, 0x4230, 0x4055, 0x4342, 0x428e, 0x4372, 0x1af5, 0xb0f6, 0x4096, 0x1a4a, 0x1c74, 0x1b7b, 0x27d4, 0x1dfb, 0x3d43, 0x0c87, 0x2649, 0xbaf2, 0x40dc, 0xbf70, 0xbff0, 0xb24a, 0x40f6, 0xbf62, 0x42da, 0x46d3, 0x407b, 0x432d, 0xb207, 0x427a, 0x417b, 0x4156, 0x291d, 0x41c1, 0x41c8, 0x42d8, 0xb27c, 0xb0a5, 0x42cf, 0x2f18, 0x4661, 0x426a, 0x4316, 0x3f27, 0xbade, 0x42fd, 0x43e9, 0xbf5b, 0x4172, 0xac9d, 0x429e, 0x444b, 0xa3d2, 0x19ac, 0x411e, 0x211e, 0x40dc, 0x4332, 0x405f, 0x2ae6, 0x1bc6, 0xba2b, 0x1f21, 0x4124, 0x404b, 0x42b0, 0x4217, 0xb2b4, 0xb0cc, 0x2e76, 0x1f91, 0x23f0, 0xb206, 0x0d78, 0x4323, 0x312a, 0xbf46, 0x4198, 0x42de, 0x430c, 0x433a, 0x40f4, 0x41ea, 0x1c1d, 0x447e, 0xbae1, 0xa070, 0x415a, 0x4111, 0xb026, 0x4323, 0xbf72, 0x4419, 0x410e, 0x04c5, 0x422f, 0x16fa, 0x3627, 0x24fc, 0x42b9, 0xbf6e, 0x42ef, 0x1589, 0x05b3, 0xbf84, 0x3920, 0x01a7, 0x0d74, 0x4240, 0x241c, 0x1b1f, 0x408d, 0x1f58, 0x40d2, 0xb072, 0xbf29, 0x1994, 0x16a6, 0x403f, 0x4145, 0x2d03, 0xa4f8, 0x1b74, 0x4195, 0xba79, 0x4418, 0x4037, 0x40f8, 0xba19, 0xba7c, 0x44c8, 0x435a, 0x40e3, 0x43c3, 0x431b, 0x4124, 0x409b, 0x4355, 0x42db, 0x4300, 0x17a9, 0x3dfd, 0xbf0b, 0x41eb, 0xbac5, 0x0b64, 0x1f66, 0x098c, 0xbf60, 0x456a, 0xb0c2, 0x223c, 0x4301, 0xb20e, 0x4628, 0x4062, 0x1d7d, 0x43b7, 0x4369, 0x424c, 0xbaf7, 0xbfa9, 0xbace, 0x41c5, 0x4390, 0x420e, 0x40ff, 0x41f4, 0x4186, 0x4022, 0xbfa0, 0x4151, 0xa81d, 0x4019, 0x4220, 0x41f5, 0xbfcf, 0x43b6, 0xb290, 0xbfc0, 0x4311, 0x437d, 0xbfa0, 0xb2a4, 0xba2a, 0x41e6, 0xbf81, 0x0d08, 0x1e47, 0x1cb0, 0x4370, 0x12b8, 0xb261, 0xb26a, 0x437c, 0xbafe, 0x42b8, 0x43e5, 0xb0e5, 0xba58, 0x4158, 0x42a0, 0x4311, 0x4144, 0x1fc0, 0x1dc0, 0x08a7, 0x171f, 0x4637, 0x445d, 0x0a52, 0x1f8e, 0x1a78, 0xb2e5, 0xbf42, 0xba74, 0xba51, 0x1dd6, 0x45b6, 0x4278, 0x4389, 0x4230, 0xa3e1, 0x0f67, 0x42d0, 0xba44, 0xbfb0, 0x4383, 0x4583, 0xad13, 0x404e, 0x43b2, 0x4295, 0x0b4d, 0x42e6, 0x42bf, 0x107b, 0xbfb8, 0x1fa3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5384f6cd, 0x93b1eb86, 0x519fcf8e, 0x564cda67, 0xcde58997, 0xf3ec3c43, 0x6c08d4ea, 0x6e3c1d46, 0xe8c48cc6, 0xcea89f8e, 0x2f68758c, 0x0e162d4d, 0xc7597952, 0x77abbef9, 0x00000000, 0xa00001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000007, 0x00000000, 0xdcbede57, 0xcea8b10a, 0x00011309, 0x0e162d4d, 0x00011309, 0xf5b62f3f, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1cdd, 0x1c98, 0xa82c, 0x22bd, 0xabb4, 0x4105, 0x3601, 0x432b, 0x406b, 0xb08c, 0xbacf, 0xb2ee, 0x382b, 0xb055, 0x4418, 0x16af, 0x4171, 0xb284, 0xb2c7, 0x4273, 0x43c5, 0xbf28, 0x4374, 0xb22c, 0xa063, 0x4242, 0x43f9, 0xba6a, 0x42a1, 0x4311, 0x4298, 0xb2b4, 0xbfcf, 0xba03, 0x41ed, 0xba7d, 0x444d, 0x1db4, 0x1de9, 0x192c, 0xbac1, 0x2cd3, 0xbfc9, 0xb224, 0xba09, 0x4098, 0x3ae8, 0x4310, 0xba6a, 0x4366, 0xb05a, 0xbf43, 0x406f, 0xb0a2, 0x0a86, 0x41c1, 0x413c, 0x1e56, 0x224c, 0xbf0e, 0xb016, 0x40d3, 0x40eb, 0x3b28, 0x4131, 0x30c8, 0xb23a, 0x40db, 0x2ff3, 0xb293, 0x1ea8, 0x4283, 0x4071, 0x40a5, 0x40db, 0x1539, 0xb2ff, 0xb0d4, 0xb2a4, 0xb2f4, 0xbf60, 0x433c, 0xbf0b, 0x3bac, 0x134e, 0xb241, 0xba6a, 0x1bf6, 0xb28b, 0xba3f, 0x43b1, 0x1f92, 0xba68, 0xb21e, 0x4272, 0x40a4, 0x2fde, 0xaec9, 0xad72, 0x42f0, 0x433f, 0xaa74, 0x1032, 0xa843, 0x19d7, 0xaf1b, 0xbf23, 0x14cb, 0x41a3, 0x1ff5, 0x40ca, 0xba1e, 0x4205, 0x4123, 0x401d, 0x1f2f, 0xbfc0, 0x3d68, 0x4368, 0x3441, 0x43e1, 0x04bf, 0x1934, 0xb299, 0x41bc, 0x4331, 0x00c2, 0xbf74, 0x4424, 0xb0de, 0x3563, 0xb2ea, 0x4325, 0x425f, 0x4331, 0x1516, 0x2bfa, 0xaa20, 0x4001, 0xb2dd, 0x1b4a, 0x435d, 0x1e15, 0x46fb, 0x1ca3, 0xb240, 0xba1e, 0x43ac, 0x04d1, 0x4288, 0xb053, 0x0649, 0x40ab, 0xbf6a, 0xa873, 0x41bd, 0x40bf, 0x0734, 0x410d, 0xbaf5, 0x431d, 0xb21d, 0x43bc, 0x1a8d, 0x43d1, 0x182f, 0xb2a7, 0xba53, 0xa1e5, 0x1f14, 0x40fa, 0x4595, 0x4373, 0x3a9e, 0x4316, 0x4382, 0x2233, 0x3f1e, 0xbfac, 0x285f, 0x435e, 0xb2e4, 0x1a0c, 0xbae3, 0x432c, 0xbf96, 0x3dd0, 0x4388, 0xb233, 0x1dd2, 0x0628, 0x40f6, 0xad19, 0xbfb0, 0xb01e, 0x3f19, 0x42fb, 0x4397, 0x42fb, 0x4380, 0x088d, 0x1c55, 0x4125, 0x41db, 0x4343, 0x285a, 0xb297, 0x0708, 0x1742, 0x438f, 0x40c9, 0x41ba, 0xb2d7, 0x42f4, 0xbf84, 0xb2be, 0x0ed1, 0xaa8e, 0x1766, 0xb2a2, 0xb23f, 0xb246, 0x42e7, 0xa98e, 0x2ed6, 0xba33, 0x4275, 0x3d19, 0xba01, 0xaf24, 0xba0e, 0x10fe, 0xbf70, 0x40ae, 0x0b85, 0x1b11, 0xba72, 0x1dbd, 0x4218, 0x420f, 0x43df, 0xba42, 0xbf27, 0xba3d, 0x4318, 0x4152, 0x431f, 0x417d, 0xbfaa, 0x1b13, 0x401f, 0x426e, 0xbfb1, 0xb098, 0x1cb5, 0xba05, 0x2f51, 0xa7e9, 0x4238, 0xb2de, 0x4374, 0x3239, 0xb0c3, 0xb2c4, 0xb2db, 0x43d1, 0x17fa, 0x1918, 0xba45, 0x4073, 0x19dc, 0x433b, 0x095c, 0x1f3f, 0x2dfd, 0x2105, 0x4120, 0x4174, 0x42cf, 0x32ac, 0xbf4a, 0xb28e, 0x401d, 0x445e, 0xbf00, 0xbf0d, 0x4353, 0xbfb0, 0x406e, 0x3f79, 0x2e94, 0xb225, 0x40fe, 0xa79f, 0x43fa, 0xb21f, 0x3c1d, 0xbaf9, 0x4178, 0xb224, 0x455b, 0x2da3, 0x36c0, 0x410b, 0xb2bd, 0x1fd4, 0x43ed, 0x4658, 0xbf45, 0xa2bb, 0xba4b, 0x1848, 0x407f, 0x0d0c, 0xb0d2, 0xbfb8, 0xb013, 0x41ff, 0xba13, 0x4246, 0x4413, 0x3ac4, 0xbf25, 0x0c28, 0xba3d, 0xbf00, 0xba77, 0x4115, 0x425e, 0xad5a, 0x44b9, 0x16b5, 0x445c, 0x4030, 0x41ba, 0x4337, 0x435f, 0x25e8, 0xbfe0, 0x43bf, 0x2e8c, 0x2603, 0x1d02, 0x339f, 0xb2da, 0xbf12, 0x4073, 0x46ed, 0x408a, 0x42a4, 0xbfa0, 0x428c, 0x3f60, 0xb264, 0x4656, 0xbf54, 0x4212, 0x1b30, 0x2cc3, 0xb2d3, 0x1c32, 0x4279, 0x2647, 0x4222, 0xba14, 0x1f8d, 0x414a, 0x4244, 0x4316, 0xa3df, 0x42d7, 0x42fc, 0xbf90, 0x4183, 0xa6b7, 0x4002, 0x2fb3, 0x4304, 0xba10, 0xbf80, 0xb21c, 0x414e, 0x1bc7, 0xbfad, 0xb2cd, 0x2f6f, 0xa397, 0x180a, 0xab69, 0xb0db, 0x0d11, 0xba35, 0xbfdb, 0x00c5, 0xa4b1, 0x400a, 0xba74, 0xa723, 0x050c, 0x4318, 0xaa07, 0x4135, 0x4181, 0xba71, 0x4090, 0x41a2, 0x412a, 0x4230, 0x4398, 0xbf18, 0x42c6, 0x441a, 0xb297, 0xb2fa, 0x41b7, 0x4134, 0xb290, 0x41c4, 0xb290, 0x195c, 0x4586, 0x433e, 0x402c, 0x34a4, 0x1add, 0x4316, 0x403b, 0xba51, 0x4257, 0x3d75, 0xba2d, 0x170f, 0x45bb, 0xbf26, 0x43ca, 0xb247, 0x427e, 0x4109, 0x1284, 0xbaf7, 0xbae0, 0xb200, 0x428b, 0x4376, 0x1cfc, 0x4091, 0x2b38, 0x1f17, 0x41c4, 0x40f0, 0xba52, 0x0c05, 0xbfaa, 0x4278, 0x41b2, 0x4334, 0x43b9, 0x42e4, 0x4338, 0x46cb, 0xaecc, 0x2b5b, 0x2407, 0xbace, 0x1ff5, 0x41ba, 0xb013, 0xb283, 0xbf94, 0x1c7b, 0x4444, 0xaf5b, 0x1e40, 0xbfe0, 0x4401, 0x40be, 0xa46a, 0x4229, 0x40a3, 0x43b4, 0x43e9, 0xb268, 0x419f, 0x41b5, 0x43b9, 0x4203, 0xbfa7, 0xb0a5, 0x46b9, 0x42ec, 0x1b9e, 0x400b, 0x06d9, 0xb0b2, 0x4230, 0xb0c6, 0x420e, 0x19d4, 0xbfe1, 0x40f8, 0x1f60, 0x19b8, 0x43c1, 0xb2a7, 0x4357, 0x4364, 0x4320, 0xba6a, 0x15cc, 0x40a4, 0x4378, 0xb0fe, 0x2e96, 0x4183, 0xba02, 0x4387, 0xb27f, 0x42cf, 0xaa6a, 0x404c, 0x4269, 0x036c, 0x42cf, 0x43df, 0x42a2, 0xbfda, 0x078b, 0x402b, 0x1d71, 0x43ee, 0x1a55, 0x1d58, 0x4077, 0x4021, 0x4165, 0xbf4a, 0x42c7, 0x4322, 0xb00e, 0x435a, 0x1fd9, 0xb227, 0x40da, 0x123a, 0x4323, 0x4353, 0xaa7b, 0xbf16, 0x40b0, 0x42df, 0x4334, 0x054d, 0x422d, 0xba0b, 0x46da, 0x44f2, 0xb214, 0x1a61, 0x20d7, 0x43b2, 0xb062, 0x369a, 0xa256, 0xb2f1, 0xb20f, 0x43ba, 0x3b86, 0x4278, 0x1b0f, 0x2c13, 0x4211, 0x46a0, 0xbf88, 0x43b1, 0x4094, 0xba06, 0x45d8, 0x410e, 0x1fc8, 0x1e0c, 0x429a, 0x3c95, 0x45ec, 0x42a6, 0xb2d2, 0x4039, 0x42c0, 0x40a9, 0x41f3, 0x1eb9, 0x04a4, 0xba5d, 0x4256, 0x4469, 0x40f4, 0x1687, 0x4244, 0x400d, 0x2eb0, 0xbf6f, 0xb018, 0xba24, 0x1911, 0x115e, 0x43c1, 0x406a, 0x4008, 0xb076, 0xba5f, 0xb0f5, 0x40a6, 0xb2f4, 0x0d52, 0xba72, 0x4172, 0xbf91, 0x1310, 0x4572, 0xad5e, 0x4399, 0x4081, 0x2c80, 0x258f, 0x1d8c, 0xb067, 0x426e, 0x40c0, 0xb2d0, 0x4544, 0xbf70, 0x4236, 0x43d2, 0x422f, 0x43e9, 0xb0a2, 0xbf3a, 0xa951, 0xb051, 0x1995, 0x42af, 0x4099, 0x40bb, 0x4221, 0x40b8, 0x4234, 0x422e, 0x444a, 0x2d03, 0x145f, 0x0cf6, 0x438c, 0x4358, 0x41e5, 0x4336, 0x0b76, 0x43e5, 0x4305, 0xb0fe, 0x412c, 0xa59e, 0x3a6d, 0xb23e, 0x4010, 0xbf23, 0x3890, 0x00ee, 0x40ec, 0x1a97, 0x1810, 0x436c, 0x43df, 0x4019, 0xb20f, 0xbfc0, 0x17e0, 0x0e47, 0xbfc8, 0x4068, 0x43bf, 0xbafa, 0x41be, 0x355a, 0xbfc3, 0x18e7, 0xbff0, 0x40e2, 0xbff0, 0x4132, 0x4207, 0xb042, 0x26ad, 0x4329, 0xb285, 0x425e, 0x46da, 0x100a, 0x070c, 0x40d9, 0xb2d3, 0xa9b2, 0x44fa, 0x3edc, 0x1d54, 0x435f, 0x3e4f, 0x2d4a, 0xbfcc, 0x4070, 0x4111, 0x42b6, 0x40f8, 0x210e, 0x3003, 0x41f4, 0x41e1, 0x1e19, 0x4152, 0xba39, 0xbaf6, 0x42aa, 0x4086, 0xa1ff, 0x42d5, 0x4294, 0x226a, 0x407c, 0x1f25, 0x3bea, 0x4158, 0x45bc, 0x445a, 0x29b5, 0xb2ed, 0xba7d, 0xbfa9, 0x41af, 0x31bf, 0x1f06, 0xb277, 0x4600, 0x40a6, 0xba24, 0x4023, 0x413b, 0xbac2, 0x1cdb, 0xba24, 0x00b5, 0x400b, 0x14e9, 0xba47, 0x42a3, 0x1eb7, 0x41b1, 0x42d2, 0x3742, 0xbfdc, 0x17d4, 0x1b82, 0x409c, 0x3640, 0x1a72, 0x4055, 0xba27, 0x4357, 0x2367, 0x43b5, 0x4607, 0xbf59, 0xb0c5, 0x1926, 0x4121, 0xba51, 0xb203, 0xbaf7, 0x066a, 0xb2b3, 0x43a6, 0x432b, 0x40b0, 0x085e, 0x0b22, 0xa46e, 0xbae7, 0xb2f0, 0x1d0e, 0xa37d, 0xba21, 0x40eb, 0x1083, 0x1bb3, 0x4223, 0x43a5, 0xb253, 0x402e, 0x4108, 0xbf37, 0xba0c, 0x43c0, 0xb079, 0x422d, 0x4081, 0x40e0, 0x45b9, 0x43b2, 0x4685, 0x189b, 0x400c, 0x0dbd, 0x425a, 0x0e61, 0xba61, 0x1ad3, 0x4335, 0x41a4, 0xbf64, 0x3714, 0x409a, 0x428d, 0x4421, 0xa9d1, 0xb298, 0x1dd4, 0xb028, 0x4600, 0x128e, 0x42da, 0x1a55, 0xb0da, 0xbf16, 0x423b, 0xba54, 0x42fa, 0x17f3, 0x0753, 0x4299, 0x414a, 0x1e9a, 0xb06c, 0x432b, 0xa23d, 0xba29, 0x4367, 0x19f0, 0x40c1, 0x1e5f, 0xb249, 0x1e21, 0x3078, 0x36fa, 0x2c53, 0xb033, 0x4362, 0x43cd, 0xa40c, 0xbf53, 0xba37, 0x4057, 0xb004, 0x411f, 0xbae2, 0x4671, 0x4276, 0x2505, 0x1044, 0x2743, 0x44cb, 0xaaed, 0x44e5, 0x04b3, 0x41bf, 0x1864, 0x4195, 0x4598, 0xb01d, 0xbf49, 0x4264, 0x4005, 0x0615, 0x43d3, 0xbf21, 0x4175, 0xb218, 0xba29, 0x4168, 0x1b8c, 0x407e, 0xbfa0, 0x4181, 0xb22d, 0x4350, 0x0803, 0x04f2, 0x1a2b, 0xba50, 0xbac6, 0xbf8d, 0x14a4, 0x459a, 0x41d1, 0xba19, 0xba51, 0x1cdb, 0xb283, 0xadd2, 0xb0a3, 0x4140, 0x1e4e, 0xb05a, 0x40ee, 0x428d, 0x37d3, 0x401b, 0x029e, 0x4351, 0x4230, 0x40ae, 0x40ab, 0x430f, 0x4121, 0x197e, 0x4307, 0x43ae, 0xb282, 0xab6a, 0xbf96, 0xb083, 0x41dc, 0x1d8b, 0x4131, 0x43f3, 0xb2af, 0x43e5, 0xb0bb, 0xb06d, 0x10ae, 0x42ef, 0x0d29, 0x4252, 0x1ef1, 0x400f, 0x4287, 0x1f0a, 0x0879, 0x38bd, 0x4306, 0x1094, 0x1b2a, 0x42ec, 0x26b2, 0x42fb, 0xbfbf, 0x1e68, 0x400f, 0x4261, 0x4248, 0x4301, 0x03a6, 0x401d, 0x43a6, 0x42ff, 0x412f, 0xb078, 0x2f4c, 0x059e, 0x404d, 0x40d0, 0x4033, 0x4067, 0x41d5, 0x40f7, 0x4044, 0xbff0, 0x4183, 0xbf23, 0xb208, 0x1922, 0x43cb, 0x2b84, 0xbf46, 0x1c9d, 0x2a1d, 0x41f7, 0x43d2, 0x4297, 0x11e1, 0x3929, 0xba74, 0x1b10, 0xb267, 0x1dc6, 0x403a, 0x420d, 0xb246, 0xb2e7, 0x0450, 0x04fc, 0xbf0e, 0x4680, 0xb015, 0x0f61, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x9761e594, 0x04a76bb3, 0xdf9b4478, 0xd3a7ec97, 0xb0bb8660, 0x52cb52ca, 0x60d713aa, 0x30959c7c, 0xca01d692, 0xd5efbf25, 0x11a77f9a, 0x39816f99, 0x60fa61fd, 0xf4de7402, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x00000000, 0xffffffd5, 0x00000000, 0x9e100020, 0x00000000, 0x9e100022, 0xffffffa5, 0x00000000, 0x00000000, 0xf4de7411, 0xd5efb38f, 0xcace1224, 0x60fa61fd, 0x60fa67ad, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x203c, 0x4138, 0xb29a, 0x402d, 0xbf0a, 0x4151, 0x42a8, 0x4489, 0x4091, 0xb2e9, 0xbf8d, 0xb205, 0x41e3, 0x08fb, 0x048b, 0x401f, 0xb2c6, 0x4106, 0xbf00, 0xba13, 0x391f, 0xba43, 0x4000, 0x407c, 0xba09, 0xb2bd, 0xbf25, 0x13a2, 0xb2ca, 0xaf42, 0x4327, 0xb2d4, 0xb2be, 0xaf98, 0x416c, 0x2cbd, 0x41c7, 0x401e, 0xb281, 0x4107, 0x42d2, 0xbfd2, 0x1939, 0xb248, 0xa7cd, 0x1ac5, 0xbf00, 0x2d7b, 0xb260, 0x43c6, 0xb28f, 0xb29a, 0x3844, 0x3701, 0xb08b, 0x1d56, 0xbff0, 0x42aa, 0xb2d2, 0xbfd8, 0x21a3, 0x34fc, 0x4617, 0x41a3, 0x284d, 0x447c, 0x42dc, 0xba10, 0x42ad, 0x1858, 0x1b99, 0x009e, 0xba48, 0xaf30, 0x413b, 0xbf6b, 0xba13, 0x1e8b, 0x29ae, 0x40bb, 0x1f0d, 0x1bf6, 0xb0fa, 0xb085, 0x1c93, 0xae0c, 0x44ad, 0xb294, 0x4327, 0x1ea8, 0x0fbb, 0x4172, 0x43e1, 0xb274, 0xab3a, 0x421e, 0x4003, 0x4370, 0x189a, 0x4289, 0xbf1d, 0xba66, 0x1d6f, 0xb2da, 0xbfb0, 0x4017, 0xa419, 0xba6f, 0x127d, 0x43f9, 0x0b3c, 0x1a9d, 0x1c19, 0x2d38, 0xbfa0, 0x38c1, 0x406f, 0x181c, 0x431a, 0x4069, 0x4359, 0xaa68, 0x1e23, 0x0012, 0x047a, 0xbf6e, 0x15c1, 0x4572, 0x4344, 0x0ff8, 0x46e4, 0x41db, 0x3216, 0x421c, 0x4195, 0x23cb, 0x4261, 0xa31a, 0xbad8, 0x06e8, 0x4197, 0xbf69, 0x420d, 0x228b, 0xbae0, 0x302d, 0x4046, 0x4016, 0x4137, 0xbf79, 0x2c74, 0x407b, 0xa65d, 0x401b, 0x43ea, 0x09f7, 0x41ad, 0x401d, 0x3324, 0xb2cb, 0xa421, 0x40b4, 0x43f1, 0xbfbe, 0x3ae0, 0x45c1, 0x4014, 0x41c2, 0x4596, 0xbaf1, 0xbf1f, 0x4035, 0xb29e, 0x4070, 0xb0cc, 0xbfb4, 0xbad1, 0xba09, 0x1b93, 0x414a, 0xbf43, 0x40b8, 0xb0ef, 0x41c2, 0x4157, 0x36c8, 0x42b7, 0x40dd, 0x43ab, 0x0d39, 0x32b9, 0x1cfc, 0x1c79, 0xa753, 0x4613, 0x418a, 0x40e8, 0xb25e, 0x0bf3, 0xbf9f, 0x4100, 0x3e0f, 0xb205, 0xb2d7, 0x12b8, 0x430e, 0x4032, 0x42a7, 0x18fe, 0x1ac1, 0x221c, 0xb086, 0x0cf7, 0x4367, 0x426e, 0xabea, 0x4282, 0x1da5, 0xaca3, 0x4277, 0xbf24, 0x4308, 0xb035, 0x41c0, 0x1434, 0xaf0d, 0xbfa0, 0xb20a, 0xbacd, 0xb2e8, 0x0248, 0x40cb, 0x41be, 0x43c5, 0x43a2, 0x40cb, 0x0c9a, 0x2979, 0x0138, 0x1f54, 0x4166, 0xb2de, 0x1e7e, 0x4132, 0x4194, 0xb2a8, 0xbfc3, 0x419d, 0xbadc, 0x43db, 0x4363, 0x42a6, 0xb02d, 0x40c1, 0x32dc, 0x467d, 0x43b0, 0x4581, 0x431c, 0x4232, 0x3daa, 0x436d, 0x4288, 0x40f7, 0x1554, 0x43f1, 0xbf0d, 0x0509, 0x2371, 0x4387, 0x42bc, 0x1f69, 0xb07f, 0x43e2, 0x29d5, 0x41d7, 0x297a, 0x4285, 0x09c3, 0xb2e8, 0x417d, 0xba00, 0x42d1, 0x4286, 0xb203, 0x40c7, 0x46e4, 0x4558, 0x40d4, 0x12b2, 0x3bf4, 0xb09a, 0x1a12, 0xbf48, 0xbaf4, 0x1858, 0x412f, 0x051a, 0x21cf, 0x4118, 0xbaf9, 0x1a50, 0xb245, 0xbf78, 0x43bd, 0x416a, 0x4178, 0xae37, 0x425a, 0x4038, 0xa7bb, 0x414d, 0x1bcd, 0x395a, 0x411d, 0xba0a, 0x41d4, 0xbf80, 0x059e, 0xb0bb, 0x0522, 0xb276, 0xb2f9, 0x2920, 0xa495, 0xbac2, 0xbfe4, 0xbfe0, 0x4081, 0xb0f1, 0xb258, 0x45b1, 0xa68e, 0x428f, 0xb015, 0x11ae, 0x0d68, 0xb255, 0x44e4, 0x43d2, 0x42ae, 0x310f, 0x1e7e, 0xb030, 0x2a4f, 0xbfa1, 0x1f11, 0xb213, 0x4237, 0x1e41, 0x43d5, 0xb2e0, 0x425f, 0xbf00, 0x432a, 0x0299, 0x18de, 0x434f, 0x4304, 0xb06a, 0x33aa, 0x1e4d, 0xba46, 0x42ca, 0x4115, 0x4116, 0xa597, 0x30fc, 0x10be, 0x41a3, 0xbf48, 0xa37b, 0x40eb, 0xb0e4, 0xb28b, 0x4002, 0x416c, 0x40aa, 0x461b, 0x43b9, 0x18d6, 0x4276, 0x4447, 0x4692, 0x464b, 0x427f, 0xba74, 0xba05, 0x2045, 0x1bad, 0x4097, 0xa1f9, 0x41a4, 0xb2e0, 0xbf34, 0xba77, 0x1ef1, 0xba6f, 0x435f, 0x44fb, 0xbf74, 0x4247, 0x4275, 0x1e8d, 0x40bc, 0x2747, 0x4001, 0x419c, 0x20f8, 0x428d, 0x02eb, 0xbac8, 0x0ba6, 0x44ac, 0x41ce, 0x4314, 0x0bdb, 0x1251, 0xb256, 0x423b, 0x4089, 0x4066, 0xbf63, 0x1cd3, 0x4095, 0xba21, 0x430c, 0x3453, 0x42b6, 0x0ace, 0x4181, 0x40ff, 0x4215, 0x43a8, 0x4244, 0x1aa6, 0x1a97, 0xa74d, 0x428c, 0x400a, 0xbf99, 0x1c2b, 0x2b70, 0x437c, 0x408d, 0x4367, 0xbf68, 0x3c30, 0x4382, 0x4227, 0x4266, 0x0e7d, 0xbae3, 0x1ab4, 0x43aa, 0x45f4, 0x4248, 0x15a1, 0x4152, 0xb2e6, 0x1a40, 0xa73a, 0x424e, 0x0b31, 0x1b75, 0xae32, 0xba41, 0x1b3a, 0x45d9, 0x1b7a, 0xbfd7, 0x4326, 0x46aa, 0x40b1, 0xb233, 0x42c9, 0x1a6b, 0xa894, 0x43ca, 0x433d, 0xbf45, 0x2f43, 0x421c, 0x1234, 0xb0be, 0xba6d, 0x1cc2, 0x43d3, 0xb248, 0x1f29, 0xb2e3, 0x4365, 0xbace, 0x31e2, 0xbf63, 0xa256, 0x1be2, 0x4420, 0x4339, 0xba0b, 0x1d57, 0x4694, 0xb28d, 0xbfad, 0x407f, 0xa64a, 0x4308, 0xbfd0, 0x443f, 0x05d3, 0xbfa3, 0xb27f, 0x3293, 0x40df, 0x41ff, 0x42e4, 0x4213, 0xb04d, 0x3ee4, 0x41d8, 0x1e59, 0xa2f5, 0x049c, 0xb2e5, 0x406f, 0xab82, 0x428f, 0xba3f, 0x41db, 0x415f, 0xbfe0, 0x406c, 0x46c8, 0xb29e, 0x1834, 0x067a, 0x4142, 0x433c, 0xb2d4, 0x045c, 0xbfdd, 0x43cb, 0x4105, 0x307f, 0x42a5, 0xaa0d, 0x1152, 0x4082, 0x338b, 0x1c40, 0x44a3, 0xbf67, 0xb28c, 0x1cfd, 0x1a7c, 0xb289, 0xbaca, 0x4056, 0x0316, 0x459d, 0x188c, 0x4341, 0x424e, 0xaaac, 0x0908, 0x41b4, 0xb233, 0x4150, 0xb2d1, 0x2241, 0x179d, 0xbfc7, 0x4249, 0xa363, 0x4179, 0x4125, 0x40d2, 0x429e, 0x41a1, 0xb23f, 0x386c, 0x0051, 0xba75, 0xba3d, 0xbfce, 0xb000, 0xbae5, 0x4402, 0x4366, 0x4374, 0x4546, 0x4185, 0x3555, 0xbf7b, 0x40ed, 0x39b8, 0x43f7, 0xb05d, 0x40cd, 0x4176, 0xbfd0, 0x2425, 0x1f85, 0x1b56, 0x4265, 0x403c, 0x4043, 0x4308, 0x4164, 0xb20f, 0xb2f7, 0x4336, 0x26a7, 0x026a, 0x45da, 0x236e, 0x4342, 0xa109, 0xa913, 0xbfd7, 0x2392, 0xba4b, 0xb270, 0xba5a, 0xb02f, 0xa1c0, 0xba4a, 0x41c4, 0x411e, 0x20bf, 0x4331, 0x1519, 0x1987, 0x1931, 0x44c5, 0x1546, 0xbac4, 0xbf48, 0xb2eb, 0x403f, 0x237f, 0x435e, 0xb229, 0x1cac, 0xbae2, 0xbacf, 0x42c9, 0x42d2, 0x0630, 0x22af, 0x407b, 0x43d6, 0x1a64, 0xba0e, 0x3269, 0x364a, 0xba0d, 0xae94, 0x4310, 0x357b, 0x41f1, 0xbfc9, 0xb2d1, 0x434d, 0x4223, 0x41b8, 0xb2b9, 0xbf80, 0x41af, 0x401a, 0xbf9e, 0x062f, 0x39e3, 0x4007, 0x41ff, 0x419a, 0x4379, 0x39f6, 0x43be, 0x2e46, 0xbadb, 0x1dc4, 0xa8b1, 0x1e25, 0x431d, 0x42af, 0x41f8, 0x232f, 0xba70, 0x4018, 0x4111, 0x4160, 0x40a1, 0xbfe8, 0x4001, 0x42a3, 0xab81, 0x420a, 0x41c6, 0x4366, 0x40f4, 0x3117, 0x4065, 0xba39, 0xb2f4, 0x1a3c, 0xb2e3, 0x435e, 0xb246, 0x423d, 0x43de, 0xbfcf, 0x42f2, 0x424c, 0x328b, 0xba0e, 0x1f34, 0x462b, 0x456b, 0xbadb, 0xb277, 0xbfb4, 0x40d1, 0xbafa, 0x4248, 0x127c, 0x324a, 0x1b77, 0x40c6, 0xba06, 0x42f0, 0x4043, 0x0313, 0x4542, 0x4084, 0x368e, 0xbf24, 0x4294, 0x1c3c, 0x408d, 0xb2b2, 0x314f, 0x1c72, 0x4267, 0x430a, 0xa020, 0xb0fa, 0x1aa7, 0xb229, 0x41af, 0x404a, 0x12c8, 0x4302, 0xb2b8, 0x406d, 0xbac6, 0x1a94, 0x4306, 0xb284, 0x41a1, 0xbf77, 0xb206, 0x41af, 0x3aea, 0x438f, 0xadb4, 0x1cc3, 0x11cd, 0xbac1, 0x1c4a, 0xa4b3, 0x415b, 0x3cba, 0x43e8, 0x41c0, 0xb29a, 0xb2a1, 0xa433, 0xbf73, 0x4553, 0x4198, 0x4243, 0x21e5, 0xb2a4, 0xbf90, 0x4319, 0x40e5, 0x433b, 0x423d, 0x42a7, 0x4149, 0x4346, 0x1abf, 0xba72, 0xb086, 0x40bf, 0x2f64, 0x43e8, 0x064e, 0xb2bd, 0x4617, 0xbfb5, 0xb2cf, 0x3673, 0x4608, 0xba4b, 0x1f3b, 0x1bcf, 0xbfb0, 0x42f8, 0x42d4, 0x4389, 0x40f2, 0x0335, 0x4337, 0x425b, 0xb26c, 0x422e, 0xb241, 0x42a4, 0x42c0, 0xbf01, 0x362f, 0x38e0, 0x42ee, 0x1944, 0x4312, 0xb2db, 0x12e0, 0x46f0, 0xba2e, 0xb0c2, 0x42c5, 0x1838, 0xbff0, 0x1d87, 0x462f, 0x428f, 0x411a, 0x1dc6, 0xa2d1, 0x45ea, 0x4261, 0x17a2, 0x4012, 0x400d, 0x1ae6, 0x4284, 0x1abf, 0xbf36, 0x40f0, 0x4389, 0x0c78, 0x1d1d, 0xb203, 0x407f, 0x14b5, 0xb2b3, 0x0fce, 0xba7c, 0xb2c9, 0xb25e, 0x2199, 0x419a, 0x4188, 0x430b, 0xb0ef, 0xba69, 0x4171, 0xb212, 0x41f1, 0xba4c, 0xbf8c, 0x45c4, 0xb211, 0x4266, 0x436a, 0x43de, 0xb0b4, 0x41e7, 0xb0d5, 0x424a, 0xba7f, 0xb212, 0x40c1, 0x4323, 0x4241, 0x42f0, 0x42d5, 0x4269, 0x433e, 0x1b80, 0x3715, 0xaafc, 0xba72, 0x4611, 0x1d89, 0xbfaa, 0xb0f9, 0x3a65, 0xb220, 0xb256, 0x4498, 0x40b6, 0x43af, 0x4334, 0x3daf, 0x43ee, 0xb2b8, 0x40fb, 0x3140, 0x4092, 0x412a, 0x4242, 0xaa96, 0x43b0, 0xb0b3, 0x1ccc, 0xbfe8, 0x4139, 0x4377, 0x41ad, 0x187d, 0xbfa4, 0xba15, 0x4166, 0x41a4, 0x3fb1, 0x4005, 0xb2c7, 0x413d, 0x3917, 0xbfe0, 0x41a5, 0xb211, 0x433d, 0x05fd, 0x4409, 0x4475, 0x41cc, 0x1e64, 0xbf89, 0xb28f, 0x1a04, 0xbf90, 0x42b3, 0xb2cc, 0x41a2, 0xa53c, 0xbfc0, 0x19b0, 0x4337, 0xbf48, 0xb282, 0x1f0e, 0x4370, 0x4358, 0xb21a, 0xbad7, 0x4043, 0x431f, 0x4551, 0x4685, 0x19cf, 0x4053, 0x408a, 0xb020, 0x43fd, 0x2dc1, 0x4603, 0xbf0f, 0x0b17, 0x415b, 0xac11, 0x407c, 0x4362, 0xba64, 0x4216, 0x1384, 0x09e0, 0xbad7, 0xba17, 0x426d, 0x438e, 0x33d3, 0x406d, 0x3b44, 0xb210, 0x2ec5, 0x40ab, 0x42e8, 0x195e, 0x1577, 0x425c, 0x4159, 0x42d0, 0x14ae, 0x1cb1, 0x4081, 0x1a6a, 0xbf57, 0xb2f2, 0x4301, 0x2184, 0x41b0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x67423155, 0xe3599f1a, 0xf8825cc1, 0xb4b9fc16, 0xcf447746, 0x4ebf74b8, 0x62e4bf2f, 0x5a51ed64, 0xce8f8c9f, 0x5b8b5b0d, 0xed67892d, 0xb797f36a, 0xb91b1a79, 0x89af40a1, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000002, 0xfffffffe, 0x43e44438, 0xbc1bbbc8, 0x00000000, 0x00000000, 0x0000021f, 0xff1bffff, 0x5b8b5b0d, 0xfffffff5, 0x5738068c, 0x89af3e59, 0x21f22254, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x351e, 0xb20f, 0x43d6, 0xba50, 0x40d6, 0x40c0, 0xb0f4, 0x41e8, 0xb0e2, 0x4618, 0xb07e, 0x4103, 0xba3c, 0xbafb, 0x0552, 0x41d7, 0xbfab, 0x41e3, 0x40ae, 0xa6e2, 0x2bec, 0xb23a, 0x007d, 0x4264, 0xb07d, 0x432c, 0xbfe8, 0x4279, 0x44d1, 0x225c, 0x21f0, 0x0f23, 0x426d, 0x0b9f, 0x429c, 0x4275, 0xba66, 0x0a83, 0x4227, 0x31d8, 0x1ca1, 0xbfc5, 0x1fc2, 0xbadf, 0x4359, 0x4171, 0x4122, 0x0782, 0xbf51, 0xb06b, 0x4121, 0x408c, 0x3364, 0xb00d, 0xba44, 0x1aaa, 0x43b5, 0x4640, 0xbaf9, 0xaf1b, 0xb2eb, 0xba59, 0x423a, 0x0a7a, 0xb0fa, 0xb2cf, 0x45c8, 0x42e2, 0x2a0c, 0x1c8f, 0xbac4, 0xbf08, 0xb22e, 0x425b, 0xb08a, 0x43f2, 0xbfb7, 0x42fa, 0x4212, 0x1a85, 0x4292, 0xbacc, 0x34d2, 0xb260, 0x1fb1, 0xa08c, 0xbae8, 0x31bf, 0x40fb, 0xa24b, 0xb270, 0x4042, 0x43a4, 0x44b4, 0xb0a3, 0x26c1, 0xbf34, 0x43dc, 0x151d, 0x1be3, 0x40ec, 0xb224, 0xb280, 0x29f4, 0x1afe, 0x4248, 0xbf1a, 0x4282, 0x3389, 0xb25a, 0x1c1f, 0xbf48, 0x3fe4, 0xb25f, 0xa162, 0x2921, 0xbafa, 0x401a, 0x410b, 0xbf0b, 0x42e9, 0xb28a, 0xb270, 0x1a56, 0x2b14, 0x1d39, 0x431f, 0x1f4e, 0xb28e, 0x4177, 0x1a5f, 0xa81b, 0xa73a, 0x4392, 0x416c, 0x420b, 0x4119, 0x40ff, 0xbf65, 0x4039, 0x41cb, 0xb098, 0x41f1, 0x4126, 0xba70, 0xa037, 0x0686, 0x31f0, 0xba02, 0x2c1b, 0xa41f, 0xbf9f, 0x4037, 0x417c, 0x4371, 0x1c48, 0x39f5, 0xb0a2, 0x4271, 0xb201, 0x3107, 0xbf1b, 0x42a8, 0x42c0, 0x465b, 0xb20c, 0x2b6d, 0xbf6c, 0xb2d6, 0xb274, 0x1344, 0x4605, 0xb2a4, 0x359b, 0xba1d, 0xb2fb, 0xbf1b, 0x1dd4, 0x1d7e, 0x4679, 0x4591, 0xba75, 0xbf60, 0x42ce, 0xba57, 0x160d, 0x415e, 0x085d, 0x18d1, 0x1f0b, 0xbfd6, 0x429a, 0x411f, 0x4079, 0x4682, 0xbfbc, 0x44a4, 0x416f, 0x4029, 0x1975, 0x294b, 0xbad1, 0x43f0, 0x2017, 0x4083, 0x416e, 0x41f7, 0xbfd0, 0x45e0, 0x3c14, 0x4647, 0x44f1, 0x4681, 0xb034, 0x4433, 0x4192, 0x4281, 0xb08e, 0xba31, 0xbfb7, 0x427f, 0x4630, 0xbad4, 0x4379, 0x40c7, 0x4318, 0x3267, 0xb2e5, 0x4280, 0x42ee, 0x059f, 0x4054, 0x4585, 0x456d, 0xa855, 0x42f3, 0x0953, 0xba44, 0xb040, 0xb2af, 0x426d, 0x4635, 0x41a1, 0x405a, 0x1d67, 0x1941, 0x43fe, 0x425a, 0xbfba, 0x32e7, 0x1621, 0x41ef, 0x3fae, 0x1be4, 0x036f, 0x40d0, 0x430c, 0xb02e, 0xbace, 0x4060, 0x4299, 0xba14, 0x4557, 0xb22d, 0xbf4f, 0x4188, 0xba3a, 0xb2a1, 0x419f, 0x408e, 0x0dbf, 0xb2de, 0xaa6f, 0x401e, 0x4315, 0xb2f2, 0xbad9, 0xb0c6, 0x0710, 0xb025, 0x431e, 0x42b7, 0x1f99, 0xbf5d, 0x4273, 0x45f4, 0xb2ba, 0x410a, 0x19da, 0xb21e, 0xbafe, 0x32a4, 0xb24e, 0xbafc, 0xa511, 0x4175, 0x4209, 0x15ec, 0x4203, 0x4016, 0x4451, 0xbafd, 0x25f3, 0x409d, 0x1c68, 0xbfd7, 0x43b2, 0x41c1, 0x432d, 0x428c, 0xba10, 0x403a, 0xa089, 0x245f, 0xbf59, 0xb2a4, 0x151d, 0x38b5, 0xa654, 0x44f9, 0x44a3, 0xbfca, 0xb216, 0x1314, 0xba41, 0x289e, 0x0797, 0x40df, 0x409a, 0x430f, 0x40a5, 0x1e10, 0x40d4, 0x41a7, 0x11f6, 0x4384, 0xbf90, 0x42f6, 0x42c5, 0x403c, 0x43a9, 0xa104, 0x1d69, 0x417b, 0xa553, 0xbfe1, 0xb2e2, 0x24d6, 0x4095, 0x1fc7, 0xb243, 0x19ca, 0x419d, 0x40ca, 0x4039, 0x4145, 0xbfa0, 0xbfbb, 0xb233, 0xbf00, 0x40cf, 0x0ac0, 0xa120, 0x1f27, 0x094c, 0x4121, 0xbad4, 0x18a6, 0xbf05, 0x4333, 0xaca2, 0x19b2, 0x0f87, 0xbadf, 0x1653, 0xb251, 0x0c0e, 0xb027, 0x423d, 0x44f3, 0x43cf, 0xb06d, 0xaa68, 0x43d7, 0xaf3d, 0xaac6, 0x30bc, 0xae02, 0x0af9, 0x0a91, 0x40ba, 0x42eb, 0xb271, 0x42a8, 0xbfb8, 0x46d5, 0xa671, 0xb0ea, 0x43d2, 0x1a79, 0x35a2, 0xba41, 0x1a5a, 0x02ce, 0x4339, 0x4053, 0xbfd9, 0xa3d6, 0x2e6e, 0x416f, 0x43f7, 0xbafd, 0x427c, 0xbf76, 0xa945, 0x20fd, 0xba28, 0xb0b4, 0x4062, 0x21e7, 0xb09c, 0x4078, 0xa702, 0x41ce, 0x417a, 0xb219, 0x4196, 0xbf3d, 0xbaf7, 0x192c, 0xa208, 0x425e, 0xb2c4, 0x400a, 0x40f1, 0x4304, 0xb0e2, 0x4031, 0xb211, 0x42c3, 0x0eca, 0x46f1, 0x4072, 0x1c40, 0x40f6, 0x1b35, 0xb2f8, 0x43e3, 0x28f6, 0x43a3, 0x419a, 0x4337, 0x4311, 0x40ff, 0xbfb9, 0x312e, 0x4189, 0xa2f4, 0x45e9, 0x42f6, 0x1db4, 0x0c19, 0x44f1, 0xba6c, 0x425f, 0x4108, 0x41ba, 0x0903, 0xba53, 0x16c7, 0x4112, 0x044b, 0xb02f, 0xb21d, 0x389c, 0x420b, 0x424e, 0x309a, 0xbf87, 0x4161, 0x1fa5, 0x40cf, 0x4060, 0xbfdd, 0x4183, 0x1107, 0x3376, 0x433b, 0xa8eb, 0x352d, 0x1918, 0x0664, 0x4225, 0x401b, 0x42e0, 0xb208, 0x46a8, 0xbf38, 0xb277, 0x0fd4, 0x434e, 0x16bf, 0x43ea, 0xb2f3, 0x4124, 0x198f, 0xba5b, 0x43be, 0x1e44, 0xb2ae, 0xa745, 0xbfde, 0x405d, 0x4151, 0x41ce, 0x1c07, 0x41a7, 0x4166, 0x40fc, 0x38fe, 0x41d9, 0x4084, 0x4128, 0x1817, 0x433a, 0xbf18, 0xb269, 0x420f, 0x442e, 0xb020, 0x42e4, 0x182e, 0x2237, 0xb067, 0x4225, 0x41b4, 0x40cc, 0xbf88, 0x45d2, 0xba2b, 0x2749, 0xadb9, 0x1b34, 0x40e8, 0x0ff1, 0xb20a, 0xba70, 0x3396, 0xb261, 0x4296, 0x4387, 0x4002, 0x436b, 0x435f, 0x434c, 0xbaf7, 0x4077, 0xbfe1, 0x1f0d, 0xba30, 0xba16, 0x4085, 0xb2ad, 0x4029, 0xba54, 0x1c2f, 0xb27b, 0x0dea, 0x4233, 0xb202, 0x0952, 0xba7e, 0x31f6, 0x43fc, 0x4117, 0x45c9, 0xae39, 0xbf3f, 0x4226, 0x43b7, 0x4221, 0x44f2, 0x4157, 0x1efc, 0xbf32, 0x40ab, 0xbfc0, 0x2669, 0x4388, 0xb0a9, 0x406c, 0x28ca, 0x15e3, 0x4337, 0xba26, 0xad3c, 0x106b, 0x340e, 0xbff0, 0xba04, 0x412d, 0x415d, 0x410e, 0xbfbd, 0x427f, 0xb232, 0xac33, 0x4042, 0x40cc, 0xa690, 0x42ce, 0x2cf1, 0xbaf3, 0x40ae, 0xba6b, 0xba26, 0x0b5f, 0xb2e8, 0x4105, 0xbaf2, 0x4348, 0x4240, 0x1f5b, 0x408f, 0xbf38, 0xb282, 0x1c4d, 0x45ab, 0x43c5, 0xb253, 0xb26d, 0x1bfe, 0xbfb5, 0xb018, 0x366a, 0x4065, 0x122e, 0x408b, 0x313d, 0xba7e, 0x4002, 0xbacc, 0x4040, 0x40c3, 0x42dd, 0xb276, 0x0d1e, 0x43ab, 0xba2d, 0x190e, 0x4082, 0xb29e, 0xba38, 0x2855, 0xb074, 0x1ca5, 0x4337, 0x1d60, 0x1cfd, 0xbad9, 0x046c, 0xbfc1, 0xbfd0, 0x42d1, 0xb211, 0x1c31, 0x4694, 0xafe8, 0x43c6, 0x4303, 0x468d, 0xb214, 0x00fa, 0xb03c, 0x4280, 0x43c6, 0x4637, 0xbfde, 0x07a2, 0x4044, 0xba6b, 0xb096, 0xa983, 0x4592, 0x413a, 0x0518, 0x46e2, 0xa242, 0x25c1, 0x4103, 0xba62, 0xb001, 0x455f, 0xbf47, 0x11f2, 0xacab, 0x3992, 0xbad6, 0x439e, 0x4388, 0xb2b8, 0xba17, 0x432a, 0xb221, 0xba03, 0x1730, 0x43cf, 0x4265, 0xba50, 0x1b03, 0x428f, 0xbf2e, 0xb20c, 0x458c, 0x4390, 0x1afe, 0x42b9, 0x1d65, 0x4463, 0x42b5, 0x0dff, 0x4095, 0x42f6, 0x3496, 0x4441, 0x0c8a, 0xba61, 0x4325, 0x20f4, 0xb09c, 0x1c27, 0x420f, 0x40e0, 0xbf70, 0x21c7, 0x4240, 0x459b, 0x40a0, 0xb2ae, 0xbfd2, 0xac93, 0xb0b0, 0x31e9, 0x4131, 0x185d, 0x417c, 0xba60, 0xb24b, 0x2874, 0x40ef, 0xb206, 0xb2e9, 0x44fc, 0xb074, 0xb03b, 0x43a8, 0x426a, 0xba38, 0x43ce, 0x443f, 0xbf4b, 0x4141, 0x1f3c, 0xb2b6, 0x430b, 0xbfcf, 0x4292, 0xba52, 0x4172, 0xa9f9, 0x3050, 0x44e0, 0x43b0, 0x43e8, 0x40b3, 0x4257, 0xba6f, 0x4006, 0x4036, 0x4320, 0x46f5, 0x41f6, 0xa8e5, 0xba6a, 0x429d, 0x0a20, 0x19d7, 0x467f, 0xb20c, 0x400d, 0x4007, 0xbfaa, 0x4367, 0x21b7, 0x4142, 0x428b, 0x4369, 0x45a0, 0x2abb, 0x1af5, 0x4216, 0x1aee, 0xb02b, 0x121e, 0xb241, 0xbaee, 0x2d8b, 0x40d2, 0xbf69, 0x4117, 0x4366, 0x2bbd, 0xba10, 0xb09f, 0x4172, 0x426c, 0x23fc, 0x338e, 0xbf43, 0x4015, 0x42ee, 0xa3f1, 0x0369, 0x42e7, 0x062e, 0x1667, 0xb23b, 0x4433, 0x438e, 0xb0c9, 0x418e, 0xb0d8, 0xba4c, 0xbf13, 0x42e0, 0x1c20, 0x19f8, 0xb232, 0x4083, 0x410f, 0xad20, 0x412c, 0x3662, 0x2a58, 0x4089, 0xb275, 0x117c, 0x3b0c, 0xbf5d, 0x41c1, 0x4011, 0x4063, 0x2178, 0xab2f, 0x42a1, 0x439e, 0x2b3e, 0x45e2, 0xb084, 0xadfb, 0xa056, 0xbaf7, 0x3e15, 0x4464, 0xb046, 0xb24d, 0xb09d, 0x4202, 0x19f1, 0x4211, 0xbf5e, 0x4035, 0x400f, 0x1e45, 0x13ab, 0x40aa, 0x1ad0, 0x429f, 0xb0a4, 0xb05b, 0x0f39, 0xaa55, 0x2864, 0x437f, 0x43ca, 0xb2d1, 0x3846, 0x46e0, 0xb0d4, 0x4411, 0x1ce1, 0xb06d, 0x1246, 0x187d, 0x08ca, 0x46da, 0xbf95, 0xb273, 0xba11, 0x4211, 0x40d3, 0xb040, 0x05fd, 0x1e61, 0xa78d, 0x46f3, 0xb0e7, 0x4168, 0xba4c, 0xba0b, 0x000f, 0x41a0, 0x340c, 0x29e0, 0x2c2e, 0x42ab, 0xb01b, 0x421e, 0x2607, 0x1b58, 0xba0f, 0x442a, 0x4219, 0xb2ac, 0xbfd4, 0x414a, 0x0524, 0x3d35, 0xb0a0, 0x4343, 0x40e8, 0x40a9, 0x18f6, 0x14df, 0x41d4, 0x4542, 0x434a, 0x4306, 0x36f2, 0x1d38, 0x4595, 0x41a2, 0x42fd, 0xbf24, 0x4596, 0xa565, 0x4650, 0xb264, 0x4139, 0xb039, 0x4261, 0x41ae, 0xbaf5, 0x4112, 0x43c3, 0x0409, 0x4168, 0x40e4, 0xbfa8, 0xba31, 0xa175, 0x41ac, 0x2089, 0x423e, 0x40fe, 0x411e, 0x2535, 0x42e2, 0xa13b, 0x44c0, 0xb20f, 0xbf6e, 0x43be, 0xb2a5, 0x40bb, 0x4068, 0x19bb, 0x459d, 0x2ab5, 0x192c, 0x427e, 0x429e, 0x4286, 0x43ee, 0x415e, 0xb0cd, 0x46c3, 0x45a3, 0x4345, 0x4170, 0x064c, 0xb2bd, 0xbf19, 0xb091, 0x02aa, 0x3a96, 0xb2a8, 0x41a0, 0xb0c3, 0x4327, 0x18c6, 0x014d, 0x402a, 0xb299, 0xb207, 0x0b66, 0x3ced, 0xb2bb, 0x4188, 0x3513, 0xbaff, 0x4356, 0x437c, 0x42f2, 0xba74, 0xb239, 0xba67, 0x42b4, 0xbf3a, 0xb2d8, 0x1c3f, 0x2787, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3be9d470, 0x621bd961, 0x561548a7, 0x32c98a3d, 0x487719ff, 0x1c15d826, 0x34d9255e, 0xaae8da54, 0xf65e2145, 0x5153589f, 0x032f5367, 0xf13554cd, 0xda5c80c1, 0x8ef09f26, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0xffffffff, 0xffff8018, 0x00031000, 0x00001880, 0x00000000, 0x00031013, 0x00000000, 0x00000087, 0x00012ad4, 0x00000000, 0xf135552c, 0x00012ad4, 0x0000956a, 0xfffffd74, 0x00000000, 0x600001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb2f7, 0x4028, 0x42dd, 0x4043, 0x422e, 0xbf5c, 0xa6cf, 0xb289, 0x06ce, 0xb229, 0x4065, 0x430d, 0x1986, 0x466b, 0x2c11, 0x0fcb, 0xba22, 0x412a, 0x40f3, 0xbf9d, 0x0cd5, 0x415a, 0x310f, 0xb07e, 0xb2da, 0xbf12, 0x3f1e, 0x3e0e, 0xb22c, 0x4414, 0xb0a3, 0x1a93, 0xbf15, 0x33ee, 0x1c22, 0x4184, 0x416b, 0x43cd, 0x4625, 0xba70, 0x438f, 0x4175, 0x4031, 0xbf53, 0x0d6a, 0x4234, 0x2c94, 0x4288, 0x4196, 0xb0ef, 0x3a5b, 0xb02b, 0xa285, 0x438c, 0x1914, 0xbfb0, 0xbf82, 0x4636, 0x4550, 0x23ad, 0x466e, 0xb04e, 0x4303, 0x4301, 0x1efc, 0xb267, 0xba31, 0x4247, 0xb01d, 0xbfa3, 0x4026, 0x11c3, 0x41fd, 0x40cc, 0x4150, 0x435d, 0xbf3f, 0xa514, 0xbfc0, 0x4293, 0x1e27, 0x42af, 0x405f, 0x3ad4, 0x40a2, 0x0423, 0x406a, 0x4236, 0xb2a5, 0x1ec3, 0x1f33, 0x29aa, 0x2865, 0x0b77, 0x2536, 0x1e74, 0x40a6, 0x43cb, 0x42e1, 0x402c, 0x4414, 0x25f3, 0x0c28, 0xbf7c, 0x0042, 0x4154, 0xbad1, 0x2d5b, 0x45c1, 0x4399, 0x3a0d, 0x41ea, 0x4398, 0xbfa0, 0xba3a, 0x41b8, 0x1f6c, 0xbf75, 0x1579, 0xb29c, 0x1d42, 0x18ce, 0x16f3, 0x44a3, 0x46b1, 0xbadc, 0x1943, 0xb2d6, 0x24b9, 0x4061, 0x2103, 0x1e64, 0x41c5, 0x41f9, 0xbf59, 0xba55, 0x40c0, 0x414f, 0xa5c1, 0x4189, 0x1f1d, 0x1afd, 0x41b6, 0xba58, 0xba3b, 0xb0b7, 0x2acf, 0x4370, 0x41e3, 0xb2d2, 0xb265, 0xb28b, 0x4133, 0xa98a, 0x403b, 0x3a58, 0x4271, 0x447e, 0x41fa, 0x40c7, 0xba58, 0x4319, 0x41e5, 0xb030, 0xbf49, 0x4074, 0xa3ca, 0xbfb0, 0x1928, 0x41a9, 0xba74, 0x4070, 0xb00f, 0x4225, 0xa687, 0x00f0, 0x406c, 0x4148, 0xbf83, 0x19c0, 0x4197, 0xb2da, 0x424c, 0xb2bc, 0xba23, 0x2e66, 0x4089, 0x28dd, 0xb2e4, 0x05cd, 0x4184, 0x05c6, 0x404d, 0xba2e, 0x0f8a, 0x4175, 0x412a, 0x44cc, 0xb0ef, 0xb042, 0xafe8, 0x420a, 0x4253, 0xbf83, 0x1b27, 0x4062, 0x424b, 0xae9c, 0xba7e, 0xb265, 0xb28a, 0x148f, 0xb2d3, 0x2e6d, 0x2efc, 0xba46, 0x40f7, 0x41fd, 0x411b, 0x4566, 0xb2b1, 0x4209, 0xb218, 0x0aeb, 0xb20d, 0x40c9, 0x226c, 0x429a, 0x22d3, 0xbfb0, 0xb286, 0x4275, 0x42a6, 0xbf96, 0x4235, 0x4222, 0x42e3, 0xb237, 0xb220, 0x4624, 0x1d89, 0xb031, 0x389e, 0x4196, 0xb0b4, 0x426e, 0x1ae5, 0x140f, 0x4398, 0x4667, 0x40e5, 0x0fe5, 0x14fe, 0x18e6, 0x44b2, 0x4285, 0x4044, 0x4233, 0x2a71, 0xb0c8, 0xb06f, 0xa688, 0xbfcb, 0x435e, 0x1af2, 0x1de1, 0x0607, 0xabf8, 0xbfc0, 0x42ef, 0x4220, 0xbf70, 0xb271, 0x3965, 0x4132, 0xab6b, 0x1665, 0xb2c5, 0x1fa6, 0x32f2, 0xb0aa, 0x41ca, 0x2064, 0x0cbe, 0x41a5, 0xbf60, 0x1fa4, 0xb02d, 0x4046, 0xa9a9, 0x41b7, 0xbf43, 0x4167, 0x0b2e, 0xa33a, 0x4202, 0xa8c2, 0x42c0, 0x425b, 0x3f6c, 0xbada, 0x1b7a, 0x40ae, 0x18c0, 0x4132, 0x407a, 0x40cb, 0xba02, 0xad3f, 0x4275, 0x4271, 0x1f2e, 0x1dec, 0xbfde, 0x4544, 0x0f82, 0x214e, 0xb284, 0x4089, 0x4334, 0x4251, 0x2dd9, 0x08e2, 0x431d, 0x4036, 0x09d7, 0x432b, 0x434c, 0xb203, 0xbf70, 0x3ce7, 0x3620, 0xbfa0, 0x155a, 0xbfb4, 0x198a, 0xb28e, 0xbf71, 0x047b, 0x1e2d, 0x42c4, 0xaa34, 0xba55, 0x4073, 0x2661, 0x4106, 0x3527, 0x1a73, 0x333f, 0x430d, 0xb26b, 0x3a6a, 0xb2f2, 0xbade, 0x42c5, 0x1d3a, 0xbf2f, 0xa17c, 0x4156, 0x4267, 0xb207, 0x4581, 0xb0f1, 0x4385, 0x1869, 0x4045, 0xb258, 0x0b70, 0x1f7d, 0xad43, 0x1f8d, 0x4235, 0xb24e, 0xb04c, 0xb0e0, 0x0d27, 0x3468, 0xbf9d, 0x189e, 0x1ccd, 0x43bc, 0x4060, 0xb2d5, 0x45bd, 0x1b74, 0x44dc, 0xbaf8, 0xab61, 0xb2d2, 0xb21c, 0x1e49, 0xafa9, 0x432a, 0x4374, 0x1b4d, 0xb24a, 0x4092, 0xb29b, 0xbf62, 0xb2d2, 0x36fc, 0x4131, 0x1d9b, 0x45f1, 0x0d6c, 0x398b, 0xb2f7, 0x424a, 0x4044, 0x03c7, 0xba2e, 0x1adb, 0x43eb, 0x40e2, 0x06a9, 0x07d8, 0x419c, 0xb291, 0xba06, 0xbf3e, 0xb28b, 0x268a, 0x41b9, 0x1119, 0x432f, 0xb29f, 0xab98, 0x4691, 0xb28e, 0x1c91, 0x128e, 0x4432, 0x435b, 0x1ae9, 0x418b, 0x05fb, 0x426f, 0xbfb2, 0x4306, 0x3ff6, 0x42f0, 0x40a1, 0x0abe, 0xba16, 0x0068, 0x4095, 0x09b3, 0x36a2, 0x4362, 0x4226, 0x407a, 0x2f81, 0xbfe2, 0x3c7d, 0x425b, 0x43f0, 0x218b, 0x3826, 0x00fc, 0x19cf, 0xb2a4, 0xba31, 0x4390, 0x43f4, 0x1d18, 0xb28e, 0xbaeb, 0x41a1, 0xbf80, 0x41e0, 0x1c8a, 0xba1e, 0x4212, 0x435d, 0x2bbc, 0xbf90, 0x432c, 0xbfae, 0x4211, 0xba70, 0x43f9, 0x42d5, 0x1ffa, 0x413d, 0xba40, 0xbfb2, 0x232a, 0x429c, 0x4490, 0x02b5, 0x4059, 0x404c, 0xba41, 0xa995, 0x415b, 0x40f7, 0x4041, 0x43e6, 0xba3b, 0xb2d7, 0x425d, 0x197f, 0x411d, 0xbf68, 0xba27, 0x1922, 0x1f2b, 0xbafe, 0x4454, 0x4310, 0x434c, 0xad88, 0x414d, 0xa4a5, 0xba15, 0xbfbc, 0x435d, 0x41ff, 0x41bc, 0x43fd, 0x1b7d, 0x3bc0, 0xb237, 0x1cf3, 0xba44, 0x4019, 0x439a, 0x4151, 0x4265, 0x406d, 0x4247, 0x2b08, 0x410e, 0x2a27, 0x42c5, 0xb2d0, 0xbf8c, 0x40d2, 0x42b5, 0xb075, 0x4324, 0x40d6, 0x431f, 0x40c8, 0x11fe, 0xa8ea, 0xbf5e, 0x4055, 0x1462, 0x091d, 0x194f, 0x0447, 0x045d, 0x4655, 0x37cb, 0xbf78, 0xba5e, 0x419f, 0x3835, 0x1c9d, 0x404a, 0x427e, 0x43c9, 0xbf60, 0x4276, 0x3259, 0x439b, 0x08a9, 0xb071, 0x4117, 0x44d3, 0xbad4, 0x42f7, 0xba45, 0x4469, 0x18e6, 0x42cd, 0x434f, 0x3f51, 0x433e, 0x45cb, 0xbf6d, 0xb268, 0x43ff, 0xbf90, 0x42c4, 0x40cc, 0x3fd6, 0x3d3e, 0x0b24, 0x40e1, 0xab12, 0xb2bf, 0xb0fc, 0xb05f, 0x43b7, 0x1dc3, 0xbf60, 0x44d8, 0xbf60, 0x43a2, 0xb2af, 0x4137, 0x41d8, 0x403e, 0x42ee, 0x32c0, 0x42a1, 0xba3b, 0x39e8, 0xbf01, 0x1a35, 0x43f6, 0x1e36, 0x4382, 0x40e2, 0xba04, 0x06a1, 0x1cc8, 0xa943, 0x4171, 0x415e, 0x4006, 0xba30, 0x411b, 0xbfc5, 0xac02, 0xb29c, 0xba4c, 0xb232, 0xba60, 0x4198, 0x42fc, 0x1a5e, 0x4269, 0xad0e, 0x0366, 0xa38a, 0x4045, 0x0a7b, 0x3899, 0x1bcc, 0xba2c, 0xb2b3, 0x45c5, 0x2575, 0x44ca, 0x4185, 0x06b8, 0x4185, 0x4018, 0xbfcd, 0x403f, 0x391d, 0xa99c, 0x17d1, 0xb280, 0xba7d, 0xba66, 0x1e75, 0x3ad1, 0x1304, 0xbfcb, 0x1e01, 0xbad0, 0x43cf, 0x4425, 0xb230, 0x43d8, 0xbae1, 0xbad7, 0x13f0, 0x42a0, 0x438c, 0x28f8, 0xb0a4, 0xb0a6, 0x196a, 0x1cc0, 0xbfc5, 0x0c99, 0x40e4, 0x1935, 0x45b4, 0xb011, 0x40ec, 0x40e9, 0x3f82, 0x42c8, 0x1aa3, 0x43be, 0x4034, 0xbf1d, 0xa56d, 0x44da, 0x19e3, 0xb2b1, 0xb2bb, 0x40e0, 0x40ef, 0x3c5d, 0x43f5, 0xb26e, 0xbfd5, 0x402f, 0x4023, 0xba3e, 0x38d4, 0xb255, 0x406d, 0x4244, 0xb06b, 0x42e1, 0xb045, 0x408a, 0x4382, 0x4102, 0x17cb, 0x4670, 0xbf21, 0xb28e, 0x4282, 0xb27c, 0x1f26, 0xb091, 0x42c9, 0x1e3a, 0xbad0, 0xbfe2, 0x1b74, 0x42a6, 0xba44, 0x42b2, 0x4291, 0xb21e, 0x404d, 0x4328, 0x4343, 0xad78, 0x42b3, 0x4169, 0x3557, 0x1b59, 0x4372, 0x403d, 0x41f5, 0xba79, 0x0e81, 0xa713, 0x2378, 0xba21, 0x4050, 0x40d7, 0xb050, 0xa3af, 0x417d, 0x4001, 0xbf95, 0xa8cc, 0x437c, 0x436e, 0x408e, 0x43aa, 0x3957, 0xb021, 0x4209, 0xb272, 0xba1e, 0x42a2, 0x4351, 0x467a, 0x13cc, 0x2db1, 0x466e, 0x4234, 0x43c0, 0xb0dc, 0x1fcc, 0xbf5e, 0x41e2, 0x17ab, 0x4173, 0x4110, 0x427b, 0x41b2, 0xa5af, 0x411e, 0xbaf2, 0xba49, 0x4197, 0x4448, 0xb068, 0x4283, 0x4354, 0xbfa0, 0x40f2, 0x1f52, 0x41d3, 0x4571, 0x1aab, 0xb2b5, 0xaa74, 0x4372, 0xbf26, 0xb2cd, 0x42e7, 0x18a8, 0x373f, 0xba7c, 0x40a7, 0x40ec, 0x1deb, 0x4380, 0xb072, 0x43b6, 0xab0a, 0xbf54, 0x42d3, 0xa8e9, 0x1d8b, 0x1f92, 0x4046, 0x1ae7, 0xafbf, 0x20de, 0x45cc, 0x189d, 0x42f4, 0x43e8, 0xb093, 0x292d, 0x412b, 0x430a, 0xbae8, 0x0ffa, 0x4080, 0x3294, 0x4665, 0x4284, 0x22c5, 0x25bf, 0xb27b, 0xbf3c, 0x3912, 0x2d61, 0xbadb, 0x44a3, 0xbf42, 0xb2a4, 0x411f, 0x1d60, 0x40f5, 0x21a9, 0x0250, 0x0a3c, 0x31e6, 0xbf77, 0x43ea, 0x1ff8, 0xbac4, 0xbac2, 0xb209, 0xbf1b, 0x43b9, 0x0791, 0x3579, 0x4447, 0xba5f, 0xbf92, 0x3c03, 0x4383, 0x4044, 0xa95a, 0xba71, 0xb229, 0x4303, 0x316a, 0x4389, 0x1ff4, 0x422e, 0xb2b3, 0x40e3, 0x4085, 0x3342, 0x42a2, 0x1d76, 0xb200, 0xae67, 0x45b2, 0xbf1d, 0xbadc, 0x27d9, 0x18e9, 0x41dd, 0x294b, 0xbf41, 0x0d48, 0x4137, 0x4015, 0x43f4, 0x317c, 0x42bf, 0x4391, 0x1774, 0x0d7f, 0x40b4, 0xba0f, 0x4111, 0xba30, 0x30fe, 0xb2b2, 0x438e, 0x4012, 0x460d, 0xbfc0, 0x2de0, 0xb2aa, 0x00da, 0x42bd, 0x4385, 0xbfa8, 0x1f7e, 0xb203, 0xb293, 0x2c9a, 0x1ea5, 0x2d34, 0x46aa, 0xb288, 0xbf7d, 0xba02, 0x2b0a, 0xb0a2, 0x43ae, 0x435c, 0x406e, 0xb09f, 0x1484, 0x416b, 0x46b3, 0x4314, 0x422f, 0xa751, 0x44e8, 0x43ae, 0xa901, 0xba66, 0x41ca, 0xbf0b, 0x43af, 0x1faf, 0x1fbf, 0xac1d, 0xac02, 0x235c, 0xbfc1, 0x4145, 0x1b22, 0x1c5c, 0x404a, 0xa600, 0x43b0, 0xbad1, 0xbf4b, 0x1b24, 0xa8e3, 0xba03, 0xaf8e, 0x295f, 0x1b40, 0x4178, 0xb205, 0x146f, 0x09ca, 0x2e3c, 0x43f7, 0x439b, 0x3295, 0x408a, 0xbf1a, 0x454f, 0x3bc1, 0xba5b, 0x4100, 0x40f2, 0x1af2, 0x1c9b, 0xa85c, 0xaf0f, 0xbfb0, 0xa8dc, 0x4218, 0x414a, 0xb205, 0xb2a2, 0x4160, 0xa22c, 0x00b4, 0x42e3, 0xba5b, 0xbf52, 0xba23, 0x40bd, 0xb01d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe4527cc5, 0xffa5922e, 0xca434c0d, 0x2c3146e6, 0x89b4f1a8, 0xd1d30b0d, 0x90e04460, 0x4d312bcd, 0xd0bf479f, 0x23e96a47, 0x987a7688, 0x7251199f, 0x0415194f, 0xea9d51e2, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0xea9d5adb, 0x00001257, 0x0000187c, 0x105e0000, 0x00005e10, 0x00005a7e, 0x00001784, 0xea9d574a, 0xc6274cdf, 0x00000000, 0xfffffffe, 0x8aff0005, 0x766633dc, 0xea9d570e, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4166, 0xba66, 0x4108, 0x43d1, 0x2c5e, 0xb2fb, 0x4067, 0xafc6, 0x1915, 0x25ef, 0xaadd, 0xb016, 0xbac4, 0xba6e, 0x4415, 0x455e, 0xba35, 0xbf09, 0x4108, 0x0586, 0x42f9, 0x41e2, 0x15df, 0x279d, 0x18cb, 0x3d65, 0x40cd, 0xb0fe, 0x430b, 0xb006, 0x40a5, 0xb21d, 0x409f, 0xa0e2, 0xbf8f, 0x4328, 0x4444, 0xbf70, 0xb2be, 0x40b1, 0xb294, 0x4068, 0x414b, 0x13ae, 0x41e2, 0xb21d, 0x43ce, 0xba3d, 0xb26b, 0x1830, 0xb2b0, 0x440c, 0x29fa, 0x4290, 0xba40, 0x37eb, 0x43db, 0x422c, 0x4337, 0x30cc, 0x41cc, 0x4309, 0xb2f7, 0x401d, 0xbf1a, 0x4322, 0x43b8, 0x41b3, 0x33c5, 0x29e3, 0x0c91, 0x4240, 0x419f, 0xbf31, 0x4059, 0x407f, 0xa4cc, 0x179f, 0x42a2, 0xb28b, 0x413f, 0x40b7, 0xb276, 0x42f8, 0x416c, 0xba03, 0x287a, 0xbfa0, 0x3811, 0x4374, 0x44fd, 0x4147, 0x1d8b, 0x2828, 0x13cb, 0x4166, 0x45e0, 0xb206, 0x184e, 0x414d, 0xbfb9, 0xba1c, 0x40c7, 0x41fb, 0x41f5, 0x433c, 0x400e, 0x42da, 0x43a5, 0x1c04, 0x4202, 0x1c75, 0xb20a, 0x41d0, 0x405a, 0xba78, 0x3bb3, 0xbf19, 0xba6a, 0xb223, 0x35a6, 0x40a4, 0x0707, 0xbf61, 0xb2ac, 0x443e, 0x408c, 0x421c, 0x4128, 0x1975, 0xa7e9, 0x09d5, 0x1ffb, 0xb25d, 0x448a, 0x41e9, 0xaf03, 0x1ea5, 0x40bd, 0xb25f, 0x13f0, 0x4639, 0xba67, 0x404e, 0x1701, 0x423f, 0x3cf4, 0x1f10, 0x431e, 0xbfb6, 0xba7c, 0xb2b0, 0x1f0e, 0xba3e, 0x37c8, 0xba46, 0xba74, 0xba30, 0x41b1, 0x4291, 0x40ad, 0xb0ae, 0x4268, 0xb0a1, 0xba4e, 0x0d33, 0x3511, 0x03b1, 0x4383, 0xbf2e, 0x1a28, 0x43ce, 0x0d7a, 0x40e1, 0x4078, 0x13ce, 0xba1b, 0xb0fe, 0x2b59, 0x402c, 0xbfb6, 0xb288, 0xb25a, 0x41ee, 0x4343, 0x1c53, 0xb21d, 0x41af, 0xb2b4, 0x423d, 0xad88, 0x43cd, 0x4382, 0x40fd, 0x1376, 0x3a63, 0x4162, 0x4091, 0xb2fc, 0x43f8, 0x4247, 0x43b5, 0x4622, 0x41da, 0x1e63, 0x40e3, 0x435f, 0x4630, 0xbfd1, 0x19a2, 0x4237, 0x433c, 0x42eb, 0x3f17, 0xbf6a, 0xbacb, 0x4340, 0xb236, 0xbace, 0x43a4, 0xb20b, 0x1f3a, 0x393d, 0x42a8, 0x22da, 0x1b3d, 0x1884, 0x33e2, 0xa02b, 0xb087, 0x2942, 0xb038, 0xbad6, 0x4055, 0x418a, 0x4373, 0x411b, 0x4382, 0x4041, 0x41cf, 0x42fe, 0xbf79, 0x43bf, 0x1c07, 0xbac0, 0xb263, 0xba01, 0x429b, 0x0774, 0xb011, 0x4135, 0x42b9, 0xbf72, 0x4466, 0xbf60, 0x422b, 0x3aea, 0xbf1f, 0x1986, 0xb2f1, 0xa844, 0x41f7, 0xb029, 0x4319, 0x3b38, 0x4017, 0xbf34, 0x1bd1, 0xbfc0, 0x4014, 0x4304, 0xb2e2, 0x1824, 0xb28b, 0x430b, 0xa850, 0x2fdd, 0x1eb3, 0x46f2, 0x1974, 0x428c, 0x43c9, 0x42f1, 0xbf6a, 0x1d2a, 0x1f8c, 0x41ef, 0x440e, 0x4440, 0x40e9, 0x43a1, 0xaa22, 0x15b9, 0xbf25, 0xb2b8, 0x43f6, 0x40a6, 0x42f1, 0xb28a, 0xb23f, 0x4350, 0x418d, 0x42ea, 0xae84, 0xb062, 0xb297, 0x405c, 0x4132, 0x1943, 0x43da, 0xb012, 0x1d91, 0x3514, 0xbf6a, 0x4202, 0x00f2, 0x4243, 0xafb2, 0x414f, 0xbf11, 0x42aa, 0x40bd, 0x416f, 0x4196, 0xb061, 0x4119, 0x1225, 0x439e, 0x0de1, 0x2cc9, 0x1cdd, 0x3081, 0x431b, 0xb21e, 0x1e40, 0x3105, 0xbf98, 0xb0b5, 0xba65, 0xb2e7, 0x42d3, 0x0bab, 0x1fc2, 0xb0ca, 0x436f, 0xb296, 0x330d, 0x4078, 0xb25d, 0x4285, 0x3335, 0x2608, 0x34e2, 0x411d, 0x38b9, 0x1ab8, 0x023f, 0x10e9, 0x095b, 0x41d9, 0xbf60, 0xbf61, 0x4363, 0x1d56, 0x4658, 0x43f8, 0x41ac, 0x3b35, 0x4203, 0x204d, 0xbfb0, 0x4609, 0x413a, 0x4029, 0xba79, 0xba33, 0x456d, 0xb278, 0xa065, 0x40d8, 0x1c95, 0xb0ee, 0xb20f, 0xbf82, 0x404e, 0x4102, 0x4005, 0xafb3, 0x43ae, 0x1adb, 0x1ef9, 0x04e7, 0xb2b1, 0xbf90, 0x3eae, 0x44d4, 0x10e7, 0xbfaf, 0x42b4, 0x41b9, 0x4661, 0xb0ba, 0xbaf0, 0x4310, 0xba07, 0xadfc, 0x370b, 0x4078, 0xa86c, 0x43bf, 0x46a9, 0xb2c3, 0x438a, 0x0e58, 0x2d29, 0xb2c2, 0x0d85, 0x2025, 0x43be, 0xbf95, 0x43e2, 0x0190, 0xb03b, 0x43af, 0xba48, 0xbf60, 0x280a, 0x45c0, 0xb225, 0xbac3, 0x4171, 0x4286, 0x27a7, 0x4084, 0x1b18, 0x3e85, 0x1d9e, 0x222f, 0x2477, 0x1cd9, 0xba7e, 0xbfd1, 0x0c6e, 0x419b, 0xb284, 0x466c, 0x4076, 0x427f, 0x4657, 0x415f, 0x1d87, 0x0221, 0x346e, 0x4217, 0xbadb, 0x41f4, 0x41e9, 0xbac1, 0x42b2, 0x4177, 0x4118, 0x4128, 0x3828, 0x04a2, 0xbac4, 0xba30, 0xb0f2, 0xbf8a, 0xa678, 0x18bf, 0xb0ad, 0x40dd, 0x416e, 0x43d7, 0x44b9, 0xa0fd, 0x434b, 0xb097, 0x1f28, 0x4320, 0xb00e, 0x4377, 0x1187, 0xba24, 0x4160, 0xb027, 0xb20e, 0x43aa, 0x43cf, 0xbfd4, 0x39e1, 0x1d32, 0x40f2, 0x404a, 0xba4f, 0x00ca, 0xbaee, 0x426d, 0xb2f1, 0x16fd, 0x3012, 0x415e, 0x412c, 0x4221, 0xbf62, 0x1eaf, 0xab08, 0x40b8, 0x0851, 0x3a0d, 0x0010, 0xa39c, 0x0bbb, 0x438e, 0x115e, 0x2965, 0x43c4, 0x448a, 0x447b, 0x4389, 0x42a7, 0xba70, 0x46a5, 0x437f, 0x1a53, 0xb2d0, 0xb2b4, 0x417a, 0xb20f, 0x3ddf, 0x4213, 0xbf03, 0x419b, 0xb229, 0x1cc5, 0x42d9, 0x4307, 0xb0b0, 0x424e, 0xb0c5, 0x2b34, 0x430e, 0x42a2, 0x409c, 0x41eb, 0x4143, 0x4092, 0xbfab, 0x419d, 0x4370, 0x173a, 0xbac9, 0x44ba, 0xba24, 0x4292, 0xbf83, 0x10f4, 0x1352, 0x435f, 0x430a, 0x4113, 0x4556, 0xba04, 0x43b0, 0x43cc, 0x3369, 0xb0c6, 0x43ca, 0x42e2, 0xbf43, 0xb04a, 0xb0cf, 0xb2c0, 0xbff0, 0x436f, 0x40e7, 0xbf72, 0x1cda, 0xba12, 0xb2b6, 0x1294, 0xb083, 0x44f3, 0xbf4a, 0x4379, 0x4206, 0x4168, 0x18aa, 0x43b3, 0xb2d4, 0x442d, 0x4363, 0x4219, 0x3fba, 0xb2f9, 0x4226, 0xbff0, 0xb050, 0x1882, 0x4143, 0x4602, 0x3889, 0x19a3, 0xbf35, 0x4110, 0x429e, 0xb2dd, 0x4087, 0xba4b, 0x4303, 0x415b, 0xb031, 0xb003, 0x4062, 0x1aa9, 0x4066, 0x1ca6, 0xbf80, 0x1970, 0x0840, 0xa6a7, 0xaf8d, 0x40ea, 0x2ad2, 0xb2b7, 0x41aa, 0xbae6, 0x393e, 0x4264, 0x40ae, 0xbaf7, 0xbf6d, 0xb244, 0xba21, 0xba7b, 0x23d2, 0x400a, 0x4059, 0xba4d, 0x1505, 0x1d64, 0x42b9, 0x4197, 0x41f8, 0x4437, 0x41ac, 0x0a56, 0x1d3d, 0x4555, 0xbf8f, 0x4319, 0xbade, 0xba6e, 0x03a1, 0xb2b4, 0x40ac, 0xa7c5, 0xbf14, 0xac04, 0x221e, 0x4246, 0xb2d9, 0x4020, 0xba06, 0xbff0, 0x40d0, 0x2539, 0xba4f, 0x4283, 0x43b5, 0x42f1, 0x43a9, 0xb215, 0xba78, 0x4353, 0xb2ad, 0x0eb2, 0x40b9, 0x05f6, 0x1b03, 0x037a, 0x1922, 0x1d2d, 0x4218, 0xbf42, 0x2674, 0x44fd, 0x41e2, 0xba4d, 0x43f9, 0x10d1, 0x4121, 0x4309, 0xb25a, 0xba68, 0xb2d2, 0x402c, 0x27fb, 0x4143, 0xba48, 0xbfc6, 0x42cd, 0x438f, 0xb23a, 0x253a, 0x19ff, 0x40ed, 0x429a, 0x03f5, 0x412e, 0xba52, 0x43e5, 0xb2bf, 0x1bae, 0x43a5, 0x4389, 0x4253, 0x1bc9, 0xbf8c, 0xb20d, 0xb2be, 0x43ea, 0xa6fd, 0x3b1d, 0x40ad, 0x43d6, 0x43fd, 0x448a, 0x17ff, 0x430b, 0x41e0, 0xba42, 0xbf57, 0xaf35, 0x423c, 0x40b0, 0x43f4, 0xb245, 0x1a09, 0x04e0, 0x398e, 0x4049, 0x4389, 0x43b2, 0xbf00, 0xb0b9, 0x1f41, 0x41a5, 0x4143, 0x403e, 0xa712, 0xab59, 0xbac2, 0x422b, 0x21e4, 0xbfdd, 0xb273, 0x2b57, 0xa6e7, 0xa357, 0xa329, 0x427e, 0xb29d, 0x0b10, 0xbae1, 0xa29b, 0x2e57, 0x41dd, 0x43b3, 0xb2ff, 0xb201, 0xa480, 0x111a, 0x434f, 0x3828, 0x0c1e, 0x40bd, 0x1ddc, 0x4120, 0xbf90, 0x1f65, 0x4387, 0x4307, 0x4235, 0xbf87, 0xbff0, 0x41c6, 0x36e9, 0x4120, 0x40ca, 0x4383, 0xb211, 0x41f5, 0x4309, 0xbfd5, 0x4332, 0xb23a, 0x4227, 0x4358, 0x2229, 0x41a0, 0xb09b, 0x4162, 0x00af, 0x4017, 0xb270, 0xbf60, 0x4289, 0xba44, 0x415f, 0x3745, 0x1d25, 0xb24f, 0x415b, 0x1dde, 0x41d5, 0x421d, 0x4061, 0xb23d, 0xb0e1, 0xb231, 0xbfce, 0x4479, 0x18ae, 0x429b, 0x3aa6, 0xb031, 0xbf08, 0x4173, 0x4025, 0xbf01, 0xbf80, 0x429a, 0xa140, 0x4241, 0x4004, 0xbaf0, 0x1950, 0xbf00, 0x4069, 0xbf90, 0x42f8, 0x4149, 0x4352, 0x40ee, 0x1a2e, 0x1888, 0x2be6, 0xa75e, 0xb040, 0x42ed, 0x4158, 0xbf7a, 0x45aa, 0x0e95, 0x1014, 0x0d0c, 0x22e9, 0x1dc7, 0xbfa0, 0xb092, 0xbfb0, 0x41c7, 0x0377, 0xb2c9, 0x4260, 0x4371, 0x1ba6, 0x4273, 0x426c, 0x1a58, 0x1e6b, 0x0f52, 0x1c50, 0x3528, 0x4157, 0x4296, 0xbf43, 0x345e, 0x40f5, 0x429c, 0x42bc, 0x4312, 0xa05d, 0x1ea4, 0x40e2, 0xbf92, 0x38a7, 0x43b2, 0xbfb0, 0x1d58, 0xba54, 0xb256, 0xb255, 0x442e, 0x4293, 0xba67, 0x1804, 0xb27c, 0x425a, 0x4173, 0xba62, 0x4221, 0x41c7, 0x3475, 0xbfb8, 0x4002, 0xb0b4, 0x412c, 0x4589, 0xa77d, 0xb020, 0xbaff, 0xa2d3, 0x4344, 0xb205, 0x46b8, 0x1801, 0x1ee2, 0x4131, 0x43ef, 0x408a, 0x1107, 0x4168, 0xbf41, 0xb20c, 0x432c, 0x2c41, 0x3ee4, 0xa926, 0x40aa, 0xb0b8, 0x3a6f, 0x469a, 0x4379, 0xbf7c, 0x1f14, 0xb2c5, 0xb2bb, 0x1c36, 0xb265, 0xba1d, 0x41af, 0x374d, 0xb28e, 0x42b4, 0xbae7, 0x4258, 0x40f9, 0xa16f, 0xae72, 0x1f30, 0xbad1, 0x1e7c, 0x4319, 0xbfd9, 0xbae6, 0xb0d2, 0xb28f, 0xb29e, 0xb03f, 0x41eb, 0x416e, 0x4095, 0x3136, 0xa0cf, 0x1dd8, 0x40e6, 0x06b2, 0xb2a3, 0x0458, 0xb036, 0x435d, 0x43f2, 0x43b0, 0x315c, 0xba07, 0xb0ba, 0x439a, 0xb21d, 0x1079, 0xba14, 0xbf22, 0x433f, 0x40cd, 0xbfe0, 0xa72f, 0xbac2, 0x4134, 0xb0ed, 0xbfda, 0xb2d7, 0x41e2, 0xba61, 0x1b57, 0x406a, 0x4217, 0x4064, 0xba66, 0x3ffd, 0x41b3, 0x343a, 0xb233, 0xbfd3, 0x4111, 0x4213, 0x4093, 0x40c3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6a1fb324, 0x37d1732c, 0x1194c5d3, 0x5a48240b, 0xf3985d54, 0xf7ed5c2f, 0xf3b4ec29, 0x13619425, 0x3b477912, 0x36e07690, 0x5d185c63, 0x32e603e4, 0xdff550bc, 0xbcb83fd1, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x1a1c0000, 0x00000e0d, 0xffff8d0e, 0x00000000, 0x0000003a, 0xffff8d0e, 0x00000000, 0x000071f5, 0x00001419, 0x868c5020, 0xffffffff, 0x32e603e4, 0xdff550bc, 0x000217f8, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb2dc, 0x425f, 0xb2d1, 0xb20b, 0x1b94, 0xbf39, 0x1983, 0x4350, 0x426e, 0x43a5, 0x4377, 0x1ee2, 0x1e44, 0x1b2d, 0x1ad2, 0x1cc0, 0x402d, 0xb2f6, 0x02f0, 0x3eb0, 0x4037, 0x0a3e, 0x418c, 0xbf8a, 0x405e, 0xa376, 0xb075, 0x420d, 0xb2b6, 0xb022, 0x440a, 0xb0cc, 0x46a9, 0x40e7, 0x21fc, 0xb27b, 0x437c, 0xa9bc, 0x1c9d, 0x42bd, 0x4197, 0xbf9f, 0x441e, 0xba11, 0x424e, 0xbff0, 0x41cb, 0xb21c, 0x428b, 0x1a18, 0x4007, 0x1795, 0x2000, 0x4006, 0x43b6, 0xbf3b, 0x42ec, 0x422d, 0x2e16, 0x29b0, 0x1bee, 0x407d, 0xb02d, 0x4075, 0xbf64, 0x407e, 0x4547, 0x4159, 0xb0eb, 0xb21d, 0x41f2, 0x09e7, 0x410d, 0x462b, 0x4583, 0xb2f8, 0x409e, 0x40a4, 0xbfd0, 0xb051, 0x4029, 0x3874, 0x4642, 0xbfba, 0x40d1, 0xb217, 0xb083, 0x2a1f, 0x4263, 0x468a, 0x322b, 0xba31, 0x0de5, 0x4091, 0xb243, 0xb296, 0x4259, 0xac7d, 0x2a5a, 0x4010, 0xbf98, 0x4116, 0x4166, 0x19d7, 0xba43, 0x0e2e, 0xbff0, 0x430f, 0x439b, 0xa2f3, 0x1914, 0x3b48, 0x18d2, 0x4065, 0xb07d, 0xb02c, 0xb2a7, 0xb2c8, 0x03b0, 0xa9b3, 0x45de, 0xbf00, 0xb27c, 0x4347, 0x1fc8, 0x40cc, 0x4095, 0xbf74, 0x2dfe, 0x4223, 0x42f8, 0x3a83, 0x1b47, 0x1be1, 0x1281, 0x42da, 0x03dc, 0x40f1, 0xb2c1, 0x4648, 0x44b0, 0xba72, 0xba36, 0x0d2d, 0x4001, 0xbf9b, 0x1890, 0x45d4, 0x42b6, 0x40f9, 0x420a, 0x410e, 0xaa69, 0xb263, 0xbf77, 0x00be, 0x4092, 0x40e7, 0xb2b1, 0x435a, 0x42b4, 0xb09f, 0x420d, 0xb06a, 0xba63, 0x4272, 0x42d0, 0x32b7, 0x43ce, 0x1af8, 0xb06d, 0x430c, 0x404c, 0x097a, 0xb2c9, 0xbf3c, 0x2533, 0xbadd, 0x46b0, 0x441f, 0xb0f8, 0x4384, 0x45da, 0x42fb, 0x4380, 0xbac6, 0x35c4, 0x1564, 0x0e66, 0xbfe0, 0x3645, 0xba64, 0x4229, 0xa6c0, 0xbf14, 0xb07b, 0x40fe, 0x4353, 0x1856, 0xa5a8, 0x1bb5, 0xaa8b, 0x40bf, 0xabf0, 0x4389, 0xbac0, 0xba7b, 0x43e4, 0x1ee2, 0x42ce, 0xbf0e, 0x46a0, 0x4279, 0x45ae, 0x4096, 0x0e54, 0x4106, 0xb25d, 0x40ec, 0x3483, 0x4090, 0x42cf, 0x1a7a, 0x406b, 0xb02c, 0x43b1, 0x4029, 0x2399, 0x19ce, 0xb233, 0x4303, 0xbfd0, 0x2c0c, 0x43d7, 0xb254, 0x27fa, 0x2ea7, 0x0a4b, 0x40b1, 0xbfc4, 0x1b4d, 0x420e, 0xb2b7, 0xb234, 0x4210, 0x43f8, 0x466b, 0x4134, 0xb2f1, 0x1f53, 0xb06b, 0x40a2, 0xb0f1, 0x40ea, 0xb0f5, 0x421d, 0x42df, 0x28d3, 0xba66, 0x42fc, 0xbfd5, 0xbfe0, 0xbfa0, 0xba2f, 0x45c1, 0xbae4, 0xb290, 0x4030, 0x436c, 0x466e, 0x44d9, 0xba3a, 0xba5f, 0xa987, 0x247e, 0x1273, 0x1925, 0xb240, 0xa2a8, 0x1d6a, 0xbf94, 0x42e4, 0x43e1, 0x42ff, 0xbf49, 0x454d, 0x442d, 0xb26b, 0xb054, 0x41d3, 0x402d, 0x40b2, 0x43e0, 0xbf2e, 0x0098, 0x4047, 0xb2af, 0x42e4, 0x4315, 0x41c2, 0xab7f, 0xb279, 0xb253, 0x4101, 0xb2fe, 0xa41d, 0xb2f0, 0x4357, 0x1b5e, 0x3f79, 0xb26e, 0x41bd, 0xbf69, 0x1813, 0x0a71, 0x0efc, 0x4215, 0x16f2, 0x4108, 0x4191, 0xbf2a, 0xb2a0, 0x1c23, 0xaeff, 0x21f2, 0x2a43, 0xbad6, 0xac3a, 0x43ee, 0x4226, 0xbf42, 0x412a, 0x04ce, 0x3044, 0x42fd, 0x430f, 0x3e1c, 0x4063, 0xbfac, 0x3557, 0x4258, 0x06d8, 0x4250, 0x43fa, 0xbaf8, 0x40e1, 0x11e8, 0x4000, 0x40ff, 0x1adf, 0x4693, 0x4257, 0x46d0, 0xbfba, 0xb2b9, 0xb23d, 0xba49, 0x4266, 0x4210, 0xa557, 0x1df4, 0xba45, 0x454c, 0xa556, 0x434e, 0x1a7a, 0x4380, 0xb09a, 0x4087, 0x1937, 0xb20f, 0x4067, 0xbfd3, 0x42b7, 0xb227, 0x41f9, 0xbaee, 0x4191, 0x4011, 0x2bf8, 0x41c0, 0x4599, 0x428a, 0x40ba, 0x1e9f, 0x12b1, 0x2cdc, 0xb285, 0xaa95, 0xbfbe, 0x4022, 0x466c, 0xba77, 0xb041, 0x42bc, 0x4016, 0x434b, 0x41c6, 0xba0a, 0x416d, 0x3ddc, 0xba28, 0x40f3, 0x3d50, 0x41fb, 0x437d, 0x42e3, 0x4343, 0x143b, 0xbf4b, 0x1a96, 0x43cd, 0x05a6, 0x41bf, 0x413e, 0x1de5, 0xb0c7, 0x436d, 0xbac4, 0x43cf, 0x42e9, 0x4294, 0x07b5, 0xa1ea, 0x4577, 0xbae6, 0x4387, 0x42eb, 0x2ba4, 0x4117, 0x4229, 0x4171, 0xb0b0, 0xbade, 0x4226, 0x43a4, 0xa8c8, 0x4213, 0xbf00, 0xbf88, 0xb07b, 0x4229, 0x322d, 0x2ed2, 0x42d6, 0x2426, 0x427b, 0xbf28, 0x039a, 0x4377, 0x1baf, 0x2d7d, 0x42fd, 0x4699, 0x4004, 0x4166, 0x4164, 0x0761, 0xb0da, 0xb2c5, 0x4178, 0x19b8, 0x42a1, 0xbfb9, 0x4299, 0x4192, 0x40f5, 0x42d0, 0xbf7f, 0x4353, 0x40d7, 0x4261, 0x09af, 0x1c71, 0x4492, 0x421b, 0xba64, 0x41a2, 0x4391, 0xbfe0, 0x18ad, 0x4114, 0x42ba, 0xbafa, 0xb0e0, 0xbf12, 0x3c22, 0x13d3, 0xb270, 0x1297, 0x4256, 0x4227, 0x4148, 0xbfa1, 0xbf80, 0x1863, 0x075c, 0xa63a, 0x192a, 0xb285, 0x415f, 0x371f, 0x36d4, 0x41f9, 0xb09f, 0x4246, 0xb2f3, 0xb000, 0x420d, 0x4315, 0xb2b3, 0x0bc4, 0x2021, 0x0dda, 0xb0c5, 0xbf7a, 0x432f, 0xa889, 0x4112, 0x42ea, 0x4310, 0xbf3e, 0xa3e2, 0x4291, 0x1be6, 0x43bd, 0x40b7, 0x205f, 0x4014, 0x31dc, 0xa38c, 0x4117, 0x4230, 0x4381, 0xbf66, 0x4002, 0x41df, 0x42f0, 0xbaf0, 0x4171, 0xba79, 0x1339, 0x45cb, 0x409d, 0xbf18, 0x4163, 0xb07c, 0x4172, 0x40c2, 0x2cb7, 0x4309, 0x4068, 0x1788, 0xba35, 0x42b2, 0x43a5, 0x43c5, 0x442f, 0xba0c, 0x014d, 0x432c, 0xb2d3, 0x3076, 0x4274, 0xb283, 0xa8fe, 0x1c15, 0x46d1, 0xb075, 0xbf1e, 0x42b9, 0x31df, 0xba3b, 0x414e, 0xbae5, 0xa161, 0xb2c0, 0x436c, 0xbf9c, 0x43c1, 0xbadb, 0x18b0, 0xbfe2, 0xaf82, 0x39f6, 0x41b3, 0x2696, 0x38ad, 0x42b5, 0xb043, 0x43dd, 0x40ca, 0x42f0, 0xba54, 0x4369, 0x0412, 0x4349, 0x438f, 0x4555, 0x4022, 0x412f, 0x41f4, 0x4139, 0xa343, 0x41c4, 0xba2c, 0xad94, 0xbfb4, 0x0d58, 0x3a42, 0x43a9, 0x41fb, 0xbaf7, 0x463c, 0x0151, 0x44e8, 0x43bf, 0x42b2, 0x42e1, 0x0479, 0x1801, 0xba2e, 0x1614, 0x41fe, 0x4204, 0xbad0, 0x18ad, 0x4056, 0xb0e2, 0xb2a8, 0x06e3, 0xb260, 0xbf3f, 0xba59, 0x4620, 0xba29, 0x3348, 0x45c4, 0xbaef, 0xb293, 0xbadb, 0x40d2, 0x4374, 0xba7f, 0x35fa, 0x1b6c, 0xb23b, 0x2a15, 0x1ed7, 0xa687, 0xa2c0, 0x01fc, 0xa940, 0xba6e, 0x459a, 0x4044, 0x437d, 0xb2ab, 0x3190, 0x2415, 0x2092, 0xbf87, 0x427d, 0xb271, 0x2f34, 0x21c7, 0x404d, 0xb08a, 0x1ce7, 0x30cd, 0x433c, 0x45ab, 0xb234, 0x438d, 0x3eae, 0x4062, 0xb224, 0x3c04, 0xb2ce, 0xb23a, 0x17a3, 0xbfbc, 0x1af8, 0x430c, 0x4264, 0x43d9, 0xb2b5, 0x40dc, 0xbad1, 0x1ad9, 0x41d7, 0x40b1, 0xb0e2, 0x43b9, 0x41ca, 0x352f, 0xbf70, 0x33cb, 0xac4b, 0xba11, 0xb0d1, 0x42c3, 0xbf0e, 0xb277, 0x1c5f, 0xb2f3, 0x4278, 0x4005, 0x40d3, 0xa013, 0x441a, 0x44b4, 0x089a, 0xad41, 0x1c00, 0xb238, 0x46a9, 0x40a7, 0xba4e, 0x43ab, 0x2692, 0x1f89, 0xb021, 0x41a7, 0xbfa2, 0x106c, 0x4213, 0xb251, 0x423c, 0x43fd, 0x423d, 0x406b, 0x40b7, 0x1cc0, 0xacca, 0x433f, 0x3dda, 0xba4a, 0x3137, 0x2707, 0x4167, 0x4016, 0xbf45, 0x40d7, 0xba6a, 0x403b, 0xae6b, 0x42ae, 0x4351, 0xb0bf, 0x420c, 0x2052, 0xb037, 0x1b19, 0x43cf, 0x42aa, 0x40cd, 0x4329, 0x0139, 0xbf32, 0x2fda, 0x1320, 0x256b, 0x421c, 0x41e4, 0x4080, 0x40b8, 0x07be, 0x3f93, 0x4270, 0xb2d8, 0x4295, 0x41ba, 0xb013, 0x440f, 0x4157, 0x42b1, 0x40f4, 0xb0f6, 0x1b7d, 0x408b, 0xbfe0, 0xba53, 0x433a, 0x464f, 0xbf25, 0x406e, 0xae0f, 0x1c12, 0xb230, 0xbacf, 0xbacb, 0x1fe3, 0xbf90, 0x4602, 0xbf82, 0xba73, 0x4481, 0x23b5, 0x4394, 0x4320, 0xbad1, 0x1aec, 0x27c6, 0x41bd, 0x4157, 0x1de8, 0xbfe0, 0x435e, 0x41de, 0x25df, 0x01fe, 0x40d6, 0x467f, 0xb081, 0xb0b2, 0x3885, 0x192d, 0x2974, 0x31f9, 0x458d, 0x2fc1, 0xb048, 0xbf18, 0x10c7, 0x4272, 0xb25a, 0x4198, 0xb221, 0x400f, 0xa6c2, 0x42d2, 0x1b23, 0x417b, 0x4456, 0x4465, 0x1fdf, 0x1c70, 0x4068, 0xb2d8, 0x4582, 0xb279, 0x40b9, 0xba17, 0x4243, 0x4060, 0x1936, 0xbf0f, 0xa1bc, 0xb0fa, 0x439d, 0x41e8, 0x39be, 0xb2e1, 0x4430, 0x40ff, 0x1a28, 0x42de, 0x1e90, 0x417e, 0x03bc, 0x4240, 0x435c, 0xbad7, 0x1de9, 0xa31a, 0x1135, 0x4198, 0x441f, 0x4276, 0xbf70, 0x1d98, 0xbfa4, 0x418f, 0xba2f, 0x41f1, 0x4182, 0x41e0, 0x425b, 0xb269, 0x1ff1, 0x4330, 0xb29e, 0x2039, 0x41b9, 0xa5d6, 0x42bb, 0xb2dd, 0x4025, 0x2734, 0x469b, 0x4368, 0x1c15, 0x41c3, 0xbfb1, 0x0bef, 0x4469, 0xb214, 0x3073, 0x423d, 0x1ecf, 0x437f, 0x41e6, 0xac31, 0xbf6d, 0x18aa, 0x4641, 0x2805, 0x43e1, 0xa531, 0x42a7, 0xb246, 0x1f48, 0x42a0, 0x1e10, 0x4263, 0x43eb, 0xb090, 0x4020, 0xa807, 0x4149, 0x42b1, 0xb291, 0xbf90, 0x2fbf, 0x1edd, 0x3d40, 0x433f, 0x3eb8, 0xbf5f, 0xb212, 0x4151, 0x43b7, 0x4415, 0x03b7, 0x40cb, 0xba3f, 0x438d, 0x41fd, 0x0518, 0x42cf, 0xa79a, 0xa354, 0x2ac3, 0xb25a, 0x4145, 0xa5e8, 0x400e, 0x1851, 0xbf87, 0x1c05, 0x4377, 0x405d, 0xba00, 0xb2d5, 0x4232, 0x319c, 0x4409, 0x1f3d, 0x41e6, 0x4069, 0xbfb4, 0x41d9, 0x18d0, 0x4307, 0x4567, 0x433e, 0x421c, 0x429c, 0x055c, 0x41ab, 0x40b1, 0x40f1, 0x353d, 0xaae0, 0x4237, 0xaadc, 0x23f3, 0xb0b0, 0xbf47, 0x401a, 0xaeea, 0x4077, 0xa7bd, 0x4209, 0x41a4, 0x46eb, 0x437f, 0x4226, 0xb2e2, 0xb04a, 0x4233, 0x4387, 0xbfc8, 0x46dc, 0x4381, 0xbada, 0x0d3d, 0x44cb, 0x405c, 0x4544, 0x438e, 0x419a, 0x40f9, 0x442a, 0xb000, 0xb2da, 0x1aa8, 0x1e51, 0x1dd4, 0x1319, 0xba33, 0x4176, 0x40eb, 0xbff0, 0xbf62, 0x42d7, 0x40e7, 0x413e, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xec5af9f4, 0x9436eb06, 0x9ba4ad42, 0x6f255f58, 0x3f12d10f, 0x0928b6c9, 0x415b7bf8, 0x054f2f3e, 0x41a3cd86, 0x6af229e8, 0xc9f61355, 0xeb58ff68, 0x0c0e2ab2, 0xe150fa69, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0xffffff39, 0x00000000, 0x000000f3, 0x00000000, 0x000000fa, 0x0000002c, 0x2eb370e8, 0x02c5a110, 0xe151009d, 0xe150fdb5, 0xffffffff, 0xc2a1f61a, 0xe150f865, 0xe150f98d, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x368f, 0x4181, 0xba43, 0x438d, 0xb2ff, 0x43b4, 0x1f09, 0xb23a, 0x4230, 0x293c, 0x4083, 0x243f, 0x2f0b, 0xba43, 0x4022, 0xbf99, 0x2d22, 0xb03a, 0xb2bd, 0x44d3, 0x08ec, 0x425e, 0xb00c, 0x407d, 0xba24, 0xafc8, 0x1de9, 0xbf70, 0x41a6, 0xbf58, 0x42e9, 0x264c, 0x45c3, 0x40bd, 0x4029, 0xba3a, 0x1fda, 0x405c, 0x4005, 0xb2cd, 0x4079, 0x3285, 0xa715, 0x411f, 0x4560, 0xbfb1, 0x406c, 0x4091, 0xa6a9, 0xb22f, 0x41d6, 0xb20e, 0x40ad, 0xbff0, 0xad3c, 0x443a, 0xbf80, 0xb256, 0x4025, 0x43b3, 0x1fd2, 0x00f7, 0x4160, 0xbf1f, 0x36b9, 0x19d7, 0xb26e, 0x4101, 0x41d5, 0x1d7a, 0xba56, 0x43a6, 0xbf83, 0xb2dc, 0x424b, 0x4124, 0xba59, 0x4307, 0xb0cf, 0x4328, 0x4065, 0xbafa, 0xb0c7, 0xb0af, 0xbf83, 0x43b6, 0xb2b8, 0x408c, 0x0730, 0xbfb4, 0xbff0, 0x02a9, 0xb286, 0x4258, 0x45b2, 0x42dd, 0x427e, 0x1faf, 0x1af5, 0x40fb, 0xb092, 0x4615, 0xbf29, 0x1b44, 0xbf60, 0xa826, 0xb250, 0x40cc, 0xb0ad, 0x23d8, 0x10b2, 0x4183, 0x426a, 0x4152, 0xbaf9, 0xacfe, 0x1cd8, 0xbfa2, 0x2419, 0x3078, 0x4083, 0xb2c7, 0x45ae, 0x38db, 0x1879, 0x40cf, 0x4220, 0xb23d, 0x20b8, 0xba32, 0x4031, 0x455b, 0x105c, 0xb295, 0x1af1, 0x12d9, 0xbfc9, 0x403c, 0x4133, 0x4261, 0x41aa, 0x4369, 0x032b, 0xb049, 0xad38, 0x27b3, 0xb20c, 0x424c, 0x4238, 0x1923, 0xb283, 0xb050, 0x424a, 0x1c55, 0x0a49, 0xb0f6, 0x43e2, 0x0e4f, 0x4219, 0x400a, 0x42ac, 0x414e, 0xa73b, 0xbf0f, 0x437b, 0x41ef, 0xa1bd, 0x4051, 0x4308, 0x16a2, 0xbfa0, 0xb26d, 0xba3c, 0x181e, 0xba7e, 0x42c2, 0xba45, 0xac7d, 0x087f, 0x4225, 0xba6c, 0x188b, 0xbf0a, 0xa98e, 0xb2da, 0x1f20, 0x19c1, 0xbf08, 0x40fe, 0x4137, 0xb23a, 0xbfb0, 0x4098, 0x1814, 0x4026, 0x2027, 0x4269, 0xb060, 0x1807, 0xb28e, 0x1ae8, 0xb2a0, 0x4308, 0x414a, 0x416e, 0x1358, 0x43c5, 0xb257, 0x4412, 0xbfc0, 0x2c84, 0x4212, 0xbf7f, 0x40b3, 0xb0cc, 0x404b, 0x0a2d, 0x1a49, 0x1eb1, 0x4342, 0x43b1, 0x1bc8, 0xa36f, 0x42b3, 0x1995, 0x43d8, 0x41c3, 0x407f, 0x410b, 0x40b9, 0x416c, 0xac31, 0x4010, 0xb08d, 0x4007, 0xb209, 0xb091, 0x4383, 0xb21e, 0x411d, 0xb02b, 0x44ad, 0xbf82, 0x18f7, 0x422a, 0xb01d, 0x3f8a, 0xb05f, 0x4589, 0x45a6, 0x4161, 0xba59, 0x41e8, 0xb280, 0x4052, 0xae0d, 0x43b8, 0x4297, 0xb2e8, 0xb260, 0x44a9, 0xbad8, 0x429b, 0x1271, 0x412a, 0x43ef, 0x403f, 0x1f4c, 0xbf7e, 0xa769, 0x40fa, 0xb2a7, 0x3cc5, 0xbf70, 0x17d1, 0x41de, 0x414f, 0xbf7f, 0xba4d, 0xa1df, 0x022b, 0xa268, 0x4237, 0x1a45, 0x40ce, 0x4324, 0x129a, 0x4087, 0x1820, 0x4287, 0x401a, 0x43a3, 0x268b, 0x1b43, 0xa18b, 0x1855, 0x4046, 0x406f, 0x43ee, 0x140b, 0xba3c, 0xb27d, 0x4227, 0xb2fc, 0xab9c, 0x418c, 0xbf2d, 0x42bc, 0x41a9, 0x02ae, 0xb203, 0x401d, 0x4250, 0x431a, 0x40b4, 0xba7c, 0x40db, 0x20f2, 0xb200, 0xa5a0, 0x31f7, 0x028e, 0xbfa0, 0xb2c0, 0x1584, 0xbadf, 0x4278, 0x43ab, 0x0582, 0x414a, 0x1a67, 0x3b4c, 0x1c8a, 0x42c3, 0x40e1, 0x4272, 0xbf41, 0xba3c, 0x42b7, 0x40fa, 0x0974, 0x42b7, 0x1c59, 0xbf17, 0x1c3b, 0x42b2, 0x42c7, 0x1feb, 0x419f, 0x24bb, 0x0932, 0x46c4, 0x42fd, 0x16d0, 0x42f3, 0xbac8, 0x125a, 0xb275, 0x18b1, 0x4432, 0x2522, 0xbf6c, 0x43e3, 0xba24, 0xb053, 0x439c, 0xa9a6, 0x1aa2, 0x3eaf, 0x402d, 0x0b2b, 0xa543, 0xbfa0, 0x4055, 0xbf12, 0x4168, 0x437e, 0x1ea7, 0x4022, 0xa4e7, 0xba4b, 0x394d, 0x195c, 0x1ab5, 0x41fd, 0x436d, 0xaa65, 0x2437, 0xbac6, 0x424a, 0xbfca, 0xba0e, 0xbaec, 0x4038, 0x1252, 0x258d, 0x4186, 0x45d2, 0x42a3, 0x45b8, 0x4160, 0xbf3e, 0x3d8d, 0xb2a1, 0xb277, 0xb2b9, 0xba2a, 0x42e5, 0x4149, 0x4345, 0xba37, 0x43a0, 0x4104, 0x20f0, 0xbac7, 0x43d0, 0x42a9, 0x4215, 0x4576, 0x0a5a, 0xb21c, 0x0657, 0xba38, 0x0f44, 0x1b98, 0x1deb, 0x43af, 0x4198, 0x40bf, 0x3af8, 0xbfd4, 0x03ec, 0x2f69, 0xb05c, 0x2d36, 0x1505, 0xb2be, 0x40ba, 0xb016, 0x40e9, 0x0007, 0xbf66, 0xb284, 0xb2fa, 0x19c8, 0x11bc, 0xba48, 0x4172, 0xa0c2, 0x41de, 0x42d9, 0xb224, 0x1e05, 0x42df, 0x1bf1, 0x272b, 0x4143, 0xbf97, 0xba04, 0x43a9, 0x16d4, 0xb2b7, 0xbafe, 0x033a, 0xba53, 0xb2a6, 0x432c, 0x425e, 0x35c4, 0x43d7, 0x43a1, 0xb209, 0x40d7, 0xbad6, 0x434c, 0x4380, 0xbf61, 0x43db, 0xa3c6, 0x4202, 0x41a0, 0x4370, 0x42d2, 0xb241, 0x0725, 0xb24a, 0x41b8, 0x1e71, 0x42dc, 0x419c, 0x1ccf, 0xb267, 0x4427, 0xb2f7, 0xb235, 0x2c94, 0x19eb, 0xb2b2, 0xbf76, 0x3401, 0x41d7, 0x4212, 0x1e3e, 0xae23, 0x3934, 0x4128, 0xb245, 0x1fcb, 0xbfcb, 0x1a32, 0x44dc, 0x41a4, 0x3ec2, 0x4381, 0x318a, 0x408b, 0x4076, 0x42be, 0x4214, 0x0530, 0x4351, 0x2735, 0xad1b, 0x436c, 0x43dc, 0xba34, 0x4276, 0x41eb, 0xb01f, 0x3246, 0x0717, 0x437e, 0x4021, 0x4043, 0xb237, 0x1876, 0x417d, 0x41d4, 0xbf57, 0xba07, 0xb2e7, 0x1f8b, 0xb28c, 0xbf21, 0x4243, 0x4157, 0x3d59, 0x40be, 0xb22b, 0xb21f, 0x2afc, 0x0525, 0x4291, 0x465e, 0x402d, 0x45ce, 0x0f5f, 0x3952, 0xaaf9, 0x414a, 0xbacc, 0x0081, 0x2e90, 0x4280, 0xbfa5, 0x404c, 0x413e, 0xb2d4, 0x38c8, 0xbf0a, 0x3124, 0x402c, 0x404e, 0x3c59, 0x41f1, 0xba4a, 0xbadc, 0x4117, 0xa1c4, 0x414e, 0x46b4, 0x40f7, 0xb251, 0x46a9, 0xa050, 0xb28c, 0x4369, 0x029c, 0x424c, 0x469b, 0x4266, 0x43b8, 0x2c0e, 0xbfbc, 0x1825, 0xadd6, 0x1b78, 0xba75, 0x10ef, 0x26bd, 0xb09b, 0x42b8, 0xba65, 0x3f0f, 0x0476, 0xb0dc, 0xb21f, 0xb02a, 0x1cb4, 0x468d, 0x3cbe, 0xbf29, 0xbad6, 0x439d, 0xb2d8, 0x1fc1, 0x4260, 0x4541, 0xb271, 0xbac0, 0x1d46, 0xb273, 0x197c, 0x4280, 0x426a, 0x1f19, 0x426c, 0xb284, 0x4230, 0xbf60, 0xb2fd, 0xb203, 0x18b1, 0xbf58, 0x39ee, 0x1ff4, 0x27de, 0xbfa0, 0xb27d, 0xbfe0, 0xb21c, 0x4165, 0x404d, 0xb2d9, 0x45c5, 0x4333, 0x1ba5, 0x4568, 0x43d8, 0xb277, 0x1ef0, 0x124f, 0xb251, 0x1d1d, 0xbf4a, 0x409b, 0xb23c, 0x4400, 0x4391, 0x293e, 0x1a99, 0x40be, 0x0314, 0x43bb, 0x41f5, 0xb28c, 0x1db7, 0x40c4, 0xb00c, 0x4175, 0x40af, 0x4291, 0x3db2, 0xa7cb, 0x4031, 0xba26, 0x43fa, 0x188c, 0x403e, 0x0de1, 0x432b, 0xb027, 0x2431, 0xbf74, 0x41e4, 0x432b, 0xb20f, 0x2b39, 0x4352, 0xaff4, 0xb2eb, 0x45ca, 0x4273, 0x4142, 0x4366, 0x40c1, 0x44fb, 0x1930, 0xb2a5, 0x1bf9, 0xb294, 0x42ec, 0x431c, 0x4345, 0xbf55, 0x0d4b, 0xb07f, 0x0095, 0x2cff, 0xb284, 0x14a1, 0x42bb, 0x4643, 0x436c, 0xb2f1, 0x44e8, 0xb23f, 0x4207, 0x2f1e, 0xb2d7, 0xa512, 0xac75, 0x3a91, 0x231d, 0x3c3d, 0x4151, 0x3341, 0x01ad, 0x44d5, 0xbf3c, 0x1763, 0xa0da, 0x41af, 0x1a77, 0x406d, 0x40fa, 0x18ed, 0x429f, 0x1a22, 0x295a, 0x2dbf, 0x4007, 0x0072, 0xbae7, 0x4343, 0x425d, 0xb2db, 0xb2d4, 0xac1e, 0x40cc, 0x1068, 0x46ca, 0x40c0, 0x417f, 0x4025, 0x4121, 0xbf1d, 0xb297, 0x1c35, 0x43f0, 0xba3f, 0x1e51, 0x43c9, 0x40de, 0x435b, 0xba05, 0x42b9, 0xb270, 0xbf05, 0x3d22, 0x4206, 0x1c39, 0xa97f, 0xba7e, 0xba1b, 0x31a6, 0x414f, 0x39ef, 0xb26a, 0xbad8, 0x2fd1, 0x1abe, 0x1f4e, 0x43a4, 0xb289, 0xb069, 0xb256, 0x4150, 0x181b, 0xbf6a, 0xa9e8, 0xb299, 0x260b, 0x1b81, 0x1356, 0x09f9, 0x16e5, 0x4353, 0x0218, 0x41e3, 0x42c2, 0x2b68, 0xba77, 0x2cc6, 0x4577, 0x42d2, 0x212f, 0x42b8, 0x4063, 0xb235, 0x195a, 0x3085, 0xbf69, 0x4208, 0x2c24, 0x0117, 0x27bc, 0x421d, 0x4165, 0x4033, 0xbf65, 0x38d0, 0x3e41, 0x46c4, 0x40d8, 0xb2d5, 0xbf60, 0x1ab8, 0x402b, 0x23ec, 0xbfc0, 0xbf21, 0xba39, 0x4176, 0x22b6, 0xa1ca, 0x4041, 0x1e24, 0x2334, 0xb079, 0x1ac9, 0x428c, 0x43a1, 0x4556, 0x1ebd, 0xa512, 0xba5c, 0x2319, 0x1ffe, 0xb25d, 0x421b, 0xbf9b, 0x4012, 0x41de, 0x41ad, 0xbaee, 0x42c6, 0x41d7, 0xb20a, 0x33f2, 0x42de, 0x41a2, 0x0d16, 0xba7a, 0x4289, 0xba26, 0xbf49, 0xb2ae, 0x129d, 0xb2eb, 0x4223, 0x1864, 0x0aad, 0xbf57, 0x412c, 0x404f, 0xaa07, 0x41c6, 0x40ef, 0x114a, 0x46f8, 0xbac8, 0x45f1, 0xa8a5, 0x4338, 0x43ab, 0x0833, 0x2140, 0x41b6, 0x2a3e, 0xb018, 0x4215, 0xbf05, 0xba24, 0xa61c, 0xbae2, 0x3218, 0x1078, 0xa5bb, 0x3cf2, 0x34b0, 0xbacd, 0xb085, 0xbf6d, 0xbac5, 0x46d2, 0x4335, 0x421e, 0xb2e6, 0x1df5, 0x412e, 0xbf08, 0x4060, 0x1c63, 0x1c85, 0xba0b, 0xb25a, 0x430b, 0xbfbe, 0xb072, 0x4307, 0x43f1, 0x436c, 0x463f, 0x0028, 0x4311, 0xa4ad, 0x4648, 0x4361, 0x4175, 0x228c, 0x42e9, 0x40e8, 0x0f81, 0x081b, 0x4146, 0xbf81, 0x42a1, 0xac30, 0x40f1, 0x4492, 0x4336, 0xb2fb, 0x1b1d, 0xb2ac, 0x23c8, 0x44ad, 0x1f91, 0x43fe, 0x42b7, 0x1bed, 0x1cb1, 0x3518, 0x0864, 0x4183, 0x42de, 0x4307, 0xb283, 0x43af, 0xb2cc, 0xba5f, 0x1411, 0xb298, 0x4071, 0xbf16, 0x40c9, 0x4637, 0x4307, 0xbf53, 0x1a70, 0x40ca, 0xb26b, 0x42e6, 0xb079, 0xa782, 0x1c70, 0x4623, 0x40b1, 0x41f2, 0x46a4, 0x2ac3, 0x4229, 0x3f2d, 0x4302, 0x030f, 0xa7bd, 0xbf81, 0x4040, 0xbafc, 0xa376, 0xb05e, 0xbf6b, 0x42ec, 0xa754, 0x41e9, 0x433d, 0x4325, 0x2462, 0x4205, 0xa968, 0xbff0, 0x00c8, 0x4378, 0x430f, 0x4136, 0x41f5, 0x2f8e, 0x44fa, 0x413c, 0xbfb9, 0xb097, 0xbaf6, 0xb03e, 0x43d0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x8b2100eb, 0x3af85a8f, 0x0bb662e6, 0x7ac5449f, 0x6c8b9706, 0x2d86ea00, 0xd9186e8c, 0xb3755866, 0x7227be38, 0x7f97e5fb, 0xc176d2f9, 0x1ff0913a, 0xcb8e1a87, 0x6b0408ce, 0x00000000, 0x800001f0 },
                FinalRegs = new uint[] { 0xffffff72, 0xc176c228, 0x0000008d, 0x00000002, 0x00000000, 0xffffff36, 0x00000000, 0xc176db3c, 0x000016e4, 0x00000000, 0x000017e2, 0x0000206e, 0x00000002, 0xc176c180, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbac5, 0x426b, 0x16bc, 0x2d44, 0x43aa, 0xbf7b, 0x43f8, 0x42b9, 0x458b, 0x1f7e, 0xb0ac, 0xb292, 0x42e3, 0xb251, 0xb2dd, 0x1b86, 0x35fd, 0x410c, 0xac53, 0x3357, 0x35fb, 0x4355, 0x413e, 0xb2b9, 0x42f8, 0xbfa3, 0x408f, 0x1d27, 0xb0dd, 0xbf70, 0x2c41, 0x1bb6, 0xbf0d, 0xb208, 0x2cad, 0xb296, 0x4132, 0x4119, 0x411d, 0x403a, 0x4332, 0xb2cb, 0xbac4, 0x4074, 0x403d, 0x18a4, 0x45ae, 0x42c9, 0xb043, 0xb0b4, 0x40c4, 0xba53, 0x409b, 0xbf63, 0x1a75, 0xb040, 0x4166, 0x3f7e, 0xa55e, 0x428d, 0x4235, 0x459c, 0x42d5, 0x4111, 0x42dc, 0x43ca, 0x4049, 0x4091, 0x29d5, 0xbf61, 0x40cd, 0x422d, 0x4170, 0x4541, 0x1062, 0x1cf3, 0x4248, 0x422e, 0x1cdc, 0xabfc, 0xa3f2, 0x0677, 0x28d6, 0x143c, 0xb2d6, 0xba55, 0xb261, 0x41f7, 0x439c, 0x2b7b, 0x1c82, 0xb008, 0x2432, 0xbf62, 0xa8ee, 0x37a9, 0x1e86, 0xa85b, 0xb25c, 0x434c, 0xbf96, 0x423e, 0x4093, 0x44e1, 0x42ff, 0x41bf, 0xa8c3, 0x43cb, 0x3a77, 0x2b99, 0x0976, 0x3160, 0x4030, 0xb2db, 0xba36, 0x3ef4, 0x4617, 0x1ab5, 0x02d3, 0xb018, 0x43ad, 0x40cd, 0xbf89, 0x4390, 0x15d6, 0x4255, 0x46dc, 0xac0e, 0x11b4, 0xbf05, 0xac46, 0x4392, 0xbac5, 0x1842, 0x40df, 0x4113, 0x4126, 0x1cac, 0x3e49, 0x4051, 0xb2cf, 0x43b7, 0xbf78, 0x07e7, 0xb210, 0xb26b, 0x1f7b, 0x03b0, 0x42db, 0x2d35, 0x43eb, 0xb23a, 0x418a, 0xaddb, 0x1c63, 0xbf33, 0x35a5, 0x4349, 0x0da2, 0x428e, 0x4132, 0x40f8, 0xba40, 0x41e8, 0x40e8, 0x2f5d, 0x417c, 0x1bc0, 0x41b3, 0xaa64, 0x1b12, 0x0f9c, 0x435b, 0x2204, 0x42d7, 0x19ce, 0x4059, 0x40b3, 0xad3a, 0x43a1, 0x44b9, 0x42f0, 0xb2d8, 0x2938, 0xbf8c, 0x0c53, 0x41c7, 0x4177, 0xbaca, 0x40a7, 0x4607, 0x4238, 0x4100, 0x14e5, 0x4271, 0x4148, 0x24b6, 0x108d, 0xb234, 0xafb1, 0x20a6, 0xbf2e, 0x1ff8, 0xba48, 0x42a8, 0x43a6, 0x40f7, 0x3723, 0x1e21, 0x4004, 0x14b3, 0xbae8, 0x43f2, 0x4223, 0xa5a7, 0x4021, 0xbf60, 0xb2dd, 0xb2fa, 0x4144, 0xb284, 0x135c, 0xbf8b, 0x40a8, 0x4238, 0xb26e, 0x42b4, 0x43c5, 0x3a5f, 0x43e5, 0x2fe1, 0xbf5e, 0x039d, 0x3102, 0x4076, 0x40ec, 0x1e43, 0xbfc1, 0x1f10, 0x212d, 0xa889, 0xb2af, 0x1ccd, 0xb22b, 0x4373, 0x09c1, 0xbf49, 0x40ab, 0xb2f7, 0x4037, 0xb28e, 0x41fd, 0x0334, 0x1a33, 0x41ce, 0x462f, 0x41a4, 0x4181, 0x1ac3, 0xbf46, 0xb054, 0xaf5f, 0x41f6, 0x3343, 0x42e6, 0x4281, 0x0710, 0x0255, 0x41fa, 0x43ca, 0x1bf1, 0x4138, 0x2a43, 0x4388, 0xb0b1, 0xb2d3, 0x41ef, 0x43f1, 0x4090, 0x4195, 0x4151, 0x15b9, 0x1eb1, 0x1923, 0x1b16, 0x4352, 0xbfde, 0x42e7, 0x4130, 0xb248, 0x43a6, 0x460a, 0xbf39, 0x1a10, 0x4375, 0x4089, 0xbacc, 0x26f9, 0x4582, 0xbf6d, 0xb20a, 0xba36, 0x41c7, 0xb22a, 0xb05e, 0xba3f, 0x1ffe, 0x34c2, 0xb244, 0x2181, 0xb2f0, 0x3c08, 0x08a6, 0xba2d, 0x4221, 0x33fb, 0x4162, 0x439d, 0x43e9, 0x1eb8, 0x430e, 0x1f07, 0xbadb, 0xba6e, 0x02d2, 0xbf4e, 0xb294, 0xb2a7, 0xba6c, 0x41a4, 0x4647, 0xb231, 0x41c2, 0xb29e, 0x0648, 0x0ca8, 0x4273, 0x425d, 0xbf33, 0xa38e, 0xba5e, 0x407a, 0x1415, 0x3523, 0x4023, 0x2600, 0x4029, 0x426a, 0xb25a, 0xb2c1, 0x1b6e, 0x42aa, 0x4040, 0xb2bd, 0xab1b, 0x40de, 0xb05c, 0x427f, 0xbf2e, 0xb004, 0xba52, 0x1b44, 0x413d, 0x4147, 0x41a7, 0xbac9, 0xb27d, 0x1c52, 0x42aa, 0x1c6e, 0x43d7, 0xb04a, 0x4676, 0x1f18, 0x41fe, 0x42db, 0x4252, 0xbf4d, 0xb098, 0x3625, 0xb29f, 0x43c0, 0xa08e, 0xbf8c, 0x4317, 0xba78, 0x40ae, 0x3641, 0x42c7, 0xaf31, 0x40ae, 0x4360, 0x4066, 0x1a32, 0x4138, 0xba1f, 0xba21, 0x23aa, 0xba5f, 0x45e9, 0xba0c, 0xbfc6, 0xb2ad, 0x4389, 0x0aea, 0x0da6, 0x17ad, 0x1de4, 0x4247, 0x4040, 0xb2e0, 0x43d1, 0x1a71, 0xb088, 0xba62, 0x4034, 0x4077, 0x40dc, 0x1a7b, 0x05a4, 0xba3d, 0x4424, 0xbf55, 0xbfb0, 0x1408, 0x408c, 0x43c3, 0x42dd, 0x4464, 0xa818, 0x1161, 0x19ea, 0xb0ce, 0x24ce, 0xb2af, 0x15fa, 0x027b, 0xba01, 0xa369, 0x014f, 0x4036, 0x43ed, 0x40e5, 0x320e, 0x4214, 0xbaed, 0x275f, 0x400c, 0xba13, 0xa866, 0xbfb0, 0x41b7, 0xbfcd, 0x2c19, 0xbaea, 0x421e, 0xb296, 0xbf42, 0x42cc, 0x1726, 0xb08d, 0x4314, 0xbf87, 0x4107, 0x0425, 0x4586, 0x3b4d, 0x40c1, 0xbafc, 0xb0d5, 0xbf1e, 0x0b9a, 0xba4c, 0xb0e4, 0x3056, 0xbf80, 0xafd4, 0xb2fb, 0x4395, 0x4669, 0xb296, 0x1914, 0x4009, 0x2ac8, 0x4102, 0x4116, 0x40c1, 0x4130, 0x4493, 0x1bc0, 0xb250, 0x463b, 0x0469, 0x424b, 0x1b6b, 0x3e46, 0xb2fa, 0x4151, 0xbfac, 0x44d9, 0x1c7a, 0x1a00, 0x420b, 0xbaf5, 0x407e, 0xb21e, 0xad7b, 0x425c, 0xbf7a, 0x1db5, 0xae92, 0x1b35, 0x1ea1, 0x2cf5, 0x1af1, 0x4210, 0xbaf4, 0x42a0, 0x4124, 0x41dd, 0x4215, 0x13b4, 0x43e6, 0xbfa4, 0x416b, 0x1e6a, 0xbfc0, 0x0d58, 0x40c0, 0x454e, 0x0202, 0x40b5, 0xb276, 0x4333, 0x4574, 0xa2ac, 0x2a76, 0x425d, 0x40fc, 0x40da, 0xba47, 0x2428, 0xbf19, 0xb2a3, 0xb269, 0x2843, 0x3a89, 0x1f48, 0xbf7f, 0x416c, 0x4208, 0x4356, 0x43f2, 0xbae5, 0xb274, 0x4037, 0xb237, 0xaa79, 0x42d2, 0x1d34, 0x426f, 0xb236, 0x43a3, 0x4200, 0xbf60, 0x18b8, 0x1d7f, 0xbf8d, 0xb273, 0x4190, 0x3d1e, 0x415c, 0xba38, 0xbfc0, 0x13f8, 0x4117, 0xba3e, 0xb274, 0xbfb5, 0x4077, 0xb279, 0x4009, 0x3ac8, 0xaa7c, 0x0983, 0xbf8b, 0xba24, 0xb27b, 0x1e7e, 0xb0d1, 0x42c2, 0xaba6, 0xba50, 0x1eea, 0x056b, 0xa171, 0x43b3, 0x35ca, 0xb0e3, 0xb0c5, 0x08e0, 0xb22e, 0x414b, 0x41e6, 0x40bc, 0x2800, 0x402f, 0x409f, 0x164c, 0x44b3, 0x4154, 0x4158, 0x418a, 0x4028, 0xbf94, 0xb28b, 0xaff1, 0x41ec, 0x0d8d, 0x41a2, 0x1b3b, 0xaf95, 0xba4a, 0x1ee5, 0x40cd, 0x1840, 0xae03, 0xb0bc, 0x4643, 0x434f, 0x0c96, 0xba07, 0x3748, 0x46d8, 0xbf26, 0x4195, 0x0b00, 0x41ab, 0x4315, 0x405a, 0x4145, 0x14f4, 0xa261, 0x42dc, 0x3318, 0x424c, 0x1fdc, 0x0500, 0x1acf, 0xba4d, 0x1f49, 0x40c1, 0x1825, 0x42cf, 0x4175, 0x4097, 0x293b, 0xba0d, 0x4014, 0x207a, 0x407d, 0x4361, 0x437a, 0xbf4b, 0x4086, 0x41a0, 0x3c12, 0xb0ca, 0x460c, 0x4162, 0xb24d, 0x425e, 0xb2cd, 0x0dcb, 0x13ba, 0x40c2, 0x41af, 0x29cb, 0x0956, 0x41aa, 0xb216, 0x4112, 0x4349, 0x4431, 0x2734, 0x422e, 0xbf2a, 0x40b8, 0x4649, 0xb068, 0x438c, 0x426d, 0xabc5, 0xb0a7, 0xbf9c, 0x4671, 0x42e1, 0x43ee, 0xb02a, 0x4217, 0xb0ce, 0x4026, 0x4269, 0x4374, 0x3931, 0x02f5, 0x42c6, 0x43aa, 0x2532, 0x2c53, 0x4231, 0xb2dd, 0x43a9, 0x1c6d, 0x4274, 0x2948, 0x4162, 0xbfa3, 0x427f, 0xb01c, 0x0396, 0x4174, 0x1327, 0x36ab, 0x41f4, 0x41b6, 0x3fda, 0x4648, 0x4095, 0x1c98, 0x41a0, 0xbf56, 0x4144, 0xb0dd, 0xb28c, 0x42d5, 0x4280, 0xbf81, 0xaccd, 0x45ee, 0x4145, 0x4342, 0xbfb7, 0x4261, 0xba45, 0xb01f, 0x418a, 0x439f, 0x42bf, 0x43fc, 0xba22, 0xbfe0, 0x2446, 0xbfde, 0x1c19, 0xb0f5, 0xba68, 0x1b4a, 0x00d7, 0x4377, 0x425d, 0x46db, 0x411e, 0xb2b8, 0x406a, 0x1945, 0x43b9, 0xa423, 0x42af, 0xbf76, 0x412e, 0x1fa8, 0xb25f, 0x4591, 0xb27f, 0x461b, 0x41e3, 0x0cc4, 0x0236, 0x4157, 0x288b, 0x05d7, 0xbf87, 0xba66, 0x43bf, 0xba33, 0x18c4, 0x416c, 0xbacb, 0x0077, 0x398d, 0x414d, 0x1fc4, 0x447d, 0xbaef, 0xb2a8, 0x1833, 0xba42, 0xbf51, 0x41c6, 0x43ae, 0x420b, 0x2a87, 0x437e, 0x4206, 0x42ff, 0x4221, 0xa594, 0x2023, 0x4311, 0xb241, 0x416f, 0x40e2, 0x0418, 0x422b, 0x43c2, 0x4385, 0xbf81, 0xb284, 0x408d, 0x40ee, 0x4377, 0x43b3, 0x416d, 0x4353, 0x432b, 0x44a1, 0x43d3, 0xbf81, 0xbaf1, 0x40a6, 0x42c2, 0x0cf8, 0x4068, 0x1ff8, 0xb231, 0x41f9, 0x430f, 0x43a0, 0x1705, 0x0110, 0x09ac, 0xbace, 0x4230, 0x46d4, 0xaec3, 0x4353, 0x416e, 0xb20e, 0x43fc, 0xba7d, 0xbf17, 0x41b0, 0x33b5, 0xba1c, 0x4046, 0x400a, 0xbf8f, 0xb0bd, 0x42d2, 0xacb0, 0x1ddc, 0xb2f1, 0x40a3, 0x4082, 0x4318, 0x2d71, 0x19ec, 0xb0fd, 0x4336, 0xb2cd, 0x40cc, 0xaba4, 0x1a86, 0x42f0, 0x4668, 0x38ff, 0x1e05, 0x4040, 0x184a, 0x4015, 0xbf38, 0x4242, 0x4626, 0xad8d, 0x43c2, 0xb277, 0x4030, 0xbae2, 0x41a5, 0x4215, 0x1513, 0xbf6a, 0x4142, 0x43ba, 0x44f0, 0x4178, 0x2703, 0x428d, 0x42c0, 0x2ffb, 0x438b, 0x42cd, 0x40b5, 0x4435, 0x445d, 0xa385, 0x4082, 0x4148, 0xb2eb, 0x022f, 0x423b, 0x42f5, 0xbf5f, 0x04f0, 0xb098, 0x40a5, 0xba1f, 0x44c1, 0x3b57, 0x4094, 0x41ed, 0xb208, 0x3d9b, 0x0060, 0xb0d1, 0x4352, 0x2507, 0x4282, 0x4273, 0xa9fa, 0xa561, 0xb2b5, 0x3a86, 0x4076, 0xa2e9, 0xbfc3, 0x2a84, 0xbad6, 0xba74, 0x4361, 0x074c, 0x4169, 0xba42, 0xb086, 0xb070, 0xb23e, 0x3b35, 0x387f, 0x4303, 0x1fb5, 0x4267, 0xba3a, 0xbade, 0xbfa8, 0x41aa, 0x41e6, 0xbf80, 0x43fb, 0xbf90, 0x1deb, 0x43bc, 0x4019, 0xb21d, 0x409f, 0xb24c, 0x197a, 0x4264, 0xb2c1, 0xba06, 0xba7d, 0xb00e, 0x43e2, 0x4624, 0xabe7, 0x4348, 0x42b3, 0xbfc5, 0xb209, 0x40ef, 0x43e5, 0x3ada, 0x3431, 0x414a, 0x41de, 0x136c, 0xb01d, 0x413b, 0xba68, 0xbfa0, 0x1ecb, 0x402f, 0x43df, 0x43ca, 0x1de1, 0xbf34, 0x1d54, 0x2ffe, 0x43bf, 0x1dcb, 0x41fd, 0x438d, 0xb2c1, 0xbf8c, 0xaf57, 0x46d3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x83fd023f, 0x982a2ada, 0x932883b1, 0x8b07a16b, 0x6509d713, 0x387a315c, 0x306e7acf, 0x817d06d5, 0xc0872b3a, 0x0acac3f9, 0x3a05fe03, 0x78554f05, 0x868e89dd, 0x6a74d709, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x80000000, 0x00000000, 0xffffff4a, 0x0000040e, 0xffffff4f, 0x00800000, 0x5a9b0180, 0x00000000, 0x78554db0, 0x8204e12c, 0x3a05fe03, 0x3a05fe03, 0x3a05fe03, 0x6a74d1d9, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x3ec5, 0x42d3, 0x06b4, 0xbf80, 0x430d, 0x40b3, 0x3a02, 0x43fd, 0x1ef3, 0x4051, 0x426b, 0xa108, 0xbf48, 0xba7c, 0xba08, 0x2831, 0xb055, 0xbad1, 0xb278, 0x0b30, 0xbff0, 0xa224, 0x1beb, 0x24d4, 0x1873, 0xb2c9, 0x44d9, 0x1eb9, 0x4086, 0xb2b6, 0x4172, 0x40e2, 0x1f7c, 0x403a, 0xbf9c, 0xa542, 0x19da, 0x429c, 0x42c9, 0x431d, 0x1d07, 0xb059, 0x3791, 0x306d, 0xa7cc, 0xb28a, 0x43c8, 0xbf62, 0xb02d, 0xba06, 0x1b78, 0x21ea, 0x41b6, 0xbfae, 0x0ca8, 0x1ae7, 0x4307, 0x39e3, 0x1e5e, 0x03f8, 0x04c8, 0x427d, 0x0676, 0x3256, 0x1b57, 0xb080, 0xb2cb, 0x418d, 0xb2ae, 0xbf51, 0x412f, 0xba3c, 0x1916, 0x4198, 0x40da, 0x3425, 0x43bb, 0x4339, 0xbf59, 0x42ff, 0x3058, 0x4108, 0xba09, 0x1c3b, 0x1f5b, 0x3703, 0x404a, 0x4259, 0x40b3, 0x40a8, 0x3e59, 0xa7ab, 0xbf66, 0x404f, 0x42bc, 0xbaf0, 0x22c3, 0xbfc0, 0xb03d, 0x4018, 0x1a56, 0x4383, 0x4446, 0x42c1, 0xb292, 0x246e, 0x4426, 0x1aec, 0x1fa7, 0xbf07, 0x3231, 0x4116, 0x1177, 0x45c6, 0x3f2b, 0xa87c, 0x4170, 0xba41, 0x436e, 0x4012, 0x4553, 0xadc9, 0xae57, 0x4034, 0xbaf0, 0xa094, 0x293a, 0x412d, 0x4145, 0xbfc7, 0xb0ed, 0x164e, 0x2376, 0x19ae, 0x404e, 0x338a, 0x42a5, 0xbad0, 0xb275, 0x4217, 0x40c3, 0x40a8, 0x4015, 0x1ce6, 0xb2f5, 0x435d, 0x403b, 0xaaed, 0x41e0, 0x1b07, 0x4390, 0x08eb, 0x2e05, 0x41c0, 0x193f, 0x421f, 0x2648, 0x4074, 0xa3c0, 0xbf0d, 0x0eb2, 0x29cd, 0x403b, 0xafe2, 0xba0a, 0xbff0, 0x40db, 0x1f18, 0xba01, 0xbf90, 0xb0e6, 0xb096, 0xbaf6, 0xbf90, 0x18a5, 0xbfbb, 0x40eb, 0x1d3b, 0x4150, 0x1993, 0xbf9d, 0x413c, 0xb2e0, 0x1b5d, 0x415a, 0xb244, 0x0152, 0xb2ed, 0x19ce, 0xba30, 0xba2e, 0xb277, 0xb271, 0x0387, 0xa74d, 0x4088, 0x4141, 0x4226, 0xbf5c, 0x46e2, 0x0897, 0x41c1, 0x428a, 0xb272, 0x1e00, 0x42d3, 0xb297, 0xba46, 0x40e1, 0x18fb, 0xbf0b, 0xba08, 0xb222, 0xbfb0, 0xbacf, 0x1a87, 0x428a, 0x42ee, 0x414c, 0x4606, 0x4375, 0x45db, 0x4171, 0x40f1, 0xb297, 0x4133, 0x3d66, 0x4206, 0x43ec, 0x43f8, 0xbf2e, 0xbfc0, 0xba27, 0x43ef, 0x407f, 0xbaef, 0x4306, 0x4350, 0x1e3c, 0x4085, 0xb25f, 0x4039, 0xb2dd, 0x40e1, 0x462b, 0x4633, 0x4048, 0x17ca, 0xba44, 0xbfa3, 0x4208, 0xb282, 0xba26, 0xb2da, 0x00d2, 0x4151, 0x40f7, 0x42e9, 0x4203, 0xba5a, 0x203f, 0x465b, 0xb20a, 0xbfdc, 0xa24b, 0x2188, 0x144b, 0xbfc0, 0x4001, 0xbf74, 0x41c4, 0x15cf, 0x42fc, 0x4269, 0xaad7, 0x4353, 0x308d, 0xb27f, 0x4242, 0x4276, 0x43b6, 0x4243, 0xa999, 0xba34, 0xa8cb, 0xbfe8, 0xb0e3, 0x40c0, 0x402e, 0xb22d, 0xb06b, 0x1b53, 0xb0bb, 0x447f, 0x459a, 0x4083, 0x31f0, 0xb018, 0x3814, 0x43e1, 0x1c61, 0x417d, 0xa3ab, 0xbac9, 0x0d83, 0xa3e9, 0xbfcb, 0x1c34, 0x0847, 0x012c, 0x34c5, 0x412f, 0xb21c, 0x4286, 0xb24c, 0x40eb, 0x1dc0, 0x4006, 0x437d, 0xb01b, 0xbf45, 0x38bb, 0x4092, 0x41ab, 0xb283, 0x424c, 0xb036, 0xa4ec, 0x1f66, 0x1f00, 0xbac8, 0xbf8a, 0x42d7, 0x42aa, 0x4140, 0x09ec, 0x183a, 0x1e06, 0x405c, 0xb05b, 0xb0ef, 0x1d87, 0x1807, 0x41bf, 0x411c, 0x4168, 0xbf19, 0xb050, 0xad8f, 0x1cd4, 0x0d12, 0x411f, 0x4071, 0x187d, 0xa451, 0x4354, 0x4170, 0x17d0, 0x419b, 0x41ca, 0x4293, 0x1d2e, 0x41e0, 0x4131, 0xba25, 0xbf4c, 0x4313, 0x22f1, 0xbf91, 0x4601, 0xb2d8, 0x39d6, 0xb2d3, 0x1e51, 0x46c0, 0xba66, 0x2408, 0xb0aa, 0xb0b7, 0xbf5d, 0xbfc0, 0x438d, 0x24bb, 0x23a9, 0x4285, 0x1d19, 0x431b, 0xa1e4, 0x43b2, 0x1015, 0xbad8, 0x408d, 0x40c5, 0x401d, 0xbf77, 0xb0a3, 0x40d9, 0x3dd4, 0x4291, 0xb2ad, 0x4611, 0x4243, 0x42d3, 0x4274, 0x430f, 0x1f8f, 0x4458, 0xbf43, 0xbaf1, 0xb05f, 0xba34, 0xb2bd, 0xb2f2, 0xa7ef, 0x4368, 0xb2b9, 0xbfae, 0xbf00, 0xbad8, 0xbaf8, 0x40f2, 0x403e, 0xbf3d, 0xbaff, 0x4352, 0xb0b8, 0x414a, 0xa4e6, 0x1278, 0x419d, 0x41be, 0x1ef6, 0x4611, 0xb240, 0x4300, 0x4151, 0xba3a, 0x2800, 0x40c0, 0x40f1, 0x433b, 0x1e6b, 0xbf83, 0x40e0, 0x43cc, 0xb089, 0x42ff, 0xafd5, 0x4288, 0x1bd6, 0xbfc0, 0xb2e9, 0x403b, 0x40e1, 0x1ff6, 0x408d, 0xb284, 0x40f6, 0x41eb, 0x1c1c, 0x411c, 0x439d, 0x4546, 0x418d, 0xb288, 0x4026, 0x31bb, 0xbfb9, 0xb032, 0x421e, 0x0492, 0x4349, 0xac8e, 0xb272, 0x40d2, 0x4017, 0xbf90, 0xbf00, 0x2e71, 0xba43, 0x421b, 0x4094, 0x4223, 0xbf49, 0xb2a3, 0xb286, 0x02b5, 0x424a, 0x426f, 0xb21c, 0xb277, 0x4137, 0x43fe, 0xbf4d, 0x418a, 0xba41, 0x05a2, 0x4005, 0xbf34, 0x4591, 0xb209, 0xba3e, 0x418c, 0x4190, 0xb2ab, 0x4243, 0x04ac, 0x40b1, 0x3fd9, 0xbfcf, 0x1bd4, 0xb2b3, 0x42ba, 0xb0e0, 0x1cf2, 0x3fc6, 0x042d, 0xbf26, 0xba72, 0xa972, 0x018f, 0x42e3, 0xab4b, 0xb253, 0x40a4, 0x43bc, 0xbad4, 0x444a, 0xb298, 0x4095, 0x05a4, 0x3e20, 0x41dc, 0xba4d, 0xb29a, 0x4210, 0xba2d, 0x40e4, 0x3b94, 0xbfe0, 0x405d, 0x43cc, 0x4282, 0xbfca, 0x4296, 0x41d1, 0x4319, 0xba33, 0x417f, 0xbfd9, 0xb24d, 0xa08c, 0x405c, 0x4397, 0x4201, 0xb2ff, 0x2561, 0x00f1, 0x3ee1, 0x43d3, 0x4177, 0xb289, 0x407f, 0xbacb, 0x30e3, 0x37a9, 0xba34, 0x432c, 0x43d7, 0x4017, 0x428b, 0xa7e4, 0x42be, 0xbf3c, 0xba26, 0xb25c, 0x414c, 0x3494, 0xaeb7, 0x42f3, 0x1325, 0xbf96, 0x089b, 0xbf60, 0x4147, 0x0189, 0x42fc, 0xbfa4, 0x4137, 0xbac5, 0x1f1c, 0x42ea, 0xb2a0, 0x39d2, 0x1068, 0xb2a7, 0x4371, 0x4086, 0xbad1, 0x418e, 0x211d, 0x1284, 0x0ab4, 0x359e, 0x4161, 0x1eab, 0x0245, 0x430b, 0x42ea, 0xba51, 0x4082, 0x43a9, 0x03f5, 0xa5cb, 0x1316, 0xbf36, 0x4611, 0xbaee, 0xba0a, 0xbfd2, 0x409c, 0xb091, 0x43b3, 0x4248, 0x426e, 0xba02, 0x0db8, 0x1a5b, 0xbfb5, 0x43b0, 0xb2ad, 0x09aa, 0x1205, 0xb266, 0x415b, 0x434d, 0x1fe3, 0xb0a3, 0xbf93, 0x4559, 0x4439, 0x212b, 0x3af6, 0x189c, 0x1be0, 0x4248, 0xb285, 0xbf96, 0x41c3, 0xb254, 0x0c55, 0x4274, 0x3ed3, 0x431c, 0x407a, 0x43fe, 0x1e2d, 0x18a3, 0x4126, 0x42ad, 0x06fe, 0x29b3, 0x144b, 0x4051, 0x4114, 0x1f76, 0x1e83, 0x1855, 0xbadb, 0xbf34, 0xb291, 0xb2d4, 0xb253, 0x414e, 0xb282, 0xba7a, 0x41f3, 0xba13, 0xb20a, 0xbf36, 0x428d, 0x1d7b, 0xbad0, 0xb2bf, 0xb254, 0xb0a6, 0xb078, 0x4053, 0x43d9, 0x4162, 0xb274, 0x41fd, 0x184a, 0x4399, 0x2dd4, 0x4307, 0x40f4, 0xba36, 0x43c2, 0x4197, 0x436c, 0x4203, 0x1fbe, 0xbf31, 0x408d, 0xb2ee, 0x4226, 0x414e, 0xbf3a, 0x23bf, 0x46c9, 0x3892, 0xba42, 0x41b5, 0x1b55, 0x4282, 0x401d, 0x1d0c, 0x1a9c, 0x257a, 0xb2b4, 0x42c1, 0x4201, 0xb23b, 0x43a1, 0x43bd, 0x407e, 0x43d6, 0x1b28, 0x13fb, 0x4064, 0x1cb2, 0x314e, 0x2ae2, 0xba18, 0xbf0b, 0xa9ec, 0x4165, 0x43d1, 0xb2ab, 0x1879, 0xb0fe, 0xb290, 0x380d, 0xb22c, 0x1d2e, 0x17c1, 0x4011, 0x43f2, 0xbf87, 0xba26, 0xb293, 0x1854, 0x34a5, 0xb2f5, 0x43c1, 0x2ea9, 0xa962, 0xb266, 0x0e46, 0xb055, 0x412d, 0x43e8, 0x419a, 0xb23e, 0x1b36, 0x180b, 0x434f, 0x1a50, 0xbac0, 0xbaf0, 0x09f6, 0x18d9, 0xba51, 0xb2ee, 0x32fe, 0x403b, 0x1db0, 0x0fa6, 0xbf97, 0xb26b, 0x4315, 0x42a2, 0xba2a, 0x431b, 0x1a42, 0x1f3a, 0xbf4e, 0x40cc, 0x41da, 0x4109, 0x422d, 0x1bd6, 0x439f, 0xb203, 0xbfd1, 0x462d, 0x404b, 0x420b, 0x4298, 0x153f, 0xb0c7, 0xba66, 0xb258, 0xb27a, 0x417e, 0x1ca0, 0x46f9, 0x0c87, 0x401d, 0x42e3, 0xbacc, 0x2a1c, 0x21ff, 0x4187, 0xb001, 0x4054, 0xbae9, 0xbf98, 0x411c, 0xb27f, 0x2f39, 0x42cc, 0x4602, 0xb07f, 0xb21b, 0x418b, 0x03f0, 0xb021, 0xb03f, 0xbf5a, 0x410c, 0x40ac, 0xa162, 0x4021, 0xba5b, 0xa977, 0x204c, 0x43f6, 0x40a9, 0x1e9c, 0x4283, 0xbf2d, 0x2abd, 0x4249, 0xba02, 0xa9c6, 0x4008, 0x4424, 0x2c94, 0xbf61, 0x18ae, 0xb07d, 0x46c3, 0x185f, 0xb286, 0xb2b7, 0x1ada, 0x4147, 0x403a, 0x2702, 0x4065, 0x404c, 0x4274, 0x42aa, 0x434f, 0xb0a5, 0x4315, 0x0ac8, 0x19eb, 0xa9ea, 0x2541, 0x460f, 0x4634, 0xbf25, 0x436c, 0x42a5, 0xb256, 0x425f, 0xb20d, 0x4225, 0x4669, 0x23f8, 0xab09, 0x43c7, 0x34cf, 0x0f49, 0x415d, 0xba25, 0x4230, 0xac19, 0xb243, 0x42d7, 0x42e5, 0x182d, 0x424d, 0x4213, 0xba2c, 0x1ad0, 0x4301, 0x2446, 0x2b9e, 0xabb5, 0x426d, 0xbfad, 0xa4d2, 0x41c1, 0x1de8, 0x28af, 0xb252, 0x18f0, 0x342f, 0x2d6e, 0xb240, 0x427e, 0xbf5a, 0x41c0, 0x446a, 0x1a36, 0xb223, 0x1b1a, 0x4319, 0xba13, 0x4332, 0x4293, 0x1cc3, 0xb092, 0x4339, 0x41b6, 0x21e9, 0xb237, 0xbaeb, 0xbf06, 0x1d64, 0xa016, 0x41ec, 0x413a, 0x1abc, 0x42fc, 0xb274, 0xb0bc, 0x4016, 0x42d6, 0xb220, 0x43b0, 0x44fa, 0x44cb, 0xbf1c, 0x1ed5, 0x4229, 0x45f4, 0xbf67, 0x4142, 0x1e8e, 0x434c, 0x418e, 0xb0fb, 0x4369, 0x46b2, 0xae1c, 0xbfa6, 0xb294, 0xba4c, 0x0ff8, 0x149e, 0x4201, 0xba72, 0x44a9, 0x465f, 0x42c9, 0xbad2, 0x3fc6, 0x4379, 0x1969, 0x417e, 0x409b, 0x46e8, 0x40a2, 0xba58, 0x1c75, 0x42f0, 0x40ba, 0x12ae, 0x32c5, 0x348d, 0x402a, 0x2812, 0x40e9, 0x41b1, 0xbf89, 0x1914, 0x41ac, 0x07c5, 0x406e, 0x43a2, 0x42f0, 0x1150, 0x3395, 0x3055, 0xbfb6, 0x00f6, 0x40d4, 0x1c99, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x12c1cca9, 0x2d72bbb5, 0x1bcf8ad0, 0x4c840915, 0x42730230, 0xb7a8e9c4, 0xcb0a2f1b, 0xf9eb0e6a, 0x3ce2afc4, 0x2182885a, 0x1ed96c99, 0x0cf55f95, 0xb8e6e008, 0xecaa7d5b, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x00000055, 0xfffcc2a2, 0x00000001, 0x00000795, 0x79854540, 0x0cf57523, 0x00033d5d, 0x0cf57521, 0xecaa795b, 0x0000164f, 0xffffff17, 0x0cf575e7, 0xb8e6e008, 0xecaa795b, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4393, 0x413f, 0x4166, 0xa433, 0x409c, 0xb2e8, 0x1cc5, 0xba5a, 0xaf1d, 0x4249, 0x158a, 0x400c, 0xbf02, 0xba4d, 0x1b2a, 0x0683, 0x409b, 0x4033, 0x2766, 0xbf00, 0x43bc, 0x4593, 0x4623, 0x40f0, 0xba2d, 0x433c, 0xbfcc, 0x0712, 0xa0ee, 0x301a, 0x400e, 0x4218, 0x428a, 0xa5a0, 0x4362, 0xb28f, 0x4270, 0x3fd6, 0x40b4, 0x2c32, 0x4494, 0xb0e0, 0xba73, 0x2cfd, 0x4118, 0xba2a, 0xb24c, 0x2d2c, 0x1a18, 0x00d8, 0xbacb, 0x4396, 0xbf0b, 0xb244, 0x1eb5, 0x4154, 0xbacc, 0xb258, 0x1b71, 0xbfb7, 0xb2f8, 0xb0c5, 0x01d6, 0xbad7, 0x240f, 0x1dd7, 0x42f5, 0x4691, 0xbfb0, 0x1ca2, 0x15b5, 0x3be3, 0x4127, 0x4122, 0x403e, 0x4065, 0x4314, 0x1a38, 0x1e93, 0xba0d, 0x1ea0, 0xbaf7, 0xa3e8, 0x42c5, 0x1f42, 0xbf1b, 0x40c5, 0x461d, 0xb013, 0x40a2, 0x1903, 0x4388, 0xb090, 0xbfc1, 0x42fe, 0x42a9, 0x4353, 0x41c2, 0xba5c, 0x406b, 0x4351, 0x41ba, 0xbfe0, 0xb254, 0x1057, 0x41ba, 0xba06, 0xb259, 0x067d, 0x40f4, 0xba18, 0x45b0, 0x3200, 0x405a, 0xba03, 0xbfde, 0x4127, 0x4379, 0x400c, 0xb20e, 0x0f5d, 0x41b4, 0x1f75, 0x416d, 0x413c, 0x1d18, 0x3c86, 0x41e6, 0x1d1e, 0xb281, 0x1f3c, 0x4218, 0x40f8, 0xbf29, 0x286f, 0xa620, 0xb2bb, 0x0150, 0xbac4, 0xac02, 0x439c, 0xb0fa, 0x40df, 0x1f98, 0x4316, 0x3d6d, 0x414e, 0x4159, 0xba17, 0xbfb1, 0x1a85, 0x4214, 0x423d, 0x0134, 0x4669, 0xb254, 0xbf95, 0xb2ed, 0x428b, 0x4328, 0x4256, 0x12d9, 0x40cc, 0xb23e, 0xb025, 0x42fe, 0x4022, 0xb0da, 0x42a9, 0xad61, 0x1f53, 0x46f9, 0x4462, 0xba7b, 0x3985, 0x4142, 0xa656, 0x21c4, 0x4255, 0xba64, 0xb20b, 0xa593, 0x43be, 0xbf8f, 0x463b, 0xb038, 0xb030, 0x4055, 0x17ca, 0xbacb, 0x1f67, 0x4055, 0x1923, 0x1a46, 0xb2a0, 0x41af, 0x4285, 0xbfa4, 0x3160, 0x42c8, 0x29ec, 0x1c50, 0x412c, 0xbaec, 0xb225, 0x44d4, 0xba56, 0xb23d, 0xbfdb, 0x275b, 0x415f, 0xb0ea, 0x21fb, 0x40d9, 0x4325, 0xb20c, 0x43d8, 0xa916, 0x4239, 0x438b, 0x418f, 0x4275, 0xb25f, 0x2252, 0xb262, 0x4688, 0x1c27, 0xb234, 0x19e4, 0x41b6, 0xbfb4, 0x3226, 0xa8bd, 0x40b1, 0xb24f, 0x1848, 0x436f, 0x4372, 0x436f, 0x426f, 0xb2b6, 0xba5a, 0xb2c6, 0x2bac, 0x4160, 0x4002, 0x40b0, 0xba46, 0x319b, 0xb251, 0x2a9a, 0x428b, 0x4239, 0x28d5, 0x46e9, 0x4030, 0x4167, 0xbf2e, 0xbf00, 0x41cc, 0xb2e1, 0xba20, 0x4080, 0x1179, 0x4484, 0xbaf4, 0xbf48, 0xb2f8, 0x4320, 0x426b, 0xb2ee, 0x350c, 0xb241, 0x4386, 0x3af4, 0x42b4, 0xa8e5, 0x4369, 0x4315, 0x4110, 0x407b, 0xbaff, 0x419e, 0xbf52, 0x40f2, 0x45b5, 0x41ec, 0x402a, 0xb21b, 0xbf6c, 0xb0ad, 0x40cd, 0x4094, 0xb07c, 0x1fbb, 0x286b, 0x422d, 0x1f2b, 0x401a, 0x42aa, 0x406b, 0x4473, 0x41d8, 0xb08d, 0xba59, 0xb03e, 0x1d7d, 0xaf13, 0x45e8, 0x4169, 0x418a, 0x4366, 0xb001, 0x3913, 0x40ef, 0x40fa, 0xbf74, 0x1bcd, 0x42e7, 0x40cf, 0x1959, 0xb20a, 0xb2ec, 0x1261, 0x40ab, 0x4062, 0x34eb, 0x40fb, 0x4255, 0xb052, 0x40e4, 0x414f, 0x4241, 0xb21f, 0xb2f9, 0x2894, 0x4546, 0x2e67, 0xbf0a, 0x2dcd, 0x3df4, 0x1431, 0x426d, 0x162d, 0xbfd8, 0x4255, 0xbaf2, 0x4263, 0xa5ca, 0x43e9, 0xb06b, 0x2b34, 0xba1b, 0xbf60, 0x42bc, 0x0a25, 0x4584, 0x40f2, 0xb07f, 0x3185, 0xb0b6, 0xbaf2, 0xbf86, 0xafbf, 0xb2a6, 0x10ea, 0x4379, 0x43ec, 0x4291, 0x2f80, 0xb2a8, 0x400b, 0xb2a7, 0x4007, 0xb23c, 0x454b, 0x40ab, 0xbad0, 0xbf84, 0x43ab, 0x4252, 0x40c8, 0x400f, 0x43ae, 0xa451, 0x458e, 0xbae0, 0x4683, 0x412d, 0x3ea7, 0x2233, 0x4564, 0x3aa8, 0x421a, 0x1fc8, 0xa2ab, 0xba49, 0x18b2, 0x2a32, 0x40b7, 0xb25e, 0xba1f, 0xb025, 0xb2d7, 0x41f2, 0xbfe1, 0x1fa1, 0xb0a2, 0x43bd, 0x1812, 0x0fdf, 0xba2f, 0x42e5, 0xb238, 0x197b, 0x1d3f, 0xbfa1, 0x1def, 0x1a08, 0x41f5, 0x352c, 0xb021, 0x1891, 0x004f, 0x427c, 0x402a, 0x41fb, 0x4303, 0x4396, 0x3d52, 0xbac4, 0xba78, 0x40b0, 0x1d86, 0xb249, 0x40c1, 0x4030, 0xbac9, 0xbf3a, 0x412d, 0x4335, 0x3fe3, 0xb2f3, 0x0403, 0xb24c, 0x4194, 0x402f, 0x4245, 0x43a7, 0x4298, 0x2c2c, 0x1d25, 0x0a9b, 0xba72, 0x41f4, 0x4083, 0x1161, 0xba17, 0xac8c, 0x4595, 0x42b2, 0x4457, 0x2b24, 0xba1b, 0x4135, 0xb0ba, 0xbf14, 0x41e8, 0x2741, 0x1bcb, 0x3afa, 0x438c, 0xbf36, 0x4028, 0x40d6, 0x42dd, 0xb2cf, 0x4334, 0x4390, 0x41f0, 0xbf8d, 0x4576, 0x2bc5, 0x46b2, 0xa44d, 0x438e, 0x2439, 0xb2c1, 0x181c, 0xa1d4, 0x4448, 0xbf67, 0x4360, 0xb00a, 0xbfa0, 0x423f, 0x32f8, 0x3dbe, 0x4301, 0x43aa, 0x1ba3, 0x4258, 0xbf80, 0x41df, 0xb242, 0x1e9a, 0x0f0a, 0x389a, 0xb2fa, 0x4220, 0x401f, 0x4561, 0x1a77, 0xbf5b, 0xb2e7, 0x1879, 0x4132, 0x46db, 0xb2e1, 0x1aa2, 0x4100, 0x25d0, 0x403b, 0x13ba, 0xbf69, 0x4238, 0xba3f, 0xa5ad, 0x1b87, 0x4406, 0xa157, 0x006d, 0xbfc0, 0x4349, 0x16fb, 0x18ca, 0x406f, 0xb049, 0x43cb, 0x435e, 0xabbe, 0xba66, 0xbf8f, 0x29de, 0xbfe0, 0x35d1, 0x21e8, 0xb2b1, 0x4019, 0x1f16, 0xb07f, 0x4124, 0x437b, 0x324a, 0x4641, 0x4192, 0x40cb, 0xbf5b, 0xb257, 0x1a8b, 0x0f49, 0x0eeb, 0x4244, 0x4038, 0x4499, 0x43ce, 0xbf18, 0x1b06, 0x1fb0, 0xbfa0, 0x2f5c, 0x44dd, 0x4381, 0x4245, 0xb223, 0x4260, 0x1c09, 0x12f5, 0x0433, 0x4285, 0xbf48, 0x43fa, 0x05a8, 0x3557, 0x42ed, 0xb03e, 0xba76, 0x4124, 0x1ab2, 0xbf95, 0x4010, 0x4581, 0x459a, 0xb229, 0x0b3c, 0x4452, 0xa755, 0x4476, 0xb066, 0x13df, 0xb0b4, 0x41f9, 0xb2ac, 0x432a, 0x409f, 0x2410, 0xba48, 0xbade, 0x4291, 0x085f, 0xb2dd, 0x4346, 0xbfc6, 0x4069, 0x408a, 0x45ed, 0x4370, 0x2b4f, 0x463d, 0x1f7a, 0xba33, 0x4119, 0x4119, 0x4277, 0x08b3, 0x24d8, 0x018b, 0x1dd1, 0x4190, 0xb259, 0x408e, 0xb094, 0x434b, 0xa16e, 0x417b, 0xb28c, 0xbf92, 0x1ae1, 0x14f3, 0x1b4a, 0x4549, 0xbf48, 0x3edd, 0x0f59, 0x3f27, 0x18da, 0xb2ee, 0x412b, 0xa606, 0xba3a, 0xb236, 0x41c3, 0xb024, 0xb0d3, 0x40f2, 0x4301, 0xb0eb, 0x40a1, 0x284b, 0x40bd, 0x1e28, 0x41d3, 0x4243, 0xbfca, 0xa24e, 0x4105, 0x0a58, 0x1fd6, 0x41f4, 0xbfc8, 0xb268, 0x42ac, 0x413b, 0xb285, 0xb0d4, 0x400b, 0x43e2, 0x3133, 0x4316, 0x4319, 0x4398, 0xb27b, 0x2780, 0x4343, 0x17d6, 0xb242, 0x1282, 0x1f3e, 0x4285, 0x42a2, 0xb0ed, 0x40a5, 0x439a, 0x1c5f, 0x1d3b, 0xb032, 0xbfbd, 0xb26f, 0x42e2, 0x4350, 0x4353, 0x1d6d, 0xb2af, 0x40ad, 0x41e6, 0x4080, 0x43b6, 0x4064, 0x4007, 0x40a8, 0x4302, 0x11e1, 0x41dd, 0xb2cd, 0x4350, 0xba24, 0x4208, 0xb24d, 0x1c89, 0x4262, 0xba71, 0xbf89, 0x4357, 0x42ef, 0xb2d6, 0xb26e, 0xb208, 0xb275, 0xbf1d, 0xb2ee, 0x1944, 0xb2ce, 0xba7f, 0x36b1, 0x388f, 0x40ce, 0x4368, 0x19b9, 0x01c7, 0x24f5, 0xba7b, 0x4267, 0x4143, 0xbf32, 0x4205, 0xba59, 0x40c0, 0x3dbe, 0x46d5, 0x1ac7, 0x4195, 0x41db, 0xb2bb, 0x43c4, 0x1fcd, 0x42f1, 0x42d8, 0x1df1, 0xb2fa, 0x41d5, 0xbf0c, 0x41a8, 0x4201, 0xa7e8, 0xbf02, 0x42d1, 0x4116, 0x1c10, 0x43f6, 0x40df, 0x0a6b, 0x41ac, 0x42f3, 0xb2b6, 0x43bb, 0x1010, 0xba5d, 0x43c6, 0xb022, 0x3144, 0x41f9, 0xac9e, 0x43b2, 0x3daf, 0xa1b3, 0x19fa, 0xbfae, 0x42b3, 0x467d, 0x4000, 0xaa59, 0xba2c, 0x090f, 0x1860, 0xb022, 0x2209, 0x1d3a, 0xba15, 0x4138, 0x43fb, 0x424b, 0xbaf1, 0x41fd, 0xba49, 0x461b, 0x41b4, 0xae22, 0xa412, 0x1d38, 0xbf4a, 0x4239, 0xb25f, 0x258a, 0xab14, 0x4021, 0x33b7, 0x400d, 0x44eb, 0x439c, 0x1ff6, 0x4056, 0x414b, 0x18e1, 0x34b5, 0xb23e, 0x465d, 0xa16f, 0x41a9, 0x1047, 0x2a0d, 0x290c, 0xa552, 0x403b, 0xbf43, 0x46d2, 0xb07c, 0x42ca, 0xb007, 0xbae7, 0x45e5, 0x41aa, 0xa900, 0x4148, 0xb203, 0x411a, 0x44d0, 0xb236, 0x40bf, 0x2b7a, 0xbfd1, 0x423c, 0x4323, 0xba6d, 0x4325, 0x45b0, 0x4173, 0xbf46, 0xbada, 0x430e, 0x42d7, 0x415f, 0x4335, 0x445e, 0x4038, 0x4170, 0x4041, 0x422f, 0x4206, 0x4306, 0xba01, 0xb080, 0x0116, 0x1e6f, 0x31c7, 0xb210, 0xba4a, 0xb26a, 0xb2a4, 0x3127, 0x4140, 0xafb4, 0xb28c, 0xb279, 0xbf31, 0x4456, 0x40c0, 0x344c, 0x43eb, 0x0763, 0x042b, 0x4160, 0x0f4f, 0x4484, 0xa6f6, 0x4115, 0x436f, 0x3b87, 0xb283, 0x39a6, 0x4013, 0x2aca, 0x414d, 0x4008, 0xb29b, 0xbfe4, 0x466c, 0x43db, 0x42c5, 0xbac7, 0x22cb, 0x4678, 0x4017, 0xb218, 0x1bfe, 0x43a6, 0xb2e2, 0x1b61, 0x4423, 0x2a87, 0x430c, 0x1d9a, 0xb206, 0x425d, 0xb21c, 0xbaeb, 0x1d87, 0x4123, 0xb030, 0xbfc8, 0xb25a, 0xa621, 0x16b0, 0xba19, 0x411f, 0x43b7, 0x26fe, 0x41c8, 0x419f, 0x40cc, 0xa286, 0xba7d, 0xbf4e, 0xb2a6, 0xb21e, 0x19e8, 0x22f9, 0x43e2, 0x1b9a, 0xb0d5, 0xbae9, 0x403a, 0xb2d7, 0x420a, 0x4345, 0xbfd8, 0x434e, 0x2382, 0x40dc, 0xb2dc, 0x1efb, 0xbfaf, 0xbfb0, 0xbaed, 0xbaea, 0x4249, 0x1806, 0x4459, 0xba0d, 0x1be8, 0xb2c6, 0x4346, 0x4302, 0x1483, 0x36d3, 0xb25d, 0x1e58, 0x0383, 0x42e7, 0xb0c2, 0x2be8, 0x1172, 0x1f26, 0x40d3, 0x1c1f, 0x42f0, 0x0b5c, 0xbf41, 0xa6e6, 0x4318, 0x3b45, 0x3690, 0x43c9, 0x40c7, 0x43c9, 0xbf38, 0x44f5, 0xbf00, 0x1d4e, 0xb055, 0x40d0, 0x298a, 0x4352, 0xba74, 0xa308, 0x1c72, 0x44f4, 0xb28c, 0xba4a, 0x438c, 0x4208, 0x1ce7, 0x4581, 0xbf93, 0x445f, 0x4041, 0xb06f, 0x40e4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd71e5741, 0xc26ca9a0, 0xc815b4b8, 0x2e87799c, 0xd6d7ab03, 0x28643399, 0xee47b812, 0x4d2e6ed2, 0xb0352da7, 0xaedffe24, 0x03886f34, 0xf8a78ddb, 0x3400fdbb, 0x65c78639, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x0388d025, 0x880325d0, 0x000017f8, 0x00000000, 0x00000074, 0x0388d02a, 0x00000003, 0x694ff1d5, 0x65c78249, 0x03886f34, 0x0388b858, 0x7789f72c, 0x038871d4, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x45a4, 0x08a7, 0xa344, 0xba0d, 0xba6a, 0x18aa, 0xb2ed, 0x027f, 0x414e, 0x3b58, 0xb06c, 0x4019, 0x443a, 0x43d5, 0x43d9, 0x3988, 0xb2fb, 0x3731, 0x42e1, 0x4198, 0xbade, 0x1852, 0x414d, 0xb2bf, 0xb272, 0xbfb3, 0x4083, 0x42cb, 0x4084, 0x023d, 0x4218, 0xba3e, 0x434c, 0xb2bb, 0x41ee, 0x1d8f, 0xbf08, 0xb2d7, 0x1ade, 0x20d7, 0x0a8f, 0x4679, 0xba77, 0x4156, 0x44c8, 0xbf0f, 0xba07, 0xb234, 0x4326, 0xb018, 0xb28c, 0x159a, 0xb050, 0x442f, 0xa0b3, 0xb0b2, 0x1cae, 0xbfc3, 0x42bf, 0xb23d, 0x4296, 0x409b, 0x46fa, 0x40c7, 0x41a1, 0xbf0f, 0x44f9, 0x28e2, 0x3304, 0x41f9, 0xbf00, 0x462c, 0x4233, 0x1f2f, 0xbad2, 0x2961, 0x2bad, 0xb0ab, 0xbf60, 0x40d6, 0x1633, 0x4280, 0xbf8f, 0x445a, 0x4398, 0x42e2, 0x40f3, 0xb05e, 0x1e98, 0x467e, 0x11ff, 0x1a98, 0x42c5, 0x4038, 0xb223, 0x1e66, 0xbae3, 0x1dcf, 0x23d7, 0xbfd1, 0x4431, 0x43ee, 0x2d7b, 0x4275, 0x07dc, 0x4320, 0x41ea, 0xb25b, 0x0fe5, 0x4131, 0x4108, 0x4338, 0xba61, 0x42d0, 0x0f3d, 0x1a3d, 0x1c5e, 0x4244, 0x40e1, 0xb094, 0xb207, 0x4499, 0xbf6e, 0xb209, 0x41a2, 0x37d8, 0xb018, 0xa0e3, 0xb217, 0x4626, 0x4466, 0x15ef, 0x433b, 0x1cf5, 0x0846, 0x191b, 0xb203, 0x4396, 0x4217, 0xbf69, 0xb090, 0x436a, 0x1b63, 0x426f, 0x0022, 0x4110, 0x2019, 0xb26d, 0x2962, 0xaf0d, 0x43d9, 0xb246, 0x085b, 0xb049, 0x43df, 0x412e, 0x42a5, 0x027f, 0xbf89, 0xb237, 0x42ae, 0x42a7, 0x36db, 0xb25a, 0x43cf, 0x1d28, 0xb0ca, 0x404e, 0xb2a9, 0x402d, 0x4268, 0xad94, 0xbf32, 0x2cef, 0x40c2, 0xba2e, 0x43fd, 0x4168, 0x43d1, 0xb2a8, 0xba30, 0x1b80, 0x40dc, 0x4035, 0x4676, 0xb08c, 0x4267, 0x409b, 0x462f, 0x408d, 0x3b28, 0x18fc, 0x4344, 0x4016, 0x42b4, 0x41a1, 0xbfdb, 0x4156, 0xa1fb, 0xb2b5, 0x1860, 0xb2aa, 0x033d, 0x42cf, 0x4399, 0x430f, 0x410f, 0xba6d, 0xbf51, 0x41fe, 0x2855, 0x45c1, 0x1740, 0x4321, 0xa3e5, 0x422e, 0x4289, 0x01e5, 0x14b8, 0xaef7, 0x4012, 0x1c72, 0x4261, 0xa856, 0x069e, 0xba61, 0x40c1, 0x0edc, 0x4040, 0xba69, 0x4012, 0x4157, 0xba5b, 0x45d6, 0xba32, 0xb000, 0x1efb, 0x4351, 0xbf36, 0x4548, 0xb0b2, 0x344a, 0xba1b, 0x1ff8, 0x4104, 0x43c9, 0xac71, 0x3da1, 0xb095, 0x423a, 0x410a, 0x43c1, 0x10b8, 0x431c, 0x42c6, 0x4100, 0xbfda, 0x3ae2, 0x46d0, 0x4241, 0xbae3, 0x0dba, 0x4297, 0xaa48, 0x286a, 0x42b0, 0x43d8, 0x00e7, 0x10ae, 0x0281, 0xb013, 0x4556, 0x1b24, 0x4093, 0x2ab6, 0x40cd, 0x41aa, 0x193f, 0x2446, 0x4344, 0x4161, 0xb2e5, 0x42e9, 0xbfb4, 0x25e1, 0x40c6, 0x4320, 0x4214, 0x4185, 0x076d, 0xb273, 0x4149, 0x0f14, 0x43bd, 0x109a, 0x4636, 0x411c, 0x43f5, 0xbfe1, 0x1ed8, 0x437e, 0x40b7, 0x0795, 0x1f94, 0x4678, 0x00c4, 0x0040, 0x1f3a, 0xbaef, 0x096f, 0x3ce4, 0x4061, 0x4241, 0x42ca, 0xb250, 0x458e, 0x308f, 0x45ba, 0x41f6, 0xb22d, 0xb0a2, 0xbaf1, 0xbfb6, 0x0318, 0x0f57, 0xac8a, 0xb2bf, 0xbfd0, 0x438d, 0x1a76, 0x4447, 0x42c0, 0xbfae, 0x182a, 0x1469, 0xa377, 0x1b30, 0xb2f1, 0x1354, 0x42e2, 0xaf4d, 0xa156, 0x40f8, 0xb26a, 0x416e, 0x2cdc, 0x2741, 0xb06b, 0x4322, 0x4689, 0x41d1, 0x462a, 0x1f1d, 0xb297, 0x415d, 0x26de, 0x4089, 0xbf78, 0x4359, 0x42f6, 0xa6c6, 0x19f5, 0x29df, 0x4285, 0x459e, 0xb297, 0x2047, 0xbf4c, 0x0551, 0x1b21, 0xb27c, 0x413a, 0x4369, 0x4052, 0xbaeb, 0x3da1, 0x06a6, 0x40de, 0x4154, 0x18f7, 0x45ea, 0x44e4, 0x43e9, 0xba2e, 0x30a4, 0xaf60, 0x430e, 0xbac3, 0x1af2, 0x4158, 0xb21a, 0xbf2b, 0xbadf, 0x4055, 0x41d9, 0x4018, 0x44cd, 0xbfd0, 0x3e73, 0xa07c, 0xbad5, 0xb2a1, 0x1a51, 0x41e0, 0xbfe0, 0x462f, 0x409f, 0x4095, 0xba69, 0x04c2, 0x426e, 0xbf00, 0x40ae, 0x409d, 0x4312, 0x417f, 0x4312, 0x098f, 0xb0e9, 0xbfa4, 0x416a, 0xb074, 0x44f1, 0x421b, 0xb02c, 0x3394, 0xaced, 0x4636, 0x43b6, 0x2b1c, 0x1f46, 0x1ab5, 0x4235, 0x4030, 0x40e0, 0xba47, 0xb207, 0xb25c, 0x1786, 0xbaf3, 0x25bc, 0x430b, 0x4179, 0x436c, 0xbf7b, 0x4616, 0xb033, 0xa35a, 0x0a2f, 0x433c, 0x434f, 0xb254, 0xb2b7, 0x40a2, 0x140c, 0x1d7a, 0xb0df, 0xbf9b, 0x1396, 0x4323, 0x04ee, 0x1ad3, 0x1df6, 0x4001, 0x0825, 0xa444, 0x4221, 0x2085, 0xbfb0, 0x42fc, 0xa67e, 0x419c, 0xba0a, 0xa828, 0x40ec, 0x28b6, 0x4364, 0x4362, 0x2892, 0x419c, 0xb206, 0xb2f1, 0x1795, 0xbf6d, 0x0f94, 0x43e5, 0x1e8c, 0xa665, 0xb0e9, 0x06c5, 0xb22e, 0xbac4, 0x1f02, 0xba21, 0x0048, 0x43f7, 0x402f, 0xbf69, 0x43e2, 0x39d0, 0x1c2b, 0x0c3d, 0x41f5, 0x40a7, 0x4254, 0x4303, 0xa98f, 0x19bb, 0x27cd, 0x43bc, 0xb219, 0x4208, 0x43db, 0xb0e1, 0x42ac, 0x43c3, 0x400d, 0x4151, 0x45e1, 0x4097, 0xba76, 0x0cb1, 0x420d, 0x42eb, 0x466a, 0xbf9b, 0x409e, 0x46c0, 0x4168, 0xaba2, 0x01b0, 0x358b, 0x42e1, 0xbad1, 0x43e7, 0x415f, 0x42bf, 0xb09d, 0x1b7c, 0x38eb, 0xb212, 0xb298, 0x1b3e, 0x3148, 0xbfbd, 0xb038, 0xb003, 0x40b0, 0x1604, 0xbf5b, 0x45a4, 0x1e35, 0x2f5a, 0x0a27, 0x40ff, 0x42d9, 0x4196, 0x0bf3, 0xb0cf, 0x43c3, 0x14c9, 0x41c2, 0x419c, 0xb05f, 0xba3d, 0x407e, 0x4118, 0xbaed, 0x2f0a, 0x066f, 0x409c, 0x4132, 0x40c5, 0xbff0, 0x4592, 0xbf26, 0x1bfb, 0x1004, 0x439a, 0x41bb, 0xbf5e, 0x4543, 0x4243, 0x1faa, 0x4099, 0xb2c4, 0x1c03, 0xb2a7, 0xb200, 0x42e0, 0x01fa, 0x4083, 0x40dd, 0x19d7, 0xb2a3, 0xb2e3, 0xbf8a, 0x19a1, 0xa995, 0x40ad, 0x4564, 0x1a6c, 0x44b1, 0xbf82, 0x29f0, 0x407f, 0x4032, 0x0d94, 0x43c7, 0x1cd1, 0xbfdc, 0xb02a, 0xba7c, 0xa53e, 0x191e, 0x3274, 0x06ff, 0xbf0d, 0xa643, 0x19ed, 0x1ab6, 0x4292, 0xade2, 0x4138, 0x35a7, 0x42a6, 0x2df3, 0xbfc8, 0x410c, 0x194b, 0x40d9, 0x4350, 0xbac2, 0x42b8, 0xba17, 0x408f, 0x430c, 0xb2b4, 0x425b, 0x4319, 0x423d, 0x42c0, 0xb2cf, 0xb233, 0x4601, 0xba6b, 0x448d, 0x435f, 0x4133, 0x421b, 0xbfd7, 0xb263, 0xba08, 0x4386, 0x40c0, 0xba74, 0xb283, 0x4275, 0x405a, 0xa597, 0x3eab, 0xa03e, 0x4317, 0x4194, 0x46d5, 0x40d6, 0x4626, 0x407a, 0x4618, 0xba4b, 0x4185, 0x1fd4, 0xbac4, 0x4676, 0xbad6, 0xbade, 0x42a0, 0x425b, 0x43ee, 0xbf5a, 0x4137, 0xba4e, 0x40aa, 0x43e6, 0xb283, 0x4295, 0xa6de, 0x418f, 0x4314, 0x0bb1, 0x4278, 0xbaea, 0xb256, 0xb0c2, 0x0164, 0x218f, 0xbfb1, 0x40d5, 0x40d7, 0x19d6, 0x4197, 0x4255, 0x42cf, 0x4220, 0x41f8, 0x455e, 0x42fc, 0x42b1, 0x40e2, 0xb01b, 0xb207, 0x4362, 0x0dc2, 0x2f3b, 0x2d06, 0xb259, 0x46ec, 0x46c9, 0xba27, 0x400c, 0xba15, 0xba27, 0x28db, 0x416d, 0x1f98, 0xbf66, 0x4101, 0x4379, 0x4008, 0x26ab, 0x2a72, 0x14da, 0x4675, 0x40b5, 0xbaf2, 0xb2a0, 0x26d3, 0x44c5, 0x094c, 0x41cc, 0xbad0, 0xb255, 0x40f8, 0xbaf0, 0x41d1, 0x1a2c, 0x19e9, 0xb213, 0x263a, 0x429b, 0x4443, 0x1d60, 0x401d, 0xb2b7, 0xbf25, 0x1edf, 0x437a, 0xb0e0, 0x1fdb, 0x1eff, 0xbf08, 0x40c6, 0x418d, 0xac60, 0x41c8, 0x4311, 0xb056, 0x43fd, 0xba3a, 0x1e0a, 0x1c07, 0x4110, 0xba6f, 0xba05, 0x465e, 0x198e, 0xba67, 0x40eb, 0xbfae, 0xb2af, 0x4272, 0xa551, 0xb29a, 0x43dc, 0x42df, 0xba5c, 0x400b, 0x1e9c, 0xae12, 0x447c, 0x429a, 0x456d, 0x41f1, 0x4650, 0xba05, 0x40b7, 0x4385, 0xb2de, 0x4067, 0xbf2a, 0x4065, 0xb20c, 0xbf80, 0x26f3, 0x4060, 0xb27a, 0xba1e, 0xb2e5, 0x1006, 0xbfd0, 0xb226, 0xb225, 0x430a, 0x412a, 0xbf83, 0x1af2, 0x1ce9, 0x413e, 0xa670, 0x43ef, 0x43c2, 0x24d3, 0x1cdf, 0xba14, 0xbf07, 0x4635, 0x432c, 0x2b26, 0x0b9d, 0x443a, 0x1dff, 0x1813, 0xb232, 0xbf77, 0x4308, 0x0d17, 0x1c3f, 0xba68, 0x1bac, 0x445e, 0xb228, 0x43a3, 0x1d1d, 0x1fa3, 0x0765, 0x421b, 0x4133, 0x422d, 0x4113, 0x4169, 0xba43, 0x404c, 0x416f, 0x423d, 0x442f, 0xba0a, 0xbacb, 0x201f, 0xb2d1, 0x42ec, 0x32d0, 0x419e, 0xbf83, 0xbac6, 0x41a1, 0x09d4, 0x1866, 0x422b, 0x4190, 0x422c, 0x4120, 0x1b77, 0x1aa2, 0x43b8, 0x400c, 0x43c2, 0x20d1, 0xb283, 0x421d, 0x1c4a, 0x4105, 0xba4e, 0xb2d5, 0xb257, 0x420e, 0x1800, 0x434b, 0xb256, 0x4365, 0xbf33, 0x412e, 0x1fb4, 0x18f4, 0x4175, 0x4122, 0x1309, 0x41dd, 0xb2a7, 0x4093, 0x2718, 0x223b, 0xb2cd, 0xba41, 0xbf57, 0x435c, 0xa9d4, 0x40c6, 0x4628, 0xa687, 0x1abd, 0xba66, 0x42ef, 0x1edb, 0xb2a1, 0x4174, 0xac29, 0x1832, 0x3f4b, 0x40c1, 0xba29, 0x428f, 0xa663, 0xbf29, 0x437a, 0x44d5, 0x3e25, 0x1e7c, 0x46d4, 0x42d5, 0x0605, 0x3dfc, 0xb25f, 0x415b, 0xbfc3, 0x26e3, 0x3d32, 0x43a2, 0xb063, 0x4216, 0xba27, 0x4147, 0xbfc1, 0xb203, 0x40bf, 0x404c, 0x4498, 0x28ea, 0xba56, 0x456c, 0xb0f5, 0xbf80, 0x230a, 0xb21b, 0x1e7c, 0x3ee1, 0x435f, 0x1e9a, 0x417d, 0x404c, 0xbf0a, 0xba00, 0x41c0, 0xb21d, 0x4261, 0x1f4a, 0x4043, 0x41ec, 0x438f, 0xb0f5, 0xb275, 0xbfc6, 0xb0e6, 0x2fb0, 0x430d, 0x06b2, 0x435a, 0xa453, 0x46a8, 0x4074, 0x43a7, 0x0363, 0x404d, 0xbaf1, 0xba12, 0x1a9d, 0x420c, 0x447c, 0x41f8, 0x434c, 0xb292, 0xbf18, 0xb2c0, 0x4294, 0x4258, 0x4060, 0xbf9f, 0xb2b3, 0x3a02, 0xae29, 0xba20, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3cccdcbb, 0xb8b42a16, 0x5515dd66, 0x79ed5e7e, 0x48648228, 0x836897c4, 0x72bc80f9, 0xe9ea3611, 0xd38985d1, 0x943bed5a, 0x921c041b, 0x24eee228, 0x42062aa5, 0xb425effb, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0x5058e007, 0x00001f30, 0x000000d6, 0x0000301f, 0x07e05850, 0x051ddf28, 0x00001ba0, 0x01000000, 0x0000001f, 0x000010d8, 0x00001080, 0x24eee228, 0x00001080, 0x00001afc, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4037, 0x4485, 0x422f, 0x1977, 0x42be, 0x400d, 0x43d8, 0x4058, 0x423d, 0x2582, 0x1864, 0xbfc1, 0x4207, 0x4056, 0x1ff5, 0xb2e1, 0x4389, 0xac7e, 0x45ac, 0x41ad, 0x408a, 0x429a, 0x1539, 0xba4c, 0xa1e6, 0xba01, 0xba09, 0xba5e, 0x421d, 0x4005, 0x247d, 0x40f4, 0xbf52, 0x42ac, 0xb214, 0xba5d, 0x100c, 0x448b, 0x080c, 0x4371, 0xbac9, 0xb2db, 0x4319, 0x43a1, 0x4219, 0x43fe, 0x422b, 0x3cd5, 0xb080, 0x4107, 0x403e, 0x4196, 0x0c01, 0xb2a8, 0x2e67, 0xb242, 0x439e, 0x42c0, 0x047e, 0xbfaf, 0x1b5d, 0x43ec, 0x406f, 0x11e0, 0xb2c9, 0x2097, 0xba48, 0x43a2, 0xbf23, 0xb213, 0x43d8, 0xba5c, 0x0e20, 0x43e3, 0x4052, 0x43ad, 0x2a92, 0x1b79, 0x421d, 0xba10, 0x4612, 0x4408, 0xba35, 0x411e, 0xb273, 0xb241, 0x43ec, 0xb017, 0x2b0d, 0x41fb, 0x3ce9, 0xb2b0, 0x40b7, 0xbff0, 0x4191, 0xb226, 0x2331, 0xbf3f, 0xb0e7, 0xba0c, 0xb016, 0xba4c, 0x4323, 0x4241, 0x40ad, 0xbf23, 0x426d, 0x4083, 0x1bd2, 0xa309, 0x1e36, 0xa492, 0xb0c1, 0x1ba7, 0xba32, 0xb2e5, 0x4340, 0x43a3, 0x41af, 0x1493, 0x11be, 0xba11, 0x4274, 0xba14, 0xb056, 0x1c8c, 0x2879, 0x4028, 0x077b, 0x406e, 0x3ff9, 0xbf85, 0x4284, 0x46b9, 0xb247, 0xa8d2, 0x409f, 0xb2c5, 0xba69, 0x050c, 0x1b37, 0x1086, 0x4247, 0x40d0, 0xac04, 0x41e5, 0x41b2, 0x40ff, 0xac10, 0x431d, 0xa41c, 0x4147, 0x44c9, 0xb2b5, 0x4333, 0xbf51, 0x46a9, 0x4598, 0x4110, 0x403c, 0xb26d, 0xb0aa, 0x2212, 0x449a, 0x2adb, 0x1f40, 0xb0c6, 0xba43, 0x3c79, 0x4152, 0xba29, 0x414a, 0x4285, 0x01b1, 0x3d68, 0x1ff0, 0x43ae, 0x410e, 0xbf6f, 0xba00, 0xa19f, 0xb071, 0x4007, 0x41f6, 0xb28f, 0x435e, 0x0d65, 0xa66a, 0x41db, 0x42d0, 0xb006, 0xb246, 0x4142, 0xbf00, 0xb28b, 0x0d98, 0xb297, 0xbf24, 0x409d, 0x4090, 0x434a, 0x4085, 0x42df, 0x40f4, 0x4009, 0x41af, 0x4157, 0x416f, 0x423d, 0x4281, 0xb05e, 0x4370, 0xba4f, 0x19da, 0xb206, 0xba78, 0xba32, 0xbfa5, 0x43f3, 0x2473, 0x28a5, 0x42ca, 0xbf78, 0xb229, 0xbaee, 0xba52, 0x4157, 0x40e9, 0xaa96, 0x409d, 0x4090, 0xb269, 0x4361, 0x466d, 0x434c, 0x43ab, 0x45b9, 0xba6b, 0x1c45, 0x18f8, 0xbfc7, 0x0236, 0x42ff, 0x35f7, 0x1e55, 0x4148, 0xb2f9, 0x16a8, 0x417e, 0x402d, 0x40cd, 0xb0a4, 0x313e, 0x225a, 0x1136, 0xa497, 0x45f4, 0xbfb6, 0x418e, 0x4639, 0x1c68, 0x419e, 0x3bee, 0x1b1a, 0xb030, 0x0ae1, 0x40bd, 0x41c8, 0xa186, 0xbf13, 0x35f5, 0x4172, 0xb2a5, 0x433d, 0x41de, 0xba70, 0x4194, 0x4157, 0x1c65, 0x2daa, 0xbfd0, 0xbf07, 0x4303, 0x4226, 0xba1d, 0xb0ad, 0xbf4c, 0x40b3, 0xbad6, 0xafc9, 0x13c7, 0x0dfe, 0xbad5, 0x4008, 0x1b04, 0x1317, 0xba12, 0x40a5, 0x43c5, 0xb045, 0x18e6, 0x414c, 0x0165, 0xb23a, 0x1c08, 0x305f, 0x3885, 0x4310, 0x4179, 0x4093, 0xb236, 0x407c, 0xb096, 0x405f, 0xbf4e, 0x1d98, 0x17f0, 0x41d5, 0x404c, 0x466b, 0x41e0, 0x41c7, 0xbfd0, 0x4302, 0xba29, 0x428b, 0x429c, 0xb01c, 0x2dec, 0x415c, 0x43d0, 0xb29e, 0x0b28, 0x2572, 0x41c2, 0xb23e, 0xbfb3, 0x1fcc, 0xb23b, 0xb29b, 0xb253, 0x4133, 0x42f0, 0x4555, 0xb00e, 0x3f9e, 0x4352, 0xba62, 0x458a, 0xbf0a, 0xb21b, 0xb2f8, 0x2bd1, 0x4347, 0xbfa8, 0x4140, 0x09a8, 0x16d2, 0x42ad, 0x40a0, 0xb29b, 0x056b, 0xa54c, 0x4224, 0xb2d8, 0x4360, 0xba52, 0x4606, 0x1ea5, 0xb20d, 0xb26c, 0x273f, 0x4062, 0x409c, 0x454c, 0xbf7f, 0x2edf, 0x4290, 0xb06e, 0x422c, 0x4244, 0x1f4d, 0xbf64, 0x432a, 0x1595, 0x413c, 0xbf2d, 0xb0c2, 0x4133, 0x41f7, 0x412b, 0x4482, 0x4347, 0xae85, 0x434d, 0x41a1, 0x42e2, 0xbae1, 0x42ed, 0xb29b, 0x4330, 0x42ba, 0x40df, 0x41eb, 0x3c9f, 0x43a2, 0x3e6f, 0x41e3, 0x229a, 0x41e4, 0x1d5e, 0xbfb6, 0x402f, 0xb201, 0x4024, 0xb030, 0x195a, 0xba4d, 0x464b, 0x4025, 0x0420, 0x1a5a, 0xb25e, 0x1a8d, 0x4133, 0x3e2c, 0x15ef, 0x45be, 0x1cb5, 0x0ec0, 0x1d99, 0x1419, 0x430b, 0x4009, 0x0d89, 0x0030, 0xb22e, 0xbfb3, 0xa6c3, 0x1c87, 0x2728, 0x2023, 0xbfb8, 0x1ee3, 0x413e, 0x41c4, 0x4384, 0x4433, 0xb04b, 0x28c5, 0xb003, 0xbf49, 0x0673, 0x1a62, 0xb2e5, 0x316f, 0xb2bb, 0xb2fe, 0x2ed3, 0x1c7c, 0x1b45, 0x41a2, 0xb29d, 0x16cb, 0x4267, 0xb284, 0x4235, 0x42d2, 0x1dff, 0x1d0a, 0x1ca3, 0x414b, 0xadb4, 0x43cd, 0x3db7, 0x410a, 0xb082, 0x1d92, 0x0461, 0xa273, 0x29f0, 0xbf77, 0x0edf, 0x1957, 0x1d7e, 0x4676, 0x4085, 0x43aa, 0xbff0, 0xb0f1, 0x4249, 0x4045, 0xb2ab, 0xad44, 0x41b4, 0x4129, 0xba75, 0x1dda, 0xaf01, 0x1aa6, 0x1e4d, 0xb27c, 0xb2eb, 0xaf98, 0x35e7, 0xbf07, 0x2f6f, 0x44f9, 0xb2b3, 0x1ed8, 0x3699, 0xbae5, 0x2d3b, 0x4017, 0x26dc, 0x2d2b, 0xba0e, 0xbf60, 0x1718, 0xbf00, 0x2c1a, 0x411e, 0x1a13, 0xb2b4, 0x219d, 0xbf07, 0x424b, 0x43bf, 0x4252, 0x098e, 0x4302, 0x250d, 0x4064, 0xb017, 0x3af3, 0x4222, 0xadfc, 0x41e5, 0xbad6, 0xa529, 0x4085, 0x41e1, 0xba1c, 0x4279, 0x43ba, 0x03fb, 0x2279, 0xb224, 0xb28e, 0x2e71, 0xb230, 0x3733, 0xb085, 0x41cb, 0xb2a8, 0xbf77, 0x4113, 0xa77d, 0x403b, 0x1d4f, 0x3e00, 0xa3df, 0xb243, 0xb2b5, 0x4396, 0x19ea, 0xba1c, 0xbfbc, 0xb23e, 0x410e, 0xba2e, 0x405f, 0x1df6, 0x1b93, 0x4148, 0x401a, 0x41ec, 0xb06d, 0x4277, 0x0f45, 0xb2c0, 0x42b2, 0x4326, 0x1426, 0x2c0d, 0xbf47, 0xba26, 0xb296, 0x1907, 0x43a4, 0x407c, 0x406f, 0x4332, 0x2146, 0x403c, 0x4173, 0x43c5, 0x1463, 0x3d72, 0xa539, 0x02d3, 0xbf85, 0xb2e9, 0x080e, 0x3996, 0xb0c8, 0x3133, 0x422c, 0xbaf6, 0x1e96, 0x414d, 0xb236, 0x4071, 0x1b45, 0x41a3, 0x1b5a, 0x04f2, 0xaaed, 0x43d5, 0x40c1, 0x43ee, 0x4631, 0xb0c0, 0x4616, 0x432f, 0x06c9, 0x1dca, 0xbfe1, 0x2a04, 0x106d, 0x40fd, 0x4417, 0xb245, 0x40c4, 0x4279, 0x1ab5, 0x424d, 0x1c8c, 0x2adf, 0xb215, 0x41fb, 0x1014, 0x41bc, 0x43a2, 0x428a, 0xba77, 0x2e89, 0xba5f, 0xb285, 0xb26f, 0x4071, 0x3346, 0xb22b, 0x4350, 0xbf01, 0x403f, 0x42ef, 0x2d94, 0xba29, 0x421b, 0xbf5c, 0x4164, 0x4616, 0x2ad2, 0xb22d, 0x1fea, 0xbf70, 0x42be, 0x409e, 0x42ef, 0x42fa, 0xb28e, 0xba3b, 0xba70, 0xb25a, 0x07ec, 0x405c, 0x40bf, 0x404e, 0x1bc1, 0xba1a, 0xba63, 0xb2e8, 0x3845, 0x4072, 0x1cc5, 0xbf6f, 0x1fd3, 0x1fd9, 0xb2f7, 0x4077, 0x423b, 0x402d, 0x1abd, 0xbf99, 0x4192, 0xbaf0, 0x3122, 0x2e08, 0x4029, 0xbafe, 0xb263, 0x42e5, 0x38e5, 0x43b0, 0xa37a, 0x3088, 0x421d, 0x2339, 0x1d50, 0xb232, 0x2ba9, 0x0059, 0x442f, 0x1e1a, 0x403d, 0x4223, 0x406b, 0xb005, 0xb2e7, 0xbf21, 0x4391, 0xbad2, 0x40bb, 0x04c5, 0x1e4a, 0x424a, 0x415e, 0x0467, 0xbad2, 0x40c5, 0xa1c5, 0x45d3, 0x43d3, 0xb24f, 0x3261, 0xb05b, 0x4364, 0x420b, 0x19f9, 0x2464, 0xbfd7, 0x240f, 0x411a, 0x0a1e, 0xbaf0, 0xb0bb, 0x40fa, 0xb232, 0x4480, 0x401a, 0x40ea, 0xaa93, 0xbff0, 0x0dd7, 0x43ce, 0x4279, 0xba2c, 0xb20b, 0x3d38, 0xbae1, 0xa965, 0x4201, 0x425a, 0x4217, 0x4469, 0x03f7, 0xbf3e, 0xb0d5, 0xb297, 0x42c4, 0x42b9, 0x42d7, 0x417f, 0x194b, 0x3e69, 0x409f, 0xbf60, 0x4219, 0xa318, 0x3a6a, 0xbf29, 0xb2a3, 0x4649, 0x467b, 0x1528, 0xb0b6, 0x40d0, 0xa445, 0xa6a6, 0x4153, 0x15db, 0x0d2b, 0xa059, 0x3ca7, 0x0813, 0x43fe, 0x199d, 0xbfd4, 0x4138, 0xb00a, 0x1936, 0x4111, 0xa06a, 0x43f8, 0x4041, 0xb2f4, 0x4293, 0xba58, 0x437f, 0x155d, 0x0878, 0xbad2, 0xb0cb, 0x1c3e, 0xbf19, 0x427a, 0x1a09, 0x17be, 0x1742, 0x1f52, 0xb282, 0x404e, 0xb263, 0xba0d, 0x425c, 0xb095, 0x41cd, 0x1d19, 0x0352, 0x1841, 0x26ff, 0xbf7c, 0xbadf, 0x1407, 0x42a3, 0x4025, 0x4330, 0xbaf6, 0xbf91, 0xa711, 0x4002, 0xa488, 0x1a65, 0x4052, 0x19a3, 0x1819, 0x43e7, 0xb239, 0x40a4, 0x31d2, 0xa11f, 0x42e6, 0x43d7, 0x4110, 0x4012, 0x2c2f, 0x43b5, 0x45a1, 0xb2dc, 0x12f9, 0x268b, 0x1993, 0x4228, 0xb20b, 0x419b, 0xb27b, 0x42bb, 0x46a2, 0xbfa9, 0x24ed, 0x41a1, 0x0d0a, 0x1559, 0x4114, 0x42f0, 0xa900, 0x4413, 0x4008, 0xa9f6, 0x05d6, 0xb2a4, 0x42e2, 0x3bdb, 0x435c, 0x1d9f, 0x183f, 0x4063, 0x0c6b, 0x30e3, 0xbf7c, 0x430f, 0x1169, 0xb20a, 0x1ce1, 0x3691, 0xa65e, 0xb20f, 0x40d9, 0x28c6, 0xba56, 0x403d, 0x43e8, 0x1e4e, 0xa2a3, 0xb21f, 0xa649, 0x4428, 0xa8cf, 0x4068, 0xb0f4, 0xb2b0, 0xbfcb, 0xb275, 0x1a56, 0x409b, 0x1ed9, 0x3908, 0x41ca, 0xb0d0, 0x1a06, 0x4230, 0x1f36, 0xbf90, 0xba26, 0x4020, 0xbf78, 0x42fe, 0x46b9, 0x3e8c, 0xa5e7, 0x4272, 0x0d53, 0x4159, 0x41b4, 0x4484, 0xbfc0, 0x4281, 0x414a, 0xad25, 0x195b, 0x43b8, 0x1f77, 0xbf67, 0x404e, 0xba33, 0xa665, 0x45ba, 0xb0ec, 0x41dd, 0x3f03, 0x343b, 0x43a5, 0xa8f0, 0x0ac1, 0x46ab, 0x4223, 0xb060, 0x4240, 0x41fc, 0x0d95, 0x1df4, 0x4299, 0xbff0, 0x0e19, 0xbaf4, 0x4137, 0x40e0, 0x4317, 0x3f08, 0xbf49, 0x1ad3, 0x1977, 0x1e1b, 0x434f, 0xb2ab, 0x22ae, 0xbfcc, 0x4435, 0x0e38, 0x414b, 0x2b42, 0x400f, 0x36e1, 0x41dd, 0xbade, 0x4190, 0xbf6f, 0xa805, 0x4073, 0x41bd, 0x05d3, 0x41b6, 0x1a96, 0x4075, 0x35a1, 0x3336, 0xa35c, 0x4332, 0x431f, 0x26bc, 0xb228, 0x0f68, 0x41ff, 0x438a, 0xb227, 0xbac0, 0xba78, 0xb073, 0x4083, 0x1d03, 0xbf41, 0x4159, 0x2f5c, 0x4081, 0x4337, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3a4c2871, 0xacd9f7e9, 0x7bcd3cce, 0xadebfcc6, 0xde673e7c, 0x7dd2cf41, 0x7006af38, 0x2bb0bab5, 0xe2b978ed, 0x142f0bcb, 0x8bc78472, 0x492faf1e, 0x7421150a, 0x1ef52200, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0xffff18fc, 0x00000000, 0x000000a6, 0xffff1900, 0xfffffc18, 0x00063f47, 0x000000bc, 0xfffffcbc, 0xe2b978f1, 0x00000000, 0x000000b8, 0xaca0a530, 0x7421150a, 0x59414be9, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xb25e, 0x4201, 0xb0f6, 0x437b, 0xbf28, 0x1a61, 0x41f3, 0xb2af, 0xba52, 0xb0be, 0x0ee9, 0x18cb, 0x2ddc, 0x3e47, 0x4216, 0xb2e8, 0x40c7, 0x33d7, 0x42a9, 0x412a, 0x4322, 0x4313, 0x42b0, 0xbfbc, 0x42d1, 0x4309, 0xb2bc, 0xa626, 0xbfaf, 0x41e6, 0x4026, 0x4028, 0xa235, 0x40c3, 0xb030, 0xbfa0, 0x4331, 0xacd4, 0x2a6c, 0x4058, 0x353b, 0x42eb, 0x199e, 0x1a78, 0x1e98, 0x1222, 0x4677, 0x180d, 0x1b21, 0x36a8, 0x1d5e, 0xba01, 0x42b0, 0x4110, 0x416f, 0x4007, 0x1bb0, 0xab34, 0xbf31, 0x40f0, 0x46d4, 0x422e, 0xb2db, 0xa831, 0x2443, 0x40a8, 0xb002, 0xba2f, 0x41df, 0x4363, 0xa95d, 0xba57, 0xb2bf, 0x43b9, 0x427a, 0x4550, 0x1af8, 0xbad9, 0xb2cf, 0xbfb0, 0x4300, 0x4301, 0xbf4f, 0x42f3, 0xb2c5, 0xba77, 0x4157, 0xb2cc, 0x12ef, 0x187c, 0x46d9, 0xa1ec, 0x1c3f, 0xa4c0, 0x34a8, 0xa8ec, 0xb25d, 0x4253, 0x40d1, 0x4122, 0xbafd, 0x42fe, 0x19fb, 0x4566, 0xba75, 0x3820, 0x4115, 0x412e, 0x1f3f, 0xbf8d, 0x10fe, 0x4231, 0x40d3, 0xb054, 0x1750, 0xb07a, 0x1be5, 0x41a9, 0xb227, 0x431e, 0x41a3, 0xbae7, 0x43c0, 0x2338, 0x41c9, 0x4026, 0x4368, 0x4371, 0x42ad, 0xb2be, 0x1d31, 0x1cfb, 0x41f2, 0x3242, 0xbf33, 0x3a13, 0xbae1, 0xb03a, 0x4698, 0x4130, 0x44e0, 0xb238, 0x417b, 0x40f2, 0xbfbe, 0x4233, 0x4115, 0x45a8, 0x41cb, 0x46c3, 0x30b0, 0x42df, 0xa26e, 0xb2b7, 0x4205, 0xbf5b, 0xba12, 0xa0ae, 0x4247, 0x46e0, 0xb2c5, 0x41ed, 0x4068, 0x41e9, 0x19fd, 0x435a, 0x41e7, 0xb2f1, 0xb2b0, 0x41a1, 0x43c4, 0x45dc, 0x4387, 0x40b8, 0x2a85, 0x19fc, 0x4360, 0x4545, 0x1930, 0x41b3, 0x24ec, 0xbfa0, 0x42bc, 0x403f, 0xbf7d, 0xaf68, 0xb2d1, 0x4238, 0xb2a9, 0x4346, 0xb09b, 0xa482, 0x41d9, 0xba36, 0x4108, 0xb281, 0xba17, 0x28a4, 0x16f0, 0x10b0, 0x22f2, 0x467d, 0xac2a, 0x133e, 0x4195, 0x0338, 0x41dc, 0xbf03, 0x41c7, 0x4375, 0x41db, 0x4608, 0x45a0, 0x401a, 0x1b82, 0x42ca, 0xb01b, 0x206b, 0x4610, 0x1db7, 0x44dd, 0xa4e6, 0x0c75, 0x4064, 0x41ec, 0x27c6, 0x4371, 0x42f1, 0x418f, 0xb2d6, 0x237a, 0x209a, 0x364f, 0x4183, 0x4252, 0xbac8, 0xbf48, 0x32a8, 0x1805, 0x42d8, 0x1fab, 0x18b3, 0x41fa, 0x04ab, 0xbf80, 0x19b6, 0x1a4d, 0x41c2, 0x1b8f, 0xb23a, 0x40a9, 0xbaff, 0x40ec, 0x4313, 0x40d2, 0xb09a, 0xb218, 0x42da, 0x1f49, 0x41ab, 0xb22b, 0xbf58, 0xba62, 0x4020, 0x1f31, 0xb293, 0x4136, 0x24c0, 0x088e, 0x4252, 0x2136, 0x3252, 0x2efa, 0xa1bf, 0x04ad, 0x1c79, 0xb01c, 0x43f7, 0x42e6, 0x426d, 0x4364, 0x4236, 0xb298, 0xbf24, 0x26e7, 0xb27d, 0xae82, 0x414b, 0xa71c, 0x44f0, 0x4395, 0x421c, 0xb29b, 0x409f, 0xbfd9, 0xb206, 0x43c8, 0x19f8, 0xb24d, 0x4265, 0x419f, 0x414d, 0x429b, 0x45c5, 0x435a, 0x11d9, 0xb259, 0x4188, 0xba06, 0x204d, 0x1b38, 0x4296, 0xa47f, 0xbf03, 0x45ad, 0x167f, 0x23eb, 0xb237, 0xb289, 0x1c1c, 0x43ef, 0x42ad, 0xa496, 0x438e, 0xb0a6, 0x3afb, 0xb229, 0x2661, 0x4159, 0x4384, 0x3b07, 0x45b8, 0x09ae, 0x3811, 0xbfa1, 0x3db5, 0xb217, 0x2dff, 0x43a2, 0xb28a, 0xbfa3, 0xba3f, 0xb2ec, 0xb2e0, 0xb067, 0x0a10, 0xb20a, 0xa2ad, 0x446b, 0xb27d, 0x05f3, 0x4248, 0xb02b, 0x43be, 0xba20, 0xbf90, 0x2133, 0x0c37, 0x254d, 0x4144, 0xb213, 0x403f, 0x1356, 0xa35e, 0x4219, 0x438d, 0xb211, 0x42ae, 0xbfbd, 0x1e52, 0x3cf3, 0x1d3c, 0xb287, 0x0781, 0x405e, 0x41a9, 0x436e, 0x1ebf, 0xbf01, 0x0041, 0x29a5, 0x422e, 0xba5a, 0xb0f3, 0x4355, 0x41b5, 0x421c, 0x4191, 0x4276, 0x43cb, 0xbae3, 0x42d1, 0xba32, 0xbfc0, 0xb232, 0xbf6a, 0x4013, 0x4256, 0xba73, 0x40a9, 0xba05, 0x1de9, 0xbf04, 0x4170, 0x42da, 0x40a6, 0x13a4, 0x4084, 0x42da, 0xbfa0, 0x4081, 0xbfd9, 0x220b, 0xb26f, 0x41ee, 0xba5b, 0xba52, 0x43e1, 0xb2b4, 0x1ffd, 0xbf6b, 0xb278, 0x4690, 0x4370, 0x3669, 0xb21e, 0xb017, 0x375d, 0x4131, 0x407e, 0x40e7, 0x4638, 0x0d44, 0x4254, 0x3476, 0x41ec, 0x42fd, 0x1ccd, 0xb295, 0xbfd0, 0x401c, 0x3235, 0x4645, 0x155b, 0x0311, 0x41db, 0x40c0, 0x396d, 0x0ba6, 0xbfaf, 0x0e6d, 0x43fd, 0xa6bc, 0x401a, 0xbafc, 0x421e, 0xbf18, 0x1886, 0x4106, 0xbf27, 0x12cb, 0x43c8, 0xba28, 0x4011, 0xba7f, 0x1103, 0xbac5, 0x193b, 0x466d, 0x42c5, 0x184a, 0x0d38, 0xbf62, 0x3f92, 0x43e9, 0xaa66, 0xb093, 0x334d, 0xba06, 0x1ebb, 0xb2ff, 0x465e, 0xb24e, 0xbade, 0xbfa3, 0x2abf, 0x3186, 0x430a, 0x448a, 0x434f, 0xb291, 0x4215, 0x1952, 0xb003, 0x41ec, 0x44ed, 0x438f, 0x42f3, 0x424c, 0x4025, 0xb0d7, 0x18f0, 0xb25d, 0x416e, 0x4120, 0x45a9, 0xb0f9, 0xb2fd, 0xbaed, 0xba7d, 0xbfd3, 0x41ec, 0xb22b, 0x428a, 0x42e5, 0xb02c, 0x1b7c, 0x37e0, 0x430a, 0x190a, 0x25db, 0x2eff, 0xb276, 0x0b11, 0xbfe2, 0xab47, 0x251c, 0xb223, 0x31c7, 0xbaf2, 0xb001, 0xb212, 0x30f1, 0xa44c, 0x41ae, 0x418d, 0x40a4, 0xb0b6, 0x4331, 0x425a, 0x4040, 0x1849, 0xb03a, 0x1959, 0xb092, 0x429d, 0x4188, 0xbf12, 0x4619, 0xba68, 0xbf60, 0xbf80, 0x42b8, 0x0a84, 0xb2fe, 0x456c, 0xb0f3, 0x4194, 0xb272, 0xb0d7, 0xb2de, 0x4353, 0x0e33, 0x1c0a, 0xbac8, 0x42c6, 0x19ff, 0x43af, 0x4089, 0xbfb8, 0x3b9b, 0x096d, 0x31ac, 0x0234, 0x40cf, 0x24b8, 0x4278, 0x417a, 0x4309, 0x427e, 0xb070, 0x42cf, 0x40a9, 0x433b, 0x418d, 0xb2be, 0x1969, 0x429e, 0x43af, 0xbf1f, 0x0b1a, 0x4286, 0x41ce, 0x426d, 0x4136, 0x1ec7, 0x3425, 0xa663, 0x4457, 0xb2a4, 0x43b1, 0x41e0, 0xb2cc, 0x26cc, 0xbad6, 0xbacc, 0xb2e0, 0x4018, 0x3a50, 0xb2bf, 0xba3b, 0xbadb, 0xb0c4, 0xbf09, 0x1167, 0x4099, 0x044b, 0xba75, 0x43d6, 0xba3a, 0x43f5, 0x2c1c, 0xba5a, 0x43bc, 0x42a9, 0x1c95, 0xad4c, 0x43e4, 0x46cc, 0x42cc, 0x2605, 0x4009, 0x1871, 0x4069, 0xba37, 0x411d, 0x1807, 0x426d, 0x423b, 0x0fab, 0x4159, 0xbf83, 0x4290, 0x1d71, 0x3601, 0xa5d0, 0xb23a, 0x1cfb, 0x42fc, 0x435f, 0x4003, 0xa2b5, 0x4163, 0x4185, 0x403c, 0xb0d0, 0x1ae1, 0xb271, 0xafbc, 0x40cb, 0x4174, 0xbf69, 0xb258, 0xb2ec, 0x42c0, 0x41dd, 0x197d, 0xbf14, 0xa735, 0x38c4, 0x4188, 0xbfb0, 0xbf0d, 0x4117, 0x41ae, 0xb0e8, 0x411b, 0x4155, 0x43bb, 0xba44, 0x4198, 0xb2cd, 0xbfc7, 0xbae2, 0xb2c4, 0x43d7, 0x1e4d, 0x34ca, 0xb223, 0x439c, 0xb294, 0xbae1, 0x1d72, 0xbfe0, 0xb2ef, 0x4363, 0xbfb9, 0x413e, 0x408b, 0xa5be, 0x2675, 0x42d3, 0xb204, 0x1ff1, 0x096f, 0x41dd, 0x3c1e, 0xb2bd, 0x40c6, 0x2575, 0xb2b1, 0x1910, 0x4607, 0x42b8, 0x43dd, 0x4329, 0x4112, 0x1ea0, 0x1b9b, 0xa27c, 0xbadf, 0xbf81, 0x4088, 0x41a7, 0xada0, 0x0db4, 0xb2fe, 0x408a, 0x18da, 0xb026, 0xb0b3, 0x43f1, 0x4467, 0xbfaf, 0x4367, 0xbad4, 0x405f, 0x4286, 0x1a0d, 0xb2b7, 0x4571, 0x183e, 0x43cf, 0x1a0d, 0x4036, 0x424f, 0x41a4, 0x410f, 0x46a2, 0xaf85, 0xa17c, 0x433b, 0xbf0e, 0x4022, 0x43f0, 0x11a4, 0x40c4, 0xba2c, 0x18a6, 0x1a93, 0x41d1, 0xa8a3, 0xbaf8, 0x4584, 0xbff0, 0xb23a, 0x4385, 0x43d8, 0xb29b, 0x4362, 0xb20f, 0xb0a7, 0xad67, 0xb252, 0xbf8a, 0x1cb1, 0xb215, 0x41a4, 0x40b3, 0x2a7b, 0x0d23, 0x4199, 0x18ce, 0x4618, 0x144a, 0x3082, 0x1c3e, 0xbf57, 0x32de, 0x412d, 0xb298, 0x4648, 0x1c4d, 0x1df9, 0x3e54, 0xbf8e, 0x4488, 0x4092, 0xb200, 0xba13, 0x41a1, 0xba23, 0x1d7d, 0xba09, 0x2b1c, 0x4691, 0x397f, 0x4032, 0xbf0e, 0xba67, 0x40e5, 0xb2f3, 0xbff0, 0xba04, 0x3ec6, 0x1a0f, 0x408b, 0xb2e2, 0xbfe2, 0x4224, 0x195a, 0x3204, 0x430d, 0x39cb, 0x1c36, 0xb23f, 0x1977, 0xb030, 0x46f4, 0xb2fe, 0xba77, 0x4141, 0xb276, 0x4217, 0xbf60, 0xb223, 0xb2ca, 0x43bd, 0x4204, 0x41d0, 0xa049, 0x4352, 0xb035, 0xbf05, 0x4396, 0x429b, 0xa066, 0x0975, 0x409b, 0x46bd, 0x2aa5, 0xb2b9, 0xba09, 0xb05e, 0x422d, 0x1d65, 0x42c5, 0x42d4, 0x0533, 0x221b, 0x42c2, 0xa9dd, 0x2ac2, 0xad89, 0x4263, 0x4378, 0x40ff, 0x429b, 0x4282, 0x421f, 0xb0c4, 0xbf56, 0xb0c4, 0xb070, 0xbac9, 0x43ff, 0x1bd4, 0x418f, 0x09b8, 0xb0a8, 0xbfc7, 0x455a, 0x04bf, 0xba69, 0x4283, 0x404a, 0x43a2, 0x449b, 0x1e46, 0x4130, 0x4305, 0xb272, 0x1203, 0x43f4, 0x437e, 0xb066, 0x4273, 0x0f4c, 0x4612, 0xaa6e, 0x08d1, 0x437c, 0x461d, 0x44ac, 0xa824, 0xb2cd, 0xb01f, 0xbf53, 0xbfa0, 0xb283, 0x4151, 0xb284, 0x41f4, 0x406e, 0xb038, 0x4330, 0x420d, 0x44b5, 0xb2b2, 0xa730, 0xb2e0, 0x2b11, 0x4698, 0x4142, 0xb2f9, 0x42ec, 0xb08e, 0x3c8b, 0x42bc, 0xb22e, 0x4226, 0x400b, 0xb219, 0xbf25, 0x43c3, 0xbf00, 0x4345, 0x40aa, 0xb0a0, 0x1a70, 0x429e, 0x4074, 0x41af, 0xb272, 0xb0af, 0xb0f4, 0x42ff, 0xba34, 0x4342, 0x399a, 0x46cd, 0x19e4, 0x3c64, 0xbfcf, 0x40e9, 0x40c2, 0x3ac5, 0x323c, 0x307e, 0xba37, 0xba17, 0xb2f7, 0x207f, 0x431e, 0xac1e, 0xb0c8, 0x41b1, 0x436b, 0x40d1, 0xb02b, 0xbfe8, 0x4335, 0x41ba, 0x1cad, 0xa482, 0xbacd, 0xb285, 0x401c, 0xa85b, 0x4199, 0x4392, 0xbfbb, 0xb045, 0x0935, 0x30e0, 0x4226, 0xb22e, 0x282a, 0xb2e5, 0x40c8, 0x4326, 0x2bf9, 0x4197, 0xbf16, 0x4371, 0xb2ea, 0x40c4, 0x3f3c, 0xb285, 0x4170, 0x40ce, 0x2706, 0x308f, 0xbfbe, 0x4113, 0x0553, 0xbad5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x9b4ed9e3, 0xfb4d9b4f, 0xf2becf95, 0xee40ea2c, 0x9b566ffa, 0x1e8d47a2, 0xeeda6eed, 0x8e4fe94e, 0x87b89292, 0x5bc99f48, 0xcdbc076e, 0x49f2a792, 0x65012e66, 0x2d75e18d, 0x00000000, 0xa00001f0 },
                FinalRegs = new uint[] { 0x40001b7a, 0x000019ff, 0x00000000, 0xffffffff, 0x00000000, 0x000000ec, 0x00000000, 0x00000006, 0xef680aac, 0x80000000, 0xffffffff, 0xc2cc9a01, 0xef680aac, 0x7fffff8c, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xba64, 0xba1a, 0x40a5, 0xba40, 0x1307, 0xb2a5, 0xb24d, 0x4102, 0xbfc0, 0xa0b9, 0xb20f, 0x181b, 0x439c, 0xa7dd, 0x1c89, 0xb011, 0x43c1, 0x1c4a, 0x418e, 0x1924, 0xa84d, 0xb235, 0x1f2a, 0x422d, 0xbf81, 0xb23e, 0x4057, 0x4036, 0xb2a5, 0xbf48, 0xb013, 0x42cf, 0x432a, 0xbfb6, 0x437b, 0x436d, 0x418a, 0x41cb, 0x2910, 0x1797, 0x424a, 0xa593, 0xb0aa, 0x429d, 0x40e6, 0x45c0, 0xb201, 0x4317, 0x4349, 0xb0a8, 0x4148, 0x19d4, 0x4255, 0x412a, 0xa20f, 0x1e2b, 0xbf1d, 0x403e, 0x42b4, 0x1a38, 0x4476, 0x436a, 0xab06, 0xb0e1, 0xbf82, 0x410d, 0xb2d0, 0x401a, 0x4356, 0xba0e, 0x40cc, 0x43e6, 0xba47, 0x408c, 0x43ae, 0xbf4a, 0x42cd, 0x1b95, 0x4046, 0x4299, 0x4380, 0x27bd, 0x431c, 0x4129, 0xbacc, 0x18be, 0x4380, 0xbf7c, 0x41ec, 0x19d7, 0xbf22, 0x40e2, 0x24c3, 0xb294, 0x4430, 0xb27b, 0x4648, 0x4360, 0xb24b, 0xb003, 0x4051, 0x42e7, 0x43d3, 0x41d4, 0x1bf1, 0xb035, 0xb2d4, 0xb2e9, 0x40e4, 0x435d, 0x3083, 0x464b, 0x41dd, 0x1b19, 0xb27e, 0x4224, 0x2dc0, 0x4181, 0x1624, 0xbf7d, 0xbae9, 0x4317, 0xabcc, 0x43d7, 0x438e, 0x4254, 0xb287, 0x40d0, 0xbf4e, 0xbaf3, 0x43f9, 0x4350, 0xa956, 0x1fc1, 0x4374, 0x44e5, 0xa792, 0xb2f3, 0x412b, 0xb201, 0xb241, 0xb2d3, 0xbfc0, 0xba76, 0x4246, 0xb019, 0x40e3, 0x198a, 0x2b9a, 0x3b90, 0x46d9, 0x072a, 0xba25, 0x3d43, 0x4097, 0x41e1, 0xbfc6, 0x4205, 0x04d8, 0x40f9, 0x1cd1, 0x1f55, 0x1f1e, 0xaa31, 0xb23e, 0x4598, 0x13f6, 0x4179, 0xaf2e, 0xbfae, 0x4357, 0x1374, 0x1ad4, 0xad33, 0x4156, 0x427e, 0xbfe4, 0x44e0, 0x4331, 0x43f4, 0x42c9, 0xbf85, 0x1a24, 0xa2f6, 0x41f2, 0xa107, 0x4232, 0xbf90, 0xa2a0, 0xb23e, 0x1f9e, 0x4187, 0x1bb0, 0xb26d, 0x426f, 0xb223, 0x4088, 0xacf5, 0x405e, 0x309e, 0x41ef, 0x22b8, 0xbfaa, 0x41bd, 0xbac2, 0x1c97, 0x4673, 0xb257, 0x41bf, 0x4055, 0xb287, 0x1bef, 0x44d5, 0x1b5a, 0xb06d, 0x42f0, 0x3c79, 0x420f, 0x1cc2, 0xbf2f, 0x44ca, 0x40ec, 0x41e5, 0xba3c, 0x4118, 0xba6d, 0xa231, 0xb266, 0xae71, 0xba56, 0xa854, 0x21c1, 0x39da, 0x18f7, 0x162a, 0x0af5, 0x4246, 0x43c3, 0x1839, 0xb0f9, 0x40d0, 0x1f0f, 0x46ac, 0x186e, 0x44a1, 0x40aa, 0xbfd1, 0x404c, 0x2780, 0x43f3, 0x4391, 0xbfc0, 0x439d, 0x0446, 0xba07, 0x411e, 0x4358, 0x40a1, 0x42eb, 0x408f, 0xa392, 0x25c7, 0x4289, 0xb203, 0x4220, 0xbfe0, 0xba43, 0xa384, 0x10d1, 0xbf4c, 0xba61, 0xbf90, 0x2fe4, 0x4339, 0x419e, 0xa2e4, 0xbf90, 0xa1b8, 0xb2ca, 0x431e, 0xa96f, 0xbaeb, 0x0eab, 0xba3f, 0x42da, 0x4092, 0xba50, 0x419f, 0x42ae, 0x428f, 0xbf7e, 0x43b6, 0x3644, 0xb0e6, 0xa3fd, 0x46c8, 0x4049, 0xb22e, 0x434c, 0x1dec, 0xb233, 0xb2de, 0x42f1, 0xb2d6, 0xb2ed, 0x3db5, 0x4326, 0x4279, 0x22e0, 0x2801, 0x42ac, 0x4331, 0x4192, 0x2136, 0xbf80, 0x4358, 0xbf48, 0xb09e, 0x0395, 0xb232, 0x41d8, 0xbfb0, 0x18fd, 0xba3e, 0x1da0, 0x45d5, 0x43a3, 0x41c0, 0x4028, 0x446a, 0x00dc, 0xbfa8, 0x2ac3, 0x292e, 0x439b, 0x421c, 0x1924, 0xb25b, 0x464b, 0x427e, 0x40a9, 0xbae4, 0x42ed, 0xb0be, 0x4650, 0x4205, 0x4062, 0x400c, 0xb283, 0x40a1, 0xb26e, 0x4438, 0x4210, 0xaf94, 0xb270, 0x18ad, 0xbf3c, 0x3d88, 0x41c3, 0x42eb, 0xbf51, 0x40f4, 0xba01, 0x4119, 0x42b6, 0x13e7, 0x2c7e, 0xb248, 0x3088, 0x2d04, 0xb22c, 0x45ce, 0x006d, 0x446f, 0x1a24, 0x4059, 0x40f6, 0x4360, 0xa82e, 0xb240, 0x42e0, 0xba21, 0x2130, 0x42b4, 0xb033, 0xbf04, 0x4020, 0x4093, 0x409d, 0x0fb2, 0x455f, 0x40d7, 0x168b, 0xba34, 0x2d4c, 0x426e, 0xba7b, 0x1b47, 0xb0a4, 0x4252, 0x38dd, 0x3e12, 0xb034, 0x429f, 0x4195, 0xbfd9, 0xb2f8, 0x427a, 0x3ecb, 0xaba8, 0xb27d, 0x44a4, 0xb2d5, 0x42cc, 0x43f0, 0x4369, 0x4281, 0x2b77, 0xbaf3, 0x1827, 0x3984, 0xb008, 0xba71, 0xbaeb, 0xbf68, 0xba04, 0xbf48, 0x40da, 0xb025, 0x324b, 0x41b9, 0xb0a9, 0xb02e, 0x45dc, 0x2fa9, 0x435c, 0x1b90, 0xbf7a, 0x41da, 0x1990, 0xba5a, 0xb206, 0x40c1, 0x4188, 0x4063, 0x4182, 0xb2d3, 0xba04, 0xbad2, 0x1f08, 0x031c, 0x179d, 0x1dcc, 0x1ed3, 0xa42e, 0x416d, 0x2dae, 0x05e5, 0x3b23, 0x09b7, 0x40c8, 0x1aa2, 0xb247, 0x1462, 0x40d1, 0x4049, 0xbf77, 0xb277, 0x1eee, 0xb298, 0xbf00, 0x42bb, 0x1f09, 0x423a, 0xba7f, 0x4217, 0xbaf5, 0xba2f, 0x3e9d, 0xbf3e, 0x40c8, 0x2bf6, 0xba24, 0xba43, 0x31ec, 0x2cae, 0x4211, 0xbf21, 0x1433, 0x467f, 0x1eab, 0xb2f5, 0x437e, 0x4143, 0xba44, 0xafe7, 0x4173, 0x418d, 0xad21, 0x0b5f, 0xa2f0, 0xba57, 0xbfaf, 0xb2ef, 0x3b63, 0x4155, 0xba08, 0xb20c, 0x4565, 0x407e, 0x4006, 0x27ce, 0xba15, 0xb2f7, 0x403e, 0xbafe, 0x4385, 0xb263, 0x4092, 0x43ba, 0xb2a9, 0x43bd, 0x424f, 0x4101, 0x45e3, 0x3773, 0xbfba, 0x1dbc, 0xb2ea, 0xb242, 0x40c7, 0xb057, 0xb2ee, 0x41d9, 0xb20d, 0x4052, 0x4391, 0xa641, 0xa730, 0xb20d, 0x3c10, 0x4477, 0x3eab, 0xa994, 0x2ce4, 0xbaf4, 0xbf62, 0xb252, 0xb2d1, 0x14c9, 0xbfc2, 0x4064, 0x4185, 0x1baf, 0x4298, 0xb2ab, 0xa44e, 0x40ea, 0xa273, 0x439d, 0x44d5, 0xba28, 0x41b2, 0x0318, 0x1e74, 0x4310, 0xbf0d, 0x1b3e, 0xba2c, 0x3286, 0x412d, 0x199a, 0xa2d6, 0x41de, 0xbfb0, 0x43e6, 0xa815, 0x162b, 0x16a3, 0x429e, 0xb210, 0x130d, 0xba4b, 0xba20, 0xbfbd, 0x2828, 0x00c1, 0x425e, 0x0047, 0x4324, 0x41af, 0x2141, 0x42c6, 0x42a2, 0xad20, 0x163f, 0x4385, 0x288d, 0xb0ad, 0x40a2, 0xba7a, 0xba4b, 0xb0ba, 0x4044, 0x0501, 0x33f5, 0x1845, 0xa3c5, 0xa7d3, 0x4274, 0xb270, 0xbafc, 0x4485, 0xbf35, 0x421e, 0x41a2, 0x4250, 0x421b, 0x04da, 0xb222, 0x1cd3, 0x0958, 0x4085, 0xa75d, 0x1d39, 0x059a, 0x43cb, 0xba2c, 0x40e8, 0x41e4, 0x4664, 0x40d9, 0xbf18, 0x43f1, 0x4350, 0x4267, 0x4027, 0xbf21, 0x1afb, 0x1264, 0xb258, 0x4575, 0x4124, 0xb215, 0x466d, 0x1870, 0x4319, 0xaa64, 0xbfc1, 0x4376, 0xb2fb, 0x437b, 0x438f, 0xb011, 0x2417, 0xb206, 0xbf00, 0xbaf8, 0xbfc5, 0xba1e, 0xa11d, 0xb27d, 0xadf2, 0x1b0f, 0x40c8, 0xbf05, 0x46d4, 0x43b0, 0x43e4, 0x4008, 0x426d, 0x1f5a, 0x423b, 0x1b37, 0x1df6, 0x2bde, 0xbadc, 0x1ff8, 0xbf3b, 0x43a8, 0x41de, 0x4239, 0x1def, 0x4184, 0x43ee, 0x431b, 0x41f0, 0x43f5, 0x4290, 0x422c, 0x3066, 0x19fd, 0xaeba, 0x409f, 0x1f41, 0x2c3e, 0x1f68, 0x40c7, 0x4146, 0x2ff8, 0xb2f6, 0xb026, 0xba41, 0xb220, 0x443f, 0x205a, 0xb2e6, 0x44cb, 0xbf3b, 0xbf90, 0xb2de, 0x0b90, 0x0e8f, 0xba26, 0x06c0, 0xbaed, 0x0acd, 0x408e, 0x31fb, 0x048f, 0x4577, 0x40db, 0xbfd5, 0x4181, 0xaa2b, 0x16a1, 0x401d, 0xbf19, 0x2bed, 0x4053, 0x3875, 0xb265, 0xad06, 0x4044, 0x11f9, 0xb289, 0x4602, 0xb0da, 0x461b, 0x40a1, 0xba3d, 0x13fd, 0x41a4, 0x41b3, 0x41eb, 0xb000, 0x4385, 0x46ec, 0x2780, 0x45be, 0x33bd, 0xbf34, 0x1963, 0x4266, 0x422e, 0x1c54, 0x1b2a, 0xa37d, 0xba7c, 0xba54, 0x4221, 0xba78, 0x1ef2, 0x425a, 0x40c8, 0x24fa, 0xbfcb, 0xb237, 0x420a, 0xb0bb, 0x1eeb, 0x40c3, 0x0071, 0x0ce2, 0xbf2e, 0x4297, 0x4227, 0x432c, 0xb01b, 0xae0a, 0xa328, 0x09dc, 0xb2cb, 0x4014, 0x1897, 0xbf54, 0xb07e, 0x234a, 0x284d, 0x44b1, 0x46fb, 0xb26b, 0x0044, 0xaf0c, 0xa318, 0x43b4, 0xa43d, 0xb2b1, 0x4082, 0x42ab, 0x108f, 0x41d8, 0xba0c, 0x4318, 0xbf12, 0x31db, 0x4296, 0x4376, 0x466e, 0x1702, 0xb08c, 0x1446, 0xb2c3, 0xb262, 0xb06b, 0x33ac, 0xb285, 0x1c39, 0x0554, 0xb254, 0x4010, 0x0807, 0x406d, 0x3fd4, 0xb2f0, 0x41d8, 0xb03d, 0xbf31, 0x0072, 0x4300, 0xabf2, 0x3778, 0x41ce, 0x425d, 0xa931, 0xb088, 0x104d, 0x1902, 0xb237, 0x1adb, 0x40fb, 0x4310, 0x03b3, 0x409c, 0xb0af, 0xb2e6, 0x06ba, 0x439d, 0x41c4, 0xbfc5, 0x4290, 0xb086, 0xb274, 0x1ff7, 0xbfca, 0x179a, 0x24c7, 0x407e, 0xba61, 0xbf1f, 0xb281, 0x41b6, 0x2aec, 0x41f5, 0x40fe, 0x27f0, 0x03f3, 0x1a45, 0x2020, 0x0c37, 0x18c6, 0x4420, 0xb225, 0x42fd, 0xbf87, 0x429e, 0x3085, 0x4240, 0x18df, 0x3b5f, 0x3e59, 0x42ba, 0x4369, 0x093c, 0x43df, 0xb2a3, 0x1f17, 0x427c, 0x1c1f, 0x17ae, 0x188a, 0x418f, 0xa2c1, 0x1b58, 0x4160, 0xbff0, 0x404f, 0x1985, 0xbf90, 0x1915, 0x434c, 0xbf11, 0x4177, 0x408b, 0x0d62, 0x186b, 0x1168, 0x18e3, 0xb235, 0x3037, 0x4695, 0xba1b, 0xa33d, 0xba31, 0xb2c3, 0x40f3, 0xb0c4, 0x08b1, 0x4207, 0x4398, 0x41d5, 0x2398, 0x440a, 0x2c54, 0xba66, 0x1c94, 0xbaf5, 0xbf9c, 0x437b, 0xb2f5, 0x4674, 0x40c3, 0x435f, 0x41d5, 0x4233, 0xac76, 0xbae1, 0xba0c, 0xba07, 0x46ba, 0xb220, 0x22a5, 0xb25d, 0xb204, 0x41f3, 0x04e6, 0x46b9, 0x43a2, 0x41bd, 0x40b6, 0x1d71, 0xbfb1, 0x41cf, 0x32a2, 0x1e0c, 0x43e4, 0xb05c, 0xbf8c, 0xb273, 0x4072, 0x462b, 0x2863, 0x415a, 0x4000, 0xb22a, 0x42f5, 0x41c1, 0xa19b, 0x436f, 0xa5ad, 0x0a9d, 0x40e3, 0x4066, 0xa244, 0x41c8, 0x40a8, 0x43e4, 0x4301, 0xbf64, 0x36fb, 0x3220, 0x45e5, 0xa505, 0xb222, 0xb2a2, 0xb09c, 0x4418, 0xb02c, 0x1dcc, 0x40ad, 0x0709, 0xa73b, 0x4374, 0x33ea, 0xbad1, 0xbfa2, 0x4021, 0x4082, 0x4449, 0x41b6, 0x1df3, 0x42a8, 0x0838, 0x41e1, 0x4104, 0x218c, 0x4384, 0x2d58, 0x006d, 0x4561, 0x4374, 0xb299, 0x4177, 0x4064, 0xa457, 0x3187, 0x1b34, 0xbf63, 0x4261, 0x4158, 0x2bf6, 0x435e, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xddc07271, 0x377926ce, 0x9a93827d, 0xcbef0283, 0x5bc182ba, 0xa9bfdede, 0x4f15e679, 0xd23d8725, 0x59c353e8, 0x9a2b180a, 0x2b318b54, 0xd61dea1b, 0x8cc70559, 0xca220005, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x0000008d, 0x00000000, 0x00000006, 0xffffe6b3, 0x00000000, 0xfffffffa, 0x000018ab, 0xc35cf61b, 0x00010000, 0x00010000, 0x00001630, 0xad4c19c4, 0x00001aa0, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x418a, 0x18b4, 0xb262, 0x1dbf, 0xb028, 0x2f39, 0xba6e, 0x1e05, 0x41b8, 0x1a37, 0xbfc2, 0x3a0d, 0xaacb, 0x435a, 0x411a, 0x43c3, 0xbfc1, 0x4241, 0x1c8d, 0x46db, 0x2c85, 0x420a, 0xb29c, 0x35a9, 0xad4a, 0xb007, 0x433d, 0x42a9, 0x2e9d, 0x42dc, 0xba41, 0xba2b, 0xb236, 0x40ad, 0x2a1f, 0xa550, 0xb2ae, 0x0dd7, 0xbaea, 0x456a, 0x4299, 0x40d9, 0xa992, 0xbf0a, 0xbaff, 0xbad1, 0x4667, 0xaf90, 0x4281, 0x4310, 0xba51, 0xbf7c, 0x30f1, 0x43ff, 0x41c9, 0xb2f2, 0x412e, 0x43f1, 0xba1d, 0x45da, 0x275c, 0xb2d2, 0x337c, 0x4200, 0xbfb5, 0xb292, 0xba70, 0x4174, 0x40b0, 0x41e1, 0x42ed, 0x41e6, 0x439f, 0x224e, 0xba2c, 0x12e8, 0x411c, 0x1dea, 0x40f4, 0x4199, 0x43ac, 0x40fb, 0xb23b, 0x0f39, 0x4093, 0x45cb, 0x431a, 0x13c1, 0xbf73, 0xb271, 0x398f, 0xb018, 0x402a, 0xb002, 0xaac0, 0x1bc9, 0x2243, 0x0ec2, 0x3f51, 0xbf1b, 0x408c, 0xb0d0, 0x1c15, 0x03fc, 0x39ff, 0xb2a6, 0x02ca, 0x46ba, 0x3b17, 0x19ea, 0x40ef, 0x064f, 0x43f8, 0xbfda, 0x4268, 0x4074, 0x35bd, 0x44fa, 0x1ceb, 0xbf2d, 0x4325, 0x4157, 0x415c, 0x431a, 0xbaf1, 0xa1b0, 0x1b83, 0x01c9, 0x4306, 0x42e5, 0x172c, 0x0b7d, 0x405a, 0x4140, 0x4393, 0x4056, 0x1e6e, 0x414e, 0x40f4, 0x43a2, 0xb292, 0xbf3f, 0x4445, 0x190a, 0x431b, 0x1b32, 0xbf59, 0x4625, 0xad4e, 0x45bb, 0x446d, 0x44cc, 0x195c, 0x4619, 0xafd9, 0x0a89, 0x23e7, 0xa8ec, 0x425c, 0x41d0, 0x362d, 0x4257, 0xb036, 0xbf39, 0x44ec, 0xb00c, 0xb214, 0x1f74, 0xba33, 0x4574, 0x44f9, 0x41aa, 0x3e33, 0xbf7e, 0xb089, 0x4260, 0x44fd, 0xbfa8, 0x4262, 0x4348, 0xa90d, 0xaee0, 0xbf97, 0xba70, 0xba28, 0x4370, 0xb2c6, 0x41d4, 0xb21b, 0xb281, 0x43fc, 0xbfd3, 0x1a9f, 0xbac9, 0xbf60, 0x3f4b, 0x25a2, 0x4149, 0xbaeb, 0x4009, 0x238f, 0xba6d, 0xba55, 0xbfa0, 0x3191, 0x22ce, 0x3c36, 0xa50f, 0xaba5, 0xb0dc, 0x4216, 0xb2ff, 0x4322, 0x40fa, 0x418f, 0xba57, 0x4553, 0x1f57, 0xbf89, 0x0fe2, 0x3db5, 0xb201, 0x4111, 0xa77b, 0x42f0, 0x18c1, 0xbfa1, 0x1797, 0xbf80, 0x415c, 0xa873, 0x1a6f, 0x43c4, 0x41dc, 0x42d8, 0x4107, 0x40fa, 0x44d3, 0x401b, 0x0e14, 0xbfe0, 0xbacb, 0x0018, 0xb029, 0x403e, 0x4259, 0x408c, 0xb021, 0x0d72, 0x409f, 0x3e97, 0x0710, 0x422a, 0xbf23, 0xb08d, 0x1777, 0x2940, 0x40b2, 0xbae8, 0x1d10, 0x2a49, 0x24f4, 0x4297, 0x44d2, 0x405b, 0xbfb4, 0x4036, 0x3cc2, 0xbac2, 0x4287, 0x408f, 0x04cf, 0x4402, 0xbf36, 0x18a3, 0x1153, 0x4222, 0xb058, 0x403b, 0x424e, 0x1f0a, 0x419f, 0x43cd, 0xba14, 0x403f, 0xbae3, 0x1a87, 0xb275, 0x4053, 0x433d, 0xb2b7, 0x20a2, 0x40ed, 0x425c, 0x405e, 0xb2f0, 0x169c, 0x43de, 0xb2b4, 0xb030, 0x401e, 0xbf75, 0x4329, 0x1f63, 0x40ed, 0xb2e5, 0xb21e, 0x2dae, 0x13ff, 0xb246, 0x19fb, 0x43a2, 0x4179, 0x4304, 0x25a9, 0x1d4e, 0x1804, 0xb2d8, 0x41ba, 0xb28c, 0x4264, 0x4237, 0x27ff, 0x400a, 0xb00e, 0xb0c7, 0x42af, 0x33d3, 0x0b19, 0x42f9, 0xbf29, 0x41be, 0xb20e, 0xba0a, 0x25c7, 0x4215, 0x1b76, 0x16d0, 0xb08c, 0xb299, 0x4305, 0xbf94, 0xa65b, 0x02f3, 0xb226, 0x40c5, 0x185a, 0x42a4, 0xbf92, 0x1a1f, 0xb29b, 0x40d5, 0xba22, 0xb27b, 0xb0ac, 0x2ee1, 0x424e, 0xb2f8, 0x4299, 0xb26d, 0x0c8c, 0x4350, 0x4191, 0x1a7a, 0x439f, 0x1f7b, 0xb0f6, 0x442d, 0x0997, 0xbf36, 0x418d, 0xba3f, 0xb096, 0x0faa, 0xb201, 0x4302, 0x1919, 0x1d75, 0xb2f2, 0x4288, 0xbf00, 0xb203, 0x413f, 0x40ea, 0xbfd0, 0x198b, 0x4117, 0x439c, 0x09e7, 0xb277, 0x0cce, 0x43a1, 0xba4e, 0x466d, 0x18cf, 0x1e3e, 0xba56, 0xad9d, 0xbf71, 0x428e, 0x0a44, 0x434f, 0x4335, 0x401c, 0x403a, 0xbac5, 0xb0eb, 0xb075, 0x1ee0, 0x300b, 0x419a, 0x4152, 0x1833, 0x412f, 0x42cd, 0x42b6, 0x0e30, 0xa3fc, 0x40d4, 0xb226, 0x409d, 0x0a68, 0x3bb8, 0x1b5d, 0x1a26, 0xb2bb, 0xb290, 0x4294, 0xbfb8, 0x1d4a, 0x431a, 0xb071, 0x41f8, 0x42d3, 0x23e3, 0xb2c7, 0xb217, 0x436d, 0xba56, 0xb2c6, 0x417d, 0x1b54, 0xba72, 0x40eb, 0x0b91, 0xbf2f, 0xb242, 0xa679, 0x41bd, 0xba15, 0x4394, 0x419e, 0x42ad, 0x2bbf, 0x405e, 0x4128, 0x3858, 0x46cb, 0xbf79, 0x40e6, 0x42e0, 0x411c, 0x23b9, 0x42c2, 0x42f9, 0x00d5, 0x435b, 0x08e8, 0x2279, 0xbf94, 0xad4a, 0x4014, 0x40b4, 0x21dc, 0x435a, 0x46b9, 0xb2c5, 0xa749, 0x43b4, 0x40d2, 0x4408, 0x43d5, 0xbf60, 0x43d3, 0x408f, 0x2e00, 0x1ec6, 0x4186, 0xa036, 0x4147, 0xb227, 0xbf9d, 0x1a4b, 0x418f, 0x44b2, 0x40d9, 0xbf37, 0xb2ac, 0x1173, 0xb0e2, 0x409d, 0xb23e, 0xb230, 0x42da, 0x4081, 0x413d, 0xafe1, 0x1d08, 0x408b, 0x412a, 0x334a, 0x4023, 0x4028, 0xb014, 0xba0d, 0x05a5, 0x118a, 0x4090, 0x40cc, 0xb2da, 0x2c35, 0x4057, 0x40ee, 0x408c, 0xb259, 0x3bb3, 0xbf3d, 0x4323, 0x4384, 0xb286, 0x43a6, 0x41e6, 0x2ac2, 0x434b, 0x2a9c, 0x40bc, 0x43c4, 0x1f48, 0x2920, 0xab0a, 0x439e, 0xb0c9, 0xbf7a, 0x4081, 0xba67, 0xbf70, 0x42bc, 0x2f3c, 0x41b1, 0xa46f, 0xbace, 0x3c3d, 0xb2f1, 0x43c7, 0x4049, 0x11d4, 0xbf45, 0x4042, 0x1115, 0x4113, 0xba1b, 0xa069, 0x4571, 0xbacd, 0x1aee, 0x4499, 0xb06a, 0x42ac, 0x4069, 0xa4cd, 0x402a, 0x416b, 0x429c, 0x4362, 0x1314, 0x38c5, 0x400a, 0x4122, 0x3473, 0x413a, 0xba66, 0x27a8, 0x42ca, 0x4369, 0xbfce, 0x4268, 0xa867, 0x4072, 0x04f5, 0xba27, 0x43b5, 0x3c3f, 0xbf25, 0xba45, 0x439c, 0x1cd7, 0xb203, 0x45a3, 0xba2b, 0x41bd, 0x4013, 0x41a0, 0x43cd, 0xb2c4, 0xba18, 0x4097, 0x1507, 0xba2e, 0xaa23, 0x417a, 0x42fc, 0x1df3, 0x40c1, 0xba68, 0x41b6, 0xa65b, 0x438f, 0x1cbe, 0x01aa, 0xbf5d, 0x42a5, 0x070b, 0x423f, 0x465f, 0x4350, 0xa044, 0xb287, 0xb289, 0x42ad, 0x40fc, 0xafbe, 0x0b8c, 0xb2aa, 0x40d3, 0x463c, 0x4314, 0x1d39, 0x40e1, 0x1d12, 0x0eb4, 0xbf23, 0x43b0, 0xb2ca, 0x1b06, 0x414d, 0x1813, 0xb2f0, 0x1cec, 0xa02b, 0x454e, 0x4111, 0x19e3, 0x41f2, 0x4228, 0xbada, 0xb20f, 0x4342, 0x418c, 0x435d, 0x2d58, 0xb2af, 0x429b, 0x4020, 0xb2ae, 0x198b, 0x4256, 0xbf7c, 0xbaf9, 0xa179, 0xb012, 0xb2ed, 0x466e, 0x42b8, 0xb003, 0xb0c2, 0x429c, 0xb2fe, 0x00cb, 0x4440, 0x428e, 0xb033, 0x467e, 0x419e, 0x4396, 0xb261, 0x29a3, 0x434c, 0x1f85, 0x2734, 0x4068, 0xbf52, 0x44cb, 0xbac6, 0x4374, 0xb013, 0x42b7, 0x3f89, 0x163d, 0x1db9, 0xbf19, 0x430c, 0x4326, 0x440e, 0xba28, 0xba31, 0x43e2, 0x41b9, 0xad58, 0x4108, 0x1ad5, 0x428b, 0x1393, 0x4290, 0xb2b9, 0x403d, 0xb23d, 0xb00e, 0xaf72, 0x420e, 0x438e, 0x1298, 0x40db, 0x41ce, 0x40b8, 0x1da4, 0xb085, 0x420b, 0xbf18, 0x025d, 0x1b8b, 0x1038, 0x4191, 0x1e3f, 0x3912, 0x4015, 0x1b40, 0xba0a, 0xb0a5, 0x2dfa, 0x405a, 0xb2dc, 0x419f, 0x020d, 0xbf03, 0x420b, 0x4185, 0xbaca, 0xb0a1, 0x32e6, 0xbae1, 0x40ad, 0x0c4c, 0x426a, 0x442d, 0x0cbc, 0x3f03, 0x4220, 0xb025, 0xb05d, 0x0a54, 0x4253, 0x40e4, 0x0628, 0xba74, 0x4130, 0xb201, 0x3d5a, 0x1f92, 0x42fb, 0x2807, 0x3fd0, 0x434e, 0xbfd5, 0x433e, 0x4171, 0xb0c9, 0x4389, 0x0633, 0xbadd, 0x4468, 0xae94, 0x1efb, 0xb0ba, 0x4066, 0x18b4, 0xbf9b, 0x0986, 0x4373, 0x3dfe, 0xb2cf, 0x4276, 0x37fc, 0x4154, 0xb03b, 0x34e0, 0x1e93, 0x4394, 0x4261, 0x43ce, 0xb0b7, 0xb234, 0xb242, 0x406a, 0x42e3, 0x2ae2, 0x1b93, 0xbf96, 0xb07a, 0x4317, 0x1e51, 0x42e5, 0x430d, 0x324b, 0x4354, 0x410d, 0x41b8, 0x1eb2, 0x4035, 0x06cb, 0x4353, 0x43ca, 0xa102, 0x203d, 0x414e, 0xba16, 0x1b5a, 0xb2d4, 0xb292, 0xbf2d, 0x411e, 0xba6a, 0xb29e, 0x430b, 0x418f, 0xb017, 0x3d40, 0x4271, 0x43ca, 0xb20d, 0x42d1, 0x1ae9, 0x419d, 0x4021, 0x466a, 0x2bb4, 0x1ef8, 0x24be, 0x4146, 0xbf77, 0x423e, 0x4075, 0x4145, 0xb23b, 0xbae8, 0x0db2, 0xb20b, 0xb2bf, 0x0cbf, 0x183f, 0x0f19, 0x21e3, 0x4231, 0x46f9, 0x400b, 0xb0bf, 0x405f, 0xb047, 0x3f77, 0x1c33, 0x3d13, 0x3476, 0xb218, 0xbf7c, 0x4060, 0x43b8, 0xbf80, 0xb0c4, 0x40e7, 0x421c, 0xbf64, 0xb0de, 0x436e, 0x41b5, 0xb0bc, 0x4065, 0x40f4, 0x2bb9, 0xaf92, 0x4624, 0x1e4c, 0x4318, 0x41e2, 0x0adf, 0xb2d4, 0x1e99, 0x1e3b, 0x46f0, 0xa8b3, 0xbf7e, 0x32ac, 0x2b7a, 0x41fb, 0x40fd, 0xba62, 0xbf21, 0x0d64, 0x42b0, 0x3a8c, 0x4270, 0x1e40, 0xbfd4, 0xb298, 0x4084, 0x43e2, 0xbaf3, 0xb256, 0x422d, 0x2965, 0xb29d, 0xb260, 0x41ec, 0x41fb, 0x071c, 0xbf4e, 0x403a, 0x41c6, 0xbad3, 0x41b6, 0x4651, 0xb2c7, 0x44e2, 0x43fd, 0x421f, 0x09c3, 0x4018, 0xbaf3, 0xbff0, 0x106b, 0x40bc, 0x46bc, 0x41ba, 0x41be, 0xb297, 0x4270, 0x4369, 0xbf6c, 0x46b4, 0xb21a, 0x0857, 0x37bc, 0x41c4, 0x1cc5, 0xaddc, 0xb014, 0xb2e3, 0xb0fb, 0x1f42, 0x06c9, 0x412d, 0xa6cd, 0xb2bc, 0x40a6, 0xb0e1, 0x4174, 0xb29f, 0x4200, 0xbf55, 0xba34, 0x4080, 0x434f, 0xb249, 0x43f7, 0x44e1, 0xa252, 0x409d, 0x42bc, 0x4195, 0x4274, 0xba0f, 0x41ce, 0x43e9, 0x45e3, 0xb21b, 0x2ffd, 0xa5c2, 0xba7b, 0x4068, 0x4062, 0x0ea1, 0xbf68, 0x0330, 0x4041, 0x2f6d, 0x0761, 0x4354, 0x3531, 0x40c5, 0xac1f, 0x0a0d, 0x4104, 0x4338, 0x428a, 0x4293, 0xba73, 0x0b8b, 0x0958, 0xbfa3, 0x00ff, 0x43b3, 0x1bca, 0x4146, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xca173107, 0x59131982, 0xd9386e54, 0x24fdf592, 0x3f36eb46, 0x7926dfe1, 0x64131655, 0x72501df8, 0xd923625a, 0xfe222377, 0xf0f7cf65, 0x2d44a91d, 0x61d44f3a, 0x2f3c5836, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0xffffffc0, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000040, 0x00000000, 0x000016c0, 0x8f32eca2, 0xfe2234ce, 0x00000000, 0x2f3c6602, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x23f8, 0xbfae, 0x415e, 0xbf00, 0x00dd, 0x1949, 0xb2d0, 0x45e8, 0x4179, 0x1814, 0xb001, 0xb239, 0x2993, 0xbfd8, 0x42ec, 0x3367, 0xbf7c, 0xb2a5, 0x3158, 0x4273, 0xbf16, 0x4099, 0x2952, 0x1ff7, 0xbfa8, 0x4059, 0xac74, 0x413a, 0xb03e, 0x1e71, 0x2513, 0x4293, 0x1e93, 0x4321, 0x16a0, 0x410e, 0x1d63, 0xb048, 0x43eb, 0x417c, 0xbf7f, 0xab52, 0xaa4f, 0x4267, 0xb2bd, 0xb227, 0x40ce, 0x0c3d, 0x09b8, 0x4218, 0xa803, 0x1b80, 0x4037, 0x185b, 0x3a2a, 0xbfdc, 0x4345, 0x42e2, 0x2885, 0xb0f7, 0xb2ca, 0x46aa, 0x2e1d, 0x436c, 0xba12, 0x4227, 0x0c58, 0xb075, 0x08a1, 0xba6b, 0xada7, 0xb034, 0xa86d, 0x426e, 0x428f, 0x4218, 0x4389, 0x1a12, 0xaab7, 0xbfe0, 0xb299, 0xbf3e, 0x41c7, 0x4200, 0x40cd, 0x4073, 0xb00b, 0x1122, 0x4270, 0x1516, 0xb0fe, 0x306c, 0x412d, 0x40ce, 0x3c5c, 0x1fab, 0x43bb, 0x4446, 0x405e, 0x40c6, 0xb03b, 0xba37, 0x403e, 0x46fc, 0xbf15, 0xb262, 0x1b66, 0xb26f, 0x4461, 0x22a8, 0x4276, 0x4289, 0x42e4, 0xbaf7, 0x404b, 0x166d, 0x44d9, 0x43fe, 0xb0eb, 0x43d9, 0xaa3b, 0x0b0e, 0xbf0a, 0x332e, 0xba4f, 0x1f48, 0x446d, 0x2fcc, 0x4009, 0x431a, 0x4006, 0x1f9b, 0x42c6, 0xb01d, 0x17a0, 0x4369, 0xbfc0, 0x07be, 0x4026, 0xb00f, 0x4412, 0x30d4, 0x4089, 0x461c, 0x432e, 0xbf36, 0xba17, 0x40ab, 0x432a, 0xa280, 0x1f32, 0xb0e4, 0x23c0, 0x1a9e, 0x4080, 0xba1f, 0xba5e, 0x2887, 0xad8b, 0x282d, 0x1882, 0x41ab, 0xb2e0, 0xb2e7, 0x4373, 0x41c2, 0x41ed, 0x45eb, 0xb00f, 0xb07f, 0xba27, 0xbf63, 0x1bcc, 0x4098, 0xa2a7, 0x189c, 0x4338, 0x4308, 0xbace, 0x1ff4, 0x40fd, 0x4279, 0x0ae8, 0x3b6d, 0x4315, 0x4654, 0x00d0, 0x425e, 0x40ef, 0x4357, 0x192b, 0xbf94, 0x43ce, 0x37fa, 0x1c40, 0x2cbf, 0x26f3, 0x1fe8, 0xb227, 0xabe3, 0x401e, 0x310b, 0x40ce, 0x4223, 0xbf65, 0xba30, 0x4352, 0x4241, 0x0f84, 0x1941, 0x4098, 0x40f7, 0x409a, 0x4081, 0x400c, 0x41da, 0xb256, 0x45cd, 0x1f42, 0x42f2, 0xbaef, 0x0335, 0xa8e0, 0x429a, 0xba0e, 0x40a3, 0x3d57, 0x439d, 0x406f, 0xbae3, 0xbfc6, 0x2f1b, 0xb2f5, 0xa396, 0x41ff, 0x40e4, 0x4401, 0x43f9, 0x4169, 0x40cb, 0x1f8b, 0x4335, 0xb25a, 0x4407, 0xba60, 0x4286, 0xa2ee, 0x4027, 0xbf3f, 0x1c86, 0x435c, 0x40bc, 0x41cc, 0x1bda, 0x2ad1, 0x1c02, 0xb04d, 0x42cb, 0x4084, 0x1dd2, 0xb2fd, 0xb0ac, 0xb280, 0x1dad, 0xb06c, 0x43a1, 0x4310, 0x40e4, 0x435f, 0xb2be, 0x4201, 0xbf7b, 0x1e70, 0x0de8, 0x02bf, 0x407d, 0x45d2, 0x1d92, 0x43c8, 0xbfa4, 0x401d, 0x431e, 0x4030, 0x40a1, 0x1fb9, 0x2bdc, 0xbf44, 0x423c, 0x41f8, 0x0525, 0xbf03, 0xbae5, 0x4316, 0x2fa1, 0xb042, 0x438d, 0x3f53, 0x424a, 0xa7ee, 0x4470, 0x0dc7, 0x4408, 0x18e7, 0xb2f9, 0xb01f, 0xa1af, 0xbf28, 0xbfc0, 0xb26b, 0xa1cc, 0xbf6c, 0x0855, 0x3e49, 0x45e1, 0x41a7, 0xba1b, 0x2d90, 0x4216, 0xb215, 0xb224, 0x4136, 0xb0fb, 0x4676, 0x1127, 0x4151, 0xba11, 0xb287, 0xb2e4, 0x4074, 0x4313, 0x060e, 0x1dc1, 0xb213, 0x4000, 0x42ca, 0x4085, 0x264b, 0xbf84, 0xa3c4, 0xb045, 0x0f04, 0x425a, 0xba22, 0x414a, 0x4155, 0x40ae, 0x12ca, 0x3ed5, 0xb0e1, 0x4080, 0xba16, 0xb04b, 0xba70, 0xbfa0, 0x40d5, 0x402b, 0xbaf1, 0x46b8, 0x1f45, 0xbf60, 0x423d, 0xb2ee, 0x2b14, 0x1e82, 0xbfc9, 0x1e6e, 0x415b, 0xba48, 0x2cf4, 0x43ee, 0x4322, 0xba57, 0xba68, 0x434d, 0xba0d, 0x31f2, 0xbf86, 0xa72b, 0x43d1, 0xb22c, 0xb29a, 0x3a3a, 0x433b, 0x41ad, 0x401a, 0x1df5, 0x0084, 0xa1db, 0xbfe0, 0x4091, 0xb294, 0x43f7, 0xbfde, 0x1d6a, 0x40c1, 0x4052, 0x14ba, 0x4315, 0x1b64, 0x4370, 0x1ef0, 0x0bf2, 0xb09b, 0xbf2e, 0x4260, 0xba0d, 0x0a68, 0xbfd0, 0x4197, 0x1e0a, 0xb204, 0x4282, 0x1afb, 0x4369, 0x438d, 0x41b4, 0x40e6, 0xba10, 0xb2e6, 0x436f, 0x40f8, 0x4207, 0x1986, 0x4342, 0xa33e, 0xb2e6, 0x1fe0, 0x402c, 0xbf48, 0xb03e, 0x40df, 0x0251, 0x1e47, 0x42dd, 0xb2d3, 0x1c88, 0xbf42, 0x36f1, 0xbaf1, 0x15ed, 0x433c, 0xbac4, 0xb0aa, 0x4315, 0xb242, 0x1fcd, 0x40b2, 0x420b, 0xb0e1, 0xba7c, 0xa872, 0x2966, 0xba76, 0x4635, 0x438b, 0x40b4, 0xbf80, 0xba5a, 0xbf80, 0x4350, 0xbfcf, 0x3a60, 0x45f4, 0x21f6, 0x2fc8, 0x425f, 0x2511, 0x42cf, 0x1945, 0x2264, 0x40ab, 0xbf8e, 0xb27e, 0xba02, 0x415f, 0xbfd7, 0x1026, 0x40a9, 0x1a50, 0xaca5, 0x4569, 0x1a72, 0x1ca3, 0x407a, 0xbac3, 0x402a, 0x1d13, 0x1ba5, 0x43bb, 0xb2fa, 0x3dc5, 0xb25f, 0xb2cd, 0x1eda, 0xba67, 0x39d3, 0x16ad, 0x1c10, 0x0b53, 0xb225, 0xbf55, 0xb20a, 0x29be, 0xb2cc, 0x43f5, 0x0c83, 0xb21f, 0x439e, 0x45d1, 0xb2b1, 0x0286, 0x441f, 0x0eb2, 0xbae9, 0xba76, 0xb2c4, 0x18f1, 0x40f6, 0xb2c9, 0x052d, 0xbaf3, 0x427f, 0x0215, 0x43ba, 0x4561, 0x14e4, 0xba1e, 0x41b5, 0xaf51, 0x42a3, 0xbf52, 0x4037, 0x410b, 0xba44, 0xbf70, 0x2dee, 0x005e, 0x4181, 0x1d78, 0x0559, 0xbf48, 0xb0ef, 0x40d1, 0x1bc3, 0x4012, 0x4613, 0x0053, 0x4351, 0xb2bd, 0x4381, 0x1a6a, 0x45c6, 0xb200, 0x4443, 0x08ef, 0xbfaf, 0x425e, 0x42e9, 0x437d, 0x10d0, 0xb25b, 0x441c, 0xbad8, 0x4263, 0x2561, 0xb02c, 0xbaf6, 0x428d, 0xb2d5, 0x406d, 0x4015, 0xb0dd, 0x45ac, 0x43d0, 0x365f, 0xb0df, 0xb2c3, 0x1829, 0xb0b3, 0x42e6, 0x02a5, 0x46d2, 0xbf0c, 0x4132, 0x409e, 0x30b6, 0xb25a, 0x42ef, 0x42f1, 0x0405, 0x41b2, 0x41e6, 0xb254, 0x4107, 0x40c6, 0x41a8, 0x0bec, 0x4134, 0xa686, 0xa857, 0xbf04, 0xab3f, 0xbfb0, 0xbf16, 0x4240, 0x42f4, 0xb2fb, 0xb2db, 0xb0f2, 0x197c, 0x0476, 0x406d, 0xb265, 0x40a1, 0x4389, 0x433e, 0xba5f, 0x41e8, 0x0e2f, 0x42f7, 0x4088, 0x3609, 0x24cb, 0x06bb, 0x428c, 0x1544, 0xbf19, 0xba5b, 0xba74, 0x4317, 0x42cc, 0xb261, 0xab8d, 0xa8bf, 0x003f, 0xa63e, 0xba06, 0x4061, 0x1a1d, 0x1baa, 0xbfb5, 0x42e0, 0x4273, 0xba55, 0xa83f, 0x46e0, 0x4324, 0x4556, 0x40e0, 0x421d, 0x2fcd, 0x42b4, 0x3700, 0xbae7, 0x19a2, 0xba1c, 0xba30, 0xb269, 0x1717, 0x413f, 0x1d62, 0x412e, 0x1196, 0x1c42, 0xbf66, 0x4017, 0x4238, 0x4351, 0x405a, 0x28e5, 0x29e7, 0x4555, 0x4052, 0x4356, 0x42bb, 0x079e, 0x42cc, 0xb245, 0xba69, 0x41c6, 0x4158, 0x3c55, 0x1ea8, 0xb25d, 0xbfdb, 0xbae1, 0x2de6, 0x19fe, 0x1b24, 0x4076, 0x46f1, 0xb020, 0xbadf, 0x4373, 0x4332, 0xacca, 0xb072, 0x440d, 0x445c, 0x192d, 0x317a, 0xbf48, 0x2038, 0x08bd, 0xaf9f, 0x0031, 0x4336, 0x15e1, 0xbfb5, 0x4387, 0xbacb, 0x4379, 0xb215, 0xb254, 0xbf7c, 0x19bc, 0x04e4, 0x4574, 0x432f, 0xb279, 0xbf6b, 0xbfd0, 0x2ec6, 0x41a4, 0x2665, 0xb0d5, 0x40e9, 0x435a, 0x4064, 0x1f9b, 0xb2ca, 0x0bee, 0x3d64, 0x3530, 0xbada, 0xa670, 0x42d5, 0x4200, 0x42f5, 0x42b8, 0xbad3, 0xba31, 0xbf71, 0x1cb2, 0xb205, 0x41eb, 0xba50, 0x4126, 0xbfad, 0xb0d7, 0x0788, 0xba09, 0x4246, 0x464a, 0x07d1, 0x438e, 0x4336, 0x2594, 0x3309, 0x4373, 0x4324, 0xb097, 0x428b, 0xbf90, 0x40e8, 0xa8ec, 0xbf3a, 0x0a6a, 0x2d9b, 0x04d6, 0x407d, 0xb233, 0x10b7, 0x40e0, 0x42a7, 0x43fd, 0xa709, 0x19a8, 0x40c3, 0x1b39, 0x19ea, 0x4492, 0x4046, 0xbae9, 0xba67, 0xaee3, 0x435c, 0x43ef, 0x409e, 0x4397, 0x3a1d, 0xbfa0, 0x4158, 0x4135, 0x1f07, 0xbf25, 0xb24f, 0x428f, 0xb2c9, 0x411e, 0xbf00, 0xba6c, 0x26ae, 0xb00f, 0x35f8, 0x419c, 0x1ea4, 0x42d4, 0xb207, 0x1611, 0x4399, 0xbf5c, 0x4283, 0x07d5, 0x10a8, 0x4387, 0x42db, 0xb259, 0xbf04, 0x1f26, 0x413c, 0x40b0, 0x4389, 0x1526, 0x212f, 0xbf63, 0xb0d6, 0xbad7, 0x33b4, 0x4322, 0x4317, 0x4188, 0x43eb, 0x40ea, 0xbfe4, 0x0285, 0x22a7, 0x4291, 0xbaf8, 0xb0ef, 0x438a, 0x1dd5, 0x4344, 0x266e, 0x41ca, 0x4155, 0x430a, 0xb2b7, 0x3f6f, 0xba27, 0x4372, 0xba32, 0x291c, 0x129c, 0xbfb2, 0xac6e, 0x4046, 0x4248, 0x4338, 0x12c1, 0xb227, 0x4168, 0x0f8e, 0x4038, 0x3caa, 0x43b0, 0x1a4b, 0xbf33, 0x21dd, 0xbae6, 0x1bab, 0x43a6, 0xbada, 0x1c96, 0x40a7, 0x4065, 0xb264, 0x1b34, 0x103f, 0x10e0, 0x007e, 0x0742, 0x405b, 0xbada, 0x466f, 0x438a, 0x40f5, 0xb287, 0x43a5, 0xb237, 0x4350, 0x44e5, 0x4222, 0x4067, 0x405e, 0xbfc6, 0x411e, 0x4222, 0x289f, 0xba0c, 0x2535, 0xb29e, 0xba3c, 0xb2a9, 0x1d5c, 0x1afb, 0x4033, 0x1ea6, 0xa772, 0x2d80, 0xa777, 0x43e4, 0x40c9, 0x401f, 0x315d, 0xba55, 0x4595, 0xb2a3, 0x409a, 0xbfaf, 0x46db, 0x4328, 0x4373, 0xb23b, 0xb0a6, 0x4093, 0xa89e, 0xb248, 0x43c9, 0x1643, 0xb04f, 0xb21b, 0x4365, 0xb2d3, 0x4660, 0x42b9, 0xb273, 0x0942, 0x4335, 0x1def, 0xab0c, 0xbf0e, 0x43af, 0x317b, 0x407e, 0x435d, 0x361a, 0x454c, 0x4304, 0x3b0b, 0x4046, 0xbf76, 0x4368, 0xb0fc, 0x4277, 0xa54e, 0x15dd, 0x43ef, 0x42ba, 0x436a, 0x3be5, 0x410e, 0x2870, 0xbf7e, 0x1890, 0x18a3, 0x4203, 0x4384, 0x4371, 0x0e8e, 0x42a8, 0x288f, 0x1920, 0x447b, 0xba58, 0x41e2, 0xb296, 0x42a4, 0x4151, 0xae23, 0x4268, 0x42ea, 0xb2dc, 0x42b5, 0x1ac1, 0x43af, 0xba4d, 0xbf3e, 0xb238, 0x4208, 0x2d85, 0xb284, 0x183c, 0x40c6, 0xa6b9, 0x429f, 0x0428, 0xb080, 0x23ff, 0x4191, 0xb286, 0xab3d, 0xbf23, 0xb2ab, 0x4101, 0xba54, 0x42ae, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe99b8527, 0x3e866f78, 0xbb8f4079, 0xc9ef19b8, 0xaaf5b29d, 0x98cec417, 0x79e91cc3, 0x8e2e7557, 0xc691e551, 0x2095f998, 0xe3609aa0, 0xb521524b, 0xcc39afbb, 0xcf08047d, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x161c0000, 0x0cd2dc16, 0xf32d3fff, 0xcf08105b, 0x000000c2, 0x0000161c, 0x00000000, 0x00000061, 0x000010d2, 0x00000000, 0x00001643, 0xb521524b, 0x000010d2, 0xcf080f67, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbf78, 0xb251, 0x1eb2, 0x15c1, 0x1b9e, 0xb26c, 0x46ba, 0x4392, 0x42d2, 0x42bd, 0x40e0, 0x1474, 0xba55, 0x4199, 0xb255, 0xbf64, 0x4214, 0x44fb, 0x437e, 0x426f, 0x4258, 0x4079, 0x409b, 0x4240, 0x42da, 0x02ff, 0x41f7, 0x1e1c, 0xbafd, 0x4108, 0x1976, 0x4304, 0x4363, 0x4376, 0x422f, 0x4448, 0x4164, 0x41f6, 0xaada, 0xbf0d, 0x43b8, 0xb0ab, 0x43c9, 0xb084, 0x0010, 0x419f, 0x4158, 0xb241, 0x4008, 0x44a8, 0x41dc, 0x42f1, 0x4083, 0x2913, 0x43e7, 0x4226, 0x1f85, 0xbf57, 0x40f3, 0xbac0, 0x43e2, 0x17b4, 0x401b, 0xa764, 0x19f8, 0xa5d3, 0xba04, 0x42c5, 0x410c, 0x0045, 0x40df, 0x45c4, 0x4370, 0xb0e6, 0x40f9, 0x1994, 0x42ea, 0xa907, 0xbf67, 0x4016, 0x1d1c, 0x0fd3, 0xb022, 0x1d1e, 0x4688, 0x44a4, 0x4357, 0x18dd, 0x3d71, 0x1ab6, 0x19b6, 0x1f8f, 0xaaf6, 0xb23a, 0x1b6a, 0x4137, 0x40a8, 0xbfe8, 0x4388, 0x422d, 0x434b, 0x0732, 0x4068, 0x42d5, 0x42f3, 0x41ad, 0xb20a, 0x4091, 0x411a, 0xba44, 0xa7d9, 0x1688, 0xb09c, 0x1d2f, 0x1cf1, 0xb2ca, 0x409c, 0x0873, 0x19b1, 0x409d, 0xb2e0, 0x4071, 0x425a, 0x04b0, 0xbf3a, 0x412f, 0x44e1, 0x3786, 0x412e, 0x419b, 0x4012, 0x4547, 0x19fb, 0xb0be, 0x4209, 0x4001, 0x43af, 0x429b, 0x20fd, 0x43d4, 0xbfa0, 0x40ee, 0x1a3d, 0xbfc0, 0x4418, 0xb2f9, 0x0a05, 0x2b88, 0xb23b, 0xbf5c, 0x183d, 0x4122, 0x463b, 0x3ed8, 0xbf70, 0x4381, 0xba5f, 0xba7d, 0x41ec, 0xba39, 0xbf70, 0x4596, 0x1660, 0x1e85, 0x4278, 0x43a4, 0xad91, 0x4046, 0x41fd, 0xba79, 0x408b, 0xbf4e, 0x0e75, 0xb22b, 0x1ee0, 0x4195, 0x1e6c, 0xb040, 0x43e2, 0x3745, 0xba65, 0x40e0, 0xa8ab, 0x4126, 0xbf41, 0xb20d, 0x4235, 0x0483, 0x43c5, 0x36d5, 0xaf62, 0xb0c7, 0x1e22, 0xbfa1, 0xb2e7, 0x17de, 0x3fa1, 0x2a2d, 0xb035, 0x4308, 0xa38f, 0x4115, 0xb001, 0xba31, 0x1d3f, 0x4093, 0xad99, 0x425f, 0x187a, 0x4206, 0xaaec, 0x41c2, 0x1b58, 0x4299, 0x419f, 0xbf59, 0x4252, 0xb275, 0x4157, 0x4314, 0x40c5, 0x4002, 0x4156, 0x4040, 0x1c34, 0xba34, 0x0b6f, 0x19d4, 0xba24, 0x40fd, 0x41cb, 0x4141, 0x3190, 0x19c1, 0xbad9, 0x1fdb, 0xbfab, 0x41ed, 0x43de, 0x4080, 0x3d58, 0x2532, 0x466a, 0xbfc0, 0xb049, 0x412a, 0xa8ca, 0xb2af, 0x420f, 0xbacd, 0x432a, 0xbfde, 0xb0e2, 0x43ea, 0x4100, 0x404d, 0x426e, 0xbf71, 0x124a, 0xb259, 0x407a, 0x43f7, 0x4215, 0x416f, 0x34fa, 0x4367, 0x40b1, 0xb239, 0x419f, 0xba28, 0xae76, 0x1a0e, 0xbf82, 0x4174, 0xb27d, 0xa2f6, 0x422b, 0x401c, 0x439a, 0xba51, 0x2d8f, 0xb063, 0x1d7c, 0x40f6, 0xb0f9, 0xb2aa, 0xbfaf, 0xaaaf, 0x40e9, 0x41a9, 0x16c4, 0xbad8, 0x1dda, 0x17ae, 0x26f9, 0xba47, 0xbad9, 0x41eb, 0x4086, 0x424f, 0x4234, 0xb290, 0x4236, 0xa8ce, 0x40fd, 0x18c3, 0x4291, 0xb2e9, 0x4240, 0xbf88, 0xbad6, 0x02a2, 0x439e, 0x1c1b, 0x1e78, 0xabba, 0x40c4, 0x42e9, 0xa4b2, 0x21f3, 0x1f64, 0x1c05, 0xbf6a, 0xb20b, 0x4034, 0x20f5, 0xbaf9, 0xac9b, 0xbfb0, 0xb02d, 0x43c9, 0x4020, 0x41f1, 0xb26e, 0x1c9d, 0xba7e, 0x4047, 0x41f4, 0x1bce, 0x412c, 0x4378, 0x3857, 0x43da, 0x419f, 0x4296, 0xba50, 0xbafd, 0x1f01, 0xbaf8, 0xa978, 0xbfb2, 0xb291, 0x42df, 0x410c, 0x40d3, 0x469b, 0x4388, 0x41eb, 0xbad7, 0x4367, 0x1aed, 0x411f, 0xba27, 0xb220, 0x417a, 0x1839, 0xb256, 0xb22e, 0xaeb8, 0xad5b, 0x0842, 0x2af8, 0x423d, 0xba02, 0xb256, 0xb093, 0xbfb5, 0x431d, 0xb01b, 0xb2cd, 0xba7c, 0x19f9, 0xbfa9, 0x1c2b, 0x40b2, 0xb024, 0xaf9c, 0xad91, 0x0191, 0x4152, 0x40b1, 0xbacb, 0xa5d4, 0x42e8, 0xb0b6, 0x080a, 0x41da, 0x4195, 0x40f4, 0x4329, 0xaa3c, 0x4602, 0x40f8, 0x1d9f, 0xbf86, 0x1d1d, 0xb041, 0x4294, 0xb215, 0x41a0, 0x4336, 0xb07c, 0xba73, 0x1537, 0xa583, 0x4332, 0xbfa3, 0xb03a, 0x2082, 0x405f, 0xb0e5, 0x4083, 0x3b48, 0x1fdd, 0x4399, 0x38f1, 0x41d9, 0xb09f, 0x1377, 0xb076, 0x195c, 0x4341, 0x1faa, 0x1a14, 0x1c98, 0x4177, 0xbf64, 0xbff0, 0x4341, 0x199c, 0xba2d, 0x4001, 0x4294, 0x271a, 0xbf49, 0xb2eb, 0x4625, 0xb2c7, 0x4282, 0x42c2, 0xbf90, 0x42cf, 0x4057, 0xbaef, 0x414a, 0xbf2b, 0x4196, 0x1071, 0x02c2, 0xa540, 0x424a, 0x032e, 0xb21b, 0x4331, 0x4049, 0x4673, 0xbfd2, 0x421f, 0x42ee, 0x417e, 0xa3ed, 0xb21e, 0xb0ba, 0xbac9, 0x4071, 0x3b78, 0x436b, 0xa792, 0xb0ea, 0xb271, 0xa5df, 0x1a45, 0x2020, 0x43ff, 0xb282, 0xb24f, 0xbfb6, 0x022c, 0xbae8, 0x3ac4, 0xb016, 0xba5a, 0x41e5, 0x407b, 0x0f5d, 0xb245, 0x1f55, 0x41ae, 0x43ba, 0x0399, 0x42df, 0x0d93, 0x4355, 0xba29, 0x4670, 0x4558, 0x047d, 0xbf86, 0x29a9, 0x19d7, 0x1f16, 0xb0fe, 0xbaf9, 0x21a5, 0x46fc, 0x2750, 0x1398, 0x2284, 0x0309, 0x461b, 0xb2cd, 0x29de, 0xbf0a, 0xba3f, 0x119d, 0xb24a, 0xa8e5, 0x4069, 0x148f, 0xb04d, 0xb264, 0xa8c6, 0x1dd3, 0x46b4, 0x18c7, 0x43bc, 0x40e3, 0xba4e, 0x405b, 0x4101, 0x1d87, 0x3d58, 0x43e6, 0x2199, 0xb21f, 0x1a00, 0x4301, 0x2a14, 0x02f4, 0xbfb7, 0x4441, 0xa4bf, 0x40bf, 0xb2d4, 0x4117, 0x11d9, 0x1f4a, 0x1923, 0x02f0, 0x4350, 0x1f6a, 0x4176, 0xb2f9, 0xb233, 0x31f5, 0x4012, 0xbfdd, 0x4297, 0x4575, 0x40ae, 0xba01, 0x42b1, 0xbacd, 0x40fa, 0x3d82, 0x4338, 0xba51, 0x411c, 0x4153, 0x2556, 0x4398, 0x1d63, 0x4305, 0x28c7, 0xbf0a, 0x4309, 0xb051, 0x42e7, 0x37b1, 0xb28c, 0x41f8, 0xb263, 0xbf60, 0x1f06, 0x421f, 0x4294, 0x401c, 0x43d3, 0xb2b3, 0x1a06, 0x428e, 0x42fe, 0xb246, 0xb27b, 0x43e7, 0x4135, 0x2f59, 0xbf6e, 0xac94, 0xbaf0, 0x12fd, 0x41da, 0x4337, 0xb294, 0xb06f, 0x4156, 0x42d8, 0x2309, 0x1de4, 0xb021, 0x0d01, 0x42fd, 0x1957, 0x44e1, 0x0a23, 0xb21d, 0xba2e, 0x2c2f, 0xa0c8, 0x40f8, 0x41f6, 0xbaed, 0x4224, 0xada8, 0x42ff, 0x4227, 0xbfc1, 0x2336, 0xbad7, 0xba01, 0xba5b, 0x1eec, 0xba44, 0x43d7, 0xbae3, 0xba1f, 0x417f, 0x3fbb, 0xb2aa, 0xb202, 0x07f6, 0xb249, 0x414f, 0x4144, 0xb25e, 0xba30, 0x413d, 0xb2c9, 0xb0a8, 0x029f, 0x19a0, 0xbf58, 0x4140, 0xb260, 0x0f52, 0xb23b, 0x42c4, 0x43db, 0xad0e, 0x43f0, 0x028d, 0xbf24, 0xbac3, 0x1c7e, 0x1094, 0xbff0, 0x41b1, 0x30a0, 0x01a0, 0x094c, 0x149b, 0xbfdf, 0x32f6, 0x41b3, 0x43d5, 0x35b7, 0x468c, 0x2936, 0x1877, 0x2db5, 0x436e, 0x18bc, 0x1c4b, 0xbf00, 0xb276, 0xbf00, 0xb262, 0xba52, 0x240e, 0x4281, 0x410b, 0xbfa2, 0x4173, 0x2aba, 0x43de, 0x2801, 0x432c, 0x40b4, 0x4119, 0xb2da, 0xaa5d, 0x43dd, 0x1c35, 0x4357, 0x4237, 0x191a, 0xba06, 0x3b92, 0x4018, 0xb299, 0x46c2, 0x1f3c, 0xba7a, 0x438e, 0x44b9, 0xba71, 0x401f, 0xbfd7, 0xbaf6, 0x41b4, 0x0e92, 0x4151, 0xb225, 0xb08b, 0x4035, 0x428c, 0x42b2, 0x41cb, 0xb2a3, 0xbae6, 0x41cd, 0xb2b9, 0xba59, 0x4022, 0xba11, 0x4246, 0x4625, 0x1b37, 0xad1c, 0x108d, 0x2de7, 0x1a8b, 0xbf9d, 0x1bc9, 0x43c4, 0x432a, 0x1a06, 0x4196, 0x3937, 0xb08f, 0xbad4, 0x18f6, 0x3f38, 0xbfc6, 0xb2d1, 0xb03a, 0x4246, 0x3ea1, 0x42bb, 0xb259, 0x43b0, 0xb2bb, 0xba47, 0xb28d, 0x1907, 0xbf6a, 0xb089, 0x4372, 0x4469, 0x182d, 0xb03d, 0xbf90, 0x4399, 0x4237, 0xb223, 0x36ba, 0x4173, 0x4233, 0x4616, 0x21de, 0x25e2, 0x4287, 0xb20a, 0x4612, 0x409c, 0xb24c, 0x40d6, 0x400f, 0xbad9, 0xbf01, 0x4324, 0x4374, 0xb0d1, 0xb21a, 0x1e51, 0xb2e8, 0x41ed, 0xbf60, 0x4664, 0x4043, 0x463a, 0xb21d, 0x430d, 0xbf19, 0x4203, 0x4339, 0x4549, 0xa026, 0x1ca7, 0xbaf4, 0xaaac, 0x4360, 0x2a0a, 0x125d, 0xb2de, 0x313b, 0x4025, 0xb221, 0xba16, 0x42e4, 0x4087, 0xba4e, 0x1f06, 0x31b6, 0x1af5, 0xbf3b, 0xb217, 0xb227, 0x3022, 0xba71, 0xbf11, 0x40db, 0x40a6, 0x4219, 0x4274, 0x4349, 0x4168, 0xbfb1, 0x4219, 0xb27f, 0x415d, 0x0685, 0x42f3, 0x4321, 0x400e, 0x4041, 0x3c1d, 0x41c8, 0xbacf, 0xa885, 0x425b, 0xa833, 0xb0ed, 0x4111, 0x408a, 0xbf60, 0x4048, 0xb2e6, 0x314a, 0xb21c, 0x2119, 0x4349, 0x1d8f, 0x43e4, 0x1ce5, 0xb2db, 0x4281, 0xbf3c, 0x1907, 0x431e, 0x41f2, 0x4309, 0x413f, 0x3c15, 0x1e57, 0x1875, 0x2a86, 0x1cde, 0x444d, 0xb26f, 0x40ad, 0x3408, 0x05f4, 0x42b4, 0x418f, 0x1bdb, 0x437c, 0xb29e, 0xa7c4, 0x44b8, 0x4254, 0x4198, 0x4568, 0xbf29, 0xba58, 0x42e5, 0x413d, 0xbae2, 0x42d2, 0x44a0, 0xb2b6, 0x431a, 0x3165, 0x1d28, 0x0b29, 0xb234, 0x409b, 0xb296, 0xbf5d, 0x13ee, 0x4310, 0x4015, 0x42b6, 0x1919, 0x21c3, 0xb249, 0x43dc, 0x459d, 0xbfc0, 0x4073, 0x4251, 0x428e, 0x42a6, 0xb212, 0xb20d, 0xbfc0, 0x40b8, 0xba7e, 0x42d8, 0xb234, 0x05c8, 0x4120, 0x1a7d, 0xb05f, 0x4608, 0x411a, 0xbf47, 0xba36, 0x19e1, 0x4019, 0x0eb2, 0x4090, 0x43f4, 0x42a7, 0xba3a, 0x0080, 0x42b9, 0xba72, 0xbf90, 0xb0ff, 0xb295, 0xa5d3, 0x4215, 0x36ac, 0x40be, 0x1a4a, 0xbad4, 0x09b5, 0xba41, 0x465c, 0xba6f, 0xb25f, 0xbfa2, 0x14b7, 0x09c9, 0x1ee3, 0x4397, 0x280b, 0xbf53, 0xb230, 0x42c9, 0x1896, 0x4454, 0xbafa, 0x0535, 0x42f0, 0x395d, 0x1d22, 0x4191, 0x4481, 0x1b2e, 0xba36, 0x4103, 0x1a29, 0x2b7c, 0x326a, 0x407a, 0x42a4, 0x0308, 0x4673, 0xbfd0, 0xb24d, 0x184f, 0x439d, 0xba62, 0x42f7, 0xbf93, 0x24a2, 0xb2d0, 0xb0f4, 0x13d7, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x7f786d81, 0xed0589af, 0xd3e5419b, 0x0129696d, 0xe25706db, 0x2c5ca927, 0x6d81e353, 0x64845d90, 0x7114f20e, 0x7890cd21, 0xe05b5922, 0xe345d064, 0xda1b31e1, 0xa9b62e07, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x00000000, 0xb6a9df2b, 0x00000000, 0x000000a2, 0x00000000, 0x21d44956, 0xffff6d53, 0x347f7fc7, 0x188dd287, 0xa9b62bdf, 0x00000000, 0xffffffff, 0xa9b62c9f, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x1e54, 0x40d8, 0x2f66, 0x276c, 0x416f, 0x1987, 0xb06a, 0x45ec, 0xba67, 0x408a, 0xba2f, 0xacd5, 0x29cf, 0x1872, 0x111e, 0x1b04, 0xbafd, 0x41a5, 0xbf97, 0x43df, 0xb06f, 0x1c82, 0x268b, 0xb2ba, 0xb08a, 0x407f, 0xbaea, 0x1d52, 0x06cb, 0x41e7, 0x4475, 0x43b8, 0xba6b, 0x1fb1, 0xbf3e, 0x4241, 0x3d38, 0x4120, 0x2a6b, 0xb217, 0xbfb8, 0x1f4e, 0xb218, 0xb2cc, 0x1ad6, 0x4128, 0x169b, 0xaa87, 0xaf67, 0xb2aa, 0xbfc0, 0xb0e8, 0x33a9, 0x43fb, 0x195b, 0x4091, 0x3613, 0x00c8, 0xbf09, 0xbff0, 0x0ab2, 0x4211, 0xbadc, 0x4100, 0xba4d, 0x0f34, 0xba3f, 0xb250, 0x1425, 0x1ad6, 0x1f20, 0xb0e6, 0xae48, 0x41f4, 0x4117, 0xb0d0, 0xba25, 0xb2f9, 0xba7d, 0x4063, 0x1e53, 0x40ef, 0x401c, 0x1b23, 0x059d, 0x4021, 0x4126, 0xbf0c, 0x3318, 0x4087, 0x4135, 0x430e, 0xbf69, 0x18c0, 0xb285, 0x4164, 0xbf90, 0x40e9, 0x43bf, 0xba12, 0x2a34, 0x4009, 0x4593, 0xbf06, 0x07c3, 0xb295, 0x369a, 0x41f4, 0x434a, 0xb01e, 0x42a5, 0x0343, 0x2ad7, 0x1be0, 0xbfce, 0x1bdd, 0x4484, 0x439c, 0x461b, 0x10f4, 0x2207, 0x0503, 0x40ab, 0xbf60, 0x4108, 0x4271, 0x1c07, 0x40d7, 0x1820, 0x01a5, 0x3519, 0xb283, 0xb0c1, 0xb28f, 0xb241, 0x238f, 0x43da, 0x2f61, 0xbfb2, 0xa2bd, 0xbae8, 0x410c, 0x46e2, 0xb012, 0x1e36, 0xb236, 0x45ba, 0xb23b, 0x2e93, 0x4369, 0x18a6, 0xb257, 0x4340, 0xba50, 0x4240, 0x4181, 0x1a86, 0xba64, 0x0cb1, 0x4065, 0xb21a, 0x1b54, 0xbf70, 0x462e, 0xb236, 0x01af, 0x41c3, 0xbf89, 0x4324, 0x4357, 0x41ae, 0x4654, 0x408c, 0x1c4d, 0x43ca, 0x405e, 0x19cc, 0x429b, 0xb026, 0x4337, 0x404b, 0x1c2f, 0xba75, 0x42f1, 0x1e97, 0xb27f, 0x425d, 0x049c, 0xb2e2, 0x1385, 0xb270, 0x1eb3, 0xa556, 0xbade, 0x4064, 0x22e3, 0xbfd1, 0x1cf0, 0x0f0c, 0x4266, 0xb0e2, 0x02c3, 0xb092, 0x1212, 0x3b16, 0x4349, 0x458d, 0x317f, 0x4263, 0x19ee, 0x3fde, 0x085c, 0x0f9e, 0x21c4, 0x41e8, 0xb27f, 0x1ae4, 0x425c, 0x2207, 0x43b9, 0xba3e, 0xb24a, 0xbfcc, 0x40b9, 0x40df, 0x4026, 0x415a, 0xb2db, 0x41e2, 0xbae2, 0x410a, 0x420e, 0xbfa5, 0x43d6, 0xb237, 0x40a9, 0x407f, 0x4075, 0xb23b, 0x1dbb, 0x4021, 0x16c5, 0x430d, 0xb258, 0x1fee, 0x44a5, 0xb03f, 0x4367, 0x43e4, 0x435a, 0x37aa, 0x1bd9, 0xbf08, 0x4143, 0xaea3, 0x2a14, 0x41ef, 0xb076, 0x1e69, 0x4644, 0x1971, 0x2fe4, 0x0aef, 0x3651, 0x43e0, 0x4171, 0x1c89, 0x417b, 0x4354, 0x43fa, 0x418d, 0x411b, 0x425a, 0xbf3a, 0x42b3, 0xa73a, 0x4396, 0xbfc0, 0xb258, 0x429c, 0x44c8, 0x4195, 0xba65, 0x46ed, 0xb0ec, 0x18f8, 0x43b9, 0xb246, 0x4306, 0x465f, 0xb2d4, 0xbf7e, 0x37f6, 0x4173, 0x4685, 0x0536, 0x3e40, 0x0691, 0x2372, 0xbf2e, 0xb213, 0xba2c, 0x1c23, 0x41e6, 0x30e0, 0x4062, 0x4030, 0xb229, 0x43d9, 0x2e16, 0x25b3, 0xb2ae, 0x439f, 0x18bb, 0x16b6, 0x4048, 0x4418, 0x42f7, 0xa615, 0xb008, 0xb2d6, 0x1958, 0x1b00, 0xb0d0, 0x4281, 0xb225, 0xab07, 0xbf82, 0x4276, 0x2565, 0x4156, 0x43eb, 0xb2fe, 0xba25, 0x4143, 0x435c, 0x41cf, 0x428c, 0xb2b4, 0x1772, 0x4398, 0x1b8d, 0xb016, 0x435d, 0xba5c, 0xbf63, 0xba66, 0x4100, 0xba5f, 0x43ac, 0xbfb1, 0x41c5, 0x3555, 0x3b44, 0x4305, 0x3e89, 0xba59, 0x1f34, 0x1828, 0x3d4b, 0xa344, 0xb2c2, 0x3ca8, 0x42ea, 0xbf88, 0x4250, 0xb2ac, 0x43fd, 0xb02a, 0x30bb, 0x0fba, 0x43b6, 0x4336, 0x432b, 0xb256, 0x432f, 0xb0bf, 0x4353, 0x4069, 0x095a, 0xa6cb, 0xbf47, 0x3074, 0x165b, 0x1fe4, 0x2ed8, 0xb272, 0x43d0, 0x43a8, 0x2770, 0xb05c, 0x4279, 0x4055, 0xbfd0, 0xbf84, 0x41aa, 0x43ec, 0x1daf, 0x4663, 0x40c5, 0x34e8, 0x447f, 0xa971, 0x145d, 0x00c3, 0x4281, 0x4392, 0xba2d, 0x3677, 0x42eb, 0x42d3, 0x28df, 0xb2c4, 0x4004, 0xa6ed, 0x40fb, 0x1947, 0xbf33, 0xbaee, 0x446b, 0x40b7, 0x402b, 0x460f, 0x1ef9, 0x4130, 0x28f6, 0xad23, 0x4042, 0x2583, 0x45b5, 0x41c8, 0x462e, 0x19f1, 0x4008, 0x4144, 0x421d, 0x4305, 0xba2b, 0x4103, 0xb2fd, 0xb239, 0x4365, 0xbf9a, 0xb219, 0x4461, 0x41a7, 0x43ac, 0xb2a4, 0x416a, 0xbfe0, 0xb09b, 0x4214, 0xae01, 0x2c45, 0x0c53, 0x4107, 0x43b4, 0x4005, 0xb22f, 0x1966, 0xbf60, 0x182c, 0x4011, 0x430e, 0x4291, 0x34de, 0xbfcc, 0xabcf, 0x4255, 0xba1e, 0xba3b, 0x4556, 0x0971, 0x1926, 0xb2ff, 0xaffe, 0x0c75, 0x401a, 0xbf69, 0x4026, 0x402a, 0x4406, 0x4213, 0xba16, 0x41da, 0x43f0, 0x41d3, 0x4312, 0x4228, 0x43fb, 0x44a8, 0x413c, 0x1e14, 0x1924, 0x435d, 0x409c, 0xb27c, 0x4113, 0x405c, 0x4180, 0xba02, 0x42c0, 0xba36, 0x4197, 0x4359, 0x319f, 0xbf82, 0x4092, 0x1fec, 0x1ee0, 0x43fa, 0x4170, 0x45b4, 0x1e51, 0x4109, 0x40e2, 0x2867, 0x30ff, 0x41f8, 0xbf69, 0x17f9, 0x0d33, 0x06dd, 0x1990, 0xa9a2, 0x3666, 0x1d0f, 0xa2cf, 0x4150, 0x4073, 0x217c, 0x1c58, 0x3267, 0x42d9, 0xb217, 0x439f, 0xbafd, 0x4054, 0x1a4c, 0x42e3, 0xbfc9, 0x2489, 0x4620, 0x00d9, 0x41c0, 0xbf90, 0xb23e, 0xb2a1, 0x27f9, 0x423a, 0x417f, 0x2922, 0xbac3, 0x1a8e, 0x1ff5, 0xb220, 0x421f, 0x4378, 0x433e, 0xa84c, 0xb081, 0x1a72, 0x4684, 0xb280, 0x1acb, 0x1d52, 0x43cb, 0x4125, 0x1b4d, 0xbf1b, 0x4190, 0x192c, 0x42fc, 0xaf1e, 0xbf3f, 0x42a6, 0x1ef6, 0x1a9b, 0xb076, 0x30ec, 0xb25b, 0x19f6, 0x29fb, 0x43f2, 0xb2b7, 0x40af, 0x438e, 0x1bbe, 0xb20b, 0x4166, 0x1a51, 0x4140, 0xb250, 0x23de, 0x4062, 0x39c9, 0xbf0d, 0x1ff7, 0xb04e, 0x4670, 0xb233, 0x40ca, 0x4280, 0x17e1, 0x4572, 0x435a, 0xb248, 0x198a, 0x427a, 0x21aa, 0x412d, 0x0d38, 0x1b5c, 0x1cda, 0x40ad, 0x081c, 0xb2ab, 0xbf98, 0x05b0, 0x157a, 0x42ad, 0x43dc, 0x09b2, 0xa299, 0x1def, 0x41dd, 0x2faa, 0x4089, 0x4161, 0x40eb, 0x400d, 0x42e7, 0xbf70, 0x408f, 0x439c, 0x11bc, 0x40c6, 0x1277, 0x2045, 0x426e, 0x420f, 0xbf70, 0xba1a, 0xbfc4, 0x088e, 0x4437, 0x19aa, 0xb095, 0xba24, 0x1d2b, 0x19d9, 0xb200, 0xb26c, 0xbac0, 0x42c3, 0xbf19, 0x3ab5, 0x185a, 0x1e03, 0x42bd, 0xb040, 0x456d, 0x0e83, 0x4244, 0x4001, 0xb257, 0xb255, 0x4016, 0x21f4, 0x134a, 0x1c09, 0xb2eb, 0xb2d5, 0x103c, 0x412f, 0xb2f8, 0x4645, 0x41ca, 0xbad7, 0x3c4c, 0x43f2, 0x4469, 0xbf7f, 0xb096, 0x404c, 0xb20e, 0x408f, 0xb293, 0x44f4, 0xb2ac, 0x4300, 0xb0d5, 0xb044, 0xb298, 0x401d, 0x40a5, 0x0257, 0xb2bd, 0x427d, 0x42d6, 0x432a, 0x41e0, 0xbf48, 0x3377, 0xb282, 0xabfc, 0xb0a9, 0xb2cf, 0x1ba6, 0x45c8, 0x4080, 0x1e90, 0xb0f0, 0x462e, 0xb229, 0x02cf, 0x41ba, 0x41b8, 0x4210, 0x3f35, 0x398c, 0xba39, 0x413b, 0xbf64, 0x21a7, 0x40b8, 0x0509, 0x4388, 0x20cc, 0xba62, 0x41b9, 0x1efd, 0x41c7, 0x413e, 0x40a3, 0x4136, 0x1b72, 0x4366, 0xba01, 0xb2e0, 0x428b, 0x438e, 0xa279, 0x43d2, 0x1b4e, 0x43ac, 0xbf99, 0x0ff8, 0x36ad, 0x46c3, 0x43e4, 0x42ae, 0xba6e, 0x368c, 0x12b7, 0x43cf, 0x405e, 0x1e52, 0xb28e, 0x0282, 0x280c, 0x4294, 0x2cf6, 0xbfd8, 0x418c, 0xa006, 0x4179, 0xb2b4, 0x415d, 0x0cba, 0x1502, 0x46da, 0x43cf, 0x42bd, 0x411a, 0x4037, 0x4265, 0x4268, 0xbfa9, 0xa149, 0x1de5, 0xb2d1, 0x416a, 0xb0af, 0x4390, 0x4302, 0xbf37, 0x36bf, 0x40ef, 0xaa8e, 0x1731, 0x2537, 0xaa7a, 0x439d, 0xbfdd, 0x416c, 0xb271, 0xb0a0, 0xb0cb, 0x462a, 0xa0ff, 0xaf7c, 0x4338, 0x41c6, 0x40e4, 0x430f, 0xbfae, 0x24ea, 0x41d3, 0x1baf, 0xbf0a, 0xb2b5, 0x415b, 0x4574, 0x42dd, 0x1886, 0xb298, 0x40e3, 0x23f6, 0x43b1, 0x4179, 0x421d, 0x00f3, 0x44fb, 0x43f2, 0xb2da, 0xbf43, 0x329d, 0x40df, 0x41f5, 0x43a4, 0x410f, 0xb070, 0x40c3, 0x0290, 0xba76, 0x40c0, 0x03b8, 0x449a, 0x1b97, 0x0f78, 0x419b, 0x422e, 0x179f, 0x4123, 0xa591, 0x1e2b, 0x1fda, 0x24bb, 0x412f, 0x4204, 0x4331, 0x40cc, 0x3aa9, 0xbfd7, 0x443b, 0x43ac, 0x130f, 0x4328, 0x2575, 0xba56, 0xb246, 0xa7fe, 0x46ac, 0x3801, 0x43dd, 0xba2b, 0x39e7, 0x0bd0, 0x42b3, 0x4038, 0x41f9, 0x4393, 0x4102, 0xb203, 0xb2f8, 0xa22e, 0xa526, 0xb08e, 0x2e59, 0x2b55, 0xb25f, 0x4663, 0xbf3e, 0x4005, 0x43c8, 0xb034, 0x3d71, 0x3d1d, 0xb258, 0x4322, 0x43a8, 0x4052, 0x2a11, 0x434d, 0xba44, 0xb283, 0x0014, 0x18b3, 0xbf85, 0x4236, 0x370f, 0xa4a4, 0x4081, 0xbfae, 0x3107, 0x4246, 0x4159, 0x40f4, 0xb262, 0x1bd6, 0xb275, 0x41ea, 0xbf8e, 0x40f6, 0x4004, 0xba73, 0xbf01, 0x42a3, 0x419b, 0x418a, 0x4309, 0x4300, 0x4053, 0xa24b, 0x43b2, 0x4105, 0x417a, 0x42ee, 0x3a99, 0x41b6, 0x1c97, 0x43e8, 0xbf95, 0x421b, 0xb22f, 0x468c, 0xba30, 0x171d, 0x42e5, 0x4283, 0xbf78, 0x1e0f, 0xb02e, 0x3bce, 0x46f2, 0xbaeb, 0x1a73, 0x1d12, 0xba2a, 0xa2cb, 0xb23a, 0x42a8, 0xa26b, 0x3c45, 0x0a4d, 0x1ce2, 0x1a89, 0xbfa3, 0x462f, 0x45f1, 0x0ccd, 0x19fc, 0x4432, 0xba68, 0x402c, 0x43bd, 0x4106, 0x2d24, 0x08ae, 0x41ee, 0x4395, 0x438a, 0x410d, 0x414e, 0x40c1, 0x41f5, 0xbae6, 0x2a67, 0x0307, 0x1b7c, 0x42fa, 0xba7a, 0x40e7, 0xbad0, 0xb218, 0x4551, 0xbfbc, 0xbfc0, 0x44e1, 0x19e1, 0xb2fc, 0x423b, 0xbf07, 0xb295, 0xb295, 0x4409, 0x42c5, 0xb285, 0x3d52, 0xbf66, 0xbf70, 0x4244, 0xba58, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x35dcd53f, 0x329b21b1, 0x5b679e3c, 0x95697b24, 0xded610ce, 0x5fb2bf66, 0xe3f3845c, 0xe4e7d67e, 0xb27d6955, 0x52a7f817, 0xf1c67637, 0xa8f2cd39, 0x50a9900e, 0xbf067896, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0xe8fef8ea, 0x08800000, 0x20020000, 0xfee8eaf8, 0x00000000, 0x0000eaa6, 0x00000000, 0x02200000, 0x0525616d, 0x52a7f817, 0x00000000, 0x052577df, 0x01171508, 0x00011300, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x2155, 0x1e10, 0x3dda, 0xa4a1, 0x40e5, 0xba3b, 0x40e6, 0x1cbe, 0xbf21, 0x3982, 0x403a, 0x3b83, 0x40d0, 0x1dbd, 0x4633, 0xbf27, 0x40fa, 0x4217, 0xba03, 0xa0e8, 0x4051, 0x43d7, 0xb2ad, 0xb23a, 0x4456, 0x40ca, 0xb28a, 0x4598, 0x414f, 0x183b, 0xbf47, 0xb04c, 0x2d55, 0xb200, 0x410d, 0x423c, 0x427c, 0x43e2, 0xbf75, 0x438e, 0x420c, 0xb040, 0x431f, 0xaaec, 0x1ea1, 0x4219, 0x0e67, 0x4107, 0x258a, 0x4363, 0x4209, 0xbfa0, 0x4244, 0x1fed, 0x082e, 0xa5ef, 0xba10, 0xb215, 0xbf60, 0xbadd, 0x441f, 0x4360, 0x4416, 0x0ced, 0x4396, 0xb2a5, 0xbfbd, 0x020b, 0x10f7, 0x40bc, 0x43fa, 0xba04, 0xb203, 0x424b, 0x3ef2, 0xb21f, 0xb2db, 0x18b7, 0x41eb, 0x32b6, 0x4415, 0x405d, 0x19be, 0x124d, 0xb067, 0x4048, 0xbf7e, 0x4163, 0x4494, 0x2c27, 0x41a4, 0x4094, 0x417b, 0xbadb, 0xb221, 0xac56, 0x4045, 0xbf00, 0xbf9e, 0xb02c, 0x1467, 0xa9eb, 0xb28c, 0x42f0, 0x42f1, 0x2b99, 0x10b3, 0x4181, 0x4565, 0x16bf, 0xbaf0, 0x3ce0, 0x0fdb, 0x182c, 0x4008, 0xb253, 0x405c, 0x4291, 0x427d, 0x1b26, 0x4275, 0x424d, 0x400e, 0xbf97, 0x24cc, 0xb0ac, 0x00e5, 0x4646, 0x2dad, 0xbfd5, 0x416b, 0x43dc, 0x15af, 0x405f, 0x42ae, 0xaffc, 0x435c, 0x407e, 0x25a2, 0xb0a1, 0x26af, 0x4440, 0x2f9f, 0x0c00, 0x44e2, 0x4079, 0x0965, 0xb21f, 0x43b0, 0x081f, 0x441c, 0xbf5c, 0x4350, 0x1b32, 0x41d9, 0x46d8, 0xba76, 0x42cf, 0x426e, 0x438e, 0x0cf5, 0xb00b, 0x414f, 0xba7f, 0x2ea0, 0x4601, 0x43de, 0x2e67, 0x0090, 0xb095, 0x43b4, 0x1edf, 0x4695, 0xb264, 0x0acf, 0xbf11, 0x42f8, 0xa71c, 0x42c7, 0xb29e, 0xb223, 0x08a7, 0x4425, 0x40c5, 0x4342, 0x466f, 0x1e7a, 0x425e, 0xba7c, 0xbaef, 0x4195, 0x40f0, 0x4105, 0x182a, 0xb249, 0x01b6, 0x43db, 0x33cd, 0xb098, 0x22ef, 0x4670, 0x1c06, 0x4143, 0xb2ad, 0xbfdd, 0x44ea, 0xba3e, 0x43d7, 0xb251, 0xba39, 0xba0d, 0x40bf, 0x46a2, 0x20a1, 0x41ab, 0x3578, 0xafc6, 0x19e4, 0x4127, 0xb06d, 0xacfe, 0x1d9e, 0xbf31, 0x43da, 0x0408, 0x42e6, 0xb2ca, 0x400e, 0xa0f7, 0x42c4, 0xb24e, 0x401f, 0xb220, 0x4494, 0x4410, 0xbf15, 0x417b, 0x42eb, 0x151d, 0x4295, 0x4188, 0xbae2, 0x40d5, 0x40c3, 0x1d6c, 0xa78d, 0x4059, 0x46ad, 0x1903, 0x1b14, 0x407f, 0xb249, 0x43cf, 0x415f, 0x4423, 0x410e, 0xba10, 0x0aac, 0x40ae, 0xa148, 0x0bda, 0xbfb6, 0x4161, 0xb0c0, 0xb2b3, 0xacde, 0x43b8, 0x40c4, 0xb2b5, 0x4476, 0x41cf, 0x4245, 0xbf2a, 0x1c2a, 0x34fe, 0x089d, 0x1943, 0xa5a5, 0x4611, 0x4013, 0x305a, 0x1a16, 0x43f1, 0x4118, 0xb2a5, 0x44a3, 0x1aa6, 0x4055, 0x412d, 0xb2c1, 0x43cf, 0x0ae2, 0xbf28, 0x4299, 0x42ac, 0xb203, 0xa899, 0x4160, 0xbf56, 0xbafe, 0x40d0, 0xb2ea, 0xb2b1, 0x1f3a, 0x1cf9, 0x43fd, 0xbf76, 0xba48, 0xbaca, 0xb2d1, 0x4268, 0x43a9, 0x465a, 0x1f70, 0xb0a5, 0xb021, 0x4016, 0xae2f, 0x4012, 0x3ad1, 0x402f, 0x3c2d, 0x425e, 0x40a7, 0xba78, 0xbf2f, 0x226a, 0x44a0, 0x417a, 0x040e, 0x0659, 0xb288, 0x41b4, 0xba78, 0x0a09, 0x4165, 0x2bfb, 0xbf05, 0x4222, 0x1bfa, 0xa40a, 0x43ab, 0x41db, 0x43e3, 0x419f, 0x4694, 0x43d0, 0x35e8, 0x1f83, 0x4116, 0x4206, 0xac92, 0x4034, 0x220e, 0x413e, 0xad23, 0x4116, 0x4430, 0xbf60, 0x0e3f, 0x1c57, 0xac22, 0x3c5b, 0xb208, 0x098b, 0xbf0c, 0x42db, 0x3b41, 0x435e, 0x4302, 0xbf19, 0x3fc7, 0x4222, 0xbac3, 0x1528, 0x427e, 0x428a, 0xb2ce, 0xb09d, 0x41f4, 0x435c, 0xba54, 0xbf3f, 0x2dc4, 0x1526, 0x4335, 0xba05, 0xbf5f, 0xb252, 0x4431, 0x2474, 0x4260, 0x4278, 0x41eb, 0x43f7, 0x41bc, 0x4274, 0x2a23, 0x1ea8, 0xb024, 0x163b, 0x4365, 0x3c89, 0x424f, 0xba1c, 0x43d1, 0x42ac, 0xbac7, 0x405a, 0xb2d1, 0x1d69, 0xb2a2, 0x41bf, 0x412b, 0x42f5, 0x141d, 0xbf5b, 0x43f7, 0x0b65, 0x4124, 0xbacc, 0x414a, 0x39dc, 0x4290, 0x160f, 0xb2c6, 0x46b2, 0x43b8, 0xbf90, 0x417c, 0x41ea, 0x4280, 0x438c, 0x40b9, 0x1fc0, 0x0291, 0x46e2, 0x432a, 0x4206, 0x2149, 0x06e2, 0xb218, 0xba0d, 0x3fc3, 0xbf68, 0x46c8, 0x4166, 0x437d, 0x41a0, 0xb00f, 0x26f5, 0x42b6, 0x466c, 0xaf14, 0xba28, 0x0364, 0x0f78, 0xa3a4, 0xbf6e, 0x4572, 0xb2fe, 0xb046, 0x413c, 0xbaf3, 0x4370, 0x4005, 0x2268, 0x42af, 0x4278, 0x4089, 0x439f, 0xbfd6, 0xaed2, 0x4305, 0xb2e2, 0xb2f6, 0xbf18, 0x42a3, 0x1ee9, 0x41bc, 0x1d48, 0x4313, 0xa6e2, 0x407a, 0xb09e, 0x459c, 0xb24b, 0xb015, 0xaa03, 0x18b9, 0xacea, 0x1e86, 0x4169, 0xbfc8, 0x1f6a, 0x4396, 0x347f, 0x182d, 0x3184, 0xba54, 0x221f, 0x40d0, 0x337e, 0x4272, 0x40c0, 0xb0de, 0x430d, 0x4105, 0x315e, 0x11cc, 0xbf90, 0x228a, 0x4273, 0xbada, 0x006d, 0xbf2a, 0x4334, 0x18c8, 0xb2d9, 0x1cf8, 0xba7c, 0xbaef, 0x4461, 0x2973, 0x2d26, 0x4042, 0xaba4, 0x4186, 0xb05c, 0xbfd4, 0x4254, 0x1e78, 0x4643, 0xb29a, 0x427a, 0x4370, 0x1f71, 0x419a, 0xa4c1, 0x07c9, 0xbf80, 0x3f05, 0x405f, 0x3150, 0xbf37, 0x40ce, 0xba27, 0xb2a0, 0x2398, 0xbfca, 0x43e4, 0xaa17, 0xba74, 0xb0d6, 0xb08f, 0x42c9, 0x42cd, 0xb224, 0x1ccf, 0xafbc, 0x4060, 0xb0f5, 0x423d, 0x43c4, 0xb2fc, 0xbac2, 0x4017, 0xb265, 0x431a, 0xbfca, 0x43a8, 0x415f, 0x40ff, 0xb0d7, 0xbae4, 0x1b93, 0xb2e9, 0x1a43, 0x28ac, 0x427c, 0xb215, 0x4017, 0xbf3d, 0xb2d0, 0x42be, 0x1d8f, 0x447c, 0x40ec, 0xba4e, 0x1910, 0xb041, 0x1fdd, 0x41b7, 0x40c9, 0x42c8, 0xbf60, 0x404f, 0xb27d, 0x4661, 0x4399, 0x43eb, 0xb20a, 0x460b, 0x1c22, 0xb26f, 0xbae0, 0xbff0, 0x421f, 0x1f27, 0x4264, 0x4290, 0xbf93, 0xb2a4, 0xb2a7, 0xbaff, 0xad42, 0x4391, 0xba27, 0x40f4, 0xbfd6, 0x42f8, 0x429f, 0x03c8, 0x40c6, 0x32d1, 0x40c8, 0xaf82, 0x02cf, 0x046b, 0x4285, 0x4291, 0xba45, 0xbadd, 0x40f2, 0x33e6, 0xba09, 0xba1e, 0x4607, 0x45d9, 0x4453, 0xb00d, 0x4222, 0x40b0, 0xbf19, 0xb20c, 0x1ab7, 0x41dc, 0x439e, 0x4221, 0x1f89, 0x400c, 0xb288, 0x42f2, 0x4543, 0x3d97, 0x43e3, 0xba58, 0xaa3d, 0x42d0, 0x1870, 0x26b1, 0xbfab, 0x4112, 0x42d9, 0x467c, 0x4052, 0x4631, 0xb286, 0xbf5e, 0x43c5, 0x44d0, 0x2fb1, 0x43b3, 0xb2ea, 0x1d46, 0x4076, 0x415a, 0x3bbe, 0xbf03, 0x0812, 0x2d4f, 0x1f62, 0x4322, 0x0a58, 0x40e4, 0x1fcd, 0xba52, 0x15b6, 0x43b0, 0x43a0, 0x400a, 0x4303, 0x4297, 0x40e7, 0xb0cf, 0x43e8, 0x15c1, 0x4134, 0x4248, 0x4576, 0x0870, 0x01c4, 0xbf63, 0xbac5, 0x42cd, 0x437d, 0x43e7, 0xb261, 0x42e2, 0xbf18, 0xbac0, 0x4037, 0x40d0, 0x1c84, 0x4149, 0xbf80, 0xb2c6, 0xbfe0, 0x30f1, 0xa467, 0x2e0e, 0xbf6b, 0x46e2, 0x4216, 0x43df, 0x1ab8, 0xbf23, 0x46d3, 0xba71, 0xb208, 0x099f, 0xbf60, 0xba1f, 0x40a5, 0x42d4, 0x21cb, 0x1ffd, 0x40c9, 0x42dc, 0x15f1, 0x279d, 0x2437, 0x0cea, 0xbfdd, 0xbae4, 0x438b, 0x4378, 0x463c, 0x0a74, 0x4005, 0xba1e, 0x41d7, 0x4481, 0x4274, 0x45da, 0xbf46, 0xb0e9, 0x3f19, 0x1bb1, 0x4581, 0x42ce, 0xb2df, 0xba6f, 0xb22f, 0x343f, 0x40d8, 0xae4d, 0x4225, 0x404d, 0x3748, 0x4641, 0x1924, 0xba34, 0xbf27, 0x42ba, 0x1e38, 0x40c4, 0x1e29, 0xbfb8, 0xa804, 0xbaf2, 0xbaec, 0x411a, 0x42e3, 0x402b, 0x404a, 0x4654, 0x403c, 0x229e, 0x4343, 0x4278, 0x420e, 0x414e, 0x1b28, 0x1ec8, 0x4137, 0xbf74, 0x4132, 0x40db, 0xba2b, 0x424e, 0x400f, 0x4342, 0xa268, 0x45cc, 0x41a1, 0xb210, 0x1e13, 0xba32, 0x405f, 0x4070, 0x40a8, 0x4495, 0xaf59, 0x1a8a, 0x4335, 0x1aaf, 0xa0de, 0x4557, 0x429b, 0xbf2b, 0x1a1f, 0xb24a, 0x20ba, 0x4026, 0x4426, 0x415d, 0xba49, 0x1a0f, 0xa6ba, 0x408e, 0xba37, 0xb067, 0xa1be, 0x43f0, 0xb26f, 0x06e2, 0x4101, 0x424b, 0xb004, 0x17fe, 0x417d, 0x4053, 0x133e, 0xbfa1, 0xa51b, 0xaf80, 0x43e6, 0x4031, 0x0893, 0x42a2, 0x113a, 0xbf86, 0xb211, 0xb2e9, 0x1955, 0xb2e0, 0x4249, 0x1fee, 0x43bf, 0x1f72, 0xb29c, 0x41ce, 0x1c6c, 0x1ac1, 0x4216, 0x4309, 0xbf90, 0x4176, 0x4114, 0x0c13, 0xb22a, 0x10be, 0x4293, 0x1255, 0x40dd, 0x1f80, 0xba69, 0xb298, 0x430c, 0x3114, 0xbfc2, 0x4209, 0x4682, 0x4160, 0x461f, 0x4003, 0x4104, 0x27e8, 0xb27f, 0x13b2, 0x4034, 0x061d, 0xb093, 0x30b2, 0x4252, 0xa0b3, 0xa03e, 0x4489, 0x4458, 0xbad3, 0xbf2e, 0x4301, 0x401f, 0x40ed, 0xba6c, 0x41bf, 0xb017, 0xbf5b, 0x4549, 0x4316, 0xb2ac, 0x2a29, 0x19a6, 0x437d, 0xaba0, 0x3226, 0xa26e, 0x1bde, 0x4143, 0x40a4, 0x46ac, 0x420e, 0x43d6, 0x40d2, 0x40e5, 0x059e, 0x41eb, 0xbfb2, 0xaf76, 0x4194, 0x425c, 0x3ca9, 0xbafa, 0x1cd4, 0x40b9, 0xb29b, 0x439a, 0x3586, 0xb28f, 0x43a5, 0x438b, 0x45ad, 0xb200, 0xbf90, 0x387c, 0x1693, 0x40a5, 0x43c8, 0x065e, 0x37af, 0xbf55, 0x4319, 0x41ee, 0x1ee5, 0x1d3e, 0x43f1, 0x1d52, 0x462d, 0x0129, 0xbae8, 0x4613, 0x1d81, 0x2a54, 0xba46, 0x432c, 0x101c, 0xb29a, 0x42c2, 0xbfd0, 0x2520, 0x32e6, 0xba32, 0xba5a, 0x40ef, 0xbf70, 0xa438, 0xbf48, 0x42cf, 0x4220, 0x40f8, 0x0b4e, 0x434e, 0x1909, 0x417e, 0x42ee, 0xbf02, 0xbfd0, 0x157f, 0x408d, 0x42dc, 0xba68, 0xb219, 0xb075, 0xbfa5, 0x2f93, 0x1fb8, 0x40f1, 0xb240, 0xb2b3, 0x2097, 0x42f4, 0x188b, 0xb0e8, 0x4326, 0x4157, 0x2694, 0x404c, 0xbaf8, 0x431a, 0x4308, 0x2760, 0x359d, 0xb290, 0x1cc4, 0x4133, 0x1771, 0x408f, 0x4159, 0xba63, 0x4222, 0xbf7d, 0x44f8, 0x4088, 0xbad6, 0x0287, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x8f725d16, 0x8f2d1fe5, 0xd2b28a66, 0x47a47346, 0xc0b4401b, 0xf8e7fb12, 0x4ba63b76, 0xa9e6ad4c, 0xa1d2cdf2, 0x35a0c989, 0xc174909a, 0x846af94c, 0xe89071ad, 0x3db39c20, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0x0000414c, 0x00000000, 0x0000414c, 0x00004f41, 0x0000414f, 0x000000bd, 0x00004c41, 0x00000060, 0x08d61086, 0x35a0d48c, 0x00000000, 0x846afcc2, 0x00000000, 0x17fffbb8, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xbafe, 0xb286, 0x0988, 0xb25a, 0x4061, 0x1ed9, 0x4663, 0x430f, 0xb280, 0x2590, 0xb0a2, 0x43b0, 0x0a42, 0xa6b5, 0x4171, 0x3320, 0x4145, 0xb286, 0xba32, 0x1307, 0xbf7c, 0x435e, 0x1bc9, 0x41ff, 0x1e42, 0xb09a, 0x444e, 0x40b1, 0x4016, 0xbaca, 0x411c, 0x2fb8, 0x431a, 0xbf01, 0x4360, 0x41d8, 0x1e44, 0x1331, 0x3c8c, 0xb04e, 0x4296, 0x4548, 0xbf2e, 0x2139, 0x403d, 0x197a, 0xbf90, 0x1c14, 0x4472, 0x3fa6, 0x0f32, 0x21ac, 0x407e, 0xbaf8, 0xb255, 0xba53, 0x1b89, 0xbf65, 0x18ef, 0xba18, 0x405b, 0xb0bd, 0x0ebd, 0xb204, 0xba30, 0x413b, 0x4143, 0x4316, 0x41e9, 0x4367, 0x41c5, 0x1673, 0x103e, 0x09a5, 0x2ef0, 0xb263, 0x43e6, 0x42ec, 0x1c44, 0x082b, 0x45c5, 0x439b, 0xb2e9, 0xa969, 0xbf7b, 0xbfd0, 0x438b, 0x4132, 0xb0ec, 0x431b, 0x34ab, 0x0f49, 0xa401, 0x4124, 0xacc3, 0x4314, 0x1cf2, 0x3c2e, 0x309c, 0xb0be, 0xb0a3, 0x40e6, 0x4314, 0x41ab, 0x122b, 0xb2c2, 0x13c9, 0x423d, 0x423e, 0xbfce, 0x4175, 0xaa5d, 0x0876, 0x1c1c, 0xbad3, 0x4248, 0xb298, 0x0069, 0x16ba, 0x40c0, 0xbfc1, 0x43c4, 0x17a8, 0xa13f, 0x41a8, 0x40d6, 0x4680, 0x4319, 0xba2a, 0x44b8, 0x4470, 0xb0c8, 0xbf6d, 0x4303, 0x1c1a, 0x439b, 0xb271, 0xba0b, 0xb236, 0x41a2, 0x4071, 0x431e, 0x395c, 0xadd7, 0xba48, 0x4163, 0x40ee, 0x4195, 0x2da0, 0x19e4, 0x32b4, 0xb2ca, 0xb0b7, 0x45f5, 0xbfbd, 0x3c50, 0x23c9, 0x2092, 0x4424, 0x410a, 0x40d8, 0x43ee, 0xb2a8, 0x1d9f, 0x416e, 0x418c, 0x3fb8, 0x1557, 0xbac8, 0x4659, 0x4131, 0x0f9d, 0x434f, 0xb2a4, 0x41b5, 0xb2a8, 0xbf6c, 0x415e, 0xbacf, 0xb2f5, 0xb20a, 0xb059, 0xbaf2, 0x4003, 0x0bf5, 0x40b9, 0x4480, 0x0151, 0x0a9f, 0x4269, 0x439f, 0xb029, 0x4224, 0x04a3, 0x4365, 0x4071, 0x4630, 0x2d00, 0x4436, 0x1904, 0x3861, 0xbf71, 0x198a, 0xb2ad, 0x0641, 0x3aad, 0x425e, 0xb2e0, 0x00d3, 0xb030, 0xbf78, 0x4089, 0xba44, 0x1789, 0x0fb3, 0x45ba, 0xb2dc, 0xbf3a, 0x186d, 0x42c9, 0x432b, 0xb2ea, 0xb0cf, 0x42d5, 0xb22c, 0xb21e, 0x2355, 0xba01, 0xa56e, 0x41fd, 0xbad1, 0x2bcf, 0x1b37, 0xb2a4, 0x411f, 0x4018, 0x4342, 0x20f4, 0x43c4, 0x43be, 0xbfa1, 0x4139, 0x468b, 0x42c4, 0x1b6e, 0x21aa, 0x42a3, 0xb0cd, 0xb0d2, 0x431e, 0x431d, 0xbfa0, 0x428d, 0x35d6, 0xbf32, 0xae52, 0x431c, 0x1e62, 0xbaf6, 0x4099, 0xb015, 0x4122, 0x19f9, 0x01c1, 0xb2ef, 0xa849, 0x2614, 0xb24a, 0x02c0, 0x43ec, 0x0216, 0x42a0, 0x42c1, 0x43e2, 0x43d9, 0x1ae7, 0x1285, 0xb23e, 0x1d33, 0x187a, 0x3872, 0x4661, 0x3d9c, 0xbfaa, 0x4033, 0x4089, 0x41c3, 0x328b, 0x43d5, 0xb2ab, 0x06f1, 0xbaf9, 0x432e, 0x3160, 0x416a, 0x2c76, 0x4232, 0xbfe8, 0xbf70, 0xbf4f, 0xb20a, 0xb29a, 0x16b0, 0xb2f3, 0xb02f, 0x1ed3, 0x42e1, 0x44f8, 0xb279, 0xba78, 0xbff0, 0xb2ae, 0x433e, 0x388b, 0x3cac, 0xbf3f, 0xb2ab, 0x31d8, 0x34a0, 0xaa1a, 0xb22d, 0x3df9, 0xb28a, 0x43d6, 0x37ab, 0xb228, 0xbf1b, 0x41d9, 0x062a, 0xb256, 0x1c7f, 0xa48c, 0x441d, 0x403b, 0x20fc, 0x1841, 0x4260, 0xb280, 0xbfb0, 0x41ee, 0x4340, 0xb0dc, 0x415d, 0x41c7, 0x41c2, 0x4565, 0xb239, 0xba36, 0x435e, 0xb2c3, 0x0432, 0x434a, 0xbfd7, 0x4127, 0x26c1, 0x38e4, 0x4210, 0x400c, 0xb0ac, 0x46f9, 0xba4d, 0x4331, 0x2faf, 0x4088, 0x4014, 0x2e91, 0x42e5, 0xb0dc, 0xbfb3, 0x0e14, 0x42b4, 0xb251, 0x45a4, 0xbf13, 0x4460, 0xba07, 0x2702, 0x406e, 0xbfc0, 0x40d8, 0x3d8d, 0xb231, 0xb067, 0xb0cf, 0xba32, 0x4334, 0x29fc, 0x4051, 0x4068, 0x4617, 0x361f, 0x4013, 0x42c3, 0x1ff9, 0x437f, 0xba51, 0xb24f, 0x4150, 0xbf4f, 0xb2b8, 0x1a70, 0xaf84, 0x410e, 0xb207, 0x4041, 0x42f2, 0x4637, 0x00ae, 0x41b1, 0x411f, 0x4333, 0x41ae, 0x1d1b, 0xbf5b, 0x4313, 0x393e, 0xb277, 0xbfc0, 0x4103, 0x18c6, 0xae36, 0x429a, 0x42c7, 0x464b, 0x1569, 0x035a, 0x41c4, 0x3350, 0x209f, 0x43c8, 0x416e, 0x4305, 0x1daa, 0x43a6, 0x4325, 0x4248, 0x3079, 0xbf01, 0x4072, 0xafc0, 0xb299, 0x42a7, 0x422d, 0x40f4, 0xb2d7, 0xb2d8, 0xb006, 0x4318, 0xbf61, 0xb2c3, 0x2017, 0x4172, 0xa800, 0x43b6, 0xba42, 0x4367, 0x43a3, 0xb08f, 0x4267, 0x43c8, 0xb0f0, 0x0970, 0xbacc, 0x1b86, 0x4177, 0x439a, 0x41ce, 0x4053, 0x4307, 0xb29b, 0x3708, 0xb29f, 0x25a5, 0xb279, 0x4329, 0xb2df, 0xbf3c, 0x40c9, 0x435c, 0xbfd9, 0x4180, 0x01f8, 0x37cd, 0x4087, 0xbf80, 0x41fb, 0x43c3, 0xba0f, 0xb214, 0x1f01, 0x43e2, 0xbf28, 0xb2ce, 0x4181, 0x19db, 0x4124, 0xbff0, 0x170e, 0xb078, 0x41d6, 0x4325, 0x4033, 0xba21, 0x42b9, 0x15d9, 0x42c2, 0x0491, 0x42b1, 0x40c8, 0xba19, 0x4411, 0xbae6, 0xbac3, 0xbf61, 0x427b, 0x4341, 0xb2ba, 0xb233, 0xb2fa, 0x40c7, 0xb0b5, 0xb2b9, 0xb23a, 0x433a, 0x418e, 0xb240, 0xbf3d, 0x12df, 0x405d, 0x0bfe, 0x1c0d, 0xbff0, 0xb2a2, 0x42dc, 0x08fb, 0x42c4, 0x4129, 0x1eee, 0xba63, 0x41f4, 0xbfdd, 0x4239, 0xa32e, 0xbac6, 0x41dc, 0x1a05, 0x41dc, 0x3616, 0x4401, 0x40da, 0x4278, 0x15fd, 0x19be, 0xafdf, 0xada3, 0xbf22, 0x0b84, 0xbf90, 0x4558, 0x425f, 0x12b9, 0x42de, 0x43a1, 0x31dc, 0x1fb3, 0x1f66, 0x4313, 0x0dee, 0xa10b, 0x4272, 0x0ff7, 0x40cf, 0xbfb0, 0x4614, 0x4263, 0xbfd7, 0xa80c, 0x43d9, 0x4690, 0xba05, 0x40b9, 0x1a89, 0x3326, 0x43eb, 0x0377, 0x43f4, 0x4182, 0xb066, 0x25ac, 0x4076, 0x1848, 0x02ff, 0xbf2b, 0xb291, 0x4646, 0xa320, 0xb077, 0x41b0, 0x412b, 0x422b, 0x0c68, 0xbf62, 0x4139, 0x413d, 0x2f7a, 0x41a3, 0x417e, 0xb0c1, 0x3b6a, 0xb0d5, 0xb2a6, 0x43ef, 0x1a5e, 0x1de5, 0x4215, 0x211f, 0xbfd5, 0x1f96, 0x4694, 0x05d2, 0x2a85, 0x4460, 0x4180, 0x41bc, 0x30a7, 0x419a, 0x401d, 0x26d1, 0xb009, 0x4267, 0x40f1, 0x413e, 0xae3b, 0x4351, 0xa082, 0x3aec, 0xbf49, 0x42b4, 0x3cdf, 0xb031, 0xb225, 0xbf90, 0xb21e, 0xa23e, 0x3d65, 0x2014, 0xa8dc, 0xba0c, 0x4217, 0x17df, 0x4234, 0x18ea, 0x41f5, 0x0639, 0x45e4, 0x34d4, 0x4207, 0x0eb1, 0x421c, 0x408c, 0x405e, 0xa02e, 0xb273, 0x4109, 0x0240, 0x42a9, 0xbfcf, 0x0d28, 0x43ce, 0x2281, 0x430e, 0x2cde, 0xb08c, 0x4013, 0x165a, 0xb0db, 0xad8f, 0x3ed1, 0x111b, 0x461d, 0x08fb, 0x4074, 0x4076, 0x4030, 0x41a4, 0xb210, 0x144d, 0x24dc, 0x4063, 0x1890, 0x4313, 0x38c2, 0x416f, 0xbfc9, 0x270d, 0xb276, 0x4308, 0x1f80, 0x2858, 0x3b1a, 0x397f, 0x3b6e, 0x19f9, 0x43b3, 0x4249, 0xb2c7, 0x1f5a, 0xba0f, 0x42f8, 0xbfc3, 0xb09c, 0x1cc4, 0xa140, 0x4315, 0x4080, 0xb22d, 0x437d, 0x461f, 0xbada, 0x41af, 0xbf3c, 0x306b, 0x43ce, 0xb244, 0xba73, 0x4193, 0x400a, 0x1af3, 0x403f, 0xb26e, 0x1bee, 0x43a8, 0xba08, 0x400d, 0x4215, 0x3529, 0xba5b, 0x2623, 0xba16, 0x4025, 0x40d8, 0x43e9, 0x4225, 0x4112, 0xb2f4, 0x41db, 0x1c09, 0x1ece, 0xbf27, 0x1ac8, 0x4338, 0x4015, 0x4299, 0x4094, 0x4662, 0xba14, 0x43bd, 0x4082, 0x0e05, 0x40c2, 0x413e, 0x43bb, 0x42a6, 0x4146, 0x0cc0, 0x1a0b, 0x1c27, 0x4104, 0xbf93, 0x2799, 0x46ed, 0x1c30, 0xbfc0, 0x422b, 0xa622, 0x1866, 0x15a6, 0xbf80, 0x4334, 0x4060, 0x07a4, 0x14d3, 0xa26f, 0xbae9, 0xbf44, 0x4033, 0xba7e, 0xad85, 0x4045, 0x1bc4, 0x43b8, 0x43ed, 0x10c5, 0x4200, 0x4330, 0x44cd, 0x36e3, 0x4004, 0x4114, 0x438e, 0xba11, 0x4297, 0x19b2, 0x4222, 0xa7bb, 0xbfb4, 0xb0db, 0xba70, 0x430d, 0x4017, 0x1848, 0x1390, 0x4078, 0x4204, 0x42df, 0xba34, 0x414f, 0x4353, 0x4205, 0x01ea, 0xb249, 0xbf17, 0x4208, 0x42a3, 0x456e, 0xba29, 0xbf32, 0x430b, 0x19a9, 0xb09a, 0x41e6, 0xb0a6, 0x401f, 0xa293, 0x40bc, 0x1df0, 0x40df, 0x3ec9, 0x063a, 0x4255, 0xba51, 0xbf34, 0x405f, 0xb0d1, 0x3863, 0x0052, 0x1541, 0xa37a, 0x429d, 0x4285, 0x2a8b, 0xbf8b, 0x429e, 0x1dc2, 0x4193, 0xb249, 0xb0fe, 0x4135, 0x1864, 0xa8fd, 0xbfc0, 0x23fa, 0x4029, 0xb2b7, 0x4319, 0x408a, 0x4315, 0x4221, 0x1257, 0xb041, 0x0dcb, 0xbfb3, 0x4262, 0xb293, 0xbac4, 0xb20c, 0xa4ca, 0xbad8, 0x4215, 0x4188, 0x401a, 0x1a27, 0x41ce, 0xb282, 0x0e98, 0x4075, 0x40a9, 0xbafe, 0x3362, 0xb25e, 0x4165, 0x1cb7, 0x4025, 0x374e, 0xb2ac, 0x1b9c, 0xb28a, 0xbacb, 0xbf57, 0x2d7c, 0xb010, 0xa274, 0x3dab, 0x41a6, 0x44b9, 0x3456, 0xa547, 0x4304, 0x4056, 0x4359, 0x1e93, 0x416e, 0x4009, 0xba31, 0xbf6f, 0x410e, 0x40ac, 0x1854, 0xb0c6, 0xba4c, 0xa9db, 0xb2c3, 0x1cc0, 0x053b, 0xbf51, 0x04d7, 0x44f3, 0x4598, 0x4157, 0x4483, 0xb2f7, 0xbf5a, 0x40f9, 0x4075, 0xa00f, 0xbae4, 0xb0d9, 0x3a47, 0xbaec, 0x43b8, 0xb28f, 0x4308, 0xa8af, 0x4159, 0xb2c7, 0x4025, 0x2d96, 0x406f, 0xb28e, 0xb0e9, 0xbf44, 0xb2ba, 0x4236, 0x0e9d, 0x43cf, 0x420d, 0xb2f9, 0xb228, 0xb038, 0x42de, 0xb283, 0x43a7, 0x013a, 0xba3d, 0xb208, 0xa6da, 0x44fa, 0xb2cf, 0x3eb1, 0x1b4e, 0x404b, 0xbf60, 0xb2ae, 0x43a6, 0x44cb, 0x4297, 0xa637, 0x403f, 0xbfd8, 0xbfe0, 0x40ee, 0x41ed, 0x407c, 0x1927, 0x1c96, 0x4239, 0x19b4, 0xb25c, 0xbf3d, 0x20cf, 0xb025, 0xb2a8, 0x1849, 0x297d, 0x427e, 0x15b8, 0x42df, 0x4050, 0xb2fa, 0x4151, 0xbf3d, 0x267a, 0xb23f, 0x4098, 0x42cf, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc83b3c92, 0xbd911bb4, 0xd5a5a9bb, 0xeab26e45, 0xe6aa6cea, 0xa200e8c6, 0x4e453677, 0x79cc53e1, 0x299e08fa, 0xe4a48c55, 0x96623e63, 0x103ae3c6, 0x15c81382, 0xac554abf, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x00000000, 0x000001fd, 0x000000fe, 0x000000fd, 0xfffffffd, 0xff5e0000, 0x0000007a, 0x000043fe, 0xbfc51110, 0x00001384, 0x966255f3, 0x00004087, 0xfffffea7, 0xac554d4d, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xba07, 0x1e95, 0x30c3, 0x4194, 0x2be0, 0x1a8a, 0x404b, 0x4110, 0x4339, 0x444c, 0x46ec, 0xb2bb, 0xba43, 0x2218, 0x4228, 0x40af, 0xa05c, 0xb02b, 0xb297, 0xb2a6, 0x41fe, 0xbf38, 0x43f7, 0x43a4, 0x06aa, 0x41ce, 0x4586, 0x42ed, 0xbafd, 0xb020, 0x4274, 0xa264, 0x356c, 0x43f1, 0x3264, 0xac1d, 0xb252, 0x430b, 0xab27, 0xb2e9, 0xbf90, 0x1aea, 0xbfdb, 0x193b, 0x4338, 0x1bee, 0xb2c8, 0xbac2, 0x420f, 0xbf0d, 0x4206, 0xb266, 0x2a23, 0xbae6, 0x40ff, 0xbfb0, 0xb06e, 0x1a93, 0x1bfa, 0x3b77, 0x4167, 0x420b, 0xba05, 0x2a9d, 0xba3d, 0x431f, 0xba6f, 0x41e5, 0x406a, 0x307a, 0x4298, 0x41cc, 0xb202, 0xb0a0, 0xafbe, 0xbfde, 0x4625, 0x43f4, 0x1b22, 0x41c9, 0x0924, 0xbac1, 0x424e, 0x4024, 0xbf28, 0x4128, 0xb2f2, 0xa079, 0xb078, 0xb252, 0x2be8, 0x41f0, 0x4207, 0x035e, 0x42cc, 0x3a3d, 0x43d3, 0xb2ac, 0xb2e1, 0x40ec, 0xa1c3, 0xb0ae, 0xb226, 0x4084, 0xb0b4, 0xae1e, 0xb206, 0xbf2c, 0x3a9c, 0x42e1, 0x40a8, 0x0650, 0xbfe0, 0x1d2d, 0x1ddf, 0x40fe, 0xbaf2, 0x400a, 0x01fa, 0x1bc9, 0x45bc, 0x4111, 0x467b, 0x1255, 0xa90c, 0x422e, 0x1d2f, 0x00a2, 0x434f, 0x383d, 0x3bbd, 0x41b1, 0x42c7, 0xb2d2, 0x0a93, 0xbf73, 0x403d, 0xba68, 0x428b, 0xb087, 0x1943, 0x1ea1, 0x1cfe, 0x43b4, 0x1c92, 0xb215, 0x1ffa, 0xb20e, 0x40a8, 0x0631, 0xb2fe, 0x42fb, 0x4690, 0xb2e8, 0x411b, 0xa461, 0x415d, 0xb0dd, 0xbfdc, 0xbac6, 0x40b3, 0xb269, 0xb2c0, 0x10c9, 0xbac9, 0xba5d, 0x40c1, 0xbf6e, 0xa5fa, 0x1681, 0xb0da, 0x1f93, 0xb20c, 0x4624, 0xb067, 0x414c, 0x179d, 0xba2a, 0x41f6, 0x4320, 0xbf90, 0x430e, 0x45b1, 0x4008, 0x4273, 0x1619, 0xbf18, 0x093d, 0x42f5, 0x101c, 0x408e, 0xb2a8, 0x4068, 0xba64, 0x1b43, 0x445e, 0x4130, 0x424f, 0x1af2, 0xb2a9, 0xb2c7, 0x1aeb, 0x0e84, 0xb2fc, 0x42a0, 0x06a4, 0x408a, 0xbf8f, 0x40e0, 0xbf70, 0x4673, 0x3809, 0x432d, 0x4011, 0x41de, 0xb03e, 0x42e6, 0xbaf3, 0xb0ef, 0x40e5, 0x4616, 0x40e3, 0x445b, 0xbf60, 0x46d3, 0x41a9, 0x4012, 0x189d, 0xba3b, 0x42ea, 0x418b, 0xbfba, 0x4306, 0x4106, 0x4177, 0x1a98, 0x1fa5, 0x4220, 0x466e, 0xb272, 0x42ab, 0xae2c, 0x41d9, 0x4335, 0x400a, 0xbf80, 0x34cf, 0x060e, 0xb233, 0xba1c, 0x2797, 0xb0a7, 0x446d, 0xb265, 0xadc0, 0xaf88, 0xbf5b, 0x418b, 0x40b3, 0xa037, 0x40fc, 0x4186, 0x4294, 0x43d1, 0x415d, 0x43a0, 0x438f, 0xbf9e, 0x461d, 0x198d, 0x388a, 0xada6, 0x41f7, 0x4335, 0xb24f, 0x2507, 0x1d8e, 0xb29a, 0x1f43, 0x42d2, 0xbae6, 0x42de, 0xa28d, 0x1aeb, 0x43dd, 0x40f9, 0x4253, 0x22a2, 0xa9b0, 0xb232, 0xbacb, 0xba56, 0xa533, 0xba38, 0x4154, 0xba5c, 0xbf15, 0x35e5, 0xb2c0, 0x43ce, 0x0e9c, 0x46eb, 0xbfc0, 0x40eb, 0x44fc, 0x42f3, 0x4337, 0x439c, 0x1aa7, 0x435e, 0x4215, 0x0762, 0x4186, 0x403d, 0x41b2, 0xbf5a, 0x42dc, 0x4097, 0x40f6, 0xb284, 0xba73, 0x4648, 0x408f, 0x407a, 0x1cd9, 0x4446, 0x43aa, 0xbf12, 0x11a8, 0x4164, 0x11f9, 0xae18, 0xb0c6, 0xb0de, 0x43af, 0x405a, 0x4328, 0x4236, 0xace1, 0xa3a0, 0xa0ac, 0x1c62, 0x1dc2, 0x431e, 0xba78, 0x433c, 0x42f8, 0x467f, 0x102a, 0xba5f, 0x422f, 0x4168, 0x431e, 0xa297, 0xbf41, 0x1ac0, 0xb0c8, 0xb2b8, 0x4290, 0x1a1a, 0x2d4d, 0x409e, 0x1b2d, 0x430c, 0xbfe1, 0x41e9, 0xba38, 0x1dc3, 0x400f, 0xba5e, 0x4158, 0xb011, 0x43bb, 0x40f6, 0x1d74, 0xba3c, 0x1af3, 0x4607, 0x2a9a, 0x44c8, 0x40cd, 0x43cd, 0x4253, 0x0b40, 0x419c, 0x1afb, 0x405f, 0x1664, 0xb076, 0x3250, 0x3abc, 0x4282, 0x425d, 0xbf8a, 0xb0dc, 0x2556, 0x41d1, 0x4251, 0x4046, 0x14af, 0x1b2e, 0xb22f, 0x1cbc, 0xb21d, 0x401d, 0x18e1, 0x436a, 0xbfd3, 0x07d1, 0xb2df, 0x24c2, 0x416b, 0x1bd4, 0x1d69, 0x4172, 0x4049, 0x3a4d, 0xacb4, 0x19c2, 0x45a5, 0x1236, 0x2647, 0x3ed1, 0xb03f, 0x43f8, 0xac19, 0x41b9, 0xbf34, 0xba5f, 0xb23c, 0x238f, 0xba3e, 0x4630, 0x4376, 0xb263, 0xb261, 0x431e, 0x4069, 0x1a93, 0x407f, 0x4000, 0x40a3, 0x42c8, 0xbfd4, 0x4322, 0xb2e4, 0xb0ad, 0xb2e1, 0xb273, 0xb203, 0x4064, 0x43b4, 0x144b, 0x41a3, 0x4143, 0xacf2, 0xa4dc, 0xbfe2, 0x4101, 0x406d, 0x4075, 0xba34, 0xb0e0, 0x4010, 0xb247, 0x437a, 0x2989, 0x4375, 0x290c, 0x464a, 0x43b0, 0xb201, 0x4320, 0x4097, 0x2155, 0xaa34, 0x1967, 0xb211, 0x3a11, 0x1522, 0xba36, 0x1b4a, 0xbf6d, 0xb216, 0xb2e3, 0x1c04, 0xa4c9, 0x1b2f, 0xbfe0, 0x43bb, 0xb2af, 0xa800, 0xbf90, 0x4016, 0xba5e, 0xbf7c, 0x4193, 0x19a4, 0x1235, 0x0b1f, 0x429d, 0x2f62, 0x43d7, 0x4341, 0x2659, 0xba31, 0x094f, 0x44db, 0x4034, 0x418e, 0x1f48, 0x46a2, 0xba6a, 0xbf4d, 0xb23d, 0x404d, 0x4111, 0x0d4b, 0x40fd, 0x41d2, 0x403e, 0x30ee, 0xb26f, 0x1ebb, 0x4201, 0xa2a4, 0x402e, 0x40df, 0x2596, 0x4395, 0x2ab3, 0xb276, 0x4287, 0x14a6, 0x4313, 0xbf85, 0x1cc0, 0x46cc, 0x440b, 0x243b, 0x4123, 0x15af, 0x40ae, 0x4336, 0x41b3, 0xb207, 0x40b7, 0x40ea, 0x4288, 0xa9af, 0x40ae, 0x4173, 0x4293, 0x40a2, 0x2d32, 0x12f0, 0x4265, 0x339c, 0x2d1d, 0x4273, 0x4033, 0xb273, 0xa8dd, 0x4088, 0x4393, 0xbf85, 0x200b, 0xbfc0, 0x4370, 0x416a, 0xba73, 0xbafc, 0x40d6, 0xb20a, 0x09bc, 0x366a, 0x3509, 0xbf84, 0x2e0f, 0x419a, 0xb06a, 0x41cd, 0x4115, 0x433e, 0x1cdb, 0x42d0, 0x46f3, 0xb06c, 0x388e, 0x43b6, 0x41ed, 0x190c, 0x43ab, 0x1654, 0x4043, 0x427d, 0x0edd, 0x4022, 0xbaca, 0x0775, 0x18b0, 0xa255, 0xbfb9, 0x1cb8, 0x41c3, 0xb245, 0xa4a7, 0x07dd, 0x37c1, 0x41fb, 0x42d0, 0x349d, 0x42db, 0xbf9b, 0xba23, 0x40a2, 0x43f6, 0x43d2, 0xba6f, 0x368e, 0x4004, 0x1d3f, 0x43ee, 0x4127, 0xbf80, 0xbaec, 0x410a, 0x2bd6, 0x2a9f, 0x40cc, 0x40dc, 0xba0b, 0x43c4, 0x4554, 0x1afc, 0xa651, 0x1ebe, 0x41aa, 0xbf60, 0xb23f, 0xa420, 0xbf4b, 0x4412, 0x4168, 0xb200, 0x42bd, 0x4439, 0x188c, 0x0cce, 0x43a1, 0x1936, 0x36b8, 0x41a5, 0x1fd1, 0x193d, 0x44c9, 0x38d3, 0x40d8, 0xb22e, 0xbfac, 0x1893, 0x2638, 0x17ae, 0x404a, 0x44c8, 0xb257, 0x4097, 0x4328, 0x414e, 0x1e25, 0x424e, 0x41f0, 0x08f4, 0xbfd0, 0xb081, 0x4348, 0x4062, 0x43af, 0xb2fa, 0xbfdf, 0xad9a, 0x14a9, 0x1b10, 0x40c4, 0x430b, 0xba6c, 0xbaee, 0x4670, 0x40cf, 0x415c, 0xb29c, 0x4095, 0xa03c, 0x45f3, 0xbfa0, 0x1b48, 0xbf53, 0xad0c, 0x1f88, 0x3697, 0x42a2, 0x2c12, 0x2442, 0xbad7, 0xa0c2, 0xb277, 0xb0d9, 0x43e8, 0xa0c3, 0x4006, 0x4399, 0xbfca, 0x0cbc, 0xb261, 0x2cc7, 0x417e, 0x40d9, 0x43f3, 0xb27c, 0x09c8, 0xb0b9, 0x125f, 0x429f, 0xbfc3, 0x3d9c, 0x0f46, 0x0047, 0x4258, 0x4081, 0x1968, 0x45a1, 0x292c, 0x4002, 0x4463, 0x1a41, 0x414a, 0xba3f, 0x3ebb, 0x152e, 0x41f9, 0xb25f, 0x4005, 0x36c5, 0x22ec, 0xba69, 0x1ab9, 0x32de, 0xba32, 0x1867, 0xb296, 0xb2c2, 0x06f3, 0xb2e9, 0xbf8d, 0x1ea4, 0x406a, 0x46ad, 0xbacc, 0x4677, 0x4097, 0xbfa0, 0x2090, 0x0a7d, 0x400a, 0x42b6, 0xba75, 0xbaf3, 0x43e5, 0xa474, 0x419e, 0x4425, 0x3cd2, 0x46fc, 0x4043, 0xbfc5, 0x32fe, 0xaf8a, 0x1802, 0x1dfe, 0x432f, 0xb002, 0x42ef, 0xba54, 0x363d, 0x1fc4, 0x4619, 0x4206, 0x4247, 0xb0ef, 0xac69, 0xb276, 0x45b6, 0x4322, 0x3c86, 0x42fd, 0xbf5c, 0x1e9d, 0x0ef5, 0x4118, 0xb037, 0x315b, 0x42a2, 0xb083, 0x4245, 0xba51, 0x46f9, 0xbf70, 0x34d3, 0xb281, 0x43b7, 0xba6b, 0xb0d4, 0xba17, 0x431b, 0x4084, 0x425a, 0x0967, 0x40e1, 0x4571, 0x4429, 0xbf87, 0xb265, 0x4011, 0x4066, 0xba0f, 0xb0f6, 0x4334, 0x1a54, 0x0d0c, 0xb271, 0x4394, 0xbf48, 0x1fae, 0xac17, 0x1fea, 0xb2d6, 0x2631, 0x3ebb, 0x1d7f, 0x46c4, 0xba4f, 0xbf7b, 0x45aa, 0x4240, 0x3de4, 0xbafa, 0x41c2, 0x45b5, 0xba4c, 0x4129, 0x4202, 0x4489, 0x4220, 0x43f3, 0xbfd0, 0x421e, 0xbfa0, 0x0784, 0x4181, 0x42f5, 0x42ae, 0xbfb6, 0x40f3, 0xbaf7, 0x030d, 0x430c, 0xb024, 0x4162, 0x4055, 0x444b, 0x4278, 0xb2f9, 0xbfc5, 0x2d34, 0x41c2, 0x4260, 0xa8c5, 0x0235, 0xb0a5, 0x44b0, 0x2efc, 0x1bbd, 0x41d0, 0x062f, 0xbfba, 0xb25a, 0xb023, 0x46fa, 0xbac7, 0xbfd0, 0x0a43, 0xbf14, 0x3ff5, 0x4073, 0x4130, 0x41a7, 0x2bd7, 0xbf90, 0xb2e5, 0x4030, 0x41bb, 0x4262, 0x2062, 0x4021, 0xb266, 0x1a01, 0x4264, 0xb234, 0x40cb, 0x425d, 0x25e2, 0x414a, 0x3c20, 0xb0ea, 0xa1c7, 0x14e0, 0xbfb7, 0x4170, 0x42fc, 0x43d7, 0x4580, 0xb26d, 0x1a13, 0x44dc, 0xb244, 0x00e2, 0xb088, 0x422a, 0xb203, 0x1ac1, 0xba09, 0xa3ad, 0xbfd0, 0x1c62, 0xbf26, 0x02ce, 0x167e, 0x4108, 0x1b88, 0x4351, 0x4259, 0xb05d, 0x40e7, 0xbf00, 0x42aa, 0x1aac, 0xb201, 0xb036, 0xba1c, 0x429f, 0x033c, 0x462b, 0xa687, 0xbf60, 0x4216, 0xb2b0, 0xb24c, 0x4381, 0xbf75, 0x3c94, 0x4360, 0x143b, 0x40a5, 0xbaca, 0x32ca, 0x109f, 0x353e, 0x1e88, 0x4123, 0x1418, 0x4040, 0x40ae, 0x4332, 0x134c, 0x4211, 0x4271, 0x1ddd, 0x2885, 0x431f, 0xbfc5, 0x1dc1, 0x4296, 0xb2af, 0xb2b3, 0xb2d0, 0x2abd, 0x296b, 0x3323, 0x418f, 0x438a, 0xa4d6, 0x1ae8, 0x0ab5, 0x4173, 0xb095, 0x42db, 0xb226, 0xb03d, 0xbf2f, 0xb210, 0xb247, 0x437f, 0xb2f5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x804206f6, 0x99583a3f, 0xad754391, 0x26941fad, 0x139e5e23, 0x688cb146, 0x1ba6feec, 0x7e394ee6, 0x68aeb641, 0x77239003, 0x176ac985, 0x4d825eb1, 0xa99614a6, 0x1716467f, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0xffffffe4, 0x00000000, 0x000021b0, 0x00000023, 0x00001b18, 0x00000018, 0x00001b18, 0x00000310, 0x4a3eb53f, 0x0000169f, 0x00000010, 0x00000000, 0x4a3eb5c9, 0x0603ff82, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x42b1, 0x434a, 0x1868, 0x1910, 0x4205, 0x3246, 0x2096, 0x439d, 0xba7d, 0xba78, 0x180e, 0xa168, 0xafc3, 0x40e6, 0xb2c6, 0xbf60, 0xbf80, 0x46ca, 0x2a71, 0xa766, 0xb269, 0x1f80, 0x14d9, 0x4050, 0xb209, 0xbf72, 0x0b06, 0x4024, 0x41fe, 0xa917, 0xba63, 0xb282, 0x41f2, 0x4171, 0x4288, 0x23f7, 0x032e, 0x4043, 0xb23d, 0xbf12, 0x430e, 0x1c8f, 0x1851, 0x4341, 0x0360, 0x4090, 0x2675, 0x4281, 0x411e, 0x4255, 0xba5b, 0xb283, 0x1af5, 0xb28d, 0xbf33, 0x41ed, 0xb07c, 0xae17, 0x1e11, 0x19ce, 0xb296, 0x33f9, 0x4392, 0xbaf3, 0x40bd, 0x433e, 0x40f9, 0xa383, 0x1b86, 0x1c9f, 0xb096, 0xbf00, 0x4369, 0x42ce, 0x1708, 0x37f1, 0xbf11, 0x431e, 0x28cb, 0x245a, 0x434f, 0x4298, 0x0366, 0x1b85, 0x22a8, 0x4214, 0x42fc, 0x43ca, 0x33ae, 0x42da, 0x1828, 0xabe1, 0x40e4, 0x39be, 0x1a59, 0xb01f, 0xbfb8, 0xb20e, 0x41cc, 0xb2ce, 0x4122, 0xac37, 0x435d, 0xbfc3, 0x4261, 0xb071, 0x24ed, 0xbadd, 0xba07, 0x19a6, 0x1e22, 0x4241, 0x4265, 0xb2f8, 0x401a, 0xb043, 0x4034, 0xb236, 0xb28b, 0x3189, 0xb085, 0xb239, 0x4409, 0x429d, 0x41c2, 0x30cc, 0xbfdf, 0xbaea, 0x40da, 0xb247, 0x1e98, 0xb20e, 0xbad2, 0x4136, 0x1f95, 0xaf3a, 0x2cae, 0x427a, 0x4015, 0x409f, 0x025c, 0xafcb, 0xb20f, 0x43a1, 0x1b77, 0x4222, 0x400b, 0x42c5, 0x1f2f, 0xbfc7, 0xb2ce, 0x4608, 0x4694, 0x42cb, 0x2963, 0x3619, 0x41dd, 0x43ac, 0xb204, 0x3c86, 0xbf05, 0xbac9, 0x1f7c, 0xba2e, 0x0be5, 0x38aa, 0x41aa, 0xa6f7, 0x432b, 0x4124, 0x43fa, 0x1d2d, 0x42d4, 0x1acf, 0x465e, 0x1aa0, 0x32b0, 0x410a, 0x4205, 0xb0b0, 0xbf9c, 0x409d, 0x3b35, 0x1b4f, 0x43eb, 0x189c, 0xbfb3, 0x43f9, 0x42a9, 0x19b1, 0x45cc, 0x1f2f, 0x2d27, 0x1f79, 0x41aa, 0xb20a, 0x420d, 0x45e0, 0xbf1e, 0xb008, 0x4254, 0x2b5d, 0x2382, 0xbf6a, 0x416f, 0xba16, 0x11bf, 0x40de, 0x1a5a, 0x182a, 0x4051, 0x40a8, 0xbf92, 0x41bc, 0xa479, 0x44ed, 0xa6bd, 0x449d, 0x35e7, 0xa2eb, 0x3f57, 0xb0a6, 0x191c, 0x401b, 0xb265, 0x4155, 0x4227, 0x42a1, 0x3f24, 0x1de3, 0x4312, 0x434f, 0x467f, 0xbfce, 0x4054, 0x42c8, 0x18a8, 0x4355, 0x42ad, 0x4139, 0x0b84, 0x2b9b, 0x3f55, 0x42ba, 0xbfc5, 0xa96c, 0x33cc, 0x41c6, 0xba77, 0x405f, 0x4009, 0x4106, 0x1ace, 0x0146, 0xbf1a, 0x1e1b, 0x41fb, 0x42b8, 0xbf82, 0x3dac, 0xaba4, 0xa622, 0x42d3, 0xb060, 0xba3f, 0xbaf0, 0x41b1, 0x4030, 0x410b, 0x1f0b, 0x03e5, 0x42a7, 0x2640, 0x416b, 0xb261, 0x1eaa, 0x4118, 0xba72, 0x4083, 0xbf31, 0x3606, 0xbac0, 0x25ff, 0xa90b, 0xa970, 0x4680, 0xb219, 0x4312, 0xb2b2, 0x4400, 0x1f32, 0xba03, 0x4371, 0x420f, 0x424b, 0xba02, 0x41f4, 0x46c1, 0x3158, 0xab7d, 0xba0b, 0xba2b, 0x42a4, 0x3e0b, 0x02c5, 0x4039, 0x4111, 0xb2f1, 0x0250, 0xbf5d, 0xbaf8, 0x464c, 0x43e7, 0x1baf, 0x41a3, 0x45c1, 0x4369, 0x4113, 0x415e, 0xbff0, 0x2a72, 0x1e9d, 0xba79, 0x083f, 0x41fe, 0x418c, 0x467e, 0x4472, 0x4484, 0xbf62, 0x4671, 0x4698, 0x4141, 0x41d0, 0x29ea, 0x1d92, 0xba36, 0xba01, 0x188a, 0x3f58, 0x43ec, 0x4396, 0xa1b9, 0x4266, 0x40ba, 0x4122, 0x43a4, 0xb269, 0x401f, 0x18f0, 0xb241, 0x4119, 0xb07a, 0xbf95, 0x08c9, 0x37d4, 0x21c0, 0x43e4, 0x4165, 0x468a, 0x40b6, 0xba3b, 0x0e8d, 0xba2f, 0xb287, 0x4223, 0x1bab, 0xb2e8, 0x41dd, 0x4065, 0xbf6c, 0xb23f, 0x2e6d, 0xb2fa, 0x4252, 0x0c56, 0xa14e, 0x4313, 0x42c3, 0x427c, 0x41ca, 0x430f, 0x4187, 0x11c1, 0x17df, 0x40ac, 0xb2c7, 0xb23e, 0xb0b3, 0xa0f6, 0x430f, 0xb25b, 0xbf81, 0xa50d, 0xb258, 0x217d, 0x40a7, 0x1bd7, 0xa080, 0x4248, 0x2643, 0xba66, 0xba2b, 0x4349, 0x1edb, 0x428e, 0x40a9, 0x2b5e, 0x439b, 0x417e, 0x402e, 0x12a6, 0x0e8d, 0xbfbb, 0x424f, 0xb06e, 0x4032, 0xa1c4, 0x4271, 0xb2e8, 0x4674, 0xb0cc, 0x427d, 0x42ac, 0x469b, 0x416b, 0xb201, 0xb2d2, 0x401a, 0x42b5, 0x4397, 0x4299, 0xb238, 0x4664, 0x40a4, 0xbafc, 0xbf7b, 0x1f88, 0x4334, 0x414b, 0x4365, 0x3112, 0x1aa7, 0x1ea7, 0x40f3, 0x1197, 0x45da, 0x411f, 0xbfdb, 0xb020, 0xba4b, 0x1e98, 0x42ea, 0x4388, 0x4344, 0x1956, 0x4225, 0x45c5, 0xb2b9, 0x435c, 0x1e1b, 0x40b7, 0x1bb0, 0x0400, 0x40d0, 0x43e1, 0x4071, 0x40ea, 0xb0b7, 0x122c, 0xb27d, 0x40d4, 0x4470, 0xbfc8, 0x4002, 0x408c, 0x0287, 0xb201, 0x4221, 0x408d, 0x42f2, 0x4009, 0xb2dd, 0x43ae, 0xba01, 0xbf17, 0xb2fb, 0xb271, 0x209a, 0x4201, 0xb203, 0x4202, 0x43e9, 0x2338, 0x40af, 0x2f11, 0x40c6, 0xbf70, 0x0018, 0xa318, 0xbae2, 0xb063, 0x1bbf, 0xba6f, 0xbf60, 0xb2fa, 0xb2c2, 0x40f1, 0xba51, 0xa87a, 0xbf1b, 0x43fe, 0x408f, 0x42b4, 0x09d9, 0xb2a1, 0xbae5, 0xbf32, 0x41fe, 0xa3c9, 0x42ec, 0x1bf6, 0x4151, 0xb2ac, 0xb26c, 0xa3e7, 0x1c0c, 0x4298, 0xb291, 0x41c3, 0x428b, 0xb278, 0xb062, 0x4054, 0xb232, 0x40c4, 0x1067, 0x43ea, 0x4261, 0x43c9, 0x1dca, 0xb044, 0xbfd4, 0xb2e7, 0xb221, 0xabdc, 0x42e2, 0x417e, 0x43fa, 0x42f2, 0xbaf1, 0xb237, 0x46b1, 0xbf70, 0x1b90, 0x1baf, 0x4029, 0xba6d, 0x4112, 0x4305, 0x2569, 0xb0c6, 0xbf2a, 0x430a, 0x434e, 0x411a, 0x4233, 0x41a0, 0x175a, 0x41c1, 0x4235, 0x404a, 0xb090, 0x40f4, 0xb25c, 0xb2b1, 0x41cb, 0xba1a, 0x4064, 0x1f35, 0x423d, 0x4124, 0x4264, 0x4344, 0xbf6b, 0xb0d3, 0x0de7, 0x40c9, 0xb204, 0xb257, 0x1d42, 0x4340, 0xba64, 0x2cc3, 0x23ec, 0xb223, 0x1ea1, 0x3050, 0x402c, 0x43e3, 0x2354, 0x4044, 0xbac8, 0xb262, 0x42c0, 0xb2c2, 0x40d2, 0x42e3, 0x45a2, 0x4176, 0x40c3, 0xbf45, 0xb069, 0x467a, 0xb00e, 0x4571, 0x412b, 0xbfb0, 0x42ab, 0x2c01, 0x461d, 0x33c4, 0x4124, 0x2d04, 0xbac0, 0xba2d, 0xb04a, 0x431c, 0xbf05, 0xb220, 0xb057, 0x2eb8, 0xba23, 0xbace, 0x38d4, 0xbfe0, 0x4076, 0xb2b3, 0x429a, 0xb041, 0x43c3, 0x3b35, 0x43f0, 0xb203, 0xa113, 0x2d45, 0xbff0, 0x1f05, 0xbf67, 0x422b, 0xba7c, 0x2794, 0x40c5, 0xbfba, 0x41a3, 0x43be, 0xbff0, 0xb27b, 0xb23c, 0x2e8b, 0x3d1e, 0x0282, 0x425e, 0xbad1, 0x4005, 0xbad3, 0xb0e5, 0x4227, 0xb010, 0x4381, 0xb2b3, 0xbf46, 0xb2ce, 0x1f53, 0x44ea, 0xa888, 0xae43, 0x424c, 0x1e4e, 0x18c7, 0x4424, 0xb0ff, 0xb274, 0x4294, 0x341d, 0x40bb, 0xa1b2, 0x4564, 0xba76, 0xbad6, 0x4177, 0xb000, 0xbae7, 0xbf8c, 0xb24e, 0x0516, 0x178a, 0xa2d3, 0x42bc, 0xbf32, 0x2fc5, 0x3c82, 0x188b, 0xb265, 0x4087, 0xb2c3, 0x293d, 0xba08, 0x438a, 0x0d71, 0x22c9, 0xb09a, 0xb2c6, 0xb259, 0x3f8c, 0x43bf, 0xbfd0, 0xbfc4, 0xb28a, 0x1f75, 0xb0dc, 0x418f, 0xae2d, 0x193f, 0xbadd, 0x31f3, 0x403c, 0x1537, 0xbaf9, 0xb299, 0x27eb, 0x4330, 0xb2df, 0xbf1d, 0xb253, 0xa5e3, 0x1b97, 0x40e3, 0x229a, 0xb0e9, 0x204d, 0x1e6b, 0x4217, 0xb056, 0x3e7d, 0x4347, 0x4151, 0x24fa, 0x231c, 0x4151, 0xb0bf, 0x4261, 0x4294, 0x1499, 0xa071, 0x4389, 0x40f0, 0xbf24, 0x43f0, 0xa34e, 0x3d28, 0x40fc, 0xba75, 0x42b5, 0xb235, 0x276f, 0xa13d, 0x4045, 0xae0d, 0x4339, 0xba7f, 0x46e0, 0x3011, 0xb222, 0x106e, 0x421e, 0x43fc, 0x306e, 0xbf6e, 0xb293, 0x410b, 0x1cf9, 0x4073, 0x38a8, 0xba4f, 0x0e8e, 0xba5a, 0x46e5, 0x0c13, 0x438c, 0x0dc5, 0xbf00, 0x42e2, 0x4098, 0x3453, 0x1904, 0xba61, 0x4475, 0xbf3c, 0x4280, 0x431e, 0x4411, 0x12e6, 0x3dce, 0x1d7d, 0x41f3, 0xb204, 0x1e51, 0x1e1c, 0x4082, 0x4220, 0x0544, 0x40eb, 0x40f7, 0x4161, 0x1ae9, 0x43e3, 0x4071, 0x2ecb, 0x401e, 0x44b5, 0xb2b8, 0xbf9a, 0x41ec, 0x324e, 0x19af, 0x44f2, 0xb2ce, 0xb248, 0x42dc, 0xba09, 0x3d53, 0x1a5c, 0xa24d, 0xa580, 0x08df, 0xba14, 0x4695, 0x4409, 0x4212, 0x0169, 0x41b5, 0xb2ae, 0x2638, 0x42b9, 0xbfcf, 0xb292, 0x441a, 0x19f3, 0x1831, 0xb200, 0xb2b0, 0xb2f8, 0xba7e, 0x43bf, 0x108a, 0x4113, 0xab7b, 0x42e2, 0x43bb, 0x406e, 0x44fa, 0x4225, 0xb2bb, 0x4416, 0x32c6, 0x41d4, 0x033a, 0x40f5, 0xba4d, 0xbfdb, 0xbf90, 0x3054, 0x1bc7, 0x2374, 0x40b9, 0xb2a7, 0x2123, 0x42c4, 0x2200, 0xb24b, 0xb096, 0xa563, 0x415a, 0x099a, 0x1bdd, 0x4279, 0x3b38, 0x421b, 0x3c50, 0x1efa, 0xb2ed, 0xb0ab, 0x1914, 0x34d6, 0x2af8, 0xba20, 0x056e, 0xbf52, 0x427b, 0x41dc, 0xbfc0, 0x0e3c, 0x4107, 0x439d, 0xb260, 0xb25d, 0x429c, 0x46b3, 0x179c, 0xbfc7, 0x43c5, 0xb27f, 0x197b, 0x402d, 0xb264, 0x4117, 0x2330, 0xbf70, 0x4073, 0xbf00, 0xbad5, 0xb069, 0x42ef, 0xaf7f, 0x4002, 0x436d, 0x4121, 0x1410, 0x1f1a, 0x042f, 0x423d, 0x3abd, 0xbf33, 0xb0cc, 0x425d, 0xb0ec, 0x41da, 0x4335, 0x4263, 0xb2b0, 0x047c, 0x0a77, 0x4326, 0xbac0, 0xabe5, 0x417c, 0x4141, 0xadbf, 0xbf62, 0xb051, 0x2cbe, 0x1cfa, 0x428d, 0x10c9, 0xa7b6, 0xba01, 0xbfd0, 0x4247, 0x43fe, 0x441a, 0x2a92, 0xab72, 0x406b, 0x2f61, 0xbfcc, 0x189e, 0x0ed2, 0x4337, 0x41fa, 0x4049, 0xb207, 0xa86f, 0x439d, 0xa2fa, 0xbf93, 0x1b7d, 0xb0b2, 0x2ea1, 0xaf72, 0xae7d, 0x436a, 0x415b, 0x412c, 0x1964, 0x1660, 0x3250, 0x40bb, 0x41b4, 0xa262, 0x430c, 0xbf22, 0xabab, 0x3c05, 0xb04a, 0x4322, 0x410c, 0xb23a, 0x26f1, 0x43db, 0xba7c, 0x42f8, 0x1fa5, 0x43f5, 0xad41, 0x4276, 0xbf85, 0x42e1, 0xb06d, 0x0bee, 0x43e4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe4628e3c, 0xb130dc40, 0x301f3068, 0x711e026e, 0x29a5a394, 0xa6eb85b1, 0xcd2f7a79, 0x2ec8b90f, 0xd90284cf, 0xe8b9b1aa, 0xa23589aa, 0x4701154b, 0x09e98069, 0x05583b86, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0xffffffff, 0x00000000, 0x00001870, 0xffffe6ab, 0x00007018, 0x000018d4, 0x00000000, 0x00001870, 0x09e98068, 0xffffffff, 0x05585795, 0x0c600000, 0x09e98068, 0x000017d0, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x190c, 0x41dd, 0xb2fe, 0x408e, 0x4222, 0x32ca, 0x40b9, 0x3388, 0xba3b, 0x455c, 0xbfd0, 0x0a17, 0x40b6, 0x1dc6, 0x316f, 0x1fe4, 0x1ddd, 0x4312, 0x1f2e, 0x41c8, 0x4142, 0x43e6, 0xbf64, 0x4407, 0x4230, 0x46a5, 0x43bc, 0x422b, 0x43b7, 0x4111, 0x4125, 0x42fc, 0xb0eb, 0xba23, 0x4375, 0xb002, 0xba10, 0x4272, 0xb282, 0xb0f4, 0x2703, 0x4192, 0xb2c8, 0xbf3a, 0x415a, 0x41dc, 0x2e6f, 0x4326, 0xa155, 0x1c7e, 0x43a5, 0x3588, 0xbf33, 0x434c, 0x3d8b, 0xa455, 0x4028, 0x42b9, 0x42e8, 0x4117, 0x12c6, 0xb2ea, 0x0555, 0x39c4, 0x4674, 0x4339, 0xb2fd, 0xb241, 0xba30, 0x42f0, 0x2e82, 0x418f, 0xbad0, 0x4253, 0x075f, 0x4412, 0x1dbd, 0xb229, 0x43c2, 0xbfae, 0xb085, 0xb2f3, 0xb0e9, 0x1b1f, 0x1f0e, 0x281f, 0x18e7, 0x30b9, 0xbadf, 0xba64, 0x42ec, 0x4208, 0x2e04, 0x2730, 0x430e, 0xb2aa, 0xb244, 0x4103, 0x1eb8, 0x4229, 0x1e5c, 0x40b5, 0x3def, 0xa5b2, 0x33b8, 0x4389, 0x42a1, 0xba14, 0xbf98, 0xa852, 0x4343, 0x40e5, 0xbaf5, 0x188e, 0x1ac6, 0x4073, 0x0c0c, 0xbfc2, 0x4028, 0x404d, 0x1860, 0x442f, 0xb0f6, 0xbf61, 0x3c74, 0xbf60, 0x41a0, 0x1c6a, 0xba14, 0x12cd, 0x40bb, 0xbf70, 0x4034, 0xbfc2, 0xbf00, 0x40d4, 0xb2f2, 0x433a, 0x001a, 0xac53, 0x0ab7, 0x428d, 0x443b, 0x4335, 0x27cc, 0x1e0a, 0x40e1, 0x17e2, 0xbad0, 0xb297, 0x405b, 0xb270, 0x429a, 0xbad7, 0x41de, 0x4210, 0x1acf, 0x42a3, 0x443b, 0x1d64, 0x460d, 0xbf06, 0xbad0, 0xb2b7, 0x4092, 0xba0f, 0x438c, 0x439f, 0x291d, 0x2fba, 0xb2b8, 0x435c, 0xbf9b, 0xac65, 0x4277, 0x4171, 0x2bd5, 0x0832, 0x46c5, 0xb275, 0xb292, 0x4012, 0x1396, 0x4014, 0xb046, 0xba1d, 0xb2a1, 0x067f, 0xb26b, 0x432a, 0x1630, 0x2119, 0xb202, 0x42cc, 0xbf54, 0xb004, 0x43df, 0xaad2, 0xa411, 0x1ea9, 0x40d3, 0x1fe2, 0xaf6c, 0xb068, 0x40ab, 0x2124, 0x39a3, 0x1a23, 0xba65, 0xb2eb, 0xbfa4, 0xb035, 0xba7c, 0x40b7, 0x4013, 0xb292, 0x444c, 0xb0d2, 0x401e, 0xba6a, 0xba5b, 0x4368, 0x46ec, 0x4493, 0x4016, 0x4601, 0x4049, 0x15b2, 0xb298, 0x18e4, 0xb22a, 0x1f07, 0x101f, 0x1908, 0xbfaf, 0x434e, 0x1125, 0x41d5, 0x1322, 0x41d0, 0xbfa2, 0x4349, 0x1fa5, 0x4289, 0x1ebd, 0x10be, 0x4328, 0x4084, 0x4076, 0xa326, 0x4262, 0xba59, 0xb017, 0x1eb2, 0x43e8, 0x43a9, 0xb014, 0x089a, 0xb22b, 0x1cf0, 0x4490, 0x1529, 0x4371, 0xbfa0, 0x43cc, 0x4127, 0xbfb7, 0xba00, 0xba0d, 0x41f4, 0xb0ea, 0x1e04, 0xbacd, 0x40bf, 0x46bc, 0x403b, 0x2642, 0x400b, 0x1d25, 0x4339, 0x4124, 0x43fb, 0x4203, 0x40f2, 0xb0f1, 0xac2d, 0xbfc0, 0x1f1c, 0x3447, 0x40b2, 0xbfa8, 0xbfb0, 0xbfb2, 0x4373, 0xabab, 0x428b, 0x400a, 0xaa61, 0xbff0, 0x46c0, 0x45e8, 0x1f40, 0xbae0, 0x2815, 0x4130, 0x39ef, 0xbace, 0x3978, 0x1c85, 0x4023, 0x1fbc, 0xbf01, 0x413b, 0x1f38, 0x4248, 0xa2f5, 0x1354, 0xa20b, 0x4287, 0x2088, 0x377f, 0x4029, 0x41a5, 0x0b8a, 0xb2ff, 0x419e, 0x003c, 0xb28e, 0xbf2a, 0xbad0, 0x3fea, 0x4138, 0x434b, 0x41c3, 0x4371, 0xb08c, 0xb240, 0x41d3, 0x42c1, 0x43ee, 0x42ae, 0x41e0, 0xb287, 0x1c7c, 0x41fc, 0x4130, 0x46e1, 0x4391, 0x1d6d, 0x401f, 0x189e, 0xbf24, 0x206b, 0xb27e, 0xb2e3, 0x432a, 0xba3f, 0x43b6, 0x1e9c, 0x41e6, 0x1943, 0x411a, 0x40d7, 0x4367, 0xa70f, 0x1088, 0x422f, 0x1d6e, 0x197a, 0x42ae, 0x3fb9, 0x01b5, 0xb226, 0x40c7, 0x3246, 0xbf58, 0xb07e, 0x4250, 0xba4e, 0xa4d2, 0xa9fa, 0x2365, 0xb28d, 0xa57c, 0xbaf7, 0xba5b, 0xb010, 0xba53, 0xbadf, 0x4122, 0xbf57, 0xb010, 0x4374, 0xbae0, 0xb2fb, 0x1fda, 0x41c7, 0x432a, 0xaa9f, 0x19da, 0x1f13, 0xb2e1, 0x1d38, 0x44a4, 0xbfb7, 0x438e, 0x054a, 0x425a, 0x3229, 0xbac4, 0x40a6, 0x404f, 0x4348, 0x41da, 0x40db, 0xafa8, 0xb0ba, 0x4004, 0x432c, 0xb058, 0x4324, 0x1daf, 0x4309, 0x42bd, 0x18cf, 0xbf6e, 0x42ee, 0x418a, 0xb270, 0xb287, 0xb209, 0x4271, 0xb0e0, 0x3c45, 0x1a47, 0xb284, 0xb224, 0x410f, 0x4230, 0xb239, 0x41d8, 0xb09f, 0x4125, 0xba3f, 0x18ea, 0x1a7b, 0x3bb4, 0xb018, 0xbf4a, 0x2a5f, 0x31f9, 0x411f, 0x4142, 0x423b, 0x4573, 0xbfb0, 0xba72, 0x1940, 0xb032, 0xbae0, 0x43e3, 0xba0c, 0x2537, 0x4081, 0xb211, 0x15fa, 0x41fa, 0xbfa0, 0xab62, 0xb0d6, 0xba2d, 0xbf51, 0x400b, 0x30b2, 0x4036, 0xaea0, 0xbf00, 0x42e8, 0x2b15, 0x1f0d, 0x381f, 0x4103, 0x4097, 0x41be, 0x235b, 0x4091, 0xbaf5, 0xb296, 0x455e, 0xb252, 0x4223, 0xba21, 0xbf90, 0x2bb5, 0x4153, 0xbf31, 0xa17a, 0x1938, 0x4210, 0x4274, 0x2b0c, 0x43b2, 0x411d, 0x1a71, 0x40e4, 0x2d1a, 0x4041, 0x4091, 0x40c5, 0x4082, 0xb256, 0x43a8, 0x4365, 0x426f, 0x42fc, 0xb058, 0xb098, 0x4077, 0xb074, 0x426f, 0x2362, 0x40fd, 0xbf22, 0xb2ba, 0xbf60, 0x438c, 0x4005, 0x4240, 0xbfd6, 0x4120, 0x42ab, 0xb0fa, 0x4674, 0x40ef, 0xbfce, 0x42fb, 0xb03e, 0x426b, 0x4116, 0x426b, 0xb29c, 0x419f, 0x4106, 0x437d, 0x43f6, 0x4625, 0x0a71, 0x424b, 0xb287, 0xa587, 0x40c2, 0xa389, 0x4043, 0xb23a, 0x0f83, 0x0e74, 0xbfa3, 0x42c8, 0x1837, 0xb04a, 0xb215, 0x1f63, 0x409b, 0x466f, 0x4020, 0x40cb, 0x1fc4, 0x424a, 0x4041, 0xab3f, 0x2513, 0xba39, 0x2761, 0x08ef, 0xb01c, 0x40ec, 0x4344, 0x432a, 0x423a, 0xbac6, 0x008b, 0x43cc, 0x1f37, 0x0f45, 0xbf11, 0x4088, 0x4359, 0x42cc, 0x1c92, 0xbacb, 0xb06c, 0xaf25, 0xb299, 0xbfde, 0x4184, 0x3ea1, 0xaab7, 0xbfc0, 0x2538, 0xbfd4, 0x4100, 0x2625, 0x419a, 0xb210, 0x231f, 0x40b5, 0xbfd9, 0x1dd1, 0x1c34, 0x41a3, 0x1c6a, 0xa756, 0xbf70, 0x430f, 0x42c1, 0x4077, 0x404f, 0x41fc, 0x4245, 0xbff0, 0x42cb, 0x3c1f, 0x1c6b, 0xab3a, 0x4083, 0x4131, 0x4234, 0x41a8, 0x41fe, 0x435e, 0x4086, 0xbf98, 0xb2c8, 0xbae1, 0x3ccd, 0x441d, 0xb29c, 0x4047, 0xbf92, 0x42ea, 0x4077, 0x1f00, 0x2d3c, 0x434f, 0x1d83, 0x2a97, 0x4238, 0x43d0, 0xb274, 0x4000, 0x438d, 0x407a, 0x199e, 0x1ec9, 0x4041, 0x4133, 0x0b38, 0x437e, 0xbadb, 0x0c6c, 0x43c9, 0x436d, 0xb2b7, 0x1b76, 0x1207, 0x403a, 0xbfa2, 0x0e8e, 0x1b82, 0x2840, 0x4018, 0xbfa5, 0x2ffa, 0x4493, 0x1cf6, 0xbfe0, 0x243e, 0x4113, 0xb28a, 0x4057, 0xb2d6, 0x43a2, 0xb2ac, 0x40d0, 0x4198, 0xae4e, 0x42f7, 0x3681, 0x4231, 0x4455, 0x35d7, 0x1e5f, 0x411e, 0xb2dc, 0x422d, 0x1edd, 0x4164, 0x2df7, 0xa616, 0x1734, 0xbf53, 0x404c, 0x4061, 0x44c8, 0x43d0, 0xbfdb, 0x415f, 0x1c73, 0xb2de, 0x40bf, 0x462e, 0x4218, 0x1dcc, 0xba45, 0xaef1, 0x425a, 0xa89f, 0xb25a, 0xa25f, 0x1fa5, 0x417a, 0xba13, 0x418e, 0x42a4, 0x35d6, 0x1b4b, 0xba13, 0xba40, 0x1b23, 0x0b14, 0xb20c, 0x2bba, 0xba45, 0x3b65, 0x256f, 0xbfd7, 0xba0d, 0xb2d5, 0x4267, 0x24e6, 0x2d6d, 0x11bc, 0x42b6, 0xbf15, 0xb217, 0xb2d8, 0x405f, 0x4143, 0xa6e6, 0xa92f, 0x42fb, 0x42bd, 0x438b, 0x458a, 0x43c3, 0x420c, 0x43bb, 0x4068, 0x43d4, 0x3fa0, 0x1968, 0x03e1, 0xb291, 0xbf90, 0x2968, 0x4029, 0x38bc, 0x431d, 0x43af, 0x1b23, 0x41a3, 0xbfd9, 0x43f3, 0x229b, 0x0925, 0xb295, 0x15ec, 0xb223, 0x19d3, 0xba1c, 0xb080, 0xb27a, 0xbf7e, 0xba7d, 0xb236, 0x4365, 0x4273, 0x1c90, 0x21b5, 0x42cf, 0x40d6, 0x18c8, 0x0777, 0x281d, 0xbfc0, 0x40ca, 0x1b92, 0x412d, 0xa3aa, 0x44c3, 0x4220, 0x4087, 0x4110, 0x43b0, 0x1aad, 0x3f60, 0x1515, 0xb2fd, 0x40ea, 0xbf0c, 0x19d3, 0x41ba, 0x4003, 0x4040, 0xb239, 0x4057, 0xbfd0, 0x439d, 0x41c6, 0x4382, 0x4122, 0x30b4, 0xbff0, 0x4285, 0x4112, 0x0eec, 0x41f2, 0x4165, 0xba79, 0xbfc0, 0xbac0, 0x43bc, 0x1fa0, 0xbf3b, 0x17fb, 0xba32, 0xba37, 0xb218, 0x4397, 0x40c6, 0xb2e3, 0xbfe4, 0xb23e, 0xb07d, 0x42cb, 0x424e, 0x4269, 0x41e7, 0x4323, 0x40f6, 0x3575, 0x4110, 0x41bc, 0xb2e8, 0x4228, 0xb242, 0x403a, 0x4262, 0xba27, 0x153a, 0xba00, 0xba70, 0x2dde, 0xb290, 0xbf91, 0x40bd, 0x427b, 0x0a83, 0x2605, 0x40ca, 0xa13e, 0x408a, 0x2078, 0xa34c, 0xbfc0, 0x4671, 0x1ae9, 0xbaed, 0x2113, 0xb27b, 0x4138, 0xb268, 0x0f57, 0xbfa5, 0xaeb6, 0xbac7, 0x4304, 0xb03a, 0x1524, 0x4439, 0xb2cf, 0xbfb0, 0xb252, 0xb083, 0x1b1f, 0x4297, 0xa384, 0x4077, 0x1b4d, 0x4386, 0x41da, 0x29a1, 0xbfc0, 0x1f1a, 0x3f17, 0x4007, 0x42f5, 0x4192, 0xbfe8, 0x2373, 0x40b8, 0x4160, 0x4621, 0xb02b, 0xae8c, 0x13f9, 0xba3d, 0x25d3, 0x3d46, 0x41df, 0x4481, 0x1289, 0xa8ac, 0x4257, 0x1157, 0x423d, 0xbff0, 0xb2f2, 0xab13, 0xbfde, 0x412c, 0xb09e, 0x40ba, 0xb0a4, 0x4356, 0x43fe, 0x4209, 0xbf3a, 0x40f3, 0x43df, 0x39a1, 0xb0c3, 0x0992, 0x46ca, 0x4269, 0x417e, 0xb200, 0x46bb, 0x41d6, 0xba1c, 0xbfe0, 0x42a5, 0x3e74, 0x4651, 0x4371, 0x435c, 0x437a, 0x23a1, 0xb217, 0x1bd7, 0x1920, 0x449b, 0xb24c, 0xbf26, 0x4551, 0x41d7, 0x43d9, 0xae4b, 0x4008, 0x403d, 0xba74, 0x43e3, 0x185b, 0xba5c, 0xbf16, 0xba06, 0xbff0, 0x4118, 0x2b3b, 0x435a, 0x409b, 0x4085, 0x434b, 0xa0a7, 0xbf02, 0x1b80, 0xb2a7, 0x1fb5, 0x2ab4, 0x4102, 0x40ec, 0x2d72, 0xba16, 0x4076, 0x23fd, 0xab93, 0xba7e, 0xbf5d, 0x41a0, 0x1d5c, 0x43cc, 0x3cb3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x96070e06, 0x60f8a631, 0x6d122e21, 0xad567bfc, 0x291715df, 0x4658e055, 0x3efa71b8, 0x42812292, 0x84889393, 0xde7eb11f, 0x47c9ca7e, 0x0ba82b44, 0x8ffeeee8, 0x71d424e5, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0xb75339fa, 0xfffcaf4b, 0x00000000, 0x8488a09f, 0x000350b4, 0x48ace04c, 0x00002fab, 0x0000ab2f, 0x84889836, 0x00000741, 0x00000741, 0x000000a0, 0x00001640, 0x84889e53, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0x4361, 0xb2d9, 0x43f4, 0x2def, 0x1e99, 0x1afe, 0xbfc7, 0x43d0, 0x404f, 0xafd3, 0x4601, 0x4586, 0x4143, 0x42bd, 0x4118, 0x4128, 0x431b, 0xa978, 0x2d56, 0x4607, 0x4124, 0x401b, 0x40a2, 0x430f, 0xba2a, 0xae2f, 0xb035, 0x3079, 0xa7cc, 0x4374, 0x2f83, 0xb20d, 0xbf6c, 0xacbf, 0x40e3, 0xb23c, 0x02c5, 0x433b, 0x40fe, 0x284b, 0x4208, 0x4616, 0x444a, 0xbfc8, 0x3a02, 0x4427, 0x3cd7, 0xa66c, 0x3bf0, 0x4176, 0xb277, 0x0c72, 0x033a, 0x4374, 0x3b46, 0x4039, 0x4646, 0xbaef, 0x4694, 0x43d8, 0xbff0, 0x18d4, 0x191f, 0x1ba0, 0x34cd, 0x09c2, 0x41f8, 0x08e6, 0x22f3, 0xbf76, 0xb0c1, 0x4267, 0xba61, 0xb2cc, 0x412e, 0x41fc, 0x2b38, 0xba61, 0x4130, 0x3f3b, 0x19f1, 0xb2e9, 0x43d3, 0x429d, 0x3b51, 0x42a9, 0x3dcd, 0x06c6, 0x42e0, 0x404e, 0xbf2b, 0xa69c, 0xb26a, 0x4013, 0x2ddf, 0xba2c, 0xafc1, 0xb0c8, 0xb0e1, 0x0dbf, 0x42e7, 0x4102, 0xbf90, 0x4034, 0xb013, 0x1c3c, 0x41fb, 0x1961, 0xb26d, 0x45d2, 0xb07e, 0xbf87, 0x4085, 0x1cf2, 0xbfc0, 0x42ff, 0xa589, 0x43c3, 0xb24f, 0x418e, 0x4261, 0xab43, 0x1b92, 0xbf05, 0x4388, 0x43bb, 0x4038, 0x4393, 0x4187, 0xa081, 0xb0d2, 0x42e2, 0xbf00, 0x2535, 0x43fe, 0x42c1, 0xb24a, 0x408a, 0x1f7d, 0xbf92, 0x1bcb, 0x1b0c, 0x3141, 0xb262, 0xbf86, 0x08aa, 0xb2f5, 0x4262, 0xbfdf, 0x42ea, 0x460c, 0x424f, 0x1ada, 0x42f2, 0xa012, 0x081b, 0x4046, 0x42b5, 0xb008, 0xb231, 0xb0e8, 0x42c3, 0x4187, 0x464b, 0x4016, 0x1abe, 0x098f, 0x1e7f, 0xba68, 0x2a45, 0x1b1a, 0xbf96, 0x402f, 0x1350, 0x3bc2, 0x411b, 0x04a8, 0xb076, 0xb03b, 0x4550, 0xba39, 0xb22a, 0x42f4, 0x4334, 0x4249, 0xb268, 0x12a7, 0x414b, 0x45b1, 0x2e3a, 0xb021, 0x10fa, 0x4249, 0x40b0, 0xbf71, 0x422f, 0x1c42, 0xba7b, 0x41f7, 0xbacd, 0x3d7d, 0xba1d, 0x44d8, 0x45e3, 0x4021, 0x1933, 0xbac1, 0x386d, 0x2777, 0x2151, 0x425a, 0xb251, 0x4376, 0xb008, 0xb2fd, 0xbf42, 0x4191, 0x1724, 0x41da, 0x418a, 0x1a0b, 0x3c2d, 0x345d, 0x0f74, 0xa44e, 0x461c, 0x18d4, 0xa030, 0xb2e0, 0xba75, 0x3710, 0xbf81, 0xba2d, 0x2b8b, 0x1c31, 0x44e1, 0xbfbc, 0x40f3, 0x398e, 0xb260, 0x4567, 0x46e8, 0xb240, 0x1c35, 0xba1b, 0x46c1, 0x4109, 0x43b7, 0xb094, 0x1676, 0x4143, 0x43f1, 0xbf87, 0x136c, 0x4058, 0x1e68, 0x4234, 0x059a, 0x430d, 0x1631, 0x41fe, 0x43b6, 0xb2ed, 0xbff0, 0x4024, 0x406d, 0x4367, 0x418a, 0x4307, 0xbfaf, 0x4374, 0xbae2, 0x4097, 0x3185, 0xb0ec, 0xb051, 0x4052, 0xba60, 0x433d, 0xb210, 0x4347, 0x402e, 0x43d1, 0x412f, 0xb0f2, 0x2310, 0xbfac, 0x4191, 0xb2fd, 0x1db4, 0x4559, 0xb08f, 0xb2aa, 0x4182, 0x42ae, 0x4036, 0xaf83, 0xb234, 0x418d, 0xbf8c, 0x407b, 0xba34, 0x41cf, 0x4351, 0x41c5, 0xba77, 0x1cfd, 0xa0cd, 0xbac7, 0xbf43, 0xba21, 0x425d, 0xba4c, 0xba13, 0x413c, 0x4095, 0xa127, 0x4198, 0xb2f3, 0x1f4e, 0x40e9, 0x19cc, 0xba4c, 0xb2e0, 0x4241, 0xbf90, 0xb214, 0x4224, 0x40f4, 0x3c7c, 0x07b9, 0x43e1, 0x41c7, 0x4339, 0x4181, 0xba11, 0x0bcc, 0xbf29, 0x4051, 0x1dbf, 0xb25a, 0x07c0, 0xba76, 0x413e, 0x417f, 0x1d87, 0xa388, 0x1d2e, 0x40e6, 0x4165, 0x432e, 0x050e, 0xb0fd, 0x43f4, 0x1895, 0xbf32, 0xb090, 0x40ba, 0x1e9a, 0x1f13, 0x45d2, 0x4044, 0x1f04, 0xb2ea, 0x43fa, 0x4197, 0x43d6, 0x1de6, 0x4671, 0x402d, 0x4058, 0x1aca, 0x43c5, 0xadd4, 0x41e8, 0xba3b, 0xb273, 0xbf9c, 0x197a, 0xbaf6, 0xbfb0, 0xba0e, 0x45d1, 0x1f11, 0x4379, 0xaf6e, 0xbff0, 0xb019, 0xa676, 0xbf4e, 0x4264, 0x4351, 0xb051, 0xbae2, 0x415e, 0x4355, 0x4016, 0x437f, 0xbfca, 0x408e, 0x42ef, 0xb240, 0x40cc, 0x311b, 0x435a, 0xab55, 0x43de, 0xaeae, 0x20fa, 0xbacb, 0x4301, 0x43bd, 0x42b6, 0xbfb0, 0x403c, 0xaf7d, 0x1941, 0xbf05, 0xba02, 0x0c49, 0x42cc, 0x1e9d, 0x1efc, 0x0a69, 0x42a6, 0x42e6, 0x1ae2, 0xba55, 0xbfd0, 0x407b, 0xbaf9, 0xa02b, 0xba5d, 0xbf16, 0xb239, 0xbfa0, 0xba47, 0x418f, 0x41e9, 0xafc4, 0x23c8, 0xb04e, 0x42fb, 0xaf00, 0x402a, 0x4270, 0x4380, 0x4214, 0x4466, 0x2297, 0x441a, 0xbf6d, 0x4298, 0x1836, 0x1918, 0x3cff, 0x42ec, 0x4363, 0xbf60, 0x4311, 0xbfa3, 0x4004, 0x0c27, 0x41fe, 0x4346, 0xb0f2, 0x1544, 0x43b7, 0x412d, 0xab14, 0x1f41, 0xb2ab, 0x190f, 0x218f, 0xb2d9, 0x0e76, 0x40b8, 0xb2d0, 0x41da, 0xbf09, 0x1a5e, 0x4340, 0x4004, 0x0884, 0x437a, 0xbf90, 0x428a, 0x19e6, 0x4283, 0x439b, 0xbfb8, 0x43d1, 0xba5f, 0x4366, 0x06b4, 0x4390, 0xb20c, 0x2116, 0x10b5, 0x4266, 0x43d3, 0x17f9, 0x42d4, 0xbac7, 0x437c, 0x434e, 0x410d, 0xaef0, 0xad99, 0x231d, 0x41a0, 0xb2ac, 0x0716, 0x3728, 0x1cbc, 0xbac7, 0x42d9, 0xbf05, 0xb2fb, 0x438a, 0x08e7, 0x42f7, 0xa411, 0x35d7, 0x4631, 0x46f2, 0xa304, 0x4650, 0x4055, 0x416c, 0x4336, 0x0e29, 0x41cb, 0x40e5, 0xb2e9, 0x0e4d, 0x4172, 0x1d2c, 0x3f5c, 0x301e, 0x4245, 0xba5e, 0x0d11, 0xbf99, 0x4155, 0x02c2, 0xbfb0, 0x40ec, 0xb0a7, 0xb20a, 0xba21, 0xba5e, 0x41b8, 0x42cf, 0x41ab, 0xbfe2, 0xbae1, 0x3b4b, 0xb01c, 0x46cd, 0x43db, 0xb2c6, 0x46f0, 0x4141, 0x1c08, 0x41e4, 0xba11, 0x4127, 0xb220, 0xba75, 0x1846, 0xbf43, 0x1be2, 0x4073, 0x418f, 0x37e6, 0x2c57, 0xaf36, 0xb23b, 0x4095, 0x412e, 0x0e71, 0x42ba, 0x19ec, 0xb0ab, 0x19d9, 0xa24a, 0x362e, 0x408b, 0x1094, 0xb21f, 0x40c9, 0x42be, 0xb2c7, 0x409e, 0xbf15, 0xb0cf, 0xb00b, 0x415c, 0x4009, 0x4186, 0xb0c7, 0x4257, 0x1992, 0x40c9, 0xbf00, 0xb2d0, 0x4168, 0xbf9d, 0x43c4, 0xbaf8, 0xba68, 0xba26, 0x45bc, 0x4214, 0x1231, 0xbfd4, 0x4009, 0xba00, 0x3e72, 0xbf1e, 0xb29b, 0xbacb, 0xbaf9, 0x4352, 0x4105, 0x42e7, 0xaa0f, 0xb2b3, 0x43a6, 0x4699, 0x2170, 0x0c8e, 0xae4a, 0xbfe0, 0xb004, 0xb237, 0x1a88, 0xbf41, 0xb2ad, 0xbf00, 0x13a8, 0x42a5, 0x00cb, 0xa3f6, 0x1bd6, 0x44d1, 0xba06, 0x4278, 0xba65, 0xbfc0, 0x098f, 0x029d, 0x2a7b, 0xb2df, 0x07a7, 0x0105, 0xad3c, 0x43ac, 0x40bf, 0x428f, 0x339a, 0x40fb, 0x43b9, 0x446d, 0x42ac, 0x1083, 0xad05, 0xbf8c, 0x4264, 0x42e4, 0xbaf2, 0xbad8, 0xb21f, 0x4244, 0x25d1, 0xb282, 0x405c, 0x1a04, 0x4399, 0x414d, 0xb2fa, 0xba6b, 0xb216, 0xb250, 0x438b, 0x2b3c, 0x40b0, 0x425b, 0x412e, 0xb220, 0xb25f, 0xb2c2, 0x2ec8, 0xbf3a, 0xba70, 0x43bc, 0xb299, 0x4284, 0x369c, 0xbaff, 0x058f, 0xba2c, 0x1699, 0x42e3, 0x41ce, 0xabfc, 0xb2cc, 0x2a1e, 0xb201, 0x19d3, 0xb099, 0xbf34, 0x332f, 0x0535, 0x4422, 0x42b9, 0xbfdf, 0xb270, 0x400a, 0x40d7, 0x4024, 0x1f9e, 0x196d, 0x434c, 0x1dd6, 0x41df, 0xb219, 0x405d, 0x4022, 0x1e04, 0x41dc, 0x4296, 0x43c4, 0xba25, 0x145b, 0x42a6, 0x4154, 0xbf0a, 0x4061, 0x28b4, 0x1c9f, 0x4310, 0x274b, 0x4191, 0x4034, 0xbf7f, 0x439a, 0x42d8, 0xb0eb, 0xba0f, 0xba5a, 0x41a7, 0x43c1, 0x0538, 0x41ad, 0x42a1, 0x43ce, 0xbf04, 0xa79a, 0x1b98, 0xb018, 0xb0e0, 0x42f5, 0x4476, 0xaf71, 0x433b, 0xb212, 0xbf73, 0x0a7e, 0x2b1f, 0xbafb, 0xba4e, 0x0425, 0x420d, 0x43e2, 0x43e0, 0x4065, 0x4117, 0xba07, 0x2562, 0xa532, 0x43b3, 0xba29, 0x2779, 0x4353, 0x440b, 0x4140, 0xbf58, 0x27dc, 0x4286, 0xa57a, 0x42b2, 0x425d, 0xb2a9, 0xba52, 0x43de, 0x41ee, 0x0796, 0x1fa7, 0x2c5f, 0x2a61, 0x1c25, 0x425a, 0x40b7, 0x42f8, 0x4180, 0x40b4, 0xbfc5, 0x428e, 0x4205, 0xba23, 0x4003, 0x1f55, 0x3e3c, 0x416c, 0x42e0, 0xb234, 0x1d80, 0x432c, 0x417f, 0x4079, 0x3538, 0x1eff, 0x4475, 0xbf34, 0xb277, 0x43b6, 0x4355, 0x40d3, 0xb023, 0x17b5, 0x1d0a, 0x2b3a, 0x4455, 0x0615, 0xbf0b, 0x4045, 0x40ad, 0x2759, 0xb068, 0x40aa, 0x1f85, 0x436b, 0x4268, 0xb282, 0x40d6, 0xb0ed, 0xb01d, 0x422d, 0x4104, 0x4114, 0x1d54, 0xbf70, 0x1d50, 0xbf98, 0x432f, 0x403a, 0x417f, 0x4622, 0xb08c, 0x1bf4, 0x3219, 0x4079, 0x23f2, 0xba10, 0x29d4, 0x4224, 0x431b, 0x41ea, 0x4440, 0x41f5, 0x40c6, 0x1e5e, 0x01c4, 0x34a9, 0x1bde, 0x43b6, 0xb2e9, 0xbf24, 0x3fa4, 0xb2c1, 0x41d5, 0x4002, 0x4215, 0x185d, 0x1a17, 0xb27f, 0xb23a, 0x1397, 0x1e78, 0xb03b, 0x1cad, 0xba6a, 0x2a81, 0x0ee6, 0x4351, 0xbfbe, 0xb2fa, 0x225e, 0x429d, 0x410b, 0x4080, 0xba53, 0x4196, 0x40a2, 0x437b, 0x0a59, 0xb2aa, 0xbae9, 0x0534, 0x40a9, 0x465a, 0x448c, 0xb26a, 0xb213, 0x41c5, 0x424c, 0x18e1, 0x30fe, 0x4046, 0x40d8, 0xbf39, 0x1f57, 0x1865, 0x4059, 0x4376, 0x4564, 0x4438, 0x41e0, 0x12d8, 0xb06b, 0x4329, 0x42b6, 0x40f5, 0x41e4, 0xbac3, 0xb265, 0x4570, 0x42f8, 0x2fb0, 0x432d, 0x438e, 0x4222, 0x438f, 0xbf02, 0x43d6, 0x4179, 0x278d, 0x25df, 0xbafc, 0x251e, 0x1cfc, 0xba69, 0xbf31, 0x0ac3, 0xb0ad, 0x4205, 0x441e, 0x410c, 0xb273, 0x42be, 0x1e0d, 0xbf07, 0x468c, 0x4016, 0xa105, 0x437e, 0x416e, 0xbaf0, 0x418d, 0x3851, 0x2464, 0x43b8, 0x43e5, 0xbac0, 0x4146, 0x3c61, 0x2797, 0x4094, 0x41fb, 0x181a, 0x4437, 0xb273, 0x4365, 0x4321, 0x1fe4, 0x4199, 0x269e, 0x1c0a, 0xbf77, 0x40b0, 0x3eaf, 0x1f30, 0xa473, 0x26a3, 0x42f4, 0xb0a2, 0x4280, 0xb2d0, 0x2fb3, 0xb017, 0xb07d, 0x43f7, 0x439e, 0x43f1, 0xbfe0, 0x459e, 0x3cb3, 0xbfd6, 0x1244, 0x2ba2, 0x4391, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xfaefb3b8, 0x78bd4750, 0x110d5992, 0x6de9a720, 0x42910210, 0xa47797d3, 0xb17bbffb, 0x69ee994d, 0xbfeace8e, 0x33a9e570, 0x6e9aadf6, 0xc053407f, 0x87f3e955, 0x34a56210, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0x000000a1, 0xffffe85e, 0x000017a1, 0x00000002, 0x000018e1, 0x00000000, 0x000000a1, 0xffffff5c, 0x00000000, 0x0000ffbc, 0x00000000, 0xc053407f, 0x00021000, 0x34a56118, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xba27, 0x42db, 0x40d3, 0x4362, 0x409f, 0x1812, 0xbac7, 0xa7df, 0x34cd, 0xa70f, 0x4280, 0x40b0, 0xb0ee, 0x01be, 0xb211, 0x4052, 0xb2a7, 0x4063, 0x406f, 0x4654, 0x434e, 0x42fb, 0xbf25, 0x4269, 0x1119, 0xb264, 0x4301, 0xba63, 0x1a18, 0xb2fb, 0x1887, 0x2f9b, 0xb233, 0x440e, 0x4206, 0xbad1, 0xb09d, 0x262b, 0xb2d3, 0xba06, 0xab5c, 0x2ee6, 0x4419, 0x339c, 0xb2d1, 0x4192, 0xbf72, 0xbae1, 0x410e, 0x41fe, 0x4193, 0xbf00, 0xba7d, 0xb2b6, 0x4318, 0x1eee, 0xb2f0, 0x413c, 0xba4b, 0x41e0, 0xb02d, 0x4287, 0x4467, 0xba35, 0x46d4, 0xa117, 0xbf5c, 0x40b4, 0xa074, 0x2102, 0x4230, 0x0167, 0x44e9, 0x0471, 0x4106, 0x42b1, 0x00af, 0x1a8c, 0x1ae2, 0xabbf, 0x42c5, 0x1b0b, 0x123f, 0x406d, 0xbf91, 0xb09b, 0xb276, 0xb07c, 0x4383, 0x18ab, 0x4156, 0xb2aa, 0xbf7b, 0x191b, 0xb228, 0xb263, 0x0dda, 0x4062, 0x46fa, 0x3720, 0x1f98, 0xb045, 0xb235, 0x1fa8, 0x4276, 0xb27a, 0xa3a8, 0x434b, 0x2044, 0x1f00, 0x169e, 0xbae6, 0xadf8, 0x4272, 0x4004, 0x42de, 0x2bee, 0xba2c, 0xbfb6, 0xb043, 0x415d, 0xb27f, 0x4213, 0x1ebb, 0x4077, 0xb284, 0xa9ab, 0x40a5, 0xb2a9, 0xb254, 0x4593, 0xa036, 0xb058, 0xb2ab, 0x3458, 0x1fa6, 0x3798, 0x4180, 0x41c0, 0x46cd, 0x1155, 0x42cd, 0xbf58, 0xba33, 0xa87e, 0xb012, 0x43eb, 0x2dc8, 0x42f8, 0x1e62, 0xb24f, 0x413d, 0x412d, 0x4335, 0x1915, 0x1ecf, 0x1358, 0x4419, 0x40ec, 0x438f, 0x2cb3, 0x360d, 0x2b61, 0x1cea, 0x0ed4, 0xb2e1, 0xb206, 0xb255, 0xbfa1, 0x1bc7, 0xb011, 0x4306, 0x3b69, 0xa9a9, 0xb20b, 0xad51, 0x420f, 0x1493, 0x4140, 0x405f, 0xb03f, 0x4107, 0x422c, 0x186c, 0xb285, 0x4350, 0x4115, 0x196d, 0xba29, 0xbaeb, 0x4340, 0x404c, 0x4195, 0xba1d, 0x4156, 0x4257, 0xbf45, 0x19bf, 0x4356, 0x3366, 0x40fa, 0x4022, 0x40e4, 0xa663, 0x284c, 0xbf4a, 0x415c, 0x0de3, 0x411d, 0x4268, 0x36e8, 0xbafe, 0xb0b5, 0x40b6, 0x406c, 0x4131, 0xb01b, 0x4281, 0x174c, 0xb06b, 0x421d, 0x412e, 0x4159, 0x4346, 0x01c4, 0x2f52, 0xbfd7, 0xb0f4, 0xb0b7, 0x4053, 0x1ce2, 0xbfdd, 0x40ae, 0xb286, 0x41d1, 0x4156, 0x1b8d, 0x41e1, 0x4281, 0x1844, 0xb297, 0x1c04, 0x405e, 0x1841, 0x412a, 0x414d, 0x45c8, 0xba0a, 0xb25d, 0xadc5, 0xa7eb, 0xbfad, 0x45e0, 0x41ce, 0x4230, 0x4232, 0x41e4, 0x42f1, 0x4566, 0x46d3, 0x1f16, 0x419c, 0x2db3, 0x4312, 0x30ff, 0x28e3, 0xbfd2, 0x4277, 0xa610, 0x41b8, 0xbad9, 0x42db, 0x1ff4, 0x088c, 0x41e5, 0x4104, 0xbf60, 0x0a11, 0x380e, 0xb24f, 0xb287, 0x42a9, 0xb006, 0xba45, 0x408d, 0x414f, 0x1d4e, 0xbf03, 0x3191, 0x41f3, 0xb260, 0x1f59, 0x4347, 0x42f8, 0x412d, 0x43f0, 0x1d4e, 0x42ba, 0xab0a, 0x1f8d, 0x432c, 0xaf72, 0x1bbd, 0x4130, 0x43ab, 0x41ba, 0x0066, 0xbf01, 0x1477, 0x410f, 0x40f3, 0x415c, 0x197b, 0x2fe1, 0x4542, 0xbf31, 0x4020, 0x420c, 0xa16c, 0xb231, 0xba34, 0x420e, 0x46f5, 0x0851, 0x4374, 0xbf22, 0xba20, 0x1df1, 0xb2fb, 0xb090, 0x431f, 0x44cc, 0x411a, 0xb2dd, 0x46b3, 0xba7b, 0x42b1, 0xb267, 0x41e5, 0x41c9, 0x414b, 0x40df, 0xa37b, 0xb2ef, 0x411d, 0xb29b, 0x1380, 0xbf2a, 0x41f5, 0xb280, 0xb2f2, 0x0ed2, 0x4329, 0xb248, 0x0969, 0x4631, 0x40f3, 0x43d0, 0xb093, 0xbf5f, 0xb204, 0x4463, 0x1d6d, 0x4382, 0x41a4, 0x40dc, 0x43f2, 0x460d, 0x42a1, 0x0a18, 0xb097, 0x23aa, 0x1acb, 0x1ff4, 0xb270, 0x0311, 0x0baa, 0x338c, 0xbf01, 0xb22a, 0x422c, 0xb2ae, 0x4369, 0x43d4, 0xba01, 0x41c8, 0xbafb, 0xb034, 0x0c66, 0xb221, 0x432c, 0x1b48, 0x413f, 0x4014, 0xa721, 0xbae1, 0xbae8, 0x192f, 0xba20, 0x416b, 0xb2cd, 0x408f, 0xbf61, 0x42a8, 0x3999, 0xbaff, 0x43a6, 0x416a, 0x4148, 0xb00e, 0x10c7, 0x1a90, 0xba63, 0xbf81, 0xb03e, 0xbaf7, 0x4046, 0x40a1, 0x439d, 0x4141, 0x2c5c, 0x29a6, 0x2b42, 0x0a7a, 0x3b4a, 0x42d6, 0x249b, 0x3240, 0x2ca8, 0xb2e1, 0xbfb8, 0x4059, 0x19ad, 0xb27c, 0xba73, 0x4270, 0x171d, 0x4183, 0xbac4, 0x43bf, 0x4418, 0x4169, 0x422e, 0x41e2, 0x43c2, 0xbf6d, 0x40de, 0xb283, 0x432a, 0x437e, 0xb2b8, 0x4277, 0x4338, 0x40d1, 0x42f7, 0xb2b2, 0xbaef, 0x16a7, 0x4124, 0x4341, 0xbf80, 0xb262, 0x021b, 0xb042, 0x18df, 0x1038, 0xb292, 0x1d14, 0xb0ee, 0x429e, 0x4019, 0xb293, 0x4216, 0xbf62, 0x07a6, 0xb281, 0x1f1b, 0x4660, 0x4568, 0x279d, 0x43e9, 0x45e1, 0xb27b, 0x40fe, 0xb242, 0x42d7, 0x2b9f, 0xb28d, 0xbfb1, 0xb2d4, 0x40ed, 0x43f7, 0xb05a, 0x34cd, 0xb2cc, 0x426d, 0x1c09, 0x414c, 0x4146, 0xb0ac, 0x434e, 0xb2f2, 0xba3c, 0xa981, 0x224b, 0xba0d, 0xb239, 0xbf8a, 0x3ae8, 0x40aa, 0x3459, 0xbf90, 0x4491, 0xb2e0, 0xba62, 0x425b, 0x41b8, 0x427e, 0xb24e, 0x4117, 0xbfc8, 0xa2f2, 0x1c4d, 0xaae2, 0x2776, 0xba4b, 0xbf03, 0x1f0e, 0x41ff, 0xa3a7, 0x21d9, 0x1520, 0xb2d1, 0x4344, 0x4669, 0x42b5, 0xba0a, 0x3274, 0xb23b, 0x1a2d, 0xbf60, 0x4424, 0x43e3, 0x4236, 0xbfa1, 0x4335, 0x41d8, 0xb2fa, 0xba3b, 0x3433, 0xb099, 0xbf26, 0x415a, 0x149b, 0x1e17, 0x2be1, 0x41c8, 0x26ac, 0x18c2, 0xbfe4, 0x27e9, 0x2413, 0x41c9, 0x29b1, 0xa042, 0xba56, 0xba44, 0x1c5f, 0xba32, 0xba60, 0xbac1, 0xa246, 0x1a6e, 0x41dc, 0xb2f1, 0x18f1, 0x1b38, 0x331a, 0x2a6f, 0x40e4, 0x0350, 0x1ff6, 0x43fb, 0xbf77, 0x2b4e, 0x1f7b, 0xa24e, 0xb05b, 0xbf51, 0x1b2f, 0xbaf4, 0x0756, 0x4238, 0x1c0d, 0x2351, 0xb03c, 0x2234, 0xb2c8, 0xb073, 0x1a1e, 0x4171, 0x3256, 0x0169, 0xa84a, 0x43f6, 0x2cb9, 0x429b, 0xba22, 0xbf9a, 0x43ba, 0x46cd, 0x30ba, 0x02fa, 0x42c5, 0x0ca2, 0x326a, 0x428b, 0x43be, 0x114e, 0xbf2c, 0x0816, 0x0903, 0x1a3d, 0xb291, 0x4025, 0xab9a, 0xa63e, 0x426f, 0x42f1, 0x1d96, 0x1e6c, 0x4117, 0x403d, 0x1ecb, 0x438c, 0xbfc4, 0x4110, 0xba01, 0x458c, 0x442e, 0xb07a, 0xb2fb, 0x43dc, 0xb2d4, 0x1bec, 0x407a, 0x19ff, 0x0fd7, 0x4321, 0x1e0c, 0x2ff7, 0xbf91, 0x4266, 0x1dd1, 0x42de, 0xbf90, 0xb208, 0x40a4, 0x1ff9, 0x41c7, 0xb276, 0x408b, 0xbfaf, 0xb2f4, 0x4056, 0xa933, 0x42c6, 0x4198, 0x42ee, 0xbad7, 0xb288, 0xb28f, 0x42b7, 0x43f8, 0xb27b, 0xaa4e, 0x4066, 0x30f5, 0x4242, 0x41e1, 0xba7e, 0xa066, 0x4038, 0x1c69, 0xba0c, 0x4372, 0xb08a, 0xbaef, 0x4314, 0x41c4, 0xbf9f, 0x4355, 0xb2be, 0xb096, 0x402d, 0xb206, 0xb2d6, 0x3674, 0x159d, 0x4093, 0x41dd, 0x427c, 0x40a7, 0xb0ca, 0xb214, 0xb25c, 0xbf1e, 0x4043, 0xa1bb, 0x41c7, 0x4655, 0x1805, 0x4313, 0xba4b, 0xb03c, 0xb2d9, 0xa67b, 0x291b, 0x0eb7, 0x425b, 0xb007, 0x4669, 0xb2aa, 0x41e8, 0x41b0, 0x4015, 0xa496, 0x3460, 0x43cd, 0x44f4, 0xbf01, 0x41cd, 0x4393, 0xad38, 0xba1a, 0xbf4b, 0x2960, 0x43a0, 0xba2d, 0x4559, 0xb0c7, 0x173a, 0x263a, 0xb23e, 0x43c1, 0x43aa, 0xa3b5, 0xba78, 0x0bf1, 0x4126, 0x067a, 0x42fc, 0x446f, 0x0b51, 0x2b2a, 0x439e, 0xb23c, 0xbf23, 0xb078, 0xbae6, 0x0398, 0xa8d3, 0x42ec, 0x431c, 0x41ed, 0x40d1, 0xba3d, 0x2317, 0xbac5, 0x42e1, 0x4268, 0x361b, 0x4235, 0x4151, 0x4596, 0x1ac6, 0x4215, 0xbfbc, 0xb0c0, 0x4296, 0x2525, 0x4287, 0xbf69, 0x24a2, 0x425c, 0xb258, 0xb01f, 0xbf3d, 0xb246, 0xb2cb, 0xbae2, 0xb284, 0xbac2, 0x4339, 0x4580, 0x3c52, 0x40f7, 0x04a0, 0xa9ab, 0x416b, 0xadba, 0x429d, 0x4270, 0x3def, 0x1a98, 0xbf65, 0x4286, 0x4056, 0x3ef0, 0xb287, 0x0bfb, 0x45e1, 0xb270, 0x0e86, 0x4006, 0x4593, 0xaf1d, 0x403a, 0x4113, 0x401e, 0x4209, 0x4018, 0x4130, 0x41ac, 0x425d, 0x42ba, 0xa052, 0x43de, 0x073f, 0x4018, 0xba5b, 0xb2e8, 0x1b83, 0x424a, 0x4000, 0xbf61, 0x1f92, 0x4086, 0x404a, 0x2004, 0x45e1, 0x4221, 0x4257, 0xbf9a, 0x1c3a, 0x20b9, 0x41b2, 0x429b, 0x4097, 0x40cb, 0xbfdb, 0x41de, 0x4107, 0xa45c, 0x426d, 0xafd3, 0x412f, 0x199b, 0x42ac, 0x4389, 0xbf41, 0xb20a, 0xa51d, 0x439f, 0x41ff, 0x4215, 0x3e5e, 0xba14, 0x18f3, 0x428a, 0x43fd, 0x438e, 0x4272, 0x1b74, 0x42ad, 0x417d, 0x1ff0, 0x1b9c, 0x41b7, 0x41b6, 0x405f, 0x17e5, 0xb2a3, 0x07dc, 0xbf22, 0xb08c, 0x409e, 0xb23e, 0xb2b4, 0x29f8, 0x44e8, 0x41ad, 0xba65, 0x4200, 0x41b5, 0xbf8c, 0x1ee4, 0x41ff, 0x40f0, 0xb2b3, 0x45dd, 0xba63, 0x1e16, 0xbfe2, 0x1ee9, 0xa4e7, 0xb07c, 0xae9f, 0xba20, 0x403a, 0x4260, 0x439a, 0x444f, 0x40bb, 0x430f, 0xbf57, 0x18d5, 0x43ba, 0x0350, 0x2a39, 0xaade, 0x0a7a, 0xb239, 0xb2c2, 0xb0a4, 0x0999, 0x4270, 0x4390, 0xbfb6, 0x4048, 0xbfa0, 0xb20d, 0x4026, 0xba4e, 0x430e, 0x4375, 0x4171, 0xb278, 0x4256, 0x40bd, 0x42e4, 0x1c1e, 0xb2de, 0xbf58, 0x059b, 0xb243, 0x197d, 0x4084, 0xa945, 0xb03f, 0x4430, 0x1c82, 0xa4a0, 0x43c7, 0x41e8, 0x41a5, 0xba57, 0x0ecc, 0x2e03, 0xba50, 0x42f7, 0xb017, 0x42b2, 0xbac1, 0xbfbc, 0x408a, 0x0c66, 0xa768, 0xb223, 0x427c, 0x448d, 0x445f, 0x437d, 0x405e, 0x42a3, 0x257d, 0x4485, 0x44ec, 0xba2b, 0x4177, 0x42ea, 0x42f5, 0x43cc, 0x4681, 0x4211, 0x416c, 0xb038, 0xba19, 0x4669, 0xbfbb, 0x08f8, 0x4338, 0xb2cf, 0x4653, 0xa218, 0xb2f6, 0x1894, 0xbaed, 0xb2c1, 0x19ae, 0x1138, 0x1b80, 0x4158, 0x41ea, 0xb083, 0xbfe2, 0xacb0, 0x1d95, 0xa07c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x53e412bb, 0xf8075b4a, 0xc06d1392, 0xad332bbd, 0x27e72f7d, 0xac8f8bd5, 0x3a5b2232, 0xc5e2c8ec, 0xe032a7a7, 0x2b035634, 0x4a720b19, 0x0f87e928, 0x73db65d8, 0x6c7bc94f, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x000019d4, 0x00000000, 0x00001828, 0x7d000000, 0x977f681b, 0x0000182e, 0x00007d12, 0x00000067, 0x77b1c835, 0x00004100, 0x000010c8, 0x0000000e, 0x79708dab, 0x977f655b, 0x00000000, 0x200001d0 },
            },
        };
    }
}
