using NUnit.Framework;

namespace Ryujinx.Tests.Cpu
{
    [Category("T32Alu")]
    public sealed class CpuTestT32Alu : CpuTest32
    {
        [Test]
        public void TestT32AluRsImm([ValueSource(nameof(RsImmTestCases))] PrecomputedThumbTestCase test)
        {
            RunPrecomputedTestCase(test);
        }

        [Test]
        public void TestT32AluImm([ValueSource(nameof(ImmTestCases))] PrecomputedThumbTestCase test)
        {
            RunPrecomputedTestCase(test);
        }

        public static readonly PrecomputedThumbTestCase[] RsImmTestCases =
        {
            // TST (reg)
            new()
            {
                Instructions = new ushort[] { 0xea18, 0x4f03, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x15a99211, 0x08a56ba3, 0x3c588032, 0xdac302ae, 0x6b7d5b2d, 0x4fe1d8dd, 0x04a574ba, 0x7873779d, 0x17a565d1, 0x63a4bf95, 0xd62594fb, 0x2b9aa84b, 0x20448ccd, 0x70b2197e, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0x15a99211, 0x08a56ba3, 0x3c588032, 0xdac302ae, 0x6b7d5b2d, 0x4fe1d8dd, 0x04a574ba, 0x7873779d, 0x17a565d1, 0x63a4bf95, 0xd62594fb, 0x2b9aa84b, 0x20448ccd, 0x70b2197e, 0x00000000, 0x300001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea11, 0x5f67, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc9754393, 0xec511f2a, 0xc365b8f1, 0xa024565a, 0x089ae8e2, 0xf0c91f23, 0x290f83f4, 0x48f2f445, 0xd3288f2b, 0x7d7b2e44, 0xe80dd37e, 0xb000697f, 0x95be1027, 0x74702206, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0xc9754393, 0xec511f2a, 0xc365b8f1, 0xa024565a, 0x089ae8e2, 0xf0c91f23, 0x290f83f4, 0x48f2f445, 0xd3288f2b, 0x7d7b2e44, 0xe80dd37e, 0xb000697f, 0x95be1027, 0x74702206, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x2fc9, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe9c49eb7, 0x2ca13a97, 0x3fded5a8, 0x30e203e9, 0x811a9ee5, 0x504f95f2, 0x746794b4, 0xfe92b6d6, 0x7608d3c4, 0xf3c5ea36, 0x6290c8f2, 0x45a4a521, 0x359a615c, 0x25674915, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0xe9c49eb7, 0x2ca13a97, 0x3fded5a8, 0x30e203e9, 0x811a9ee5, 0x504f95f2, 0x746794b4, 0xfe92b6d6, 0x7608d3c4, 0xf3c5ea36, 0x6290c8f2, 0x45a4a521, 0x359a615c, 0x25674915, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea15, 0x0f85, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3130c8d7, 0x5917d350, 0xdf48eedb, 0x23025883, 0x076175bb, 0x5402cc6c, 0x54a95806, 0x7f59c691, 0x9c3eeebf, 0x4b52b4d1, 0xb4eb9626, 0x21fa7996, 0x0ff0a95a, 0x6beb27fd, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0x3130c8d7, 0x5917d350, 0xdf48eedb, 0x23025883, 0x076175bb, 0x5402cc6c, 0x54a95806, 0x7f59c691, 0x9c3eeebf, 0x4b52b4d1, 0xb4eb9626, 0x21fa7996, 0x0ff0a95a, 0x6beb27fd, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x6feb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x39889074, 0xbea8978e, 0x0331cc7a, 0x448e3b19, 0x33285e9e, 0xdf295408, 0x8444676e, 0xe6998904, 0x819e4da4, 0xb099272c, 0x101385a7, 0x71728a87, 0x76f95b3a, 0x8d5012e4, 0x00000000, 0xc00001f0 },
                FinalRegs = new uint[] { 0x39889074, 0xbea8978e, 0x0331cc7a, 0x448e3b19, 0x33285e9e, 0xdf295408, 0x8444676e, 0xe6998904, 0x819e4da4, 0xb099272c, 0x101385a7, 0x71728a87, 0x76f95b3a, 0x8d5012e4, 0x00000000, 0x000001d0 },
            },
            // AND (reg)
            new()
            {
                Instructions = new ushort[] { 0xea18, 0x1f52, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xcbe174f1, 0x44be318c, 0x4a8a1a70, 0x1f3c8883, 0x33b316ee, 0x0591a3c5, 0x0ceff4a5, 0xd74988e2, 0xa5ef1873, 0xbd35a940, 0x52a9f4d8, 0xf8662781, 0xda558ea8, 0x4c7d50bc, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0xcbe174f1, 0x44be318c, 0x4a8a1a70, 0x1f3c8883, 0x33b316ee, 0x0591a3c5, 0x0ceff4a5, 0xd74988e2, 0xa5ef1873, 0xbd35a940, 0x52a9f4d8, 0xf8662781, 0xda558ea8, 0x4c7d50bc, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea19, 0x4f6b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x97b9423c, 0x1c25286b, 0x50f84fef, 0xd917c24e, 0x2a5116af, 0xcc65ba10, 0xf5e9dc41, 0xf9f61d10, 0x9876cfe5, 0xd0fdd4bc, 0x95913be0, 0x844c820f, 0xfdaf9519, 0xf3fb09b6, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x97b9423c, 0x1c25286b, 0x50f84fef, 0xd917c24e, 0x2a5116af, 0xcc65ba10, 0xf5e9dc41, 0xf9f61d10, 0x9876cfe5, 0xd0fdd4bc, 0x95913be0, 0x844c820f, 0xfdaf9519, 0xf3fb09b6, 0x00000000, 0x900001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x3f52, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x1900757b, 0x6914c62d, 0x5eaa28ed, 0xd927c1f7, 0xfd2052ac, 0x146bcb99, 0x604f9b1d, 0xb395bf46, 0x3723ba84, 0xb909d3ec, 0x3db4365e, 0x42df68cd, 0x5fdc10cb, 0x4955b8be, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x1900757b, 0x6914c62d, 0x5eaa28ed, 0xd927c1f7, 0xfd2052ac, 0x146bcb99, 0x604f9b1d, 0xb395bf46, 0x3723ba84, 0xb909d3ec, 0x3db4365e, 0x42df68cd, 0x5fdc10cb, 0x4955b8be, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x0f17, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6134093f, 0x115a1456, 0xa7877f6e, 0x2070e9eb, 0x9ddf4a73, 0x14266482, 0x7f98e557, 0xbaa854e0, 0xa37f89a6, 0x641325de, 0xae2dc79b, 0x5b3f2af2, 0x476476d2, 0xb99cc9fd, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x6134093f, 0x115a1456, 0xa7877f6e, 0x2070e9eb, 0x9ddf4a73, 0x14266482, 0x7f98e557, 0xbaa854e0, 0xa37f89a6, 0x641325de, 0xae2dc79b, 0x5b3f2af2, 0x476476d2, 0xb99cc9fd, 0x00000000, 0x700001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x5f17, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xbaa2bc1a, 0xee3e86d4, 0x3179d65a, 0x8a63cf55, 0x48ea14f4, 0xf85c5d5b, 0x6af50974, 0xf3ded3e9, 0xdab4d6e6, 0x930c07eb, 0x8084b2dd, 0xf6518695, 0x4a3e0f7a, 0x581bd56a, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0xbaa2bc1a, 0xee3e86d4, 0x3179d65a, 0x8a63cf55, 0x48ea14f4, 0xf85c5d5b, 0x6af50974, 0xf3ded3e9, 0xdab4d6e6, 0x930c07eb, 0x8084b2dd, 0xf6518695, 0x4a3e0f7a, 0x581bd56a, 0x00000000, 0x300001d0 },
            },
            // BIC (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1c, 0x0fbc, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xfc7b7a8b, 0xc1186d54, 0x0a83eda1, 0x88fed37c, 0x5438e8ea, 0xe0af3690, 0x6dba7b9f, 0xa7395bd6, 0xd43af274, 0xbb46f4c2, 0xb65dbcd5, 0xa6bd08b0, 0xb55971c7, 0x2244572e, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0xfc7b7a8b, 0xc1186d54, 0x0a83eda1, 0x88fed37c, 0x5438e8ea, 0xe0af3690, 0x6dba7b9f, 0xa7395bd6, 0xd43af274, 0xbb46f4c2, 0xb65dbcd5, 0xa6bd08b0, 0xb55971c7, 0x2244572e, 0x00000000, 0xb00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x5fe7, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x75f617c8, 0x12ac7ccd, 0x85e6d881, 0x30967bdd, 0xdf66b387, 0xb3d59ccf, 0xe3c824b4, 0xada7a9e4, 0x225da86f, 0x18e008ac, 0x51854224, 0xf3b43823, 0xde37f151, 0x6764b34a, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x75f617c8, 0x12ac7ccd, 0x85e6d881, 0x30967bdd, 0xdf66b387, 0xb3d59ccf, 0xe3c824b4, 0xada7a9e4, 0x225da86f, 0x18e008ac, 0x51854224, 0xf3b43823, 0xde37f151, 0x6764b34a, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x1fc3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xde174255, 0x3968e364, 0xf1efd73b, 0x9a159a4e, 0x2b906c3e, 0xf1dfb847, 0x34e3e8f0, 0x39c33745, 0xc368a812, 0x8f3fe175, 0xe3da055f, 0x7737a5d5, 0x7464344a, 0xdb3ac192, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0xde174255, 0x3968e364, 0xf1efd73b, 0x9a159a4e, 0x2b906c3e, 0xf1dfb847, 0x34e3e8f0, 0x39c33745, 0xc368a812, 0x8f3fe175, 0xe3da055f, 0x7737a5d5, 0x7464344a, 0xdb3ac192, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea18, 0x6f66, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x368ef4f3, 0x18583461, 0x94f6e104, 0x21e1c1b0, 0x009c85df, 0xe6bddfb2, 0x118e9dad, 0xcdf92eb5, 0xae18b093, 0xe24a54ab, 0x55d1a1a0, 0x0eed1bad, 0x8b6bce47, 0x20b1fdc2, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x368ef4f3, 0x18583461, 0x94f6e104, 0x21e1c1b0, 0x009c85df, 0xe6bddfb2, 0x118e9dad, 0xcdf92eb5, 0xae18b093, 0xe24a54ab, 0x55d1a1a0, 0x0eed1bad, 0x8b6bce47, 0x20b1fdc2, 0x00000000, 0x700001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x3fc6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6f913e40, 0xd1814933, 0x181eb63c, 0x287a5050, 0xe5925dd9, 0x712ee261, 0xcca2e51d, 0x0e88a1ba, 0xa4c8d4c3, 0x26887e3e, 0x83b8de36, 0xc5a5d439, 0x8d2ace7a, 0x9df36292, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x6f913e40, 0xd1814933, 0x181eb63c, 0x287a5050, 0xe5925dd9, 0x712ee261, 0xcca2e51d, 0x0e88a1ba, 0xa4c8d4c3, 0x26887e3e, 0x83b8de36, 0xc5a5d439, 0x8d2ace7a, 0x9df36292, 0x00000000, 0x200001d0 },
            },
            // MOV (reg)
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x2fdd, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x89fcb953, 0xafbf8db2, 0xad96137f, 0x7901360c, 0x513b561b, 0x2345a005, 0x0ece889b, 0xc8bb918f, 0x270458ce, 0x73bea675, 0xab735592, 0xf68e00e5, 0x88bf2dc1, 0x98601074, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x89fcb953, 0xafbf8db2, 0xad96137f, 0x7901360c, 0x513b561b, 0x2345a005, 0x0ece889b, 0xc8bb918f, 0x270458ce, 0x73bea675, 0xab735592, 0xf68e00e5, 0x88bf2dc1, 0x98601074, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea19, 0x6fd6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x35f5eea1, 0x75fe4252, 0x71165923, 0x13ad82d2, 0x01f69a1c, 0x33ff5351, 0x869c335f, 0x70ce9266, 0xf58868ad, 0x4f58e982, 0x89f7df88, 0xd0ba8d45, 0xf45e6e03, 0x7f653972, 0x00000000, 0x800001f0 },
                FinalRegs = new uint[] { 0x35f5eea1, 0x75fe4252, 0x71165923, 0x13ad82d2, 0x01f69a1c, 0x33ff5351, 0x869c335f, 0x70ce9266, 0xf58868ad, 0x4f58e982, 0x89f7df88, 0xd0ba8d45, 0xf45e6e03, 0x7f653972, 0x00000000, 0x600001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x6f5d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x1e83719a, 0x1b6405c5, 0x25d9d1d6, 0x3e5fc7f3, 0xd157d610, 0x271b5c46, 0xb65c2838, 0xe4590643, 0x2f2623d7, 0xf1155f93, 0xfa676221, 0x6fac2a1d, 0xc1fa1d8d, 0x8cfa89e1, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x1e83719a, 0x1b6405c5, 0x25d9d1d6, 0x3e5fc7f3, 0xd157d610, 0x271b5c46, 0xb65c2838, 0xe4590643, 0x2f2623d7, 0xf1155f93, 0xfa676221, 0x6fac2a1d, 0xc1fa1d8d, 0x8cfa89e1, 0x00000000, 0x500001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1c, 0x2fa2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x431278f1, 0xd3fffe52, 0xbfb4d877, 0x10af0eeb, 0xd375b791, 0xbd19aa81, 0x45eb7ba3, 0x30e47d42, 0xc274e032, 0x6da10d33, 0xfeda1ba4, 0x3dc6205e, 0xc275197e, 0x6c8b86d1, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x431278f1, 0xd3fffe52, 0xbfb4d877, 0x10af0eeb, 0xd375b791, 0xbd19aa81, 0x45eb7ba3, 0x30e47d42, 0xc274e032, 0x6da10d33, 0xfeda1ba4, 0x3dc6205e, 0xc275197e, 0x6c8b86d1, 0x00000000, 0x900001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x7f7b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x31f0830d, 0x69dda3f6, 0x983fc927, 0x0407652a, 0x32ceab65, 0xe76a77fd, 0x8a7dd0a6, 0x4892a02f, 0xeab00585, 0xa78bf230, 0x896dd5a9, 0xe3c44398, 0xc2d743d0, 0x42b03803, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x31f0830d, 0x69dda3f6, 0x983fc927, 0x0407652a, 0x32ceab65, 0xe76a77fd, 0x8a7dd0a6, 0x4892a02f, 0xeab00585, 0xa78bf230, 0x896dd5a9, 0xe3c44398, 0xc2d743d0, 0x42b03803, 0x00000000, 0x100001d0 },
            },
            // ORR (reg)
            new()
            {
                Instructions = new ushort[] { 0xea10, 0x5f72, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5834d41e, 0x7092ed2e, 0x8994242e, 0x7fac6d96, 0x4d896829, 0x1a578dec, 0x98649fd8, 0x3b713450, 0xca430792, 0xd68d5176, 0xfe0b5c4f, 0xd9caf416, 0xb0e9d5fa, 0x62c57422, 0x00000000, 0x800001f0 },
                FinalRegs = new uint[] { 0x5834d41e, 0x7092ed2e, 0x8994242e, 0x7fac6d96, 0x4d896829, 0x1a578dec, 0x98649fd8, 0x3b713450, 0xca430792, 0xd68d5176, 0xfe0b5c4f, 0xd9caf416, 0xb0e9d5fa, 0x62c57422, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x0fb4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6842aa84, 0x0711ecb6, 0xebae7374, 0x6ea58edd, 0xa6d2837c, 0xbcc1e0d1, 0xe52c9d6c, 0x7bb5fa1c, 0xa7cd6f8a, 0x4558ddb7, 0x7adb449c, 0x95986dd8, 0x7432562c, 0x80d2595c, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x6842aa84, 0x0711ecb6, 0xebae7374, 0x6ea58edd, 0xa6d2837c, 0xbcc1e0d1, 0xe52c9d6c, 0x7bb5fa1c, 0xa7cd6f8a, 0x4558ddb7, 0x7adb449c, 0x95986dd8, 0x7432562c, 0x80d2595c, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x2f78, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6be94e4c, 0x0569589b, 0xc7e6e127, 0xe5537aea, 0x323e7a85, 0x895e9a94, 0x2341f9b6, 0x9632a18a, 0xa790766f, 0x53533cf3, 0x83cec3aa, 0xa1d042af, 0xabff7e58, 0x614f9bc0, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x6be94e4c, 0x0569589b, 0xc7e6e127, 0xe5537aea, 0x323e7a85, 0x895e9a94, 0x2341f9b6, 0x9632a18a, 0xa790766f, 0x53533cf3, 0x83cec3aa, 0xa1d042af, 0xabff7e58, 0x614f9bc0, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea12, 0x4fbc, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2e43ac9a, 0xa6a8d4b6, 0xf5853279, 0xf152f284, 0xce9656e5, 0x07642918, 0xd6e25d4a, 0xdebc7fa6, 0x8c3af5e0, 0x3d00cd4c, 0x7e744bb4, 0x2a4b8015, 0x602ea481, 0xdef7571b, 0x00000000, 0x300001f0 },
                FinalRegs = new uint[] { 0x2e43ac9a, 0xa6a8d4b6, 0xf5853279, 0xf152f284, 0xce9656e5, 0x07642918, 0xd6e25d4a, 0xdebc7fa6, 0x8c3af5e0, 0x3d00cd4c, 0x7e744bb4, 0x2a4b8015, 0x602ea481, 0xdef7571b, 0x00000000, 0xb00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x7f4c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x67be4dae, 0xff0f74a8, 0xd769f9e1, 0xb4a98e0a, 0x2988a7dc, 0xb5726464, 0xb7b3fb27, 0x077e539c, 0x9c817cd4, 0xa8cc3981, 0xbe5a7591, 0xc753850a, 0xb8c612a7, 0x6d913c9b, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x67be4dae, 0xff0f74a8, 0xd769f9e1, 0xb4a98e0a, 0x2988a7dc, 0xb5726464, 0xb7b3fb27, 0x077e539c, 0x9c817cd4, 0xa8cc3981, 0xbe5a7591, 0xc753850a, 0xb8c612a7, 0x6d913c9b, 0x00000000, 0x100001d0 },
            },
            // MVN (reg)
            new()
            {
                Instructions = new ushort[] { 0xea15, 0x0ffb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4a0a7b4c, 0x4e58d907, 0x386b8207, 0xcd71b0c4, 0x86dcf525, 0x8ae9dba4, 0xf5d6a418, 0xfac79f2e, 0x44cf918b, 0x5d38193b, 0xc17adeaf, 0xa4ad8a86, 0x69527ece, 0x69b75c61, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0x4a0a7b4c, 0x4e58d907, 0x386b8207, 0xcd71b0c4, 0x86dcf525, 0x8ae9dba4, 0xf5d6a418, 0xfac79f2e, 0x44cf918b, 0x5d38193b, 0xc17adeaf, 0xa4ad8a86, 0x69527ece, 0x69b75c61, 0x00000000, 0xb00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x4f01, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xce60a6df, 0xb1127c3f, 0x410d7b4a, 0xd9cfc917, 0xd1b2fc52, 0x8be1e03c, 0xde9b256d, 0xff989abd, 0x07e3c46a, 0x780e7d7c, 0xd807ce82, 0x5e5c8f2b, 0x09232f6d, 0x00746338, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0xce60a6df, 0xb1127c3f, 0x410d7b4a, 0xd9cfc917, 0xd1b2fc52, 0x8be1e03c, 0xde9b256d, 0xff989abd, 0x07e3c46a, 0x780e7d7c, 0xd807ce82, 0x5e5c8f2b, 0x09232f6d, 0x00746338, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea18, 0x2f5e, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x18b1240a, 0xa896f734, 0xcd4a40bc, 0x28346a77, 0xbdf09586, 0x3c74ed70, 0x3e255ea3, 0xe55679b4, 0xcc602510, 0x9cd73bfb, 0xf21a6ddb, 0x263a4338, 0x06beb332, 0x0790ac93, 0x00000000, 0xa00001f0 },
                FinalRegs = new uint[] { 0x18b1240a, 0xa896f734, 0xcd4a40bc, 0x28346a77, 0xbdf09586, 0x3c74ed70, 0x3e255ea3, 0xe55679b4, 0xcc602510, 0x9cd73bfb, 0xf21a6ddb, 0x263a4338, 0x06beb332, 0x0790ac93, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x7f41, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x0c25f69d, 0xc32dc28a, 0xf5e2fe71, 0xe46af209, 0x2d1b6ac8, 0xccac564c, 0x567cc561, 0x63707d28, 0xeae934c8, 0xab78e6f6, 0x2d78d86d, 0x76471cdc, 0x9b909f76, 0xa2cc099d, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x0c25f69d, 0xc32dc28a, 0xf5e2fe71, 0xe46af209, 0x2d1b6ac8, 0xccac564c, 0x567cc561, 0x63707d28, 0xeae934c8, 0xab78e6f6, 0x2d78d86d, 0x76471cdc, 0x9b909f76, 0xa2cc099d, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea19, 0x6ff6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6e79c449, 0xe9449bf7, 0x51f8fcb8, 0x138e0b80, 0x715312f2, 0x601ea894, 0xb01f9369, 0x02738c29, 0xee35545f, 0xb61ae4a2, 0xba412f08, 0x1d349e02, 0x56a0dfc0, 0x68cd5bfe, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0x6e79c449, 0xe9449bf7, 0x51f8fcb8, 0x138e0b80, 0x715312f2, 0x601ea894, 0xb01f9369, 0x02738c29, 0xee35545f, 0xb61ae4a2, 0xba412f08, 0x1d349e02, 0x56a0dfc0, 0x68cd5bfe, 0x00000000, 0x100001d0 },
            },
            // ORN (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x3fd0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x77034e34, 0xd0727e58, 0x4748dbf2, 0x2becd551, 0x0a650329, 0x005548fa, 0xcfb963c2, 0x9561c965, 0xf157c850, 0x180a1a6c, 0x0252e103, 0x29d0f25a, 0xbd9bbecd, 0xbfd1347c, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x77034e34, 0xd0727e58, 0x4748dbf2, 0x2becd551, 0x0a650329, 0x005548fa, 0xcfb963c2, 0x9561c965, 0xf157c850, 0x180a1a6c, 0x0252e103, 0x29d0f25a, 0xbd9bbecd, 0xbfd1347c, 0x00000000, 0x300001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x4f72, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x71cc7ddf, 0x67d4ce81, 0x60b04501, 0xcc90b805, 0xc5f34081, 0x5e83e9f7, 0xb5a78fa9, 0xc2497a71, 0xb20cdf14, 0x4de9f773, 0xf79525ec, 0x26534abd, 0xcd7b59d1, 0x5cfc9554, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x71cc7ddf, 0x67d4ce81, 0x60b04501, 0xcc90b805, 0xc5f34081, 0x5e83e9f7, 0xb5a78fa9, 0xc2497a71, 0xb20cdf14, 0x4de9f773, 0xf79525ec, 0x26534abd, 0xcd7b59d1, 0x5cfc9554, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1d, 0x4fa7, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x73259d94, 0xc85643db, 0xbf238eb1, 0x51648d99, 0xce2971c9, 0xf9e0e440, 0x90de33c9, 0xcf8ac8e9, 0xda964c21, 0x539eb057, 0x3a681b87, 0x11993d47, 0x05a1358f, 0xa8282529, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x73259d94, 0xc85643db, 0xbf238eb1, 0x51648d99, 0xce2971c9, 0xf9e0e440, 0x90de33c9, 0xcf8ac8e9, 0xda964c21, 0x539eb057, 0x3a681b87, 0x11993d47, 0x05a1358f, 0xa8282529, 0x00000000, 0xb00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea12, 0x3fdb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd274d46b, 0x8937836f, 0x33b78178, 0xc250b807, 0xd3323d2f, 0x82e03ba2, 0xf93bf1a6, 0xb31e0c74, 0xc9238070, 0x957331d1, 0xfaadd1ee, 0x073d40fb, 0x05b3e8b4, 0x93e5233b, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0xd274d46b, 0x8937836f, 0x33b78178, 0xc250b807, 0xd3323d2f, 0x82e03ba2, 0xf93bf1a6, 0xb31e0c74, 0xc9238070, 0x957331d1, 0xfaadd1ee, 0x073d40fb, 0x05b3e8b4, 0x93e5233b, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea15, 0x5f92, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x24755d61, 0xb65b742d, 0xb46476cf, 0x771a9fcd, 0x465b367f, 0x3daa2a47, 0x6984eeb8, 0x238e3187, 0xa9717261, 0x4592be1d, 0x46d19147, 0x6a6e4dc8, 0x4ddd896f, 0x2f899425, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x24755d61, 0xb65b742d, 0xb46476cf, 0x771a9fcd, 0x465b367f, 0x3daa2a47, 0x6984eeb8, 0x238e3187, 0xa9717261, 0x4592be1d, 0x46d19147, 0x6a6e4dc8, 0x4ddd896f, 0x2f899425, 0x00000000, 0x300001d0 },
            },
            // TEQ (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x2f54, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x36675cac, 0xd9259257, 0x0b8ab9ad, 0xbfef324e, 0xf9623cd6, 0xfc1919ff, 0x616b25f5, 0x2d26a3d3, 0x61eb12c8, 0xbb8d48f0, 0xbfb9c232, 0x10383506, 0x31d10885, 0xf29cb615, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0x36675cac, 0xd9259257, 0x0b8ab9ad, 0xbfef324e, 0xf9623cd6, 0xfc1919ff, 0x616b25f5, 0x2d26a3d3, 0x61eb12c8, 0xbb8d48f0, 0xbfb9c232, 0x10383506, 0x31d10885, 0xf29cb615, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea17, 0x1f43, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf7dce25d, 0xbe296478, 0xe1aee674, 0x0414c126, 0xa258cf11, 0x5347cc5f, 0x6f8ed2c9, 0xed554dbe, 0xd3073560, 0x627dbd64, 0xca8bb3fc, 0x9590e3a9, 0xe4bea6bc, 0x557934a6, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0xf7dce25d, 0xbe296478, 0xe1aee674, 0x0414c126, 0xa258cf11, 0x5347cc5f, 0x6f8ed2c9, 0xed554dbe, 0xd3073560, 0x627dbd64, 0xca8bb3fc, 0x9590e3a9, 0xe4bea6bc, 0x557934a6, 0x00000000, 0x900001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea13, 0x1f5b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x84ad5535, 0xc1f15e65, 0x5ea0078b, 0x79df457d, 0x1c735fe5, 0x06dcfd95, 0x6db96dae, 0x572f572d, 0xac88a919, 0x56d850a6, 0xd5ce3a30, 0x2be992e8, 0x497a47ce, 0x38a74019, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x84ad5535, 0xc1f15e65, 0x5ea0078b, 0x79df457d, 0x1c735fe5, 0x06dcfd95, 0x6db96dae, 0x572f572d, 0xac88a919, 0x56d850a6, 0xd5ce3a30, 0x2be992e8, 0x497a47ce, 0x38a74019, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x3ff3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xa85ea277, 0x6028643d, 0xa5a85f15, 0x47f0f2a5, 0x9b6eebdb, 0x9f064bc7, 0xab59939f, 0x1a278260, 0xb9f91cfa, 0xf913c49c, 0x2b5c0052, 0x1bf2d6dc, 0x81da80a4, 0xced90006, 0x00000000, 0x000001f0 },
                FinalRegs = new uint[] { 0xa85ea277, 0x6028643d, 0xa5a85f15, 0x47f0f2a5, 0x9b6eebdb, 0x9f064bc7, 0xab59939f, 0x1a278260, 0xb9f91cfa, 0xf913c49c, 0x2b5c0052, 0x1bf2d6dc, 0x81da80a4, 0xced90006, 0x00000000, 0xa00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x3f09, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xdf494128, 0xbc975c16, 0x62c66823, 0x95be3737, 0xa07e8778, 0x6ce80cc7, 0xfea03385, 0x4c5bf35a, 0x5cd0bcdf, 0xc47451ab, 0x3849af70, 0x1329c14a, 0xb1f96f79, 0x321eaf12, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0xdf494128, 0xbc975c16, 0x62c66823, 0x95be3737, 0xa07e8778, 0x6ce80cc7, 0xfea03385, 0x4c5bf35a, 0x5cd0bcdf, 0xc47451ab, 0x3849af70, 0x1329c14a, 0xb1f96f79, 0x321eaf12, 0x00000000, 0x300001d0 },
            },
            // EOR (reg)
            new()
            {
                Instructions = new ushort[] { 0xea17, 0x6fc0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x9c803ce5, 0x38e325f9, 0x4d32aea8, 0x0120f77b, 0x8e34b507, 0xee41aabf, 0x7e6d8a0c, 0x761a3f21, 0x99b57f1d, 0x32a4bbf3, 0x9902c1f4, 0xd5e2dd41, 0xe2a08209, 0x2896ceba, 0x00000000, 0xc00001f0 },
                FinalRegs = new uint[] { 0x9c803ce5, 0x38e325f9, 0x4d32aea8, 0x0120f77b, 0x8e34b507, 0xee41aabf, 0x7e6d8a0c, 0x761a3f21, 0x99b57f1d, 0x32a4bbf3, 0x9902c1f4, 0xd5e2dd41, 0xe2a08209, 0x2896ceba, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1c, 0x4f58, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe9c3e9b7, 0x26c9e052, 0x708b6153, 0x72dbdc3c, 0xdd3e922d, 0xd0260aca, 0x38dcf6be, 0x4164575f, 0x5d8e03dc, 0x30bfa694, 0xe72a6609, 0xba632c43, 0x1f768178, 0x6b4f56a6, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0xe9c3e9b7, 0x26c9e052, 0x708b6153, 0x72dbdc3c, 0xdd3e922d, 0xd0260aca, 0x38dcf6be, 0x4164575f, 0x5d8e03dc, 0x30bfa694, 0xe72a6609, 0xba632c43, 0x1f768178, 0x6b4f56a6, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x7f31, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x9205c1f5, 0x80dd7e44, 0x8ecd5272, 0x3d8d3691, 0x35d45cca, 0x4b2d9eb3, 0xa1652285, 0x6a1cb7f1, 0x8e08b99f, 0xdf8f0c57, 0x28dd0dfa, 0xf2c0abbd, 0x167a6539, 0x75163a9d, 0x00000000, 0xa00001f0 },
                FinalRegs = new uint[] { 0x9205c1f5, 0x80dd7e44, 0x8ecd5272, 0x3d8d3691, 0x35d45cca, 0x4b2d9eb3, 0xa1652285, 0x6a1cb7f1, 0x8e08b99f, 0xdf8f0c57, 0x28dd0dfa, 0xf2c0abbd, 0x167a6539, 0x75163a9d, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x4f80, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x96547d6c, 0x7afda83c, 0xb3edea2d, 0x49d05904, 0xb661ef76, 0xf1cce5ff, 0x2986ab1b, 0xcb39b044, 0x88937ad3, 0x962cf736, 0x80d6f109, 0xb73dd0d6, 0xb93f9f60, 0xb93a02c9, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x96547d6c, 0x7afda83c, 0xb3edea2d, 0x49d05904, 0xb661ef76, 0xf1cce5ff, 0x2986ab1b, 0xcb39b044, 0x88937ad3, 0x962cf736, 0x80d6f109, 0xb73dd0d6, 0xb93f9f60, 0xb93a02c9, 0x00000000, 0xb00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea12, 0x1f35, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xb1a94a51, 0xa8679784, 0xd7558ceb, 0x58d63d95, 0x6e5cf7eb, 0x0d40398d, 0xf88fa339, 0xbe88a56f, 0x7180f980, 0x0795ba21, 0x0732b252, 0xa51be7c8, 0x47c02749, 0xb0fbbd9f, 0x00000000, 0x800001f0 },
                FinalRegs = new uint[] { 0xb1a94a51, 0xa8679784, 0xd7558ceb, 0x58d63d95, 0x6e5cf7eb, 0x0d40398d, 0xf88fa339, 0xbe88a56f, 0x7180f980, 0x0795ba21, 0x0732b252, 0xa51be7c8, 0x47c02749, 0xb0fbbd9f, 0x00000000, 0xa00001d0 },
            },
            // CMN (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x4fc5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf4440521, 0x26b151d9, 0x90053d26, 0x8c3bbde1, 0x4a757fa1, 0x34b63513, 0xd1d1a182, 0xa9123bc1, 0xadfbf652, 0xec28d3e6, 0x6ca54af1, 0x385d5637, 0x46280bac, 0x18f38d39, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0xf4440521, 0x26b151d9, 0x90053d26, 0x8c3bbde1, 0x4a757fa1, 0x34b63513, 0xd1d1a182, 0xa9123bc1, 0xadfbf652, 0xec28d3e6, 0x6ca54af1, 0x385d5637, 0x46280bac, 0x18f38d39, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x4fe4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x20abeba4, 0x77f9bd90, 0x5c09b098, 0xdebadd51, 0x6e114dcf, 0xd8212cf8, 0x2a6a57d2, 0xb9667ed8, 0x93817fef, 0x639cd8b4, 0x52b67bce, 0x681ee61c, 0x50bb5414, 0x9f297765, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x20abeba4, 0x77f9bd90, 0x5c09b098, 0xdebadd51, 0x6e114dcf, 0xd8212cf8, 0x2a6a57d2, 0xb9667ed8, 0x93817fef, 0x639cd8b4, 0x52b67bce, 0x681ee61c, 0x50bb5414, 0x9f297765, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1d, 0x6f51, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x12b54404, 0x9ad5df58, 0x8a73af2a, 0xd89dd454, 0x57f5a14b, 0xcee0a06f, 0xb53e67ca, 0x92730368, 0xab12a843, 0x929ae15d, 0xea1e4f49, 0xd7fadfbc, 0x9defdd99, 0xff22c9c8, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0x12b54404, 0x9ad5df58, 0x8a73af2a, 0xd89dd454, 0x57f5a14b, 0xcee0a06f, 0xb53e67ca, 0x92730368, 0xab12a843, 0x929ae15d, 0xea1e4f49, 0xd7fadfbc, 0x9defdd99, 0xff22c9c8, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea15, 0x5f77, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x64dfdc90, 0xd570bc25, 0xae804ff7, 0x491ad040, 0xfe5f6d58, 0x850c1223, 0x39afac7b, 0xcc5a165a, 0x956a9473, 0x89d3941f, 0x6e520e57, 0x804dca75, 0xbd40cde8, 0xff68c0e7, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0x64dfdc90, 0xd570bc25, 0xae804ff7, 0x491ad040, 0xfe5f6d58, 0x850c1223, 0x39afac7b, 0xcc5a165a, 0x956a9473, 0x89d3941f, 0x6e520e57, 0x804dca75, 0xbd40cde8, 0xff68c0e7, 0x00000000, 0xb00001d0 },
            },
            // ADD (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1c, 0x4f0b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x8a81d956, 0x65b2b25b, 0x34dd981e, 0x924542f4, 0xeed4b95a, 0x096d832c, 0x8ddcb715, 0x2df1897b, 0x696d0d5c, 0xfa6853c1, 0xcbb52912, 0xe37a3fda, 0x54dd595d, 0x652e5a2b, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0x8a81d956, 0x65b2b25b, 0x34dd981e, 0x924542f4, 0xeed4b95a, 0x096d832c, 0x8ddcb715, 0x2df1897b, 0x696d0d5c, 0xfa6853c1, 0xcbb52912, 0xe37a3fda, 0x54dd595d, 0x652e5a2b, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea12, 0x6faa, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4b774bbf, 0xe1f3168c, 0xfcdf56d4, 0x0a9feca9, 0x8d832cd1, 0x27af5bb2, 0xe7123c8f, 0x5ae971a8, 0x7c86287f, 0x5e69f0a7, 0x43e672d3, 0xb552a0f4, 0xb8b4fc17, 0xa9cc9a9d, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x4b774bbf, 0xe1f3168c, 0xfcdf56d4, 0x0a9feca9, 0x8d832cd1, 0x27af5bb2, 0xe7123c8f, 0x5ae971a8, 0x7c86287f, 0x5e69f0a7, 0x43e672d3, 0xb552a0f4, 0xb8b4fc17, 0xa9cc9a9d, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea18, 0x2f2c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x907613be, 0x807d314f, 0x10328bb5, 0xf3433f78, 0x0fa6c190, 0x473e8ac5, 0x0019b12e, 0xa24d7590, 0x0fdac8d5, 0x24e4feea, 0xf5eadcbf, 0xdfd73f71, 0xee2c8957, 0xaef12e15, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x907613be, 0x807d314f, 0x10328bb5, 0xf3433f78, 0x0fa6c190, 0x473e8ac5, 0x0019b12e, 0xa24d7590, 0x0fdac8d5, 0x24e4feea, 0xf5eadcbf, 0xdfd73f71, 0xee2c8957, 0xaef12e15, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x3f00, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2f9a7f7c, 0xc6ad7d01, 0x12b220a4, 0x57b4e83c, 0x2132b566, 0xb4afd045, 0x2b5d39bf, 0xceeecd89, 0x724bff21, 0xb527620e, 0xa9fba943, 0xd2d70658, 0x4e69f57b, 0x55df6b8f, 0x00000000, 0xf00001f0 },
                FinalRegs = new uint[] { 0x2f9a7f7c, 0xc6ad7d01, 0x12b220a4, 0x57b4e83c, 0x2132b566, 0xb4afd045, 0x2b5d39bf, 0xceeecd89, 0x724bff21, 0xb527620e, 0xa9fba943, 0xd2d70658, 0x4e69f57b, 0x55df6b8f, 0x00000000, 0x300001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x2fdb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x854d0c91, 0x895ded29, 0x3e81cd89, 0x9ed269cf, 0x8a7354fa, 0x95cfe79f, 0x07248663, 0x3ec81b86, 0x6f1086e0, 0x51b4c91c, 0xb2d0946b, 0x1b81a616, 0x2b03fe57, 0xfbde03fd, 0x00000000, 0x700001f0 },
                FinalRegs = new uint[] { 0x854d0c91, 0x895ded29, 0x3e81cd89, 0x9ed269cf, 0x8a7354fa, 0x95cfe79f, 0x07248663, 0x3ec81b86, 0x6f1086e0, 0x51b4c91c, 0xb2d0946b, 0x1b81a616, 0x2b03fe57, 0xfbde03fd, 0x00000000, 0x300001d0 },
            },
            // ADC (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x3fe4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4333cbde, 0xb61e8731, 0x2121c6ec, 0x796ecc75, 0xb570472d, 0x203aa9ea, 0xdad7bf7e, 0x93654919, 0x9e4c4e08, 0x1e352004, 0x104a06d7, 0x6b9a4b8a, 0xa0bcd372, 0x1713789a, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x4333cbde, 0xb61e8731, 0x2121c6ec, 0x796ecc75, 0xb570472d, 0x203aa9ea, 0xdad7bf7e, 0x93654919, 0x9e4c4e08, 0x1e352004, 0x104a06d7, 0x6b9a4b8a, 0xa0bcd372, 0x1713789a, 0x00000000, 0x300001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x3f40, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xad935c4c, 0x4c7dcbfc, 0x802965ca, 0x1dccd2e8, 0xe960e9e1, 0x3bff0055, 0x204f6a43, 0xe31b010d, 0x33c8f3c5, 0x8e27b912, 0x1351e4a6, 0x90195167, 0xd9112661, 0xee319b51, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0xad935c4c, 0x4c7dcbfc, 0x802965ca, 0x1dccd2e8, 0xe960e9e1, 0x3bff0055, 0x204f6a43, 0xe31b010d, 0x33c8f3c5, 0x8e27b912, 0x1351e4a6, 0x90195167, 0xd9112661, 0xee319b51, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea15, 0x3fe8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x26deb79f, 0x01ac9258, 0x5982ae34, 0x2c3c6df6, 0xabf6a749, 0x319d4b48, 0xce8cca6d, 0xe4b70851, 0x135c049c, 0xd8839d5e, 0xd3171a30, 0xfb09e096, 0x06d67a32, 0x0bab9825, 0x00000000, 0x100001f0 },
                FinalRegs = new uint[] { 0x26deb79f, 0x01ac9258, 0x5982ae34, 0x2c3c6df6, 0xabf6a749, 0x319d4b48, 0xce8cca6d, 0xe4b70851, 0x135c049c, 0xd8839d5e, 0xd3171a30, 0xfb09e096, 0x06d67a32, 0x0bab9825, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea12, 0x2fec, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd2c9ebca, 0x3ace22dc, 0x41603f45, 0x8cc7e2c2, 0xa160b815, 0xac1e7eb8, 0x57c49232, 0x62547a9b, 0xa18407fd, 0x5c549424, 0xf3eec0e1, 0x4185299f, 0x329a9063, 0x649d9b44, 0x00000000, 0x900001f0 },
                FinalRegs = new uint[] { 0xd2c9ebca, 0x3ace22dc, 0x41603f45, 0x8cc7e2c2, 0xa160b815, 0xac1e7eb8, 0x57c49232, 0x62547a9b, 0xa18407fd, 0x5c549424, 0xf3eec0e1, 0x4185299f, 0x329a9063, 0x649d9b44, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea15, 0x2f73, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf3f8be7b, 0x05e02ec4, 0x78bd9fce, 0x6fb329c8, 0x3094f103, 0x7c93c61d, 0xaade3d9f, 0x381bf77c, 0x738389cd, 0xc68dc0e2, 0x60a06e86, 0x9b717afd, 0x9e51e4eb, 0xf7966699, 0x00000000, 0x500001f0 },
                FinalRegs = new uint[] { 0xf3f8be7b, 0x05e02ec4, 0x78bd9fce, 0x6fb329c8, 0x3094f103, 0x7c93c61d, 0xaade3d9f, 0x381bf77c, 0x738389cd, 0xc68dc0e2, 0x60a06e86, 0x9b717afd, 0x9e51e4eb, 0xf7966699, 0x00000000, 0x300001d0 },
            },
            // SBC (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1d, 0x3faa, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd839dfb1, 0x7cc5425c, 0xc55b5255, 0x37b845aa, 0x59f892d4, 0x7ef8e6ec, 0x3491a7fb, 0x87b88546, 0x72b4c444, 0x24cf48d7, 0x7f530fb7, 0x5b243902, 0x6c39c38f, 0x10e3165c, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0xd839dfb1, 0x7cc5425c, 0xc55b5255, 0x37b845aa, 0x59f892d4, 0x7ef8e6ec, 0x3491a7fb, 0x87b88546, 0x72b4c444, 0x24cf48d7, 0x7f530fb7, 0x5b243902, 0x6c39c38f, 0x10e3165c, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea12, 0x1f7d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xcd24aba8, 0x1d7d0523, 0xc150da45, 0x2f7eb96b, 0x9c1ed9af, 0x75056b89, 0x91c818d1, 0x8a07d574, 0x67ff1d4a, 0x6aca4429, 0xc4b5fb7c, 0x21e9ca50, 0xb95cbd15, 0xce3752e7, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0xcd24aba8, 0x1d7d0523, 0xc150da45, 0x2f7eb96b, 0x9c1ed9af, 0x75056b89, 0x91c818d1, 0x8a07d574, 0x67ff1d4a, 0x6aca4429, 0xc4b5fb7c, 0x21e9ca50, 0xb95cbd15, 0xce3752e7, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea10, 0x2fc2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x401285a0, 0x7ab1e348, 0xf48a2615, 0x46d49913, 0xff5e9911, 0x4b4d7920, 0x8e1f921e, 0x05075455, 0x24e4acea, 0x8652e355, 0x11d0fe46, 0x0cfe7c08, 0xf326adee, 0x7fcde7ac, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0x401285a0, 0x7ab1e348, 0xf48a2615, 0x46d49913, 0xff5e9911, 0x4b4d7920, 0x8e1f921e, 0x05075455, 0x24e4acea, 0x8652e355, 0x11d0fe46, 0x0cfe7c08, 0xf326adee, 0x7fcde7ac, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x3f22, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5bb504b3, 0x3dd293c9, 0x3b2b2d7c, 0x30c2876a, 0x1c99a70e, 0x741294e7, 0xfd5f7315, 0x0149b9db, 0x3975aa1c, 0x9269e207, 0xdc42fd14, 0xea6a1c89, 0xa03e7d65, 0x171c30ad, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x5bb504b3, 0x3dd293c9, 0x3b2b2d7c, 0x30c2876a, 0x1c99a70e, 0x741294e7, 0xfd5f7315, 0x0149b9db, 0x3975aa1c, 0x9269e207, 0xdc42fd14, 0xea6a1c89, 0xa03e7d65, 0x171c30ad, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x1f86, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe0f5bbe3, 0xd96f3c62, 0x11944b25, 0x372e4b0e, 0x7c956b35, 0x03df46ac, 0x8f11684b, 0x3044502e, 0x6ebf2992, 0x4f3a0366, 0x9f36f014, 0x4c55f6aa, 0x6473e494, 0x8b6310d6, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0xe0f5bbe3, 0xd96f3c62, 0x11944b25, 0x372e4b0e, 0x7c956b35, 0x03df46ac, 0x8f11684b, 0x3044502e, 0x6ebf2992, 0x4f3a0366, 0x9f36f014, 0x4c55f6aa, 0x6473e494, 0x8b6310d6, 0x00000000, 0x200001d0 },
            },
            // CMP (reg)
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x6f45, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe72e848a, 0x97499b66, 0xcde944bc, 0xf6a7e4e1, 0xd8860029, 0xc55c7e43, 0x58dc13d7, 0x5e1cf6ac, 0x8094a819, 0xdba64363, 0xd8f5423f, 0x6ae843f0, 0x69766600, 0x2814e4e6, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0xe72e848a, 0x97499b66, 0xcde944bc, 0xf6a7e4e1, 0xd8860029, 0xc55c7e43, 0x58dc13d7, 0x5e1cf6ac, 0x8094a819, 0xdba64363, 0xd8f5423f, 0x6ae843f0, 0x69766600, 0x2814e4e6, 0x00000000, 0x800001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x7fd8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x323e5ef9, 0x46e23bdf, 0x8d69d89a, 0x9ffddd37, 0x53f4a07b, 0xe923f9bb, 0x5ea62678, 0x1709127c, 0xc0c20492, 0x0ee47a0c, 0xe137cc2e, 0x7d72db37, 0xca9eb971, 0x4447b224, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x323e5ef9, 0x46e23bdf, 0x8d69d89a, 0x9ffddd37, 0x53f4a07b, 0xe923f9bb, 0x5ea62678, 0x1709127c, 0xc0c20492, 0x0ee47a0c, 0xe137cc2e, 0x7d72db37, 0xca9eb971, 0x4447b224, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea10, 0x3f43, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4cce7ac7, 0x03055e03, 0x479ec669, 0x8b1d9783, 0xa59509e1, 0xa46866ef, 0x654578c4, 0x700e322b, 0xa4191329, 0xb1b8479a, 0xe555a2ce, 0x1ef22472, 0xd41fb2ae, 0x2d794684, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x4cce7ac7, 0x03055e03, 0x479ec669, 0x8b1d9783, 0xa59509e1, 0xa46866ef, 0x654578c4, 0x700e322b, 0xa4191329, 0xb1b8479a, 0xe555a2ce, 0x1ef22472, 0xd41fb2ae, 0x2d794684, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea18, 0x7fd2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xeecfbfb2, 0xbe6288fd, 0x34c0fc94, 0x3a01b105, 0xe7dc6252, 0xc05813fa, 0x6613d82d, 0x90dc7a0c, 0x34637299, 0x58f6d0e7, 0xb151d65e, 0xca975eca, 0xf83b6533, 0x10177f01, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0xeecfbfb2, 0xbe6288fd, 0x34c0fc94, 0x3a01b105, 0xe7dc6252, 0xc05813fa, 0x6613d82d, 0x90dc7a0c, 0x34637299, 0x58f6d0e7, 0xb151d65e, 0xca975eca, 0xf83b6533, 0x10177f01, 0x00000000, 0x400001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x2f6e, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x54dcd0c5, 0x629131da, 0xc4010f0b, 0xf28a7f0f, 0x866d92a3, 0xbb9a1b32, 0xc8c7f425, 0x8d13d61f, 0x1f9a5d13, 0x83e0b2b7, 0x7ef44e14, 0x24c291a3, 0x851cc882, 0x31a056cb, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0x54dcd0c5, 0x629131da, 0xc4010f0b, 0xf28a7f0f, 0x866d92a3, 0xbb9a1b32, 0xc8c7f425, 0x8d13d61f, 0x1f9a5d13, 0x83e0b2b7, 0x7ef44e14, 0x24c291a3, 0x851cc882, 0x31a056cb, 0x00000000, 0x500001d0 },
            },
            // SUB (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x6f56, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf997cd9f, 0xf94c5bd7, 0x5411a289, 0x21311b8f, 0xee8a1fe4, 0x73808b62, 0x4daadf68, 0x14a1c57c, 0x92d98c4c, 0x31f999c9, 0x953b94b9, 0x108acc75, 0xcc38ea73, 0x5dc27e61, 0x00000000, 0x600001f0 },
                FinalRegs = new uint[] { 0xf997cd9f, 0xf94c5bd7, 0x5411a289, 0x21311b8f, 0xee8a1fe4, 0x73808b62, 0x4daadf68, 0x14a1c57c, 0x92d98c4c, 0x31f999c9, 0x953b94b9, 0x108acc75, 0xcc38ea73, 0x5dc27e61, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea19, 0x0f94, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5f8de6f4, 0x09e82020, 0x480dc701, 0xd3303ca3, 0x8739e87a, 0x3da0b6d2, 0x10093787, 0xd30606fc, 0xd81d45da, 0xa66f5e86, 0xd8ddf48e, 0xa8321bd1, 0x62a75c1c, 0x3cffac30, 0x00000000, 0x800001f0 },
                FinalRegs = new uint[] { 0x5f8de6f4, 0x09e82020, 0x480dc701, 0xd3303ca3, 0x8739e87a, 0x3da0b6d2, 0x10093787, 0xd30606fc, 0xd81d45da, 0xa66f5e86, 0xd8ddf48e, 0xa8321bd1, 0x62a75c1c, 0x3cffac30, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea14, 0x7fc6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x001f39cf, 0x76b925c8, 0x292b283a, 0x9d142282, 0x2cda04fa, 0x87f29de5, 0x9e9a98e4, 0x9d48ddbb, 0x9ea329fd, 0x653f2346, 0xfc116785, 0x6e565e16, 0x9a7f8c11, 0x46f1ecbb, 0x00000000, 0xd00001f0 },
                FinalRegs = new uint[] { 0x001f39cf, 0x76b925c8, 0x292b283a, 0x9d142282, 0x2cda04fa, 0x87f29de5, 0x9e9a98e4, 0x9d48ddbb, 0x9ea329fd, 0x653f2346, 0xfc116785, 0x6e565e16, 0x9a7f8c11, 0x46f1ecbb, 0x00000000, 0x500001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea19, 0x5fa5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xfb6a4a50, 0xd074ee0e, 0x599131ef, 0x5db48236, 0xf287fcd1, 0xadea3b9f, 0xf2529f30, 0x6717a5af, 0xe1a3bc40, 0xd92e291b, 0x9b0337eb, 0xcab803ed, 0x255dd8a9, 0xea0e7824, 0x00000000, 0xb00001f0 },
                FinalRegs = new uint[] { 0xfb6a4a50, 0xd074ee0e, 0x599131ef, 0x5db48236, 0xf287fcd1, 0xadea3b9f, 0xf2529f30, 0x6717a5af, 0xe1a3bc40, 0xd92e291b, 0x9b0337eb, 0xcab803ed, 0x255dd8a9, 0xea0e7824, 0x00000000, 0xb00001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1c, 0x6f86, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3a000492, 0xc16be6fa, 0x20053393, 0x597617c9, 0xc30c0ac0, 0x0ed34739, 0xf964a3d4, 0x4dcf9b40, 0x93109692, 0x7ed22040, 0x1f57a26e, 0x008d29d2, 0x99b2dae8, 0xe8a14948, 0x00000000, 0x200001f0 },
                FinalRegs = new uint[] { 0x3a000492, 0xc16be6fa, 0x20053393, 0x597617c9, 0xc30c0ac0, 0x0ed34739, 0xf964a3d4, 0x4dcf9b40, 0x93109692, 0x7ed22040, 0x1f57a26e, 0x008d29d2, 0x99b2dae8, 0xe8a14948, 0x00000000, 0x200001d0 },
            },
            // RSB (reg)
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x6f72, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x9a603e20, 0x10004fe3, 0x8c33bcef, 0x8a23db09, 0x47244c0c, 0x53417661, 0x6486ac8b, 0x5276c43b, 0x577f49a7, 0x34542492, 0xb4ac7c99, 0x5de5cb55, 0x8f6e1d72, 0x077d4a02, 0x00000000, 0xe00001f0 },
                FinalRegs = new uint[] { 0x9a603e20, 0x10004fe3, 0x8c33bcef, 0x8a23db09, 0x47244c0c, 0x53417661, 0x6486ac8b, 0x5276c43b, 0x577f49a7, 0x34542492, 0xb4ac7c99, 0x5de5cb55, 0x8f6e1d72, 0x077d4a02, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x0ff3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6fdd73d7, 0xc6c4c438, 0x772312e2, 0xa57de93f, 0xa1edd64b, 0x8ee41d33, 0x85849a41, 0xac34953a, 0xb3d7c6b5, 0x439ceff1, 0xa3096172, 0x5d8f0654, 0x2e2993a3, 0xca221149, 0x00000000, 0x400001f0 },
                FinalRegs = new uint[] { 0x6fdd73d7, 0xc6c4c438, 0x772312e2, 0xa57de93f, 0xa1edd64b, 0x8ee41d33, 0x85849a41, 0xac34953a, 0xb3d7c6b5, 0x439ceff1, 0xa3096172, 0x5d8f0654, 0x2e2993a3, 0xca221149, 0x00000000, 0x200001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1b, 0x1f34, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xbefd78a5, 0x6d071150, 0xe9ce2c88, 0x2251ed54, 0x30610b17, 0x6428697e, 0xf6e940a4, 0x2395634f, 0xdabff1a3, 0x89988d57, 0x85dd20b0, 0x2ca1311d, 0xcd0748d9, 0xedf55a6f, 0x00000000, 0x800001f0 },
                FinalRegs = new uint[] { 0xbefd78a5, 0x6d071150, 0xe9ce2c88, 0x2251ed54, 0x30610b17, 0x6428697e, 0xf6e940a4, 0x2395634f, 0xdabff1a3, 0x89988d57, 0x85dd20b0, 0x2ca1311d, 0xcd0748d9, 0xedf55a6f, 0x00000000, 0x000001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea16, 0x5f83, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x57018e40, 0xc4027d19, 0x33a32bd7, 0x6a75787a, 0x18f8569a, 0xbbf3a50d, 0x7f35656f, 0x66fbdad7, 0x3aa48c57, 0x39709ea2, 0x5972e4ba, 0xb2c2c772, 0x52f35620, 0x7ef9f1d6, 0x00000000, 0xd00001f0 },
                FinalRegs = new uint[] { 0x57018e40, 0xc4027d19, 0x33a32bd7, 0x6a75787a, 0x18f8569a, 0xbbf3a50d, 0x7f35656f, 0x66fbdad7, 0x3aa48c57, 0x39709ea2, 0x5972e4ba, 0xb2c2c772, 0x52f35620, 0x7ef9f1d6, 0x00000000, 0x100001d0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xea1a, 0x0fd8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x79108ff6, 0x0cb1e662, 0x9eb9ffed, 0x1ee4d3de, 0x7a8fa20a, 0x1db7e216, 0x6fc42752, 0x9cb6cdad, 0xa497a582, 0x654c446f, 0xcbb31efc, 0x601e6995, 0xe328af35, 0x824026e7, 0x00000000, 0xd00001f0 },
                FinalRegs = new uint[] { 0x79108ff6, 0x0cb1e662, 0x9eb9ffed, 0x1ee4d3de, 0x7a8fa20a, 0x1db7e216, 0x6fc42752, 0x9cb6cdad, 0xa497a582, 0x654c446f, 0xcbb31efc, 0x601e6995, 0xe328af35, 0x824026e7, 0x00000000, 0x100001d0 },
            },
        };

        public static readonly PrecomputedThumbTestCase[] ImmTestCases =
        {
            // TST (imm)
            new()
            {
                Instructions = new ushort[] { 0xf018, 0x0fd4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf5a1b919, 0x37ee0ad4, 0xec1bbb30, 0x8345ecb1, 0xf733e93e, 0x76668927, 0xa9b16176, 0x34b9678e, 0xa6167f8b, 0xea4f20a9, 0x45345e75, 0xc8a2ea55, 0xae108472, 0x67b5e3a4, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0xf5a1b919, 0x37ee0ad4, 0xec1bbb30, 0x8345ecb1, 0xf733e93e, 0x76668927, 0xa9b16176, 0x34b9678e, 0xa6167f8b, 0xea4f20a9, 0x45345e75, 0xc8a2ea55, 0xae108472, 0x67b5e3a4, 0x00000001, 0x300001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf41b, 0x1fff, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xfc928b9b, 0xeff1f0a6, 0xe1cd22ba, 0xef1e4a75, 0x10779ef3, 0x3b003004, 0x7a532842, 0x2e71a8c4, 0x62b71ce6, 0x5ffcf3ce, 0xdbe8efa1, 0x86822f2b, 0x560da6b6, 0x46550850, 0x00000001, 0x700001f0 },
                FinalRegs = new uint[] { 0xfc928b9b, 0xeff1f0a6, 0xe1cd22ba, 0xef1e4a75, 0x10779ef3, 0x3b003004, 0x7a532842, 0x2e71a8c4, 0x62b71ce6, 0x5ffcf3ce, 0xdbe8efa1, 0x86822f2b, 0x560da6b6, 0x46550850, 0x00000001, 0x100001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf416, 0x7f97, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd0ba5d0a, 0xc8fa5c53, 0xac3069cb, 0x4be76d89, 0xcc9b4f47, 0x36984914, 0xd49fe0a5, 0x7d80c756, 0x8210fb6d, 0xcb498541, 0xc366597f, 0xacef4405, 0xdf6341a9, 0x6a1124b8, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0xd0ba5d0a, 0xc8fa5c53, 0xac3069cb, 0x4be76d89, 0xcc9b4f47, 0x36984914, 0xd49fe0a5, 0x7d80c756, 0x8210fb6d, 0xcb498541, 0xc366597f, 0xacef4405, 0xdf6341a9, 0x6a1124b8, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf016, 0x0f12, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xb8568ac2, 0x67a7ee09, 0x266abe3b, 0x9d93101d, 0x504b4adb, 0x45838822, 0x62126cc4, 0xf4198159, 0xf24a524c, 0x163fa3e9, 0x3c6d489e, 0xacef0dff, 0x73fc8fdd, 0x9d34fc09, 0x00000001, 0x800001f0 },
                FinalRegs = new uint[] { 0xb8568ac2, 0x67a7ee09, 0x266abe3b, 0x9d93101d, 0x504b4adb, 0x45838822, 0x62126cc4, 0xf4198159, 0xf24a524c, 0x163fa3e9, 0x3c6d489e, 0xacef0dff, 0x73fc8fdd, 0x9d34fc09, 0x00000001, 0x400001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf017, 0x2fd1, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5548cee0, 0x59a88eb4, 0x3624d775, 0x98fe9a19, 0x84d1b83f, 0xed2c476a, 0x046a6aea, 0x0c92fadb, 0xdff5abe1, 0x91a16e82, 0xbb0f8ba4, 0x87c1888c, 0xa2df958e, 0x6cebba03, 0x00000001, 0x400001f0 },
                FinalRegs = new uint[] { 0x5548cee0, 0x59a88eb4, 0x3624d775, 0x98fe9a19, 0x84d1b83f, 0xed2c476a, 0x046a6aea, 0x0c92fadb, 0xdff5abe1, 0x91a16e82, 0xbb0f8ba4, 0x87c1888c, 0xa2df958e, 0x6cebba03, 0x00000001, 0x000001f0 },
            },
            // AND (imm)
            new()
            {
                Instructions = new ushort[] { 0xf403, 0x3ce5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xddd90378, 0xc3a2891f, 0x3d5007a2, 0x0f0c0756, 0xbdc5c113, 0xeee78000, 0x90693126, 0x8763b349, 0xbf6814b4, 0x40160bf9, 0xfff4a26d, 0x16a11d59, 0x26b3b8cc, 0xeb09487f, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0xddd90378, 0xc3a2891f, 0x3d5007a2, 0x0f0c0756, 0xbdc5c113, 0xeee78000, 0x90693126, 0x8763b349, 0xbf6814b4, 0x40160bf9, 0xfff4a26d, 0x16a11d59, 0x00000200, 0xeb09487f, 0x00000001, 0x200001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf00a, 0x177d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xce4db266, 0x67f21be8, 0x0de0c290, 0x2615cfc5, 0x6e3c46cb, 0x44a52240, 0xb12e0470, 0x903e0182, 0x61c32a9a, 0x58bf0753, 0xa9b3c209, 0x68ec1f37, 0x9320a3c9, 0xab952fd9, 0x00000001, 0xa00001f0 },
                FinalRegs = new uint[] { 0xce4db266, 0x67f21be8, 0x0de0c290, 0x2615cfc5, 0x6e3c46cb, 0x44a52240, 0xb12e0470, 0x00310009, 0x61c32a9a, 0x58bf0753, 0xa9b3c209, 0x68ec1f37, 0x9320a3c9, 0xab952fd9, 0x00000001, 0xa00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf014, 0x2913, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4257c8eb, 0xd9eaf199, 0xe9381b0a, 0x39fcb309, 0xb1f24181, 0xebb2b7e4, 0x73799a0f, 0xc70a2fc7, 0xe2af6496, 0xb9014f5b, 0xe22ff568, 0x12dd4afe, 0x6d8544ac, 0x9293d043, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0x4257c8eb, 0xd9eaf199, 0xe9381b0a, 0x39fcb309, 0xb1f24181, 0xebb2b7e4, 0x73799a0f, 0xc70a2fc7, 0xe2af6496, 0x11000100, 0xe22ff568, 0x12dd4afe, 0x6d8544ac, 0x9293d043, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf004, 0x27c2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x8d7ceb35, 0x000e9260, 0x6825b561, 0xbcf66952, 0x3fbc7775, 0xd5afaa83, 0xe4fde261, 0xa35fa71e, 0xbefc5c9f, 0x667d9163, 0x8c2543b0, 0xd8489b89, 0x661ffec5, 0x45ccdaa8, 0x00000001, 0xd00001f0 },
                FinalRegs = new uint[] { 0x8d7ceb35, 0x000e9260, 0x6825b561, 0xbcf66952, 0x3fbc7775, 0xd5afaa83, 0xe4fde261, 0x02004200, 0xbefc5c9f, 0x667d9163, 0x8c2543b0, 0xd8489b89, 0x661ffec5, 0x45ccdaa8, 0x00000001, 0xd00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf008, 0x6ce5, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x78a2a638, 0x54ac2bd7, 0x60ad5509, 0x9c38b11f, 0x8dd109de, 0xd07aea77, 0xf5a0bcfc, 0xbcd81a17, 0x2f159ebd, 0xcc7e8454, 0x04621cce, 0xd0e9eca5, 0xb33f4ba6, 0x1e2bb5b2, 0x00000001, 0xf00001f0 },
                FinalRegs = new uint[] { 0x78a2a638, 0x54ac2bd7, 0x60ad5509, 0x9c38b11f, 0x8dd109de, 0xd07aea77, 0xf5a0bcfc, 0xbcd81a17, 0x2f159ebd, 0xcc7e8454, 0x04621cce, 0xd0e9eca5, 0x07000000, 0x1e2bb5b2, 0x00000001, 0xf00001f0 },
            },
            // BIC (imm)
            new()
            {
                Instructions = new ushort[] { 0xf420, 0x6425, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x28c2ef44, 0x206bdfed, 0x30c780a9, 0x440ab4ab, 0x666fc882, 0x92a4aa1d, 0x3ceb6b36, 0xca757a75, 0xdf2f77b7, 0xae012305, 0x06b5c956, 0x0ff05e78, 0xad918973, 0x73778e28, 0x00000001, 0xe00001f0 },
                FinalRegs = new uint[] { 0x28c2ef44, 0x206bdfed, 0x30c780a9, 0x440ab4ab, 0x28c2e504, 0x92a4aa1d, 0x3ceb6b36, 0xca757a75, 0xdf2f77b7, 0xae012305, 0x06b5c956, 0x0ff05e78, 0xad918973, 0x73778e28, 0x00000001, 0xe00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf430, 0x44ed, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5cd96673, 0xa1c27ac5, 0xbb205490, 0xa844d844, 0x1e66662f, 0x0f259402, 0xbe81472f, 0x36d55b13, 0x02c6d2a2, 0xc39c31b1, 0x59b71936, 0xf1914252, 0xef8188b8, 0x0c18bea1, 0x00000001, 0x700001f0 },
                FinalRegs = new uint[] { 0x5cd96673, 0xa1c27ac5, 0xbb205490, 0xa844d844, 0x5cd90073, 0x0f259402, 0xbe81472f, 0x36d55b13, 0x02c6d2a2, 0xc39c31b1, 0x59b71936, 0xf1914252, 0xef8188b8, 0x0c18bea1, 0x00000001, 0x100001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf038, 0x1334, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe0afc0e0, 0x7ffd49d1, 0x6fe858fa, 0x7564882e, 0xaa895bf1, 0x725f4071, 0x612b9956, 0xb28fd700, 0xf0acd15c, 0xb62cf0bb, 0x1d9d5c1f, 0xdc3942a2, 0x9b3248ea, 0x3b7593ca, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0xe0afc0e0, 0x7ffd49d1, 0x6fe858fa, 0xf088d148, 0xaa895bf1, 0x725f4071, 0x612b9956, 0xb28fd700, 0xf0acd15c, 0xb62cf0bb, 0x1d9d5c1f, 0xdc3942a2, 0x9b3248ea, 0x3b7593ca, 0x00000001, 0xb00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf438, 0x51d7, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x75f070b0, 0x24e410c6, 0x128174fa, 0x04a10755, 0x6728fd35, 0xf6007f21, 0x0cb9efa3, 0x260e061c, 0xc5e02c94, 0x4aaa3354, 0x00796ab8, 0x897274d2, 0xe87dcffc, 0xa47bd3ab, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0x75f070b0, 0xc5e02414, 0x128174fa, 0x04a10755, 0x6728fd35, 0xf6007f21, 0x0cb9efa3, 0x260e061c, 0xc5e02c94, 0x4aaa3354, 0x00796ab8, 0x897274d2, 0xe87dcffc, 0xa47bd3ab, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf438, 0x1db2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x62dcbe9e, 0x55f42016, 0x6689f461, 0x31e32805, 0x1fc90a1e, 0x02e3a47f, 0xf236bafd, 0x65006290, 0x0065bd7f, 0xc1752579, 0x59528615, 0x6ef68c79, 0x138b8bb3, 0x0761d66c, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0x62dcbe9e, 0x55f42016, 0x6689f461, 0x31e32805, 0x1fc90a1e, 0x02e3a47f, 0xf236bafd, 0x65006290, 0x0065bd7f, 0xc1752579, 0x59528615, 0x6ef68c79, 0x138b8bb3, 0x0061bd7f, 0x00000001, 0x000001f0 },
            },
            // MOV (imm)
            new()
            {
                Instructions = new ushort[] { 0xf45f, 0x5032, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6ed98807, 0x536a9cb7, 0x79c2d5bb, 0xc2e9a860, 0x185b4e57, 0x77c1c99f, 0x99a24897, 0xc6cc4ea1, 0xe3d294a6, 0xb8e525ae, 0xca245840, 0x27943892, 0xa76ed6fe, 0xecbcffe8, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0x00002c80, 0x536a9cb7, 0x79c2d5bb, 0xc2e9a860, 0x185b4e57, 0x77c1c99f, 0x99a24897, 0xc6cc4ea1, 0xe3d294a6, 0xb8e525ae, 0xca245840, 0x27943892, 0xa76ed6fe, 0xecbcffe8, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf04f, 0x23f9, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x7e6bf90e, 0x87533a8f, 0x29389323, 0x96a37f30, 0x63e6e58e, 0xb8bb21d0, 0x5bd9ae04, 0x26b7a586, 0xfa359510, 0x131a4e95, 0x5d0adb02, 0xa8148f64, 0xbfe74669, 0xea2cdf2d, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0x7e6bf90e, 0x87533a8f, 0x29389323, 0xf900f900, 0x63e6e58e, 0xb8bb21d0, 0x5bd9ae04, 0x26b7a586, 0xfa359510, 0x131a4e95, 0x5d0adb02, 0xa8148f64, 0xbfe74669, 0xea2cdf2d, 0x00000001, 0xb00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf45f, 0x5351, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd8b5d8b2, 0x7ae44f2b, 0xda909eb2, 0xdb2fe423, 0xb2486971, 0x23427a2c, 0x96c88749, 0xb88d6d78, 0x2f4aa092, 0xbf40760f, 0x88d72a3f, 0x88854e62, 0x8d459486, 0x82a8ba9f, 0x00000001, 0x300001f0 },
                FinalRegs = new uint[] { 0xd8b5d8b2, 0x7ae44f2b, 0xda909eb2, 0x00003440, 0xb2486971, 0x23427a2c, 0x96c88749, 0xb88d6d78, 0x2f4aa092, 0xbf40760f, 0x88d72a3f, 0x88854e62, 0x8d459486, 0x82a8ba9f, 0x00000001, 0x100001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf45f, 0x207c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x73063041, 0x038b5f4a, 0xf7e85421, 0xe935f110, 0x9a5c34de, 0xbc3dbed9, 0x1c57c517, 0x3294067f, 0x01cd78b5, 0xe7bfe428, 0x6e297fce, 0xccb2c833, 0x2e8bb930, 0xeb6e2004, 0x00000001, 0x300001f0 },
                FinalRegs = new uint[] { 0x000fc000, 0x038b5f4a, 0xf7e85421, 0xe935f110, 0x9a5c34de, 0xbc3dbed9, 0x1c57c517, 0x3294067f, 0x01cd78b5, 0xe7bfe428, 0x6e297fce, 0xccb2c833, 0x2e8bb930, 0xeb6e2004, 0x00000001, 0x100001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf45f, 0x5073, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd0d531fe, 0xeb1676df, 0x0acf5912, 0x6061b3fe, 0xecac2ae2, 0x40075143, 0x88a47781, 0x3ecb7baa, 0x6aee3603, 0x53133f32, 0x1e891e57, 0x4d7f8f94, 0xd09c727a, 0x28a79c93, 0x00000001, 0x400001f0 },
                FinalRegs = new uint[] { 0x00003cc0, 0xeb1676df, 0x0acf5912, 0x6061b3fe, 0xecac2ae2, 0x40075143, 0x88a47781, 0x3ecb7baa, 0x6aee3603, 0x53133f32, 0x1e891e57, 0x4d7f8f94, 0xd09c727a, 0x28a79c93, 0x00000001, 0x000001f0 },
            },
            // ORR (imm)
            new()
            {
                Instructions = new ushort[] { 0xf45c, 0x03c9, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc7d9e145, 0xc2a6bd5b, 0x92c8ca0b, 0x1fde4d2f, 0xa4705c7f, 0x47559e93, 0x9f9d3e22, 0x7b3719c0, 0x3746ffc9, 0xa1476ae8, 0x88f45e36, 0x0fc7d2a7, 0xaa94b64c, 0xe9fee33b, 0x00000001, 0x700001f0 },
                FinalRegs = new uint[] { 0xc7d9e145, 0xc2a6bd5b, 0x92c8ca0b, 0xaaf4b64c, 0xa4705c7f, 0x47559e93, 0x9f9d3e22, 0x7b3719c0, 0x3746ffc9, 0xa1476ae8, 0x88f45e36, 0x0fc7d2a7, 0xaa94b64c, 0xe9fee33b, 0x00000001, 0x900001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf441, 0x60ec, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2262c23f, 0xf8ba9254, 0x2a870feb, 0xa66d2c1a, 0xa3bb8f6d, 0x2f754de2, 0xb3b0b9be, 0xc3cf59e8, 0xebaa6300, 0x22ea8a3d, 0xf3bcf0f4, 0xffb0aae8, 0x4982d5ab, 0x4c945119, 0x00000001, 0x800001f0 },
                FinalRegs = new uint[] { 0xf8ba9774, 0xf8ba9254, 0x2a870feb, 0xa66d2c1a, 0xa3bb8f6d, 0x2f754de2, 0xb3b0b9be, 0xc3cf59e8, 0xebaa6300, 0x22ea8a3d, 0xf3bcf0f4, 0xffb0aae8, 0x4982d5ab, 0x4c945119, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf44c, 0x5343, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x844cd5c4, 0x5a244353, 0xd74ff677, 0x25eefc9f, 0xa040f56f, 0x06e237a6, 0x7ccb1c91, 0xc9aa6d32, 0xf9e18bd6, 0xc0780954, 0x955d8f60, 0xa9cb014e, 0x64d583e2, 0x3e50533a, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0x844cd5c4, 0x5a244353, 0xd74ff677, 0x64d5b3e2, 0xa040f56f, 0x06e237a6, 0x7ccb1c91, 0xc9aa6d32, 0xf9e18bd6, 0xc0780954, 0x955d8f60, 0xa9cb014e, 0x64d583e2, 0x3e50533a, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf040, 0x48e2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x24423eae, 0x40f7667e, 0x017f283e, 0x72887399, 0x063f4da0, 0x9b57a1c5, 0x5500c630, 0x6a304cac, 0xf9f10e9a, 0x02cdd193, 0x3f42bccd, 0x3c52ef2e, 0x15858a11, 0x25fd30bf, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0x24423eae, 0x40f7667e, 0x017f283e, 0x72887399, 0x063f4da0, 0x9b57a1c5, 0x5500c630, 0x6a304cac, 0x75423eae, 0x02cdd193, 0x3f42bccd, 0x3c52ef2e, 0x15858a11, 0x25fd30bf, 0x00000001, 0xc00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf455, 0x1de0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc8c22d0e, 0x98a19d05, 0x61b4ea5e, 0x52f6f9a0, 0x2f8ceae4, 0x15649771, 0x61953174, 0x45b9d93f, 0x4e0629af, 0x30f43259, 0x863e8e5c, 0x3310b69e, 0xae5e5b9d, 0xf00e065a, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0xc8c22d0e, 0x98a19d05, 0x61b4ea5e, 0x52f6f9a0, 0x2f8ceae4, 0x15649771, 0x61953174, 0x45b9d93f, 0x4e0629af, 0x30f43259, 0x863e8e5c, 0x3310b69e, 0xae5e5b9d, 0x157c9771, 0x00000001, 0x100001f0 },
            },
            // MVN (imm)
            new()
            {
                Instructions = new ushort[] { 0xf46f, 0x1681, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xb1267a38, 0xe72b03aa, 0x50dc392a, 0xaff74b0d, 0xf83a17ba, 0xb8edf09d, 0x799df56d, 0x1ecbd371, 0xb4a74b9a, 0xe79f52fb, 0xbcec8b62, 0xbb0b01ea, 0x26d72e8c, 0x1d2ac349, 0x00000001, 0x900001f0 },
                FinalRegs = new uint[] { 0xb1267a38, 0xe72b03aa, 0x50dc392a, 0xaff74b0d, 0xf83a17ba, 0xb8edf09d, 0xffefdfff, 0x1ecbd371, 0xb4a74b9a, 0xe79f52fb, 0xbcec8b62, 0xbb0b01ea, 0x26d72e8c, 0x1d2ac349, 0x00000001, 0x900001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf07f, 0x572f, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xa95387ad, 0x256c4ece, 0x32084d7a, 0x84935d58, 0x12f6880b, 0x3b386e47, 0xbeb69796, 0xdcf3fac5, 0xee2f9386, 0x25372541, 0x56499ba6, 0x06fa7586, 0xd114f908, 0x3442736e, 0x00000001, 0x400001f0 },
                FinalRegs = new uint[] { 0xa95387ad, 0x256c4ece, 0x32084d7a, 0x84935d58, 0x12f6880b, 0x3b386e47, 0xbeb69796, 0xd43fffff, 0xee2f9386, 0x25372541, 0x56499ba6, 0x06fa7586, 0xd114f908, 0x3442736e, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf46f, 0x17e3, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd7f2d1e1, 0x1d12b22c, 0x2c26620c, 0xeadb8ead, 0x73560a2e, 0xf521b384, 0x4094f3d2, 0x17ed0f6f, 0x79d30498, 0x6d47211a, 0x8fdfef1d, 0xce6cbfa7, 0x75dc1c1b, 0x2ffd5d28, 0x00000001, 0x700001f0 },
                FinalRegs = new uint[] { 0xd7f2d1e1, 0x1d12b22c, 0x2c26620c, 0xeadb8ead, 0x73560a2e, 0xf521b384, 0x4094f3d2, 0xffe39fff, 0x79d30498, 0x6d47211a, 0x8fdfef1d, 0xce6cbfa7, 0x75dc1c1b, 0x2ffd5d28, 0x00000001, 0x700001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf07f, 0x1431, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4154dce7, 0x66c452e9, 0xff9bea1b, 0x228a4a5e, 0xe9fee66b, 0xddd7117f, 0x303cdcb6, 0x4bdf78a2, 0xfbcca92c, 0x2f628d24, 0x51816529, 0xcdea5042, 0x77a1e4a2, 0x8a745cb4, 0x00000001, 0xa00001f0 },
                FinalRegs = new uint[] { 0x4154dce7, 0x66c452e9, 0xff9bea1b, 0x228a4a5e, 0xffceffce, 0xddd7117f, 0x303cdcb6, 0x4bdf78a2, 0xfbcca92c, 0x2f628d24, 0x51816529, 0xcdea5042, 0x77a1e4a2, 0x8a745cb4, 0x00000001, 0xa00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf07f, 0x73ac, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd7b60274, 0x1ff3baba, 0xfdc8fa51, 0xcfacae9d, 0xd27a8214, 0xbbfb1abf, 0x3766111f, 0x89af2196, 0x4bd14cd6, 0x5af84659, 0xd279ed2f, 0x7abdf656, 0x868a6980, 0xd343d52a, 0x00000001, 0xd00001f0 },
                FinalRegs = new uint[] { 0xd7b60274, 0x1ff3baba, 0xfdc8fa51, 0xfea7ffff, 0xd27a8214, 0xbbfb1abf, 0x3766111f, 0x89af2196, 0x4bd14cd6, 0x5af84659, 0xd279ed2f, 0x7abdf656, 0x868a6980, 0xd343d52a, 0x00000001, 0x900001f0 },
            },
            // ORN (imm)
            new()
            {
                Instructions = new ushort[] { 0xf464, 0x0976, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x02e1c999, 0x40c2ff04, 0x16f00059, 0xd360cd62, 0xcb34f9d2, 0x303b434a, 0x53e0151f, 0x188b36bc, 0x84868958, 0xebad0ada, 0xdcd0cb74, 0x64bc056c, 0xd17a7256, 0xb71ddae3, 0x00000001, 0x500001f0 },
                FinalRegs = new uint[] { 0x02e1c999, 0x40c2ff04, 0x16f00059, 0xd360cd62, 0xcb34f9d2, 0x303b434a, 0x53e0151f, 0x188b36bc, 0x84868958, 0xff3dffff, 0xdcd0cb74, 0x64bc056c, 0xd17a7256, 0xb71ddae3, 0x00000001, 0x500001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf477, 0x3c66, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x71910713, 0xd17f8e75, 0x2652c7ac, 0xfad0527a, 0xc52b726d, 0x29e66793, 0xa1011225, 0x00c8ecc1, 0x48af4edd, 0x5c4e2e67, 0xc5393bd5, 0x702fcda1, 0x4549b1cf, 0x72d5a971, 0x00000001, 0xd00001f0 },
                FinalRegs = new uint[] { 0x71910713, 0xd17f8e75, 0x2652c7ac, 0xfad0527a, 0xc52b726d, 0x29e66793, 0xa1011225, 0x00c8ecc1, 0x48af4edd, 0x5c4e2e67, 0xc5393bd5, 0x702fcda1, 0xfffcefff, 0x72d5a971, 0x00000001, 0x900001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf479, 0x1270, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x91060c85, 0x9b9c9033, 0x771ac325, 0x001e17c8, 0xb1adee43, 0xbaa9ec02, 0xf57f9f83, 0x3fed4e5c, 0x198cc3ea, 0x1a40edde, 0x6844391b, 0xa03319a0, 0xf741e11b, 0xc1892487, 0x00000001, 0x600001f0 },
                FinalRegs = new uint[] { 0x91060c85, 0x9b9c9033, 0xffc3ffff, 0x001e17c8, 0xb1adee43, 0xbaa9ec02, 0xf57f9f83, 0x3fed4e5c, 0x198cc3ea, 0x1a40edde, 0x6844391b, 0xa03319a0, 0xf741e11b, 0xc1892487, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf46f, 0x19d4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4fd5b2bd, 0x1c8f29ae, 0x12803c79, 0x93683874, 0xccd779c1, 0x6978c335, 0x06eb789d, 0xc8b74ef8, 0x51ca145a, 0x242d8047, 0x5036f51f, 0x13a4a4a2, 0x08818ae4, 0xe1687e67, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0x4fd5b2bd, 0x1c8f29ae, 0x12803c79, 0x93683874, 0xccd779c1, 0x6978c335, 0x06eb789d, 0xc8b74ef8, 0x51ca145a, 0xffe57fff, 0x5036f51f, 0x13a4a4a2, 0x08818ae4, 0xe1687e67, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf07f, 0x614f, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x83c9ef5a, 0xb5933c7e, 0x2dc23d71, 0x5723ae27, 0x1218bc2c, 0x456f3dbd, 0xf6ee7d22, 0xde4df878, 0x3e800973, 0x39c4c131, 0x0676384d, 0xef62a558, 0x2acc92f2, 0x9cd71aa1, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0x83c9ef5a, 0xf30fffff, 0x2dc23d71, 0x5723ae27, 0x1218bc2c, 0x456f3dbd, 0xf6ee7d22, 0xde4df878, 0x3e800973, 0x39c4c131, 0x0676384d, 0xef62a558, 0x2acc92f2, 0x9cd71aa1, 0x00000001, 0x900001f0 },
            },
            // TEQ (imm)
            new()
            {
                Instructions = new ushort[] { 0xf49b, 0x2fe4, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc87047c3, 0x99001273, 0xa963adc7, 0xaba3d1a1, 0x4b9c13a0, 0xc42566ba, 0xee0b7ab1, 0x3e4423ec, 0x5d874e97, 0xfffb5799, 0xdb88f462, 0xbdc4a9e2, 0x3933e52b, 0xe1839111, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0xc87047c3, 0x99001273, 0xa963adc7, 0xaba3d1a1, 0x4b9c13a0, 0xc42566ba, 0xee0b7ab1, 0x3e4423ec, 0x5d874e97, 0xfffb5799, 0xdb88f462, 0xbdc4a9e2, 0x3933e52b, 0xe1839111, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf09b, 0x0f59, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6d2c2ac7, 0xdd2b59f4, 0x3fc013f4, 0x567e744e, 0xc4feb096, 0x188454f3, 0xae13338b, 0x66a0a40b, 0xac995945, 0x7e27f097, 0x547cbd54, 0xd2abf0ab, 0x02c08b3e, 0xe6d1283f, 0x00000001, 0x500001f0 },
                FinalRegs = new uint[] { 0x6d2c2ac7, 0xdd2b59f4, 0x3fc013f4, 0x567e744e, 0xc4feb096, 0x188454f3, 0xae13338b, 0x66a0a40b, 0xac995945, 0x7e27f097, 0x547cbd54, 0xd2abf0ab, 0x02c08b3e, 0xe6d1283f, 0x00000001, 0x900001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf494, 0x6f3d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x16342d21, 0xff794fb0, 0x513ba230, 0x7b9e4b2b, 0x9a2d1ba9, 0xebce0dae, 0xe792f2b8, 0xf4932236, 0x0bcd9542, 0x12bcab94, 0x0110b845, 0xdde237b0, 0xa401d5b9, 0xc3162f6d, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0x16342d21, 0xff794fb0, 0x513ba230, 0x7b9e4b2b, 0x9a2d1ba9, 0xebce0dae, 0xe792f2b8, 0xf4932236, 0x0bcd9542, 0x12bcab94, 0x0110b845, 0xdde237b0, 0xa401d5b9, 0xc3162f6d, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf09c, 0x6f59, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x8d9e2002, 0xfa519294, 0x700740d6, 0x29220c73, 0x8f0ad8b2, 0x6ce9d5e8, 0x12f9da7a, 0x286a9813, 0x2be49d73, 0x16241aa1, 0xe096f43b, 0x1fd0d3e2, 0x31791bb5, 0xa4943f4e, 0x00000001, 0xe00001f0 },
                FinalRegs = new uint[] { 0x8d9e2002, 0xfa519294, 0x700740d6, 0x29220c73, 0x8f0ad8b2, 0x6ce9d5e8, 0x12f9da7a, 0x286a9813, 0x2be49d73, 0x16241aa1, 0xe096f43b, 0x1fd0d3e2, 0x31791bb5, 0xa4943f4e, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf094, 0x6f35, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x222e0e7c, 0xa89d1fdf, 0xa7d67bc3, 0x658e1ee9, 0x10b41780, 0x5cd566a4, 0xce03a58a, 0x63fb9a9e, 0x4f5cb2bd, 0x14e72619, 0x296a9bd5, 0xbf7b1fb1, 0x705a45cc, 0xba8540ae, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0x222e0e7c, 0xa89d1fdf, 0xa7d67bc3, 0x658e1ee9, 0x10b41780, 0x5cd566a4, 0xce03a58a, 0x63fb9a9e, 0x4f5cb2bd, 0x14e72619, 0x296a9bd5, 0xbf7b1fb1, 0x705a45cc, 0xba8540ae, 0x00000001, 0x000001f0 },
            },
            // EOR (imm)
            new()
            {
                Instructions = new ushort[] { 0xf496, 0x54fb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x6267728b, 0xc834f7c7, 0xa136a1d6, 0xfd9533e9, 0x096db729, 0x8fff8a73, 0x6a45348e, 0xd52111ed, 0xa5640aff, 0xa4cf82a6, 0x5ab70b5c, 0x5b3c4563, 0xf1a91ab7, 0x5718fdd1, 0x00000001, 0x500001f0 },
                FinalRegs = new uint[] { 0x6267728b, 0xc834f7c7, 0xa136a1d6, 0xfd9533e9, 0x6a452bee, 0x8fff8a73, 0x6a45348e, 0xd52111ed, 0xa5640aff, 0xa4cf82a6, 0x5ab70b5c, 0x5b3c4563, 0xf1a91ab7, 0x5718fdd1, 0x00000001, 0x100001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf08a, 0x339d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xbf1e6da6, 0x2c10408a, 0xe961ddde, 0x5add8306, 0xc266064d, 0xa79569e1, 0x945c28ed, 0xb996f578, 0x68082b6e, 0x14cdd2c7, 0x7d0cc6a2, 0x8d6edfbf, 0x9151e24c, 0x63eaee32, 0x00000001, 0x300001f0 },
                FinalRegs = new uint[] { 0xbf1e6da6, 0x2c10408a, 0xe961ddde, 0xe0915b3f, 0xc266064d, 0xa79569e1, 0x945c28ed, 0xb996f578, 0x68082b6e, 0x14cdd2c7, 0x7d0cc6a2, 0x8d6edfbf, 0x9151e24c, 0x63eaee32, 0x00000001, 0x300001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf490, 0x27d8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd6826a86, 0x39aa35f5, 0x2a8a913e, 0xd9dbf560, 0xcb1a9957, 0xe6779d2f, 0x0eeab3f9, 0xa463d4c2, 0xb3187660, 0xa51778c3, 0x73817179, 0x6d6dae92, 0x864a3e80, 0x43d8f181, 0x00000001, 0xe00001f0 },
                FinalRegs = new uint[] { 0xd6826a86, 0x39aa35f5, 0x2a8a913e, 0xd9dbf560, 0xcb1a9957, 0xe6779d2f, 0x0eeab3f9, 0xd684aa86, 0xb3187660, 0xa51778c3, 0x73817179, 0x6d6dae92, 0x864a3e80, 0x43d8f181, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf485, 0x3d32, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x063885c0, 0xa183a44d, 0x5cb2f961, 0xe44b8670, 0x8ec25495, 0xb8f5a831, 0x1c2fecb4, 0xfc15fcff, 0x28dd902e, 0xf0c875f4, 0x0af03bb5, 0xefe4ba8b, 0x10e57000, 0x4cd51767, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0x063885c0, 0xa183a44d, 0x5cb2f961, 0xe44b8670, 0x8ec25495, 0xb8f5a831, 0x1c2fecb4, 0xfc15fcff, 0x28dd902e, 0xf0c875f4, 0x0af03bb5, 0xefe4ba8b, 0x10e57000, 0xb8f76031, 0x00000001, 0xb00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf095, 0x58e8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5a60610b, 0x4d178413, 0x3b12edd0, 0x23afc7fc, 0x47f0647d, 0x327bd294, 0x52351d80, 0x36733323, 0x490a0d2a, 0x75d5888c, 0x9b45f4e6, 0x89ebf7dc, 0xd278dd78, 0x1b9b0bbd, 0x00000001, 0x400001f0 },
                FinalRegs = new uint[] { 0x5a60610b, 0x4d178413, 0x3b12edd0, 0x23afc7fc, 0x47f0647d, 0x327bd294, 0x52351d80, 0x36733323, 0x2f7bd294, 0x75d5888c, 0x9b45f4e6, 0x89ebf7dc, 0xd278dd78, 0x1b9b0bbd, 0x00000001, 0x000001f0 },
            },
            // CMN (imm)
            new()
            {
                Instructions = new ushort[] { 0xf514, 0x6f12, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xec864396, 0xe2f483b8, 0x18df08c9, 0xae7780ba, 0xd16bc913, 0x892037de, 0x84a3589e, 0x3a468960, 0x004f92e4, 0x6fd793c2, 0x81b048c6, 0xe044e7cf, 0x2199ccda, 0x4667415d, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0xec864396, 0xe2f483b8, 0x18df08c9, 0xae7780ba, 0xd16bc913, 0x892037de, 0x84a3589e, 0x3a468960, 0x004f92e4, 0x6fd793c2, 0x81b048c6, 0xe044e7cf, 0x2199ccda, 0x4667415d, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf517, 0x2f38, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x154548b0, 0x28aed64c, 0x533306b3, 0x8eace432, 0x9a6523f1, 0x22b08ccb, 0xe7fceaf6, 0x45429c2c, 0xf58378c1, 0x0ef49416, 0x88dbd472, 0xf6a35b6c, 0x46b19364, 0x52e4982d, 0x00000001, 0x900001f0 },
                FinalRegs = new uint[] { 0x154548b0, 0x28aed64c, 0x533306b3, 0x8eace432, 0x9a6523f1, 0x22b08ccb, 0xe7fceaf6, 0x45429c2c, 0xf58378c1, 0x0ef49416, 0x88dbd472, 0xf6a35b6c, 0x46b19364, 0x52e4982d, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf116, 0x7fe2, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x30b90186, 0xec7b038f, 0xcb392feb, 0x10c09c2f, 0x8619521d, 0xcf8d7075, 0x108f8f49, 0x6e44275d, 0x1728faed, 0xf2a0b2a4, 0x783cf97f, 0x201d6d0b, 0x317f276d, 0x5a7186e2, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0x30b90186, 0xec7b038f, 0xcb392feb, 0x10c09c2f, 0x8619521d, 0xcf8d7075, 0x108f8f49, 0x6e44275d, 0x1728faed, 0xf2a0b2a4, 0x783cf97f, 0x201d6d0b, 0x317f276d, 0x5a7186e2, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf51b, 0x7f4a, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xedd3c03d, 0xd7cce5a2, 0xfc40b40a, 0x6a9c96f3, 0x40ca8c2d, 0xaa2973e1, 0xd7953408, 0xfa11d2df, 0x7cec28c2, 0x4e523380, 0x007a4ac6, 0x03890c29, 0xd1495b3e, 0xdf1af969, 0x00000001, 0x500001f0 },
                FinalRegs = new uint[] { 0xedd3c03d, 0xd7cce5a2, 0xfc40b40a, 0x6a9c96f3, 0x40ca8c2d, 0xaa2973e1, 0xd7953408, 0xfa11d2df, 0x7cec28c2, 0x4e523380, 0x007a4ac6, 0x03890c29, 0xd1495b3e, 0xdf1af969, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf11c, 0x5f9c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd8c0c360, 0xd50bcc87, 0xe265e8b2, 0xca49cc71, 0xa6bb11c8, 0x13649388, 0x034a4c8c, 0xa3b4c570, 0x014d32ac, 0x1847d102, 0x7fc3678d, 0xb0e0f469, 0x9508a619, 0x2a2372e0, 0x00000001, 0xa00001f0 },
                FinalRegs = new uint[] { 0xd8c0c360, 0xd50bcc87, 0xe265e8b2, 0xca49cc71, 0xa6bb11c8, 0x13649388, 0x034a4c8c, 0xa3b4c570, 0x014d32ac, 0x1847d102, 0x7fc3678d, 0xb0e0f469, 0x9508a619, 0x2a2372e0, 0x00000001, 0x800001f0 },
            },
            // ADD (imm)
            new()
            {
                Instructions = new ushort[] { 0xf10b, 0x00e0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xadc2fe68, 0xa8a14518, 0x5baf5e87, 0x17e2b502, 0x638227c2, 0xba11428f, 0x98c5b963, 0x5b9cbcd3, 0xb4c11f97, 0x0ca6832e, 0xea26efa6, 0x7bb19ec8, 0x8ea04a89, 0x62d597c2, 0x00000001, 0x300001f0 },
                FinalRegs = new uint[] { 0x7bb19fa8, 0xa8a14518, 0x5baf5e87, 0x17e2b502, 0x638227c2, 0xba11428f, 0x98c5b963, 0x5b9cbcd3, 0xb4c11f97, 0x0ca6832e, 0xea26efa6, 0x7bb19ec8, 0x8ea04a89, 0x62d597c2, 0x00000001, 0x300001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf114, 0x7b41, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x9366f694, 0x02670e58, 0x6f3e74b8, 0x567d3e30, 0xeebb29c4, 0xc25ce8e6, 0x942b94c8, 0xc7dccdd9, 0xccfe17a9, 0xeacc4db1, 0xbbbc0fde, 0x248b7093, 0x7f66c92d, 0xfc063cb6, 0x00000001, 0xe00001f0 },
                FinalRegs = new uint[] { 0x9366f694, 0x02670e58, 0x6f3e74b8, 0x567d3e30, 0xeebb29c4, 0xc25ce8e6, 0x942b94c8, 0xc7dccdd9, 0xccfe17a9, 0xeacc4db1, 0xbbbc0fde, 0xf1bf29c4, 0x7f66c92d, 0xfc063cb6, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf51c, 0x21d1, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2b53ae1e, 0x733046c3, 0xbcc33a3a, 0x2f7bbd50, 0xed2a39f2, 0xfee631ec, 0xeb6d3bc3, 0x9f9b502d, 0x30d20f7b, 0xdc75211b, 0xdb234e2b, 0x85008c86, 0x43beb508, 0x6a8303d5, 0x00000001, 0xd00001f0 },
                FinalRegs = new uint[] { 0x2b53ae1e, 0x43c53d08, 0xbcc33a3a, 0x2f7bbd50, 0xed2a39f2, 0xfee631ec, 0xeb6d3bc3, 0x9f9b502d, 0x30d20f7b, 0xdc75211b, 0xdb234e2b, 0x85008c86, 0x43beb508, 0x6a8303d5, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf513, 0x22e8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xbb3b52c1, 0x40ff59f3, 0x05ca09c5, 0x440114be, 0xec3a4022, 0x0ff93d8c, 0x38868879, 0x824d36d8, 0xf513a9d8, 0xf1d0ad5a, 0xc453fdd8, 0xe3dc8d52, 0x1fc5a9ef, 0x809dbe9b, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0xbb3b52c1, 0x40ff59f3, 0x440854be, 0x440114be, 0xec3a4022, 0x0ff93d8c, 0x38868879, 0x824d36d8, 0xf513a9d8, 0xf1d0ad5a, 0xc453fdd8, 0xe3dc8d52, 0x1fc5a9ef, 0x809dbe9b, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf518, 0x68c7, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf9c00a78, 0xf47ee408, 0xdc31e40b, 0x167902da, 0x03b23f2f, 0x6d41efdc, 0x9cb99b17, 0x21bfbf63, 0x9fbe8105, 0x250087d0, 0xe0588965, 0x0f0f669c, 0x2ed04b37, 0xc65c6e2e, 0x00000001, 0x100001f0 },
                FinalRegs = new uint[] { 0xf9c00a78, 0xf47ee408, 0xdc31e40b, 0x167902da, 0x03b23f2f, 0x6d41efdc, 0x9cb99b17, 0x21bfbf63, 0x9fbe873d, 0x250087d0, 0xe0588965, 0x0f0f669c, 0x2ed04b37, 0xc65c6e2e, 0x00000001, 0x800001f0 },
            },
            // ADC (imm)
            new()
            {
                Instructions = new ushort[] { 0xf54d, 0x379a, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x09eb57e5, 0xc9981095, 0x94b0bf26, 0x27080c39, 0x9fba115a, 0xde0e1533, 0xaa5916aa, 0x1bfc2313, 0x32a96f13, 0x5b8f2d6c, 0x9098dcf2, 0x86143a3f, 0x5c004908, 0xd233cd08, 0x00000001, 0x300001f0 },
                FinalRegs = new uint[] { 0x09eb57e5, 0xc9981095, 0x94b0bf26, 0x27080c39, 0x9fba115a, 0xde0e1533, 0xaa5916aa, 0xd2350109, 0x32a96f13, 0x5b8f2d6c, 0x9098dcf2, 0x86143a3f, 0x5c004908, 0xd233cd08, 0x00000001, 0x300001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf149, 0x3a77, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xe32aaf45, 0x05fe0eac, 0x9c782c15, 0x9301164b, 0xa2f59aea, 0xe6b2618b, 0xfceb237a, 0xcfeb98bd, 0xaaa75e8d, 0xbb57f750, 0xd282f40d, 0xa181d4d7, 0x93313b48, 0x9a64c67f, 0x00000001, 0xf00001f0 },
                FinalRegs = new uint[] { 0xe32aaf45, 0x05fe0eac, 0x9c782c15, 0x9301164b, 0xa2f59aea, 0xe6b2618b, 0xfceb237a, 0xcfeb98bd, 0xaaa75e8d, 0xbb57f750, 0x32cf6ec8, 0xa181d4d7, 0x93313b48, 0x9a64c67f, 0x00000001, 0xf00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf549, 0x57c8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x67da941e, 0x5d744410, 0x1f93bf8f, 0xb52727e0, 0x77ce10fe, 0xe7a40291, 0x40ac5a1f, 0x127e801f, 0x68233546, 0xdbe8086f, 0x82b65e68, 0xcf35c09b, 0x8846e02d, 0x5fd54256, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0x67da941e, 0x5d744410, 0x1f93bf8f, 0xb52727e0, 0x77ce10fe, 0xe7a40291, 0x40ac5a1f, 0xdbe82170, 0x68233546, 0xdbe8086f, 0x82b65e68, 0xcf35c09b, 0x8846e02d, 0x5fd54256, 0x00000001, 0x200001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf15c, 0x1649, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x86cf07b1, 0x1c86e00f, 0x8dc39789, 0xe8fafb40, 0xb837bf22, 0xe9c2c765, 0xb9e8b84b, 0xdbc9663e, 0x979b81da, 0xfb7a5636, 0x9012981d, 0xf52ec47c, 0xf98f6294, 0xaf70ff24, 0x00000001, 0xe00001f0 },
                FinalRegs = new uint[] { 0x86cf07b1, 0x1c86e00f, 0x8dc39789, 0xe8fafb40, 0xb837bf22, 0xe9c2c765, 0xf9d862de, 0xdbc9663e, 0x979b81da, 0xfb7a5636, 0x9012981d, 0xf52ec47c, 0xf98f6294, 0xaf70ff24, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf144, 0x6ab6, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x151549e7, 0xbdfa6ced, 0x47ba5025, 0xaba24048, 0x17c38ef8, 0xf92095ec, 0xdccd5b6f, 0xcb3878a5, 0x30d25594, 0x94886d84, 0xaec74633, 0xbe39725f, 0x439d8ef1, 0xcd66a204, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0x151549e7, 0xbdfa6ced, 0x47ba5025, 0xaba24048, 0x17c38ef8, 0xf92095ec, 0xdccd5b6f, 0xcb3878a5, 0x30d25594, 0x94886d84, 0x1d738ef8, 0xbe39725f, 0x439d8ef1, 0xcd66a204, 0x00000001, 0x000001f0 },
            },
            // SBC (imm)
            new()
            {
                Instructions = new ushort[] { 0xf565, 0x3beb, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x23efd21b, 0x78e2f658, 0x37a4e044, 0x8feab92a, 0x9795995f, 0x66c7ddab, 0x1c29040f, 0x10034172, 0x2eede540, 0x961c1400, 0x34cf45b9, 0xdb736f38, 0xd601c8ed, 0x99a714af, 0x00000001, 0xf00001f0 },
                FinalRegs = new uint[] { 0x23efd21b, 0x78e2f658, 0x37a4e044, 0x8feab92a, 0x9795995f, 0x66c7ddab, 0x1c29040f, 0x10034172, 0x2eede540, 0x961c1400, 0x34cf45b9, 0x66c607ab, 0xd601c8ed, 0x99a714af, 0x00000001, 0xf00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf172, 0x1b0d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x596b63ec, 0xb659c798, 0x300ca58e, 0x52200fa5, 0x0db74ebe, 0x01e5b394, 0xed83d480, 0x1a524b19, 0x593d9bd1, 0x1152a751, 0xf3e1cb1c, 0xfb9392e3, 0x08fc2cd9, 0xc3910cf3, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0x596b63ec, 0xb659c798, 0x300ca58e, 0x52200fa5, 0x0db74ebe, 0x01e5b394, 0xed83d480, 0x1a524b19, 0x593d9bd1, 0x1152a751, 0xf3e1cb1c, 0x2fffa580, 0x08fc2cd9, 0xc3910cf3, 0x00000001, 0x200001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf57c, 0x14da, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5eab5df9, 0x3cfdd390, 0xcfd20097, 0xc8986688, 0xa714c17c, 0xc9eee620, 0x6626498e, 0x2de48d3c, 0xc27c794f, 0xf7d0c67f, 0x75b6b9d9, 0xbaf9f630, 0x7bd89fad, 0xe5a2e298, 0x00000001, 0xe00001f0 },
                FinalRegs = new uint[] { 0x5eab5df9, 0x3cfdd390, 0xcfd20097, 0xc8986688, 0x7bbd5fad, 0xc9eee620, 0x6626498e, 0x2de48d3c, 0xc27c794f, 0xf7d0c67f, 0x75b6b9d9, 0xbaf9f630, 0x7bd89fad, 0xe5a2e298, 0x00000001, 0x200001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf57a, 0x6bbf, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xaee56760, 0xa9f9b7d4, 0x9dd85a8c, 0x4c8cea6b, 0x7807b53d, 0xd1349b90, 0xcf320f62, 0x7af6d0c9, 0xc61fac5f, 0x23b43bbd, 0xef7466b3, 0x98e322a8, 0x1e10ae81, 0xb6987dcc, 0x00000001, 0xa00001f0 },
                FinalRegs = new uint[] { 0xaee56760, 0xa9f9b7d4, 0x9dd85a8c, 0x4c8cea6b, 0x7807b53d, 0xd1349b90, 0xcf320f62, 0x7af6d0c9, 0xc61fac5f, 0x23b43bbd, 0xef7466b3, 0xef7460bb, 0x1e10ae81, 0xb6987dcc, 0x00000001, 0xa00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf171, 0x47e8, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x4164d035, 0x72eecb21, 0xbb63329c, 0x8883a249, 0x230b524b, 0x40c059ae, 0x529e2950, 0xd0f7b958, 0xae900a4a, 0xa5a3f2b5, 0xe68da7f3, 0x68fececb, 0x91a2f476, 0x3986b8a0, 0x00000001, 0x400001f0 },
                FinalRegs = new uint[] { 0x4164d035, 0x72eecb21, 0xbb63329c, 0x8883a249, 0x230b524b, 0x40c059ae, 0x529e2950, 0xfeeecb20, 0xae900a4a, 0xa5a3f2b5, 0xe68da7f3, 0x68fececb, 0x91a2f476, 0x3986b8a0, 0x00000001, 0x800001f0 },
            },
            // CMP (imm)
            new()
            {
                Instructions = new ushort[] { 0xf5ba, 0x7f0c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x32876eff, 0x3127746f, 0x25274f4b, 0x50ba4fa5, 0xa3013fb5, 0x4985e2cb, 0x43dad09c, 0xfb6e47f2, 0x673ee708, 0x3beee172, 0x4866bb83, 0x9368060a, 0x565ecf8e, 0xecc22394, 0x00000001, 0xc00001f0 },
                FinalRegs = new uint[] { 0x32876eff, 0x3127746f, 0x25274f4b, 0x50ba4fa5, 0xa3013fb5, 0x4985e2cb, 0x43dad09c, 0xfb6e47f2, 0x673ee708, 0x3beee172, 0x4866bb83, 0x9368060a, 0x565ecf8e, 0xecc22394, 0x00000001, 0x200001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf1b4, 0x5f0c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xabb3ffca, 0x7dbdda85, 0xe413a0d4, 0xf2ea8958, 0x81be2593, 0x8b0997e0, 0x5319660b, 0xd4edc3d0, 0x4b147c71, 0xa60a6a5f, 0x9984a94a, 0xbabe5540, 0x24df8017, 0x1e97e9f5, 0x00000001, 0xd00001f0 },
                FinalRegs = new uint[] { 0xabb3ffca, 0x7dbdda85, 0xe413a0d4, 0xf2ea8958, 0x81be2593, 0x8b0997e0, 0x5319660b, 0xd4edc3d0, 0x4b147c71, 0xa60a6a5f, 0x9984a94a, 0xbabe5540, 0x24df8017, 0x1e97e9f5, 0x00000001, 0x300001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf5b1, 0x0f4b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xf6edbf76, 0xd3f53e21, 0x37679835, 0x6af58147, 0x143dd6be, 0x4f6339d1, 0x0261fa88, 0x38fe033f, 0x1b503fb3, 0x802af22b, 0x22901e74, 0xae61d40e, 0xe1e850ee, 0xe353701c, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0xf6edbf76, 0xd3f53e21, 0x37679835, 0x6af58147, 0x143dd6be, 0x4f6339d1, 0x0261fa88, 0x38fe033f, 0x1b503fb3, 0x802af22b, 0x22901e74, 0xae61d40e, 0xe1e850ee, 0xe353701c, 0x00000001, 0xa00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf5b2, 0x7f57, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x350b2e14, 0xcc603c9e, 0xa7a56491, 0x1f4fe90b, 0x6bb14aba, 0x325154ef, 0xc7655249, 0xe1a6077b, 0x145fc2f0, 0x21e0bc5e, 0x18275d8b, 0x0d8f37f0, 0xfdb56518, 0x405f5649, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0x350b2e14, 0xcc603c9e, 0xa7a56491, 0x1f4fe90b, 0x6bb14aba, 0x325154ef, 0xc7655249, 0xe1a6077b, 0x145fc2f0, 0x21e0bc5e, 0x18275d8b, 0x0d8f37f0, 0xfdb56518, 0x405f5649, 0x00000001, 0xa00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf1b7, 0x0fd0, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x5a7f551b, 0x624d7cb7, 0xdc3e4dab, 0xd242610e, 0x8b7213db, 0x3c4f81df, 0x353e713e, 0x0ffdfd5c, 0xe56efdf9, 0x59330bc2, 0x1b91689c, 0x5497152e, 0x7ce02ab7, 0x0127aeca, 0x00000001, 0xd00001f0 },
                FinalRegs = new uint[] { 0x5a7f551b, 0x624d7cb7, 0xdc3e4dab, 0xd242610e, 0x8b7213db, 0x3c4f81df, 0x353e713e, 0x0ffdfd5c, 0xe56efdf9, 0x59330bc2, 0x1b91689c, 0x5497152e, 0x7ce02ab7, 0x0127aeca, 0x00000001, 0x200001f0 },
            },
            // SUB (imm)
            new()
            {
                Instructions = new ushort[] { 0xf5a6, 0x2902, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x688a6dd6, 0xcabb9832, 0xa187c464, 0xe4474634, 0x19316c88, 0x8b99d147, 0xd67bc441, 0x48cfa0cf, 0x4cd8b792, 0x9593d34d, 0x66b5a570, 0x9065cc35, 0x6ddf1e6f, 0xd49a2985, 0x00000001, 0xf00001f0 },
                FinalRegs = new uint[] { 0x688a6dd6, 0xcabb9832, 0xa187c464, 0xe4474634, 0x19316c88, 0x8b99d147, 0xd67bc441, 0x48cfa0cf, 0x4cd8b792, 0xd673a441, 0x66b5a570, 0x9065cc35, 0x6ddf1e6f, 0xd49a2985, 0x00000001, 0xf00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf1a5, 0x4730, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x69e8d900, 0x3ca9d66e, 0x91788f4e, 0x6e821399, 0xd710747f, 0xc8e72a37, 0xf9f9702f, 0x8e689c3f, 0x87ef1e3c, 0xc8270c3e, 0xd76f0d87, 0x5482900c, 0xec43f474, 0x72617560, 0x00000001, 0x000001f0 },
                FinalRegs = new uint[] { 0x69e8d900, 0x3ca9d66e, 0x91788f4e, 0x6e821399, 0xd710747f, 0xc8e72a37, 0xf9f9702f, 0x18e72a37, 0x87ef1e3c, 0xc8270c3e, 0xd76f0d87, 0x5482900c, 0xec43f474, 0x72617560, 0x00000001, 0x000001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf5bd, 0x7d6b, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x56f27741, 0xdf3a0328, 0x49864f87, 0xd8b84caa, 0xd7a4cc2b, 0x85467faf, 0x6e972a47, 0xc2440b53, 0xa56fc6fa, 0xe86c3322, 0x19e1532d, 0x2984be63, 0xd7302738, 0xbf00369c, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0x56f27741, 0xdf3a0328, 0x49864f87, 0xd8b84caa, 0xd7a4cc2b, 0x85467faf, 0x6e972a47, 0xc2440b53, 0xa56fc6fa, 0xe86c3322, 0x19e1532d, 0x2984be63, 0xd7302738, 0xbf0032f0, 0x00000001, 0xa00001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf5aa, 0x048c, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc48ce68c, 0x33c654cc, 0xa31ea382, 0x398c4095, 0xfff680a5, 0x5886b5f4, 0xb1debf0b, 0x8bd529bb, 0x1354ba05, 0xcf80960a, 0x18582cbe, 0x37ca8996, 0x08f95e3c, 0xc87fdb04, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0xc48ce68c, 0x33c654cc, 0xa31ea382, 0x398c4095, 0x18122cbe, 0x5886b5f4, 0xb1debf0b, 0x8bd529bb, 0x1354ba05, 0xcf80960a, 0x18582cbe, 0x37ca8996, 0x08f95e3c, 0xc87fdb04, 0x00000001, 0x200001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf5ba, 0x13aa, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xd2de6567, 0x993624bf, 0xcfbd492f, 0x7b922424, 0x9fa01912, 0x04225225, 0x3a812a6d, 0xe62792b8, 0xb47cee9a, 0x5694288e, 0x6c669666, 0x213701a6, 0xe423ad2d, 0xc7d5362b, 0x00000001, 0xb00001f0 },
                FinalRegs = new uint[] { 0xd2de6567, 0x993624bf, 0xcfbd492f, 0x6c515666, 0x9fa01912, 0x04225225, 0x3a812a6d, 0xe62792b8, 0xb47cee9a, 0x5694288e, 0x6c669666, 0x213701a6, 0xe423ad2d, 0xc7d5362b, 0x00000001, 0x200001f0 },
            },
            // RSB (imm)
            new()
            {
                Instructions = new ushort[] { 0xf5dc, 0x767d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x8496100e, 0x93007a60, 0x0d33d3dc, 0xd932c4e1, 0x6e05ad8d, 0xde3cc68e, 0x74400ff8, 0xce309ee7, 0x188e0ebd, 0xe10837ab, 0x6b2534e2, 0x280add20, 0x3adc0489, 0x8ef32355, 0x00000001, 0x600001f0 },
                FinalRegs = new uint[] { 0x8496100e, 0x93007a60, 0x0d33d3dc, 0xd932c4e1, 0x6e05ad8d, 0xde3cc68e, 0xc523ff6b, 0xce309ee7, 0x188e0ebd, 0xe10837ab, 0x6b2534e2, 0x280add20, 0x3adc0489, 0x8ef32355, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf1dc, 0x377d, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xc5d7fe20, 0xade81daf, 0xba65ccf8, 0xa101ee00, 0x3a2b70d9, 0xc90238d9, 0xc3b54049, 0x436bf83f, 0x99c96b58, 0xd134cb19, 0x4de47e7f, 0x6a175e2d, 0xd9e49229, 0x174d24ac, 0x00000001, 0x400001f0 },
                FinalRegs = new uint[] { 0xc5d7fe20, 0xade81daf, 0xba65ccf8, 0xa101ee00, 0x3a2b70d9, 0xc90238d9, 0xc3b54049, 0xa398eb54, 0x99c96b58, 0xd134cb19, 0x4de47e7f, 0x6a175e2d, 0xd9e49229, 0x174d24ac, 0x00000001, 0x900001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf5c5, 0x34bd, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0xcea4f214, 0xdc15a8f8, 0xd22be9ef, 0x42c400c5, 0x2fd1fc9b, 0xca724b52, 0x5582071d, 0xd01b7816, 0xa4f5a435, 0xcfd50db5, 0x24e0c80b, 0x7b52178d, 0x11cd0449, 0xd6daa84a, 0x00000001, 0x800001f0 },
                FinalRegs = new uint[] { 0xcea4f214, 0xdc15a8f8, 0xd22be9ef, 0x42c400c5, 0x358f2eae, 0xca724b52, 0x5582071d, 0xd01b7816, 0xa4f5a435, 0xcfd50db5, 0x24e0c80b, 0x7b52178d, 0x11cd0449, 0xd6daa84a, 0x00000001, 0x800001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf1ce, 0x7846, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x3c676ff3, 0x511ea0cb, 0x15e79c80, 0x51a3a8c1, 0x535cc233, 0x6ae729a3, 0x4e5726da, 0x81260fb9, 0x24dd423a, 0x9e81d6c0, 0x812b3bd1, 0x55bd0f44, 0x1871ec65, 0x87087126, 0x00000001, 0x200001f0 },
                FinalRegs = new uint[] { 0x3c676ff3, 0x511ea0cb, 0x15e79c80, 0x51a3a8c1, 0x535cc233, 0x6ae729a3, 0x4e5726da, 0x81260fb9, 0x0317ffff, 0x9e81d6c0, 0x812b3bd1, 0x55bd0f44, 0x1871ec65, 0x87087126, 0x00000001, 0x200001f0 },
            },
            new()
            {
                Instructions = new ushort[] { 0xf5c5, 0x2418, 0x4770, 0xe7fe },
                StartRegs = new uint[] { 0x2bb00694, 0x1c56a4c0, 0xc5cc4a3e, 0xc627c1ab, 0x4e4a8dfc, 0x1f3d71a4, 0x897d57b8, 0x0d4a7208, 0x433b7b88, 0xaaf24fd6, 0x2438f5f8, 0x9875e64a, 0xda475f22, 0x66d5e2e7, 0x00000001, 0x700001f0 },
                FinalRegs = new uint[] { 0x2bb00694, 0x1c56a4c0, 0xc5cc4a3e, 0xc627c1ab, 0xe0cc0e5c, 0x1f3d71a4, 0x897d57b8, 0x0d4a7208, 0x433b7b88, 0xaaf24fd6, 0x2438f5f8, 0x9875e64a, 0xda475f22, 0x66d5e2e7, 0x00000001, 0x700001f0 },
            },
        };
    }
}
