// SDD64 license generator

#include <windows.h>
#include <stdio.h>

#include "private_key.h"

/*************************************************************
 *
 * CORE MQ FUNCTIONS
 *  
 *************************************************************/

UINT64 transform(UINT64 x, UINT64 v[64])
{
   UINT64 result=0;

   for(INT i=0; i<64; ++i)
   {
       UINT64 a = x & v[i];
       UINT64 b = 0;

       while(a)
       {
           b ^= (a & 1);
           a /= 2;
       }

       result |= (b << (63-i));
   }

   return result;
}

UINT64 f_evaluate(UINT64 x)
{
    UINT64 y = 0;

    INT cursor = 0;

    for(INT i=0; i<64; ++i)
        for(INT j=i; j<64; ++j)
        {
            if( (x & ((UINT64)1<<i)) &&
                (x & ((UINT64)1<<j))
            )
            {
                y ^= lsvec[cursor];
            }

            cursor++;
        }

    return y;
}

/*************************************************************
 *
 * GF(2) Functions 
 * 
 *************************************************************/

UINT64 FieldMul(UINT64 multiplicand, UINT64 multiplier, UINT64 modulus)
{
    UINT64 d_mask;
    UINT64 result;

    // measure the degree of the modulus
    d_mask = (UINT64)1 << 63;

    while(!(d_mask & modulus))
        d_mask >>= 1;

    // calculate the product
    result = 0;
        
    while(multiplier)
    {
        // if the current coeff of the muliplier is set
        //   then add the multiplicand to the result
        if(multiplier & 1)
            result ^= multiplicand;

        multiplier >>= 1;

        multiplicand <<= 1;

        // if degree(multiplicand) >= degree(modulus)
        //   then subtract the modulus
        if(multiplicand & d_mask)
            multiplicand ^= modulus;
    }

    return result;
}

UINT64 FieldAdd(UINT64 addend_a, UINT64 addend_b)
{
    return addend_a ^ addend_b;
}

UINT64 FieldDivide(UINT64 dividend, UINT64 divisor, UINT64 *remainder)
{
    UINT64 quotient = 0;

    // position the divisor all the way to the left of a 64-bit data type
    // count the degree of this divisor
    INT degree = 63;

    while(!(divisor & 0x8000000000000000))
    {
        divisor <<= 1;
        degree--;
    }

    quotient = 0;

    for(INT bit = 63; bit >= degree; --bit)
    {
        if(dividend & ((UINT64)1<<bit))
        {
            quotient |= ((UINT64)1<<(bit-degree));

            dividend ^= divisor;
        }

        // slide the divisor to the right across the dividend
        divisor >>= 1;
    }

    *remainder = dividend;

    return quotient;
}

// purposely did not collapse this into a routine that swaps the a and b
// the code is a twice as long this way, but more than twice as clear
//
UINT64 FieldEGCD(UINT64 a, UINT64 b, UINT64 primitive, UINT64 *r, UINT64 *s)
{
    UINT64 sum;
    UINT64 product;
    UINT64 quotient;
    UINT64 remainder;

    // 
    UINT64 gcd = 0;

    // throughout the computation we'll keep track of the current a, b
    // expressed in terms of the original a and b
    UINT64 r_a = 1, s_a = 0; // a = 1*a + 0*b
    UINT64 r_b = 0, s_b = 1; // b = 0*b + 1*a

    while(a && b)
    {
        if(a > b)
        {
            quotient = FieldDivide(a, b, &remainder);
            a = remainder;

            // update the expression for a in terms of original a and b
            //
            product = FieldMul(quotient, r_b, primitive);
            r_a = FieldAdd(r_a, product);       // r_a = r_a - quotient * (r_b);

            product = FieldMul(quotient, s_b, primitive);
            s_a = FieldAdd(s_a, product);       // s_a = s_a - quotient * (s_b);
        }
        else
        {
            quotient = FieldDivide(b, a, &remainder);
            b = remainder;

            // update the expression for b in terms of original a and b
            //
            product = FieldMul(quotient, r_a, primitive);
            r_b = FieldAdd(r_b, product);       // r_a = r_a - quotient * (r_a);

            product = FieldMul(quotient, s_a, primitive);
            s_b = FieldAdd(s_b, product);       // s_b = s_b - quotient * (s_a);
        }
    }

    if(a)
    {
        *s = s_a;
        *r = r_a;
        return a;
    }
    else
    {
        *s = s_b;
        *r = r_b;
        return b;
    }
}

UINT64 FieldInvert(UINT64 a, UINT64 primitive)
{
    UINT64 r, s;

    FieldEGCD(a, primitive, primitive, &r, &s);

    return r;
}

/*************************************************************
 *
 * MAIN
 *
 *************************************************************/

INT main(INT ac, PCHAR *av)
{
    
    // check/parse arg
    //
    if(ac<2)
    {
        printf("provide license ID\n");
        return -1;
    }

    INT id = atoi(av[1]);
    if(id < 1 || id > 1048575)
    {
	printf("licence ID must be in range [1,1048575]\n");
	return -1;
    }

    // license = parity + id
    //
    printf("generating license with id: %d\n", id);

    UINT64 inverse = FieldInvert(id, 0x17A6DC8D861);
    UINT64 license = (inverse << 20) | id;
    printf("parity+id: %016I64X\n", license);

    // decrypt
    //
    printf("encoding (could be a couple minutes)...\n");

    // L2^-1
    //
    #ifdef DO_OUTPUT_TRANSFORM
    license = transform(license, L2_inv);
    #endif

    //
    UINT64 x1[64];  // the x's that pass sbox 1
    CHAR n_x1=0;
    UINT64 x2[64];  // " that pass sbox 2
    CHAR n_x2=0;
    UINT64 x3[64];  // " that pass sbox 3
    CHAR n_x3=0;
    UINT64 x4[64];  // " that pass sbox 4
    CHAR n_x4=0;

    UINT64 x=0;

    #define BOX1_MASK ((UINT64)0x7FFF)          
    #define BOX2_MASK ((UINT64)0x7FFF << 15)
    #define BOX3_MASK ((UINT64)0x7FFF << 30)
    #define BOX4_MASK ((UINT64)0x7FFF << 45)

    n_x1 = n_x2 = n_x3 = n_x4 = 0;

    UINT64 y_box1 = license & BOX1_MASK;   // y00..y14
    UINT64 y_box2 = license & BOX2_MASK;   // y15..y29
    UINT64 y_box3 = license & BOX3_MASK;   // y30..y44
    UINT64 y_box4 = license & BOX4_MASK;   // y45..y59

    // brute box 1
    for(INT i=0; i<=0xFFFF; ++i)
    {
        x = (UINT64)i;

        if((f_evaluate(x) & BOX1_MASK) == y_box1)
            x1[n_x1++]=x;
    }

    if(!n_x1)
        goto abort_search;

    // brute box 2
    for(INT j=0; j<n_x1; ++j)       // for each x satisfying box1
    {
        x = x1[j];

        for(INT i=0; i<=0xFFFF; ++i)
        {
            x &= 0xFFFF;            // keep x_15..x_0
            x |= (UINT64)i << 16;   // 

            if((f_evaluate(x) & BOX2_MASK) == y_box2)
                x2[n_x2++]=x;
        }
    }
    if(!n_x2)
        goto abort_search;

    // brute box 3
    for(INT j=0; j<n_x2; ++j)
    {
        x = x2[j];

        for(INT i=0; i<=0xFFFF; ++i)
        {
            x &= 0xFFFFFFFF;        // keep x_31..x_0
            x |= (UINT64)i << 32;

            if((f_evaluate(x) & BOX3_MASK) == y_box3)
                x3[n_x3++]=x;
        }
    }
    if(!n_x3)
        goto abort_search;

    // brute box 4
    for(INT j=0; j<n_x3; ++j)
    {
        x = x3[j];

        for(INT i=0; i<=0xFFFF; ++i)
        {
            x &= 0xFFFFFFFFFFFF;    // keep x_47..x_0
            x |= (UINT64)i << 48;

            if((f_evaluate(x) & BOX4_MASK) == y_box4)
                x4[n_x4++]=x;
        }
    }

    abort_search:

	if(n_x4)
	{
    	printf("\nEncoded licenses:\n");

	    for(INT i=0; i<n_x4; ++i)
	    {
	        #ifdef DO_INPUT_TRANSFORM
	        x4[i] = transform(x4[i],L1_inv);
	        #endif
	        printf("%016I64X\n", x4[i]);
		}
	}
	else
        printf("no license code exists for this ID! sorry!\n");

    return 0;
}
