﻿using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
namespace RhythmBase.Components
{
	/// <summary>
	/// Represents a rotated rectangle with non-integer coordinates.
	/// </summary>
	[DebuggerDisplay($"{{{nameof(GetDebuggerDisplay)}(),nq}}")]
	public struct RDRotatedRectE(RDPointE? location, RDSizeE? size, RDPointE? pivot, RDExpression? angle = null) : IEquatable<RDRotatedRectE>
	{
		/// <summary>
		/// Gets or sets the location of the rectangle.
		/// </summary>
		public RDPointE? Location { get; set; } = location;

		/// <summary>
		/// Gets or sets the size of the rectangle.
		/// </summary>
		public RDSizeE? Size { get; set; } = size;

		/// <summary>
		/// Gets or sets the pivot point of the rotation.
		/// </summary>
		public RDPointE? Pivot { get; set; } = pivot;

		/// <summary>
		/// Gets or sets the angle of rotation in degrees.
		/// </summary>
		public RDExpression? Angle { get; set; } = angle;

		/// <summary>
		/// Gets the rectangle without rotation.
		/// </summary>
		public readonly RDRectE? WithoutRotate => Location is null && Pivot is null && Size is null ? null : new(Location - (RDSizeE)(Pivot??default), Size);

		/// <summary>
		/// IItializes a new instance of the <see cref="RDRotatedRectE"/> struct.
		/// </summary>
		/// <param name="rect">The rectangle.</param>
		/// <param name="pivot">The pivot point.</param>
		/// <param name="angle">The angle of rotation.</param>
		public RDRotatedRectE(RDRectE? rect, RDPointE? pivot, float angle) : this(rect?.Location, rect?.Size, pivot, angle) { }

		/// <summary>
		/// IItializes a new instance of the <see cref="RDRotatedRectE"/> struct.
		/// </summary>
		/// <param name="rect">The rectangle.</param>
		public RDRotatedRectE(RDRectE? rect) : this(rect?.Location, rect?.Size, null, 0f) { }

		/// <summary>
		/// Inflates the specified rectangle by the specified size.
		/// </summary>
		/// <param name="rect">The rectangle to inflate.</param>
		/// <param name="size">The size to inflate by.</param>
		/// <returns>The inflated rectangle.</returns>
		public static RDRotatedRectE Inflate(RDRotatedRectE rect, RDSizeE size)
		{
			RDRotatedRectE result = rect;
			result.Inflate(size);
			return result;
		}

		/// <summary>
		/// Inflates the specified rectangle by the specified width and height.
		/// </summary>
		/// <param name="rect">The rectangle to inflate.</param>
		/// <param name="x">The width to inflate by.</param>
		/// <param name="y">The height to inflate by.</param>
		/// <returns>The inflated rectangle.</returns>
		public static RDRotatedRectE Inflate(RDRotatedRectE rect, int x, int y)
		{
			RDRotatedRectE result = rect;
			result.Inflate(x, y);
			return result;
		}

		/// <summary>
		/// Offsets the rectangle by the specified x and y values.
		/// </summary>
		/// <param name="x">The x value to offset by.</param>
		/// <param name="y">The y value to offset by.</param>
		public void Offset(RDExpression? x, RDExpression? y) => Location += (RDSizeE)new RDPointE(x, y);

		/// <summary>
		/// Offsets the rectangle by the specified point.
		/// </summary>
		/// <param name="p">The point to offset by.</param>
		public void Offset(RDPointE p) => Offset(p.X, p.Y);

		/// <summary>
		/// Inflates the rectangle by the specified size.
		/// </summary>
		/// <param name="size">The size to inflate by.</param>
		public void Inflate(RDSizeE size)
		{
			Size += new RDSizeE(size.Width * 2, size.Height * 2);
			Pivot -= (RDSizeE)new RDPointE(size.Width, size.Height);
		}

		/// <summary>
		/// Inflates the rectangle by the specified width and height.
		/// </summary>
		/// <param name="width">The width to inflate by.</param>
		/// <param name="height">The height to inflate by.</param>
		public void Inflate(int width, int height)
		{
			Size += new RDSizeE(width * 2, height * 2);
			Pivot -= (RDSizeE)new RDPointE(width, height);
		}
		/// <inheritdoc/>
		public static bool operator ==(RDRotatedRectE rect1, RDRotatedRectE rect2) => rect1.Equals(rect2);
		/// <inheritdoc/>
		public static bool operator !=(RDRotatedRectE rect1, RDRotatedRectE rect2) => !rect1.Equals(rect2);
		/// <inheritdoc/>
		public override readonly bool Equals([NotNullWhen(true)] object? obj) => obj is RDRotatedRectE e && Equals(e);
		/// <inheritdoc/>
		public override readonly int GetHashCode() => HashCode.Combine(Location, Size, Pivot, Angle);
		/// <inheritdoc/>
		public override readonly string ToString() => $"{{Location=[{Location}],Size=[{Size}],Pivot[{Pivot}],Angle={Angle}}}";
		/// <inheritdoc/>
		public readonly bool Equals(RDRotatedRectE other) =>
			Location == other.Location &&
			Size == other.Size && Pivot
			== other.Pivot &&
			Angle == other.Angle;
		private readonly string GetDebuggerDisplay() => ToString();
	}
}