/**
   @file RAK1903_OPT3001_OLED.ino
   @author rakwireless.com
   @brief Setup and read values from a opt3001 sensor using OLED
   @version 0.1
   @date 2021-11-18
   @copyright Copyright (c) 2021
**/

#include <Wire.h>
#include <ClosedCube_OPT3001.h> // Click here to get the library: http://librarymanager/All#OPT3001
#include <U8g2lib.h>       // Click to install library: http://librarymanager/All#u8g2

ClosedCube_OPT3001 g_opt3001;
U8G2_SSD1306_128X64_NONAME_F_HW_I2C u8g2(U8G2_R0);
#define OPT3001_ADDRESS 0x44

void configureSensor()
{
  OPT3001_Config newConfig;

  newConfig.RangeNumber = B1100;
  newConfig.ConvertionTime = B0;
  newConfig.Latch = B1;
  newConfig.ModeOfConversionOperation = B11;

  OPT3001_ErrorCode errorConfig = g_opt3001.writeConfig(newConfig);
  if (errorConfig != NO_ERROR)
    printError("OPT3001 configuration", errorConfig);
  else
  {
    OPT3001_Config sensorConfig = g_opt3001.readConfig();
    Serial.println("OPT3001 Current Config:");
    Serial.println("------------------------------");

    Serial.print("Conversion ready (R):");
    Serial.println(sensorConfig.ConversionReady, HEX);

    Serial.print("Conversion time (R/W):");
    Serial.println(sensorConfig.ConvertionTime, HEX);

    Serial.print("Fault count field (R/W):");
    Serial.println(sensorConfig.FaultCount, HEX);

    Serial.print("Flag high field (R-only):");
    Serial.println(sensorConfig.FlagHigh, HEX);

    Serial.print("Flag low field (R-only):");
    Serial.println(sensorConfig.FlagLow, HEX);

    Serial.print("Latch field (R/W):");
    Serial.println(sensorConfig.Latch, HEX);

    Serial.print("Mask exponent field (R/W):");
    Serial.println(sensorConfig.MaskExponent, HEX);

    Serial.print("Mode of conversion operation (R/W):");
    Serial.println(sensorConfig.ModeOfConversionOperation, HEX);

    Serial.print("Polarity field (R/W):");
    Serial.println(sensorConfig.Polarity, HEX);

    Serial.print("Overflow flag (R-only):");
    Serial.println(sensorConfig.OverflowFlag, HEX);

    Serial.print("Range number (R/W):");
    Serial.println(sensorConfig.RangeNumber, HEX);

    Serial.println("------------------------------");
  }
}

void opt3001_read_data()
{
  
  
  OPT3001 result = g_opt3001.readResult();
  if (result.error == NO_ERROR)
  {
    Serial.print("OPT3001");
    Serial.print(": ");
    Serial.print(result.lux);
    Serial.println(" lux");

    uint16_t luminosity = result.lux;
    char data[32] = {0};
    // display OPT3001 sensor data on OLED
    u8g2.clearBuffer();         // clear the internal memory
    u8g2.setFont(u8g2_font_ncenB10_tr); // choose a suitable font

    memset(data, 0, sizeof(data));
    sprintf(data, "LUX METER");
    u8g2.drawStr(15, 15, data);

    memset(data, 0, sizeof(data));
    sprintf(data, "LI = %.2f lux", result.lux);
    u8g2.drawStr(12, 45, data);
    u8g2.sendBuffer(); // transfer internal memory to the display

  }
  else
  {
    printError("OPT3001", result.error);
  }
}

void printResult(String text, OPT3001 result)
{
  if (result.error == NO_ERROR)
  {
    Serial.print(text);
    Serial.print(": ");
    Serial.print(result.lux);
    Serial.println(" lux");
  }
  else
  {
    printError(text, result.error);
  }
}
void printError(String text, OPT3001_ErrorCode error)
{
  Serial.print(text);
  Serial.print(": [ERROR] Code #");
  Serial.println(error);
}

static const unsigned char PROGMEM bitmap[] =
{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xC0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x7F, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xFC, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFE, 0x00, 0x00, 0xFE, 0xFF, 
  0x03, 0x80, 0x1F, 0x80, 0x0F, 0xE0, 0x07, 0x00, 0x00, 0x00, 0x3E, 0xFF, 
  0x00, 0x00, 0xFC, 0xFF, 0x07, 0xC0, 0x1F, 0x80, 0x0F, 0xF0, 0x03, 0x00, 
  0x00, 0x00, 0xDF, 0xFF, 0x06, 0x00, 0xFC, 0xFF, 0x0F, 0xC0, 0x3F, 0x80, 
  0x0F, 0xF8, 0x01, 0x00, 0x00, 0x00, 0xCE, 0xC7, 0x0E, 0x00, 0x7C, 0xE0, 
  0x0F, 0xC0, 0x3F, 0x80, 0x0F, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x3E, 0x00, 0x3C, 0x80, 0x1F, 0xE0, 0x3F, 0x80, 0x0F, 0x7E, 0x00, 0x00, 
  0x00, 0xCE, 0x7B, 0xFF, 0x7E, 0x00, 0x3C, 0x00, 0x1F, 0xE0, 0x7F, 0x80, 
  0x0F, 0x3F, 0x00, 0x00, 0x00, 0xDE, 0x7B, 0xFF, 0xFC, 0x00, 0x3C, 0x00, 
  0x1F, 0xE0, 0x7D, 0x80, 0x8F, 0x1F, 0x00, 0x00, 0x00, 0xDE, 0x7D, 0xFF, 
  0xF8, 0x00, 0x3C, 0x00, 0x1F, 0xF0, 0x79, 0x80, 0xCF, 0x0F, 0x00, 0x00, 
  0x00, 0xDE, 0x7D, 0xFF, 0xF6, 0x01, 0x3C, 0x00, 0x1F, 0xF0, 0xF8, 0x80, 
  0xEF, 0x07, 0x00, 0x00, 0x00, 0xDE, 0x7D, 0x00, 0xE6, 0x01, 0x3C, 0x80, 
  0x0F, 0xF0, 0xF8, 0x80, 0xFF, 0x07, 0x00, 0x00, 0x00, 0xDE, 0x7B, 0xF0, 
  0xEE, 0x03, 0x7C, 0xE0, 0x0F, 0xF8, 0xF0, 0x80, 0xFF, 0x07, 0x00, 0x00, 
  0x00, 0xDE, 0x7B, 0xF0, 0xCE, 0x03, 0xFC, 0xFF, 0x03, 0x78, 0xF0, 0x81, 
  0xFF, 0x0F, 0x00, 0x00, 0x00, 0xBE, 0x7B, 0xF0, 0xDE, 0x03, 0xFC, 0xFF, 
  0x03, 0x7C, 0xF0, 0x81, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0xBE, 0x03, 0xF0, 
  0xDE, 0x03, 0xFC, 0xFF, 0x07, 0x7C, 0xE0, 0x83, 0x3F, 0x1F, 0x00, 0x00, 
  0x00, 0x7C, 0xFB, 0xF7, 0xDE, 0x03, 0x3C, 0xC0, 0x0F, 0xFC, 0xFF, 0x83, 
  0x1F, 0x3F, 0x00, 0x00, 0x00, 0x78, 0xFA, 0xF7, 0xDE, 0x03, 0x3C, 0x80, 
  0x0F, 0xFE, 0xFF, 0x83, 0x0F, 0x7E, 0x00, 0x00, 0x00, 0xF8, 0xFC, 0xF7, 
  0xDE, 0x03, 0x3C, 0x00, 0x1F, 0xFE, 0xFF, 0x87, 0x0F, 0xFC, 0x00, 0x00, 
  0x00, 0xF0, 0xF9, 0xF7, 0xDE, 0x03, 0x3C, 0x00, 0x1F, 0xFF, 0xFF, 0x87, 
  0x0F, 0xFC, 0x00, 0x00, 0x00, 0xE0, 0x03, 0x00, 0x0C, 0x00, 0x3C, 0x00, 
  0x1F, 0x1F, 0xC0, 0x87, 0x0F, 0xF8, 0x01, 0x00, 0x00, 0xC0, 0x0B, 0x1E, 
  0x03, 0x00, 0x3C, 0x00, 0x1F, 0x1F, 0x80, 0x8F, 0x0F, 0xF0, 0x03, 0x00, 
  0x00, 0x00, 0xFD, 0xDF, 0x07, 0x00, 0x3C, 0x00, 0x9F, 0x0F, 0x80, 0x8F, 
  0x0F, 0xF0, 0x03, 0x00, 0x00, 0x00, 0xFC, 0xEF, 0x03, 0x00, 0x3C, 0x00, 
  0x9F, 0x0F, 0x80, 0x9F, 0x0F, 0xE0, 0x07, 0x00, 0x00, 0x00, 0xF8, 0xF3, 
  0x03, 0x00, 0x3C, 0x00, 0x9F, 0x07, 0x00, 0x9F, 0x0F, 0xC0, 0x0F, 0x00, 
  0x00, 0x00, 0x00, 0xFC, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x7F, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xF8, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x07, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00,
};
void setup()
{
  time_t timeout = millis();
  Serial.begin(115200);
  
  while (!Serial)
  {
    if ((millis() - timeout) < 5000)
    {
      delay(100);
    }
    else
    {
      break;
    }
  }

  u8g2.begin();
  u8g2.clearBuffer();          // clear the internal memory
  u8g2.drawXBM(0,0, 128, 64, bitmap);
  u8g2.sendBuffer();          // transfer internal memory to the display
  /* opt3001 init */
  g_opt3001.begin(OPT3001_ADDRESS);
  Serial.print("OPT3001 Manufacturer ID");
  Serial.println(g_opt3001.readManufacturerID());
  Serial.print("OPT3001 Device ID");
  Serial.println(g_opt3001.readDeviceID());

  configureSensor();
  printResult("High-Limit", g_opt3001.readHighLimit());
  printResult("Low-Limit", g_opt3001.readLowLimit());
  delay(3000);
  
}

void loop()
{
  opt3001_read_data();
  delay(1000);
}
