"use strict";

const fs = require("fs");
const ut = require("../../node/util");
require("../../js/utils");

const schemaTemplate = {
	$schema: "http://json-schema.org/draft-07/schema#",
	$id: "traits.json",
	title: "Traits Schema",
	description: "A generated file containing a list of all traits in Pf2eTools.",
	type: "object",
	version: "0.0.1",
	definitions: {
		anyTrait: {
			type: "string",
			description: "Any Trait in the game.",
			anyOf: [],
			uniqueItems: true,
		},
	},
};

function generateTraitSchema (file) {
	const traitsFile = ut.readJson("./data/traits.json");
	const traits = traitsFile.trait;
	if (traits.map(it => it.name).findDuplicates()) throw new Error(`Duplicate trait: ${traits.map(it => it.name).findDuplicates()}!`);
	let schema = schemaTemplate;

	// For each trait, add it to the schema.
	traits.forEach(trait => {
		// If trait doesn't have ANY category, give it an "Unknown" category.
		if (!trait.categories || trait.categories.length === 0) {
			trait.categories = ["Unknown"];
		}

		// For each category, add it to the schema.
		trait.categories.forEach(category => {
			let categoryName = `${category.replace(/[^a-zA-Z ]/g, "").toCamelCase()}Trait`;

			// If category doesn't exist, create it.
			if (!schema.definitions[categoryName]) {
				schema.definitions[categoryName] = {
					type: "string",
					description: `A list of all ${category} traits.`,
					enum: [],
				};
			}
			schema.definitions[categoryName].enum.push(trait.name.toLowerCase());
			(trait.alias || []).forEach(x => schema.definitions[categoryName].enum.push(x.toLowerCase()));
			schema.definitions[categoryName].enum.sort(SortUtil.ascSort);
		});

		// If the trait has a variable value, create a regex entry for it.
		if (trait.variable) {
			// If the category doesn't exist, create it.
			if (!schema.definitions["variableTraits"]) {
				schema.definitions["variableTraits"] = {
					type: "string",
					description: `A list of all traits with variable values, enclosed in <these 'brackets'>.`,
					anyOf: [],
				};
			}
			schema.definitions["variableTraits"].anyOf.push({
				type: "string",
				pattern: `^${trait.name.toLowerCase()} <.*>$`,
			});
			(trait.alias || []).forEach(x => {
				schema.definitions["variableTraits"].anyOf.push({
					type: "string",
					pattern: `^${x.toLowerCase()} <.*>$`,
				});
			});
		}
	});
	let allCategories = Object.keys(schema.definitions).filter(e => e !== "anyTrait");
	console.log("Created the following categories:", allCategories);

	// Put every category in the "anyTrait" category.
	schema.definitions.anyTrait.anyOf = allCategories.map(category => {
		return {
			$ref: `#/definitions/${category}`,
		};
	}).sort(SortUtil.ascSort);

	// Write it all.
	fs.writeFileSync(file, CleanUtil.getCleanJson(schema), "utf-8");
}

generateTraitSchema(`./test/schema-template/_generated/traits.json`);
console.log("Traits schema created.");
