# 5.4. 自定义层
深度学习成功背后的一个因素是，可以用创造性的方式组合广泛的层，从而设计出适用于各种任务的结构。例如，研究人员发明了专门用于处理图像、文本、序列数据和执行动态编程的层。早晚有一天，你会遇到或要自己发明一个在深度学习框架中还不存在的层。在这些情况下，你必须构建自定义层。在本节中，我们将向你展示如何操作。

# 5.4.1. 不带参数的层
首先，我们构造一个没有任何参数的自定义层。如果你还记得我们在 5.1节 对块的介绍，这应该看起来很眼熟。下面的CenteredLayer类要从其输入中减去均值。要构建它，我们只需继承基础层类并实现正向传播功能。


```python
import paddle
import paddle.nn.functional as F
from paddle import nn

class CenteredLayer(nn.Layer):
    def __init__(self):
        super().__init__()

    def forward(self, X):
        return X - X.mean()
```

让我们通过向其提供一些数据来验证该层是否按预期工作。


```python
layer = CenteredLayer()
layer(paddle.to_tensor([1, 2, 3,4,5],dtype='float32'))
```




    Tensor(shape=[5], dtype=float32, place=CPUPlace, stop_gradient=True,
           [-2., -1.,  0.,  1.,  2.])



现在，我们可以将层作为组件合并到构建更复杂的模型中。


```python
net = nn.Sequential(nn.Linear(8, 128), CenteredLayer())
```

作为额外的健全性检查，我们可以向网络发送随机数据后，检查均值是否为0。由于我们处理的是浮点数，因为存储精度的原因，我们仍然可能会看到一个非常小的非零数。


```python
Y = net(paddle.rand([4, 8]))
Y.mean()
```




    Tensor(shape=[1], dtype=float32, place=CPUPlace, stop_gradient=False,
           [0.00000000])



# 5.4.2. 带参数的层
既然我们知道了如何定义简单的层，那么让我们继续定义具有参数的层，这些参数可以通过训练进行调整。我们可以使用内置函数来创建参数，这些函数提供一些基本的管理功能。比如管理访问、初始化、共享、保存和加载模型参数。这样做的好处之一是，我们不需要为每个自定义层编写自定义序列化程序。

现在，让我们实现自定义版本的全连接层。回想一下，该层需要两个参数，一个用于表示权重，另一个用于表示偏置项。在此实现中，我们使用ReLU作为激活函数。该层需要输入参数：in_units和units，分别表示输入和输出的数量。


```python
class MyLinear(nn.Layer):
    def __init__(self, in_units, units):
        super().__init__()
        self.weight = paddle.create_parameter(shape=(in_units, units), dtype='float32')
        self.bias = paddle.create_parameter(shape=(units,), dtype='float32')
    def forward(self, X):
        linear = paddle.matmul(X, self.weight) + self.bias
        return F.relu(linear)
```

接下来，我们实例化MyLinear类并访问其模型参数。


```python
linear = MyLinear(5, 3)
linear.weight
```




    Parameter containing:
    Tensor(shape=[5, 3], dtype=float32, place=CPUPlace, stop_gradient=False,
           [[-0.75326788, -0.21927863, -0.52132845],
            [ 0.60195655,  0.71570390,  0.26937658],
            [-0.55248767, -0.85275090,  0.34603995],
            [ 0.11166179,  0.33433622,  0.79770511],
            [-0.59691632, -0.43368688, -0.34520006]])



我们可以使用自定义层直接执行正向传播计算。


```python
linear(paddle.randn([2,5]))
```




    Tensor(shape=[2, 3], dtype=float32, place=CPUPlace, stop_gradient=False,
           [[0.39164445, 0.        , 0.54029500],
            [0.08022110, 0.05805540, 0.        ]])



我们还可以使用自定义层构建模型。我们可以像使用内置的全连接层一样使用自定义层。


```python
net = nn.Sequential(MyLinear(64, 8), MyLinear(8, 1))
net(paddle.rand([2,64]))
```




    Tensor(shape=[2, 1], dtype=float32, place=CPUPlace, stop_gradient=False,
           [[0.],
            [0.]])



# 5.4.3. 小结
* 我们可以通过基本层类设计自定义层。这允许我们定义灵活的新层，其行为与库中的任何现有层不同。
* 在自定义层定义完成后，就可以在任意环境和网络结构中调用该自定义层。
* 层可以有局部参数，这些参数可以通过内置函数创建。

# 5.4.4. 练习
* 设计一个接受输入并计算张量降维的层，它返回![](https://ai-studio-static-online.cdn.bcebos.com/79af2c3041e642d9960dd10653fab101ff0c7ce2fb084b96889fb60aed8a8716)
* 设计一个返回输入数据的傅立叶系数前半部分的层。


```python

```
