# 多层感知机的从零开始实现
:label:`sec_mlp_scratch`

我们已经在数学上描述了多层感知机（MLP），现在让我们尝试自己实现一个多层感知机。为了与我们之前使用softmax回归（ :numref:`sec_softmax_scratch` ）获得的结果进行比较，我们将继续使用Fashion-MNIST图像分类数据集（ :numref:`sec_fashion_mnist`）。



```python
import paddle
from paddle import nn
from d2l import torch as d2l
import pd2l
```

    C:\Users\WeiWu-GU\anaconda3\envs\pte\lib\site-packages\ipykernel\pylab\backend_inline.py:164: DeprecationWarning: `configure_inline_support` is deprecated since IPython 7.23, directly use `matplotlib_inline.backend_inline.configure_inline_support()`
      configure_inline_support(ip, backend)



```python
batch_size = 256
train_iter, test_iter = pd2l.load_data_fashion_mnist(batch_size)
```

    C:\Users\WeiWu-GU\anaconda3\envs\pte\lib\site-packages\ipykernel\ipkernel.py:287: DeprecationWarning: `should_run_async` will not call `transform_cell` automatically in the future. Please pass the result to `transformed_cell` argument and any exception that happen during thetransform in `preprocessing_exc_tuple` in IPython 7.17 and above.
      and should_run_async(code)


## 初始化模型参数

回想一下，Fashion-MNIST中的每个图像由$28 \times 28 = 784$个灰度像素值组成。所有图像共分为10个类别。忽略像素之间的空间结构，我们可以将每个图像视为具有784个输入特征和10个类的简单分类数据集。首先，我们将[**实现一个具有单隐藏层的多层感知机，它包含256个隐藏单元**]。注意，我们可以将这两个量都视为超参数。通常，我们选择2的若干次幂作为层的宽度。因为内存在硬件中的分配和寻址方式，这么做往往可以在计算上更高效。

我们用几个张量来表示我们的参数。注意，对于每一层我们都要记录一个权重矩阵和一个偏置向量。跟以前一样，我们要为这些参数的损失的梯度分配内存。



```python
num_inputs, num_outputs, num_hiddens = 784, 10, 256

W1 = paddle.randn([num_inputs, num_hiddens]) * 0.01
W1.stop_gradient = False
b1 = paddle.zeros([num_hiddens])
b1.stop_gradient = False
W2 = paddle.randn([num_hiddens, num_outputs]) * 0.01
W2.stop_gradient = False
b2 = paddle.zeros([num_outputs])
b2.stop_gradient = False

params = [W1, b1, W2, b2]
```

## 激活函数

为了确保我们知道一切是如何工作的，我们将使用最大值函数自己[**实现ReLU激活函数**]，而不是直接调用内置的`relu`函数。



```python
def relu(X):
    a = paddle.zeros_like(X)
    return paddle.max(X, a)
```

## 模型

因为我们忽略了空间结构，所以我们使用`reshape`将每个二维图像转换为一个长度为`num_inputs`的向量。我们只需几行代码就可以(**实现我们的模型**)。



```python
def net(X):
    X = X.reshape((-1, num_inputs))
    H = relu(X@W1 + b1)  # 这里“@”代表矩阵乘法
    return (H@W2 + b2)
```

## 损失函数

为了确保数值稳定性，同时由于我们已经从零实现过softmax函数（ :numref:`sec_softmax_scratch` ），因此在这里我们直接使用高级API中的内置函数来计算softmax和交叉熵损失。回想一下我们之前在 :numref:`subsec_softmax-implementation-revisited` 中对这些复杂问题的讨论。我们鼓励感兴趣的读者查看损失函数的源代码，以加深对实现细节的了解。



```python
loss = nn.CrossEntropyLoss()
```

## 训练

幸运的是，[**多层感知机的训练过程与softmax回归的训练过程完全相同**]。可以直接调用`d2l`包的`train_ch3`函数（参见 :numref:`sec_softmax_scratch` ），将迭代周期数设置为10，并将学习率设置为0.1.



```python
num_epochs, lr = 10, 0.1
updater = paddle.optimizer.SGD(learning_rate=lr, parameters=params)
pd2l.train_ch3(net, train_iter, test_iter, loss, num_epochs, updater)
```


    ---------------------------------------------------------------------------

    ValueError                                Traceback (most recent call last)

    <ipython-input-18-00a2951881b8> in <module>
          1 num_epochs, lr = 10, 0.1
          2 updater = paddle.optimizer.SGD(learning_rate=lr, parameters=params)
    ----> 3 pd2l.train_ch3(net, train_iter, test_iter, loss, num_epochs, updater)
    

    D:\workspace\d2ltopaddle\chapter_multilayer-perceptrons\pd2l.py in train_ch3(net, train_iter, test_iter, loss, num_epochs, updater)
        324                         legend=['train loss', 'train acc', 'test acc'])
        325     for epoch in range(num_epochs):
    --> 326         train_metrics = train_epoch_ch3(net, train_iter, loss, updater)
        327         test_acc = evaluate_accuracy(net, test_iter)
        328         animator.add(epoch + 1, train_metrics + (test_acc,))


    D:\workspace\d2ltopaddle\chapter_multilayer-perceptrons\pd2l.py in train_epoch_ch3(net, train_iter, loss, updater)
        259     for X, y in train_iter:
        260         # Compute gradients and update parameters
    --> 261         y_hat = net(X)
        262         l = loss(y_hat, y)
        263         if isinstance(updater, paddle.optimizer.Optimizer):


    <ipython-input-16-872339a7efc2> in net(X)
          2     X = X.reshape((-1, num_inputs))
          3     H = relu(X@W1 + b1)  # 这里“@”代表矩阵乘法
    ----> 4     return (H@W2 + b2)
    

    ~\anaconda3\envs\pte\lib\site-packages\paddle\fluid\dygraph\math_op_patch.py in __impl__(self, other_var)
        248             axis = -1
        249             math_op = getattr(core.ops, op_type)
    --> 250             return math_op(self, other_var, 'axis', axis)
        251 
        252         comment = OpProtoHolder.instance().get_op_proto(op_type).comment


    ValueError: (InvalidArgument) Input(Y) has error dim.Y'dims[0] must be equal to 1But received Y'dims[0] is 256
      [Hint: Expected y_dims[y_ndim - 2] == N, but received y_dims[y_ndim - 2]:256 != N:1.] (at C:\home\workspace\Paddle_release\paddle/fluid/operators/matmul_v2_op.h:164)
      [operator < matmul_v2 > error]


为了对学习到的模型进行评估，我们将[**在一些测试数据上应用这个模型**]。



```python
d2l.predict_ch3(net, test_iter)
```


![svg](output_14_0.svg)


## 小结

* 我们看到即使手动实现一个简单的多层感知机也是很容易的。
* 然而，如果有大量的层，从零开始实现多层感知机会变得很麻烦（例如，要命名和记录模型的参数）。

## 练习

1. 在所有其他参数保持不变的情况下，更改超参数`num_hiddens`的值，并查看此超参数的变化对结果有何影响。确定此超参数的最佳值。
1. 尝试添加更多的隐藏层，并查看它对结果有何影响。
1. 改变学习速率会如何影响结果？保持模型结构和其他超参数(包括迭代周期数)不变，学习率设置为多少会带来最好的结果？
1. 通过对所有超参数(学习率、迭代周期数、隐藏层数、每层的隐藏单元数)进行联合优化，可以得到的最佳结果是什么？
1. 描述为什么涉及多个超参数更具挑战性。
1. 如果要构建多个超参数的搜索方法，你能想到的最聪明的策略是什么？


[Discussions](https://discuss.d2l.ai/t/1804)

