# 3.3 线性回归的简洁实现
使用深度学习框架简洁实现上一节中的线性回归模型


```python
!pip install -U d2l
!python -m pip install paddlepaddle -i https://mirror.baidu.com/pypi/simple
```


```python
import random
import numpy as np
# torch
import torch
from d2l import torch as d2l
from torch.utils import data
# paddle
import paddle
import matplotlib.pyplot as plt
```


```python
def synthetic_data(w, b, num_examples):
  """Generate y = Xw + b + noise."""
  X = np.random.normal(0, 1, (num_examples, len(w)))
  y = np.dot(X, w) + b
  y += np.random.normal(0, 0.01, y.shape)
  X = paddle.to_tensor(X)
  y = paddle.to_tensor(y)
  return X, y.reshape((-1, 1))
```

## 3.3.1 生成数据集
### torch版


```python
torch_w = torch.tensor([2, -3.4])
torch_b = 4.2
torch_features, torch_labels = d2l.synthetic_data(torch_w, torch_b, 1000)
```


```python
print(torch_features.shape, torch_labels.shape)
```

### paddle版


```python
paddle_w = paddle.to_tensor([2,-3.4])
paddle_b = 4.2
paddle_features, paddle_labels = synthetic_data(paddle_w, paddle_b, 1000)
```


```python
print(paddle_features.shape, paddle_labels.shape)
```

## 3.3.2 读取数据集
可以调用框架中现有的API来读取数据。我们将features和labels作为API的参数传递，并在实例化数据迭代器对象时指定batch_size。此外，布尔值is_train表示是否希望数据迭代器对象在每个迭代周期内打乱数据。

### torch版


```python
def torch_load_array(data_arrays, batch_size, is_train=True):
    """构造一个PyTorch数据迭代器。"""
    # print(len(data_arrays))
    dataset = data.TensorDataset(*data_arrays)
    return data.DataLoader(dataset, batch_size, shuffle=is_train)

torch_batch_size = 10
torch_data_iter = torch_load_array((torch_features, torch_labels), torch_batch_size)
```

使用data_iter的方式与我们在 3.2节中使用data_iter函数的方式相同。为了验证是否正常工作，让我们读取并打印第一个小批量样本。 与 3.2节不同，这里我们使用iter构造Python迭代器，并使用next从迭代器中获取第一项。


```python
# print(type(torch_data_iter))
next(iter(torch_data_iter))
```

### paddle版


```python
# 测试TensorDataset
from paddle.io import TensorDataset
input_np = np.random.random([2, 3, 4]).astype('float32')
input = paddle.to_tensor(input_np)
label_np = np.random.random([2, 1]).astype('int32')
label = paddle.to_tensor(label_np)

dataset = TensorDataset([input, label])
print(len(dataset))
loader = DataLoader(dataset, 10, shuffle=True)

def test_load_array(data_arrays, batch_size, is_train=True):
    """构造一个PyTorch数据迭代器。"""
    print(len(data_arrays))
    dataset = TensorDataset(data_arrays)
    return DataLoader(dataset, batch_size, shuffle=is_train)

test_batch_size = 10
test_data_iter = test_load_array((input, label), test_batch_size)
```


```python
from paddle.io import TensorDataset, DataLoader
def paddle_load_array(data_arrays, batch_size, is_train=True):
    """构造一个paddle数据迭代器。"""
    print(len(data_arrays))
    dataset = TensorDataset(data_arrays)
    return DataLoader(dataset, batch_size, shuffle=is_train)

paddle_batch_size = 10
paddle_data_iter = paddle_load_array((paddle_features, paddle_labels), paddle_batch_size)
```


```python
print(type(paddle_data_iter))
for X, y in paddle_data_iter:
  print(X, y)
# next(iter(paddle_data_iter))
```

## 3.3.3 定义模型
当我们在 3.2节中实现线性回归时，我们明确定义了模型参数变量，并编写了计算的代码，这样通过基本的线性代数运算得到输出。但是，如果模型变得更加复杂，而且当你几乎每天都需要实现模型时，你会想简化这个过程。这种情况类似于从头开始编写自己的博客。做一两次是有益的、有启发性的，但如果每次你每需要一个博客就花一个月的时间重新发明轮子，那你将是一个糟糕的网页开发者。

对于标准操作，我们可以使用框架的预定义好的层。这使我们只需关注使用哪些层来构造模型，而不必关注层的实现细节。我们首先定义一个模型变量net，它是一个Sequential类的实例。Sequential类为串联在一起的多个层定义了一个容器。当给定输入数据，Sequential实例将数据传入到第一层，然后将第一层的输出作为第二层的输入，依此类推。在下面的例子中，我们的模型只包含一个层，因此实际上不需要Sequential。但是由于以后几乎所有的模型都是多层的，在这里使用Sequential会让你熟悉标准的流水线。

回顾 图3.1.2中的单层网络架构，这一单层被称为全连接层（fully-connected layer），因为它的每一个输入都通过矩阵-向量乘法连接到它的每个输出。

### torch版

在PyTorch中，全连接层在Linear类中定义。值得注意的是，我们将两个参数传递到nn.Linear中。第一个指定输入特征形状，即2，第二个指定输出特征形状，输出特征形状为单个标量，因此为1。


```python
# `nn` 是神经网络的缩写
from torch import nn

torch_net = nn.Sequential(nn.Linear(2, 1))
```

### paddle版


```python
paddle_net = paddle.nn.Sequential(paddle.nn.Linear(in_features=2, out_features=1))
print(paddle_net)
```

## 3.3.4 初始化模型参数
在使用net之前，我们需要初始化模型参数。如在线性回归模型中的权重和偏置。 深度学习框架通常有预定义的方法来初始化参数。 在这里，我们指定每个权重参数应该从均值为0、标准差为0.01的正态分布中随机采样，偏置参数将初始化为零。

### torch版

正如我们在构造nn.Linear时指定输入和输出尺寸一样。现在我们直接访问参数以设定初始值。我们通过net[0]选择网络中的第一个图层，然后使用weight.data和bias.data方法访问参数。然后使用替换方法normal_和fill_来重写参数值。


```python
torch_net[0].weight.data.normal_(0, 0.01)
torch_net[0].bias.data.fill_(0)
```

### paddle版


```python
# 先放着
# paddle_net[0].weight.data.normal_(0, 0.01)
# paddle_net[0].weight.data
# paddle_net[0].bias.data.fill_(0)
```

## 3.3.5 定义损失函数
计算均方误差使用的是MSELoss类，也称为平方 𝐿2
 范数。默认情况下，它返回所有样本损失的平均值。

### torch版


```python
torch_loss = nn.MSELoss()
```

### paddle版


```python
paddle_loss = paddle.nn.MSELoss()
```

## 3.3.6 定义优化算法

小批量随机梯度下降算法是一种优化神经网络的标准工具，PyTorch在optim模块中实现了该算法的许多变种。当我们实例化SGD实例时，我们要指定优化的参数（可通过net.parameters()从我们的模型中获得）以及优化算法所需的超参数字典。小批量随机梯度下降只需要设置lr值，这里设置为0.03。

### torch版


```python
torch_trainer = torch.optim.SGD(torch_net.parameters(), lr=0.03)
```

### paddle版


```python
paddle_trainer = paddle.optimizer.SGD(learning_rate=0.03, parameters=paddle_net.parameters())
```

## 3.3.7 训练

通过深度学习框架的高级API来实现我们的模型只需要相对较少的代码。 我们不必单独分配参数、不必定义我们的损失函数，也不必手动实现小批量随机梯度下降。 当我们需要更复杂的模型时，高级API的优势将大大增加。 当我们有了所有的基本组件，训练过程代码与我们从零开始实现时所做的非常相似。

回顾一下：在每个迭代周期里，我们将完整遍历一次数据集（train_data），不停地从中获取一个小批量的输入和相应的标签。对于每一个小批量，我们会进行以下步骤:
- 通过调用net(X)生成预测并计算损失l（正向传播）。
- 通过进行反向传播来计算梯度。
- 通过调用优化器来更新模型参数。

为了更好的衡量训练效果，我们计算每个迭代周期后的损失，并打印它来监控训练过程。

### torch版



```python
num_epochs = 3
for epoch in range(num_epochs):
    for X, y in torch_data_iter:
        l = torch_loss(torch_net(X) ,y)
        torch_trainer.zero_grad()
        l.backward()
        torch_trainer.step()
    l = torch_loss(torch_net(torch_features), torch_labels)
    print(f'epoch {epoch + 1}, loss {l:f}')
```

下面我们比较生成数据集的真实参数和通过有限数据训练获得的模型参数。 要访问参数，我们首先从net访问所需的层，然后读取该层的权重和偏置。 正如在从零开始实现中一样，我们估计得到的参数与生成数据的真实参数非常接近。


```python
w = torch_net[0].weight.data
print('w的估计误差：', torch_w - w.reshape(torch_w.shape))
b = net[0].bias.data
print('b的估计误差：', torch_b - b)
```

### paddle版


```python
num_epochs = 3
for epoch in range(num_epochs):
    for X, y in paddle_data_iter:
        l = paddle_loss(paddle_net(X) ,y)
        paddle_trainer.zero_grad()
        l.backward()
        paddle_trainer.step()
    l = paddle_loss(paddle_net(paddle_features), paddle_labels)
    print(f'epoch {epoch + 1}, loss {l:f}')
# for epoch in range(num_epochs):
#   y_predict = paddle_net(paddle_features)
#   loss=paddle_loss(y_predict, paddle_labels)
#   loss.backward()
#   paddle_trainer .step()
#   paddle_trainer .clear_grad()
#   print(f'epoch {epoch + 1}, loss {loss:f}')
```
