import React, { PureComponent } from 'react';
import { createPortal } from 'react-dom';
import Types from 'prop-types';
import clickOutside from 'react-click-outside';
import AutocompleteWidget from './AutocompleteWidget';
import { getSlateEditor } from '../../utils';
import { getAccents, getPosAfterEmphasis } from '../../slate/transforms';
import { matchUnderCursor } from './utils';

const escapeCode = 27;
const arrowUpCode = 38;
const arrowDownCode = 40;
const enterCode = 13;
const tabCode = 9;

@clickOutside
class AutocompleteContainer extends PureComponent {
  static propTypes = {
    state: Types.object,
    locale: Types.string,
    options: Types.object,
    onChange: Types.func,
    onScroll: Types.func,
    onMouseUp: Types.func,
    onToggle: Types.func,
  };

  static defaultProps = {
    state: {},
    locale: 'en',
    options: {},
    onChange: () => {},
    onScroll: () => {},
    onMouseUp: () => {},
    onToggle: () => {}
  };

  state = {
    show: false,
    selectedItem: 0,
    items: [],
    ref: null
  }

  componentDidMount = () => {
    this._isMounted = true;
    this.searchItems(this.props);
  };

  componentWillReceiveProps = (nextProps) => {
    this.searchItems(nextProps);
  };

  componentWillUnmount() {
    this._isMounted = false;
  }

  matchExtension = (extensions, token) => {
    for (let i = 0, count = extensions.length; i < count; i++) {
      const extension = extensions[i];

      if (token.match(extension.termRegex)) {
        return extension;
      }
    }
    return undefined;
  };

  getSymbolPos = term => {
    let offset = [' ', '[', '('].
      map(symbol => term.lastIndexOf(symbol)).
      reduce((offset, currOffset) => (offset < currOffset ? currOffset : offset), -1);

    if (offset !== -1) {
      offset++;
    }

    return offset;
  };

  getSearchToken = state => {
    const text = state.focusBlock.text;
    const { anchorOffset } = state.selection; // FIXME: works differently for different directions of selecting.

    let term;
    let offset = -1;

    if (text && anchorOffset > 0) {
      term = text.substring(0, anchorOffset);

      if (!term.endsWith(' ')) { // FIXME: ending with TAB should behave similarly.
        offset = term.lastIndexOf(' ');
        if (offset !== -1) { offset++ }
        const accents = getAccents(state);
        if (accents.length > 0) { // this position has some accent
          const currLeftPos = getPosAfterEmphasis(state, accents);
          if (offset < currLeftPos) {
            offset = currLeftPos;
          }
        }

        if (offset !== -1) {
          term = term.substring(offset);
        }
      }
    }

    return { term, text, offset };
  };

  handleClickOutside = () => {
    if (this.state.show) {
      this.hideWidget();
    }
  }

  handleKeyDown = (e) => {
    let { show, items, selectedItem } = this.state; // eslint-disable-line prefer-const

    if (e.keyCode === tabCode) {
      e.preventDefault();
      e.stopPropagation();
      this.hideWidget();
    }

    if (show) {
      if (e.keyCode === escapeCode) {
        e.preventDefault();
        e.stopPropagation();
        this.hideWidget();
      }
    }

    if (show && items) {
      if (e.keyCode === enterCode) {
        e.preventDefault();
        this.handleSelectItem(selectedItem, e);
        this.hideWidget();
      } else if (e.keyCode === arrowUpCode || e.keyCode === arrowDownCode) {
        e.preventDefault();
        const length = items.length;
        selectedItem = e.keyCode === arrowDownCode ? selectedItem + 1 : selectedItem - 1;
        if (e.keyCode === arrowDownCode && selectedItem < length ||
          e.keyCode === arrowUpCode && selectedItem >= 0) {
          this.setState({ selectedItem });
        }
      }
    }
  };

  removeSpecialCharacter = (state, specialCharacter) => {
    const change = state.change();
    const text = state.startBlock.text;
    const charLength = specialCharacter.length;
    const specialCharPos = text.lastIndexOf(specialCharacter, state.endOffset);
    change.moveOffsetsTo(specialCharPos).deleteForward(charLength).
      moveOffsetsTo(state.endOffset - charLength).focus();
    this.props.onChange(change.state);
  };

  handleSelectItem = (index, event = null) => {
    const { items } = this.state;
    const { state, options } = this.props;
    const { term } = this.getSearchToken(state);

    if (term) {
      const item = items[index];

      const { extensions } = options;
      const extension = this.matchExtension(extensions, term);

      if (extension && item) {
        const change = state.change();

        const text = change.anchorText.text
        const cursor = change.anchorOffset;

        const { start, end } = matchUnderCursor({ text, cursor, regexp: extension.termRegex });

        change.
          moveOffsetsTo(start, start).
          deleteForward(end - start).
          insertText(extension.markdownText(item, term)).
          focus();

        this.props.onChange(change.state, true);
      }

      if (!item && event) {
        this.removeSpecialCharacter(state, extension.specialCharacter);
      }
    }

    this.hideWidget();
  };

  searchItems = ({ state, options }) => {
    const { editorIsActive } = this.props.options;
    if (!editorIsActive) {
      this.setState({ items: [], show: false, loading: false });
      return
    }

    const { term } = this.getSearchToken(state);
    const extension = term && this.matchExtension(options.extensions, term);

    if (!extension) {
      this.hideWidget();
      return
    }

    this.setState({ items: [], loading: true });
    this.toggleWidget(true);
    const request = this.currentRequest = extension.searchItems(term).
      then(items => {
        if (this.currentRequest === request && this._isMounted) {
          this.setState({ items, selectedItem: 0, loading: false })
        }
      }).
      catch(err => {
        if (this.currentRequest === request && this._isMounted) {
          this.setState({ items: [], loading: false })
        }
      });
  };

  toggleWidget = flag => {
    this.props.onToggle(flag);
    this.setState({ show: flag });
  };

  hideWidget = () => {
    this.toggleWidget(false);
  };

  handleRef = (ref) => {
    this.setState({ ref });
  };

  getItemRenderFunc = _ => {
    const { state, options: { extensions } } = this.props;
    const { term } = this.getSearchToken(state);
    const extension = term && this.matchExtension(extensions, term);
    return (extension || {}).renderItem;
  }

  render() {
    const { show, selectedItem, items, ref, loading } = this.state;
    const { children, locale, options: { editorIsActive, containerRef } } = this.props;

    const selection = window.getSelection();
    if (selection.anchorNode) {
      const slateEditor = getSlateEditor(selection);
      if (slateEditor && !show) {
        slateEditor.style.overflow = 'auto';
      }
    }

    return (
      <div className="AutocompleteContainer"
        onKeyDown={this.handleKeyDown}
        ref={this.handleRef}
      >
        {show && editorIsActive ? createPortal(
          <AutocompleteWidget
            items={items}
            itemRenderer={this.getItemRenderFunc()}
            loading={loading}
            selectedItem={selectedItem}
            locale={locale}
            onChange={this.handleSelectItem}
            restrictorRef={ref}
            containerRef={containerRef}
          />,
          containerRef
        ) : null}
        {children}
      </div>
    );
  }
}

export default AutocompleteContainer;
