#include "stdafx.h"
#pragma hdrstop

#include "FS_internal.h"

#if defined(XR_PLATFORM_WINDOWS)
#include <io.h>
#include <direct.h>
#elif defined(XR_PLATFORM_LINUX) || defined(XR_PLATFORM_BSD) || defined(XR_PLATFORM_APPLE)
#include <sys/mman.h>
#endif
#include <sys/stat.h>
#include <fcntl.h>

#ifdef FS_DEBUG
XRCORE_API u32 g_file_mapped_memory = 0;
u32 g_file_mapped_count = 0;
typedef xr_map<u32, std::pair<u32, shared_str>> FILE_MAPPINGS;
FILE_MAPPINGS g_file_mappings;

void register_file_mapping(void* address, const u32& size, pcstr file_name)
{
    FILE_MAPPINGS::const_iterator I = g_file_mappings.find(*(u32*)&address);
    VERIFY(I == g_file_mappings.end());
    g_file_mappings.insert(std::make_pair(*(u32*)&address, std::make_pair(size, shared_str(file_name))));

    // Msg ("++register_file_mapping(%2d): [0x%08x]%s", g_file_mapped_count + 1, *((u32*)&address), file_name);

    g_file_mapped_memory += size;
    ++g_file_mapped_count;
#ifdef USE_MEMORY_MONITOR
    // memory_monitor::monitor_alloc (addres,size,"file mapping");
    string512 temp;
    xr_sprintf(temp, sizeof(temp), "file mapping: %s", file_name);
    memory_monitor::monitor_alloc(address, size, temp);
#endif // USE_MEMORY_MONITOR
}

void unregister_file_mapping(void* address, const u32& size)
{
    FILE_MAPPINGS::iterator I = g_file_mappings.find(*(u32*)&address);
    VERIFY(I != g_file_mappings.end());
    // VERIFY2 ((*I).second.first == size,make_string("file mapping sizes are different: %d ->
    // %d",(*I).second.first,size));
    g_file_mapped_memory -= (*I).second.first;
    --g_file_mapped_count;

    // Msg ("--unregister_file_mapping(%2d): [0x%08x]%s", g_file_mapped_count + 1, *((u32*)&address),
    // (*I).second.second.c_str());

    g_file_mappings.erase(I);

#ifdef USE_MEMORY_MONITOR
    memory_monitor::monitor_free(address);
#endif // USE_MEMORY_MONITOR
}

XRCORE_API void dump_file_mappings()
{
    Msg("* active file mappings (%d):", g_file_mappings.size());

    FILE_MAPPINGS::const_iterator I = g_file_mappings.begin();
    FILE_MAPPINGS::const_iterator E = g_file_mappings.end();
    for (; I != E; ++I)
        Msg("* [0x%08x][%d][%s]", (*I).first, (*I).second.first, (*I).second.second.c_str());
}
#endif // DEBUG
//////////////////////////////////////////////////////////////////////
// Tools
//////////////////////////////////////////////////////////////////////
//---------------------------------------------------
void VerifyPath(pcstr path)
{
    string1024 tmp;
    for (int i = 0; path[i]; i++)
    {
        if (path[i] != _DELIMITER || i == 0)
            continue;
        CopyMemory(tmp, path, i);
        tmp[i] = 0;
        convert_path_separators(tmp);
        _mkdir(tmp);
    }
}

static int open_internal(pcstr fn, int& handle)
{
#if defined(XR_PLATFORM_WINDOWS)
    return (_sopen_s(&handle, fn, _O_RDONLY | _O_BINARY, _SH_DENYNO, _S_IREAD));
#elif defined(XR_PLATFORM_LINUX) || defined(XR_PLATFORM_BSD) || defined(XR_PLATFORM_APPLE)
    pstr conv_fn = xr_strdup(fn);
    convert_path_separators(conv_fn);
    handle = open(conv_fn, _O_RDONLY);
    xr_free(conv_fn);

    return (handle == -1);
#else
#   error Select or add implementation for your platform
#endif
}

bool file_handle_internal(pcstr file_name, size_t& size, int& file_handle)
{
    if (open_internal(file_name, file_handle))
    {
        Sleep(1);
        if (open_internal(file_name, file_handle))
            return (false);
    }

    size = _filelength(file_handle);
    return (true);
}

void* FileDownload(pcstr file_name, const int& file_handle, size_t& file_size)
{
    VERIFY(file_size != 0);
    void* buffer = xr_malloc(file_size);

    const auto r_bytes = _read(file_handle, buffer, file_size);
    R_ASSERT3(file_size == static_cast<size_t>(r_bytes), "Can't read from file : ", file_name);

    // file_size = r_bytes;

    R_ASSERT3(!_close(file_handle), "can't close file : ", file_name);

    return (buffer);
}

void* FileDownload(pcstr file_name, size_t* buffer_size)
{
    int file_handle;
    R_ASSERT3(file_handle_internal(file_name, *buffer_size, file_handle), "can't open file : ", file_name);

    return (FileDownload(file_name, file_handle, *buffer_size));
}

typedef char MARK[9];
IC void mk_mark(MARK& M, const char* S) { strncpy_s(M, sizeof(M), S, 8); }
void FileCompress(pcstr fn, pcstr sign, void* data, size_t size)
{
    MARK M;
    mk_mark(M, sign);

    int H = _open(fn, O_BINARY | O_CREAT | O_WRONLY | O_TRUNC, S_IREAD | S_IWRITE);
    R_ASSERT2(H > 0, fn);
    std::ignore = _write(H, &M, 8);
    _writeLZ(H, data, size);
    _close(H);
}

void* FileDecompress(pcstr fn, pcstr sign, size_t* size)
{
    MARK M, F;
    mk_mark(M, sign);

    int H = _open(fn, O_BINARY | O_RDONLY);
    R_ASSERT2(H > 0, fn);
    std::ignore = _read(H, &F, 8);
    if (strncmp(M, F, 8) != 0)
    {
        F[8] = 0;
        Msg("FATAL: signatures doesn't match, file(%s) / requested(%s)", F, sign);
    }
    R_ASSERT(strncmp(M, F, 8) == 0);

    void* ptr = 0;
    const size_t SZ = _readLZ(H, ptr, _filelength(H) - 8);
    _close(H);
    if (size)
        *size = SZ;
    return ptr;
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//---------------------------------------------------
// memory
CMemoryWriter::~CMemoryWriter() { xr_free(data); }
void CMemoryWriter::w(const void* ptr, size_t count)
{
    if (position + count > mem_size)
    {
        // reallocate
        if (mem_size == 0)
            mem_size = 128;
        while (mem_size <= (position + count))
            mem_size *= 2;
        if (0 == data)
            data = (u8*)xr_malloc(mem_size);
        else
            data = (u8*)xr_realloc(data, mem_size);
    }
    CopyMemory(data + position, ptr, count);
    position += count;
    if (position > file_size)
        file_size = position;
}

// static const u32 mb_sz = 0x1000000;
bool CMemoryWriter::save_to(pcstr fn) const
{
    IWriter* F = FS.w_open(fn);
    if (F)
    {
        F->w(pointer(), size());
        FS.w_close(F);
        return true;
    }
    return false;
}

void IWriter::open_chunk(u32 type)
{
    w_u32(type);
    chunk_pos.push(tell());
    w_u32(0); // the place for 'size'
}
void IWriter::close_chunk()
{
    VERIFY(!chunk_pos.empty());

    const size_t pos = tell();
    seek(chunk_pos.top());
    w_u32(pos - chunk_pos.top() - 4);
    seek(pos);
    chunk_pos.pop();
}
size_t IWriter::chunk_size()
{
    if (chunk_pos.empty())
        return 0;
    return tell() - chunk_pos.top() - 4;
}

void IWriter::w_compressed(void* ptr, size_t count)
{
    u8* dest = 0;
    size_t dest_sz = 0;
    _compressLZ(&dest, &dest_sz, ptr, count);

    // if (g_dummy_stuff)
    // g_dummy_stuff (dest,dest_sz,dest);

    if (dest && dest_sz)
        w(dest, dest_sz);
    xr_free(dest);
}

void IWriter::w_chunk(u32 type, void* data, size_t size)
{
    open_chunk(type);
    if (type & CFS_CompressMark)
        w_compressed(data, size);
    else
        w(data, size);
    close_chunk();
}
void IWriter::w_sdir(const Fvector& D)
{
    Fvector C;
    float mag = D.magnitude();
    if (mag > EPS_S)
    {
        C.div(D, mag);
    }
    else
    {
        C.set(0, 0, 1);
        mag = 0;
    }
    w_dir(C);
    w_float(mag);
}
// XXX: reimplement to prevent buffer overflows
void IWriter::w_printf(const char* format, ...)
{
    va_list args;
    va_start(args, format);
    VPrintf(format, args);
    va_end(args);
}

void IWriter::VPrintf(const char* format, va_list args)
{
    char buf[1024];
    std::vsnprintf(buf, sizeof(buf), format, args);
    w(buf, xr_strlen(buf));
}

//---------------------------------------------------
// base stream
IReader* IReader::open_chunk(u32 ID)
{
    bool bCompressed;

    const size_t dwSize = find_chunk(ID, &bCompressed);
    if (dwSize != 0)
    {
        if (bCompressed)
        {
            u8* dest;
            size_t dest_sz;
            _decompressLZ(&dest, &dest_sz, pointer(), dwSize);
            return xr_new<CTempReader>(dest, dest_sz, tell() + dwSize);
        }
        else
        {
            return xr_new<IReader>(pointer(), dwSize, tell() + dwSize);
        }
    }
    else
        return 0;
};
void IReader::close()
{
    IReader* self = this;
    xr_delete(self);
}

#include "FS_impl.h"

#ifdef TESTING_IREADER
IReaderTestPolicy::~IReaderTestPolicy() { xr_delete(m_test); };
#endif // TESTING_IREADER

#ifdef FIND_CHUNK_BENCHMARK_ENABLE
find_chunk_counter g_find_chunk_counter;
#endif // FIND_CHUNK_BENCHMARK_ENABLE

size_t IReader::find_chunk(u32 ID, bool* bCompressed) { return inherited::find_chunk(ID, bCompressed); }
IReader* IReader::open_chunk_iterator(u32& ID, IReader* _prev)
{
    if (0 == _prev)
    {
        // first
        rewind();
    }
    else
    {
        // next
        seek(_prev->iterpos);
        _prev->close();
    }

    // open
    if (elapsed() < 8)
        return NULL;
    ID = r_u32();
    const size_t _size = r_u32();
    if (ID & CFS_CompressMark)
    {
        // compressed
        u8* dest;
        size_t dest_sz;
        _decompressLZ(&dest, &dest_sz, pointer(), _size);
        return xr_new<CTempReader>(dest, dest_sz, tell() + _size);
    }
    else
    {
        // normal
        return xr_new<IReader>(pointer(), _size, tell() + _size);
    }
}

void IReader::r(void* p, size_t cnt)
{
    VERIFY(Pos + cnt <= Size);
    CopyMemory(p, pointer(), cnt);
    advance(cnt);
#ifdef DEBUG
    bool bShow = false;
    if (dynamic_cast<CFileReader*>(this))
        bShow = true;
    if (dynamic_cast<CVirtualFileReader*>(this))
        bShow = true;
    if (bShow)
    {
        FS.dwOpenCounter++;
    }
#endif
};

IC bool is_term(char a) { return (a == 13) || (a == 10); };
IC size_t IReader::advance_term_string()
{
    size_t sz = 0;
    char* src = (char*)data;
    while (!eof())
    {
        Pos++;
        sz++;
        if (!eof() && is_term(src[Pos]))
        {
            while (!eof() && is_term(src[Pos]))
                Pos++;
            break;
        }
    }
    return sz;
}
void IReader::r_string(char* dest, size_t tgt_sz)
{
    char* src = (char*)data + Pos;
    size_t sz = advance_term_string();
    R_ASSERT2(sz < (tgt_sz - 1), "Dest string less than needed.");
#if defined(XR_PLATFORM_WINDOWS)
    R_ASSERT(!IsBadReadPtr((void*)src, sz));
#endif

    strncpy_s(dest, tgt_sz, src, sz);
    dest[sz] = 0;
}
void IReader::r_string(xr_string& dest)
{
    char* src = (char*)data + Pos;
    size_t sz = advance_term_string();
    dest.assign(src, sz);
}
void IReader::r_stringZ(char* dest, size_t tgt_sz)
{
    char* src = (char*)data;
    size_t sz = xr_strlen(src);
    R_ASSERT2(sz < tgt_sz, "Dest string less than needed.");
    while ((src[Pos] != 0) && (!eof()))
        *dest++ = src[Pos++];
    *dest = 0;
    Pos++;
}
void IReader::r_stringZ(shared_str& dest)
{
    dest = (char*)(data + Pos);
    Pos += (dest.size() + 1);
}
void IReader::r_stringZ(xr_string& dest)
{
    dest = (char*)(data + Pos);
    Pos += dest.size() + 1;
};

void IReader::skip_stringZ()
{
    char* src = (char*)data;
    while ((src[Pos] != 0) && (!eof()))
        Pos++;
    Pos++;
};

bool IReader::try_r_string(char* dest, size_t tgt_sz)
{
    char* src = (char*)data + Pos;
    size_t sz = advance_term_string();
    if (sz >= tgt_sz)
        return false;

#if defined(XR_PLATFORM_WINDOWS)
    R_ASSERT(!IsBadReadPtr((void*)src, sz));
#endif

    strncpy_s(dest, tgt_sz, src, sz);
    dest[sz] = 0;

    return true;
}

//---------------------------------------------------
// temp stream
CTempReader::~CTempReader() { xr_free(data); };
//---------------------------------------------------
// pack stream
CPackReader::~CPackReader()
{
#ifdef FS_DEBUG
    unregister_file_mapping(base_address, Size);
#endif // DEBUG
#if defined(XR_PLATFORM_WINDOWS)
    UnmapViewOfFile(base_address);
#elif defined(XR_PLATFORM_LINUX) || defined(XR_PLATFORM_BSD) || defined(XR_PLATFORM_APPLE)
    ::munmap(base_address, Size);
#else
#   error Select or add implementation for your platform
#endif
};
//---------------------------------------------------
// file stream
CFileReader::CFileReader(pcstr name)
{
    data = (char*)FileDownload(name, (size_t*)&Size);
    Pos = 0;
};
CFileReader::~CFileReader() { xr_free(data); };
//---------------------------------------------------
// compressed stream
CCompressedReader::CCompressedReader(const char* name, const char* sign)
{
    data = (char*)FileDecompress(name, sign, (size_t*)&Size);
    Pos = 0;
}
CCompressedReader::~CCompressedReader() { xr_free(data); };
CVirtualFileRW::CVirtualFileRW(pcstr cFileName)
{
#if defined(XR_PLATFORM_WINDOWS)
    // Open the file
    hSrcFile = CreateFile(cFileName, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, 0, OPEN_EXISTING, 0, 0);
    R_ASSERT3(hSrcFile != INVALID_HANDLE_VALUE, cFileName, xrDebug::ErrorToString(GetLastError()));
    Size = (int)GetFileSize(hSrcFile, NULL);
    R_ASSERT3(Size, cFileName, xrDebug::ErrorToString(GetLastError()));

    hSrcMap = CreateFileMapping(hSrcFile, 0, PAGE_READWRITE, 0, 0, 0);
    R_ASSERT3(hSrcMap != INVALID_HANDLE_VALUE, cFileName, xrDebug::ErrorToString(GetLastError()));

    data = (char*)MapViewOfFile(hSrcMap, FILE_MAP_ALL_ACCESS, 0, 0, 0);
    R_ASSERT3(data, cFileName, xrDebug::ErrorToString(GetLastError()));
#elif defined(XR_PLATFORM_LINUX) || defined(XR_PLATFORM_BSD) || defined(XR_PLATFORM_APPLE)
    pstr conv_path = xr_strdup(cFileName);
    convert_path_separators(conv_path);
    hSrcFile = ::open(conv_path, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH); //за такое использование указателя нужно убивать, но пока пусть будет
    xr_free(conv_path);
    R_ASSERT3(hSrcFile != -1, cFileName, xrDebug::ErrorToString(GetLastError()));
    struct stat file_info;
    ::fstat(hSrcFile, &file_info);
    Size = (int)file_info.st_size;
    R_ASSERT3(Size, cFileName, xrDebug::ErrorToString(GetLastError()));
    data = (char*)::mmap(NULL, Size, PROT_READ | PROT_WRITE, MAP_SHARED, hSrcFile, 0);
    R_ASSERT3(data && data != MAP_FAILED, cFileName, xrDebug::ErrorToString(GetLastError()));
#else
#   error Select or add implementation for your platform
#endif
#ifdef FS_DEBUG
    register_file_mapping(data, Size, cFileName);
#endif // DEBUG
}

CVirtualFileRW::~CVirtualFileRW()
{
#ifdef FS_DEBUG
    unregister_file_mapping(data, Size);
#endif // DEBUG
#if defined(XR_PLATFORM_WINDOWS)
    UnmapViewOfFile((void*)data);
    CloseHandle(hSrcMap);
    CloseHandle(hSrcFile);
#elif defined(XR_PLATFORM_LINUX) || defined(XR_PLATFORM_BSD) || defined(XR_PLATFORM_APPLE)
    ::munmap((void*)data, Size);
    ::close(hSrcFile);
    hSrcFile = -1;
#else
#   error Select or add implementation for your platform
#endif
}

CVirtualFileReader::CVirtualFileReader(pcstr cFileName)
{
#if defined(XR_PLATFORM_WINDOWS)
    // Open the file
    hSrcFile = CreateFile(cFileName, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, 0, OPEN_EXISTING, 0, 0);
    R_ASSERT3(hSrcFile != INVALID_HANDLE_VALUE, cFileName, xrDebug::ErrorToString(GetLastError()));
    Size = (int)GetFileSize(hSrcFile, NULL);
    R_ASSERT3(Size, cFileName, xrDebug::ErrorToString(GetLastError()));

    hSrcMap = CreateFileMapping(hSrcFile, 0, PAGE_READONLY, 0, 0, 0);
    R_ASSERT3(hSrcMap != INVALID_HANDLE_VALUE, cFileName, xrDebug::ErrorToString(GetLastError()));

    data = (char*)MapViewOfFile(hSrcMap, FILE_MAP_READ, 0, 0, 0);
    R_ASSERT3(data, cFileName, xrDebug::ErrorToString(GetLastError()));
#elif defined(XR_PLATFORM_LINUX) || defined(XR_PLATFORM_BSD) || defined(XR_PLATFORM_APPLE)
    pstr conv_path = xr_strdup(cFileName);
    convert_path_separators(conv_path);
    hSrcFile = ::open(conv_path, O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH); //за такое использование указателя нужно убивать, но пока пусть будет
    xr_free(conv_path);
    R_ASSERT3(hSrcFile != -1, cFileName, xrDebug::ErrorToString(GetLastError()));
    struct stat file_info;
    ::fstat(hSrcFile, &file_info);
    Size = (int)file_info.st_size;
    R_ASSERT3(Size, cFileName, xrDebug::ErrorToString(GetLastError()));
    data = (char*)::mmap(NULL, Size, PROT_READ, MAP_SHARED, hSrcFile, 0);
    R_ASSERT3(data && data != MAP_FAILED, cFileName, xrDebug::ErrorToString(GetLastError()));
#else
#   error Select or add implementation for your platform
#endif

#ifdef FS_DEBUG
    register_file_mapping(data, Size, cFileName);
#endif // DEBUG
}

CVirtualFileReader::~CVirtualFileReader()
{
#ifdef FS_DEBUG
    unregister_file_mapping(data, Size);
#endif // DEBUG
#if defined(XR_PLATFORM_WINDOWS)
    UnmapViewOfFile((void*)data);
    CloseHandle(hSrcMap);
    CloseHandle(hSrcFile);
#elif defined(XR_PLATFORM_LINUX) || defined(XR_PLATFORM_BSD) || defined(XR_PLATFORM_APPLE)
    ::munmap((void*)data, Size);
    ::close(hSrcFile);
    hSrcFile = -1;
#else
#   error Select or add implementation for your platform
#endif
}
