/* Copyright (c) 1996-2022 The OPC Foundation. All rights reserved.
   The source code in this file is covered under a dual-license scenario:
     - RCL: for OPC Foundation Corporate Members in good-standing
     - GPL V2: everybody else
   RCL license terms accompanied with this source code. See http://opcfoundation.org/License/RCL/1.00/
   GNU General Public License as published by the Free Software Foundation;
   version 2 of the License are accompanied with this source code. See http://opcfoundation.org/License/GPLv2
   This source code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/


using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Opc.Ua.Security.Certificates;

namespace Opc.Ua
{
    /// <summary>
    /// Utility functions for X509 certificates.
    /// </summary>
    public static class X509Utils
    {
        /// <summary>
        /// Extracts the DNS names specified in the certificate.
        /// </summary>
        /// <param name="certificate">The certificate.</param>
        /// <returns>The DNS names.</returns>
        public static IList<string> GetDomainsFromCertificate(X509Certificate2 certificate)
        {
            List<string> dnsNames = new List<string>();

            // extracts the domain from the subject name.
            List<string> fields = X509Utils.ParseDistinguishedName(certificate.Subject);

            StringBuilder builder = new StringBuilder();

            for (int ii = 0; ii < fields.Count; ii++)
            {
                if (fields[ii].StartsWith("DC=", StringComparison.Ordinal))
                {
                    if (builder.Length > 0)
                    {
                        builder.Append('.');
                    }
#if NET5_0_OR_GREATER || NETSTANDARD2_1
                    builder.Append(fields[ii].AsSpan(3));
#else
                    builder.Append(fields[ii].Substring(3));
#endif
                }
            }

            if (builder.Length > 0)
            {
                dnsNames.Add(builder.ToString().ToUpperInvariant());
            }

            // extract the alternate domains from the subject alternate name extension.
            X509SubjectAltNameExtension alternateName = X509Extensions.FindExtension<X509SubjectAltNameExtension>(certificate);
            if (alternateName != null)
            {
                for (int ii = 0; ii < alternateName.DomainNames.Count; ii++)
                {
                    string hostname = alternateName.DomainNames[ii];

                    // do not add duplicates to the list.
                    bool found = false;

                    for (int jj = 0; jj < dnsNames.Count; jj++)
                    {
                        if (String.Equals(dnsNames[jj], hostname, StringComparison.OrdinalIgnoreCase))
                        {
                            found = true;
                            break;
                        }
                    }

                    if (!found)
                    {
                        dnsNames.Add(hostname.ToUpperInvariant());
                    }
                }

                for (int ii = 0; ii < alternateName.IPAddresses.Count; ii++)
                {
                    string ipAddress = alternateName.IPAddresses[ii];

                    if (!dnsNames.Contains(ipAddress))
                    {
                        dnsNames.Add(ipAddress);
                    }
                }
            }

            // return the list.
            return dnsNames;
        }

        /// <summary>
        /// Returns the size of the public key and disposes RSA key.
        /// </summary>
        /// <param name="certificate">The certificate</param>
        public static int GetRSAPublicKeySize(X509Certificate2 certificate)
        {
            using (RSA rsaPublicKey = certificate.GetRSAPublicKey())
            {
                if (rsaPublicKey != null)
                {
                    return rsaPublicKey.KeySize;
                }
                return -1;
            }
        }

        /// <summary>
        /// Returns the size of the public key of a given certificate
        /// </summary>
        /// <param name="certificate">The certificate</param>
        public static int GetPublicKeySize(X509Certificate2 certificate)
        {
            using (RSA rsaPublicKey = certificate.GetRSAPublicKey())
            {
                if (rsaPublicKey != null)
                {
                    return rsaPublicKey.KeySize;
                }
            }

            using (ECDsa ecdsaPublicKey = certificate.GetECDsaPublicKey())
            {
                if (ecdsaPublicKey != null)
                {
                    return ecdsaPublicKey.KeySize;
                }
            }

            return -1;
        }

        /// <summary>
        /// Extracts the application URI specified in the certificate.
        /// </summary>
        /// <param name="certificate">The certificate.</param>
        /// <returns>The application URI.</returns>
        public static string GetApplicationUriFromCertificate(X509Certificate2 certificate)
        {
            // extract the alternate domains from the subject alternate name extension.
            X509SubjectAltNameExtension alternateName = X509Extensions.FindExtension<X509SubjectAltNameExtension>(certificate);

            // get the application uri.
            if (alternateName != null && alternateName.Uris.Count > 0)
            {
                return alternateName.Uris[0];
            }

            return string.Empty;
        }

        /// <summary>
        /// Check if certificate has an application urn.
        /// </summary>
        /// <param name="certificate">The certificate.</param>
        /// <returns>true if the application URI starts with urn: </returns>
        public static bool HasApplicationURN(X509Certificate2 certificate)
        {
            // extract the alternate domains from the subject alternate name extension.
            X509SubjectAltNameExtension alternateName = X509Extensions.FindExtension<X509SubjectAltNameExtension>(certificate);

            // find the application urn.
            if (alternateName != null && alternateName.Uris.Count > 0)
            {
                string urn = "urn:";
                for (int i = 0; i < alternateName.Uris.Count; i++)
                {
                    if (string.Compare(alternateName.Uris[i], 0, urn, 0, urn.Length, StringComparison.OrdinalIgnoreCase) == 0)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// Checks that the domain in the URL provided matches one of the domains in the certificate.
        /// </summary>
        /// <param name="certificate">The certificate.</param>
        /// <param name="endpointUrl">The endpoint url to verify.</param>
        /// <returns>True if the certificate matches the url.</returns>
        public static bool DoesUrlMatchCertificate(X509Certificate2 certificate, Uri endpointUrl)
        {
            if (endpointUrl == null || certificate == null)
            {
                return false;
            }

            IList<string> domainNames = GetDomainsFromCertificate(certificate);

            for (int jj = 0; jj < domainNames.Count; jj++)
            {
                if (String.Equals(domainNames[jj], endpointUrl.DnsSafeHost, StringComparison.OrdinalIgnoreCase))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Determines whether the certificate is allowed to be an issuer.
        /// </summary>
        public static bool IsIssuerAllowed(X509Certificate2 certificate)
        {
            X509BasicConstraintsExtension constraints = X509Extensions.FindExtension<X509BasicConstraintsExtension>(certificate);

            if (constraints != null)
            {
                return constraints.CertificateAuthority;
            }

            return false;
        }

        /// <summary>
        /// Determines whether the certificate is issued by a Certificate Authority.
        /// </summary>
        public static bool IsCertificateAuthority(X509Certificate2 certificate)
        {
            var constraints = X509Extensions.FindExtension<X509BasicConstraintsExtension>(certificate);
            if (constraints != null)
            {
                return constraints.CertificateAuthority;
            }
            return false;
        }

        /// <summary>
        /// Return the key usage flags of a certificate.
        /// </summary>
        public static X509KeyUsageFlags GetKeyUsage(X509Certificate2 cert)
        {
            var allFlags = X509KeyUsageFlags.None;
            foreach (X509KeyUsageExtension ext in cert.Extensions.OfType<X509KeyUsageExtension>())
            {
                allFlags |= ext.KeyUsages;
            }
            return allFlags;
        }

        /// <summary>
        /// Check for self signed certificate if there is match of the Subject/Issuer.
        /// </summary>
        /// <param name="certificate">The certificate to test.</param>
        /// <returns>True if self signed.</returns>
        public static bool IsSelfSigned(X509Certificate2 certificate)
        {
            return X509Utils.CompareDistinguishedName(certificate.SubjectName, certificate.IssuerName);
        }

        /// <summary>
        /// Compares two distinguished names.
        /// </summary>
        public static bool CompareDistinguishedName(X500DistinguishedName name1, X500DistinguishedName name2)
        {
            // check for simple binary equality.
            return Utils.IsEqual(name1.RawData, name2.RawData);
        }

        /// <summary>
        /// Compares two distinguished names as strings.
        /// </summary>
        /// <remarks>
        /// Where possible, distinguished names should be compared
        /// by using the <see cref="X500DistinguishedName"/> version.
        /// </remarks>
        public static bool CompareDistinguishedName(string name1, string name2)
        {
            // check for simple equality.
            if (String.Equals(name1, name2, StringComparison.Ordinal))
            {
                return true;
            }

            // parse the names.
            List<string> fields1 = ParseDistinguishedName(name1);
            List<string> fields2 = ParseDistinguishedName(name2);

            // can't be equal if the number of fields is different.
            if (fields1.Count != fields2.Count)
            {
                return false;
            }

            return CompareDistinguishedNameFields(fields1, fields2);
        }

        /// <summary>
        /// Compares string fields of two distinguished names.
        /// </summary>
        private static bool CompareDistinguishedNameFields(IList<string> fields1, IList<string> fields2)
        {
            // compare each.
            for (int ii = 0; ii < fields1.Count; ii++)
            {
                var comparison = StringComparison.Ordinal;
                if (fields1[ii].StartsWith("DC=", StringComparison.OrdinalIgnoreCase))
                {
                    // DC hostnames may have different case
                    comparison = StringComparison.OrdinalIgnoreCase;
                }
                if (!String.Equals(fields1[ii], fields2[ii], comparison))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// Compares two distinguished names.
        /// </summary>
        public static bool CompareDistinguishedName(X509Certificate2 certificate, List<string> parsedName)
        {
            // can't compare if the number of fields is 0.
            if (parsedName.Count == 0)
            {
                return false;
            }

            // parse the names.
            List<string> certificateName = ParseDistinguishedName(certificate.Subject);

            // can't be equal if the number of fields is different.
            if (parsedName.Count != certificateName.Count)
            {
                return false;
            }

            return CompareDistinguishedNameFields(parsedName, certificateName);
        }

        private static readonly char[] anyOf = new char[] { '/', ',', '=' };

        /// <summary>
        /// Parses a distingushed name.
        /// </summary>
        public static List<string> ParseDistinguishedName(string name)
        {
            List<string> fields = new List<string>();

            if (String.IsNullOrEmpty(name))
            {
                return fields;
            }

            // determine the delimiter used.
            char delimiter = ',';
            bool found = false;
            bool quoted = false;

            for (int ii = name.Length - 1; ii >= 0; ii--)
            {
                char ch = name[ii];

                if (ch == '"')
                {
                    quoted = !quoted;
                    continue;
                }

                if (!quoted && ch == '=')
                {
                    ii--;

                    while (ii >= 0 && Char.IsWhiteSpace(name[ii])) ii--;
                    while (ii >= 0 && (Char.IsLetterOrDigit(name[ii]) || name[ii] == '.')) ii--;
                    while (ii >= 0 && Char.IsWhiteSpace(name[ii])) ii--;

                    if (ii >= 0)
                    {
                        delimiter = name[ii];
                    }

                    break;
                }
            }

            StringBuilder buffer = new StringBuilder();

            string key = null;
            string value = null;
            found = false;

            for (int ii = 0; ii < name.Length; ii++)
            {
                while (ii < name.Length && Char.IsWhiteSpace(name[ii])) ii++;

                if (ii >= name.Length)
                {
                    break;
                }

                char ch = name[ii];

                if (found)
                {
                    char end = delimiter;

                    if (ii < name.Length && name[ii] == '"')
                    {
                        ii++;
                        end = '"';
                    }

                    while (ii < name.Length)
                    {
                        ch = name[ii];

                        if (ch == end)
                        {
                            while (ii < name.Length && name[ii] != delimiter) ii++;
                            break;
                        }

                        buffer.Append(ch);
                        ii++;
                    }

                    value = buffer.ToString().TrimEnd();
                    found = false;

                    buffer.Length = 0;
                    buffer.Append(key);
                    buffer.Append('=');

                    if (value.IndexOfAny(anyOf) != -1)
                    {
                        if (value.Length > 0 && value[0] != '"')
                        {
                            buffer.Append('"');
                        }

                        buffer.Append(value);

                        if (value.Length > 0 && value[value.Length - 1] != '"')
                        {
                            buffer.Append('"');
                        }
                    }
                    else
                    {
                        buffer.Append(value);
                    }

                    fields.Add(buffer.ToString());
                    buffer.Length = 0;
                }

                else
                {
                    while (ii < name.Length)
                    {
                        ch = name[ii];

                        if (ch == '=')
                        {
                            break;
                        }

                        buffer.Append(ch);
                        ii++;
                    }

                    key = buffer.ToString().TrimEnd().ToUpperInvariant();
                    buffer.Length = 0;
                    found = true;
                }
            }

            return fields;
        }

        /// <summary>
        /// Return if a certificate has a ECDsa signature.
        /// </summary>
        /// <param name="cert">The certificate to test.</param>
        public static bool IsECDsaSignature(X509Certificate2 cert)
        {
            return X509PfxUtils.IsECDsaSignature(cert);
        }

        /// <summary>
        /// Return a qualifier string if a ECDsa signature algorithm used.
        /// </summary>
        /// <param name="certificate">The certificate.</param>
        public static string GetECDsaQualifier(X509Certificate2 certificate)
        {
            return EccUtils.GetECDsaQualifier(certificate);
        }

        /// <summary>
        /// Verify RSA key pair of two certificates.
        /// </summary>
        public static bool VerifyKeyPair(
            X509Certificate2 certWithPublicKey,
            X509Certificate2 certWithPrivateKey,
            bool throwOnError = false)
        {
            return X509PfxUtils.VerifyKeyPair(certWithPublicKey, certWithPrivateKey, throwOnError);
        }

        /// <summary>
        /// Verify ECDsa key pair of two certificates.
        /// </summary>
        public static bool VerifyECDsaKeyPair(
            X509Certificate2 certWithPublicKey,
            X509Certificate2 certWithPrivateKey,
            bool throwOnError = false)
        {
#if ECC_SUPPORT  
            return X509PfxUtils.VerifyECDsaKeyPair(certWithPublicKey, certWithPrivateKey, throwOnError);
#else
            throw new NotSupportedException();
#endif
        }

        /// <summary>
        /// Verify RSA key pair of two certificates.
        /// </summary>
        public static bool VerifyRSAKeyPair(
            X509Certificate2 certWithPublicKey,
            X509Certificate2 certWithPrivateKey,
            bool throwOnError = false)
        {
            return X509PfxUtils.VerifyRSAKeyPair(certWithPublicKey, certWithPrivateKey, throwOnError);
        }

        /// <summary>
        /// Verify the signature of a self signed certificate.
        /// </summary>
        public static bool VerifySelfSigned(X509Certificate2 cert)
        {
            try
            {
                var signature = new X509Signature(cert.RawData);
                return signature.Verify(cert);
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// Creates a copy of a certificate with a private key.
        /// If the platform defaults to an ephemeral key set,
        /// the private key requires an extra copy.
        /// </summary>
        /// <returns>The certificate</returns>
        public static X509Certificate2 CreateCopyWithPrivateKey(X509Certificate2 certificate, bool persisted)
        {
            // a copy is only necessary on windows
            if (certificate.HasPrivateKey
#if !NETFRAMEWORK
                && RuntimeInformation.IsOSPlatform(OSPlatform.Windows)
#endif
                )
            {
                // see https://github.com/dotnet/runtime/issues/29144
                string passcode = GeneratePasscode();
                X509KeyStorageFlags storageFlags = persisted ? X509KeyStorageFlags.PersistKeySet : X509KeyStorageFlags.Exportable;
                return new X509Certificate2(certificate.Export(X509ContentType.Pfx, passcode), passcode, storageFlags);
            }
            return certificate;
        }

        /// <summary>
        /// Creates a certificate from a PKCS #12 store with a private key.
        /// </summary>
        /// <param name="rawData">The raw PKCS #12 store data.</param>
        /// <param name="password">The password to use to access the store.</param>
        /// <param name="noEphemeralKeySet">Set to true if the key should not use the ephemeral key set.</param>
        /// <returns>The certificate with a private key.</returns>
        public static X509Certificate2 CreateCertificateFromPKCS12(
            byte[] rawData,
            string password,
            bool noEphemeralKeySet = false
            )
        {
            return X509PfxUtils.CreateCertificateFromPKCS12(rawData, password, noEphemeralKeySet);
        }

        /// <summary>
        /// Get the certificate by issuer and serial number.
        /// </summary>
        public static async Task<X509Certificate2> FindIssuerCABySerialNumberAsync(
            ICertificateStore store,
            X500DistinguishedName issuer,
            string serialnumber)
        {
            X509Certificate2Collection certificates = await store.Enumerate().ConfigureAwait(false);

            foreach (var certificate in certificates)
            {
                if (X509Utils.CompareDistinguishedName(certificate.SubjectName, issuer) &&
                    Utils.IsEqual(certificate.SerialNumber, serialnumber))
                {
                    return certificate;
                }
            }

            return null;
        }

        /// <summary>
        /// Extension to add a certificate to a <see cref="ICertificateStore"/>.
        /// </summary>
        /// <remarks>
        /// Saves also the private key, if available.
        /// If written to a Pfx file, the password is used for protection.
        /// </remarks>
        /// <param name="certificate">The certificate to store.</param>
        /// <param name="storeType">Type of certificate store (Directory) <see cref="CertificateStoreType"/>.</param>
        /// <param name="storePath">The store path (syntax depends on storeType).</param>
        /// <param name="password">The password to use to protect the certificate.</param>
        /// <returns></returns>
        public static X509Certificate2 AddToStore(
            this X509Certificate2 certificate,
            string storeType,
            string storePath,
            string password = null)
        {
            // add cert to the store.
            if (!String.IsNullOrEmpty(storePath) && !String.IsNullOrEmpty(storeType))
            {
                var certificateStoreIdentifier = new CertificateStoreIdentifier(storePath, storeType, false);
                using (ICertificateStore store = certificateStoreIdentifier.OpenStore())
                {
                    if (store == null)
                    {
                        throw new ArgumentException("Invalid store type");
                    }

                    store.Open(storePath, false);
                    store.Add(certificate, password).Wait();
                    store.Close();
                }
            }
            return certificate;
        }

        /// <summary>
        /// Extension to add a certificate to a <see cref="ICertificateStore"/>.
        /// </summary>
        /// <remarks>
        /// Saves also the private key, if available.
        /// If written to a Pfx file, the password is used for protection.
        /// </remarks>
        /// <param name="certificate">The certificate to store.</param>
        /// <param name="storeIdentifier">The certificate store.</param>
        /// <param name="password">The password to use to protect the certificate.</param>
        /// <returns></returns>
        public static X509Certificate2 AddToStore(
            this X509Certificate2 certificate,
            CertificateStoreIdentifier storeIdentifier,
            string password = null)
        {
            // add cert to the store.
            if (storeIdentifier != null)
            {
                ICertificateStore store = storeIdentifier.OpenStore();
                try
                {
                    if (store == null || store.NoPrivateKeys == true)
                    {
                        throw new ArgumentException("Invalid store type");
                    }

                    store.Add(certificate, password).Wait();
                }
                finally
                {
                    store?.Close();
                }
            }
            return certificate;
        }

        /// <summary>e
        /// Extension to add a certificate to a <see cref="ICertificateStore"/>.
        /// </summary>
        /// <remarks>
        /// Saves also the private key, if available.
        /// If written to a Pfx file, the password is used for protection.
        /// </remarks>
        /// <param name="certificate">The certificate to store.</param>
        /// <param name="storeType">Type of certificate store (Directory) <see cref="CertificateStoreType"/>.</param>
        /// <param name="storePath">The store path (syntax depends on storeType).</param>
        /// <param name="password">The password to use to protect the certificate.</param>
        /// <param name="ct">The cancellation token.</param>
        public static async Task<X509Certificate2> AddToStoreAsync(
            this X509Certificate2 certificate,
            string storeType,
            string storePath,
            string password = null,
            CancellationToken ct = default)
        {
            // add cert to the store.
            if (!String.IsNullOrEmpty(storePath) && !String.IsNullOrEmpty(storeType))
            {
                var certificateStoreIdentifier = new CertificateStoreIdentifier(storePath, storeType, false);
                using (ICertificateStore store = certificateStoreIdentifier.OpenStore())
                {
                    if (store == null)
                    {
                        throw new ArgumentException("Invalid store type");
                    }

                    await store.Add(certificate, password).ConfigureAwait(false);
                    store.Close();
                }
            }
            return certificate;
        }

        /// <summary>e
        /// Extension to add a certificate to a <see cref="ICertificateStore"/>.
        /// </summary>
        /// <remarks>
        /// Saves also the private key, if available.
        /// If written to a Pfx file, the password is used for protection.
        /// </remarks>
        /// <param name="certificate">The certificate to store.</param>
        /// <param name="storeIdentifier">Type of certificate store (Directory) <see cref="CertificateStoreType"/>.</param>
        /// <param name="password">The password to use to protect the certificate.</param>
        /// <param name="ct">The cancellation token.</param>
        public static async Task<X509Certificate2> AddToStoreAsync(
            this X509Certificate2 certificate,
            CertificateStoreIdentifier storeIdentifier,
            string password = null,
            CancellationToken ct = default)
        {
            // add cert to the store.
            if (storeIdentifier != null)
            {
                ICertificateStore store = storeIdentifier.OpenStore();
                try
                {
                    if (store == null)
                    {
                        throw new ArgumentException("Invalid store type");
                    }
                    await store.Add(certificate, password).ConfigureAwait(false);
                }
                finally
                {
                    store?.Close();
                }
            }
            return certificate;
        }


        /// <summary>
        /// Get the hash algorithm from the hash size in bits.
        /// </summary>
        /// <param name="hashSizeInBits"></param>
        public static HashAlgorithmName GetRSAHashAlgorithmName(uint hashSizeInBits)
        {
            if (hashSizeInBits <= 160)
            {
                return HashAlgorithmName.SHA1;
            }
            else if (hashSizeInBits <= 256)
            {
                return HashAlgorithmName.SHA256;
            }
            else if (hashSizeInBits <= 384)
            {
                return HashAlgorithmName.SHA384;
            }
            else
            {
                return HashAlgorithmName.SHA512;
            }
        }

        /// <summary>
        /// Create secure temporary passcode.
        /// </summary>
        internal static string GeneratePasscode()
        {
            const int kLength = 18;
            byte[] tokenBuffer = Nonce.CreateRandomNonceData(kLength);
            return Convert.ToBase64String(tokenBuffer);
        }

    }
}
