/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * License); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * AS IS BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Copyright (c) 2021, OPEN AI LAB
 * Author: bhu@openailab.com
 */

#include "batchnorm_param.h"

#include "graph/tensor.h"
#include "graph/node.h"
#include "graph/graph.h"
#include "module/module.h"
#include "utility/sys_port.h"

static int infer_shape(struct node* node)
{
    struct graph* ir_graph = node->graph;
    struct tensor* input = get_ir_graph_tensor(ir_graph, node->input_tensors[0]);
    struct tensor* output = get_ir_graph_tensor(ir_graph, node->output_tensors[0]);

    return set_ir_tensor_shape(output, input->dims, input->dim_num);
}

static int init_op(struct op* op)
{
    batchnorm_param_t* batchnorm_param = (batchnorm_param_t*)sys_malloc(sizeof(batchnorm_param_t));

    if (batchnorm_param == NULL)
    {
        return -1;
    }

    batchnorm_param->eps = 1e-5f;
    batchnorm_param->rescale_factor = 1.0f;
    batchnorm_param->caffe_flavor = 0;

    op->param_mem = batchnorm_param;
    op->param_size = sizeof(batchnorm_param_t);
    op->same_shape = 0;
    op->infer_shape = infer_shape;

    return 0;
}

static void release_op(struct op* op)
{
    sys_free(op->param_mem);
}

int register_batchnorm_op()
{
    struct method m;

    m.version = 1;
    m.init = init_op;
    m.release = release_op;

    return register_op(OP_BATCHNORM, OP_BATCHNORM_NAME, &m);
}

int unregister_batchnorm_op()
{
    return unregister_op(OP_BATCHNORM, 1);
}
