/*
 * Copyright (c) 2015-2022 Morwenn
 * SPDX-License-Identifier: MIT
 */
#ifndef CPPSORT_TESTSUITE_DISTRIBUTIONS_H_
#define CPPSORT_TESTSUITE_DISTRIBUTIONS_H_

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <random>
#include <cpp-sort/detail/bitops.h>
#include "random.h"

namespace dist
{
    template<typename Derived>
    struct distribution
    {
        template<typename OutputIterator>
        using fptr_t = void(*)(OutputIterator, long long int);

        template<typename OutputIterator>
        operator fptr_t<OutputIterator>() const
        {
            return [](OutputIterator out, long long int size) {
                return Derived{}(out, size);
            };
        }

        // Make it easier to specify explicit template parameters
        template<typename T=long long int, typename OutputIterator>
        auto call(OutputIterator out, long long int size) const
            -> decltype(auto)
        {
            return static_cast<const Derived&>(*this).template operator()<T>(out, size);
        }

        template<typename T=long long int, typename OutputIterator>
        auto call(OutputIterator out, long long int size, long long int start) const
            -> decltype(auto)
        {
            return static_cast<const Derived&>(*this).template operator()<T>(out, size, start);
        }
    };

    struct shuffled:
        distribution<shuffled>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size, long long int start=0ll) const
            -> void
        {
            hasard::fill_with_shuffle<T>(out, size, start, hasard::bit_gen());
        }
    };

    struct shuffled_16_values:
        distribution<shuffled_16_values>
    {
        static constexpr auto mod_16(long long int value)
            -> long long int
        {
            return value % 16;
        }

        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            hasard::fill_with_shuffle<T>(out, size, 0, hasard::bit_gen(), &mod_16);
        }
    };

    struct all_equal:
        distribution<all_equal>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            for (long long int i = 0 ; i < size ; ++i) {
                *out++ = static_cast<T>(0);
            }
        }
    };

    struct ascending:
        distribution<ascending>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            for (long long int i = 0 ; i < size ; ++i) {
                *out++ = static_cast<T>(i);
            }
        }
    };

    struct descending:
        distribution<descending>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            while (size--) {
                *out++ = static_cast<T>(size);
            }
        }
    };

    struct ascending_duplicates:
        distribution<ascending_duplicates>
    {
        // Ascending (sorted) distribution with series of 10
        // times the same integer value, used to test specific
        // algorithms against inputs with duplicate values

        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            for (long long int i = 0 ; i < size ; ++i) {
                *out++ = static_cast<T>(i / 10);
            }
        }
    };

    struct pipe_organ:
        distribution<pipe_organ>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            for (long long int i = 0 ; i < size / 2 ; ++i) {
                *out++ = static_cast<T>(i);
            }
            for (long long int i = size / 2 ; i < size ; ++i) {
                *out++ = static_cast<T>(size - i);
            }
        }
    };

    struct push_front:
        distribution<push_front>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            if (size > 0) {
                for (long long int i = 0 ; i < size - 1 ; ++i) {
                    *out++ = static_cast<T>(i);
                }
                *out = static_cast<T>(0);
            }
        }
    };

    struct push_middle:
        distribution<push_middle>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            if (size > 0) {
                for (long long int i = 0 ; i < size ; ++i) {
                    if (i != size / 2) {
                        *out++ = static_cast<T>(i);
                    }
                }
                *out = static_cast<T>(size / 2);
            }
        }
    };

    struct ascending_sawtooth:
        distribution<ascending_sawtooth>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            auto limit = static_cast<long long int>(size / cppsort::detail::log2(size) * 0.9);
            for (long long int i = 0 ; i < size ; ++i) {
                *out++ = static_cast<T>(i % limit);
            }
        }
    };

    struct descending_sawtooth:
        distribution<descending_sawtooth>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            auto limit = static_cast<long long int>(size / cppsort::detail::log2(size) * 0.9);
            while (size--) {
                *out++ = static_cast<T>(size % limit);
            }
        }
    };

    struct alternating:
        distribution<alternating>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            for (long long int i = 0 ; i < size ; ++i) {
                *out++ = static_cast<T>((i % 2) ? i : -i);
            }
        }
    };

    struct descending_plateau:
        distribution<descending_plateau>
    {
        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            long long int i = size;
            while (i > 2 * size / 3) {
                *out++ = static_cast<T>(i);
                --i;
            }
            while (i > size / 3) {
                *out++ = static_cast<T>(size / 2);
                --i;
            }
            while (i > 0) {
                *out++ = static_cast<T>(i);
                --i;
            }
        }
    };

    struct inversions:
        distribution<inversions>
    {
        // Percentage of chances that an "out-of-place" value
        // is produced for each position, the goal is to test
        // Inv-adaptive algorithms
        double factor;

        constexpr explicit inversions(double factor) noexcept:
            factor(factor)
        {}

        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            // Generate a percentage of error
            std::uniform_real_distribution<double> percent_dis(0.0, 1.0);
            // Generate a random value
            std::uniform_int_distribution<long long int> value_dis(0, size - 1);

            for (long long int i = 0; i < size; ++i) {
                if (percent_dis(hasard::engine()) < factor) {
                    *out++ = static_cast<T>(value_dis(hasard::engine()));
                } else {
                    *out++ = static_cast<T>(i);
                }
            }
        }
    };

    struct median_of_3_killer:
        distribution<median_of_3_killer>
    {
        // This distribution comes from *A Killer Adversary for Quicksort*
        // by M. D. McIlroy, and is supposed to trick several quicksort
        // implementations with common pivot selection methods go quadratic

        template<typename T=long long int, typename OutputIterator>
        auto operator()(OutputIterator out, long long int size) const
            -> void
        {
            long long int j = size / 2;
            for (long long int i = 1 ; i < j + 1 ; ++i) {
                if (i % 2 != 0) {
                    *out++ = static_cast<T>(i);
                } else {
                    *out++ = static_cast<T>(j + i - 1);
                }
            }
            for (long long int i = 1 ; i < j + 1 ; ++i) {
                *out++ = static_cast<T>(2 * i);
            }
        }
    };
}

#endif // CPPSORT_TESTSUITE_DISTRIBUTIONS_H_
