//
// thread.cpp
//
//      Copyright (c) Microsoft Corporation. All rights reserved.
//
// The _beginthread(), _beginthreadex(), _endthread(), and _endthreadex().
//
// There are several key differences in behavior between _beginthread() and
// _beginthreadex():
//
//  * _beginthreadex() takes three additional parameters, which are passed on to
//    CreateThread():  a security descriptor for the new thread, the initial
//    thread state (running or asleep), and an optional out parameter to which
//    the thread id of the newly created thread will be stored.
//
//  * The procedure passed to _beginthread() must be __cdecl and has no return
//    code.  The routine passed to _beginthreadex() must be __stdcall and must
//    return a return code, which will be used as the thread exit code.
//    Likewise, _endthread() takes no parameter and always returns a thread exit
//    code of 0 if the thread exits without error, whereas _endthreadex() takes
//    an exit code.
//
//  * _endthread() calls CloseHandle() on the handle returned from CreateThread().
//    Note that this means that a caller should not use this handle, since it is
//    possible that the thread will have terminated and the handle will have been
//    closed by the time that _beginthread() returns.
//
//    _endthreadex() does not call CloseHandle() to close the handle:  the caller
//    of _beginthreadex() is required to close the handle.
//
//  * _beginthread() returns -1 on failure.  _beginthreadex() returns zero on
//    failure (just as CreateThread() does).
//
#include <corecrt_internal.h>
#include <process.h>
//#include <roapi.h>

// In some compilation models, the compiler is able to detect that the return
// statement at the end of thread_start is unreachable.  We cannot suppress the
// warning locally because it is a backend warning.
#pragma warning(disable: 4702) // unreachable code
#pragma warning(disable: 4984) // 'if constexpr' is a C++17 language extension


namespace
{
    struct thread_parameter_free_policy
    {
        void operator()(__acrt_thread_parameter* const parameter) throw()
        {
            if (!parameter)
            {
                return;
            }

            if (parameter->_thread_handle)
            {
                (void)ZwClose(parameter->_thread_handle);
            }

            _free_crt(parameter);
        }
    };

    using unique_thread_parameter = __crt_unique_heap_ptr<
        __acrt_thread_parameter,
        thread_parameter_free_policy>;
}

template <typename ThreadProcedure, bool Ex>
static void WINAPI thread_start(void* const parameter) throw()
{
    if (!parameter)
    {
        (void)PsTerminateSystemThread(STATUS_INVALID_PARAMETER);
    }

    __acrt_thread_parameter* const context = static_cast<__acrt_thread_parameter*>(parameter);

    __acrt_getptd()->_beginthread_context = context;

    __try
    {
        ThreadProcedure const procedure = reinterpret_cast<ThreadProcedure>(context->_procedure);
        if constexpr (Ex)
        {
            _endthreadex(procedure(context->_context));
        }
        else
        {
            procedure(context->_context);
            _endthreadex(0);
        }
    }
    __except (_seh_filter_dll(GetExceptionCode(), GetExceptionInformation()))
    {
        // Execution should never reach here:
        _exit(GetExceptionCode());
    }
}



static __acrt_thread_parameter* __cdecl create_thread_parameter(
    void* const procedure,
    void* const context
    ) throw()
{
    unique_thread_parameter parameter(_calloc_crt_t(__acrt_thread_parameter, 1).detach());
    if (!parameter)
    {
        return nullptr;
    }

    parameter.get()->_procedure = procedure;
    parameter.get()->_context   = context;

    return parameter.detach();
}

extern "C" uintptr_t __cdecl _beginthread(
    _beginthread_proc_type const procedure,
    unsigned int           const stack_size,
    void*                  const context
    )
{
    // kernel thread can't set this.
    UNREFERENCED_PARAMETER(stack_size);

    _VALIDATE_RETURN(procedure != nullptr, EINVAL, reinterpret_cast<uintptr_t>(INVALID_HANDLE_VALUE));

    unique_thread_parameter parameter(create_thread_parameter(procedure, context));
    if (!parameter)
    {
        return reinterpret_cast<uintptr_t>(INVALID_HANDLE_VALUE);
    }

    // We create the new thread in a suspended state so that we can update
    // the parameter structure with the thread handle.  The newly created
    // thread is responsible for closing this handle.

    OBJECT_ATTRIBUTES object_attributes;
    InitializeObjectAttributes(&object_attributes, nullptr, OBJ_KERNEL_HANDLE, nullptr, nullptr);

    HANDLE thread_handle = nullptr;
    long status = PsCreateSystemThread(&thread_handle, THREAD_ALL_ACCESS, &object_attributes,
        nullptr, nullptr, thread_start<_beginthread_proc_type, false>, parameter.get());
    if (!NT_SUCCESS(status))
    {
        __acrt_errno_map_os_error(status);
        return reinterpret_cast<uintptr_t>(INVALID_HANDLE_VALUE);
    }

    parameter.get()->_thread_handle = thread_handle;

    // Now we can start the thread...
    //status = ZwResumeThread(thread_handle, nullptr);
    //if (!NT_SUCCESS(status))
    //{
    //    __acrt_errno_map_os_error(status);
    //    return reinterpret_cast<uintptr_t>(INVALID_HANDLE_VALUE);
    //}

    // If we successfully created the thread, the thread now owns its parameter:
    parameter.detach();

    return reinterpret_cast<uintptr_t>(thread_handle);
}

extern "C" uintptr_t __cdecl _beginthreadex(
    void*                    const security_descriptor,
    unsigned int             const stack_size,
    _beginthreadex_proc_type const procedure,
    void*                    const context,
    unsigned int             const creation_flags,
    unsigned int*            const thread_id_result
    )
{
    // kernel thread can't set these.
    UNREFERENCED_PARAMETER(stack_size);
    UNREFERENCED_PARAMETER(creation_flags);

    _VALIDATE_RETURN(procedure != nullptr, EINVAL, 0);

    unique_thread_parameter parameter(create_thread_parameter(procedure, context));
    if (!parameter)
    {
        return 0;
    }

    OBJECT_ATTRIBUTES object_attributes;
    InitializeObjectAttributes(&object_attributes, nullptr, OBJ_KERNEL_HANDLE, nullptr, security_descriptor);

    CLIENT_ID client_id{};
    HANDLE thread_handle = nullptr;
    const long status = PsCreateSystemThread(&thread_handle, THREAD_ALL_ACCESS, &object_attributes,
        nullptr, &client_id, thread_start<_beginthreadex_proc_type, true>, parameter.get());
    if (!NT_SUCCESS(status)) {
        __acrt_errno_map_os_error(status);
        return 0;
    }

    if (thread_id_result)
    {
        *thread_id_result = static_cast<unsigned>(reinterpret_cast<size_t>(client_id.UniqueThread));
    }

    // If we successfully created the thread, the thread now owns its parameter:
    parameter.detach();

    return reinterpret_cast<uintptr_t>(thread_handle);
}



static void __cdecl common_end_thread(unsigned int const return_code) throw()
{
    __acrt_ptd* const ptd = __acrt_getptd_noexit();
    if (!ptd)
    {
        (void)PsTerminateSystemThread(return_code);
    }

    __acrt_thread_parameter* const parameter = ptd->_beginthread_context;
    if (!parameter)
    {
        (void)PsTerminateSystemThread(return_code);
    }

    if (parameter->_thread_handle != INVALID_HANDLE_VALUE && parameter->_thread_handle != nullptr)
    {
        (void)ZwClose(parameter->_thread_handle);
    }

    (void)PsTerminateSystemThread(return_code);
}

extern "C" void __cdecl _endthread()
{
    return common_end_thread(0);
}

extern "C" void __cdecl _endthreadex(unsigned int const return_code)
{
    return common_end_thread(return_code);
}
