// MIT License
//
// Copyright (c) 2017-2022 MessageKit
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

import UIKit

extension UIView {
  internal func fillSuperview() {
    guard let superview = superview else {
      return
    }
    translatesAutoresizingMaskIntoConstraints = false

    let constraints: [NSLayoutConstraint] = [
      leftAnchor.constraint(equalTo: superview.leftAnchor),
      rightAnchor.constraint(equalTo: superview.rightAnchor),
      topAnchor.constraint(equalTo: superview.topAnchor),
      bottomAnchor.constraint(equalTo: superview.bottomAnchor),
    ]
    NSLayoutConstraint.activate(constraints)
  }

  internal func centerInSuperview() {
    guard let superview = superview else {
      return
    }
    translatesAutoresizingMaskIntoConstraints = false
    let constraints: [NSLayoutConstraint] = [
      centerXAnchor.constraint(equalTo: superview.centerXAnchor),
      centerYAnchor.constraint(equalTo: superview.centerYAnchor),
    ]
    NSLayoutConstraint.activate(constraints)
  }

  internal func constraint(equalTo size: CGSize) {
    guard superview != nil else { return }
    translatesAutoresizingMaskIntoConstraints = false
    let constraints: [NSLayoutConstraint] = [
      widthAnchor.constraint(equalToConstant: size.width),
      heightAnchor.constraint(equalToConstant: size.height),
    ]
    NSLayoutConstraint.activate(constraints)
  }

  @discardableResult
  internal func addConstraints(
    _ top: NSLayoutYAxisAnchor? = nil,
    left: NSLayoutXAxisAnchor? = nil,
    bottom: NSLayoutYAxisAnchor? = nil,
    right: NSLayoutXAxisAnchor? = nil,
    centerY: NSLayoutYAxisAnchor? = nil,
    centerX: NSLayoutXAxisAnchor? = nil,
    topConstant: CGFloat = 0,
    leftConstant: CGFloat = 0,
    bottomConstant: CGFloat = 0,
    rightConstant: CGFloat = 0,
    centerYConstant: CGFloat = 0,
    centerXConstant: CGFloat = 0,
    widthConstant: CGFloat = 0,
    heightConstant: CGFloat = 0) -> [NSLayoutConstraint]
  {
    if superview == nil {
      return []
    }
    translatesAutoresizingMaskIntoConstraints = false

    var constraints = [NSLayoutConstraint]()

    if let top = top {
      let constraint = topAnchor.constraint(equalTo: top, constant: topConstant)
      constraint.identifier = "top"
      constraints.append(constraint)
    }

    if let left = left {
      let constraint = leftAnchor.constraint(equalTo: left, constant: leftConstant)
      constraint.identifier = "left"
      constraints.append(constraint)
    }

    if let bottom = bottom {
      let constraint = bottomAnchor.constraint(equalTo: bottom, constant: -bottomConstant)
      constraint.identifier = "bottom"
      constraints.append(constraint)
    }

    if let right = right {
      let constraint = rightAnchor.constraint(equalTo: right, constant: -rightConstant)
      constraint.identifier = "right"
      constraints.append(constraint)
    }

    if let centerY = centerY {
      let constraint = centerYAnchor.constraint(equalTo: centerY, constant: centerYConstant)
      constraint.identifier = "centerY"
      constraints.append(constraint)
    }

    if let centerX = centerX {
      let constraint = centerXAnchor.constraint(equalTo: centerX, constant: centerXConstant)
      constraint.identifier = "centerX"
      constraints.append(constraint)
    }

    if widthConstant > 0 {
      let constraint = widthAnchor.constraint(equalToConstant: widthConstant)
      constraint.identifier = "width"
      constraints.append(constraint)
    }

    if heightConstant > 0 {
      let constraint = heightAnchor.constraint(equalToConstant: heightConstant)
      constraint.identifier = "height"
      constraints.append(constraint)
    }

    NSLayoutConstraint.activate(constraints)
    return constraints
  }

  internal func addSubviews(_ subviews: UIView...) {
    subviews.forEach { addSubview($0) }
  }
}
